/*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*
 ** Copyright (c) 1992, UCAR
 ** University Corporation for Atmospheric Research(UCAR)
 ** National Center for Atmospheric Research(NCAR)
 ** Research Applications Program(RAP)
 ** P.O.Box 3000, Boulder, Colorado, 80307-3000, USA
 ** All rights reserved. Licenced use only.
 ** Do not copy or distribute without authorization
 ** 1993/3/2 13:59:21
 *=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*/
/**********************************************************************
 * ucompress.c
 *
 * Compression utilities
 *
 **********************************************************************/

#include "./umisc.h"
#include "bigend.h"
#include "./mdv_file.h"
#include "./mdv_print.h"
#include <memory.h>
#include <netinet/in.h>

#define RL8_FLAG 0xfe0103fdU
#define MDV_RL8_FLAG 0xfe0104fdU   /* This flag was used by early MDV */
                                   /* files.  Now, MDV files use the */
                                   /* RL8_FLAG, but we must keep this */
                                   /* flag also so the early MDV data can */
                                   /* still be decoded.  This is only used */
                                   /* in the check at the beginning of the */
                                   /* decode routine. */

/**********************************************************************
 * uRLEncode8() - performs run-length encoding on byte data which uses
 *                all 8 bits
 *
 * In the coded data, the first 20 bytes are as follows:
 *
 * (si32) Magic cookie - RL8_FLAG
 * (si32) key
 * (si32) nbytes_array
 * (si32) nbytes_full
 * (si32) nbytes_coded
 *
 * The coded data follows these 20 bytes. The coded data is padded out
 * to end on a 4-byte boundary.
 *
 * The memory for the encoded array is allocated by this routine.
 *
 * Returns pointer to the encoded array. The number of bytes in the
 * encodeded data array is returned via nbytes_array.
 *
 * utility routine
 *
 * N. Rehak RAP, NCAR, Boulder CO 5 Oct 1990
 *
 * hacked from uRLEncode.c by Mike Dixon
 *
 **********************************************************************/

ui08 *uRLEncode8(ui08 *full_data,
		 ui32 nbytes_full,
		 ui32 key,
		 ui32 *nbytes_array)
     
{

  register ui08 byteval;
  register ui08 *coded_data;
  register ui08 *fdata, *cdata, *end;

  register ui32 runcount;

  si32 *lptr;

  ui32 nbytes_coded;
  ui32 nbytes_extra;
  ui32 nbytes_alloc;
  ui32 nbytes_unpadded;

  int i;

  /*
   * full_data is original array
   * fdata is pointer into original array
   * cdata is pointer into coded array
   * end is pointer to last byte in original array
   */

  /*
   * initial allocation of encoded array, the size of the original array
   * plus the extra bytes at the start for the nbyte values, plus enough
   * bytes to pass a word boundary. Twice this will be sufficient for the
   * worst case in which the pattern is
   *    key non-key key non-key key ...
   * which expands to twice the original size.
   */

  nbytes_extra = RL8_NBYTES_EXTRA;
  nbytes_alloc = 2 * (nbytes_full + nbytes_extra + NBYTES_WORD);
  coded_data = calloc (nbytes_alloc, 1);

  /*
   * set the number of bytes in the full data, and the pointer to the
   * number of encoded bytes
   */

  /*
   * set pointers to data arrays
   */

  fdata = full_data;
  cdata = coded_data + nbytes_extra;
  
  /*
   * set pointer to last data byte
   */

  end = fdata + nbytes_full;

  if (full_data != NULL) {

    while (fdata < end) {

      /*
       * get byte value
       */

      byteval = *fdata;
      runcount = 1;
      fdata++;
      
      while ((fdata < end) &&
	     (runcount < 255) &&
	     (*fdata == byteval)) {
	
	/*
	 * count up adjacent bytes of same value
	 */

	fdata++;
	runcount++;
	
      }

      if (runcount <= 3 && byteval != key) {
	
	/*
	 * count <= 3, and byteval is not key,
	 * so store as single bytes because there is no
	 * advantage to RLE
	 */
	
	for (i = 0; i < runcount; i++) {
	  *cdata = byteval;
	  cdata++;
	}
	
      } else {

	/*
	 *  count > 3, store as RLE indicator, count then byte value
	 */

	*cdata = key;
	*(cdata + 1) = runcount;
	*(cdata + 2) = byteval;
	cdata += 3;

      }

    } /* while (fdata < end) */

    /*
     * compute the number of bytes in the encoded data, including the 
     * leading 8 bytes and the padding to go to a word boundary
     */

    nbytes_coded = cdata - coded_data - nbytes_extra;
    nbytes_unpadded = nbytes_coded + nbytes_extra;
    *nbytes_array =
      ((ui32) ((nbytes_unpadded - 1) / NBYTES_WORD) + 1) * NBYTES_WORD;

    /*
     * realloc the coded_data array
     */

    coded_data = (ui08 *) realloc
      ((char *) coded_data, (ui32) *nbytes_array);

    /*
     * set the bytes counts
     */

    lptr = (si32 *) coded_data;

    *lptr = BE_from_si32(RL8_FLAG);
    lptr++;
    *lptr = BE_from_si32(key);
    lptr++;
    *lptr = BE_from_si32(*nbytes_array);
    lptr++;
    *lptr = BE_from_si32(nbytes_full);
    lptr++;
    *lptr = BE_from_si32(nbytes_coded);

    return (coded_data);

  } else {

    return ((ui08 *) NULL);

  } /* if (full_data != NULL) */

}

/**********************************************************************
 * uRLDecode8() - performs run-length decoding on byte data which was
 *                compressed using uRLEncode8
 *
 * Returns the full data array. The size of the array is passed back via
 * nbytes_full.
 *
 * utility routine
 *
 * N. Rehak RAP, NCAR, Boulder CO 6 October 1992
 *
 * hacked from "uRLDecode.c" by Mike Dixon
 *
 **********************************************************************/

ui08 *uRLDecode8(ui08 *coded_data,
  ui32 *nbytes_full)
     
{

  register int runcount;
  int swap;
  
  si32 *lptr;
  si32 compress_flag;
  si32 key;

  ui32 nbytes_coded;
  ui32 nbytes_extra;

  register ui08 byteval;
  ui08 *full_data;
  register ui08 *end;
  register ui08 *fdata, *cdata;

  if (coded_data != NULL) {

    lptr = (si32 *) coded_data;
    compress_flag = *lptr;
    lptr++;

    if (compress_flag == RL8_FLAG ||
	compress_flag == MDV_RL8_FLAG) {
      swap = FALSE;
    } else if (BE_to_si32(compress_flag) == RL8_FLAG ||
	       BE_to_si32(compress_flag) == MDV_RL8_FLAG) {
      swap = TRUE;
    } else {
      /*
       * not compressed with this alg, return NULL
       */
      return ((ui08 *) NULL);
    }

    /*
     * get number of bytes for the coded and full data
     */

    if (swap) {
      key = BE_to_si32(*lptr);
    } else {
      key = *lptr;
    }

    lptr += 2;
    if (swap) {
      *nbytes_full = BE_to_si32(*lptr);
    } else {
      *nbytes_full = *lptr;
    }

    lptr++;
    if (swap) {
      nbytes_coded = BE_to_si32(*lptr);
    } else {
      nbytes_coded = *lptr;
    }

    nbytes_extra = RL8_NBYTES_EXTRA;

    /*
     * get space for full data
     */

    full_data = calloc(*nbytes_full, 1);
    
    fdata = full_data;
    cdata = coded_data + nbytes_extra;

    end = cdata + nbytes_coded;

    while (cdata < end) {

      byteval = *cdata;

      if (byteval == key) {

	/*
	 * if RLE flag value, the next byte contains the run count and
	 * the byte after contains the byte value
	 */

	cdata ++;
	runcount = *cdata;
	cdata++;
	byteval  = *cdata;

	/*
	 * set runcount values
	 */

	memset((char *) fdata, (int) byteval, (int) runcount);
	fdata += runcount;

      } else {

	/*
	 * if not RLE flag value, set single byte
	 */

	*fdata = byteval;
	fdata++;

      } /* if (byteval == key) */

      cdata++;

    } /* while (cdata < end) */

    return (full_data);

  } else {

    return ((ui08 *) NULL);

  }

}

/**********************************************************************
 * uRLCheck()
 *
 * Checks for compression type, and number of bytes in compressed
 * array.
 *
 * Returns 0 on success, -1 on error
 *
 * utility routine
 *
 * Mike Dixon, RAP, NCAR, Boulder CO
 *
 * Feb 1994
 *
 **********************************************************************/

int uRLCheck(ui08 *coded_data,
	     ui32 nbytes_passed,
	     int *eight_bit,
	     ui32 *nbytes_compressed)
     
{

  ui32 *lptr;
  ui32 first_int, third_int;
  ui32 compress_flag;
  
  if (coded_data == NULL)
    return (-1);
  
  if (nbytes_passed < sizeof(ui32))
    return (-1);
  
  lptr = (ui32 *) coded_data;
  first_int = BE_to_si32(*lptr);

  compress_flag = first_int;

  if (compress_flag == RL8_FLAG ||
      compress_flag == MDV_RL8_FLAG) {
    
    *eight_bit = TRUE;

    if (nbytes_passed < 3 * sizeof(ui32))
      return (-1);
  
    lptr += 2;
    third_int = BE_to_si32(*lptr);
    *nbytes_compressed = third_int;

  } else {

    *eight_bit = FALSE;
    *nbytes_compressed = first_int;

  }

  return (0);

}

/**********************************************************************
 * uRLEncode() - performs run-length encoding on byte data which uses
 *               only the lower 7 bits
 *
 * In the coded data, the first 12 bytes are as follows:
 *
 * (si32) nbytes_array, (si32) nbytes_full, (si32) nbytes_coded.
 *
 * The coded data follows these 12 bytes. The coded data is padded out
 * to end on a 4-byte boundary.
 *
 * The memory for the encoded array is allocated by this routine.
 *
 * Returns pointer to the encoded array. The number of bytes in the
 * encodeded data array is returned via nbytes_array.
 *
 * utility routine
 *
 * Mike Dixon RAP, NCAR, Boulder CO November 1990
 *
 **********************************************************************/

ui08 *uRLEncode(ui08 *full_data, ui32 nbytes_full,
		ui32 *nbytes_array)
     
{

  register ui08 byteval;
  register ui08 *coded_data;
  register ui08 *fdata, *cdata, *end;

  register ui32 runcount;

  ui32 nbytes_coded;
  ui32 nbytes_extra;
  ui32 nbytes_unpadded;

  /*
   * full_data is original array
   * fdata is pointer into original array
   * cdata is pointer into coded array
   * end is pointer to last byte in original array
   */

  /*
   * initial allocation of encoded array, the size of the original array
   * plus the extra bytes at the start for the nbyte values, plus enough
   * bytes to pass a word boundary. This will be sufficient for the
   * worst case in which there is no compression
   */

  nbytes_extra = RL7_NBYTES_EXTRA;
  
  coded_data = (ui08 *) malloc
    ((ui32) (nbytes_full + nbytes_extra + NBYTES_WORD));

  /*
   * set the number of bytes in the full data, and the pointer to the
   * number of encoded bytes
   */

  /*
   * set pointers to data arrays
   */

  fdata = full_data;
  cdata = coded_data + nbytes_extra;
  
  /*
   * set pointer to last data byte
   */

  end = full_data + nbytes_full;

  if (full_data != NULL) {

    while (fdata < end) {

      /*
       * get byte value
       */

      byteval = *fdata;
      fdata++;

      /*
       * return with NULL pointer if data exceeds 127
       */

      if (byteval > 127) {

	fprintf(stderr, "ERROR - uRLEncode\n");
	fprintf(stderr, "Byte value exceeds 127.\n");
	free(coded_data);
	return ((ui08 *) NULL);

      } /* if (byteval .... */
  
      runcount = 1;
      
      while ((fdata < end) &&
	     (runcount < 127) &&
	     (*fdata == byteval)) {

	/*
	 * count up adjacent bytes of same value
	 */

	fdata++;
	runcount++;
	
      }

      if (runcount == 1) {

	/*
	 * count = 1, store as single byte
	 */

	*cdata = byteval;
	cdata++;

      } else {

	/*
	 *  count > 1, store as count then byte value
	 */

	*cdata = 0x80 | runcount;
	*(cdata + 1) = byteval;
	cdata += 2;

      }

    } /* while (fdata < end) */

    /*
     * compute the number of bytes in the encoded data, including the 
     * leading 8 bytes and the padding to go to a word boundary
     */

    nbytes_coded = cdata - coded_data - nbytes_extra;
    nbytes_unpadded = nbytes_coded + nbytes_extra;
    *nbytes_array =
      ((ui32) ((nbytes_unpadded - 1) / NBYTES_WORD) + 1) * NBYTES_WORD;

    /*
     * realloc the coded_data array
     */

    coded_data = (ui08 *) realloc
      ((char *) coded_data, (ui32) *nbytes_array);

    /*
     * set the bytes counts
     */

    *((si32 *) coded_data) = BE_from_si32(*nbytes_array);
    *((si32 *) coded_data + 1) = BE_from_si32(nbytes_full);
    *((si32 *) coded_data + 2) = BE_from_si32(nbytes_coded);

    return (coded_data);

  } else {

    return ((ui08 *) NULL);

  } /* if (full_data != NULL) */

}

/**********************************************************************
 * uRLDecode() - performs run-length decoding on byte data which was
 *               compressed using uRLEncode
 *
 * Returns the full data array. The size of the array is passed back via
 * nbytes_full.
 *
 * utility routine
 *
 * Mike Dixon RAP, NCAR, Boulder CO November 1990
 *
 **********************************************************************/

ui08 *uRLDecode(ui08 *coded_data, ui32 *nbytes_full)
     
{

  register int runcount;
  ui32 nbytes_coded;
  ui32 nbytes_extra;

  register ui08 byteval;
  ui08 *full_data;
  register ui08 *end;
  register ui08 *fdata, *cdata;

  if (coded_data != NULL) {

    /*
     * get number of bytes for the coded and full data
     */

    *nbytes_full = BE_to_si32(*((si32 *) coded_data + 1));
    nbytes_coded = BE_to_si32(*((si32 *) coded_data + 2));
    nbytes_extra = RL7_NBYTES_EXTRA;

    /*
     * get space for full data
     */

    full_data = (ui08 *) malloc(*nbytes_full);
    
    fdata = full_data;
    cdata = coded_data + nbytes_extra;

    end = cdata + nbytes_coded;

    while (cdata < end) {

      byteval = *cdata;

      if ((byteval & 0x80) == 0x80) {

	/*
	 * if most significant bit set, mask off lower 7 bits and
	 * use as the count on the next byte value
	 */

	runcount = byteval & 0x7f;
	cdata++;
	byteval  = *cdata;

	/*
	 * set runcount values
	 */

	memset((char *) fdata, (int) byteval, (int) runcount);
	fdata += runcount;

      } else {

	/*
	 * if most significant bit not set, set single byte
	 */

	*fdata = byteval;
	fdata++;

      } /* if ((byteval & 0x80) == 0x80) */

      cdata++;

    } /* while (cdata < end) */

    return (full_data);

  } else {

    return ((ui08 *) NULL);

  }

}

static int BigEnd = 1;

/***********************************************
 * BE_reverse()
 *
 * Reverses the sense of this library. Therefore,
 * is called once, SmallEndian values are set.
 * If called twice, goes back to BigEndian.
 */

void BE_reverse(void)

{
  BigEnd = !BigEnd;
}

/************************************************
 * Return 1 if host is big_endian and 0 otherwise
 *
 * For debugging, if FORCE_SWAP is set, this routine will
 * always return FALSE, forcing a swap.
 */

int
BE_is_big_endian()
{
  
#ifdef FORCE_SWAP

  return (0);

#else

  union 
    {
      ui16    d;
      ui08     bytes[2];
    }
  short_int;

  short_int.d = 1;
  if (short_int.bytes[1] != 0)
    {
    printf("\n Reading On a Big Endian machine");
    return (BigEnd);
    }
  else
    {
    printf("\n Reading On a Little Endian machine");
    return (!BigEnd);
    }
#endif

}

/**********************************************************************
 * BE_swap_array_32()
 *
 * Performs an in-place 32-bit word byte swap, if necessary, to produce
 * BE representation from machine representation, or vice-versa.
 *
 * Array must be aligned.
 *
 * Returns the number of bytes converted.
 *
 */

si32
BE_swap_array_32(void *array, ui32 nbytes)
     
{

  static int big_endian;
  static int one_time = 0;
  ui32 i, l, nlongs;
  ui32 *this_long;
  ui32 *array32 = array;

  /* check for little or big endian */
  if (one_time == 0) {
    big_endian = BE_is_big_endian ();
    one_time = 1;
  }
  
  if (big_endian) {
    return (0);
  }
  
  nlongs = nbytes / sizeof(ui32);
  this_long = array32;
  
  for (i = 0; i < nlongs; i++) {

    l = *this_long;
    
    *this_long = (((l & 0xff000000) >> 24) |
		  ((l & 0x00ff0000) >> 8) |
		  ((l & 0x0000ff00) << 8) |
		  ((l & 0x000000ff) << 24));
    
    this_long++;

  }

  return (nbytes);

}

/**********************************************************************
 * BE_swap_array_16()
 *
 * Performs an in-place 16-bit word byte swap, if necessary, to produce
 * BE representation from machine representation, or vice-versa.
 *
 * Array must be aligned.
 *
 * Returns the number of bytes converted.
 *
 */

si32
BE_swap_array_16(void *array, ui32 nbytes)
     
{

  static int big_endian;
  static int one_time = 0;
  ui32 i, l, nlongs, nshorts;
  ui32 *this_long;
  ui16 *array16 = array;
  ui16 s;

  /* check for little or big endian */
  if (one_time == 0) {
    big_endian = BE_is_big_endian ();
    one_time = 1;
  }
  
  if (big_endian) {
    return (0);
  }
  
  nlongs = nbytes / sizeof(ui32);
  this_long = (ui32 *)array16;

  for (i = 0; i < nlongs; i++) {
    
    l = *this_long;
    
    *this_long = (((l & 0xff000000) >> 8) |
		   ((l & 0x00ff0000) << 8) |
		   ((l & 0x0000ff00) >> 8) |
		   ((l & 0x000000ff) << 8));

    this_long++;
  }
  
  if (nlongs * sizeof(ui32) != nbytes) {
    nshorts = nbytes / sizeof(ui16);
    s = array16[nshorts-1];
    array16[nshorts-1]= (((s & 0xff00) >> 8) | ((s & 0x00ff) << 8));
  }

  return (nbytes);
  
}

/*****************************************************
 * the to- and from- routines are identical,
 * and may be implemented in terms of BE_swap routines
 *
 * Later we may duplicate the code for effiency, to
 * avoid the calling overhead.
 */

/********************
 * BE_from_array_32()
 * Converts an array of 32s
 */

si32 BE_from_array_32(void *array, ui32 nbytes)

{
  return (BE_swap_array_32(array, nbytes));
}
     
/******************
 * BE_to_array_32()
 * Converts an array of 32s
 */

si32 BE_to_array_32(void *array, ui32 nbytes)

{
  return (BE_swap_array_32(array, nbytes));
}
     
/********************
 * BE_from_array_16()
 * Converts an array of 16s
 */

si32 BE_from_array_16(void *array, ui32 nbytes)

{
  return (BE_swap_array_16(array, nbytes));
}
     
/******************
 * BE_to_array_16()
 * Converts an array of 16s
 */

si32 BE_to_array_16(void *array, ui32 nbytes)

{
  return (BE_swap_array_16(array, nbytes));
}

/********************
 * BE_from_fl32()           
 * Converts a single fl32
 */                   

void BE_from_fl32(fl32 *dst, fl32 *src)                       
 
{
  memcpy( dst, src, sizeof(fl32) );
  BE_swap_array_32(dst, sizeof(fl32));   
}
 
/******************************
 *  BE_to_fl32       
 *  Converts a single fl32
 */

void BE_to_fl32(fl32 *src, fl32 *dst)

{
  memcpy( dst, src, sizeof(fl32) );
  BE_swap_array_32(dst, sizeof(fl32));
}

/********************************
 *  BE_from_si32 replaces htonl()
 *  Converts a single si32
 */

si32 BE_from_si32(si32 x)

{
  BE_swap_array_32((void *) &x, sizeof(si32));
  return (x);
}    

/******************************
 *  BE_to_si32 replaces ntohl()
 *  Converts a single si32
 */

si32 BE_to_si32(si32 x)

{
  BE_swap_array_32(&x, sizeof(si32));
  return (x);
}    

/********************************
 *  BE_from_si16 replaces htons()
 *  Converts a single si16
 */

si16 BE_from_si16(si16 x)

{
  BE_swap_array_16(&x, sizeof(si16));
  return (x);
}    

/******************************
 *  BE_to_si16 replaces ntohs()
 *  Converts a single si16
 */

si16 BE_to_si16(si16 x)

{
  BE_swap_array_16(&x, sizeof(si16));
  return (x);
}    

/********************************
 *  BE_from_ui32
 *  Converts a single ui32
 */

ui32 BE_from_ui32(ui32 x)

{
  BE_swap_array_32(&x, sizeof(ui32));
  return (x);
}    

/******************************
 *  BE_to_ui32
 *  Converts a single ui32
 */

ui32 BE_to_ui32(ui32 x)

{
  BE_swap_array_32(&x, sizeof(ui32));
  return (x);
}    

/********************************
 *  BE_from_ui16
 *  Converts a single ui16
 */

ui16 BE_from_ui16(ui16 x)

{
  BE_swap_array_16(&x, sizeof(ui16));
  return (x);
}    

/******************************
 *  BE_to_ui16
 *  Converts a single ui16
 */

ui16 BE_to_ui16(ui16 x)

{
  BE_swap_array_16(&x, sizeof(ui16));
  return (x);
}    



/*****************************************************************
 * MDV_MASTER_HEADER_FROM_BE: Converts master header from big endian
 * format to native format.  Nancy Rehak 6/97
 */

void MDV_master_header_from_BE(MDV_master_header_t *m_hdr) 
{
  /* swap header si32 and fl32's */
  BE_to_array_32((ui32 *)(&m_hdr->record_len1),
		 MDV_NUM_MASTER_HEADER_32 * sizeof(si32));

  /* swap the last record length */
  m_hdr->record_len2 = BE_to_si32(m_hdr->record_len2);

  return;
}


 
/*****************************************************************
 * MDV_FIELD_HEADER_FROM_BE: Converts field header from big endian
 * format to native format.  Nancy Rehak 6/97
 */

void MDV_field_header_from_BE(MDV_field_header_t *f_hdr)
{
  /* swap header si32 and fl32's */
  BE_to_array_32((ui32 *)(&f_hdr->record_len1),
		 MDV_NUM_FIELD_HEADER_32 * sizeof(si32));
 
  /* swap the last record length */
  f_hdr->record_len2 = BE_to_si32(f_hdr->record_len2);
 
  return;
}




/*****************************************************************
 * MDV_UNENCODED_VOLUME_FROM_BE: Converts the data in an unencoded
 * data volume from big endian format to native format.  N. Rehak 6/97
 *
 * returns MDV_SUCCESS or MDV_FAILURE
 */
 
int MDV_unencoded_volume_from_BE(void *volume_data,
				 ui32 volume_size,
				 int data_type)
{
  char *routine_name = "MDV_unencoded_volume_from_BE";
  
  switch(data_type)
  {
  case MDV_INT8 :
    /* No data swapping necessary */
    break;
    
  case MDV_INT16 :
    BE_to_array_16((ui16 *)volume_data, volume_size);
    break;
    
  case MDV_INT32 :
  case MDV_FLOAT32 :
    BE_to_array_32((ui32 *)volume_data, volume_size);
    break;
    
  default:
    fprintf(stderr,
	    "%s: Do not know how to byte swap data in %s format\n",
	    routine_name, MDV_encode2string(data_type));
    return(MDV_FAILURE);
    
  } /* endswitch - data_type */

  return(MDV_SUCCESS);
}
/*****************************************************************
 * MDV_ORDER2STRING: Convert the data order integer to an
 * ascii string.  See mdv_macros for the data order declarations.
 * --Nancy Rehak 4/96
 */

char * MDV_order2string(int order_type)
{
  switch(order_type)
  {
  case MDV_ORDER_XYZ :
    return("MDV_ORDER_XYZ");
  case MDV_ORDER_YXZ :
    return("MDV_ORDER_YXZ");
  case MDV_ORDER_XZY :
    return("MDV_ORDER_XZY");
  case MDV_ORDER_YZX :
    return("MDV_ORDER_YZX");
  case MDV_ORDER_ZXY :
    return("MDV_ORDER_ZXY");
  case MDV_ORDER_ZYX :
    return("MDV_ORDER_ZYX");
  default:
    return("Unknown Data Order");
  }
}

/*****************************************************************
 * MDV_COLLTYPE2STRING: Convert the collection type integer to an
 * ascii string.  See mdv_macros for the vertical type declarations.
 * --Rachel Ames 1/96
 */

char * MDV_colltype2string(int coll_type)
{

   switch(coll_type)  {
      case MDV_DATA_MEASURED :
        return("Measured");
      case MDV_DATA_EXTRAPOLATED :
	return("Extrapolated");
      case MDV_DATA_FORECAST :
	return("Forecast");
      case MDV_DATA_SYNTHESIS :
	return("Synthesis");
      case MDV_DATA_MIXED :
	return("Mixed");
      default:
	return("Unknown Collection Type");
   }
}

/*****************************************************************
 * MDV_ORIENT2STRING: Convert the data orientation integer to an
 * ascii string.  See mdv_macros for the orientation type declarations.
 * --Nancy Rehak 4/96
 */

char * MDV_orient2string(int orient_type)
{
  switch(orient_type)
  {
  case MDV_ORIENT_OTHER :
    return("MDV_ORIENT_OTHER");
  case MDV_ORIENT_SN_WE :
    return("MDV_ORIENT_SN_WE");
  case MDV_ORIENT_NS_WE :
    return("MDV_ORIENT_NS_WE");
  case MDV_ORIENT_SN_EW :
    return("MDV_ORIENT_SN_EW");
  case MDV_ORIENT_NS_EW :
    return("MDV_ORIENT_NS_EW");
  default:
    return("Unknown Orientation");
  }
}

/*****************************************************************
 * MDV_VERTTYPE2STRING: Convert the vertical type integer to ascii 
 * string.  See mdv_macros for the vertical type declarations.
 * --Rachel Ames 1/96
 */

char * MDV_verttype2string(int vert_type)
{

   switch(vert_type)  {
      case MDV_VERT_TYPE_SURFACE : 
        return("Surface"); 
      case MDV_VERT_TYPE_SIGMA_P :
	return("Sigma P"); 
      case MDV_VERT_TYPE_PRESSURE :
	return("Pressure (units mb)"); 
      case MDV_VERT_TYPE_Z :
	return("Constant Altitude (units KM MSL)"); 
      case MDV_VERT_TYPE_SIGMA_Z :
	return("Sigma Z"); 
      case MDV_VERT_TYPE_ETA :
	return("ETA"); 
      case MDV_VERT_TYPE_THETA :
	return("Theta"); 
      case MDV_VERT_TYPE_MIXED :
	return("Mixed"); 
      case MDV_VERT_TYPE_ELEV :
	return("Elevation Angles"); 
      case MDV_VERT_TYPE_COMPOSITE :
	return("Composite"); 
      case MDV_VERT_TYPE_CROSS_SEC :
	return("Cross Secional View"); 
      case MDV_VERT_SATELLITE_IMAGE :
	return("Satelite Image"); 
      default:
	return("Unknown Vertical Type"); 
   }
}  



