#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <memory.h>
#include <errno.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include "./bigend.h"
#include "./port_types.h"
#include "./mdv_file.h"
#include "./mdv_read.h"
#include "./os_config.h"
#include "./mdv_print.h"
#include "./ced_cdf.h"
#define MDV_READ_SUCCESSFUL             0
#define MDV_READ_OPEN_FAILURE           1
#define MDV_READ_BAD_MASTER_HDR         2
#define MDV_READ_INVALID_FIELD_NUM      3
#define MDV_READ_BAD_FIELD_HDR          4
#define MDV_READ_BAD_VLEVEL_HDR         5
#define MDV_READ_NO_VLEVEL_HDRS         6
#define MDV_READ_INVALID_CHUNK_NUM      7
#define MDV_READ_BAD_CHUNK_HDR          8
#define MDV_READ_DATA_ARRAY_TOO_SMALL   9
#define MDV_READ_DATA_ERROR  10

#ifndef BOOL_STR
#define BOOL_STR(a) (a == FALSE ? "false" : "true")
#endif

static struct files *open_files;
static struct variables *var;

// I copied these from mdv_utils.h to avoid compilation warnings about implicit function declarations. MDV_unencoded_volume_from_BE, MDV_plane_rle8_from_BE
 
/*
 * Swap routines.
 */

/*****************************************************************
 * MDV_UNENCODED_VOLUME_FROM_BE: Converts the data in an unencoded
 * data volume from big endian format to native format.  N. Rehak 6/97
 *
 * returns MDV_SUCCESS or MDV_FAILURE
 */
 
int MDV_unencoded_volume_from_BE(void *volume_data,
				 ui32 volume_size,
				 int data_type);

/*****************************************************************
 * MDV_PLANE_RLE8_FROM_BE: Converts the data in a plane of data
 * encoded in the MDV_PLANE_RLE8 format from big endian format to
 * native format.  N. Rehak 6/97
 *
 * returns MDV_SUCCESS or MDV_FAILURE
 */
 
int MDV_plane_rle8_from_BE(void *plane_data);



/*****************************************************************
 * MDV_ENCODE2STRING: Convert the encoding type integer to ascii 
 * string.  See mdv_macros for the encoding type declarations.
 * --Rachel Ames 1/96
 */

char * MDV_encode2string(int encodeing_type)
{

   switch(encodeing_type)  {
      case MDV_NATIVE :
        return("NATIVE"); 
      case MDV_INT8 :
	return("MDV_INT8 (CHAR/BYTE)"); 
      case MDV_INT16:
	return("MDV_INT16 (SHORT)"); 
      case MDV_INT32 :
	return("MDV_INT32 (INT)"); 
      case MDV_INT64 :
	return("MDV_INT64 (LONG)"); 
      case MDV_FLOAT32 :
	return("MDV_FLOAT32 (FLOAT)"); 
      case MDV_FLOAT64 :
	return("MDV_FLOAT64 (DOUBLE)"); 
      case MDV_PLANE_RLE8 :
	return("MDV_PLANE_RLE8"); 
      case MDV_ROW_RLE8 :
	return("MDV_ROW_RLE8"); 
      default:
	return("Unknown Encoding Type"); 
   }
}  
 
/***********************************************************************
 * mf_field_hdr_from_fortran:  Fills in the MDV_field_header_t structure
 *                             from the given FORTRAN arrays.
 */

void mf_field_hdr_from_fortran(si32 *field_hdr_ints,
			       fl32 *field_hdr_reals,
			       char *field_name_long,
			       char *field_name_short,
			       char *field_units,
			       char *field_transform,
			       char *field_unused_char,
			       MDV_field_header_t *field_hdr)
{
  /*
   * Load the information into the field header.
   */

  memcpy(&field_hdr->struct_id, field_hdr_ints,
	 sizeof(si32) * MDV_NUM_FIELD_HEADER_SI32);

  memcpy(&field_hdr->proj_origin_lat, field_hdr_reals,
	 sizeof(fl32) * MDV_NUM_FIELD_HEADER_FL32);

  memcpy(field_hdr->field_name_long, field_name_long, MDV_LONG_FIELD_LEN);
  
  memcpy(field_hdr->field_name, field_name_short, MDV_SHORT_FIELD_LEN);
  
  memcpy(field_hdr->units, field_units, MDV_UNITS_LEN);
  
  memcpy(field_hdr->transform, field_transform, MDV_TRANSFORM_LEN);
  
  memcpy(field_hdr->unused_char, field_unused_char, MDV_UNITS_LEN);
  
  /*
   * Make sure the strings are NULL terminated.
   */

  field_hdr->field_name_long[MDV_LONG_FIELD_LEN-1] = '\0';
  field_hdr->field_name[MDV_SHORT_FIELD_LEN-1] = '\0';
  field_hdr->units[MDV_UNITS_LEN] = '\0';
  field_hdr->transform[MDV_TRANSFORM_LEN] = '\0';
  
  /*
   * Set the record length values
   */

  field_hdr->record_len1 = sizeof(MDV_field_header_t) - 2 * sizeof(si32);
  field_hdr->record_len2 = field_hdr->record_len1;
  
  return;
}


/***********************************************************************
* read_mdv_data: Read the data for the indicated field
* from a data file and return the information in FORTRAN-usable structures.
*
* Use the following to call the subroutine:
*
*    CHARACTER*1024 FNAME
*    INTEGER        FIELD_NUM
*    CHARACTER*(ARRAY_SIZE) FIELD_DATA_ARRAY
*    INTEGER        ARRAY_SIZE
*    INTEGER        REQESTED_DATA_FORMAT
*    INTEGER        RETURN_DATA_SIZE
*    INTEGER        RETURN_STATUS
*
*   CALL open_MDV(FNAME, FIELD_NUM,FIELD_DATA_ARRAY, ARRAY_SIZE,
*                        REQUESTED_DATA_FORMAT,RETURN_DATA_SIZE,
*	                 RETURN_STATUS)
*
* This function will open the file named FNAME and read the indicated field
* data.  The data is returned in the given arrays.
*/

#ifdef LINUX
void openMDV__(int *unit, int *return_status)
#else
void openMDV_(int *unit,int *return_status)
#endif

{
  char filename[9];
  int  num1,num2,temp;
  extern struct variables *var;
  FILE *mdv_file;


  /* 
   *Construct filename. 
   */
  if(*unit < 100){
     num1 = *unit/10;
     temp = num1 * 10;
     num2 = *unit - temp;

  }
  filename[0] = 'f';
  filename[1] = 'o';
  filename[2] = 'r';
  filename[3] = 't';
  filename[4] = '.';
  filename[5] = num1 + 48;  /* convert to ascii */
  filename[6] = num2 + 48;  /* convert to ascii */
  filename[7] = '\0';  

  
  /*
   * Open the input file
   */

  if ((mdv_file = fopen(filename,"r")) == NULL)
  {
    fprintf(stderr, "Error opening input file\n");
    perror(filename);
    *return_status = MDV_READ_OPEN_FAILURE;
    return;
  }

  *return_status = MDV_SUCCESS;
  var->fptr =  mdv_file;   
 }/*read_mdv_data*/



  /* These static function declarations used to be inside MDV_GET_VOLUME_SIZE, but moved them here per http://lists.apple.com/archives/Unix-porting/2005/Jul/msg00038.html 
   ahijevyc - 20060919 */
  static ui08 *read_field_int8();
  static ui08 *read_field_plane_rle8();
  static ui08 *read_field_row_rle8();
/******************************************************************************
 * MDV_GET_VOLUME_SIZE: Allocate space for a data volume (data for all levels)
 * and read the desired volume into the buffer from the Open file. Returns the
 * size of the volume returned.  Caller is responsible for freeing up the
 * buffer when done. Caller is responsible for making sure the field_header is
 * correct and proper for the open file.
 *
 * Inputs: infile - pointer to the input file.  This is assumed to currently
 *                  be open for read.
 *         f_hdr - field header for the volume to be loaded.  This header
 *                 must contain the correct encoding_type, volume_size and
 *                 field_data_offset for the data.
 *         return_type - format in memory for the returned data volume.
 *
 * Outputs: volume_size - number of bytes in the volume of data returned.
 *
 * Returns: returns a pointer to the data volume information, byte swapped if
 *          necessary, or NULL if there is an error.  This space for this
 *          data volume is allocated by this routine and must be freed by the
 *          calling routine.
 */

void * MDV_get_volume_size(FILE *infile,
			   MDV_field_header_t *f_hdr,
			   int return_type,
			   int *volume_size)
{
  static char *routine_name = "MDV_get_volume_size";
  
  void *return_buf = NULL;              /* buffer returned to caller */
  si32 record_size;                     /* field for reading in reclen */

  /* moved these outside the function per http://lists.apple.com/archives/Unix-porting/2005/Jul/msg00038.html
   ahijevyc - 20060919
  static ui08 *read_field_int8();
  static ui08 *read_field_plane_rle8();
  static ui08 *read_field_row_rle8();
	*/
  
  /* Do some sanity checking */
  if (infile == NULL || f_hdr == NULL) return NULL;
  if (f_hdr->volume_size <= 0) return NULL;

  /*
   * Seek to the record length record before the start of this
   * field's data volume
   */

  if (fseek(infile, f_hdr->field_data_offset - sizeof(si32), SEEK_SET) != 0) 
  {
    printf("%s: Error seeking to beginning of volume data, offset = %ld\n",
	    routine_name,
	    (long int)(f_hdr->field_data_offset - sizeof(si32)));
    return (NULL);
  }
  
  /*
   * Read in the record size.
   */

  if (fread(&record_size, sizeof(record_size), 1, infile) != 1)
  {
    printf("%s: Error reading in volume record size\n",
	    routine_name);
    return(NULL);
  }
  
  /*
   * Process the field data based on the encoding type in the file
   * and the return type.  Use the encoding type in the file as
   * the outside switch statement because the reading code will
   * be the same regardless of the processing for return type.
   */

  switch (f_hdr->encoding_type)
  {

  case MDV_INT8 :

   
    /*
    return_buf = read_field_int8(infile,
				 f_hdr,
				 return_type,
				 volume_size);
    */




    return_buf = read_field_plane_rle8(infile,
				       f_hdr,
				       return_type,
				       volume_size);


    break;
    
  case MDV_PLANE_RLE8 :
    return_buf = read_field_plane_rle8(infile,
				       f_hdr,
				       return_type,
				       volume_size);
    break;
    
  case MDV_ROW_RLE8 :
    return_buf = read_field_row_rle8(infile,
				     f_hdr,
				     return_type,
				     volume_size);
    break;
    
  default:
    printf("%s: Cannot read files with field data in %s format.\n",
	    routine_name, MDV_encode2string(f_hdr->encoding_type));
    *volume_size = 0;
    return(NULL);
    break;    /* endcase - input format default */

  } /* endswitch - f_hdr->encoding_type */
    
  return(return_buf); 
}

/*****************************************************************
 * MDV_PROJ2STRING: Convert the projection type integer to 
 * an ascii string.  See mdv_macros.h for enumeration types.
 * --Rachel Ames 1/96
 */

char * MDV_proj2string(int proj_type)
{

   switch(proj_type)  {
      case MDV_PROJ_NATIVE :
        return("Native"); 
      case MDV_PROJ_LATLON :
	return("Latitude/Longitude Grid (units in degrees)"); 
      case MDV_PROJ_ARTCC :
	return("ARTCC"); 
      case MDV_PROJ_STEREOGRAPHIC :
	return("Stereographice"); 
      case MDV_PROJ_LAMBERT_CONF :
	return("Lambert Conformal"); 
      case MDV_PROJ_MERCATOR :
	return("Mercator"); 
      case MDV_PROJ_POLAR_STEREO :
	return("Polar Stereographic"); 
      case MDV_PROJ_POLAR_ST_ELLIP :
	return("Polar Sterographic Equidistant"); 
      case MDV_PROJ_CYL_EQUIDIST :
	return("Cylindrical Equidistant"); 
      case MDV_PROJ_FLAT :
	return("Flat (Cartesian) (units in KM)"); 
      case MDV_PROJ_POLAR_RADAR :
	return("Polar Radar"); 
      case MDV_PROJ_RADIAL :
	return("Radial"); 
      default:
	return("Unknown Projection Type"); 
   }

}  
 
/******************************************************************************
 * MDV_GET_VOLUME_SIZE: Allocate space for a data volume (data for all levels)
 * and read the desired volume into the buffer from the Open file. Returns the
 * size of the volume returned.  Caller is responsible for freeing up the
 * buffer when done. Caller is responsible for making sure the field_header is
 * correct and proper for the open file.
 *
 * Inputs: infile - pointer to the input file.  This is assumed to currently
 *                  be open for read.
 *         f_hdr - field header for the volume to be loaded.  This header
 *                 must contain the correct encoding_type, volume_size and
 *                 field_data_offset for the data.
 *         return_type - format in memory for the returned data volume.
 *
 * Outputs: volume_size - number of bytes in the volume of data returned.
 *
 * Returns: returns a pointer to the data volume information, byte swapped if
 *          necessary, or NULL if there is an error.  This space for this
 *          data volume is allocated by this routine and must be freed by the
 *          calling routine.
 */

/********************************************************
 * STATIC ROUTINES
 ********************************************************/

  // moved here ahijevych 2006-09-19 
  static void *read_unencoded_volume();
/********************************************************
 * READ_FIELD_INT8
 * Read a field of data in MDV_INT8 format and return it
 * in the specified format.
 *
 * Returns a pointer to the read data, or NULL on error.
 * The caller must free the space allocated.
 */

/********************************************************
 * READ_FIELD_INT8
 * Read a field of data in MDV_INT8 format and return it
 * in the specified format.
 */

static ui08 *read_field_int8(FILE *infile,
			     MDV_field_header_t *field_hdr,
			     int return_type,
			     int *volume_size_returned)
{
  static char *routine_name = "read_field_int8";
  // commented out and moved to main level before read_field_int8. ahijevych 2006-09-19 
  //static void *read_unencoded_volume();
  ui08 *input_buf;
  ui08 *return_buf;
  
  /*
   * Read the input buffer from the file.
   */

  if ((input_buf = (ui08 *)read_unencoded_volume(infile,
						 field_hdr->volume_size,
						 MDV_INT8))
      == NULL)
  {
    fprintf(stderr,
	    "%s: Error reading in input buffer\n",
	    routine_name);
    return(NULL);
  }
    
  /*
   * Convert the data to the return format, if necessary.
   */

  switch (return_type)
  {
  case MDV_INT8 :
    return_buf = (void *)input_buf;
    *volume_size_returned = field_hdr->volume_size;
    break;  /* endcase - output format MDV_INT8 */
	
  default:
    fprintf(stderr,
	    "%s: Cannot convert input %s data into %s format -- not yet implemented\n",
	    routine_name, MDV_encode2string(MDV_INT8),
	    MDV_encode2string(return_type));
    free(input_buf);
    *volume_size_returned = 0;
    return(NULL);
    break;  /* endcase - output format default */
      
  } /* endswitch - return_type */
      
  return(return_buf);
  
} /* end read_field_int8 */

// moved out here from previous place inside read_field_plane_rle8 -- ahijevyc 20060919
  static ui08 *read_plane_rle8_volume();

/********************************************************
 * READ_FIELD_PLANE_RLE8
 * Read a field of data in MDV_PLANE_RLE8 format and return it
 * in the specified format.
 *
 * Returns a pointer to the read data, or NULL on error.
 * The caller must free the space allocated.
 */

static ui08 *read_field_plane_rle8(FILE *infile,
				   MDV_field_header_t *field_hdr,
				   int return_type,
				   int *volume_size_returned)

{
  static char *routine_name = "read_field_plane_rle8";

  si32 vlevel_loc[MDV_MAX_VLEVELS];
  si32 vlevel_size[MDV_MAX_VLEVELS];
  ui08 *encoded_buf = NULL;
  ui08 *decoded_buf = NULL;
  
  ui08 *return_buf;
  ui08 *uRLDecode8();
  ui32 nbytes_decoded;

  ui08 *decoded_plane;
  ui08 *buf_ptr;
  int plane_size = field_hdr->nx * field_hdr->ny * sizeof(ui08);
  int vlevel;
      

  
  /*
   * Read in the encoded buffer.
   */


  if ((encoded_buf = read_plane_rle8_volume(infile,
					    field_hdr->nz,
					    field_hdr->volume_size,
					    vlevel_loc,
					    vlevel_size)) == NULL)
  {
    fprintf(stderr,
	    "%s: Error reading in encoded buffer\n",
	    routine_name);
    return(NULL);
  }
    
  /*
   * Decode the data into the return type.
   */

  switch (return_type)
  {
  case MDV_INT8 :
  {
    /*
     * Allocate space for the decoded buffer.
     */

    if ((decoded_buf = (ui08 *)malloc(field_hdr->nz * plane_size)) == NULL)
    {
      fprintf(stderr,
	      "%s: Error allocating %d bytes for decoded volume\n",
	      routine_name, field_hdr->nz * plane_size);
      free(encoded_buf);
      *volume_size_returned = 0;
      return(NULL);
    }
      
    /*
     * Decode each plane and copy it into the decoded buffer.
     */

    buf_ptr = decoded_buf;
      
    for (vlevel = 0; vlevel < field_hdr->nz; vlevel++)
    {
	
      /*
       * Decode the plane of data.
       */

      if ((decoded_plane =
	   uRLDecode8(encoded_buf + vlevel_loc[vlevel],
	     &nbytes_decoded)) == NULL)

      {
	fprintf(stderr,
		"%s: Error decoding plane %d\n",
		routine_name, vlevel);
	free(encoded_buf);
	free(decoded_buf);
	*volume_size_returned = 0;
	return(NULL);
      }
	
      if (nbytes_decoded != plane_size)
      {
	fprintf(stderr,
		"%s: Error in decoding, decoded plane has %d bytes, should have %d bytes\n",
		routine_name, nbytes_decoded, plane_size);
	free(encoded_buf);
	free(decoded_buf);
	free(decoded_plane);
	*volume_size_returned = 0;
	return(NULL);
      }
	
      /*
       * Copy the plane of data into the volume buffer.
       */

      memcpy((char *)buf_ptr, (char *)decoded_plane, plane_size);
      buf_ptr += plane_size;
      free(decoded_plane);
	
    } /* endfor - vlevel */
      
    return_buf = (void *)decoded_buf;
    *volume_size_returned = field_hdr->nz * plane_size;
    
    break;
  }    /* endcase - output format MDV_INT8 */
      
  case MDV_PLANE_RLE8 :
  {
    int array_size;
      
    /*
     * We need to return this buffer as one large buffer with
     * the vlevel information in the front.  We could have just
     * read the entire return buffer from disk and byte swapped
     * in place, but that would have ruined the "flow" of this
     * routine.
     */

    if ((return_buf = (void *)malloc(field_hdr->volume_size)) == NULL)
    {
      fprintf(stderr,
	      "%s: Error allocating %d bytes for return buffer\n",
	      routine_name, field_hdr->volume_size);
      free(encoded_buf);
      *volume_size_returned = 0;
      return(NULL);
    }
      
    array_size = field_hdr->nz * sizeof(si32);
      
    memcpy((char *)return_buf,
	   (char *)vlevel_loc,
	   array_size);
      
    memcpy((char *)return_buf + array_size,
	   (char *)vlevel_size,
	   array_size);
      
    memcpy((char *)return_buf + (2 * array_size),
	   (char *)encoded_buf,
	   field_hdr->volume_size - (2 * array_size));
      
    *volume_size_returned = field_hdr->volume_size;
    
    free(encoded_buf);
    
    break;
  } /* endcase - output format MDV_PLANE_RLE8 */

  default:
    fprintf(stderr,
	    "%s: Cannot convert input %s data into %s format -- not yet implemented\n",
	    routine_name, MDV_encode2string(MDV_PLANE_RLE8),
	    MDV_encode2string(return_type));
    free(encoded_buf);
    *volume_size_returned = 0;
    
    return(NULL);
    break;  /* endcase - output format default */
      
  } /* endswitch - return_tyoe */
    
  return(return_buf);
  
} /* end read_field_plane_rle8 */


/********************************************************
 * READ_FIELD_ROW_RLE8
 * Read a field of data in MDV_ROW_RLE8 format and return it
 * in the specified format.
 *
 * Returns a pointer to the read data, or NULL on error.
 * The caller must free the space allocated.
 */

static ui08 *read_field_row_rle8(FILE *infile,
				 MDV_field_header_t *field_hdr,
				 int return_type,
				 int *volume_size_returned)
{
  static char *routine_name = "read_field_row_rle8";

  si32 *row_locs;
  si32 *row_sizes;
  ui08 *encoded_buf = NULL;
  ui08 *decoded_buf = NULL;
  int num_rows = field_hdr->ny * field_hdr->nz;
  
  ui08 *return_buf;
  
  /*
   * Allocate space for row information arrays.
   */

  if ((row_locs = (si32 *)malloc(num_rows * sizeof(si32))) == NULL)
  {
    fprintf(stderr,
	    "%s: Error allocating %ld bytes for row locations array\n",
	    routine_name, (long int)(num_rows * sizeof(si32)));
    return(NULL);
  }
  
  if ((row_sizes = (si32 *)malloc(num_rows * sizeof(si32))) == NULL)
  {
    fprintf(stderr,
	    "%s: Error allocating %ld bytes for row sizes array\n",
	    routine_name, (long int)(num_rows * sizeof(si32)));
    free(row_locs);
    return(NULL);
  }

  /*
   * Read in the encoded buffer.
   */

  /*if ((encoded_buf = read_row_rle8_volume(infile,
					  num_rows,
					  field_hdr->volume_size,
					  row_locs,
					  row_sizes)) == NULL)
  {
    fprintf(stderr,
	    "%s: Error reading in encoded buffer\n",
	    routine_name);
    free(row_locs);
    free(row_sizes);
    return(NULL);
    }*/
    
  /*
   * Decode the data into the return type.
   */

  switch (return_type)
  {
  case MDV_INT8 :
  {
    ui08 *decoded_row;
    ui08 *buf_ptr;
    int row_size = field_hdr->nx * sizeof(ui08);
    int volume_size = field_hdr->ny * field_hdr->nz * row_size;
    int irow;
      
    /*
     * Allocate space for the decoded buffer.
     */

    if ((decoded_buf = (ui08 *)malloc(volume_size)) == NULL)
    {
      fprintf(stderr,
	      "%s: Error allocating %d bytes for decoded volume\n",
	      routine_name, volume_size);
      free(encoded_buf);
      free(row_locs);
      free(row_sizes);
      *volume_size_returned = 0;
      return(NULL);
    }
      
    /*
     * Decode each row and copy it into the decoded buffer.
     */

    buf_ptr = decoded_buf;
      
    for (irow = 0; irow < num_rows; irow++)
    {
      ui32 nbytes_decoded;
	
      /*
       * Decode the plane of data.
       */

      /*if ((decoded_row = uRLDecode8(encoded_buf + row_locs[irow],
				    &nbytes_decoded)) == NULL)
      {
	fprintf(stderr,
		"%s: Error decoding row %d\n",
		routine_name, irow);
	free(encoded_buf);
	free(decoded_buf);
	free(row_locs);
	free(row_sizes);
	*volume_size_returned = 0;
	return(NULL);
	}*/
	
      if (nbytes_decoded != row_size)
      {
	fprintf(stderr,
		"%s: Error in decoding, decoded row has %d bytes, should have %d bytes\n",
		routine_name, nbytes_decoded, row_size);
	free(encoded_buf);
	free(decoded_buf);
	free(decoded_row);
	free(row_locs);
	free(row_sizes);
	*volume_size_returned = 0;
	return(NULL);
      }
	
      /*
       * Copy the row of data into the volume buffer.
       */

      memcpy((char *)buf_ptr, (char *)decoded_row, row_size);
      buf_ptr += row_size;
      free(decoded_row);
	
    } /* endfor - irow */
      
    return_buf = (void *)decoded_buf;
    *volume_size_returned = volume_size;
    
    break;
  }    /* endcase - output format MDV_INT8 */
      
  case MDV_ROW_RLE8 :
  {
    int array_size;
      
    /*
     * We need to return this buffer as one large buffer with
     * the row information in the front.  We could have just
     * read the entire return buffer from disk and byte swapped
     * in place, but that would have ruined the "flow" of this
     * routine.
     */

    if ((return_buf = (void *)malloc(field_hdr->volume_size)) == NULL)
    {
      fprintf(stderr,
	      "%s: Error allocating %d bytes for return buffer\n",
	      routine_name, field_hdr->volume_size);
      free(encoded_buf);
      return(NULL);
    }
      
    array_size = num_rows * sizeof(si32);
      
    memcpy((char *)return_buf,
	   (char *)row_locs,
	   array_size);
      
    memcpy((char *)return_buf + array_size,
	   (char *)row_sizes,
	   array_size);
      
    memcpy((char *)return_buf + (2 * array_size),
	   (char *)encoded_buf,
	   field_hdr->volume_size - (2 * array_size));
    
    *volume_size_returned = field_hdr->volume_size;
    
    free(encoded_buf);

    break;
  } /* endcase - output format MDV_ROW_RLE8 */

  default:
    fprintf(stderr,
	    "%s: Cannot convert input %s data into %s format -- not yet implemented\n",
	    routine_name, MDV_encode2string(MDV_ROW_RLE8),
	    MDV_encode2string(return_type));
    free(encoded_buf);
    *volume_size_returned = 0;
    return(NULL);
    break;  /* endcase - output format default */
      
  } /* endswitch - return_tyoe */
    
  /*
   * Free the row information arrays.
   */

  free(row_locs);
  free(row_sizes);
  
  return(return_buf);
  
} /* end read_field_row_rle8 */


/********************************************************
 * READ_PLANE_INT8
 * Routine for reading a plane of data in MDV_INT8 format
 * and returning it in the specified format. All necessary
 * byte swapping is done in this routine.
 *
 * This routine assumes that the plane number is valid for
 * this data.
 *
 * Returns pointer to data buffer or NULL on error.  Note
 * that the calling routine must free the data pointer
 * returned.
 */

static void *read_plane_int8(FILE *infile,
			     MDV_field_header_t *field_hdr,
			     int data_type,
			     int plane_num)
{
  static char *routine_name = "read_plane_int8";
  
  ui08 *input_buf;
  void *return_buf;
  
  int plane_size;
  int plane_loc;
  int bytes_read;
  
  /*
   * Compute size of plane on disk.
   */

  plane_size = field_hdr->nx * field_hdr->ny * sizeof(ui08);

  /*
   * Compute where the data is located in the file 
   */

  plane_loc = field_hdr->field_data_offset + plane_num * plane_size;
    
  /*
   * Seek to the data position.
   */

  if (fseek(infile, plane_loc, SEEK_SET) != 0)
  {
    fprintf(stderr,
	    "%s: Error seeking to file offset %d to get plane data\n",
	    routine_name, plane_loc);
    return(NULL);
  }
  
  /*
   * Allocate space for the buffer to be read in.
   */

  if ((input_buf = (ui08 *)malloc(plane_size)) == NULL) 
  {
    fprintf(stderr,
	    "%s: Error allocating %d bytes for input buffer\n",
	    routine_name, plane_size);
    return(NULL);
  }
  
  /*
   * Read the data into the input buffer
   */

  if ((bytes_read = fread(input_buf, 1, plane_size, infile))
      != plane_size)
  {
    fprintf(stderr,
	    "%s: Error reading plane data from disk (expected bytes = %d, read bytes = %d)\n",
	    routine_name, plane_size, bytes_read);
    free(input_buf);
    return(NULL);
  }

  /*
   * No swapping necessary - byte data
   */

  /*
   * Convert the data to the desired return type.
   */

  switch(data_type)
  {
  case MDV_INT8 :
    return_buf = (void *)input_buf;
    break;
    
  default:
    fprintf(stderr,
	    "%s: Cannot convert %s plane to %s format -- not yet implemented\n",
	    routine_name, MDV_encode2string(MDV_INT8),
	    MDV_encode2string(data_type));
    free(input_buf);
    return(NULL);
  } /* endswitch - data_type */
  
  return(return_buf);
  
} /* end read_plane_int8 */


/********************************************************
 * READ_PLANE_PLANE_RLE8
 * Routine for reading a plane of data in MDV_PLANE_RLE8 format
 * and returning it in the specified format. All necessary
 * byte swapping is done in this routine.
 *
 * This routine assumes that the plane number is valid for
 * this data.
 *
 * Returns pointer to data buffer or NULL on error.  Note
 * that the calling routine must free the data pointer
 * returned.
 */

static void *read_plane_plane_rle8(FILE *infile,
				   MDV_field_header_t *field_hdr,
				   int data_type,
				   int plane_num)
{
  static char *routine_name = "read_plane_plane_rle8";
  
  void *return_buf;
  void *encode_buf;

  int vlevel_offset;
  si32 *vlevel_locs;
  si32 *vlevel_sizes;
  int array_size = field_hdr->nz * sizeof(si32);
  ui32 nbytes;
  int bytes_read;
  si32 *lptr;
 
 // added explicit function declaration ahijevyc 20060919 
  ui08 *uRLDecode8();
 
  /*
   * Move to the plane information in the input file.
   */

  if (fseek(infile, field_hdr->field_data_offset, SEEK_SET) != 0)
  {
    fprintf(stderr,
	    "%s: Error seeking to plane information offset %d\n",
	    routine_name, field_hdr->field_data_offset);
    return(NULL);
  }
  
  /*
   * Allocate memory for location array
   */

  if ((vlevel_locs = (si32 *)malloc(array_size)) == NULL)
  {
    fprintf(stderr,
	    "%s: Error allocating %d bytes for vlevel_locs array\n",
	    routine_name, array_size);
    return(NULL);
  }

  /*
   * Allocate memory for size array
   */

  if ((vlevel_sizes = (si32 *)malloc(array_size)) == NULL)
  {
    fprintf(stderr,
	    "%s: Error allocating %d bytes for vlevel_sizes array\n",
	    routine_name, array_size);
    free(vlevel_locs);
    return(NULL);
  }

  /*
   * Read in location info
   */
  
  if ((fread(vlevel_locs, sizeof(si32),
	      field_hdr->nz, infile)) != field_hdr->nz)
  {
    fprintf(stderr,
	    "%s: Error reading vlevel_locs array\n",
	    routine_name);
    free(vlevel_locs);
    free(vlevel_sizes);
    return(NULL);
  }
       
  /*
   * Read in size info
   */

  if ((fread(vlevel_sizes, sizeof(si32),
	      field_hdr->nz, infile)) != field_hdr->nz)
  {
    fprintf(stderr,
	    "%s: Error reading vlevel_locs array\n",
	    routine_name);
    free(vlevel_locs);
    free(vlevel_sizes);
    return(NULL);
  }
       
  /*
   * Swap the plane information, if necessary.
   */

  BE_to_array_32(vlevel_locs, array_size);
    
  BE_to_array_32(vlevel_sizes, array_size);
  
  /*
   * Allocate memory for encoded data plane
   */

  if ((encode_buf = (ui08 *)malloc(vlevel_sizes[plane_num])) == NULL)
  {
    fprintf(stderr,
	    "%s: Error allocating %d bytes for encode_buf\n",
	    routine_name, vlevel_sizes[plane_num]);
    free(vlevel_locs);
    free(vlevel_sizes);
    return(NULL);
  }

  /*
   * Calculate where plane data actually starts, skip over
   * loc/size arrays
   */

  vlevel_offset = field_hdr->field_data_offset +
    2 * array_size + vlevel_locs[plane_num];
   
  /*
   * Go to beginning of this vlevel data 
   */

  if (fseek(infile, vlevel_offset, SEEK_SET) != 0)
  {
    fprintf(stderr,
	    "%s: Error seeking to beginning of plane data (offset = %d)\n",
	    routine_name, vlevel_offset);
    free(vlevel_locs);
    free(vlevel_sizes);
    free(encode_buf);
    return(NULL);
  }

  /*
   * Read encoded data
   */
  
  if ((bytes_read = fread(encode_buf, 1,
			   vlevel_sizes[plane_num], infile))
      != vlevel_sizes[plane_num])
  {
    fprintf(stderr,
	    "%s: Error reading encoded data (expected bytes = %d, bytes read = %d)\n",
	    routine_name, vlevel_sizes[plane_num], bytes_read);
    free(vlevel_locs);
    free(vlevel_sizes);
    free(encode_buf);
    return(NULL);
  }

  /*
   * Swap the encoded plane.
   */

  lptr = encode_buf;
    
  lptr[0] = BE_to_si32(lptr[0]);   /* RL8_FLAG */
  lptr[1] = BE_to_si32(lptr[1]);   /* key */
  lptr[2] = BE_to_si32(lptr[2]);   /* nbytes_array */
  lptr[3] = BE_to_si32(lptr[3]);   /* nbytes_full */
  lptr[4] = BE_to_si32(lptr[4]);   /* nbytes_coded */
  
  /*
   * Put the data into the desired format.
   */

  switch(data_type)
  {
  case MDV_INT8:
    /*
     * Decode this plane of data
     */

    return_buf = (void *)uRLDecode8(encode_buf, &nbytes);

    if (return_buf == NULL ||
	nbytes != field_hdr->nx * field_hdr->ny * INT8_SIZE)
    {
      fprintf(stderr,
	      "%s: Error decoding buffer into %s format\n",
	      routine_name, MDV_encode2string(MDV_INT8));

      if (return_buf != NULL)
      {
	free(return_buf);
	return_buf = NULL;
      }
    }
    
    free(encode_buf);
    break;
    
  default:
    fprintf(stderr,
	    "%s: Cannot convert %s data into %s format -- not yet implemented\n",
	    routine_name, MDV_encode2string(MDV_PLANE_RLE8),
	    MDV_encode2string(data_type));
    free(vlevel_locs);
    free(vlevel_sizes);
    free(encode_buf);
    return(NULL);
    break;
  } /* endswitch - data_type */
  
  /*
   * Clean up
   */

  free(vlevel_locs);
  free(vlevel_sizes);

  return(return_buf);
  
} /* end read_plane_plane_rle8 */


/********************************************************
 * READ_PLANE_ROW_RLE8
 * Routine for reading a plane of data in MDV_ROW_RLE8 format
 * and returning it in the specified format. All necessary
 * byte swapping is done in this routine.
 *
 * This routine assumes that the plane number is valid for
 * this data.
 *
 * Returns pointer to data buffer or NULL on error.  Note
 * that the calling routine must free the data pointer
 * returned.
 */

static void *read_plane_row_rle8(FILE *infile,
				 MDV_field_header_t *field_hdr,
				 int data_type,
				 int plane_num)
{
  static char *routine_name = "read_plane_row_rle8";
  
  void *return_buf;
  void *encode_buf;

  int plane_offset;
  int plane_size;
  si32 *row_locs;
  si32 *row_sizes;
  ui08 **rows;
  int num_rows = field_hdr->ny * field_hdr->nz;
  int array_size = num_rows * sizeof(si32);
  ui32 nbytes;
  int bytes_read;
  
  int irow;

 // added explicit function declaration ahijevyc 20060919 
  ui08 *uRLDecode8();

  /*
   * Move to the row information in the input file.
   */

  if (fseek(infile, field_hdr->field_data_offset, SEEK_SET) != 0)
  {
    fprintf(stderr,
	    "%s: Error seeking to row information offset %d\n",
	    routine_name, field_hdr->field_data_offset);
    return(NULL);
  }
  
  /*
   * Allocate memory for location array
   */

  if ((row_locs = (si32 *)malloc(array_size)) == NULL)
  {
    fprintf(stderr,
	    "%s: Error allocating %d bytes for row_locs array\n",
	    routine_name, array_size);
    return(NULL);
  }

  /*
   * Allocate memory for size array
   */

  if ((row_sizes = (si32 *)malloc(array_size)) == NULL)
  {
    fprintf(stderr,
	    "%s: Error allocating %d bytes for row_sizes array\n",
	    routine_name, array_size);
    free(row_locs);
    return(NULL);
  }

  /*
   * Read in location info
   */
  
  if ((fread(row_locs, sizeof(si32),
	      num_rows, infile)) != num_rows)
  {
    fprintf(stderr,
	    "%s: Error reading row_locs array\n",
	    routine_name);
    free(row_locs);
    free(row_sizes);
    return(NULL);
  }
       
  /*
   * Read in size info
   */

  if ((fread(row_sizes, sizeof(si32),
	      num_rows, infile)) != num_rows)
  {
    fprintf(stderr,
	    "%s: Error reading row_locs array\n",
	    routine_name);
    free(row_locs);
    free(row_sizes);
    return(NULL);
  }
       
  /*
   * Swap the row information.
   */

  BE_to_array_32(row_locs, array_size);
    
  BE_to_array_32(row_sizes, array_size);
  
  /*
   * Calculate the encoded data plane size.
   */

  plane_size = 0;
  
  for (irow = field_hdr->ny * plane_num;
       irow < (field_hdr->ny * plane_num) + field_hdr->ny;
       irow++)
    plane_size += row_sizes[irow];
  
  /*
   * Allocate memory for encoded data plane
   */

  if ((encode_buf = (ui08 *)malloc(plane_size)) == NULL)
  {
    fprintf(stderr,
	    "%s: Error allocating %d bytes for encode_buf\n",
	    routine_name, plane_size);
    free(row_locs);
    free(row_sizes);
    return(NULL);
  }

  /*
   * Calculate where plane data actually starts, skip over
   * loc/size arrays
   */

  plane_offset = field_hdr->field_data_offset +
    2 * array_size + row_locs[plane_num * field_hdr->ny];
   
  /*
   * Go to beginning of this plane data 
   */

  if (fseek(infile, plane_offset, SEEK_SET) != 0)
  {
    fprintf(stderr,
	    "%s: Error seeking to beginning of plane data (offset = %d)\n",
	    routine_name, plane_offset);
    free(row_locs);
    free(row_sizes);
    free(encode_buf);
    return(NULL);
  }

  /*
   * Read encoded data
   */
  
  if ((bytes_read = fread(encode_buf, 1,
			   plane_size, infile))
      != plane_size)
  {
    fprintf(stderr,
	    "%s: Error reading encoded data (expected bytes = %d, bytes read = %d)\n",
	    routine_name, plane_size, bytes_read);
    free(row_locs);
    free(row_sizes);
    free(encode_buf);
    return(NULL);
  }

  /*
   * Set pointers to each of the rows of data.
   */

  if ((rows = (ui08 **)malloc(field_hdr->ny * sizeof(ui08 *))) == NULL)
  {
    fprintf(stderr,
	    "%s: Error allocating %ld bytes for rows array\n",
	    routine_name, (long int)(field_hdr->ny * sizeof(ui08 *)));
    free(row_locs);
    free(row_sizes);
    free(encode_buf);
    return(NULL);
  }
  
  for (irow = 0; irow < field_hdr->ny; irow++)
  {
    rows[irow] = (ui08 *) encode_buf + 
      row_locs[irow + plane_num * field_hdr->ny] -
	row_locs[plane_num * field_hdr->ny];
  }
    
  /*
   * Swap the encoded plane.
   */

  for (irow = 0; irow < field_hdr->ny; irow++)
  {
    si32 *lptr = (si32 *)rows[irow];
    
    lptr[0] = BE_to_si32(lptr[0]);   /* RL8_FLAG */
    lptr[1] = BE_to_si32(lptr[1]);   /* key */
    lptr[2] = BE_to_si32(lptr[2]);   /* nbytes_array */
    lptr[3] = BE_to_si32(lptr[3]);   /* nbytes_full */
    lptr[4] = BE_to_si32(lptr[4]);   /* nbytes_coded */
  } /* endfor - irow */
  
  /*
   * Put the data into the desired format.
   */

  switch(data_type)
  {
  case MDV_INT8:
  {
    ui08 *return_buf_ptr;
    
    /*
     * Allocate space for the return buffer.
     */

    if ((return_buf =
	 (void *)malloc(field_hdr->nx * field_hdr->ny * sizeof(ui08)))
	== NULL)
    {
      fprintf(stderr,
	      "%s: Error allocating %ld bytes for return buffer\n",
	      routine_name,
	      (long int)(field_hdr->nx * field_hdr->ny * sizeof(ui08)));
      free(row_locs);
      free(row_sizes);
      free(rows);
      free(encode_buf);
      return(NULL);
    }
    
    /*
     * Decode each row of data and add it to the return buffer.
     */

    return_buf_ptr = return_buf;
    
    for (irow = 0; irow < field_hdr->ny; irow++)
    {
      ui08 *decode_buf;
      
      decode_buf = (void *)uRLDecode8(rows[irow], &nbytes);

      if (decode_buf == NULL ||
	  nbytes != field_hdr->nx * sizeof(ui08))
      {
	fprintf(stderr,
		"%s: Error decoding buffer into %s format\n",
		routine_name, MDV_encode2string(MDV_INT8));

	if (decode_buf != NULL)
	{
	  free(decode_buf);
	  decode_buf = NULL;
	}

	free(row_locs);
	free(row_sizes);
	free(rows);
	free(encode_buf);
	return(NULL);
      }
    
      memcpy(return_buf_ptr, decode_buf, nbytes);
      return_buf_ptr += nbytes;
      
      free(decode_buf);
    }
    break;
  } /* endcase - MDV_INT8 */
    
  default:
    fprintf(stderr,
	    "%s: Cannot convert %s data into %s format -- not yet implemented\n",
	    routine_name, MDV_encode2string(MDV_ROW_RLE8),
	    MDV_encode2string(data_type));
    free(row_locs);
    free(row_sizes);
    free(rows);
    free(encode_buf);
    return(NULL);
    break;
  } /* endswitch - data_type */
  
  /*
   * Clean up
   */

  free(row_locs);
  free(row_sizes);
  free(rows);
  
  return(return_buf);
  
} /* end read_plane_row_rle8 */

/********************************************************
 * READ_UNENCODED_VOLUME
 * Routine for reading a volume of data that's not
 * encoded.
 */

static void *read_unencoded_volume(FILE *infile,
				   int volume_size,
				   int data_type)
{
  static char *routine_name = "read_unencoded_volume";
  
  ui08 *input_buf;
  int bytes_read;
      
  /*
   * Make sure this is a data type we have implemented.
   */

  if (data_type != MDV_INT8 &&
      data_type != MDV_INT16 &&
      data_type != MDV_INT32 &&
      data_type != MDV_FLOAT32)
  {
    fprintf(stderr,
	    "%s: Cannot read unencoded %s data from file -- have not implemented byte swapping\n",
	    routine_name, MDV_encode2string(data_type));
    return(NULL);
  }
  
  /*
   * Allocate space for the input buffer.
   */

  if ((input_buf = (ui08 *)malloc(volume_size)) == NULL)
  {
    fprintf(stderr,
	    "%s: Error allocating %d bytes for input buffer.\n",
	    routine_name, volume_size);
    return(NULL);
  }
      
  /*
   * Read the data into the buffer.
   */

  if ((bytes_read = fread(input_buf, 1, volume_size, infile))
      != volume_size)
  {
    fprintf(stderr,
	    "%s: Error reading volume data from input file (bytes expected = %d, bytes read = %d)\n",
	    routine_name, volume_size, bytes_read);
    free(input_buf);
    return(NULL);
  }
   
  /*
   * Byte swap the data.
   */

  if (MDV_unencoded_volume_from_BE(input_buf,
				   volume_size,
				   data_type) != MDV_SUCCESS)
  {
    fprintf(stderr,
	    "%s: Error swapping unencoded data\n",
	    routine_name);
    free(input_buf);
    return(NULL);
  }
  
  return((void *)input_buf);
}

/********************************************************
 * READ_PLANE_RLE8_VOLUME
 * Routine for reading a volume of data that's encoded
 * in the MDV_PLANE_RLE8 format.
 */

static ui08 *read_plane_rle8_volume(FILE *infile,
				    int nlevels,
				    int data_size,
				    si32 *vlevel_loc,
				    si32 *vlevel_size)
{
  static char *routine_name = "read_plane_rle8_volume";
  
  ui08 *input_buf;
  ui32 array_size = nlevels * sizeof(si32);
  int volume_size = data_size - (2 * array_size);
  int bytes_read;
  int plane;
  
  /*
   * Read in the vlevel location and size information.
   */

  if (fread(vlevel_loc, sizeof(si32), nlevels, infile)
      != nlevels)
  {
    fprintf(stderr,
	    "%s: Error reading in vlevel location information\n",
	    routine_name);
    return(NULL);
  }
    
  if (fread(vlevel_size, sizeof(si32), nlevels, infile)
      != nlevels)
  {
    fprintf(stderr,
	    "%s: Error reading in vlevel size information\n",
	    routine_name);
    return(NULL);
  }
    
  /*
   * Byte swap the vlevel information.
   */

  BE_to_array_32((ui32 *)vlevel_loc, array_size);
    
  BE_to_array_32((ui32 *)vlevel_size, array_size);
  
  /*
   * Allocate space for and read the encoded data.
   */

  if ((input_buf = (ui08 *)malloc(volume_size))
      == NULL)
  {
    fprintf(stderr,
	    "%s: Error allocating %d bytes for input encoded buffer\n",
	    routine_name, volume_size);
    return(NULL);
  }
   
  if ((bytes_read = fread(input_buf, 1, volume_size, infile))
      != volume_size)
  {
    fprintf(stderr,
	    "%s: Error reading encoded data (expected bytes = %d, read bytes = %d)\n",
	    routine_name, volume_size, bytes_read);
    free(input_buf);
    return(NULL);
  }
  
  /*
   * Byte swap the encoded data for each plane.
   */

  for (plane = 0; plane < nlevels; plane++)
  {
    void *plane_ptr;
    
    plane_ptr = (void *)(input_buf + vlevel_loc[plane]);
    
    if (MDV_plane_rle8_from_BE(plane_ptr) != MDV_SUCCESS)
    {
      fprintf(stderr,
	      "%s: Error swapping plane %d data\n",
	      routine_name, plane);
      free(input_buf);
      return(NULL);
    }
      
  }
  
  return(input_buf);
} /* end read_plane_rle8_volume */

/*****************************************************************
 * MDV_PLANE_RLE8_FROM_BE: Converts the data in a plane of data
 * encoded in the MDV_PLANE_RLE8 format from big endian format to
 * native format.  N. Rehak 6/97
 *
 * returns MDV_SUCCESS or MDV_FAILURE
 */
 
int MDV_plane_rle8_from_BE(void *plane_data)
{
  si32 *plane_ptr = (si32 *)plane_data;
  
  plane_ptr[0] = BE_to_si32(plane_ptr[0]);  /* RL8_FLAG */
  plane_ptr[1] = BE_to_si32(plane_ptr[1]);  /* key */
  plane_ptr[2] = BE_to_si32(plane_ptr[2]);  /* nbytes_array */
  plane_ptr[3] = BE_to_si32(plane_ptr[3]);  /* nbytes_full */
  plane_ptr[4] = BE_to_si32(plane_ptr[4]);  /* nbytes_coded */

  /* All of the rest of the data is byte data so doesn't need swapping */

  return(MDV_SUCCESS);
}
