#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <memory.h>
#include <errno.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include "./bigend.h"
#include "./port_types.h"
#include "./mdv_read.h"
#include "./os_config.h"
#include "./mdv_print.h"

#define MDV_READ_SUCCESSFUL             0
#define MDV_READ_OPEN_FAILURE           1
#define MDV_READ_BAD_MASTER_HDR         2
#define MDV_READ_INVALID_FIELD_NUM      3
#define MDV_READ_BAD_FIELD_HDR          4
#define MDV_READ_BAD_VLEVEL_HDR         5
#define MDV_READ_NO_VLEVEL_HDRS         6
#define MDV_READ_INVALID_CHUNK_NUM      7
#define MDV_READ_BAD_CHUNK_HDR          8
#define MDV_READ_DATA_ARRAY_TOO_SMALL   9
#define MDV_READ_DATA_ERROR  10
#define NID 510


#ifndef BOOL_STR
#define BOOL_STR(a) (a == FALSE ? "false" : "true")
#endif

static struct files *open_files;

// I copied these from mdv_utils.h to avoid compilation warnings about implicit function declarations. MDV_master_header_from_BE MDV_field_header_from_BE 
 
/*
 * Swap routines.
 */

/*****************************************************************
 * MDV_MASTER_HEADER_FROM_BE: Converts master header from big endian
 * format to native format.  Nancy Rehak 6/97
 */

void MDV_master_header_from_BE(MDV_master_header_t *m_hdr) ;

/*****************************************************************
 * MDV_FIELD_HEADER_FROM_BE: Converts field header from big endian
 * format to native format.  Nancy Rehak 6/97
 */

void MDV_field_header_from_BE(MDV_field_header_t *f_hdr);


// I moved  MDV_vlevel_header_from_BE up here from below and added `void' before it.
// ahijevyc 20060920
/*****************************************************************
 * MDV_VLEVEL_HEADER_FROM_BE: Swaps vlevel header from big endian
 * format to native format.  Nancy Rehak 6/97
 */
 
void MDV_vlevel_header_from_BE(MDV_vlevel_header_t *v_hdr)
{
  /* swap header si32 and fl32's. No chars so everything done at once */
  BE_to_array_32((ui32 *)(&v_hdr->record_len1),
		 MDV_NUM_VLEVEL_HEADER_32 * sizeof(si32));
 
  /* swap the last record length */
  v_hdr->record_len2 = BE_to_si32(v_hdr->record_len2);
 
  return;
}



/***********************************************************************
* MF_RM_READ_MASTER_HDR: Read the master header from a data file and
* return the information in FORTRAN-usable structures.
*
* Use the following to call the subroutine:
*
*    CHARACTER*8 FNAME
*    INTEGER*4      MASTER_HDR_INTS(MDV_NUM_MASTER_HEADER_SI32)
*    REAL*4         MASTER_HDR_REALS(MDV_NUM_MASTER_HEADER_FL32)
*    CHARACTER*(MDV_INFO_LEN)  DATASET_INFO
*    CHARACTER*(MDV_NAME_LEN)  DATASET_NAME
*    CHARACTER*(MDV_NAME_LEN)  DATASET_SOURCE
*    INTEGER        RETURN_STATUS
*
*   CALL MF_RM_READ_MASTER_HDR(FNAME, MASTER_HDR_INTS, MASTER_HDR_REALS,
*                              DATASET_INFO, DATASET_NAME, DATASET_SOURCE,
*		               RETURN_STATUS)
*
* This function will open the file named FNAME, read the master header.
* The header information is returned in the given arrays.
* 
* When finished, RETURN_STATUS can be MDV_READ_SUCCESSFUL,
*                                     MDV_READ_OPEN_FAILURE,
*                                     MDV_READ_BAD_MASTER_HDR
* Author : Nancy Rehak, RAP
*/


/* void mf_rm_read_master_hdr__( */

void readmasterhdr_(

                            char *fname,
			    si32 *master_hdr_ints,
			    fl32 *master_hdr_reals,
			    char  *dataset_info,
			    char  *dataset_name,
			    char  *dataset_source,
			    int   *return_status)
{
  FILE *mdv_file;

  MDV_master_header_t master_hdr;
  int i;
  int i1,i2;
  void mf_master_hdr_to_fortran();

  /* strip off trailing spaces */
  i=0;
  while (fname[i] != ' ')
    i++;
  fname[i]='\0';

 	 
  /*
   * Open the input file
   */

  if ((mdv_file = fopen(fname,"r")) == NULL)
  {
    fprintf(stderr, "Error opening input file\n");
    perror(fname);
    *return_status = MDV_READ_OPEN_FAILURE;
    return;
  }

  /*
   * Read the master header
   */

  if (MDV_load_master_header(mdv_file, &master_hdr) == MDV_FAILURE)
  {
    fprintf(stderr, "Error reading master header from file <%s>\n",
	    fname);
    fclose(mdv_file);
    *return_status = MDV_READ_BAD_MASTER_HDR;
    return;
  }

  /*
   * Copy the master header info
   */

  mf_master_hdr_to_fortran(&master_hdr,
			   master_hdr_ints,
			   master_hdr_reals,
			   dataset_info,
			   dataset_name,
			   dataset_source);
  
  *return_status = MDV_READ_SUCCESSFUL;
  fclose(mdv_file);
  return;
}/*mf_rm_read_master_hdr*/

/******************************************************************************
 * MDV_LOAD_MASTER_HEADER: Load mdv data file header into given area from
 * disk.  Memory for the master header is assumed to be allocated before
 * this routine is called.  The bytes in the header are swapped if necessary
 * to put the data in native format.
 *
 * Inputs: infile - pointer to the input file.  This is assumed to currently
 *                  be open for read.
 *         m_hdr - pointer to the master header to be loaded.
 *
 * Outputs: m_hdr - updated to include the values read in from disk,
 *                  byte swapped as necessary.
 *
 * Returns: MDV_SUCCESS or MDV_FAILURE.
 * Author : Nancy Rehak, RAP
 */

int MDV_load_master_header( FILE *infile, MDV_master_header_t *m_hdr)
{
   /* master headers are always at the beginning */
    if (fseek(infile,0,SEEK_SET)) 
        return MDV_FAILURE;

    if((fread(m_hdr,sizeof(MDV_master_header_t),1,infile)) != 1) {
        return MDV_FAILURE;
    }

    MDV_master_header_from_BE(m_hdr);

    return MDV_SUCCESS;
}


/***********************************************************************
 * mf_master_hdr_to_fortran:  Fills in the FORTRAN arrays from the given
 *                            MDV_master_header_t structure.
 */

void mf_master_hdr_to_fortran(MDV_master_header_t *master_hdr,
			      si32 *master_hdr_ints,
			      fl32 *master_hdr_reals,
			      char *dataset_info,
			      char *dataset_name,
			      char *dataset_source)
{
  /*
   * Load the information into the arrays.
   */

  memcpy(master_hdr_ints, &master_hdr->struct_id,
	 sizeof(si32) * MDV_NUM_MASTER_HEADER_SI32);

  memcpy(master_hdr_reals, master_hdr->user_data_fl32,
	 sizeof(fl32) * MDV_NUM_MASTER_HEADER_FL32);

  memcpy(dataset_info, master_hdr->data_set_info, MDV_INFO_LEN);

  memcpy(dataset_name, master_hdr->data_set_name, MDV_NAME_LEN);

  memcpy(dataset_source, master_hdr->data_set_source, MDV_NAME_LEN);

  return;
}


/***********************************************************************
* GET_MASTER_HDR_INFO: This routine will get information needed by the 
* Cedric 510 word header from the MDV master header.
*
*
*    SI32 and fl32 are defined in port_types.h
*    int         master_hdr_ints(MDV_NUM_MASTER_HEADER_SI32)
*    float        master_hdr_reals(MDV_NUM_MASTER_HEADER_FL32)
*    character*(MDV_INFO_LEN) dataset_info
*    character*(MDV_NAME_LEN) dataset_name
*    character*(MDV_NAME_LEN) dataset_source
*    int        SCAN_TYPE  CART,ELEV OR COPLANE 
*    float      location - array containing the lat,lon and altitude of the sensor.
*    int        beg_time - beginning time of the data set
*    int        end_time - ending time of the data set
*    int        nflds    - the number of fields on the tape
*    int        numxyz   - the max number of array elements in the x,y,z direction.    
*    int        mdv_order- the ordering of cells in the arrray.
*  
*
*/
void get_master_hdr_info_(   si32 *master_hdr_ints,fl32 *master_hdr_reals,
                             char *dataset_info,char *dataset_name,char *dataset_source,
                             int  *scan_type,float location[3],
                             int  *beg_time,int *end_time,
                             int  *nflds, int *numxyz, int *mdv_order )

{
  MDV_master_header_t master_hdr;
  void MDV_print_master_header_full();
  void mf_master_hdr_from_fortran();


  /*
   * Load the information in the master header
   */

  mf_master_hdr_from_fortran(master_hdr_ints,
			     master_hdr_reals,
			     dataset_info,
			     dataset_name,
			     dataset_source,
			     &master_hdr);



  *scan_type  = master_hdr.vlevel_type;
  location[0] = master_hdr.sensor_lat;
  location[1] = master_hdr.sensor_lon;
  location[2] = master_hdr.sensor_alt * 1000.;
  *beg_time   = master_hdr.time_begin;
  *end_time   = master_hdr.time_end;
  *nflds      = master_hdr.n_fields;
  numxyz[0]   = master_hdr.max_nx;
  numxyz[1]   = master_hdr.max_ny; 
  numxyz[2]   = master_hdr.max_nz;
  *mdv_order  = master_hdr.grid_order_indices;

  return;
}

/*****************************************************************
 * MDV_PRINT_MASTER_HEADER_FULL: print out all of the master
 * header info
 * --Nancy Rehak 4/96
 */

void MDV_print_master_header_full(MDV_master_header_t *mmh, FILE *outfile)
{
  int i;
  time_t print_time;         /* temporary storage area for printing time */
                             /*   values -- necessary on systems which have */
                             /*   time_t defined as other than 32-bit */
  
  fprintf(outfile, "\n");
  fprintf(outfile, "           MDV_print_master_header\n");
  fprintf(outfile, "           -----------------------\n");
  fprintf(outfile, "\n");
  fprintf(outfile, "record_len1:          %d\n", mmh->record_len1);
  fprintf(outfile, "struct_id:            %d\n", mmh->struct_id);
  fprintf(outfile, "revision_number:      %d\n", mmh->revision_number);
  fprintf(outfile, "\n");
  print_time = mmh->time_gen;
  fprintf(outfile, "time_gen:             %s",
	  asctime(gmtime(&print_time)));
  fprintf(outfile, "user_time:           %d\n", mmh->user_time);
  print_time = mmh->time_begin;
  fprintf(outfile, "time_begin:           %s",
	  asctime(gmtime(&print_time)));
  print_time = mmh->time_end;
  fprintf(outfile, "time_end:             %s",
	  asctime(gmtime(&print_time)));
  print_time = mmh->time_centroid;
  fprintf(outfile, "time_centroid:        %s",
	  asctime(gmtime(&print_time)));
  print_time = mmh->time_expire;
  if (mmh->time_expire == 0)
     fprintf(outfile, "time_expire:          %d\n",mmh->time_expire);
  else
     fprintf(outfile, "time_expire:          %s",
	     asctime(gmtime(&print_time)));
  fprintf(outfile, "num_data_times:       %d\n", mmh->num_data_times);
  fprintf(outfile, "index_number:         %d\n", mmh->index_number);
  fprintf(outfile, "data_dimension:       %d\n", mmh->data_dimension);
  fprintf(outfile, "data_collection_type: %s\n",
	  MDV_colltype2string(mmh->data_collection_type));
  fprintf(outfile, "user_data:            %d\n", mmh->user_data);
  fprintf(outfile, "native_vlevel_type:   %s\n",
	  MDV_verttype2string(mmh->native_vlevel_type));
  fprintf(outfile, "vlevel_type:          %s\n",
	  MDV_verttype2string(mmh->vlevel_type));
  fprintf(outfile, "vlevel_included:      %s\n",
	  BOOL_STR(mmh->vlevel_included));
  fprintf(outfile, "grid_order_direction: %s\n",
	  MDV_orient2string(mmh->grid_order_direction));
  fprintf(outfile, "grid_order_indices:   %s\n",
	  MDV_order2string(mmh->grid_order_indices));
  fprintf(outfile, "n_fields:             %d\n", mmh->n_fields);
  fprintf(outfile, "max_nx:               %d\n", mmh->max_nx);
  fprintf(outfile, "max_ny:               %d\n", mmh->max_ny);
  fprintf(outfile, "max_nz:               %d\n", mmh->max_nz);
  fprintf(outfile, "n_chunks:             %d\n", mmh->n_chunks);
  fprintf(outfile, "field_hdr_offset:     %d\n", mmh->field_hdr_offset);
  fprintf(outfile, "vlevel_hdr_offset:    %d\n", mmh->vlevel_hdr_offset);
  fprintf(outfile, "chunk_hdr_offset:     %d\n", mmh->chunk_hdr_offset);
  fprintf(outfile, "field_grids_differ:   %s\n",
          BOOL_STR(mmh->field_grids_differ));
  for (i = 0; i < 8; i++)
    fprintf(outfile, "user_data_si32[%d]:    %d\n",
	    i, mmh->user_data_si32[i]);
  
  fprintf(outfile, "\n");
  for (i = 0; i < 6; i++)
    fprintf(outfile, "user_data_fl32[%d]:    %f\n",
	    i, mmh->user_data_fl32[i]);
  fprintf(outfile, "sensor_lon:           %f\n", mmh->sensor_lon);
  fprintf(outfile, "sensor_lat:           %f\n", mmh->sensor_lat);
  fprintf(outfile, "sensor_alt:           %f\n", mmh->sensor_alt);
  fprintf(outfile, "\n");
  fprintf(outfile, "data_set_info:        <%s>\n", mmh->data_set_info);
  fprintf(outfile, "data_set_name:        <%s>\n", mmh->data_set_name);
  fprintf(outfile, "data_set_source:      <%s>\n", mmh->data_set_source);
  fprintf(outfile, "\n");
  fprintf(outfile, "record_len2:          %d\n", mmh->record_len2);
  fprintf(outfile, "\n\n");

  return;
}  
 

/***********************************************************************
 * mf_master_hdr_from_fortran:  Fills in the MDV_master_header_t structure
 *                              from the given FORTRAN arrays.
 */

void mf_master_hdr_from_fortran(si32 *master_hdr_ints,
				fl32 *master_hdr_reals,
				char *dataset_info,
				char *dataset_name,
				char *dataset_source,
				MDV_master_header_t *master_hdr)
{
  /*
   * Load the information from the arrays
   */

  memcpy(&master_hdr->struct_id, master_hdr_ints,
         MDV_NUM_MASTER_HEADER_SI32 * sizeof(si32));
  
  memcpy(master_hdr->user_data_fl32, master_hdr_reals,
         MDV_NUM_MASTER_HEADER_FL32 * sizeof(fl32));
  
  memcpy(master_hdr->data_set_info, dataset_info, MDV_INFO_LEN);
  
  memcpy(master_hdr->data_set_name, dataset_name, MDV_NAME_LEN);
  
  memcpy(master_hdr->data_set_source, dataset_source, MDV_NAME_LEN);
  
  /*
   * Make sure the strings are NULL terminated
   */

  master_hdr->data_set_info[MDV_INFO_LEN-1] = '\0';
  master_hdr->data_set_name[MDV_NAME_LEN-1] = '\0';
  master_hdr->data_set_source[MDV_NAME_LEN-1] = '\0';
  
  /*
   * Set the record length values
   */

  master_hdr->record_len1 = sizeof(MDV_master_header_t) - 2 * sizeof(si32);
  master_hdr->record_len2 = master_hdr->record_len1;
  
  return;
}
 
/***********************************************************************
* MF_pm_print_master_hdr: This routine will print an MDV master header
* from FORTRAN format data.
*
* Use the following to call the subroutine:
*
*    INTEGER*4     MASTER_HDR_INTS(MDV_NUM_MASTER_HEADER_SI32)
*    REAL*4        MASTER_HDR_REALS(MDV_NUM_MASTER_HEADER_FL32)
*    CHARACTER*(MDV_INFO_LEN) DATASET_INFO
*    CHARACTER*(MDV_NAME_LEN) DATASET_NAME
*    CHARACTER*(MDV_NAME_LEN) DATASET_SOURCE
*
*   CALL MF_PM_PRINT_MASTER_HDR(MASTER_HDR_INTS, MASTER_HDR_REALS,
*                               DATASET_INFO, DATASET_NAME, DATASET_SOURCE)
*
*/

void printmasterhdr_(

			     si32 *master_hdr_ints,
                             fl32 *master_hdr_reals,
                             char *dataset_info,
                             char *dataset_name,
			     char *dataset_source)
{
  MDV_master_header_t master_hdr;
  
  /*
   * Load the information in the master header
   */

  mf_master_hdr_from_fortran(master_hdr_ints,
			     master_hdr_reals,
			     dataset_info,
			     dataset_name,
			     dataset_source,
			     &master_hdr);
  
  /*
   * Print the master header
   */

  MDV_print_master_header_full(&master_hdr, stdout);
  
  return;
}



/**********************************************************************/
void mdv_radar_name_(char dataset_info[MDV_INFO_LEN],char source[8],
                     char radar[6],int *baseangle)
{

   int    index,city_index,save_index;
   char   mdvcity[25],city[25],state[4],radar_name[4];
   char   line[128],character;
   int    latd,latm,lats,lond,lonm,lons,found;
   int    site_number,ii,number,length;
   float  latitude,longitude,altitude,frequency_mhz,short_pulse_ns,long_pulse_ns;
   FILE   *fs;

   *baseangle = 0;
   for(index = 0; index < 6; index++) source[index] = dataset_info[index];
   if(strncmp(source,"NEXRAD",6) != 0) return;

   *baseangle = 90;
   for(index = 7; index < MDV_INFO_LEN; index++){

     if(dataset_info[index] == ','){
        save_index = index;
        break;
     }
   }

   save_index = save_index + 1;
   if(dataset_info[save_index] == ' ')save_index = save_index + 1;

   city_index = 0;
   for(index = save_index; index < MDV_INFO_LEN; index++){
     mdvcity[city_index] = dataset_info[index];
     if(dataset_info[index] == ','){
        mdvcity[city_index] = '\0';
        break;
     }
     city_index = city_index + 1;
   }  
   length = strlen(mdvcity);
   length = length-1;
   city_index = 0;
   if(!(fs = fopen("nexrad_radar_sites.txt", "r"))) {
     printf("unable to open radar sites file\n");
     exit(0);
  }
  found = 0;
  while (fgets(line,sizeof(line),fs) != NULL) {
      if(*line == '#')
         continue;
      ii = sscanf(line, "%d%s%s%s%d%d%d%d%d%d%f%f%f%f"
                    , &site_number
                    , radar_name
                    , city
                    , state
                    , &latd
                    , &latm
                    , &lats
                    , &lond
                    , &lonm
                    , &lons
                    , &altitude
                    , &frequency_mhz
                    , &short_pulse_ns
                    , &long_pulse_ns
                    );

      if(strncmp(mdvcity,city,length) == 0){
         found = 1;
         strncpy(radar,radar_name,4);
         radar_name[4] = ' ';
         radar_name[5] = ' ';
      }     
      else city_index++;
   }/*end while*/
   

  if(found == 0){
     strcpy(radar,"     ");
  }
}/*mdv_radar_name*/


/***********************************************************************
* MF_RF_READ_FIELD_HDR: Read the indicated field header from a data file
* and return the information in FORTRAN-usable structures.
*
* Use the following to call the subroutine:
*
*    CHARACTER*1024 FNAME
*    INTEGER        FIELD_NUM
*    INTEGER*4      FIELD_HDR_INTS(MDV_NUM_FIELD_HEADER_SI32)
*    REAL*4         FIELD_HDR_REALS(MDV_NUM_FIELD_HEADER_FL32)
*    CHARACTER*(MDV_LONG_FIELD_LEN) FIELD_NAME_LONG
*    CHARACTER*(MDV_SHORT_FIELD_LEN) FIELD_NAME_SHORT
*    CHARACTER*(MDV_UNITS_LEN) FIELD_UNITS
*    CHARACTER*(MDV_TRANSFORM_LEN) FIELD_TRANSFORM
*    CHARACTER*(MDV_UNITS_LEN) FIELD_UNUSED_CHAR
*    INTEGER        RETURN_STATUS
*
*   CALL MF_RF_READ_FIELD_HDR(INUNIT, FIELD_NUM,
*                             FIELD_HDR_INTS, FIELD_HDR_REALS,
*                             FIELD_NAME_LONG, FIELD_NAME_SHORT,
*                             FIELD_UNITS, FIELD_TRANSFORM, FIELD_UNUSED_CHAR,
*		              F_NAME,RETURN_STATUS)
*
* This function will open the file named FNAME and read the indicated field
* header.  The header information is returned in the given arrays.
* 
* When finished, RETURN_STATUS can be MDV_READ_SUCCESSFUL,
*                                     MDV_READ_OPEN_FAILURE,
*                                     MDV_READ_BAD_MASTER_HDR,
*                                     MDV_READ_INVALID_FIELD_NUM
*                                     MDV_READ_BAD_FIELD_HDR
*/

void readfieldhdr_(

			   char *fname,
                           int *field_num,
                           si32 *field_hdr_ints,
                           fl32 *field_hdr_reals,
                           char *field_name_long,
                           char *field_name_short,
                           char *field_units,
                           char *field_transform,
                           char *field_unused_char,
                           char f_name[8],
                           int  *return_status)
{

  extern struct files *open_files;
  MDV_field_header_t field_hdr;
  int i,i1,i2,len;
  void mf_field_hdr_to_fortran(); 
  int MDV_load_field_header();
  FILE *mdv_file;


  /* strip off trailing spaces */
  i=0;
  while (fname[i] != ' ')
    i++;
  fname[i]='\0';


  /*
   * Open the input file
   */

  if ((mdv_file = fopen(fname,"r")) == NULL)
  {
    fprintf(stderr, "Error opening input file\n");
    perror(fname);
    *return_status = MDV_READ_OPEN_FAILURE;
    return;
  }

  /*
   * Read the field header
   */

  if (MDV_load_field_header(mdv_file, &field_hdr, *field_num) != MDV_SUCCESS)
  {
    fprintf(stderr, "Error loading field %d header from file\n", *field_num);
    fclose(mdv_file);
    *return_status = MDV_READ_BAD_FIELD_HDR;
    return;
  }
  
  /*
   * Close the input file
   */

  fclose(mdv_file);

  /*
   * Copy the field header info
   */

  mf_field_hdr_to_fortran(&field_hdr,
                          field_hdr_ints,
                          field_hdr_reals,
                          field_name_long,
                          field_name_short,
                          field_units,
                          field_transform,
                          field_unused_char);
  
  *return_status = MDV_READ_SUCCESSFUL;

  strcpy(f_name,field_name_short);
  len = strlen(field_name_short);
  for(i = len; i < 8; i++) f_name[i] = ' ';
  return;
}
/******************************************************************************
 * MDV_LOAD_FIELD_HEADER: Load mdv field header data into the given structure
 * from disk.  Memory for the field header is assumed to be allocated before
 * this routine is called.  The bytes in the header are swapped if necessary
 * to put the data in native format.
 *
 * Inputs: infile - pointer to the input file.  This is assumed to currently
 *                  be open for read.
 *         f_hdr - pointer to the field header to be loaded.
 *         field_num - number of the field being loaded.  This is used to
 *                     determine the position on disk where the field header
 *                     information is located.
 *
 * Outputs: f_hdr - updated to include the values read in from disk, byte
 *                  swapped as necessary.
 *
 * Returns: MDV_SUCCESS or MDV_FAILURE.
 */

int MDV_load_field_header(FILE *infile, MDV_field_header_t *f_hdr,
			  int field_num)
{
    long hdr_offset;

     hdr_offset = sizeof(MDV_master_header_t) +
                  (field_num * sizeof(MDV_field_header_t));

     if (fseek(infile,hdr_offset,SEEK_SET)) 
     return MDV_FAILURE;

     if((fread(f_hdr,sizeof(MDV_field_header_t),1,infile)) != 1) {
        return MDV_FAILURE;
     }

     MDV_field_header_from_BE(f_hdr);

     return MDV_SUCCESS;
}
/***********************************************************************
 * mf_field_hdr_to_fortran:  Fills in the FORTRAN arrays from the given
 *                           MDV_field_header_t structure.
 */

void mf_field_hdr_to_fortran(MDV_field_header_t *field_hdr,
			     si32 *field_hdr_ints,
			     fl32 *field_hdr_reals,
			     char *field_name_long,
			     char *field_name_short,
			     char *field_units,
			     char *field_transform,
			     char *field_unused_char)
{
  /*
   * Load the information into the arrays.
   */

  memcpy(field_hdr_ints, &field_hdr->struct_id,
	 sizeof(si32) * MDV_NUM_FIELD_HEADER_SI32);

  memcpy(field_hdr_reals, &field_hdr->proj_origin_lat,
	 sizeof(fl32) * MDV_NUM_FIELD_HEADER_FL32);

  memcpy(field_name_long, field_hdr->field_name_long, MDV_LONG_FIELD_LEN);
  
  memcpy(field_name_short, field_hdr->field_name, MDV_SHORT_FIELD_LEN);
  
  memcpy(field_units, field_hdr->units, MDV_UNITS_LEN);
  
  memcpy(field_transform, field_hdr->transform, MDV_TRANSFORM_LEN);
  
  memcpy(field_unused_char, field_hdr->unused_char, MDV_UNITS_LEN);
  
  return;
}

/***********************************************************************
* MF_pf_print_field_hdr: This routine will print an MDV field header
* from FORTRAN format data.
*
* Use the following to call the subroutine:
*
*    INTEGER*4     FIELD_HDR_INTS(MDV_NUM_FIELD_HEADER_SI32)
*    REAL*4        FIELD_HDR_REALS(MDV_NUM_FIELD_HEADER_FL32)
*    CHARACTER*(MDV_LONG_FIELD_LEN) FIELD_NAME_LONG
*    CHARACTER*(MDV_SHORT_FIELD_LEN) FIELD_NAME_SHORT
*    CHARACTER*(MDV_UNITS_LEN) FIELD_UNITS
*    CHARACTER*(MDV_TRANSFORM_LEN) FIELD_TRANSFORM
*    CHARACTER*(MDV_UNITS_LEN) FIELD_UNUSED_CHAR
*
*   CALL MF_PF_PRINT_FIELD_HDR(FIELD_HDR_INTS, FIELD_HDR_REALS,
*                              FIELD_NAME_LONG, FIELD_NAME_SHORT,
*                              FIELD_UNITS, FIELD_TRANSFORM,
*                              FIELD_UNUSED_CHAR)
*
*/

void printfieldhdr_(

			    si32 *field_hdr_ints,
			    fl32 *field_hdr_reals,
			    char *field_name_long,
			    char *field_name_short,
			    char *field_units,
			    char *field_transform,
			    char *field_unused_char)
{
  MDV_field_header_t field_hdr;
  void mf_field_hdr_from_fortran();
  
  /*
   * Load the information in the field header
   */

  mf_field_hdr_from_fortran(field_hdr_ints,
			    field_hdr_reals,
			    field_name_long,
			    field_name_short,
			    field_units,
			    field_transform,
			    field_unused_char,
			    &field_hdr);
  
  /*
   * Print the field header
   */

  MDV_print_field_header_full(&field_hdr, stdout);
  
  return;
}

/*****************************************************************
 * MDV_PRINT_FIELD_HEADER_FULL: print out all of the field header
 * info
 * --Nancy Rehak 4/96
 */
 
void MDV_print_field_header_full(MDV_field_header_t *fld_hdr, FILE *outfile)
{
  int i;
  time_t print_time;         /* temporary storage area for printing time */
                             /*   values -- necessary on systems which have */
                             /*   time_t defined as other than 32-bit */
  
  
  fprintf(outfile, "\n");
  fprintf(outfile, "           MDV_print_field_header\n");
  fprintf(outfile, "           -----------------------\n");
  fprintf(outfile, "\n");
  fprintf(outfile, "record_len1:            %d\n", fld_hdr->record_len1);
  fprintf(outfile, "struct_id:              %d\n", fld_hdr->struct_id);
  fprintf(outfile, "\n");
  fprintf(outfile, "field_code:             %d\n", fld_hdr->field_code);
  fprintf(outfile, "user_time1:             %d\n", fld_hdr->user_time1);
  fprintf(outfile, "forecast_delta:         %d\n", fld_hdr->forecast_delta);
  fprintf(outfile, "user_time2:             %d\n", fld_hdr->user_time2);
  fprintf(outfile, "user_time3:             %d\n", fld_hdr->user_time3);
  print_time = fld_hdr->forecast_time;
  fprintf(outfile, "forecast_time:          %s",
	  asctime(gmtime(&print_time)));
  fprintf(outfile, "user_time4:             %d\n", fld_hdr->user_time4);
  fprintf(outfile, "nx:                     %d\n", fld_hdr->nx);
  fprintf(outfile, "ny:                     %d\n", fld_hdr->ny);
  fprintf(outfile, "nz:                     %d\n", fld_hdr->nz);
  fprintf(outfile, "proj_type:              %s\n",
	  MDV_proj2string(fld_hdr->proj_type));
  fprintf(outfile, "encoding_type:          %s\n",
	  MDV_encode2string(fld_hdr->encoding_type));
  fprintf(outfile, "data_element_nbytes:    %d\n",
	  fld_hdr->data_element_nbytes);
  fprintf(outfile, "field_data_offset:      %d\n", fld_hdr->field_data_offset);
  fprintf(outfile, "volume_size:            %d\n", fld_hdr->volume_size);
  for (i = 0; i < 10; i++)
    fprintf(outfile, "user_data_si32[%d]:      %d\n",
	    i, fld_hdr->user_data_si32[i]);
  
  fprintf(outfile, "\n");
  fprintf(outfile, "proj_origin_lon:        %f\n", fld_hdr->proj_origin_lon);
  fprintf(outfile, "proj_origin_lat:        %f\n", fld_hdr->proj_origin_lat);
  fprintf(outfile, "proj_rotation:          %f\n", fld_hdr->proj_rotation);
  for (i = 0; i < MDV_MAX_PROJ_PARAMS; i++)
    fprintf(outfile, "proj_param[%02d]:         %f\n",
	    i, fld_hdr->proj_param[i]);
  fprintf(outfile, "vert_reference:         %f\n", fld_hdr->vert_reference);
  fprintf(outfile, "\n");
  fprintf(outfile, "grid_dx:                %f\n", fld_hdr->grid_dx);
  fprintf(outfile, "grid_dy:                %f\n", fld_hdr->grid_dy);
  fprintf(outfile, "grid_dz:                %f\n", fld_hdr->grid_dz);
  fprintf(outfile, "grid_minx:              %f\n", fld_hdr->grid_minx);
  fprintf(outfile, "grid_miny:              %f\n", fld_hdr->grid_miny);
  fprintf(outfile, "grid_minz:              %f\n", fld_hdr->grid_minz);
  fprintf(outfile, "scale:                  %f\n", fld_hdr->scale);
  fprintf(outfile, "bias:                   %f\n", fld_hdr->bias);
  fprintf(outfile, "bad_data_value:         %f\n", fld_hdr->bad_data_value);
  fprintf(outfile, "missing_data_value:     %f\n",
	  fld_hdr->missing_data_value);
  fprintf(outfile, "proj_rotation:          %f\n", fld_hdr->proj_rotation);
  for (i = 0; i < 4; i++)
    fprintf(outfile, "user_data_fl32[%d]:      %f\n",
	    i, fld_hdr->user_data_fl32[i]);
  fprintf(outfile, "\n");
  fprintf(outfile, "field_name_long:        <%s>\n", fld_hdr->field_name_long);
  fprintf(outfile, "field_name:             <%s>\n", fld_hdr->field_name);
  fprintf(outfile, "units:                  <%s>\n", fld_hdr->units);
  fprintf(outfile, "transform:              <%s>\n", fld_hdr->transform);
  fprintf(outfile, "\n");
  fprintf(outfile, "record_len2:            %d\n", fld_hdr->record_len2);
  fprintf(outfile, "\n\n");
  
  return;
}

/***********************************************************************
* MF_RV_READ_VLEVEL_HDR: Read the indicated vlevel header from a data file
* and return the information in FORTRAN-usable structures.
*
* Use the following to call the subroutine:
*
*    CHARACTER*1024 FNAME
*    INTEGER        FIELD_NUM
*    INTEGER*4      VLEVEL_HDR_INTS(MDV_NUM_VLEVEL_HEADER_SI32)
*    REAL*4         VLEVEL_HDR_REALS(MDV_NUM_VLEVEL_HEADER_FL32)
*    INTEGER        RETURN_STATUS
*
*   CALL MF_RV_READ_VLEVEL_HDR(FNAME, FIELD_NUM,
*                              VLEVEL_HDR_INTS, VLEVEL_HDR_REALS,
*	                       RETURN_STATUS)
*
* This function will open the file named FNAME and read the indicated vlevel
* header.  The header information is returned in the given arrays.
* 
* When finished, RETURN_STATUS can be MDV_READ_SUCCESSFUL,
*                                     MDV_READ_OPEN_FAILURE,
*                                     MDV_READ_BAD_MASTER_HDR,
*                                     MDV_READ_INVALID_FIELD_NUM,
*                                     MDV_READ_BAD_VLEVEL_HDR
*/

void readvlevelhdr_(

			    char *fname,
			    int *field_num,
			    si32 *vlevel_hdr_ints,
			    fl32 *vlevel_hdr_reals,
                           int  *return_status)
{
  FILE *mdv_file;

  MDV_master_header_t master_hdr;
  MDV_vlevel_header_t vlevel_hdr;
  int i,i1,i2;
  void mf_vlevel_hdr_to_fortran();

  
  /*
   * Open the input file
   */

  /* strip off trailing spaces */
  i=0;
  while (fname[i] != ' ')
    i++;
  fname[i]='\0';


  if ((mdv_file = fopen(fname,"r")) == NULL)
  {
    fprintf(stderr, "Error opening input file\n");
    perror(fname);
    *return_status = MDV_READ_OPEN_FAILURE;
    return;
  }

  /*
   * Read the master header
   */

  if (MDV_load_master_header(mdv_file, &master_hdr) == MDV_FAILURE)
  {
    fprintf(stderr, "Error reading master header from file <%s>\n",
	    fname);
    fclose(mdv_file);
    *return_status = MDV_READ_BAD_MASTER_HDR;
    return;
  }

  /*
   * Check to make sure the field number is valid
   */

  if (*field_num >= master_hdr.n_fields)
  {
    fprintf(stderr, "Invalid field number %d given, file only has %d fields\n",
            *field_num, master_hdr.n_fields);
    fclose(mdv_file);
    *return_status = MDV_READ_INVALID_FIELD_NUM;
    return;
  }
  
  /*
   * Check to make sure vlevel headers are included in this file
   */

  if (!master_hdr.vlevel_included)
  {
    fprintf(stderr,
	    "Trying to read vlevel headers from file not including them\n");
    fclose(mdv_file);
    *return_status = MDV_READ_NO_VLEVEL_HDRS;
    return;
  }
  
  /*
   * Read the vlevel header
   */

  if (MDV_load_vlevel_header(mdv_file, &vlevel_hdr,
			     &master_hdr, *field_num) != MDV_SUCCESS)
  {
    fprintf(stderr, "Error loading field %d vlevelheader from file\n",
	    *field_num);
    fclose(mdv_file);
    *return_status = MDV_READ_BAD_VLEVEL_HDR;
    return;
  }
  
  /*
   * Close the input file
   */

  fclose(mdv_file);

  /*
   * Copy the vlevel header info
   */

  mf_vlevel_hdr_to_fortran(&vlevel_hdr,
			   vlevel_hdr_ints,
			   vlevel_hdr_reals);
  
  *return_status = MDV_READ_SUCCESSFUL;
     
  return;
}

/***********************************************************************
 * mf_vlevel_hdr_to_fortran:  Fills in the FORTRAN arrays from the given
 *                            MDV_vlevel_header_t structure.
 */

void mf_vlevel_hdr_to_fortran(MDV_vlevel_header_t *vlevel_hdr,
			      si32 *vlevel_hdr_ints,
			      fl32 *vlevel_hdr_reals)
{
  /*
   * Load the information into the arrays.
   */

  memcpy(vlevel_hdr_ints, &vlevel_hdr->struct_id,
	 sizeof(si32) * MDV_NUM_VLEVEL_HEADER_SI32);

  memcpy(vlevel_hdr_reals, vlevel_hdr->vlevel_params,
	 sizeof(fl32) * MDV_NUM_VLEVEL_HEADER_FL32);

  return;
}

/***********************************************************************
* MF_pv_print_vlevel_hdr: This routine will print an MDV vlevel header
* from FORTRAN format data.
*
* Use the following to call the subroutine:
*
*    INTEGER*4     VLEVEL_HDR_INTS(MDV_NUM_VLEVEL_HEADER_SI32)
*    REAL*4        VLEVEL_HDR_REALS(MDV_NUM_VLEVEL_HEADER_FL32)
*    INTEGER*4     NUM_Z
*    CHARACTER*(MDV_LONG_FIELD_LEN) FIELD_NAME
*
*   CALL MF_PV_PRINT_VLEVEL_HDR(VLEVEL_HDR_INTS, VLEVEL_HDR_REALS,
*                               NUM_Z, FIELD_NAME)
*
*/

void printvlevelhdr_(

			     si32 *vlevel_hdr_ints,
			     fl32 *vlevel_hdr_reals,
			     int *num_z,
			     char *field_name)
{
  MDV_vlevel_header_t vlevel_hdr;
  void mf_vlevel_hdr_from_fortran();


  /*
   * Load the information in the vlevel header
   */

  mf_vlevel_hdr_from_fortran(vlevel_hdr_ints,
			     vlevel_hdr_reals,
			     &vlevel_hdr);
  
  /*
   * Print the vlevel header
   */

  MDV_print_vlevel_header_full(&vlevel_hdr, *num_z,
			       field_name, stdout);
  
  return;
}

/***********************************************************************
 * mf_vlevel_hdr_from_fortran:  Fills in the MDV_vlevel_header_t structure
 *                              from the given FORTRAN arrays.
 */

void mf_vlevel_hdr_from_fortran(si32 *vlevel_hdr_ints,
				fl32 *vlevel_hdr_reals,
				MDV_vlevel_header_t *vlevel_hdr)
{
  /*
   * Load the information into the vlevel header.
   */

  memcpy(&vlevel_hdr->struct_id, vlevel_hdr_ints,
	 sizeof(si32) * MDV_NUM_VLEVEL_HEADER_SI32);

  memcpy(vlevel_hdr->vlevel_params, vlevel_hdr_reals,
	 sizeof(fl32) * MDV_NUM_VLEVEL_HEADER_FL32);

  /*
   * Set the record length values
   */

  vlevel_hdr->record_len1 = sizeof(MDV_vlevel_header_t) - 2 * sizeof(si32);
  vlevel_hdr->record_len2 = vlevel_hdr->record_len1;
  
  return;
}

/*****************************************************************
 * MDV_PRINT_VLEVEL_HEADER_FULL: print out all of the vlevel
 * header info
 * --Nancy Rehak 4/96
 */

void MDV_print_vlevel_header_full(MDV_vlevel_header_t *mvh, int nz, 
				  char *field_name,FILE *outfile)
{ 
  int i;
  
  fprintf(outfile, "\n");
  fprintf(outfile, "           Vlevel_header for %s\n",field_name);
  fprintf(outfile, "           -----------------------\n");
  fprintf(outfile, "\n");
  fprintf(outfile, "record_len1:             %d\n", mvh->record_len1);
  fprintf(outfile, "struct_id:               %d\n", mvh->struct_id);
  fprintf(outfile, "\n");
  for (i = 0; i < nz; i++)
  {
    fprintf(outfile, "vlevel_type[%02d]:         %s\n",
	    i, MDV_verttype2string(mvh->vlevel_type[i]));
    
    fprintf(outfile, "vlevel_params[%02d]:       %f\n",
	    i, mvh->vlevel_params[i]);
  }
  fprintf(outfile, "\n");
  fprintf(outfile, "record_len2:             %d\n", mvh->record_len2);
  fprintf(outfile, "\n\n");
  
  return;
}
 
/******************************************************************************
 * MDV_LOAD_VLEVEL_HEADER: Load mdv vlevel header data into the given
 * structure from disk.  Memory for the vlevel header is assumed to be
 * allocated before this routine is called.  The bytes in the header are
 * swapped if necessary to put the data in native format.
 *
 * Inputs: infile - pointer to the input file.  This is assumed to currently
 *                  be open for read.
 *         v_hdr - pointer to the vlevel header to be loaded.
 *         m_hdr - pointer to the associated master header information.  This
 *                 information is used to determine the position on disk
 *                 where the vlevel header information is located.
 *         field_num - the field number.  This information is also used to
 *                     determine the disk location.
 *
 * Output: v_hdr - updated to include the values read in from disk,
 *                 byte swapped as necessary.
 *
 * Returns: MDV_SUCCESS or MDV_FAILURE.
 */
 
int MDV_load_vlevel_header( FILE *infile, MDV_vlevel_header_t *v_hdr, 
                             MDV_master_header_t *m_hdr, int field_num)
{
     long hdr_offset;
 
     /*
      * Make sure that vlevel headers are included in this data.
      */

     if (!m_hdr->vlevel_included)
       return MDV_FAILURE;
     
     hdr_offset = m_hdr->vlevel_hdr_offset + 
                  (field_num * sizeof(MDV_vlevel_header_t));
 
     if (fseek(infile,hdr_offset,SEEK_SET)) 
         return MDV_FAILURE;

     if((fread(v_hdr,sizeof(MDV_vlevel_header_t),1,infile)) != 1) {
        return MDV_FAILURE;
     }

     MDV_vlevel_header_from_BE(v_hdr);
 
     return MDV_SUCCESS;
}


/******************************************************************************
 * MDV_LOAD_VLEVEL_HEADER_OFFSET: Load mdv vlevel header data into the given
 * structure from disk given the offset for the first vlevel header in the
 * file rather than the master header.  Memory for the vlevel header is
 * assumed to be allocated before this routine is called.  The bytes in the
 * header are swapped if necessary to put the data in native format.
 *
 * Inputs: infile - pointer to the input file.  This is assumed to currently
 *                  be open for read.
 *         v_hdr - pointer to the vlevel header to be loaded.
 *         vlevel_offset - offset in the file to the first vlevel header.
 *                         This information is used to determine the disk
 *                         location of the desired vlevel header.
 *         field_num - the field number.  This information is also used to
 *                     determine the disk location.
 *
 * Output: v_hdr - updated to include the values read in from disk,
 *                 byte swapped as necessary.
 *
 * Returns: MDV_SUCCESS or MDV_FAILURE.
 */
 
int MDV_load_vlevel_header_offset(FILE *infile,
				  MDV_vlevel_header_t *v_hdr, 
				  int vlevel_offset,
				  int field_num)
{
     long hdr_offset;
     //MDV_vlevel_header_from_BE();
 
     hdr_offset = vlevel_offset + 
                  (field_num * sizeof(MDV_vlevel_header_t));
 
     if (fseek(infile, hdr_offset, SEEK_SET)) 
         return MDV_FAILURE;

     if ((fread(v_hdr, sizeof(MDV_vlevel_header_t), 1, infile)) != 1)
     {
        return MDV_FAILURE;
     }

     MDV_vlevel_header_from_BE(v_hdr);
 
     return MDV_SUCCESS;
}

/***********************************************************************
* MF_RFD_READ_FIELD_DATA: Read the data for the indicated field
* from a data file and return the information in FORTRAN-usable structures.
*
* Use the following to call the subroutine:
*
*    INTEGER        INUNIT
*    INTEGER        FIELD_NUM
*    CHARACTER*(ARRAY_SIZE) FIELD_DATA_ARRAY
*    INTEGER        ARRAY_SIZE
*    INTEGER        REQESTED_DATA_FORMAT
*    INTEGER        RETURN_DATA_SIZE
*    INTEGER        RETURN_STATUS
*
*   CALL MF_RFD_READ_FIELD_DATA(INUNIT, FIELD_NUM,
*                               FIELD_DATA_ARRAY, ARRAY_SIZE,
*                               REQUESTED_DATA_FORMAT,
*                               RETURN_DATA_SIZE,
*	                        RETURN_STATUS)
*
* This function will open the file named FNAME and read the indicated field
* data.  The data is returned in the given arrays.
* 
* When finished, RETURN_STATUS can be MDV_READ_SUCCESSFUL,
*                                     MDV_READ_OPEN_FAILURE,
*                                     MDV_READ_BAD_MASTER_HDR,
*                                     MDV_READ_INVALID_FIELD_NUM,
*                                     MDV_READ_BAD_FIELD_HDR,
*                                     MDV_READ_DATA_ARRAY_TOO_SMALL,
*                                     MDV_READ_DATA_ERROR
*/

void readfielddata_(

			     char *fname,
                             int *field_num,
                             char *field_data_array,
                             int *array_size,
                             int *requested_data_format,
                             int *return_data_size,
                             int *return_status)
{
  FILE *mdv_file;

  MDV_master_header_t master_hdr;
  MDV_field_header_t field_hdr;
  void *volume_data;
  int i,i1,i2;

  
  /*
   * Open the input file
   */

  /* strip off trailing spaces */
  i=0;
  while (fname[i] != ' ')
    i++;
  fname[i]='\0';

 	
  if ((mdv_file = fopen(fname,"r")) == NULL)
  {
    fprintf(stderr, "Error opening input file\n");
    perror(fname);
    *return_status = MDV_READ_OPEN_FAILURE;
    return;
  }

  /*
   * Read the master header
   */

  if (MDV_load_master_header(mdv_file, &master_hdr) == MDV_FAILURE)
  {
    fprintf(stderr, "Error reading master header from file <%s>\n",
	    fname);
    fclose(mdv_file);
    *return_status = MDV_READ_BAD_MASTER_HDR;
    return;
  }

  /*
   * Check to make sure the field number is valid
   */

  if (*field_num >= master_hdr.n_fields)
  {
    fprintf(stderr, "Invalid field number %d given, file only has %d fields\n",
            *field_num, master_hdr.n_fields);
    fclose(mdv_file);
    *return_status = MDV_READ_INVALID_FIELD_NUM;
    return;
  }
  
  /*
   * Read the field header
   */

  if (MDV_load_field_header(mdv_file, &field_hdr,
                            *field_num) != MDV_SUCCESS)
  {
    fprintf(stderr, "Error loading field %d header from file\n",
	    *field_num);
    fclose(mdv_file);
    *return_status = MDV_READ_BAD_FIELD_HDR;
    return;
  }
  
  /*
   * Read the data
   */

// added (void *) on 20060919 ahijevych to avoid compilation warning about making pointer from integer w/o cast
  if ((volume_data = (void *)MDV_get_volume_size(mdv_file, &field_hdr,
					 *requested_data_format,
					 return_data_size)) == NULL)
  {
    printf("Error loading data for field %d from file\n",
            *field_num);
    fclose(mdv_file);
    *return_data_size = 0;
    *return_status = MDV_READ_DATA_ERROR;
    return;
  }
  
  /*
   * Close the input file
   */

  fclose(mdv_file);

  /*
   * Make sure the return array is big enough for the data
   */

  if (*return_data_size > *array_size)
  {
    fprintf(stderr,
	    "Given array not big enough for data -- array has %d bytes, data has %d bytes\n",
	    *array_size, *return_data_size);
    fclose(mdv_file);
    *return_status = MDV_READ_DATA_ARRAY_TOO_SMALL;
    return;
  }
  
  /*
   * Copy the data into the returned array
   */

  memcpy(field_data_array, volume_data, *return_data_size);
  free(volume_data);
  
  
  *return_status = MDV_READ_SUCCESSFUL;
     
  return;
}


