MODULE final_analysis

   USE first_guess

CONTAINS

!-------------------------------------------------------------------------------

SUBROUTINE open_final_analysis ( unit , domain_id )

!  This routine OPENs the file for the final analysis.

   IMPLICIT NONE

   INTEGER                 :: unit , domain_id
   CHARACTER (LEN=16)      :: name

   LOGICAL                 :: tf

   INCLUDE 'error.inc'

   INTERFACE
      INCLUDE 'error.int'
   END INTERFACE

   !  Check to see if this file is already OPENed.

   WRITE ( name , '("LITTLE_R_DOMAIN",I1)' ) domain_id

   INQUIRE ( FILE = name , OPENED = tf )

   !  OPEN the file, yes or no?

   already_open : IF ( tf ) THEN
      error_number = 03912001
      error_message(1:31) = 'open_final_analysis            '
      error_message(32:)  = ' The file ' // TRIM(name) // &
      ' is already OPEN.'
      fatal = .false.
      listing = .false.
      CALL error_handler ( error_number , error_message , &
      fatal , listing )

   ELSE already_open
      OPEN ( UNIT   = unit          , &
             FILE   = name          , &
             ACCESS = 'SEQUENTIAL'  , &
             STATUS = 'REPLACE'     , &
             FORM   = 'UNFORMATTED' , &
             IOSTAT = error_number      )
      error_open : IF ( error_number .NE. 0 ) THEN
         error_number = error_number + 03912000
         error_message(1:31) = 'open_final_analysis            '
         error_message(32:)  = ' Error OPENing final analysis file: '// TRIM(name)  // '.'
         fatal = .true.
         listing = .false.
         CALL error_handler ( error_number , error_message , &
         fatal , listing )
      ENDIF error_open
   ENDIF already_open

END SUBROUTINE open_final_analysis


!-------------------------------------------------------------------------------

SUBROUTINE open_final_analysis_fdda ( unit , domain_id )

!  This routine OPENs the file for the final analysis for the surface FDDA file.

   IMPLICIT NONE

   INTEGER                 :: unit , domain_id
   CHARACTER (LEN=15)      :: name

   LOGICAL                 :: tf

   INCLUDE 'error.inc'

   INTERFACE
      INCLUDE 'error.int'
   END INTERFACE

   !  Check to see if this file is already OPENed.

   WRITE ( name , '("SFCFDDA_DOMAIN",I1)' ) domain_id

   INQUIRE ( FILE = name , OPENED = tf )

   !  OPEN the file, yes or no?

   already_open : IF ( tf ) THEN
      error_number = 03912002
      error_message(1:31) = 'open_final_analysis            '
      error_message(32:)  = ' The file ' // TRIM(name) // &
      ' is already OPEN.'
      fatal = .false.
      listing = .false.
      CALL error_handler ( error_number , error_message , &
      fatal , listing )

   ELSE already_open
      OPEN ( UNIT   = unit          , &
             FILE   = name          , &
             ACCESS = 'SEQUENTIAL'  , &
             STATUS = 'REPLACE'     , &
             FORM   = 'UNFORMATTED' , &
             IOSTAT = error_number      )
      error_open : IF ( error_number .NE. 0 ) THEN
         error_number = error_number + 03912003
         error_message(1:31) = 'open_final_analysis            '
         error_message(32:)  = ' Error OPENing final analysis SFC FDDA file: '// TRIM(name)  // '.'
         fatal = .true.
         listing = .false.
         CALL error_handler ( error_number , error_message , &
         fatal , listing )
      ENDIF error_open
   ENDIF already_open

END SUBROUTINE open_final_analysis_fdda

!------------------------------------------------------------------------------

SUBROUTINE printout_header(bhi , bhr , bhic , bhrc ) 

!  This routine prints out the entire record header.  The data is broken into
!  integer and real data, with the character data printed along side, as an
!  aid for understanding.

   IMPLICIT NONE

   INCLUDE 'big_header.inc'

   INTEGER :: i , & ! loop index
              j     ! loop index

   program_loop : DO j = 1 , bhi(1,1)

      IF      ( j .eq. 1 ) THEN
         WRITE ( UNIT = * , FMT = &
         ' ( / "TERRAIN portion of record header:" ) ' )
      ELSE IF ( j .eq. 2 ) THEN
         WRITE ( UNIT = * , FMT = &
         ' ( / "REGRID portion of record header:" ) ' )
      ELSE IF ( j .eq. 3 ) THEN
         WRITE ( UNIT = * , FMT = &
         ' ( / "little_r portion of record header:" ) ' )
      ELSE IF ( j .eq. 4 ) THEN
         WRITE ( UNIT = * , FMT = &
         ' ( / "RAWINS - FDDA portion of record header:" ) ' )
      ENDIF

      WRITE ( UNIT = * , FMT = ' ( "  Integers: " ) ' )

      integers : DO i = 1 , 50
         IF ( bhi(i,j) .ne. -999 ) THEN
            WRITE ( UNIT = * , FMT = 100 ) i , j , bhi(i,j) ,  bhic(i,j)
         ENDIF
      END DO integers
 100  FORMAT( '     BHI(' , I4 , ',' , I2 , ') = ' , I10 , ' : ' , A80 )

      WRITE ( UNIT = * , FMT = ' ( / "  Floating Points: " ) ' )

      reals : DO i = 1 , 20
         IF ( ABS ( bhr(i,j) + 999. ) .GT. 1.E-10 ) THEN
            WRITE ( UNIT = * , FMT = 101 ) i , j , bhr(i,j) , bhrc(i,j)
         ENDIF
      END DO reals
 101  FORMAT( '     BHR(' , I4 , ',' , I2 , ') = ' , F10.5 , ' : ' , A80 )
   END DO program_loop

END SUBROUTINE printout_header

!------------------------------------------------------------------------------

SUBROUTINE write_analysis ( unit , &
t , u , v , h , rh , &
terrain , land_use , map_factor_x , map_factor_d , coriolis , &
latitude_x , longitude_x , latitude_d , longitude_d , &
snow_cover , slp_d , slp_x , sst , tobbox , &
iew_alloc , jns_alloc , kbu_alloc , iewd , jnsd , print_analysis )

!  This routine assembles the correct data fields and outputs them with the
!  appropriate small header.

   USE input_data
   USE header_flags

   IMPLICIT NONE 

   INTEGER         :: unit

   INCLUDE 'big_header.inc'
   INCLUDE 'first_guess_size.inc'
   INCLUDE 'first_guess.inc'
   INCLUDE 'error.inc'
 
   REAL , DIMENSION ( jns_alloc , iew_alloc )   :: tobbox
   INTEGER                                      :: iewd , jnsd
   REAL , DIMENSION ( jnsd , iewd , kbu_alloc ) :: dum3d
   REAL , DIMENSION ( jnsd , iewd )             :: dum2d

   INTERFACE 
      INCLUDE 'error.int'
      INCLUDE 'sample.int'
   END INTERFACE
   
   INTEGER :: loop_count , &
              num3d , &
              num2d , &
              num1d , &
              num0d
   REAL,PARAMETER    :: dummy_value = -99999.
   INTEGER :: extra_loop
   LOGICAL :: print_analysis

   !  We need to keep track of where we are sticking the data for the FDDA option.

   IF ( initial_time ) THEN
      tt = first_time
   ELSE
      tt = second_time
   END IF
 
   !  Do all of the 3d data.
             
   three_D : DO loop_count = 1 , loop3
      IF      ( all_3d(loop_count,tt)%small_header%name(1:8) .EQ. 'T       ' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_3d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, kbu_alloc , 1 /)
         WRITE ( UNIT = unit )  &
         all_3d(loop_count,tt)%small_header%num_dims    , all_3d(loop_count,tt)%small_header%start_dims  , &
         all_3d(loop_count,tt)%small_header%end_dims    , all_3d(loop_count,tt)%small_header%xtime       , &
         all_3d(loop_count,tt)%small_header%staggering  , all_3d(loop_count,tt)%small_header%ordering    , &
         all_3d(loop_count,tt)%small_header%current_date, all_3d(loop_count,tt)%small_header%name        , &
         all_3d(loop_count,tt)%small_header%units       , all_3d(loop_count,tt)%small_header%description
         CALL unexpand3 ( t  , iew_alloc , jns_alloc , kbu_alloc , dum3d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum3d
      ELSE IF ( all_3d(loop_count,tt)%small_header%name(1:8) .EQ. 'U       ' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_3d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, kbu_alloc , 1 /)
         WRITE ( UNIT = unit )  &
         all_3d(loop_count,tt)%small_header%num_dims    , all_3d(loop_count,tt)%small_header%start_dims  , &
         all_3d(loop_count,tt)%small_header%end_dims    , all_3d(loop_count,tt)%small_header%xtime       , &
         all_3d(loop_count,tt)%small_header%staggering  , all_3d(loop_count,tt)%small_header%ordering    , &
         all_3d(loop_count,tt)%small_header%current_date, all_3d(loop_count,tt)%small_header%name        , &
         all_3d(loop_count,tt)%small_header%units       , all_3d(loop_count,tt)%small_header%description
         CALL unexpand3 ( u  , iew_alloc , jns_alloc , kbu_alloc , dum3d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum3d
      ELSE IF ( all_3d(loop_count,tt)%small_header%name(1:8) .EQ. 'V       ' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_3d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, kbu_alloc , 1 /)
         WRITE ( UNIT = unit )  &
         all_3d(loop_count,tt)%small_header%num_dims    , all_3d(loop_count,tt)%small_header%start_dims  , &
         all_3d(loop_count,tt)%small_header%end_dims    , all_3d(loop_count,tt)%small_header%xtime       , &
         all_3d(loop_count,tt)%small_header%staggering  , all_3d(loop_count,tt)%small_header%ordering    , &
         all_3d(loop_count,tt)%small_header%current_date, all_3d(loop_count,tt)%small_header%name        , &
         all_3d(loop_count,tt)%small_header%units       , all_3d(loop_count,tt)%small_header%description
         CALL unexpand3 ( v  , iew_alloc , jns_alloc , kbu_alloc , dum3d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum3d
      ELSE IF ( all_3d(loop_count,tt)%small_header%name(1:8) .EQ. 'H       ' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_3d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, kbu_alloc , 1 /)
         WRITE ( UNIT = unit )  &
         all_3d(loop_count,tt)%small_header%num_dims    , all_3d(loop_count,tt)%small_header%start_dims  , &
         all_3d(loop_count,tt)%small_header%end_dims    , all_3d(loop_count,tt)%small_header%xtime       , &
         all_3d(loop_count,tt)%small_header%staggering  , all_3d(loop_count,tt)%small_header%ordering    , &
         all_3d(loop_count,tt)%small_header%current_date, all_3d(loop_count,tt)%small_header%name        , &
         all_3d(loop_count,tt)%small_header%units       , all_3d(loop_count,tt)%small_header%description
         CALL unexpand3 ( h  , iew_alloc , jns_alloc , kbu_alloc , dum3d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum3d
      ELSE IF ( all_3d(loop_count,tt)%small_header%name(1:8) .EQ. 'RH      ' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_3d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, kbu_alloc , 1 /)
         WRITE ( UNIT = unit )  &
         all_3d(loop_count,tt)%small_header%num_dims    , all_3d(loop_count,tt)%small_header%start_dims  , &
         all_3d(loop_count,tt)%small_header%end_dims    , all_3d(loop_count,tt)%small_header%xtime       , &
         all_3d(loop_count,tt)%small_header%staggering  , all_3d(loop_count,tt)%small_header%ordering    , &
         all_3d(loop_count,tt)%small_header%current_date, all_3d(loop_count,tt)%small_header%name        , &
         all_3d(loop_count,tt)%small_header%units       , all_3d(loop_count,tt)%small_header%description
         CALL unexpand3 ( rh , iew_alloc , jns_alloc , kbu_alloc , dum3d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum3d
      ELSE IF ( ( all_3d(loop_count,tt)%small_header%name(1:8) .EQ. 'W       ' ) .OR. &
                ( all_3d(loop_count,tt)%small_header%name(1:8) .EQ. 'PP      ' ) .OR. &
                ( all_3d(loop_count,tt)%small_header%name(1:8) .EQ. 'RAD TEND' ) ) THEN
!        PRINT '(A,A,A)','Skipping the output of the 3d field, ', all_3d(loop_count,tt)%small_header%name(1:8), &
!                        ' since this would confuse INTERPF.'
      ELSE
         WRITE ( UNIT = unit ) sh_flag
         all_3d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, kbu_alloc , 1 /)
         WRITE ( UNIT = unit )  &
         all_3d(loop_count,tt)%small_header%num_dims    , all_3d(loop_count,tt)%small_header%start_dims  , &
         all_3d(loop_count,tt)%small_header%end_dims    , all_3d(loop_count,tt)%small_header%xtime       , &
         all_3d(loop_count,tt)%small_header%staggering  , all_3d(loop_count,tt)%small_header%ordering    , &
         all_3d(loop_count,tt)%small_header%current_date, all_3d(loop_count,tt)%small_header%name        , &
         all_3d(loop_count,tt)%small_header%units       , all_3d(loop_count,tt)%small_header%description
         CALL unexpand3 ( all_3d(loop_count,tt)%array , iew_alloc , jns_alloc , kbu_alloc , dum3d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum3d
      END IF
      
   END DO three_D

   !  Now, do the 2d data.

   extra_loop = 0
   two_D : DO loop_count = 1 , loop2
      IF      (all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'TERRAIN ' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_2d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, 1 , 1 /)
         WRITE ( UNIT = unit )  &
         all_2d(loop_count,tt)%small_header%num_dims    , all_2d(loop_count,tt)%small_header%start_dims  , &
         all_2d(loop_count,tt)%small_header%end_dims    , all_2d(loop_count,tt)%small_header%xtime       , &
         all_2d(loop_count,tt)%small_header%staggering  , all_2d(loop_count,tt)%small_header%ordering    , &
         all_2d(loop_count,tt)%small_header%current_date, all_2d(loop_count,tt)%small_header%name        , &
         all_2d(loop_count,tt)%small_header%units       , all_2d(loop_count,tt)%small_header%description
         CALL unexpand2 ( terrain      , iew_alloc , jns_alloc , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
      ELSE IF (all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'LAND USE' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_2d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, 1 , 1 /)
         WRITE ( UNIT = unit )  &
         all_2d(loop_count,tt)%small_header%num_dims    , all_2d(loop_count,tt)%small_header%start_dims  , &
         all_2d(loop_count,tt)%small_header%end_dims    , all_2d(loop_count,tt)%small_header%xtime       , &
         all_2d(loop_count,tt)%small_header%staggering  , all_2d(loop_count,tt)%small_header%ordering    , &
         all_2d(loop_count,tt)%small_header%current_date, all_2d(loop_count,tt)%small_header%name        , &
         all_2d(loop_count,tt)%small_header%units       , all_2d(loop_count,tt)%small_header%description
         CALL unexpand2 ( land_use     , iew_alloc , jns_alloc , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
      ELSE IF (all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'MAPFACCR' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_2d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, 1 , 1 /)
         WRITE ( UNIT = unit )  &
         all_2d(loop_count,tt)%small_header%num_dims    , all_2d(loop_count,tt)%small_header%start_dims  , &
         all_2d(loop_count,tt)%small_header%end_dims    , all_2d(loop_count,tt)%small_header%xtime       , &
         all_2d(loop_count,tt)%small_header%staggering  , all_2d(loop_count,tt)%small_header%ordering    , &
         all_2d(loop_count,tt)%small_header%current_date, all_2d(loop_count,tt)%small_header%name        , &
         all_2d(loop_count,tt)%small_header%units       , all_2d(loop_count,tt)%small_header%description
         CALL unexpand2 ( map_factor_x , iew_alloc , jns_alloc , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
      ELSE IF (all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'MAPFACDT' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_2d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, 1 , 1 /)
         WRITE ( UNIT = unit )  &
         all_2d(loop_count,tt)%small_header%num_dims    , all_2d(loop_count,tt)%small_header%start_dims  , &
         all_2d(loop_count,tt)%small_header%end_dims    , all_2d(loop_count,tt)%small_header%xtime       , &
         all_2d(loop_count,tt)%small_header%staggering  , all_2d(loop_count,tt)%small_header%ordering    , &
         all_2d(loop_count,tt)%small_header%current_date, all_2d(loop_count,tt)%small_header%name        , &
         all_2d(loop_count,tt)%small_header%units       , all_2d(loop_count,tt)%small_header%description
         CALL unexpand2 ( map_factor_d , iew_alloc , jns_alloc , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
      ELSE IF (all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'CORIOLIS' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_2d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, 1 , 1 /)
         WRITE ( UNIT = unit )  &
         all_2d(loop_count,tt)%small_header%num_dims    , all_2d(loop_count,tt)%small_header%start_dims  , &
         all_2d(loop_count,tt)%small_header%end_dims    , all_2d(loop_count,tt)%small_header%xtime       , &
         all_2d(loop_count,tt)%small_header%staggering  , all_2d(loop_count,tt)%small_header%ordering    , &
         all_2d(loop_count,tt)%small_header%current_date, all_2d(loop_count,tt)%small_header%name        , &
         all_2d(loop_count,tt)%small_header%units       , all_2d(loop_count,tt)%small_header%description
         CALL unexpand2 ( coriolis     , iew_alloc , jns_alloc , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
      ELSE IF (all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'LATITCRS' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_2d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, 1 , 1 /)
         WRITE ( UNIT = unit )  &
         all_2d(loop_count,tt)%small_header%num_dims    , all_2d(loop_count,tt)%small_header%start_dims  , &
         all_2d(loop_count,tt)%small_header%end_dims    , all_2d(loop_count,tt)%small_header%xtime       , &
         all_2d(loop_count,tt)%small_header%staggering  , all_2d(loop_count,tt)%small_header%ordering    , &
         all_2d(loop_count,tt)%small_header%current_date, all_2d(loop_count,tt)%small_header%name        , &
         all_2d(loop_count,tt)%small_header%units       , all_2d(loop_count,tt)%small_header%description
         CALL unexpand2 ( latitude_x   , iew_alloc , jns_alloc , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
      ELSE IF (all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'LONGICRS' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_2d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, 1 , 1 /)
         WRITE ( UNIT = unit )  &
         all_2d(loop_count,tt)%small_header%num_dims    , all_2d(loop_count,tt)%small_header%start_dims  , &
         all_2d(loop_count,tt)%small_header%end_dims    , all_2d(loop_count,tt)%small_header%xtime       , &
         all_2d(loop_count,tt)%small_header%staggering  , all_2d(loop_count,tt)%small_header%ordering    , &
         all_2d(loop_count,tt)%small_header%current_date, all_2d(loop_count,tt)%small_header%name        , &
         all_2d(loop_count,tt)%small_header%units       , all_2d(loop_count,tt)%small_header%description
         CALL unexpand2 ( longitude_x  , iew_alloc , jns_alloc , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
      ELSE IF (all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'LATITDOT' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_2d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, 1 , 1 /)
         WRITE ( UNIT = unit )  &
         all_2d(loop_count,tt)%small_header%num_dims    , all_2d(loop_count,tt)%small_header%start_dims  , &
         all_2d(loop_count,tt)%small_header%end_dims    , all_2d(loop_count,tt)%small_header%xtime       , &
         all_2d(loop_count,tt)%small_header%staggering  , all_2d(loop_count,tt)%small_header%ordering    , &
         all_2d(loop_count,tt)%small_header%current_date, all_2d(loop_count,tt)%small_header%name        , &
         all_2d(loop_count,tt)%small_header%units       , all_2d(loop_count,tt)%small_header%description
         CALL unexpand2 ( latitude_d   , iew_alloc , jns_alloc , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
      ELSE IF (all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'LONGIDOT' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_2d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, 1 , 1 /)
         WRITE ( UNIT = unit )  &
         all_2d(loop_count,tt)%small_header%num_dims    , all_2d(loop_count,tt)%small_header%start_dims  , &
         all_2d(loop_count,tt)%small_header%end_dims    , all_2d(loop_count,tt)%small_header%xtime       , &
         all_2d(loop_count,tt)%small_header%staggering  , all_2d(loop_count,tt)%small_header%ordering    , &
         all_2d(loop_count,tt)%small_header%current_date, all_2d(loop_count,tt)%small_header%name        , &
         all_2d(loop_count,tt)%small_header%units       , all_2d(loop_count,tt)%small_header%description
         CALL unexpand2 ( longitude_d  , iew_alloc , jns_alloc , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
      ELSE IF (all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'SNOWCOVR' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_2d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, 1 , 1 /)
         WRITE ( UNIT = unit )  &
         all_2d(loop_count,tt)%small_header%num_dims    , all_2d(loop_count,tt)%small_header%start_dims  , &
         all_2d(loop_count,tt)%small_header%end_dims    , all_2d(loop_count,tt)%small_header%xtime       , &
         all_2d(loop_count,tt)%small_header%staggering  , all_2d(loop_count,tt)%small_header%ordering    , &
         all_2d(loop_count,tt)%small_header%current_date, all_2d(loop_count,tt)%small_header%name        , &
         all_2d(loop_count,tt)%small_header%units       , all_2d(loop_count,tt)%small_header%description
         CALL unexpand2 ( snow_cover   , iew_alloc , jns_alloc , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
      ELSE IF (all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'PSEALVLD' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_2d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, 1 , 1 /)
         WRITE ( UNIT = unit )  &
         all_2d(loop_count,tt)%small_header%num_dims    , all_2d(loop_count,tt)%small_header%start_dims  , &
         all_2d(loop_count,tt)%small_header%end_dims    , all_2d(loop_count,tt)%small_header%xtime       , &
         all_2d(loop_count,tt)%small_header%staggering  , all_2d(loop_count,tt)%small_header%ordering    , &
         all_2d(loop_count,tt)%small_header%current_date, all_2d(loop_count,tt)%small_header%name        , &
         all_2d(loop_count,tt)%small_header%units       , all_2d(loop_count,tt)%small_header%description
         CALL unexpand2 ( slp_d        , iew_alloc , jns_alloc , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
      ELSE IF (all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'PSEALVLC' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_2d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, 1 , 1 /)
         WRITE ( UNIT = unit )  &
         all_2d(loop_count,tt)%small_header%num_dims    , all_2d(loop_count,tt)%small_header%start_dims  , &
         all_2d(loop_count,tt)%small_header%end_dims    , all_2d(loop_count,tt)%small_header%xtime       , &
         all_2d(loop_count,tt)%small_header%staggering  , all_2d(loop_count,tt)%small_header%ordering    , &
         all_2d(loop_count,tt)%small_header%current_date, all_2d(loop_count,tt)%small_header%name        , &
         all_2d(loop_count,tt)%small_header%units       , all_2d(loop_count,tt)%small_header%description
         CALL unexpand2 ( slp_x        , iew_alloc , jns_alloc , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
      ELSE IF (all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'TSEASFC ' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_2d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, 1 , 1 /)
         WRITE ( UNIT = unit )  &
         all_2d(loop_count,tt)%small_header%num_dims    , all_2d(loop_count,tt)%small_header%start_dims  , &
         all_2d(loop_count,tt)%small_header%end_dims    , all_2d(loop_count,tt)%small_header%xtime       , &
         all_2d(loop_count,tt)%small_header%staggering  , all_2d(loop_count,tt)%small_header%ordering    , &
         all_2d(loop_count,tt)%small_header%current_date, all_2d(loop_count,tt)%small_header%name        , &
         all_2d(loop_count,tt)%small_header%units       , all_2d(loop_count,tt)%small_header%description
         CALL unexpand2 ( sst          , iew_alloc , jns_alloc , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
      ELSE IF (all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'SKINTEMP' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_2d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, 1 , 1 /)
         WRITE ( UNIT = unit )  &
         all_2d(loop_count,tt)%small_header%num_dims    , all_2d(loop_count,tt)%small_header%start_dims  , &
         all_2d(loop_count,tt)%small_header%end_dims    , all_2d(loop_count,tt)%small_header%xtime       , &
         all_2d(loop_count,tt)%small_header%staggering  , all_2d(loop_count,tt)%small_header%ordering    , &
         all_2d(loop_count,tt)%small_header%current_date, all_2d(loop_count,tt)%small_header%name        , &
         all_2d(loop_count,tt)%small_header%units       , all_2d(loop_count,tt)%small_header%description
         CALL unexpand2 ( sst          , iew_alloc , jns_alloc , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
      ELSE IF (all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'TOBBOX  ' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_2d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, 1 , 1 /)
         WRITE ( UNIT = unit )  &
         all_2d(loop_count,tt)%small_header%num_dims    , all_2d(loop_count,tt)%small_header%start_dims  , &
         all_2d(loop_count,tt)%small_header%end_dims    , all_2d(loop_count,tt)%small_header%xtime       , &
         all_2d(loop_count,tt)%small_header%staggering  , all_2d(loop_count,tt)%small_header%ordering    , &
         all_2d(loop_count,tt)%small_header%current_date, all_2d(loop_count,tt)%small_header%name        , &
         all_2d(loop_count,tt)%small_header%units       , all_2d(loop_count,tt)%small_header%description
         CALL unexpand2 ( tobbox       , iew_alloc , jns_alloc , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
      ELSE IF (all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'RH SFC  ' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_2d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, 1 , 1 /)
         WRITE ( UNIT = unit )  &
         all_2d(loop_count,tt)%small_header%num_dims    , all_2d(loop_count,tt)%small_header%start_dims  , &
         all_2d(loop_count,tt)%small_header%end_dims    , all_2d(loop_count,tt)%small_header%xtime       , &
         all_2d(loop_count,tt)%small_header%staggering  , 'YX  '                                         , &
         all_2d(loop_count,tt)%small_header%current_date, all_2d(loop_count,tt)%small_header%name        , &
         all_2d(loop_count,tt)%small_header%units       , all_2d(loop_count,tt)%small_header%description
         CALL unexpand2 ( all_2d(loop_count,tt)%array , iew_alloc , jns_alloc , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
      ELSE IF ( ( all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'PSTARCRS' ) .OR. &
                ( all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'GROUND T' ) .OR. &
                ( all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'RES TEMP' ) ) THEN
!        PRINT '(A,A,A)','Skipping the output of the 2d field, ', all_2d(loop_count,tt)%small_header%name(1:8), &
!                        ' since this would confuse INTERPF.'
      ELSE
         WRITE ( UNIT = unit ) sh_flag
         all_2d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, 1 , 1 /)
         WRITE ( UNIT = unit )  &
         all_2d(loop_count,tt)%small_header%num_dims    , all_2d(loop_count,tt)%small_header%start_dims  , &
         all_2d(loop_count,tt)%small_header%end_dims    , all_2d(loop_count,tt)%small_header%xtime       , &
         all_2d(loop_count,tt)%small_header%staggering  , all_2d(loop_count,tt)%small_header%ordering    , &
         all_2d(loop_count,tt)%small_header%current_date, all_2d(loop_count,tt)%small_header%name        , &
         all_2d(loop_count,tt)%small_header%units       , all_2d(loop_count,tt)%small_header%description
         extra_loop = extra_loop + 1
         CALL unexpand2 ( all_2d(loop_count,tt)%array , iew_alloc , jns_alloc , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
      END IF

   END DO two_D

   !  Output the 1d array.  Should just be pressure.

   one_D : DO loop_count = 1 , loop1
      WRITE ( UNIT = unit ) sh_flag
         WRITE ( UNIT = unit )  &
         all_1d(loop_count)%small_header%num_dims    , all_1d(loop_count)%small_header%start_dims  , &
         all_1d(loop_count)%small_header%end_dims    , all_1d(loop_count)%small_header%xtime       , &
         all_1d(loop_count)%small_header%staggering  , all_1d(loop_count)%small_header%ordering    , &
         all_1d(loop_count)%small_header%current_date, all_1d(loop_count)%small_header%name        , &
         all_1d(loop_count)%small_header%units       , all_1d(loop_count)%small_header%description
#ifdef PRESSURE
all_1d(loop_count)%array = all_1d(loop_count)%array * 100
#endif
      WRITE ( UNIT = unit ) all_1d(loop_count)%array
   END DO one_D

   !  Stick the end of time flag on this data.

   WRITE ( UNIT = unit ) eot_flag

END SUBROUTINE write_analysis

!------------------------------------------------------------------------------

SUBROUTINE write_analysis_fdda ( unit , &
t , u , v , h , rh , &
terrain , slp_x , tobbox , pressure , &
iew_alloc , jns_alloc , kbu_alloc , iewd , jnsd , fdda_date_24 , ptop )

!  This routine assembles the correct data fields and outputs them with the
!  appropriate small header in the format expected for the surface FDDA file.

   USE input_data
   USE header_flags

   IMPLICIT NONE 

   INTEGER         :: unit

   INCLUDE 'big_header.inc'
 
   INTEGER                                                :: iewd , jnsd , iew_alloc , jns_alloc , kbu_alloc
   REAL , DIMENSION ( jns_alloc , iew_alloc , kbu_alloc ) :: t , u , v , h , rh , qv
   REAL , DIMENSION ( jns_alloc , iew_alloc )             :: terrain , slp_x , tobbox , psfc
   REAL , DIMENSION ( jnsd , iewd )                       :: dum2d
   REAL , DIMENSION ( kbu_alloc )                         :: pressure

   CHARACTER (LEN=24)                                     :: fdda_date_24
   REAL                                                   :: ptop

   INTEGER :: loop_count , &
              num3d , &
              num2d , &
              num1d , &
              num0d
#ifdef NCARG
real,dimension(iewd,jnsd)::ddd
integer :: i , j
#endif

   !  We need to keep track of where we are sticking the data for the FDDA option.

   IF ( initial_time ) THEN
      tt = first_time
   ELSE
      tt = second_time
   END IF
 
   !  Do all of the 3d data.
             
   three_D : DO loop_count = 1 , loop3
      IF      ( all_3d(loop_count,tt)%small_header%name(1:8) .EQ. 'T       ' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         WRITE ( UNIT = unit )  &
         2                                              , all_3d(loop_count,tt)%small_header%start_dims  , &
          (/ jnsd , iewd, 1         , 1 /)              , 0.                                             , &
         all_3d(loop_count,tt)%small_header%staggering  , 'YX  '                                         , &
         fdda_date_24                                   , all_3d(loop_count,tt)%small_header%name        , &
         all_3d(loop_count,tt)%small_header%units       , all_3d(loop_count,tt)%small_header%description
         CALL unexpand3 ( t  , iew_alloc , jns_alloc , 1 , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
#ifdef NCARG
do i=1,iewd
do j=1,jnsd
ddd(i,j)=dum2d(j,i)
enddo
enddo
call conrec(ddd,iewd,iewd-1,jnsd-1,0.,0.,0.,0,0,0)
call plchlq(0.5,0.95,'T ',0.02,0.,0.)
call frame
#endif
      ELSE IF ( all_3d(loop_count,tt)%small_header%name(1:8) .EQ. 'U       ' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         WRITE ( UNIT = unit )  &
         2                                              , all_3d(loop_count,tt)%small_header%start_dims  , &
          (/ jnsd , iewd, 1         , 1 /)              , 0.                                             , &
         all_3d(loop_count,tt)%small_header%staggering  , 'YX  '                                         , &
         fdda_date_24                                   , all_3d(loop_count,tt)%small_header%name        , &
         all_3d(loop_count,tt)%small_header%units       , all_3d(loop_count,tt)%small_header%description
         CALL unexpand3 ( u  , iew_alloc , jns_alloc , 1 , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
#ifdef NCARG
do i=1,iewd
do j=1,jnsd
ddd(i,j)=dum2d(j,i)
enddo
enddo
call conrec(ddd,iewd,iewd,jnsd,0.,0.,0.,0,0,0)
call plchlq(0.5,0.95,'U ',0.02,0.,0.)
call frame
#endif
      ELSE IF ( all_3d(loop_count,tt)%small_header%name(1:8) .EQ. 'V       ' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         WRITE ( UNIT = unit )  &
         2                                              , all_3d(loop_count,tt)%small_header%start_dims  , &
          (/ jnsd , iewd, 1         , 1 /)              , 0.                                             , &
         all_3d(loop_count,tt)%small_header%staggering  , 'YX  '                                         , &
         fdda_date_24                                   , all_3d(loop_count,tt)%small_header%name        , &
         all_3d(loop_count,tt)%small_header%units       , all_3d(loop_count,tt)%small_header%description
         CALL unexpand3 ( v  , iew_alloc , jns_alloc , 1 , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
#ifdef NCARG
do i=1,iewd
do j=1,jnsd
ddd(i,j)=dum2d(j,i)
enddo
enddo
call conrec(ddd,iewd,iewd,jnsd,0.,0.,0.,0,0,0)
call plchlq(0.5,0.95,'V ',0.02,0.,0.)
call frame
#endif
      ELSE IF ( all_3d(loop_count,tt)%small_header%name(1:8) .EQ. 'RH      ' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         WRITE ( UNIT = unit )  &
         2                                              , all_3d(loop_count,tt)%small_header%start_dims  , &
          (/ jnsd , iewd, 1         , 1 /)              , 0.                                             , &
         all_3d(loop_count,tt)%small_header%staggering  , 'YX  '                                         , &
         fdda_date_24                                   , all_3d(loop_count,tt)%small_header%name        , &
         all_3d(loop_count,tt)%small_header%units       , all_3d(loop_count,tt)%small_header%description
         CALL unexpand3 ( rh , iew_alloc , jns_alloc , 1 , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
#ifdef NCARG
do i=1,iewd
do j=1,jnsd
ddd(i,j)=dum2d(j,i)
enddo
enddo
call conrec(ddd,iewd,iewd-1,jnsd-1,0.,0.,0.,0,0,0)
call plchlq(0.5,0.95,'RH',0.02,0.,0.)
call frame
#endif

         WRITE ( UNIT = unit ) sh_flag
         WRITE ( UNIT = unit )  &
         2                                              , all_3d(loop_count,tt)%small_header%start_dims  , &
          (/ jnsd , iewd, 1         , 1 /)              , 0.                                             , &
         all_3d(loop_count,tt)%small_header%staggering  , 'YX  '                                         , &
         fdda_date_24                                   , 'Q        '                                    , &
         'kg kg{-1}                '                    , all_3d(loop_count,tt)%small_header%description
         CALL mixing_ratio ( rh , t , h , terrain , slp_x , pressure , iew_alloc , jns_alloc , kbu_alloc , qv , psfc )
         CALL unexpand3 ( qv , iew_alloc , jns_alloc , 1 , dum2d , iewd , jnsd )
         WRITE ( UNIT = unit ) dum2d
#ifdef NCARG
do i=1,iewd
do j=1,jnsd
ddd(i,j)=dum2d(j,i)
enddo
enddo
call conrec(ddd,iewd,iewd-1,jnsd-1,0.,0.,0.,0,0,0)
call plchlq(0.5,0.95,'Q ',0.02,0.,0.)
call frame
#endif

         WRITE ( UNIT = unit ) sh_flag
         WRITE ( UNIT = unit )  &
         2                                              , all_3d(loop_count,tt)%small_header%start_dims  , &
          (/ jnsd , iewd, 1         , 1 /)              , 0.                                             , &
         all_3d(loop_count,tt)%small_header%staggering  , 'YX  '                                         , &
         fdda_date_24                                   , 'PSTARCRS '                                    , &
         'Pa                       '                    , 'Hydrostatic total surface pressure - PTOP     '
         CALL unexpand2 ( psfc , iew_alloc , jns_alloc , dum2d , iewd , jnsd )
         dum2d(:jnsd-1,:iewd-1) = dum2d(:jnsd-1,:iewd-1) - ptop
         dum2d(:,iewd)=dum2d(:,iewd-1)
         dum2d(jnsd,:)=dum2d(jnsd-1,:)
         WRITE ( UNIT = unit ) dum2d
#ifdef NCARG
do i=1,iewd-1
do j=1,jnsd-1
ddd(i,j)=dum2d(j,i)/100.
enddo
enddo
call conrec(ddd,iewd,iewd-1,jnsd-1,0.,0.,0.,0,0,0)
call plchlq(0.5,0.95,'PSFC ',0.02,0.,0.)
call frame
#endif
      END IF
      
   END DO three_D

   !  Now, do the 2d data.

   two_D : DO loop_count = 1 , loop2
      IF      (all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'PSEALVLC' ) THEN
         WRITE ( UNIT = unit ) sh_flag
         all_2d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, 1 , 1 /)
         WRITE ( UNIT = unit )  &
         all_2d(loop_count,tt)%small_header%num_dims    , all_2d(loop_count,tt)%small_header%start_dims  , &
         all_2d(loop_count,tt)%small_header%end_dims    , 0.                                             , &
         all_2d(loop_count,tt)%small_header%staggering  , all_2d(loop_count,tt)%small_header%ordering    , &
         fdda_date_24                                   , all_2d(loop_count,tt)%small_header%name        , &
         all_2d(loop_count,tt)%small_header%units       , all_2d(loop_count,tt)%small_header%description
         CALL unexpand2 ( slp_x        , iew_alloc , jns_alloc , dum2d , iewd , jnsd ) 
         dum2d(:,iewd)=dum2d(:,iewd-1)
         dum2d(jnsd,:)=dum2d(jnsd-1,:)
         WRITE ( UNIT = unit ) dum2d
#ifdef NCARG
do i=1,iewd-1
do j=1,jnsd-1
ddd(i,j)=dum2d(j,i)/100.
enddo
enddo
call conrec(ddd,iewd,iewd-1,jnsd-1,0.,0.,4.,0,0,0)
call plchlq(0.5,0.95,'PSLV',0.02,0.,0.)
call frame
#endif

         WRITE ( UNIT = unit ) sh_flag
         all_2d(loop_count,tt)%small_header%end_dims = (/ jnsd , iewd, 1 , 1 /)
         WRITE ( UNIT = unit )  &
         all_2d(loop_count,tt)%small_header%num_dims    , all_2d(loop_count,tt)%small_header%start_dims  , &
         all_2d(loop_count,tt)%small_header%end_dims    , 0.                                             , &
         all_2d(loop_count,tt)%small_header%staggering  , all_2d(loop_count,tt)%small_header%ordering    , &
         fdda_date_24                                   , 'TOBBOX   '                                    , &
         'Obs within 250 km        '                    , 'Avg # of obs within 250 km (tot obs/ # vars)  '
         CALL unexpand2 ( tobbox       , iew_alloc , jns_alloc , dum2d , iewd , jnsd ) 
         WRITE ( UNIT = unit ) dum2d
#ifdef NCARG
do i=1,iewd
do j=1,jnsd
ddd(i,j)=dum2d(j,i)
enddo
enddo
call conrec(ddd,iewd,iewd-1,jnsd-1,0.,0.,1.,0,0,0)
call plchlq(0.5,0.95,'OBS',0.02,0.,0.)
call frame
#endif
      END IF

   END DO two_D

   !  Stick the end of time flag on this data.

   WRITE ( UNIT = unit ) eot_flag

END SUBROUTINE write_analysis_fdda

!-------------------------------------------------------------------------------

SUBROUTINE unexpand3 ( orig , iew_big , jns_big , kx , new , iew_small , jns_small ) 

   INTEGER                                         :: iew_big   , jns_big   , &
                                                      kx , &
                                                      iew_small , jns_small
   REAL , DIMENSION ( jns_big   , iew_big   , kx ) :: orig
   REAL , DIMENSION ( jns_small , iew_small , kx ) :: new

   INTEGER                                         :: i , j , k

   DO k = 1 , kx
      DO i = 1 , iew_small
         DO j = 1 , jns_small
            new(j,i,k) = orig(j+(jns_big-jns_small)/2,i+(iew_big-iew_small)/2,k)
         END DO
      END DO
   END DO

END SUBROUTINE unexpand3


!-------------------------------------------------------------------------------

SUBROUTINE unexpand2 ( orig , iew_big , jns_big , new , iew_small , jns_small ) 

   INTEGER                                         :: iew_big   , jns_big   , &
                                                      kx , &
                                                      iew_small , jns_small
   REAL , DIMENSION ( jns_big   , iew_big   )      :: orig
   REAL , DIMENSION ( jns_small , iew_small )      :: new

   INTEGER                                         :: i , j , k

   DO i = 1 , iew_small
      DO j = 1 , jns_small
         new(j,i) = orig(j+(jns_big-jns_small)/2,i+(iew_big-iew_small)/2)
      END DO
   END DO

END SUBROUTINE unexpand2

!-------------------------------------------------------------------------------

SUBROUTINE write_big_header ( unit , bhi , bhr , bhic , bhrc , start_date , kbu , interval , &
                              max_error_t , max_error_uv , max_error_p , buddy_weight )
  
   USE header_flags
   USE date_pack

   IMPLICIT NONE

   INTEGER :: unit , kbu , interval
   INCLUDE 'big_header.inc'
   CHARACTER (LEN=19) :: start_date
   REAL :: max_error_t , max_error_uv , max_error_p , buddy_weight

   INTEGER :: yy , mm , dd , hh, mi , se 

   WRITE ( UNIT = unit ) bh_flag

   bhi ( 1,1) = 3
   bhic( 1,1) = 'Objectively Analyzed Pressure Level Data from little_r                          '

   bhi ( 2,3) = 1
   bhic( 2,3) = 'little_r Version 3 MM5 System Format Edition Number                             '

   bhi ( 3,3) = 7
   bhic( 3,3) = 'little_r Program Version Number                                                 '

   bhi ( 4,3) = 0
   bhic( 4,3) = 'little_r Program Minor Revision Number                                          '

   CALL split_date_char ( start_date , yy , mm , dd , hh, mi , se )

   bhi ( 5,3) =  yy
   bhic( 5,3) = 'FOUR-DIGIT YEAR OF THE STAR TIME (1900 - 2099)                                  '

   bhi ( 6,3) =  mm
   bhic( 6,3) = 'TWO-DIGIT MONTH OF THE START TIME (01-12)                                       '

   bhi ( 7,3) =  dd
   bhic( 7,3) = 'TWO-DIGIT DAY OF THE START TIME (01-31)                                         '

   bhi ( 8,3) =  hh
   bhic( 8,3) = 'TWO-DIGIT HOUR OF THE START TIME (00-23)                                        '

   bhi ( 9,3) =  mi
   bhic( 9,3) = 'TWO-DIGIT MINUTE OF THE START TIME (00-59)                                      '

   bhi (10,3) =  se
   bhic(10,3) = 'TWO-DIGIT SECOND OF THE START TIME (00-59)                                      '

   bhi (11,3) =  0
   bhic(11,3) = 'FOUR-DIGIT TEN-THOUSANDTH OF A SECOND OF THE START TIME (0000-9999)             '

   bhi (12,3) =  kbu
   bhic(12,3) = 'NUMBER OF PRESSURE LEVELS IN OUTPUT, INCLUDING SURFACE LEVEL                    '

   bhr ( 1,3) =  interval
   bhrc( 1,3) = 'TIME DIFFERENCE (s) BETWEEN OUTPUT ANALYSIS TIMES                               '

   bhr ( 2,3) =  max_error_t
   bhrc( 2,3) = 'MAXIMUM TEMPERATURE DIFFERENCE ALLOWED IN ERROR MAX (K)                         '

   bhr ( 3,3) =  max_error_uv
   bhrc( 3,3) = 'MAXIMUM SPEED DIFFERENCE ALLOWED IN ERROR MAX (m/s)                             '

   bhr ( 4,3) =  max_error_p
   bhrc( 4,3) = 'MAXIMUM SEA-LEVEL PRESSURE DIFFERENCE ALLOWED IN ERROR MAX (Pa)                 '

   bhr ( 5,3) =  buddy_weight
   bhrc( 5,3) = 'TOLERANCE FOR BUDDY CHECK (0 = NO BUDDY CHECK)                                  '

   WRITE ( UNIT = unit ) bhi , bhr , bhic , bhrc

END SUBROUTINE write_big_header

!-------------------------------------------------------------------------------

SUBROUTINE write_big_header_fdda ( unit , bhi , bhr , bhic , bhrc , start_date , kbu , interval , &
                              max_error_t , max_error_uv , max_error_p , buddy_weight , interval_analysis )
  
   USE header_flags
   USE date_pack

   IMPLICIT NONE


   INTEGER :: unit , kbu , interval , interval_analysis
   INCLUDE 'big_header.inc'
   CHARACTER (LEN=19) :: start_date
   REAL :: max_error_t , max_error_uv , max_error_p , buddy_weight

   INTEGER :: yy , mm , dd , hh, mi , se 

   WRITE ( UNIT = unit ) bh_flag

   bhi ( 2,3) = 1
   bhic( 2,3) = 'little_r Version 3 MM5 System Format Edition Number                             '

   bhi ( 3,3) = 7
   bhic( 3,3) = 'little_r Program Version Number                                                 '

   bhi ( 4,3) = 0
   bhic( 4,3) = 'little_r Program Minor Revision Number                                          '

   CALL split_date_char ( start_date , yy , mm , dd , hh, mi , se )

   bhi ( 5,3) =  yy
   bhic( 5,3) = 'FOUR-DIGIT YEAR OF THE STAR TIME (1900 - 2099)                                  '

   bhi ( 6,3) =  mm
   bhic( 6,3) = 'TWO-DIGIT MONTH OF THE START TIME (01-12)                                       '

   bhi ( 7,3) =  dd
   bhic( 7,3) = 'TWO-DIGIT DAY OF THE START TIME (01-31)                                         '

   bhi ( 8,3) =  hh
   bhic( 8,3) = 'TWO-DIGIT HOUR OF THE START TIME (00-23)                                        '

   bhi ( 9,3) =  mi
   bhic( 9,3) = 'TWO-DIGIT MINUTE OF THE START TIME (00-59)                                      '

   bhi (10,3) =  se
   bhic(10,3) = 'TWO-DIGIT SECOND OF THE START TIME (00-59)                                      '

   bhi (11,3) =  0
   bhic(11,3) = 'FOUR-DIGIT TEN-THOUSANDTH OF A SECOND OF THE START TIME (0000-9999)             '

   bhi (12,3) =  kbu
   bhic(12,3) = 'NUMBER OF PRESSURE LEVELS IN OUTPUT, INCLUDING SURFACE LEVEL                    '

   bhr ( 1,3) =  interval_analysis
   bhrc( 1,3) = 'TIME DIFFERENCE (s) BETWEEN OUTPUT ANALYSIS TIMES                               '

   bhr ( 2,3) =  max_error_t
   bhrc( 2,3) = 'MAXIMUM TEMPERATURE DIFFERENCE ALLOWED IN ERROR MAX (K)                         '

   bhr ( 3,3) =  max_error_uv
   bhrc( 3,3) = 'MAXIMUM SPEED DIFFERENCE ALLOWED IN ERROR MAX (m/s)                             '

   bhr ( 4,3) =  max_error_p
   bhrc( 4,3) = 'MAXIMUM SEA-LEVEL PRESSURE DIFFERENCE ALLOWED IN ERROR MAX (Pa)                 '

   bhr ( 5,3) =  buddy_weight
   bhrc( 5,3) = 'TOLERANCE FOR BUDDY CHECK (0 = NO BUDDY CHECK)                                  '

   bhi ( 1,1) = 4
   bhic( 1,1) = 'Surface FDDA File from little_r                                                 '

   bhi ( 2,4) = 1
   bhic( 2,4) = 'little_r Version 3 MM5 System Format Edition Number                             '

   bhi ( 3,4) = 7
   bhic( 3,4) = 'little_r Program Version Number                                                 '

   bhi ( 4,4) = 0
   bhic( 4,4) = 'little_r Program Minor Revision Number                                          '

   CALL split_date_char ( start_date , yy , mm , dd , hh, mi , se )

   bhi ( 5,4) =  yy
   bhic( 5,4) = 'FOUR-DIGIT YEAR OF THE STAR TIME (1900 - 2099)                                  '

   bhi ( 6,4) =  mm
   bhic( 6,4) = 'TWO-DIGIT MONTH OF THE START TIME (01-12)                                       '

   bhi ( 7,4) =  dd
   bhic( 7,4) = 'TWO-DIGIT DAY OF THE START TIME (01-31)                                         '

   bhi ( 8,4) =  hh
   bhic( 8,4) = 'TWO-DIGIT HOUR OF THE START TIME (00-23)                                        '

   bhi ( 9,4) =  mi
   bhic( 9,4) = 'TWO-DIGIT MINUTE OF THE START TIME (00-59)                                      '

   bhi (10,4) =  se
   bhic(10,4) = 'TWO-DIGIT SECOND OF THE START TIME (00-59)                                      '

   bhi (11,4) =  0
   bhic(11,4) = 'FOUR-DIGIT TEN-THOUSANDTH OF A SECOND OF THE START TIME (0000-9999)             '

   bhi (12,4) =  1
   bhic(12,4) = 'NUMBER OF PRESSURE LEVELS IN OUTPUT, INCLUDING SURFACE LEVEL                    '

   bhr ( 1,4) =  interval
   bhrc( 1,4) = 'TIME DIFFERENCE (seconds) BETWEEN SURFACE ANALYSES                              '

   WRITE ( UNIT = unit ) bhi , bhr , bhic , bhrc

END SUBROUTINE write_big_header_fdda

!-------------------------------------------------------------------------------

SUBROUTINE mixing_ratio     ( rh , t , h , &
                              terrain , slp_x , &
                              pressure , &
                              iew_alloc , jns_alloc , kbu_alloc , &
                              qv , psfc )
                                      
   IMPLICIT NONE 

   INTEGER , INTENT(IN) :: iew_alloc , jns_alloc , kbu_alloc
  
   REAL , INTENT(IN) , DIMENSION(kbu_alloc) :: pressure
   REAL , INTENT(IN) , DIMENSION(jns_alloc,iew_alloc) :: terrain , slp_x
   REAL , INTENT(INOUT) , DIMENSION(jns_alloc,iew_alloc,kbu_alloc) :: rh
   REAL , INTENT(IN) , DIMENSION(jns_alloc,iew_alloc,kbu_alloc) :: t , h

   REAL , INTENT(OUT) , DIMENSION(jns_alloc,iew_alloc,kbu_alloc) :: qv
   REAL , INTENT(OUT) , DIMENSION(jns_alloc,iew_alloc) :: psfc

   INTEGER :: i , j , k 
   REAL                        :: ES
   REAL                        :: QS
   REAL,         PARAMETER     :: EPS         = 0.622
   REAL,         PARAMETER     :: SVP1        = 0.6112
   REAL,         PARAMETER     :: SVP2        = 17.67
   REAL,         PARAMETER     :: SVP3        = 29.65
   REAL,         PARAMETER     :: SVPT0       = 273.15

   !  Compute Qv at all of the mandatory levels - not the surface.

   rh(1:jns_alloc-1,1:iew_alloc-1,:) = MIN ( MAX ( rh(1:jns_alloc-1,1:iew_alloc-1,:) ,  1. ) , 100. )

   DO k = 2 , kbu_alloc
      DO j = 1, jns_alloc-1
         DO i = 1, iew_alloc-1
            es = svp1 * 10. * EXP(svp2 * (t(j,i,k) - svpt0) / (t(j,i,k) - svp3))
            qs = eps * es / (pressure(k) - es)
            qv(j,i,k) = MAX(0.01 * rh(j,i,k) * qs,0.0)
         END DO
      END DO
   END DO

   !  Now, with a 3d Qv, we can do the surface pressure.

   CALL sfcprs ( t , qv , h , slp_x , terrain , pressure , &
   iew_alloc , jns_alloc , kbu_alloc , psfc )

   !  Now, with a surface pressure, we can do the Qv at the surface, which was what
   !  we wanted any who.

   DO j = 1, jns_alloc-1
      DO i = 1, iew_alloc-1
         es = svp1 * 10. * EXP(svp2 * (t(j,i,1) - svpt0) / (t(j,i,1) - svp3))
         qs = eps * es / (psfc(j,i)/100. - es)
         qv(j,i,1) = MAX(0.01 * rh(j,i,1) * qs,0.0)
      END DO
   END DO

END SUBROUTINE mixing_ratio

!-------------------------------------------------------------------------------

SUBROUTINE sfcprs ( t , q , h , slp_x , terrain , pressure , &
iew_alloc , jns_alloc , kbu_alloc , psfc )

   IMPLICIT NONE

   INTEGER , INTENT(IN) ::  iew_alloc , jns_alloc , kbu_alloc
   REAL , INTENT(IN) , DIMENSION(jns_alloc,iew_alloc,kbu_alloc) ::  t , q , h
   REAL , INTENT(IN) , DIMENSION(jns_alloc,iew_alloc) ::  slp_x , terrain
   REAL , INTENT(IN) , DIMENSION(kbu_alloc) :: pressure

   REAL , INTENT(OUT) , DIMENSION(jns_alloc,iew_alloc) ::  psfc

   !  Local variables, counters, etc.

   INTEGER :: k850=0 , k700=0 , k500=0
   INTEGER :: i , j , k    
   REAL , PARAMETER :: R=287. , g = 9.806 , gamma = 6.5e-3
   REAL , PARAMETER :: gammarg = gamma*R/g 
   REAL :: p78 , p57 , gamma78 , gamma57
   REAL , DIMENSION(jns_alloc,iew_alloc) ::  ht , p1 , tslv , tsfc
   REAL :: t850 , t700 , t500 , t1

   !  Find the levels with which we wish to play.

   DO k = 2 , kbu_alloc
      IF      ( NINT(pressure(k) ) .EQ. 850 ) THEN
         k850 = k
      ELSE IF ( NINT(pressure(k) ) .EQ. 700 ) THEN
         k700 = k
      ELSE IF ( NINT(pressure(k) ) .EQ. 500 ) THEN
         k500 = k
      END IF
   END DO

   IF ( ( k850 .EQ. 0 ) .OR. & 
        ( k700 .EQ. 0 ) .OR. & 
        ( k500 .EQ. 0 ) ) THEN
      PRINT '(A)','Problems with find the 850, 700, or 500 mb level.'
      PRINT '(3(A,I3),A)','k850=',k850,', k700=',k700,', k500=',k500,'.'
      STOP 'No_find_levels'
   END IF

   !  First guess at surface pressure to choose the correct lapse rates.

   DO j = 1 , jns_alloc-1
      DO i = 1 , iew_alloc-1
         ht(j,i)= - terrain(j,i) / h(j,i,k850)
         psfc(j,i) = slp_x(j,i) * ( slp_x(j,i)/85000. ) ** ht(j,i)
      END DO
   END DO

   !  Pressure about 100 hPa above the surface.

   DO j = 1 , jns_alloc-1
      DO i = 1 , iew_alloc-1
         IF ( psfc(j,i) .GT. 95000 ) THEN
            p1(j,i) = 85000.
         ELSE IF ( psfc(j,i) .GT. 70000 ) THEN
            p1(j,i) = psfc(j,i) - 10000.
         ELSE 
            p1(j,i) = 50000.
         END IF
      END DO
   END DO

   !  Compute the necessary sea level temp and the surface temp.

   p78 = 1./LOG(850./700.) 
   p57 = 1./LOG(700./500.) 

   DO j = 1 , jns_alloc-1
      DO i = 1 , iew_alloc-1

         !  Virtual temperatures.

         t850 = t(j,i,k850) * ( 1. + 0.608 * q(j,i,k850) ) 
         t700 = t(j,i,k700) * ( 1. + 0.608 * q(j,i,k700) ) 
         t500 = t(j,i,k500) * ( 1. + 0.608 * q(j,i,k500) ) 
  
         !  Lapse rates.

         gamma78 = LOG(t850/t700) * p78
         gamma57 = LOG(t700/t500) * p57

         !  Approximate temperature, about 100 hPa above the surface.

         IF ( psfc(j,i) .GT. 95000 ) THEN
            t1 = t850
         ELSE IF ( psfc(j,i) .GT. 85000 ) THEN
            t1 = t700 * (p1(j,i)/70000.)**gamma78
         ELSE IF ( psfc(j,i) .GT. 70000 ) THEN
            t1 = t500 * (p1(j,i)/50000.)**gamma57
         ELSE 
            t1 = t500
         END IF
     
         !  Extrapolate to get the sea level temperature.

         tslv(j,i) = t1 * ( slp_x(j,i)/p1(j,i) ) ** gammarg

         !  Compute the surface temperature from the sea level temp.

         tsfc(j,i) = tslv(j,i) - gamma * terrain(j,i)
      END DO
   END DO
   
   !  Ta Da - we now do the sea level pressure computation.

   DO j = 1 , jns_alloc-1
      DO i = 1 , iew_alloc-1
         psfc(j,i) = slp_x(j,i) * EXP ( (-terrain(j,i)*g) / (R/2. * (tsfc(j,i) + tslv(j,i))) )
      END DO
   END DO

END SUBROUTINE sfcprs

!-------------------------------------------------------------------------------

END MODULE final_analysis
