MODULE first_guess

   USE small_header_data
   USE header_flags
   USE input_data
   USE date_pack

   REAL , DIMENSION(:,:,:) , ALLOCATABLE              :: extra_2d_data
   CHARACTER(LEN=80) , DIMENSION(:) , ALLOCATABLE     :: extra_2d_ch
   TYPE(sh) , DIMENSION(:) , ALLOCATABLE              :: extra_2d_sh
   INTEGER                                            :: extra_2d_count

CONTAINS

!------------------------------------------------------------------------------

SUBROUTINE alloc_lots_of_2d_data ( jns_alloc , iew_alloc , num2d_extra )

   IMPLICIT NONE

   INTEGER :: iew_alloc , jns_alloc , num2d_extra

   extra_2d_count = num2d_extra

   IF ( extra_2d_count .GT. 0 ) THEN
      ALLOCATE (extra_2d_data(jns_alloc,iew_alloc,num2d_extra))
      ALLOCATE (extra_2d_ch(num2d_extra))
      ALLOCATE (extra_2d_sh(num2d_extra))
   END IF

END SUBROUTINE alloc_lots_of_2d_data

!------------------------------------------------------------------------------

SUBROUTINE clean_fg_from_model ( slp_d , slp_x , land_use , &
iew_alloc , jns_alloc , kbu_alloc )

!  This routine cleans the first guess input data from the interpolated 
!  model output.

   INCLUDE 'first_guess_size.inc'

   REAL , DIMENSION ( jns_alloc , iew_alloc ) :: slp_d , &
                                                 slp_x , &
                                                 land_use

   !  Dot point sea level pressure does not exist, so create it, though it is 
   !  never used anywhere.

   slp_d = slp_x
   CALL crs2dot ( slp_d , jns_alloc , iew_alloc )

   !  The land_use array from the model output usually has the bottom row
   !  contaminated.  And if it is not contaminated by the HIRPBL scheme, 
   !  then no biggy, since there is no normal boundary gradient for the
   !  land use.

   land_use(1,:iew_alloc-1) = land_use(1,:iew_alloc-1) -  ( int(land_use(1,:iew_alloc-1)) / 100 ) * 100

END SUBROUTINE clean_fg_from_model

!------------------------------------------------------------------------------

SUBROUTINE crs2dot(field,dim1,dim2)
  
   IMPLICIT NONE

   INTEGER :: dim1 , dim2
   REAL , DIMENSION(dim1,dim2) :: field,dummy
   INTEGER :: i , j

   dummy(2:dim1-1,2:dim2-1)           = ( field(1:dim1-2,1:dim2-2) + &
                                          field(1:dim1-2,2:dim2-1) + &
                                          field(2:dim1-1,1:dim2-2) + &
                                          field(2:dim1-1,2:dim2-1) ) * 0.25
  
   dummy(2:dim1-1,1:dim2:dim2-1)      = ( field(1:dim1-2,1:dim2-1:dim2-2) + &
                                          field(2:dim1-1,1:dim2-1:dim2-2) ) * 0.5
  
   dummy(1:dim1:dim1-1,2:dim2-1)      = ( field(1:dim1-1:dim1-2,1:dim2-2) + &
                                          field(1:dim1-1:dim1-2,2:dim2-1) ) * 0.5
  
   dummy(1:dim1:dim1-1,1:dim2:dim2-1) =   field(1:dim1-1:dim1-2,1:dim2-1:dim2-2)
  
   field                              =   dummy
  
END SUBROUTINE crs2dot

!------------------------------------------------------------------------------

SUBROUTINE open_first_guess ( unit , filename )

!  This routine OPENs the first guess analysis.

   IMPLICIT NONE

   INTEGER                 :: unit
   CHARACTER ( LEN = 132 ) :: filename , &
                              test_name
   LOGICAL                 :: tf

   INCLUDE 'error.inc'

   INTERFACE
      INCLUDE 'error.int'
   END INTERFACE

   !  Check to see if this file is already OPENed.

   INQUIRE ( UNIT = unit , NAME = test_name , OPENED = tf )

   !  OPEN the file, yes or no?

   already_open : IF ( tf ) THEN
      error_number = 00321001
      error_message(1:31) = 'open_first_guess               '
      error_message(32:)  = ' The file ' // TRIM(filename) //&
      ' is already OPEN as ' // TRIM(test_name)
      fatal = .false.
      listing = .false.
      CALL error_handler ( error_number , error_message , &
      fatal , listing )

   ELSE already_open
      OPEN ( UNIT   = unit          , &
             FILE   = filename      , &
             ACCESS = 'SEQUENTIAL'  , &
             STATUS = 'OLD'         , &
             FORM   = 'UNFORMATTED' , &
             IOSTAT = error_number      )
      error_open : IF ( error_number .NE. 0 ) THEN
         error_number = error_number + 00321000
         error_message(1:31) = 'open_first_guess               '
         error_message(32:)  = ' Error OPENing first guess analysis file.'
         fatal = .true.
         listing = .false.
         CALL error_handler ( error_number , error_message , &
         fatal , listing )
      ENDIF error_open
   ENDIF already_open

END SUBROUTINE open_first_guess

!------------------------------------------------------------------------------

SUBROUTINE read_first_guess ( unit , &
bhi , bhr , bhic , bhrc , &
num3d , num2d , num1d , &
t , u , v , h , rh , &
terrain , land_use , map_factor_x , map_factor_d , coriolis , &
latitude_x , longitude_x , latitude_d , longitude_d , &
snow_cover , slp_d , slp_x , sst , &
pressure , &
iew_alloc , jns_alloc , kbu_alloc , &
current_date_8 , current_time_6 , icount , print_analysis ) 

!  This routine ingests this time period of the first guess analysis.  The
!  data is passed back through the argument list after a successful
!  sequence of READs.

   IMPLICIT NONE

   INTEGER                                      :: unit
   INTEGER                                      :: num3d , &
                                                   num2d , &
                                                   num1d 

   INCLUDE 'first_guess_size.inc'
   INCLUDE 'first_guess.inc'

   INTEGER                                      :: current_date_8 , & 
                                                   current_time_6

   INTEGER                                      :: loop_count     , & 
                                                   header_date_8  , &
                                                   header_time_6  , &
                                                   icount
   REAL , DIMENSION(jns_alloc,iew_alloc)        :: dummy
   INTEGER                                      :: extra_counter
   CHARACTER(LEN=8)                             :: dum_char
   INTEGER                                      :: yy, mm, dd, hh, mi, ss
   LOGICAL                                      :: print_analysis , &
                                                   first_loop
   REAL ,    DIMENSION(kbu_alloc)               :: pressure

   INCLUDE 'error.inc'
   INCLUDE 'big_header.inc'

   INTERFACE
      INCLUDE 'error.int'
   END INTERFACE

   !  We need to keep track of where we are sticking the data for the FDDA option.

   IF ( initial_time ) THEN
      tt = first_time
   ELSE
      tt = second_time
   END IF 

   !  This loop allows the routine to READ multiple time periods, looking
   !  for the correct time (it will match current_date_8 and 
   !  current_time_6).  The first time through this loop we allocate the
   !  extra space for the additional 2d fields that may be present.  A test
   !  is required so that if we are not starting at the intitial time period
   !  in the first-guess, we don't try to re-allocate space which has not
   !  been de-allocated.

   first_loop = .TRUE.
   time_loop : DO

      loop3 = 0
      loop2 = 0
      loop1 = 0
      variable_loop : DO 

         READ ( UNIT = unit , IOSTAT = error_number ) flag
      
         !  Was there an error on the read.  If so, then process the error.
      
         IF ( error_number .LT. 0 ) THEN
            error_number = error_number + 00322000
            error_message(1:31) = 'read_first_guess               '
            error_message(32:)  = ' Unexpected EOF read for first-guess data.  The requested time is not in the data.'
            fatal = .true.
            listing = .false.
            CALL error_handler ( error_number , error_message ,  &
            fatal , listing )
         ELSE IF ( error_number .NE. 0 ) THEN
            error_number = error_number + 00322001
            error_message(1:31) = 'read_first_guess               '
            error_message(32:)  = ' Error in small header read for the first guess.'
            fatal = .true.
            listing = .false.
            CALL error_handler ( error_number , error_message ,  &
            fatal , listing )
         ELSE IF ( flag .EQ. bh_flag ) THEN
            IF ( print_analysis ) THEN
               WRITE ( UNIT = * , FMT = * ) 'Found big header flag, so, we''ll just read it.'
            END IF
            READ ( UNIT = unit ) bhi , bhr , bhic , bhrc
            CYCLE time_loop
         ELSE IF ( flag .EQ. eot_flag ) THEN
            IF ( print_analysis ) THEN
               WRITE ( UNIT = * , FMT = * ) 'Found end of time flag.'
            END IF
            EXIT variable_loop
         ELSE IF ( flag .EQ. sh_flag ) THEN
            IF ( print_analysis ) THEN
               WRITE ( UNIT = * , FMT = * ) 'Found small header flag, proceeding.'
            END IF
         ELSE
            WRITE ( UNIT = * , FMT = * ) 'Did not find a regular flag value.'
            STOP 'weird_flag_value'
         END IF
   
         !  After the small header flag, we read the header.
   
         READ ( UNIT = unit ) small_header%num_dims     ,  &
                              small_header%start_dims   ,  &
                              small_header%end_dims     ,  &
                              small_header%xtime        ,  &
                              small_header%staggering   ,  &
                              small_header%ordering     ,  &
                              small_header%current_date ,  &
                              small_header%name         ,  &
                              small_header%units        ,  &
                              small_header%description

         !  After the flag, and the small header is an array of data.  But is it 3d, 2d or 1d?
   
         IF     ( small_header%num_dims .EQ. 3 ) THEN
            loop3=loop3+1
            ALLOCATE ( all_3d(loop3,tt)%array(jns_alloc,iew_alloc,kbu_alloc) ) 
            READ ( UNIT = unit ) all_3d(loop3,tt)%array
            all_3d(loop3,tt)%small_header = small_header
         ELSE IF( small_header%num_dims .EQ. 2 ) THEN
            loop2=loop2+1
            ALLOCATE ( all_2d(loop2,tt)%array(jns_alloc,iew_alloc) ) 
            READ ( UNIT = unit ) all_2d(loop2,tt)%array
            all_2d(loop2,tt)%small_header = small_header
         ELSE IF( small_header%num_dims .EQ. 1 ) THEN
            loop1=loop1+1
            ALLOCATE ( all_1d(loop1)%array(kbu_alloc) ) 
            READ ( UNIT = unit ) all_1d(loop1)%array
            all_1d(loop1)%small_header = small_header
         END IF

      END DO variable_loop

      !  We have read in all of the data for this time period.  Was it the right time?

      CALL split_date_char ( small_header%current_date , yy , mm , dd , hh , mi , ss )
      header_date_8 = yy * 10000 + mm * 100 + dd
      header_time_6 = hh * 10000 + mi * 100 + ss

      IF ( ( header_date_8 .NE. current_date_8 ) .OR. &
           ( header_time_6 .NE. current_time_6 ) ) THEN
         PRINT '(A,A,A)','Found wrong time, skipping past ',small_header%current_date,'.'
         CYCLE time_loop
     
      ELSE

         !  We have a valid time, so save the input data and exit out of here.

         num3d = loop3
         num2d = loop2
         num1d = loop1
   
         !  If this is the first time in this routine, we need to ALLOCATE some
         !  space for the extra 2d fields that will be coming this way.  There
         !  are usually only 13 2d arrays.  If there are more than that, we gotta
         !  have room for the data and associated header information.
   
         IF      ( ( first_loop) .AND. ( icount .EQ. 1 ) .AND. ( bhi(1,1) .EQ. 8 ) ) THEN
            CALL alloc_lots_of_2d_data ( jns_alloc , iew_alloc , num2d-13+1 )
         ELSE IF ( ( first_loop) .AND. ( icount .EQ. 1 ) ) THEN
            CALL alloc_lots_of_2d_data ( jns_alloc , iew_alloc , num2d-13   )
         END IF
         first_loop = .FALSE.
      
         !  Put the 3d data that we know about into storage arrays.
      
         three_D : DO loop_count = 1 , num3d
            IF      ( all_3d(loop_count,tt)%small_header%name(1:8) .EQ. 'T       ' ) THEN
               t = all_3d(loop_count,tt)%array
               t ( jns_alloc,:iew_alloc-1,kbu_alloc) = t ( jns_alloc-1,:iew_alloc-1,kbu_alloc)
               t (:jns_alloc, iew_alloc  ,kbu_alloc) = t (:jns_alloc  , iew_alloc-1,kbu_alloc)
            ELSE IF ( all_3d(loop_count,tt)%small_header%name(1:8) .EQ. 'U       ' ) THEN
               u = all_3d(loop_count,tt)%array
            ELSE IF ( all_3d(loop_count,tt)%small_header%name(1:8) .EQ. 'V       ' ) THEN
               v = all_3d(loop_count,tt)%array
            ELSE IF ( all_3d(loop_count,tt)%small_header%name(1:8) .EQ. 'H       ' ) THEN
               h = all_3d(loop_count,tt)%array
            ELSE IF ( all_3d(loop_count,tt)%small_header%name(1:8) .EQ. 'RH      ' ) THEN
               rh= all_3d(loop_count,tt)%array
               rh( jns_alloc,:iew_alloc-1,kbu_alloc) = rh( jns_alloc-1,:iew_alloc-1,kbu_alloc)
               rh(:jns_alloc,iew_alloc   ,kbu_alloc) = rh(:jns_alloc  , iew_alloc-1,kbu_alloc)
            ELSE
               error_number = 00322003
               error_message(1:31) = 'read_first_guess               '
               error_message(32:)  = ' Found a 3D field that is not standard: ' &
               // TRIM ( all_3d(loop_count,tt)%small_header%name ) 
               fatal = .false.
               listing = .false.
               IF ( print_analysis ) THEN
                  CALL error_handler ( error_number , error_message ,  &
                  fatal , listing )
               END IF
               error_number = 0
            END IF
            
         END DO three_D
      
         !  Process the 2d data.  Even the fields we don't recognize get saved to be
         !  passed through to the rest of the modeling system.
      
         extra_counter = 0
         two_D : DO loop_count = 1 , num2d
            IF      ( all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'TERRAIN ' ) THEN
               terrain = all_2d(loop_count,tt)%array
               terrain     ( jns_alloc,:iew_alloc-1) = terrain     ( jns_alloc-1,:iew_alloc-1)
               terrain     (:jns_alloc,iew_alloc   ) = terrain     (:jns_alloc  , iew_alloc-1)
            ELSE IF ( all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'LAND USE' ) THEN
               land_use = all_2d(loop_count,tt)%array
               land_use    ( jns_alloc,:iew_alloc-1) = land_use    ( jns_alloc-1,:iew_alloc-1)
               land_use    (:jns_alloc,iew_alloc   ) = land_use    (:jns_alloc  , iew_alloc-1)
            ELSE IF ( all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'MAPFACCR' ) THEN
               map_factor_x = all_2d(loop_count,tt)%array
               map_factor_x( jns_alloc,:iew_alloc-1) = map_factor_x( jns_alloc-1,:iew_alloc-1)
               map_factor_x(:jns_alloc,iew_alloc   ) = map_factor_x(:jns_alloc  , iew_alloc-1)
            ELSE IF ( all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'MAPFACDT' ) THEN
               map_factor_d = all_2d(loop_count,tt)%array
            ELSE IF ( all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'CORIOLIS' ) THEN
               coriolis = all_2d(loop_count,tt)%array
            ELSE IF ( all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'LATITCRS' ) THEN
               latitude_x = all_2d(loop_count,tt)%array
               latitude_x  ( jns_alloc,:iew_alloc-1) = latitude_x  ( jns_alloc-1,:iew_alloc-1)
               latitude_x  (:jns_alloc,iew_alloc   ) = latitude_x  (:jns_alloc  , iew_alloc-1)
            ELSE IF ( all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'LONGICRS' ) THEN
               longitude_x = all_2d(loop_count,tt)%array
               longitude_x ( jns_alloc,:iew_alloc-1) = longitude_x ( jns_alloc-1,:iew_alloc-1)
               longitude_x (:jns_alloc,iew_alloc   ) = longitude_x (:jns_alloc  , iew_alloc-1)
            ELSE IF ( all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'LATITDOT' ) THEN
               latitude_d = all_2d(loop_count,tt)%array
            ELSE IF ( all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'LONGIDOT' ) THEN
               longitude_d = all_2d(loop_count,tt)%array
            ELSE IF ( all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'SNOWCOVR' ) THEN
               snow_cover = all_2d(loop_count,tt)%array
               snow_cover  ( jns_alloc,:iew_alloc-1) = snow_cover  ( jns_alloc-1,:iew_alloc-1)
               snow_cover  (:jns_alloc,iew_alloc   ) = snow_cover  (:jns_alloc  , iew_alloc-1)
            ELSE IF ( all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'PSEALVLD' ) THEN
               slp_d = all_2d(loop_count,tt)%array
            ELSE IF ( all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'PSEALVLC' ) THEN
               slp_x = all_2d(loop_count,tt)%array
               slp_x       ( jns_alloc,:iew_alloc-1) = slp_x       ( jns_alloc-1,:iew_alloc-1)
               slp_x       (:jns_alloc,iew_alloc   ) = slp_x       (:jns_alloc  , iew_alloc-1)
            ELSE IF ( all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'GROUND T' ) THEN
               sst = all_2d(loop_count,tt)%array
               sst         ( jns_alloc,:iew_alloc-1) = sst         ( jns_alloc-1,:iew_alloc-1)
               sst         (:jns_alloc,iew_alloc   ) = sst         (:jns_alloc  , iew_alloc-1)
            ELSE IF ( all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'TSEASFC ' ) THEN
               sst = all_2d(loop_count,tt)%array
               sst         ( jns_alloc,:iew_alloc-1) = sst         ( jns_alloc-1,:iew_alloc-1)
               sst         (:jns_alloc,iew_alloc   ) = sst         (:jns_alloc  , iew_alloc-1)
            ELSE IF ( all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'SKINTEMP' ) THEN
               sst = all_2d(loop_count,tt)%array
               sst         ( jns_alloc,:iew_alloc-1) = sst         ( jns_alloc-1,:iew_alloc-1)
               sst         (:jns_alloc,iew_alloc   ) = sst         (:jns_alloc  , iew_alloc-1)
            ELSE IF ( all_2d(loop_count,tt)%small_header%name(1:8) .EQ. 'TOBBOX  ' ) THEN
               error_number = 00322003
               error_message(1:31) = 'read_first_guess               '
               error_message(32:)  = ' Found old observation density, TOBBOX, discarding it.'
               fatal = .false.
               listing = .false.
               IF ( print_analysis ) THEN
                  CALL error_handler ( error_number , error_message ,  &
                  fatal , listing )
               END IF
               error_number = 0
            ELSE
               extra_counter = extra_counter + 1
               dummy = all_2d(loop_count,tt)%array
               dum_char(1:8) = all_2d(loop_count,tt)%small_header%name(1:8)
               extra_2d_data(:,:,extra_counter) = dummy
               extra_2d_ch(extra_counter) = all_2d(loop_count,tt)%small_header%name // &
                                            all_2d(loop_count,tt)%small_header%units // &
                                            all_2d(loop_count,tt)%small_header%description
               error_number = 00322002
               error_message(1:31) = 'read_first_guess               '
               error_message(32:)  = ' Found a 2D field that is not standard: ' &
               // dum_char(1:8)
               fatal = .false.
               listing = .false.
               IF ( print_analysis ) THEN
                  CALL error_handler ( error_number , error_message ,  &
                  fatal , listing )
               END IF
               error_number = 0
            END IF
      
         END DO two_D
   
         !  We need to pick up the only important 1d array - pressure.
   
         one_D : DO loop_count = 1 , num1d
            IF      ( all_1d(loop_count)%small_header%name(1:8) .EQ. 'PRESSURE' ) THEN
               pressure = all_1d(loop_count)%array
            ELSE
               error_number = 00322002
               error_message(1:31) = 'read_first_guess               '
               error_message(32:)  = ' Found a 1D field that is not standard: ' &
               // all_1d(loop_count)%small_header%name(1:8)
               fatal = .false.
               listing = .false.
               IF ( print_analysis ) THEN
                  CALL error_handler ( error_number , error_message ,  &
                  fatal , listing )
               END IF
               error_number = 0
            END IF
         END DO one_D
      
         !  If this data came from the interpolated model output (bhi(1,1) = 8),
         !  then some data needs to be fixed.  The dot point sea level pressure,
         !  and the sea surface temperature do not exist.  Also the lower row of
         !  the land use array has miter steps included if the Blackadar PBL 
         !  was used.
      
         IF ( bhi(1,1) .EQ. 8 ) THEN
            CALL clean_fg_from_model ( slp_d , slp_x , land_use , &
            iew_alloc , jns_alloc , kbu_alloc )
         END IF

         !  If we are in this part of the IF test, we found the right time.  We can
         !  therefore exit the time loop.

         EXIT time_loop
  
      END IF
   
   END DO time_loop

END SUBROUTINE read_first_guess

!------------------------------------------------------------------------------

SUBROUTINE temporal_interp ( t , u , v , h , rh , slp_x , & 
iew_alloc , jns_alloc , kbu_alloc , num3d , num2d , &
icount_fdda , icount_1 , icount_2 ) 

!  This routine ingests this time period of the first guess analysis.  The
!  data is passed back through the argument list after a successful
!  sequence of READs.

   IMPLICIT NONE

   INTEGER , INTENT(IN) ::  iew_alloc , jns_alloc , kbu_alloc , num3d , num2d , icount_fdda , icount_1 , icount_2

   REAL , INTENT(OUT) , DIMENSION(jns_alloc,iew_alloc,kbu_alloc) :: t , u , v , h , rh
   REAL , INTENT(OUT) , DIMENSION(jns_alloc,iew_alloc)           :: slp_x

   INTEGER :: loop_count

   !  Linearly interpolate the 3d data.

   the_3d_search : DO loop_count = 1 , num3d
      IF      ( all_3d(loop_count,second_time)%small_header%name(1:8) .EQ. 'T       ' ) THEN
         t (1:jns_alloc-1,1:iew_alloc-1,kbu_alloc) = &
         ( REAL(icount_2-icount_fdda) * all_3d(loop_count,first_time )%array(1:jns_alloc-1,1:iew_alloc-1,kbu_alloc) + &
           REAL(icount_fdda-icount_1) * all_3d(loop_count,second_time)%array(1:jns_alloc-1,1:iew_alloc-1,kbu_alloc) ) / &
           REAL(icount_2-icount_1) 
         t ( jns_alloc,:iew_alloc-1,kbu_alloc) = t ( jns_alloc-1,:iew_alloc-1,kbu_alloc)
         t (:jns_alloc, iew_alloc  ,kbu_alloc) = t (:jns_alloc  , iew_alloc-1,kbu_alloc)
      ELSE IF ( all_3d(loop_count,second_time)%small_header%name(1:8) .EQ. 'U       ' ) THEN
         u (1:jns_alloc  ,1:iew_alloc  ,kbu_alloc) = &
         ( REAL(icount_2-icount_fdda) * all_3d(loop_count,first_time )%array(1:jns_alloc  ,1:iew_alloc  ,kbu_alloc) + &
           REAL(icount_fdda-icount_1) * all_3d(loop_count,second_time)%array(1:jns_alloc  ,1:iew_alloc  ,kbu_alloc) ) / &
           REAL(icount_2-icount_1) 
      ELSE IF ( all_3d(loop_count,second_time)%small_header%name(1:8) .EQ. 'V       ' ) THEN
         v (1:jns_alloc  ,1:iew_alloc  ,kbu_alloc) = &
         ( REAL(icount_2-icount_fdda) * all_3d(loop_count,first_time )%array(1:jns_alloc  ,1:iew_alloc  ,kbu_alloc) + &
           REAL(icount_fdda-icount_1) * all_3d(loop_count,second_time)%array(1:jns_alloc  ,1:iew_alloc  ,kbu_alloc) ) / &
           REAL(icount_2-icount_1) 
      ELSE IF ( all_3d(loop_count,second_time)%small_header%name(1:8) .EQ. 'H       ' ) THEN
         h (1:jns_alloc  ,1:iew_alloc  ,kbu_alloc) = &
         ( REAL(icount_2-icount_fdda) * all_3d(loop_count,first_time )%array(1:jns_alloc  ,1:iew_alloc  ,kbu_alloc) + &
           REAL(icount_fdda-icount_1) * all_3d(loop_count,second_time)%array(1:jns_alloc  ,1:iew_alloc  ,kbu_alloc) ) / &
           REAL(icount_2-icount_1) 
      ELSE IF ( all_3d(loop_count,second_time)%small_header%name(1:8) .EQ. 'RH      ' ) THEN
         rh(1:jns_alloc-1,1:iew_alloc-1,kbu_alloc) = &
         ( REAL(icount_2-icount_fdda) * all_3d(loop_count,first_time )%array(1:jns_alloc-1,1:iew_alloc-1,kbu_alloc) + &
           REAL(icount_fdda-icount_1) * all_3d(loop_count,second_time)%array(1:jns_alloc-1,1:iew_alloc-1,kbu_alloc) ) / &
           REAL(icount_2-icount_1) 
         rh( jns_alloc,:iew_alloc-1,kbu_alloc) = rh( jns_alloc-1,:iew_alloc-1,kbu_alloc)
         rh(:jns_alloc, iew_alloc  ,kbu_alloc) = rh(:jns_alloc  , iew_alloc-1,kbu_alloc)
      END IF
   END DO the_3d_search

   !  Linearly interpolate the 2d data.

   the_2d_search : DO loop_count = 1 , num2d
      IF      ( all_2d(loop_count,second_time)%small_header%name(1:8) .EQ. 'PSEALVLC' ) THEN
         slp_x(1:jns_alloc-1,1:iew_alloc-1) = &
         ( REAL(icount_2-icount_fdda) * all_2d(loop_count,first_time )%array(1:jns_alloc-1,1:iew_alloc-1) + &
           REAL(icount_fdda-icount_1) * all_2d(loop_count,second_time)%array(1:jns_alloc-1,1:iew_alloc-1) ) / &
           REAL(icount_2-icount_1) 
         slp_x( jns_alloc,:iew_alloc-1) = slp_x( jns_alloc-1,:iew_alloc-1)
         slp_x(:jns_alloc, iew_alloc  ) = slp_x(:jns_alloc  , iew_alloc-1)
      END IF
   END DO the_2d_search

END SUBROUTINE temporal_interp

!------------------------------------------------------------------------------

END MODULE first_guess
