MODULE all_io

   USE header_data
   USE util

   TYPE input_fields_3d
      REAL , POINTER , DIMENSION(:,:,:) :: array
      REAL , POINTER , DIMENSION(:,:,:) :: ebdy , wbdy , nbdy , sbdy
      TYPE(sh)                          :: small_header
   END TYPE input_fields_3d

   TYPE input_fields_2d
      REAL , POINTER , DIMENSION(:,:)   :: array
      TYPE(sh)                          :: small_header
   END TYPE input_fields_2d

   TYPE input_fields_1d
      REAL , POINTER , DIMENSION(:)     :: array
      TYPE(sh)                          :: small_header
   END TYPE input_fields_1d

   TYPE(input_fields_3d) , ALLOCATABLE , DIMENSION(:) :: pall_3d
   TYPE(input_fields_2d) , ALLOCATABLE , DIMENSION(:) :: pall_2d
   TYPE(input_fields_1d) , ALLOCATABLE , DIMENSION(:) :: pall_1d

   TYPE(input_fields_3d) , ALLOCATABLE , DIMENSION(:) :: nall_3d
   TYPE(input_fields_2d) , ALLOCATABLE , DIMENSION(:) :: nall_2d
   TYPE(input_fields_1d) , ALLOCATABLE , DIMENSION(:) :: nall_1d

   TYPE(input_fields_3d) , ALLOCATABLE , DIMENSION(:) :: vall_3d

   TYPE(input_fields_2d) , ALLOCATABLE , DIMENSION(:) :: all_terrain
   TYPE(input_fields_2d) , ALLOCATABLE , DIMENSION(:) :: pall_lowbdy
   TYPE(input_fields_2d) , ALLOCATABLE , DIMENSION(:) :: nall_lowbdy

   INTEGER :: num_3d , num_2d , num_1d
   INTEGER :: num_terrain_2d
   INTEGER :: num_lowbdy_2d

   CHARACTER(LEN=19) :: sh_date

   INTEGER , PRIVATE :: return_code

CONTAINS

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   SUBROUTINE read_terrain_data ( unit_fg )

      IMPLICIT NONE

      !  Input variables.

      INTEGER :: unit_fg

      !  Local variables.

      INTEGER :: loop2
      INTEGER :: flag

      !  Loop over all of the variables.

      loop2 = 0

      var_loop : DO

         !  Start off with a flag, as ususal.  We are here after a big header
         !  flag, so we can just look for the small header without trying to 
         !  do too many error tests for which flag is expected.
   
         READ ( unit_fg ) flag
   
         IF ( flag .EQ. bh_flag ) THEN
            PRINT '(A,I1,A)','Wrong flag for TERRAIN, wanted small header flag: ',flag,'.'
            STOP 'wrong_flag_small_header'
         ELSE IF ( flag .EQ. eot_flag ) THEN
            PRINT '(A)','Found the end of the TERRAIN data.'
            EXIT var_loop
         END IF
   
         !  We have the right flag, so get the header.
   
         READ ( unit_fg ) small_header%num_dims , small_header%start_dims , small_header%end_dims , &
                          small_header%xtime , small_header%staggering , small_header%ordering , &
                          small_header%current_date , small_header%name , small_header%units , &
                          small_header%description
   
         !  Increment the count of the fields.
         
         loop2 = loop2 + 1
    
         !  Allocate space for the data in the input array area.

         ALLOCATE ( all_terrain(loop2)%array(small_header%end_dims(1),small_header%end_dims(2)) , STAT = return_code )
         IF ( return_code .NE. 0 ) THEN
            PRINT '(A,A,A,I1,A)','ALLOCATE return code for TERRAIN field ',small_header%name,' is ',return_code,'.'
            STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
         ELSE
            PRINT *,small_header%end_dims(1),small_header%end_dims(2)
#endif
         END IF

         !  Use this space to input the data.

         READ ( unit_fg ) all_terrain(loop2)%array

         !  Assign the small_header data to this variable's small_header storage.

         all_terrain(loop2)%small_header = small_header
   
      END DO var_loop

      num_terrain_2d = loop2
      
   END SUBROUTINE read_terrain_data

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   SUBROUTINE read_lowbdy_data ( unit_fg )

      IMPLICIT NONE

      !  Input variables.

      INTEGER :: unit_fg

      !  Local variables.

      INTEGER :: loop2
      INTEGER :: flag 

      !  Loop over all of the variables.

      loop2 = 0

      var_loop : DO

         !  Start off with a flag, as ususal.  We are here after a big header
         !  flag, so we can just look for the small header without trying to 
         !  do too many error tests for which flag is expected.
   
         READ ( unit_fg ) flag
   
         IF ( flag .EQ. bh_flag ) THEN
            PRINT '(A,I1,A)','Wrong flag for LOWBDY, wanted small header flag: ',flag,'.'
            STOP 'wrong_flag_small_header'
         ELSE IF ( flag .EQ. eot_flag ) THEN
            PRINT '(A)','Found the end of the LOWBDY data.'
            EXIT var_loop
         END IF
   
         !  We have the right flag, so get the header.
   
         READ ( unit_fg ) small_header%num_dims , small_header%start_dims , small_header%end_dims , &
                          small_header%xtime , small_header%staggering , small_header%ordering , &
                          small_header%current_date , small_header%name , small_header%units , &
                          small_header%description
   
         !  Increment the count of the fields.
         
         loop2 = loop2 + 1
    
         !  Allocate space for the data in the input array area.

         ALLOCATE ( pall_lowbdy(loop2)%array(small_header%end_dims(1),small_header%end_dims(2)) , STAT = return_code )
         IF ( return_code .NE. 0 ) THEN
            PRINT '(A,A,A,I1,A)','ALLOCATE return code for P LOWBDY field ',small_header%name,' is ',return_code,'.'
            STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
         ELSE
            PRINT *,small_header%end_dims(1),small_header%end_dims(2)
#endif
         END IF

         !  Use this space to input the data.

         READ ( unit_fg ) pall_lowbdy(loop2)%array

         !  Assign the small_header data to this variable's small_header storage.

         pall_lowbdy(loop2)%small_header = small_header
   
      END DO var_loop

      num_lowbdy_2d = loop2
      
   END SUBROUTINE read_lowbdy_data

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   SUBROUTINE read_data ( unit_fg , file_fg , count_fg )

      IMPLICIT NONE

      !  Input variables.

      INTEGER :: unit_fg , count_fg
      CHARACTER(LEN=132) , DIMENSION(100) :: file_fg

      !  Local variables.

      REAL    :: dum2d
      INTEGER :: loop3 , loop2 , loop1 , loop
      INTEGER :: flag , ok , bhi_i1

      !  Loop over all of the variables.

      loop3 = 0
      loop2 = 0
      loop1 = 0

      var_loop : DO

         !  Start off with a flag, as ususal.  We are here after a big header
         !  flag, so we can just look for the small header without trying to 
         !  do too many error tests for which flag is expected.
   
         READ ( unit_fg , IOSTAT = ok ) flag
   
         IF      ( ( ok .LT. 0 ) .AND. ( count_fg .EQ. 100 ) ) THEN
            PRINT '(A,I8,A)','Unexpected EOF in MM5 data: ',ok,'.'
            PRINT '(A)','Exhausted list of input file names in the namelist.input file.'
            PRINT '(A)','You may have asked for a time period that is not available.'
            STOP 'EOF_MM5_read_all_files'
         ELSE IF ( ( ok .LT. 0 ) .AND. ( LEN_TRIM ( file_fg(count_fg+1) ) .EQ. 0 ) ) THEN
            PRINT '(A,I8,A)','Unexpected EOF in MM5 data: ',ok,'.'
            PRINT '(A)','No subsequent file specified for input in the namelist.input file.'
            PRINT '(A)','You may have asked for a time period that is not available.'
            STOP 'EOF_in_MM5_data'
         ELSE IF ( ( ok .LT. 0 ) .AND. ( LEN_TRIM ( file_fg(count_fg+1) ) .NE. 0 ) ) THEN
            PRINT '(A,I8,A)','Unexpected EOF in MM5 data: ',ok,'.'
            PRINT '(A,A,A)','Moving to next file in the namelist.input file: ',TRIM ( file_fg(count_fg+1) ) ,'.' 
            CLOSE ( unit_fg )
            count_fg = count_fg + 1
            CALL get_fg_file ( file_fg(count_fg) ,  unit_fg ) 
            DO loop = 1 , loop3
               DEALLOCATE ( pall_3d(loop)%array )
            END DO
            DO loop = 1 , loop2
               DEALLOCATE ( pall_2d(loop)%array )
            END DO
            DO loop = 1 , loop1
               DEALLOCATE ( pall_1d(loop)%array )
            END DO
            loop3 = 0
            loop2 = 0
            loop1 = 0
            CYCLE var_loop
         END IF

         IF ( flag .EQ. bh_flag ) THEN
            PRINT '(A,I1,A)','Wrong flag, wanted small header flag: ',flag,'.'
            PRINT '(A)','Assuming that this is an instance of multiple input files or cat''ed MM5 input files.'
            READ ( unit_fg ) bhi_i1           
            IF ( bhi_i1 .EQ. 11 ) THEN
               CYCLE var_loop
            ELSE
               PRINT '(A,I8,A)','This did not turn out to be MM5 data, program ID = ',bhi_i1,'.'
               STOP 'Messed_up_multiple_input_files'               
            END IF
         ELSE IF ( flag .EQ. eot_flag ) THEN
            PRINT '(A)','Found the end of the time period.'
            EXIT var_loop
         END IF
   
         !  We have the right flag, so get the header.
   
         READ ( unit_fg , IOSTAT = ok ) small_header%num_dims , small_header%start_dims , small_header%end_dims , &
                        small_header%xtime , small_header%staggering , small_header%ordering , &
                        small_header%current_date , small_header%name , small_header%units , &
                        small_header%description

         IF ( ( ok .LT. 0 ) .AND. ( LEN_TRIM ( file_fg(count_fg+1) ) .NE. 0 ) ) THEN
            PRINT '(A,I8,A)','Unexpected EOF in MM5 data: ',ok,'.'
            PRINT '(A,A,A)','Moving to next file in the namelist.input file: ',TRIM ( file_fg(count_fg+1) ) ,'.' 
            CLOSE ( unit_fg )
            count_fg = count_fg + 1
            CALL get_fg_file ( file_fg(count_fg) ,  unit_fg ) 
            DO loop = 1 , loop3
               DEALLOCATE ( pall_3d(loop)%array )
            END DO
            DO loop = 1 , loop2
               DEALLOCATE ( pall_2d(loop)%array )
            END DO
            DO loop = 1 , loop1
               DEALLOCATE ( pall_1d(loop)%array )
            END DO
            loop3 = 0
            loop2 = 0
            loop1 = 0
            CYCLE var_loop
         END IF
   
         !  Let's allocate space for this one array.  Then read it in.

         IF      ( small_header%num_dims .eq. 3 ) THEN
            
            !  Increment the count of the fields.
          
            loop3 = loop3 + 1
    
            !  Allocate space for the data in the input array area.

            ALLOCATE ( pall_3d(loop3)%array(small_header%end_dims(1),small_header%end_dims(2),small_header%end_dims(3)) , &
                       STAT = return_code )
            IF ( return_code .NE. 0 ) THEN
               PRINT '(A,A,A,I1,A)','ALLOCATE return code for P 3D field ',small_header%name,' is ',return_code,'.'
               STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
            ELSE
               PRINT *,small_header%end_dims(1),small_header%end_dims(2),small_header%end_dims(3)
#endif
            END IF

            NULLIFY ( pall_3d(loop3)%ebdy )
            NULLIFY ( pall_3d(loop3)%wbdy )
            NULLIFY ( pall_3d(loop3)%nbdy )
            NULLIFY ( pall_3d(loop3)%sbdy )

            !  Use this space to input the data.

            READ ( unit_fg , IOSTAT = ok ) pall_3d(loop3)%array

            IF ( ( ok .LT. 0 ) .AND. ( LEN_TRIM ( file_fg(count_fg+1) ) .NE. 0 ) ) THEN
               PRINT '(A,I8,A)','Unexpected EOF in MM5 data: ',ok,'.'
               PRINT '(A,A,A)','Moving to next file in the namelist.input file: ',TRIM ( file_fg(count_fg+1) ) ,'.' 
               CLOSE ( unit_fg )
               count_fg = count_fg + 1
               CALL get_fg_file ( file_fg(count_fg) ,  unit_fg ) 
               DO loop = 1 , loop3
                  DEALLOCATE ( pall_3d(loop)%array )
               END DO
               DO loop = 1 , loop2
                  DEALLOCATE ( pall_2d(loop)%array )
               END DO
               DO loop = 1 , loop1
                  DEALLOCATE ( pall_1d(loop)%array )
               END DO
               loop3 = 0
               loop2 = 0
               loop1 = 0
               CYCLE var_loop
            END IF
          
            !  Assign the small_header data to this variable's small_header storage.

            pall_3d(loop3)%small_header = small_header
   
         ELSE IF ( small_header%num_dims .eq. 2 ) THEN

            IF ( small_header%ordering .EQ. 'YX  ' ) THEN

               !  Increment the count of the fields.
            
               loop2 = loop2 + 1

               !  Allocate space for the data in the input array area.
  
               ALLOCATE ( pall_2d(loop2)%array(small_header%end_dims(1),small_header%end_dims(2)) , STAT = return_code )
               IF ( return_code .NE. 0 ) THEN
                  PRINT '(A,A,A,I1,A)','ALLOCATE return code for P 2D field ',small_header%name,' is ',return_code,'.'
                  STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
               ELSE
                  PRINT *,small_header%end_dims(1),small_header%end_dims(2)
#endif
               END IF
 
  
               !  Use this space to input the data.
  
               READ ( unit_fg , IOSTAT = ok ) pall_2d(loop2)%array

               IF ( ( ok .LT. 0 ) .AND. ( LEN_TRIM ( file_fg(count_fg+1) ) .NE. 0 ) ) THEN
                  PRINT '(A,I8,A)','Unexpected EOF in MM5 data: ',ok,'.'
                  PRINT '(A,A,A)','Moving to next file in the namelist.input file: ',TRIM ( file_fg(count_fg+1) ) ,'.' 
                  CLOSE ( unit_fg )
                  count_fg = count_fg + 1
                  CALL get_fg_file ( file_fg(count_fg) ,  unit_fg ) 
                  DO loop = 1 , loop3
                     DEALLOCATE ( pall_3d(loop)%array )
                  END DO
                  DO loop = 1 , loop2
                     DEALLOCATE ( pall_2d(loop)%array )
                  END DO
                  DO loop = 1 , loop1
                     DEALLOCATE ( pall_1d(loop)%array )
                  END DO
                  loop3 = 0
                  loop2 = 0
                  loop1 = 0
                  CYCLE var_loop
               END IF
     
               !  Assign the small_header data to this variable's small_header storage.
  
               pall_2d(loop2)%small_header = small_header

            ELSE
        
               !  There are some 2d fields that are dimensioned by the number of land use
               !  categories - not a regular (i,j) field.  We can skip past them, but let's
               !  mention that we are doing so.

               PRINT '(A,A,A)','Skipping 2d field ',TRIM ( small_header%name ) ,'.'
!              PRINT '(A)','   ---> This field will not be sent back to the model as input.'

               READ ( unit_fg , IOSTAT = ok ) dum2d

               IF ( ( ok .LT. 0 ) .AND. ( LEN_TRIM ( file_fg(count_fg+1) ) .NE. 0 ) ) THEN
                  PRINT '(A,I8,A)','Unexpected EOF in MM5 data: ',ok,'.'
                  PRINT '(A,A,A)','Moving to next file in the namelist.input file: ',TRIM ( file_fg(count_fg+1) ) ,'.' 
                  CLOSE ( unit_fg )
                  count_fg = count_fg + 1
                  CALL get_fg_file ( file_fg(count_fg) ,  unit_fg ) 
                  DO loop = 1 , loop3
                     DEALLOCATE ( pall_3d(loop)%array )
                  END DO
                  DO loop = 1 , loop2
                     DEALLOCATE ( pall_2d(loop)%array )
                  END DO
                  DO loop = 1 , loop1
                     DEALLOCATE ( pall_1d(loop)%array )
                  END DO
                  loop3 = 0
                  loop2 = 0
                  loop1 = 0
                  CYCLE var_loop
               END IF

            END IF
   
         ELSE IF ( small_header%num_dims .eq. 1 ) THEN
            
            !  Increment the count of the fields.
          
            loop1 = loop1 + 1
    
            !  Allocate space for the data in the input array area.

            ALLOCATE ( pall_1d(loop1)%array(small_header%end_dims(1)) , STAT = return_code )
            IF ( return_code .NE. 0 ) THEN
               PRINT '(A,A,A,I1,A)','ALLOCATE return code for P 1D field ',small_header%name,' is ',return_code,'.'
               STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
            ELSE
               PRINT *,small_header%end_dims(1)
#endif
            END IF
 

            !  Use this space to input the data.

            READ ( unit_fg , IOSTAT = ok ) pall_1d(loop1)%array

            IF ( ( ok .LT. 0 ) .AND. ( LEN_TRIM ( file_fg(count_fg+1) ) .NE. 0 ) ) THEN
               PRINT '(A,I8,A)','Unexpected EOF in MM5 data: ',ok,'.'
               PRINT '(A,A,A)','Moving to next file in the namelist.input file: ',TRIM ( file_fg(count_fg+1) ) ,'.' 
               CLOSE ( unit_fg )
               count_fg = count_fg + 1
               CALL get_fg_file ( file_fg(count_fg) ,  unit_fg ) 
               DO loop = 1 , loop3
                  DEALLOCATE ( pall_3d(loop)%array )
               END DO
               DO loop = 1 , loop2
                  DEALLOCATE ( pall_2d(loop)%array )
               END DO
               DO loop = 1 , loop1
                  DEALLOCATE ( pall_1d(loop)%array )
               END DO
               loop3 = 0
               loop2 = 0
               loop1 = 0
               CYCLE var_loop
            END IF

            !  Assign the small_header data to this variable's small_header storage.

            pall_1d(loop1)%small_header = small_header
   
         END IF
   
      END DO var_loop

      !  A few values that we want out of here: the date, and the number of
      !  fields.

      sh_date = small_header%current_date(1:19)

      num_3d = loop3
      num_2d = loop2
      num_1d = loop1
      
   END SUBROUTINE read_data

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   SUBROUTINE add_lbc ( ground_t , ground_t_sh , &
                        tseasfc  ,  tseasfc_sh , unit_lowerbc )

      IMPLICIT NONE

      REAL , DIMENSION(:,:) :: ground_t , tseasfc
      TYPE(sh)              :: ground_t_sh , tseasfc_sh
      INTEGER               :: unit_lowerbc

      WRITE ( unit_lowerbc )  tseasfc_sh%num_dims ,  tseasfc_sh%start_dims ,  tseasfc_sh%end_dims , &
                              tseasfc_sh%xtime , &
                              tseasfc_sh%staggering ,  tseasfc_sh%ordering ,  tseasfc_sh%current_date , &
                              tseasfc_sh%name ,  tseasfc_sh%units ,  tseasfc_sh%description
      WRITE ( unit_lowerbc )  tseasfc

      WRITE ( unit_lowerbc ) ground_t_sh%num_dims , ground_t_sh%start_dims , ground_t_sh%end_dims , &
                             ground_t_sh%xtime , &
                             ground_t_sh%staggering , ground_t_sh%ordering , ground_t_sh%current_date , &
                             ground_t_sh%name , ground_t_sh%units , ground_t_sh%description
      WRITE ( unit_lowerbc ) ground_t

   END SUBROUTINE add_lbc
   
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   SUBROUTINE sum_lbc ( unit_lowerbc , itimes , interval , less_than_24h , &
                        start_year , start_month , start_day , start_hour , &
                        start_minute , start_second , start_frac )

      IMPLICIT NONE

      !  Input data.

      INTEGER               :: unit_lowerbc , itimes , interval , &
                               start_year , start_month , start_day , start_hour , &
                               start_minute , start_second , start_frac
      LOGICAL               :: less_than_24h

      !  Local data.

      REAL , ALLOCATABLE , DIMENSION(:,:) :: tseasfc , sfc_t , sum_tseasfc , sum_sfc_t
      TYPE(sh)              :: tseasfc_sh , sfc_t_sh
      INTEGER :: imx , jmx , loop , loopmax
      CHARACTER (LEN=24) :: tseasfc_current_date , sfc_t_current_date
      
      !  Read back the data from the lower boundary file, add up the
      !  the arrays, get a mean, and write the mean field back out as a standard
      !  v3 formatted file.  Since we have been writing to this unit, there is the
      !  small matter of a rewind.

      REWIND ( unit_lowerbc )
 
      !  Loop of the number of full days.  For example, if interval = 21600 s, and itimes = 9 
      !  (so, 0, 6, 12, 18, 24, 30, 36, 42, and 48h), then loop from 1 to 8.  If the user
      !  has SPECIFICALLY said "less than 24 h, please", then we acquiesce.

      IF ( ( less_than_24h ) .AND. &
           ( interval * itimes / 86400 .LT. 1 ) ) THEN
         loopmax = itimes
      ELSE IF ( interval * itimes / 86400 .GE. 1 ) THEN
         loopmax = ( interval * itimes / 86400 ) * ( 86400 / interval )
      ELSE
         PRINT '(A)','You should not be allowed to be here.'
         STOP 'wrong_number_of_times'
      END IF

      DO loop = 1 , loopmax

         !  The first field in the LOWBDY file is the sea surface temperature.

         READ ( unit_lowerbc ) tseasfc_sh%num_dims , tseasfc_sh%start_dims , tseasfc_sh%end_dims , &
                               tseasfc_sh%xtime , &
                               tseasfc_sh%staggering , tseasfc_sh%ordering , tseasfc_sh%current_date , &
                               tseasfc_sh%name , tseasfc_sh%units , tseasfc_sh%description

         !  To read in the data, we need an input array.  Just ALLOCATE this the first time,
         !  the array is the same size.  There are a few small header values that we glom onto
         !  so that we have them when we do the single small header write below.

         IF ( loop .EQ. 1 ) THEN
            tseasfc_current_date = tseasfc_sh%current_date
            imx                  = tseasfc_sh%end_dims(1)
            jmx                  = tseasfc_sh%end_dims(2)
            ALLOCATE ( tseasfc(imx,jmx) , STAT = return_code )
            IF ( return_code .NE. 0 ) THEN
               PRINT '(A,A,A,I1,A)','ALLOCATE return code for SST field ',tseasfc_sh%name,' is ',return_code,'.'
               STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
            ELSE
               PRINT *,imx,jmx
#endif
            END IF

            ALLOCATE ( sum_tseasfc(imx,jmx) , STAT = return_code )
            IF ( return_code .NE. 0 ) THEN
               PRINT '(A,A,A,I1,A)','ALLOCATE return code for SUM SST field ',tseasfc_sh%name,' is ',return_code,'.'
               STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
            ELSE
               PRINT *,imx,jmx
#endif
            END IF
 
            sum_tseasfc          = 0
         END IF

         READ ( unit_lowerbc ) tseasfc

         !  Fill in the outer row and column for sloppy array notation.

         tseasfc(imx,:)=tseasfc(imx-1,:)
         tseasfc(:,jmx)=tseasfc(:,jmx-1)
  
         sum_tseasfc = sum_tseasfc + tseasfc

         !  The second field in the LOWBDY file is the surface air temperature.

         READ ( unit_lowerbc ) sfc_t_sh%num_dims , sfc_t_sh%start_dims , sfc_t_sh%end_dims , &
                               sfc_t_sh%xtime , &
                               sfc_t_sh%staggering , sfc_t_sh%ordering , sfc_t_sh%current_date , &
                               sfc_t_sh%name , sfc_t_sh%units , sfc_t_sh%description

         !  To read in the data, we need an input array.  Just ALLOCATE this the first time,
         !  the array is the same size.  There are a few small header values that we glom onto
         !  so that we have them when we do the single small header write below.

         IF ( loop .EQ. 1 ) THEN
            sfc_t_current_date = sfc_t_sh%current_date
            imx                = sfc_t_sh%end_dims(1)
            jmx                = sfc_t_sh%end_dims(2)
            ALLOCATE ( sfc_t(imx,jmx) , STAT = return_code )
            IF ( return_code .NE. 0 ) THEN
               PRINT '(A,A,A,I1,A)','ALLOCATE return code for SFC T field ',sfc_t_sh%name,' is ',return_code,'.'
               STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
            ELSE
               PRINT *,imx,jmx
#endif
            END IF

            ALLOCATE ( sum_sfc_t(imx,jmx) , STAT = return_code )
            IF ( return_code .NE. 0 ) THEN
               PRINT '(A,A,A,I1,A)','ALLOCATE return code for SUM SFC T field ',sfc_t_sh%name,' is ',return_code,'.'
               STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
            ELSE
               PRINT *,imx,jmx
#endif
            END IF

            sum_sfc_t          = 0
         END IF

         READ ( unit_lowerbc ) sfc_t

         !  Fill in the outer row and column for sloppy array notation.

         sfc_t(imx,:)=sfc_t(imx-1,:)
         sfc_t(:,jmx)=sfc_t(:,jmx-1)
  
         sum_sfc_t = sum_sfc_t + sfc_t

      END DO

      !  Big, cool average computation about to happen, we use the same number of time
      !  periods as above.  This gives the model an average of the surface air
      !  temperature and an average of the sea surface temperature.  Though the sea surface
      !  temperature is slowly varying compared to the surface air temperature, we 
      !  sometimes have to use a skin temperature.  This is an attempt to fix the diurnal
      !  variation of the skin temperature when used for the sst.

      tseasfc = sum_tseasfc / REAL (  loopmax )
      sfc_t   = sum_sfc_t   / REAL (  loopmax )

      !  We are ready to write out the file.  Make the big header for this file.  Again,
      !  switching back to WRITE mode, so do a REWIND.

      REWIND ( unit_lowerbc )
      WRITE ( unit_lowerbc ) bh_flag

      !  Just in case no one else cleared the big header, we should do it.  All
      !  of the header values for the lower boundary condition index can be set to -999.

      bhi (:,6) = -999
      bhr (:,6) = -999
      bhic(:,6) = '                                                                                '
      bhrc(:,6) = '                                                                                '

      !  Identify the data as the lower boundary condition for MM5.

      bhi( 1,1) = 6 ; bhic( 1,1) = 'Model lower boundary condition for MM5                                          '
 
      !  A few lines about data format and version numbers.  If we make a change in the
      !  IC INTERP output version numbers, those will now be reflected here.

      bhi( 2,6) = bhi(2,5) ; bhic( 2,6) = bhic(2,5)
      bhi( 3,6) = bhi(3,5) ; bhic( 3,6) = bhic(3,5)
      bhi( 4,6) = bhi(4,5) ; bhic( 4,6) = bhic(4,5)
      
      !  Starting date of the INTERP data, this was specified in the namelist.

      bhi ( 5,6) = start_year   
      bhic( 5,6) = 'Four-digit year of start time                                                   '
      bhi ( 6,6) = start_month
      bhic( 6,6) = 'Month of the year of the start time (1-12)                                      '
      bhi ( 7,6) = start_day
      bhic( 7,6) = 'Day of the month of the start time (1-31)                                       '
      bhi ( 8,6) = start_hour
      bhic( 8,6) = 'Hour of the day of the start time (0-23)                                        '
      bhi ( 9,6) = start_minute
      bhic( 9,6) = 'Minute of the start time (0-59)                                                 '
      bhi (10,6) = start_second
      bhic(10,6) = 'Second of the start time (0-59)                                                 '
      bhi (11,6) = start_frac
      bhic(11,6) = 'Ten thousandths of a second of the start time (0-9999)                          '

      !  How many levels to expect.

      bhi (12,6) = 1  
      bhic(12,6) = 'Number of levels in the lower boundary condition file                           '

      !  Time interval between output files.  Notice that this is the length of the
      !  entire INTERP program run, not just a single interval.

      bhr ( 1,6) = interval * (itimes-1)
      bhrc( 1,6) = 'Time difference (seconds) through which the lower boundary condition is valid   '

      !  Output the big header.

      WRITE ( unit_lowerbc ) bhi , bhr , bhic , bhrc

      !  Now, the small header things: 1) flag, 2) header, and 3) data.

      WRITE ( unit_lowerbc ) sh_flag

      WRITE ( unit_lowerbc ) tseasfc_sh%num_dims , tseasfc_sh%start_dims , tseasfc_sh%end_dims , &
                             tseasfc_sh%xtime , &
                             tseasfc_sh%staggering , tseasfc_sh%ordering , tseasfc_current_date , &
                             tseasfc_sh%name , tseasfc_sh%units , tseasfc_sh%description

      WRITE ( unit_lowerbc ) tseasfc

      WRITE ( unit_lowerbc ) sh_flag

      WRITE ( unit_lowerbc ) sfc_t_sh%num_dims , sfc_t_sh%start_dims , sfc_t_sh%end_dims , &
                             sfc_t_sh%xtime , &
                             sfc_t_sh%staggering , sfc_t_sh%ordering , sfc_t_current_date , &
                             sfc_t_sh%name , sfc_t_sh%units , sfc_t_sh%description

      WRITE ( unit_lowerbc ) sfc_t

      !  Finish the file with an end of time flag.

      WRITE ( unit_lowerbc ) eot_flag

      !  DEALLOCATE the heap arrays.

      DEALLOCATE ( tseasfc )
      DEALLOCATE ( sum_tseasfc )
      DEALLOCATE ( sfc_t )
      DEALLOCATE ( sum_sfc_t )


   END SUBROUTINE sum_lbc
   
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   SUBROUTINE sum_lbc2 ( local_nlbhi , local_nlbhr , local_nlbhic , local_nlbhrc , &
                         unit_lowerbc , iprocess , interval , &
                         start_year , start_month , start_day , start_hour , &
                         start_minute , start_second , start_frac )

      IMPLICIT NONE

      !  Input data.

      INTEGER               :: local_nlbhi     (   50 , 20 )
      REAL                  :: local_nlbhr     (   20 , 20 )
      CHARACTER*80          :: local_nlbhic    (   50 , 20 )
      CHARACTER*80          :: local_nlbhrc    (   20 , 20 )
      
      INTEGER               :: unit_lowerbc , iprocess , interval , &
                               start_year , start_month , start_day , start_hour , &
                               start_minute , start_second , start_frac

      !  Local data.

      TYPE(sh)              :: ground_t_sh
      INTEGER               :: loop
      
      !  Just in case no one else cleared the big header, we should do it.  All
      !  of the header values for the lower boundary condition index can be set to -999.

      local_nlbhi (:,6) = -999
      local_nlbhr (:,6) = -999
      local_nlbhic(:,6) = '                                                                                '
      local_nlbhrc(:,6) = '                                                                                '

      !  Identify the data as the lower boundary condition for MM5.

      local_nlbhi ( 1,1) = 6
      local_nlbhic( 1,1) = '1-way model lower boundary condition for MM5                                    '
 
      !  A few lines about data format and version numbers.  If we make a change in the
      !  IC INTERP output version numbers, those will now be reflected here.

      local_nlbhi( 2,6) = local_nlbhi(2,5) ; local_nlbhic( 2,6) = local_nlbhic(2,5)
      local_nlbhi( 3,6) = local_nlbhi(3,5) ; local_nlbhic( 3,6) = local_nlbhic(3,5)
      local_nlbhi( 4,6) = local_nlbhi(4,5) ; local_nlbhic( 4,6) = local_nlbhic(4,5)
      
      !  Starting date of the NESTDOWN data, this was specified in the namelist.

      local_nlbhi ( 5,6) = start_year   
      local_nlbhic( 5,6) = 'Four-digit year of start time                                                   '
      local_nlbhi ( 6,6) = start_month
      local_nlbhic( 6,6) = 'Month of the year of the start time (1-12)                                      '
      local_nlbhi ( 7,6) = start_day
      local_nlbhic( 7,6) = 'Day of the month of the start time (1-31)                                       '
      local_nlbhi ( 8,6) = start_hour
      local_nlbhic( 8,6) = 'Hour of the day of the start time (0-23)                                        '
      local_nlbhi ( 9,6) = start_minute
      local_nlbhic( 9,6) = 'Minute of the start time (0-59)                                                 '
      local_nlbhi (10,6) = start_second
      local_nlbhic(10,6) = 'Second of the start time (0-59)                                                 '
      local_nlbhi (11,6) = start_frac
      local_nlbhic(11,6) = 'Ten thousandths of a second of the start time (0-9999)                          '

      !  How many levels to expect.

      local_nlbhi (12,6) = 1  
      local_nlbhic(12,6) = 'Number of levels in the lower boundary condition file                           '

      !  Time interval between output files.  Notice that this is the length of the
      !  entire NESTDOWN program run, not just a single interval.

      local_nlbhr ( 1,6) = interval * (iprocess-1)
      local_nlbhrc( 1,6) = 'Time difference (seconds) through which the lower boundary condition is valid   '

      !  Output the big header.

      WRITE ( unit_lowerbc ) bh_flag
      WRITE ( unit_lowerbc ) local_nlbhi , local_nlbhr , local_nlbhic , local_nlbhrc

      !  Now, the small header things: 1) flag, 2) header, and 3) data.

      DO loop = 1 , num_lowbdy_2d

         WRITE ( unit_lowerbc ) sh_flag
 
         ground_t_sh = nall_lowbdy(loop)%small_header

         WRITE ( unit_lowerbc ) ground_t_sh%num_dims , ground_t_sh%start_dims , ground_t_sh%end_dims , &
                                ground_t_sh%xtime , &
                                ground_t_sh%staggering , ground_t_sh%ordering , ground_t_sh%current_date , &
                                ground_t_sh%name , ground_t_sh%units , ground_t_sh%description

         WRITE ( unit_lowerbc ) nall_lowbdy(loop)%array

      END DO

      !  Finish the file with an end of time flag.

      WRITE ( unit_lowerbc ) eot_flag

      !  This was the only WRITE, so we can CLOSE the file.

      CLOSE ( unit_lowerbc )

   END SUBROUTINE sum_lbc2
   
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   SUBROUTINE do_namelist_2 ( input_file ,  input_lowbdy_file , input_terrain_file , &
                              start_year , start_month , start_day , start_hour , &
                              start_minute , start_second , start_frac , &
                              end_year ,   end_month ,   end_day ,   end_hour , &
                              end_minute ,   end_second ,   end_frac , &
                              interval , less_than_24h , &
                              sigma_f_bu , sst_to_ice_threshold , &
                              wrth2o ,  &
                              ifdatim , &
                              interp_method , print_info , use_mm5_lowbdy )

      IMPLICIT NONE

      !  Input variables.
   
      !  RECORD0
      
      CHARACTER(LEN=132) , DIMENSION(100) :: input_file
      CHARACTER(LEN=132)                  :: input_lowbdy_file , input_terrain_file
      NAMELIST /RECORD0/ input_file , input_lowbdy_file , input_terrain_file
      
      !  RECORD1
      
      INTEGER :: start_year , start_month , start_day , start_hour , &
                 start_minute , start_second , start_frac
      INTEGER ::   end_year ,   end_month ,   end_day ,   end_hour , &
                 end_minute ,   end_second ,   end_frac
      INTEGER :: interval
      LOGICAL :: less_than_24h

      NAMELIST /RECORD1/ start_year , start_month , start_day ,   &
                         start_hour ,  start_minute , start_second , start_frac ,  &
                         end_year ,   end_month ,   end_day ,     &
                         end_hour ,  end_minute ,   end_second ,   end_frac ,  &
                         interval , less_than_24h
      
      !  RECORD2

      REAL , DIMENSION(1000) :: sigma_f_bu
      REAL    :: sst_to_ice_threshold

      NAMELIST /RECORD2/ sigma_f_bu , sst_to_ice_threshold

      !  RECORD4
      
      LOGICAL :: wrth2o   

      NAMELIST /RECORD4/ wrth2o
      
      !  RECORD5
      
      INTEGER :: ifdatim

      NAMELIST /RECORD5/ ifdatim
      
      !  RECORD6
      
      INTEGER :: interp_method
      LOGICAL :: print_info
      LOGICAL :: use_mm5_lowbdy

      NAMELIST /RECORD6/ interp_method , print_info , use_mm5_lowbdy

      !  Local variables.

      LOGICAL :: is_it_there = .FALSE.
      INTEGER , PARAMETER :: unit_nml=10

      !  Initialize the variable "input_file" to blank

      input_file = '                                                                                ' // &
                   '                                                    '
   
      !     Does the file exist?
      
      INQUIRE ( FILE = 'namelist.input' , EXIST = is_it_there )
      
      IF ( is_it_there ) THEN
      
         !  The file exists, get a unit number.
      
         OPEN ( FILE   = 'namelist.input' , &
                UNIT   =  unit_nml        , &
                STATUS = 'OLD'            , &
                FORM   = 'FORMATTED'      , &
                ACTION = 'READ'           , &
                ACCESS = 'SEQUENTIAL'     )
   
         !  File is opened, so read it.
   
         READ (unit_nml , NML = RECORD0 )
!        WRITE (6    , NML = RECORD0 )
         READ (unit_nml , NML = RECORD1 )
         WRITE (6    , NML = RECORD1 )
         READ (unit_nml , NML = RECORD2 )
         WRITE (6    , NML = RECORD2 )
         READ (unit_nml , NML = RECORD4 )
         WRITE (6    , NML = RECORD4 )
         READ (unit_nml , NML = RECORD5 )
         WRITE (6    , NML = RECORD5 )
         READ (unit_nml , NML = RECORD6 )
         WRITE (6    , NML = RECORD6 )
   
      ELSE
         PRINT '(A)','Could not find the namelist: "namelist.input".'
         STOP 'No_namelist_found'
      END IF
   
      CLOSE ( unit_nml )
   
   END SUBROUTINE do_namelist_2

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
   
   SUBROUTINE get_fg_file ( input_file , unit_fg )
   
      IMPLICIT NONE
   
      !  Input variables.

      CHARACTER(LEN=132) , INTENT(IN) :: input_file

      INTEGER , INTENT(IN) :: unit_fg

      !  Local variables.

      LOGICAL :: is_it_there = .FALSE.
   
      !  Does the file exist?
   
      INQUIRE ( FILE = TRIM(input_file) , EXIST = is_it_there )
   
      IF ( is_it_there ) THEN
         OPEN ( FILE   = TRIM(input_file) , &
                UNIT   =  unit_fg         , &
                STATUS = 'OLD'            , &
                FORM   = 'UNFORMATTED'    , &
                ACCESS = 'SEQUENTIAL'     )
         PRINT '(A,A,A)','Opened gridded data file (',TRIM(input_file),') for input.'
      
      ELSE
         PRINT '(A,A,A)','Could not find file ',TRIM(input_file),'.'
         STOP 'No_file_found'
      
      END IF
   
   END SUBROUTINE get_fg_file
   
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
   
   SUBROUTINE open_out_file ( mm_unit , bdy_unit , lbc_unit , domain_id )
   
      IMPLICIT NONE
      
      !  Input variables.

      INTEGER , INTENT(IN) :: mm_unit , bdy_unit , lbc_unit , domain_id
      
      !  Local variables

      LOGICAL :: is_it_there = .FALSE.
      CHARACTER(LEN=132) :: mm_name , bdy_name , lbc_name

      WRITE (  mm_name , '("MMINPUT_DOMAIN",i1)' ) domain_id
      WRITE ( bdy_name ,  '("BDYOUT_DOMAIN",i1)' ) domain_id
      WRITE ( lbc_name , ' ("LOWBDY_DOMAIN",i1)' ) domain_id
      
      OPEN ( FILE   =  TRIM(mm_name)   , &
             UNIT   =  mm_unit         , &
             STATUS = 'UNKNOWN'        , &
             FORM   = 'UNFORMATTED'    , &
             ACCESS = 'SEQUENTIAL'     )
             PRINT '(A)','Opened MMINPUT file for output.'

      OPEN ( FILE   =  TRIM(bdy_name)  , &
             UNIT   =  bdy_unit        , &
             STATUS = 'UNKNOWN'        , &
             FORM   = 'UNFORMATTED'    , &
             ACCESS = 'SEQUENTIAL'     )
             PRINT '(A)','Opened BDYOUT file for output.'

      OPEN ( FILE   =  TRIM(lbc_name)  , &
             UNIT   =  lbc_unit        , &
             STATUS = 'UNKNOWN'        , &
             FORM   = 'UNFORMATTED'    , &
             ACCESS = 'SEQUENTIAL'     )
             PRINT '(A)','Opened LOWBDY file for output.'

   END SUBROUTINE open_out_file

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   SUBROUTINE outmodeln ( current_date19 , &
                          imx , jmx , kxs , immout , itimes , interval , &
                          start_year , start_month , start_day , start_hour , &
                          start_minute , start_second , start_frac , &
                          p0 , tlp , ts0 , tiso, &
                          sighup , &
                          local_pbhi  , local_pbhic  , local_pbhr  , local_pbhrc  , &
                          local_nbhi  , local_nbhic  , local_nbhr  , local_nbhrc  )

      IMPLICIT NONE

      !  Input data.

      CHARACTER (LEN=19)      :: current_date19

      INTEGER                 :: imx , jmx , kxs , immout , itimes , interval , &
                                 start_year , start_month , start_day , start_hour , &
                                 start_minute , start_second , start_frac

      REAL                    :: p0 , tlp , ts0, tiso

      REAL , DIMENSION(:)     :: sighup

      INTEGER            , DIMENSION(50,20) :: local_pbhi  , local_nbhi
      CHARACTER (LEN=80) , DIMENSION(50,20) :: local_pbhic , local_nbhic
      REAL               , DIMENSION(20,20) :: local_pbhr  , local_nbhr
      CHARACTER (LEN=80) , DIMENSION(20,20) :: local_pbhrc , local_nbhrc

      INTEGER :: loop , loop_regular , loop_terrain

      !  Data for small header.

      INTEGER                :: num_dims
      INTEGER , DIMENSION(4) :: start_dims
      INTEGER , DIMENSION(4) :: end_dims
      REAL                   :: xtime
      CHARACTER (LEN= 4)     :: staggering
      CHARACTER (LEN= 4)     :: ordering
      CHARACTER (LEN=24)     :: cur_date
      CHARACTER (LEN= 9)     :: name
      CHARACTER (LEN=25)     :: units
      CHARACTER (LEN=46)     :: description

      !  If this is the first time in here, then we need to build and output 
      !  the big header.

      IF ( itimes .EQ. 1 ) THEN

         !  First, the flag, then the data.

         WRITE ( immout ) bh_flag

         !  Clear the header information.

         bhi  = -999
         bhr  = -999
         bhic = '                                                                                '
         bhrc = '                                                                                '
  
         !  Initialize the big header to the information contained in the MM5 output
         !  header, through the objective analysis level and excluding the the TERRAIN 
         !  information.

         bhi (:,2:4)  = local_pbhi (:,2:4)
         bhic(:,2:4)  = local_pbhic(:,2:4)
         bhr (:,2:4)  = local_pbhr (:,2:4)
         bhrc(:,2:4)  = local_pbhrc(:,2:4)

         !  Adjust the big header to reflect the nesting.  All of this information is
         !  contained in the TERRAIN location.

         bhi (:,1)  = local_nbhi (:,1)
         bhic(:,1)  = local_nbhic(:,1)
         bhr (:,1)  = local_nbhr (:,1)
         bhrc(:,1)  = local_nbhrc(:,1)

         !  Identify the data as model input.

         bhi( 1,1) = 5 ; bhic( 1,1) = '1-way nest meteorological initial condition for MM5                             '
 
         !  A few lines about data format and version numbers.

         bhi( 2,5) = 1 ; bhic( 2,5) = 'NESTDOWN Version 3 MM5 System Format Edition Number                             '
         bhi( 3,5) = 7 ; bhic( 3,5) = 'NESTDOWN Program Version Number                                                 '
         bhi( 4,5) = 0 ; bhic( 4,5) = 'NESTDOWN Program Minor Revision Number                                          '
         
         !  Starting date of the NESTDOWN data, this was specified in the namelist.

         bhi ( 5,5) = start_year   
         bhic( 5,5) = 'Four-digit year of start time                                                   '
         bhi ( 6,5) = start_month
         bhic( 6,5) = 'Month of the year of the start time (1-12)                                      '
         bhi ( 7,5) = start_day
         bhic( 7,5) = 'Day of the month of the start time (1-31)                                       '
         bhi ( 8,5) = start_hour
         bhic( 8,5) = 'Hour of the day of the start time (0-23)                                        '
         bhi ( 9,5) = start_minute
         bhic( 9,5) = 'Minute of the start time (0-59)                                                 '
         bhi (10,5) = start_second
         bhic(10,5) = 'Second of the start time (0-59)                                                 '
         bhi (11,5) = start_frac
         bhic(11,5) = 'Ten thousandths of a second of the start time (0-9999)                          '

         !  How many half sigma levels to expect.

         bhi (12,5) = kxs
         bhic(12,5) = 'Number of half-sigma layers in the model input data (top down)                  '

         !  Time interval between output files.

         bhr ( 1,5) = interval
         bhrc( 1,5) = 'Time difference (seconds) between model IC input files                          '

         !  Base state information.

         bhr ( 2,5) = p0
         bhrc( 2,5) = 'Non-hydrostatic base state sea-level pressure (Pa)                              '
         bhr ( 3,5) = ts0
         bhrc( 3,5) = 'Non-hydrostatic base state sea-level temperature (K)                            '
         bhr ( 4,5) = tlp
         bhrc( 4,5) = 'Non-hydrostatic base state lapse rate d(T)/d(ln P)                              '
         bhr ( 5,5) = tiso
         bhrc( 5,5) = 'Non-hydrostatic base state stratospheric isothermal temperature (K)             '

         !  Alrighty, we can output it now.
  
         WRITE ( immout ) bhi , bhr , bhic , bhrc
         
      END IF

      !  Triplets of data follow: 1) small head flag, 2) small header, 3) data.
      !  Each triplet is for a single array (3d, 2d, or 1d).

      output_3d : DO loop = 1 , num_3d
         WRITE ( immout ) sh_flag
         num_dims        = nall_3d(loop)%small_header%num_dims
         start_dims      = nall_3d(loop)%small_header%start_dims
         end_dims        = nall_3d(loop)%small_header%end_dims
         xtime           = 0
         staggering      = nall_3d(loop)%small_header%staggering
         ordering        = nall_3d(loop)%small_header%ordering
         cur_date        = current_date19 // '.0000'
         name            = nall_3d(loop)%small_header%name
         units           = nall_3d(loop)%small_header%units
         description     = nall_3d(loop)%small_header%description
         WRITE ( immout ) num_dims , start_dims , end_dims , xtime , &
                          staggering , ordering , cur_date , name , units , description
         WRITE ( immout ) nall_3d(loop)%array
      END DO output_3d

      output_2d : DO loop = 1 , num_2d
         WRITE ( immout ) sh_flag
         num_dims        = nall_2d(loop)%small_header%num_dims
         start_dims      = nall_2d(loop)%small_header%start_dims
         end_dims        = nall_2d(loop)%small_header%end_dims
         xtime           = 0
         staggering      = nall_2d(loop)%small_header%staggering
         ordering        = nall_2d(loop)%small_header%ordering
         cur_date        = current_date19 // '.0000'
         name            = nall_2d(loop)%small_header%name
         units           = nall_2d(loop)%small_header%units
         description     = nall_2d(loop)%small_header%description
         WRITE ( immout ) num_dims , start_dims , end_dims , xtime , &
                          staggering , ordering , cur_date , name , units , description
         WRITE ( immout ) nall_2d(loop)%array
      END DO output_2d

      !  This is the 2d data that comes from the TERRAIN file that is not already included
      !  in the MM5 file that was used as input.  

      output_terrain : DO loop_terrain = 1 , num_terrain_2d
         regular_2d : DO loop_regular = 1 , num_2d
            IF ( nall_2d(loop_regular)%small_header%name .EQ. all_terrain(loop_terrain)%small_header%name ) THEN
               CYCLE output_terrain
            END IF
         END DO regular_2d
         WRITE ( immout ) sh_flag
         num_dims        = all_terrain(loop_terrain)%small_header%num_dims
         start_dims      = all_terrain(loop_terrain)%small_header%start_dims
         end_dims        = all_terrain(loop_terrain)%small_header%end_dims
         xtime           = 0
         staggering      = all_terrain(loop_terrain)%small_header%staggering
         ordering        = all_terrain(loop_terrain)%small_header%ordering
         cur_date        = current_date19 // '.0000'
         name            = all_terrain(loop_terrain)%small_header%name
         units           = all_terrain(loop_terrain)%small_header%units
         description     = all_terrain(loop_terrain)%small_header%description
         WRITE ( immout ) num_dims , start_dims , end_dims , xtime , &
                          staggering , ordering , cur_date , name , units , description
         WRITE ( immout ) all_terrain(loop_terrain)%array
      END DO output_terrain

      !  There is an important 1D array, the vertical coordinate sigma (top-down, half layers).

      WRITE ( immout ) sh_flag
      num_dims        = 1
      start_dims      = (/ 1, 1, 1, 1 /)
      end_dims        = (/ kxs, 1, 1, 1 /)
      xtime           = 0
      staggering      = 'H   '
      ordering        = 'S   '
      cur_date        = current_date19 // '.0000'
      name            = 'SIGMAH   '
      units           = 'Pa Pa{-1}                '
      description     = 'Top-down, half sigma layers                   '
      WRITE ( immout ) num_dims , start_dims , end_dims , xtime , &
                       staggering , ordering , cur_date , name , units , description
      WRITE ( immout ) sighup
      
      !  The last thing you do before going to sleep at night is to put out the 
      !  end of time flag.

      WRITE ( immout ) eot_flag

   END SUBROUTINE outmodeln

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   SUBROUTINE outmodelv ( current_date19 , &
                          imx , jmx , kxs , immout , itimes , interval , &
                          start_year , start_month , start_day , start_hour , &
                          start_minute , start_second , start_frac , &
                          p0 , tlp , ts0 , tiso, &
                          sighup , vert_nesting , &
                          local_pbhi  , local_pbhic  , local_pbhr  , local_pbhrc  , &
                          local_nbhi  , local_nbhic  , local_nbhr  , local_nbhrc  )

      IMPLICIT NONE

      !  Input data.

      CHARACTER (LEN=19)      :: current_date19

      INTEGER                 :: imx , jmx , kxs , immout , itimes , interval , &
                                 start_year , start_month , start_day , start_hour , &
                                 start_minute , start_second , start_frac

      REAL                    :: p0 , tlp , ts0, tiso

      REAL , DIMENSION(:)     :: sighup
      LOGICAL , INTENT(IN)    :: vert_nesting

      INTEGER            , DIMENSION(50,20) :: local_pbhi  , local_nbhi
      CHARACTER (LEN=80) , DIMENSION(50,20) :: local_pbhic , local_nbhic
      REAL               , DIMENSION(20,20) :: local_pbhr  , local_nbhr
      CHARACTER (LEN=80) , DIMENSION(20,20) :: local_pbhrc , local_nbhrc

      INTEGER :: loop , loop_regular , loop_terrain

      !  Data for small header.

      INTEGER                :: num_dims
      INTEGER , DIMENSION(4) :: start_dims
      INTEGER , DIMENSION(4) :: end_dims
      REAL                   :: xtime
      CHARACTER (LEN= 4)     :: staggering
      CHARACTER (LEN= 4)     :: ordering
      CHARACTER (LEN=24)     :: cur_date
      CHARACTER (LEN= 9)     :: name
      CHARACTER (LEN=25)     :: units
      CHARACTER (LEN=46)     :: description

      !  If this is the first time in here, then we need to build and output 
      !  the big header.

      IF ( itimes .EQ. 1 ) THEN

         !  First, the flag, then the data.

         WRITE ( immout ) bh_flag

         !  Clear the header information.

         bhi  = -999
         bhr  = -999
         bhic = '                                                                                '
         bhrc = '                                                                                '
  
         !  Initialize the big header to the information contained in the MM5 output
         !  header, through the objective analysis level and excluding the the TERRAIN 
         !  information.

         bhi (:,2:4)  = local_pbhi (:,2:4)
         bhic(:,2:4)  = local_pbhic(:,2:4)
         bhr (:,2:4)  = local_pbhr (:,2:4)
         bhrc(:,2:4)  = local_pbhrc(:,2:4)

         !  Adjust the big header to reflect the nesting.  All of this information is
         !  contained in the TERRAIN location.

         bhi (:,1)  = local_nbhi (:,1)
         bhic(:,1)  = local_nbhic(:,1)
         bhr (:,1)  = local_nbhr (:,1)
         bhrc(:,1)  = local_nbhrc(:,1)

         !  Identify the data as model input.

         bhi( 1,1) = 5 ; bhic( 1,1) = '1-way nest meteorological initial condition for MM5                             '
 
         !  A few lines about data format and version numbers.

         bhi( 2,5) = 1 ; bhic( 2,5) = 'NESTDOWN Version 3 MM5 System Format Edition Number                             '
         bhi( 3,5) = 7 ; bhic( 3,5) = 'NESTDOWN Program Version Number                                                 '
         bhi( 4,5) = 0 ; bhic( 4,5) = 'NESTDOWN Program Minor Revision Number                                          '
         
         !  Starting date of the NESTDOWN data, this was specified in the namelist.

         bhi ( 5,5) = start_year   
         bhic( 5,5) = 'Four-digit year of start time                                                   '
         bhi ( 6,5) = start_month
         bhic( 6,5) = 'Month of the year of the start time (1-12)                                      '
         bhi ( 7,5) = start_day
         bhic( 7,5) = 'Day of the month of the start time (1-31)                                       '
         bhi ( 8,5) = start_hour
         bhic( 8,5) = 'Hour of the day of the start time (0-23)                                        '
         bhi ( 9,5) = start_minute
         bhic( 9,5) = 'Minute of the start time (0-59)                                                 '
         bhi (10,5) = start_second
         bhic(10,5) = 'Second of the start time (0-59)                                                 '
         bhi (11,5) = start_frac
         bhic(11,5) = 'Ten thousandths of a second of the start time (0-9999)                          '

         !  How many half sigma levels to expect.

         bhi (12,5) = kxs
         bhic(12,5) = 'Number of half-sigma layers in the model input data (top down)                  '

         !  Time interval between output files.

         bhr ( 1,5) = interval
         bhrc( 1,5) = 'Time difference (seconds) between model IC input files                          '

         !  Base state information.

         bhr ( 2,5) = p0
         bhrc( 2,5) = 'Non-hydrostatic base state sea-level pressure (Pa)                              '
         bhr ( 3,5) = ts0
         bhrc( 3,5) = 'Non-hydrostatic base state sea-level temperature (K)                            '
         bhr ( 4,5) = tlp
         bhrc( 4,5) = 'Non-hydrostatic base state lapse rate d(T)/d(ln P)                              '
         bhr ( 5,5) = tiso
         bhrc( 5,5) = 'Non-hydrostatic base state stratospheric isothermal temperature (K)             '

         !  Alrighty, we can output it now.
  
         WRITE ( immout ) bhi , bhr , bhic , bhrc
         
      END IF

      !  Triplets of data follow: 1) small head flag, 2) small header, 3) data.
      !  Each triplet is for a single array (3d, 2d, or 1d).

      IF ( vert_nesting ) THEN 
         output_3dv : DO loop = 1 , num_3d
            WRITE ( immout ) sh_flag
            num_dims        = vall_3d(loop)%small_header%num_dims
            start_dims      = vall_3d(loop)%small_header%start_dims
            end_dims        = vall_3d(loop)%small_header%end_dims
            xtime           = 0
            staggering      = vall_3d(loop)%small_header%staggering
            ordering        = vall_3d(loop)%small_header%ordering
            cur_date        = current_date19 // '.0000'
            name            = vall_3d(loop)%small_header%name
            units           = vall_3d(loop)%small_header%units
            description     = vall_3d(loop)%small_header%description
            WRITE ( immout ) num_dims , start_dims , end_dims , xtime , &
                             staggering , ordering , cur_date , name , units , description
            WRITE ( immout ) vall_3d(loop)%array
         END DO output_3dv
      ELSE
         output_3d : DO loop = 1 , num_3d
            WRITE ( immout ) sh_flag
            num_dims        = nall_3d(loop)%small_header%num_dims
            start_dims      = nall_3d(loop)%small_header%start_dims
            end_dims        = nall_3d(loop)%small_header%end_dims
            xtime           = 0
            staggering      = nall_3d(loop)%small_header%staggering
            ordering        = nall_3d(loop)%small_header%ordering
            cur_date        = current_date19 // '.0000'
            name            = nall_3d(loop)%small_header%name
            units           = nall_3d(loop)%small_header%units
            description     = nall_3d(loop)%small_header%description
            WRITE ( immout ) num_dims , start_dims , end_dims , xtime , &
                             staggering , ordering , cur_date , name , units , description
            WRITE ( immout ) nall_3d(loop)%array
         END DO output_3d
      END IF

      output_2d : DO loop = 1 , num_2d
         WRITE ( immout ) sh_flag
         num_dims        = nall_2d(loop)%small_header%num_dims
         start_dims      = nall_2d(loop)%small_header%start_dims
         end_dims        = nall_2d(loop)%small_header%end_dims
         xtime           = 0
         staggering      = nall_2d(loop)%small_header%staggering
         ordering        = nall_2d(loop)%small_header%ordering
         cur_date        = current_date19 // '.0000'
         name            = nall_2d(loop)%small_header%name
         units           = nall_2d(loop)%small_header%units
         description     = nall_2d(loop)%small_header%description
         WRITE ( immout ) num_dims , start_dims , end_dims , xtime , &
                          staggering , ordering , cur_date , name , units , description
         WRITE ( immout ) nall_2d(loop)%array
      END DO output_2d

      !  This is the 2d data that comes from the TERRAIN file that is not already included
      !  in the MM5 file that was used as input.  

      output_terrain : DO loop_terrain = 1 , num_terrain_2d
         regular_2d : DO loop_regular = 1 , num_2d
            IF ( nall_2d(loop_regular)%small_header%name .EQ. all_terrain(loop_terrain)%small_header%name ) THEN
               CYCLE output_terrain
            END IF
         END DO regular_2d
         WRITE ( immout ) sh_flag
         num_dims        = all_terrain(loop_terrain)%small_header%num_dims
         start_dims      = all_terrain(loop_terrain)%small_header%start_dims
         end_dims        = all_terrain(loop_terrain)%small_header%end_dims
         xtime           = 0
         staggering      = all_terrain(loop_terrain)%small_header%staggering
         ordering        = all_terrain(loop_terrain)%small_header%ordering
         cur_date        = current_date19 // '.0000'
         name            = all_terrain(loop_terrain)%small_header%name
         units           = all_terrain(loop_terrain)%small_header%units
         description     = all_terrain(loop_terrain)%small_header%description
         WRITE ( immout ) num_dims , start_dims , end_dims , xtime , &
                          staggering , ordering , cur_date , name , units , description
         WRITE ( immout ) all_terrain(loop_terrain)%array
      END DO output_terrain

      !  There is an important 1D array, the vertical coordinate sigma (top-down, half layers).

      WRITE ( immout ) sh_flag
      num_dims        = 1
      start_dims      = (/ 1, 1, 1, 1 /)
      end_dims        = (/ kxs, 1, 1, 1 /)
      xtime           = 0
      staggering      = 'H   '
      ordering        = 'S   '
      cur_date        = current_date19 // '.0000'
      name            = 'SIGMAH   '
      units           = 'Pa Pa{-1}                '
      description     = 'Top-down, half sigma layers                   '
      WRITE ( immout ) num_dims , start_dims , end_dims , xtime , &
                       staggering , ordering , cur_date , name , units , description
      WRITE ( immout ) sighup
      
      !  The last thing you do before going to sleep at night is to put out the 
      !  end of time flag.

      WRITE ( immout ) eot_flag

   END SUBROUTINE outmodelv

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

END MODULE all_io
