MODULE base_state

CONTAINS

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   SUBROUTINE dfdp (f, ips0dot, nps0dot, ptop, sigma, nimax, njmax, kxs)
   
      !  This SUBROUTINE adjusts each column of wind to the new terrain.
   
      IMPLICIT NONE
   
      REAL                    :: f          ( : , : , : )
      REAL                    :: ips0dot    ( : , : )
      REAL                    :: nps0dot    ( : , : )
      REAL                    :: sigma      ( : )
      REAL                    :: ptop
      INTEGER                 :: nimax
      INTEGER                 :: njmax
      INTEGER                 :: kxs
      INTEGER                 :: i
      INTEGER                 :: j
      INTEGER                 :: k
      REAL                    :: mbpt
      REAL                    :: pf
      REAL                    :: pfu
      REAL                    :: pfl
      REAL                    :: pfn
      REAL                    :: dp
      REAL                    :: dfldp
      INTEGER                 :: km1
      INTEGER                 :: kp1
   
      !  Loop over each (i,j,k) for the vertical adjustment of the horizontal wind
      !  components due to changes in the terrain elevation.
   
!$OMP PARALLEL DO DEFAULT ( SHARED ) &
!$OMP PRIVATE ( i , j , k , pf , km1 , kp1 , pfu , pfl , dp , dfldp )
      DO j = 1, njmax
         DO k = kxs, 1, -1
   
            !  Bound the vertical levels above and below with 1 and kxs.
  
            km1 = MAX(k-1,1)
            kp1 = MIN(k+1,kxs)
   
            DO i = 1, nimax
   
               !  Interpolated reference pressure at this (i,j,k).
    
               pf  = sigma(k)   * ips0dot(i,j) + ptop

               !  Interpolated reference pressure above and below current k-level.
   
               pfu = sigma(km1) * ips0dot(i,j) + ptop
               pfl = sigma(kp1) * ips0dot(i,j) + ptop
   
               !  Nest-domain (non-interpolated) reference pressure at this (i,j,k).
   
               pfn = sigma(k)   * nps0dot(i,j)   + ptop
            
               !  Difference in pressure at each (i,j,k) between the interpolated
               !  reference pressure (based on the interpolated terrain elevation) and
               !  the nest domain reference pressure (based on the fine-grid terrain
               !  elevation).
              
               dp  = pfn - pf
      
               !  To remove errors caused by having the wind near the surface in affected
               !  by free atmosphere winds, we bound the vertical depth of the 
               !  interpolation/extrapolation to 1 sigma level above or 1 sigma level 
               !  below the current level.
      
               dp = MIN(dp, pfl -  pf)
               dp = MAX(dp, pfu - pf)
   
               !  Vertical gradient. 
   
               dfldp = (f(i,j,kp1) - f(i,j,km1)) / (pfl - pfu)
   
               !  Adjust wind component at this (i,j,k). 
   
               f(i,j,k) = f(i,j,k) + dp * dfldp
            END DO
         END DO
      END DO
   
   END SUBROUTINE dfdp

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   SUBROUTINE nhbase (terr, imx, jmx, kxs, ps0, pr0, t0, &
                         ptop, p0, tlp, ts0, tiso, sigma)
   
      !  Purpose: compute the nonhydrostatic base state
   
      IMPLICIT NONE
   
      REAL,          PARAMETER    :: g           =    9.81
      REAL,          PARAMETER    :: r           =  287.04
   
      INTEGER                     :: imx
      INTEGER                     :: jmx
      INTEGER                     :: i,j,k
      INTEGER                     :: kxs
   
      REAL                        :: sigma       ( : )
      REAL                        :: p0
      REAL                        :: pr0         ( : , : , : )
      REAL                        :: ps0         ( : , : )
      REAL                        :: ptop
      REAL                        :: terr        ( : , : )
      REAL                        :: tlp
      REAL                        :: tiso
      REAL                        :: invtlp
      REAL                        :: t0          ( : , : , : )
      REAL                        :: ts0
      REAL                        :: r1
      REAL                        :: r2
      REAL                        :: r3
      REAL                        :: ts0ovtlp
   
      !  Non-hydrostatic base state calculation is described on page 7-8
      !  of the MM5 Tutorial Manual, January 1998.  These calculations should be 
      !  done in pascals, kelvin, meters, etc.
      !  p0 and ptop are already in pascals
      !  terrain is in meters
      !  ts0, tiso are in kelvin
      !  g is meters per second squared
      !  r is joules per Kelvin per kilogram
   
      ts0ovtlp = ts0/tlp
      r1 = r*tlp/(2*g)
      r2 = r*ts0/g
      r3 = 1./r1
   
      !  Compute reference pstar array: ps0
   
      ps0 = p0 * EXP( -ts0ovtlp + sqrt((ts0ovtlp)**2 - r3*terr)) - ptop

      !  Compute reference state pressure array: pr0
      !  Compute reference state temperature array: t0
   
!$OMP PARALLEL DO DEFAULT ( SHARED ) &
!$OMP PRIVATE ( i , j , k )
      DO j = 1, jmx
         DO k = 1, kxs
            DO i = 1, imx
              pr0(i,j,k) = ps0(i,j) * sigma(k) + ptop
              t0 (i,j,k) = MAX ( ts0 + tlp  *  LOG(pr0(i,j,k) / p0) , tiso ) 
            END DO
         END DO
      END DO
   
   END SUBROUTINE nhbase

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   SUBROUTINE rhtoq (rh, t, pp, sighup, p, ptop, imxm, jmxm, kxs, wrth2o , q)

      IMPLICIT NONE

      INTEGER                     :: I
      INTEGER                     :: IMXM
      INTEGER                     :: J
      INTEGER                     :: JMXM
      INTEGER                     :: K
      INTEGER                     :: KXS

      LOGICAL                     :: wrth2o

      REAL                        :: EW
      REAL                        :: P           ( : , : )
      REAL                        :: PP          ( : , : , : )
      REAL                        :: PTOP
      REAL                        :: Q           ( : , : , : )
      REAL                        :: Q1
      REAL                        :: RH          ( : , : , : )
      REAL                        :: SIGHUP      ( : )
      REAL                        :: T           ( : , : , : )
      REAL                        :: T1
      REAL,DIMENSION(imxm,jmxm,kxs)::es1,qs1

      REAL,         PARAMETER     :: T_REF       = 0.0
      REAL,         PARAMETER     :: MW_AIR      = 28.966
      REAL,         PARAMETER     :: MW_VAP      = 18.0152

      REAL,         PARAMETER     :: A0       = 6.107799961
      REAL,         PARAMETER     :: A1       = 4.436518521e-01
      REAL,         PARAMETER     :: A2       = 1.428945805e-02
      REAL,         PARAMETER     :: A3       = 2.650648471e-04
      REAL,         PARAMETER     :: A4       = 3.031240396e-06
      REAL,         PARAMETER     :: A5       = 2.034080948e-08
      REAL,         PARAMETER     :: A6       = 6.136820929e-11

      REAL,         PARAMETER     :: ES0 = 6.1121

      REAL,         PARAMETER     :: C1       = 9.09718
      REAL,         PARAMETER     :: C2       = 3.56654
      REAL,         PARAMETER     :: C3       = 0.876793
      REAL,         PARAMETER     :: EIS      = 6.1071
      REAL                        :: RHS
      REAL,         PARAMETER     :: TF       = 273.16
      REAL                        :: TK

      REAL                        :: ES
      REAL                        :: QS
      REAL,         PARAMETER     :: EPS         = 0.622
      REAL,         PARAMETER     :: SVP1        = 0.6112
      REAL,         PARAMETER     :: SVP2        = 17.67
      REAL,         PARAMETER     :: SVP3        = 29.65
      REAL,         PARAMETER     :: SVPT0       = 273.15

      !  This function is designed to compute (q) from basic variables
      !  p (mb), t(K) and rh(0-100%) to give (q) in (kg/kg).  The reference
      !  temperature t_ref (c) is used to describe the temperature at which
      !  the liquid and ice phase change occurs.

      IF ( wrth2o ) THEN
         es1=svp1*10.*EXP(svp2*(t-svpt0)/(t-svp3))
         DO k = 1, kxs
!           qs1(:,:,k)=eps*es1(:,:,k)/(sighup(k)*p/100.+ptop/100.+pp(:,:,k)/100.-(1.-eps)*es1(:,:,k))
            qs1(:,:,k)=eps*es1(:,:,k)/(sighup(k)*p/100.+ptop/100.+pp(:,:,k)/100.-         es1(:,:,k))
         END DO
         q=MAX(.01*rh*qs1,0.0)

      ELSE
         DO K = 1, KXS
            DO J = 1, JMXM
               DO I = 1, IMXM

                  t1 = t(i,j,k) - 273.16

                  !  Obviously dry.

                  IF ( t1 .lt. -200. ) THEN
                     q(i,j,k) = 0

                  ELSE

                     !  First compute the ambient vapor pressure of water

                     IF ( ( t1 .GE. t_ref ) .AND. ( t1 .GE. -47.) ) THEN    ! liq phase ESLO
                        ew = a0 + t1 * (a1 + t1 * (a2 + t1 * (a3 + t1 * (a4 + t1 * (a5 + t1 *  a6)))))

!         elseif (t1 .le. t_ref .and. t1 .ge. -47.) then !liq phas poor ES
                     ELSE IF ( ( t1 .GE. t_ref ) .AND. ( t1 .LT. -47. ) ) then !liq phas poor ES
                        ew = es0 * exp(17.67 * t1 / ( t1 + 243.5))

                     ELSE
                        tk = t(i,j,k)
                        rhs = -c1 * (tf / tk - 1.) - c2 * alog10(tf / tk) +  c3 * (1. - tk / tf) + alog10(eis)
                        ew = 10. ** rhs

                     END IF

                     ew = MAX ( ew , 0. )

                     !  Now sat vap pres obtained compute local vapor pressure
  
                     ew = ew * rh(i,j,k) * 0.01

                     !  Now compute the specific humidity using the partial vapor
                     !  pressures of water vapor (ew) and dry air (p-ew).

                     q1 = mw_vap * ew
                     q1 = q1 / (q1 + mw_air * (p(i,j)*sighup(k)+ptop+pp(i,j,k) - ew))

                     q(i,j,k) = q1 / (1. - q1 )

                  END IF

               END DO
            END DO
         END DO

      END IF

   END SUBROUTINE rhtoq

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   SUBROUTINE qtorh (q, t, pp, sigh, ps, ptop, imx, jmx, kx, wrth2o, rh)
   
      IMPLICIT NONE
   
      !  Compute rh on sigma.  This computation comes from Rogers
      !  and Yau, A Short Course in Cloud Physics, 3rd edition, pages 16
      !  and 17.  Incoming units are:
      !        [t] = Kelvin
      !        [pp] = Pascals
      !        [ps] = Pascals
      !        [ptop] = Pascals
      !  The computation requires [t] = celcius and [pressure] = millibars.
   
      REAL                        :: q      ( : , : , : )
      REAL                        :: t      ( : , : , : )
      REAL                        :: pp     ( : , : , : )
      REAL                        :: sigh   ( : )
      REAL                        :: ps     ( : , : )
      REAL                        :: es
      REAL                        :: qs
      REAL                        :: ptop
      REAL                        :: ppptmb
      REAL                        :: tc
      INTEGER                     :: imx
      INTEGER                     :: jmx
      INTEGER                     :: kx
      REAL                        :: rh     ( : , : , : )
      INTEGER                     :: i
      INTEGER                     :: j
      INTEGER                     :: k
      LOGICAL                     :: wrth2o

      REAL                        :: psmb   ( imx , jmx )
      REAL,DIMENSION(imx,jmx,kx)  :: es1,qs1
   
      REAL,        PARAMETER      :: svp1  = 6.112
      REAL,        PARAMETER      :: svp2  = 17.67
      REAL,        PARAMETER      :: svp3  = 243.5
      REAL,        PARAMETER      :: eps   = 0.622
   
      !  Incoming pressure arguments are not in mb.  Make one array by
      !  combining, pp and ptop and correct for units while you do it. Make
      !  a second array for pstar in millibars.  Incoming temperature array 
      !  is not in celcius.  Convert to celcius and store in tc.
   
      psmb = ps * 0.01
   
      !  Compute sat. vap. pres. (es), sat. mixrat. (qs), then rel hum (rh)
   
      IF ( wrth2o ) THEN
 
         es1 = svp1 * EXP(svp2 * (t - 273.15) / (t - 273.15 + svp3))
         DO k = 1, kx
            qs1(:,:,k) = eps * es1(:,:,k) / (sigh(k) * psmb + ((pp(:,:,k) + ptop) * .01) - (0.378 * es1(:,:,k)))
         END DO
         rh = MIN ( MAX ( 100. * q / qs1 , 1. ) , 100. )

      ELSE
print *,'no rh to q computation wrt ice yet.'
stop 'under_development_ice'
      END IF
   
      rh(:,jmx,:) = rh(:,jmx-1,:)
      rh(imx,:,:) = rh(imx-1,:,:)
   
   END SUBROUTINE qtorh

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

END MODULE base_state
