PROGRAM nestdown

   USE all_io
   USE base_state
   USE bdy
   USE date_pack
   USE file
   USE header_data
   USE horiz_interp
   USE lateral_bdy
   USE vert_interp
   USE util

   IMPLICIT NONE

   !  Declarations for MM5 output variables: "p"varname means "parent" grid.

   INTEGER                    :: pbhi      (   50 , 20 )
   REAL                       :: pbhr      (   20 , 20 )
   CHARACTER*80               :: pbhic     (   50 , 20 )
   CHARACTER*80               :: pbhrc     (   20 , 20 )
   INTEGER                    :: plbhi     (   50 , 20 )
   REAL                       :: plbhr     (   20 , 20 )
   CHARACTER*80               :: plbhic    (   50 , 20 )
   CHARACTER*80               :: plbhrc    (   20 , 20 )
   REAL,          ALLOCATABLE :: pps0      ( : , : )
   REAL,          ALLOCATABLE :: ppr0      ( : , : , : )
   REAL,          ALLOCATABLE :: pt0       ( : , : , : )
   REAL,          ALLOCATABLE :: pps0dot   ( : , : )

   !  Declarations for nestdown output variables: "n"varname is "nest" grid.

   INTEGER                    :: nbhi      (   50 , 20 )
   REAL                       :: nbhr      (   20 , 20 )
   CHARACTER*80               :: nbhic     (   50 , 20 )
   CHARACTER*80               :: nbhrc     (   20 , 20 )
   INTEGER                    :: nlbhi     (   50 , 20 )
   REAL                       :: nlbhr     (   20 , 20 )
   CHARACTER*80               :: nlbhic    (   50 , 20 )
   CHARACTER*80               :: nlbhrc    (   20 , 20 )
   REAL,          ALLOCATABLE :: nseaice   ( : , : )
   REAL,          ALLOCATABLE :: nps0      ( : , : )
   REAL,          ALLOCATABLE :: nps0dot   ( : , : )
   REAL,          ALLOCATABLE :: npr0      ( : , : , : )
   REAL,          ALLOCATABLE :: nt0       ( : , : , : )
   REAL,          ALLOCATABLE :: nrh       ( : , : , : )

   !  Declarations for nestdown output variables: "v"varname is "vertically nested" grid.

   INTEGER                    :: vbhi      (   50 , 20 )
   REAL                       :: vbhr      (   20 , 20 )
   CHARACTER*80               :: vbhic     (   50 , 20 )
   CHARACTER*80               :: vbhrc     (   20 , 20 )
   INTEGER                    :: vlbhi     (   50 , 20 )
   REAL                       :: vlbhr     (   20 , 20 )
   CHARACTER*80               :: vlbhic    (   50 , 20 )
   CHARACTER*80               :: vlbhrc    (   20 , 20 )
   REAL,          ALLOCATABLE :: vpr0      ( : , : , : )
   REAL,          ALLOCATABLE :: vt0       ( : , : , : )
   REAL,          ALLOCATABLE :: vrh       ( : , : , : )

   !     declarations for other required variables to run this PROGRAM

   REAL,          ALLOCATABLE :: sigma     ( : ) , sigma_full_old ( : ) , sigma_half_new ( : ) , sigma_full_new ( : ) 
   REAL,          ALLOCATABLE :: ips0      ( : , : )
   REAL,          ALLOCATABLE :: ips0dot   ( : , : )
   REAL,          ALLOCATABLE :: ipt0      ( : , : , : )
   REAL,          ALLOCATABLE :: t0diff    ( : , : , : )
   REAL,          ALLOCATABLE :: newcoord  ( : )
   REAL                       :: ptop
   REAL                       :: p0
   REAL                       :: tlp
   REAL                       :: ts0
   REAL                       :: tiso
   INTEGER                    :: icrsdot
   INTEGER                    :: iratio
   INTEGER                    :: kxs , kxs_full_new , kxs_half_new , k_loop , numk
   INTEGER                    :: nesti
   INTEGER                    :: nestj
   INTEGER                    :: nimax
   INTEGER                    :: njmax
   INTEGER                    :: loopcnt , loop_count , loop , iprocess , itimes , input_count
   LOGICAL                    :: found
   INTEGER                    :: i , j , k , local_k , kx
   INTEGER                    :: imax
   INTEGER                    :: jmax
   INTEGER                    :: nratio
   INTEGER                    :: return_code
   LOGICAL                    :: vert_nesting = .FALSE.
   INTEGER                    :: time_difference_seconds

   !  Specific locations from the big arrays of all data.

   INTEGER :: index_monalb01 = 0 , index_monalb02 = 0 , index_monalb03 = 0 , &
              index_monalb04 = 0 , index_monalb05 = 0 , index_monalb06 = 0 , &
              index_monalb07 = 0 , index_monalb08 = 0 , index_monalb09 = 0 , &
              index_monalb10 = 0 , index_monalb11 = 0 , index_monalb12 = 0 , &
              index_albsnomx = 0 , index_albedo   = 0
   INTEGER :: index_terrain  = 0 , index_pstarcrs = 0 , index_snowcovr = 0 , &
              index_tseasfc  = 0 , index_groundt  = 0 ,                      &
              index_tempgrd  = 0 , index_restemp  = 0 ,                      &
              index_landuse  = 0 , index_seaice   = 0 , index_latitude = 0 , &
              index_soilt400 = 0 , index_soilt200 = 0 , index_soilt100 = 0 , &
              index_soilt040 = 0 , index_soilt010 = 0 ,                      &
              index_soilt1   = 0 , index_soilt2   = 0 , index_soilt3   = 0 , &
              index_soilt4   = 0 , index_soilt5   = 0 , index_soilt6   = 0 , &
              index_soilm400 = 0 , index_soilm200 = 0 , index_soilm100 = 0 , &
              index_soilm040 = 0 , index_soilm010 = 0 ,                      &
              index_soilm1   = 0 , index_soilm2   = 0 , index_soilm3   = 0 , &
              index_soilm4   = 0 , index_soilm5   = 0 , index_soilm6   = 0
   INTEGER :: index_t        = 0 , index_q        = 0 , index_pp       = 0 , &
              index_u        = 0 , index_v        = 0

   !  NAMELIST data

   !  Record 0

   CHARACTER(LEN=132) , DIMENSION(100) :: input_file
   CHARACTER(LEN=132)                  :: input_lowbdy_file , input_terrain_file

   !  Record 1

   INTEGER :: start_year , start_month , start_day , start_hour , &
              start_minute = 0, start_second = 0, start_frac = 0
   INTEGER ::   end_year ,   end_month ,   end_day ,   end_hour , &
              end_minute = 0,   end_second = 0,   end_frac = 0
   INTEGER :: interval
   LOGICAL :: less_than_24h = .FALSE.

   !  Record 2

   REAL , DIMENSION(1000) :: sigma_f_bu = -1.
   REAL    :: sst_to_ice_threshold = 0

   !  Record 4

   LOGICAL :: wrth2o = .TRUE.

   !  Record 5

   INTEGER :: ifdatim = -1

   !  Record 6

   INTEGER :: interp_method = 1
   LOGICAL :: print_info = .FALSE.
   LOGICAL :: use_mm5_lowbdy = .TRUE.

   !  ALLOCATE space for the 3d, 2d, and 1d data for the coarse grid MM5 data set.
   !  The leading "p" is for parent.

   ALLOCATE ( pall_3d( 30) , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for init P 3D field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,30
#endif
   END IF

   ALLOCATE ( pall_2d(100) , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for init P 2D field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,100
#endif
   END IF

   ALLOCATE ( pall_1d(  5) , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for init P 1D field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,1
#endif
   END IF

   !  ALLOCATE space for the 3d, 2d, and 1d data for the nest data set that we will 
   !  create.  This is the model input format.  The leading "n" is for nest.

   ALLOCATE ( nall_3d( 30) , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for init N 3D field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,30
#endif
   END IF

   ALLOCATE ( nall_2d(100) , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for init N 2D field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,100
#endif
   END IF

   ALLOCATE ( nall_1d(  5) , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for init N 1D field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,5
#endif
   END IF

   !  ALLOCATE space for the 3d data for the vertically nested data set that we will 
   !  create.  This is the model input format.  The leading "v" is for vertically nested.

   ALLOCATE ( vall_3d( 30) , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for init V 3D field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,30
#endif
   END IF

   !  ALLOCATE space for the TERRAIN data from the nested domain.  This is required
   !  to input all of the terrestial fields.  This is the only input which does not
   !  have a "parent" and "nest" affiliation, since this comes in as the fine grid
   !  only.

   ALLOCATE ( all_terrain(100) , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for init TERRAIN field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,100
#endif
   END IF

   !  ALLOCATE space for the LOWBDY data from the coarse grid domain.  This is used
   !  to create the LOWBDY file for the fine grid.

   ALLOCATE ( pall_lowbdy(10) , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for init P LOWBDY field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,10
#endif
   END IF
 
   ALLOCATE ( nall_lowbdy(10) , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for init N LOWBDY field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,10
#endif
   END IF

   !  Get NAMELIST information.

   CALL do_namelist_2 ( input_file , input_lowbdy_file , input_terrain_file , &
                        start_year , start_month , start_day , start_hour , &
                        start_minute , start_second , start_frac , &
                        end_year ,   end_month ,   end_day ,   end_hour , &
                        end_minute ,   end_second ,   end_frac , &
                        interval , less_than_24h , &
                        sigma_f_bu , sst_to_ice_threshold , &
                        wrth2o ,  &
                        ifdatim , &
                        interp_method , print_info , use_mm5_lowbdy )

   !  The date info was contained in the namelist.  When to start and
   !  when to end, and interval.

   WRITE ( start_date ,  &
           '(I4.4,"-",I2.2,"-",I2.2,"_",I2.2,":",I2.2,":",I2.2)') &
           start_year , start_month , start_day , start_hour , &
           start_minute , start_second

   current_date = start_date

   WRITE ( end_date ,  &
           '(I4.4,"-",I2.2,"-",I2.2,"_",I2.2,":",I2.2,":",I2.2)') &
           end_year , end_month , end_day , end_hour , &
           end_minute , end_second

   PRINT '(A)','Time periods to process'
   loop_count = 0
   which_times : DO

      loop_count = loop_count + 1
      IF ( loop_count .GT. 1000 ) THEN
         PRINT '(A)','Seems like a lot of loops.'
         STOP 'lots_of_loops_check_dates'
      END IF

      IF ( current_date .EQ. end_date ) THEN
         PRINT '(A,A,A)','Found ending time ',current_date,'.'
         iprocess = loop_count
         EXIT which_times
      ELSE IF ( current_date .LT. end_date ) THEN
            PRINT '(A,A,A)','Found valid time ',current_date,'.'
      ELSE
         iprocess = loop_count -1
         EXIT which_times
      END IF

      CALL geth_newdate ( new_date , current_date , interval )
      current_date = new_date

   END DO which_times

   !  If the lower boundary condition is to be computed from the MM5 fields (use_mm5_lowbdy = .FALSE.), 
   !  then we need at least one full day's worth of processing.  If the lower boundary is to simply
   !  be interpolated from the coarse grid (use_mm5_lowbdy = .TRUE.), then we can do with less than
   !  a full day.  We can also force less than a full day with the less_than_24h flag.

   IF ( ( iprocess * interval .LT. 86400 ) .AND. &
        ( .NOT. less_than_24h  ) .AND. &
        ( .NOT. use_mm5_lowbdy ) ) THEN
      PRINT '(A)','The lower boundary condition needs at least one full day of data.'
      PRINT '(A,I2,A,I5,A,F5.2,A)','You have specified only ',iprocess,' time periods at ',interval,&
                                   ' s. (approximately ',REAL(interval*(iprocess-1))/3600.,' hours).'
      PRINT '(A)','You may set use_mm5_lowbdy = .TRUE. in the namelist.input file.'
      PRINT '(A)','You may also set less_than_24h = .TRUE. in the namelist.input file.'
      STOP 'need_full_day'
   END IF

   !  Compute the number of NEW half and full sigma layers.  If we are going to do any vertical
   !  nesting, we need these values.  Vertical nesting is just an option.  If there is not a listing
   !  of the new vertical distribution, then we keep the old one and only do horizontal interpolating.

   find_sigma_nml : DO k_loop = 1 , 1000
      IF ( sigma_f_bu(k_loop) .LT. -0.5 ) THEN
         kxs_half_new = k_loop -2
         kxs_full_new = k_loop -1
         EXIT find_sigma_nml
      END IF
   END DO find_sigma_nml
!  kxs_half_new = MINLOC(sigma_f_bu,DIM=1,MASK=sigma_f_bu.LT.-0.5) - 2
!  kxs_full_new = kcs_half_new + 1
   
   IF ( kxs_half_new .LT. 1 ) THEN
      PRINT '(A)','Welp, didn''t find any new sigma levels, so we just assume that there is no vertical nesting.'
      vert_nesting = .FALSE.
   ELSE
      PRINT '(A)','Looks like we are doing the vertical nesting option.'
      PRINT '(A)','The new full sigma level distribution will be:'
      DO k_loop = 1 , kxs_full_new
         PRINT '(I4,1X,F7.5)',k_loop , sigma_f_bu(k_loop)
      END DO
      vert_nesting = .TRUE.
   END IF

   !  We know how many sigma levels that we will be interpolating to when we
   !  do the vertical nesting.  But wait, Batfans, do we do it at all?  You
   !  can run this without any new sigma levels, and you end up with the vertical
   !  distribution that you started with.  We are here for you.

   IF ( vert_nesting ) THEN
      ALLOCATE ( sigma_full_new(1:kxs_full_new) ) 
      sigma_full_new(1:kxs_full_new) = sigma_f_bu(kxs_full_new:1:-1)
      ALLOCATE ( sigma_half_new(1:kxs_half_new) ) 
      sigma_half_new = ( sigma_full_new(1:kxs_half_new) + sigma_full_new(2:kxs_full_new) ) * 0.5
   END IF

   !  Justin Case was a careful person.  Justin Case would always make sure the sigma levels
   !  were in the right order, not upside down or with one "oops" included.  Justin Case.

   check_sigma : DO k_loop = 2 , kxs_full_new
      IF ( sigma_f_bu(k_loop) .GE. sigma_f_bu(k_loop-1) ) THEN
         PRINT '(A)','Error in defining full sigma levels in namelist'
         PRINT '(A,I3,A,F6.4,A,I3,A,F6.4)','sigma_f_bu(',k_loop,')=',sigma_f_bu(k_loop), &
                ' is greater than sigma_f_bu(',k_loop-1,')=',sigma_f_bu(k_loop-1)
         PRINT '(A)','Sigma levels must start with 1 and end with 0, and they must vary monotonically'
         STOP 're-order the levels in the namelist.input file, and give it another whirl'
      END IF
   END DO check_sigma

   !  OPEN the input file from the fine grid TERRAIN file.

   CALL get_fg_file ( input_terrain_file , input_terrain ) 

   !  Get the terrain big header data.

   CALL read_bh ( input_terrain )

   !  Save the big header for the TERRAIN data.   Remember, the "n" means "nest"
   !  domain.

   nbhi  = bhi
   nbhr  = bhr
   nbhic = bhic
   nbhrc = bhrc

   nimax  = nbhi(16,1)
   njmax  = nbhi(17,1)
   nesti  = nbhi(18,1)
   nestj  = nbhi(19,1)
   nratio = nbhi(21,1)
   PRINT '(A,2I8)','Fine grid info: IMAX,JMAX  =',nimax,njmax
   PRINT '(A,2I8)','Fine grid info: NESTI,NESTJ=',nesti,nestj

   !  Get the TERRAIN data loaded.

   CALL read_terrain_data ( input_terrain ) 

   !  If we did not ask for the MM5 input lower boundary condition file to be horizontally interpolated, there is
   !  no need to input the file for this program.  We can/will generate the file from the reservoir temperature 
   !  from MM5.

   IF ( use_mm5_lowbdy ) THEN

      !  OPEN the input file from the coarse grid lower boundary condition.

      CALL get_fg_file ( input_lowbdy_file , input_lowbdy )

      !  Get the coarse grid lower boudary condition big header data.

      CALL read_bh ( input_lowbdy )
   
      !  Save the big header for the lower boundary condition data.   Remember, the 
      !  "p" means "parent" domain.
   
      plbhi  = bhi
      plbhr  = bhr
      plbhic = bhic
      plbhrc = bhrc
   
      !  The fine grid lower boundary file is just like the coarse grid, so save
      !  the big header, but just a few modifications.  The important grid information
      !  is in the TERRAIN header (hence the following (:,1) assignments).  Additionally,
      !  the (1,1) location needs to say lower boundary file.  Remember, "n" is the 
      !  "nest" domain.
   
      nlbhi  = plbhi
      nlbhr  = plbhr
      nlbhic = plbhic
      nlbhrc = plbhrc
   
      nlbhi (:,1) = nbhi (:,1)
      nlbhr (:,1) = nbhr (:,1)
      nlbhic(:,1) = nbhic(:,1)
      nlbhrc(:,1) = nbhrc(:,1)
      
      nlbhi (1,1) = plbhi (1,1) 
      nlbhic(1,1) = plbhic(1,1) 
   
      !  Get the coarse grid lower boundary data loaded.
   
      CALL read_lowbdy_data ( input_lowbdy ) 

   ELSE

      !  No input LOWBDY file.

      PRINT '(A)','Not attempting to use the LOWBDY file as input.'

   END IF

   !  OPEN the input file for the MM5 data.

   input_count = 1
   CALL get_fg_file ( input_file(input_count) , input_mm5 ) 

   !  Once we have the sigma level data ready to read, we
   !  can proceed.  We process the data in the big header first.
   !  The big header data comes out of this routine, and is available
   !  via the USE association for header_data.

   CALL read_bh ( input_mm5 )

   !  As an "oh, by the way", if this is not MM5 data that we just ingested, there had better
   !  be a LOWBDY file as input: it is required with INTERPF input.  Let's check that out.

   IF ( ( bhi(1,1) .EQ. 5 ) .AND. ( .NOT. use_mm5_lowbdy ) ) THEN
      PRINT '(A)','This program requires the use of the LOWBDY file if the input data is from INTERPF.'
      PRINT '(A)','Modify the namelist to turn use_mm5_lowbdy = .TRUE., and fill in the LOWBDY filename.'
      STOP 'need_LOWBDY_with_INTERPF'
   END IF

   !  Save the big header for the MM5 data.  Remember, the "p" means "parent"
   !  domain.

   pbhi  = bhi
   pbhr  = bhr
   pbhic = bhic
   pbhrc = bhrc

   IF ( .NOT. use_mm5_lowbdy ) THEN

      !  Save the big header for the lower boundary condition data.  Remember, the 
      !  "p" means "parent" domain.
   
      plbhi  = bhi
      plbhr  = bhr
      plbhic = bhic
      plbhrc = bhrc
   
      !  The fine grid lower boundary file is just like the coarse grid, so save
      !  the big header, but just a few modifications.  The important grid information
      !  is in the TERRAIN header (hence the following (:,1) assignments).  Additionally,
      !  the (1,1) location needs to say lower boundary file.
   
      nlbhi  = plbhi
      nlbhr  = plbhr
      nlbhic = plbhic
      nlbhrc = plbhrc
   
      nlbhi (:,1) = nbhi (:,1)
      nlbhr (:,1) = nbhr (:,1)
      nlbhic(:,1) = nbhic(:,1)
      nlbhrc(:,1) = nbhrc(:,1)
      
      nlbhi (1,1) = plbhi (1,1) 
      nlbhic(1,1) = plbhic(1,1) 

   END IF

   !  Here are some constants that we use often enough to warrant pulling them out of the header.

   imax = pbhi(16,1)
   jmax = pbhi(17,1)
   kxs  = pbhi(12,pbhi(1,1))
   ptop = pbhr(2,2)
   p0   = pbhr(2,5)
   ts0  = pbhr(3,5)
   tlp  = pbhr(4,5)
   tiso = MAX ( pbhr(5,5) , 0. )

   !  We have three input data sets: mm5 output from the coarse grid,
   !  lower boundary from the coarse grid and the terrain data from 
   !  the nest grid.  A few error tests are in order to make sure that
   !  these data sets are consistent.

   CALL compare_bh ( pbhi , pbhr , plbhi , plbhr , nbhi , nbhr , use_mm5_lowbdy )

   !  Some more I/O things.  This sets up the names of the output files
   !  for the INTERP program.  UNIT=20 is the MMINPUT_DOMAINx, UNIT=30
   !  is the BDYOUT_DOMAINx, UNIT=40 is the LOWBDY_DOMAINx file.
   !  The bhi(13,1) is the domain ID, so that we can use a number for
   !  the "x" in DOMAINx.

   CALL open_out_file ( output_mminput , output_bdyout , output_lowerbc , nbhi(13,1) )

   !  ALLOCATE some more arrays that we'll need later.  First, the
   !  vertical coordinate sigma.  This is the top-down version of the
   !  half-sigma layers.  The next one is for the relative humidity to
   !  mixing ratio computation.

   ALLOCATE ( sigma(kxs) , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for SIGMA field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,kxs
#endif
   END IF

   ALLOCATE ( sigma_full_old(kxs+1) , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for SIGMA full field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,kxs+1
#endif
   END IF
 
   ALLOCATE ( nrh ( nimax , njmax , kxs ) , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for N RH field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,nimax , njmax , kxs
#endif
   END IF

   !  These are a collection of coarse grid (parent "p"), fine grid
   !  (nest "n"), vertically nested (vertical "v"), and interpolated from 
   !  coarse to fine ("i") arrays that  are not in the input data.  These 
   !  variables are for the base state computation.

   ALLOCATE ( ips0dot (nimax, njmax)      , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for N IPS0DOT field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,nimax, njmax
#endif
   END IF

   ALLOCATE ( ipt0    (nimax, njmax, kxs) , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for N IPT0 field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,nimax, njmax, kxs
#endif
   END IF

   ALLOCATE ( npr0    (nimax, njmax, kxs) , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for N NPR0 field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,nimax, njmax, kxs
#endif
   END IF

   ALLOCATE ( nps0    (nimax, njmax)      , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for N NPS0 field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,nimax, njmax
#endif
   END IF

   ALLOCATE ( nps0dot (nimax, njmax)      , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for N NPS0DOT field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,nimax, njmax
#endif
   END IF

   ALLOCATE ( nt0     (nimax, njmax, kxs) , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for N NT0 field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,nimax, njmax, kxs
#endif
   END IF

   ALLOCATE ( t0diff  (nimax, njmax, kxs) , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for N T0DIFF field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,nimax, njmax, kxs
#endif
   END IF

   IF ( vert_nesting ) THEN
      ALLOCATE ( vpr0    (nimax, njmax, kxs_half_new) , STAT = return_code )
      IF ( return_code .NE. 0 ) THEN
         PRINT '(A,I1,A)','ALLOCATE return code for V VPR0 field is ',return_code,'.'
         STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
      ELSE
         PRINT *,nimax, njmax, kxs_half_new
#endif
      END IF

      ALLOCATE ( vt0     (nimax, njmax, kxs_half_new) , STAT = return_code )
      IF ( return_code .NE. 0 ) THEN
         PRINT '(A,I1,A)','ALLOCATE return code for V VT0 field is ',return_code,'.'
         STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
      ELSE
         PRINT *,nimax, njmax, kxs_half_new_half_new
#endif
      END IF
   END IF

   ALLOCATE ( ppr0    (imax,  jmax, kxs)  , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for C PPR0 field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,imax,  jmax, kxs
#endif
   END IF

   ALLOCATE ( pps0    (imax,  jmax )      , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for C PPS0 field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,imax,  jmax
#endif
   END IF

   ALLOCATE ( pt0     (imax,  jmax, kxs)  , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for C PT0 field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,imax,  jmax, kxs
#endif
   END IF

   ALLOCATE ( nseaice (nimax, njmax )     , STAT = return_code )
   IF ( return_code .NE. 0 ) THEN
      PRINT '(A,I1,A)','ALLOCATE return code for N SEAICE field is ',return_code,'.'
      STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
   ELSE
      PRINT *,nimax, njmax
#endif
   END IF

   !  Loop over all of the input MM5 data time periods.  Time periods
   !  are chosen that are consistent with those requested from the
   !  NAMELIST.

   itimes = 0
   current_date = start_date
   time_loop : DO
   
      !  Get the sigma-level model-output data loaded for this time period.

      CALL read_data ( input_mm5 , input_file , input_count ) 

      test_date = sh_date

      !  How far apart is the model time from the current requested time?
      !  If we are within a coarse grid time step, this is the correct time.

      CALL geth_idts ( test_date , current_date , time_difference_seconds )

      !  Is this one of the times that we requested?  We are allowing model output
      !  as well as model input, so let's be careful with these tests.

      IF      ( ( pbhi(1,1) .EQ. 11 ) .AND. ABS ( time_difference_seconds ) .LE. 2.*pbhr(2,12) ) THEN
         PRINT '(A,A,A)','Will process time ',test_date,' for model output.'
      ELSE IF ( ( pbhi(1,1) .EQ.  5 ) .AND. ABS ( time_difference_seconds ) .EQ.         0  ) THEN
         PRINT '(A,A,A)','Will process time ',test_date,' for model input.'
      ELSE IF ( test_date .LT. current_date ) THEN
         PRINT '(A,A,A)','Skipping ',test_date,'.  Reading another time.'
print *,'num_3d etc=', num_3d,num_2d , num_1d
         DO loop = 1 , num_3d
            DEALLOCATE ( pall_3d(loop)%array )
         END DO
         DO loop = 1 , num_2d
            DEALLOCATE ( pall_2d(loop)%array )
         END DO
         DO loop = 1 , num_1d
            DEALLOCATE ( pall_1d(loop)%array )
         END DO
         CYCLE time_loop
      ELSE IF ( test_date .GT. current_date ) THEN
         PRINT '(A,A,A)','Seemed to have gone past the right time with ',test_date,'.'
         PRINT '(A)','This happens if you ask for a time period beyond the last valid time in the file.'
         PRINT '(A)','Also, this could happen if you have the incorrect time interval specified.'
         STOP 'past_right_time'
      END IF

      itimes = itimes + 1

      !  If this is the first time in this loop, get the sigma data, the
      !  actual values of the vertical coordinate from the 1d arrays.

      IF ( itimes .EQ. 1 ) THEN
         found = .FALSE.
         find_sigma_input : DO loop = 1 , num_1d
            IF ( pall_1d(loop)%small_header%name(1:8) .EQ. 'SIGMAH  ' ) THEN
               sigma = pall_1d(loop)%array
               found = .TRUE.
               EXIT find_sigma_input
            END IF
         END DO find_sigma_input

         !  Do we now have the sigma levels?

         IF ( .NOT. found ) THEN
            PRINT '(A)','Could not find the sigma level array from MM5.'
            STOP 'no_sigma_array'
         END IF

         !  If we have the half sigma levels, let's get the full ones, too.

         sigma_full_old(1) = 0.
         DO k = 1 , kxs
            sigma_full_old(k+1) = 2.* sigma(k) - sigma_full_old(k)
         END DO
      END IF

      !  Loop over the input "parent" data, and allocate space for the fine
      !  grid "nest" data.  The only difference between the two data sets is
      !  the horizontal dimensions (which are modified).  All of the 3d data 
      !  have lateral boundary space allocated.  Both types of allocations are 
      !  required only for the first time period, since the data remain the 
      !  same horizontal and vertical size as the initial time period.  Even if there
      !  is vertical nesting, we first need the fine-grid (horizontally interpolated)
      !  data on the original sigma levels, so, we always want the "n" variables for
      !  for the 3d data.

      nest_alloc_3 : DO loop = 1 , num_3d
         small_header = pall_3d(loop)%small_header
         small_header%end_dims(1:2) = (/ nimax , njmax /)
         nall_3d(loop)%small_header = small_header
         ALLOCATE (nall_3d(loop)%array(small_header%end_dims(1),small_header%end_dims(2),small_header%end_dims(3)),&
                   STAT = return_code )
         IF ( return_code .NE. 0 ) THEN
            PRINT '(A,A,A,I1,A)','ALLOCATE return code for N 3D field ',small_header%name,' is ',return_code,'.'
            STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
         ELSE
            PRINT *,small_header%end_dims(1),small_header%end_dims(2),small_header%end_dims(3)
#endif
         END IF

         !  We need to allocate space for the vertically nested "v" variables.  This only affects the 3d arrays, and
         !  it affects all of the 3d arrays.  The only thing different between the nested data and the vertically nested
         !  data is the number of vertical levels.  The number of levels is the number of new half levels + the 
         !  difference between the old number of half levels the the current number of levels.  This just picks
         !  up an extra level for us with the vertical velocity.

         IF ( vert_nesting ) THEN
            vall_3d(loop)%small_header = small_header
            numk = kxs_half_new + ( small_header%end_dims(3) - kxs )
            vall_3d(loop)%small_header%end_dims(3) = numk
            ALLOCATE (vall_3d(loop)%array(small_header%end_dims(1),small_header%end_dims(2),numk),&
                      STAT = return_code )
            IF ( return_code .NE. 0 ) THEN
               PRINT '(A,A,A,I1,A)','ALLOCATE return code for V 3D field ',small_header%name,' is ',return_code,'.'
               STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
            ELSE
               PRINT *,small_header%end_dims(1),small_header%end_dims(2),numk
#endif
            END IF
         END IF

         !  The lateral boundary data is allocated for the first time period that is processed.  Only the
         !  3d data is used for lateral boundaries, and all of the 3d data is required for the lateral 
         !  boundaries.  If this is a vertically nested run, the "v" (vertical nesting) arrays are allocated.
         !  If no vertical nesting was requested in the namelist, then the "n" (nested) boundary arrays 
         !  are required.  We never need both of them.

         IF      ( ( itimes .EQ. 1 ) .AND. ( .NOT. vert_nesting ) ) THEN
            ALLOCATE (nall_3d(loop)%ebdy(small_header%end_dims(1),small_header%end_dims(3),nspgd),STAT = return_code )
            IF ( return_code .NE. 0 ) THEN
               PRINT '(A,A,A,I1,A)','ALLOCATE return code for N 3D EBDY field ',small_header%name,' is ',return_code,'.'
               STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
            ELSE
               PRINT *,small_header%end_dims(1),small_header%end_dims(3),nspgd
#endif
            END IF

            ALLOCATE (nall_3d(loop)%wbdy(small_header%end_dims(1),small_header%end_dims(3),nspgd),STAT = return_code )
            IF ( return_code .NE. 0 ) THEN
               PRINT '(A,A,A,I1,A)','ALLOCATE return code for N 3D WBDY field ',small_header%name,' is ',return_code,'.'
               STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
            ELSE
               PRINT *,small_header%end_dims(1),small_header%end_dims(3),nspgd
#endif
            END IF

            ALLOCATE (nall_3d(loop)%nbdy(small_header%end_dims(2),small_header%end_dims(3),nspgd),STAT = return_code )
            IF ( return_code .NE. 0 ) THEN
               PRINT '(A,A,A,I1,A)','ALLOCATE return code for N 3D NBDY field ',small_header%name,' is ',return_code,'.'
               STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
            ELSE
               PRINT *,small_header%end_dims(2),small_header%end_dims(3),nspgd
#endif
            END IF

            ALLOCATE (nall_3d(loop)%sbdy(small_header%end_dims(2),small_header%end_dims(3),nspgd),STAT = return_code )
            IF ( return_code .NE. 0 ) THEN
               PRINT '(A,A,A,I1,A)','ALLOCATE return code for N 3D SBDY field ',small_header%name,' is ',return_code,'.'
               STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
            ELSE
               PRINT *,small_header%end_dims(2),small_header%end_dims(3),nspgd
#endif
            END IF
         ELSE IF ( ( itimes .EQ. 1 ) .AND. ( vert_nesting ) ) THEN
            ALLOCATE (vall_3d(loop)%ebdy(small_header%end_dims(1),numk,nspgd),STAT = return_code )
            IF ( return_code .NE. 0 ) THEN
               PRINT '(A,A,A,I1,A)','ALLOCATE return code for V 3D EBDY field ',small_header%name,' is ',return_code,'.'
               STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
            ELSE
               PRINT *,small_header%end_dims(1),numk,nspgd
#endif
            END IF

            ALLOCATE (vall_3d(loop)%wbdy(small_header%end_dims(1),numk,nspgd),STAT = return_code )
            IF ( return_code .NE. 0 ) THEN
               PRINT '(A,A,A,I1,A)','ALLOCATE return code for V 3D WBDY field ',small_header%name,' is ',return_code,'.'
               STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
            ELSE
               PRINT *,small_header%end_dims(1),numk,nspgd
#endif
            END IF

            ALLOCATE (vall_3d(loop)%nbdy(small_header%end_dims(2),numk,nspgd),STAT = return_code )
            IF ( return_code .NE. 0 ) THEN
               PRINT '(A,A,A,I1,A)','ALLOCATE return code for V 3D NBDY field ',small_header%name,' is ',return_code,'.'
               STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
            ELSE
               PRINT *,small_header%end_dims(2),numk,nspgd
#endif
            END IF

            ALLOCATE (vall_3d(loop)%sbdy(small_header%end_dims(2),numk,nspgd),STAT = return_code )
            IF ( return_code .NE. 0 ) THEN
               PRINT '(A,A,A,I1,A)','ALLOCATE return code for V 3D SBDY field ',small_header%name,' is ',return_code,'.'
               STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
            ELSE
               PRINT *,small_header%end_dims(2),numk,nspgd
#endif
            END IF

         END IF
      END DO nest_alloc_3

      nest_alloc_2 : DO loop = 1 , num_2d
         small_header = pall_2d(loop)%small_header
         small_header%end_dims(1:2) = (/ nimax , njmax /)
         nall_2d(loop)%small_header = small_header
         ALLOCATE (nall_2d(loop)%array(small_header%end_dims(1),small_header%end_dims(2)), STAT = return_code )
         IF ( return_code .NE. 0 ) THEN
            PRINT '(A,A,A,I1,A)','ALLOCATE return code for N 2D field ',small_header%name,' is ',return_code,'.'
            STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
         ELSE
            PRINT *,small_header%end_dims(1),small_header%end_dims(2)
#endif
         END IF

      END DO nest_alloc_2

      nest_alloc_1 : DO loop = 1 , num_1d
         nall_1d(loop)%small_header = pall_1d(loop)%small_header
         ALLOCATE (nall_1d(loop)%array(nall_1d(loop)%small_header%end_dims(1)), STAT = return_code )
         IF ( return_code .NE. 0 ) THEN
            PRINT '(A,A,A,I1,A)','ALLOCATE return code for N 2D field ',small_header%name,' is ',return_code,'.'
            STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
         ELSE
            PRINT *,nall_1d(loop)%small_header%end_dims(1)
#endif
         END IF

      END DO nest_alloc_1

      !  Horizontally interpolate the data from the parent grid to the 
      !  nest grid.  This is performmed for each 3d and 2d array.  There
      !  are necessary cleanup functions that will precede and follow the 
      !  interpolations.  First we do the 3d arrays, they are simpler due to the lack
      !  of searching for duplicate fields in the TERRAIN file, no masking, etc.

      horiz_interp_3 : DO loop = 1 , num_3d

         PRINT '(A,A,A)','... Interpolating ',TRIM(nall_3d(loop)%small_header%name),'.'

         IF      ( nall_3d(loop)%small_header%staggering .EQ. 'D   ' ) THEN
            icrsdot = 0
         ELSE IF ( nall_3d(loop)%small_header%staggering .EQ. 'C   ' ) THEN
            icrsdot = 1
         ELSE
            PRINT '(A)','Weird staggering: ',nall_3d(loop)%small_header%staggering,'.'
            STOP 'wrong_staggering'
         END IF
 
         !  Fill outer row and column for cross point data.

         IF ( icrsdot .EQ. 1 ) THEN
            pall_3d(loop)%array(imax,:,:) = pall_3d(loop)%array(imax-1,:,:)
            pall_3d(loop)%array(:,jmax,:) = pall_3d(loop)%array(:,jmax-1,:)
         END IF
 
         !  Check for small values.

         WHERE ( ABS(pall_3d(loop)%array) .LT. 1.E-10 ) 
            pall_3d(loop)%array = 0
         END WHERE

         !  And just how many k levels do we want to process?  This changes with the
         !  vertical component of velocity.

         local_k = pall_3d(loop)%small_header%end_dims(3)
         
         !  The horizontal interpolation method is chosen from the NAMELIST, unless
         !  the ratio is not 3::1.  If the ratio is other than 3::1, then the BINT
         !  FUNCTION is required.

         IF        ( nratio .NE. 3 ) THEN
            CALL slowint ( pall_3d(loop)%array ,  imax ,  jmax , local_k , &
                           nall_3d(loop)%array , nimax , njmax , nesti , nestj , icrsdot , nratio )
         ELSE IF ( ( nratio .EQ. 3 ) .AND. ( interp_method .EQ. 1 ) ) THEN
            CALL quaint  ( pall_3d(loop)%array ,  imax ,  jmax , local_k , &
                           nall_3d(loop)%array , nimax , njmax , nesti , nestj , icrsdot )
         ELSE IF ( ( nratio .EQ. 3 ) .AND. ( interp_method .EQ. 2 ) ) THEN
            CALL exaint  ( pall_3d(loop)%array ,  imax ,  jmax , local_k , &
                           nall_3d(loop)%array , nimax , njmax , nesti , nestj , icrsdot )
         END IF
 
         !  Fill outer row and column for cross point data.

         IF ( icrsdot .EQ. 1 ) THEN
            nall_3d(loop)%array(nimax,:,:) = nall_3d(loop)%array(nimax-1,:,:)
            nall_3d(loop)%array(:,njmax,:) = nall_3d(loop)%array(:,njmax-1,:)
         END IF
 
         !  Check for small values caused by the interpolation.

         WHERE ( ABS(nall_3d(loop)%array) .LT. 1.E-10 ) 
            nall_3d(loop)%array = 0
         END WHERE
 
         !  Positive definite moisture fields are forced to be .GE. 0.

         IF ( ( nall_3d(loop)%small_header%name(1:8) .EQ. 'Q       ' ) .OR. &
              ( nall_3d(loop)%small_header%name(1:8) .EQ. 'CLW     ' ) .OR. &
              ( nall_3d(loop)%small_header%name(1:8) .EQ. 'RNW     ' ) .OR. &
              ( nall_3d(loop)%small_header%name(1:8) .EQ. 'ICE     ' ) .OR. &
              ( nall_3d(loop)%small_header%name(1:8) .EQ. 'SNOW    ' ) .OR. &
              ( nall_3d(loop)%small_header%name(1:8) .EQ. 'GRAUPEL ' ) .OR. &
              ( nall_3d(loop)%small_header%name(1:8) .EQ. 'NCI     ' ) ) THEN
            WHERE ( nall_3d(loop)%array .LT. 1.e-10 ) 
               nall_3d(loop)%array = 0
            END WHERE
         END IF
      
         !  Some of the 3d fields will need to be used a bit later.

         IF      ( nall_3d(loop)%small_header%name(1:8) .EQ. 'T       ' ) THEN
            index_t        = loop
         ELSE IF ( nall_3d(loop)%small_header%name(1:8) .EQ. 'Q       ' ) THEN
            index_q        = loop
         ELSE IF ( nall_3d(loop)%small_header%name(1:8) .EQ. 'PP      ' ) THEN
            index_pp       = loop
         ELSE IF ( nall_3d(loop)%small_header%name(1:8) .EQ. 'U       ' ) THEN
            index_u        = loop
         ELSE IF ( nall_3d(loop)%small_header%name(1:8) .EQ. 'V       ' ) THEN
            index_v        = loop
         END IF

      END DO horiz_interp_3

      !  Loop over all of the 2d data now for the horizontal interpolation.

      horiz_interp_2 : DO loop = 1 , num_2d
         
         !  Some of the 2d data should come from the TERRAIN fine grid file, not from
         !  an interpolated coarse grid.  For these fields, we "find better data" in the
         !  TERRAIN file (such as elevation, land use, etc.).  For each of the 2d fields
         !  from the input data, we first test to see if the same field exists in the
         !  TERRAIN file.  If it does, we use the info from the TERRAIN program.  If not,
         !  then we perform the horizontal interpolation.

         found = .FALSE.
         find_better_data : DO loop_count = 1 , num_terrain_2d

            IF ( all_terrain(loop_count)%small_header%name(1:8) .EQ. 'TEMPGRD ' ) THEN
               index_tempgrd  = loop_count
            END IF

            IF ( all_terrain(loop_count)%small_header%name .EQ. nall_2d(loop)%small_header%name ) THEN
               found = .TRUE.
               EXIT find_better_data
            END IF

         END DO find_better_data

         IF ( found ) THEN
            PRINT '(A,A,A)','... From TERRAIN  ',TRIM(nall_2d(loop)%small_header%name),'.'
            nall_2d(loop)%array = all_terrain(loop_count)%array

         ELSE
            PRINT '(A,A,A)','... Interpolating ',TRIM(nall_2d(loop)%small_header%name),'.'

            IF      ( nall_2d(loop)%small_header%staggering .EQ. 'D   ' ) THEN
               icrsdot = 0
            ELSE IF ( nall_2d(loop)%small_header%staggering .EQ. 'C   ' ) THEN
               icrsdot = 1
            ELSE
               PRINT '(A)','Weird staggering: ',nall_2d(loop)%small_header%staggering,'.'
               STOP 'wrong_staggering'
            END IF

            !  A bit of clean up for the parent data.

            IF ( icrsdot .EQ. 1 ) THEN
               pall_2d(loop)%array(imax,:) = pall_2d(loop)%array(imax-1,:)
               pall_2d(loop)%array(:,jmax) = pall_2d(loop)%array(:,jmax-1)
            END IF

            !  Another clean up tip: if the number is really small, it is probably some
            !  precipitation or moisture variable.  Set it to zero to avoid interpolation
            !  troubles with really tiny values.

            WHERE ( ABS(pall_2d(loop)%array) .LT. 1.E-10 ) 
               pall_2d(loop)%array = 0
            END WHERE

            !  The horizontal interpolation method is chosen from the NAMELIST, unless
            !  the ratio is not 3::1.  If the ratio is other than 3::1, then the BINT
            !  FUNCTION is required.
   
            IF        ( nratio .NE. 3 ) THEN
               CALL slowint ( pall_2d(loop)%array ,  imax ,  jmax ,   1 , &
                              nall_2d(loop)%array , nimax , njmax , nesti , nestj , icrsdot , nratio )
            ELSE IF ( ( nratio .EQ. 3 ) .AND. ( interp_method .EQ. 1 ) ) THEN
               CALL quaint  ( pall_2d(loop)%array ,  imax ,  jmax ,   1 , &
                              nall_2d(loop)%array , nimax , njmax , nesti , nestj , icrsdot )
            ELSE IF ( ( nratio .EQ. 3 ) .AND. ( interp_method .EQ. 2 ) ) THEN
               CALL exaint  ( pall_2d(loop)%array ,  imax ,  jmax ,   1 , &
                              nall_2d(loop)%array , nimax , njmax , nesti , nestj , icrsdot )
            END IF

         END IF

         !  Set the outer row and column of the cross point variables to the value of the
         !  element in the row and column next door.

         IF ( icrsdot .EQ. 1 ) THEN
            nall_2d(loop)%array(nimax,:) = nall_2d(loop)%array(nimax-1,:)
            nall_2d(loop)%array(:,njmax) = nall_2d(loop)%array(:,njmax-1)
         END IF

         !  Check for small values in the 2d data caused by the interpolation.

         WHERE ( ABS(nall_2d(loop)%array) .LT. 1.E-10 ) 
            nall_2d(loop)%array = 0
         END WHERE
      
         !  Some of the 2d fields will need to be used a bit later.

         IF      ( nall_2d(loop)%small_header%name(1:8) .EQ. 'TERRAIN ' ) THEN
            index_terrain  = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'PSTARCRS' ) THEN
            index_pstarcrs = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SNOWCOVR' ) THEN
            index_snowcovr = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'TSEASFC ' ) THEN
            index_tseasfc  = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'GROUND T' ) THEN
            index_groundt  = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'LAND USE' ) THEN
            index_landuse  = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SEAICE  ' ) THEN
            index_seaice   = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'LATITCRS' ) THEN
            index_latitude = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'RES TEMP' ) THEN
            index_restemp  = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOILT400' ) THEN
            index_soilt400 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOILT200' ) THEN
            index_soilt200 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOILT100' ) THEN
            index_soilt100 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOILT040' ) THEN
            index_soilt040 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOILT010' ) THEN
            index_soilt010 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOIL T 1' ) THEN
            index_soilt1   = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOIL T 2' ) THEN
            index_soilt2   = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOIL T 3' ) THEN
            index_soilt3   = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOIL T 4' ) THEN
            index_soilt4   = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOIL T 5' ) THEN
            index_soilt5   = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOIL T 6' ) THEN
            index_soilt6   = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOILM400' ) THEN
            index_soilm400 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOILM200' ) THEN
            index_soilm200 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOILM100' ) THEN
            index_soilm100 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOILM040' ) THEN
            index_soilm040 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOILM010' ) THEN
            index_soilm010 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOIL M 1' ) THEN
            index_soilm1   = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOIL M 2' ) THEN
            index_soilm2   = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOIL M 3' ) THEN
            index_soilm3   = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOIL M 4' ) THEN
            index_soilm4   = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOIL M 5' ) THEN
            index_soilm5   = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SOIL M 6' ) THEN
            index_soilm6   = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'MONALB01' ) THEN
            index_monalb01 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'MONALB02' ) THEN
            index_monalb02 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'MONALB03' ) THEN
            index_monalb03 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'MONALB04' ) THEN
            index_monalb04 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'MONALB05' ) THEN
            index_monalb05 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'MONALB06' ) THEN
            index_monalb06 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'MONALB07' ) THEN
            index_monalb07 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'MONALB08' ) THEN
            index_monalb08 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'MONALB09' ) THEN
            index_monalb09 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'MONALB10' ) THEN
            index_monalb10 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'MONALB11' ) THEN
            index_monalb11 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'MONALB12' ) THEN
            index_monalb12 = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'ALBSNOMX' ) THEN
            index_albsnomx = loop
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'ALBEDO  ' ) THEN
            index_albedo   = loop
         END IF

      END DO horiz_interp_2

      !  We have already done this, but let's do it again.  Interpolate the coarse grid 
      !  sea ice to the fine grid locations using water masking.  This has to be a REAL
      !  sea ice, not one that we just invented (and set to zero).

      mask_interp_1 : DO loop = 1 , num_2d

         IF      ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SEAICE  ' ) THEN
            CALL mask_water ( pall_2d(loop)%array ,  imax ,  jmax , &
                              nall_2d(loop)%array , nimax , njmax , nesti , nestj , icrsdot , nratio , &
                              pall_2d(index_landuse)%array , nall_2d(index_landuse)%array , &
                              nall_2d(index_latitude)%array , &
                              nbhi(23,1) , nall_2d(loop)%small_header%name(1:8) , .TRUE., 0. , .FALSE. )
         ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SEAICEFR' ) THEN
            CALL mask_water ( pall_2d(loop)%array ,  imax ,  jmax , &
                              nall_2d(loop)%array , nimax , njmax , nesti , nestj , icrsdot , nratio , &
                              pall_2d(index_landuse)%array , nall_2d(index_landuse)%array , &
                              nall_2d(index_latitude)%array , &
                              nbhi(23,1) , nall_2d(loop)%small_header%name(1:8) , .TRUE., 0. , .FALSE. )
!        ELSE IF ( nall_2d(loop)%small_header%name(1:6) .EQ. 'ALBEDO'   ) THEN
!           CALL mask_water ( pall_2d(loop)%array ,  imax ,  jmax , &
!                             nall_2d(loop)%array , nimax , njmax , nesti , nestj , icrsdot , nratio , &
!                             pall_2d(index_landuse)%array , nall_2d(index_landuse)%array , &
!                             nall_2d(index_latitude)%array , &
!                             nbhi(23,1) , nall_2d(loop)%small_header%name(1:8) , .TRUE., 16. , .FALSE. )
!        ELSE IF ( nall_2d(loop)%small_header%name(1:6) .EQ. 'MONALB'   ) THEN
!           CALL mask_water ( pall_2d(loop)%array ,  imax ,  jmax , &
!                             nall_2d(loop)%array , nimax , njmax , nesti , nestj , icrsdot , nratio , &
!                             pall_2d(index_landuse)%array , nall_2d(index_landuse)%array , &
!                             nall_2d(index_latitude)%array , &
!                             nbhi(23,1) , nall_2d(loop)%small_header%name(1:8) , .TRUE., 16. , .FALSE. )
!        ELSE IF ( nall_2d(loop)%small_header%name(1:8) .EQ. 'ALBSNOMX' ) THEN
!           CALL mask_water ( pall_2d(loop)%array ,  imax ,  jmax , &
!                             nall_2d(loop)%array , nimax , njmax , nesti , nestj , icrsdot , nratio , &
!                             pall_2d(index_landuse)%array , nall_2d(index_landuse)%array , &
!                             nall_2d(index_latitude)%array , &
!                             nbhi(23,1) , nall_2d(loop)%small_header%name(1:8) , .TRUE., 60. , .FALSE. )
         END IF
     
      END DO mask_interp_1

      !  Did we have a sea ice field?  We want to use it later in the modified interpolation
      !  for the ground temperature.  Under the sea ice, we have a soil temperature, for
      !  example.  If we don't have a sea ice field as input, this is probably not a big
      !  deal: most of the time you get the soil temperatures and soil moisture from the
      !  same file that provides the sea ice.

      IF ( index_seaice .NE. 0 ) THEN
         nseaice = nall_2d(index_seaice)%array
      ELSE
         nseaice = 0
      END IF

      !  We have interpolated all of the 2d fields to the grid.  If the SST is less than
      !  the prescribed value in the namelist, sst_to_ice_threshold, turn the land use
      !  category to ice.  There are three IF tests, one for each type of available soil 
      !  input source.

      IF      ( nbhic(23,1)(1:4) .EQ. 'OLD ' ) THEN
         WHERE ( ( nall_2d(index_landuse)%array .EQ. nbhi(23,1)           ) .AND. &
                 ( nall_2d(index_tseasfc)%array .LT. sst_to_ice_threshold ) )
            nall_2d(index_landuse)%array = 11
         END WHERE
      ELSE IF ( nbhic(23,1)(1:4) .EQ. 'USGS' ) THEN
         WHERE ( ( nall_2d(index_landuse)%array .EQ. nbhi(23,1)           ) .AND. &
                 ( nall_2d(index_tseasfc)%array .LT. sst_to_ice_threshold ) )
            nall_2d(index_landuse)%array = 24
         END WHERE
      ELSE IF ( nbhic(23,1)(1:4) .EQ. 'SiB ' ) THEN
         WHERE ( ( nall_2d(index_landuse)%array .EQ. nbhi(23,1)           ) .AND. &
                 ( nall_2d(index_tseasfc)%array .LT. sst_to_ice_threshold ) )
            nall_2d(index_landuse)%array = 16
         END WHERE
      END IF

      !  These 2d snow fields are simply set to 0 over water, if there is no sea ice at the grid point.

      snow_interp : DO loop = 1 , num_2d

         IF ( ( nall_2d(loop)%small_header%name(1:5) .EQ. 'WEASD'    ) .OR. &
              ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SNODPTH ' ) .OR. &
              ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SNOWH   ' ) ) THEN
            WHERE ( ( nall_2d(index_landuse)%array .EQ. bhi(23,1) ) .AND. ( nseaice .EQ. 0 ) ) 
               nall_2d(loop)%array = 0
            END WHERE
            WHERE ( nall_2d(loop)%array .LT. 0 ) 
               nall_2d(loop)%array = 0
            END WHERE
         END IF
      END DO snow_interp

      !  A few fields are between 0 and 1.  Make sure their interpolated values respect those 
      !  imposed limits.

      bounded_check : DO loop = 1 , num_2d

         IF ( ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SNOWCOVR' ) .OR. &
              ( nall_2d(loop)%small_header%name(1:8) .EQ. 'SEAICEFR' ) ) THEN
            WHERE ( nall_2d(loop)%array .LT. 0. ) 
               nall_2d(loop)%array = 0.
            END WHERE
            WHERE ( nall_2d(loop)%array .GT. 1. ) 
               nall_2d(loop)%array = 1.
            END WHERE
         END IF
      END DO bounded_check

      !  Set all of the albedo fields to a water value.  Then, when we mask it to only
      !  use the land values, the water values will be ignored, and will have been previously
      !  set to our requested 8%.

      mask_interp_1b : DO loop = 1 , num_2d
         IF ( ( nall_2d(loop)%small_header%name(1:6) .EQ. 'MONALB'   ) .OR. &
              ( nall_2d(loop)%small_header%name(1:8) .EQ. 'ALBSNOMX' ) .OR. &
              ( nall_2d(loop)%small_header%name(1:6) .EQ. 'ALBEDO'   ) ) THEN
            nall_2d(loop)%array = 8.
         END IF
      END DO mask_interp_1b

      !  Canopy moisture is zero everywhere, then the interpolation handles the values
      !  that it can.

      mask_interp_1c : DO loop = 1 , num_2d
         IF ( ( nall_2d(loop)%small_header%name(1:7) .EQ. 'CANOPYM'  ) ) THEN
            nall_2d(loop)%array = 0.
         END IF
      END DO mask_interp_1c

      !  All of these 2d variables will be interpolated again, this time using some masking
      !  assumptions.  

      mask_interp_2 : DO loop = 1 , num_2d

         IF ( ( nall_2d(loop)%small_header%name(1:5) .EQ. 'SOILT'    ) .OR. &
              ( nall_2d(loop)%small_header%name(1:5) .EQ. 'SOILM'    ) .OR. &
              ( nall_2d(loop)%small_header%name(1:5) .EQ. 'SOILW'    ) .OR. &
              ( nall_2d(loop)%small_header%name(1:6) .EQ. 'SOIL T'   ) .OR. &
              ( nall_2d(loop)%small_header%name(1:6) .EQ. 'SOIL M'   ) .OR. &
              ( nall_2d(loop)%small_header%name(1:6) .EQ. 'SOIL W'   ) .OR. & 
              ( nall_2d(loop)%small_header%name(1:8) .EQ. 'GROUND T' ) .OR. &
              ( nall_2d(loop)%small_header%name(1:8) .EQ. 'RES TEMP' ) .OR. &
              ( nall_2d(loop)%small_header%name(1:6) .EQ. 'MONALB'   ) .OR. &
              ( nall_2d(loop)%small_header%name(1:8) .EQ. 'ALBSNOMX' ) .OR. &
              ( nall_2d(loop)%small_header%name(1:6) .EQ. 'ALBEDO'   ) .OR. &
              ( nall_2d(loop)%small_header%name(1:7) .EQ. 'CANOPYM'  ) ) THEN
 
            !  Do a masked interpolation over the land points for the fields selected in the
            !  big IF test.

            CALL mask_land ( pall_2d(loop)%array ,  imax ,  jmax , &
                             nall_2d(loop)%array , nimax , njmax , nesti , nestj , icrsdot , nratio , &
                             pall_2d(index_landuse)%array , nall_2d(index_landuse)%array , &
                             nall_2d(index_latitude)%array , bhi(23,1) , &
                             nall_2d(loop)%small_header%name(1:8) , .FALSE., 0. , .FALSE. )

            !  Set the various temperatures to a combination of the current 
            !  temperature (from horizontal interpolation using land masking) and
            !  SST (from horizontal interpolation).  That means we have these masked
            !  temperature fields even over the water.  They are never used over the
            !  water, but you can at least plot the darn things now.

            IF ( ( ( nall_2d(loop)%small_header%name(1:5) .EQ. 'SOILT'    )    .OR. &
                   ( nall_2d(loop)%small_header%name(1:6) .EQ. 'SOIL T'   )    .OR. &
                   ( nall_2d(loop)%small_header%name(1:8) .EQ. 'RES TEMP' )    .OR. &
                   ( nall_2d(loop)%small_header%name(1:8) .EQ. 'GROUND T' ) ) .AND. &
                 ( index_tseasfc .NE. 0 ) .AND. ( index_landuse .NE. 0 ) )  THEN
               WHERE ( NINT(nall_2d(index_landuse)%array) .EQ. bhi(23,1) )
                  nall_2d(loop)%array = nall_2d(index_tseasfc)%array
               END WHERE
            END IF

            !  Soil moisture over water is defined as 1.

            IF ( ( nall_2d(loop)%small_header%name(1:5) .EQ. 'SOILM'    ) .OR. &
                 ( nall_2d(loop)%small_header%name(1:5) .EQ. 'SOILW'    ) .OR. &
                 ( nall_2d(loop)%small_header%name(1:6) .EQ. 'SOIL M'   ) .OR. &
                 ( nall_2d(loop)%small_header%name(1:6) .EQ. 'SOIL W'   ) ) THEN
               WHERE ( nall_2d(index_landuse)%array .EQ. nbhi(23,1) ) 
                  nall_2d(loop)%array = 1.
               END WHERE
            END IF

         END IF

      END DO mask_interp_2

      !  All of these 2d variables will be interpolated again, this time using some masking
      !  assumptions.   These are the fields without "reasonable" defaults.

      mask_interp_3 : DO loop = 1 , num_2d

         IF    ( ( nall_2d(loop)%small_header%name(1:7) .EQ. 'SOILHGT'  ) .OR. &
                 ( nall_2d(loop)%small_header%name(5:8) .EQ.     'NOFF' ) .OR. & 
                 ( nall_2d(loop)%small_header%name(1:5) .EQ. 'SNOWH'    ) ) THEN
 
            !  Do a masked interpolation over the land points for the fields selected in the
            !  big IF test.

            nall_2d(loop)%array  = 0.
            CALL mask_land ( pall_2d(loop)%array ,  imax ,  jmax , &
                             nall_2d(loop)%array , nimax , njmax , nesti , nestj , icrsdot , nratio , &
                             pall_2d(index_landuse)%array , nall_2d(index_landuse)%array , &
                             nall_2d(index_latitude)%array , bhi(23,1) , &
                             nall_2d(loop)%small_header%name(1:8) , .TRUE., 0. , .FALSE. )

         END IF

      END DO mask_interp_3

      !  Make an interpolated pstar array on dot points for later use.

      ips0dot = nall_2d(index_pstarcrs)%array
      CALL crs2dot (ips0dot, nimax, njmax)

      !  Ensure snow cover array meets acceptable criteria.  If there is snow
      !  at this grid point, it is = 1.; else if there is no snow, the grid point
      !  value is -0.01.  This only applies to non-LSM MM5 output.

      IF ( ( pbhi(5,13) .EQ. 0 ) .OR. ( pbhi(5,13) .EQ. 1 ) ) THEN
         IF ( index_snowcovr .NE. 0 ) THEN
            WHERE ( nall_2d(index_snowcovr)%array .GE. 0.5)
               nall_2d(index_snowcovr)%array =  1.0
            ELSEWHERE
               nall_2d(index_snowcovr)%array = -0.01
            END WHERE
         END IF
      END IF

      !  The next step required is to compute the nonhydrostatic base
      !  state atmosphere for both the parent and the nest.  We then
      !  interpolate the parent's reference state temperature array to
      !  the nest, determine the difference in base states, and use the
      !  difference to correct temperature, ground temperature, and rh.
      !  By definition, the nonhydrostatic base state is independent of
      !  time; it is a function only of p0, ptop, ts0, tlp, tiso and terrain elevation.
      !  We only need to compute this the first time through the time_loop, but
      !  the perturbation is applied to all of the time levels.

      IF ( itimes .EQ. 1 ) THEN

         !  Compute the non-hydrostatic base state for the parent grid.

         pall_2d(index_terrain)%array(imax,:) = pall_2d(index_terrain)%array(imax-1,:)
         pall_2d(index_terrain)%array(:,jmax) = pall_2d(index_terrain)%array(:,jmax-1)
         CALL nhbase ( pall_2d(index_terrain)%array,  imax,  jmax, kxs, pps0, ppr0, pt0, ptop, p0, tlp, ts0, tiso, sigma )

         !  Compute the non-hydrostatic base state for the nest grid and
         !  create nonhydrostatic pstar on dot points for later use.  This
         !  pstar is not interpolated from the coarse data but comes directly
         !  from the fine grid terrain elevation.

         nall_2d(index_terrain)%array(nimax,:) = nall_2d(index_terrain)%array(nimax-1,:)
         nall_2d(index_terrain)%array(:,njmax) = nall_2d(index_terrain)%array(:,njmax-1)
         CALL nhbase ( nall_2d(index_terrain)%array, nimax, njmax, kxs, nps0, npr0, nt0, &
                       ptop, p0, tlp, ts0, tiso, sigma )

         !  Is there to be any, shall we say, vertical nesting?  Well, by all means, if so then
         !  we should produce a proper base state, eh?
 
         IF ( vert_nesting ) THEN
            CALL nhbase ( nall_2d(index_terrain)%array, nimax, njmax, kxs_half_new, nps0, vpr0, vt0, &
                          ptop, p0, tlp, ts0, tiso, sigma_half_new )
         END IF
         nps0(nimax,:) = nps0(nimax-1,:)
         nps0(:,njmax) = nps0(:,njmax-1)

         nps0dot = nps0
         CALL crs2dot (nps0dot, nimax, njmax)

         !  Interpolate parent reference state temperature to the nest.

         IF        ( nratio .NE. 3 ) THEN
            CALL slowint (pt0, imax, jmax, kxs, ipt0, nimax, njmax, nesti, nestj, 1, nratio)
         ELSE IF ( ( nratio .EQ. 3 ) .AND. ( interp_method .EQ. 1 ) ) THEN
            CALL quaint  (pt0, imax, jmax, kxs, ipt0, nimax, njmax, nesti, nestj, 1)
         ELSE IF ( ( nratio .EQ. 3 ) .AND. ( interp_method .EQ. 2 ) ) THEN
            CALL exaint  (pt0, imax, jmax, kxs, ipt0, nimax, njmax, nesti, nestj, 1)
         END IF

         ipt0(nimax,:,:) = ipt0(nimax-1,:,:)
         ipt0(:,njmax,:) = ipt0(:,njmax-1,:)

         !  Compute the difference in the nest base state temperature and
         !  base state temperature interpolated above.

         t0diff = nt0 - ipt0

      END IF

      !  Convert q to rh, then keeping RH constant, use the new temperature
      !  on the fine grid and recompute the q on the fine grid.  Use the 
      !  interpolated pstar from q -> RH, and the "correct" pstar for the
      !  RH -> q computation.

      CALL qtorh (nall_3d(index_q)%array, nall_3d(index_t)%array, nall_3d(index_pp)%array, &
                  sigma, nall_2d(index_pstarcrs)%array, ptop, nimax, njmax, kxs, wrth2o, nrh)

      !  Adjust the 3d temperature.

      nall_3d(index_t)%array = nall_3d(index_t)%array + t0diff

      !  Use adjusted temperature array and the correct pstar for the fine grid to compute 
      !  an adjusted mixing ratio.

      CALL rhtoq (nrh, nall_3d(index_t)%array, nall_3d(index_pp)%array, &
                  sigma, nps0, ptop, nimax, njmax, kxs, wrth2o, nall_3d(index_q)%array)

      !  Apply the difference to adjust the 2d temperature fields: ground temperature, 
      !  reservoir temperature, soil temperatures.

      nall_2d(index_groundt)%array = nall_2d(index_groundt)%array + t0diff(:,:,kxs)

      IF ( index_restemp  .NE. 0 ) THEN
         nall_2d(index_restemp)%array  = nall_2d(index_restemp)%array  + t0diff(:,:,kxs)
      END IF

      IF ( index_soilt400 .NE. 0 ) THEN
         nall_2d(index_soilt400)%array = nall_2d(index_soilt400)%array + t0diff(:,:,kxs)
      END IF

      IF ( index_soilt200 .NE. 0 ) THEN
         nall_2d(index_soilt200)%array = nall_2d(index_soilt200)%array + t0diff(:,:,kxs)
      END IF

      IF ( index_soilt100 .NE. 0 ) THEN
         nall_2d(index_soilt100)%array = nall_2d(index_soilt100)%array + t0diff(:,:,kxs)
      END IF

      IF ( index_soilt040 .NE. 0 ) THEN
         nall_2d(index_soilt040)%array = nall_2d(index_soilt040)%array + t0diff(:,:,kxs)
      END IF

      IF ( index_soilt010 .NE. 0 ) THEN
         nall_2d(index_soilt010)%array = nall_2d(index_soilt010)%array + t0diff(:,:,kxs)
      END IF

      IF ( index_soilt1   .NE. 0 ) THEN
         nall_2d(index_soilt1)%array   = nall_2d(index_soilt1)%array   + t0diff(:,:,kxs)
      END IF

      IF ( index_soilt2   .NE. 0 ) THEN
         nall_2d(index_soilt2)%array   = nall_2d(index_soilt2)%array   + t0diff(:,:,kxs)
      END IF

      IF ( index_soilt3   .NE. 0 ) THEN
         nall_2d(index_soilt3)%array   = nall_2d(index_soilt3)%array   + t0diff(:,:,kxs)
      END IF

      IF ( index_soilt4   .NE. 0 ) THEN
         nall_2d(index_soilt4)%array   = nall_2d(index_soilt4)%array   + t0diff(:,:,kxs)
      END IF

      IF ( index_soilt5   .NE. 0 ) THEN
         nall_2d(index_soilt5)%array   = nall_2d(index_soilt5)%array   + t0diff(:,:,kxs)
      END IF

      IF ( index_soilt6   .NE. 0 ) THEN
         nall_2d(index_soilt6)%array   = nall_2d(index_soilt6)%array   + t0diff(:,:,kxs)
      END IF

      !  Need to adjust wind components to new base state/terrain height.  NOTE: This has been
      !  discontinued as a recommended option.

!     CALL dfdp ( nall_3d(index_u)%array, ips0dot, nps0dot, ptop, sigma, nimax, njmax, kxs )

!     CALL dfdp ( nall_3d(index_v)%array, ips0dot, nps0dot, ptop, sigma, nimax, njmax, kxs )

      !  Put the non-interpolated value of pstar in the nested location for use by the output
      !  routines.

      nall_2d(index_pstarcrs)%array = nps0 
  
      !  Let's check on doing some vertical interpolation.

      IF ( vert_nesting ) THEN

         vert_interp3 : DO loop = 1 , num_3d

            PRINT '(A,A,A)','Vertically Interpolating ',TRIM(nall_3d(loop)%small_header%name),'.'

            IF      ( vall_3d(loop)%small_header%staggering .EQ. 'D   ' ) THEN
               icrsdot = 0
            ELSE IF ( vall_3d(loop)%small_header%staggering .EQ. 'C   ' ) THEN
               icrsdot = 1
            ELSE
               PRINT '(A)','Weird staggering: ',vall_3d(loop)%small_header%staggering,'.'
               STOP 'wrong_staggering'
            END IF

            kx = vall_3d(loop)%small_header%end_dims(3) 

            IF      ( ( kx .EQ. kxs_half_new ) .AND. ( icrsdot .EQ. 0 ) ) THEN
               CALL vinterp ( vall_3d(loop)%array , sigma_half_new , kxs_half_new , &
                              nall_3d(loop)%array , sigma          , kxs          , &
                              nimax , njmax , icrsdot , nps0dot , &
                              ptop , ts0 , p0 , tlp , tiso )
            ELSE IF ( ( kx .EQ. kxs_half_new ) .AND. ( icrsdot .EQ. 1 ) ) THEN
               CALL vinterp ( vall_3d(loop)%array , sigma_half_new , kxs_half_new , &
                              nall_3d(loop)%array , sigma          , kxs          , &
                              nimax , njmax , icrsdot , nps0    , &
                              ptop , ts0 , p0 , tlp , tiso )
            ELSE IF ( ( kx .EQ. kxs_full_new ) .AND. ( icrsdot .EQ. 0 ) ) THEN
               CALL vinterp ( vall_3d(loop)%array , sigma_full_new , kxs_full_new , &
                              nall_3d(loop)%array , sigma_full_old , kxs+1        , &
                              nimax , njmax , icrsdot , nps0dot , &
                              ptop , ts0 , p0 , tlp , tiso )
            ELSE IF ( ( kx .EQ. kxs_full_new ) .AND. ( icrsdot .EQ. 1 ) ) THEN
               CALL vinterp ( vall_3d(loop)%array , sigma_full_new , kxs_full_new , &
                              nall_3d(loop)%array , sigma_full_old , kxs+1        , &
                              nimax , njmax , icrsdot , nps0    , &
                              ptop , ts0 , p0 , tlp , tiso )
            END IF

         END DO vert_interp3

      END IF

      !  We're done for this time period.  Output the model data if this is one of the time
      !  periods requested for output.  If IFDATIM .GT. 1, the user will probably want to
      !  do analysis FDDA.  If this is the last time period to process, close the model
      !  input file.

      IF ( ( itimes .LE. ifdatim ) .OR. ( ifdatim .EQ. -1 ) ) THEN

         IF ( vert_nesting ) THEN
            CALL outmodelv (current_date,nimax,njmax,kxs_half_new,output_mminput,itimes,interval, &
                            start_year, start_month, start_day, start_hour, &
                            start_minute, start_second, start_frac, &
                            p0, tlp, ts0, tiso, &
                            sigma_half_new, vert_nesting , &
                            pbhi  , pbhic  , pbhr  , pbhrc  , &
                            nbhi  , nbhic  , nbhr  , nbhrc  )
         ELSE
            CALL outmodelv (current_date,nimax,njmax,kxs,output_mminput,itimes,interval, &
                            start_year, start_month, start_day, start_hour, &
                            start_minute, start_second, start_frac, &
                            p0, tlp, ts0, tiso, &
                            sigma, vert_nesting , &
                            pbhi  , pbhic  , pbhr  , pbhrc  , &
                            nbhi  , nbhic  , nbhr  , nbhrc  )
         END IF
   
         PRINT '(A,A,A,I1,A)','Initial conditions written for ',current_date, &
                               ' on MMINPUT_DOMAIN',nbhi(13,1),'.'

         IF      ( ( ifdatim .NE. -1 ) .AND. ( itimes .EQ. MIN ( ifdatim , iprocess ) ) ) THEN
            CLOSE ( output_mminput )
         ELSE IF ( ( ifdatim .EQ. -1 ) .AND. ( itimes .EQ. iprocess ) ) THEN
            CLOSE ( output_mminput )
         END IF

      END IF

      !  There are two choices for the lower boundary condition file (the file 
      !  that holds the model substrate temperature and the mean SST).  Either the 
      !  user can have the original lower boundary from the model interpolated to the 
      !  fine grid domain, or the user can have the lowest sigma-level temperature
      !  data summed to create the file.  If we are choosing to use the original
      !  LOWBDY file, we only need to interpolate the data and and write it at the
      !  initial time.

      IF      ( ( use_mm5_lowbdy ) .AND. ( itimes .EQ. 1 ) ) THEN
         DO loop = 1 , num_lowbdy_2d
            
            !  Fix the small header for each of the fields in the lower boundary condition
            !  file.  The headers should be identical to the parental version, except for
            !  the date and the horizontal size.

            nall_lowbdy(loop)%small_header = pall_lowbdy(loop)%small_header
            nall_lowbdy(loop)%small_header%end_dims = (/ nimax , njmax , 1 , 1 /)
            nall_lowbdy(loop)%small_header%current_date = start_date // '.0000'

            !  The space for the fine grid lower boundary "array" data has to be allocated.

            ALLOCATE ( nall_lowbdy(loop)%array(nimax,njmax) , STAT = return_code )
            IF ( return_code .NE. 0 ) THEN
               PRINT '(A,A,I1,A)','ALLOCATE return code for N LOWBDY field ',nall_lowbdy(loop)%small_header%name,&
                                  ' is ',return_code,'.'
               STOP 'allocate_error'
#ifdef DEBUG_ALLOCATE
            ELSE
               PRINT *,nimax,njmax
#endif
            END IF

            !  Some of the fields in the LOWBDY file need to be masked on the interpolation.

            IF      ( nall_lowbdy(loop)%small_header%name(1:8) .EQ. 'SEAICE  ' ) THEN
               nall_lowbdy(loop)%array = 0.
               CALL mask_water ( pall_lowbdy(loop)%array ,  imax ,  jmax , &
                                 nall_lowbdy(loop)%array , nimax , njmax , nesti , nestj , icrsdot , nratio , &
                                 pall_2d(index_landuse)%array , nall_2d(index_landuse)%array , &
                                 nall_2d(index_latitude)%array , &
                                 nbhi(23,1) , nall_lowbdy(loop)%small_header%name(1:8) , .TRUE., 0. , .FALSE. )
            ELSE IF ( nall_lowbdy(loop)%small_header%name(1:8) .EQ. 'SEAICEFR' ) THEN
               nall_lowbdy(loop)%array = 0.
               CALL mask_water ( pall_lowbdy(loop)%array ,  imax ,  jmax , &
                                 nall_lowbdy(loop)%array , nimax , njmax , nesti , nestj , icrsdot , nratio , &
                                 pall_2d(index_landuse)%array , nall_2d(index_landuse)%array , &
                                 nall_2d(index_latitude)%array , &
                                 nbhi(23,1) , nall_lowbdy(loop)%small_header%name(1:8) , .TRUE., 0. , .FALSE. )

            ELSE

               !  Since this is coming directly from the parent grid, we have to horizontally interpolate
               !  the data.  The choice of techniques is consistent with the previous interpolations.
   
               IF        ( nratio .NE. 3 ) THEN
                  CALL slowint (pall_lowbdy(loop)%array, imax, jmax,   1, &
                                nall_lowbdy(loop)%array, nimax, njmax, nesti, nestj, 1, nratio)
               ELSE IF ( ( nratio .EQ. 3 ) .AND. ( interp_method .EQ. 1 ) ) THEN
                  CALL quaint  (pall_lowbdy(loop)%array, imax, jmax,   1, &
                                nall_lowbdy(loop)%array, nimax, njmax, nesti, nestj, 1)
               ELSE IF ( ( nratio .EQ. 3 ) .AND. ( interp_method .EQ. 2 ) ) THEN
                  CALL exaint  (pall_lowbdy(loop)%array, imax, jmax,   1, &
                                nall_lowbdy(loop)%array, nimax, njmax, nesti, nestj, 1)
               END IF

            END IF

            !  A few fields are between 0 and 1.  Make sure their interpolated values respect those 
            !  imposed limits.  The SEAICE field is just a 0/1 flag.
      
            IF ( ( nall_lowbdy(loop)%small_header%name(1:8) .EQ. 'SNOWCOVR' ) .OR. &
                 ( nall_lowbdy(loop)%small_header%name(1:8) .EQ. 'SEAICEFR' ) ) THEN
               WHERE ( nall_lowbdy(loop)%array .LT. 0. ) 
                  nall_lowbdy(loop)%array = 0.
               END WHERE
               WHERE ( nall_lowbdy(loop)%array .GT. 1. ) 
                  nall_lowbdy(loop)%array = 1.
               END WHERE
            END IF
      
            IF   ( nall_lowbdy(loop)%small_header%name(1:8) .EQ. 'SEAICE  ' ) THEN
               WHERE ( nall_lowbdy(loop)%array .LT. 0.5 )
                  nall_lowbdy(loop)%array = 0.
               END WHERE
               WHERE ( nall_lowbdy(loop)%array .GE. 0.5 )
                  nall_lowbdy(loop)%array = 1.
               END WHERE
            END IF

            !  The outer row and column are filled, since these fields are known to be
            !  cross point.

            nall_lowbdy(loop)%array(nimax,:) = nall_lowbdy(loop)%array(nimax-1,:)
            nall_lowbdy(loop)%array(:,njmax) = nall_lowbdy(loop)%array(:,njmax-1)

            !  Correct the reservoir temperature for the change in elevation.  This is the
            !  same temperature adjustment that is applied to the other temperatures.

            IF ( nall_lowbdy(loop)%small_header%name(1:8) .EQ. 'RES TEMP' ) THEN
               nall_lowbdy(loop)%array = nall_lowbdy(loop)%array + t0diff(:,:,kxs)
            END IF

         END DO

         !  Output the lower boundary file.

         CALL sum_lbc2 ( nlbhi , nlbhr , nlbhic , nlbhrc , &
                         output_lowerbc , iprocess , interval , &
                         start_year , start_month , start_day , start_hour , &
                         start_minute , start_second , start_frac )

      ELSE IF   ( .NOT. use_mm5_lowbdy ) THEN

         !  If we are in this IF test (do not use the LOWBDY file from the coarse grid), we have to
         !  be using MM5 data as input.  No real reason, just consider it as some sort of   
         !  micromanagement or punishment.  Since this is MM5 data, we either have
         !  ( ( MM5 was using ISOIL=2 ) && ( TEMPGRD in input from TERRAIN ) ) or not.  If we do
         !  meet these criteria, then we use the TEMPGRD as the reservoir temperature in the 
         !  LOWBDY file.  If we do not meet the criteria (either MM5 did not run ISOIL=2 or the
         !  TEMPGRD field is not in the terrain input), then we horizontally interpolate the 
         !  coarse grid reservoir temperature (already done) and correct it for the change in
         !  elevation (and hey, we have already done that as well).

         !  Note that the TEMPGRD field comes from the TERRAIN file.  The rest of the data comes
         !  from the regular input data set (in this IF test, that is MM5).  The header for the
         !  TEMPGRD file is from a different field.  Don't say "Ah ha! I found a bug."  If this is
         !  an ISOIL=2 run, the TEMPGRD field is only used over land, so we fill it in with the
         !  lowest level air temperature over the water.  This way we can plot it out.

         IF ( ( pbhi(5,13) .EQ. 2 ) .AND. ( index_tempgrd .NE. 0 ) ) THEN

!           WHERE ( NINT(nall_2d(index_landuse)%array) .EQ. bhi(23,1) )
            WHERE ( all_terrain(index_tempgrd)%array .LT. 200 )
               all_terrain(index_tempgrd)%array = nall_3d(index_t)%array(:,:,kxs)
            END WHERE
            
            CALL add_lbc  ( all_terrain(index_tempgrd)%array , nall_2d(index_restemp)%small_header , &
                            nall_2d(index_tseasfc)%array     , nall_2d(index_tseasfc)%small_header , output_lowerbc )

         ELSE

            CALL add_lbc  ( nall_2d(index_restemp)%array     , nall_2d(index_restemp)%small_header , &
                            nall_2d(index_tseasfc)%array     , nall_2d(index_tseasfc)%small_header , output_lowerbc )

         END IF

      END IF

      !  Some of the 3d fields coming out of the boundary file have to be coupled.  Loop    
      !  over all of the 3d fields, and select the variables that are to be coupled.  For
      !  each of the selected variables, the field has to be coupled with either the cross
      !  or dot point pstar in kPa.

      CALL scale ( nps0dot , 0.001 , nimax , njmax , 1 , 0 )
      CALL scale ( nps0    , 0.001 , nimax , njmax , 1 , 1 )

      IF ( vert_nesting ) THEN
         couple_3dv : DO loop = 1 , num_3d
            IF ( vall_3d(loop)%small_header%name(1:8) .NE. 'RAD TEND' ) THEN
               IF        ( vall_3d(loop)%small_header%staggering(1:1) .EQ. 'D' ) THEN
                  CALL couple(vall_3d(loop)%array,nps0dot,nimax,njmax,kxs_half_new,0)
               ELSE IF ( ( vall_3d(loop)%small_header%staggering(1:1) .EQ. 'C' ) .AND. &
                         ( vall_3d(loop)%small_header%name(1:8) .NE. 'W       ' ) ) THEN
                  CALL couple(vall_3d(loop)%array,nps0,nimax,njmax,kxs_half_new,1)
               ELSE IF ( ( vall_3d(loop)%small_header%staggering(1:1) .EQ. 'C' ) .AND. &
                         ( vall_3d(loop)%small_header%name(1:8) .EQ. 'W       ' ) ) THEN
                  CALL couple(vall_3d(loop)%array,nps0,nimax,njmax,kxs_half_new+1,1)
               END IF
            END IF
         END DO couple_3dv
      ELSE
         couple_3d : DO loop = 1 , num_3d
            IF ( nall_3d(loop)%small_header%name(1:8) .NE. 'RAD TEND' ) THEN
               IF        ( nall_3d(loop)%small_header%staggering(1:1) .EQ. 'D' ) THEN
                  CALL couple(nall_3d(loop)%array,nps0dot,nimax,njmax,kxs,0)
               ELSE IF ( ( nall_3d(loop)%small_header%staggering(1:1) .EQ. 'C' ) .AND. &
                         ( nall_3d(loop)%small_header%name(1:8) .NE. 'W       ' ) ) THEN
                  CALL couple(nall_3d(loop)%array,nps0,nimax,njmax,kxs,1)
               ELSE IF ( ( nall_3d(loop)%small_header%staggering(1:1) .EQ. 'C' ) .AND. &
                         ( nall_3d(loop)%small_header%name(1:8) .EQ. 'W       ' ) ) THEN
                  CALL couple(nall_3d(loop)%array,nps0,nimax,njmax,kxs+1,1)
               END IF
            END IF
         END DO couple_3d
      END IF

      CALL scale ( nps0dot , 1000. , nimax , njmax , 1 , 0 )
      CALL scale ( nps0    , 1000. , nimax , njmax , 1 , 1 )

      !  Compute the lateral boundary conditions and tendencies.

      IF ( vert_nesting ) THEN
         CALL bdyoutv (output_bdyout , nimax, njmax, kxs_half_new, itimes, interval , &
                       start_year , start_month , start_day , start_hour , &
                       start_minute , start_second , start_frac )
      ELSE
         CALL bdyoutn (output_bdyout , nimax, njmax, kxs, itimes, interval , &
                       start_year , start_month , start_day , start_hour , &
                       start_minute , start_second , start_frac )
      END IF

      !  Boundary conditions are valid through the following times.

      IF ( itimes .GT. 1 ) THEN
         CALL geth_newdate ( old_date , current_date , -1* interval )
         PRINT '(A,A,A,A,A)','Lateral boundary conditions valid from ',old_date,' through ',current_date,'.'
      END IF

      !  Is this the last time?

      IF ( current_date .EQ. end_date ) THEN
         PRINT '(A)','At end of requested time periods.'
         EXIT time_loop
      END IF

      !  OK, this is not the last time, so we increment the current time by the 
      !  time interval given in the NAMELIST, and go to the next time.

      CALL geth_newdate ( new_date , current_date , interval )
      current_date = new_date
 
      !  DEALLOCATE the array space.

      DO loop = 1 , num_3d
         DEALLOCATE ( pall_3d(loop)%array )
         DEALLOCATE ( nall_3d(loop)%array )
         IF ( vert_nesting ) THEN
            DEALLOCATE ( vall_3d(loop)%array )
         END IF
      END DO
      DO loop = 1 , num_2d
         DEALLOCATE ( pall_2d(loop)%array )
         DEALLOCATE ( nall_2d(loop)%array )
      END DO
      DO loop = 1 , num_1d
         DEALLOCATE ( pall_1d(loop)%array )
         DEALLOCATE ( nall_1d(loop)%array )
      END DO
      
   END DO time_loop

   !  After all of the time periods have been processed, we need to do the summing
   !  and averaging for the lower boundary condition file.

   IF ( .NOT. use_mm5_lowbdy ) THEN
      CALL sum_lbc ( output_lowerbc , itimes , interval , less_than_24h , &
                     start_year , start_month , start_day , start_hour , &
                     start_minute , start_second , start_frac )
   END IF

   PRINT '(A)','STOP 99999'

END PROGRAM nestdown
