
!-------------------------------------------------------------
!
!  This subroutine writes data in NetCDF files.
!
!  Code originally written by Daniel Kirshbaum
!  Code last modified by George Bryan, 080516
!
!-------------------------------------------------------------


      subroutine writeout_cdf(nwrite,qname,sigma,sigmaf,xh,xf,yh,yf,zh,zf,    &
                          pi0,prs0,rho0,th0,qv0,u0,v0,                        &
                          zs,rain,sws,thflux,qvflux,dum1,dum2,                &
                          rho,prs,ua,dumu,va,dumv,wa,dumw,ppi,tha,            &
                          qa,kmh,kmv,khh,khv,tkea,pta)
      implicit none

      include 'input.incl'
      include 'constants.incl'

#ifdef MPI
      include 'mpif.h'
#endif

      integer :: nwrite
      character*3, dimension(maxq) :: qname
      real, dimension(kb:ke) :: sigma
      real, dimension(kb:ke+1) :: sigmaf
      real, dimension(ib:ie) :: xh
      real, dimension(ib:ie+1) :: xf
      real, dimension(jb:je) :: yh
      real, dimension(jb:je+1) :: yf
      real, dimension(ib:ie,jb:je,kb:ke) :: zh,pi0,prs0,rho0,th0,qv0
      real, dimension(ib:ie,jb:je,kb:ke+1) :: zf
      real, dimension(itb:ite,jtb:jte) :: zs
      real, dimension(ib:ie,jb:je,nrain) :: rain,sws
      real, dimension(ib:ie,jb:je) :: thflux,qvflux
      real, dimension(ib:ie,jb:je,kb:ke) :: dum1,dum2,rho,prs
      real, dimension(ib:ie+1,jb:je,kb:ke) :: u0,ua,dumu
      real, dimension(ib:ie,jb:je+1,kb:ke) :: v0,va,dumv
      real, dimension(ib:ie,jb:je,kb:ke+1) :: wa,dumw
      real, dimension(ib:ie,jb:je,kb:ke) :: ppi,tha
      real, dimension(ibm:iem,jbm:jem,kbm:kem,numq) :: qa
      real, dimension(ibc:iec,jbc:jec,kbc:kec) :: kmh,kmv,khh,khv
      real, dimension(ibt:iet,jbt:jet,kbt:ket) :: tkea
      real, dimension(ibp:iep,jbp:jep,kbp:kep,npt) :: pta

#ifdef NETCDF

      include 'netcdf.inc'
      integer i,j,k,n,irec
#ifdef MPI
      character*80 sname,uname,vname,wname
#endif

      logical, parameter :: coards = .false.

      integer :: cdfid    ! ID for the netCDF file to be created
      integer :: time_index
      character *16  cdf_out    ! Name of the netCDF output file
      character *11  cdl_out
      character *80 command,command2
      character*8 chid

      real d2d(ni,nj)
      real ds(ni,nj,nk)
      real du(ni+1,nj,nk)
      real dv(ni,nj+1,nk)
      real dw(ni,nj,nk+1)

!-------------------------------------------------------------
! Declare and set integer values for the netCDF dimensions 
!-------------------------------------------------------------

      integer nip1,njp1,nkp1

      integer         &
          start(4),   &   ! for indexing the netCDF data array;
          start2d(3), &
          count2d(3), &
          count(4),   &   ! for indexing the netCDF data array;
          countxp(4), &   ! for indexing the netCDF data array;
          countyp(4), &   ! for indexing the netCDF data array;
          countzp(4)      ! for indexing the netCDF data array;

      real actual_time
      real x_min, x_max, y_min, y_max, z_min, z_max
      real x_mns, x_mxs, y_mns, y_mxs, z_mns, z_mxs

      integer idtime
      logical :: allinfo

!--------------------------------------------------------------
! Initializing some things
!--------------------------------------------------------------

      time_index = nwrite
      if(output_filetype.eq.2) time_index = 1

    if(coards)then
      if(tapfrq.le.60.0)then
        print *
        print *,'  Output frequency cannot be less than 60 s for coards format'
        print *
        call stopcm1
      endif
      actual_time = (nwrite-1)*tapfrq/60.0
    else
      actual_time = (nwrite-1)*tapfrq
    endif

      nip1=ni+1
      njp1=nj+1
      nkp1=nk+1

      chid = '        '

!--------------------------------------------------------------
!  Write data to cdf file
!--------------------------------------------------------------

#ifndef MPI
    IF(output_filetype.eq.1)THEN
      cdl_out = 'cm1out.nl'
      cdf_out = 'cm1out.nc'
    ELSEIF(output_filetype.eq.2)THEN
      cdl_out = 'cm1out_XXXX.nl'
      cdf_out = 'cm1out_XXXX.nc'
      write(cdl_out(8:11),100) nwrite
      write(cdf_out(8:11),100) nwrite
    ELSE
      write(outfile,*) '  Invalid option for output_filetype'
      call stopcm1
    ENDIF
#else

      cdl_out = 'cm1out_XXXX.cdl'
      cdf_out = 'cm1out_XXXX_YYYY.cdf'

      write(cdl_out( 8:11),100) myid
      write(cdf_out( 8:11),100) myid
      write(cdf_out(13:16),100) nwrite
#endif

100   format(i4.4)


      start=(/1, 1, 1, 1/)
      start2d=(/1, 1, 1/)
      count2d=(/ni, nj, 1/)
      count=(/ni, nj, nk, 1/)
      countxp=(/nip1, nj, nk, 1/)
      countyp=(/ni, njp1, nk, 1/)
      countzp=(/ni, nj, nkp1, 1/)

    if(output_filetype.eq.1)then
      start(4) = nwrite
      start2d(3) = nwrite
    elseif(output_filetype.eq.1)then
      start(4) = 1
      start2d(3) = 1
    endif

!--------------------------------------------------------------
!  if this is the start of a file, then do this stuff:

    allinfo = .false.
    IF(nwrite.eq.1) allinfo=.true.
    IF(output_filetype.eq.2) allinfo=.true.

    ifallinfo: IF(allinfo)THEN

!--------------------------------------------------------------
! Determine start and end locations for data
!--------------------------------------------------------------

      x_min = xf(1)
      x_max = xf(nip1)
      x_mns = xh(1)
      x_mxs = xh(ni)

      y_min = yf(1)
      y_max = yf(njp1)
      y_mns = yh(1)
      y_mxs = yh(nj)

    if(terrain_flag)then
      z_min = sigmaf(1)
      z_max = sigmaf(nkp1)
      z_mns = sigma(1)
      z_mxs = sigma(nk)
    else
      z_min = zf(1,1,1)
      z_max = zf(1,1,nkp1)
      z_mns = zh(1,1,1)
      z_mxs = zh(1,1,nk)
    endif

      open(unit=7, file=cdl_out)
      write(7,*) 'netcdf mine{'
  
! Dimensions

      write(7,*) 'dimensions:'
      write(7,*) 'ni=', ni, ';'
      write(7,*) 'nj=', nj, ';'
      write(7,*) 'nk=', nk, ';'
      write(7,*) 'nip1=', nip1, ';'
      write(7,*) 'njp1=', njp1, ';'
      write(7,*) 'nkp1=', nkp1, ';'
      write(7,*) 'time=UNLIMITED;'
      write(7,*) 'one=',1,';'

! Write the variable declarations along with their attributes

      write(7,*) 'variables:'

      write(7,*) 'float f_cor(one);'
      write(7,*) 'f_cor:units="1/s";'

    if(.not.coards)then
      write(7,*) 'float ztop(one);'
      write(7,*) 'ztop:units="km";'
    endif

! 1D variables

    IF(coards)THEN

      write(7,*) 'float time(time);'
      write(7,*) 'time:description="time since beginning of simulation";'
      write(7,*) 'time:units="minutes since 2000-07-03 00:00:00";'

      write(7,*) 'float ni(ni);'
      write(7,*) 'ni:description="west-east location of scalar grid points";'
      write(7,*) 'ni:units="degree_east";'

      write(7,*) 'float nip1(nip1);'
      write(7,*) 'nip1:description="west-east location of staggered u grid points";'
      write(7,*) 'nip1:units="degree_east";'

      write(7,*) 'float nj(nj);'
      write(7,*) 'nj:description="south-north location of scalar grid points";'
      write(7,*) 'nj:units="degree_north";'

      write(7,*) 'float njp1(njp1);'
      write(7,*) 'njp1:description="south-north location of staggered v grid points";'
      write(7,*) 'njp1:units="degree_north";'

      write(7,*) 'float nk(nk);'
      write(7,*) 'nk:description="nominal height of scalar grid points";'
      write(7,*) 'nk:units="m";'

      write(7,*) 'float nkp1(nkp1);'
      write(7,*) 'nkp1:description="nominal height of staggered w grid points";'
      write(7,*) 'nkp1:units="m";'

    ELSE

      write(7,*) 'float time(time);'
      write(7,*) 'time:description="time since beginning of simulation";'
      write(7,*) 'time:units="seconds since 2000-07-03 00:00:00";'

      write(7,*) 'float xh(ni);'
      write(7,*) 'xh:description="west-east location of scalar grid points";'
      write(7,*) 'xh:units="km";'

      write(7,*) 'float xf(nip1);'
      write(7,*) 'xf:description="west-east location of staggered u grid points";'
      write(7,*) 'xf:units="km";'

      write(7,*) 'float yh(nj);'
      write(7,*) 'yh:description="south-north location of scalar grid points";'
      write(7,*) 'yh:units="km";'

      write(7,*) 'float yf(njp1);'
      write(7,*) 'yf:description="south-north location of staggered v grid points";'
      write(7,*) 'yf:units="km";'

      write(7,*) 'float zh(nk);'
      write(7,*) 'zh:description="nominal height of scalar grid points";'
      write(7,*) 'zh:units="km";'

      write(7,*) 'float zf(nkp1);'
      write(7,*) 'zf:description="nominal height of staggered w grid points";'
      write(7,*) 'zf:units="km";'

    ENDIF

      if (output_zs.eq.1.and.terrain_flag) then
        write(7,*) 'float zs(nj,ni);'
        write(7,*) 'zs:description="terrain height";'
        write(7,*) 'zs:units="m";'
        write(7,*) 'zs:x_min=',0.001*x_mns,';'
        write(7,*) 'zs:y_min=',0.001*y_mns,';'
      endif

      if (output_zh.eq.1.and.terrain_flag) then
        write(7,*) 'float zh(nk,nj,ni);'
        write(7,*) 'zh:description="height (above ground) of scalar grid points";'
        write(7,*) 'zh:units="km";'
        write(7,*) 'zh:x_min=',0.001*x_mns,';'
        write(7,*) 'zh:y_min=',0.001*y_mns,';'
        write(7,*) 'zh:z_min=',0.001*z_mns,';'
      endif

      if (output_basestate.eq.1) then

        write(7,*) 'float pi0(nk,nj,ni);'
        write(7,*) 'pi0:description="base-state nondimensional pressure";'
        write(7,*) 'pi0:units="dimensionless";'
        write(7,*) 'pi0:x_min=',0.001*x_mns,';'
        write(7,*) 'pi0:y_min=',0.001*y_mns,';'
        write(7,*) 'pi0:z_min=',0.001*z_mns,';'

        write(7,*) 'float th0(nk,nj,ni);'
        write(7,*) 'th0:description="base-state potential temperature";'
        write(7,*) 'th0:units="K";'
        write(7,*) 'th0:x_min=',0.001*x_mns,';'
        write(7,*) 'th0:y_min=',0.001*y_mns,';'
        write(7,*) 'th0:z_min=',0.001*z_mns,';'

        write(7,*) 'float prs0(nk,nj,ni);'
        write(7,*) 'prs0:description="base-state pressure";'
        write(7,*) 'prs0:units="Pa";'
        write(7,*) 'prs0:x_min=',0.001*x_mns,';'
        write(7,*) 'prs0:y_min=',0.001*y_mns,';'
        write(7,*) 'prs0:z_min=',0.001*z_mns,';'

        write(7,*) 'float qv0(nk,nj,ni);'
        write(7,*) 'qv0:description="base-state water vapor mixing ratio";'
        write(7,*) 'qv0:units="kg/kg";'
        write(7,*) 'qv0:x_min=',0.001*x_mns,';'
        write(7,*) 'qv0:y_min=',0.001*y_mns,';'
        write(7,*) 'qv0:z_min=',0.001*z_mns,';'

        write(7,*) 'float u0(nk,nj,nip1);'
        write(7,*) 'u0:description="base-state x-component of velocity";'
        write(7,*) 'u0:units="m/s";'
        write(7,*) 'u0:x_min=',0.001*x_min,';'
        write(7,*) 'u0:y_min=',0.001*y_mns,';'
        write(7,*) 'u0:z_min=',0.001*z_mns,';'

        write(7,*) 'float v0(nk,njp1,ni);'
        write(7,*) 'v0:description="base-state y-component of velocity";'
        write(7,*) 'v0:units="m/s";'
        write(7,*) 'v0:x_min=',0.001*x_mns,';'
        write(7,*) 'v0:y_min=',0.001*y_min,';'
        write(7,*) 'v0:z_min=',0.001*z_mns,';'

      endif

! 2D variables

      if (output_rain.eq.1) then
        write(7,*) 'float rain(time,nj,ni);'
        write(7,*) 'rain:description="accumulated surface rainfall";'
        write(7,*) 'rain:units="cm";'
        write(7,*) 'rain:x_min=',0.001*x_mns,';'
        write(7,*) 'rain:y_min=',0.001*y_mns,';'
      endif

      if (output_sws.eq.1) then
        write(7,*) 'float sws(time,nj,ni);'
        write(7,*) 'sws:description="max surface windspeed";'
        write(7,*) 'sws:units="m/s";'
        write(7,*) 'sws:x_min=',0.001*x_mns,';'
        write(7,*) 'sws:y_min=',0.001*y_mns,';'
      endif

      IF(nrain.eq.2)THEN

        if (output_rain.eq.1) then
          write(7,*) 'float rain2(time,nj,ni);'
          write(7,*) 'rain2:description="accumulated surface rainfall, translated with moving domain";'
          write(7,*) 'rain2:units="cm";'
          write(7,*) 'rain2:x_min=',0.001*x_mns,';'
          write(7,*) 'rain2:y_min=',0.001*y_mns,';'
        endif

        if (output_sws.eq.1) then
          write(7,*) 'float sws2(time,nj,ni);'
          write(7,*) 'sws2:description="max surface windspeed, translated with moving domain";'
          write(7,*) 'sws2:units="m/s";'
          write(7,*) 'sws2:x_min=',0.001*x_mns,';'
          write(7,*) 'sws2:y_min=',0.001*y_mns,';'
        endif

      ENDIF

      if (isfcflx.eq.1) then
        write(7,*) 'float thflux(time,nj,ni);'
        write(7,*) 'thflux:description="surface potential temperature flux";'
        write(7,*) 'thflux:units="K m s^{-1}";'
        write(7,*) 'thflux:x_min=',0.001*x_mns,';'
        write(7,*) 'thflux:y_min=',0.001*y_mns,';'

        write(7,*) 'float qvflux(time,nj,ni);'
        write(7,*) 'qvflux:description="surface water vapor flux flux";'
        write(7,*) 'qvflux:units="kg kg^{-1} m s^{-1}";'
        write(7,*) 'qvflux:x_min=',0.001*x_mns,';'
        write(7,*) 'qvflux:y_min=',0.001*y_mns,';'
      endif

!     3D variables

      if (output_u.eq.1) then
        write(7,*) 'float u(time,nk,nj,nip1);'
        write(7,*) 'u:description="velocity in x-direction";'
        write(7,*) 'u:units="m/s";'
        write(7,*) 'u:x_min=',0.001*x_min,';'
        write(7,*) 'u:y_min=',0.001*y_mns,';'
        write(7,*) 'u:z_min=',0.001*z_mns,';'
      endif

      if (output_upert.eq.1) then
        write(7,*) 'float upert(time,nk,nj,nip1);'
        write(7,*) 'upert:description="perturbation velocity in x-direction";'
        write(7,*) 'upert:units="m/s";'
        write(7,*) 'upert:x_min=',0.001*x_min,';'
        write(7,*) 'upert:y_min=',0.001*y_mns,';'
        write(7,*) 'upert:z_min=',0.001*z_mns,';'
      endif

      if (output_v.eq.1) then
        write(7,*) 'float v(time,nk,njp1,ni);'
        write(7,*) 'v:description="velocity in y-direction";'
        write(7,*) 'v:units="m/s";'
        write(7,*) 'v:x_min=',0.001*x_mns,';'
        write(7,*) 'v:y_min=',0.001*y_min,';'
        write(7,*) 'v:z_min=',0.001*z_mns,';'
      endif

      if (output_vpert.eq.1) then
        write(7,*) 'float vpert(time,nk,njp1,ni);'
        write(7,*) 'vpert:description="perturbation velocity in y-direction";'
        write(7,*) 'vpert:units="m/s";'
        write(7,*) 'vpert:x_min=',0.001*x_mns,';'
        write(7,*) 'vpert:y_min=',0.001*y_min,';'
        write(7,*) 'vpert:z_min=',0.001*z_mns,';'
      endif

      if (output_w.eq.1) then
        write(7,*) 'float w(time,nkp1,nj,ni);'
        write(7,*) 'w:description="velocity in z-direction";'
        write(7,*) 'w:units="m/s";'
        write(7,*) 'w:x_min=',0.001*x_mns,';'
        write(7,*) 'w:y_min=',0.001*y_mns,';'
        write(7,*) 'w:z_min=',0.001*z_min,';'
      endif

      if (output_rho.eq.1) then
        write(7,*) 'float rho(time,nk,nj,ni);'
        write(7,*) 'rho:description="density of dry air";'
        write(7,*) 'rho:units="kg/m^3";'
        write(7,*) 'rho:x_min=',0.001*x_mns,';'
        write(7,*) 'rho:y_min=',0.001*y_mns,';'
        write(7,*) 'rho:z_min=',0.001*z_mns,';'
      endif

      if (output_rhopert.eq.1) then
        write(7,*) 'float rhopert(time,nk,nj,ni);'
        write(7,*) 'rhopert:description="perturbation density of dry air";'
        write(7,*) 'rhopert:units="kg/m^3";'
        write(7,*) 'rhopert:x_min=',0.001*x_mns,';'
        write(7,*) 'rhopert:y_min=',0.001*y_mns,';'
        write(7,*) 'rhopert:z_min=',0.001*z_mns,';'        
      endif
 
      if (output_th.eq.1) then
        write(7,*) 'float th(time,nk,nj,ni);'
        write(7,*) 'th:description="potential temperature";'
        write(7,*) 'th:units="K";'
        write(7,*) 'th:x_min=',0.001*x_mns,';'
        write(7,*) 'th:y_min=',0.001*y_mns,';'
        write(7,*) 'th:z_min=',0.001*z_mns,';'
      endif

      if (output_thpert.eq.1) then
        write(7,*) 'float thpert(time,nk,nj,ni);'
        write(7,*) 'thpert:description="perturbation potential temperature";'
        write(7,*) 'thpert:units="K";'
        write(7,*) 'thpert:x_min=',0.001*x_mns,';'
        write(7,*) 'thpert:y_min=',0.001*y_mns,';'
        write(7,*) 'thpert:z_min=',0.001*z_mns,';'     
      endif

      if (output_pi.eq.1) then
        write(7,*) 'float pii(time,nk,nj,ni);'
        write(7,*) 'pii:description="nondimensional pressure";'
        write(7,*) 'pii:units="dimensionless";'
        write(7,*) 'pii:x_min=',0.001*x_mns,';'
        write(7,*) 'pii:y_min=',0.001*y_mns,';'
        write(7,*) 'pii:z_min=',0.001*z_mns,';' 
      endif

      if (output_pipert.eq.1) then
        write(7,*) 'float ppi(time,nk,nj,ni);'
        write(7,*) 'ppi:description="perturbation nondimensional pressure";'
        write(7,*) 'ppi:units="dimensionless";'
        write(7,*) 'ppi:x_min=',0.001*x_mns,';'
        write(7,*) 'ppi:y_min=',0.001*y_mns,';'
        write(7,*) 'ppi:z_min=',0.001*z_mns,';'
      endif

      if (output_prs.eq.1) then
        write(7,*) 'float prs(time,nk,nj,ni);'
        write(7,*) 'prs:description="pressure";'
        write(7,*) 'prs:units="Pa";'
        write(7,*) 'prs:x_min=',0.001*x_mns,';'
        write(7,*) 'prs:y_min=',0.001*y_mns,';'
        write(7,*) 'prs:z_min=',0.001*z_mns,';'
      endif

      if (output_prspert.eq.1) then
        write(7,*) 'float prspert(time,nk,nj,ni);'
        write(7,*) 'prspert:description="perturbation pressure";'
        write(7,*) 'prspert:units="Pa";'
        write(7,*) 'prspert:x_min=',0.001*x_mns,';'
        write(7,*) 'prspert:y_min=',0.001*y_mns,';'
        write(7,*) 'prspert:z_min=',0.001*z_mns,';'
      endif

      if (imoist.eq.1) then

        if(output_qv.eq.1)then
          write(7,121) qname(nqv)
          write(7,129) qname(nqv)
129       format(1x,a3,':description="water vapor mixing ratio";')
          write(7,122) qname(nqv)
          write(7,124) qname(nqv),0.001*x_mns
          write(7,125) qname(nqv),0.001*y_mns
          write(7,126) qname(nqv),0.001*z_mns
        endif
        if(output_qv.eq.1)then
          write(7,*) 'float qvpert(time,nk,nj,ni);'
          write(7,*) 'qvpert:description="perturbation water vapor mixing ratio";'
          write(7,*) 'qvpert:units="kg/kg";'
          write(7,*) 'qvpert:x_min=',0.001*x_mns,';'
          write(7,*) 'qvpert:y_min=',0.001*y_mns,';'
          write(7,*) 'qvpert:z_min=',0.001*z_mns,';'
        endif
        if(output_q.eq.1)then
          do n=1,numq
            if(n.ne.nqv)then
              write(7,121) qname(n)
              if(idm.eq.1.and.n.ge.nnc1.and.n.le.nnc2)then
                write(7,123) qname(n)
                write(7,128) qname(n)
              else
                write(7,122) qname(n)
                write(7,127) qname(n)
              endif
              write(7,124) qname(n),0.001*x_mns
              write(7,125) qname(n),0.001*y_mns
              write(7,126) qname(n),0.001*z_mns
            endif
          enddo
121       format(' float ',a3,'(time,nk,nj,ni);')
127       format(1x,a3,':description="mixing ratio";')
128       format(1x,a3,':description="number concentration";')
122       format(1x,a3,':units="kg/kg";')
123       format(1x,a3,':units="kg^{-1}";')
124       format(1x,a3,':x_min=',f12.3,';')
125       format(1x,a3,':y_min=',f12.3,';')
126       format(1x,a3,':z_min=',f12.3,';')
        endif
        if(iptra.eq.1)then
          do n=1,npt
            write(7,131) n
            write(7,132) n
            write(7,133) n
            write(7,134) n,0.001*x_mns
            write(7,135) n,0.001*y_mns
            write(7,136) n,0.001*z_mns
          enddo
131       format(' float pt',i1,'(time,nk,nj,ni);')
132       format(' pt',i1,':description="mixing ratio of passive tracer";')
133       format(' pt',i1,':units="kg/kg";')
134       format(' pt',i1,':x_min=',f12.3,';')
135       format(' pt',i1,':y_min=',f12.3,';')
136       format(' pt',i1,':z_min=',f12.3,';')
        endif
      endif


      if (iturb.gt.0) then
        if (output_km.eq.1) then
!----------------
          write(7,*) 'float kmh(time,nkp1,nj,ni);'
          write(7,*) 'kmh:description="eddy mixing coefficient for momentum in the horizontal direction";'
          write(7,*) 'kmh:units="m/s^2";'
          write(7,*) 'kmh:x_min=',0.001*x_mns,';'
          write(7,*) 'kmh:y_min=',0.001*y_mns,';'
          write(7,*) 'kmh:z_min=',0.001*z_min,';'
!----------------
          write(7,*) 'float kmv(time,nkp1,nj,ni);'
          write(7,*) 'kmv:description="eddy mixing coefficient for momentum in the vertical direction";'
          write(7,*) 'kmv:units="m/s^2";'
          write(7,*) 'kmv:x_min=',0.001*x_mns,';'
          write(7,*) 'kmv:y_min=',0.001*y_mns,';'
          write(7,*) 'kmv:z_min=',0.001*z_min,';'
!----------------
        endif
        if (output_kh.eq.1) then
!----------------
          write(7,*) 'float khh(time,nkp1,nj,ni);'
          write(7,*) 'khh:description="eddy mixing coefficient for scalars in the horizontal direction";'
          write(7,*) 'khh:units="m/s^2";'
          write(7,*) 'khh:x_min=',0.001*x_mns,';'
          write(7,*) 'khh:y_min=',0.001*y_mns,';'
          write(7,*) 'khh:z_min=',0.001*z_min,';'
!----------------
          write(7,*) 'float khv(time,nkp1,nj,ni);'
          write(7,*) 'khv:description="eddy mixing coefficient for scalars in the vertical direction";'
          write(7,*) 'khv:units="m/s^2";'
          write(7,*) 'khv:x_min=',0.001*x_mns,';'
          write(7,*) 'khv:y_min=',0.001*y_mns,';'
          write(7,*) 'khv:z_min=',0.001*z_min,';'
!----------------
        endif
      endif

      if ((iturb.eq.1).and.(output_tke.eq.1)) then
        write(7,*) 'float tke(time,nkp1,nj,ni);'
        write(7,*) 'tke:description="subgrid turbulence kinetic energy";'
        write(7,*) 'tke:units="kg m^2/s^2";'
        write(7,*) 'tke:x_min=',0.001*x_mns,';'
        write(7,*) 'tke:y_min=',0.001*y_mns,';'
        write(7,*) 'tke:z_min=',0.001*z_min,';'
      endif

      if(output_uinterp.eq.1)then
        write(7,*) 'float uinterp(time,nk,nj,ni);'
        write(7,*) 'uinterp:description="velocity in x-direction, interpolated to scalar points";'
        write(7,*) 'uinterp:units="m/s";'
        write(7,*) 'uinterp:x_min=',0.001*x_mns,';'
        write(7,*) 'uinterp:y_min=',0.001*y_mns,';'
        write(7,*) 'uinterp:z_min=',0.001*z_mns,';'
      endif
      if(output_vinterp.eq.1)then
        write(7,*) 'float vinterp(time,nk,nj,ni);'
        write(7,*) 'vinterp:description="velocity in y-direction, interpolated to scalar points";'
        write(7,*) 'vinterp:units="m/s";'
        write(7,*) 'vinterp:x_min=',0.001*x_mns,';'
        write(7,*) 'vinterp:y_min=',0.001*y_mns,';'
        write(7,*) 'vinterp:z_min=',0.001*z_mns,';'
      endif
      if(output_winterp.eq.1)then
        write(7,*) 'float winterp(time,nk,nj,ni);'
        write(7,*) 'winterp:description="velocity in z-direction, interpolated to scalar points";'
        write(7,*) 'winterp:units="m/s";'
        write(7,*) 'winterp:x_min=',0.001*x_mns,';'
        write(7,*) 'winterp:y_min=',0.001*y_mns,';'
        write(7,*) 'winterp:z_min=',0.001*z_mns,';'
      endif

! Write the global attributes

      write(7,*) ':Conventions = "COARDS";'
      write(7,*) '// Cooperative Ocean/Atmosphere Research Data Service'

 60   format(1x,':x_min=',f12.4,';')
 61   format(1x,':x_max=',f12.4,';')
 62   format(1x,':x_delta=',f12.4,';')
 63   format(1x,':y_min=',f12.4,';')
 64   format(1x,':y_max=',f12.4,';')
 65   format(1x,':y_delta=',f12.4,';')
 66   format(1x,':z_min=',f12.4,';')
 67   format(1x,':z_max=',f12.4,';')
 68   format(1x,':z_delta=',f12.4,';')

      write(7,60) 0.001*x_min
      write(7,61) 0.001*x_max
      write(7,62) 0.001*dx
      write(7,*) ':x_units="km";'
      write(7,*) ':x_label="x";'
      write(7,63) 0.001*y_min
      write(7,64) 0.001*y_max
      write(7,65) 0.001*dy
      write(7,*) ':y_units="km";'
      write(7,*) ':y_label="y";'
      write(7,66) 0.001*z_min
      write(7,67) 0.001*z_max
      write(7,68) 0.001*dz
      write(7,*) ':z_units="km";'
      write(7,*) ':z_label="z";'


! Begin writing the data

      write(7,*) 'data:'
      write(7,*) 'f_cor=',fcor,';'
    if(.not.coards)then
      write(7,*) 'ztop=',0.001*ztop,';'
    endif

    if(coards)then
      write(7,*) 'ni='
    else
      write(7,*) 'xh='
    endif
      do i = 1, ni-1
         write(7, *) 0.001*xh(i), ','
      enddo
      write(7,*) 0.001*xh(ni), ';'

    if(coards)then
      write(7,*) 'nip1='
    else
      write(7,*) 'xf='
    endif
      do i = 1, ni
         write(7, *) 0.001*xf(i), ','
      enddo
      write(7,*) 0.001*xf(ni+1), ';'

    if(coards)then
      write(7,*) 'nj='
    else
      write(7,*) 'yh='
    endif
      do j = 1, nj-1
         write(7, *) 0.001*yh(j), ','
      enddo
      write(7,*) 0.001*yh(nj), ';'

    if(coards)then
      write(7,*) 'njp1='
    else
      write(7,*) 'yf='
    endif
      do j = 1, nj
         write(7, *) 0.001*yf(j), ','
      enddo
      write(7,*) 0.001*yf(nj+1), ';'

    if(terrain_flag)then
    if(coards)then
      write(7,*) 'nk='
    else
      write(7,*) 'zh='
    endif
      do k = 1, nk-1
         write(7, *) 0.001*sigma(k), ','
      enddo
      write(7,*) 0.001*sigma(nk), ';'
    if(coards)then
      write(7,*) 'nkp1='
    else
      write(7,*) 'zf='
    endif
      do k = 1, nk
         write(7, *) 0.001*sigmaf(k), ','
      enddo
      write(7,*) 0.001*sigmaf(nk+1), ';'
    else
    if(coards)then
      write(7,*) 'nk='
    else
      write(7,*) 'zh='
    endif
      do k = 1, nk-1
         write(7, *) 0.001*zh(1,1,k), ','
      enddo
      write(7,*) 0.001*zh(1,1,nk), ';'
    if(coards)then
      write(7,*) 'nkp1='
    else
      write(7,*) 'zf='
    endif
      do k = 1, nk
         write(7, *) 0.001*zf(1,1,k), ','
      enddo
      write(7,*) 0.001*zf(1,1,nk+1), ';'
    endif

      write(7,*) '}'

      close (unit=7)
 
      write(command, *) 'ncgen -o ', cdf_out,' ',cdl_out
      call system(command)

      write(command2, *) 'rm -f ', cdl_out
      call system(command2)

    ENDIF ifallinfo

!------------------------------------------------------------------
!------------------------------------------------------------------
!------------------------------------------------------------------

      cdfid=ncopn(cdf_out, NCWRITE, ierr)

    moreinfo: IF(allinfo)THEN
      if (output_zs.eq.1.and.terrain_flag) then
        do j=1,nj
        do i=1,ni
          d2d(i,j)=zs(i,j)
        enddo
        enddo
        call write2d_nc('zs      ',cdfid,ierr,start2d,count2d,ni,nj,d2d)
      endif
      if (output_zh.eq.1.and.terrain_flag) then
        do k=1,nk
        do j=1,nj
        do i=1,ni
          ds(i,j,k)=zh(i,j,k)
        enddo
        enddo
        enddo
        call write3d_nc('zh      ',cdfid,ierr,start,count,ni,nj,nk,ds)
      endif
      if (output_basestate.eq.1) then
        do k=1,nk
        do j=1,nj
        do i=1,ni
          ds(i,j,k)=pi0(i,j,k)
        enddo
        enddo
        enddo
        call write3d_nc('pi0     ',cdfid,ierr,start,count,ni,nj,nk,ds)
        do k=1,nk
        do j=1,nj
        do i=1,ni
          ds(i,j,k)=th0(i,j,k)
        enddo
        enddo
        enddo
        call write3d_nc('th0     ',cdfid,ierr,start,count,ni,nj,nk,ds)
        do k=1,nk
        do j=1,nj
        do i=1,ni
          ds(i,j,k)=prs0(i,j,k)
        enddo
        enddo
        enddo
        call write3d_nc('prs0    ',cdfid,ierr,start,count,ni,nj,nk,ds)
        do k=1,nk
        do j=1,nj
        do i=1,ni
          ds(i,j,k)=qv0(i,j,k)
        enddo
        enddo
        enddo
        call write3d_nc('qv0     ',cdfid,ierr,start,count,ni,nj,nk,ds)
        do k=1,nk
        do j=1,nj
        do i=1,ni+1
          du(i,j,k)=u0(i,j,k)
        enddo
        enddo
        enddo
        call write3d_nc('u0      ',cdfid,ierr,start,countxp,ni+1,nj,nk,du)
        do k=1,nk
        do j=1,nj+1
        do i=1,ni
          dv(i,j,k)=v0(i,j,k)
        enddo
        enddo
        enddo
        call write3d_nc('v0      ',cdfid,ierr,start,countyp,ni,nj+1,nk,dv)
      endif
    ENDIF moreinfo

!------------------------------------------------------------------
! Write 1D output variables
!------------------------------------------------------------------
      
      idtime=ncvid(cdfid, 'time'  , ierr)
      call ncvpt1(cdfid, idtime, time_index, actual_time, ierr)

!------------------------------------------------------------------
! Write 2D output variables
!------------------------------------------------------------------
      
      if (imoist.eq.1) then
        if (output_rain.eq.1) then
          do j=1,nj
          do i=1,ni
            d2d(i,j)=rain(i,j,1)
          enddo
          enddo
          call write2d_nc('rain    ',cdfid,ierr,start2d,count2d,ni,nj,d2d)
        endif
        if (output_sws.eq.1) then
          do j=1,nj
          do i=1,ni
            d2d(i,j)=sws(i,j,1)
          enddo
          enddo
          call write2d_nc('sws     ',cdfid,ierr,start2d,count2d,ni,nj,d2d)
        endif
        IF(nrain.eq.2)THEN
          if (output_rain.eq.1) then
            do j=1,nj
            do i=1,ni
              d2d(i,j)=rain(i,j,2)
            enddo
            enddo
            call write2d_nc('rain2   ',cdfid,ierr,start2d,count2d,ni,nj,d2d)
          endif
          if (output_sws.eq.1) then
            do j=1,nj
            do i=1,ni
              d2d(i,j)=sws(i,j,2)
            enddo
            enddo
            call write2d_nc('sws2    ',cdfid,ierr,start2d,count2d,ni,nj,d2d)
          endif
        ENDIF
      endif
      if (isfcflx.eq.1) then
        do j=1,nj
        do i=1,ni
          d2d(i,j)=thflux(i,j)
        enddo
        enddo
        call write2d_nc('thflux  ',cdfid,ierr,start2d,count2d,ni,nj,d2d)
        do j=1,nj
        do i=1,ni
          d2d(i,j)=qvflux(i,j)
        enddo
        enddo
        call write2d_nc('qvflux  ',cdfid,ierr,start2d,count2d,ni,nj,d2d)
      endif

!------------------------------------------------------------------
! Write 3D output variables
!------------------------------------------------------------------
      
!-------------
      if (output_u.eq.1) then
        do k=1,nk
        do j=1,nj
        do i=1,ni+1
          du(i,j,k)=ua(i,j,k)
        enddo
        enddo
        enddo
        call write3d_nc('u       ',cdfid,ierr,start,countxp,ni+1,nj,nk,du)
      endif
      if (output_upert.eq.1) then
        do k=1,nk
        do j=1,nj
        do i=1,ni+1
          du(i,j,k)=ua(i,j,k)-u0(i,j,k)
        enddo
        enddo
        enddo
        call write3d_nc('upert   ',cdfid,ierr,start,countxp,ni+1,nj,nk,du)
      endif
      if (output_v.eq.1) then
        do k=1,nk
        do j=1,nj+1
        do i=1,ni
          dv(i,j,k)=va(i,j,k)
        enddo
        enddo
        enddo
        call write3d_nc('v       ',cdfid,ierr,start,countyp,ni,nj+1,nk,dv)
      endif
      if (output_vpert.eq.1) then
        do k=1,nk
        do j=1,nj+1
        do i=1,ni
          dv(i,j,k)=va(i,j,k)-v0(i,j,k)
        enddo
        enddo
        enddo
        call write3d_nc('vpert   ',cdfid,ierr,start,countyp,ni,nj+1,nk,dv)
      endif     
      if (output_w.eq.1) then
        do k=1,nk+1
        do j=1,nj
        do i=1,ni
          dw(i,j,k)=wa(i,j,k)
        enddo
        enddo
        enddo
        call write3d_nc('w       ',cdfid,ierr,start,countzp,ni,nj,nk+1,dw)
      endif
      if (output_rho.eq.1) then
        do k=1,nk
        do j=1,nj
        do i=1,ni
          ds(i,j,k)=rho(i,j,k)
        enddo
        enddo
        enddo
        call write3d_nc('rho     ',cdfid,ierr,start,count,ni,nj,nk,ds)
      endif
!-------------
      if (output_rhopert.eq.1) then
        do k=1,nk
        do j=1,nj
        do i=1,ni
          ds(i,j,k)=rho(i,j,k)-rho0(i,j,k)
        enddo
        enddo
        enddo
        call write3d_nc('rhopert ',cdfid,ierr,start,count,ni,nj,nk,ds)
      endif
      if (output_th.eq.1) then
        do k=1,nk
        do j=1,nj
        do i=1,ni
          ds(i,j,k)=tha(i,j,k)+th0(i,j,k)
        enddo
        enddo
        enddo
        call write3d_nc('th      ',cdfid,ierr,start,count,ni,nj,nk,ds)
      endif
      if (output_thpert.eq.1) then
        do k=1,nk
        do j=1,nj
        do i=1,ni
          ds(i,j,k)=tha(i,j,k)
        enddo
        enddo
        enddo
        call write3d_nc('thpert  ',cdfid,ierr,start,count,ni,nj,nk,ds)
      endif
      if (output_pi.eq.1) then
        do k=1,nk
        do j=1,nj
        do i=1,ni
          ds(i,j,k)=pi0(i,j,k)+ppi(i,j,k)
        enddo
        enddo
        enddo
        call write3d_nc('pii     ',cdfid,ierr,start,count,ni,nj,nk,ds)
      endif
      if (output_pipert.eq.1) then
        do k=1,nk
        do j=1,nj
        do i=1,ni
          ds(i,j,k)=ppi(i,j,k)
        enddo
        enddo
        enddo
        call write3d_nc('ppi     ',cdfid,ierr,start,count,ni,nj,nk,ds)
      endif
      if (output_prs.eq.1) then
        do k=1,nk
        do j=1,nj
        do i=1,ni
          ds(i,j,k)=prs(i,j,k)
        enddo
        enddo
        enddo
        call write3d_nc('prs     ',cdfid,ierr,start,count,ni,nj,nk,ds)
      endif
      if (output_prspert.eq.1) then
        do k=1,nk
        do j=1,nj
        do i=1,ni
          ds(i,j,k)=prs(i,j,k)-p00*pi0(i,j,k)**cpdrd
        enddo
        enddo
        enddo
        call write3d_nc('prspert ',cdfid,ierr,start,count,ni,nj,nk,ds)
      endif
      if (imoist.eq.1) then
        if (output_qv.eq.1) then
          do k=1,nk
          do j=1,nj
          do i=1,ni
            ds(i,j,k)=qa(i,j,k,nqv)
          enddo
          enddo
          enddo
          write(chid(1:3),110) qname(nqv)
          call write3d_nc(chid,cdfid,ierr,start,count,ni,nj,nk,ds)
        endif
        if (output_qvpert.eq.1) then
          do k=1,nk
          do j=1,nj
          do i=1,ni
            ds(i,j,k)=qa(i,j,k,nqv)-qv0(i,j,k)
          enddo
          enddo
          enddo
          call write3d_nc('qvpert  ',cdfid,ierr,start,count,ni,nj,nk,ds)
        endif
        if (output_q.eq.1) then
          do n=1,numq
            if(n.ne.nqv)then
              do k=1,nk
              do j=1,nj
              do i=1,ni
                ds(i,j,k)=qa(i,j,k,n)
              enddo
              enddo
              enddo
              write(chid(1:3),110) qname(n)
110           format(a3)
              call write3d_nc(chid,cdfid,ierr,start,count,ni,nj,nk,ds)
            endif
          enddo
        endif
        if(iptra.eq.1)then
          do n=1,npt
            do k=1,nk
            do j=1,nj
            do i=1,ni
              ds(i,j,k)=pta(i,j,k,n)
            enddo
            enddo
            enddo
            chid = 'pt      '
            write(chid(3:3),111) n
111         format(i1)
            call write3d_nc(chid,cdfid,ierr,start,count,ni,nj,nk,ds)
          enddo
        endif
      endif

      IF (iturb.gt.0) THEN
!-------------
        if (output_km.eq.1) then
            do k=1,nk+1
            do j=1,nj
            do i=1,ni
              dw(i,j,k)=kmh(i,j,k)
            enddo
            enddo
            enddo
            call write3d_nc('kmh     ',cdfid,ierr,start,countzp,ni,nj,nk+1,dw)
            do k=1,nk+1
            do j=1,nj
            do i=1,ni
              dw(i,j,k)=kmv(i,j,k)
            enddo
            enddo
            enddo
            call write3d_nc('kmv     ',cdfid,ierr,start,countzp,ni,nj,nk+1,dw)
        endif
!-------------
        if (output_kh.eq.1) then
            do k=1,nk+1
            do j=1,nj
            do i=1,ni
              dw(i,j,k)=khh(i,j,k)
            enddo
            enddo
            enddo
            call write3d_nc('khh     ',cdfid,ierr,start,countzp,ni,nj,nk+1,dw)
            do k=1,nk+1
            do j=1,nj
            do i=1,ni
              dw(i,j,k)=khv(i,j,k)
            enddo
            enddo
            enddo
            call write3d_nc('khv     ',cdfid,ierr,start,countzp,ni,nj,nk+1,dw)
        endif
!-------------
        if ((iturb.eq.1).and.(output_tke.eq.1)) then
            do k=1,nk+1
            do j=1,nj
            do i=1,ni
              dw(i,j,k)=tkea(i,j,k)
            enddo
            enddo
            enddo
            call write3d_nc('tke     ',cdfid,ierr,start,countzp,ni,nj,nk+1,dw)
        endif
!-------------
      ENDIF

      if (output_uinterp.eq.1) then
        do k=1,nk
        do j=1,nj
        do i=1,ni
          ds(i,j,k)=0.5*(ua(i,j,k)+ua(i+1,j,k))
        enddo
        enddo
        enddo
        call write3d_nc('uinterp ',cdfid,ierr,start,count,ni,nj,nk,ds)
      endif
      if (output_vinterp.eq.1) then
        do k=1,nk
        do j=1,nj
        do i=1,ni
          ds(i,j,k)=0.5*(va(i,j,k)+va(i,j+1,k))
        enddo
        enddo
        enddo
        call write3d_nc('vinterp ',cdfid,ierr,start,count,ni,nj,nk,ds)
      endif
      if (output_winterp.eq.1) then
        do k=1,nk
        do j=1,nj
        do i=1,ni
          ds(i,j,k)=0.5*(wa(i,j,k)+wa(i,j,k+1))
        enddo
        enddo
        enddo
        call write3d_nc('winterp ',cdfid,ierr,start,count,ni,nj,nk,ds)
      endif

      call ncclos(cdfid, ierr)
#endif

      write(outfile,*)
      write(outfile,*) 'Done Writing Data to File: nwrite=',nwrite
      write(outfile,*)

      return
      end


#ifdef NETCDF
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

      subroutine write2d_nc(chid,cdfid,ierr,start,count,numi,numj,d2d)
      implicit none

      include 'netcdf.inc'

      character*8 :: chid
      integer :: cdfid,ierr
      integer, dimension(3) :: start,count
      integer :: numi,numj
      real, dimension(numi,numj) :: d2d

      integer :: id

!----------------------------------

      id = ncvid(cdfid, chid , ierr)
      call ncvpt(cdfid, id , start , count , d2d ,ierr)

!----------------------------------

      return
      end

!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc


      subroutine write3d_nc(chid,cdfid,ierr,start,count,numi,numj,numk,d3d)
      implicit none

      include 'netcdf.inc'

      character*8 :: chid
      integer :: cdfid,ierr
      integer, dimension(4) :: start,count
      integer :: numi,numj,numk
      real, dimension(numi,numj,numk) :: d3d

      integer :: id

!----------------------------------

      id = ncvid(cdfid, chid , ierr)
      call ncvpt(cdfid, id , start , count , d3d ,ierr)

!----------------------------------

      return
      end
#endif
