C******************************************************************************
C PADCIRC RELEASE VERSION 43.03 05/20/2003                                    *
C last changes in this file VERSION 41.11                                     *
C                                                                             *
C  mod history                                                                *
C  v43.03     - 05/20/03 - rl - from 43.02 - parallel wind stuff (m.brown)    *
C                                          output buffer flush (m.cobb)       *
C                                          3D fixes (k.dresback)              *
C                                          drop MNPROC in fort.15 (t.campbell)*
C                                          various bug fixes in RBCs          *
c                                          ZSURFBUOY/BCPG calc                *
C  v41.11 - 09/14/01 - rl - from 41.09 - elim MNWLON,MNWLAT from NWS11GET     *
C  v41.09 - 06/30/01 - jw - from 41.08 - minor mods per vp version 41.05      *
C  v41.06 - 04/02/01 - rl - corrected dimensioning on most of wind reading    *
C                          subroutines.  Eliminated MNWP.  Note MNWLON and    *
C                          MNWLAT still need to be eliminated from NWS11GET   *
C                          subroutine.                                        *
C******************************************************************************
C
       MODULE WIND
       USE SIZES

C
C
C***********************************************************************
C                                                                      *
C   THE FOLLOWING SUBROUTINES READ IN AND IN SOME CASES INTERPOLATE    *
C   ONTO THE ADCIRC GRID WIND AND PRESSURE FIELDS IN VARIOUS INPUT     *
C   FORMATS.                                                           *
C                                                                      *
C   ALL WIND SPEEDS ARE CONVERTED TO M/S AND ALL PRESSURES TO M OF H20 *
C   BEFORE THEY ARE RETURNED.                                          *
C                                                                      *
C***********************************************************************
C
      REAL(8),PRIVATE,PARAMETER :: PI=3.141592653589793D0, 
     &                             TWOPI=PI*2.D0,
     &                             HFPI=PI/2.D0,
     &                             RAD2DEG = 180.D0/PI,
     &                             DEG2RAD = PI/180.D0

C------------------------end of data declarations______________________________C


       CONTAINS


C***********************************************************************
C                                                                      *
C   Convert time from year,month,day,hour,min,sec into seconds since   *
C   the beginning of the year.                                         *
C                                                                      *
C***********************************************************************

      SUBROUTINE TIMECONV(IYR,IMO,IDAY,IHR,IMIN,SEC,TIMESEC)
      IMPLICIT NONE
      INTEGER IYR,IMO,IDAY,IHR,IMIN,ILEAP
      REAL*8 TIMESEC,SEC
C
      TIMESEC = (IDAY-1)*86400 + IHR*3600 + IMIN*60 + SEC
      IF(IMO.GE.2)  TIMESEC = TIMESEC + 31*86400
      ILEAP = (IYR/4)*4
      IF((ILEAP.EQ.IYR).AND.(IMO.GE.3)) TIMESEC = TIMESEC + 29*86400
      IF((ILEAP.NE.IYR).AND.(IMO.GE.3)) TIMESEC = TIMESEC + 28*86400
      IF(IMO.GE.4)  TIMESEC = TIMESEC + 31*86400
      IF(IMO.GE.5)  TIMESEC = TIMESEC + 30*86400
      IF(IMO.GE.6)  TIMESEC = TIMESEC + 31*86400
      IF(IMO.GE.7)  TIMESEC = TIMESEC + 30*86400
      IF(IMO.GE.8)  TIMESEC = TIMESEC + 31*86400
      IF(IMO.GE.9)  TIMESEC = TIMESEC + 31*86400
      IF(IMO.GE.10) TIMESEC = TIMESEC + 30*86400
      IF(IMO.GE.11) TIMESEC = TIMESEC + 31*86400
      IF(IMO.EQ.12) TIMESEC = TIMESEC + 30*86400
      IF(IMO.GT.12) THEN
        WRITE(6,*) 'FATAL ERROR IN SUBROUTINE TIMECONV - MONTH > 12 '
        WRITE(16,*) 'FATAL ERROR IN SUBROUTINE TIMECONV - MONTH > 12 '
        STOP
        ENDIF
      RETURN
      END SUBROUTINE

C***********************************************************************
C                                                                      *
C   READ IN AND INTERPOLATE ONTO THE ADCIRC GRID WIND FIELDS FROM U.S. *
C   NAVY FLEET NUMERIC WIND FILES.                                     *
C                                                                      *
C   NOTE: The ADCIRC grid information consists only of the Lon and Lat *
C   of the nodes.  THE LONS AND LATS MUST BE IN RADIANS!               *
C                                                                      *
C                                                                      *
C   NWLAT = MAXIMUM NUMBER OF LATITUDES IN FLEET NUMERIC WIND FILE     *
C            SET = 1 IF FLEET NUMERIC WIND FILE NOT IN USE             *
C   NWLON = MAXIMUM NUMBER OF LONGITUDES IN FLEET NUMERIC WIND FILE    *
C            SET = 1 IF FLEET NUMERIC WIND FILE NOT IN USE             *
C                                                                      *
C                        R.L. 4/17/96                                  *
C                                                                      *
C   R.L. 4/2/01  changed MNWLAT,MNWLON in ALLOCATE statement to        *
C                NWLAT,NWLON                                           *
C***********************************************************************

      SUBROUTINE NWS3GET(X,Y,SLAM,SFEA,WVNX,WVNY,IWTIME,IWYR,WTIMED,NP,
     &                  NWLON,NWLAT,WLATMAX,WLONMIN,WLATINC,WLONINC,ICS)
      USE SIZES
      IMPLICIT NONE
      INTEGER, SAVE :: FIRSTCALL = 0
      INTEGER IWTIME,IWYR,IWMO,IWDAY,IWHR,NP,NWLON,NWLAT,ICS,I,J
      REAL*8 WTIMED
      REAL*8 X(*),Y(*),SLAM(*),SFEA(*),XCOOR,YCOOR
      INTEGER  LATIND1,LATIND2,LONIND1,LONIND2
      REAL(SZ) WLATMAX,WLONMIN,WLATINC,WLONINC,WSPEED,WDIR
      REAL(SZ) WLATM,WLONM,XWRATIO,YWRATIO
      REAL(SZ),ALLOCATABLE,SAVE :: WVXFN(:,:),WVYFN(:,:),PRN(:,:)
      REAL(SZ) WVNX(*),WVNY(*)
C
      IF (FIRSTCALL.EQ.0) THEN
         FIRSTCALL = 1
         ALLOCATE ( WVXFN(NWLAT,NWLON),WVYFN(NWLAT,NWLON),
     &             PRN(NWLAT,NWLON) )
      ENDIF
C
      READ(22,*) IWTIME
      IWYR = IWTIME/1000000
      IWMO = IWTIME/10000 - IWYR*100
      IWDAY = IWTIME/100 - IWYR*10000 - IWMO*100
      IWHR = IWTIME - IWYR*1000000 - IWMO*10000 - IWDAY*100
      CALL TIMECONV(IWYR,IWMO,IWDAY,IWHR,0,0.0D0,WTIMED)
C
      DO I=1,NWLAT
         READ(22,*) (WVXFN(I,J),J=1,NWLON)
      END DO
      DO I=1,NWLAT
         READ(22,*) (WVYFN(I,J),J=1,NWLON)
      END DO
C
      DO I=1,NWLAT              !CONVERT TO X AND Y COMPONENTS
         DO J=1,NWLON
            WSPEED=WVXFN(I,J)
            WDIR=WVYFN(I,J)*DEG2RAD
            WVXFN(I,J)=-WSPEED*SIN(WDIR)
            WVYFN(I,J)=-WSPEED*COS(WDIR)
         END DO
      END DO
      
      DO I=1,NP                 !INTERPOLATE TO ADCIRC GRID
         IF(ICS.EQ.2) THEN
            YCOOR=SFEA(I)*RAD2DEG
            XCOOR=SLAM(I)*RAD2DEG
         ENDIF
         IF(ICS.EQ.1) THEN
            YCOOR=Y(I)
            XCOOR=X(I)
         ENDIF
         LATIND2=(WLATMAX-YCOOR)/WLATINC + 1
         IF(LATIND2.EQ.NWLAT) LATIND2=LATIND2-1
         LATIND1=LATIND2 + 1
         LONIND1=(XCOOR-WLONMIN)/WLONINC + 1
         IF(LONIND1.EQ.NWLON) LONIND1=LONIND1-1
         LONIND2=LONIND1+1
         WLONM = WLONMIN + (LONIND1-1)*WLONINC
         WLATM = WLATMAX - (LATIND1-1)*WLATINC
         XWRATIO=(XCOOR-WLONM)/WLONINC
         YWRATIO=(YCOOR-WLATM)/WLATINC
C     
         WVNX(I) = WVXFN(LATIND2,LONIND2)*XWRATIO*YWRATIO
     &        + WVXFN(LATIND2,LONIND1)*(1.d0-XWRATIO)*YWRATIO
     &        + WVXFN(LATIND1,LONIND2)*XWRATIO*(1.d0-YWRATIO)
     &        + WVXFN(LATIND1,LONIND1)*(1.d0-XWRATIO)*(1.d0-YWRATIO)
         WVNY(I) = WVYFN(LATIND2,LONIND2)*XWRATIO*YWRATIO
     &        + WVYFN(LATIND2,LONIND1)*(1.d0-XWRATIO)*YWRATIO
     &        + WVYFN(LATIND1,LONIND2)*XWRATIO*(1.d0-YWRATIO)
     &        + WVYFN(LATIND1,LONIND1)*(1.d0-XWRATIO)*(1.d0-YWRATIO)
      END DO
C     
      RETURN
      END SUBROUTINE



C***********************************************************************
C                                                                      *
C   Read onto the ADCIRC grid wind fields from the PBL-JAG model       *
C                                                                      *
C   Output from this subroutine is U,V (M/S) and P (M H20) on the      *
C   ADCIRC grid.                                                       *
C                                                                      *
C   The background pressure is assumed to be 1013 Mbars                *
C                                                                      *
C                           R.L.11/06/96                               *
C   R.L.09/04/00 added RHOWAT0 to call                                 *
C   R.L. 4/2/01  changed MNP dimensions to *                           *
C   R.L. 3/15/03 accounted for PRN=0                                   *   !RAL0315+ OK
C***********************************************************************

      SUBROUTINE NWS4GET(WVNX,WVNY,PRN,NP,RHOWAT0,G)
      USE SIZES
      IMPLICIT NONE
      INTEGER   NP,I,NHG
      REAL(SZ)  WVNX(*),WVNY(*),PRN(*)
      REAL(SZ)  RHOWAT0,RHOWATG,G
      CHARACTER*80 PBLJAGF
C
      RHOWATG=RHOWAT0*G
      DO I=1,NP
        WVNX(I)=0.d0
        WVNY(I)=0.d0
        PRN(I)=101300.d0/RHOWATG
      END DO
 170  READ(22,'(A80)') PBLJAGF
      IF(PBLJAGF(2:2).EQ.'#') GOTO 170
 171  READ(PBLJAGF,'(I8,5E13.5)') NHG,WVNX(NHG),WVNY(NHG),
     &                            PRN(NHG)
      WVNX(NHG)=WVNX(NHG)*1.04d0*0.5144d0 !CONVERT 30-MIN WINDS IN
      WVNY(NHG)=WVNY(NHG)*1.04d0*0.5144d0 !KNOTS TO 10-MIN WIND IN M/S
      PRN(NHG)=100.d0*PRN(NHG)/RHOWATG !CONVERT MILLIBARS TO M OF WATER
      IF(PRN(NHG).EQ.0.) PRN(NHG)=101300.d0/RHOWATG                       !RAL0315+ OK
      READ(22,'(A80)') PBLJAGF
      IF(PBLJAGF(2:2).NE.'#') GOTO 171
      RETURN
      END SUBROUTINE


C***********************************************************************
C                                                                      *
C   READ IN AND INTERPOLATE ONTO THE ADCIRC GRID WIND AND PRESSURE     *
C   FIELDS FROM A MET FILE ON A RECTILINEAR GRID.                      *
C                                                                      *
C   NOTE: The ADCIRC grid information consists only of the Lon and Lat *
C   of the nodes.  THE LONS AND LATS MUST BE IN RADIANS!               *
C                                                                      *
C   NOTE:  It is assumed that the met file data is oriented so that    *
C          the outer loop is on latitude and the inner loop is on      *
C          longitude.  For example:                                    *
C          line 1             lat 1,     lon 1                         *
C          line 2             lat 1,     lon 2                         *
C            .                                                         *
C          line nwlon         lat 1,     lon nwlon                     *
C          line nwlon+1       lat 2,     lon 1                         *
C          line nwlon+2       lat 2,     lon 2                         *
C            .                                                         *
C          line 2*nwlon       lat 2,     lon nwlon                     *
C          line 2*nwlon+1     lat 3,     lon 1                         *
C          line 2*nwlon+2     lat 3,     lon 2                         *
C            .                                                         *
C          line nwlon*nwlat   lat nwlat, lon nwlon                     *
C                                                                      *
C   NOTE:  It is assumed that he met file data is oriented so that     *
C          latitude varies from the northern most value (lat 1) to the *
C          southern most value (lat nwlat) and longitude varies in an  *
C          easterly direction (e.g. from 0 to 360 where positive       *
C          longitudes are angles measured easterly of the GM.          *
C                                                                      *
C   NOTE:  For the global AVN grid running from 0.5 - 359.5 deg lon    *
C          and 90 - -90 deg lat in 1 degree increments, NWLAT=181 and  *
C          NWLON=360 yielding a total number of entries in the file    *
C          of 65160.                                                   *    
C                                                                      *
C   NOTE:  It is assumed that wind velocity is in EAST,NORTH components*
C          in M/2 and pressure is in N/M^2                             *
C                                                                      *
C   NOTE:  WLATMAX,WLONMIN,WLATINC,WLONINC should be in deg.           *
C                                                                      *
C   NOTE:  This should wrap if XCOORD > WLONMIN+NWLON*WLONINC  or      *
C          XCOORD < WLONMIN                                            *
C                                                                      *
C                                                                      *
C   MNWLAT = MAXIMUM NUMBER OF LATITUDES IN WIND FILE                  *
C            SET = 1 IF FLEET NUMERIC WIND FILE NOT IN USE             *
C   MNWLON = MAXIMUM NUMBER OF LONGITUDES IN WIND FILE                 *
C            SET = 1 IF FLEET NUMERIC WIND FILE NOT IN USE             *
C                                                                      *
C                           R.L. 4/13/99                               *
C                           R.L.09/04/00 added RHOWAT0 to call         *
C   R.L.09/04/00 added RHOWAT0 to call                                 *
C   R.L. 4/2/01  changed MNWLAT,MNWLON in ALLOCATE statement to        *
C                NWLAT,NWLON                                           *
C***********************************************************************

      SUBROUTINE NWS6GET(X,Y,SLAM,SFEA,WVNX,WVNY,PRESS,NP,NWLON,NWLAT,
     &     WLATMAX,WLONMIN,WLATINC,WLONINC,ICS,RHOWAT0,G)
      USE SIZES

      IMPLICIT NONE
      INTEGER, SAVE :: FIRSTCALL = 0
      INTEGER NP,NWLON,NWLAT,I,J,ICS 
      REAL(SZ) RHOWAT0,RHOWATG,G
      INTEGER  LATIND1,LATIND2,LONIND1,LONIND2
      REAL(SZ) WLATMAX,WLONMIN,WLATINC,WLONINC,XWRATIO,YWRATIO
      REAL(SZ) WLATM,WLONM
      REAL*8 X(*),Y(*),SLAM(*),SFEA(*),XCOOR,YCOOR
      REAL(SZ) WVNX(*),WVNY(*),PRESS(*)
      REAL(SZ),SAVE,ALLOCATABLE :: WVXFN(:,:),WVYFN(:,:),PRN(:,:)
C     
      IF (FIRSTCALL.EQ.0) THEN
         FIRSTCALL = 1
         ALLOCATE ( WVXFN(NWLAT,NWLON),WVYFN(NWLAT,NWLON),
     &        PRN(NWLAT,NWLON) )
      ENDIF
C     
      RHOWATG=RHOWAT0*G
      DO I=1,NWLAT
         DO J=1,NWLON
            READ(22,*) PRN(I,J),WVXFN(I,J),WVYFN(I,J)
         END DO
      END DO

      DO I=1,NP                 !INTERPOLATE TO ADCIRC GRID
         IF(ICS.EQ.2) THEN
            YCOOR=SFEA(I)*RAD2DEG
            XCOOR=SLAM(I)*RAD2DEG
            IF(XCOOR.LT.0.) XCOOR=XCOOR+360.d0
         ENDIF
         IF(ICS.EQ.1) THEN
            YCOOR=Y(I)
            XCOOR=X(I)
         ENDIF
         LATIND2=(WLATMAX-YCOOR)/WLATINC + 1
         IF(LATIND2.EQ.NWLAT) LATIND2=LATIND2-1
         LATIND1=LATIND2 + 1
         LONIND1=(XCOOR-WLONMIN)/WLONINC + 1
         LONIND2=LONIND1 + 1
C     
         WLONM = WLONMIN + (LONIND1-1)*WLONINC 
         WLATM = WLATMAX - (LATIND1-1)*WLATINC
         XWRATIO=(XCOOR-WLONM)/WLONINC
         YWRATIO=(YCOOR-WLATM)/WLATINC
C     
         IF(LONIND1.EQ.0) LONIND1=NWLON
         IF(LONIND1.EQ.NWLON) LONIND2=1
C     
         WVNX(I) = WVXFN(LATIND2,LONIND2)*XWRATIO*YWRATIO
     &        + WVXFN(LATIND2,LONIND1)*(1.d0-XWRATIO)*YWRATIO
     &        + WVXFN(LATIND1,LONIND2)*XWRATIO*(1.d0-YWRATIO)
     &        + WVXFN(LATIND1,LONIND1)*(1.d0-XWRATIO)*(1.d0-YWRATIO)
         WVNY(I) = WVYFN(LATIND2,LONIND2)*XWRATIO*YWRATIO
     &        + WVYFN(LATIND2,LONIND1)*(1.d0-XWRATIO)*YWRATIO
     &        + WVYFN(LATIND1,LONIND2)*XWRATIO*(1.d0-YWRATIO)
     &        + WVYFN(LATIND1,LONIND1)*(1.d0-XWRATIO)*(1.d0-YWRATIO)
         PRESS(I) = PRN(LATIND2,LONIND2)*XWRATIO*YWRATIO
     &        + PRN(LATIND2,LONIND1)*(1.d0-XWRATIO)*YWRATIO
     &        + PRN(LATIND1,LONIND2)*XWRATIO*(1.d0-YWRATIO)
     &        + PRN(LATIND1,LONIND1)*(1.d0-XWRATIO)*(1.d0-YWRATIO)
         PRESS(I) = PRESS(I)/RHOWATG

      END DO
C     
      RETURN
      END SUBROUTINE


C***********************************************************************
C                                                                      *
C   Read in and interpolate onto the ADCIRC grid wind fields from U.S. *
C   National Weather Service AVN model SFLUX meteorological files.     *
C                                                                      *
C   The input files are in binary and have been created by the GRIB    *
C   unpacking program unpkgrb1.f to extract only the U 10M, V 10M, and *
C   surface P fields.    THE BINARY INPUT HAS BEEN ELIMINATED!!!!      *
C   The input files are in ASCII and contain surface P, U 10M and V 10M*
C   fields.                                                            *
C                                                                      *
C   The SFLUX files utilize a global Gaussian Lon/Lat grid which is    *
C   constructed in these subroutines.                                  *
C                                                                      *
C   NOTE: The ADCIRC grid information consists only of the Lon and Lat *
C   of the nodes.  THE LONS AND LATS MUST BE IN RADIANS!               *
C                                                                      *
C   Output from this subroutine is U,V (M/S) and P (M H20) on the      *
C   ADCIRC grid.                                                       *
C                                                                      *
C   MNWLAT = LATB = 190    FOR GAUSSIAN GRID                           *
C   MNWLON = LONB = 384    FOR GAUSSIAN GRID                           *
C                                                                      *
C                           R.L. 4/14/99                               *
C                           R.L.09/04/00 added RHOWAT0 to call         *
C   R.L. 4/2/01  changed MNWLAT,MNWLON in ALLOCATE statement to        *
C                LATB,LONB; elminiated MNWP as a dimension             *
C***********************************************************************

      SUBROUTINE NWS10GET(NWSGGWI,FLON,FLAT,ULL,VLL,PLL,NP,RHOWAT0,G,
     &     LONB,LATB,WTIMINC)
      USE SIZES
      IMPLICIT NONE
      INTEGER, SAVE :: FIRSTCALL = 0
      INTEGER N,NP,NWSGGWI,LONB,LATB,I,J,JJ,IEXT,IDIG1,IDIG2,
     &     IDIG3,KERR
      REAL*8 WTIMINC
      REAL*8 FLAT(*),FLON(*)
      REAL(SZ)  ULL(*),VLL(*),PLL(*)
      REAL(SZ) RHOWAT0,RHOWATG,G,GDLON,P1,P2,P3,P4,U1,U2,U3,U4,
     &     V1,V2,V3,V4
      INTEGER KGDS(200)
      INTEGER,SAVE,ALLOCATABLE ::  N00(:),N10(:),N11(:),N01(:)
      REAL(SZ),SAVE,ALLOCATABLE :: D00(:),D10(:),D11(:),D01(:)
      REAL(SZ),SAVE,ALLOCATABLE :: COLRAB(:),DUMMY(:),
     &     GCLAT(:),GCLON(:)
      REAL(SZ),SAVE,ALLOCATABLE ::  UG(:),VG(:),PG(:)
      CHARACTER*1 PDS(50),FNAME2(8)
      CHARACTER*8 FNAME1
      EQUIVALENCE (FNAME1,FNAME2)
      LOGICAL FOUND
C     
      IF (FIRSTCALL.EQ.0) THEN
         FIRSTCALL = 1
         ALLOCATE ( UG(LATB*LONB),VG(LATB*LONB),
     &        PG(LATB*LONB) )
         ALLOCATE ( N00(MNP),N10(MNP),N11(MNP),N01(MNP) )
         ALLOCATE ( D00(MNP),D10(MNP),D11(MNP),D01(MNP) )
         ALLOCATE ( COLRAB(LATB),DUMMY(LATB),GCLAT(LATB),
     &        GCLON(LONB) )
      ENDIF
C     
      RHOWATG=RHOWAT0*G
C     
C...  The first time the subroutine is called, setup the Gaussian grid and
C...  determine the interpolating factors for the ADCIRC grid.
C     
      IF (NWSGGWI.EQ.-1) THEN
         CALL GLATS(LATB/2,COLRAB,DUMMY,DUMMY,DUMMY)
         DO J=1,LATB/2
            GCLAT(J)=COLRAB(J)
            JJ=LATB-J+1
            GCLAT(JJ)=PI-COLRAB(J)
         ENDDO
         GDLON=TWOPI/LONB
         DO J=1,LONB
            GCLON(J)=GDLON*(J-1)
         END DO
         CALL G2RINI(GCLON,GCLAT,FLON,FLAT,N00,N10,N11,N01,D00,D10,D11,
     &        D01,NP,LONB,LATB)
         RETURN
      ENDIF

C...  Figure out the data file name

      FNAME1='fort.   '
      IEXT=200 + NWSGGWI*(WTIMINC/3600)
      IDIG1=IEXT/100
      IDIG2=(IEXT-100*IDIG1)/10
      IDIG3=(IEXT-100*IDIG1-10*IDIG2)
      FNAME2(6)=CHAR(IDIG1+48)
      FNAME2(7)=CHAR(IDIG2+48)
      FNAME2(8)=CHAR(IDIG3+48)


C...  Enter, locate and open the data file

 1010 FORMAT(' File ',A8,' WAS NOT FOUND!  FATAL ERROR',/)
 1011 FORMAT(' File ',A8,' WAS FOUND!  Opening & Processing file',/)

      WRITE(*,*) '  '
      INQUIRE(FILE=FNAME1,EXIST=FOUND)
      IF(FOUND) GOTO 32
      WRITE(*,1010) FNAME1
      WRITE(16,1010) FNAME1
      STOP
 32   WRITE(*,1011) FNAME1

C...Open and read the GRIB BINARY data file
c     OPEN(IEXT,FILE=FNAME1,status='old',access='sequential',
c    &     form='unformatted',iostat=kerr)
c     READ(IEXT,END=1100) LENPDS,LENKGDS,NWORDS
c     IF(LENPDS.GT.0) READ(IEXT,END=1100) (pds(j),j=1,lenpds)
c     IF(LENKGDS.GT.0) READ(IEXT,END=1100) (kgds(j),j=1,lenkgds)
c     IF(NWORDS.GT.0) READ(IEXT,END=1100) (UG(J),J=1,NWORDS)
c
c     READ(IEXT,END=1100) LENPDS,LENKGDS,NWORDS
c     IF(LENPDS.GT.0) READ(IEXT,END=1100) (pds(j),j=1,lenpds)
c     IF(LENKGDS.GT.0) READ(IEXT,END=1100) (kgds(j),j=1,lenkgds)
c     IF(NWORDS.GT.0) READ(IEXT,END=1100) (VG(J),J=1,NWORDS)
c
c     READ(IEXT,END=1100) LENPDS,LENKGDS,NWORDS
c     IF(LENPDS.GT.0) READ(IEXT,END=1100) (pds(j),j=1,lenpds)
c     IF(LENKGDS.GT.0) READ(IEXT,END=1100) (kgds(j),j=1,lenkgds)
c     IF(NWORDS.GT.0) READ(IEXT,END=1100) (PG(J),J=1,NWORDS)

C...Open and read the ASCII data file

      OPEN(IEXT,FILE=FNAME1,status='old',iostat=kerr)
      DO I=1,LONB*LATB
         READ(IEXT,*) PG(I),UG(I),VG(I)
      ENDDO

 1100 CLOSE(IEXT)


C.....Go from the Gaussian grid to the ADCIRC grid
C.....Convert pressure from N/M^2 to M of H20

      DO N=1,NP
         P1=PG(N00(N))
         P2=PG(N10(N))
         P3=PG(N11(N))
         P4=PG(N01(N))
         U1=UG(N00(N))
         U2=UG(N10(N))
         U3=UG(N11(N))
         U4=UG(N01(N))
         V1=VG(N00(N))
         V2=VG(N10(N))
         V3=VG(N11(N))
         V4=VG(N01(N))
         PLL(N)=P1*D00(N)+P2*D10(N)+P3*D11(N)+P4*D01(N)
         ULL(N)=U1*D00(N)+U2*D10(N)+U3*D11(N)+U4*D01(N)
         VLL(N)=V1*D00(N)+V2*D10(N)+V3*D11(N)+V4*D01(N)
         PLL(N)=PLL(N)/RHOWATG
      END DO
C     
      RETURN
      END SUBROUTINE


C***********************************************************************
C  Subroutine to compute the latutudes in a Global Gaussian Lat/Lon    *
C  grid with T126 resolution (GRIB Grid type 126).                     *
C                                                                      *
C       modified from the original GLATS by R.L. 4/24/96               *
C***********************************************************************

      SUBROUTINE GLATS(LGGHAF,COLRAD,WGT,WGTCS,RCS2)
      USE SIZES
      IMPLICIT NONE
      REAL(SZ) COLRAD(*),WGT(*),WGTCS(*),RCS2(*)
      INTEGER LGGHAF,L2,K,K1,ITER
      REAL(SZ) SI,SCALE,RL2,DRAD,DRADZ,RAD,P1,P2,EPS,PHI,X,W,SN,RC
C     
      EPS=1.d-6
C     EPS=1.d-12
C     PRINT 101
C     101  FORMAT ('0 I   COLAT   COLRAD     WGT', 12X, 'WGTCS',
CCCC  1 10X, 'ITER  RES')
C     
      SI = 1.0d0
      L2=2*LGGHAF
      RL2=L2
      SCALE = 2.0d0/(RL2*RL2)
      K1=L2-1
      DRADZ = PI / 360.d0
      RAD = 0.0
      DO 1000 K=1,LGGHAF
         ITER=0
         DRAD=DRADZ
 1       CALL POLY(L2,RAD,P2)
 2       P1 =P2
         ITER=ITER+1
         RAD=RAD+DRAD
         CALL POLY(L2,RAD,P2)
         IF(SIGN(SI,P1).EQ.SIGN(SI,P2)) GO TO 2
         IF(DRAD.LT.EPS)GO TO 3
         RAD=RAD-DRAD
         DRAD = DRAD * 0.25d0
         GO TO 1
 3       CONTINUE
         COLRAD(K)=RAD
         PHI = RAD * 180.d0 / PI
         CALL POLY(K1,RAD,P1)
         X = COS(RAD)
         W = SCALE * (1.0d0 - X*X)/ (P1*P1)
         WGT(K) = W
         SN = SIN(RAD)
         W=W/(SN*SN)
         WGTCS(K) = W
         RC=1.d0/(SN*SN)
         RCS2(K) = RC
         CALL POLY(L2,RAD,P1)
C     PRINT 102,K,PHI,COLRAD(K),WGT(K),WGTCS(K),ITER,P1
C     102  FORMAT(1H ,I2,2X,F6.2,2X,F10.7,2X,E13.7,2X,E13.7,2X,I4,2X,D13.7)
 1000 CONTINUE
c     PRINT 100,LGGHAF
c     100  FORMAT(1H ,'SHALOM FROM 0.0 GLATS FOR ',I3)
      RETURN
      END SUBROUTINE


C***********************************************************************
C  Subroutine used by GLATS.                                           *
C***********************************************************************

      SUBROUTINE POLY(N,RAD,P)
      USE SIZES
      IMPLICIT NONE
      INTEGER N,I
      REAL(SZ) RAD,P,X,Y1,Y2,Y3,G
C     
      X = COS(RAD)
      Y1 = 1.0d0
      Y2=X
      DO 1 I=2,N
         G=X*Y2
         Y3=G-Y1+G-(G-Y1)/FLOAT(I)
         Y1=Y2
         Y2=Y3
 1    CONTINUE
      P=Y3
      RETURN
      END SUBROUTINE

C***********************************************************************
C  Subroutine to compute the factors to interpolate from a global      *
C  Gaussian Lat/Lon grid with T126 resolution (GRIB Grid type 126)     *
C  onto another grid.                                                  *
C                                                                      *
C  The new grid is a series of longitude and latitude points contained *
C  in the FLON and FLAT arrays with a total number of points NP        *
C                                                                      *
C       modified from the original G2RINI by R.L. 4/17/96              *
C***********************************************************************

      SUBROUTINE G2RINI(GCLON,GCLAT,FLON,FLAT,N00,N10,N11,N01,D00,D10,
     &     D11,D01,NP,LONB,LATB)
      USE SIZES
      IMPLICIT NONE
      INTEGER,SAVE :: ICALL = 0
      INTEGER NP,N,I,LONB,LATB,NLAT,NLON,LON,LONP1,LAT,LATP1
      REAL*8 DLAT,DLON,FLONWORK,COLAT,DDLAT,XLAT,DFLAT,DFLAT1,
     &     DDLON,XLON,DFLON,DFLON1
      REAL*8 FLAT(*),FLON(*)
      REAL(SZ) GCLAT(*),GCLON(*)
      INTEGER  N00(*),N10(*),N11(*),N01(*)
      REAL(SZ) D00(*),D10(*),D11(*),D01(*)
C     
      IF( ICALL .EQ. 0 ) THEN
         ICALL = 1
c       PRINT 1234
c1234   FORMAT(' = IN ROUTINE G2RINI FOR HORIZONTAL INTERPOLATION = ')

C...Compute estimated DLAT, true DLON for Gaussian grid

         NLAT=LATB
         NLON=LONB
         DLAT=PI/FLOAT(NLAT-1)
         DLON=TWOPI/FLOAT(NLON)
         N=0

C...Loop through all the nodes in the grid to be interpolated onto and
C.....compute the interpolating factors.

         DO I=1,NP
           
C.....Compute initial guess of which lon value FLON(I) is in the Gaussian file
C.......Check that this value is reasonable.

            FLONWORK=FLON(I)
            IF(FLONWORK.LT.0.) FLONWORK=FLONWORK+TWOPI
            LON=FLONWORK/DLON + 1
            LONP1=LON+1
            IF(LON.EQ.NLON) LONP1=1 !Circle condition
            IF((LON.LT.1).OR.(LON.GT.NLON)) THEN
               PRINT *,' ***** ERROR IN LON ****'
               PRINT *,' I ',I
               PRINT *,' LON ',LON
               PRINT *,' DLON ',DLON
               PRINT *,' FLON ',FLON(I)
               STOP
            ENDIF  
            
C.....Compute initial guess of which lat value FLAT(I) is in the Gaussian file
C.......Check that this value is reasonable.

            COLAT=HFPI-FLAT(I)
            LAT=COLAT/DLAT + 1
            IF(LAT.EQ.NLAT) LAT=LAT-1
            LATP1=LAT+1
            IF((LAT.LT.1).OR.(LAT.GT.NLAT)) THEN
               PRINT *,' ***** ERROR IN LAT ****'
               PRINT *,' I ',I
               PRINT *,' LAT ',LAT
               PRINT *,' DLAT ',DLAT
               PRINT *,' FLAT ',FLAT(I)
               STOP
            ENDIF

 5          CONTINUE
        IF((COLAT.GE.GCLAT(LAT)).AND.(COLAT.LE.GCLAT(LATP1))) GO TO 9
            IF(COLAT.LT.GCLAT(LAT)) THEN
               LATP1=LAT
               LAT=LAT-1
               IF(LAT.LE.0) THEN
                  LAT=1
                  LATP1=2
                  GOTO 9
               ENDIF
               GOTO 5
            ENDIF
            IF(COLAT.GT.GCLAT(LATP1)) THEN
               LAT=LAT+1
               LATP1=LAT+1
               IF(LAT.GE.NLAT ) THEN
                  LAT=NLAT-1
                  LATP1=NLAT
                  GOTO 9
               ENDIF
               GOTO 5
            ENDIF
            
 9          CONTINUE
            DDLAT=GCLAT(LATP1)-GCLAT(LAT)
            XLAT=GCLAT(LAT)
            DFLAT1=(COLAT-XLAT)/DDLAT
            IF(LAT.EQ.1) DFLAT1=MAX(0.d0,DFLAT1) !MODIFY THIS FOR POLAR POINTS
            IF(LATP1.EQ.NLAT) DFLAT1=MIN(1.d0,DFLAT1) !MODIFY THIS FOR POLAR POINTS
            DFLAT=1.d0-DFLAT1
            DDLON=DLON
            XLON=GCLON(LON)
            DFLON1=(FLONWORK-XLON)/DDLON
            DFLON=1.d0-DFLON1
            N=N+1
            D00(N)=DFLON*DFLAT
            D10(N)=DFLON1*DFLAT
            D11(N)=DFLON1*DFLAT1
            D01(N)=DFLON*DFLAT1
            N00(N)=LON+(LAT-1)*NLON
            N10(N)=LONP1+(LAT-1)*NLON
            N11(N)=LONP1+(LATP1-1)*NLON
            N01(N)=LON+(LATP1-1)*NLON
            
         END DO
         
c     WRITE(*,*) ' D00 TO D11 SHOULD BE ALL POSITIVE.'
         
      ELSE
c     WRITE(*,*) ' G2RINI ALREADY CALLED '
      ENDIF
      
      RETURN
      END SUBROUTINE


C***********************************************************************
C                                                                      *
C   Read in and interpolate onto the ADCIRC grid wind fields from U.S. *
C   National Weather Service ETA-29 model that have been stripped down *
C   and given to us by NOAA.                                           *
C                                                                      *
C   The input files are in binary and have been created by NOAA and    *
C   contain only the U 10M, V 10M, (M/S) and surface P fields (mbars). *
C                                                                      *
C   The ETA-29 model uses an E grid and therefore the U and V          *
C   components are not oriented along lines of constant latitute and   *
C   longitude. These must be converted to be useful in ADCIRC.         *
C                                                                      *
C   NOTE: The ADCIRC grid information consists only of the Lon and Lat *
C   of the nodes.  THE LONS AND LATS MUST BE IN RADIANS!               *
C                                                                      *
C   Output from this subroutine is U,V (M/S) and P (M H20) on the      *
C   ADCIRC grid.                                                       *
C                                                                      *
C   MNWLAT = LATB = 271    FOR ETA-29 GRID                             *
C   MNWLON = LONB = 181    FOR ETA-29 GRID                             *
C                                                                      *
C                           R.L. 1/11/97                               *
C   R.L.09/04/00 added RHOWAT0 to call                                 *
C   R.L. 4/02/01  elminiated MNWP as a dimension                       *
C   R.L. 9/14/01  changed MNWLAT,MNWLON in ALLOCATE statement to       *
C                271,181                                               *
C***********************************************************************

      SUBROUTINE NWS11GET(NWSEGWI,IDSETFLG,FLON,FLAT,ULL,VLL,PLL,NP,
     &     RHOWAT0,G)
      USE SIZES
      IMPLICIT NONE
      INTEGER,SAVE  ::  ICALL = 0
      INTEGER NWSEGWI,IDSETFLG,NP,I,IEXT,IDIG1,IDIG2,IDIG3,KERR,N
      INTEGER IYEAR,IMONTH,IDAY,IHOUR
      REAL*8 RHOWATG100,FLONDEG,FLATDEG
      REAL(SZ) P1,P2,P3,U1,U2,U3,V1,V2,V3,UE29,VE29,CBETAU,SBETAU,G
      REAL(SZ) RHOWAT0
      REAL(SZ) ULL(*),VLL(*),PLL(*)
      REAL*8 FLAT(*),FLON(*)
C     
      INTEGER,SAVE,ALLOCATABLE ::  N1(:),N2(:),N3(:)
      REAL(SZ),SAVE,ALLOCATABLE :: D1(:),D2(:),D3(:),BETAU(:)
      REAL(SZ),SAVE,ALLOCATABLE :: UE(:),VE(:),PE(:)
C     
      CHARACTER*1 FNAME2(8)
      CHARACTER*8 FNAME1
      EQUIVALENCE (FNAME1,FNAME2)
      LOGICAL FOUND
C     
      IF (ICALL.EQ.0) THEN
         ICALL = 1
         ALLOCATE ( N1(MNP),N2(MNP),N3(MNP) )
         ALLOCATE ( D1(MNP),D2(MNP),D3(MNP),BETAU(MNP) )
         ALLOCATE ( UE(181*271),VE(181*271),PE(181*271) )
      ENDIF
C     
      RHOWATG100=RHOWAT0*G*100.d0

C...  The first time the subroutine is called, setup the interpolating factors
C...  between the Eta-29 grid and the ADCIRC grid.

      IF((NWSEGWI.EQ.0).AND.(IDSETFLG.EQ.0)) THEN
         WRITE(*,*) 'Computing ETA29 met field interpolating factors'
         DO I=1,NP
            flondeg=rad2deg*flon(i)
            flatdeg=rad2deg*flat(i)
            CALL E29SEARCH(I,FLONDEG,FLATDEG,N1(I),N2(I),N3(I),
     &           D1(I),D2(I),D3(I),betau(i))
         END DO
         RETURN
      ENDIF

C...  Figure out the met data file name

      FNAME1='fort.   '
      IEXT=200 + NWSEGWI
      IDIG1=IEXT/100
      IDIG2=(IEXT-100*IDIG1)/10
      IDIG3=(IEXT-100*IDIG1-10*IDIG2)
      FNAME2(6)=CHAR(IDIG1+48)
      FNAME2(7)=CHAR(IDIG2+48)
      FNAME2(8)=CHAR(IDIG3+48)

C...  If appropriate, enter, locate and open the met data file

 1010 FORMAT(' File ',A8,' WAS NOT FOUND!  FATAL ERROR',/)
 1011 FORMAT(' File ',A8,' WAS FOUND!  Opening & Processing file',/)

      WRITE(*,*) '  '
      INQUIRE(FILE=FNAME1,EXIST=FOUND)
      IF(FOUND) GOTO 32
      WRITE(*,1010) FNAME1
      WRITE(16,1010) FNAME1
      STOP
 32   WRITE(*,1011) FNAME1
      IF((NWSEGWI.EQ.0).OR.(IDSETFLG.EQ.1)) OPEN(IEXT,FILE=FNAME1,
     &status='old',access='sequential',form='unformatted',iostat=kerr)

C...  Read the met data file

      READ(IEXT,END=1100) IYEAR,IMONTH,IDAY,IHOUR
      READ(IEXT,END=1100) UE,VE,PE

      IF(NWSEGWI.EQ.0) THEN     !If the first file, read until the end
         DO I=2,IDSETFLG
            READ(IEXT,END=1100) IYEAR,IMONTH,IDAY,IHOUR
            READ(IEXT,END=1100) UE,VE,PE
         ENDDO
      ENDIF

 1100 IF(IDSETFLG.EQ.8) CLOSE(IEXT)

C.....Interpolate onto ADCIRC grid
C.....Convert velocity from the E grid reference to a lat/lon reference
C.....Convert pressure from millibars to N/M^2 to M of H20

      DO N=1,NP
         P1=PE(N1(N))
         P2=PE(N2(N))
         P3=PE(N3(N))
         U1=UE(N1(N))
         U2=UE(N2(N))
         U3=UE(N3(N))
         V1=VE(N1(N))
         V2=VE(N2(N))
         V3=VE(N3(N))
         UE29=U1*D1(N)+U2*D2(N)+U3*D3(N)
         VE29=V1*D1(N)+V2*D2(N)+V3*D3(N)
         CBETAU=COS(BETAU(N))
         SBETAU=SIN(BETAU(N))
         ULL(N)=UE29*CBETAU - VE29*SBETAU
         VLL(N)=UE29*SBETAU + VE29*CBETAU
         PLL(N)=P1*D1(N)+P2*D2(N)+P3*D3(N)
         PLL(N)=PLL(N)/RHOWATG100
      END DO

      RETURN
      END SUBROUTINE



C***********************************************************************
C  Subroutine to find where a given lon,lat falls in the Eta29 grid,   *
C     determine the interpolating factors to interpolate Eta29 fields  *
C     to that position, and finally to compute the angle to rotate the *
C     Eta29 velocity field to get to a lon, lat coordinated system.    *
C                                                                      *
C                    Written by R.L.       1/12/98                     *
C***********************************************************************

      subroutine e29search(node,FLON,FLAT,NN1,NN2,NN3,DD1,DD2,DD3,betau)
      implicit none
      integer nn1,nn2,nn3,node,icode,nwlon,nwlat,ifflag
      integer i,j,im2,jm2,n,ia,ja,na,ib,jb,nb,ic,jc,nc,id,jd,nd,
     &  ie,je,ne,ig,jg,ng,if
      real(sz) dd1,dd2,dd3,betau,ri,x1,x2,x3,x4,y1,y2,y3,y4
      real(sz) aemin,areas,a1,a2,a3,aa,ae,lambda
      real(8) lamda0,phi0,rphi0,cphi0,sphi0,tphi0,dlamda,dphi,rdlamda,
     &       rdphi,rflat,tflat,sflat,cflat,a,rlamar,cphiicrlamda,phiarg,
     &       rphii,rlamda,ri1,ri2,rj,dgtora,flon,flat
      real(sz) lamda,lamdaa,lamdab,lamdac,lamdad,lamdae,lamdag
      real(sz) phi,phia,phib,phic,phid,phie,phig
c
      icode=0
      nwlon=181
      nwlat=271
      dgtora=deg2rad
      lamda0=-97.0d0
      phi0=41.0d0
      rphi0=dgtora*phi0
      cphi0=cos(rphi0)
      sphi0=sin(rphi0)
      tphi0=tan(rphi0)
      dlamda=7.d0/36.d0
      dphi=5.d0/27.d0
      rdlamda=dgtora*dlamda
      rdphi=dgtora*dphi
c
      rflat=flat*dgtora
        tflat=tan(rflat)
      sflat=sin(rflat)
      cflat=cos(rflat)

c     compute the position of the closest node in the E29 grid

      a=flon-lamda0
      rlamar=cos(a*dgtora)
      cphiicrlamda=(rlamar+tflat*tphi0)*cflat*cphi0
      phiarg=sflat
      rphii=asin((phiarg-sphi0*cphiicrlamda)/cphi0)
      rlamda=acos(cphiicrlamda/cos(rphii))
      if(flon.lt.lamda0) rlamda=-rlamda
c
      ri2=(rlamda/rdlamda+nwlon+1)/2.
      ri1=(rlamda/rdlamda+nwlon)/2.
      rj=rphii/rdphi+(nwlat+1)/2
      j=(rj+0.5d0)
      ri=ri1
      if(mod(j,2).eq.0) ri=ri2
      i=(ri+0.5d0)

c     write(*,*) "lamda, phi = ",flon,flat
c     write(*,*) "ri1, ri2, ri, rj = ",ri1,ri2,ri,rj
c     write(*,*) "i, j = ",i,j

      if ((rj.lt.1).or.(rj.gt.nwlat)) then
c        write(333,*) 'ADCIRC grid node ',node,
c     &             ' falls outside of the ETA 29 grid'
        icode=1
        NN1=1
        NN2=1
        NN3=1
        DD1=0
        DD2=0
        DD3=0
        return
      endif

      if (mod(j,2).eq.0) then
         if ((ri.lt.1).or.(ri.gt.(nwlon+0.5d0))) then
c          write(333,*) 'ADCIRC grid node ',node,
c     &                 ' falls outside of the ETA 29 grid'
            icode=1
            NN1=1
            NN2=1
            NN3=1
            DD1=0
            DD2=0
            DD3=0
            return
         endif
      endif
      
      if (mod(j,2).ne.0) then
         if ((ri.lt.0.5).or.(ri.gt.nwlon)) then
c           write(333,*) 'ADCIRC grid node ',node,
c     &                 ' falls outside of the ETA 29 grid'
            icode=1
            NN1=1
            NN2=1
            NN3=1
            DD1=0
            DD2=0
            DD3=0
            return
         endif
      endif
      
c     compute the coordinates of the closest Eta29 grid node

      jm2=(nwlat+1)/2
      im2=nwlon*2
      call e29calc(i,j,lamda,phi,n)

c     compute the coordinates of neighbor node "a" (located SW of closest node)

      if ((i.eq.1).and.(mod(j,2).eq.0)) then
         ia=i
         ja=j-2
      else
         ia=i
         if(mod(j,2).eq.0) ia=i-1
         ja=j-1
      endif
c                                 this neighbor lies outside of Eta29 grid
      if ((ia.lt.1).or.(ja.lt.1)) then
         na=0
      else
         call e29calc(ia,ja,lamdaa,phia,na)
      endif

c     compute the coordinates of neighbor node "b" (located W of closest node)

      ib=i-1
      jb=j
      if (ib.lt.1) then         !this neighbor lies outside of Eta29 grid
         nb=0
      else
         call e29calc(ib,jb,lamdab,phib,nb)
      endif

c     compute the coordinates of neighbor node "c" (located NW of closest node)

      if ((i.eq.1).and.(mod(j,2).eq.0)) then
         ic=i
         jc=j+2
      else
         ic=ia
         jc=j+1
      endif
c                                    this neighbor lies outside of Eta29 grid
      if ((ic.lt.1).or.(jc.gt.nwlat)) then  
         nc=0
      else
         call e29calc(ic,jc,lamdac,phic,nc)
      endif

c     compute the coordinates of neighbor node "d" (located NE of closest node)

      if ((i.eq.181).and.(mod(j,2).ne.0)) then
         id=i
         jd=j+2
      else
         id=ic+1
         jd=j+1
      endif
c                                    this neighbor lies outside of Eta29 grid
      if ((id.gt.nwlon).or.(jd.gt.nwlat)) then  
         nd=0
      else
         call e29calc(id,jd,lamdad,phid,nd)
      endif

c     compute the coordinates of neighbor node "e" (located E of closest node)

      ie=i+1
      je=j
      if (ie.gt.nwlon) then     !this neighbor lies outside of Eta29 grid
         ne=0
      else
         call e29calc(ie,je,lamdae,phie,ne)
      endif
      
c     compute the coordinates of neighbor node "g" (located SE of closest node)

      if ((i.eq.181).and.(mod(j,2).ne.0)) then
         ig=i
         jg=j-2
      else
         ig=id
         jg=j-1
      endif
c                                    this neighbor lies outside of Eta29 grid
      if ((ig.gt.nwlon).or.(jg.lt.1)) then  
         ng=0
      else
         call e29calc(ig,jg,lamdag,phig,ng)
      endif

c      write(*,*) 'closest E29 node i,j = ',n,i,j,lamda,phi
c      if(na.eq.0) write(*,*) 'point a falls outside of Eta29 grid'
c      if(na.ne.0) write(*,*) 'point a   = ',na,ia,ja,lamdaa,phia
c      if(nb.eq.0) write(*,*) 'point b falls outside of Eta29 grid'
c      if(nb.ne.0) write(*,*) 'point b   = ',nb,ib,jb,lamdab,phib
c      if(nc.eq.0) write(*,*) 'point c falls outside of Eta29 grid'
c      if(nc.ne.0) write(*,*) "point c   = ",nc,ic,jc,lamdac,phic
c      if(nd.eq.0) write(*,*) 'point d falls outside of Eta29 grid'
c      if(nd.ne.0) write(*,*) "point d   = ",nd,id,jd,lamdad,phid
c      if(ne.eq.0) write(*,*) 'point e falls outside of Eta29 grid'
c      if(ne.ne.0) write(*,*) "point e   = ",ne,ie,je,lamdae,phie
c      if(ng.eq.0) write(*,*) 'point g falls outside of Eta29 grid'
c      if(ng.ne.0) write(*,*) "point g   = ",ng,ig,jg,lamdag,phig

      NN1=1
      NN2=1
      NN3=1
      DD1=0
      DD2=0
      DD3=0
      X1=lamda
      X4=flon
      Y1=phi
      Y4=flat
      ifflag=0
      AEMIN=99999.d0

c     test if the point is in triangle ij - b - a

      if ((na.ne.0).and.(nb.ne.0)) then
         X2=lamdab
         X3=lamdaa
         Y2=phib
         Y3=phia
         AREAS=ABS((X1-X3)*(Y2-Y3)+(X3-X2)*(Y1-Y3))
         A1=(X4-X3)*(Y2-Y3)+(X2-X3)*(Y3-Y4)
         A2=(X4-X1)*(Y3-Y1)-(Y4-Y1)*(X3-X1)
         A3=(Y4-Y1)*(X2-X1)-(X4-X1)*(Y2-Y1)
         AA=ABS(A1)+ABS(A2)+ABS(A3)
         AE=ABS(AA-AREAS)/AREAS
c     write(333,*) "AE = ",AE
         IF((AE.LT.1.0d-5).AND.(AE.LT.AEMIN)) THEN
            AEMIN=AE
            NN1=n
            NN2=nb
            NN3=na
            DD1=((X4-X3)*(Y2-Y3)+(X2-X3)*(Y3-Y4))/AREAS
            DD2=((X4-X1)*(Y3-Y1)-(Y4-Y1)*(X3-X1))/AREAS
            DD3=(-(X4-X1)*(Y2-Y1)+(Y4-Y1)*(X2-X1))/AREAS
            call betaucalc(i,j,DD1,ib,jb,DD2,ia,ja,DD3,betau)
            ifflag=ifflag+1
c     write(333,*) 'position found in triangle ij - b - a'
         ENDIF
      endif

c     if along the west boundary, test if the point is in triangle ij - c - a

      if((i.eq.1).and.(mod(j,2).ne.0)) then
         if((na.ne.0).and.(nc.ne.0)) then
            X2=lamdac
            X3=lamdaa
            Y2=phic
            Y3=phia
            AREAS=ABS((X1-X3)*(Y2-Y3)+(X3-X2)*(Y1-Y3))
            A1=(X4-X3)*(Y2-Y3)+(X2-X3)*(Y3-Y4)
            A2=(X4-X1)*(Y3-Y1)-(Y4-Y1)*(X3-X1)
            A3=(Y4-Y1)*(X2-X1)-(X4-X1)*(Y2-Y1)
            AA=ABS(A1)+ABS(A2)+ABS(A3)
            AE=ABS(AA-AREAS)/AREAS
c     write(333,*) "AE = ",AE
            IF((AE.LT.1.0d-5).AND.(AE.LT.AEMIN)) THEN
               NN1=n
               NN2=nc
               NN3=na
               DD1=((X4-X3)*(Y2-Y3)+(X2-X3)*(Y3-Y4))/AREAS
               DD2=((X4-X1)*(Y3-Y1)-(Y4-Y1)*(X3-X1))/AREAS
               DD3=(-(X4-X1)*(Y2-Y1)+(Y4-Y1)*(X2-X1))/AREAS
               call betaucalc(i,j,DD1,ic,jc,DD2,ia,ja,DD3,betau)
               ifflag=ifflag+1
c     write(333,*) 'position found in triangle ij - c - a'
            ENDIF
         endif
      endif

c     test if the point is in triangle ij - c - b

      if((nb.ne.0).and.(nc.ne.0)) then
         X2=lamdac
         X3=lamdab
         Y2=phic
         Y3=phib
         AREAS=ABS((X1-X3)*(Y2-Y3)+(X3-X2)*(Y1-Y3))
         A1=(X4-X3)*(Y2-Y3)+(X2-X3)*(Y3-Y4)
         A2=(X4-X1)*(Y3-Y1)-(Y4-Y1)*(X3-X1)
         A3=(Y4-Y1)*(X2-X1)-(X4-X1)*(Y2-Y1)
         AA=ABS(A1)+ABS(A2)+ABS(A3)
         AE=ABS(AA-AREAS)/AREAS
c     write(333,*) "AE = ",AE
         IF((AE.LT.1.0d-5).AND.(AE.LT.AEMIN)) THEN
            NN1=n
            NN2=nc
            NN3=nb
            DD1=((X4-X3)*(Y2-Y3)+(X2-X3)*(Y3-Y4))/AREAS
            DD2=((X4-X1)*(Y3-Y1)-(Y4-Y1)*(X3-X1))/AREAS
            DD3=(-(X4-X1)*(Y2-Y1)+(Y4-Y1)*(X2-X1))/AREAS
            call betaucalc(i,j,DD1,ic,jc,DD2,ib,jb,DD3,betau)
            ifflag=ifflag+1
c     write(333,*) 'position found in triangle ij - c - b'
         ENDIF
      endif

c     test if the point is in triangle ij - d - c

      if((nc.ne.0).and.(nd.ne.0)) then
         X2=lamdad
         X3=lamdac
         Y2=phid
         Y3=phic
         AREAS=ABS((X1-X3)*(Y2-Y3)+(X3-X2)*(Y1-Y3))
         A1=(X4-X3)*(Y2-Y3)+(X2-X3)*(Y3-Y4)
         A2=(X4-X1)*(Y3-Y1)-(Y4-Y1)*(X3-X1)
         A3=(Y4-Y1)*(X2-X1)-(X4-X1)*(Y2-Y1)
         AA=ABS(A1)+ABS(A2)+ABS(A3)
         AE=ABS(AA-AREAS)/AREAS
c     write(333,*) "AE = ",AE
         IF((AE.LT.1.0d-5).AND.(AE.LT.AEMIN)) THEN
            NN1=n
            NN2=nd
            NN3=nc
            DD1=((X4-X3)*(Y2-Y3)+(X2-X3)*(Y3-Y4))/AREAS
            DD2=((X4-X1)*(Y3-Y1)-(Y4-Y1)*(X3-X1))/AREAS
            DD3=(-(X4-X1)*(Y2-Y1)+(Y4-Y1)*(X2-X1))/AREAS
            call betaucalc(i,j,DD1,id,jd,DD2,ic,jc,DD3,betau)
            ifflag=ifflag+1
c     write(333,*) 'position found in triangle ij - d - c'
         ENDIF
      endif

c     if along the east boundary, test if the point is in triangle ij - g - d

      if((i.eq.181).and.(mod(j,2).eq.0)) then
         if((nd.ne.0).and.(ng.ne.0)) then
            X2=lamdag
            X3=lamdad
            Y2=phig
            Y3=phid
            AREAS=ABS((X1-X3)*(Y2-Y3)+(X3-X2)*(Y1-Y3))
            A1=(X4-X3)*(Y2-Y3)+(X2-X3)*(Y3-Y4)
            A2=(X4-X1)*(Y3-Y1)-(Y4-Y1)*(X3-X1)
            A3=(Y4-Y1)*(X2-X1)-(X4-X1)*(Y2-Y1)
            AA=ABS(A1)+ABS(A2)+ABS(A3)
            AE=ABS(AA-AREAS)/AREAS
c     write(333,*) "AE = ",AE
            IF((AE.LT.1.0d-5).AND.(AE.LT.AEMIN)) THEN
               NN1=n
               NN2=ng
               NN3=nd
               DD1=((X4-X3)*(Y2-Y3)+(X2-X3)*(Y3-Y4))/AREAS
               DD2=((X4-X1)*(Y3-Y1)-(Y4-Y1)*(X3-X1))/AREAS
               DD3=(-(X4-X1)*(Y2-Y1)+(Y4-Y1)*(X2-X1))/AREAS
               call betaucalc(i,j,DD1,ig,jg,DD2,id,jd,DD3,betau)
               ifflag=ifflag+1
c     write(333,*) 'position found in triangle ij - g - d'
            ENDIF
         endif
      endif

c     test if the point is in triangle ij - e - d

      if((nd.ne.0).and.(ne.ne.0)) then
         X2=lamdae
         X3=lamdad
         Y2=phie
         Y3=phid
         AREAS=ABS((X1-X3)*(Y2-Y3)+(X3-X2)*(Y1-Y3))
         A1=(X4-X3)*(Y2-Y3)+(X2-X3)*(Y3-Y4)
         A2=(X4-X1)*(Y3-Y1)-(Y4-Y1)*(X3-X1)
         A3=(Y4-Y1)*(X2-X1)-(X4-X1)*(Y2-Y1)
         AA=ABS(A1)+ABS(A2)+ABS(A3)
         AE=ABS(AA-AREAS)/AREAS
c     write(333,*) "AE = ",AE
         IF((AE.LT.1.0d-5).AND.(AE.LT.AEMIN)) THEN
            NN1=n
            NN2=ne
            NN3=nd
            DD1=((X4-X3)*(Y2-Y3)+(X2-X3)*(Y3-Y4))/AREAS
            DD2=((X4-X1)*(Y3-Y1)-(Y4-Y1)*(X3-X1))/AREAS
            DD3=(-(X4-X1)*(Y2-Y1)+(Y4-Y1)*(X2-X1))/AREAS
            call betaucalc(i,j,DD1,ie,je,DD2,id,jd,DD3,betau)
            ifflag=ifflag+1
c     write(333,*) 'position found in triangle ij - e - d'
         ENDIF
      endif

c     test if the point is in triangle ij - g - e

      if((ne.ne.0).and.(ng.ne.0)) then
         X2=lamdag
         X3=lamdae
         Y2=phig
         Y3=phie
         AREAS=ABS((X1-X3)*(Y2-Y3)+(X3-X2)*(Y1-Y3))
         A1=(X4-X3)*(Y2-Y3)+(X2-X3)*(Y3-Y4)
         A2=(X4-X1)*(Y3-Y1)-(Y4-Y1)*(X3-X1)
         A3=(Y4-Y1)*(X2-X1)-(X4-X1)*(Y2-Y1)
         AA=ABS(A1)+ABS(A2)+ABS(A3)
         AE=ABS(AA-AREAS)/AREAS
c     write(333,*) "AE = ",AE
         IF((AE.LT.1.0d-5).AND.(AE.LT.AEMIN)) THEN
            NN1=n
            NN2=ng
            NN3=ne
            DD1=((X4-X3)*(Y2-Y3)+(X2-X3)*(Y3-Y4))/AREAS
            DD2=((X4-X1)*(Y3-Y1)-(Y4-Y1)*(X3-X1))/AREAS
            DD3=(-(X4-X1)*(Y2-Y1)+(Y4-Y1)*(X2-X1))/AREAS
            call betaucalc(i,j,DD1,ig,jg,DD2,ie,je,DD3,betau)
            ifflag=ifflag+1
c     write(333,*) 'position found in triangle ij - g - e'
         ENDIF
      endif

c     test if the point is in triangle ij - a - g

      if((na.ne.0).and.(ng.ne.0)) then
         X2=lamdaa
         X3=lamdag
         Y2=phia
         Y3=phig
         AREAS=ABS((X1-X3)*(Y2-Y3)+(X3-X2)*(Y1-Y3))
         A1=(X4-X3)*(Y2-Y3)+(X2-X3)*(Y3-Y4)
         A2=(X4-X1)*(Y3-Y1)-(Y4-Y1)*(X3-X1)
         A3=(Y4-Y1)*(X2-X1)-(X4-X1)*(Y2-Y1)
         AA=ABS(A1)+ABS(A2)+ABS(A3)
         AE=ABS(AA-AREAS)/AREAS
c     write(333,*) "AE = ",AE
         IF((AE.LT.1.0d-5).AND.(AE.LT.AEMIN)) THEN
            NN1=n
            NN2=na
            NN3=ng
            DD1=((X4-X3)*(Y2-Y3)+(X2-X3)*(Y3-Y4))/AREAS
            DD2=((X4-X1)*(Y3-Y1)-(Y4-Y1)*(X3-X1))/AREAS
            DD3=(-(X4-X1)*(Y2-Y1)+(Y4-Y1)*(X2-X1))/AREAS
            call betaucalc(i,j,DD1,ia,ja,DD2,ig,jg,DD3,betau)
            ifflag=ifflag+1
c     write(333,*) 'position found in triangle ij - a - g'
         ENDIF
      endif

c      if(ifflag.eq.0) then
c        write(333,*) 'position not found'
c        write(*,*) 'position not found in subroutine E29SEARCH'
c        icode=3
c        else
c       write(*,*) 'i,j,NN1,NN2,NN3,DD1,DD2,DD3'
c        write(333,999) i,j,NN1,NN2,NN3,DD1,DD2,DD3,betau/dgtora
c 999    format(5I8,1x,3E13.6)
c        endif

      return
      end subroutine



C***********************************************************************
C  Subroutine to compute the longititude and latitude of a given i,j   *
C       position in the Eta29 grid.                                    *
C                                                                      *
C                    Written by R.L.       1/11/98                     *
C***********************************************************************

      subroutine e29calc(i,j,lamda,phi,n)
      implicit none
      integer i,j,n,nwlon,nwlat,im2,jm2,i1,i2,i1p1,i1m1,i2p1,i2m1,
     &     i3p1,i3m1
      real(sz) lamda,phi,phii,dlon,dlat,dlnt,arg,betau1,betau2,betau3
      real(8) lamda0,phi0,rphi0,cphi0,sphi0,tphi0,dlamda,dphi,rdlamda,
     &     rdphi,a,rlamar,phiarg,rlamda,dgtora
c     
      nwlon=181
      nwlat=271
      dgtora=deg2rad
      lamda0=-97.0d0
      phi0=41.0d0
      rphi0=dgtora*phi0
      cphi0=cos(rphi0)
      sphi0=sin(rphi0)
      tphi0=tan(rphi0)
      dlamda=7.d0/36.d0
      dphi=5.d0/27.d0
      rdlamda=dgtora*dlamda
      rdphi=dgtora*dphi
c     
      jm2=(nwlat+1)/2
      im2=nwlon*2
c     
      phii=rdphi*float(j-jm2)
      i1=2*i-1
      i2=2*i
      if(mod(j,2).ne.0) then
         rlamda=rdlamda*float(i2-nwlon)
      else
         rlamda=rdlamda*float(i1-nwlon)
      endif
      phiarg= sin(phii)*cphi0+cos(phii)*sphi0*cos(rlamda)
      if(phiarg.gt.1.0d0) phiarg=1.0d0
      if(phiarg.lt.-1.0d0) phiarg=-1.0d0
      phi=asin(phiarg)
      rlamar= cos(phii)*cos(rlamda)/(cos(phi)*cphi0)-tan(phi)*tphi0
      if(rlamar.gt.1.0d0) rlamar=1.0d0
      if(rlamar.lt.-1.d0) rlamar=-1.d0
      a=acos(rlamar)/dgtora
      if(rlamda.le.0.) then
         lamda=lamda0-a
      else
         lamda=lamda0+a
      endif
      phi=phi/dgtora
      n=nwlon*(j-1)+i
C     
      return
      end subroutine


C***********************************************************************
C  Subroutine to compute the conversion angle between the E29 velocity *
C       field and a lon,lat coordinate system.                         *
C                                                                      *
C                    Written by R.L.       1/12/98                     *
C***********************************************************************

      subroutine betaucalc(i1,j1,dd1,i2,j2,dd2,i3,j3,dd3,betau)
      implicit none
      integer i1,j1,i2,j2,i3,j3,n,i1p1,i1m1,i2p1,i2m1,i3p1,i3m1
      real(sz) dd1,dd2,dd3,betau
      real(sz) lamda,lamdap1,lamdam1,phi,phip1,phim1,dlon,dlat,
     &     dlnt,arg,betau1,betau2,betau3,dgtora
c     
      dgtora=deg2rad
c     
      if(i1.ne.181) then
         i1p1=i1+1
      else
         i1p1=i1
      endif
      if(i1.ne.1) then
         i1m1=i1-1
      else
         i1m1=i1
      endif
      call e29calc(i1,j1,lamda,phi,n)
      call e29calc(i1p1,j1,lamdap1,phip1,n)
      call e29calc(i1m1,j1,lamdam1,phim1,n)
      dlon=(lamdap1-lamdam1)*cos(phi*dgtora)
      dlat=phip1-phim1
      dlnt=sqrt(dlon*dlon+dlat*dlat)
      arg=dlat/dlnt
      if(arg.gt.1.d0) arg=1.d0
      if(arg.lt.-1.d0) arg=-1.d0
      betau1=asin(arg)
c     
      if(i2.ne.181) then
         i2p1=i2+1
      else
         i2p1=i2
      endif
c     
      if(i2.ne.1) then
         i2m1=i2-1
      else
         i2m1=i2
      endif
c     
      call e29calc(i2,j2,lamda,phi,n)
      call e29calc(i2p1,j2,lamdap1,phip1,n)
      call e29calc(i2m1,j2,lamdam1,phim1,n)
      dlon=(lamdap1-lamdam1)*cos(phi*dgtora)
      dlat=phip1-phim1
      dlnt=sqrt(dlon*dlon+dlat*dlat)
      arg=dlat/dlnt
      if(arg.gt.1.d0) arg=1.d0
      if(arg.lt.-1.d0) arg=-1.d0
      betau2=asin(arg)
c     
      if(i3.ne.181) then
         i3p1=i3+1
      else
         i3p1=i3
      endif
c     
      if(i3.ne.1) then
         i3m1=i3-1
      else
         i3m1=i3
      endif
c     
      call e29calc(i3,j3,lamda,phi,n)
      call e29calc(i3p1,j3,lamdap1,phip1,n)
      call e29calc(i3m1,j3,lamdam1,phim1,n)
      dlon=(lamdap1-lamdam1)*cos(phi*dgtora)
      dlat=phip1-phim1
      dlnt=sqrt(dlon*dlon+dlat*dlat)
      arg=dlat/dlnt
      if(arg.gt.1.d0) arg=1.d0
      if(arg.lt.-1.d0) arg=-1.d0
      betau3=asin(arg)
      betau=dd1*betau1+dd2*betau2+dd3*betau3
C     
      return
      end subroutine

C***********************************************************************
C                                                                      *
C   End of subroutines to read wind and pressure fields                * 
C                                                                      *
C***********************************************************************


C***********************************************************************
C                                                                      *
C   Read onto the ADCIRC grid radiation stress fields in the PBL-JAG   *
C   (hurricane) model format.                                          *
C                                                                      *
C                                                                      *
C                           R.L.05/12/99                               *
C***********************************************************************

      SUBROUTINE RSGET(RSNX,RSNY,NP)
      USE SIZES
      IMPLICIT NONE
      INTEGER NP,I,NHG
      REAL(SZ) RSNX(*),RSNY(*)
      CHARACTER*80 PBLJAGF
C     
      DO I=1,NP
         RSNX(I)=0.d0
         RSNY(I)=0.d0
      END DO
 170  READ(23,'(A80)') PBLJAGF
      IF(PBLJAGF(2:2).EQ.'#') GOTO 170
 171  READ(PBLJAGF,'(I8,5E13.5)') NHG,RSNX(NHG),RSNY(NHG)
      READ(23,'(A80)') PBLJAGF
      IF(PBLJAGF(2:2).NE.'#') GOTO 171
C     
      RETURN
      END SUBROUTINE


      END MODULE
