SUBROUTINE ckwrfhdrs (fl1, fl2)

!-------------------------------------------------------------------------------
! Name:     Check WRF Headers
! Purpose:  Compares headers of two WRF-ARW files to ensure that they are for
!           the same geospatial domain.
! Revised:  11 May 2007  Original version.  (T. Otte)
!           23 Aug 2007  Revised to make code general.  (T. Otte)
!           30 Nov 2007  Removed restriction that required the Pleim surface
!                        layer model to be used with the Pleim-Xiu land-surface
!                        model.  (T. Otte and R. Gilliam)
!           10 Dec 2009  Changed the file (from fl1 to fl2) that is examined to ensure
!                        the PX LSM is the land surface model (SF_SURFACE_PHYSICS). This is 
!                        neccessary because the new wrfsfdda file does not have this configuration
!                        information, only the wrfinput_d01 file
!-------------------------------------------------------------------------------

  USE wrf_netcdf

  IMPLICIT NONE

  INCLUDE 'netcdf.inc'

  INTEGER                      :: cdfid
  REAL                         :: cenlat_fl1
  REAL                         :: cenlat_fl2
  REAL                         :: cenlon_fl1
  REAL                         :: cenlon_fl2
  REAL                         :: dx_fl1
  REAL                         :: dx_fl2
  REAL                         :: dy_fl1
  REAL                         :: dy_fl2
  REAL,          PARAMETER     :: eps     = 1.0e-3
  CHARACTER*256, INTENT(IN)    :: fl1
  CHARACTER*256, INTENT(IN)    :: fl2
  INTEGER                      :: mapproj_fl1
  INTEGER                      :: mapproj_fl2
  REAL                         :: moadcenlat_fl1
  REAL                         :: moadcenlat_fl2
  INTEGER                      :: nx_fl1
  INTEGER                      :: nx_fl2
  INTEGER                      :: ny_fl1
  INTEGER                      :: ny_fl2
  INTEGER                      :: nz_fl1
  INTEGER                      :: nz_fl2
  LOGICAL                      :: ok      = .TRUE.
  INTEGER                      :: phy_surface_id
  CHARACTER*16,  PARAMETER     :: pname   = 'CKWRFHDRS'
  INTEGER                      :: rcode
  REAL                         :: standlon_fl1
  REAL                         :: standlon_fl2
  REAL                         :: trulat1_fl1
  REAL                         :: trulat1_fl2
  REAL                         :: trulat2_fl1
  REAL                         :: trulat2_fl2
  INTEGER                      :: varid

!-------------------------------------------------------------------------------
! Extract various global attributes from WRF file 1 to compare with attributes
! from WRF file 2.
!-------------------------------------------------------------------------------

  cdfid = ncopn (fl1, ncnowrit, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9000) TRIM(fl1)
    GOTO 1001
  ENDIF

  ! Make sure Pleim-Xiu LSM is the land surface model in wrfinput_d0* before continuing with this code.

  CALL get_gl_att_int_cdf (fl2, 'SF_SURFACE_PHYSICS', phy_surface_id,   rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl1), 'SF_SURFACE_PHYSICS', rcode
    GOTO 1001
  ENDIF

  IF ( phy_surface_id /= 7 ) THEN
    WRITE (6,9200) TRIM(fl1), phy_surface_id
    GOTO 1001
  ENDIF

  ! Extract global attributes.

  CALL get_gl_att_int_cdf  (fl1, 'WEST-EAST_GRID_DIMENSION',   nx_fl1, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl1), 'WEST-EAST_GRID_DIMENSION', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_int_cdf  (fl1, 'SOUTH-NORTH_GRID_DIMENSION', ny_fl1, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl1), 'SOUTH-NORTH_GRID_DIMENSION', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_int_cdf  (fl1, 'BOTTOM-TOP_GRID_DIMENSION',  nz_fl1, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl1), 'BOTTOM-TOP_GRID_DIMENSION', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_real_cdf (fl1, 'DX',           dx_fl1,            rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl1), 'DX', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_real_cdf (fl1, 'DY',           dy_fl1,            rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl1), 'DY', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_int_cdf  (fl1, 'MAP_PROJ',     mapproj_fl1,       rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl1), 'MAP_PROJ', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_real_cdf (fl1, 'STAND_LON',    standlon_fl1,      rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl1), 'STAND_LON', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_real_cdf (fl1, 'MOAD_CEN_LAT', moadcenlat_fl1,    rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl1), 'MOAD_CEN_LAT', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_real_cdf (fl1, 'CEN_LON',      cenlon_fl1,        rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl1), 'CEN_LON', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_real_cdf (fl1, 'CEN_LAT',      cenlat_fl1,        rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl1), 'CEN_LAT', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_real_cdf (fl1, 'TRUELAT1',     trulat1_fl1,       rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl1), 'TRUELAT1', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_real_cdf (fl1, 'TRUELAT2',     trulat2_fl1,       rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl1), 'TRUELAT2', rcode
    GOTO 1001
  ENDIF

  ! Close netCDF file.

  CALL ncclos (cdfid, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9300) TRIM(fl1)
    GOTO 1001
  ENDIF

!-------------------------------------------------------------------------------
! Extract various global attributes from WRF file 2 to compare with attributes
! from WRF file 1.
!-------------------------------------------------------------------------------

  cdfid = ncopn (fl2, ncnowrit, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9000) TRIM(fl2)
    GOTO 1001
  ENDIF

  ! Extract global attributes.

  CALL get_gl_att_int_cdf  (fl2, 'WEST-EAST_GRID_DIMENSION',   nx_fl2, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl2), 'WEST-EAST_GRID_DIMENSION', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_int_cdf  (fl2, 'SOUTH-NORTH_GRID_DIMENSION', ny_fl2, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl2), 'SOUTH-NORTH_GRID_DIMENSION', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_int_cdf  (fl2, 'BOTTOM-TOP_GRID_DIMENSION',  nz_fl2, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl2), 'BOTTOM-TOP_GRID_DIMENSION', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_real_cdf (fl2, 'DX',           dx_fl2,            rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl2), 'DX', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_real_cdf (fl2, 'DY',           dy_fl2,            rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl2), 'DY', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_int_cdf  (fl2, 'MAP_PROJ',     mapproj_fl2,       rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl2), 'MAP_PROJ', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_real_cdf (fl2, 'STAND_LON',    standlon_fl2,      rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl2), 'STAND_LON', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_real_cdf (fl2, 'MOAD_CEN_LAT', moadcenlat_fl2,    rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl2), 'MOAD_CEN_LAT', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_real_cdf (fl2, 'CEN_LON',      cenlon_fl2,        rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl2), 'CEN_LON', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_real_cdf (fl2, 'CEN_LAT',      cenlat_fl2,        rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl2), 'CEN_LAT', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_real_cdf (fl2, 'TRUELAT1',     trulat1_fl2,       rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl2), 'TRUELAT1', rcode
    GOTO 1001
  ENDIF

  CALL get_gl_att_real_cdf (fl2, 'TRUELAT2',     trulat2_fl2,       rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl2), 'TRUELAT2', rcode
    GOTO 1001
  ENDIF

  ! Close netCDF file.

  CALL ncclos (cdfid, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9300) TRIM(fl2)
    GOTO 1001
  ENDIF

!-------------------------------------------------------------------------------
! Check to see if first-order domain information is the same in the two
! WRF files.
!-------------------------------------------------------------------------------

  ok = ok .AND. ( nx_fl1                                == nx_fl2      )
  ok = ok .AND. ( ny_fl1                                == ny_fl2      )
  ok = ok .AND. ( ABS(dx_fl1         - dx_fl2 )         < eps          )
  ok = ok .AND. ( ABS(dy_fl1         - dy_fl2 )         < eps          )
  ok = ok .AND. ( mapproj_fl1                           == mapproj_fl2 )
  ok = ok .AND. ( ABS(standlon_fl1   - standlon_fl2   ) < eps          )
  ok = ok .AND. ( ABS(moadcenlat_fl1 - moadcenlat_fl2 ) < eps          )
  ok = ok .AND. ( ABS(cenlon_fl1     - cenlon_fl2     ) < eps          )
  ok = ok .AND. ( ABS(cenlat_fl1     - cenlat_fl2     ) < eps          )
  ok = ok .AND. ( ABS(trulat1_fl1    - trulat1_fl2    ) < eps          )
  ok = ok .AND. ( ABS(trulat2_fl1    - trulat2_fl2    ) < eps          )

  IF ( .NOT. ok ) THEN
    WRITE (6,9400)
    GOTO 1001
  ENDIF

  RETURN

!-------------------------------------------------------------------------------
! Error-handling section.
!-------------------------------------------------------------------------------

 9000 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: CKWRFHDRS',                           &
              /, 1x, '***   ERROR OPENING WRF FILE',                        &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, 70('*'))

 9100 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: CKWRFHDRS',                           &
              /, 1x, '***   ERROR RETRIEVING VARIABLE FROM WRF FILE',       &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, '***   VARIABLE = ', a,                                &
              /, 1x, '***   RCODE = ', i3,                                  &
              /, 1x, 70('*'))

 9200 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: CKWRFHDRS',                           &
              /, 1x, '***   DOES NOT APPEAR TO BE P-X RUN IN WRF FILE',     &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, '***   SF_SURFACE_PHYSICS = ', i3,                     &
              /, 1x, 70('*'))

 9300 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: CKWRFHDRS',                           &
              /, 1x, '***   ERROR CLOSING WRF NETCDF FILE',                 &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, 70('*'))

 9400 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: CKWRFHDRS',                           &
              /, 1x, '***   FILES APPEAR TO BE FROM DIFFERENT GRIDS',       &
              /, 1x, 70('*'))

 1001 CALL graceful_stop (pname)
      RETURN

END SUBROUTINE ckwrfhdrs
