SUBROUTINE readnml

!-------------------------------------------------------------------------------
! Name:     Read Namelist
! Purpose:  Reads input namelist to get user control variables.
! Revised:  11 May 2007  Original version.  (T. Otte)
!           24 Aug 2007  Added option to replace deep soil temperature with
!                        2-m temperature averaged from analyses over
!                        user-defined period.  (T. Otte)
!-------------------------------------------------------------------------------

  USE files

  IMPLICIT NONE

  INTEGER                      :: istat
  CHARACTER*16,  PARAMETER     :: pname   = 'READNML'

  NAMELIST /filenames/ file_wrfout_last, file_wrfin_next, file_wrffdda_next,  &
                       do_tsoil, do_msoil, do_tsoildeep_from2m, avg_period_2m

!-------------------------------------------------------------------------------
! Open namelist file.
!-------------------------------------------------------------------------------

  OPEN (iutnml, FILE=file_nml, STATUS='OLD', ERR=8000, IOSTAT=istat)

!-------------------------------------------------------------------------------
! Initialize input file names.
!-------------------------------------------------------------------------------

  file_wrfout_last    = " "
  file_wrfin_next     = " "
  file_wrffdda_next   = " "

  do_tsoil            = .TRUE.
  do_msoil            = .TRUE.
  do_tsoildeep_from2m = .FALSE.

!-------------------------------------------------------------------------------
! Read namelist to get user definitions.  Rewind namelist file after each
! read in case namelists are not in the correct order in the namelist.
!-------------------------------------------------------------------------------

  READ (iutnml, filenames, ERR=8100, IOSTAT=istat)
  REWIND (iutnml)

!-------------------------------------------------------------------------------
! Close namelist file.
!-------------------------------------------------------------------------------

  CLOSE (iutnml)

!-------------------------------------------------------------------------------
! Crop blank spaces off ends of file names.
!-------------------------------------------------------------------------------

  file_wrfout_last  = TRIM( ADJUSTL( file_wrfout_last  ) )
  file_wrfin_next   = TRIM( ADJUSTL( file_wrfin_next   ) )
  file_wrffdda_next = TRIM( ADJUSTL( file_wrffdda_next ) )

!-------------------------------------------------------------------------------
! Verify that some option was selected by the user for processing.
!-------------------------------------------------------------------------------

  IF ( ( .NOT. do_tsoil ) .AND. ( .NOT. do_msoil ) .AND.  &
       ( .NOT. do_tsoildeep_from2m ) ) THEN
    GOTO 8200
  ENDIF

!-------------------------------------------------------------------------------
! Verify that the averaging period for the 2-m temperature is at least one
! full diurnal cycle.
!-------------------------------------------------------------------------------

  IF ( ( do_tsoildeep_from2m ) .AND. ( avg_period_2m < 1 ) ) THEN
    GOTO 8300
  ENDIF

  RETURN

!-------------------------------------------------------------------------------
! Error-handling section.
!-------------------------------------------------------------------------------

 8000 WRITE (6,9000) iutnml, TRIM(file_nml), istat
      GOTO 1001

 8100 WRITE (6,9100) iutnml, TRIM(file_nml), "filenames", istat
      GOTO 1001

 8200 WRITE (6,9200)
      GOTO 1001

 8300 WRITE (6,9300) avg_period_2m
      GOTO 1001

 9000 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: READNML',                             &
              /, 1x, '***   ERROR OPENING NAMELIST FILE ON UNIT ', i3,      &
              /, 1x, '***   NAMELIST FILE NAME = ', a,                      &
              /, 1x, '***   IOSTAT = ', i4,                                 &
              /, 1x, 70('*'))

 9100 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: READNML',                             &
              /, 1x, '***   ERROR READING NAMELIST FILE ON UNIT ', i3,      &
              /, 1x, '***   NAMELIST FILE NAME = ', a,                      &
              /, 1x, '***   NAMELIST = ', a,                                &
              /, 1x, '***   IOSTAT = ', i4,                                 &
              /, 1x, 70('*'))

 9200 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: READNML',                             &
              /, 1x, '***   ALL PROCESSING OPTIONS WERE FALSE',             &
              /, 1x, '***   STOPPING...',                                   &
              /, 1x, 70('*'))

 9300 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: READNML',                             &
              /, 1x, '***   AVERAGING PERIOD FOR 2-M TEMP IS LESS THAN 1 D',&
              /, 1x, '***   AVG_PERIOD_2M = ', i4,                          &
              /, 1x, 70('*'))

 1001 CALL graceful_stop (pname)
      RETURN

END SUBROUTINE readnml
