SUBROUTINE wrfaverage

!-------------------------------------------------------------------------------
! Name:     WRF Average and Swap Data
! Purpose:  Extracts data from a specific time period from one file, averages
!           the field, and overwrites a different field in another file with
!           those data.
! Revised:  24 Aug 2007  Original version.  (T. Otte)
!
!-------------------------------------------------------------------------------

  USE date_pack
  USE files
  USE wrf_netcdf

  IMPLICIT NONE

  INCLUDE 'netcdf.inc'

  INTEGER                      :: cdfid
  INTEGER                      :: count     ( 10 )
  REAL                         :: deltat_in_h
  INTEGER                      :: dimids    ( 10 )
  CHARACTER*256                :: fl
  INTEGER                      :: i
  INTEGER                      :: idtsec
  INTEGER                      :: it
  INTEGER                      :: j
  INTEGER,       PARAMETER     :: max_times = 150
  INTEGER                      :: n
  INTEGER                      :: n_times
  INTEGER                      :: ns
  INTEGER                      :: nt
  INTEGER                      :: nx
  INTEGER                      :: ny
  INTEGER                      :: nz
  CHARACTER*16,  PARAMETER     :: pname     = 'WRFAVERAGE'
  INTEGER                      :: rcode
  INTEGER                      :: start     ( 10 )
  REAL,          ALLOCATABLE   :: t2m       ( : , : , : )
  CHARACTER*19                 :: time_i_want
  CHARACTER*80                 :: times     ( max_times )
  REAL,          ALLOCATABLE   :: tsdeep    ( : , : )
  REAL,          ALLOCATABLE   :: tslb      ( : , : , : )
  INTEGER,       PARAMETER     :: ttol      = 300           ! [sec]
  INTEGER                      :: varid

!-------------------------------------------------------------------------------
! If initializing deep soil temperature by averaging 2-m temperature,
! make sure that 2-m temperature analyses are in the WRF FDDA file.
!-------------------------------------------------------------------------------

  fl = file_wrffdda_next

  cdfid = ncopn (fl, ncnowrit, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9000) TRIM(fl)
    GOTO 1001
  ENDIF

  rcode = nf_inq_varid (cdfid, 'T2_NDG_OLD', varid)  ! 2-m temperature
  IF ( rcode /= 0 ) THEN
    WRITE (6,9050) 'SMOIS', TRIM(fl)
    GOTO 1001
  ENDIF

  CALL ncclos (cdfid, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9200) TRIM(fl)
    GOTO 1001
  ENDIF

!-------------------------------------------------------------------------------
! Get initial time from WRF input for next simulation.  This is TIME_I_WANT.
!-------------------------------------------------------------------------------

  fl = file_wrfin_next

  cdfid = ncopn (fl, ncnowrit, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9000) TRIM(fl)
    GOTO 1001
  ENDIF

  CALL get_times_cdf (fl, times, n_times, max_times, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl), rcode
    GOTO 1001
  ENDIF

  time_i_want = times(1)(1:19)
  print*,'...TIME_I_WANT = ', TRIM(time_i_want)

!-------------------------------------------------------------------------------
! Read soil temperature.
!-------------------------------------------------------------------------------

  rcode = nf_inq_varid (cdfid, 'TSLB', varid)  ! soil temperature (all layers)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9400) 'TSLB', TRIM(fl)
    GOTO 1001
  ENDIF

  rcode = nf_inq_vardimid (cdfid, varid, dimids)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9500) 'TSLB', TRIM(fl)
    GOTO 1001
  ENDIF

  rcode = nf_inq_dimlen (cdfid, dimids(1), nx)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9600) TRIM(fl), 'TSLB', 'nx'
    GOTO 1001
  ENDIF

  rcode = nf_inq_dimlen (cdfid, dimids(2), ny)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9600) TRIM(fl), 'TSLB', 'ny'
    GOTO 1001
  ENDIF

  rcode = nf_inq_dimlen (cdfid, dimids(3), ns)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9600) TRIM(fl), 'TLSB', 'ns'
    GOTO 1001
  ENDIF

  print*,'+++ Preparing to replace initial deep soil temp with avg 2-m temp'
  print*,'...TSLB:        nx, ny, ns = ', nx, ny, ns

  ALLOCATE ( tslb (nx, ny, ns) )

  CALL get_times_cdf (fl, times, n_times, max_times, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl), rcode
    GOTO 1001
  ENDIF

  findit1: DO i = 1, n_times
    CALL geth_idts (times(i)(1:19), time_i_want, idtsec)
    IF ( ABS(idtsec) <= ttol ) THEN  ! found TIME_I_WANT
      it = i
      EXIT findit1
    ENDIF
  ENDDO findit1
  IF ( i > n_times ) THEN
    WRITE (6,9300) TRIM(fl), time_i_want
    GOTO 1001
  ENDIF

  CALL get_var_3d_real_cdf (fl, 'TSLB', tslb, nx, ny, ns, it, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9700) 'TSLB', TRIM(fl)
    GOTO 1001
  ENDIF

  CALL ncclos (cdfid, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9200) TRIM(fl)
    GOTO 1001
  ENDIF

!-------------------------------------------------------------------------------
! Search for desired time (TIME_I_WANT) in WRF FDDA file.
!-------------------------------------------------------------------------------

  fl = file_wrffdda_next

  cdfid = ncopn (fl, ncnowrit, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9000) TRIM(fl)
    GOTO 1001
  ENDIF

  CALL get_times_cdf (fl, times, n_times, max_times, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl), rcode
    GOTO 1001
  ENDIF

  findit2: DO i = 1, n_times
    CALL geth_idts (times(i)(1:19), time_i_want, idtsec)
    IF ( ABS(idtsec) <= ttol ) THEN  ! found TIME_I_WANT
      it = i
      EXIT findit2
    ENDIF
  ENDDO findit2
  IF ( i > n_times ) THEN
    WRITE (6,9300) TRIM(fl), time_i_want
    GOTO 1001
  ENDIF

!-------------------------------------------------------------------------------
! Determine temporal spacing of analyses in WRF FDDA file.  Assume uniform
! temporal spacing in file, so only check between first two analyses.  Then
! determine the number of analyses to extract based on user input variable
! AVG_PERIOD_2M.
!-------------------------------------------------------------------------------

  CALL geth_idts (times(2)(1:19), times(1)(1:19), idtsec)

  deltat_in_h = FLOAT( ABS(idtsec) / 3600 )

  nt = 24.0 * FLOAT(avg_period_2m) / deltat_in_h

  IF ( nt > n_times ) THEN
    WRITE (6,9350) nt, n_times
    GOTO 1001
  ENDIF

!-------------------------------------------------------------------------------
! Extract 2-m temperature from WRF FDDA file.
!-------------------------------------------------------------------------------

  rcode = nf_inq_varid (cdfid, 'T2_NDG_OLD', varid)  ! 2-m temperature
  IF ( rcode /= 0 ) THEN
    WRITE (6,9400) 'T2_NDG_OLD', TRIM(fl)
    GOTO 1001
  ENDIF

  rcode = nf_inq_vardimid (cdfid, varid, dimids)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9500) 'T2_NDG_OLD', TRIM(fl)
    GOTO 1001
  ENDIF

  rcode = nf_inq_dimlen (cdfid, dimids(1), nx)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9600) TRIM(fl), 'T2_NDG_OLD', 'nx'
    GOTO 1001
  ENDIF

  rcode = nf_inq_dimlen (cdfid, dimids(2), ny)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9600) TRIM(fl), 'T2_NDG_OLD', 'ny'
    GOTO 1001
  ENDIF

  rcode = nf_inq_dimlen (cdfid, dimids(3), nz)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9600) TRIM(fl), 'T2_NDG_OLD', 'nz'
    GOTO 1001
  ENDIF

  print*,'...T2_NDG_OLD:  nx, ny, nz = ', nx, ny, nz

  ALLOCATE ( t2m    (nx, ny, nt) )
  ALLOCATE ( tsdeep (nx, ny)     )

  print*,'...NT in average = ', nt


  DO n = 1, nt
    ! Note:  As of now, 2D variables are stored in 3D arrays in WRF FDDA file.
    CALL get_var_3d_real_cdf (fl, 'T2_NDG_OLD', t2m(:,:,n), nx, ny, nz, &
                              it+n-1, rcode)
    IF ( rcode /= 0 ) THEN
      WRITE (6,9750) 'T2_NDG_OLD', TRIM(fl), n
      GOTO 1001
    ENDIF
  ENDDO

  CALL ncclos (cdfid, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9200) TRIM(fl)
    GOTO 1001
  ENDIF

!-------------------------------------------------------------------------------
! Compute average 2-m temperature over user-defined period to use for
! initial deep soil temperature.
!-------------------------------------------------------------------------------

  DO j = 1, ny
    DO i = 1, nx
      tsdeep(i,j) = SUM(t2m(i,j,1:nt)) / nt
    ENDDO
  ENDDO

  tslb(:,:,ns) = tsdeep(:,:)
!-------------------------------------------------------------------------------
! Write deep soil temperature to WRF input file for next simulation.
!-------------------------------------------------------------------------------

  fl = file_wrfin_next

  cdfid = ncopn (fl, ncwrite, rcode)  ! <-- open with WRITE option
  IF ( rcode /= 0 ) THEN
    WRITE (6,9000) TRIM(fl)
    GOTO 1001
  ENDIF

  rcode = nf_inq_varid (cdfid, 'TSLB', varid)  ! soil temperature (all layers)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9400) 'TSLB', TRIM(fl)
    GOTO 1001
  ENDIF

  start(1) = 1   ;  count(1) = nx
  start(2) = 1   ;  count(2) = ny
  start(3) = 1   ;  count(3) = ns
  start(4) = 1   ;  count(4) = 1     ! only replace the first time period

  rcode = nf_put_vara_real (cdfid, varid, start, count, tslb)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9800) 'TSLB', TRIM(fl)
    GOTO 1001
  ENDIF

  CALL ncclos (cdfid, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9200) TRIM(fl)
    GOTO 1001
  ENDIF

!-------------------------------------------------------------------------------
! Deallocate arrays.
!-------------------------------------------------------------------------------

  DEALLOCATE ( t2m    )
  DEALLOCATE ( tsdeep )

  RETURN

!-------------------------------------------------------------------------------
! Error-handling section.
!-------------------------------------------------------------------------------

 9000 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFAVERAGE',                          &
              /, 1x, '***   ERROR OPENING WRF FILE',                        &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, 70('*'))

 9050 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFAVERAGE',                          &
              /, 1x, '***   VARIABLE ', a, ' WAS NOT FOUND',                &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, 70('*'))

 9100 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFAVERAGE',                          &
              /, 1x, '***   ERROR RETRIEVING TIMES FROM WRF FILE',          &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1X, '***   RCODE = ', i3,                                  &
              /, 1x, 70('*'))

 9200 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFAVERAGE',                          &
              /, 1x, '***   ERROR CLOSING WRF NETCDF FILE',                 &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, 70('*'))

 9300 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFAVERAGE',                          &
              /, 1x, '***   DID NOT FIND DESIRED TIME IN WRF OUTPUT',       &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, '***   TIME = ', a,                                    &
              /, 1x, 70('*'))

 9350 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFAVERAGE',                          &
              /, 1x, '***   AVERAGING TIME PERIOD EXCEEDS AVAILABLE DATA',  &
              /, 1x, '***   NT BASED ON AVG_PERIOD_2M = ', i3,              &
              /, 1x, '***   N_TIMES IN FILE = ', i3,                        &
              /, 1x, 70('*'))

 9400 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFAVERAGE',                          &
              /, 1x, '***   VARIABLE ', a, ' WAS NOT FOUND',                &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, 70('*'))

 9500 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFAVERAGE',                          &
              /, 1x, '***   ERROR RETRIEVING VARIABLE DIMIDS',              &
              /, 1X, '***   VARIABLE = ', a,                                &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, 70('*'))

 9600 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFAVERAGE',                          &
              /, 1x, '***   ERROR RETRIEVING VARIABLE DIMENSIONS',          &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1X, '***   VARIABLE = ', a,                                &
              /, 1x, '***   DIMENSION = ', a,                               &
              /, 1x, 70('*'))

 9700 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFAVERAGE',                          &
              /, 1x, '***   ERROR RETRIEVING VARIABLE FROM WRF FILE',       &
              /, 1X, '***   VARIABLE = ', a,                                &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, 70('*'))

 9750 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFAVERAGE',                          &
              /, 1x, '***   ERROR RETRIEVING VARIABLE FROM WRF FILE',       &
              /, 1X, '***   VARIABLE = ', a,                                &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, '***   COUNTER = ', i3,                                &
              /, 1x, 70('*'))

 9800 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFAVERAGE',                          &
              /, 1x, '***   ERROR WRITING VARIABLE TO WRF FILE',            &
              /, 1X, '***   VARIABLE = ', a,                                &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, 70('*'))

 1001 CALL graceful_stop (pname)
      RETURN

END SUBROUTINE wrfaverage
