SUBROUTINE wrfswapdata

!-------------------------------------------------------------------------------
! Name:     WRF Swap Data
! Purpose:  Extracts data for a specific time from one file and overwrites
!           those fields in another file with those data.
! Revised:  11 May 2007  Original version.  (T. Otte)
!           23 Aug 2007  Added option to replace deep soil temperature with
!                        2-m temperature averaged from analyses over
!                        user-defined period.  (T. Otte)
!-------------------------------------------------------------------------------

  USE date_pack
  USE files
  USE wrf_netcdf

  IMPLICIT NONE

  INCLUDE 'netcdf.inc'

  INTEGER                      :: cdfid
  INTEGER                      :: count     ( 10 )
  INTEGER                      :: dimids    ( 10 )
  CHARACTER*256                :: fl
  INTEGER                      :: i
  INTEGER                      :: idtsec
  INTEGER                      :: it
  INTEGER,       PARAMETER     :: max_times = 150
  INTEGER                      :: n_times
  INTEGER                      :: ns
  INTEGER                      :: nx
  INTEGER                      :: ny
  CHARACTER*16,  PARAMETER     :: pname     = 'WRFSWAPDATA'
  INTEGER                      :: rcode
  REAL,          ALLOCATABLE   :: smois     ( : , : , : )
  INTEGER                      :: start     ( 10 )
  CHARACTER*19                 :: time_i_want
  CHARACTER*80                 :: times     ( max_times )
  REAL,          ALLOCATABLE   :: tslb      ( : , : , : )
  INTEGER,       PARAMETER     :: ttol      = 300           ! [sec]
  INTEGER                      :: varid

!-------------------------------------------------------------------------------
! If persisting soil moisture and/or soil temperature from previous run,
! make sure that soil moisture and soil temperature are in the output file.
!-------------------------------------------------------------------------------

  fl = file_wrfout_last

  cdfid = ncopn (fl, ncnowrit, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9000) TRIM(fl)
    GOTO 1001
  ENDIF

  IF ( do_msoil ) THEN
    rcode = nf_inq_varid (cdfid, 'SMOIS', varid)  ! soil moisture (all layers)
    IF ( rcode /= 0 ) THEN
      WRITE (6,9050) 'SMOIS', TRIM(fl)
      GOTO 1001
    ENDIF
  ENDIF

  IF ( do_tsoil ) THEN
    rcode = nf_inq_varid (cdfid, 'TSLB', varid)  ! soil temp (all layers)
    IF ( rcode /= 0 ) THEN
      WRITE (6,9050) 'TSLB', TRIM(fl)
      GOTO 1001
    ENDIF
  ENDIF

  CALL ncclos (cdfid, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9200) TRIM(fl)
    GOTO 1001
  ENDIF

!-------------------------------------------------------------------------------
! Get initial time from WRF input for next simulation.  This is TIME_I_WANT.
!-------------------------------------------------------------------------------

  fl = file_wrfin_next

  cdfid = ncopn (fl, ncnowrit, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9000) TRIM(fl)
    GOTO 1001
  ENDIF

  CALL get_times_cdf (fl, times, n_times, max_times, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl), rcode
    GOTO 1001
  ENDIF

  time_i_want = times(1)(1:19)
  print*,'...TIME_I_WANT = ', TRIM(time_i_want)

  CALL ncclos (cdfid, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9200) TRIM(fl)
    GOTO 1001
  ENDIF

!-------------------------------------------------------------------------------
! Search for desired time (TIME_I_WANT) in WRF output from last simulation.
!-------------------------------------------------------------------------------

  fl = file_wrfout_last

  cdfid = ncopn (fl, ncnowrit, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9000) TRIM(fl)
    GOTO 1001
  ENDIF

  CALL get_times_cdf (fl, times, n_times, max_times, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9100) TRIM(fl), rcode
    GOTO 1001
  ENDIF

  findit: DO i = 1, n_times
    CALL geth_idts (times(i)(1:19), time_i_want, idtsec)
    IF ( ABS(idtsec) <= ttol ) THEN  ! found TIME_I_WANT
      it = i
      EXIT findit
    ENDIF
  ENDDO findit
  IF ( i > n_times ) THEN
    WRITE (6,9300) TRIM(fl), time_i_want
    GOTO 1001
  ENDIF

!-------------------------------------------------------------------------------
! Extract soil moisture and soil temperature from WRF output file from
! last simulation.
!-------------------------------------------------------------------------------

  IF ( do_msoil ) THEN  ! read soil moisture

  rcode = nf_inq_varid (cdfid, 'SMOIS', varid)  ! soil moisture (all layers)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9400) 'SMOIS', TRIM(fl)
    GOTO 1001
  ENDIF

  rcode = nf_inq_vardimid (cdfid, varid, dimids)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9500) 'SMOIS', TRIM(fl)
    GOTO 1001
  ENDIF

  rcode = nf_inq_dimlen (cdfid, dimids(1), nx)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9600) TRIM(fl), 'SMOIS', 'nx'
    GOTO 1001
  ENDIF

  rcode = nf_inq_dimlen (cdfid, dimids(2), ny)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9600) TRIM(fl), 'SMOIS', 'ny'
    GOTO 1001
  ENDIF

  rcode = nf_inq_dimlen (cdfid, dimids(3), ns)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9600) TRIM(fl), 'SMOIS', 'ns'
    GOTO 1001
  ENDIF

  print*,'+++ Preparing to continue soil moisture from previous run'
  print*,'...SMOIS: nx, ny, ns = ', nx, ny, ns

  ALLOCATE ( smois (nx, ny, ns) )

  CALL get_var_3d_real_cdf (fl, 'SMOIS', smois, nx, ny, ns, it, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9700) 'SMOIS', TRIM(fl)
    GOTO 1001
  ENDIF

  ENDIF  ! do_msoil read


  IF ( do_tsoil ) THEN  ! read soil temperature

  rcode = nf_inq_varid (cdfid, 'TSLB', varid)  ! soil temperature (all layers)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9400) 'TSLB', TRIM(fl)
    GOTO 1001
  ENDIF

  rcode = nf_inq_vardimid (cdfid, varid, dimids)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9500) 'TSLB', TRIM(fl)
    GOTO 1001
  ENDIF

  rcode = nf_inq_dimlen (cdfid, dimids(1), nx)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9600) TRIM(fl), 'TSLB', 'nx'
    GOTO 1001
  ENDIF

  rcode = nf_inq_dimlen (cdfid, dimids(2), ny)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9600) TRIM(fl), 'TSLB', 'ny'
    GOTO 1001
  ENDIF

  rcode = nf_inq_dimlen (cdfid, dimids(3), ns)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9600) TRIM(fl), 'TLSB', 'ns'
    GOTO 1001
  ENDIF

  print*,'+++ Preparing to continue soil temperature from previous run'
  print*,'...TSLB:  nx, ny, ns = ', nx, ny, ns

  ALLOCATE ( tslb (nx, ny, ns) )

  CALL get_var_3d_real_cdf (fl, 'TSLB', tslb, nx, ny, ns, it, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9700) 'TSLB', TRIM(fl)
    GOTO 1001
  ENDIF

  ENDIF  ! do_tsoil read

  CALL ncclos (cdfid, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9200) TRIM(fl)
    GOTO 1001
  ENDIF

!-------------------------------------------------------------------------------
! Write soil moisture and soil temperature to WRF input file for next
! simulation.
!-------------------------------------------------------------------------------

  fl = file_wrfin_next

  cdfid = ncopn (fl, ncwrite, rcode)  ! <-- open with WRITE option
  IF ( rcode /= 0 ) THEN
    WRITE (6,9000) TRIM(fl)
    GOTO 1001
  ENDIF

  IF ( do_msoil ) THEN

  rcode = nf_inq_varid (cdfid, 'SMOIS', varid)  ! soil moisture (all layers)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9400) 'SMOIS', TRIM(fl)
    GOTO 1001
  ENDIF

  start(1) = 1   ;  count(1) = nx
  start(2) = 1   ;  count(2) = ny
  start(3) = 1   ;  count(3) = ns
  start(4) = 1   ;  count(4) = 1     ! only replace the first time period

  rcode = nf_put_vara_real (cdfid, varid, start, count, smois)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9800) 'SMOIS', TRIM(fl)
    GOTO 1001
  ENDIF

  ENDIF  ! do_msoil write


  IF ( do_tsoil ) THEN

  rcode = nf_inq_varid (cdfid, 'TSLB', varid)  ! soil temperature (all layers)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9400) 'TSLB', TRIM(fl)
    GOTO 1001
  ENDIF

  start(1) = 1   ;  count(1) = nx
  start(2) = 1   ;  count(2) = ny
  start(3) = 1   ;  count(3) = ns
  start(4) = 1   ;  count(4) = 1     ! only replace the first time period

  rcode = nf_put_vara_real (cdfid, varid, start, count, tslb)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9800) 'TSLB', TRIM(fl)
    GOTO 1001
  ENDIF

  ENDIF  ! do_tsoil write

  CALL ncclos (cdfid, rcode)
  IF ( rcode /= 0 ) THEN
    WRITE (6,9200) TRIM(fl)
    GOTO 1001
  ENDIF

!-------------------------------------------------------------------------------
! Deallocate arrays.
!-------------------------------------------------------------------------------

  IF ( do_msoil ) THEN
    DEALLOCATE ( smois )
  ENDIF

  IF ( do_tsoil ) THEN
    DEALLOCATE ( tslb  )
  ENDIF

  RETURN

!-------------------------------------------------------------------------------
! Error-handling section.
!-------------------------------------------------------------------------------

 9000 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFSWAPDATA',                         &
              /, 1x, '***   ERROR OPENING WRF FILE',                        &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, 70('*'))

 9050 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFSWAPDATA',                         &
              /, 1x, '***   VARIABLE ', a, ' WAS NOT FOUND',                &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, 70('*'))

 9100 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFSWAPDATA',                         &
              /, 1x, '***   ERROR RETRIEVING TIMES FROM WRF FILE',          &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1X, '***   RCODE = ', i3,                                  &
              /, 1x, 70('*'))

 9200 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFSWAPDATA',                         &
              /, 1x, '***   ERROR CLOSING WRF NETCDF FILE',                 &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, 70('*'))

 9300 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFSWAPDATA',                         &
              /, 1x, '***   DID NOT FIND DESIRED TIME IN WRF OUTPUT',       &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, '***   TIME = ', a,                                    &
              /, 1x, 70('*'))

 9400 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFSWAPDATA',                         &
              /, 1x, '***   VARIABLE ', a, ' WAS NOT FOUND',                &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, 70('*'))

 9500 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFSWAPDATA',                         &
              /, 1x, '***   ERROR RETRIEVING VARIABLE DIMIDS',              &
              /, 1X, '***   VARIABLE = ', a,                                &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, 70('*'))

 9600 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFSWAPDATA',                         &
              /, 1x, '***   ERROR RETRIEVING VARIABLE DIMENSIONS',          &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1X, '***   VARIABLE = ', a,                                &
              /, 1x, '***   DIMENSION = ', a,                               &
              /, 1x, 70('*'))

 9700 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFSWAPDATA',                         &
              /, 1x, '***   ERROR RETRIEVING VARIABLE FROM WRF FILE',       &
              /, 1X, '***   VARIABLE = ', a,                                &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, 70('*'))

 9800 FORMAT (/, 1x, 70('*'),                                               &
              /, 1x, '*** SUBROUTINE: WRFSWAPDATA',                         &
              /, 1x, '***   ERROR WRITING VARIABLE TO WRF FILE',            &
              /, 1X, '***   VARIABLE = ', a,                                &
              /, 1x, '***   FILE = ', a,                                    &
              /, 1x, 70('*'))

 1001 CALL graceful_stop (pname)
      RETURN

END SUBROUTINE wrfswapdata
