!
!##################################################################
!##################################################################
!######                                                      ######
!######                   PROGRAM JOINWRF                    ######
!######                                                      ######
!######                     Developed by                     ######
!######     Center for Analysis and Prediction of Storms     ######
!######                University of Oklahoma                ######
!######                                                      ######
!##################################################################
!##################################################################
!
PROGRAM joinwrf
!
!-----------------------------------------------------------------------
!
!  PURPOSE:
!
!  This program joins WRF history files in patches into one large piece.
!
!-----------------------------------------------------------------------
!
!  AUTHOR: Yunheng Wang (04/25/2007)
!
!  MODIFICATION HISTORY:
!
!-----------------------------------------------------------------------
!
#ifdef _PARALLEL_
  !USE mpi
#endif  
  IMPLICIT NONE
#ifdef _PARALLEL_
  include "mpif.h"
#endif  

  INTEGER, PARAMETER :: nmaxvars   = 300
  INTEGER, PARAMETER :: nmaxwrffil = 1000
  INTEGER, PARAMETER :: nmaxprocs  = 10000

!-----------------------------------------------------------------------
!
! NAMLIST variables
!
!-----------------------------------------------------------------------

  INTEGER            :: namelist_input_unit

  CHARACTER(LEN=256) :: dir_extd            ! directory of external data
  INTEGER            :: io_form
  CHARACTER(LEN=19)  :: init_time_str,start_time_str,end_time_str
  CHARACTER(LEN=11)  :: history_interval
  INTEGER            :: grid_id

  NAMELIST /wrfdfile/ dir_extd,init_time_str,io_form,grid_id,           &
                      start_time_str,history_interval,end_time_str

  INTEGER            :: proc_start_x
  INTEGER            :: proc_start_y
  INTEGER            :: nproc_x
  INTEGER            :: nproc_y
  INTEGER            :: nproc_xin
  NAMELIST /patches/ proc_start_x, proc_start_y, nproc_x, nproc_y,nproc_xin

  CHARACTER(LEN=256)                      :: outdirname
  CHARACTER(LEN=5)                        :: outfiletail
  INTEGER                                 :: nvarout
  CHARACTER(LEN=20), DIMENSION(NMAXVARS)  :: varlist
  LOGICAL                                 :: attadj
  INTEGER                                 :: deflate_level
  NAMELIST /output/ outdirname,outfiletail,nvarout,varlist,attadj,deflate_level

  INTEGER :: debug
  NAMELIST /debugging/ debug

!-----------------------------------------------------------------------
!
! Misc. local variables
!
!-----------------------------------------------------------------------

  CHARACTER(LEN=256) :: namelist_filename

  INTEGER :: strlen,istatus
  INTEGER :: i,j,n
  INTEGER, DIMENSION(NMAXPROCS) :: nprocs ! Processor number for a given tile.

  CHARACTER(LEN=256), DIMENSION(NMAXWRFFIL) :: filenames
  INTEGER :: nfiles
  INTEGER :: abstimes, abstimei, abstimee
  INTEGER :: ids,ide,jds,jde,idss,idse,jdss,jdse

  INTEGER :: year,month,day,hour,minute,second

  INTEGER :: rank
  INTEGER :: numtasks

#ifdef _PARALLEL_
  INTEGER :: ierr
  integer :: itag
  integer :: isize
  integer :: other_rank
  integer :: mpistatus
#endif

!-----------------------------------------------------------------------
!
! External functions
!
!-----------------------------------------------------------------------

  CHARACTER(LEN=20) :: upcase
!
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
!
! Begining of executable code below
!
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
!


#ifdef _PARALLEL_     

  CALL MPI_INIT(ierr)
  IF ( ierr /= MPI_SUCCESS ) STOP "MPI_INIT"

  CALL MPI_COMM_RANK(MPI_COMM_WORLD, rank, ierr)
  IF ( ierr /= MPI_SUCCESS ) STOP "MPI_COMM_RANK"

  CALL MPI_COMM_SIZE(MPI_COMM_WORLD, numtasks, ierr)
  IF ( ierr /= MPI_SUCCESS ) STOP "MPI_COMM_WORLD"

#else

  rank = 0
  numtasks = 1
  
#endif

  IF ( rank == 0 ) THEN
     WRITE(6,'(10(/5x,a),/)')                                                   &
          '###################################################################',&
          '###################################################################',&
          '####                                                           ####',&
          '####                Welcome to JOINWRF                         ####',&
          '####                                                           ####',&
          '####   A program that reads in patches of WRF history files    ####',&
          '####          and join them into one large piece.              ####',&
          '####                                                           ####',&
          '###################################################################',&
          '###################################################################'
  ENDIF

!
!-----------------------------------------------------------------------
!
!  Read in namelist &wrfdfile
!
!-----------------------------------------------------------------------
!

  CALL getarg(1, namelist_filename)

  dir_extd = './'

  init_time_str         = '0000-00-00_00:00:00'
  start_time_str        = '0000-00-00_00:00:00'
  history_interval      = '00_00:00:00'
  end_time_str          = '0000-00-00_00:00:00'

  io_form               = 7
  grid_id               = 1

  IF ( namelist_filename == "-" ) THEN
     namelist_input_unit = 5
  ELSE
     namelist_input_unit = 15
     OPEN(namelist_input_unit, file=TRIM(namelist_filename), status='old', form='formatted', action='read')
  ENDIF

  READ(namelist_input_unit,wrfdfile,ERR=999)
  IF ( RANK == 0 ) THEN
     WRITE(6,'(2x,a)') 'Namelist wrfdfile read in successfully.'
  ENDIF

  strlen = LEN_TRIM(dir_extd)
  IF ( strlen > 0 ) THEN
    IF ( dir_extd(strlen:strlen) /= '/' ) THEN
      dir_extd(strlen+1:strlen+1) = '/'
      strlen = strlen + 1
    ENDIF
  ELSE
    dir_extd = './'
  ENDIF
  
  IF ( io_form /= 7 ) THEN
    WRITE(6,'(1x,a)') 'ERROR: Only netCDF format is supported at present.'
    STOP
  ENDIF

  IF ( rank == 0 ) THEN
     WRITE(6,'(5x,3a)')     'dir_extd = ''', TRIM(dir_extd),''','
     WRITE(6,'(5x,a,i3,a)') 'io_form  = ', io_form,','
     WRITE(6,'(5x,a,i3,a)') 'grid_id  = ', grid_id,','
     WRITE(6,'(5x,3a)')     'init_time_str    = ''', init_time_str,''','
     WRITE(6,'(5x,3a)')     'start_time_str   = ''', start_time_str,''','
     WRITE(6,'(5x,a,8x,2a)')'history_interval = ''', history_interval,''','
     WRITE(6,'(5x,3a)')     'end_time_str     = ''', end_time_str,''','
  ENDIF
!
!-----------------------------------------------------------------------
!
!  Read in namelist &patches
!
!-----------------------------------------------------------------------
!
  proc_start_x = 0
  proc_start_y = 0
  nproc_x = 1
  nproc_y = 1
  nproc_xin = 0

  READ(namelist_input_unit,patches,ERR=999)
  IF ( rank == 0 ) THEN
     WRITE(6,'(/,2x,a)') 'Namelist arpsgrid read in successfully.'
     WRITE(6,'(5(5x,a,i3,a,/))') 'proc_start_x   = ', proc_start_x,',',         &
          &                      'proc_start_y   = ', proc_start_y,',',         &
          &                      'nproc_x        = ', nproc_x,',',              &
          &                      'nproc_y        = ', nproc_y,',',              &
          &                      'nproc_xin      = ', nproc_xin,','
  ENDIF

!
!-----------------------------------------------------------------------
!
!  Read in namelist &output and &debugging
!
!-----------------------------------------------------------------------
!
  outdirname = './'
  outfiletail= ''
  nvarout    = 0
  deflate_level = 0
  varlist(:) = ' '
  attadj     = .FALSE.

  READ(namelist_input_unit,output,ERR=999)
  IF ( RANK == 0 ) THEN
     WRITE(6,'(/,2x,a)') 'Namelist output was successfully read.'
  ENDIF

  strlen = LEN_TRIM(outdirname)
  IF(strlen > 0) THEN
    IF(outdirname(strlen:strlen) /= '/') THEN
      outdirname(strlen+1:strlen+1) = '/'
      strlen = strlen + 1
    ENDIF
  ELSE
    outdirname = './'
  ENDIF
  
  IF ( rank == 0 ) THEN
     WRITE(6,'(5x,3a)' )    'outdirname    = ''', TRIM(outdirname),''','
     WRITE(6,'(5x,3a)' )    'outfiltail    = ''', TRIM(outfiletail),''','
     WRITE(6,'("     deflate_level = ", I4)' )  deflate_level
     WRITE(6,'(5x,a,I3,a)') 'nvarout    = ', nvarout,','
  ENDIF
  DO n = 1,nvarout-1
    varlist(n) = upcase(varlist(n))
    IF ( RANK == 0 ) THEN
       WRITE(6,'(7x,a,I3,3a)') 'varlist(',n,') = ''', TRIM(varlist(n)),''','
    ENDIF
  ENDDO
  IF (nvarout > 0) THEN
    nvarout = nvarout+1
    varlist(nvarout) = 'Times'
    IF ( rank == 0 ) THEN
       WRITE(6,'(7x,a,I3,3a)') 'varlist(',nvarout,') = ''', TRIM(varlist(nvarout)),''','
    ENDIF
  ENDIF
  IF ( rank == 0 ) THEN
     WRITE(6,'(5x,a,L4,a)') 'attadj    = ', attadj,','
  ENDIF

  debug = 0
  READ(namelist_input_unit,debugging,ERR=999)
  IF ( rank == 0 ) THEN
     WRITE(6,'(/,2x,a)'   ) 'Namelist debugging was successfully read.'
     WRITE(6,'(5x,a,i3,a,/)') 'debug = ', debug,','
  ENDIF

  IF ( namelist_filename /= "-" ) THEN
     CLOSE(namelist_input_unit)
  ENDIF

  istatus = 0

!-----------------------------------------------------------------------
!
! Prepare for reading WRF files
!
!-----------------------------------------------------------------------

  READ(end_time_str,    '(I4.4,5(1X,I2.2))')      &
                  year,month,day,hour,minute,second
  CALL ctim2abss(year,month,day,hour,minute,second,abstimee)

  READ(history_interval,'(I2.2,3(1X,I2.2))')      &
                                     day,hour,minute,second
  abstimei = day*24*3600+hour*3600+minute*60+second

  READ(start_time_str,  '(I4.4,5(1X,I2.2))')      &
                  year,month,day,hour,minute,second
  CALL ctim2abss(year,month,day,hour,minute,second,abstimes)

  if ( nproc_xin < 1 ) then
     write(6,'("Set NPROC_XIN in namelist")')
     STOP
  endif

  n = 0
  DO j = 0,nproc_y-1
    DO i = 0,nproc_x-1
      n = n+1
      nprocs(n) = (j+proc_start_y)*nproc_xin + proc_start_x + i
    ENDDO
  ENDDO

!-----------------------------------------------------------------------
!
! Check file and get dimensions
!
!-----------------------------------------------------------------------

  filenames(:) = ' '

  IF ( rank == 0 ) THEN

     CALL check_files_dimensions(NMAXWRFFIL,grid_id,io_form,               &
          &     nprocs,nproc_x,nproc_y,abstimes,abstimei,abstimee,dir_extd,   &
          &     filenames,nfiles,ids,ide,idss,idse,jds,jde,jdss,jdse,istatus)

#ifdef _PARALLEL_

  ELSE

     nfiles = -999999
     ids    = -999999
     ide    = -999999
     idss   = -999999
     idse   = -999999
     jds    = -999999
     jde    = -999999
     jdss   = -999999
     jdse   = -999999

#endif


  ENDIF

#ifdef _PARALLEL_

  call mpi_bcast(nfiles, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
  IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BCAST"

  call mpi_bcast(ids, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
  IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BCAST"

  call mpi_bcast(ide, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
  IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BCAST"

  call mpi_bcast(jds, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
  IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BCAST"

  call mpi_bcast(jde, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
  IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BCAST"

  call mpi_bcast(idss, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
  IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BCAST"

  call mpi_bcast(idse, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
  IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BCAST"

  call mpi_bcast(jdss, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
  IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BCAST"

  call mpi_bcast(jdse, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
  IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BCAST"

  call mpi_bcast(filenames, nfiles*256, MPI_CHAR, 0, MPI_COMM_WORLD, ierr)
  IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BCAST"

#endif

  IF ( istatus /= 0 ) GO TO 100

  IF ( rank == 0 ) THEN
     WRITE(6,'(/,1x,a)') '*****************************'

     WRITE(6,'(1x,2(2(a,I4),a,/33x,2(a,I4),a,/,24x))')                      &
          'The joined subdomain is: stag - ids = ',ids, ', ide = ',ide,';', &
          &                               'jds = ',jds, ', jde = ',jde,'.', &
          &                      'unstag - idss= ',idss,', idse= ',idse,';',&
          &                               'jdss= ',jdss,', jdse= ',jdse,'.'
  ENDIF

!-----------------------------------------------------------------------
!
! Join files
!
!-----------------------------------------------------------------------

  IF (nvarout == 0) nvarout = nmaxvars

#ifdef _PARALLEL_
  CALL mpi_barrier(MPI_COMM_WORLD, ierr)
  IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BARRIER"
#endif

  CALL joinwrfncdf(filenames,nfiles,rank, numtasks, attadj, deflate_level, nprocs,n,  &
       &           ids,ide,idss,idse,jds,jde,jdss,jdse,                  &
       &           outdirname,outfiletail,nvarout,varlist,debug,istatus)

#ifdef _PARALLEL_
  CALL mpi_barrier(MPI_COMM_WORLD, ierr)
  IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BARRIER"
#endif

  GO TO 100

!-----------------------------------------------------------------------
!
! Just before termination
!
!-----------------------------------------------------------------------

  999  WRITE(6,'(1x, a,a)') 'Error reading NAMELIST file. Job stopped.'

#ifdef _PARALLEL_
  CALL mpi_barrier(MPI_COMM_WORLD, ierr)
  IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BARRIER"
  CALL mpi_finalize(ierr)
  IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_FINALIZE"
#endif

  STOP

  100    CONTINUE
  IF ( istatus == 0 ) THEN
    WRITE(6,'(/,4x,a)') '==== Program JOINWRF terminated normally ===='
  ELSE
    WRITE(6,'(/,4x,a,I3,a/)') '**** Program JOINWRF terminated with error = ',istatus,' ****'
  ENDIF

#ifdef _PARALLEL_
  CALL mpi_barrier(MPI_COMM_WORLD, ierr)
  IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BARRIER"
  CALL mpi_finalize(ierr)
  IF ( ierr /= MPI_SUCCESS ) stop "Problem with MPI_FINALIZE"
#endif

END PROGRAM joinwrf

!
! Convert a character string to upper case
!
FUNCTION upcase(string) RESULT(upper)

  IMPLICIT NONE

  INTEGER, PARAMETER :: lenstr = 20 

  CHARACTER(LEN=lenstr), INTENT(IN) :: string
  CHARACTER(LEN=lenstr)             :: upper

  INTEGER :: j

  DO j = 1,lenstr
    IF(string(j:j) >= "a" .AND. string(j:j) <= "z") THEN
      upper(j:j) = ACHAR(IACHAR(string(j:j)) - 32)
    ELSE
      upper(j:j) = string(j:j)
    ENDIF
  ENDDO
END FUNCTION upcase
