!
!##################################################################
!##################################################################
!######                                                      ######
!######           SUBROUTINE check_files_dimensions          ######
!######                                                      ######
!######                     Developed by                     ######
!######     Center for Analysis and Prediction of Storms     ######
!######                University of Oklahoma                ######
!######                                                      ######
!##################################################################
!##################################################################
!
SUBROUTINE check_files_dimensions(MAXWRFFIL,grid_id,io_form,            &
     nprocs,nproc_x,nproc_y,abstimes,abstimei,abstimee,      &
     dir_extd,extdname,nextdfil,                             &
     ids,ide,idss,idse,jds,jde,jdss,jdse,istatus)
  !
  !-----------------------------------------------------------------------
  !
  ! PURPOSE: Check the existence of WRF files to be read and return the 
  !          valid file number, file names and the domain grid indices.
  !          
  !-----------------------------------------------------------------------
  !
  !  AUTHOR:
  !  Yunheng Wang (04/26/2007)
  !
  !  MODIFICATION HISTORY:
  !
  !-----------------------------------------------------------------------

  IMPLICIT NONE

  INTEGER, INTENT(IN)    :: MAXWRFFIL
  INTEGER, INTENT(IN)    :: grid_id
  INTEGER, INTENT(IN)    :: io_form
  INTEGER, INTENT(IN)    :: abstimes, abstimei, abstimee
  INTEGER, INTENT(IN)    :: nproc_x,nproc_y
  INTEGER, INTENT(IN)    :: nprocs(nproc_x*nproc_y)
  CHARACTER(LEN=256), INTENT(IN)  :: dir_extd
  CHARACTER(LEN=256), INTENT(OUT) :: extdname(MAXWRFFIL)
  INTEGER,            INTENT(OUT) :: nextdfil
  INTEGER,            INTENT(OUT) :: ids, ide, jds, jde
  INTEGER,            INTENT(OUT) :: idss,idse,jdss,jdse
  INTEGER,            INTENT(OUT) :: istatus

  !-----------------------------------------------------------------------
  !
  ! Misc. local variables
  !
  !-----------------------------------------------------------------------

  INTEGER :: year, month, day, hour, minute, second
  INTEGER :: ifile, npx, npy, n
  INTEGER :: ips, ipe, jps, jpe, ipss, ipse, jpss, jpse
  INTEGER :: ipssv, ipesv, jpssv, jpesv
  INTEGER :: nx

  CHARACTER(LEN=256) :: tmpstr

  LOGICAL :: fexist
  LOGICAL :: dset = .FALSE.
  LOGICAL :: in_a_row = .FALSE.

  !@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
  !
  ! Begining of executable code ....
  !
  !@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

  ids  = 99999999
  ide  = 0
  idss = 99999999
  idse = 0

  jds  = 99999999
  jde  = 0
  jdss = 99999999
  jdse = 0

  nextdfil    = 0
  extdname(:) = ' ' 
  istatus     = 0

  WRITE(6,'(1x,a,/,1x,a,/)')                             &
       '============================','WRF files to be read are:'

  ifile = abstimes
  IFILE_LOOP : DO WHILE (ifile <= abstimee)

     CALL abss2ctim(ifile,year,month,day,hour,minute,second)

     nextdfil = nextdfil + 1
     WRITE(extdname(nextdfil),'(a,a,I2.2,a,I4.4,5(a,I2.2))')             &
          &       TRIM(dir_extd),'wrfout_d',grid_id,'_',                 &
          &       year,'-',month,'-',day,'_',hour,':',minute,':',second

     ipssv = 0
     ipesv = 0
     jpssv = 0
     jpesv = 0

     n = 0
     YLOOP : DO npy = 1,nproc_y
        in_a_row = .FALSE.
        XLOOP : DO npx = 1,nproc_x

           IF (npx > 1) in_a_row = .TRUE.

           n = n+1
           WRITE(tmpstr,'(a,a,I4.4)') TRIM(extdname(nextdfil)),'_',nprocs(n)

           INQUIRE(FILE=TRIM(tmpstr), EXIST = fexist )

           IF(.NOT. fexist) THEN
              WRITE(6,'(1x,a)') 'ERROR: The WRF file ',                       &
                   TRIM(tmpstr),' does not exist.'
              STOP
           ENDIF

           CALL get_wrf_patch_indices(TRIM(tmpstr),io_form,                &
                ips,ipe,ipss,ipse,jps,jpe,jpss,jpse,nx,istatus)
           IF (istatus /= 0) EXIT

           IF (.NOT. dset) THEN
              IF (npx == 1) THEN
                 ids  = ips
                 idss = ipss
              END IF

              IF (npx == nproc_x) THEN
                 ide  = ipe
                 idse = ipse
              END IF

              IF (npy == 1) THEN
                 jds  = jps
                 jdss = jpss
              END IF

              IF (npy == nproc_y) THEN
                 jde  = jpe
                 jdse = jpse
              END IF

           END IF

           IF ( n > 1) THEN
              IF (in_a_row) THEN
                 IF (jps /= jpssv .OR. jpe /= jpesv .OR. ips /= ipesv+1) THEN
                    WRITE(6,'(/,1x,a,I4,2a,/,8x,2(a,I2),a,/,8x,a,/,8x,a,/)')  &
                         'ERROR: Patch ',n,' for file ',TRIM(tmpstr),            &
                         'at relative patch (',npx,',',npy,                      &
                         ') is not aligned in a row with its former patch.',     &
                         'Please check parameter nproc_xin. Either it was specified with a wrong number', &
                         'or the program has made a bad guess about it.'
                    STOP
                 END IF
              ELSE
                 IF (jps /= jpesv+1) THEN
                    WRITE(6,'(/,1x,a,I4,2a,/,8x,2(a,I2),a,/,8x,a,/,8x,a,/)')  &
                         'ERROR: Patch ',n,' for file ',TRIM(tmpstr),            &
                         'at relative patch (',npx,',',npy,                      &
                         ') is not aligned in column with its former patch.',    &
                         'Please check parameter nproc_xin. Either it was specified with a wrong number', &
                         'or the program has made a bad guess about it.'
                    STOP
                 ENDIF
              ENDIF
           ENDIF

           ipssv = ips
           ipesv = ipe
           jpssv = jps
           jpesv = jpe

           WRITE(6,'(3x,a,I2.2,a,I4,a,5x,a, 5x)')                            &
                'WRF file ',nextdfil,': patch - ',n,' =', TRIM(tmpstr)

        ENDDO XLOOP
     ENDDO YLOOP

     ifile = ifile + abstimei
     dset = .TRUE.

  ENDDO IFILE_LOOP


  !-----------------------------------------------------------------------
  !
  ! Validate nextdfil before return
  !
  !-----------------------------------------------------------------------

  IF(nextdfil < 1) THEN
     WRITE(6,'(a)') 'No input WRF file was valid. Please check the input file.'
     istatus = -3
     RETURN
  END IF

  IF (ide < ids .OR. jde < jds) THEN
     WRITE(6,'(1x,2(a,I4),/36x,2(a,I4),a)')                              &
          'ERROR: Domain indices are invalid: ids = ',ids,', ide = ',ide,     &
          '; jds = ',jds,', jde = ',jde,'.'
     istatus = -3
     RETURN
  END IF

  print*, 'nextdfil = ', nextdfil
  print*, 'ids, ide, jds, jde = ', ids, ide, jds, jde
  print*, 'idss, idse, jdss, jdse = ', idss, idse, jdss, jdse

END SUBROUTINE check_files_dimensions
!
!##################################################################
!##################################################################
!######                                                      ######
!######       SUBROUTINE get_wrf_patch_indices               ######
!######                                                      ######
!######                     Developed by                     ######
!######     Center for Analysis and Prediction of Storms     ######
!######                University of Oklahoma                ######
!######                                                      ######
!##################################################################
!##################################################################
!
SUBROUTINE get_wrf_patch_indices(filename,io_form,ips,ipe,ipss,ipse,    &
                                 jps,jpe,jpss,jpse,nx,istatus)

!-----------------------------------------------------------------------
!
!  PURPOSE:
!    Get the size of data patch stored in the WRF data file
!
!-----------------------------------------------------------------------
!
!  AUTHOR:
!  Yunheng Wang (04/26/2007)
!
!  MODIFICATION HISTORY:
!
!-----------------------------------------------------------------------
  use netcdf
 
  IMPLICIT NONE
 
  CHARACTER(LEN=*), INTENT(IN)  :: filename
  INTEGER,          INTENT(IN)  :: io_form
  INTEGER,          INTENT(OUT) :: ips, ipe, jps, jpe
  INTEGER,          INTENT(OUT) :: ipss,ipse,jpss,jpse
  INTEGER,          INTENT(OUT) :: nx
  INTEGER,          INTENT(OUT) :: istatus

!------------------------------------------------------------------
!
!  Misc. local variables
!
!------------------------------------------------------------------
 
  INTEGER           :: ncid
  CHARACTER(LEN=80) :: errmsg

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
!
!  Beginning of executable code...
!
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

  istatus = 0
  IF (io_form == 7) THEN

    istatus = NF90_OPEN(TRIM(filename),NF90_NOWRITE,ncid)
    IF(istatus /= NF90_NOERR)  GO TO 999

    istatus = NF90_GET_ATT(ncid,NF90_GLOBAL,'WEST-EAST_PATCH_START_STAG',ips)
    IF(istatus /= NF90_NOERR)  GO TO 999

    istatus = NF90_GET_ATT(ncid,NF90_GLOBAL,'WEST-EAST_PATCH_END_STAG',ipe)
    IF(istatus /= NF90_NOERR)  GO TO 999

    istatus = NF90_GET_ATT(ncid,NF90_GLOBAL,'WEST-EAST_PATCH_START_UNSTAG',ipss)
    IF(istatus /= NF90_NOERR)  GO TO 999

    istatus = NF90_GET_ATT(ncid,NF90_GLOBAL,'WEST-EAST_PATCH_END_UNSTAG',ipse)
    IF(istatus /= NF90_NOERR)  GO TO 999

    istatus = NF90_GET_ATT(ncid,NF90_GLOBAL,'SOUTH-NORTH_PATCH_START_STAG',jps)
    IF(istatus /= NF90_NOERR)  GO TO 999

    istatus = NF90_GET_ATT(ncid,NF90_GLOBAL,'SOUTH-NORTH_PATCH_END_STAG',jpe)
    IF(istatus /= NF90_NOERR)  GO TO 999

    istatus = NF90_GET_ATT(ncid,NF90_GLOBAL,'SOUTH-NORTH_PATCH_START_UNSTAG',jpss)
    IF(istatus /= NF90_NOERR)  GO TO 999

    istatus = NF90_GET_ATT(ncid,NF90_GLOBAL,'SOUTH-NORTH_PATCH_END_UNSTAG',jpse)
    IF(istatus /= NF90_NOERR)  GO TO 999

    istatus = NF90_GET_ATT(ncid,NF90_GLOBAL,'WEST-EAST_GRID_DIMENSION',nx)
    IF(istatus /= NF90_NOERR)  GO TO 999

    istatus = NF90_CLOSE(ncid)
    IF (istatus /= NF90_NOERR) GO TO 999
  ELSE
    istatus   = -1
    ips = 0
    ipe = 0
    ipse= 0
    jps = 0
    jpe = 0
    jpse= 0
    WRITE(6,'(1x,a,/)')       &
      'WARNING: Only support netCDF file at present for patch indices.'
  END IF
 
  RETURN

  999 CONTINUE
  errmsg = NF90_STRERROR(istatus)
  WRITE(6,'(1x,2a)') 'NetCDF error: ',errmsg
  STOP

END SUBROUTINE get_wrf_patch_indices
!
!##################################################################
!##################################################################
!######                                                      ######
!######           SUBROUTINE joinwrfncdf                     ######
!######                                                      ######
!######                     Developed by                     ######
!######     Center for Analysis and Prediction of Storms     ######
!######                University of Oklahoma                ######
!######                                                      ######
!##################################################################
!##################################################################
!
SUBROUTINE  joinwrfncdf(filenames,nfile,rank,numtasks,attadj,procs,npatch,            &
     &                  ids,ide,idss,idse,jds,jde,jdss,jdse,                  &
     &                  outdirname,filetail,nvarout,varlists,debug,istatus)
  !
  !-----------------------------------------------------------------------
  !
  ! PURPOSE: 
  !
  !    Join WRF files in netCDF patches into one large piece. 
  !
  !-----------------------------------------------------------------------
  !
  ! Author: Yunheng Wang (04/27/2007)
  !
  ! MODIFICATIONS:
  !
  !-----------------------------------------------------------------------
  !
  USE NETCDF
#ifdef _PARALLEL_
  use mpi
#endif
  IMPLICIT NONE
  INTEGER, INTENT(IN)            :: nfile
  INTEGER, INTENT(IN)            :: rank
  INTEGER, INTENT(IN)            :: numtasks
  LOGICAL, INTENT(IN)            :: attadj
  INTEGER, INTENT(IN)            :: npatch
  INTEGER, INTENT(IN)            :: procs(npatch)
  INTEGER, INTENT(IN)            :: ids,ide,idss,idse,jds,jde,jdss,jdse
  INTEGER, INTENT(INOUT)         :: nvarout
  INTEGER, INTENT(IN)            :: debug
  INTEGER, INTENT(OUT)           :: istatus

  CHARACTER(LEN=*),  INTENT(IN)  :: filenames(nfile)
  CHARACTER(LEN=*),  INTENT(IN)  :: outdirname
  CHARACTER(LEN=5),  INTENT(IN)  :: filetail
  CHARACTER(LEN=20), INTENT(IN)  :: varlists(nvarout)

  !-----------------------------------------------------------------------
  !
  ! Misc. local variables
  !
  !-----------------------------------------------------------------------

  INTEGER :: nf, nvar, n, i
  INTEGER :: strlen
  LOGICAL :: ispatch(NF90_MAX_VARS)

  CHARACTER(LEN=256) :: infilename, outfilename
  INTEGER :: finid, foutid

  INTEGER :: idsout, ideout, jdsout, jdeout
  INTEGER :: idssout, idseout, jdssout, jdseout
  !
  ! Dimension variables
  !
  CHARACTER(LEN=32), PARAMETER :: xdimname  = 'west_east_stag'
  CHARACTER(LEN=32), PARAMETER :: ydimname  = 'south_north_stag'
  CHARACTER(LEN=32), PARAMETER :: xsdimname = 'west_east'
  CHARACTER(LEN=32), PARAMETER :: ysdimname = 'south_north'
  CHARACTER(LEN=32) :: diminnames(NF90_MAX_DIMS)
  CHARACTER(LEN=32) :: dimname

  INTEGER :: nxid, nyid, nxlg, nylg, nxsid, nysid, nxslg, nyslg
  INTEGER :: narrsize, narrisizemax, narrasizemax
  INTEGER :: unlimdimid, odimid
  INTEGER :: ndims, dimid, dimlen

  INTEGER :: dimina(NF90_MAX_DIMS)         ! Dimension size in original file
  INTEGER :: dimouta(NF90_MAX_DIMS)        ! Dimension size in joined files
  INTEGER, dimension(NF90_MAX_DIMS) :: chunksizes

  !
  ! Attribute variables
  !
  CHARACTER(LEN=32), PARAMETER :: attnm_ips  = 'WEST-EAST_PATCH_START_STAG'
  CHARACTER(LEN=32), PARAMETER :: attnm_ipe  = 'WEST-EAST_PATCH_END_STAG'
  CHARACTER(LEN=32), PARAMETER :: attnm_ipss = 'WEST-EAST_PATCH_START_UNSTAG'
  CHARACTER(LEN=32), PARAMETER :: attnm_ipse = 'WEST-EAST_PATCH_END_UNSTAG'
  CHARACTER(LEN=32), PARAMETER :: attnm_jps  = 'SOUTH-NORTH_PATCH_START_STAG'
  CHARACTER(LEN=32), PARAMETER :: attnm_jpe  = 'SOUTH-NORTH_PATCH_END_STAG'
  CHARACTER(LEN=32), PARAMETER :: attnm_jpss = 'SOUTH-NORTH_PATCH_START_UNSTAG'
  CHARACTER(LEN=32), PARAMETER :: attnm_jpse = 'SOUTH-NORTH_PATCH_END_UNSTAG'
  CHARACTER(LEN=32) :: attname
  INTEGER :: ipsid,  ipeid,  jpsid,  jpeid
  INTEGER :: ipssid, ipseid, jpssid, jpseid
  INTEGER :: ips, ipe, ipss, ipse
  INTEGER :: jps, jpe, jpss, jpse
  INTEGER :: attnum, ngatts

  CHARACTER(LEN=32), PARAMETER :: attnm_ndx = 'WEST-EAST_GRID_DIMENSION'
  CHARACTER(LEN=32), PARAMETER :: attnm_ndy = 'SOUTH-NORTH_GRID_DIMENSION'
  INTEGER :: ndxid, ndyid

  !
  ! Dataset varaibles
  !
  INTEGER, PARAMETER :: MAX_RANK = 4    ! Assume the max rank is 5
  CHARACTER(LEN=32) :: varname
  INTEGER :: varid, nvars, ovarid
  INTEGER :: vartype, varndims, varnatts
  INTEGER :: vardimids(MAX_RANK),startidx(MAX_RANK), countidx(MAX_RANK)
  INTEGER :: outstart(MAX_RANK)
  INTEGER :: vardim, vdimid

  INTEGER :: varidlists(NF90_MAX_VARS), varoutidlists(NF90_MAX_VARS)

  INTEGER, ALLOCATABLE :: varari(:)
  REAL,    ALLOCATABLE :: vararr(:)
  CHARACTER(LEN=256)   :: tmpstr

#ifdef _PARALLEL_     
  integer :: ierr
  integer :: mpistatus
  integer :: itag
  integer :: other_rank
  integer, dimension(1) :: sendn, recvn
#endif

  INTERFACE
     SUBROUTINE handle_err(istat, message)
#ifdef _PARALLEL_
       USE mpi
#endif
       USE netcdf
       IMPLICIT NONE
       INTEGER, INTENT(IN) :: istat
       character(len=*), optional, intent(in) :: message
     END SUBROUTINE handle_err
  END INTERFACE



  !@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
  !
  ! Beginning of executable code below
  !
  !@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

  varidlists(:) = 0
  nxlg  = ide-ids+1
  nylg  = jde-jds+1
  nxslg = idse-idss+1
  nyslg = jdse-jdss+1

  startidx(:)  = 1
  narrisizemax = 0
  narrasizemax = 0

  istatus = 0
  NFILE_LOOP : DO nf = 1,nfile
     strlen = LEN_TRIM(filenames(nf))
     n = INDEX(filenames(nf),'/',.TRUE.)

     WRITE(outfilename,'(3a)') TRIM(outdirname),                         &
          filenames(nf)(n+1:strlen),filetail

     WRITE(infilename, '(2a,I4.4)') TRIM(filenames(nf)),'_',procs(1)

     IF (debug > 0) WRITE(6,'(1x,2a)') 'Opening file -- ',TRIM(infilename)
     istatus = nf90_open(infilename,NF90_NOWRITE,finid)   ! Open file
     IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem opening file "//trim(infilename))

     if (rank == 0) then
        IF (debug > 0) WRITE(6,'(1x,2a)') 'Creating file -- ',TRIM(outfilename)

        istatus = NF90_CREATE ( TRIM(outfilename), IOR(NF90_CLOBBER,NF90_HDF5),  foutid )
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem creating file "//trim(outfilename))

     endif

     !
     ! Set dimensions
     !
     istatus = nf90_inq_dimid(finid,xdimname,nxid)
     IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring dimid")
     istatus = nf90_inq_dimid(finid,ydimname,nyid)
     IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring dimid")

     istatus = nf90_inq_dimid(finid,xsdimname,nxsid)
     IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring dimid")
     istatus = nf90_inq_dimid(finid,ysdimname,nysid)
     IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring dimid")

     istatus = nf90_inquire(finid,unlimitedDimid=unlimdimid)
     IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring unlimited dimension")

     istatus = nf90_inquire(finid,nDimensions=ndims)
     IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring ndims")

     IF ( RANK == 0 ) THEN

        IF (debug > 0) WRITE(6,'(5x,a,I2)') 'Copying dimensions - ',ndims

        DIMID_LOOP : DO dimid = 1,ndims
           istatus = nf90_inquire_dimension(finid,dimid,name=dimname,len=dimlen)
           IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring dimension")

           diminnames(dimid) = dimname
           dimina(dimid)  = dimlen             ! Save dimension id and len
           dimouta(dimid) = dimlen             ! Output dimension id and len
           IF (dimid == nxid) THEN
              dimlen = nxlg
              dimouta(dimid) = dimlen
           ELSEIF (dimid == nxsid) THEN
              dimlen = nxslg
              dimouta(dimid) = dimlen
           ELSEIF (dimid == nyid) THEN
              dimlen = nylg
              dimouta(dimid) = dimlen
           ELSEIF (dimid == nysid) THEN
              dimlen = nyslg
              dimouta(dimid) = dimlen
           ELSEIF (dimid == unlimdimid) THEN
              dimlen = NF90_UNLIMITED
           ENDIF

           IF (debug > 0) WRITE(6,'(9x,2a)') 'Dimension name - ',TRIM(dimname)
           istatus = nf90_def_dim(foutid,dimname,dimlen,odimid)
           IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem defining dimension")
        ENDDO DIMID_LOOP

        !
        ! Set Global attributes
        !
        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_ips),attnum=ipsid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")
        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_ipe),attnum=ipeid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")

        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_ipss),attnum=ipssid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")
        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_ipse),attnum=ipseid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")

        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_jps),attnum=jpsid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")
        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_jpe),attnum=jpeid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")

        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_jpss),attnum=jpssid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")
        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_jpse),attnum=jpseid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")

        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_ndx),attnum=ndxid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")
        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_ndy),attnum=ndyid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")

        istatus = nf90_inquire(finid,nAttributes=ngatts)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring nAttributes")

        IF (debug > 0) WRITE(6,'(5x,a,I2)') 'Copying global attributes - ',ngatts

        IF (attadj) THEN
           idsout  = 1
           ideout  = ide  - ids  + 1
           idssout = 1
           idseout = idse - idss + 1

           jdsout  = 1
           jdeout  = jde  - jds  + 1
           jdssout = 1
           jdseout = jdse - jdss + 1
        ELSE
           idsout  = ids
           ideout  = ide
           idssout = idss
           idseout = idse

           jdsout  = jds
           jdeout  = jde
           jdssout = jdss
           jdseout = jdse
        ENDIF

        DO attnum = 1,ngatts

           istatus = nf90_inq_attname(finid,NF90_GLOBAL,attnum,attname)
           IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attname")

           IF (debug > 0) WRITE(6,'(9x,2a)') 'Attribute name - ',TRIM(attname)

           IF (attadj) THEN
              IF (attnum == ndxid) THEN
                 istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_ndx),nxlg)
                 IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
                 CYCLE
              ELSE IF (attnum == ndyid) THEN
                 istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_ndy),nylg)
                 IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
                 CYCLE
              END IF
           END IF

           IF (attnum == ipsid) THEN
              istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_ips),idsout)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
           ELSE IF (attnum == ipeid) THEN
              istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_ipe),ideout)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
           ELSE IF (attnum == jpsid) THEN
              istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_jps),jdsout)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
           ELSE IF (attnum == jpeid) THEN
              istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_jpe),jdeout)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
           ELSE IF (attnum == ipssid) THEN
              istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_ipss),idssout)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
           ELSE IF (attnum == ipseid) THEN
              istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_ipse),idseout)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
           ELSE IF (attnum == jpssid) THEN
              istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_jpss),jdssout)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
           ELSE IF (attnum == jpseid) THEN
              istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_jpse),jdseout)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
           ELSE 
              istatus = nf90_copy_att(finid,NF90_GLOBAL,attname,foutid,NF90_GLOBAL)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem copying attribute")
           END IF
        ENDDO

     ENDIF


     !
     ! Define variables
     !
     istatus = nf90_inquire ( finid , nVariables=nvars )
     IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem inquiring nVariables")

     IF (nvarout >= nvars) THEN
        nvarout = nvars
        DO n = 1,nvars
           varidlists(n) = n
        END DO
     ELSE
        nvar = nvarout         ! supposed to process this number
        nvarout = 0            ! actually got
        DO n = 1,nvar
           istatus = nf90_inq_varid(finid,TRIM(varlists(n)),ovarid)
           IF (istatus /= NF90_NOERR) THEN
              WRITE(6,'(1x,3a)') 'WARNING: Variable ',TRIM(varlists(n)),' not found. Skipped.'
              CYCLE
           END IF
           nvarout = nvarout + 1
           varidlists(nvarout) = ovarid
        END DO
     END IF

     IF ( rank == 0 ) THEN

        IF (debug > 0) WRITE(6,'(5x,a,I4)') 'Defining variables - ',nvarout

        DO n = 1,nvarout

           varid = varidlists(n)  
           istatus = nf90_inquire_variable ( finid, varid, name=varname, xtype=vartype, &
                ndims=varndims, dimids=vardimids, nAtts=varnatts )
           IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem (1) inquiring variable")

           IF (debug > 0) WRITE(6,'(9x,2a)') 'Variables - ',TRIM(varname)

!KWM           istatus = nf90_def_var ( foutid, varname, vartype, vardimids(1:varndims), ovarid )
!KWM           IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem defining variable")

           istatus = nf90_def_var ( foutid, varname, vartype, vardimids(1:varndims), ovarid, &
                deflate_level=2, shuffle=.TRUE.)
           IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem defining variable")

           varoutidlists(n) = ovarid

           DO attnum = 1,varnatts          ! Copy variable attributes
              istatus = nf90_inq_attname(finid,varid,attnum,attname)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem inquiring attname")

              istatus = nf90_copy_att(finid,varid,attname,foutid,ovarid)
              IF (istatus /= NF90_NOERR) &
                   CALL handle_err(istatus, "Problem (1) copying attribute "//trim(attname))
           ENDDO

        ENDDO

        istatus = nf90_enddef(foutid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem ending define mode")

        IF(debug > 0) WRITE(6,'(1x,a)') 'Merged file has been defined.'

     ENDIF


     istatus = nf90_close(finid)                              ! Close file
     IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem closing input file")

     !
     ! Write each patch to the merged file
     !

     ispatch(:) = .FALSE.
     PATCH_LOOP : DO n = 1,npatch

        if ( rank /= mod(n-1, numtasks) ) cycle

#ifdef _PARALLEL_
        call mpi_barrier(MPI_COMM_WORLD, ierr)
        if (ierr /= MPI_SUCCESS) stop "Problem with MPI_BARRIER"
#endif

        WRITE(infilename, '(2a,I4.4)') TRIM(filenames(nf)),'_',procs(n)

        IF (debug > 0) WRITE(6,'(1x,2a)') 'Opening file - ',TRIM(infilename)

        istatus = nf90_open(TRIM(infilename),NF90_NOWRITE,finid)   ! Open file
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem opening file "//trim(infilename))

        !
        ! Get patch indices
        !
        istatus = nf90_get_att(finid,NF90_GLOBAL,TRIM(attnm_ips),ips)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem getting attribute")
        istatus = nf90_get_att(finid,NF90_GLOBAL,TRIM(attnm_ipe),ipe)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem getting attribute")

        istatus = nf90_get_att(finid,NF90_GLOBAL,TRIM(attnm_ipss),ipss)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem getting attribute")
        istatus = nf90_get_att(finid,NF90_GLOBAL,TRIM(attnm_ipse),ipse)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem getting attribute")

        istatus = nf90_get_att(finid,NF90_GLOBAL,TRIM(attnm_jps),jps)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem getting attribute")
        istatus = nf90_get_att(finid,NF90_GLOBAL,TRIM(attnm_jpe),jpe)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem getting attribute")

        istatus = nf90_get_att(finid,NF90_GLOBAL,TRIM(attnm_jpss),jpss)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem getting attribute")
        istatus = nf90_get_att(finid,NF90_GLOBAL,TRIM(attnm_jpse),jpse)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem getting attribute")

        !
        ! Get and save dimension size for this patch
        !
        istatus = nf90_inquire(finid,nDimensions=ndims)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring ndims")

        dimina(:) = 0
        DO dimid = 1,ndims
           istatus = nf90_inquire_dimension(finid,dimid,name=dimname,len=dimlen)
           IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring dimension")

           diminnames(dimid) = dimname
           dimina(dimid)  = dimlen             ! Save dimension id and len
        ENDDO

        !
        ! loop over each variable
        !

        VAR_LOOP : DO nvar = 1,nvarout

           varid = varidlists(nvar)

           vardimids(:) = 0
           istatus = nf90_inquire_variable( finid, varid,name=varname, xtype=vartype, &
                ndims=varndims, dimids=vardimids)
           IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem (2) inquiring variable")

           countidx(:) = 1
           outstart(:) = 1
           narrsize = 1
           DO vardim = 1, varndims
              vdimid = vardimids(vardim)
              countidx(vardim) = dimina (vdimid)
              IF ( vdimid == nxid) THEN 
                 outstart(vardim) = ips - ids + 1     ! start relative in subdomain
                 ispatch(nvar) = .TRUE.
              ELSE IF ( vdimid == nyid) THEN
                 outstart(vardim) = jps - jds + 1
                 ispatch(nvar) = .TRUE.
              ELSE IF ( vdimid == nxsid) THEN
                 outstart(vardim) = ipss - idss + 1
                 ispatch(nvar) = .TRUE.
              ELSE IF ( vdimid == nysid) THEN
                 outstart(vardim) = jpss - jdss + 1
                 ispatch(nvar) = .TRUE.
              ELSE
                 outstart(vardim) = 1
              ENDIF

              narrsize = countidx(vardim)*narrsize
           END DO

#ifdef _PARALLEL_
           call mpi_barrier(MPI_COMM_WORLD, ierr)
           if (ierr /= MPI_SUCCESS) stop "Problem with MPI_BARRIER"
#endif

           IF ( n > 1 .AND. (.NOT. ispatch(nvar)) ) THEN
              IF (debug > 2) THEN
                 WRITE(100+rank,'(9x,3a)') 'Variable ',TRIM(varname),' skipped.'
              ENDIF
              CYCLE
           ELSE
              IF (debug > 2) THEN
                 WRITE(100+rank,'(9x,3a,I2)')                                       &
                      'Processing variables - ',TRIM(varname),' with rank = ',varndims

                 DO vardim = 1,varndims
                    vdimid = vardimids(vardim)
                    WRITE(100+rank,'(12x,a,2(a,I4))') diminnames(vdimid),            &
                         ', startidx = ',outstart(vardim),', size = ', countidx(vardim)
                 ENDDO
              ENDIF
           ENDIF

           ! do not have to merge, use values from the first file

           if (rank == 0) then
              ovarid = varoutidlists(nvar)
           endif

           SELECT CASE (vartype)

           CASE (NF90_INT)

              IF (narrsize > narrisizemax) THEN   ! Allocate input array only when necessary
                 IF (ALLOCATED(varari)) DEALLOCATE(varari, STAT = istatus)
                 ALLOCATE(varari(narrsize), STAT = istatus)
                 narrisizemax = narrsize
              END IF

              istatus = NF90_GET_VAR(finid,varid,varari,start=startidx,count=countidx)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem getting integer variable")

              if (rank == 0) then

                 !
                 ! First, put this task's data (i.e., data from task 0) to the final
                 ! NetCDF file.  
                 !
                 
                 istatus = nf90_put_var(foutid,ovarid,varari,start=outstart,count=countidx)
                 IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting integer variable")

#ifdef _PARALLEL_
                 !
                 !  Next, collect the data from the other tasks, and write them to the final
                 !  NetCDF file.
                 !

                 IF ( ispatch(nvar) ) THEN
                    do other_rank = 1, numtasks-1
                       varari = -999999
                       itag       = other_rank
                       recvn = -99999
                       outstart = -999999
                       countidx = -999999

                       call mpi_recv ( recvn , 1, MPI_INTEGER, other_rank, itag+100+nvar, &
                            MPI_COMM_WORLD, mpistatus, ierr)
                       if (ierr /= MPI_SUCCESS) stop "MPI_RECV_ERROR"
                       narrsize = recvn(size(recvn))

                       call mpi_recv ( outstart , MAX_RANK , MPI_INTEGER , other_rank, &
                            itag+200+nvar, MPI_COMM_WORLD, mpistatus, ierr)
                       if (ierr /= MPI_SUCCESS) stop "MPI_RECV_ERROR"

                       call mpi_recv ( countidx , MAX_RANK , MPI_INTEGER , other_rank, &
                            itag+300+nvar, MPI_COMM_WORLD, mpistatus, ierr)
                       if (ierr /= MPI_SUCCESS) stop "MPI_RECV_ERROR"

                       if (narrsize > size(varari)) then
                          IF (ALLOCATED(varari)) DEALLOCATE(varari, STAT = istatus)
                          ALLOCATE(varari(narrsize), STAT = istatus)
                          narrasizemax = narrsize
                       endif

                       call mpi_recv ( varari , narrsize , MPI_INTEGER , other_rank, &
                            &          itag+nvar, MPI_COMM_WORLD, mpistatus, ierr)
                       if (ierr /= MPI_SUCCESS) stop "MPI_RECV_ERROR"

                       istatus = nf90_put_var ( foutid,ovarid,varari, &
                            &                   start=outstart,count=countidx)
                       IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting float variable")

                    enddo
                 ENDIF
#endif

              else
                 
#ifdef _PARALLEL_

                 !
                 !  Send the data from this process over to task 0.
                 ! 

                 itag = rank
                 sendn = narrsize
                 call mpi_send( sendn  , 1 , MPI_INTEGER, 0, itag+100+nvar, MPI_COMM_WORLD, ierr)
                 if (ierr /= MPI_SUCCESS) stop "MPI_SEND_ERROR"

                 call mpi_send( outstart  , MAX_RANK , MPI_INTEGER, 0, itag+200+nvar, &
                      &         MPI_COMM_WORLD, ierr)
                 if (ierr /= MPI_SUCCESS) stop "MPI_SEND_ERROR"
                 call mpi_send( countidx  , MAX_RANK , MPI_INTEGER, 0, itag+300+nvar, &
                      &         MPI_COMM_WORLD, ierr)
                 if (ierr /= MPI_SUCCESS) stop "MPI_SEND_ERROR"

                 call mpi_send ( varari , narrsize , MPI_FLOAT , 0 , itag+nvar , & 
                      &          MPI_COMM_WORLD, ierr)
                 if (ierr /= MPI_SUCCESS) stop "MPI_SEND_ERROR" 

#endif

              endif

           CASE (NF90_FLOAT)

              IF (narrsize > narrasizemax) THEN   ! Allocate input array only when necessary
                 IF (ALLOCATED(vararr)) DEALLOCATE(vararr, STAT = istatus)
                 ALLOCATE(vararr(narrsize), STAT = istatus)
                 narrasizemax = narrsize
              END IF

              istatus = NF90_GET_VAR(finid,varid,vararr,start=startidx,count=countidx)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem getting float variable")

              IF ( rank == 0 ) THEN

                 !
                 ! First, put this task's data (i.e., data from task 0) to the final
                 ! NetCDF file.  
                 !

                 istatus = nf90_put_var(foutid,ovarid,vararr,start=outstart,count=countidx)
                 IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting float variable")

#ifdef _PARALLEL_
                 !
                 !  Next, collect the data from the other tasks, and write them to the final
                 !  NetCDF file.
                 !

                 IF ( ispatch(nvar) ) THEN
                    do other_rank = 1, numtasks-1
                       vararr = -1.E36
                       itag       = other_rank
                       recvn = -99999
                       outstart = -999999
                       countidx = -999999

                       call mpi_recv ( recvn , 1, MPI_INTEGER, other_rank, itag+100+nvar, &
                            MPI_COMM_WORLD, mpistatus, ierr)
                       if (ierr /= MPI_SUCCESS) stop "MPI_RECV_ERROR"
                       narrsize = recvn(size(recvn))

                       call mpi_recv ( outstart , MAX_RANK , MPI_INTEGER , other_rank, &
                            itag+200+nvar, MPI_COMM_WORLD, mpistatus, ierr)
                       if (ierr /= MPI_SUCCESS) stop "MPI_RECV_ERROR"

                       call mpi_recv ( countidx , MAX_RANK , MPI_INTEGER , other_rank, &
                            itag+300+nvar, MPI_COMM_WORLD, mpistatus, ierr)
                       if (ierr /= MPI_SUCCESS) stop "MPI_RECV_ERROR"

                       if (narrsize > size(vararr)) then
                          IF (ALLOCATED(vararr)) DEALLOCATE(vararr, STAT = istatus)
                          ALLOCATE(vararr(narrsize), STAT = istatus)
                          narrasizemax = narrsize
                       endif
                       call mpi_recv ( vararr , narrsize , MPI_FLOAT , other_rank, &
                            &          itag+nvar, MPI_COMM_WORLD, mpistatus, ierr)
                       if (ierr /= MPI_SUCCESS) stop "MPI_RECV_ERROR"

                       istatus = nf90_put_var ( foutid,ovarid,vararr, &
                            &                   start=outstart,count=countidx)
                       IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting float variable")

                    enddo
                 endif
#endif

              else

#ifdef _PARALLEL_
                 !
                 !  Send the data from this process over to task 0.
                 ! 

                 itag = rank
                 sendn = narrsize
                 call mpi_send( sendn  , 1 , MPI_INTEGER, 0, itag+100+nvar, MPI_COMM_WORLD, ierr)
                 if (ierr /= MPI_SUCCESS) stop "MPI_SEND_ERROR"

                 call mpi_send( outstart  , MAX_RANK , MPI_INTEGER, 0, itag+200+nvar, &
                      &         MPI_COMM_WORLD, ierr)
                 if (ierr /= MPI_SUCCESS) stop "MPI_SEND_ERROR"
                 call mpi_send( countidx  , MAX_RANK , MPI_INTEGER, 0, itag+300+nvar, &
                      &         MPI_COMM_WORLD, ierr)
                 if (ierr /= MPI_SUCCESS) stop "MPI_SEND_ERROR"

                 call mpi_send ( vararr , narrsize , MPI_FLOAT , 0 , itag+nvar , & 
                      &          MPI_COMM_WORLD, ierr)
                 if (ierr /= MPI_SUCCESS) stop "MPI_SEND_ERROR" 
#endif
              endif

#ifdef _PARALLEL_
              IF ( ispatch(nvar) ) THEN
                 call mpi_barrier(MPI_COMM_WORLD, ierr)
                 if (ierr /= MPI_SUCCESS) stop "Problem with MPI_BARRIER"
              endif
#endif

           CASE (NF90_CHAR)

              IF ( ispatch(nvar) ) THEN
                 write(*, '("Expected ISPATCH(NVAR) == F for character")')
                 istatus = -4
                 RETURN
              ENDIF

              istatus = NF90_GET_VAR(finid,varid,tmpstr,start=startidx,count=countidx)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem getting char variable")

              if (rank == 0) then
                 istatus = nf90_put_var(foutid,ovarid,TRIM(tmpstr),start=outstart,count=countidx)
                 IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting char variable")
              endif

           CASE DEFAULT
              WRITE(6,'(1x,a,I2)') 'ERROR: unsupported variable type = ',vartype
              istatus = -4
              RETURN
           END SELECT

        ENDDO VAR_LOOP

        istatus = nf90_close(finid)                              ! Close file
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem closing file")

#ifdef _PARALLEL_
        call mpi_barrier(MPI_COMM_WORLD, ierr)
        if (ierr /= MPI_SUCCESS) stop "Problem with MPI_BARRIER"
#endif
     ENDDO PATCH_LOOP

     if (rank == 0) then
        istatus = nf90_close(foutid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus)
     endif

  ENDDO NFILE_LOOP

END SUBROUTINE joinwrfncdf
!
SUBROUTINE handle_err(istat, message)
#ifdef _PARALLEL_
  use mpi
#endif
  use netcdf
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: istat
  character(len=*), optional, intent(in) :: message
  integer :: mpi_ierr
 
  IF (istat /= NF90_NOERR) THEN
     write(*, '(A)') TRIM(nf90_strerror(istat))
    if ( present ( message ) ) then
       write(*, '(A)') trim(message)
    endif
#ifdef _PARALLEL_
    call mpi_finalize(mpi_ierr)
    if (mpi_ierr /= MPI_SUCCESS) stop "Problem with MPI_FINALIZE"
#endif
    STOP 'NetCDF error!'
  ENDIF
 
END SUBROUTINE handle_err
