!
!##################################################################
!##################################################################
!######                                                      ######
!######           SUBROUTINE joinwrfncdf                     ######
!######                                                      ######
!######                     Developed by                     ######
!######     Center for Analysis and Prediction of Storms     ######
!######                University of Oklahoma                ######
!######                                                      ######
!##################################################################
!##################################################################
!
SUBROUTINE  joinwrfncdf(filenames,nfile,rank,numtasks,attadj,deflate_level, procs,npatch,            &
     &                  ids,ide,idss,idse,jds,jde,jdss,jdse,                  &
     &                  outdirname,filetail,nvarout,varlists,debug,istatus)
  !
  !-----------------------------------------------------------------------
  !
  ! PURPOSE: 
  !
  !    Join WRF files in netCDF patches into one large piece. 
  !
  !-----------------------------------------------------------------------
  !
  ! Author: Yunheng Wang (04/27/2007)
  !
  ! MODIFICATIONS:
  ! 
  !    KWM:  April 2012 -- Parallization, read a variable into a big array
  !                        before output.  Advantage:  Faster write,
  !                        especially for compressed variables.  
  !                        Disadvantage:  More memory needed to store the
  !                        full array before writing it out.
  !
  !-----------------------------------------------------------------------
  !
  USE NETCDF
#ifdef _PARALLEL_
  ! USE mpi
#endif
  IMPLICIT NONE
#ifdef _PARALLEL_
  INCLUDE "mpif.h"
#endif
  INTEGER, INTENT(IN)            :: nfile
  INTEGER, INTENT(IN)            :: rank
  INTEGER, INTENT(IN)            :: numtasks
  LOGICAL, INTENT(IN)            :: attadj
  INTEGER, INTENT(IN)            :: deflate_level
  INTEGER, INTENT(IN)            :: npatch
  INTEGER, INTENT(IN)            :: procs(npatch)
  INTEGER, INTENT(IN)            :: ids,ide,idss,idse,jds,jde,jdss,jdse
  INTEGER, INTENT(INOUT)         :: nvarout
  INTEGER, INTENT(IN)            :: debug
  INTEGER, INTENT(OUT)           :: istatus

  CHARACTER(LEN=*),  DIMENSION(NFILE),   INTENT(IN)  :: filenames
  CHARACTER(LEN=*),                      INTENT(IN)  :: outdirname
  CHARACTER(LEN=5),                      INTENT(IN)  :: filetail
  CHARACTER(LEN=20), DIMENSION(NVAROUT), INTENT(IN)  :: varlists

  !-----------------------------------------------------------------------
  !
  ! Misc. local variables
  !
  !-----------------------------------------------------------------------

  INTEGER :: nf, nvar, n, i
  INTEGER :: strlen
  LOGICAL :: ispatch(NF90_MAX_VARS)

  CHARACTER(LEN=256) :: infilename, outfilename, flagfilename
  INTEGER :: finid, foutid

  INTEGER :: idsout, ideout, jdsout, jdeout
  INTEGER :: idssout, idseout, jdssout, jdseout
  !
  ! Dimension variables
  !
  CHARACTER(LEN=32), PARAMETER :: xdimname  = 'west_east_stag'
  CHARACTER(LEN=32), PARAMETER :: ydimname  = 'south_north_stag'
  CHARACTER(LEN=32), PARAMETER :: zdimname  = 'bottom_top_stag'
  CHARACTER(LEN=32), PARAMETER :: xsdimname = 'west_east'
  CHARACTER(LEN=32), PARAMETER :: ysdimname = 'south_north'
  CHARACTER(LEN=32) :: diminnames(NF90_MAX_DIMS)
  CHARACTER(LEN=32) :: dimname

  INTEGER :: nxlg  ! Number of x-points in the large (merged) unstaggered grid.
  INTEGER :: nylg  ! Number of y-points in the large (merged) unstaggered grid.
  INTEGER :: nxslg ! Number of x-points in the large (merged) staggered grid (i.e., mass points)
  INTEGER :: nyslg ! Number of y-points in the large (merged) staggered grid (i.e., mass points)
  INTEGER :: nxid, nyid, nxsid, nysid
  INTEGER :: narrsize, narrisizemax, narrasizemax
  INTEGER :: unlimdimid, odimid
  INTEGER :: ndims, dimid, dimlen

  INTEGER :: dimina(NF90_MAX_DIMS)         ! Dimension size in original file
  INTEGER :: dimouta(NF90_MAX_DIMS)        ! Dimension size in joined files

  !
  ! Attribute variables
  !
  CHARACTER(LEN=32), PARAMETER :: attnm_ips  = 'WEST-EAST_PATCH_START_STAG'
  CHARACTER(LEN=32), PARAMETER :: attnm_ipe  = 'WEST-EAST_PATCH_END_STAG'
  CHARACTER(LEN=32), PARAMETER :: attnm_ipss = 'WEST-EAST_PATCH_START_UNSTAG'
  CHARACTER(LEN=32), PARAMETER :: attnm_ipse = 'WEST-EAST_PATCH_END_UNSTAG'
  CHARACTER(LEN=32), PARAMETER :: attnm_jps  = 'SOUTH-NORTH_PATCH_START_STAG'
  CHARACTER(LEN=32), PARAMETER :: attnm_jpe  = 'SOUTH-NORTH_PATCH_END_STAG'
  CHARACTER(LEN=32), PARAMETER :: attnm_jpss = 'SOUTH-NORTH_PATCH_START_UNSTAG'
  CHARACTER(LEN=32), PARAMETER :: attnm_jpse = 'SOUTH-NORTH_PATCH_END_UNSTAG'
  CHARACTER(LEN=32) :: attname
  INTEGER :: ipsid,  ipeid,  jpsid,  jpeid
  INTEGER :: ipssid, ipseid, jpssid, jpseid
  INTEGER :: attnum, ngatts

  CHARACTER(LEN=32), PARAMETER :: attnm_ndx = 'WEST-EAST_GRID_DIMENSION'
  CHARACTER(LEN=32), PARAMETER :: attnm_ndy = 'SOUTH-NORTH_GRID_DIMENSION'
  INTEGER :: ndxid ! Attribute ID (i.e., number) of the 'WEST-EAST_GRID_DIMENSION' attribute
  INTEGER :: ndyid ! Attribute ID (i.e., number) of the 'SOUTH-NORTH_GRID_DIMENSION' attribute

  !
  ! Dataset varaibles
  !
  INTEGER, PARAMETER :: MAX_RANK = 4    ! Assume the max rank is 5
  CHARACTER(LEN=32) :: varname
  INTEGER :: varid, nvars, ovarid
  INTEGER :: vartype, varndims, varnatts
  INTEGER, DIMENSION(MAX_RANK) :: vardimids,startidx, countidx
  INTEGER :: outstart(MAX_RANK)
  INTEGER, DIMENSION(MAX_RANK) :: countidxlg
  INTEGER :: vardim, vdimid

  INTEGER, DIMENSION(NF90_MAX_VARS) :: varidlists
  INTEGER, DIMENSION(NF90_MAX_VARS) :: varoutidlists

  INTEGER, ALLOCATABLE :: varari(:)
  REAL,    ALLOCATABLE :: vararr(:)
  CHARACTER(LEN=256)   :: tmpstr

  INTEGER, DIMENSION(npatch)           :: ips_patch
  INTEGER, DIMENSION(npatch)           :: ipss_patch
  INTEGER, DIMENSION(npatch)           :: jps_patch
  INTEGER, DIMENSION(npatch)           :: jpss_patch

  INTEGER, DIMENSION(npatch)           :: fnid
  REAL,    ALLOCATABLE, DIMENSION(:,:,:) :: varptr
  INTEGER, ALLOCATABLE, DIMENSION(:,:,:) :: varpti

  INTEGER :: j, k, m

#ifdef _PARALLEL_     
  ! INTEGER, dimension(NF90_MAX_DIMS) :: chunksizes
  integer :: ierr
  integer :: mpistatus
  integer :: itag
  integer :: other_rank
  integer, DIMENSION(1) :: sendn
  integer :: isize
#endif

  INTERFACE
     SUBROUTINE handle_err(istat, message)
       USE netcdf
       IMPLICIT NONE
       INTEGER, INTENT(IN) :: istat
       character(len=*), OPTIONAL, INTENT(IN) :: message
     END SUBROUTINE handle_err
  END INTERFACE

  !@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
  !
  ! Beginning of executable code below
  !
  !@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

  istatus = 0

  varidlists(:) = 0
  nxlg  = ide-ids+1
  nylg  = jde-jds+1
  nxslg = idse-idss+1
  nyslg = jdse-jdss+1

  startidx(:)  = 1
  narrisizemax = 0
  narrasizemax = 0

  NFILE_LOOP : DO nf = 1,nfile

     IF ( rank == 0 ) THEN

        strlen = LEN_TRIM(filenames(nf))
        n = INDEX(filenames(nf),'/',.TRUE.)

        WRITE(outfilename,'(3a)') TRIM(outdirname), filenames(nf)(n+1:strlen),filetail

        WRITE(flagfilename,'(a,"/DONE.",a,a)') TRIM(outdirname), filenames(nf)(n+1:strlen),filetail

        WRITE(infilename, '(2a,I4.4)') TRIM(filenames(nf)),'_',procs(1)

        IF (debug > 0) WRITE(6,'(1x,a,I4,2a)') 'Opening file -- ',rank, ' -- ', TRIM(infilename)
        istatus = nf90_open(infilename,NF90_NOWRITE,finid)   ! Open file
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem opening file "//trim(infilename))

        IF (debug > 0) WRITE(6,'(1x,2a)') 'Creating file -- ',TRIM(outfilename)
        istatus = NF90_CREATE ( TRIM(outfilename), IOR(NF90_CLOBBER,NF90_HDF5),  foutid )
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem creating file "//trim(outfilename))

        !
        ! Set dimensions
        !
        istatus = nf90_inq_dimid(finid,xdimname,nxid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring dimid")

        istatus = nf90_inq_dimid(finid,ydimname,nyid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring dimid")

        istatus = nf90_inq_dimid(finid,xsdimname,nxsid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring dimid")

        istatus = nf90_inq_dimid(finid,ysdimname,nysid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring dimid")

        istatus = nf90_inquire(finid,unlimitedDimid=unlimdimid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring unlimited dimension")

        istatus = nf90_inquire(finid,nDimensions=ndims)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem (1) inquiring ndims")

        IF (debug > 0) WRITE(6,'(5x,a,I2)') 'Copying dimensions - ',ndims

        DIMID_LOOP : DO dimid = 1,ndims
           istatus = nf90_inquire_dimension(finid,dimid,name=dimname,len=dimlen)
           IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring dimension")

           diminnames(dimid) = dimname
           dimina(dimid)  = dimlen             ! Save dimension id and len
           dimouta(dimid) = dimlen             ! Output dimension id and len
           IF (dimid == nxid) THEN
              dimlen = nxlg
              dimouta(dimid) = dimlen
           ELSEIF (dimid == nxsid) THEN
              dimlen = nxslg
              dimouta(dimid) = dimlen
           ELSEIF (dimid == nyid) THEN
              dimlen = nylg
              dimouta(dimid) = dimlen
           ELSEIF (dimid == nysid) THEN
              dimlen = nyslg
              dimouta(dimid) = dimlen
           ELSEIF (dimid == unlimdimid) THEN
              dimlen = NF90_UNLIMITED
           ENDIF

           IF (debug > 0) WRITE(6,'(9x,2a)') 'Dimension name - ',TRIM(dimname)
           istatus = nf90_def_dim(foutid,dimname,dimlen,odimid)
           IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem defining dimension")
        ENDDO DIMID_LOOP

        !
        ! Set Global attributes
        !
        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_ips),attnum=ipsid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")
        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_ipe),attnum=ipeid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")

        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_ipss),attnum=ipssid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")
        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_ipse),attnum=ipseid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")

        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_jps),attnum=jpsid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")
        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_jpe),attnum=jpeid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")

        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_jpss),attnum=jpssid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")
        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_jpse),attnum=jpseid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")

        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_ndx),attnum=ndxid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")
        istatus = nf90_inquire_attribute(finid,NF90_GLOBAL,TRIM(attnm_ndy),attnum=ndyid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attribute")

        istatus = nf90_inquire(finid,nAttributes=ngatts)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring nAttributes")

        IF (debug > 0) WRITE(6,'(5x,a,I2)') 'Copying global attributes - ',ngatts

        IF (attadj) THEN
           idsout  = 1
           ideout  = ide  - ids  + 1
           idssout = 1
           idseout = idse - idss + 1

           jdsout  = 1
           jdeout  = jde  - jds  + 1
           jdssout = 1
           jdseout = jdse - jdss + 1
        ELSE
           idsout  = ids
           ideout  = ide
           idssout = idss
           idseout = idse

           jdsout  = jds
           jdeout  = jde
           jdssout = jdss
           jdseout = jdse
        ENDIF

        DO attnum = 1,ngatts

           istatus = nf90_inq_attname(finid,NF90_GLOBAL,attnum,attname)
           IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring attname")

           IF (debug > 0) WRITE(6,'(9x,2a)') 'Attribute name - ',TRIM(attname)

           IF (attadj) THEN
              IF (attnum == ndxid) THEN
                 istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_ndx),nxlg)
                 IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
                 CYCLE
              ELSE IF (attnum == ndyid) THEN
                 istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_ndy),nylg)
                 IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
                 CYCLE
              ENDIF
           ENDIF

           IF (attnum == ipsid) THEN
              istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_ips),idsout)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
           ELSE IF (attnum == ipeid) THEN
              istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_ipe),ideout)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
           ELSE IF (attnum == jpsid) THEN
              istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_jps),jdsout)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
           ELSE IF (attnum == jpeid) THEN
              istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_jpe),jdeout)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
           ELSE IF (attnum == ipssid) THEN
              istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_ipss),idssout)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
           ELSE IF (attnum == ipseid) THEN
              istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_ipse),idseout)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
           ELSE IF (attnum == jpssid) THEN
              istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_jpss),jdssout)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
           ELSE IF (attnum == jpseid) THEN
              istatus = NF90_PUT_ATT(foutid,NF90_GLOBAL,TRIM(attnm_jpse),jdseout)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting attribute")
           ELSE 
              istatus = nf90_copy_att(finid,NF90_GLOBAL,attname,foutid,NF90_GLOBAL)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem copying attribute")
           ENDIF
        ENDDO

     !
     ! Define variables
     !

        istatus = nf90_inquire ( finid , nVariables=nvars )
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem inquiring nVariables")

        IF (nvarout >= nvars) THEN
           nvarout = nvars
           DO n = 1,nvars
              varidlists(n) = n
           ENDDO
        ELSE
           nvar = nvarout         ! supposed to process this number
           nvarout = 0            ! actually got
           DO n = 1,nvar
              istatus = nf90_inq_varid(finid,TRIM(varlists(n)),ovarid)
              IF (istatus /= NF90_NOERR) THEN
                 WRITE(6,'(1x,3a)') 'WARNING: Variable ',TRIM(varlists(n)),' not found. Skipped.'
                 CYCLE
              ENDIF
              nvarout = nvarout + 1
              varidlists(nvarout) = ovarid
           ENDDO
        ENDIF

        IF (debug > 0) WRITE(6,'(5x,a,I4)') 'Defining variables - ',nvarout

        DO n = 1,nvarout

           varid = varidlists(n)  
           istatus = nf90_inquire_variable ( finid, varid, name=varname, xtype=vartype, &
                ndims=varndims, dimids=vardimids, nAtts=varnatts )
           IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem (1) inquiring variable")

           IF (debug > 0) WRITE(6,'(9x,2a)') 'Variables - ',TRIM(varname)

           IF ( deflate_level == 0 ) THEN
              istatus = nf90_def_var ( foutid, varname, vartype, vardimids(1:varndims), ovarid )
           ELSE
              istatus = nf90_def_var ( foutid, varname, vartype, vardimids(1:varndims), ovarid , &
                   &                   deflate_level=deflate_level, shuffle=.TRUE.)
           ENDIF

           IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem defining variable")

           varoutidlists(n) = ovarid

           ! Copy variable attributes
           DO attnum = 1,varnatts
              istatus = nf90_inq_attname(finid,varid,attnum,attname)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem inquiring attname")

              istatus = nf90_copy_att(finid,varid,attname,foutid,ovarid)
              IF (istatus /= NF90_NOERR) &
                   CALL handle_err(istatus, "Problem (1) copying attribute "//trim(attname))
           ENDDO

        ENDDO

        istatus = nf90_enddef(foutid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem ending define mode")

        IF(debug > 0) WRITE(6,'(1x,a)') 'Merged file has been defined.'

        ! 
        ! Close file
        !

        istatus = nf90_close(finid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem closing input file")

     ENDIF  ! End the initialization tasks only performed on rank==0

     !
     !  Broadcast selected bits of information to all tasks
     !

     CALL mpi_bcast(nxid, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
     IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BCAST"

     CALL mpi_bcast(nyid, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
     IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BCAST"

     CALL mpi_bcast(nxsid, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
     IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BCAST"

     CALL mpi_bcast(nysid, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
     IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BCAST"

     CALL mpi_bcast(nvarout, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
     IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BCAST"

     CALL mpi_bcast(varidlists, NF90_MAX_VARS, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
     IF ( ierr /= MPI_SUCCESS ) STOP "Problem with MPI_BCAST"

     !
     ! loop over each variable
     !

     fnid(:) = -1

     VAR_LOOP : DO nvar = 1,nvarout

        varid = varidlists(nvar)

        vardimids(:) = 0
        
#ifdef _PARALLEL_
        CALL mpi_barrier(MPI_COMM_WORLD, ierr)
        IF (ierr /= MPI_SUCCESS) STOP "Problem with MPI_BARRIER"
#endif

        PATCH_LOOP : DO n = 1,npatch

           !
           ! Don't process a patch that's being processed by another task
           !

           IF ( rank /= MOD(n-1, numtasks) ) CYCLE PATCH_LOOP

           !
           ! If the input file for the patch hasn't already been opened, 
           ! open the input file for this patch
           !

           IF ( fnid(n) < 0 ) THEN

              WRITE(infilename, '(2a,I4.4)') TRIM(filenames(nf)),'_',procs(n)
              IF (debug > 0) WRITE(6,'(1x,a,I4,2a)') 'Opening file - ',rank, " - ", TRIM(infilename)

              istatus = nf90_open(TRIM(infilename),NF90_NOWRITE,fnid(n))
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem opening file "//trim(infilename))

              !
              !  When we open a file (the first time through the variable list), read and save the 
              !  patch indices.
              !

              istatus = nf90_get_att(fnid(n),NF90_GLOBAL,TRIM(attnm_ips),ips_patch(n))
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem getting attribute")

              istatus = nf90_get_att(fnid(n),NF90_GLOBAL,TRIM(attnm_ipss),ipss_patch(n))
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem getting attribute")

              istatus = nf90_get_att(fnid(n),NF90_GLOBAL,TRIM(attnm_jps),jps_patch(n))
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem getting attribute")

              istatus = nf90_get_att(fnid(n),NF90_GLOBAL,TRIM(attnm_jpss),jpss_patch(n))
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "problem getting attribute")

           endif

           !
           ! Get and save dimension size for this patch
           !
           istatus = nf90_inquire(fnid(n),nDimensions=ndims)
           IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem (2) inquiring ndims")

           dimina(:) = 0
           DO dimid = 1,ndims
              istatus = nf90_inquire_dimension(fnid(n),dimid,name=dimname,len=dimlen)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem inquiring dimension")

              ! Save dimension name and len              
              diminnames(dimid) = dimname
              dimina(dimid)     = dimlen
           ENDDO

           istatus = nf90_inquire_variable( fnid(n), varid,name=varname, xtype=vartype, &
                ndims=varndims, dimids=vardimids)
           IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem (2) inquiring variable")

           countidx(:) = 1
           outstart(:) = 1
           narrsize = 1
           ispatch(nvar) = .FALSE.
           DO vardim = 1, varndims
              vdimid = vardimids(vardim)
              IF ( vdimid == nxid) THEN 
                 outstart(vardim) = ips_patch(n) - ids + 1     ! start relative in subdomain
                 ispatch(nvar) = .TRUE.
              ELSE IF ( vdimid == nyid) THEN
                 outstart(vardim) = jps_patch(n) - jds + 1
                 ispatch(nvar) = .TRUE.
              ELSE IF ( vdimid == nxsid) THEN
                 outstart(vardim) = ipss_patch(n) - idss + 1
                 ispatch(nvar) = .TRUE.
              ELSE IF ( vdimid == nysid) THEN
                 outstart(vardim) = jpss_patch(n) - jdss + 1
                 ispatch(nvar) = .TRUE.
              ELSE
                 outstart(vardim) = 1
              ENDIF

              countidx(vardim) = dimina (vdimid)
              narrsize = countidx(vardim)*narrsize
           ENDDO

           IF ( n > 1 .AND. (.NOT. ispatch(nvar)) ) THEN
              IF (debug > 2) THEN
                 WRITE(*,'(9x,3a)') 'Variable ',TRIM(varname),' skipped.'
              ENDIF
              CYCLE PATCH_LOOP
           ELSE
              IF (debug > 2) THEN
                 WRITE(*,'(I4, 9x,3a,I2)')                                       &
                      rank, 'Processing variables - ',TRIM(varname),' with ndims = ',varndims

                 DO vardim = 1,varndims
                    vdimid = vardimids(vardim)
                    WRITE(*,'(12x,a,2(a,I4))') diminnames(vdimid),            &
                         ', startidx = ',outstart(vardim),', size = ', countidx(vardim)
                 ENDDO
              ENDIF
           ENDIF

           SELECT CASE (vartype)

           CASE ( NF90_INT )

              IF ( narrsize > narrisizemax ) THEN   ! Allocate input array only when necessary
                 IF (ALLOCATED(varari)) DEALLOCATE(varari, STAT = istatus)
                 ALLOCATE(varari(narrsize), STAT = istatus)
                 narrisizemax = narrsize
              ENDIF

              istatus = NF90_GET_VAR(fnid(n),varid,varari,start=startidx,count=countidx)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem getting integer variable")

              IF ( rank == 0 ) THEN

                 IF ( .NOT. ALLOCATED (varpti) ) THEN
                    countidxlg(:) = 1
                    DO vardim = 1, varndims
                       countidxlg(vardim) = dimouta(vardimids(vardim))
                    ENDDO

                    ALLOCATE(varpti(countidxlg(1),countidxlg(2),countidxlg(3)))
                    varpti = -999999
                 ENDIF

                 m=0
                 DO k = outstart(3), outstart(3)+countidx(3)-1
                    DO j = outstart(2), outstart(2)+countidx(2)-1
                       DO i = outstart(1), outstart(1)+countidx(1)-1
                          m = m + 1
                          varpti(i,j,k) = varari(m)
                       ENDDO
                    ENDDO
                 ENDDO

#ifdef _PARALLEL_
                 !
                 !  Collect the data from the other tasks into the pointer array
                 !

                 IF ( ispatch(nvar) ) THEN

                    INT_RANK_LOOP : DO other_rank = 1, numtasks-1

                       IF (n+other_rank > npatch) CYCLE INT_RANK_LOOP

                       itag       = other_rank*1000000 + nvar*1000
                       sendn = -99999
                       outstart = -999999
                       countidx = -999999

                       isize = 1
                       CALL mpi_recv ( sendn , isize, MPI_INTEGER, other_rank, itag+1, &
                            MPI_COMM_WORLD, mpistatus, ierr)
                       IF (ierr /= MPI_SUCCESS) STOP "MPI_RECV_ERROR"
                       narrsize = sendn(1)

                       CALL mpi_recv ( outstart , MAX_RANK , MPI_INTEGER , other_rank, &
                            itag+2, MPI_COMM_WORLD, mpistatus, ierr)
                       IF (ierr /= MPI_SUCCESS) STOP "MPI_RECV_ERROR"

                       CALL mpi_recv ( countidx , MAX_RANK , MPI_INTEGER , other_rank, &
                            itag+3, MPI_COMM_WORLD, mpistatus, ierr)
                       IF (ierr /= MPI_SUCCESS) STOP "MPI_RECV_ERROR"

                       IF (narrsize > narrisizemax) THEN   ! Allocate input array only when necessary
                          IF (ALLOCATED(varari)) DEALLOCATE(varari, STAT = istatus)
                          ALLOCATE(varari(narrsize), STAT = istatus)
                          narrisizemax = narrsize
                       ENDIF

                       CALL mpi_recv ( varari , narrsize , MPI_INTEGER , other_rank, itag, MPI_COMM_WORLD, mpistatus, ierr)
                       IF (ierr /= MPI_SUCCESS) STOP "MPI_RECV_ERROR"

                       m=0
                       DO k = outstart(3), outstart(3)+countidx(3)-1
                          DO j = outstart(2), outstart(2)+countidx(2)-1
                             DO i = outstart(1), outstart(1)+countidx(1)-1
                                m = m + 1
                                varpti(i,j,k) = varari(m)
                             ENDDO
                          ENDDO
                       ENDDO
                       
                    ENDDO INT_RANK_LOOP
                 ENDIF

              ELSE

                 !
                 !  Send the data from this process over to task 0.
                 ! 

                 itag = rank*1000000 + nvar*1000
                 sendn = narrsize

                 CALL mpi_send( sendn  , 1 , MPI_INTEGER, 0, itag+1, MPI_COMM_WORLD, ierr)
                 IF (ierr /= MPI_SUCCESS) STOP "MPI_SEND_ERROR"

                 CALL mpi_send( outstart  , MAX_RANK , MPI_INTEGER, 0, itag+2, MPI_COMM_WORLD, ierr)
                 IF (ierr /= MPI_SUCCESS) STOP "MPI_SEND_ERROR"
                 
                 CALL mpi_send( countidx  , MAX_RANK , MPI_INTEGER, 0, itag+3, MPI_COMM_WORLD, ierr)
                 IF (ierr /= MPI_SUCCESS) STOP "MPI_SEND_ERROR"

                 CALL mpi_send ( varari , narrsize , MPI_INTEGER , 0 , itag , MPI_COMM_WORLD, ierr)
                 IF (ierr /= MPI_SUCCESS) STOP "MPI_SEND_ERROR" 

#endif

              ENDIF

           CASE (NF90_FLOAT)

              ! Re-allocate input array only when necessary
              IF (narrsize > narrasizemax) THEN
                 IF (ALLOCATED(vararr)) DEALLOCATE(vararr, STAT = istatus)
                 ALLOCATE(vararr(narrsize), STAT = istatus)
                 narrasizemax = narrsize
              ENDIF

              istatus = NF90_GET_VAR(fnid(n),varid,vararr(1:narrsize),count=countidx)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem getting float variable")

              IF ( rank == 0 ) THEN

                 IF ( .NOT. ALLOCATED (varptr) ) THEN
                    countidxlg(:) = 1
                    DO vardim = 1, varndims
                       countidxlg(vardim) = dimouta(vardimids(vardim))
                    ENDDO

                    ALLOCATE ( varptr ( countidxlg(1) , countidxlg(2) , countidxlg(3) ) )
                    varptr = -1.E36
                 ENDIF

                 m=0
                 DO k = outstart(3), outstart(3)+countidx(3)-1
                    DO j = outstart(2), outstart(2)+countidx(2)-1
                       DO i = outstart(1), outstart(1)+countidx(1)-1
                          m = m + 1
                          varptr(i,j,k) = vararr(m)
                       ENDDO
                    ENDDO
                 ENDDO

#ifdef _PARALLEL_
                 !
                 !  Collect the data from the other tasks into the pointer array
                 !

                 IF ( ispatch(nvar) ) THEN

                    FLOAT_RANK_LOOP : DO other_rank = 1, numtasks-1

                       IF (n+other_rank > npatch) CYCLE FLOAT_RANK_LOOP

                       itag       = other_rank*1000000 + nvar*1000
                       sendn = -99999
                       outstart = -999999
                       countidx = -999999

                       isize = 1
                       CALL mpi_recv ( sendn , isize, MPI_INTEGER, other_rank, itag+1, MPI_COMM_WORLD, mpistatus, ierr)
                       IF (ierr /= MPI_SUCCESS) STOP "MPI_RECV_ERROR"
                       narrsize = sendn(1)
                       
                       CALL mpi_recv ( outstart , MAX_RANK , MPI_INTEGER , other_rank, itag+2, MPI_COMM_WORLD, mpistatus, ierr)
                       IF (ierr /= MPI_SUCCESS) STOP "MPI_RECV_ERROR"

                       CALL mpi_recv ( countidx , MAX_RANK , MPI_INTEGER , other_rank, itag+3, MPI_COMM_WORLD, mpistatus, ierr)
                       IF (ierr /= MPI_SUCCESS) STOP "MPI_RECV_ERROR"

                       IF (narrsize > narrasizemax) THEN   ! Allocate input array only when necessary
                          IF (ALLOCATED(vararr)) DEALLOCATE(vararr, STAT = istatus)
                          ALLOCATE(vararr(narrsize), STAT = istatus)
                          narrasizemax = narrsize
                       ENDIF

                       CALL mpi_recv (vararr, narrsize, MPI_FLOAT, other_rank, itag, MPI_COMM_WORLD, mpistatus, ierr)
                       IF (ierr /= MPI_SUCCESS) STOP "MPI_RECV_ERROR"

                       m=0
                       DO k = outstart(3), outstart(3)+countidx(3)-1
                          DO j = outstart(2), outstart(2)+countidx(2)-1
                             DO i = outstart(1), outstart(1)+countidx(1)-1
                                m = m + 1
                                varptr(i,j,k) = vararr(m)
                             ENDDO
                          ENDDO
                       ENDDO

                    ENDDO FLOAT_RANK_LOOP
                 ENDIF

              ELSE

                 !
                 !  Send the data from this process over to task 0.
                 ! 

                 itag = rank*1000000 + nvar*1000
                 sendn = narrsize

                 CALL mpi_send ( sendn  , 1 , MPI_INTEGER , 0 , itag+1 , MPI_COMM_WORLD , ierr )
                 IF (ierr /= MPI_SUCCESS) STOP "MPI_SEND_ERROR"

                 CALL mpi_send( outstart  , MAX_RANK , MPI_INTEGER, 0, itag+2, &
                      &         MPI_COMM_WORLD, ierr)
                 IF (ierr /= MPI_SUCCESS) STOP "MPI_SEND_ERROR"

                 CALL mpi_send( countidx  , MAX_RANK , MPI_INTEGER, 0, itag+3, &
                      &         MPI_COMM_WORLD, ierr)
                 IF (ierr /= MPI_SUCCESS) STOP "MPI_SEND_ERROR"

                 CALL mpi_send ( vararr , narrsize , MPI_FLOAT , 0 , itag , & 
                      &          MPI_COMM_WORLD, ierr)
                 IF (ierr /= MPI_SUCCESS) STOP "MPI_SEND_ERROR" 
#endif
              ENDIF

           CASE (NF90_CHAR)

              IF ( ispatch(nvar) ) THEN
                 WRITE(*, '("Expected ISPATCH(NVAR) == F for character")')
                 istatus = -4
                 RETURN
              ENDIF

              istatus = NF90_GET_VAR(fnid(n),varid,tmpstr,start=startidx,count=countidx)
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem getting char variable")

              IF ( rank == 0 ) THEN

                 ovarid = varoutidlists(nvar)
                 istatus = nf90_put_var(foutid,ovarid,TRIM(tmpstr),start=outstart,count=countidx)
                 IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting char variable")

                 WRITE(*,*) "Done writing nopatch variable ", trim(varname)

              ENDIF

           CASE DEFAULT
              WRITE(6,'(1x,a,I2)') 'ERROR: unsupported variable type = ',vartype
              istatus = -4
              RETURN
           END SELECT

        ENDDO PATCH_LOOP

#ifdef _PARALLEL_
        CALL mpi_barrier(MPI_COMM_WORLD, ierr)
        IF (ierr /= MPI_SUCCESS) STOP "Problem with MPI_BARRIER"
#endif

        IF ( rank == 0 ) THEN

           countidx(:) = 1
           outstart(:) = 1
           DO vardim = 1, varndims
              vdimid = vardimids(vardim)
              countidx(vardim) = dimouta(vdimid)
           ENDDO

           ovarid = varoutidlists(nvar)

           SELECT CASE (vartype)

           CASE (NF90_INT)

              istatus = nf90_put_var ( foutid, ovarid, varpti(1:countidx(1),1:countidx(2),1:countidx(3)), count=countidx )
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting integer variable")
              
              write (*, '("Done writing  int  variable ", A20, " :: Var ", I3, " out of ", I3)') TRIM(varname), nvar, nvarout

              deallocate(varpti)

           CASE (NF90_FLOAT)
              
              istatus = nf90_put_var ( foutid, ovarid, varptr(1:countidx(1),1:countidx(2),1:countidx(3)), count=countidx )
              IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem putting float variable")

              WRITE (*, '("Done writing float variable ", A20, " :: Var ", I3, " out of ", I3)') TRIM(varname), nvar, nvarout

              DEALLOCATE(varptr)

           END SELECT

        ENDIF

#ifdef _PARALLEL_
        CALL mpi_barrier(MPI_COMM_WORLD, ierr)
        IF (ierr /= MPI_SUCCESS) STOP "Problem with MPI_BARRIER"
#endif

     ENDDO VAR_LOOP

     IF ( rank == 0 ) THEN
        istatus = nf90_close(foutid)
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus)

        open(22, file=trim(flagfilename), status="unknown", form='formatted', action='write')
        close(22)
     ENDIF

     DO n = 1,npatch
        IF ( rank /= MOD(n-1, numtasks) ) CYCLE
        istatus = nf90_close(fnid(n))
        IF (istatus /= NF90_NOERR) CALL handle_err(istatus, "Problem closing file")
     ENDDO

  ENDDO NFILE_LOOP

END SUBROUTINE joinwrfncdf


SUBROUTINE handle_err(istat, message)
#ifdef _PARALLEL_
  !USE mpi
#endif
  USE netcdf
  IMPLICIT NONE
#ifdef _PARALLEL_
  INCLUDE "mpif.h"
#endif
  INTEGER, INTENT(IN) :: istat
  character(len=*), optional, intent(in) :: message
#ifdef _PARALLEL_
  integer :: mpi_ierr
#endif
 
  IF (istat /= NF90_NOERR) THEN
     WRITE(*, '(A)') TRIM(nf90_strerror(istat))
    IF ( PRESENT ( message ) ) THEN
       WRITE(*, '(A)') TRIM(message)
    ENDIF
#ifdef _PARALLEL_
    CALL mpi_finalize(mpi_ierr)
    IF (mpi_ierr /= MPI_SUCCESS) STOP "Problem with MPI_FINALIZE"
#endif
    STOP 'NetCDF error!'
  ENDIF
 
END SUBROUTINE handle_err
