!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: iniTimeVar
!
! !INTERFACE:
subroutine iniTimeVar(snlx     ,snowdpx ,snowagex  ,dzclmx  ,zclmx      ,&
                     ziclmx    ,h2osnox ,h2osoi_liqx,h2osoi_icex,t_grndx,&
                     t_soisnox ,t_lakex ,t_vegx    ,h2ocanx ,h2ocan_colx,&
                     h2osoi_volx, lake_icefracx, savedtke1x)
!
! !DESCRIPTION:
! Initializes the following time varying variables:
! water : h2osno, h2ocan, h2osoi_liq, h2osoi_ice, h2osoi_vol
! snow : snowdp, snowage, snl, dz, z, zi
! temperature: t_soisno, t_veg, t_grnd
! The variable, h2osoi_vol, is needed by the soil albedo routine - this is not needed
! on restart since it is computed before the soil albedo computation is called.
! The remaining variables are initialized by calls to ecosystem dynamics and
! albedo subroutines.
!
! !USES:
  use shr_kind_mod         , only : r8 => shr_kind_r8
  use clmtype
  use decompMod            , only : get_proc_bounds
  use filterMod            , only : filter
  use clm_varpar           , only : nlevsoi, nlevsno, nlevlak,maxpatch
  use clm_varcon           , only : denice, denh2o, zlnd,istsoil,istdlak
  use FracWetMod           , only : FracWet
  use SurfaceAlbedoMod     , only : SurfaceAlbedo
  use globals              , only : month, day, calday
#ifdef DGVM
  use DGVMMod              , only : resetTimeConstDGVM
  use DGVMEcosystemDynMod  , only : DGVMEcosystemDyn
#else
  use STATICEcosysDynMod, only : EcosystemDyn, interpMonthlyVeg
#endif
!
! !ARGUMENTS:
  implicit none
!
! !CALLED FROM:
! subroutine initialize in module initializeMod
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein

! The following vraiables for MM5 and restart run
    integer   :: snlx(maxpatch)
    real(r8)  :: snowdpx(maxpatch)
    real(r8)  :: snowagex(maxpatch)
    real(r8)  :: h2osnox(maxpatch)
    real(r8)  :: t_grndx(maxpatch)
    real(r8)  :: t_vegx(maxpatch)
    real(r8)  :: h2ocanx(maxpatch)
    real(r8)  :: h2ocan_colx(maxpatch)

    real(r8)  :: t_lakex(maxpatch,nlevlak)
    real(r8)  :: t_soisnox(maxpatch,-nlevsno+1:nlevsoi)
    real(r8)  :: h2osoi_liqx(maxpatch,-nlevsno+1:nlevsoi)
    real(r8)  :: h2osoi_icex(maxpatch,-nlevsno+1:nlevsoi)
    real(r8)  :: dzclmx(maxpatch,-nlevsno+1:nlevsoi)
    real(r8)  :: zclmx(maxpatch,-nlevsno+1:nlevsoi)
    real(r8)  :: ziclmx(maxpatch,-nlevsno:nlevsoi)
    real(r8)  :: h2osoi_volx(maxpatch,nlevsoi)
    real(r8), intent(in)  :: lake_icefracx(nlevsoi), savedtke1x

!
! !LOCAL VARIABLES:
!
! local pointers to implicit in arguments
!
  integer , pointer :: plandunit(:)      ! landunit index associated with each pft
  logical , pointer :: lakpoi(:)         ! true => landunit is a lake point
  real(r8), pointer :: dz(:,:)           ! layer thickness depth (m)
  real(r8), pointer :: h2osoi_ice(:,:)   ! ice lens (kg/m2)
  real(r8), pointer :: h2osoi_liq(:,:)   ! liquid water (kg/m2)
  integer , pointer :: frac_veg_nosno_alb(:) ! fraction of vegetation not covered by snow (0 OR 1) [-]
!
! local pointers to implicit out arguments
!
  real(r8), pointer :: h2osoi_vol(:,:)   ! volumetric soil water (0<=h2osoi_vol<=watsat) [m3/m3]
  real(r8), pointer :: snowdp(:)         ! snow height (m)
  real(r8), pointer :: frac_sno(:)       ! fraction of ground covered by snow (0 to 1)
  integer , pointer :: frac_veg_nosno(:) ! fraction of vegetation not covered by snow (0 OR 1) [-]
  real(r8), pointer :: fwet(:)           ! fraction of canopy that is wet (0 to 1) (pft-level)
!
! local pointers to implicit out arguments (lake points only)
!
  real(r8), pointer :: fdry(:)     ! fraction of foliage that is green and dry [-] (new)
  real(r8), pointer :: tlai(:)     ! one-sided leaf area index, no burying by snow
  real(r8), pointer :: tsai(:)     ! one-sided stem area index, no burying by snow
  real(r8), pointer :: htop(:)     ! canopy top (m)
  real(r8), pointer :: hbot(:)     ! canopy bottom (m)
  real(r8), pointer :: elai(:)     ! one-sided leaf area index with burying by snow
  real(r8), pointer :: esai(:)     ! one-sided stem area index with burying by snow

!
!EOP
!
! !OTHER LOCAL VARIABLES:
  integer :: nc,j,l,c,p,fp,fc  ! indices
  integer :: begp, endp   ! per-clump beginning and ending pft indices
  integer :: begc, endc   ! per-clump beginning and ending column indices
  integer :: begl, endl   ! per-clump beginning and ending landunit indices
  integer :: begg, endg   ! per-clump gridcell ending gridcell indices

#if (defined DGVM)
  integer , pointer :: clandunit(:)     ! column's landunit
  integer , pointer :: pcolumn(:)        ! column index of corresponding pft
  integer , pointer :: ityplun(:)       ! landunit type
  
  real(r8), pointer :: z(:,:)            ! (m)
  real(r8), pointer :: tsoi25(:)         ! soil temperature to 0.25 m (Kelvin)
  real(r8), pointer :: t_soisno(:,:)     ! soil temperature (Kelvin)  (-nlevsno+1:nlevsoi)
  real(r8), pointer :: watsat(:,:)      ! volumetric soil water at saturation (porosity)
  real(r8), pointer :: sucsat(:,:)      ! minimum soil suction (mm)
  real(r8), pointer :: bsw(:,:)         ! Clapp and Hornberger "b"
  real(r8), pointer :: wf(:)            ! soil water as frac. of whc for top 0.5 m

  real(r8) ,allocatable :: tsoi(:) ! temporary
  real(r8) ,allocatable :: dep(:)  ! temporary
  real(r8) ,allocatable :: rwat(:) ! soil water wgted by depth to maximum depth of 0.5 m
  real(r8) ,allocatable :: swat(:) ! same as rwat but at saturation
  real(r8) ,allocatable :: rz(:)   ! thickness of soil layers contributing to rwat (m)
  
  real(r8) :: watdry                     ! temporary
  real(r8) :: tsw                        ! volumetric soil water to 0.5 m
  real(r8) :: stsw                       ! volumetric soil water to 0.5 m at saturation
#endif
!-----------------------------------------------------------------------

  ! Assign local pointers to derived subtypes components (landunit-level)

  lakpoi              => clm3%g%l%lakpoi

  ! Assign local pointers to derived subtypes components (column-level)

  dz                  => clm3%g%l%c%cps%dz
  h2osoi_ice          => clm3%g%l%c%cws%h2osoi_ice
  h2osoi_liq          => clm3%g%l%c%cws%h2osoi_liq
  h2osoi_vol          => clm3%g%l%c%cws%h2osoi_vol
  snowdp              => clm3%g%l%c%cps%snowdp
  frac_sno            => clm3%g%l%c%cps%frac_sno

#if (defined DGVM)
  ityplun             => clm3%g%l%itype
  pcolumn             => clm3%g%l%c%p%column
  clandunit           => clm3%g%l%c%landunit

  z                   => clm3%g%l%c%cps%z
  t_soisno            => clm3%g%l%c%ces%t_soisno
  watsat              => clm3%g%l%c%cps%watsat
  tsoi25              => clm3%g%l%c%p%pdgvs%tsoi25
  sucsat              => clm3%g%l%c%cps%sucsat
  bsw                 => clm3%g%l%c%cps%bsw
  wf                  => clm3%g%l%c%cps%wf
#endif

  ! Assign local pointers to derived subtypes components (pft-level)

  plandunit          => clm3%g%l%c%p%landunit
  frac_veg_nosno_alb => clm3%g%l%c%p%pps%frac_veg_nosno_alb
  frac_veg_nosno     => clm3%g%l%c%p%pps%frac_veg_nosno
  fwet               => clm3%g%l%c%p%pps%fwet

  ! Assign local pointers to derived subtypes components (pft-level)
  ! The folowing pointers will only be used for lake points in this routine

  htop               => clm3%g%l%c%p%pps%htop
  hbot               => clm3%g%l%c%p%pps%hbot
  tlai               => clm3%g%l%c%p%pps%tlai
  tsai               => clm3%g%l%c%p%pps%tsai
  elai               => clm3%g%l%c%p%pps%elai
  esai               => clm3%g%l%c%p%pps%esai
  fdry               => clm3%g%l%c%p%pps%fdry

  ! ========================================================================
  ! Initialize water and temperature based on:
  ! readini = true : read initial data set -- requires netCDF codes
  ! readini = false: arbitrary initialization
  ! ========================================================================

    call mkarbinit(snlx     ,snowdpx ,snowagex  ,dzclmx  ,zclmx      ,&
                  ziclmx    ,h2osnox ,h2osoi_liqx,h2osoi_icex,t_grndx,&
                  t_soisnox ,t_lakex ,t_vegx    ,h2ocanx ,h2ocan_colx,&
                  h2osoi_volx, lake_icefracx, savedtke1x)

  ! after this subroutine, t_soisno has values -- Jiming Jin
  ! ========================================================================
  ! Remaining variables are initialized by calls to ecosystem dynamics and
  ! albedo subroutines.
  ! Note: elai, esai, frac_veg_nosno_alb are computed in
  ! Ecosysdyn and needed by routines FracWet and SurfaceAlbedo
  ! frac_veg_nosno is needed by FracWet
  ! fwet is needed in routine TwoStream (called by SurfaceAlbedo)
  ! frac_sno is needed by SoilAlbedo (called by SurfaceAlbedo)
  ! ========================================================================


#if (!defined DGVM)
  ! Read monthly vegetation data for interpolation to daily values
  call interpMonthlyVeg(month, day)
#endif


     call get_proc_bounds(begg, endg, begl, endl, begc, endc, begp, endp)

     ! Determine variables needed by SurfaceAlbedo for lake points

!dir$ concurrent
!cdir nodep
     do p = begp,endp
        l = plandunit(p)
        if (lakpoi(l)) then
           fwet(p) = 0.
           fdry(p) = 0.
           elai(p) = 0.
           esai(p) = 0.
           htop(p) = 0.
           hbot(p) = 0.
           tlai(p) = 0.
           tsai(p) = 0.
           frac_veg_nosno_alb(p) = 0.
           frac_veg_nosno(p) = 0.
        end if
     end do

#ifdef DGVM
!------------------------------------------------------------
! This addition by Jiming Jin is for calculating tsoi25
! The calculation is also repeated in DGVMRespiration

    ! Soil temperature to a depth of 0.25 m.
    allocate(tsoi(begc:endc))
    allocate(dep(begc:endc))
!dir$ concurrent
!cdir nodep
    do fc = 1,filter%num_nolakec
       c = filter%nolakec(fc)
       tsoi(c) = 0._r8
       dep(c) = 0._r8
    end do
    do j = 1, nlevsoi
!dir$ concurrent
!cdir nodep
       do fc = 1,filter%num_nolakec
          c = filter%nolakec(fc)
          if (z(c,j)+0.5*dz(c,j) <= 0.25) then
             tsoi(c) = tsoi(c) + t_soisno(c,j)*dz(c,j)
             dep(c) = dep(c) + dz(c,j)
          end if
       end do
    end do

    ! Loop over pfts

!dir$ concurrent
!cdir nodep
    do fp = 1,filter%num_nolakep
       p = filter%nolakep(fp)
       c = pcolumn(p)

       if (dep(c) /= 0.) then
          tsoi25(p) = tsoi(c)/dep(c)
       else
          tsoi25(p) = t_soisno(c,1)
       end if
    end do
    deallocate(tsoi)
    deallocate(dep)

! calculate wf()

    allocate(rwat(begc:endc))
    allocate(swat(begc:endc))
    allocate(rz(begc:endc))

    do c = begc,endc
       l = clandunit(c)
       if (ityplun(l) == istsoil .or. ityplun(l) == istdlak) then
          rwat(c) = 0._r8
          swat(c) = 0._r8
          rz(c)   = 0._r8
       end if
    end do

    do j = 1, nlevsoi
!dir$ concurrent
!cdir nodep
       do c = begc,endc
          l = clandunit(c)
          if (ityplun(l) == istsoil .or. ityplun(l) == istdlak) then
             if (z(c,j)+0.5*dz(c,j) <= 0.5) then
                watdry = watsat(c,j) * (316230./sucsat(c,j)) ** (-1./bsw(c,j))
                rwat(c) = rwat(c) + (h2osoi_vol(c,j)-watdry) * dz(c,j)
                swat(c) = swat(c) + (watsat(c,j)    -watdry) * dz(c,j)
                rz(c) = rz(c) + dz(c,j)
             end if
          end if
       end do
    end do

!dir$ concurrent
!cdir nodep
    do c = begc,endc
       l = clandunit(c)
       if (ityplun(l) == istsoil .or. ityplun(l) == istdlak) then
          if (rz(c) /= 0._r8) then
             tsw  = rwat(c)/rz(c)
             stsw = swat(c)/rz(c)
          else
             watdry = watsat(c,1) * (316230./sucsat(c,1)) ** (-1./bsw(c,1))
             tsw = h2osoi_vol(c,1) - watdry
             stsw = watsat(c,1) - watdry
          end if
          wf(c) = tsw/stsw
       else
          wf(c) = 1.0
       end if
    end do
    deallocate(rwat)
    deallocate(swat)
    deallocate(rz)
!------------------------------------------------------------
#endif

#ifdef DGVM
     call resetTimeConstDGVM(begp, endp)
     call DGVMEcosystemDyn(begp, endp, filter%num_nolakep, filter%nolakep, &
                           doalb=.true., endofyr=.false.)
#else
     call EcosystemDyn(begp, endp, filter%num_nolakep, filter%nolakep, &
                       doalb=.true.)
#endif

!dir$ concurrent
!cdir nodep
     do p = begp, endp
        l = plandunit(p)
        if (.not. lakpoi(l)) then
           frac_veg_nosno(p) = frac_veg_nosno_alb(p)
           fwet(p) = 0.
        end if
     end do

     call FracWet(filter%num_nolakep, filter%nolakep)

     ! Compute Surface Albedo - all land points (including lake)
     ! Needs as input fracion of soil covered by snow (Z.-L. Yang U. Texas)

!dir$ concurrent
!cdir nodep
     do c = begc, endc
        snowdp(c)  = snowdpx(c)
        frac_sno(c) = snowdp(c)/(10.*zlnd + snowdp(c))
     end do

     call SurfaceAlbedo(begg, endg, begc, endc, begp, endp, calday)

end subroutine iniTimeVar

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: mkarbinit
!
! !INTERFACE:
!Edited by Zack Subin to make consistent with new variables in CLM3.5 mkarbinit.
!7/16/08

subroutine mkarbinit(snlx   ,snowdpx ,snowagex  ,dzclmx  ,zclmx      ,&
                  ziclmx    ,h2osnox ,h2osoi_liqx,h2osoi_icex,t_grndx,&
                  t_soisnox ,t_lakex ,t_vegx    ,h2ocanx ,h2ocan_colx,&
                  h2osoi_volx, lake_icefracx, savedtke1x)

! !DESCRIPTION:
! Initializes the following time varying variables:
! (New for lake code: lake_icefrac, savedtke1)
! water      : h2osno, h2ocan, h2osoi_liq, h2osoi_ice, h2osoi_vol
! snow       : snowdp, snowage, snl, dz, z, zi
! temperature: t_soisno, t_veg, t_grnd
! The variable, h2osoi_vol, is needed by clm_soilalb -this is not needed on
! restart since it is computed before the soil albedo computation is called.
! The remaining variables are initialized by calls to ecosystem dynamics
! and albedo subroutines.
!
! !USES:
  use shr_kind_mod , only : r8 => shr_kind_r8
  use clmtype
  use decompMod    , only : get_proc_bounds
  use clm_varpar   , only : nlevsoi, nlevsno, nlevlak,maxpatch
  use clm_varcon   , only : bdsno, istice, istwet, istsoil, &
                            denice, denh2o, spval, sb, tfrz
#ifdef DGVM
  use DGVMMod      , only : resetWeightsDGVM
#endif
!
! !ARGUMENTS:
  implicit none
!
! !CALLED FROM:
! subroutine iniTimeVar
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
! !LOCAL VARIABLES:
!
! local pointers to implicit in arguments
!
  integer , pointer :: pcolumn(:)        ! column index associated with each pft
  integer , pointer :: clandunit(:)      ! landunit index associated with each column
  integer , pointer :: ltype(:)          ! landunit type
  logical , pointer :: lakpoi(:)         ! true => landunit is a lake point
  real(r8), pointer :: dz(:,:)           ! layer thickness depth (m)
  real(r8), pointer :: zi(:,:)       ! interface depth (m) over snow only
  real(r8), pointer :: z(:,:)       ! interface depth (m) over snow only
!New variables: Zack Subin 7/16/08
  real(r8), pointer :: watsat(:,:)       ! volumetric soil water at saturation (porosity) (nlevsoi)
  real(r8), pointer :: h2osoi_ice(:,:)   ! ice lens (kg/m2)
  real(r8), pointer :: h2osoi_liq(:,:)   ! liquid water (kg/m2)
  real(r8), pointer :: bsw2(:,:)         ! Clapp and Hornberger "b" for CN code
  real(r8), pointer :: psisat(:,:)       ! soil water potential at saturation for CN code (MPa)
  real(r8), pointer :: vwcsat(:,:)       ! volumetric water content at saturation for CN code (m3/m3)
!  real(r8), pointer :: zi(:,:)           ! interface level below a "z" level (m)
!Is it different now, not just for snow?????
  real(r8), pointer :: wa(:)             ! water in the unconfined aquifer (mm)
  real(r8), pointer :: wt(:)             ! total water storage (unsaturated soil water + groundwater) (mm)
  real(r8), pointer :: zwt(:)            ! water table depth (m)
  real(r8), pointer :: h2ocan_loss(:)    ! canopy water mass balance term (column)
! New for lakes
  real(r8), pointer :: lake_icefrac(:,:)  ! mass fraction of lake layer that is frozen 
  real(r8), pointer :: savedtke1(:)       ! top level eddy conductivity (W/mK)

!!!!!!!!!!!!!!!!!!!!!!!!

!
! local pointers to implicit out arguments
!
  integer , pointer :: snl(:)            ! number of snow layers
  real(r8), pointer :: t_soisno(:,:)     ! soil temperature (Kelvin)  (-nlevsno+1:nlevsoi)
  real(r8), pointer :: t_lake(:,:)       ! lake temperature (Kelvin)  (1:nlevlak)
  real(r8), pointer :: t_grnd(:)         ! ground temperature (Kelvin)
  real(r8), pointer :: t_veg(:)          ! vegetation temperature (Kelvin)
  real(r8), pointer :: h2osoi_vol(:,:)   ! volumetric soil water (0<=h2osoi_vol<=watsat) [m3/m3]
  real(r8), pointer :: h2ocan_col(:)     ! canopy water (mm H2O) (column-level)
  real(r8), pointer :: h2ocan_pft(:)     ! canopy water (mm H2O) (pft-level)
  real(r8), pointer :: h2osno(:)         ! snow water (mm H2O)
  real(r8), pointer :: snowdp(:)         ! snow height (m)
  real(r8), pointer :: snowage(:)        ! non dimensional snow age [-] (new)
  real(r8), pointer :: eflx_lwrad_out(:) ! emitted infrared (longwave) radiation (W/m**2)
!New variables
  real(r8), pointer :: soilpsi(:,:)    ! soil water potential in each soil layer (MPa)
!!!!!!!!!!!!!!!

! The following vraiables for a  WRF restart run
    integer   :: snlx(maxpatch)
    real(r8)  :: snowdpx(maxpatch)
    real(r8)  :: snowagex(maxpatch)
    real(r8)  :: h2osnox(maxpatch)
    real(r8)  :: t_grndx(maxpatch)
    real(r8)  :: t_vegx(maxpatch)
    real(r8)  :: h2ocanx(maxpatch)
    real(r8)  :: h2ocan_colx(maxpatch)

    real(r8)  :: t_lakex(maxpatch,nlevlak)
    real(r8)  :: t_soisnox(maxpatch,-nlevsno+1:nlevsoi)
    real(r8)  :: h2osoi_liqx(maxpatch,-nlevsno+1:nlevsoi)
    real(r8)  :: h2osoi_icex(maxpatch,-nlevsno+1:nlevsoi)
    real(r8)  :: dzclmx(maxpatch,-nlevsno+1:nlevsoi)
    real(r8)  :: zclmx(maxpatch,-nlevsno+1:nlevsoi)
    real(r8)  :: ziclmx(maxpatch,-nlevsno:nlevsoi)
    real(r8)  :: h2osoi_volx(maxpatch,nlevsoi)
    real(r8), intent(in)  :: lake_icefracx(nlevsoi), savedtke1x

!
!EOP
!
! !OTHER LOCAL VARIABLES:
  integer :: j,l,c,p      ! indices
  integer :: begp, endp   ! per-proc beginning and ending pft indices
  integer :: begc, endc   ! per-proc beginning and ending column indices
  integer :: begl, endl   ! per-proc beginning and ending landunit indices
  integer :: begg, endg   ! per-proc gridcell ending gridcell indices
!-----------------------------------------------------------------------

  ! Assign local pointers to derived subtypes components (landunit-level)

  ltype => clm3%g%l%itype
  lakpoi => clm3%g%l%lakpoi

  ! Assign local pointers to derived subtypes components (column-level)

  clandunit  => clm3%g%l%c%landunit
  snl        => clm3%g%l%c%cps%snl
  dz         => clm3%g%l%c%cps%dz
  zi         => clm3%g%l%c%cps%zi
  z          => clm3%g%l%c%cps%z
  watsat     => clm3%g%l%c%cps%watsat
  h2osoi_ice => clm3%g%l%c%cws%h2osoi_ice
  h2osoi_liq => clm3%g%l%c%cws%h2osoi_liq
  h2osoi_vol => clm3%g%l%c%cws%h2osoi_vol
  h2ocan_col => clm3%g%l%c%cws%pws_a%h2ocan
  snowage    => clm3%g%l%c%cps%snowage
  snowdp     => clm3%g%l%c%cps%snowdp
  h2osno     => clm3%g%l%c%cws%h2osno
  t_soisno   => clm3%g%l%c%ces%t_soisno
  t_lake     => clm3%g%l%c%ces%t_lake
  t_grnd     => clm3%g%l%c%ces%t_grnd
!New variables
    bsw2       => clm3%g%l%c%cps%bsw2
    vwcsat     => clm3%g%l%c%cps%vwcsat
    psisat     => clm3%g%l%c%cps%psisat
    soilpsi    => clm3%g%l%c%cps%soilpsi
    wa         => clm3%g%l%c%cws%wa
    wt         => clm3%g%l%c%cws%wt
    zwt        => clm3%g%l%c%cws%zwt
    h2ocan_loss => clm3%g%l%c%cwf%h2ocan_loss
!!!!!!
! For new lake code
    savedtke1       => clm3%g%l%c%cps%savedtke1
    lake_icefrac => clm3%g%l%c%cws%lake_icefrac

  ! Assign local pointers to derived subtypes components (pft-level)

  pcolumn => clm3%g%l%c%p%column
  h2ocan_pft => clm3%g%l%c%p%pws%h2ocan
  t_veg => clm3%g%l%c%p%pes%t_veg
  eflx_lwrad_out => clm3%g%l%c%p%pef%eflx_lwrad_out

  ! Determine subgrid bounds on this processor

  call get_proc_bounds(begg, endg, begl, endl, begc, endc, begp, endp)

  ! ========================================================================
  ! Set snow water
  ! ========================================================================

  ! NOTE: h2ocan, h2osno, snowdp and snowage has valid values everywhere

  ! canopy water (pft level)

  do p = begp, endp
     h2ocan_pft(p) = h2ocanx(p)
  end do

!dir$ concurrent
!cdir nodep
  do c = begc,endc

     ! canopy water (column level)

     h2ocan_col(c) = h2ocan_colx(c)

!New variable: canopy water loss
     h2ocan_loss(c) = 0._r8

     ! snow water

        h2osno(c) = h2osnox(c)

     ! snow depth

     snowdp(c)  = snowdpx(c)

     ! snow age

     snowage(c) = snowagex(c)

  end do


  ! ========================================================================
  ! Set snow layer number, depth and thickiness
  ! ========================================================================

  do c = begc,endc
   snl(c) = snlx(c)
   dz(c,-nlevsno+1:0)  = dzclmx(c,-nlevsno+1:0)
   z(c,-nlevsno+1:0)   = zclmx(c,-nlevsno+1:0)
   zi(c,-nlevsno+0:0)  = ziclmx(c,-nlevsno+0:0)
  end do


  ! ========================================================================
  ! Set snow/soil temperature
  ! ========================================================================

  ! NOTE:
  ! t_lake   only has valid values over lake
  ! t_grnd has valid values over all land
  ! t_veg  has valid values over all land

!dir$ concurrent
!cdir nodep
  do c = begc,endc
     t_soisno(c,-nlevsno+1:nlevsoi) = t_soisnox(c,-nlevsno+1:nlevsoi)
     t_lake(c,1:nlevlak) = t_lakex(c,1:nlevlak)
     t_grnd(c)  = t_grndx(c)
  !New lake variables
     l = clandunit(c)
     if (lakpoi(l)) then
        lake_icefrac(c,1:nlevlak) = lake_icefracx(1:nlevlak)
        savedtke1(c)              = savedtke1x
     end if
  end do

!dir$ concurrent
!cdir nodep
  do p = begp, endp
     c = pcolumn(p)
     t_veg(p) = t_vegx(c)
     eflx_lwrad_out(p) = sb * (t_grnd(c))**4
  end do

  ! ========================================================================
  ! Set snow/soil ice and liquid mass
  ! ========================================================================

  ! volumetric water is set first and liquid content and ice lens are obtained
  ! NOTE: h2osoi_vol, h2osoi_liq and h2osoi_ice only have valid values over soil

  do c = begc,endc
      do j=1,nlevsoi
          h2osoi_vol(c,j)  = h2osoi_volx(c,j)
      end do
      do j=-nlevsno+1,nlevsoi
          h2osoi_liq(c,j)  = h2osoi_liqx(c,j)
          h2osoi_ice(c,j)  = h2osoi_icex(c,j)
      end do
  end do

!New variables
    wa(begc:endc)  = 5000._r8
    wt(begc:endc)  = 5000._r8
    zwt(begc:endc) = 0._r8
!!!!!!!!

!Switched loop order
  do c = begc,endc
!For new variables
        l = clandunit(c)
!        if (.not. lakpoi(l)) then  !not lake
          wa(c)  = 4800._r8
          wt(c)  = wa(c)
          zwt(c) = (25._r8 + zi(c,nlevsoi)) - wa(c)/0.2_r8 /1000._r8  ! One meter below soil column
!       end if
!!!!!!!
     do j = 1,nlevsoi

        l = clandunit(c)
! New shallow lakes
!        if (.not. lakpoi(l)) then  !not lake
           ! volumetric water
           if(h2osoi_vol(c,j) > watsat(c,j)) then
              h2osoi_vol(c,j) = watsat(c,j)
              if(h2osoi_liq(c,j)/(dz(c,j)*denh2o)+ &
                 h2osoi_ice(c,j)/(dz(c,j)*denice)> &
                 h2osoi_vol(c,j) ) then
                 if(t_soisno(c,j) > tfrz) then
                    h2osoi_liq(c,j) = dz(c,j)*denh2o*watsat(c,j) 
                    h2osoi_ice(c,j) = 0.0
                 else
                    h2osoi_liq(c,j) = 0.0
                    h2osoi_ice(c,j) = dz(c,j)*denice*watsat(c,j)
                 end if
              end if
   
           endif
!For CN
#if (defined CN)
             ! soil water potential (added 10/21/03, PET)
             ! required for CN code
             if (ltype(l) == istsoil) then
                if (h2osoi_liq(c,j) > 0._r8) then
                   vwc = h2osoi_liq(c,j)/(dz(c,j)*denh2o)
                   psi = psisat(c,j) * (vwc/vwcsat(c,j))**bsw2(c,j)
                   soilpsi(c,j) = max(psi, -15.0_r8)
                   soilpsi(c,j) = min(soilpsi(c,j),0.0_r8)
                end if
             end if
#endif
!!!!!!!!!!!!!!

!        end if
     end do
  end do
#ifdef DGVM
  ! Determine new subgrid weights and areas (obtained
  ! from new value of fpcgrid read in above) - this is needed
  ! here to avoid round off level errors on restart before
  ! lpj is called the first time

  call resetWeightsDGVM(begg, endg, begc, endc, begp, endp)
#endif

end subroutine mkarbinit
