module initializeMod

!-----------------------------------------------------------------------
!BOP
!
! !MODULE: initializeMod
!
! !DESCRIPTION:
! Performs land model initialization
!
! !PUBLIC TYPES:
  implicit none
  save
!
! !PUBLIC MEMBER FUNCTIONS:
  public :: initialize
!
! !REVISION HISTORY:
! Created by Gordon Bonan, Sam Levis and Mariana Vertenstein
!
!EOP
!
! !PRIVATE MEMBER FUNCTIONS:
!-----------------------------------------------------------------------

contains

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: initialize
!
! !INTERFACE:
  subroutine initialize(snl    ,snowdp  ,snowage   ,dzclm     ,zclm        &
                  ,ziclm       ,h2osno  ,h2osoi_liq,h2osoi_ice,t_grnd      &
                  ,t_soisno    ,t_lake  ,t_veg     ,h2ocan    ,h2ocan_col  &
                  ,h2osoi_vol  ,xlat    ,xlon      ,areaxy    ,iveg        &
                  ,isl         ,lndmsk  ,t2m_max   ,t2m_min   ,t2m_max_inst&
                  ,t2m_min_inst,t_ref2m                                    &
! For new lake code
                  ,lake_icefrac,lakedepth,dzlake,zlake,savedtke1     &
#if (defined DGVM)
                  ,t_mo_min  ,annpsn    ,annpsnpot                         &
                  ,fmicr       ,bm_inc  ,afmicr    ,t10min    ,tmomin20    &
                  ,agdd20      ,fpcgrid ,lai_ind   ,crownarea ,dphen       &
                  ,leafon      ,leafof  ,firelength,litterag  ,litterbg    &
                  ,cpool_fast  ,cpool_slow         ,k_fast_ave,k_slow_ave  &
                  ,nind        ,lm_ind  ,sm_ind    ,hm_ind    ,rm_ind      &
                  ,present     ,htop    ,tsai      ,tda       ,t10         &
                  ,prec365     ,agdd0   ,agdd5     ,agddtw    ,agdd        &
                  ,litter_decom_ave                                        &
#endif
                                                                           )
!
! !DESCRIPTION:
! Land model initialization.
! o Initializes run control variables via the [clmexp] namelist.
! o Reads surface data on model grid.
! o Defines the multiple plant types and fraction areas for each surface type.
! o Builds the appropriate subgrid <-> grid mapping indices and weights.
! o Set up parallel processing.
! o Initializes time constant variables.
! o Reads restart data for a restart or branch run.
! o Reads initial data and initializes the time variant variables for an initial run.
! o Initializes history file output.
! o Initializes river routing model.
! o Initializes accumulation variables.
!
! !USES:
    use shr_kind_mod    , only : r8 => shr_kind_r8
    use clmtypeInitMod  , only : initClmtype
    use initGridCellsMod, only : initGridCells
    use clm_varpar      , only : lsmlon, lsmlat, maxpatch,nlevsoi,nlevsno,&
                                 nlevlak
    use clm_varsur      , only : varsur_alloc, varsur_dealloc,longxy,latixy,&
                                 area
    use filterMod       , only : initFilters
    use decompMod       , only : initDecomp
    use accFldsMod      , only : initAccFlds, initAccClmtype
    use surfFileMod     , only : surfrd
    use pftvarcon       , only : pftconrd
    use clm_varcon      , only : var_par


#if (defined DGVM)
    use DGVMEcosystemDynMod, only : DGVMEcosystemDynini
#else
    use STATICEcosysDynMod , only : EcosystemDynini
#endif

#if (defined DGVM)
    use DGVMMod            , only : resetTimeConstDGVM
#endif
!
! !ARGUMENTS:
    implicit none
!
! !REVISION HISTORY:
! Created by Gordon Bonan, Sam Levis and Mariana Vertenstein
!
!EOP
!
! !LOCAL VARIABLES:
    integer  :: i,j,k                 !indices
    integer  :: yr                    !current year (0, ...)
    integer  :: mon                   !current month (1 -> 12)
    integer  :: day                   !current day (1 -> 31)
    integer  :: ncsec                 !current time of day [seconds]
    logical  :: readini               !true if read in initial data set
    integer  :: vegxy(lsmlon,lsmlat,maxpatch) !vegetation type
    real(r8) :: wtxy(lsmlon,lsmlat,maxpatch)  !subgrid weights
    integer  :: ier

    integer   :: snl(maxpatch)
    real(r8)  :: snowdp(maxpatch)
    real(r8)  :: snowage(maxpatch)
    real(r8)  :: h2osno(maxpatch)
    real(r8)  :: t_grnd(maxpatch)
    real(r8)  :: t_veg(maxpatch)
    real(r8)  :: h2ocan(maxpatch)
    real(r8)  :: h2ocan_col(maxpatch)

    real(r8)  :: t_lake(maxpatch,nlevlak)
    real(r8)  :: t_soisno(maxpatch,-nlevsno+1:nlevsoi)
    real(r8)  :: h2osoi_liq(maxpatch,-nlevsno+1:nlevsoi)
    real(r8)  :: h2osoi_ice(maxpatch,-nlevsno+1:nlevsoi)
    real(r8)  :: dzclm(maxpatch,-nlevsno+1:nlevsoi)
    real(r8)  :: zclm(maxpatch,-nlevsno+1:nlevsoi)
    real(r8)  :: ziclm(maxpatch,-nlevsno:nlevsoi)
    real(r8)  :: h2osoi_vol(maxpatch,nlevsoi)

    real(r8)  :: xlon
    real(r8)  :: xlat
    real(r8)  :: areaxy

    integer   :: iveg
    integer   :: isl
    integer   :: lndmsk

! The following vraiables for MM5 and restart run

#if (defined DGVM)
    real(r8):: t_mo_min(maxpatch)            !annual min of t_mo (Kelvin)
    real(r8):: annpsn(maxpatch)              !annual photosynthesis (umol CO2 /m**2)
    real(r8):: annpsnpot(maxpatch)           !annual potential photosynthesis (same units)
    real(r8):: fmicr(maxpatch)               !microbial respiration (umol CO2 /m**2 /s)
    real(r8):: bm_inc(maxpatch)              !biomass increment
    real(r8):: afmicr(maxpatch)              !microbial respiration (Rh) for each naturally-vegetated pft
    real(r8):: t10min(maxpatch)              !annual minimum of 10-day running mean (K)
    real(r8):: tmomin20(maxpatch)            !20-yr running mean of tmomin
    real(r8):: agdd20(maxpatch)              !20-yr running mean of agdd
    real(r8):: fpcgrid(maxpatch)             !foliar projective cover on gridcell (fraction)
    real(r8):: lai_ind(maxpatch)             !LAI per individual
    real(r8):: crownarea(maxpatch)           !area that each individual tree takes up (m^2)
    real(r8):: dphen(maxpatch)               !phenology [0 to 1]
    real(r8):: leafon(maxpatch)              !leafon days
    real(r8):: leafof(maxpatch)              !leafoff days
    real(r8):: firelength(maxpatch)          !fire season in days
    real(r8):: litterag(maxpatch)            !above ground litter
    real(r8):: litterbg(maxpatch)            !below ground litter
    real(r8):: cpool_fast(maxpatch)          !fast carbon pool
    real(r8):: cpool_slow(maxpatch)          !slow carbon pool
    real(r8):: k_fast_ave(maxpatch)          !decomposition rate
    real(r8):: k_slow_ave(maxpatch)          !decomposition rate
    real(r8):: litter_decom_ave(maxpatch)    !decomposition rate
    real(r8):: nind(maxpatch)                !number of individuals (#/m**2)
    real(r8):: lm_ind(maxpatch)              !individual leaf mass
    real(r8):: sm_ind(maxpatch)              !individual sapwood mass
    real(r8):: hm_ind(maxpatch)              !individual heartwood mass
    real(r8):: rm_ind(maxpatch)              !individual root mass
    real(r8):: htop(maxpatch)
    real(r8):: tsai(maxpatch)
    logical :: present(maxpatch)             !whether PFT present in patch

    real(r8) :: tda(maxpatch)
    real(r8) :: t10(maxpatch)
    real(r8) :: fnpsn10(maxpatch)
    real(r8) :: prec365(maxpatch)
    real(r8) :: agdd0(maxpatch)
    real(r8) :: agdd5(maxpatch)
    real(r8) :: agddtw(maxpatch)
    real(r8) :: agdd(maxpatch)
#endif

    real(r8) :: t2m_max(maxpatch)
    real(r8) :: t2m_min(maxpatch)
    real(r8) :: t2m_max_inst(maxpatch)
    real(r8) :: t2m_min_inst(maxpatch)
    real(r8) :: t_ref2m(maxpatch)

    ! New lake code variables
    real(r8), intent(in) :: lakedepth, savedtke1, dzlake(1:nlevsoi), zlake(1:nlevsoi)
    real(r8), intent(in) :: lake_icefrac(1:nlevsoi)

    call CLMDebug('Now in Initialize. Next call varsur_alloc.')

!-----------------------------------------------------------------------
    do i=1,lsmlon
       do j=1,lsmlat
           longxy(i,j) = xlon
           latixy(i,j) = xlat
           area(i,j)   = areaxy
       end do
    end do
!-----------------------------------------------------------------------

    ! Allocate surface grid dynamic memory

     call varsur_alloc ()

     ! Read list of PFTs and their corresponding parameter values
     ! This is independent of the model resolution

     call CLMDebug('pftconrd')
     call pftconrd ()

     ! If no surface dataset name is specified then make surface dataset
     ! from original data sources. Always read surface boundary data in.
     ! This insures that bit for bit results are obtained for a run where a
     ! surface dataset file is generated and a run where a surface dataset
     ! is specified and read in. Set up vegetation type [veg] and weight [wt]
     ! arrays for [maxpatch] subgrid patches.

     call var_par

     call CLMDebug('surfrd')
     call surfrd (vegxy, wtxy, iveg, isl, lndmsk)

    ! Initialize clump and processor decomposition

    call initDecomp(wtxy)
! Is this is actually necessary?  Probably only for multiple gridcells. Or maybenot...

    ! Allocate memory and initialize values of clmtype data structures

    call initClmtype()

    ! Build hierarchy and topological info for derived typees

    call CLMDebug('initGridCells')
    call initGridCells(vegxy, wtxy)

    ! Initialize filters

    call initFilters()

     ! Initialize Ecosystem Dynamics

#if (defined DGVM)
 call DGVMEcosystemDynini(t_mo_min   ,annpsn     ,annpsnpot  ,fmicr     ,&
                          bm_inc     ,afmicr     ,t10min     ,tmomin20  ,&
                          agdd20     ,fpcgrid    ,lai_ind    ,crownarea ,&
                          dphen      ,leafon     ,leafof     ,firelength,&
                          litterag   ,litterbg   ,cpool_fast ,cpool_slow,&
                          k_fast_ave ,k_slow_ave ,nind       ,lm_ind    ,&
                          sm_ind     ,hm_ind     ,rm_ind     ,present   ,&
                          htop       ,tsai       ,litter_decom_ave)

#else
 call CLMDebug('EcosystemDynini')
 call EcosystemDynini()
#endif

    ! Initialize time constant variables

    call CLMDebug('iniTimeConst')
    call iniTimeConst (zclm,dzclm,ziclm,lakedepth,dzlake,zlake)


    ! Initialize accumulator fields to be time accumulated for various purposes.
#if (defined DGVM)
    call  initAccFlds(t_ref2m  ,tda   ,t10    ,fnpsn10   ,prec365,&
                      agdd0    ,agdd5 ,agddtw ,agdd)
#else
    call CLMDebug('initAccFlds')
    call  initAccFlds(t_ref2m) 
#endif


    ! Initialize clmtype variables that are obtained from accumulated fields.
    ! This routine is called in an initial run at nstep=0 for cam and csm mode
    ! and at nstep=1 for offline mode. This routine is also always called for a
    ! restart run and must therefore be called after the restart file is read in

    call CLMDebug('initAccClmtype') 
    call initAccClmtype(t2m_max,t2m_min,t2m_max_inst,t2m_min_inst)

    call CLMDebug('iniTimeVar')
    call iniTimeVar(snl      ,snowdp  ,snowage   ,dzclm     ,zclm      ,&
                   ziclm     ,h2osno  ,h2osoi_liq,h2osoi_ice,t_grnd    ,&
                   t_soisno  ,t_lake  ,t_veg     ,h2ocan    ,h2ocan_col,&
                   h2osoi_vol, lake_icefrac, savedtke1)

    ! Deallocate surface grid dynamic memory

    call CLMDebug('entering varsurdealloc')
    call varsur_dealloc()
    call CLMDebug('done varsurdealloc')
    ! End initialization

  end subroutine initialize

end module initializeMod
