module surfFileMod

!----------------------------------------------------------------------- 
!BOP
!
! !MODULE: surfFileMod
! 
! !DESCRIPTION: 
! Contains methods for reading in surface data file and determining
! two-dimensional subgrid weights as well as writing out new surface
! dataset. When reading in the surface dataset, determines array 
! which sets the PFT for each of the [maxpatch] patches and 
! array which sets the relative abundance of the PFT. 
! Also fills in the PFTs for vegetated portion of each grid cell. 
! Fractional areas for these points pertain to "vegetated" 
! area not to total grid area. Need to adjust them for fraction of grid 
! that is vegetated. Also fills in urban, lake, wetland, and glacier patches.
!
! !PUBLIC TYPES:
  implicit none
  save
!
! !PUBLIC MEMBER FUNCTIONS:
  public :: surfrd  !Read surface dataset and determine subgrid weights
!  public :: surfwrt !Write surface dataset
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!----------------------------------------------------------------------- 

contains

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: surfrd
!
! !INTERFACE:
  subroutine surfrd(veg, wt,iveg,isl,lndmsk)

!
! !DESCRIPTION: 
! Read the surface dataset and create subgrid weights.
! The model's surface dataset recognizes 5 basic land cover types within 
! a grid cell: lake, wetland, urban, glacier, and vegetated. The vegetated 
! portion of the grid cell is comprised of up to [maxpatch_pft] PFTs. These
! subgrid patches are read in explicitly for each grid cell. This is in
! contrast to LSMv1, where the PFTs were built implicitly from biome types.
! Read surface boundary data with the exception of 
! monthly lai,sai,and heights at top and bottom of canopy 
! on [lsmlon] x [lsmlat] grid. 
!    o real edges of grid
!    o integer  number of longitudes per latitude
!    o real latitude  of grid cell (degrees)
!    o real longitude of grid cell (degrees)
!    o integer surface type: 0 = ocean or 1 = land
!    o integer soil color (1 to 9) for use with soil albedos
!    o real soil texture, %sand, for thermal and hydraulic properties
!    o real soil texture, %clay, for thermal and hydraulic properties
!    o real % of cell covered by lake    for use as subgrid patch
!    o real % of cell covered by wetland for use as subgrid patch
!    o real % of cell that is urban      for use as subgrid patch
!    o real % of cell that is glacier    for use as subgrid patch
!    o integer PFTs
!    o real % abundance PFTs (as a percent of vegetated area)
!
! OFFLINE MODE ONLY:
! Surface grid edges -- Grids do not have to be global. 
! If grid is read in from dataset, grid is assumed to be global 
! (does not have to be regular, however)
! If grid is generated by model, grid does not have to be global but must then
! define the north, east, south, and west edges:
!
!    o lsmedge(1)    = northern edge of grid (degrees): >  -90 and <= 90
!    o lsmedge(2)    = eastern edge of grid (degrees) : see following notes
!    o lsmedge(3)    = southern edge of grid (degrees): >= -90 and <  90
!    o lsmedge(4)    = western edge of grid (degrees) : see following notes
!
!      For partial grids, northern and southern edges are any latitude
!      between 90 (North Pole) and -90 (South Pole). Western and eastern
!      edges are any longitude between -180 and 180, with longitudes
!      west of Greenwich negative. That is, western edge >= -180 and < 180;
!      eastern edge > western edge and <= 180.
!
!      For global grids, northern and southern edges are 90 (North Pole)
!      and -90 (South Pole). The western and eastern edges depend on
!      whether the grid starts at Dateline or Greenwich. Regardless,
!      these edges must span 360 degrees. Examples:
!
!                              West edge    East edge
!                            ---------------------------------------------------
!  (1) Dateline            :        -180 to 180        (negative W of Greenwich)
!  (2) Greenwich (centered):    0 - dx/2 to 360 - dx/2
!
!    Grid 1 is the grid for offline mode
!    Grid 2 is the grid for cam and csm mode since the NCAR CAM 
!    starts at Greenwich, centered on Greenwich 
!
! !USES:
    use shr_kind_mod, only: r8 => shr_kind_r8
    use clm_varpar                      !parameters
    use clm_varsur                      !surface data 
    use pftvarcon, only : noveg, crop,nwheat  !vegetation type (PFT) 
    use clm_varcon,only : sand,clay,soic,plant,cover,num_landcover_types
!
! !ARGUMENTS:
    implicit none
    integer , intent(out) :: veg(lsmlon,lsmlat,maxpatch) !PFT 
    real(r8), intent(out) :: wt(lsmlon,lsmlat,maxpatch)  !subgrid weights
!
! !CALLED FROM:
! subroutine initialize in module initializeMod
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein, Sam Levis and Gordon Bonan
!
!EOP
!   variables from MM5 11/25/2003 Jiming Jin

    integer  :: iveg,isl,lndmsk
!
! !LOCAL VARIABLES:
    integer  :: i,j,k,m,k1,k2                            ! indices
    integer  :: ncid,dimid,varid                         ! netCDF id's
    integer  :: ier                                      ! error status
    integer  :: pft(lsmlon,lsmlat,maxpatch_pft)          ! PFT
    integer  :: cft(lsmlon,lsmlat,maxpatch_cft)          ! CFT
    real(r8) :: pctpft_lunit(lsmlon,lsmlat,maxpatch_pft) ! % of vegetated landunit area for PFTs
    real(r8) :: pctcft_lunit(lsmlon,lsmlat,maxpatch_cft) ! % of crop landunit area for CFTs
    real(r8), allocatable :: pctpft(:,:,:)               ! percent of vegetated gridcell area for PFTs
    real(r8) :: pctspec                                  ! percent of gridcell made up of special landunits
    integer  :: cropcount                                ! temporary counter
    real(r8) :: sumscl                                   ! temporory scalar sum
    real(r8) :: sumvec(lsmlon,lsmlat)                    ! temporary vector sum
    logical  :: found                                    ! temporary for error check
    integer  :: iindx, jindx                             ! temporary for error check
    integer  :: miss = 99999                             ! missing data indicator
    real(r8) :: wst(0:numpft)                            ! as[signed?] pft at specific i, j
    integer  :: wsti(maxpatch_pft)                       ! ranked indices of largest values in wst
    real(r8) :: wst_sum                                  ! sum of %pft
    real(r8) :: sumpct                                   ! sum of %pft over maxpatch_pft
    real(r8) :: diff                                     ! the difference (wst_sum - sumpct)
    real(r8) :: rmax                                     ! maximum patch cover
!-------------------------------------------------------------------------
       ! Initialize surface data to fill value

       landmask(:,:) = -999
       landfrac(:,:) = -999.
       soic2d(:,:)   = -999
       sand3d(:,:,:) = -999.
       clay3d(:,:,:) = -999.
       pctlak(:,:)   = 0.0
       pctwet(:,:)   = 0.0
       pcturb(:,:)   = 0.0
       pctgla(:,:)   = 0.0
       pft(:,:,:)    = 0

       allocate (pctpft(lsmlon,lsmlat,0:numpft), stat=ier)
       pctpft(:,:,:) = 0.0


       ! Obtain netcdf file and read surface data

       do j=1,lsmlat
          numlon(j) = lsmlon
       end do

       do j=1,lsmlat
          do i=1,numlon(j)

           landmask(i,j)    = lndmsk

           soic2d(i,j) = soic(isl)

           do k=1,nlevsoi
             sand3d(i,j,k)  = sand(isl)
             clay3d(i,j,k)  = clay(isl)
           end do

!Debug
!call flush(6)
!print*, 'zzzz', iveg

!To prevent errors when adding new vegetation types
           if (iveg > num_landcover_types) then
              print*, 'Vegtype exceeds number of allowed landcover types in clm_varcon.F.'
              call endrun()
           end if
!Added by Zack Subin, 8/4/08 

           do m=1,maxpatch_pft
              pft(i,j,m)     = plant(iveg,m)
              if(cover(iveg,m).ne.0.0) then
                 pctpft(i,j,pft(i,j,m))  = cover(iveg,m)
              end if
           end do

!Urban not yet implemented?  Leave bare ground type and let urban module in WRF deal with
!roughness, heat island, albedo?
!           if(iveg.eq.1) then 
!              pcturb(i,j)   = 0.0
!           if(iveg == 1 .or. iveg == 31 .or. iveg == 32 .or. iveg == 33)
!              pcturb(i,j) = 100.0
!           elseif(iveg.eq.17) then 
!Zack Subin, 7/21/08: 18 is also a wetland.
           if(iveg == 17 .or. iveg == 18) then
              pctwet(i,j)      = 100.0 
           elseif(iveg.eq.16.and.landmask(i,j).eq.1) then
              pctlak(i,j)      = 100.0
           elseif(iveg.eq.24) then
              pctgla(i,j)      = 100.0
           end if
            
         end do
       end do

       ! Error check: valid PFTs and sum of cover must equal 100

#ifndef DGVM
       sumvec(:,:) = abs(sum(pctpft,dim=3)-100.)
       do j=1,lsmlat
          do i=1,numlon(j)
             do m = 1, maxpatch_pft
                if (pft(i,j,m)<0 .or. pft(i,j,m)>numpft) then
                   write(6,*)'SURFRD error: invalid PFT for i,j,m=',i,j,m,pft(i,j,m)
                   call endrun
                end if
             end do
             if (sumvec(i,j)>1.e-04 .and. landmask(i,j)==1) then
                write(6,*)'SURFRD error: PFT cover ne 100 for i,j=',i,j
                do m=1,maxpatch_pft
                   write(6,*)'m= ',m,' pft= ',pft(i,j,m)
                end do
                write(6,*)'sumvec= ',sumvec(i,j)
                call endrun
             end if
          end do
       end do
#endif

          ! 1. pctpft must go back to %vegetated landunit instead of %gridcell
          ! 2. pctpft bare = 100 when landmask = 1 and 100% special landunit
          ! NB: (1) and (2) do not apply to crops.
          ! For now keep all cfts (< 4 anyway) instead of 4 most dominant cfts

          do j=1,lsmlat
             do i=1,numlon(j)
                cft(i,j,:) = 0
                pctcft_lunit(i,j,:) = 0.
                cropcount = 0
                pctspec = pcturb(i,j) + pctgla(i,j) + pctlak(i,j) + pctwet(i,j)
                if (pctspec < 100.) then
                   do m = 0, numpft
                      if (crop(m) == 1. .and. pctpft(i,j,m) > 0.) then
                         cropcount = cropcount + 1
                         if (cropcount > maxpatch_cft) then
                            write(6,*) 'ERROR surfFileMod: cropcount>maxpatch_cft'
                            call endrun()
                         end if
                         cft(i,j,cropcount) = m
                         pctcft_lunit(i,j,cropcount) = pctpft(i,j,m) * 100./(100.-pctspec)
                         pctpft(i,j,m) = 0.0
                      else if (crop(m) == 0.) then
                         pctpft(i,j,m) = pctpft(i,j,m) * 100./(100.-pctspec)
                      end if
                   end do
                else if (pctspec == 100.) then
                   pctpft(i,j,0)        = 100.
                   pctpft(i,j,1:numpft) =   0.
                end if
             end do
          end do

          ! Find pft and pct arrays
          ! Save percent cover by PFT [wst] and total percent cover [wst_sum]

          do j=1,lsmlat
             do i=1,numlon(j)

                wst_sum = 0.
                sumpct = 0
                do m = 0, numpft
                   wst(m) = pctpft(i,j,m)
                   wst_sum = wst_sum + pctpft(i,j,m)
                end do

                ! Rank [wst] in ascendg order to obtain the top [maxpatch_pft] PFTs

                if (landmask(i,j) == 1) call mkrank (numpft, wst, miss, wsti, maxpatch_pft)

                ! Fill in [pft] and [pctpft] with data for top [maxpatch_pft] PFTs.
                ! If land model grid cell is ocean, set to no PFTs.
                ! If land model grid cell is land then:
                !  1. If [pctlnd_o] = 0, there is no PFT data from the input grid.
                !     Since need land data, use bare ground.
                !  2. If [pctlnd_o] > 0, there is PFT data from the input grid but:
                !     a. use the chosen PFT so long as it is not a missing value
                !     b. missing value means no more PFTs with cover > 0

                if (landmask(i,j) == 1) then       ! model grid wants land
                   do m = 1, maxpatch_pft
                      if (wsti(m) /=  miss) then
                         pft(i,j,m) = wsti(m)
                         pctpft_lunit(i,j,m) = wst(wsti(m))
                      else
                         pft(i,j,m) = noveg
                         pctpft_lunit(i,j,m) = 0.
                      end if
                      sumpct = sumpct + pctpft_lunit(i,j,m)
                   end do
                else                               ! model grid wants ocean
                   do m = 1, maxpatch_pft
                      pft(i,j,m) = 0
                      pctpft_lunit(i,j,m) = 0.
                   end do
                end if

                ! Correct for the case of more than [maxpatch_pft] PFTs present

                if (sumpct < wst_sum) then
                   diff  = wst_sum - sumpct
                   sumpct = 0.
                   do m = 1, maxpatch_pft
                      pctpft_lunit(i,j,m) = pctpft_lunit(i,j,m) + diff/maxpatch_pft
                      sumpct = sumpct + pctpft_lunit(i,j,m)
                   end do
                end if

                ! Error check: make sure have a valid PFT

                do m = 1,maxpatch_pft
                   if (pft(i,j,m) < 0 .or. pft(i,j,m) > numpft) then
                      write (6,*)'surfrd error: invalid PFT at gridcell i,j=',i,j,pft(i,j,m)
                      call endrun()
                   end if
                end do

                ! As done in mksrfdatMod.F90 for other percentages, truncate pctpft to
                ! ensure that weight relative to landunit is not nonzero
                ! (i.e. a very small number such as 1e-16) where it really should be zero

                do m=1,maxpatch_pft
                   pctpft_lunit(i,j,m) = float(nint(pctpft_lunit(i,j,m)))
                end do
                do m=1,maxpatch_cft
                   pctcft_lunit(i,j,m) = float(nint(pctcft_lunit(i,j,m)))
                end do

                ! Make sure sum of PFT cover == 100 for land points. If not,
                ! subtract excess from most dominant PFT.

                rmax = -9999.
                k1 = -9999
                k2 = -9999
                sumpct = 0.
                do m = 1, maxpatch_pft
                   sumpct = sumpct + pctpft_lunit(i,j,m)
                   if (pctpft_lunit(i,j,m) > rmax) then
                      k1 = m
                      rmax = pctpft_lunit(i,j,m)
                   end if
                end do
                do m = 1, maxpatch_cft
                   sumpct = sumpct + pctcft_lunit(i,j,m)
                   if (pctcft_lunit(i,j,m) > rmax) then
                      k2 = m
                      rmax = pctcft_lunit(i,j,m)
                   end if
                end do
                if (k1 == -9999 .and. k2 == -9999) then
                   write(6,*)'surfrd error: largest PFT patch not found'
                   call endrun()
                else if (landmask(i,j) == 1) then
                   if (sumpct < 95 .or. sumpct > 105.) then
                      write(6,*)'surfrd error: sum of PFT cover =',sumpct,' at i,j=',i,j
                      call endrun()
                   else if (sumpct /= 100. .and. k2 /= -9999) then
                      pctcft_lunit(i,j,k2) = pctcft_lunit(i,j,k2) - (sumpct-100.)
                   else if (sumpct /= 100.) then
                      pctpft_lunit(i,j,k1) = pctpft_lunit(i,j,k1) - (sumpct-100.)
                   end if
                end if

                ! Error check: make sure PFTs sum to 100% cover

                sumpct = 0.
                do m = 1, maxpatch_pft
                   sumpct = sumpct + pctpft_lunit(i,j,m)
                end do
                do m = 1, maxpatch_cft
                   sumpct = sumpct + pctcft_lunit(i,j,m)
                end do
                if (landmask(i,j) == 1) then
                   if (abs(sumpct - 100.) > 0.000001) then
                      write(6,*)'surfFileMod error: sum(pct) over maxpatch_pft is not = 100.'
                      write(6,*)sumpct, i,j
                      call endrun()
                   end if
                   if (sumpct < -0.000001) then
                      write(6,*)'surfFileMod error: sum(pct) over maxpatch_pft is < 0.'
                      write(6,*)sumpct, i,j
                      call endrun()
                   end if
                end if

             end do   ! end of longitude loop
          end do   ! end of latitude loop

          deallocate(pctpft)


       ! Error check: glacier, lake, wetland, urban sum must be less than 100

       found = .false.
       do j=1,lsmlat
          do i=1,numlon(j)
             sumscl = pctlak(i,j)+pctwet(i,j)+pcturb(i,j)+pctgla(i,j)
             if (sumscl > 100.+1.e-04) then
                found = .true.
                iindx = i
                jindx = j
                exit
             end if
          end do
          if (found) exit
       end do
       if ( found ) then
          write(6,*)'surfrd error: PFT cover>100 for i,j=',iindx,jindx
          call endrun()
       end if

       ! Error check that urban parameterization is not yet finished

#ifndef TESTZACK
       found = .false.
       do j=1,lsmlat
          do i=1,numlon(j)
             if (pcturb(i,j) /= 0.) then
                found = .true.
                iindx = i
                jindx = j
                exit
             end if
          end do
          if (found) exit
       end do
       if ( found ) then
          write (6,*)'surfrd error: urban parameterization not implemented at i,j= ',iindx,jindx
          call endrun()
       end if
#endif

    ! Determine array [veg], which sets the PFT for each of the [maxpatch]
    ! patches and array [wt], which sets the relative abundance of the PFT.
    ! Fill in PFTs for vegetated portion of grid cell. Fractional areas for
    ! these points [pctpft] pertain to "vegetated" area not to total grid area.
    ! So need to adjust them for fraction of grid that is vegetated.
    ! Next, fill in urban, lake, wetland, and glacier patches.

    veg(:,:,:) = 0
    wt(:,:,:)  = 0.
    do j=1,lsmlat
       do i=1,numlon(j)
          if (landmask(i,j) == 1) then
             sumscl = pcturb(i,j)+pctlak(i,j)+pctwet(i,j)+pctgla(i,j)
             do m = 1, maxpatch_pft
#ifdef DGVM
                veg(i,j,m) = noveg !spinup initialization
                wt(i,j,m) = 1.0/maxpatch_pft * (100.-sumscl)/100.
#else
                veg(i,j,m) = pft(i,j,m)
                wt(i,j,m) = pctpft_lunit(i,j,m) * (100.-sumscl)/10000.
#endif
             end do
             veg(i,j,npatch_urban) = noveg
             wt(i,j,npatch_urban) = pcturb(i,j)/100.
             veg(i,j,npatch_lake)  = noveg
             wt(i,j,npatch_lake)  = pctlak(i,j)/100.
             veg(i,j,npatch_wet)   = noveg
             wt(i,j,npatch_wet)   = pctwet(i,j)/100.
             veg(i,j,npatch_glacier) = noveg
             wt(i,j,npatch_glacier) = pctgla(i,j)/100.

             do m = 1,maxpatch_cft
#ifdef DGVM
                   veg(i,j,npatch_glacier+m) = noveg ! currently assume crop=0 if DGVM mode
                   wt(i,j,npatch_glacier+m)= 0.
#else
                   veg(i,j,npatch_glacier+m) = cft(i,j,m)
                   wt(i,j,npatch_glacier+m)= pctcft_lunit(i,j,m) * (100.-sumscl)/10000.
#endif
            end do
          end if
       end do
    end do

    found = .false.
    sumvec(:,:) = abs(sum(wt,dim=3)-1.)
    do j=1,lsmlat
       do i=1,numlon(j)
          if (sumvec(i,j) > 1.e-06 .and. landmask(i,j)==1) then
             found = .true.
             iindx = i
             jindx = j
             exit
          endif
       end do
       if (found) exit
    end do
    if ( found ) then
       write (6,*)'surfrd error: WT > 1 occurs at i,j= ',iindx,jindx
       call endrun()
    end if

  end subroutine surfrd

end module surfFileMod
