module ShalLakeHydrologyMod

!-----------------------------------------------------------------------
!BOP
!
! !MODULE: ShalLakeHydrologyMod
!
! !DESCRIPTION:
! Calculation of Shallow Lake Hydrology. Full hydrology of snow layers is
! done. However, there is no infiltration, and the water budget is balanced with 
! qflx_qrgwl. Lake water mass is kept constant. The soil is simply maintained at
! volumetric saturation if ice melting frees up pore space. Likewise, if the water
! portion alone at some point exceeds pore capacity, it is reduced. This is consistent
! with the possibility of initializing the soil layer with excess ice. The only
! real error with that is that the thermal conductivity will ignore the excess ice
! (and accompanying thickness change).
! 
! If snow layers are present over an unfrozen lake, and the top layer of the lake
! is capable of absorbing the latent heat without going below freezing, 
! the snow-water is runoff and the latent heat is subtracted from the lake.
!
!
! !PUBLIC TYPES:
  implicit none
  save
!
! !PUBLIC MEMBER FUNCTIONS:
  public :: ShalLakeHydrology        ! Calculates soil/snow hydrology
!
! !REVISION HISTORY:
! Created by Zack Subin, 2009
!
!EOP
!-----------------------------------------------------------------------

contains

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: ShalLakeHydrology
!
! !INTERFACE:
  subroutine ShalLakeHydrology(lbc, ubc, lbp, ubp, num_shlakec, filter_shlakec, &
                               num_shlakep, filter_shlakep &
                        )
!
! !DESCRIPTION:
!
! WARNING: This subroutine assumes lake columns have one and only one pft.
!
! Sequence is:
!  ShalLakeHydrology:
!    Do needed tasks from Hydrology1, Biogeophysics2, & top of Hydrology2.
!    -> SnowWater:             change of snow mass and snow water onto soil
!    -> SnowCompaction:        compaction of snow layers
!    -> CombineSnowLayers:     combine snow layers that are thinner than minimum
!    -> DivideSnowLayers:      subdivide snow layers that are thicker than maximum
!    Add water to soil if melting has left it with open pore space.
!    Cleanup and do water balance.
!    If snow layers are found above a lake with unfrozen top layer, whose top
!    layer has enough heat to melt all the snow ice without freezing, do so
!    and eliminate the snow layers.
!
! !USES:
    use shr_kind_mod, only: r8 => shr_kind_r8
    use clmtype
!    use clm_atmlnd      , only : clm_a2l
    use clm_varcon      , only : denh2o, denice, spval, hfus, tfrz, cpliq, cpice
    use clm_varpar      , only : nlevsoi, nlevsno
    use SnowHydrologyMod, only : SnowCompaction, CombineSnowLayers, DivideSnowLayers, &
                                 SnowWater, BuildSnowFilter
!    use clm_time_manager, only : get_step_size, is_perpetual
    use globals, only : dtime
#if (defined CLMDEBUG)
!    use abortutils,   only: endrun
#endif
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: lbc, ubc                    ! column bounds
    integer, intent(in) :: lbp, ubp                    ! pft bounds
    integer, intent(in) :: num_shlakec                 ! number of column shal lake points in column filter
    integer, intent(in) :: filter_shlakec(ubc-lbc+1)   ! column filter for shal lake points
    integer, intent(in) :: num_shlakep                 ! number of pft shal lake points in column filter
    integer, intent(in) :: filter_shlakep(ubp-lbp+1)   ! pft filter for shal lake points
!
! !CALLED FROM:
! subroutine driver
!
! !REVISION HISTORY:
! Created by Zack Subin
!
! !LOCAL VARIABLES:
!
! local pointers to implicit in arguments
!
    integer , pointer :: pcolumn(:)         !pft's column index
    integer , pointer :: pgridcell(:)       ! pft's gridcell index
    integer , pointer :: cgridcell(:)     ! column's gridcell
    integer , pointer :: clandunit(:)     ! column's landunit
!    integer , pointer :: ityplun(:)       ! landunit type
    real(r8), pointer :: watsat(:,:)      ! volumetric soil water at saturation (porosity)
    real(r8), pointer :: z(:,:)           ! layer depth  (m)
    real(r8), pointer :: dz_lake(:,:)     ! layer thickness for lake (m)
    real(r8), pointer :: forc_rain(:)     ! rain rate [mm/s]
    real(r8), pointer :: forc_snow(:)     ! snow rate [mm/s]
    real(r8), pointer :: begwb(:)         ! water mass begining of the time step
    real(r8), pointer :: qflx_evap_tot(:) ! qflx_evap_soi + qflx_evap_veg + qflx_tran_veg
    real(r8), pointer :: forc_t(:)        ! atmospheric temperature (Kelvin)
#if (defined OFFLINE)
    real(r8), pointer :: flfall(:)        ! fraction of liquid water within falling precipitation
#endif
    logical , pointer :: do_capsnow(:)     ! true => do snow capping
    real(r8), pointer :: t_grnd(:)          ! ground temperature (Kelvin)
    real(r8), pointer :: qflx_evap_soi(:)   ! soil evaporation (mm H2O/s) (+ = to atm)
!
! local pointers to implicit inout arguments
!
    real(r8), pointer :: dz(:,:)          ! layer thickness depth (m)
    real(r8), pointer :: zi(:,:)          ! interface depth (m)
    integer , pointer :: snl(:)           ! number of snow layers
    real(r8), pointer :: h2osno(:)        ! snow water (mm H2O)
    real(r8), pointer :: snowdp(:)        ! snow height (m)
    real(r8), pointer :: lake_icefrac(:,:)  ! mass fraction of lake layer that is frozen
    real(r8), pointer :: t_lake(:,:)        ! lake temperature (Kelvin)
!
! local pointers to implicit out arguments
!
    real(r8), pointer :: endwb(:)         ! water mass end of the time step
    real(r8), pointer :: snowage(:)       ! non dimensional snow age [-]
    real(r8), pointer :: snowice(:)       ! average snow ice lens
    real(r8), pointer :: snowliq(:)       ! average snow liquid water
    real(r8), pointer :: t_snow(:)        ! vertically averaged snow temperature
    real(r8), pointer :: t_soisno(:,:)    ! snow temperature (Kelvin)
    real(r8), pointer :: h2osoi_ice(:,:)  ! ice lens (kg/m2)
    real(r8), pointer :: h2osoi_liq(:,:)  ! liquid water (kg/m2)
    real(r8), pointer :: h2osoi_vol(:,:)  ! volumetric soil water (0<=h2osoi_vol<=watsat) [m3/m3]
    real(r8), pointer :: qflx_drain(:)    ! sub-surface runoff (mm H2O /s)
    real(r8), pointer :: qflx_surf(:)     ! surface runoff (mm H2O /s)
    real(r8), pointer :: qflx_infl(:)     ! infiltration (mm H2O /s)
    real(r8), pointer :: qflx_qrgwl(:)    ! qflx_surf at glaciers, wetlands, lakes
    real(r8), pointer :: qcharge(:)       ! aquifer recharge rate (mm/s)
    real(r8), pointer :: qflx_prec_grnd(:)     ! water onto ground including canopy runoff [kg/(m2 s)]
    real(r8), pointer :: qflx_snowcap(:)       ! excess precipitation due to snow capping (mm H2O /s) [+]
    real(r8), pointer :: qflx_snowcap_col(:)   ! excess precipitation due to snow capping (mm H2O /s) [+]
    real(r8), pointer :: qflx_snow_grnd_pft(:) ! snow on ground after interception (mm H2O/s) [+]
    real(r8), pointer :: qflx_snow_grnd_col(:) ! snow on ground after interception (mm H2O/s) [+]
    real(r8), pointer :: qflx_rain_grnd(:)     ! rain on ground after interception (mm H2O/s) [+]
    real(r8), pointer :: frac_iceold(:,:)      ! fraction of ice relative to the tot water
    real(r8), pointer :: qflx_evap_tot_col(:) !pft quantity averaged to the column (assuming one pft)
    real(r8) ,pointer :: soilalpha(:)     !factor that reduces ground saturated specific humidity (-)
    real(r8), pointer :: zwt(:)           !water table depth
    real(r8), pointer :: fcov(:)          !fractional area with water table at surface
    real(r8), pointer :: rootr_column(:,:) !effective fraction of roots in each soil layer
    real(r8), pointer :: qflx_evap_grnd(:)  ! ground surface evaporation rate (mm H2O/s) [+]
    real(r8), pointer :: qflx_sub_snow(:)   ! sublimation rate from snow pack (mm H2O /s) [+]
    real(r8), pointer :: qflx_dew_snow(:)   ! surface dew added to snow pack (mm H2O /s) [+]
    real(r8), pointer :: qflx_dew_grnd(:)   ! ground surface dew formation (mm H2O /s) [+]
    real(r8), pointer :: qflx_rain_grnd_col(:)   !rain on ground after interception (mm H2O/s) [+]

! Block of biogeochem currently not used.
#ifndef SHLAKE
    real(r8), pointer :: sucsat(:,:)      ! minimum soil suction (mm)
    real(r8), pointer :: bsw(:,:)         ! Clapp and Hornberger "b"
    real(r8), pointer :: bsw2(:,:)        ! Clapp and Hornberger "b" for CN code
    real(r8), pointer :: psisat(:,:)      ! soil water potential at saturation for CN code (MPa)
    real(r8), pointer :: vwcsat(:,:)      ! volumetric water content at saturation for CN code (m3/m3)
    real(r8), pointer :: wf(:)            ! soil water as frac. of whc for top 0.5 m
    real(r8), pointer :: soilpsi(:,:)     ! soil water potential in each soil layer (MPa)
    real(r8) :: psi,vwc,fsat               ! temporary variables for soilpsi calculation
#if (defined DGVM) || (defined CN)
    real(r8) :: watdry                     ! temporary
    real(r8) :: rwat(lbc:ubc)              ! soil water wgted by depth to maximum depth of 0.5 m
    real(r8) :: swat(lbc:ubc)              ! same as rwat but at saturation
    real(r8) :: rz(lbc:ubc)                ! thickness of soil layers contributing to rwat (m)
    real(r8) :: tsw                        ! volumetric soil water to 0.5 m
    real(r8) :: stsw                       ! volumetric soil water to 0.5 m at saturation
#endif
#endif
!!!!!!
!
!EOP
!
! !OTHER LOCAL VARIABLES:
!
    integer  :: p,fp,g,l,c,j,fc,jtop             ! indices
    integer  :: num_shlakesnowc                  ! number of column snow points
    integer  :: filter_shlakesnowc(ubc-lbc+1)    ! column filter for snow points
    integer  :: num_shlakenosnowc                ! number of column non-snow points
    integer  :: filter_shlakenosnowc(ubc-lbc+1)  ! column filter for non-snow points
!    real(r8) :: dtime                      ! land model time step (sec)
    integer  :: newnode                      ! flag when new snow node is set, (1=yes, 0=no)
    real(r8) :: dz_snowf                     ! layer thickness rate change due to precipitation [mm/s]
    real(r8) :: bifall                       ! bulk density of newly fallen dry snow [kg/m3]
    real(r8) :: fracsnow(lbp:ubp)            ! frac of precipitation that is snow
    real(r8) :: fracrain(lbp:ubp)            ! frac of precipitation that is rain
    real(r8) :: qflx_prec_grnd_snow(lbp:ubp) ! snow precipitation incident on ground [mm/s]
    real(r8) :: qflx_prec_grnd_rain(lbp:ubp) ! rain precipitation incident on ground [mm/s]
    real(r8) :: qflx_evap_soi_lim            ! temporary evap_soi limited by top snow layer content [mm/s]
    real(r8) :: h2osno_temp                  ! temporary h2osno [kg/m^2]
    real(r8), parameter :: snow_bd = 250._r8  !constant snow bulk density (only used in special case here) [kg/m^3]
    real(r8) :: sumsnowice(lbc:ubc)             ! sum of snow ice if snow layers found above unfrozen lake [kg/m&2]
    logical  :: unfrozen(lbc:ubc)            ! true if top lake layer is unfrozen with snow layers above
    real(r8) :: heatrem                      ! used in case above [J/m^2]
    real(r8) :: heatsum(lbc:ubc)             ! used in case above [J/m^2]
#if (defined CLMDEBUG)
    real(r8) :: snow_water(lbc:ubc)           ! temporary sum of snow water for Bal Check [kg/m^2]
#endif
!-----------------------------------------------------------------------

    ! Assign local pointers to derived subtypes components (gridcell-level)

    forc_rain => clm_a2l%forc_rain
    forc_snow => clm_a2l%forc_snow
    forc_t    => clm_a2l%forc_t
#if (defined OFFLINE)
    flfall    => clm_a2l%flfall
#endif

    ! Assign local pointers to derived subtypes components (landunit-level)

!    ityplun => clm3%g%l%itype

    ! Assign local pointers to derived subtypes components (column-level)

    cgridcell     => clm3%g%l%c%gridcell
    clandunit     => clm3%g%l%c%landunit
    snl           => clm3%g%l%c%cps%snl
    snowage       => clm3%g%l%c%cps%snowage
    t_snow        => clm3%g%l%c%ces%t_snow
    t_grnd        => clm3%g%l%c%ces%t_grnd
    h2osno        => clm3%g%l%c%cws%h2osno
    snowice       => clm3%g%l%c%cws%snowice
    snowliq       => clm3%g%l%c%cws%snowliq
    zwt           => clm3%g%l%c%cws%zwt
    fcov          => clm3%g%l%c%cws%fcov
    qcharge       => clm3%g%l%c%cws%qcharge
    watsat        => clm3%g%l%c%cps%watsat
    z             => clm3%g%l%c%cps%z
    dz            => clm3%g%l%c%cps%dz
    zi            => clm3%g%l%c%cps%zi
    t_soisno      => clm3%g%l%c%ces%t_soisno
    h2osoi_ice    => clm3%g%l%c%cws%h2osoi_ice
    h2osoi_liq    => clm3%g%l%c%cws%h2osoi_liq
    h2osoi_vol    => clm3%g%l%c%cws%h2osoi_vol
    qflx_drain    => clm3%g%l%c%cwf%qflx_drain
    qflx_surf     => clm3%g%l%c%cwf%qflx_surf
    qflx_infl     => clm3%g%l%c%cwf%qflx_infl
    qflx_qrgwl    => clm3%g%l%c%cwf%qflx_qrgwl
    endwb         => clm3%g%l%c%cwbal%endwb
    begwb         => clm3%g%l%c%cwbal%begwb
    dz_lake       => clm3%g%l%c%cps%dz_lake
    t_lake         => clm3%g%l%c%ces%t_lake
    lake_icefrac   => clm3%g%l%c%cws%lake_icefrac
    do_capsnow    => clm3%g%l%c%cps%do_capsnow
    snowdp        => clm3%g%l%c%cps%snowdp
    qflx_snow_grnd_col => clm3%g%l%c%cwf%pwf_a%qflx_snow_grnd
    frac_iceold   => clm3%g%l%c%cps%frac_iceold
    qflx_evap_tot_col => clm3%g%l%c%cwf%pwf_a%qflx_evap_tot
    soilalpha    => clm3%g%l%c%cws%soilalpha
    zwt          => clm3%g%l%c%cws%zwt
    fcov         => clm3%g%l%c%cws%fcov
    rootr_column => clm3%g%l%c%cps%rootr_column
    qflx_rain_grnd_col => clm3%g%l%c%cwf%pwf_a%qflx_rain_grnd
    ! Use column variables here.
    qflx_sub_snow  => clm3%g%l%c%cwf%pwf_a%qflx_sub_snow
    qflx_evap_grnd => clm3%g%l%c%cwf%pwf_a%qflx_evap_grnd
    qflx_dew_snow  => clm3%g%l%c%cwf%pwf_a%qflx_dew_snow
    qflx_dew_grnd  => clm3%g%l%c%cwf%pwf_a%qflx_dew_grnd
    qflx_snowcap_col   => clm3%g%l%c%cwf%pwf_a%qflx_snowcap

    ! Assign local pointers to derived type members (pft-level)

    pcolumn       => clm3%g%l%c%p%column
    pgridcell      => clm3%g%l%c%p%gridcell
    qflx_prec_grnd     => clm3%g%l%c%p%pwf%qflx_prec_grnd
    qflx_snowcap       => clm3%g%l%c%p%pwf%qflx_snowcap
    qflx_snow_grnd_pft => clm3%g%l%c%p%pwf%qflx_snow_grnd
    qflx_rain_grnd     => clm3%g%l%c%p%pwf%qflx_rain_grnd
    qflx_evap_tot => clm3%g%l%c%p%pwf%qflx_evap_tot
    qflx_evap_soi  => clm3%g%l%c%p%pwf%qflx_evap_soi

! Block of biogeochem currently not used.
#ifdef CN
    wf            => clm3%g%l%c%cps%wf
    bsw2          => clm3%g%l%c%cps%bsw2
    psisat        => clm3%g%l%c%cps%psisat
    vwcsat        => clm3%g%l%c%cps%vwcsat
    soilpsi       => clm3%g%l%c%cps%soilpsi
#endif
!!!!!!

    ! Determine step size

!    dtime = get_step_size()

    ! Add soil water to water balance.
    do j = 1, nlevsoi
!dir$ concurrent
!cdir nodep
      do fc = 1, num_shlakec
         c = filter_shlakec(fc)
         begwb(c) = begwb(c) + h2osoi_ice(c,j) + h2osoi_liq(c,j)
      end do
    end do

!!!!!!!!!!!!!!!!!!!!!!!!!!!

    ! Do precipitation onto ground, etc., from Hydrology1.

!dir$ concurrent
!cdir nodep
    do fp = 1, num_shlakep
       p = filter_shlakep(fp)
       g = pgridcell(p)
!       l = plandunit(p)
       c = pcolumn(p)

       ! Precipitation onto ground (kg/(m2 s))
!       ! PET, 1/18/2005: Added new terms for mass balance correction
!       ! due to dynamic pft weight shifting (column-level h2ocan_loss)
!       ! Because the fractionation between rain and snow is indeterminate if
!       ! rain + snow = 0, I am adding this very small flux only to the rain
!       ! components.
       ! Not relevant unless PFTs are added to lake later.
!       if (frac_veg_nosno(p) == 0) then
          qflx_prec_grnd_snow(p) = forc_snow(g)
          qflx_prec_grnd_rain(p) = forc_rain(g) !+ h2ocan_loss(c)
!       else
!          qflx_prec_grnd_snow(p) = qflx_through_snow(p) + (qflx_candrip(p) * fracsnow(p))
!          qflx_prec_grnd_rain(p) = qflx_through_rain(p) + (qflx_candrip(p) * fracrain(p)) + h2ocan_loss(c)
!       end if
       qflx_prec_grnd(p) = qflx_prec_grnd_snow(p) + qflx_prec_grnd_rain(p)

       if (do_capsnow(c)) then
          qflx_snowcap(p) = qflx_prec_grnd_snow(p) + qflx_prec_grnd_rain(p)
          qflx_snow_grnd_pft(p) = 0._r8
          qflx_rain_grnd(p) = 0._r8
       else
          qflx_snowcap(p) = 0._r8
#if (defined OFFLINE)
          qflx_snow_grnd_pft(p) = qflx_prec_grnd(p)*(1._r8-flfall(g)) ! ice onto ground (mm/s)
          qflx_rain_grnd(p)     = qflx_prec_grnd(p)*flfall(g)      ! liquid water onto ground (mm/s)
#else
          qflx_snow_grnd_pft(p) = qflx_prec_grnd_snow(p)           ! ice onto ground (mm/s)
          qflx_rain_grnd(p)     = qflx_prec_grnd_rain(p)           ! liquid water onto ground (mm/s)
#endif
       end if
       ! Assuming one PFT; needed for below
       qflx_snow_grnd_col(c) = qflx_snow_grnd_pft(p)
       qflx_rain_grnd_col(c) = qflx_rain_grnd(p)

    end do ! (end pft loop)

    ! Determine snow height and snow water

!dir$ concurrent
!cdir nodep
    do fc = 1, num_shlakec
       c = filter_shlakec(fc)
!       l = clandunit(c)
       g = cgridcell(c)

       ! Use Alta relationship, Anderson(1976); LaChapelle(1961),
       ! U.S.Department of Agriculture Forest Service, Project F,
       ! Progress Rep. 1, Alta Avalanche Study Center:Snow Layer Densification.

       if (do_capsnow(c)) then
          dz_snowf = 0._r8
       else
          if (forc_t(g) > tfrz + 2._r8) then
             bifall=50._r8 + 1.7_r8*(17.0_r8)**1.5_r8
          else if (forc_t(g) > tfrz - 15._r8) then
             bifall=50._r8 + 1.7_r8*(forc_t(g) - tfrz + 15._r8)**1.5_r8
          else
             bifall=50._r8
          end if
          dz_snowf = qflx_snow_grnd_col(c)/bifall
          snowdp(c) = snowdp(c) + dz_snowf*dtime
          h2osno(c) = h2osno(c) + qflx_snow_grnd_col(c)*dtime  ! snow water equivalent (mm)
       end if

!       if (itype(l)==istwet .and. t_grnd(c)>tfrz) then
!          h2osno(c)=0._r8
!          snowdp(c)=0._r8
!          snowage(c)=0._r8
!       end if
       ! Take care of this later in function.

       ! When the snow accumulation exceeds 10 mm, initialize snow layer
       ! Currently, the water temperature for the precipitation is simply set
       ! as the surface air temperature

       newnode = 0    ! flag for when snow node will be initialized
       if (snl(c) == 0 .and. qflx_snow_grnd_col(c) > 0.0_r8 .and. snowdp(c) >= 0.01_r8) then
          newnode = 1
          snl(c) = -1
          dz(c,0) = snowdp(c)                       ! meter
          z(c,0) = -0.5_r8*dz(c,0)
          zi(c,-1) = -dz(c,0)
          snowage(c) = 0._r8                        ! snow age
          t_soisno(c,0) = min(tfrz, forc_t(g))      ! K
          h2osoi_ice(c,0) = h2osno(c)               ! kg/m2
          h2osoi_liq(c,0) = 0._r8                   ! kg/m2
          frac_iceold(c,0) = 1._r8
       end if

       ! The change of ice partial density of surface node due to precipitation.
       ! Only ice part of snowfall is added here, the liquid part will be added
       ! later.

       if (snl(c) < 0 .and. newnode == 0) then
          h2osoi_ice(c,snl(c)+1) = h2osoi_ice(c,snl(c)+1)+dtime*qflx_snow_grnd_col(c)
          dz(c,snl(c)+1) = dz(c,snl(c)+1)+dz_snowf*dtime
       end if

    end do

    ! Calculate sublimation and dew, adapted from HydrologyLake and Biogeophysics2.

!dir$ concurrent
!cdir nodep
    do fp = 1,num_shlakep
       p = filter_shlakep(fp)
       c = pcolumn(p)
       jtop = snl(c)+1

       ! Use column variables here
       qflx_evap_grnd(c) = 0._r8
       qflx_sub_snow(c) = 0._r8
       qflx_dew_snow(c) = 0._r8
       qflx_dew_grnd(c) = 0._r8

       if (jtop <= 0) then ! snow layers
          j = jtop
          ! Assign ground evaporation to sublimation from soil ice or to dew
          ! on snow or ground

          if (qflx_evap_soi(p) >= 0._r8) then
          ! for evaporation partitioning between liquid evap and ice sublimation, 
          ! use the ratio of liquid to (liquid+ice) in the top layer to determine split
          ! Since we're not limiting evap over lakes, but still can't remove more from top
          ! snow layer than there is there, create temp. limited evap_soi.
             qflx_evap_soi_lim = min(qflx_evap_soi(p), (h2osoi_liq(c,j)+h2osoi_ice(c,j))/dtime)
             if ((h2osoi_liq(c,j)+h2osoi_ice(c,j)) > 0._r8) then
                qflx_evap_grnd(c) = max(qflx_evap_soi_lim*(h2osoi_liq(c,j)/(h2osoi_liq(c,j)+h2osoi_ice(c,j))), 0._r8)
             else
                qflx_evap_grnd(c) = 0._r8
             end if
             qflx_sub_snow(c) = qflx_evap_soi_lim - qflx_evap_grnd(c)     
          else
             if (t_grnd(c) < tfrz) then
                qflx_dew_snow(c) = abs(qflx_evap_soi(p))
             else
                qflx_dew_grnd(c) = abs(qflx_evap_soi(p))
             end if
          end if
          ! Update the pft-level qflx_snowcap
          ! This was moved in from Hydrology2 to keep all pft-level
          ! calculations out of Hydrology2
          if (do_capsnow(c)) qflx_snowcap(p) = qflx_snowcap(p) + qflx_dew_snow(c) + qflx_dew_grnd(c)

       else ! No snow layers: do as in HydrologyLake but with actual clmtype variables
          if (qflx_evap_soi(p) >= 0._r8) then
             ! Sublimation: do not allow for more sublimation than there is snow
             ! after melt.  Remaining surface evaporation used for infiltration.
             qflx_sub_snow(c) = min(qflx_evap_soi(p), h2osno(c)/dtime)
             qflx_evap_grnd(c) = qflx_evap_soi(p) - qflx_sub_snow(c)
          else
             if (t_grnd(c) < tfrz-0.1_r8) then
                qflx_dew_snow(c) = abs(qflx_evap_soi(p))
             else
                qflx_dew_grnd(c) = abs(qflx_evap_soi(p))
             end if
          end if

          ! Update snow pack for dew & sub.
          h2osno_temp = h2osno(c)
          if (do_capsnow(c)) then
             h2osno(c) = h2osno(c) - qflx_sub_snow(c)*dtime
             qflx_snowcap(p) = qflx_snowcap(p) + qflx_dew_snow(c) + qflx_dew_grnd(c)
          else
             h2osno(c) = h2osno(c) + (-qflx_sub_snow(c)+qflx_dew_snow(c))*dtime
          end if
          if (h2osno_temp > 0._r8) then
             snowdp(c) = snowdp(c) * h2osno(c) / h2osno_temp
          else
             snowdp(c) = h2osno(c)/snow_bd !Assume a constant snow bulk density = 250.
          end if

#if (defined PERGRO)
          if (abs(h2osno(c)) < 1.e-10_r8) h2osno(c) = 0._r8
#else
          h2osno(c) = max(h2osno(c), 0._r8)
#endif

       end if

    qflx_snowcap_col(c) = qflx_snowcap(p)

    end do


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    ! Determine initial snow/no-snow filters (will be modified possibly by
    ! routines CombineSnowLayers and DivideSnowLayers below

    call BuildSnowFilter(lbc, ubc, num_shlakec, filter_shlakec, &
         num_shlakesnowc, filter_shlakesnowc, num_shlakenosnowc, filter_shlakenosnowc)

    ! Determine the change of snow mass and the snow water onto soil

    call SnowWater(lbc, ubc, num_shlakesnowc, filter_shlakesnowc, num_shlakenosnowc, filter_shlakenosnowc)

    ! Determine soil hydrology
    ! Here this consists only of making sure that soil is saturated even as it melts and 10%
    ! of pore space opens up. Conversely, if excess ice is melting and the liquid water exceeds the
    ! saturation value, then remove water.

    do j = 1,nlevsoi
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

          if (h2osoi_vol(c,j) < watsat(c,j)) then
             h2osoi_liq(c,j) = (watsat(c,j)*dz(c,j) - h2osoi_ice(c,j)/denice)*denh2o
          ! h2osoi_vol will be updated below, and this water addition will come from qflx_qrgwl
          else if (h2osoi_liq(c,j) > watsat(c,j)*denh2o*dz(c,j)) then
             h2osoi_liq(c,j) = watsat(c,j)*denh2o*dz(c,j)
          end if

       end do
    end do
!!!!!!!!!!

!    if (.not. is_perpetual()) then
    if (1==1) then

       ! Natural compaction and metamorphosis.

       call SnowCompaction(lbc, ubc, num_shlakesnowc, filter_shlakesnowc)

       ! Combine thin snow elements

       call CombineSnowLayers(lbc, ubc, num_shlakesnowc, filter_shlakesnowc)

       ! Divide thick snow elements

       call DivideSnowLayers(lbc, ubc, num_shlakesnowc, filter_shlakesnowc)

    else

       do fc = 1, num_shlakesnowc
          c = filter_shlakesnowc(fc)
          h2osno(c) = 0._r8
       end do
       do j = -nlevsno+1,0
          do fc = 1, num_shlakesnowc
             c = filter_shlakesnowc(fc)
             if (j >= snl(c)+1) then
                h2osno(c) = h2osno(c) + h2osoi_ice(c,j) + h2osoi_liq(c,j)
             end if
          end do
       end do

    end if

    ! Check for snow layers above lake with unfrozen top layer.  Mechanically,
    ! the snow will fall into the lake and melt or turn to ice.  If the top layer has
    ! sufficient heat to melt the snow without freezing, then that will be done.
    ! Otherwise, the top layer will undergo freezing, but only if the top layer will
    ! not freeze completely.  Otherwise, let the snow layers persist and melt by diffusion.
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

          if (t_lake(c,1) > tfrz .and. lake_icefrac(c,1) == 0._r8 .and. snl(c) < 0) then
             unfrozen(c) = .true.
          else
             unfrozen(c) = .false.
          end if
       end do

    do j = -nlevsno+1,0
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

          if (unfrozen(c)) then
             if (j == -nlevsno+1) then
                sumsnowice(c) = 0._r8
                heatsum(c) = 0._r8
             end if
             if (j >= snl(c)+1) then
                sumsnowice(c) = sumsnowice(c) + h2osoi_ice(c,j)
                heatsum(c) = heatsum(c) + h2osoi_ice(c,j)*cpice*(tfrz - t_soisno(c,j)) &
                           + h2osoi_liq(c,j)*cpliq*(tfrz - t_soisno(c,j))
             end if
          end if
       end do
    end do

!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

          if (unfrozen(c)) then
             heatsum(c) = heatsum(c) + sumsnowice(c)*hfus
             heatrem = (t_lake(c,1) - tfrz)*cpliq*denh2o*dz_lake(c,1) - heatsum(c)

             if (heatrem + denh2o*dz_lake(c,1)*hfus > 0._r8) then            
                ! Remove snow and subtract the latent heat from the top layer.
                h2osno(c) = 0._r8
                snl(c) = 0
                ! The rest of the bookkeeping for the removed snow will be done below.
#if (defined CLMDEBUG)
                write(6,*)'Snow layers removed above unfrozen lake for column, snowice:', &
                          c, sumsnowice(c)
#endif
                if (heatrem > 0._r8) then ! simply subtract the heat from the layer
                   t_lake(c,1) = t_lake(c,1) - heatrem/(cpliq*denh2o*dz_lake(c,1))
                else !freeze part of the layer
                   t_lake(c,1) = tfrz
                   lake_icefrac(c,1) = -heatrem/(denh2o*dz_lake(c,1)*hfus)
                end if
             end if
          end if
       end do
!!!!!!!!!!!!

    ! Set snow age to zero if no snow

!dir$ concurrent
!cdir nodep
    do fc = 1, num_shlakesnowc
       c = filter_shlakesnowc(fc)
       if (snl(c) == 0) then
          snowage(c) = 0._r8
       end if
    end do

    ! Set empty snow layers to zero

    do j = -nlevsno+1,0
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakesnowc
          c = filter_shlakesnowc(fc)
          if (j <= snl(c) .and. snl(c) > -nlevsno) then
             h2osoi_ice(c,j) = 0._r8
             h2osoi_liq(c,j) = 0._r8
             t_soisno(c,j) = 0._r8
             dz(c,j) = 0._r8
             z(c,j) = 0._r8
             zi(c,j-1) = 0._r8
          end if
       end do
    end do

    ! Build new snow filter

    call BuildSnowFilter(lbc, ubc, num_shlakec, filter_shlakec, &
         num_shlakesnowc, filter_shlakesnowc, num_shlakenosnowc, filter_shlakenosnowc)

    ! Vertically average t_soisno and sum of h2osoi_liq and h2osoi_ice
    ! over all snow layers for history output

!dir$ concurrent
!cdir nodep
    do fc = 1, num_shlakesnowc
       c = filter_shlakesnowc(fc)
       t_snow(c)  = 0._r8
       snowice(c) = 0._r8
       snowliq(c) = 0._r8
    end do
!dir$ concurrent
!cdir nodep
    do fc = 1, num_shlakenosnowc
       c = filter_shlakenosnowc(fc)
       t_snow(c)  = spval
       snowice(c) = spval
       snowliq(c) = spval
    end do

    do j = -nlevsno+1, 0
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakesnowc
          c = filter_shlakesnowc(fc)
          if (j >= snl(c)+1) then
             t_snow(c)  = t_snow(c) + t_soisno(c,j)
             snowice(c) = snowice(c) + h2osoi_ice(c,j)
             snowliq(c) = snowliq(c) + h2osoi_liq(c,j)
          end if
       end do
    end do

    ! Determine ending water balance and volumetric soil water

!dir$ concurrent
!cdir nodep
    do fc = 1, num_shlakec
       
       c = filter_shlakec(fc)
       if (snl(c) < 0) t_snow(c) = t_snow(c)/abs(snl(c))
       endwb(c) = h2osno(c)
    end do

    do j = 1, nlevsoi
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)
          endwb(c) = endwb(c) + h2osoi_ice(c,j) + h2osoi_liq(c,j)
          h2osoi_vol(c,j) = h2osoi_liq(c,j)/(dz(c,j)*denh2o) + h2osoi_ice(c,j)/(dz(c,j)*denice)
       end do
    end do

#if (defined CLMDEBUG)
    ! Check to make sure snow water adds up correctly.
    do j = -nlevsno+1,0
!dir$ concurrent
!cdir nodep
      do fc = 1, num_shlakec
         c = filter_shlakec(fc)
 
         jtop = snl(c)+1
         if(j == jtop) snow_water(c) = 0._r8
         if(j >= jtop) then
            snow_water(c) = snow_water(c) + h2osoi_ice(c,j) + h2osoi_liq(c,j)
            if(j == 0 .and. abs(snow_water(c)-h2osno(c))>1.e-7_r8) then
               write(6,*)'h2osno does not equal sum of snow layers in ShalLakeHydrology:', &
                         'column, h2osno, sum of snow layers =', c, h2osno(c), snow_water(c)
               call endrun()
            end if
         end if
      end do
    end do
#endif

!!!!!!!!!!!!!
    ! Do history variables and set special landunit runoff (adapted from end of HydrologyLake)
!dir$ concurrent
!cdir nodep
    do fp = 1,num_shlakep
       p = filter_shlakep(fp)
       c = pcolumn(p)
       g = pgridcell(p)

       qflx_infl(c)      = 0._r8
       qflx_surf(c)      = 0._r8
       qflx_drain(c)     = 0._r8
       rootr_column(c,:) = spval
       soilalpha(c)      = spval
       zwt(c)            = spval
       fcov(c)           = spval
       qcharge(c)        = spval
!       h2osoi_vol(c,:)   = spval

       ! Insure water balance using qflx_qrgwl
       qflx_qrgwl(c)     = forc_rain(g) + forc_snow(g) - qflx_evap_tot(p) - (endwb(c)-begwb(c))/dtime
#if (defined CLMDEBUG)
    write(6,*)'c, rain, snow, evap, endwb, begwb, qflx_qrgwl:', &
       c, forc_rain(g), forc_snow(g), qflx_evap_tot(p), endwb(c), begwb(c), qflx_qrgwl(c)
#endif

       ! The pft average must be done here for output to history tape
       qflx_evap_tot_col(c) = qflx_evap_tot(p)
    end do

!!!!!!!!!!!!!
!For now, bracket off the remaining biogeochem code.  May need to bring it back
!to do soil carbon and methane beneath lakes.
#if (defined CN)
#ifndef SHLAKE
    do j = 1, nlevsoi
!dir$ concurrent
!cdir nodep
       do fc = 1, num_soilc
          c = filter_soilc(fc)
          
          if (h2osoi_liq(c,j) > 0._r8) then
             vwc = h2osoi_liq(c,j)/(dz(c,j)*denh2o)
            
             ! the following limit set to catch very small values of 
             ! fractional saturation that can crash the calculation of psi
           
             fsat = max(vwc/vwcsat(c,j), 0.001_r8)
             psi = psisat(c,j) * (fsat)**bsw2(c,j)
             soilpsi(c,j) = min(max(psi,-15.0_r8),0._r8)
          else 
             soilpsi(c,j) = -15.0_r8
          end if
       end do
    end do
#endif
#endif

#if (defined DGVM) || (defined CN)
#ifndef SHLAKE
    ! Available soil water up to a depth of 0.5 m.
    ! Potentially available soil water (=whc) up to a depth of 0.5 m.
    ! Water content as fraction of whc up to a depth of 0.5 m.

!dir$ concurrent
!cdir nodep
    do c = lbc,ubc
       l = clandunit(c)
       if (ityplun(l) == istsoil) then
          rwat(c) = 0._r8
          swat(c) = 0._r8
          rz(c)   = 0._r8
       end if
    end do

    do j = 1, nlevsoi
!dir$ concurrent
!cdir nodep
       do c = lbc,ubc
          l = clandunit(c)
          if (ityplun(l) == istsoil) then
             if (z(c,j)+0.5_r8*dz(c,j) <= 0.5_r8) then
                watdry = watsat(c,j) * (316230._r8/sucsat(c,j)) ** (-1._r8/bsw(c,j))
                rwat(c) = rwat(c) + (h2osoi_vol(c,j)-watdry) * dz(c,j)
                swat(c) = swat(c) + (watsat(c,j)    -watdry) * dz(c,j)
                rz(c) = rz(c) + dz(c,j)
             end if
          end if
       end do
    end do

!dir$ concurrent
!cdir nodep
    do c = lbc,ubc
       l = clandunit(c)
       if (ityplun(l) == istsoil) then
          if (rz(c) /= 0._r8) then
             tsw  = rwat(c)/rz(c)
             stsw = swat(c)/rz(c)
          else
             watdry = watsat(c,1) * (316230._r8/sucsat(c,1)) ** (-1._r8/bsw(c,1))
             tsw = h2osoi_vol(c,1) - watdry
             stsw = watsat(c,1) - watdry
          end if
          wf(c) = tsw/stsw
       else
          wf(c) = 1.0_r8
       end if
    end do

#endif
#endif

  end subroutine ShalLakeHydrology

end module ShalLakeHydrologyMod
