module clm_varcon
!----------------------------------------------------------------------- 
!BOP
!
! !MODULE: clm_varcon
!
! !DESCRIPTION: 
! Module containing various model constants 
!
! !USES:
  use shr_kind_mod, only: r8 => shr_kind_r8
  use clm_varpar, only : numcol,numrad,nlevsoi,nlevlak,&
                         maxpatch_pft,numpft
!
! !PUBLIC TYPES:
  implicit none
  save
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!----------------------------------------------------------------------- 
  
  !------------------------------------------------------------------
  ! Initialize physical constants
  !------------------------------------------------------------------
  real(r8), parameter :: cday   = 86400.0_r8   !sec in calendar day ~ sec

  real(r8), parameter :: pie    = 3.141592653589793_r8 ! pi
  real(r8), parameter :: grav   = 9.80616_r8   !gravity constant [m/s2]
  real(r8), parameter :: sb     = 5.67e-8_r8   !stefan-boltzmann constant  [W/m2/K4]
  real(r8), parameter :: vkc    = 0.4_r8       !von Karman constant [-]
  real(r8), parameter :: rgas   = 8314.468_r8  !Universal gas constant ~ J/K/kmole
  real(r8), parameter :: rwat   = 461.5046_r8  !gas constant for water vapor [J/(kg K)]
  real(r8), parameter :: rair   = 287.0423_r8  !gas constant for dry air [J/kg/K]
  real(r8), parameter :: roverg = 47062.73_r8  !Rw/g constant = (8.3144/0.018)/(9.80616)*1000. mm/K
  real(r8), parameter :: cpliq  = 4.188e3_r8   !Specific heat of water [J/kg-K]
  real(r8), parameter :: cpice  = 2.11727e3_r8 !Specific heat of ice [J/kg-K]
  real(r8), parameter :: cpair  = 1.00464e3_r8 !specific heat of dry air [J/kg/K]
  real(r8), parameter :: hvap   = 2.501e6_r8   !Latent heat of evap for water [J/kg]
  real(r8), parameter :: hfus   = 3.337e5_r8   !Latent heat of fusion for ice [J/kg]
  real(r8), parameter :: hsub   = 2.501e6_r8+3.337e5_r8 !Latent heat of sublimation    [J/kg]
  real(r8), parameter :: denh2o = 1.000e3_r8   !density of liquid water [kg/m3]
  real(r8), parameter :: denice = 0.917e3_r8   !density of ice [kg/m3]
  real(r8), parameter :: tkair  = 0.023        !thermal conductivity of air   [W/m/k]
  real(r8), parameter :: tkice  = 2.290        !thermal conductivity of ice   [W/m/k]
  real(r8), parameter :: tkwat  = 0.6          !thermal conductivity of water [W/m/k]
  real(r8), parameter :: tfrz   = 273.16_r8    !freezing temperature [K]
  real(r8), parameter :: tcrit  = 2.5          !critical temperature to determine rain or snow
  real(r8), parameter :: po2    = 0.209        !constant atmospheric partial pressure  O2 (mol/mol)
  real(r8), parameter :: pco2   = 355.e-06     !constant atmospheric partial pressure CO2 (mol/mol)
  real(r8), parameter :: pstd   = 101325.0_r8  !standard pressure ~ pascals

  real(r8), parameter :: bdsno = 250.            !bulk density snow (kg/m**3)

  real(r8), parameter :: re = 6.37122e6_r8*0.001 !radius of earth (km)

  real(r8), public, parameter :: spval = 1.e36  !special value for missing data (ocean)

  ! These are tunable constants from clm2_3

  real(r8), parameter :: zlnd = 0.01      !Roughness length for soil [m]
  real(r8), parameter :: zsno = 0.0024    !Roughness length for snow [m]
  real(r8), parameter :: csoilc = 0.004   !Drag coefficient for soil under canopy [-]
  real(r8), parameter :: capr   = 0.34    !Tuning factor to turn first layer T into surface T  
  real(r8), parameter :: cnfac  = 0.5     !Crank Nicholson factor between 0 and 1
  real(r8), parameter :: ssi    = 0.033   !Irreducible water saturation of snow
  real(r8), parameter :: wimp   = 0.05    !Water impremeable if porosity less than wimp
  real(r8), parameter :: pondmx = 10.0    !Ponding depth (mm)

!Additional CLM 3.5 constants, Zack Subin 7/15/08
  real(r8) :: o2_molar_const = 0.209_r8   !constant atmospheric O2 molar ratio (mol/mol)
#if (defined CASA)
  real(r8) :: co2_ppmv_const = 283.1878_r8    !constant atmospheric CO2 molar ratio (umol/mol)
#elif (defined CN)
  real(r8) :: co2_ppmv_const = 283.1878_r8    !constant atmospheric CO2 molar ratio (umol/mol)
#else
  real(r8) :: co2_ppmv_const = 355._r8    !constant atmospheric CO2 molar ratio (umol/mol)
#endif
  integer , public, parameter :: ispval = -9999     ! special value for int data
  ! 4/14/05: PET
  ! Adding isotope code
#if (defined CN)
  real(r8), parameter :: preind_atm_del13c = -6.0   ! preindustrial value for atmospheric del13C
  real(r8), parameter :: preind_atm_ratio = SHR_CONST_PDB + (preind_atm_del13c * SHR_CONST_PDB)/1000.0  ! 13C/12C
  real(r8) :: c13ratio = preind_atm_ratio/(1.0+preind_atm_ratio) ! 13C/(12+13)C preind atmosphere
#endif
  real(r8), parameter :: maxwattabfract = 0.3 !Max water table fraction for landunit that is not wet or ice
!!!!!


  !------------------------------------------------------------------
  ! Initialize water type constants
  !------------------------------------------------------------------

  ! "water" types 
  !   1     soil
  !   2     land ice (glacier)
  !   3     deep lake
  !   4     shallow lake
  !   5     wetland: swamp, marsh, etc

  integer,parameter :: istsoil = 1  !soil         "water" type
  integer,parameter :: istice  = 2  !land ice     "water" type
  integer,parameter :: istdlak = 3  !deep lake    "water" type
  integer,parameter :: istslak = 4  !shallow lake "water" type
  integer,parameter :: istwet  = 5  !wetland      "water" type
  integer,parameter :: isturb  = 6  !urban        landunit type

  !------------------------------------------------------------------
  ! Initialize miscellaneous radiation constants
  !------------------------------------------------------------------

  integer, private :: i  ! loop index

  integer,parameter :: num_landcover_types  = 24  !24 (USGS)


  ! saturated soil albedos for 8 color classes: 1=vis, 2=nir

  real(r8) :: albsat(numcol,numrad) !wet soil albedo by color class and waveband
  data(albsat(i,1),i=1,8)/0.12,0.11,0.10,0.09,0.08,0.07,0.06,0.05/
  data(albsat(i,2),i=1,8)/0.24,0.22,0.20,0.18,0.16,0.14,0.12,0.10/

  ! dry soil albedos for 8 color classes: 1=vis, 2=nir 

  real(r8) :: albdry(numcol,numrad) !dry soil albedo by color class and waveband
  data(albdry(i,1),i=1,8)/0.24,0.22,0.20,0.18,0.16,0.14,0.12,0.10/
  data(albdry(i,2),i=1,8)/0.48,0.44,0.40,0.36,0.32,0.28,0.24,0.20/

  ! albedo land ice: 1=vis, 2=nir

  real(r8) :: albice(numrad)        !albedo land ice by waveband
  data (albice(i),i=1,numrad) /0.80, 0.55/

  ! albedo frozen lakes: 1=vis, 2=nir 

  real(r8) :: alblak(numrad)        !albedo frozen lakes by waveband
  data (alblak(i),i=1,numrad) /0.60, 0.40/

  ! omega,betad,betai for snow 

  real(r8),parameter :: betads  = 0.5       !two-stream parameter betad for snow
  real(r8),parameter :: betais  = 0.5       !two-stream parameter betai for snow
  real(r8) :: omegas(numrad)      !two-stream parameter omega for snow by band
  data (omegas(i),i=1,numrad) /0.8, 0.4/

  !------------------------------------------------------------------
  ! Soil and Lake depths are constants for now
  ! The values for the following arrays are set in routine iniTimeConst
  !------------------------------------------------------------------

  real(r8) :: zlak(1:nlevlak)     !lake z  (layers) 
  real(r8) :: dzlak(1:nlevlak)    !lake dz (thickness)
  real(r8) :: zsoi(1:nlevsoi)     !soil z  (layers)
  real(r8) :: dzsoi(1:nlevsoi)    !soil dz (thickness)
  real(r8) :: zisoi(0:nlevsoi)    !soil zi (interfaces)  

    real(r8) :: sand(19)                           ! percent sand
    real(r8) :: clay(19)                           ! percent clay
    integer  :: soic(19)
    integer  :: plant(24,maxpatch_pft)
    real(r8) :: cover(24,maxpatch_pft)

    data(sand(i), i=1,19)/92.,80.,66.,20.,5.,43.,60.,&
      10.,32.,51., 6.,22.,39.7,0.,100.,54.,17.,100.,92./

    data(clay(i), i=1,19)/ 3., 5.,10.,15.,5.,18.,27.,&
      33.,33.,41.,47.,58.,14.7,0., 0., 8.5,54.,  0., 3./

    data(soic(i), i=1,19)/1,2,2,3,3,4,5,5,6,7,7,8,8,0,&
                          1,1,4,7,1/

!  soil type from MM5
! (1)  sand
! (2)  loamy-sand
! (3)  sandy-loam
! (4)  silt-loam
! (5)  silt
! (6)  loam
! (7)  sandy-clay-loam
! (8)  silty-clay-loam
! (9)  clay-loam
! (10) sandy-clay
! (11) silty-clay
! (12) clay
! (13) organic-material,
! (14) water
! (15) bedrock
! (16) other(land-ice)
! (17) playa
! (18) lava
! (19) white-sand
!----------------------------------------------------------------------------
  data (plant(i,1),i=1,24) / 0,  15,  15,  15,  15,  15, &
                            14,   9,   9,  14,   7,   3, &
                             4,   1,   1,   0,   0,   4, &
                            11,  11,   2,  11,  11,   0/
  data (cover(i,1),i=1,24) /100.,  85.,  85.,  85.,  50.,  40., &
                             60.,  80.,  50.,  70.,  75.,  50., &
                             95.,  75.,  37., 100., 100.,  80., &
                             10.,  30.,  13.,  20.,  10., 100./

  data (plant(i,2),i=1,24) / 0,   0,   0,   0,  14,   3, &
                            13,   0,  14,   6,   0,   0, &
                             0,   0,   7,   0,   0,   0, &
                             0,  12,   3,  12,  12,   0/
  data (cover(i,2),i=1,24) /  0.,  15.,  15.,  15.,  35.,  30., &
                             20.,  20.,  30.,  30.,  25.,  50., &
                              5.,  25.,  37.,   0.,   0.,  20., &
                             90.,  30.,  13.,  20.,  10.,   0./

  data (plant(i,3),i=1,24) / 0,   0,   0,   0,   0,   0, &
                             0,   0,   0,   0,   0,   0, &
                             0,   0,   0,   0,   0,   0, &
                             0,   0,  10,   0,   0,   0/

  data (cover(i,3),i=1,24) /  0.,   0.,   0.,   0.,  15.,  30., &
                             20.,   0.,  20.,   0.,   0.,   0., &
                              0.,   0.,  26.,   0.,   0.,   0., &
                              0.,  40.,  24.,  60.,  80.,   0./

  data (plant(i,4),i=1,24) / 0,   0,   0,   0,   0,   0, &
                             0,   0,   0,   0,   0,   0, &
                             0,   0,   0,   0,   0,   0, &
                             0,   0,   0,   0,   0,   0/

  data (cover(i,4),i=1,24) / 0.,   0.,   0.,   0.,   0.,   0., &
                             0.,   0.,   0.,   0.,   0.,   0., &
                             0.,   0.,   0.,   0.,   0.,   0., &
                             0.,   0.,  50.,   0.,   0.,   0./

!-----------------------------------------------------------------------

!USGS vegetation 24 categories
!
!Urban and Built-Up Land            1
!Dryland Cropland and Pasture       2
!Irrigated Cropland and Pasture     3
!Mixed Dryland/Irrg. C.P.           4
!Cropland/Grassland Mosaic          5
!Cropland/Woodland Mosaic           6
!Grassland                          7
!Shrubland                          8
!Mixed Shrubland/Grassland          9
!Savanna                           10
!Deciduous Broadleaf Forest        11
!Deciduous Needleleaf Forest       12
!Evergreen Broadleaf Forest        13
!Evergreen Needleleaf Forest       14
!Mixed Forest                      15
!Water Bodies                      16
!Herbaceous Wetland                17
!Wooded Wetland                    18
!Barren or Sparsely Vegetated      19
!Herbaceous Tundra                 20
!Wooded Tundra                     21
!Mixed Tundra                      22
!Bare Ground Tundra                23
!Snow or Ice                       24
!-----------------------------------------------------------------------

!Note: don't forget new PFT parameters defined for CLM 3.5 in pftvarcon.

    real(r8):: lai(numpft,12),sai(numpft,12),hvt(numpft),hvb(numpft)

    data (lai(1,i),i=1,12) &
        /4.1,4.2,4.6,4.8,4.9,5.0,4.8,4.7,4.6,4.2,4.0,4.0/
    data (lai(2,i),i=1,12) &
        /4.1,4.2,4.6,4.8,4.9,5.0,4.8,4.7,4.6,4.2,4.0,4.0/
    data (lai(3,i),i=1,12) &
        /0.0,0.0,0.0,0.6,1.2,2.0,2.6,1.7,1.0,0.5,0.2,0.0/
    data (lai(4,i),i=1,12) &
        /4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5/
    data (lai(5,i),i=1,12) &
        /4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5/
    data (lai(6,i),i=1,12) &
        /0.8,0.7,0.4,0.5,0.5,0.7,1.7,3.0,2.5,1.6,1.0,1.0/
    data (lai(7,i),i=1,12) &
        /0.0,0.0,0.3,1.2,3.0,4.7,4.5,3.4,1.2,0.3,0.0,0.0/
    data (lai(8,i),i=1,12) &
        /0.0,0.0,0.3,1.2,3.0,4.7,4.5,3.4,1.2,0.3,0.0,0.0/
    data (lai(9,i),i=1,12) &
        /1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0/
    data (lai(10,i),i=1,12) &
        /0.9,0.8,0.2,0.2,0.0,0.0,0.0,0.2,0.4,0.5,0.6,0.8/
    data (lai(11,i),i=1,12) &
        /0.0,0.0,0.0,0.0,0.0,0.2,1.4,1.2,0.0,0.0,0.0,0.0/
    data (lai(12,i),i=1,12) &
        /0.4,0.5,0.6,0.7,1.2,3.0,3.5,1.5,0.7,0.6,0.5,0.4/
    data (lai(13,i),i=1,12) &
        /0.0,0.0,0.0,0.0,0.0,0.2,1.4,1.2,0.0,0.0,0.0,0.0/
    data (lai(14,i),i=1,12) &
        /0.4,0.5,0.6,0.7,1.2,3.0,3.5,1.5,0.7,0.6,0.5,0.4/
#ifndef CACROP
    data (lai(15,i),i=1,12) &
        /0.0,0.0,0.0,0.0,1.0,2.0,3.0,3.0,1.5,0.0,0.0,0.0/
    data (lai(16,i),i=1,12) &
        /0.0,0.0,0.0,0.0,1.0,2.0,3.0,3.0,1.5,0.0,0.0,0.0/
#else
! for California Crop Land
    data (lai(15,i),i=1,12) &
        /1.7,3.7,4.3,3.7,2.4,2.5,4.2,4.2,3.6,1.7,1.0,0.9/
    data (lai(16,i),i=1,12) &
        /1.7,3.7,4.3,3.7,2.4,2.5,4.2,4.2,3.6,1.7,1.0,0.9/
#endif
!-----------------------------------------------------------------------

    data (sai(1,i),i=1,12) &
       /0.4,0.5,0.4,0.3,0.4,0.5,0.5,0.6,0.6,0.7,0.6,0.5/
    data (sai(2,i),i=1,12) &
       /0.4,0.5,0.4,0.3,0.4,0.5,0.5,0.6,0.6,0.7,0.6,0.5/
    data (sai(3,i),i=1,12) &
       /0.3,0.3,0.3,0.4,0.4,0.4,1.7,1.2,1.0,0.8,0.6,0.5/
    data (sai(4,i),i=1,12) &
       /0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5/
    data (sai(5,i),i=1,12) &
       /0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5/
    data (sai(6,i),i=1,12) &
       /0.4,0.3,0.5,0.3,0.3,0.3,0.3,0.7,0.7,1.1,0.9,0.2/
    data (sai(7,i),i=1,12) &
       /0.4,0.4,0.4,0.4,0.5,0.4,0.9,1.4,2.6,1.4,0.6,0.4/
    data (sai(8,i),i=1,12) &
       /0.4,0.4,0.4,0.4,0.5,0.4,0.9,1.4,2.6,1.4,0.6,0.4/
    data (sai(9,i),i=1,12) &
       /0.3,0.3,0.3,0.3,0.3,0.3,0.3,0.3,0.3,0.3,0.3,0.3/
    data (sai(10,i),i=1,12) &
       /0.1,0.2,0.6,0.1,0.6,0.0,0.1,0.1,0.1,0.1,0.1,0.1/
    data (sai(11,i),i=1,12) &
       /0.1,0.1,0.1,0.1,0.1,0.1,0.1,0.5,1.4,0.1,0.1,0.1/
    data (sai(12,i),i=1,12) &
       /0.3,0.3,0.3,0.3,0.3,0.4,0.8,2.3,1.1,0.4,0.4,0.4/
    data (sai(13,i),i=1,12) &
       /0.1,0.1,0.1,0.1,0.1,0.1,0.1,0.5,1.4,0.1,0.1,0.1/
    data (sai(14,i),i=1,12) &
       /0.3,0.3,0.3,0.3,0.3,0.4,0.8,2.3,1.1,0.4,0.4,0.4/
#ifndef CACROP
    data (sai(15,i),i=1,12) &
       /0.0,0.0,0.0,0.0,1.0,2.0,3.0,3.0,1.5,0.0,0.0,0.0/
    data (sai(16,i),i=1,12) &
       /0.0,0.0,0.0,0.0,1.0,2.0,3.0,3.0,1.5,0.0,0.0,0.0/
#else
! for California Crop Land
    data (sai(15,i),i=1,12) &
       /0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0/
    data (sai(16,i),i=1,12) &
       /0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0/
#endif
    data hvt /17.0,17.0,14.0,35.0,35.0,18.0,20.0,20.0,&
      0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5/
    data hvb /8.50, 8.50,7.00,1.00,1.00,10.00,11.50,11.50,&
      0.10,0.10,0.10,0.01,0.10,0.01,0.01,0.01/
!-----------------------------------------------------------------------
  character(len=40) pftname(0:numpft)
  real(r8) dleaf(0:numpft)       !characteristic leaf dimension (m)
  real(r8) c3psn(0:numpft)       !photosynthetic pathway: 0. = c4, 1. = c3
  real(r8) vcmx25(0:numpft)      !max rate of carboxylation at 25C (umol CO2/m**2/s)
  real(r8) mp(0:numpft)          !slope of conductance-to-photosynthesis relationship
  real(r8) qe25(0:numpft)        !quantum efficiency at 25C (umol CO2 / umol photon)
  real(r8) xl(0:numpft)          !leaf/stem orientation index
  real(r8) rhol(0:numpft,numrad) !leaf reflectance: 1=vis, 2=nir
  real(r8) rhos(0:numpft,numrad) !stem reflectance: 1=vis, 2=nir
  real(r8) taul(0:numpft,numrad) !leaf transmittance: 1=vis, 2=nir
  real(r8) taus(0:numpft,numrad) !stem transmittance: 1=vis, 2=nir
  real(r8) z0mr(0:numpft)        !ratio of momentum roughness length to canopy top height (-)
  real(r8) displar(0:numpft)     !ratio of displacement height to canopy top height (-)
  real(r8) roota_par(0:numpft)   !CLM rooting distribution parameter [1/m]
  real(r8) rootb_par(0:numpft)   !CLM rooting distribution parameter [1/m]

data (pftname(i),i=1,16)/'needleleaf_evergreen_temperate_tree',&
                           'needleleaf_evergreen_boreal_tree'   ,&
                           'needleleaf_deciduous_boreal_tree'   ,&
                           'broadleaf_evergreen_tropical_tree'  ,&
                           'broadleaf_evergreen_temperate_tree' ,&
                           'broadleaf_deciduous_tropical_tree'  ,&
                           'broadleaf_deciduous_temperate_tree' ,&
                           'broadleaf_deciduous_boreal_tree'    ,&
                           'broadleaf_evergreen_shrub'          ,&
                           'broadleaf_deciduous_temperate_shrub',&
                           'broadleaf_deciduous_boreal_shrub'   ,&
                           'c3_arctic_grass'                    ,&
                           'c3_non-arctic_grass'                ,&
                           'c4_grass'                           ,&
                           'corn'                               ,&
                           'wheat'/

  data (z0mr(i),i=1,16)/ 0.055, 0.055, 0.055, 0.075, 0.075,  &
       0.055,0.055, 0.055, 0.120, 0.120, 0.120, 0.120, 0.120,&
       0.120, 0.120, 0.120/
  
  data (displar(i),i=1,16)/ 0.67, 0.67, 0.67, 0.67, 0.67, &
         0.67, 0.67, 0.67, 0.68, 0.68, 0.68, 0.68, 0.68,  &
         0.68, 0.68, 0.68/

  data (dleaf(i),i=1,16)/ 0.04, 0.04, 0.04, 0.04, 0.04,&
         0.04, 0.04, 0.04, 0.04, 0.04, 0.04, 0.04, 0.04,&
         0.04, 0.04, 0.04/

  data (c3psn(i),i=1,16)/1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,&
        1.0,1.0,1.0,1.0,1.0,0.0,1.0,1.0/

  data (vcmx25(i),i=1,16)/51.0,43.0,43.0,75.0,69.0,40.0,&
       51.0,51.0,17.0,17.0,33.0,43.0,43.0,24.0,50.0,50.0/

  data (mp(i),i=1,16)/6.0,6.0,6.0,9.0,9.0,9.0,9.0,9.0,&
        9.0,9.0,9.0,9.0,9.0,5.0,9.0,9.0/

  data (qe25(i),i=1,16)/ 0.06, 0.06, 0.06, 0.06, 0.06,&
        0.06, 0.06, 0.06, 0.06, 0.06, 0.06, 0.06, 0.06,&
        0.04, 0.06, 0.06/

  data (rhol(i,1),i=1,16)/ 0.07, 0.07, 0.07, 0.10, 0.10,&
        0.10, 0.10, 0.10, 0.07, 0.10, 0.10, 0.11, 0.11,&
        0.11, 0.11, 0.11/
 
  data (rhol(i,2),i=1,16)/ 0.35, 0.35, 0.35, 0.45, 0.45,&
        0.45, 0.45, 0.45, 0.35, 0.45, 0.45, 0.58, 0.58, &
        0.58, 0.58, 0.58/

  data (rhos(i,1),i=1,16) /0.16, 0.16, 0.16, 0.16, 0.16,&
         0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.36, 0.36,&
         0.36, 0.36, 0.36/  

  data (rhos(i,2),i=1,16)/ 0.39, 0.39, 0.39, 0.39, 0.39,&
        0.39, 0.39, 0.39, 0.39, 0.39, 0.39, 0.58, 0.58, &
        0.58, 0.58, 0.58/

  data (taul(i,1),i=1,16)/ 0.05, 0.05, 0.05, 0.05, 0.05,&
        0.05, 0.05, 0.05, 0.05, 0.05, 0.05, 0.07, 0.07,&
        0.07, 0.07, 0.07/

  data (taul(i,2),i=1,16)/ 0.10, 0.10, 0.10, 0.25, 0.25,&
        0.25, 0.25, 0.25, 0.10, 0.25, 0.25, 0.25, 0.25, &
        0.25, 0.25, 0.25/

  data (taus(i,1),i=1,16)/0.001, 0.001, 0.001, 0.001,&
       0.001,0.001, 0.001, 0.001, 0.001, 0.001, 0.001,&
       0.220, 0.220, 0.220, 0.220, 0.220/
   
  data (taus(i,2),i=1,16)/ 0.001, 0.001, 0.001, 0.001,&
       0.001, 0.001, 0.001, 0.001, 0.001, 0.001, &
       0.001, 0.380, 0.380, 0.380, 0.380, 0.380/

  data (xl(i),i=1,16)/0.01,0.01,0.01,0.10,0.10, 0.01,&
       0.25, 0.25, 0.01, 0.25, 0.25, -0.30, -0.30,&
       -0.30, -0.30, -0.30/

  data (roota_par(i),i=1,16)/ 7.0, 7.0, 7.0, 7.0,&
      7.0, 6.0, 6.0, 6.0, 7.0, 7.0, 7.0, 11.0, &
      11.0, 11.0,  6.0,  6.0/

  data (rootb_par(i),i=1,16)/ 2.0, 2.0, 2.0, &
     1.0, 1.0, 2.0, 2.0, 2.0, 1.5, 1.5, 1.5, &
     2.0, 2.0, 2.0, 3.0, 3.0/



contains

 subroutine var_par
 end subroutine var_par

end module clm_varcon
