module decompMod
!------------------------------------------------------------------------------
!BOP
!
! !MODULE: decompMod
!
! !USES:
  use shr_kind_mod, only : r8 => shr_kind_r8
  use clm_varpar  , only : lsmlon, lsmlat, maxpatch, maxpatch_pft, &
                           npatch_crop, npatch_urban, npatch_glacier
  use clm_varsur  , only : numlon, landmask

  implicit none

  integer, public :: ncells
  integer, public :: nlunits
  integer, public :: ncols
  integer, public :: npfts

  public initDecomp              ! initializes land surface decomposition
                                 ! into clumps and processors
  public get_gcell_info          ! updates gridcell, landunits, columns and
                                 ! pfts counters
  public get_gcell_xyind         ! returns ixy and jxy for each grid cell

  public get_proc_bounds         ! beg and end gridcell, landunit, column,
                                 ! pft indices for this processor

  save 

  private

  type gcell_decomp
     integer :: gsn     ! corresponding cell index in south->north gridcell array
     integer :: ixy     ! cell longitude index
     integer :: jxy     ! cell latitude index
     integer :: li      ! beginning landunit index
     integer :: lf      ! ending landunit index
     integer :: ci      ! beginning column index
     integer :: cf      ! ending column index
     integer :: pi      ! beginning pft index
     integer :: pf      ! ending pft index
  end type gcell_decomp
  type(gcell_decomp), allocatable :: gcelldc(:)

contains
!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: initDecomp
!
! !INTERFACE:
  subroutine initDecomp(wtxy)
!
! !DESCRIPTION:
! This subroutine initializes the land surface decomposition into a clump
! data structure.
!
! !USES:
    use clmtype
!
! !ARGUMENTS:
    implicit none
    real(r8), intent(in) :: wtxy(lsmlon, lsmlat, maxpatch) ! subgrid patch
                                                           ! weights
!
! !LOCAL VARIABLES:
    integer :: ppc                    ! min number of pfts per clump
    integer :: lpc                    ! min number of landunits per clump
    integer :: ppclump                ! min pfts per clump
    integer :: i,j,cid,pid            ! indices
    integer :: gi,li,ci,pi            ! indices
    integer :: gf,lf,cf,pf            ! indices
    integer :: g,l,c,p,n,m            ! indices
    integer :: gdc,gsn                ! indices
    integer :: nzero                  ! first clump with zero gridcells
!    integer :: ncells                 ! total gridcells
!    integer :: nlunits                ! total landunits
!    integer :: ncols                  ! total columns
!    integer :: npfts                  ! total pfts
    integer :: nveg                   ! number of pfts in vegetated landunit
    integer :: numg                   ! total number of gridcells across all
                                      ! processors
    integer :: numl                   ! total number of landunits across all
                                      ! processors
    integer :: numc                   ! total number of columns across all
                                      ! processors
    integer :: nump                   ! total number of pfts across all
                                      ! processors
    logical :: error = .false.        ! temporary for finding full clump
    integer :: ilunits, icols, ipfts  ! temporaries
    integer :: ng                     ! temporaries
    integer :: nl                     ! temporaries
    integer :: nc                     ! temporaries
    integer :: np                     ! temporaries
    integer :: ier                    ! error code
!
! !CALLED FROM:
! subroutine initialize
!
! !REVISION HISTORY:
! 2002.09.11  Forrest Hoffman  Creation.
!
!EOP
!------------------------------------------------------------------------------

    ! Find total global number of grid cells, landunits, columns and pfts

    ncells = 0
    nlunits = 0
    ncols = 0
    npfts = 0
    do j = 1, lsmlat
       do i = 1, numlon(j)
          if (landmask(i,j) == 1) then
             call get_gcell_info (i, j, wtxy, nlunits=ilunits, ncols=icols, npfts=ipfts)
             ncells  = ncells  + 1
             nlunits = nlunits + ilunits
             ncols   = ncols   + icols
             npfts   = npfts   + ipfts
          end if
       end do
    end do

  end subroutine initDecomp
!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: get_gcell_info
!
! !INTERFACE:
   subroutine get_gcell_info (i, j, wtxy, nlunits, ncols, npfts, &
                              nveg, wtveg, ncrop, wtcrop)
!
! !DESCRIPTION:
! Obtain gridcell properties.
!
! !ARGUMENTS:
     implicit none
     integer , intent(in)  :: i                  ! longitude index
     integer , intent(in)  :: j                  ! latitude index
     real(r8), intent(in)  :: wtxy(lsmlon, lsmlat, maxpatch) ! subgrid pft
                                                             ! weights
     integer , optional, intent(out) :: nlunits  ! number of landunits
     integer , optional, intent(out) :: ncols    ! number of columns
     integer , optional, intent(out) :: npfts    ! number of pfts
     integer , optional, intent(out) :: nveg     ! number of vegetated pfts
                                                 ! in naturally vegetated
                                                 ! landunit
     real(r8), optional, intent(out) :: wtveg    ! weight (relative to
                                                 ! gridcell) of naturally
                                                 ! vegetated landunit
     integer , optional, intent(out) :: ncrop    ! number of crop pfts in
                                                 ! crop landunit
     real(r8), optional, intent(out) :: wtcrop   ! weight (relative to
                                                 ! gridcell) of crop landunit
!
! !CALLED FROM:
! subroutines initDecomp
!
! !REVISION HISTORY:
! 2002.09.11  Mariana Vertenstein  Creation.
!
!EOP
!
! !LOCAL VARIABLES:
     integer  :: m       ! loop index
     integer  :: nvegl   ! number of vegetated pfts in naturally vegetated landunit
     real(r8) :: wtvegl  ! weight (relative to gridcell) of vegetated landunit
     integer  :: nvegc   ! number of crop pfts in crop landunit
     real(r8) :: wtvegc  ! weight (relative to gridcell) of crop landunit
     integer  :: ilunits ! number of landunits in gridcell
     integer  :: icols   ! number of columns in gridcell
     integer  :: ipfts   ! number of pfts in gridcell
!------------------------------------------------------------------------------

     ! Initialize pfts, columns and landunits counters for gridcell

     ipfts   = 0
     icols   = 0
     ilunits = 0

     ! Set total number of pfts in gridcell

     do m = 1,maxpatch
        if (wtxy(i,j,m) > 0.0) ipfts = ipfts + 1
     end do

     ! Set naturally vegetated landunit

     nvegl = 0
     wtvegl = 0.0
     do m = 1, maxpatch_pft
        if (wtxy(i,j,m) > 0.0) then
           nvegl = nvegl + 1
           wtvegl = wtvegl + wtxy(i,j,m)
        end if
     end do
     if (nvegl > 0) ilunits = ilunits + 1
#if (defined NOCOMPETE)
     if (nvegl > 0) icols = icols + nvegl ! each pft on vegetated landunit has its own column
#else
     if (nvegl > 0) icols = icols + 1     ! the vegetated landunit has one column
#endif

     ! Set special landunits

     do m = npatch_urban, npatch_glacier
        if (wtxy(i,j,m) > 0.0) ilunits = ilunits + 1
        if (wtxy(i,j,m) > 0.0) icols = icols + 1
     end do

     ! Set crop landunit if appropriate

     nvegc = 0
     wtvegc = 0.0

        do m = npatch_glacier+1, npatch_crop
           if (wtxy(i,j,m) > 0.0) then
              nvegc = nvegc + 1
              wtvegc = wtvegc + wtxy(i,j,m)
           end if
        end do
        if (nvegc > 0) ilunits = ilunits + 1
        if (nvegc > 0) icols = icols + nvegc

     if (present(nlunits)) nlunits = ilunits
     if (present(ncols))   ncols   = icols
     if (present(npfts))   npfts   = ipfts
     if (present(nveg))    nveg    = nvegl
     if (present(wtveg))   wtveg   = wtvegl
     if (present(ncrop))   ncrop   = nvegc
     if (present(wtcrop))  wtcrop  = wtvegc

 end subroutine get_gcell_info

!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: get_proc_bounds
!
! !INTERFACE:
   subroutine get_proc_bounds (begg, endg, begl, endl, begc, endc, &
                               begp, endp)
!
! !ARGUMENTS:
     implicit none
     integer, intent(out) :: begp, endp  ! proc beginning and ending
                                         ! pft indices
     integer, intent(out) :: begc, endc  ! proc beginning and ending
                                         ! column indices
     integer, intent(out) :: begl, endl  ! proc beginning and ending
                                         ! landunit indices
     integer, intent(out) :: begg, endg  ! proc beginning and ending
                                         ! gridcell indices
! !DESCRIPTION:
! Retrieve gridcell, landunit, column, and pft bounds for process.
!
! !REVISION HISTORY:
! 2003.09.12  Mariana Vertenstein  Creation.
!
!EOP
!------------------------------------------------------------------------------

     begp = 1
     endp = npfts
     begc = 1
     endc = ncols
     begl = 1
     endl = nlunits
     begg = 1
     endg = 1

   end subroutine get_proc_bounds

!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: get_gcell_xyind
!
! !INTERFACE:
   subroutine get_gcell_xyind(lbg, ubg, ixy, jxy)
!
! !DESCRIPTION:
! Retrieve x,y indices of a gridcell.
!
! !ARGUMENTS:
     implicit none
     integer, intent(in) :: lbg
     integer, intent(in) :: ubg
     integer, pointer    :: ixy(:)
     integer, pointer    :: jxy(:)
!
! !REVISION HISTORY:
! 2003.09.12  Mariana Vertenstein  Creation.
!
!EOP
!
! !LOCAL VARIABLES:
     integer :: g     ! indices
     integer :: i, j
     integer :: ier                    ! error code
!------------------------------------------------------------------------------

!dir$ concurrent
!cdir nodep
    allocate(gcelldc(ncells), stat=ier)
    g = 0
    do j = 1, lsmlat
       do i = 1, numlon(j)
          g = g + 1
          gcelldc(g)%ixy = i
          gcelldc(g)%jxy = j
       end do
    end do
    do g = lbg,ubg
        ixy(g) = gcelldc(g)%ixy
        jxy(g) = gcelldc(g)%jxy
    end do
    deallocate(gcelldc)

   end subroutine get_gcell_xyind

end module decompMod
