module STATICEcosysdynMOD

#if (!defined DGVM)

!-----------------------------------------------------------------------
!BOP
!
! !MODULE: STATICEcosysDynMod
!
! !DESCRIPTION:
! Static Ecosystem dynamics: phenology, vegetation.
!
! !USES:
  use shr_kind_mod, only: r8 => shr_kind_r8
   use decompMod    , only : get_proc_bounds
!
! !PUBLIC TYPES:
  implicit none
  save
!
! !PUBLIC MEMBER FUNCTIONS:
  public :: EcosystemDyn       ! Ecosystem dynamics: phenology, vegetation
  public :: EcosystemDynini    ! Dynamically allocate memory
  public :: interpMonthlyVeg   ! interpolate monthly vegetation data
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!
! !PRIVATE MEMBER FUNCTIONS:
  private :: readMonthlyVegetation   ! read monthly vegetation data for two months
!
! PRIVATE TYPES:
  integer , private :: InterpMonths1         ! saved month index
  real(r8), private :: timwt(2)              ! time weights for month 1 and month 2
  real(r8), private, allocatable :: mlai1(:) ! lai for interpolation (month 1)
  real(r8), private, allocatable :: mlai2(:) ! lai for interpolation (month 2)
  real(r8), private, allocatable :: msai1(:) ! sai for interpolation (month 1)
  real(r8), private, allocatable :: msai2(:) ! sai for interpolation (month 2)
  real(r8), private, allocatable :: mhvt1(:) ! top vegetation height for interpolation (month 1)
  real(r8), private, allocatable :: mhvt2(:) ! top vegetation height for interpolation (month 2)
  real(r8), private, allocatable :: mhvb1(:) ! bottom vegetation height for interpolation(month 1)
  real(r8), private, allocatable :: mhvb2(:) ! bottom vegetation height for interpolation(month 2)
!-----------------------------------------------------------------------

contains

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: EcosystemDynini
!
! !INTERFACE:
  subroutine EcosystemDynini ()
!
! !DESCRIPTION:
! Dynamically allocate memory and set to signaling NaN.
!
! !USES:
    use nanMod
! !ARGUMENTS:
    implicit none
!
! !REVISION HISTORY:
!
!EOP
!
! LOCAL VARIABLES:
    integer :: ier    ! error code
    integer :: begg
    integer :: endg
    integer :: begl
    integer :: endl
    integer :: begc
    integer :: endc
    integer :: begp
    integer :: endp
!-----------------------------------------------------------------------

    InterpMonths1 = -999  ! saved month index
! begg,begl,begc,begp are all equal to 1
    call get_proc_bounds (begg, endg, begl, endl, begc, endc, begp, endp)

    ier = 0
    if(.not.allocated(mlai1))allocate (mlai1(endp), mlai2(endp), &
              msai1(endp), msai2(endp), &
              mhvt1(endp), mhvt2(endp), &
              mhvb1(endp), mhvb2(endp), stat=ier)

    if (ier /= 0) then
       write (6,*) 'EcosystemDynini allocation error'
       call endrun
    end if

    mlai1(:) = nan
    mlai2(:) = nan
    msai1(:) = nan
    msai2(:) = nan
    mhvt1(:) = nan
    mhvt2(:) = nan
    mhvb1(:) = nan
    mhvb2(:) = nan

  end subroutine EcosystemDynini

!-----------------------------------------------------------------------

!
! !IROUTINE: EcosystemDyn
!
! !INTERFACE:
  subroutine EcosystemDyn(lbp, ubp, num_nolakep, filter_nolakep, doalb)
!
! !DESCRIPTION:
! Ecosystem dynamics: phenology, vegetation
! Calculates leaf areas (tlai, elai),  stem areas (tsai, esai) and
! height (htop).
!
! !USES:
    use clmtype
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: lbp, ubp                    ! pft bounds
    integer, intent(in) :: num_nolakep                 ! number of column non-lake points in pft filter
    integer, intent(in) :: filter_nolakep(ubp-lbp+1)   ! pft filter for non-lake points
    logical, intent(in) :: doalb                       ! true = surface albedo calculation time step
!
! !CALLED FROM:
!
! !REVISION HISTORY:
! Author: Gordon Bonan
! 2/1/02, Peter Thornton: Migrated to new data structure.
!
! !LOCAL VARIABLES:
!
! local pointers to implicit in arguments
!
    integer , pointer :: pcolumn(:)  ! column index associated with each pft
    real(r8), pointer :: snowdp(:)   ! snow height (m)
!
! local pointers to implicit out arguments
!
    real(r8), pointer :: tlai(:)     ! one-sided leaf area index, no burying by snow
    real(r8), pointer :: tsai(:)     ! one-sided stem area index, no burying by snow
    real(r8), pointer :: htop(:)     ! canopy top (m)
    real(r8), pointer :: hbot(:)     ! canopy bottom (m)
    real(r8), pointer :: elai(:)     ! one-sided leaf area index with burying by snow
    real(r8), pointer :: esai(:)     ! one-sided stem area index with burying by snow
    integer , pointer :: frac_veg_nosno_alb(:) ! frac of vegetation not covered by snow [-]
!
!EOP
!
! !OTHER LOCAL VARIABLES:
!
    integer  :: fp,p,c   ! indices
    real(r8) :: ol       ! thickness of canopy layer covered by snow (m)
    real(r8) :: fb       ! fraction of canopy layer covered by snow
!-----------------------------------------------------------------------

    if (doalb) then

       ! Assign local pointers to derived type scalar members (column-level)

       snowdp  => clm3%g%l%c%cps%snowdp

       ! Assign local pointers to derived type scalar members (pftlevel)

       pcolumn => clm3%g%l%c%p%column
       tlai    => clm3%g%l%c%p%pps%tlai
       tsai    => clm3%g%l%c%p%pps%tsai
       elai    => clm3%g%l%c%p%pps%elai
       esai    => clm3%g%l%c%p%pps%esai
       htop    => clm3%g%l%c%p%pps%htop
       hbot    => clm3%g%l%c%p%pps%hbot
       frac_veg_nosno_alb => clm3%g%l%c%p%pps%frac_veg_nosno_alb

!dir$ concurrent
!cdir nodep
       do fp = 1, num_nolakep
          p = filter_nolakep(fp)
          c = pcolumn(p)

          ! need to update elai and esai only every albedo time step so do not
          ! have any inconsistency in lai and sai between SurfaceAlbedo calls (i.e.,
          ! if albedos are not done every time step).
          ! leaf phenology
          ! Set leaf and stem areas based on day of year
          ! Interpolate leaf area index, stem area index, and vegetation heights
          ! between two monthly
          ! The weights below (timwt(1) and timwt(2)) were obtained by a call to
          ! routine InterpMonthlyVeg in subroutine NCARlsm.
          !                 Field   Monthly Values
          !                -------------------------
          ! leaf area index LAI  <- mlai1 and mlai2
          ! leaf area index SAI  <- msai1 and msai2
          ! top height      HTOP <- mhvt1 and mhvt2
          ! bottom height   HBOT <- mhvb1 and mhvb2

          tlai(p) = timwt(1)*mlai1(p) + timwt(2)*mlai2(p)
          tsai(p) = timwt(1)*msai1(p) + timwt(2)*msai2(p)
          htop(p) = timwt(1)*mhvt1(p) + timwt(2)*mhvt2(p)
          hbot(p) = timwt(1)*mhvb1(p) + timwt(2)*mhvb2(p)

          ! adjust lai and sai for burying by snow. if exposed lai and sai
          ! are less than 0.05, set equal to zero to prevent numerical
          ! problems associated with very small lai and sai.

          ol = min( max(snowdp(c)-hbot(p), 0._r8), htop(p)-hbot(p))
          fb = 1. - ol / max(1.e-06, htop(p)-hbot(p))
          elai(p) = max(tlai(p)*fb, 0.0_r8)
          esai(p) = max(tsai(p)*fb, 0.0_r8)
          if (elai(p) < 0.05) elai(p) = 0._r8
          if (esai(p) < 0.05) esai(p) = 0._r8

          ! Fraction of vegetation free of snow

          if ((elai(p) + esai(p)) >= 0.05) then
             frac_veg_nosno_alb(p) = 1
          else
             frac_veg_nosno_alb(p) = 0
          end if

       end do ! end of pft loop

    end if  !end of if-doalb block

  end subroutine EcosystemDyn

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: interpMonthlyVeg
!
! !INTERFACE:
  subroutine interpMonthlyVeg (kmo, kda)
!
! !DESCRIPTION:
! Determine if 2 new months of data are to be read.
!
! !USES:
! !ARGUMENTS:
    implicit none
!
! !REVISION HISTORY:
! Created by Mariana Vertenstein
!
!EOP
!
! LOCAL VARIABLES:
    integer :: kyr         ! year (0, ...) for nstep+1
    integer :: kmo         ! month (1, ..., 12)
    integer :: kda         ! day of month (1, ..., 31)
    integer :: ksec        ! seconds into current date for nstep+1
    real(r8):: dtime       ! land model time step (sec)
    real(r8):: t           ! a fraction: kda/ndaypm
    integer :: it(2)       ! month 1 and month 2 (step 1)
    integer :: months(2)   ! months to be interpolated (1 to 12)
    integer, dimension(12) :: ndaypm= &
         (/31,28,31,30,31,30,31,31,30,31,30,31/) !days per month
!-----------------------------------------------------------------------

    t = (kda-0.5) / ndaypm(kmo)
    it(1) = t + 0.5
    it(2) = it(1) + 1
    months(1) = kmo + it(1) - 1
    months(2) = kmo + it(2) - 1
    if (months(1) <  1) months(1) = 12
    if (months(2) > 12) months(2) = 1
    timwt(1) = (it(1)+0.5) - t
    timwt(2) = 1.-timwt(1)

!    if (InterpMonths1 /= months(1)) then
       call readMonthlyVegetation (kmo, kda, months)
!       InterpMonths1 = months(1)
!    end if

  end subroutine interpMonthlyVeg

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: readMonthlyVegetation
!
! !INTERFACE:
  subroutine readMonthlyVegetation (kmo, kda, months)
!
! !DESCRIPTION:
! Read monthly vegetation data for two consec. months.
!
! !USES:
    use clmtype
    use clm_varpar  , only : lsmlon, lsmlat, maxpatch_pft, maxpatch, npatch_crop, numpft
    use clm_varcon  , only : lai,sai,hvt,hvb

!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: kmo            ! month (1, ..., 12)
    integer, intent(in) :: kda            ! day of month (1, ..., 31)
    integer, intent(in) :: months(2)      ! months to be interpolated (1 to 12)
!
! !REVISION HISTORY:
! Created by Sam Levis
!
!EOP
!
! LOCAL VARIABLES:
    integer :: i,j,k,l,m,p,ivt            ! indices
    integer :: begg
    integer :: endg
    integer :: begl
    integer :: endl
    integer :: begc
    integer :: endc
    integer :: begp
    integer :: endp

    integer :: ier                        ! error code
    real(r8), allocatable :: mlai(:,:,:)  ! lai read from input files
    real(r8), allocatable :: msai(:,:,:)  ! sai read from input files
    real(r8), allocatable :: mhgtt(:,:,:) ! top vegetation height
    real(r8), allocatable :: mhgtb(:,:,:) ! bottom vegetation height
!-----------------------------------------------------------------------

! begg,begl,begc,begp are all equal to 1
      call get_proc_bounds (begg, endg, begl, endl, begc, endc, begp, endp)
      do k=1,2
          do p = begp, endp
             i = clm3%g%l%c%p%ixy(p)
             j = clm3%g%l%c%p%jxy(p)
             m = clm3%g%l%c%p%mxy(p)
             ivt = clm3%g%l%c%p%itype(p)

             ! Assign lai/sai/hgtt/hgtb to the top [maxpatch_pft] pfts
             ! as determined in subroutine surfrd

                if((m <= maxpatch_pft.and.ivt/=0).or.ivt==15.or.ivt==16)then!vegetated pft
                   if (k == 1) then
                      mlai1(p) = lai(ivt,months(k))
                      msai1(p) = sai(ivt,months(k))
                      mhvt1(p) = hvt(ivt)
                      mhvb1(p) = hvb(ivt)
                   else !if (k == 2)
                      mlai2(p) = lai(ivt,months(k))
                      msai2(p) = sai(ivt,months(k))
                      mhvt2(p) = hvt(ivt)
                      mhvb2(p) = hvb(ivt)
                   end if
                else                        ! non-vegetated pft
                   if (k == 1) then
                      mlai1(p) = 0.
                      msai1(p) = 0.
                      mhvt1(p) = 0.
                      mhvb1(p) = 0.
                   else !if (k == 2)
                      mlai2(p) = 0.
                      msai2(p) = 0.
                      mhvt2(p) = 0.
                      mhvb2(p) = 0.
                   end if
                end if

          end do   ! end of loop over pfts

       end do   ! end of loop over months

  end subroutine readMonthlyVegetation

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: EcosystemDyn_dealloc
!
! !INTERFACE:
  subroutine EcosystemDyn_dealloc ()
!
    implicit none
!
!EOP
!-----------------------------------------------------------------------

    if(allocated(mlai1)) deallocate (mlai1)
    if(allocated(mlai2)) deallocate (mlai2)
    if(allocated(msai1)) deallocate (msai1)
    if(allocated(msai2)) deallocate (msai2)
    if(allocated(mhvt1)) deallocate (mhvt1)
    if(allocated(mhvt2)) deallocate (mhvt2)
    if(allocated(mhvb1)) deallocate (mhvb1)
    if(allocated(mhvb2)) deallocate (mhvb2)

  end subroutine EcosystemDyn_dealloc

#endif

end module STATICEcosysDynMod
