      subroutine clmzen(calday, cosz, lbg, ubg, declination)
!Added declination for use in Surface Albedo Mod, Zack Subin 7/16/08
      use shr_kind_mod, only: r8 => shr_kind_r8     
      use clmtype

      implicit none

! ------------------------ code history ---------------------------
! source file:       lsmzen.F
! purpose:           cosine of solar zenith angle
! date last revised: March 1996 - lsm version 1
! author:            Gordon Bonan
! standardized:      J. Truesdale, Feb. 1996
! reviewed:          G. Bonan, Feb. 1996
! -----------------------------------------------------------------

! ------------------------ notes ----------------------------------
! cosine solar zenith angle from:
!    o day (1.x to 365.x), where x=0 (e.g. 213.0) denotes 00:00 at greenwich
!    o latitude,  where SH = - and NH = + 
!    o longitude, where WH = - and EH = +

! the solar declination must match that used in the atmospheric model.
! for ccm2, this code matches the ccm2 cosz to within +/- 0.0001.

! this discrepancy between lsm cosz and atm cosz causes a problem.
! lsm cosz may be <= zero (sun below horizon), in which case albedos
! equal zero, but atm cosz may be > zero (sun above horizon), in which
! case atm model needs albedos. no problem if atm model has sun below 
! horizon but lsm has sun above horizon so long as atm solar fluxes 
! equal zero. a possible solution then is to reset points with sun 
! slightly below horizon to slightly above horizon. 

! in practice this error is not very large. e.g., if albedo error is 
! 0.0001 (atm cosz = 0.0001, lsm cosz = 0) absorbed solar radiation 
! error is incident flux * 0.0001. since incident flux is itself 
! multiplied by atm cosz, incident flux is small. hence, error is small.
! in fact the error is smaller than the difference between atm net solar 
! radiation at surface and lsm net solar radiation at surface, which arises
! due to the different surface radiation parameterizations.

! however, reset points as discussed above just in case atm model may
! blow up if albedos equal zero when atm cosz > 0.
! -----------------------------------------------------------------
    integer :: g
    integer , intent(in) :: lbg, ubg ! gridcell bounds
    real(r8), intent(in) :: calday   ! calendar day at Greenwich (1.00, ..., 365.99)
! ------------------------ input/output variables -----------------
! output
   real(r8), intent(inout) :: cosz(lbg:ubg) ! cosine solar zenith angle for next time step (gridcell level)
   real(r8), intent(inout) :: declination  !in radians
!Added declination output for use in SurfaceAlbedoMod, Zack Subin 7/16/08
! -----------------------------------------------------------------

! ------------------------ local variables ------------------------
      real(r8) :: dayspy              !days per year
      real(r8) :: pi                  !pi
      real(r8) :: theta               !earth orbit seasonal angle in radians
      real(r8) :: delta               !solar declination angle  in radians
      real(r8) :: sind                !sine   of declination
      real(r8) :: cosd                !cosine of declination
      real(r8) :: phi                 !greenwich calendar day + longitude offset
      real(r8) :: loctim              !local time (hour)
      real(r8) :: hrang               !solar hour angle, 24 hour periodicity (radians)

      real(r8), pointer :: lat(:)       ! gridcell latitude (radians):+ = NH
      real(r8), pointer :: lon(:)       ! gridcell longitude (radians):+ = EH
      real(r8) :: lonx(lbg:ubg)
      real(r8) :: latx(lbg:ubg)

      integer mcsec            !current seconds in day (0, ..., 86400)
! -----------------------------------------------------------------
     dayspy = 365.
     pi = 4.*atan(1.)

     lat       => clm3%g%lat
     lon       => clm3%g%lon

     
     do g = lbg, ubg
       latx(g) = lat(g)
       lonx(g) = lon(g)
       if(lonx(g).lt.0.0) then
          lonx(g) = 2.*pi + lonx(g) 
       end if
     end do


! solar declination: match ccm2  

      theta = (2.*pi*calday)/dayspy
      delta = .006918 - .399912*cos(   theta) + .070257*sin(   theta)&
                      - .006758*cos(2.*theta) + .000907*sin(2.*theta)&
                      - .002697*cos(3.*theta) + .001480*sin(3.*theta)
      sind = sin(delta)
      cosd = cos(delta)
!Added for output of declination, Zack Subin 7/16/08
      declination = delta
!!!!!!!
      do g = lbg, ubg

! local time
         mcsec = (calday - int(calday)) * 86400.
         phi = calday + lonx(g)/(2.*pi)
         loctim = (mcsec + (phi-calday)*86400.) / 3600.
         if(loctim.gt.24.0) loctim = loctim - 24.0
 
! hour angle

         hrang = 360./24. * (loctim-12.) * pi/180.

! cosine solar zenith angle. reset points with sun slightly below horizon 
! to slightly above horizon, as discussed in notes.

         cosz(g) = sin(latx(g))*sind + cos(latx(g))*cosd*cos(hrang)
         if (cosz(g) .ge. -0.001 .and. cosz(g).le. 0.) cosz(g) = 0.001
      end do

      return
      end
