!Edited by Zack Subin for coupling, 7/16/08, from the CLM3 coupled version
!to the CLM3.5 version.  New CN, DUST, CASA, dynpft features left out for
!now.


!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: driver
!
! !INTERFACE:
subroutine driver (doalb,ilx,jlx)
!
! !DESCRIPTION:
! This subroutine provides the main CLM driver calling sequence.  Most
! computations occurs over ``clumps'' of gridcells (and associated subgrid
! scale entities) assigned to each MPI process.  Computation is further
! parallelized by looping over clumps on each process using shared memory
! OpenMP or Cray Streaming Directives.
!
! The main CLM driver calling sequence is as follows:
! \begin{verbatim}
! * Communicate with flux coupler [COUP_CSM]
! + interpMonthlyVeg      interpolate monthly vegetation data [!DGVM]
!   + readMonthlyVegetation read vegetation data for two months [!DGVM]
! ==== Begin Loop 1 over clumps ====
!  -> DriverInit          save of variables from previous time step
!  -> Hydrology1          canopy interception and precip on ground
!     -> FracWet          fraction of wet vegetated surface and dry elai
!  -> SurfaceRadiation    surface solar radiation
!  -> Biogeophysics1      leaf temperature and surface fluxes
!  -> BareGroundFluxes    surface fluxes for bare soil or snow-covered
!                         vegetation patches
!     -> MoninObukIni     first-guess Monin-Obukhov length and wind speed
!     -> FrictionVelocity friction velocity and potential temperature and
!                         humidity profiles
!  -> CanopyFluxes        leaf temperature and surface fluxes for vegetated
!                         patches
!     -> QSat             saturated vapor pressure, specific humidity, &
!                         derivatives at leaf surface
!     -> MoninObukIni     first-guess Monin-Obukhov length and wind speed
!     -> FrictionVelocity friction velocity and potential temperature and
!                         humidity profiles
!     -> Stomata          stomatal resistance and photosynthesis for
!                         sunlit leaves
!     -> Stomata          stomatal resistance and photosynthesis for
!                         shaded leaves
!     -> QSat             recalculation of saturated vapor pressure,
!                         specific humidity, & derivatives at leaf surface
!  -> Biogeophysics_Lake  lake temperature and surface fluxes
!   + VOCEmission         compute VOC emission [VOC]
!   + DGVMRespiration     CO2 respriation and plant production [DGVM]
!   + DGVMEcosystemDyn    DGVM ecosystem dynamics: vegetation phenology [!DGVM]
!  -> EcosystemDyn        "static" ecosystem dynamics: vegetation phenology
!                         and soil carbon [!DGVM]
!  -> SurfaceAlbedo       albedos for next time step
!  -> Biogeophysics2      soil/snow & ground temp and update surface fluxes
!  -> pft2col             Average from PFT level to column level
!  ====  End Loop 1 over clumps  ====
! * Average fluxes over time interval and send to flux coupler [COUP_CSM]
!  ==== Begin Loop 2 over clumps ====
!  -> Hydrology2          surface and soil hydrology
!  -> Hydrology_Lake      lake hydrology
!  -> SnowAge             update snow age for surface albedo calcualtion
!  -> BalanceCheck        check for errors in energy and water balances
!  ====  End Loop 2 over clumps  ====
!  -> write_diagnostic    output diagnostic if appropriate
!   + Rtmriverflux        calls RTM river routing model [RTM]
!  -> updateAccFlds       update accumulated fields
!  -> update_hbuf         accumulate history fields for time interval
!  Begin DGVM calculations at end of model year [DGVM]
!    ==== Begin Loop over clumps ====
!     + lpj                 LPJ ecosystem dynamics: reproduction, turnover,
!                           kill, allocation, light, mortality, fire
!     + lpjreset1           reset variables & initialize for next year
!    ====  End Loop over clumps  ====
!  End DGVM calculations at end of model year [DGVM]
!  -> htapes_wrapup       write history tapes if appropriate
!  Begin DGVM calculations at end of model year [DGVM]
!    ==== Begin Loop over clumps ====
!     + lpjreset2           reset variables and patch weights
!    ====  End Loop over clumps  ====
!  End DGVM calculations at end of model year [DGVM]
!  -> restart             write restart file if appropriate
!  -> inicfile            write initial file if appropriate
! \end{verbatim}
! Optional subroutines are denoted by an plus (+) with the associated
! CPP variable in brackets at the end of the line.  Coupler communication
! when coupled with CCSM components is denoted by an asterisk (*).
!
! !USES:
  use shr_kind_mod, only: r8 => shr_kind_r8
  use globals  
  use clmtype
  use decompMod           , only : get_proc_bounds
  use filterMod           , only : filter,filters_dealloc
  use clm_varcon          , only : zlnd
  use DriverInitMod       , only : DriverInit
  use BalanceCheckMod     , only : BalanceCheck, BeginWaterBalance
  use SurfaceRadiationMod , only : SurfaceRadiation
  use Hydrology1Mod       , only : Hydrology1
  use Hydrology2Mod       , only : Hydrology2
!  use HydrologyLakeMod    , only : HydrologyLake
  use Biogeophysics1Mod   , only : Biogeophysics1
  use BareGroundFluxesMod , only : BareGroundFluxes
  use CanopyFluxesMod     , only : CanopyFluxes
  use Biogeophysics2Mod   , only : Biogeophysics2
!  use BiogeophysicsLakeMod, only : BiogeophysicsLake
#ifdef COUPLAKE
  use ShalLakeFluxesMod   , only : ShalLakeFluxes
  use ShalLakeTemperatureMod, only : ShalLakeTemperature
  use ShalLakeHydrologyMod, only : ShalLakeHydrology
#endif
!!!!!!
  use SurfaceAlbedoMod    , only : SurfaceAlbedo, Snowage
  use pft2colMod          , only : pft2col
  use accFldsMod          , only : updateAccFlds
  use accumulMod          , only : accum_dealloc
#if (defined DGVM)
  use DGVMEcosystemDynMod , only : DGVMEcosystemDyn, DGVMRespiration
  use DGVMMod             , only : lpj, lpjreset1, lpjreset2
#else
  use STATICEcosysDynMod  , only : EcosystemDyn,interpMonthlyVeg,EcosystemDyn_dealloc
#endif
#if (defined VOC)
  use VOCEmissionMod      , only : VOCEmission
#endif

#if (defined CN)
  use CNEcosystemDynMod   , only : CNEcosystemDyn
  use CNBalanceCheckMod   , only : BeginCBalance, BeginNBalance, &
                                   CBalanceCheck, NBalanceCheck
  use ndepFileMod         , only : ndepdyn_interp
#endif
#if (defined DUST)
  use DUSTMod             , only : DustDryDep, DustEmission
#endif
#if (defined CASA)
  use CASAPhenologyMod    , only : CASAPhenology
  use CASAMod             , only : Casa
#endif
#if (defined RTM)
  use RtmMod              , only : Rtmriverflux
#endif

!
! !ARGUMENTS:
  implicit none
  logical , intent(in) :: doalb  !true if time for surface albedo
                                 !calculation
!
! !REVISION HISTORY:
! 2002.10.01  Mariana Vertenstein latest update to new data structures
!
!EOP
!
! !LOCAL VARIABLES:
  integer  :: ilx,jlx
  integer  :: c         ! indices
  integer  :: ncdate        ! current date
  integer  :: kyr           ! thousand years, equals 2 at end of first year
  integer  :: begp, endp    ! clump beginning and ending pft indices
  integer  :: begc, endc    ! clump beginning and ending column indices
  integer  :: begl, endl    ! clump beginning and ending landunit indices
  integer  :: begg, endg    ! clump beginning and ending gridcell indices
  type(column_type)  , pointer :: cptr    ! pointer to column derived subtype
!  logical, external :: do_restwrite ! determine if time to write restart

!-----------------------------------------------------------------------

  ! Set pointers into derived type

  cptr => clm3%g%l%c

  ! ============================================================================
  ! Calendar information for next time step
  ! o caldayp1 = calendar day (1.00 -> 365.99) for cosine solar zenith angle
  !   calday is based on Greenwich time
  ! o get_curr_calday in the cam time manager know about perpetual mode
  !   and perpetual model is only used within cam
  ! ============================================================================

#if (!defined DGVM)
  ! ============================================================================
  ! Determine weights for time interpolation of monthly vegetation data.
  ! This also determines whether it is time to read new monthly vegetation and
  ! obtain updated leaf area index [mlai1,mlai2], stem area index [msai1,msai2],
  ! vegetation top [mhvt1,mhvt2] and vegetation bottom [mhvb1,mhvb2]. The
  ! weights obtained here are used in subroutine ecosystemdyn to obtain time
  ! interpolated values.
  ! ============================================================================

  if (doalb) call interpMonthlyVeg (monp1,dayp1)
#endif

  ! ============================================================================
  ! Loop1
  ! ============================================================================

     ! ============================================================================
     ! Determine clump boundaries
     ! ============================================================================

     call get_proc_bounds(begg, endg, begl, endl, begc, endc, begp, endp)

!Added for 3.5, Zack Subin 7/16/08
     call CLMDebug('BeginWaterBalance')
     call BeginWaterBalance(begc, endc, begp, endp, &
          filter%num_nolakec, filter%nolakec, &
          filter%num_lakec, filter%lakec, ilx, jlx)

#if (defined CN)
!Fill in later
#endif

!Leaving out pftdyn_wbal_init and canopy water loss

!!!!!!!!


     ! ============================================================================
     ! Initialize variables from previous time step and
     ! Determine canopy interception and precipitation onto ground surface.
     ! Determine the fraction of foliage covered by water and the fraction
     ! of foliage that is dry and transpiring. Initialize snow layer if the
     ! snow accumulation exceeds 10 mm.
     ! ============================================================================

     call CLMDebug('DriverInit')
     call DriverInit(begc, endc, begp, endp, &
          filter%num_nolakec, filter%nolakec, &
          filter%num_lakec, filter%lakec, ilx, jlx)

     ! ============================================================================
     ! Hydrology1
     ! ============================================================================

     call CLMDebug('Hydrology1')
     call Hydrology1(begc, endc, begp, endp, &
                     filter%num_nolakec, filter%nolakec, &
                     filter%num_nolakep, filter%nolakep)

     ! ============================================================================
     ! Surface Radiation
     ! ============================================================================

     call CLMDebug('SurfaceRadiation')
     call SurfaceRadiation(begp, endp)

     ! ============================================================================
     ! Determine leaf temperature and surface fluxes based on ground
     ! temperature from previous time step.
     ! ============================================================================

     call CLMDebug('Biogeophysics1')
     call Biogeophysics1(begg, endg, begc, endc, begp, endp, &
                         filter%num_nolakec, filter%nolakec, &
                         filter%num_nolakep, filter%nolakep)

     ! ============================================================================
     ! Determine bare soil or snow-covered vegetation surface temperature and fluxes
     ! Calculate Ground fluxes (frac_veg_nosno is either 1 or 0)
     ! ============================================================================

     call CLMDebug('BareGroundFluxes')
     call BareGroundFluxes(begp, endp, &
                           filter%num_nolakep, filter%nolakep)

     ! ============================================================================
     ! Determine non snow-covered vegetation surface temperature and fluxes
     ! Calculate canopy temperature, latent and sensible fluxes from the canopy,
     ! and leaf water change by evapotranspiration
     ! ============================================================================

     call CLMDebug('CanopyFluxes')
     call CanopyFluxes(begg, endg, begc, endc, begp, endp, &
                       filter%num_nolakep, filter%nolakep)

     ! ============================================================================
     ! Determine lake temperature and surface fluxes
     ! ============================================================================

     call CLMDebug('BiogeophysicsLake')
!     call BiogeophysicsLake(begc, endc, begp, endp, &
!                            filter%num_lakec, filter%lakec, &
!                            filter%num_lakep, filter%lakep)
#ifdef COUPLAKE
     call ShalLakeFluxes(begc, endc, begp, endp, &
                         filter%num_lakec, filter%lakec, &
                         filter%num_lakep, filter%lakep)
     call ShalLakeTemperature(begc, endc, begp, endp, &
                              filter%num_lakec, filter%lakec, &
                              filter%num_lakep, filter%lakep)
#endif

!Added for CLM3.5, Zack Subin, 7/16/08
#if (defined DUST)
     ! Dust mobilization (C. Zender's modified codes)
     call DustEmission(begp, endp, begc, endc, begl, endl, &
                       filter%num_nolakep, filter%nolakep)

     ! Dust dry deposition (C. Zender's modified codes)
     call DustDryDep(begp, endp)
#endif
!!!!!!!!!!!!!!

     ! ============================================================================
     ! Determine VOC and DGVM Respiration if appropriate
     ! ============================================================================

#if (defined VOC)
     ! VOC emission (A. Guenther's model)
     call VOCEmission(begp, endp, &
                      filter%num_nolakep, filter%nolakep)
#endif


     ! ============================================================================
     ! Ecosystem dynamics: phenology, vegetation, soil carbon, snow fraction
     ! ============================================================================

#if (defined DGVM)
     ! Surface biogeochemical fluxes: co2 respiration and plant production
     call DGVMRespiration(begc, endc, begp, endp, &
                          filter%num_nolakec, filter%nolakec, &
                          filter%num_nolakep, filter%nolakep)

     call DGVMEcosystemDyn(begp, endp, &
                       filter%num_nolakep, filter%nolakep, &
                       doalb, endofyr=.false.)
!#elif call CNEcosystemDyn
#else
     call CLMDebug('EcosystemDyn')
     call EcosystemDyn(begp, endp, &
                       filter%num_nolakep, filter%nolakep, &
                       doalb)
#endif

     ! ============================================================================
     ! Determine albedos for next time step
     ! ============================================================================

     if (doalb) then
        call CLMDebug('SurfaceAlbedo')
        call SurfaceAlbedo(begg, endg, begc, endc, begp, endp, caldayp1)
     end if

     ! ============================================================================
     ! Determine soil/snow temperatures including ground temperature and
     ! update surface fluxes for new ground temperature.
     ! ============================================================================

     call CLMDebug('Biogeophysics2')
     call Biogeophysics2(begc, endc, begp, endp, &
                         filter%num_nolakec, filter%nolakec, &
                         filter%num_nolakep, filter%nolakep)

     ! ============================================================================
     ! Perform averaging from PFT level to column level
     ! ============================================================================

     call CLMDebug('pft2col')
     call pft2col(begc, endc, filter%num_nolakec, filter%nolakec)

  ! ============================================================================
  ! Loop2
  ! ============================================================================

     ! ============================================================================
     ! Determine clump boundaries
     ! ============================================================================

     call get_proc_bounds(begg, endg, begl, endl, begc, endc, begp, endp)

     ! ============================================================================
     ! Vertical (column) soil and surface hydrology
     ! ============================================================================

     call CLMDebug('Hydrology2')
     call Hydrology2(begc, endc, ilx ,jlx, &
!New in 3.5
                     begp, endp, &
                     filter%num_nolakec, filter%nolakec, &
                     filter%num_soilc, filter%soilc, &
                     filter%num_snowc, filter%snowc, &
                     filter%num_nosnowc, filter%nosnowc)

     ! ============================================================================
     ! Lake hydrology
     ! ============================================================================

     call CLMDebug('HydrologyLake')
!     call HydrologyLake(begp, endp, &
!                        filter%num_lakep, filter%lakep)
#ifdef COUPLAKE
     call ShalLakeHydrology(begc, endc, begp, endp, filter%num_lakec, filter%lakec, &
                            filter%num_lakep, filter%lakep)
#endif

     ! ============================================================================
     ! Update Snow Age (needed for surface albedo calculation
     ! ============================================================================

     call CLMDebug('SnowAge')
     call SnowAge(begc, endc)

     ! ============================================================================
     ! ! Fraction of soil covered by snow (Z.-L. Yang U. Texas)
     ! ============================================================================

     do c = begc,endc
        cptr%cps%frac_sno(c) = cptr%cps%snowdp(c) / (10.*zlnd + cptr%cps%snowdp(c))
     end do

!Added for CLM3.5
     ! ============================================================================
     ! Ecosystem dynamics: Uses CN, DGVM, or static parameterizations
     ! ============================================================================

#if (defined CASA)
     call CASAPhenology(begp, endp, filter%num_soilp, filter%soilp)
     call Casa(begp, endp, filter%num_soilp, filter%soilp)
#endif

!!!!!!!!!



     ! ============================================================================
     ! Check the energy and water balance
     ! ============================================================================

     call CLMDebug('BalanceCheck')
     call BalanceCheck(begp, endp, begc, endc ,ilx, jlx)
!#if (defined CN) call CBalanceCheck, NBalanceCheck

     ! ============================================================================
     ! Update accumulators
     ! ============================================================================

     call CLMDebug('updateAccFlds')
     call updateAccFlds()

#if (defined DGVM)
  ! ============================================================================
  ! Call DGVM (Dynamic Global Vegetation Model)
  ! LPJ is called at last time step of year. Then reset vegetation distribution
  ! and some counters for the next year.
  ! NOTE: monp1, dayp1, and secp1 correspond to nstep+1
  ! NOTE: lpjreset1 must be called after update_accum and update_hbuf
  ! in order to have the correct values of the accumulated variables
  ! NOTE: lpjreset2 is called after htape_wrapup call in order to use the old
  ! old weights in December's history calculations.
  ! ============================================================================

  if (monp1==1 .and. dayp1==1 .and. secp1==dtime .and. nstep>0)  then

     ! Get date info.  kyr is used in lpj().  At end of first year, kyr = 2.
     ncdate = year*10000 + month*100 + day
     kyr = ncdate/10000 -  nbdate/10000 + 1

     call get_proc_bounds(begg, endg, begl, endl, begc, endc, begp, endp)
     call lpj(begg, endg, begp, endp, filter%num_natvegp, filter%natvegp, kyr)
     call lpjreset1(begg, endg, begc, endc, begp, endp, filter%num_nolakep, filter%nolakep, &
                     caldayp1)
  end if
#endif

#if (defined DGVM)
  ! ============================================================================
  ! Finish DGVM calculation
  ! ============================================================================

  if (monp1==1 .and. dayp1==1 .and. secp1==dtime .and. nstep>0)  then
        call get_proc_bounds(begg, endg, begl, endl, begc, endc, begp, endp)
        call lpjreset2(begg, endg, begl, endl, begc, endc, begp, endp)
  end if
#endif

  call accum_dealloc
  call filters_dealloc
#if (!defined DGVM)
  call EcosystemDyn_dealloc
#endif

  call CLMDebug('Leaving Driver')

end subroutine driver
