
































































module module_fr_sfire_driver

use module_model_constants, only: cp,xlv,reradius,pi2
use module_fr_sfire_model
use module_fr_sfire_phys
use module_fr_sfire_atm
use module_fr_sfire_util

private
public :: sfire_driver_em_init, sfire_driver_em_step


contains

subroutine sfire_driver_em_init (grid , config_flags               & 
            ,ids,ide, kds,kde, jds,jde                              &
            ,ims,ime, kms,kme, jms,jme                              &
            ,ips,ipe, kps,kpe, jps,jpe)

    

    USE module_domain
    USE module_configure
    implicit none

    TYPE(domain) , TARGET          :: grid   
    TYPE (grid_config_rec_type) , INTENT(IN)          :: config_flags
    integer, intent(in):: &
             ids,ide, kds,kde, jds,jde                              &
            ,ims,ime, kms,kme, jms,jme                              &
            ,ips,ipe, kps,kpe, jps,jpe

    
    integer :: &  
             ifds,ifde, kfds,kfde, jfds,jfde,                              &
             ifms,ifme, kfms,kfme, jfms,jfme,                              &
             ifps,ifpe, kfps,kfpe, jfps,jfpe                              
    

    real,dimension(1,1,1)::rho,z_at_w,dz8w

    call message('sfire_driver_em_init: SFIRE initialization start')

    
    CALL get_ijk_from_subgrid (  grid ,                   &
                            ifds,ifde, jfds,jfde,kfds,kfde,                        &
                            ifms,ifme, jfms,jfme,kfms,kfme,                        &
                            ifps,ifpe, jfps,jfpe,kfps,kfpe) 

    call sfire_driver_em ( grid , config_flags               & 
            ,1,2,0                        & 
            ,ids,ide, kds,kde, jds,jde                              &
            ,ims,ime, kms,kme, jms,jme                              &
            ,ips,ipe, kps,kpe, jps,jpe                              &
            ,ifds,ifde, jfds,jfde                                   &
            ,ifms,ifme, jfms,jfme                                   &
            ,ifps,ifpe, jfps,jfpe                                   &
            ,rho,z_at_w,dz8w ) 

    call message('sfire_driver_em_init: SFIRE initialization complete')

end subroutine sfire_driver_em_init





subroutine sfire_driver_em_step (grid , config_flags               & 
            ,ids,ide, kds,kde, jds,jde                              &
            ,ims,ime, kms,kme, jms,jme                              &
            ,ips,ipe, kps,kpe, jps,jpe                              &
            ,rho,z_at_w,dz8w ) 

    

    USE module_domain
    USE module_configure
    implicit none

    TYPE(domain) , TARGET          :: grid   
    TYPE (grid_config_rec_type) , INTENT(IN)          :: config_flags
    integer, intent(in):: &
             ids,ide, kds,kde, jds,jde                              &
            ,ims,ime, kms,kme, jms,jme                              &
            ,ips,ipe, kps,kpe, jps,jpe

    
    integer :: &  
             ifds,ifde, kfds,kfde, jfds,jfde,                              &
             ifms,ifme, kfms,kfme, jfms,jfme,                              &
             ifps,ifpe, kfps,kfpe, jfps,jfpe                              
    

    real,dimension(ims:ime, kms:kme, jms:jme)::rho,z_at_w,dz8w

    call message('sfire_driver_em_step: SFIRE step start')

    
    CALL get_ijk_from_subgrid (  grid ,                   &
                            ifds,ifde, jfds,jfde,kfds,kfde,                        &
                            ifms,ifme, jfms,jfme,kfms,kfme,                        &
                            ifps,ifpe, jfps,jfpe,kfps,kfpe) 

    call sfire_driver_em ( grid , config_flags               & 
            ,3,6,fire_test_steps                                &
            ,ids,ide, kds,kde, jds,jde                              &
            ,ims,ime, kms,kme, jms,jme                              &
            ,ips,ipe, kps,kpe, jps,jpe                              &
            ,ifds,ifde, jfds,jfde                                   &
            ,ifms,ifme, jfms,jfme                                   &
            ,ifps,ifpe, jfps,jfpe                                   &
            ,rho,z_at_w,dz8w ) 

    call message('sfire_driver_em_step: SFIRE step complete')

end subroutine sfire_driver_em_step





subroutine sfire_driver_em ( grid , config_flags                    & 
            ,fire_ifun_start,fire_ifun_end,tsteps                   &
            ,ids,ide, kds,kde, jds,jde                              &
            ,ims,ime, kms,kme, jms,jme                              &
            ,ips,ipe, kps,kpe, jps,jpe                              &
            ,ifds,ifde, jfds,jfde                                   &
            ,ifms,ifme, jfms,jfme                                   &
            ,ifps,ifpe, jfps,jfpe                                   &
            ,rho,z_at_w,dz8w ) 




    USE module_domain
    USE module_configure
    USE module_driver_constants
    USE module_machine
    USE module_tiles







    implicit none

    TYPE(domain) , TARGET :: grid                             
    TYPE (grid_config_rec_type) , INTENT(IN)  :: config_flags 
    integer, intent(in)::     fire_ifun_start,fire_ifun_end,tsteps 
    integer, intent(in):: &
             ids,ide, kds,kde, jds,jde,                              &
             ims,ime, kms,kme, jms,jme,                              &
             ips,ipe, kps,kpe, jps,jpe,                              &
             ifds,ifde, jfds,jfde,                                   &
             ifms,ifme, jfms,jfme,                                   &
             ifps,ifpe, jfps,jfpe 
    real,intent(in),dimension(ims:ime, kms:kme, jms:jme)::rho,  &
             z_at_w,dz8w                 


    INTEGER:: fire_num_ignitions
    integer, parameter::fire_max_ignitions=5
    REAL, DIMENSION(fire_max_ignitions)::  fire_ignition_start_x, &
        fire_ignition_start_y, &
        fire_ignition_end_x, &
        fire_ignition_end_y, &
        fire_ignition_time, &
        fire_ignition_radius
    integer::fire_ifun,ir,jr,fire_ignition_longlat,istep,itimestep
    logical::need_lfn_update,restart
    
    
    real, dimension(ifms:ifme, jfms:jfme)::lfn_out  
    real::lat_ctr,lon_ctr
    real:: corner_ll,corner_ul,corner_ur,corner_lr
    character(len=128)msg
    real:: unit_fxlong ,unit_fxlat




    call print_id


    
    call fire_ignition_convert (config_flags,fire_max_ignitions,fire_ignition_longlat, &
        fire_ignition_start_x,fire_ignition_start_y,fire_ignition_end_x,fire_ignition_end_y, &
        fire_ignition_radius,fire_ignition_time,fire_num_ignitions)
    
    call set_flags(config_flags)

    if(fire_ignition_longlat .eq.0)then
       
       
       unit_fxlong=1.  
       unit_fxlat=1.
       
    else
       
       lat_ctr=config_flags%cen_lat
       lon_ctr=config_flags%cen_lon
       
       unit_fxlat=pi2/(360.*reradius)  
       unit_fxlong=cos(lat_ctr*pi2/360.)*unit_fxlat  
    endif

    
    ir=grid%sr_x 
    jr=grid%sr_y
    itimestep=grid%itimestep
    restart=config_flags%restart

    

    write(msg,'(a,i1,a,i1,a,i4)') &
       'sfire_driver_em: ifun from ',fire_ifun_start,' to ',fire_ifun_end,' test steps',tsteps
    call message(msg)

    do istep=0,tsteps 
      itimestep = grid%itimestep + istep 

      need_lfn_update=.false.
      do fire_ifun=fire_ifun_start,fire_ifun_end

        
        
        
        
        
        

        
        call sfire_driver_phys ( &
            fire_ifun,need_lfn_update,                  &
            ids,ide-1, kds,kde, jds,jde-1,                          &
            ims,ime, kms,kme, jms,jme,                          &
            ips,min(ipe,ide-1), kps,kpe, jps,min(jpe,jde-1),                          & 
            ifds,ifde-ir, jfds,jfde-jr,                    &
            ifms,ifme, jfms,jfme,                    &
            ifps,min(ifpe,ifde-ir), jfps,min(jfpe,jfde-jr),      &
            ir,jr,                                      & 
            grid%num_tiles,                             & 
            grid%i_start,min(grid%i_end,ide-1),                    &
            grid%j_start,min(grid%j_end,jde-1),                    &                 
            itimestep,restart,config_flags%fire_fuel_read,config_flags%fire_fuel_cat, &  
            grid%dt,grid%dx,grid%dy,                    &
            grid%u_frame,grid%v_frame,                  &
            unit_fxlong,unit_fxlat,                           & 
            config_flags%fire_ext_grnd,config_flags%fire_ext_crwn,config_flags%fire_crwn_hgt, &
            fire_num_ignitions,                                & 
            fire_ignition_longlat,      &
            fire_ignition_start_x,fire_ignition_start_y, & 
            fire_ignition_end_x,fire_ignition_end_y,     &
            fire_ignition_radius,fire_ignition_time,     &
            grid%u_2,grid%v_2,grid%mut,rho,grid%ht,      & 
            z_at_w,dz8w,                                  &
            grid%xlong,grid%xlat,                         & 
            grid%lfn,grid%tign_g,grid%fuel_frac,          & 
            grid%fire_area,                               & 
            grid%uf,grid%vf,                              & 
            lfn_out,                                      & 
            grid%rthfrten,grid%rqvfrten,                & 
            grid%grnhfx,grid%grnqfx,grid%canhfx,grid%canqfx, & 
            grid%fgrnhfx,grid%fgrnqfx,grid%fcanhfx,grid%fcanqfx, & 
            grid%ros,                                   & 
            grid%fxlong,grid%fxlat,                           &       
            grid%nfuel_cat,                               & 
            grid%fuel_time,grid%zsf,                      & 
            grid%bbb,grid%betafl,grid%phiwc,grid%r_0,grid%fgip,grid%ischap&
        )


                

      enddo
    enddo
    if(tsteps>0)call crash('sfire_driver_em: test run of uncoupled fire model completed')

end subroutine sfire_driver_em






subroutine sfire_driver_phys (ifun,need_lfn_update,    &
    ids,ide, kds,kde, jds,jde,                    & 
    ims,ime, kms,kme, jms,jme,                    &
    ips,ipe, kps,kpe, jps,jpe,                    &
    ifds, ifde, jfds, jfde,                       & 
    ifms, ifme, jfms, jfme,                       &
    ifps, ifpe, jfps, jfpe,                       & 
    ir,jr,                                        & 
    num_tiles,i_start,i_end,j_start,j_end,        & 
    itimestep,restart,ifuelread,nfuel_cat0,dt,dx,dy,      & 
    u_frame,v_frame,                              &
    unit_fxlong,unit_fxlat,                       & 
    fire_ext_grnd,fire_ext_crwn,fire_crwn_hgt,    &
    num_ignitions,                                & 
    ignition_longlat,                             &
    ignition_start_x,ignition_start_y,            & 
    ignition_end_x,ignition_end_y,                &
    ignition_radius,                              &
    ignition_time,                                &
    u,v,mu,rho,zs,                                & 
    z_at_w,dz8w,                                  &
    xlong,xlat,                                   &
    lfn,tign,fuel_frac,                           & 
    fire_area,                                    & 
    uf,vf,lfn_out,                                & 
    rthfrten,rqvfrten,                            & 
    grnhfx,grnqfx,canhfx,canqfx,                  & 
    fgrnhfx,fgrnqfx,fcanhfx,fcanqfx,              & 
    ros,                                          &
    fxlong,fxlat,                                 & 
    nfuel_cat,                                    & 
    fuel_time,zsf,                                & 
    bbb,betafl,phiwc,r_0,fgip,ischap&
    )
USE module_dm, only:wrf_dm_maxval

implicit none



integer, intent(in)::ifun,                        &
    ids,ide, kds,kde, jds,jde,                    & 
    ims,ime, kms,kme, jms,jme,                    & 
    ips,ipe, kps,kpe, jps,jpe,                    & 
    ifds, ifde, jfds, jfde,                       & 
    ifms, ifme, jfms, jfme,                       & 
    ifps, ifpe, jfps, jfpe,                       & 
    ir,jr,                                        & 
    itimestep,                                    & 
    ifuelread,                                    & 
                                                       
                                                       
                                                       
                                                       
    nfuel_cat0,                                   & 
    num_tiles                                       

logical, intent(in)::restart
    

logical, intent(out)::need_lfn_update

integer,dimension(num_tiles),intent(in) :: i_start,i_end,j_start,j_end  

real, intent(in):: &
    dt,                                           & 
    dx,dy,                                        & 
    u_frame,v_frame,                              & 
    unit_fxlong,unit_fxlat,                       & 
    fire_crwn_hgt,                                & 
    fire_ext_grnd,                                & 
    fire_ext_crwn                                   


integer, intent(in):: num_ignitions                 
real, dimension(num_ignitions), intent(in):: &   
    ignition_start_x,ignition_start_y, &
    ignition_end_x,ignition_end_y,ignition_radius, & 
    ignition_time                           
integer, intent(in):: ignition_longlat       

real,intent(in),dimension(ims:ime,kms:kme,jms:jme)::u,v 
real,intent(in),dimension(ims:ime,jms:jme)::mu          
real,intent(in),dimension(ims:ime, jms:jme)::  zs 
real,intent(in),dimension(ims:ime,kms:kme,jms:jme)::rho, &  
                z_at_w,dz8w                         

real, dimension(ims:ime, jms:jme), intent(inout)::xlong, xlat 
    
real, intent(inout), dimension(ifms:ifme,jfms:jfme):: &
    nfuel_cat                                       

real, intent(inout), dimension(ifms:ifme, jfms:jfme)::     &
    lfn,tign,fuel_frac,                        &     
    uf,vf,lfn_out                                    

real, intent(out), dimension(ifms:ifme, jfms:jfme)::  &
    fire_area                                        

real, intent(out), dimension(ims:ime, kms:kme, jms:jme):: &
    rthfrten,rqvfrten                              

real, intent(out), dimension(ims:ime, jms:jme):: &  
    grnhfx,                                      &  
    grnqfx,                                      &  
    canhfx,                                      &  
    canqfx                                         

real, intent(out), dimension(ifms:ifme, jfms:jfme):: &  
    fgrnhfx,                                      &  
    fgrnqfx,                                      &  
    fcanhfx,                                      &  
    fcanqfx,                                      &   
    ros                                             



real, dimension(ifms:ifme, jfms:jfme), intent(inout)::fxlong,fxlat 

real, intent(inout), dimension(ifms:ifme, jfms:jfme):: &
    fuel_time,zsf,                               &
    bbb,betafl,phiwc,r_0,fgip
integer, intent(inout), dimension(ifms:ifme, jfms:jfme):: ischap
    

real :: dxf,dyf,time_start,latm
integer :: its,ite,jts,jte,kts,kte, &            
    ij,i,j,k,id,pid,kpe1, &
    ifts,ifte,jfts,jfte                          
character(len=128)::msg
character(len=3)::kk
integer::ignitions_done_tile(num_tiles),ignited_tile(num_ignitions,num_tiles)
integer::ignitions_done,ignited_patch(num_ignitions),idex,jdex





time_start = itimestep * dt


dxf=dx/ir
dyf=dy/jr



write(msg,'(a,2f15.6)')'atmosphere mesh step:',dx,dy
call message(msg)
write(msg,'(a,2f15.6)')'fire mesh step:      ',dxf,dyf
call message(msg)
write(msg,7001)'atm domain      ','ids',ids,ide,jds,jde
call message(msg)                    
write(msg,7001)'atm memory      ','ims',ims,ime,jms,jme
call message(msg)                    
write(msg,7001)'atm patch       ','ips',ips,ipe,jps,jpe
call message(msg)                    
write(msg,7001)'fire domain     ','ifds',ifds,ifde,jfds,jfde
call message(msg)                    
write(msg,7001)'fire memory     ','ifms',ifms,ifme,jfms,jfme
call message(msg)                    
write(msg,7001)'fire patch      ','ifps',ifps,ifpe,jfps,jfpe
call message(msg)                    


call check_fmesh(ids,ide,ifds,ifde,ir,'id')           
call check_fmesh(jds,jde,jfds,jfde,jr,'jd')
call check_fmesh(ips,ipe,ifps,ifpe,ir,'ip')
call check_fmesh(jps,jpe,jfps,jfpe,jr,'jp')
call check_mesh_2dim(ips,ipe,jps,jpe,ims,ime,jms,jme)        
call check_mesh_2dim(ifps,ifpe,jfps,jfpe,ifms,ifme,jfms,jfme) 
call check_mesh_2dim(ips,ipe,jps,jpe,ids,ide,jds,jde)        
call check_mesh_2dim(ifps,ifpe,jfps,jfpe,ifds,ifde,jfds,jfde) 

!$OMP SINGLE

if(ifun.eq.1) then
   call init_fuel_cats  
endif
!$OMP END SINGLE


pid=0
if(itimestep.le.10.or.mod(itimestep,10).eq.0)pid=itimestep 




kts=kps
kte=kpe


!$OMP PARALLEL DO PRIVATE(ij,its,ite,jts,jte,ifts,ifte,jfts,jfte,msg,id) &
!$OMP SCHEDULE(STATIC)
do ij=1,num_tiles

    id=0  
    if(itimestep.le.10.or.mod(itimestep,10).eq.0)id=itimestep+ij*10000

    ignitions_done_tile(ij)=0

    
    its = i_start(ij)  
    ite = i_end(ij)    
    jts = j_start(ij)  
    jte = j_end(ij)    
    ifts= (its-ids)*ir+ifds       
    ifte= (ite-ids+1)*ir+ifds-1   
    jfts= (jts-jds)*jr+jfds       
    jfte= (jte-jds+1)*jr+jfds-1   
        
    write(msg,*)'tile=',ij,' id=',id,' ifun=',ifun
    call message(msg)
    write(msg,7001)'atm tile   ','its',its,ite,jts,jte
    call message(msg)                   
    write(msg,7001)'fire tile  ','ifts',ifts,ifte,jfts,jfte
    call message(msg)                    

    
    call check_mesh_2dim(its,ite,jts,jte,ips,ipe,jps,jpe)                 
    call check_mesh_2dim(ifts,ifte,jfts,jfte,ifps,ifpe,jfps,jfpe)         
    call check_mesh_2dim(ifts-2,ifte+2,jfts-2,jfte+2,ifms,ifme,jfms,jfme)


    write(msg,'(a,i6,a,2(f15.6,a))')'time step',itimestep,' at',time_start,' duration',dt,'s'
    call message(msg)
    7001 format(a,' dimensions ',a4,':',i6,' to ',i6,' by ',i6,' to ',i6)
    write(msg,'(a,2i9)')'refinement ratio:',ir,jr

    if(ifun.eq.1)then   

      if(restart)then
          
          call message('restart - topo initialization skipped')

      else

        call print_2d_stats(ips,ipe,jps,jpe,ims,ime,jms,jme,zs,'driver:zs')
    
        
        
        if(fire_topo_from_atm.eq.1)then
            call interpolate_z2fire(id,                 & 
                ids,ide,  jds,jde,                    & 
                ims,ime,  jms,jme,                    &
                ips,ipe,jps,jpe,                              &
                its,ite,jts,jte,                              &
                ifds, ifde, jfds, jfde,                       & 
                ifms, ifme, jfms, jfme,                       &
                ifts,ifte,jfts,jfte,                          &
                ir,jr,                                        & 
                zs,                                       & 
                zsf)                                      
        else
           write(msg,'(a,i3,a)')'fire_topo_from_atm=',fire_topo_from_atm,' assuming ZSF set, interpolation skipped'
        endif

        if(ignition_longlat .eq.0)then
            
            
            call set_ideal_coord( dxf,dyf, &
                ifds,ifde,jfds,jfde,  &
                ifms,ifme,jfms,jfme,  &
                ifts,ifte,jfts,jfte,  &
                fxlong,fxlat          )
        else
            
            

         call write_array_m(its,ite,jts,jte,ims,ime,jms,jme,xlat,'xlat',id)
         call write_array_m(its,ite,jts,jte,ims,ime,jms,jme,xlong,'xlong',id)
        call interpolate_z2fire(id,                 & 
            ids,ide,  jds,jde,                    & 
            ims,ime,  jms,jme,                    &
            ips,ipe,jps,jpe,                              &
            its,ite,jts,jte,                              &
            ifds, ifde, jfds, jfde,                       & 
            ifms, ifme, jfms, jfme,                       &
            ifts,ifte,jfts,jfte,                          &
            ir,jr,                                        & 
            xlat,                                       & 
            fxlat)                                      

        call interpolate_z2fire(id,                 & 
            ids,ide,  jds,jde,                    & 
            ims,ime,  jms,jme,                    &
            ips,ipe,jps,jpe,                              &
            its,ite,jts,jte,                              &
            ifds, ifde, jfds, jfde,                       & 
            ifms, ifme, jfms, jfme,                       &
            ifts,ifte,jfts,jfte,                          &
            ir,jr,                                        & 
            xlong,                                       & 
            fxlong)                                      

        endif

     endif

    elseif(ifun.eq.2)then  
               
        
        call print_2d_stats(ifts,ifte,jfts,jfte,ifms,ifme,jfms,jfme,zsf,'driver_phys:zsf')        

    elseif(ifun.eq.3)then  
    
        call interpolate_atm2fire(id,                     & 
            ids,ide, kds,kde, jds,jde,                    & 
            ims,ime, kms,kme, jms,jme,                    &
            ips,ipe,jps,jpe,                              &
            its,ite,jts,jte,                              &                    
            ifds, ifde, jfds, jfde,                       & 
            ifms, ifme, jfms, jfme,                       &
            ifts,ifte,jfts,jfte,                          &
            ir,jr,                                        & 
            u_frame, v_frame,                             & 
            u,v,                                       & 
            uf,vf)                                      
    
    endif



    call sfire_model (id,ifun,restart,need_lfn_update,  &
        num_ignitions,                          & 
        ifuelread,nfuel_cat0,                   & 
        ifds,ifde,jfds,jfde,                    & 
        ifms,ifme,jfms,jfme,                    & 
        ifps,ifpe,jfps,jfpe,                    &
        ifts,ifte,jfts,jfte,                    & 
        time_start,dt,                          & 
        dxf,dyf,                                & 
        ignition_start_x,ignition_start_y,      & 
        ignition_end_x,ignition_end_y,          &
        ignition_radius,                        &
        ignition_time,                          &
        ignitions_done_tile(ij),ignited_tile(1,ij),  &
        fxlong,fxlat,unit_fxlong,unit_fxlat,      & 
        zsf,                                    & 
        uf,vf,                                  & 
        lfn,lfn_out,tign,fuel_frac,                     & 
        fire_area,                              & 
        fgrnhfx,fgrnqfx,                        & 
        ros,                                    & 
        nfuel_cat,                              & 
        fuel_time,                              & 
        bbb,betafl,phiwc,r_0,fgip,ischap &
    )
    
    if(ifun.eq.6)then 
    
        call print_2d_stats(ifts,ifte,jfts,jfte,ifms,ifme,jfms,jfme,fgrnhfx,'fire_driver:fgrnhfx')
        call print_2d_stats(ifts,ifte,jfts,jfte,ifms,ifme,jfms,jfme,fgrnqfx,'fire_driver:fgrnqfx')
    
        
        call sum_2d_cells(        &
            ifms,ifme,jfms,jfme,  &
            ifts,ifte,jfts,jfte,  &
            fgrnhfx,              &
            ims, ime, jms, jme,   &
            its,ite,jts,jte,      &
            grnhfx)

        call sum_2d_cells(        &
            ifms,ifme,jfms,jfme,  &
            ifts,ifte,jfts,jfte,  &
            fgrnqfx,              &
            ims, ime, jms, jme,   &
            its,ite,jts,jte,      &
            grnqfx)

        write(msg,'(a,f6.3)')'fire-atmosphere feedback scaling ',fire_atm_feedback
        do j=jts,jte
            do i=its,ite
                
                grnhfx(i,j)=fire_atm_feedback*grnhfx(i,j)/(ir*jr)
                grnqfx(i,j)=fire_atm_feedback*grnqfx(i,j)/(ir*jr)
                
                canhfx(i,j)=0
                canqfx(i,j)=0
            enddo
        enddo

        do j=jts,jte
            do k=kts,min(kte+1,kde)
               do i=its,ite
                   rthfrten(i,k,j)=0.
                   rqvfrten(i,k,j)=0.
               enddo
            enddo
        enddo


        

       call print_2d_stats(its,ite,jts,jte,ims,ime,jms,jme,grnhfx,'fire_driver:grnhfx')
       call print_2d_stats(its,ite,jts,jte,ims,ime,jms,jme,grnqfx,'fire_driver:grnqfx')

       call fire_tendency(                 &
            ids,ide, kds,kde, jds,jde,      & 
            ims,ime, kms,kme, jms,jme,      &
            its,ite, kts,kte, jts,jte,      & 
            grnhfx,grnqfx,canhfx,canqfx,        & 
            fire_ext_grnd,fire_ext_crwn,fire_crwn_hgt,                &
            zs,z_at_w,dz8w,mu,rho,          &
            rthfrten,rqvfrten)                

       

       call print_3d_stats(its,ite,kts,kte,jts,jte,ims,ime,kms,kme,jms,jme,rthfrten,'fire_driver_phys:rthfrten')
       call print_3d_stats(its,ite,kts,kte,jts,jte,ims,ime,kms,kme,jms,jme,rqvfrten,'fire_driver_phys:rqvfrten')

            
    endif 

enddo 
!$OMP END PARALLEL DO

if (ifun.eq.3)then
    ignitions_done=ignitions_done_tile(1) 
    do i=1,ignitions_done
        write(msg,*)'sfire_driver_phys: checking ignition ',i,' of ',ignitions_done
        call message(msg)
        ignited_patch(i)=0
        do ij=1,num_tiles
            ignited_patch(i)=ignited_patch(i)+ignited_tile(i,ij)
        enddo
        if(ignited_patch(i).eq.0)then
            call crash('sfire_driver_phys: Ignition failed, no nodes ignited. Bad coordinates?')
        endif
    enddo
endif

if(ifun.eq.1)then
    if(pid.ne.0)then
        call write_array_m(ips,ipe,jps,jpe,ims,ime,jms,jme,zs,'zs',pid)
        call write_array_m(ifps,ifpe,jfps,jfpe,ifms,ifme,jfms,jfme,zsf,'zsf',pid)
    endif
elseif(ifun.eq.3)then
    if(pid.gt.0)then
        call write_array_m3(ips,ipe+1,kds,kds+1,jps,jpe+1,ims,ime,kms,kme,jms,jme,u,'u',pid)
        call write_array_m3(ips,ipe+1,kds,kds+1,jps,jpe+1,ims,ime,kms,kme,jms,jme,v,'v',pid)
        call write_array_m(ifps,ifpe,jfps,jfpe,ifms,ifme,jfms,jfme,uf,'uf',pid)
        call write_array_m(ifps,ifpe,jfps,jfpe,ifms,ifme,jfms,jfme,vf,'vf',pid)
    endif
elseif(ifun.eq.5)then
    if(pid.gt.0)then
        call write_array_m(ifps,ifpe,jfps,jfpe,ifms,ifme,jfms,jfme,lfn,'lfn',pid)
        call write_array_m(ifps,ifpe,jfps,jfpe,ifms,ifme,jfms,jfme,tign,'tign',pid)
    endif
elseif(ifun.eq.6)then
    if(pid.gt.0)then
        call write_array_m(ips,ipe,jps,jpe,ims,ime,jms,jme,grnhfx,'grnhfx',pid)
        call write_array_m(ips,ipe,jps,jpe,ims,ime,jms,jme,grnqfx,'grnqfx',pid)
        call write_array_m3(ips,ipe,kps,kpe,jps,jpe,ims,ime,kms,kme,jms,jme,rthfrten,'rthfrten',pid)
        call write_array_m3(ips,ipe,kps,kpe,jps,jpe,ims,ime,kms,kme,jms,jme,rqvfrten,'rqvfrten',pid)
        call write_array_m(ifps,ifpe,jfps,jfpe,ifms,ifme,jfms,jfme,fuel_frac,'fuel_frac',pid)
        call write_array_m(ifps,ifpe,jfps,jfpe,ifms,ifme,jfms,jfme,fgrnhfx,'fgrnhfx',pid)
        call write_array_m(ifps,ifpe,jfps,jfpe,ifms,ifme,jfms,jfme,fgrnqfx,'fgrnqfx',pid)
    endif
    
    
    kpe1=min(kps+1,kpe)
    
    do k=kts,min(kte,kts+3)
        write(kk,'(i2)')k
        call print_3d_stats(ips,ipe,k,k,jps,jpe,ims,ime,kms,kme,jms,jme,rthfrten,kk//'driver_phys:rthfrten')
        call print_3d_stats(ips,ipe,k,k,jps,jpe,ims,ime,kms,kme,jms,jme,rqvfrten,kk//'driver_phys:rqvfrten')
    enddo
endif

end subroutine sfire_driver_phys




subroutine fire_ignition_convert (config_flags,fire_max_ignitions,ignition_longlat, &
    fire_ignition_start_x,fire_ignition_start_y,fire_ignition_end_x,fire_ignition_end_y, &
    fire_ignition_radius,fire_ignition_time,fire_num_ignitions)
    USE module_configure
    implicit none


    TYPE (grid_config_rec_type) , INTENT(IN)          :: config_flags
    integer, intent(in)::fire_max_ignitions
    real, dimension(fire_max_ignitions), intent(out):: &
        fire_ignition_start_x,fire_ignition_start_y,fire_ignition_end_x,fire_ignition_end_y, &
        fire_ignition_radius,fire_ignition_time
    integer, intent(out)::fire_num_ignitions,ignition_longlat

    integer::i
    logical:: real,ideal

    
    if(fire_max_ignitions.lt.5)call crash('fire_max_ignitions too small')
    
    ideal=config_flags%fire_ignition_start_x1 .ne.0. .or. config_flags%fire_ignition_start_y1 .ne. 0.
    real=config_flags%fire_ignition_start_lon1 .ne. 0. .or. config_flags%fire_ignition_start_lat1 .ne. 0.
    if(ideal)call message('Using ideal ignition coordinates, m from the lower left domain corner')
    if(real)call message('Using real ignition coordinates, longitude and latitude')
    if(ideal.and.real)call crash('Only one of the ideal or real coordinates may be given')

    if(ideal)then
        
        ignition_longlat=0
        fire_ignition_start_x(1)=config_flags%fire_ignition_start_x1
        fire_ignition_start_y(1)=config_flags%fire_ignition_start_y1
        fire_ignition_end_x(1)=config_flags%fire_ignition_end_x1
        fire_ignition_end_y(1)=config_flags%fire_ignition_end_y1
        fire_ignition_start_x(2)=config_flags%fire_ignition_start_x2
        fire_ignition_start_y(2)=config_flags%fire_ignition_start_y2
        fire_ignition_end_x(2)=config_flags%fire_ignition_end_x2
        fire_ignition_end_y(2)=config_flags%fire_ignition_end_y2
        fire_ignition_start_x(3)=config_flags%fire_ignition_start_x3
        fire_ignition_start_y(3)=config_flags%fire_ignition_start_y3
        fire_ignition_end_x(3)=config_flags%fire_ignition_end_x3
        fire_ignition_end_y(3)=config_flags%fire_ignition_end_y3
        fire_ignition_start_x(4)=config_flags%fire_ignition_start_x4
        fire_ignition_start_y(4)=config_flags%fire_ignition_start_y4
        fire_ignition_end_x(4)=config_flags%fire_ignition_end_x4
        fire_ignition_end_y(4)=config_flags%fire_ignition_end_y4
        fire_ignition_start_x(5)=config_flags%fire_ignition_start_x5
        fire_ignition_start_y(5)=config_flags%fire_ignition_start_y5
        fire_ignition_end_x(5)=config_flags%fire_ignition_end_x5
        fire_ignition_end_y(5)=config_flags%fire_ignition_end_y5
    endif
    if(real)then
        
        ignition_longlat=1
        fire_ignition_start_x(1)=config_flags%fire_ignition_start_lon1
        fire_ignition_start_y(1)=config_flags%fire_ignition_start_lat1
        fire_ignition_end_x(1)=config_flags%fire_ignition_end_lon1
        fire_ignition_end_y(1)=config_flags%fire_ignition_end_lat1
        fire_ignition_start_x(2)=config_flags%fire_ignition_start_lon2
        fire_ignition_start_y(2)=config_flags%fire_ignition_start_lat2
        fire_ignition_end_x(2)=config_flags%fire_ignition_end_lon2
        fire_ignition_end_y(2)=config_flags%fire_ignition_end_lat2
        fire_ignition_start_x(3)=config_flags%fire_ignition_start_lon3
        fire_ignition_start_y(3)=config_flags%fire_ignition_start_lat3
        fire_ignition_end_x(3)=config_flags%fire_ignition_end_lon3
        fire_ignition_end_y(3)=config_flags%fire_ignition_end_lat3
        fire_ignition_start_x(4)=config_flags%fire_ignition_start_lon4
        fire_ignition_start_y(4)=config_flags%fire_ignition_start_lat4
        fire_ignition_end_x(4)=config_flags%fire_ignition_end_lon4
        fire_ignition_end_y(4)=config_flags%fire_ignition_end_lat4
        fire_ignition_start_x(5)=config_flags%fire_ignition_start_lon5
        fire_ignition_start_y(5)=config_flags%fire_ignition_start_lat5
        fire_ignition_end_x(5)=config_flags%fire_ignition_end_lon5
        fire_ignition_end_y(5)=config_flags%fire_ignition_end_lat5
    endif
    
        fire_ignition_radius(1)=config_flags%fire_ignition_radius1 
        fire_ignition_time(1)=config_flags%fire_ignition_time1 
        fire_ignition_radius(2)=config_flags%fire_ignition_radius2 
        fire_ignition_time(2)=config_flags%fire_ignition_time2 
        fire_ignition_radius(3)=config_flags%fire_ignition_radius3 
        fire_ignition_time(3)=config_flags%fire_ignition_time3 
        fire_ignition_radius(4)=config_flags%fire_ignition_radius4 
        fire_ignition_time(4)=config_flags%fire_ignition_time4 
        fire_ignition_radius(5)=config_flags%fire_ignition_radius5 
        fire_ignition_time(5)=config_flags%fire_ignition_time5

    
        fire_num_ignitions=0      
        do i=1,min(5,config_flags%fire_num_ignitions)
            
            if(fire_ignition_radius(i).gt.0.)fire_num_ignitions=i
            
            if(fire_ignition_end_x(i).eq.0.)fire_ignition_end_x(i)=fire_ignition_start_x(i)
            if(fire_ignition_end_y(i).eq.0.)fire_ignition_end_y(i)=fire_ignition_start_y(i)
        enddo

end subroutine fire_ignition_convert






subroutine interpolate_atm2fire(id,               & 
    ids,ide, kds,kde, jds,jde,                    & 
    ims,ime, kms,kme, jms,jme,                    &
    ips,ipe,jps,jpe,                              &
    its,ite,jts,jte,                              &
    ifds, ifde, jfds, jfde,                       & 
    ifms, ifme, jfms, jfme,                       &
    ifts,ifte,jfts,jfte,                          &
    ir,jr,                                        & 
    u_frame, v_frame,                             & 
    u,v,                                          & 
    uf,vf)                                          
    
implicit none



integer, intent(in)::id,                          &
    ids,ide, kds,kde, jds,jde,                    & 
    ims,ime, kms,kme, jms,jme,                    & 
    ips,ipe,jps,jpe,                              &
    its,ite,jts,jte,                              & 
    ifds, ifde, jfds, jfde,                       & 
    ifms, ifme, jfms, jfme,                       & 
    ifts,ifte,jfts,jfte,                          & 
    ir,jr                                         
real, intent(in):: u_frame, v_frame                 
real,intent(in),dimension(ims:ime,kms:kme,jms:jme)::&
    u,v                                             
real,intent(out), dimension(ifms:ifme,jfms:jfme)::&
    uf,vf                                           
    
    

real, dimension(its-1:ite+2,jts-1:jte+2):: ua,va   
integer:: i,j,k,ifts1,ifte1,jfts1,jfte1



    k=kds             
    do j = jts-1,jte+2
        do i = its-1,ite+2 
            
            ua(i,j)=0.5*( u(i,k,j) + u(i,k+1,j)) + u_frame
            va(i,j)=0.5*( v(i,k,j) + v(i,k+1,j)) + v_frame
        enddo
    enddo

    
    call continue_at_boundary(1,0,0., & 
    its-1,ite+2,jts-1,jte+2,           &                
    ids,ide+1,jds,jde+1, &            
    ips,ipe+1,jps,jpe+1, &            
    its,ite+1,jts,jte+1, &                
    va)                               

    call continue_at_boundary(0,1,0., & 
    its-1,ite+2,jts-1,jte+2,           &                
    ids,ide+1,jds,jde+1, &            
    ips,ipe+1,jps,jpe+1, &            
    its,ite+1,jts,jte+1, &                
    ua)                               






call print_2d_stats_vec(its,ite+1,jts,jte+1,its-1,ite+2,jts-1,jte+2,ua,va, &
    'driver: atm wind (m/s)')
    




















    
    ifts1=snode(ifts,ifds,-1) 
    ifte1=snode(ifte,ifde,+1)
    jfts1=snode(jfts,jfds,-1)
    jfte1=snode(jfte,jfde,+1)
    
    call interpolate_2d(  &
        its-1,ite+2,jts-1,jte+2,                  & 
        its-1,ite+2,jts-1,jte+2,                  & 
        ifms,ifme,jfms,jfme,    & 
        ifts1,ifte1,jfts1,jfte1,& 
        ir,jr,                  & 
        real(ids),real(jds),ifds-.5,jfds+(jr+1)*.5, & 
        ua,                     & 
        uf)                      

    call interpolate_2d(  &
        its-1,ite+2,jts-1,jte+2,                  & 
        its-1,ite+2,jts-1,jte+2,                  & 
        ifms,ifme,jfms,jfme,    & 
        ifts1,ifte1,jfts1,jfte1,& 
        ir,jr,                  & 
        real(ids),real(jds),ifds+(ir+1)*.5,jfds-0.5, & 
        va,                     & 
        vf)                      




end subroutine interpolate_atm2fire





subroutine interpolate_z2fire(id,                 & 
    ids,ide, jds,jde,                    & 
    ims,ime, jms,jme,                    &
    ips,ipe,jps,jpe,                              &
    its,ite,jts,jte,                              &
    ifds, ifde, jfds, jfde,                       & 
    ifms, ifme, jfms, jfme,                       &
    ifts,ifte,jfts,jfte,                          &
    ir,jr,                                        & 
    zs,                                       & 
    zsf)                                      
    
implicit none



integer, intent(in)::id,                          &
    ids,ide, jds,jde,                    & 
    ims,ime,jms,jme,                    & 
    ips,ipe,jps,jpe,                              &
    its,ite,jts,jte,                              & 
    ifds, ifde, jfds, jfde,                       & 
    ifms, ifme, jfms, jfme,                       & 
    ifts,ifte,jfts,jfte,                          & 
    ir,jr                                         
real, intent(in), dimension(ims:ime, jms:jme):: zs  
real,intent(out), dimension(ifms:ifme,jfms:jfme)::&
    zsf                                             
    
    

real, dimension(its-2:ite+2,jts-2:jte+2):: za      
integer:: i,j,jts1,jte1,its1,ite1,jfts1,jfte1,ifts1,ifte1



    jts1=max(jts-1,jds) 
    its1=max(its-1,ids) 
    jte1=min(jte+1,jde) 
    ite1=min(ite+1,ide)
    do j = jts1,jte1
        do i = its1,ite1 
            
            za(i,j)=zs(i,j)           
        enddo
    enddo

    call continue_at_boundary(1,1,0., & 
    its-2,ite+2,jts-2,jte+2,           &                
    ids,ide,jds,jde, &            
    ips,ipe,jps,jpe, &            
    its1,ite1,jts1,jte1, &                
    za)                               

    
    jfts1=snode(jfts,jfds,-1) 
    ifts1=snode(ifts,ifds,-1)
    jfte1=snode(jfte,jfde,+1) 
    ifte1=snode(ifte,ifde,+1)
                     
    call interpolate_2d(  &
        its-2,ite+2,jts-2,jte+2, & 
        its1-1,ite1+1,jts1-1,jte1+1, & 
        ifms,ifme,jfms,jfme,    & 
        ifts1,ifte1,jfts1,jfte1,  & 
        ir,jr,                  & 
        real(ids),real(jds),ifds+(ir+1)*.5,jfds+(jr+1)*.5, & 
        za,                     & 
        zsf)                      

end subroutine interpolate_z2fire




subroutine check_fmesh(ids,ide,ifds,ifde,ir,s)

implicit none

integer, intent(in)::ids,ide,ifds,ifde,ir
character(len=*),intent(in)::s

character(len=128)msg

if ((ide-ids+1)*ir.ne.(ifde-ifds+1))then
    write(msg,1)s,ids,ide,ifds,ifde,ir
1   format('module_fr_sfire_driver: incompatible bounds ',a,' atm ',i5,':',i5,' fire ',i5,':',i5,' ratio ',i3)    
    call crash(msg)
endif
end subroutine check_fmesh




subroutine set_flags(config_flags)
USE module_configure
use module_fr_sfire_util
implicit none
TYPE (grid_config_rec_type) , INTENT(IN)          :: config_flags




fire_print_msg          = config_flags%fire_print_msg
fire_print_file         = config_flags%fire_print_file
fuel_left_method        = config_flags%fire_fuel_left_method
fuel_left_irl           = config_flags%fire_fuel_left_irl
fuel_left_jrl           = config_flags%fire_fuel_left_jrl
fire_const_time         = config_flags%fire_const_time
fire_const_grnhfx       = config_flags%fire_const_grnhfx
fire_const_grnqfx       = config_flags%fire_const_grnqfx
fire_atm_feedback       = config_flags%fire_atm_feedback
boundary_guard          = config_flags%fire_boundary_guard
fire_back_weight        = config_flags%fire_back_weight
fire_grows_only         = config_flags%fire_grows_only
fire_upwinding          = config_flags%fire_upwinding
fire_upwind_split       = config_flags%fire_upwind_split 
fire_viscosity          = config_flags%fire_viscosity 
fire_lfn_ext_up         = config_flags%fire_lfn_ext_up 
fire_test_steps         = config_flags%fire_test_steps 
fire_topo_from_atm      = config_flags%fire_topo_from_atm
fire_advection          = config_flags%fire_advection




end subroutine set_flags

subroutine print_id
character(len=128)::id,msg
id='b6fe89aeb71d941e91530aafcf2f5b183a44fc37'
msg=id
call message(msg)
end subroutine print_id

end module module_fr_sfire_driver
