module DGVMMod

#if (defined DGVM)

!-----------------------------------------------------------------------
!BOP
!
! !MODULE: DGVMMod
!
! !DESCRIPTION:
! Module containing routines to drives the annual portion of lpj
! (called once per year), reset variables related to lpj,
! and initialize/Reset time invariant dgvm variables
!
! !USES:
  use shr_kind_mod   , only : r8 => shr_kind_r8
  use clm_varpar     , only : maxpatch_pft, lsmlon, lsmlat, nlevsoi
!
! !PUBLIC TYPES:
  implicit none
  private
  save
!
! !PUBLIC MEMBER FUNCTIONS:
  public lpj                ! Drives the annual portion of lpj, called once
                            ! per year
  public lpjreset1          ! Resets variables related to lpj
  public lpjreset2          ! Resets variables related to lpj
  public resetTimeConstDGVM ! Initialize/Reset time invariant dgvm variables
  public resetWeightsDGVM   ! Reset DGVM subgrid weights and areas
!
! !REVISION HISTORY:
! Module created by Mariana Vertenstein
!
!EOP
!-----------------------------------------------------------------------

contains

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: lpj
!
! !INTERFACE:
  subroutine lpj(lbg, ubg, lbp, ubp, num_natvegp, filter_natvegp, kyr)
!
! !DESCRIPTION:
! Drives the annual portion of lpj, called once per year
!
! !USES:
    use clmtype
    use DGVMReproductionMod , only : Reproduction
    use DGVMTurnoverMod     , only : Turnover
    use DGVMAllocationMod   , only : Allocation
    use DGVMLightMod        , only : Light
    use DGVMMortalityMod    , only : Mortality
    use DGVMFireMod         , only : Fire
    use DGVMEstablishmentMod, only : Establishment
    use DGVMKillMod         , only : Kill
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: lbg, ubg       ! gridcell bounds
    integer, intent(in) :: lbp, ubp       ! pft bounds
    integer, intent(inout) :: num_natvegp ! number of naturally-vegetated
                                          ! pfts in filter
    integer, intent(inout) :: filter_natvegp(ubp-lbp+1) ! filter for
                                          ! naturally-vegetated pfts
    integer, intent(in) :: kyr            ! used in routine climate20 below
!
! !CALLED FROM:
!
! !REVISION HISTORY:
! Author: Sam Levis
!
! !LOCAL VARIABLES:
!
! local pointers to implicit in arguments
!
   integer , pointer :: mxy(:)         ! pft m index (for laixy(i,j,m),etc.)
   integer , pointer :: pgridcell(:)   ! gridcell of corresponding pft
   real(r8), pointer :: fpcgrid(:)     ! foliar projective cover on gridcell (fraction)
   real(r8), pointer :: agdd(:)        ! accumulated growing degree days above 5
   real(r8), pointer :: t_mo_min(:)    ! annual min of t_mo (Kelvin)
!
! local pointers to implicit inout arguments
!
   real(r8), pointer :: tmomin20(:)         ! 20-yr running mean of tmomin
   real(r8), pointer :: agdd20(:)           ! 20-yr running mean of agdd
   real(r8), pointer :: bm_inc(:)           ! biomass increment
   real(r8), pointer :: afmicr(:)           ! annual microbial respiration
   real(r8), pointer :: afirefrac_gcell(:)  ! fraction of gridcell affected by fire
   real(r8), pointer :: acfluxfire_gcell(:) ! gridcell C flux to atmosphere from biomass burning
   real(r8), pointer :: bmfm_gcell(:,:)     ! gridcell biomass
   real(r8), pointer :: afmicr_gcell(:,:)   ! gridcell microbial respiration
!
!EOP
!
! !LOCAL VARIABLES:
    integer  :: g,p                    ! indices
    real(r8) :: afirefrac(lbp:ubp)     ! for history write
    real(r8) :: acfluxfire(lbp:ubp)    ! for history write
!-----------------------------------------------------------------------

    ! Assign local pointers to derived type members (gridcell-level)

    afirefrac_gcell  => clm3%g%gdgvs%afirefrac
    acfluxfire_gcell => clm3%g%gdgvs%acfluxfire
    bmfm_gcell       => clm3%g%gdgvs%bmfm
    afmicr_gcell     => clm3%g%gdgvs%afmicr

    ! Assign local pointers to derived type members (pft-level)

    mxy       => clm3%g%l%c%p%mxy
    pgridcell => clm3%g%l%c%p%gridcell
    fpcgrid   => clm3%g%l%c%p%pdgvs%fpcgrid
    tmomin20  => clm3%g%l%c%p%pdgvs%tmomin20
    t_mo_min  => clm3%g%l%c%p%pdgvs%t_mo_min
    agdd      => clm3%g%l%c%p%pdgvs%agdd
    agdd20    => clm3%g%l%c%p%pdgvs%agdd20
    bm_inc    => clm3%g%l%c%p%pdgvs%bm_inc
    afmicr    => clm3%g%l%c%p%pdgvs%afmicr

    ! *************************************************************************
    ! S. Levis version of LPJ's routine climate20 - 'Returns' tmomin20 and agdd20
    ! for use in routine bioclim, which I have placed in routine Establishment
    ! Instead of 20-yr running mean of coldest monthly temperature,
    ! use 20-yr running mean of minimum 10-day running mean
    ! *************************************************************************

!dir$ concurrent
!cdir nodep
    do p = lbp,ubp
       if (kyr == 2) then
          tmomin20(p) = t_mo_min(p)
          agdd20(p) = agdd(p)
       end if
       tmomin20(p) = (19.0 * tmomin20(p) + t_mo_min(p)) / 20.0
       agdd20(p)   = (19.0 * agdd20(p)   + agdd(p)    ) / 20.0
    end do

    ! Determine grid values of npp and microbial respiration

    bmfm_gcell(lbg:ubg,1:maxpatch_pft) = 0._r8
!dir$ concurrent
!cdir nodep
    do p = lbp,ubp
       if (mxy(p) <= maxpatch_pft) then
          g = pgridcell(p)
          bmfm_gcell(g,mxy(p)) = bm_inc(p)  ! [gC/m2 patch] for output
       end if
       bm_inc(p) = bm_inc(p) * fpcgrid(p)   ! [gC/m2 cell vegetated area]
    end do

    afmicr_gcell(lbg:ubg,1:maxpatch_pft) = 0._r8
!dir$ concurrent
!cdir nodep
    do p = lbp,ubp
       if (mxy(p) <= maxpatch_pft) then
          g = pgridcell(p)
          afmicr_gcell(g,mxy(p)) = afmicr(p) * fpcgrid(p) ![gC/m2 cell veg'd area]
       end if
    end do

    ! Build filter of present natually-vegetated pfts

    call BuildNatVegFilter(lbp, ubp, num_natvegp, filter_natvegp)

    ! Returns updated bm_inc, litterag

    call Reproduction(lbp, ubp, num_natvegp, filter_natvegp)

    ! Returns turnover_ind and updated litterag,bg, l,s,h,rm_ind

    call Turnover(lbp, ubp, num_natvegp, filter_natvegp)

    ! Returns updated litterag, bg, and present

    call Kill(lbp, ubp, num_natvegp, filter_natvegp)

    ! Rebuild filter of present natually-vegetated pfts after Kill()

    call BuildNatVegFilter(lbp, ubp, num_natvegp, filter_natvegp)

    ! Returns lai_ind, lai_inc, updates crownarea, htop, l, s, h, rm_ind, litterag, litterbg

    call Allocation(lbp, ubp, num_natvegp, filter_natvegp)

    ! Returns lm,rm_ind, fpcgrid, nind, litterag,bg via modules
    ! reason for different set up (ie, no external patch loop):
    ! in this routine sub-grid patches (k) communicate at the grid cell level (i,j)

    call Light(lbg, ubg, lbp, ubp, num_natvegp, filter_natvegp)

    ! Obtain updated present, nind, litterag and bg

    call Mortality(lbp, ubp, num_natvegp, filter_natvegp)

    ! Returns updated litterag and nind

    call Fire(lbp, ubp, afirefrac, acfluxfire)

    afirefrac_gcell(lbg:ubg) = 0.0
    acfluxfire_gcell(lbg:ubg) = 0.0
    do p = lbp,ubp
       g = pgridcell(p)
       afirefrac_gcell(g) = afirefrac_gcell(g) + afirefrac(p)*fpcgrid(p)
       acfluxfire_gcell(g) = acfluxfire_gcell(g) + acfluxfire(p)
    end do

    ! Returns updated present, nind, *m_ind, crownarea, fpcgrid, htop,
    ! litter*g, and vegetation type
    ! reason for different set up (ie, no external patch loop):
    ! in this routine sub-grid patches (k) communicate at the grid cell level (i,j)

    call Establishment(lbp, ubp, lbg, ubg)

  end subroutine lpj

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: lpjreset1
!
! !INTERFACE:
  subroutine lpjreset1(lbg, ubg, lbc, ubc, lbp, ubp, &
                       num_nolakep, filter_nolakep, caldayp1)
!
! !DESCRIPTION:
! Resets variables related to lpj!
!
! !USES:
    use clmtype
    use SurfaceAlbedoMod   , only : SurfaceAlbedo
    use DGVMEcosystemDynMod, only : DGVMEcosystemDyn
!
! !ARGUMENTS:
    implicit none
    integer , intent(in) :: lbg, ubg       ! gridcell bounds
    integer , intent(in) :: lbc, ubc       ! column bounds
    integer , intent(in) :: lbp, ubp       ! pft bounds
    integer , intent(in) :: num_nolakep    ! number of non-lake pfts in filter
    integer , intent(in) :: filter_nolakep(ubp-lbp+1) ! pft filter for non-lake points
    real(r8), intent(in) :: caldayp1 !calendar day at Greenwich (1.00, ..., 365.99) for nstep+1
!
! !CALLED FROM:
! subroutine driver() in driver.F90
!
! !REVISION HISTORY:
! Author: Sam Levis
!
!EOP
!
! !LOCAL VARIABLES:
    integer :: p         ! indices
!-----------------------------------------------------------------------

    ! Reset a few variables here at the very end of the year
    ! First determine necessary per processor subgrid bounds

!dir$ concurrent
!cdir nodep
    do p = lbp,ubp
       clm3%g%l%c%p%pdgvs%annpsn(p)     = 0.
       clm3%g%l%c%p%pdgvs%annpsnpot(p)  = 0.
       clm3%g%l%c%p%pdgvs%bm_inc(p)     = 0.
       clm3%g%l%c%p%pdgvs%afmicr(p)     = 0.
       clm3%g%l%c%p%pdgvs%firelength(p) = 0.
       clm3%g%l%c%p%pdgvs%agddtw(p)     = 0.
       clm3%g%l%c%p%pdgvs%agdd(p)       = 0.
       clm3%g%l%c%p%pdgvs%t10min(p)     = 1.0e+36
       clm3%g%l%c%p%pdgvs%t_mo_min(p)   = 1.0e+36
    end do

    ! Call DGVMEcosystemDyn and SurfaceAlbedo because need information
    ! for first timestep of next year.

    call DGVMEcosystemDyn(lbp, ubp, num_nolakep, filter_nolakep, &
         doalb=.false., endofyr=.true.)

    call SurfaceAlbedo(lbg, ubg, lbc, ubc, lbp, ubp,caldayp1)

    call resetTimeConstDGVM(lbp, ubp)

  end subroutine lpjreset1

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: lpjreset2
!
! !INTERFACE:
  subroutine lpjreset2(lbg, ubg, lbl, ubl, lbc, ubc, lbp, ubp)
!
! !DESCRIPTION:
! Resets variables related to lpj
!
! !USES:
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: lbg, ubg       ! gridcell bounds
    integer, intent(in) :: lbl, ubl       ! landunit bounds
    integer, intent(in) :: lbc, ubc       ! column bounds
    integer, intent(in) :: lbp, ubp       ! pft bounds
!
! !CALLED FROM:
!
! !REVISION HISTORY:
! Author: Sam Levis
!
!EOP
!-----------------------------------------------------------------------

    !------------------------------------------------------------------
    ! Determine new subgrid weights and areas
    ! In CLM2 with satellite data, the number of veg pfts is determined once
    ! and is less than maxpatch_pft (4) in some cells.
    ! In LSM with LPJ, the number of veg patches could be dynamic. Until we
    ! implement it as such, we will make all grid cells have 10 veg patches.
    !------------------------------------------------------------------

    call resetWeightsDGVM(lbg, ubg, lbc, ubc, lbp, ubp)

  end subroutine lpjreset2

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: resetTimeConstDGVM
!
! !INTERFACE:
  subroutine resetTimeConstDGVM(lbp, ubp)
!
! !DESCRIPTION:
! Initialize/reset time invariant DGVM variables
!
! !USES:
    use clmtype
    use pftvarcon , only : roota_par, rootb_par, noveg
    use clm_varcon, only : spval
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: lbp, ubp       ! pft bounds
!
! !CALLED FROM:
! lpjreset1() in this module
! initialize() in initializeMod.F90
! iniTimeVar() in iniTimeVar.F90
!
! !REVISION HISTORY:
! Author: Gordon Bonan
!
! !LOCAL VARIABLES:
!
! local pointers to implicit in arguments
!
   real(r8), pointer :: zi(:,:)        ! interface level below a "z" level (m) (-nlevsno+0:nlevsoi)
   integer , pointer :: ivt(:)         ! pft vegetation
   integer , pointer :: pcolumn(:)     ! column of corresponding pft
   real(r8), pointer :: rootfr(:,:)    ! fraction of roots in each soil layer  (nlevsoi)
!
!EOP
!
! !LOCAL VARIABLES:
    integer :: p,c,j        ! indices
!-----------------------------------------------------------------------

    ! Assign local pointers to derived subtypes components (column-level)

    zi      => clm3%g%l%c%cps%zi

    ! Assign local pointers to derived subtypes components (pft-level)

    ivt     => clm3%g%l%c%p%itype
    pcolumn => clm3%g%l%c%p%column
    rootfr  => clm3%g%l%c%p%pps%rootfr

    ! Initialize root fraction (computing from surface, d is depth in meter):
    ! Y = 1 -1/2 (exp(-ad)+exp(-bd) under the constraint that
    ! Y(d =0.1m) = 1-beta^(10 cm) and Y(d=d_obs)=0.99 with
    ! beta & d_obs given in Zeng et al. (1998).

!dir$ concurrent
!cdir nodep
    do p = lbp,ubp
       c = pcolumn(p)
       if (ivt(p) /= noveg) then
          do j = 1, nlevsoi-1
             rootfr(p,j) = .5*( exp(-roota_par(ivt(p)) * zi(c,j-1))  &
                              + exp(-rootb_par(ivt(p)) * zi(c,j-1))  &
                              - exp(-roota_par(ivt(p)) * zi(c,j  ))  &
                              - exp(-rootb_par(ivt(p)) * zi(c,j  )) )
          end do
          rootfr(p,nlevsoi) = .5*( exp(-roota_par(ivt(p)) * zi(c,nlevsoi-1))  &
                                 + exp(-rootb_par(ivt(p)) * zi(c,nlevsoi-1)) )
       else
          rootfr(p,1:nlevsoi) = spval
       end if
    end do

  end subroutine resetTimeConstDGVM

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: resetWeightsDGVM
!
! !INTERFACE:
  subroutine resetWeightsDGVM(lbg, ubg, lbc, ubc, lbp, ubp)
!
! !DESCRIPTION:
! Reset DGVM weights
!
! !USES:
    use clmtype
!
! !ARGUMENTS:
    implicit none
    integer, intent(in) :: lbg, ubg       ! gridcell bounds
    integer, intent(in) :: lbc, ubc       ! column bounds
    integer, intent(in) :: lbp, ubp       ! pft bounds
!
! !CALLED FROM:
!  subroutine lpjreset2 in this module: as part of the DGVM calculation
!  subroutine restart_dgvm in module DGVMRestMod: if the restart file is read
!  subroutine inicrd in module inicFileMod: if the initial file is read
!  subroutine mkarbinit in module iniTimeVar
!
! !REVISION HISTORY:
! Author: Gordon Bonan
!
! !LOCAL VARIABLES:
!
! local pointers to implicit in arguments

!CLM 3.5 got rid of l%area, c%area, and p%area.  Need to revise code.
!Zack Subin, 7/17/08
!
   integer , pointer :: ixy(:)            ! gridcell lon index (gridcell level)
   integer , pointer :: jxy(:)            ! gridcell lat index (gridcell level)
   real(r8), pointer :: garea(:)          ! total land area for this gridcell (km^2)
   integer , pointer :: ltype(:)          ! landunit type
   real(r8), pointer :: larea(:)          ! total land area for this landunit (km^2)
   logical , pointer :: ifspecial(:)      ! true=>landunit is not vegetated
   integer , pointer :: clandunit(:)      ! index into landunit for each column
   integer , pointer :: cgridcell(:)      ! index into gridcell for each column
   real(r8), pointer :: fpcgrid(:)        ! weight of pft relative to vegetated landunit
   real(r8), pointer :: h2ocan_pft(:)     ! canopy water (mm H2O) (pft-level)
!
! local pointers to implicit out arguments
!
   real(r8), pointer :: cwtgcell(:)       ! weight (relative to gridcell) for this column (0-1)
   real(r8), pointer :: cwtlunit(:)       ! weight (relative to landunit) for this column (0-1)
   real(r8), pointer :: carea(:)          ! total land area for this column (km^2)
   real(r8), pointer :: h2ocan_col(:)     ! canopy water (mm H2O) (column-level)
   integer , pointer :: pcolumn(:)        ! index into column for each pft
   integer , pointer :: plandunit(:)      ! index into landunit for each pft
   integer , pointer :: pgridcell(:)      ! index into gridcell for each pft
   real(r8), pointer :: pwtcol(:)         ! weight (relative to column) for this pft (0-1)
   real(r8), pointer :: pwtlunit(:)       ! weight (relative to landunit) for this pft (0-1)
   real(r8), pointer :: pwtgcell(:)       ! weight (relative to gridcell) for this pft (0-1)
   real(r8), pointer :: parea(:)          ! total land area for this pft (km^2)
!
!EOP
!
! !LOCAL VARIABLES:
    integer  :: g,p,c,l             ! indices
    integer  :: fn,filterg(lbg:ubg) ! local gridcell filter for error check
    real(r8) :: sumwt(lbg:ubg)      ! consistency check
!-----------------------------------------------------------------------

    ! Assign local pointers to derived subtypes components (gridcell-level)

    ixy        => clm3%g%ixy
    jxy        => clm3%g%jxy
    garea      => clm3%g%area

    ! Assign local pointers to derived subtypes components (landunit-level)

    ltype      => clm3%g%l%itype
    larea      => clm3%g%l%area
    ifspecial  => clm3%g%l%ifspecial

    ! Assign local pointers to derived subtypes components (column-level)

    cgridcell  => clm3%g%l%c%gridcell
    clandunit  => clm3%g%l%c%landunit
    cwtlunit   => clm3%g%l%c%wtlunit
    cwtgcell   => clm3%g%l%c%wtgcell
    carea      => clm3%g%l%c%area
    h2ocan_col => clm3%g%l%c%cws%pws_a%h2ocan

    ! Assign local pointers to derived subtypes components (pft-level)

    pgridcell  => clm3%g%l%c%p%gridcell
    plandunit  => clm3%g%l%c%p%landunit
    pcolumn    => clm3%g%l%c%p%column
    pwtcol     => clm3%g%l%c%p%wtcol
    pwtlunit   => clm3%g%l%c%p%wtlunit
    pwtgcell   => clm3%g%l%c%p%wtgcell
    parea      => clm3%g%l%c%p%area
    fpcgrid    => clm3%g%l%c%p%pdgvs%fpcgrid
    h2ocan_pft => clm3%g%l%c%p%pws%h2ocan

    ! Determine new pft properties

!dir$ concurrent
!cdir nodep
    do p = lbp,ubp
       g = pgridcell(p)
       l = plandunit(p)
       if (.not. ifspecial(l)) then

          ! Determine pft weight relative to column and relative to landunit
#if (defined NOCOMPETE)
          ! One column per pft - column and pft areas are equal
          pwtcol(p) = 1.
          pwtlunit(p) = fpcgrid(p)
#else
          ! One column per landunit - column and landunit areas are equal
          ! Weight relative to column and weight relative to landunit are equal
          pwtcol(p) = fpcgrid(p)
          pwtlunit(p) = fpcgrid(p)
#endif
          ! Determine new pft area
          parea(p) = pwtlunit(p) * larea(l)

          ! Determine new pft weight relative to grid cell
          pwtgcell(p) = parea(p) /  garea(g)

       end if
    end do

#if (defined NOCOMPETE)
    ! Determine new column properties - only needed if compete is
    ! note defined and each column has exactly one pft. In this case
    ! pft and column indices will be the same and fpcgrid(c) can be
    ! used below instead of fpcgrid(p)

!dir$ concurrent
!cdir nodep
    do c = lbc,ubc
       g = cgridcell(c)
       l = clandunit(c)
       if (.not. ifspecial(l)) then
          ! Determine new column weight relative to landunit
          ! When competition is not on, each column has only one pft
          cwtlunit(c) = fpcgrid(c)

          ! Determine new column area
          carea(c) = cwtlunit(c) * larea(l)

          ! Determine new column weight relative to gridcell
          cwtgcell(c) = carea(c) / garea(g)
       end if
    end do
#endif

    ! Consistency check - add up all the pft weights for a given gridcell
    ! and make sure they are not greater than one.

    sumwt(:) = 0._r8
    do p = lbp,ubp
       g = pgridcell(p)
       sumwt(g) = sumwt(g) + pwtgcell(p)
    end do
    fn = 0
    do g = lbg,ubg
       if (abs(sumwt(g) - 1.0) > 1.0e-6) then
          fn = fn + 1
          filterg(fn) = g
       end if
    end do
    if (fn > 0) then
       g = filterg(1)
       write(6,*) 'resetWeightsDGVM: sumwt of pfts for grid cell ',&
         'i,j = ',ixy(g),jxy(g),' not equal to 1'
       write(6,*) 'sum of pft weights for gridcell =',sumwt(g)
       call endrun
    end if

    ! Determine average over all column pfts for h2ocan using new weights
    ! This is needed by begwb in DriverInitMod.F90.

    h2ocan_col(lbc:ubc) = 0._r8
    do p = lbp,ubp
       c = pcolumn(p)
       h2ocan_col(c) =  h2ocan_col(c) + h2ocan_pft(p) * pwtcol(p)
    end do

  end subroutine resetWeightsDGVM

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: BuildNatVegFilter
!
! !INTERFACE:
  subroutine BuildNatVegFilter(lbp, ubp, num_natvegp, filter_natvegp)
!
! !DESCRIPTION:
! Reconstruct a filter of naturally-vegetated PFTs for use in DGVM
!
! !USES:
    use clmtype
    use pftvarcon , only : crop
!
! !ARGUMENTS:
    implicit none
    integer, intent(in)  :: lbp, ubp                   ! pft bounds
    integer, intent(out) :: num_natvegp                ! number of pfts in naturally-vegetated filter
    integer, intent(out) :: filter_natvegp(ubp-lbp+1)  ! pft filter for naturally-vegetated points
!
! !CALLED FROM:
! subroutine lpj in this module
!
! !REVISION HISTORY:
! Author: Forrest Hoffman
!
! !LOCAL VARIABLES:
! local pointers to implicit in arguments
    integer , pointer :: ivt(:)         ! pft vegetation (pft level)
    logical , pointer :: present(:)     ! whether this pft present in patch
!EOP
!
! !LOCAL VARIABLES:
    integer :: p
!-----------------------------------------------------------------------

    ! Assign local pointers to derived type members (pft-level)
    ivt       => clm3%g%l%c%p%itype
    present   => clm3%g%l%c%p%pdgvs%present

    num_natvegp = 0
    do p = lbp,ubp
       if (ivt(p) > 0 .and. present(p) .and. crop(ivt(p)) == 0.) then
          num_natvegp = num_natvegp + 1
          filter_natvegp(num_natvegp) = p
       end if
    end do

  end subroutine BuildNatVegFilter

#endif

end module DGVMMod
