!======================================================================================
! 03/04/2010
! Created by Hongping Gu, Jiming Jin 
! For coupling lake model with WRF
! The lake model is extracted from CLM3.5
! Structures:
!     MODULE module_sf_lake
!       |- SUBROUTINE Lake
!       |              |- CALL ShalLakeFluxes
!       |              |                |- CALL QSat              
!       |              |                |- CALL MoninObukIni       (use FrictionVeloc_lake)
!       |              |                |- CALL FrictionVelocity   (use FrictionVeloc_lake)
!       |              |                |- CALL QSat             
!       |              |- CALL ShalLakeTemperature
!       |              |                |- CALL SoilThermProp_Lake
!       |              |                |- CALL Tridiagonal     
!       |              |                |- CALL PhaseChange_Lake
!       |              |                |- CALL SoilThermProp_Lake
!       |              |- CALL ShalLakeHydrology
!       |                               |- CALL BuildSnowFilter    (use SnowHydro_lake)
!       |                               |- CALL SnowWater 
!       |                               |- CALL SnowCompaction
!       |                               |- CALL CombineSnowLayers
!       |                               |- CALL DivideSnowLayers 
!       |                               |- CALL BuildSnowFilter
!       |- SUBROUTINE ShalLakeFluxes
!       |- SUBROUTINE ShalLakeTemperature
!       |- SUBROUTINE ShalLakeHydrology
!       |- SUBROUTINE SoilThermProp_Lake
!       |- SUBROUTINE PhaseChange_Lake
!       |- SUBROUTINE QSat
!       |- SUBROUTINE Tridiagonal
!       |- SUBROUTINE lakeini
!     END MODULE module_sf_lake
!=====================================================================================!
MODULE module_sf_lake

 use lake_const
 USE module_model_constants, ONLY : rcp
    real(r8) :: watsat(1,nlevsoi)      ! volumetric soil water at saturation (porosity)
    real(r8) :: tksatu(1,nlevsoi)      ! thermal conductivity, saturated soil [W/m-K]
    real(r8) :: tkmg(1,nlevsoi)        ! thermal conductivity, soil minerals  [W/m-K]
    real(r8) :: tkdry(1,nlevsoi)       ! thermal conductivity, dry soil (W/m/Kelvin)
    real(r8) :: csol(1,nlevsoi)        ! heat capacity, soil solids (J/m**3/Kelvin)
    CONTAINS
 

    SUBROUTINE Lake( t_phy        ,p8w            ,dz8w         ,qvcurr          ,&  
                     u_phy        ,v_phy          , glw         ,emiss           ,&
                     rainbl       ,dtbl           ,swdown       ,albedo          ,&
                     xlat_urb2d   ,z_lake3d       ,dz_lake3d    ,lakedepth2d     ,&
                     watsat3d     ,csol3d         ,tkmg3d       ,tkdry3d         ,&
                     tksatu3d     ,ivgtyp         ,ht           ,xland           ,& 
                     lakeminalt                                                  ,&
                     ids          ,ide            ,jds          ,jde             ,&
                     kds          ,kde            ,ims          ,ime             ,&
                     jms          ,jme            ,kms          ,kme             ,&
                     its          ,ite            ,jts          ,jte             ,&
                     kts          ,kte                                           ,&
                     h2osno2d     ,snowdp2d       ,snl2d        ,z3d             ,&  
                     dz3d         ,zi3d           ,h2osoi_vol3d ,h2osoi_liq3d    ,&
                     h2osoi_ice3d ,t_grnd2d       ,t_soisno3d   ,t_lake3d        ,&
                     savedtke12d  ,lake_icefrac3d                                ,&
                     hfx          ,lh             ,grdflx       ,tsk             ,&  
                     qfx          ,t2             ,th2          ,q2              ,&
                     ISWATER ) 
    IMPLICIT NONE
    
!in:
    
    INTEGER,  INTENT(IN   )   ::     ids,ide, jds,jde, kds,kde,  &
                                     ims,ime, jms,jme, kms,kme,  &
                                     its,ite, jts,jte, kts,kte
    
    REAL,           DIMENSION( ims:ime, kms:kme, jms:jme ),INTENT(IN)  :: t_phy  
    REAL,           DIMENSION( ims:ime, kms:kme, jms:jme ),INTENT(IN)  :: p8w    
    REAL,           DIMENSION( ims:ime, kms:kme, jms:jme ),INTENT(IN)  :: dz8w
    REAL,           DIMENSION( ims:ime, kms:kme, jms:jme ),INTENT(IN)  :: qvcurr
    REAL, OPTIONAL, DIMENSION( ims:ime, kms:kme, jms:jme ),INTENT(IN)  :: U_PHY
    REAL, OPTIONAL, DIMENSION( ims:ime, kms:kme, jms:jme ),INTENT(IN)  :: V_PHY
    REAL,           DIMENSION( ims:ime, jms:jme )         ,INTENT(IN)  :: glw
    REAL,           DIMENSION( ims:ime, jms:jme )         ,INTENT(IN)  :: emiss
    REAL,           DIMENSION( ims:ime, jms:jme )         ,INTENT(IN)  :: rainbl
    REAL,           DIMENSION( ims:ime, jms:jme )         ,INTENT(IN)  :: swdown
    REAL,           DIMENSION( ims:ime, jms:jme )         ,INTENT(IN)  :: albedo
    REAL,           DIMENSION( ims:ime, jms:jme )         ,INTENT(IN)  :: XLAND
    REAL, OPTIONAL, DIMENSION( ims:ime, jms:jme )         ,INTENT(IN)  :: XLAT_URB2D
    INTEGER,        DIMENSION( ims:ime, jms:jme )         ,INTENT(IN)  :: IVGTYP
    REAL,                                                  INTENT(IN)  :: dtbl
    
    REAL,           DIMENSION( ims:ime,1:nlevlak,jms:jme ),INTENT(IN)  :: z_lake3d
    REAL,           DIMENSION( ims:ime,1:nlevlak,jms:jme ),INTENT(IN)  :: dz_lake3d
    REAL,           DIMENSION( ims:ime,1:nlevsoi,jms:jme ),INTENT(IN)  :: watsat3d
    REAL,           DIMENSION( ims:ime,1:nlevsoi,jms:jme ),INTENT(IN)  :: csol3d
    REAL,           DIMENSION( ims:ime,1:nlevsoi,jms:jme ),INTENT(IN)  :: tkmg3d
    REAL,           DIMENSION( ims:ime,1:nlevsoi,jms:jme ),INTENT(IN)  :: tkdry3d
    REAL,           DIMENSION( ims:ime,1:nlevsoi,jms:jme ),INTENT(IN)  :: tksatu3d
    REAL,           DIMENSION( ims:ime, jms:jme )         ,INTENT(IN)  :: lakedepth2d    
    REAL,           DIMENSION( ims:ime, jms:jme )         ,INTENT(IN)  :: ht
    REAL                                                  ,INTENT(IN)  :: lakeminalt

!out:
    REAL,           DIMENSION( ims:ime, jms:jme )         ,INTENT(OUT) :: HFX
    REAL,           DIMENSION( ims:ime, jms:jme )         ,INTENT(OUT) :: LH
    REAL,           DIMENSION( ims:ime, jms:jme )         ,INTENT(OUT) :: GRDFLX
    REAL,           DIMENSION( ims:ime, jms:jme )         ,INTENT(OUT) :: TSK
    REAL,           DIMENSION( ims:ime, jms:jme )         ,INTENT(OUT) :: QFX   
    REAL,           DIMENSION( ims:ime, jms:jme )         ,INTENT(OUT) :: T2
    REAL,           DIMENSION( ims:ime, jms:jme )         ,INTENT(OUT) :: TH2
    REAL,           DIMENSION( ims:ime, jms:jme )         ,INTENT(OUT) :: Q2

!in&out:

    real,           dimension(ims:ime,jms:jme )                ,intent(inout)  :: savedtke12d 
    real,           dimension(ims:ime,jms:jme )                ,intent(inout)  :: snowdp2d,       &    
                                                                                  h2osno2d,       &    
                                                                                  snl2d,          &    
                                                                                  t_grnd2d
    
    real,    dimension( ims:ime,1:nlevlak, jms:jme )           ,INTENT(inout)  :: t_lake3d,       &    
                                                                                  lake_icefrac3d
    real,    dimension( ims:ime,-nlevsno+1:nlevsoi, jms:jme )  ,INTENT(inout)  :: t_soisno3d,     &    
                                                                                  h2osoi_ice3d,   &    
                                                                                  h2osoi_liq3d,   &    
                                                                                  h2osoi_vol3d,   &    
                                                                                  z3d,            &    
                                                                                  dz3d 
    real,    dimension( ims:ime,-nlevsno+0:nlevsoi, jms:jme )  ,INTENT(inout)  :: zi3d    
    integer, intent(in) :: ISWATER
       

!local variable:

    REAL     :: SFCTMP,PBOT,PSFC,ZLVL,Q2K,EMISSI,LWDN,PRCP,SOLDN,SOLNET
    INTEGER  :: C,i,j,k
    LOGICAL  :: lakeprint


      !tempory varibles in:
      real(r8)  :: forc_t(1)          ! atmospheric temperature (Kelvin)
      real(r8)  :: forc_pbot(1)       ! atm bottom level pressure (Pa) 
      real(r8)  :: forc_psrf(1)       ! atmospheric surface pressure (Pa)
      real(r8)  :: forc_hgt(1)        ! atmospheric reference height (m)
      real(r8)  :: forc_hgt_q(1)      ! observational height of humidity [m]
      real(r8)  :: forc_hgt_t(1)      ! observational height of temperature [m]
      real(r8)  :: forc_hgt_u(1)      ! observational height of wind [m]
      real(r8)  :: forc_q(1)          ! atmospheric specific humidity (kg/kg)
      real(r8)  :: forc_u(1)          ! atmospheric wind speed in east direction (m/s)
      real(r8)  :: forc_v(1)          ! atmospheric wind speed in north direction (m/s)
     ! real(r8)  :: forc_rho(1)        ! density (kg/m**3)
      real(r8)  :: forc_lwrad(1)      ! downward infrared (longwave) radiation (W/m**2)
      real(r8)  :: prec(1)               ! snow or rain rate [mm/s]
      real(r8)  :: sabg(1)            ! solar radiation absorbed by ground (W/m**2)
      real(r8)  :: lat(1)             ! latitude (radians)
      real(r8)  :: z_lake(1,nlevlak)  ! layer depth for lake (m)
      real(r8)  :: dz_lake(1,nlevlak)                  ! layer thickness for lake (m)

      real(r8)  :: lakedepth(1)       ! column lake depth (m)
      logical   :: do_capsnow(1)     ! true => do snow capping

      !in&out
      real(r8)  :: h2osoi_vol(1,-nlevsno+1:nlevsoi)  ! volumetric soil water (0<=h2osoi_vol<=watsat)[m3/m3]
      real(r8)  :: t_grnd(1)          ! ground temperature (Kelvin)
      real(r8)  :: h2osno(1)          ! snow water (mm H2O)
      real(r8)  :: snowdp(1)          ! snow height (m)
      real(r8)  :: z(1,-nlevsno+1:nlevsoi)             ! layer depth for snow & soil (m)
      real(r8)  :: dz(1,-nlevsno+1:nlevsoi)            ! layer thickness for soil or snow (m)
      real(r8)  :: t_soisno(1,-nlevsno+1:nlevsoi)      ! soil (or snow) temperature (Kelvin)
      real(r8)  :: t_lake(1,nlevlak)                   ! lake temperature (Kelvin)
      integer   :: snl(1)                              ! number of snow layers
      real(r8)  :: h2osoi_liq(1,-nlevsno+1:nlevsoi)    ! liquid water (kg/m2)
      real(r8)  :: h2osoi_ice(1,-nlevsno+1:nlevsoi)    ! ice lens (kg/m2)
      real(r8)  :: savedtke1(1)       ! top level eddy conductivity from previous timestep (W/m.K)
      real(r8)  :: zi(1,-nlevsno+0:nlevsoi)            ! interface level below a "z" level (m)
      real(r8)  :: lake_icefrac(1,nlevlak)  ! mass fraction of lake layer that is frozen


      !out:
      real(r8)  :: eflx_gnet(1)       !net heat flux into ground (W/m**2)
      real(r8)  :: eflx_lwrad_net(1)  ! net infrared (longwave) rad (W/m**2) [+ = to atm]
      real(r8)  :: eflx_sh_tot(1)     ! total sensible heat flux (W/m**2) [+ to atm]
      real(r8)  :: eflx_lh_tot(1)     ! total latent heat flux (W/m8*2)  [+ to atm]
      real(r8)  :: t_ref2m(1)         ! 2 m height surface air temperature (Kelvin)
      real(r8)  :: q_ref2m(1)         ! 2 m height surface specific humidity (kg/kg)
      real(r8)  :: taux(1)            ! wind (shear) stress: e-w (kg/m/s**2)
      real(r8)  :: tauy(1)            ! wind (shear) stress: n-s (kg/m/s**2)
      real(r8)  :: ram1(1)            ! aerodynamical resistance (s/m)
                                               ! for calculation of decay of eddy diffusivity with depth
                                               ! Change the type variable to pass back to WRF.
      real(r8)  :: z0mg(1)            ! roughness length over ground, momentum (m(


      dtime = dtbl
      lakeprint = .false.       

        DO J = jts,jte
        DO I = its,ite

           SFCTMP  = t_phy(i,1,j)
           PBOT    = p8w(i,2,j)
           PSFC    = P8w(i,1,j) 
           ZLVL    = 0.5 * dz8w(i,1,j) 
           Q2K     = qvcurr(i,1,j)/(1.0 + qvcurr(i,1,j))
           EMISSI  = EMISS(I,J) 
           LWDN    = GLW(I,J)*EMISSI 
           PRCP    = RAINBL(i,j)/dtbl
           SOLDN   = SWDOWN(I,J)                        ! SOLDN is total incoming solar
           SOLNET  = SOLDN*(1.-ALBEDO(I,J))             ! use mid-day albedo to determine net downward solar
                                                        ! (no solar zenith angle correction) 
        IF (XLAND(I,J).GT.1.5) THEN    
        if (ivgtyp(i,j)==ISWATER.and.ht(i,j)>= lakeminalt ) THEN
    
           do c = 1,column
     
            forc_t(c)          = SFCTMP           ! [K]
            forc_pbot(c)       = PBOT 
            forc_psrf(c)       = PSFC
            forc_hgt(c)        = ZLVL             ! [m]
            forc_hgt_q(c)      = ZLVL             ! [m]
            forc_hgt_t(c)      = ZLVL             ! [m]
            forc_hgt_u(c)      = ZLVL             ! [m]
            forc_q(c)          = Q2K              ! [kg/kg]
            forc_u(c)          = U_PHY(I,1,J)
            forc_v(c)          = V_PHY(I,1,J)
           ! forc_rho(c)        = SFCPRS / (287.04 * SFCTMP * (1.0+ 0.61 * Q2K)) ![kg/m/m/m] 
            forc_lwrad(c)      = LWDN             ! [W/m/m]
            prec(c)            = PRCP             ! [mm/s]
            sabg(c)            = SOLNET
            lat(c)             = XLAT_URB2D(I,J)*pie/180  ! [radian] 
            do_capsnow(c)      = .false.

            lakedepth(c)           = lakedepth2d(i,j)
            savedtke1(c)           = savedtke12d(i,j)
            snowdp(c)              = snowdp2d(i,j)
            h2osno(c)              = h2osno2d(i,j)
            snl(c)                 = snl2d(i,j)
            t_grnd(c)              = t_grnd2d(i,j)
            do k = 1,nlevlak
               t_lake(c,k)        = t_lake3d(i,k,j)
               lake_icefrac(c,k)  = lake_icefrac3d(i,k,j)
               z_lake(c,k)        = z_lake3d(i,k,j)
               dz_lake(c,k)       = dz_lake3d(i,k,j)
            enddo
            do k = -nlevsno+1,nlevsoi
               t_soisno(c,k)      = t_soisno3d(i,k,j)
	       h2osoi_ice(c,k)    = h2osoi_ice3d(i,k,j)
               h2osoi_liq(c,k)    = h2osoi_liq3d(i,k,j)
               h2osoi_vol(c,k)    = h2osoi_vol3d(i,k,j)
               z(c,k)             = z3d(i,k,j)
               dz(c,k)            = dz3d(i,k,j)
            enddo   
            do k = -nlevsno+0,nlevsoi
               zi(c,k)            = zi3d(i,k,j)
            enddo
            do k = 1,nlevsoi
               watsat(c,k)        = watsat3d(i,k,j)
               csol(c,k)          = csol3d(i,k,j)
               tkmg(c,k)          = tkmg3d(i,k,j)
               tkdry(c,k)         = tkdry3d(i,k,j)
               tksatu(c,k)        = tksatu3d(i,k,j)
            enddo
            
            if ( lakeprint ) then
               print *,"before call LakeMain :"
               print *,"t_soisno = ", t_soisno 
               print *,"sabg = ",sabg
            endif
          enddo
            CALL LakeMain(forc_t,forc_pbot,forc_psrf,forc_hgt,forc_hgt_q,   & !I  
                          forc_hgt_t,forc_hgt_u,forc_q, forc_u,         &
                          forc_v,forc_lwrad,prec, sabg,lat,             &
                          z_lake,dz_lake,lakedepth,do_capsnow,          &
                          h2osno,snowdp,snl,z,dz,zi,                    & !H
                          h2osoi_vol,h2osoi_liq,h2osoi_ice,             &
                          t_grnd,t_soisno,t_lake,                       &
                          savedtke1,lake_icefrac,                       &
                          eflx_lwrad_net,eflx_gnet,                     & !O 
                          eflx_sh_tot,eflx_lh_tot,                      &
                          t_ref2m,q_ref2m,                              &
                          taux,tauy,ram1,z0mg)


           do c = 1,column
            HFX(I,J)          = eflx_sh_tot(c)            ![W/m/m]
            LH(I,J)           = eflx_lh_tot(c)            !W/m/m]
            GRDFLX(I,J)       = eflx_gnet(c)              !W/m/m]
            TSK(I,J)          = t_grnd(c)                 ![K]
            T2(I,J)           = t_ref2m(c)
            TH2(I,J)          = T2(I,J)*(1.E5/PSFC)**RCP
            Q2(I,J)           = q_ref2m(c) 

            if( tsk(i,j) >= tfrz ) then
                qfx(i,j)      = eflx_lh_tot(c)/hvap
            else
                qfx(i,j)      = eflx_lh_tot(c)/hsub       ! heat flux (W/m^2)=>mass flux(kg/(sm^2))
            endif
           enddo

! Renew Lake State Varialbes:(14)
           do c = 1,column

            savedtke12d(i,j)         = savedtke1(c)
            snowdp2d(i,j)            = snowdp(c)
            h2osno2d(i,j)            = h2osno(c)
	    snl2d(i,j)               = snl(c)
            t_grnd2d(i,j)            = t_grnd(c)
            do k = 1,nlevlak
               t_lake3d(i,k,j)       = t_lake(c,k)
	       lake_icefrac3d(i,k,j) = lake_icefrac(c,k)
            enddo
	    do k = -nlevsno+1,nlevsoi
	       z3d(i,k,j)            = z(c,k)
	       dz3d(i,k,j)           = dz(c,k) 
	       t_soisno3d(i,k,j)     = t_soisno(c,k)
	       h2osoi_liq3d(i,k,j)   = h2osoi_liq(c,k)
	       h2osoi_ice3d(i,k,j)   = h2osoi_ice(c,k)
               h2osoi_vol3d(i,k,j)   = h2osoi_vol(c,k)
	   enddo
           do k = -nlevsno+0,nlevsoi
               zi3d(i,k,j)           = zi(c,k)
           enddo
        
         enddo

        endif
        ENDIF    ! if xland = 2
        ENDDO
        ENDDO

    END SUBROUTINE Lake


    SUBROUTINE LakeMain(forc_t,forc_pbot,forc_psrf,forc_hgt,forc_hgt_q,     &   
                          forc_hgt_t,forc_hgt_u,forc_q, forc_u,         &   
                          forc_v,forc_lwrad,prec, sabg,lat,             &   
                          z_lake,dz_lake,lakedepth,do_capsnow,          &
                          h2osno,snowdp,snl,z,dz,zi,                    & 
                          h2osoi_vol,h2osoi_liq,h2osoi_ice,             &
                          t_grnd,t_soisno,t_lake,                       &  
                          savedtke1,lake_icefrac,                       &
                          eflx_lwrad_net,eflx_gnet,                     &  
                          eflx_sh_tot,eflx_lh_tot,                      &
                          t_ref2m,q_ref2m,                              &
                          taux,tauy,ram1,z0mg)
    implicit none
!in: 

    real(r8),intent(in) :: forc_t(1)          ! atmospheric temperature (Kelvin)
    real(r8),intent(in) :: forc_pbot(1)       ! atm bottom level pressure (Pa) 
    real(r8),intent(in) :: forc_psrf(1)       ! atmospheric surface pressure (Pa)
    real(r8),intent(in) :: forc_hgt(1)        ! atmospheric reference height (m)
    real(r8),intent(in) :: forc_hgt_q(1)      ! observational height of humidity [m]
    real(r8),intent(in) :: forc_hgt_t(1)      ! observational height of temperature [m]
    real(r8),intent(in) :: forc_hgt_u(1)      ! observational height of wind [m]
    real(r8),intent(in) :: forc_q(1)          ! atmospheric specific humidity (kg/kg)
    real(r8),intent(in) :: forc_u(1)          ! atmospheric wind speed in east direction (m/s)
    real(r8),intent(in) :: forc_v(1)          ! atmospheric wind speed in north direction (m/s)
   ! real(r8),intent(in) :: forc_rho(1)        ! density (kg/m**3)
    real(r8),intent(in) :: forc_lwrad(1)      ! downward infrared (longwave) radiation (W/m**2)
    real(r8),intent(in) :: prec(1)               ! snow or rain rate [mm/s]
    real(r8),intent(in) :: sabg(1)            ! solar radiation absorbed by ground (W/m**2)
    real(r8),intent(in) :: lat(1)             ! latitude (radians)
    real(r8),intent(in) :: z_lake(1,nlevlak)  ! layer depth for lake (m)
    real(r8),intent(in) :: dz_lake(1,nlevlak)                  ! layer thickness for lake (m)
    real(r8), intent(in) :: lakedepth(1)       ! column lake depth (m)
    logical , intent(in) :: do_capsnow(1)     ! true => do snow capping
   


!in&out
    real(r8),intent(inout) :: h2osoi_vol(1,-nlevsno+1:nlevsoi)  ! volumetric soil water (0<=h2osoi_vol<=watsat)[m3/m3]
    real(r8),intent(inout) :: t_grnd(1)          ! ground temperature (Kelvin)
    real(r8),intent(inout) :: h2osno(1)          ! snow water (mm H2O)
    real(r8),intent(inout) :: snowdp(1)          ! snow height (m)
    real(r8),intent(inout) :: z(1,-nlevsno+1:nlevsoi)             ! layer depth for snow & soil (m)
    real(r8),intent(inout) :: dz(1,-nlevsno+1:nlevsoi)            ! layer thickness for soil or snow (m)
    real(r8),intent(inout) :: t_soisno(1,-nlevsno+1:nlevsoi)      ! soil (or snow) temperature (Kelvin)
    real(r8),intent(inout) :: t_lake(1,nlevlak)                   ! lake temperature (Kelvin)
    integer ,intent(inout) :: snl(1)                              ! number of snow layers
    real(r8),intent(inout) :: h2osoi_liq(1,-nlevsno+1:nlevsoi)    ! liquid water (kg/m2)
    real(r8),intent(inout) :: h2osoi_ice(1,-nlevsno+1:nlevsoi)    ! ice lens (kg/m2)
    real(r8),intent(inout) :: savedtke1(1)       ! top level eddy conductivity from previous timestep (W/m.K)
    real(r8),intent(inout) :: zi(1,-nlevsno+0:nlevsoi)            ! interface level below a "z" level (m)
    real(r8),intent(inout) :: lake_icefrac(1,nlevlak)  ! mass fraction of lake layer that is frozen


!out:
    real(r8),intent(out) :: eflx_gnet(1)       !net heat flux into ground (W/m**2)
    real(r8),intent(out) :: eflx_lwrad_net(1)  ! net infrared (longwave) rad (W/m**2) [+ = to atm]
    real(r8),intent(out) :: eflx_sh_tot(1)     ! total sensible heat flux (W/m**2) [+ to atm]
    real(r8),intent(out) :: eflx_lh_tot(1)     ! total latent heat flux (W/m8*2)  [+ to atm]
    real(r8),intent(out) :: t_ref2m(1)         ! 2 m height surface air temperature (Kelvin)
    real(r8),intent(out) :: q_ref2m(1)         ! 2 m height surface specific humidity (kg/kg)
    real(r8),intent(out) :: taux(1)            ! wind (shear) stress: e-w (kg/m/s**2)
    real(r8),intent(out) :: tauy(1)            ! wind (shear) stress: n-s (kg/m/s**2)
    real(r8),intent(out) :: ram1(1)            ! aerodynamical resistance (s/m)
                                               ! for calculation of decay of eddy diffusivity with depth
                                               ! Change the type variable to pass back to WRF.
    real(r8),intent(out) :: z0mg(1)            ! roughness length over ground, momentum (m(


!local output
    
    real(r8) :: begwb(1)           ! water mass begining of the time step
    real(r8) :: t_veg(1)           ! vegetation temperature (Kelvin)
    real(r8) :: eflx_soil_grnd(1)  ! soil heat flux (W/m**2) [+ = into soil]
    real(r8) :: eflx_lh_grnd(1)    ! ground evaporation heat flux (W/m**2) [+ to atm]
    real(r8) :: eflx_sh_grnd(1)    ! sensible heat flux from ground (W/m**2) [+ to atm]
    real(r8) :: eflx_lwrad_out(1)  ! emitted infrared (longwave) radiation (W/m**2)
    real(r8) :: qflx_evap_tot(1)   ! qflx_evap_soi + qflx_evap_veg + qflx_tran_veg
    real(r8) :: qflx_evap_soi(1)   ! soil evaporation (mm H2O/s) (+ = to atm)
    real(r8) :: qflx_prec_grnd(1)  ! water onto ground including canopy runoff [kg/(m2 s)]
    real(r8) :: forc_snow(1)       ! snow rate [mm/s]
    real(r8) :: forc_rain(1)       ! rain rate [mm/s]
    real(r8) :: ws(1)              ! surface friction velocity (m/s)
    real(r8) :: ks(1)              ! coefficient passed to ShalLakeTemperature
    real(r8) :: qflx_snomelt(1)    !snow melt (mm H2O /s) tem(out),snowwater(in)
    integer  :: imelt(1,-nlevsno+1:nlevsoi)      !flag for melting (=1), freezing (=2), Not=0 (new)
    real(r8) :: endwb(1)         ! water mass end of the time step
    real(r8) :: snowage(1)       ! non dimensional snow age [-]
    real(r8) :: snowice(1)       ! average snow ice lens
    real(r8) :: snowliq(1)       ! average snow liquid water
    real(r8) :: t_snow(1)        ! vertically averaged snow temperature
    real(r8) :: qflx_drain(1)    ! sub-surface runoff (mm H2O /s)
    real(r8) :: qflx_surf(1)     ! surface runoff (mm H2O /s)
    real(r8) :: qflx_infl(1)     ! infiltration (mm H2O /s)
    real(r8) :: qflx_qrgwl(1)    ! qflx_surf at glaciers, wetlands, lakes
    real(r8) :: qcharge(1)       ! aquifer recharge rate (mm/s)
    real(r8) :: qflx_snowcap(1)       ! excess precipitation due to snow capping (mm H2O /s) [+]
    real(r8) :: qflx_snowcap_col(1)   ! excess precipitation due to snow capping (mm H2O /s) [+]
    real(r8) :: qflx_snow_grnd_pft(1) ! snow on ground after interception (mm H2O/s) [+]
    real(r8) :: qflx_snow_grnd_col(1) ! snow on ground after interception (mm H2O/s) [+]
    real(r8) :: qflx_rain_grnd(1)     ! rain on ground after interception (mm H2O/s) [+]
    real(r8) :: frac_iceold(1,-nlevsno+1:nlevsoi)      ! fraction of ice relative to the tot water
    real(r8) :: qflx_evap_tot_col(1) !pft quantity averaged to the column (assuming one pft)
    real(r8) :: soilalpha(1)     !factor that reduces ground saturated specific humidity (-)
    real(r8) :: zwt(1)           !water table depth
    real(r8) :: fcov(1)          !fractional area with water table at surface
    real(r8) :: rootr_column(1,1:nlevsoi) !effective fraction of roots in each soil layer
    real(r8) :: qflx_evap_grnd(1)  ! ground surface evaporation rate (mm H2O/s) [+]
    real(r8) :: qflx_sub_snow(1)   ! sublimation rate from snow pack (mm H2O /s) [+]
    real(r8) :: qflx_dew_snow(1)   ! surface dew added to snow pack (mm H2O /s) [+]
    real(r8) :: qflx_dew_grnd(1)   ! ground surface dew formation (mm H2O /s) [+]
    real(r8) :: qflx_rain_grnd_col(1)   !rain on ground after interception (mm H2O/s) [+]
    
!for debug:
    logical  :: ifprint
    ifprint = .false.

!    lat  = lat*pie/180  ! [radian]

    if (prec(1)> 0.) then
        if ( forc_t(1) > (tfrz + tcrit)) then
            forc_rain(1) = prec(1)
            forc_snow(1) = 0.
          !   flfall(1) = 1.
         else
            forc_rain(1) = 0.
            forc_snow(1) = prec(1)

          !  if ( forc_t(1) <= tfrz) then
          !      flfall(1) = 0.
          !  else if ( forc_t(1) <= tfrz+2.) then
          !      flfall(1) = -54.632 + 0.2 *  forc_t(1)
          !  else
          !      flfall(1) = 0.4
         endif
    else
         forc_rain(1) = 0.
         forc_snow(1) = 0.
       !  flfall(1) = 1.
    endif

    CALL ShalLakeFluxes(forc_t,forc_pbot,forc_psrf,forc_hgt,forc_hgt_q,   &  
                          forc_hgt_t,forc_hgt_u,forc_q,                   &
                          forc_u,forc_v,forc_lwrad,forc_snow,             &
                          forc_rain,t_grnd,h2osno,snowdp,sabg,lat,        &
                          dz,dz_lake,t_soisno,t_lake,snl,h2osoi_liq,      &
                          h2osoi_ice,savedtke1,                           &
                          qflx_prec_grnd,qflx_evap_soi,qflx_evap_tot,     &  
                          eflx_sh_grnd,eflx_lwrad_out,eflx_lwrad_net,     &
                          eflx_soil_grnd,eflx_sh_tot,eflx_lh_tot,         &
                          eflx_lh_grnd,t_veg,t_ref2m,q_ref2m,taux,tauy,   &
                          ram1,ws,ks,eflx_gnet,z0mg)
 

    CALL ShalLakeTemperature(t_grnd,h2osno,sabg,dz,dz_lake,z,zi,             & 
                                 z_lake,ws,ks,snl,eflx_gnet,lakedepth,       &
                                 lake_icefrac,snowdp,                        & 
                                 eflx_sh_grnd,eflx_sh_tot,eflx_soil_grnd,    & 
                                 t_lake,t_soisno,h2osoi_liq,                 &
                                 h2osoi_ice,savedtke1,                       &
                                 frac_iceold,qflx_snomelt,imelt)



    CALL ShalLakeHydrology(dz_lake,forc_rain,forc_snow,                          & 
                               begwb,qflx_evap_tot,forc_t,do_capsnow,            &
                               t_grnd,qflx_evap_soi,                             &
                               qflx_snomelt,imelt,frac_iceold,                   & 
                               z,dz,zi,snl,h2osno,snowdp,lake_icefrac,t_lake,    & 
                               endwb,snowage,snowice,snowliq,t_snow,             & 
                               t_soisno,h2osoi_ice,h2osoi_liq,h2osoi_vol,        &
                               qflx_drain,qflx_surf,qflx_infl,qflx_qrgwl,        &
                               qcharge,qflx_prec_grnd,qflx_snowcap,              &
                               qflx_snowcap_col,qflx_snow_grnd_pft,              &
                               qflx_snow_grnd_col,qflx_rain_grnd,                &
                               qflx_evap_tot_col,soilalpha,zwt,fcov,             &
                               rootr_column,qflx_evap_grnd,qflx_sub_snow,        &
                               qflx_dew_snow,qflx_dew_grnd,qflx_rain_grnd_col)
                       
!==================================================================================
! !DESCRIPTION:
! Calculation of Shallow Lake Hydrology. Full hydrology of snow layers is
! done. However, there is no infiltration, and the water budget is balanced with 
                       
   END SUBROUTINE LakeMain


SUBROUTINE ShalLakeFluxes(forc_t,forc_pbot,forc_psrf,forc_hgt,forc_hgt_q,           &  !i
                          forc_hgt_t,forc_hgt_u,forc_q,                   &
                          forc_u,forc_v,forc_lwrad,forc_snow,             &
                          forc_rain,t_grnd,h2osno,snowdp,sabg,lat,        &
                          dz,dz_lake,t_soisno,t_lake,snl,h2osoi_liq,      &
                          h2osoi_ice,savedtke1,                           &
                          qflx_prec_grnd,qflx_evap_soi,qflx_evap_tot,     &  !o
                          eflx_sh_grnd,eflx_lwrad_out,eflx_lwrad_net,     &
                          eflx_soil_grnd,eflx_sh_tot,eflx_lh_tot,         &
                          eflx_lh_grnd,t_veg,t_ref2m,q_ref2m,taux,tauy,   &
                          ram1,ws,ks,eflx_gnet,z0mg)            
!==============================================================================
! DESCRIPTION:
! Calculates lake temperatures and surface fluxes for shallow lakes.
!
! Shallow lakes have variable depth, possible snow layers above, freezing & thawing of lake water,
! and soil layers with active temperature and gas diffusion below.
!
! WARNING: This subroutine assumes lake columns have one and only one pft.
!
! REVISION HISTORY:
! Created by Zack Subin, 2009
! Changed by guhp for coupling,2010
!
!==============================================================================
    use lake_const       
    use FrictionVeloc_lake , only : FrictionVelocity, MoninObukIni
    implicit none

!in: 

    real(r8),intent(in) :: forc_t(1)          ! atmospheric temperature (Kelvin)
    real(r8),intent(in) :: forc_pbot(1)       ! atmospheric pressure (Pa)
    real(r8),intent(in) :: forc_psrf(1)       ! atmospheric surface pressure (Pa)
    real(r8),intent(in) :: forc_hgt(1)        ! atmospheric reference height (m)
    real(r8),intent(in) :: forc_hgt_q(1)      ! observational height of humidity [m]
    real(r8),intent(in) :: forc_hgt_t(1)      ! observational height of temperature [m]
    real(r8),intent(in) :: forc_hgt_u(1)      ! observational height of wind [m]
    real(r8),intent(in) :: forc_q(1)          ! atmospheric specific humidity (kg/kg)
    real(r8),intent(in) :: forc_u(1)          ! atmospheric wind speed in east direction (m/s)
    real(r8),intent(in) :: forc_v(1)          ! atmospheric wind speed in north direction (m/s)
    real(r8),intent(in) :: forc_lwrad(1)      ! downward infrared (longwave) radiation (W/m**2)
   ! real(r8),intent(in) :: forc_rho(1)        ! density (kg/m**3)
    real(r8),intent(in) :: forc_snow(1)       ! snow rate [mm/s]
    real(r8),intent(in) :: forc_rain(1)       ! rain rate [mm/s]
    real(r8),intent(in) :: h2osno(1)          ! snow water (mm H2O)
    real(r8),intent(in) :: snowdp(1)          ! snow height (m)
    real(r8),intent(in) :: sabg(1)            ! solar radiation absorbed by ground (W/m**2)
    real(r8),intent(in) :: lat(1)             ! latitude (radians)
    real(r8),intent(in) :: dz(1,-nlevsno+1:nlevsoi)            ! layer thickness for soil or snow (m)
    real(r8),intent(in) :: dz_lake(1,nlevlak)                  ! layer thickness for lake (m)
    real(r8),intent(in) :: t_soisno(1,-nlevsno+1:nlevsoi)      ! soil (or snow) temperature (Kelvin)
    real(r8),intent(in) :: t_lake(1,nlevlak)                   ! lake temperature (Kelvin)
    integer ,intent(in) :: snl(1)                              ! number of snow layers
    real(r8),intent(in) :: h2osoi_liq(1,-nlevsno+1:nlevsoi)    ! liquid water (kg/m2)
    real(r8),intent(in) :: h2osoi_ice(1,-nlevsno+1:nlevsoi)    ! ice lens (kg/m2)
    real(r8),intent(in) :: savedtke1(1)       ! top level eddy conductivity from previous timestep (W/m.K)

!inout:
    real(r8),intent(inout) :: t_grnd(1)          ! ground temperature (Kelvin)
!out:
    real(r8),intent(out):: qflx_prec_grnd(1)  ! water onto ground including canopy runoff [kg/(m2 s)]
    real(r8),intent(out):: qflx_evap_soi(1)   ! soil evaporation (mm H2O/s) (+ = to atm)
    real(r8),intent(out):: qflx_evap_tot(1)   ! qflx_evap_soi + qflx_evap_veg + qflx_tran_veg
    real(r8),intent(out):: eflx_sh_grnd(1)    ! sensible heat flux from ground (W/m**2) [+ to atm]
    real(r8),intent(out):: eflx_lwrad_out(1)  ! emitted infrared (longwave) radiation (W/m**2)
    real(r8),intent(out):: eflx_lwrad_net(1)  ! net infrared (longwave) rad (W/m**2) [+ = to atm]
    real(r8),intent(out):: eflx_soil_grnd(1)  ! soil heat flux (W/m**2) [+ = into soil]
    real(r8),intent(out):: eflx_sh_tot(1)     ! total sensible heat flux (W/m**2) [+ to atm]
    real(r8),intent(out):: eflx_lh_tot(1)     ! total latent heat flux (W/m8*2)  [+ to atm]
    real(r8),intent(out):: eflx_lh_grnd(1)    ! ground evaporation heat flux (W/m**2) [+ to atm]
    real(r8),intent(out):: t_veg(1)           ! vegetation temperature (Kelvin)
    real(r8),intent(out):: t_ref2m(1)         ! 2 m height surface air temperature (Kelvin)
    real(r8),intent(out):: q_ref2m(1)         ! 2 m height surface specific humidity (kg/kg)
    real(r8),intent(out):: taux(1)            ! wind (shear) stress: e-w (kg/m/s**2)
    real(r8),intent(out):: tauy(1)            ! wind (shear) stress: n-s (kg/m/s**2)
    real(r8),intent(out):: ram1(1)            ! aerodynamical resistance (s/m)
    real(r8),intent(out):: ws(1)              ! surface friction velocity (m/s)
    real(r8),intent(out):: ks(1)              ! coefficient passed to ShalLakeTemperature
                                               ! for calculation of decay of eddy diffusivity with depth
    real(r8),intent(out):: eflx_gnet(1)       !net heat flux into ground (W/m**2)
                                               ! Change the type variable to pass back to WRF.
    real(r8),intent(out):: z0mg(1)            ! roughness length over ground, momentum (m(



!OTHER LOCAL VARIABLES:

    integer , parameter :: islak  = 2       ! index of lake, 1 = deep lake, 2 = shallow lake
    integer , parameter :: niters = 3       ! maximum number of iterations for surface temperature
    real(r8), parameter :: beta1  = 1._r8   ! coefficient of convective velocity (in computing W_*) [-]
    real(r8), parameter :: emg    = 0.97_r8 ! ground emissivity (0.97 for snow)
    real(r8), parameter :: zii    = 1000._r8! convective boundary height [m]
    real(r8), parameter :: tdmax  = 277._r8 ! temperature of maximum water density
    real(r8) :: forc_th(1)         ! atmospheric potential temperature (Kelvin)
    real(r8) :: forc_vp(1)         !atmospheric vapor pressure (Pa)
    real(r8) :: forc_rho(1)        ! density (kg/m**3)
    integer  :: i,fc,fp,g,c,p           ! do loop or array index
    integer  :: fncopy                  ! number of values in pft filter copy
    integer  :: fnold                   ! previous number of pft filter values
    integer  :: fpcopy(num_shlakep)     ! pft filter copy for iteration loop
    integer  :: iter                    ! iteration index
    integer  :: nmozsgn(lbp:ubp)        ! number of times moz changes sign
    integer  :: jtop(lbc:ubc)           ! top level for each column (no longer all 1)
!    real(r8) :: dtime                   ! land model time step (sec)
    real(r8) :: ax                      ! used in iteration loop for calculating t_grnd (numerator of NR solution)
    real(r8) :: bx                      ! used in iteration loop for calculating t_grnd (denomin. of NR solution)
    real(r8) :: degdT                   ! d(eg)/dT
    real(r8) :: dqh(lbp:ubp)            ! diff of humidity between ref. height and surface
    real(r8) :: dth(lbp:ubp)            ! diff of virtual temp. between ref. height and surface
    real(r8) :: dthv                    ! diff of vir. poten. temp. between ref. height and surface
    real(r8) :: dzsur(lbc:ubc)          ! 1/2 the top layer thickness (m)
    real(r8) :: eg                      ! water vapor pressure at temperature T [pa]
    real(r8) :: htvp(lbc:ubc)           ! latent heat of vapor of water (or sublimation) [j/kg]
    real(r8) :: obu(lbp:ubp)            ! monin-obukhov length (m)
    real(r8) :: obuold(lbp:ubp)         ! monin-obukhov length of previous iteration
    real(r8) :: qsatg(lbc:ubc)          ! saturated humidity [kg/kg]
    real(r8) :: qsatgdT(lbc:ubc)        ! d(qsatg)/dT
    real(r8) :: qstar                   ! moisture scaling parameter
    real(r8) :: ram(lbp:ubp)            ! aerodynamical resistance [s/m]
    real(r8) :: rah(lbp:ubp)            ! thermal resistance [s/m]
    real(r8) :: raw(lbp:ubp)            ! moisture resistance [s/m]
    real(r8) :: stftg3(lbp:ubp)         ! derivative of fluxes w.r.t ground temperature
    real(r8) :: temp1(lbp:ubp)          ! relation for potential temperature profile
    real(r8) :: temp12m(lbp:ubp)        ! relation for potential temperature profile applied at 2-m
    real(r8) :: temp2(lbp:ubp)          ! relation for specific humidity profile
    real(r8) :: temp22m(lbp:ubp)        ! relation for specific humidity profile applied at 2-m
    real(r8) :: tgbef(lbc:ubc)          ! initial ground temperature
    real(r8) :: thm(lbc:ubc)            ! intermediate variable (forc_t+0.0098*forc_hgt_t)
    real(r8) :: thv(lbc:ubc)            ! virtual potential temperature (kelvin)
    real(r8) :: thvstar                 ! virtual potential temperature scaling parameter
    real(r8) :: tksur                   ! thermal conductivity of snow/soil (w/m/kelvin)
    real(r8) :: tsur                    ! top layer temperature
    real(r8) :: tstar                   ! temperature scaling parameter
    real(r8) :: um(lbp:ubp)             ! wind speed including the stablity effect [m/s]
    real(r8) :: ur(lbp:ubp)             ! wind speed at reference height [m/s]
    real(r8) :: ustar(lbp:ubp)          ! friction velocity [m/s]
    real(r8) :: wc                      ! convective velocity [m/s]
    real(r8) :: zeta                    ! dimensionless height used in Monin-Obukhov theory
    real(r8) :: zldis(lbp:ubp)          ! reference height "minus" zero displacement height [m]
    real(r8) :: displa(lbp:ubp)         ! displacement (always zero) [m]
!    real(r8) :: z0mg(lbp:ubp)           ! roughness length over ground, momentum [m]
    real(r8) :: z0hg(lbp:ubp)           ! roughness length over ground, sensible heat [m]
    real(r8) :: z0qg(lbp:ubp)           ! roughness length over ground, latent heat [m]
    real(r8) :: beta(2)                 ! fraction solar rad absorbed at surface: depends on lake type
    real(r8) :: u2m                     ! 2 m wind speed (m/s)
    real(r8) :: u10(1)         ! 10-m wind (m/s) (for dust model)
    real(r8) :: fv(1)          ! friction velocity (m/s) (for dust model)

    real(r8) :: fm(lbp:ubp)             ! needed for BGC only to diagnose 10m wind speed
    real(r8) :: bw                       ! partial density of water (ice + liquid)
    real(r8) :: t_grnd_temp              ! Used in surface flux correction over frozen ground
    real(r8) :: betaprime(lbc:ubc)       ! Effective beta: 1 for snow layers, beta(islak) otherwise
      ! This assumes all radiation is absorbed in the top snow layer and will need
      ! to be changed for CLM 4.
!
! Constants for lake temperature model
!
    data beta/0.4_r8, 0.4_r8/  ! (deep lake, shallow lake)
    ! This is the energy absorbed at the lake surface if no snow.
!    data za  /0.6_r8, 0.5_r8/
!    data eta /0.1_r8, 0.5_r8/
!-----------------------------------------------------------------------


!    dtime = get_step_size()

! Begin calculations

!dir$ concurrent
!cdir nodep
    forc_th(1)  = forc_t(1) * (forc_psrf(1)/ forc_pbot(1))**(rair/cpair)
    forc_vp(1)  = forc_q(1) * forc_pbot(1)/ (0.622 + 0.378 * forc_q(1))
    forc_rho(1) = (forc_pbot(1) - 0.378 * forc_vp(1)) / (rair * forc_t(1))
    do fc = 1, num_shlakec
       c = filter_shlakec(fc)
       g = cgridcell(c)

       ! Surface temperature and fluxes

       ! Find top layer
#if (defined CLMDEBUG)
       if (snl(c) > 0 .or. snl(c) < -5) then
           write(6,*)'snl is not defined in ShalLakeFluxesMod'
           call endrun()
       end if
!       if (snl(c) /= 0) then
!           write(6,*)'snl is not equal to zero in ShalLakeFluxesMod'
!           call endrun()
!       end if
       write(6,*)'Now in ShalLakeFluxes. snl = ', snl(c)
#endif
       jtop(c) = snl(c) + 1

#if (defined SHLAKETEST)
!       dz_lake(c,1:nlevlak) = dz(c,1:nlevlak)
#endif

       if (snl(c) < 0) then
           betaprime(c) = 1._r8  !Assume all solar rad. absorbed at the surface of the top snow layer. 
           dzsur(c) = dz(c,jtop(c))/2._r8
       else
           betaprime(c) = beta(islak)
           dzsur(c) = dz_lake(c,1)/2._r8
       end if
       ! Originally this was 1*dz, but shouldn't it be 1/2?

       ! Saturated vapor pressure, specific humidity and their derivatives
       ! at lake surface

       call QSat(t_grnd(c), forc_pbot(g), eg, degdT, qsatg(c), qsatgdT(c))

       ! Potential, virtual potential temperature, and wind speed at the
       ! reference height

       thm(c) = forc_t(g) + 0.0098_r8*forc_hgt_t(g)   ! intermediate variable
       thv(c) = forc_th(g)*(1._r8+0.61_r8*forc_q(g))     ! virtual potential T
    end do

!dir$ concurrent
!cdir nodep
    do fp = 1, num_shlakep
       p = filter_shlakep(fp)
       c = pcolumn(p)
       g = pgridcell(p)

       nmozsgn(p) = 0
       obuold(p) = 0._r8
       displa(p) = 0._r8

       ! Roughness lengths
 
       if (t_grnd(c) >= tfrz) then   ! for unfrozen lake
          z0mg(p) = 0.01_r8
       else                          ! for frozen lake
       ! Is this okay even if it is snow covered?  What is the roughness over non-veg. snow?
          z0mg(p) = 0.04_r8
       end if
       z0hg(p) = z0mg(p)
       z0qg(p) = z0mg(p)

       ! Latent heat

#if (defined PERGRO)
       htvp(c) = hvap
#else
       if (t_grnd(c) > tfrz) then
          htvp(c) = hvap
       else
          htvp(c) = hsub
       end if
#endif
       ! Zack Subin, 3/26/09: Shouldn't this be the ground temperature rather than the air temperature above?
       ! I'll change it for now.

       ! Initialize stability variables

       ur(p)    = max(1.0_r8,sqrt(forc_u(g)*forc_u(g)+forc_v(g)*forc_v(g)))
       dth(p)   = thm(c)-t_grnd(c)
       dqh(p)   = forc_q(g)-qsatg(c)
       dthv     = dth(p)*(1._r8+0.61_r8*forc_q(g))+0.61_r8*forc_th(g)*dqh(p)
       zldis(p) = forc_hgt_u(g) - 0._r8

       ! Initialize Monin-Obukhov length and wind speed

       call MoninObukIni(ur(p), thv(c), dthv, zldis(p), z0mg(p), um(p), obu(p))

    end do

    iter = 1
    fncopy = num_shlakep
    fpcopy(1:num_shlakep) = filter_shlakep(1:num_shlakep)

    ! Begin stability iteration

    ITERATION : do while (iter <= niters .and. fncopy > 0)

       ! Determine friction velocity, and potential temperature and humidity
       ! profiles of the surface boundary layer

       call FrictionVelocity(pgridcell,forc_hgt,forc_hgt_u,          & !i
                             forc_hgt_t,forc_hgt_q,                  & !i
                             lbp, ubp, fncopy, fpcopy,               & !i
                             displa, z0mg, z0hg, z0qg,               & !i
                             obu, iter, ur, um,                      & !i
                             ustar,temp1, temp2, temp12m, temp22m,   & !o
                             u10,fv,                                 & !o
                             fm)  !i&o

!dir$ concurrent
!cdir nodep
       do fp = 1, fncopy
          p = fpcopy(fp)
          c = pcolumn(p)
          g = pgridcell(p)

          tgbef(c) = t_grnd(c)
          if (t_grnd(c) > tfrz .and. t_lake(c,1) > tfrz .and. snl(c) == 0) then
#if (defined SHLAKETEST)
!             if (savedtke1(c) > 1000) savedtke1(c) = tkwat !not initialized
#endif
             tksur = savedtke1(c)
#ifdef CLMDEBUG
             write(6,*)'savedtke1 = ',savedtke1(c)
#endif
             ! Set this to the eddy conductivity from the last
             ! timestep, as the molecular conductivity will be orders of magnitude too small.
             ! Will have to deal with first timestep.
             tsur = t_lake(c,1)
          else if (snl(c) == 0) then  !frozen but no snow layers
             tksur = tkice
             tsur = t_lake(c,1)
          else
          !Need to calculate thermal conductivity of the top snow layer
             bw = (h2osoi_ice(c,jtop(c))+h2osoi_liq(c,jtop(c)))/dz(c,jtop(c))
             tksur = tkair + (7.75e-5_r8 *bw + 1.105e-6_r8*bw*bw)*(tkice-tkair)
             tsur = t_soisno(c,jtop(c))
          end if

          ! Determine aerodynamic resistances

          ram(p)  = 1._r8/(ustar(p)*ustar(p)/um(p))
          rah(p)  = 1._r8/(temp1(p)*ustar(p))
          raw(p)  = 1._r8/(temp2(p)*ustar(p))
          ram1(p) = ram(p)   !pass value to global variable
#ifdef CLMDEBUG
          write(6,*)'ustar=',ustar(p)
#endif

          ! Get derivative of fluxes with respect to ground temperature

          stftg3(p) = emg*sb*tgbef(c)*tgbef(c)*tgbef(c)

          ! Changed surface temperature from t_lake(c,1) to tsur.
          ! Also adjusted so that if there are snow layers present, all radiation is absorbed in the top layer.
          ax  = betaprime(c)*sabg(p) + emg*forc_lwrad(g) + 3._r8*stftg3(p)*tgbef(c) &
               + forc_rho(g)*cpair/rah(p)*thm(c) &
               - htvp(c)*forc_rho(g)/raw(p)*(qsatg(c)-qsatgdT(c)*tgbef(c) - forc_q(g)) &
               + tksur*tsur/dzsur(c)
          !Changed sabg(p) and to betaprime(c)*sabg(p).
          bx  = 4._r8*stftg3(p) + forc_rho(g)*cpair/rah(p) &
               + htvp(c)*forc_rho(g)/raw(p)*qsatgdT(c) + tksur/dzsur(c)

          t_grnd(c) = ax/bx
#ifdef CLMDEBUG
          write(6,*)'t_grnd iter = ',t_grnd(c)
#endif

          ! Update htvp
#ifndef PERGRO
       if (t_grnd(c) > tfrz) then
          htvp(c) = hvap
       else
          htvp(c) = hsub
       end if
#endif

          ! Surface fluxes of momentum, sensible and latent heat
          ! using ground temperatures from previous time step

          eflx_sh_grnd(p) = forc_rho(g)*cpair*(t_grnd(c)-thm(c))/rah(p)
          qflx_evap_soi(p) = forc_rho(g)*(qsatg(c)+qsatgdT(c)*(t_grnd(c)-tgbef(c))-forc_q(g))/raw(p)

          ! Re-calculate saturated vapor pressure, specific humidity and their
          ! derivatives at lake surface

          call QSat(t_grnd(c), forc_pbot(g), eg, degdT, qsatg(c), qsatgdT(c))

          dth(p)=thm(c)-t_grnd(c)
          dqh(p)=forc_q(g)-qsatg(c)

          tstar = temp1(p)*dth(p)
          qstar = temp2(p)*dqh(p)

          thvstar=tstar*(1._r8+0.61_r8*forc_q(g)) + 0.61_r8*forc_th(g)*qstar
          zeta=zldis(p)*vkc * grav*thvstar/(ustar(p)**2*thv(c))

          if (zeta >= 0._r8) then     !stable
             zeta = min(2._r8,max(zeta,0.01_r8))
             um(p) = max(ur(p),0.1_r8)
          else                     !unstable
             zeta = max(-100._r8,min(zeta,-0.01_r8))
             wc = beta1*(-grav*ustar(p)*thvstar*zii/thv(c))**0.333_r8
             um(p) = sqrt(ur(p)*ur(p)+wc*wc)
          end if
          obu(p) = zldis(p)/zeta

          if (obuold(p)*obu(p) < 0._r8) nmozsgn(p) = nmozsgn(p)+1

          obuold(p) = obu(p)

       end do   ! end of filtered pft loop

       iter = iter + 1
       if (iter <= niters ) then
          ! Rebuild copy of pft filter for next pass through the ITERATION loop

          fnold = fncopy
          fncopy = 0
          do fp = 1, fnold
             p = fpcopy(fp)
             if (nmozsgn(p) < 3) then
                fncopy = fncopy + 1
                fpcopy(fncopy) = p
             end if
          end do   ! end of filtered pft loop
       end if

    end do ITERATION   ! end of stability iteration

!dir$ concurrent
!cdir nodep
    do fp = 1, num_shlakep
       p = filter_shlakep(fp)
       c = pcolumn(p)
       g = pgridcell(p)

       ! If there is snow on the ground and t_grnd > tfrz: reset t_grnd = tfrz.
       ! Re-evaluate ground fluxes.
       ! h2osno > 0.5 prevents spurious fluxes.
       ! note that qsatg and qsatgdT should be f(tgbef) (PET: not sure what this
       ! comment means)
       ! Zack Subin, 3/27/09: Since they are now a function of whatever t_grnd was before cooling
       !    to freezing temperature, then this value should be used in the derivative correction term.
       ! Should this happen if the lake temperature is below freezing, too? I'll assume that for now.
       ! Also, allow convection if ground temp is colder than lake but warmer than 4C, or warmer than 
       !    lake which is warmer than freezing but less than 4C.
!#ifndef SHLAKETEST
       if ( (h2osno(c) > 0.5_r8 .or. t_lake(c,1) <= tfrz) .and. t_grnd(c) > tfrz) then
!#else
!       if ( t_lake(c,1) <= tfrz .and. t_grnd(c) > tfrz) then
!#endif
          t_grnd_temp = t_grnd(c)
          t_grnd(c) = tfrz
          eflx_sh_grnd(p) = forc_rho(g)*cpair*(t_grnd(c)-thm(c))/rah(p)
          qflx_evap_soi(p) = forc_rho(g)*(qsatg(c)+qsatgdT(c)*(t_grnd(c)-t_grnd_temp) - forc_q(g))/raw(p)
       else if ( (t_lake(c,1) > t_grnd(c) .and. t_grnd(c) > tdmax) .or. &
                 (t_lake(c,1) < t_grnd(c) .and. t_lake(c,1) > tfrz .and. t_grnd(c) < tdmax) ) then
                 ! Convective mixing will occur at surface
          t_grnd_temp = t_grnd(c)
          t_grnd(c) = t_lake(c,1)
          eflx_sh_grnd(p) = forc_rho(g)*cpair*(t_grnd(c)-thm(c))/rah(p)
          qflx_evap_soi(p) = forc_rho(g)*(qsatg(c)+qsatgdT(c)*(t_grnd(c)-t_grnd_temp) - forc_q(g))/raw(p)
       end if

          ! Update htvp
#ifndef PERGRO
       if (t_grnd(c) > tfrz) then
          htvp(c) = hvap
       else
          htvp(c) = hsub
       end if
#endif

       ! Net longwave from ground to atmosphere

!       eflx_lwrad_out(p) = (1._r8-emg)*forc_lwrad(g) + stftg3(p)*(-3._r8*tgbef(c)+4._r8*t_grnd(c))
       ! What is tgbef doing in this equation? Can't it be exact now? --Zack Subin, 4/14/09
       eflx_lwrad_out(p) = (1._r8-emg)*forc_lwrad(g) + emg*sb*t_grnd(c)**4

       ! Ground heat flux

       eflx_soil_grnd(p) = sabg(p) + forc_lwrad(g) - eflx_lwrad_out(p) - &
            eflx_sh_grnd(p) - htvp(c)*qflx_evap_soi(p)
       !Why is this sabg(p) and not beta*sabg(p)??
       !I've kept this as the incorrect sabg so that the energy balance check will be correct.
       !This is the effective energy flux into the ground including the lake solar absorption
       !below the surface.  The variable eflx_gnet will be used to pass the actual heat flux
       !from the ground interface into the lake.

       taux(p) = -forc_rho(g)*forc_u(g)/ram(p)
       tauy(p) = -forc_rho(g)*forc_v(g)/ram(p)

       eflx_sh_tot(p)   = eflx_sh_grnd(p)
       qflx_evap_tot(p) = qflx_evap_soi(p)
       eflx_lh_tot(p)   = htvp(c)*qflx_evap_soi(p)
       eflx_lh_grnd(p)  = htvp(c)*qflx_evap_soi(p)
#if (defined CLMDEBUG)
       write(6,*) 'c, sensible heat = ', c, eflx_sh_tot(p), 'latent heat = ', eflx_lh_tot(p) &
              , 'ground temp = ', t_grnd(c), 'h2osno = ', h2osno(c)
       if (abs(eflx_sh_tot(p)) > 1500 .or. abs(eflx_lh_tot(p)) > 1500) then
           write(6,*)'WARNING: SH, LH = ', eflx_sh_tot(p), eflx_lh_tot(p)
       end if
       if (abs(eflx_sh_tot(p)) > 10000 .or. abs(eflx_lh_tot(p)) > 10000 &
             .or. abs(t_grnd(c)-288)>200 ) call endrun()
#endif
       ! 2 m height air temperature
       t_ref2m(p) = thm(c) + temp1(p)*dth(p)*(1._r8/temp12m(p) - 1._r8/temp1(p))

       ! 2 m height specific humidity
       q_ref2m(p) = forc_q(g) + temp2(p)*dqh(p)*(1._r8/temp22m(p) - 1._r8/temp2(p))

       ! Energy residual used for melting snow
       ! Effectively moved to ShalLakeTemp

       ! Prepare for lake layer temperature calculations below
       ! fin(c) = betaprime * sabg(p) + forc_lwrad(g) - (eflx_lwrad_out(p) + &
       !          eflx_sh_tot(p) + eflx_lh_tot(p))
       ! NOW this is just the net ground heat flux calculated below.

       eflx_gnet(p) = betaprime(c) * sabg(p) + forc_lwrad(g) - (eflx_lwrad_out(p) + &
            eflx_sh_tot(p) + eflx_lh_tot(p))
       ! This is the actual heat flux from the ground interface into the lake, not including
       ! the light that penetrates the surface.

!       u2m = max(1.0_r8,ustar(p)/vkc*log(2._r8/z0mg(p)))
       ! u2 often goes below 1 m/s; it seems like the only reason for this minimum is to
       ! keep it from being zero in the ks equation below; 0.1 m/s is a better limit for
       ! stable conditions --ZS
       u2m = max(0.1_r8,ustar(p)/vkc*log(2._r8/z0mg(p)))

       ws(c) = 1.2e-03_r8 * u2m
       ks(c) = 6.6_r8*sqrt(abs(sin(lat(g))))*(u2m**(-1.84_r8))

    end do

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    ! End of surface flux relevant code in original BiogeophysicsLakeMod until history loop.

    ! The following are needed for global average on history tape.

!dir$ concurrent
!cdir nodep
    do fp = 1, num_shlakep
       p = filter_shlakep(fp)
       c = pcolumn(p)
       g = pgridcell(p)
!       t_veg(p) = forc_t(g)
        !This is an odd choice, since elsewhere t_veg = t_grnd for bare ground.
        !Zack Subin, 4/09
       t_veg(p) = t_grnd(c)
       eflx_lwrad_net(p)  = eflx_lwrad_out(p) - forc_lwrad(g)
       qflx_prec_grnd(p) = forc_rain(g) + forc_snow(g)
    end do

END SUBROUTINE ShalLakeFluxes
 
SUBROUTINE ShalLakeTemperature(t_grnd,h2osno,sabg,dz,dz_lake,z,zi,           & 
                                 z_lake,ws,ks,snl,eflx_gnet,lakedepth,       &
                                 lake_icefrac,snowdp,                        & 
                                 eflx_sh_grnd,eflx_sh_tot,eflx_soil_grnd,    & 
                                 t_lake,t_soisno,h2osoi_liq,                 &
                                 h2osoi_ice,savedtke1,                       &
                                 frac_iceold,qflx_snomelt,imelt)
!=======================================================================================================
! !DESCRIPTION:
! Calculates temperatures in the 20-25 layer column of (possible) snow,
! lake water, and soil beneath lake.
! Snow and soil temperatures are determined as in SoilTemperature, except
! for appropriate boundary conditions at the top of the snow (the flux is fixed
! to be the ground heat flux calculated in ShalLakeFluxes), the bottom of the snow
! (adjacent to top lake layer), and the top of the soil (adjacent to the bottom
! lake layer). Also, the soil is assumed to be always fully saturated (ShalLakeHydrology
! will have to insure this). The whole column is solved simultaneously as one tridiagonal matrix.
! Lake temperatures are determined from the Hostetler model as before, except now:
!    i) Lake water layers can freeze by any fraction and release latent heat; thermal
!       and mechanical properties are adjusted for ice fraction.
!   ii) Convective mixing (though not eddy diffusion) still occurs for frozen lakes.
!  iii) No sunlight is absorbed in the lake if there are snow layers.
!   iv) Light is allowed to reach the top soil layer (where it is assumed to be completely absorbed).
!    v) Lakes have variable depth, set ultimately in surface data set but now in initShalLakeMod.
!
! Eddy + molecular diffusion:
! d ts    d            d ts     1 ds
! ---- = -- [(km + ke) ----] + -- --
!  dt    dz             dz     cw dz
!
! where: ts = temperature (kelvin)
!         t = time (s)
!         z = depth (m)
!        km = molecular diffusion coefficient (m**2/s)
!        ke = eddy diffusion coefficient (m**2/s)
!        cw = heat capacity (j/m**3/kelvin)
!         s = heat source term (w/m**2)
!
!   Shallow lakes are allowed to have variable depth, set in _____.
!
!   For shallow lakes:    ke > 0 if unfrozen,
!       and convective mixing occurs WHETHER OR NOT frozen. (See e.g. Martynov...)
!
! Use the Crank-Nicholson method to set up tridiagonal system of equations to
! solve for ts at time n+1, where the temperature equation for layer i is
! r_i = a_i [ts_i-1] n+1 + b_i [ts_i] n+1 + c_i [ts_i+1] n+1
!
! The solution conserves energy as:
!
! [For lake layers]
! cw*([ts(      1)] n+1 - [ts(      1)] n)*dz(      1)/dt + ... +
! cw*([ts(nlevlak)] n+1 - [ts(nlevlak)] n)*dz(nlevlak)/dt = fin
! But now there is phase change, so cv is not constant and there is
! latent heat.
!
! where:
! [ts] n   = old temperature (kelvin)
! [ts] n+1 = new temperature (kelvin)
! fin      = heat flux into lake (w/m**2)
!          = betaprime*sabg + forc_lwrad - eflx_lwrad_out - eflx_sh_tot - eflx_lh_tot
!          (This is now the same as the ground heat flux.)
!            + phi(1) + ... + phi(nlevlak) + phi(top soil level)
! betaprime = beta(islak) for no snow layers, and 1 for snow layers.
! This assumes all radiation is absorbed in the top snow layer and will need
! to be changed for CLM 4.
!
! WARNING: This subroutine assumes lake columns have one and only one pft.
!
! Outline:
! 1!) Initialization
! 2!) Lake density
! 3!) Diffusivity
! 4!) Heat source term from solar radiation penetrating lake
! 5!) Set thermal props and find initial energy content
! 6!) Set up vectors for tridiagonal matrix solution
! 7!) Solve tridiagonal and back-substitute
! 8!) (Optional) Do first energy check using temperature change at constant heat capacity.
! 9!) Phase change
! 9.5!) (Optional) Do second energy check using temperature change and latent heat, considering changed heat capacity.
!                  Also do soil water balance check.
!10!) Convective mixing 
!11!) Do final energy check to detect small numerical errors (especially from convection)
!     and dump small imbalance into sensible heat, or pass large errors to BalanceCheckMod for abort.
!
! REVISION HISTORY:
! Created by Zack Subin, 2009.
! Changed by guhp for coupling,2010
!=========================================================================================================


! USES:
    use lake_const
    
    implicit none

!in:
    real(r8), intent(in) :: t_grnd(1)          ! ground temperature (Kelvin)
    real(r8), intent(in) :: h2osno(1)          ! snow water (mm H2O)
    real(r8), intent(in) :: sabg(1)            ! solar radiation absorbed by ground (W/m**2)
    real(r8), intent(in) :: dz(1,-nlevsno + 1:nlevsoi)          ! layer thickness for snow & soil (m)
    real(r8), intent(in) :: dz_lake(1,nlevlak)                  ! layer thickness for lake (m)
    real(r8), intent(in) :: z(1,-nlevsno+1:nlevsoi)             ! layer depth for snow & soil (m)
    real(r8), intent(in) :: zi(1,-nlevsno+0:nlevsoi)            ! interface level below a "z" level (m)
                                                                ! the other z and dz variables
    real(r8), intent(in) :: z_lake(1,nlevlak)  ! layer depth for lake (m)
    real(r8), intent(in) :: ws(1)              ! surface friction velocity (m/s)
    real(r8), intent(in) :: ks(1)              ! coefficient passed to ShalLakeTemperature
                                               ! for calculation of decay of eddy diffusivity with depth
    integer , intent(in) :: snl(1)             ! negative of number of snow layers
    real(r8), intent(in) :: eflx_gnet(1)       ! net heat flux into ground (W/m**2) at the surface interface
    real(r8), intent(in) :: lakedepth(1)       ! column lake depth (m)
    real(r8), intent(inout) :: snowdp(1)        !snow height (m)
!out: 

    real(r8), intent(out) :: eflx_sh_grnd(1)    ! sensible heat flux from ground (W/m**2) [+ to atm]
    real(r8), intent(out) :: eflx_sh_tot(1)     ! total sensible heat flux (W/m**2) [+ to atm]
    real(r8), intent(out) :: eflx_soil_grnd(1)  ! heat flux into snow / lake (W/m**2) [+ = into soil]
                                               ! Here this includes the whole lake radiation absorbed.
#if (defined SHLAKETEST)
    real(r8), intent(out) :: qmelt(1)           ! snow melt [mm/s] [temporary]
#endif
    real(r8), intent(inout) :: t_lake(1,nlevlak)                 ! lake temperature (Kelvin)
    real(r8), intent(inout) :: t_soisno(1,-nlevsno+1:nlevsoi)    ! soil (or snow) temperature (Kelvin)
    real(r8), intent(inout) :: h2osoi_liq(1,-nlevsno+1:nlevsoi)  ! liquid water (kg/m2) [for snow & soil layers]
    real(r8), intent(inout) :: h2osoi_ice(1,-nlevsno+1:nlevsoi)  ! ice lens (kg/m2) [for snow & soil layers]
    real(r8), intent(inout) :: lake_icefrac(1,nlevlak)           ! mass fraction of lake layer that is frozen
    real(r8), intent(out) :: savedtke1(1)                      ! top level thermal conductivity (W/mK)
    real(r8), intent(out) :: frac_iceold(1,-nlevsno+1:nlevsoi) ! fraction of ice relative to the tot water
    real(r8), intent(out) :: qflx_snomelt(1)  !snow melt (mm H2O /s)
    integer, intent(out)  :: imelt(1,-nlevsno+1:nlevsoi)        !flag for melting (=1), freezing (=2), Not=0 (new)


! OTHER LOCAL VARIABLES:

    integer , parameter  :: islak = 2     ! index of lake, 1 = deep lake, 2 = shallow lake
    real(r8), parameter  :: p0 = 1._r8     ! neutral value of turbulent prandtl number
    integer  :: i,j,fc,fp,g,c,p         ! do loop or array index
!    real(r8) :: dtime                   ! land model time step (sec)
    real(r8) :: beta(2)                 ! fraction solar rad absorbed at surface: depends on lake type
    real(r8) :: za(2)                   ! base of surface absorption layer (m): depends on lake type
    real(r8) :: eta(2)                  ! light extinction coefficient (/m): depends on lake type
    real(r8) :: cwat                    ! specific heat capacity of water (j/m**3/kelvin)
    real(r8) :: cice_eff                ! effective heat capacity of ice (using density of
                                          ! water because layer depth is not adjusted when freezing
    real(r8) :: cfus                    ! effective heat of fusion per unit volume
                                          ! using water density as above
    real(r8) :: km                      ! molecular diffusion coefficient (m**2/s)
    real(r8) :: tkice_eff               ! effective conductivity since layer depth is constant
    real(r8) :: a(lbc:ubc,-nlevsno+1:nlevlak+nlevsoi)      ! "a" vector for tridiagonal matrix
    real(r8) :: b(lbc:ubc,-nlevsno+1:nlevlak+nlevsoi)      ! "b" vector for tridiagonal matrix
    real(r8) :: c1(lbc:ubc,-nlevsno+1:nlevlak+nlevsoi)     ! "c" vector for tridiagonal matrix
    real(r8) :: r(lbc:ubc,-nlevsno+1:nlevlak+nlevsoi)      ! "r" vector for tridiagonal solution
    real(r8) :: rhow(lbc:ubc,nlevlak)   ! density of water (kg/m**3)
    real(r8) :: phi(lbc:ubc,nlevlak)    ! solar radiation absorbed by layer (w/m**2)
    real(r8) :: kme(lbc:ubc,nlevlak)    ! molecular + eddy diffusion coefficient (m**2/s)
    real(r8) :: rsfin                   ! relative flux of solar radiation into layer
    real(r8) :: rsfout                  ! relative flux of solar radiation out of layer
    real(r8) :: phi_soil(lbc:ubc)       ! solar radiation into top soil layer (W/m**2)
    real(r8) :: ri                      ! richardson number
    real(r8) :: fin(lbc:ubc)            ! net heat flux into lake at ground interface (w/m**2)
    real(r8) :: ocvts(lbc:ubc)          ! (cwat*(t_lake[n  ])*dz
    real(r8) :: ncvts(lbc:ubc)          ! (cwat*(t_lake[n+1])*dz
    real(r8) :: ke                      ! eddy diffusion coefficient (m**2/s)
    real(r8) :: zin                     ! depth at top of layer (m)
    real(r8) :: zout                    ! depth at bottom of layer (m)
    real(r8) :: drhodz                  ! d [rhow] /dz (kg/m**4)
    real(r8) :: n2                      ! brunt-vaisala frequency (/s**2)
    real(r8) :: num                     ! used in calculating ri
    real(r8) :: den                     ! used in calculating ri
    real(r8) :: tav_froz(lbc:ubc)       ! used in aver temp for convectively mixed layers (C)
    real(r8) :: tav_unfr(lbc:ubc)       ! "
    real(r8) :: nav(lbc:ubc)            ! used in aver temp for convectively mixed layers
    real(r8) :: phidum                  ! temporary value of phi
    real(r8) :: iceav(lbc:ubc)          ! used in calc aver ice for convectively mixed layers
    real(r8) :: qav(lbc:ubc)            ! used in calc aver heat content for conv. mixed layers
    integer  :: jtop(lbc:ubc)           ! top level for each column (no longer all 1)
    real(r8) :: cv (lbc:ubc,-nlevsno+1:nlevsoi)  !heat capacity of soil/snow [J/(m2 K)]
    real(r8) :: tk (lbc:ubc,-nlevsno+1:nlevsoi)  !thermal conductivity of soil/snow [W/(m K)]
                                                 !(at interface below, except for j=0)
    real(r8) :: cv_lake (lbc:ubc,1:nlevlak)      !heat capacity [J/(m2 K)]
    real(r8) :: tk_lake (lbc:ubc,1:nlevlak)  !thermal conductivity at layer node [W/(m K)]
    real(r8) :: cvx (lbc:ubc,-nlevsno+1:nlevlak+nlevsoi) !heat capacity for whole column [J/(m2 K)]
    real(r8) :: tkix(lbc:ubc,-nlevsno+1:nlevlak+nlevsoi) !thermal conductivity at layer interfaces
                                                         !for whole column [W/(m K)]
    real(r8) :: tx(lbc:ubc,-nlevsno+1:nlevlak+nlevsoi) ! temperature of whole column [K]
    real(r8) :: tktopsoillay(lbc:ubc)          ! thermal conductivity [W/(m K)]
    real(r8) :: fnx(lbc:ubc,-nlevsno+1:nlevlak+nlevsoi)  !heat diffusion through the layer interface below [W/m2]
    real(r8) :: phix(lbc:ubc,-nlevsno+1:nlevlak+nlevsoi) !solar source term for whole column [W/m**2]
    real(r8) :: zx(lbc:ubc,-nlevsno+1:nlevlak+nlevsoi)   !interface depth (+ below surface) for whole column [m]
    real(r8) :: dzm                              !used in computing tridiagonal matrix [m]
    real(r8) :: dzp                              !used in computing tridiagonal matrix [m]
    integer  :: jprime                   ! j - nlevlak
    real(r8) :: factx(lbc:ubc,-nlevsno+1:nlevlak+nlevsoi) !coefficient used in computing tridiagonal matrix
    real(r8) :: t_lake_bef(lbc:ubc,1:nlevlak)    !beginning lake temp for energy conservation check [K]
    real(r8) :: t_soisno_bef(lbc:ubc,-nlevsno+1:nlevsoi) !beginning soil temp for E cons. check [K]
    real(r8) :: lhabs(lbc:ubc)       ! total per-column latent heat abs. from phase change  (J/m^2)
    real(r8) :: esum1(lbc:ubc)        ! temp for checking energy (J/m^2)
    real(r8) :: esum2(lbc:ubc)        ! ""
    real(r8) :: zsum(lbc:ubc)        ! temp for putting ice at the top during convection (m)
    real(r8) :: wsum(lbc:ubc)        ! temp for checking water (kg/m^2)
    real(r8) :: wsum_end(lbc:ubc)    ! temp for checking water (kg/m^2)
    real(r8) :: errsoi(1)                         ! soil/lake energy conservation error (W/m**2)
    real(r8) :: eflx_snomelt(1)  !snow melt heat flux (W/m**2)
!
! Constants for lake temperature model
!
    data beta/0.4_r8, 0.4_r8/  ! (deep lake, shallow lake)
!    data za  /0.6_r8, 0.5_r8/
!    data eta /0.1_r8, 0.5_r8/
    data za  /0.6_r8, 0.6_r8/
!   data eta /0.1_r8, 0.1_r8/
!   For now, keep beta and za for shallow lake the same as deep lake, until better data is found.
!   It looks like eta is key and that larger values give better results for shallow lakes.  Use
!   empirical expression from Hakanson (below). This is still a very unconstrained parameter
!   that deserves more attention.
!   Some radiation will be allowed to reach the soil.
!-----------------------------------------------------------------------


    ! 1!) Initialization
    ! Determine step size

!    dtime = get_step_size()

    ! Initialize constants
    cwat = cpliq*denh2o ! water heat capacity per unit volume
    cice_eff = cpice*denh2o !use water density because layer depth is not adjusted
                              !for freezing
    cfus = hfus*denh2o  ! latent heat per unit volume
    tkice_eff = tkice * denice/denh2o !effective conductivity since layer depth is constant
    km = tkwat/cwat     ! a constant (molecular diffusivity)

    ! Begin calculations

!dir$ concurrent
!cdir nodep
    do fc = 1, num_shlakec
       c = filter_shlakec(fc)

       ! Initialize Ebal quantities computed below

       ocvts(c) = 0._r8
       ncvts(c) = 0._r8
       esum1(c) = 0._r8
       esum2(c) = 0._r8

    end do

    ! Initialize set of previous time-step variables as in DriverInit,
    ! which is currently not called over lakes. This has to be done
    ! here because phase change will occur in this routine.
    ! Ice fraction of snow at previous time step

    do j = -nlevsno+1,0
!dir$ concurrent
!cdir nodep
      do fc = 1, num_shlakec
         c = filter_shlakec(fc)
         if (j >= snl(c) + 1) then
            frac_iceold(c,j) = h2osoi_ice(c,j)/(h2osoi_liq(c,j)+h2osoi_ice(c,j))
         end if
      end do
    end do

#if (defined CLMDEBUG)
    ! Sum soil water.
    do j = 1, nlevsoi
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)
          if (j == 1) wsum(c) = 0._r8
          wsum(c) = wsum(c) + h2osoi_ice(c,j) + h2osoi_liq(c,j)
       end do
    end do
#endif

!dir$ concurrent
!cdir nodep
    do fp = 1, num_shlakep
       p = filter_shlakep(fp)
       c = pcolumn(p)

#if (defined SHLAKETEST)
!        qmelt(c) = 0
#endif

       ! Prepare for lake layer temperature calculations below

       ! fin(c) = betaprime * sabg(p) + forc_lwrad(g) - (eflx_lwrad_out(p) + &
       !     eflx_sh_tot(p) + eflx_lh_tot(p)) 
       ! fin(c) now passed from ShalLakeFluxes as eflx_gnet
       fin(c) = eflx_gnet(p)

    end do

    ! 2!) Lake density

    do j = 1, nlevlak
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)
          rhow(c,j) = (1._r8 - lake_icefrac(c,j)) * & 
                      1000._r8*( 1.0_r8 - 1.9549e-05_r8*(abs(t_lake(c,j)-277._r8))**1.68_r8 ) &
                    + lake_icefrac(c,j)*denice
                    ! Allow for ice fraction; assume constant ice density.
                    ! Is this the right weighted average?
                    ! Using this average will make sure that surface ice is treated properly during
                    ! convective mixing.
       end do
    end do

    ! 3!) Diffusivity and implied thermal "conductivity" = diffusivity * cwat
    do j = 1, nlevlak-1
!dir$ prefervector
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)
          drhodz = (rhow(c,j+1)-rhow(c,j)) / (z_lake(c,j+1)-z_lake(c,j))
          n2 = grav / rhow(c,j) * drhodz
          ! Fixed sign error here: our z goes up going down into the lake, so no negative
          ! sign is needed to make this positive unlike in Hostetler. --ZS
          num = 40._r8 * n2 * (vkc*z_lake(c,j))**2
          den = max( (ws(c)**2) * exp(-2._r8*ks(c)*z_lake(c,j)), 1.e-10_r8 )
          ri = ( -1._r8 + sqrt( max(1._r8+num/den, 0._r8) ) ) / 20._r8
          if (t_grnd(c) > tfrz .and. t_lake(c,1) > tfrz .and. snl(c) == 0) then
             ke = vkc*ws(c)*z_lake(c,j)/p0 * exp(-ks(c)*z_lake(c,j)) / (1._r8+37._r8*ri*ri)
             kme(c,j) = km + ke
             tk_lake(c,j) = kme(c,j)*cwat
             ! If there is some ice in this layer (this should rarely happen because the surface
             ! is unfrozen and it will be unstable), still use the cwat to get out the tk b/c the eddy
             ! diffusivity equation assumes water.
          else
             kme(c,j) = km
             tk_lake(c,j) = tkwat*tkice_eff / ( (1._r8-lake_icefrac(c,j))*tkice_eff &
                            + tkwat*lake_icefrac(c,j) )
             ! Assume the resistances add as for the calculation of conductivities at layer interfaces.
          end if
       end do
    end do

!dir$ concurrent
!cdir nodep
    do fc = 1, num_shlakec
       c = filter_shlakec(fc)

       j = nlevlak
       kme(c,nlevlak) = kme(c,nlevlak-1)

       if (t_grnd(c) > tfrz .and. t_lake(c,1) > tfrz .and. snl(c) == 0) then
          tk_lake(c,j) = tk_lake(c,j-1)
       else
          tk_lake(c,j) = tkwat*tkice_eff / ( (1._r8-lake_icefrac(c,j))*tkice_eff &
                            + tkwat*lake_icefrac(c,j) )
       end if

       ! Use in surface flux calculation for next timestep.
       savedtke1(c) = kme(c,1)*cwat ! Will only be used if unfrozen
       ! set number of column levels for use by Tridiagonal below
       jtop(c) = snl(c) + 1
    end do

    ! 4!) Heat source term: unfrozen lakes only
    do j = 1, nlevlak
!dir$ concurrent
!cdir nodep
       do fp = 1, num_shlakep
          p = filter_shlakep(fp)
          c = pcolumn(p)

          ! Set eta(:), the extinction coefficient, according to L Hakanson, Aquatic Sciences, 1995
          ! (regression of Secchi Depth with lake depth for small glacial basin lakes), and the
          ! Poole & Atkins expression for extinction coeffient of 1.7 / Secchi Depth (m).
#ifndef ETALAKE
          eta(:) = 1.1925_r8*lakedepth(c)**(-0.424)
#else
          eta(:) = ETALAKE
#endif

          zin  = z_lake(c,j) - 0.5_r8*dz_lake(c,j)
          zout = z_lake(c,j) + 0.5_r8*dz_lake(c,j)
          rsfin  = exp( -eta(islak)*max(  zin-za(islak),0._r8 ) )
          rsfout = exp( -eta(islak)*max( zout-za(islak),0._r8 ) )

          ! Let rsfout for bottom layer go into soil.
          ! This looks like it should be robust even for pathological cases,
            ! like lakes thinner than za.
          if (t_grnd(c) > tfrz .and. t_lake(c,1) > tfrz .and. snl(c) == 0) then
             phidum = (rsfin-rsfout) * sabg(p) * (1._r8-beta(islak))
             if (j == nlevlak) then
                phi_soil(c) = rsfout * sabg(p) * (1._r8-beta(islak))
             end if
          else if (j == 1 .and. snl(c) == 0) then !if frozen but no snow layers
             phidum = sabg(p) * (1._r8-beta(islak))
          else !radiation absorbed at surface
             phidum = 0._r8
             if (j == nlevlak) phi_soil(c) = 0._r8
          end if
          phi(c,j) = phidum

#if (defined CLMDEBUG)
          write(6,*) 'at c, level ', c, j, ' t_lake = ', t_lake(c,j)
          if(abs(t_lake(c,j)-288) > 100) call endrun()
#endif
       end do
    end do

    ! 5!) Set thermal properties and check initial energy content.

    ! For lake
    do j = 1, nlevlak
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

          cv_lake(c,j) = dz_lake(c,j) * (cwat*(1._r8-lake_icefrac(c,j)) + cice_eff*lake_icefrac(c,j))
       end do
    end do

    ! For snow / soil
  !  call SoilThermProp_Lake(lbc, ubc, num_shlakec, filter_shlakec, tk, cv, tktopsoillay)
  call SoilThermProp_Lake (snl,dz,zi,z,t_soisno,h2osoi_liq,h2osoi_ice,    &
                           tk, cv, tktopsoillay)

    ! Sum cv*t_lake for energy check
    ! Include latent heat term, and correction for changing heat capacity with phase change.

    ! This will need to be over all soil / lake / snow layers. Lake is below.
    do j = 1, nlevlak
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

!          ocvts(c) = ocvts(c) + cv_lake(c,j)*t_lake(c,j) &
          ocvts(c) = ocvts(c) + cv_lake(c,j)*(t_lake(c,j)-tfrz) &
                   + cfus*dz_lake(c,j)*(1._r8-lake_icefrac(c,j)) !&
!                   + (cwat-cice_eff)*lake_icefrac(c)*tfrz*dz_lake(c,j) !enthalpy reconciliation term
          t_lake_bef(c,j) = t_lake(c,j)
       end do
    end do

    ! Now do for soil / snow layers
    do j = -nlevsno + 1, nlevsoi
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

          if (j >= jtop(c)) then
!             ocvts(c) = ocvts(c) + cv(c,j)*t_soisno(c,j) &
             ocvts(c) = ocvts(c) + cv(c,j)*(t_soisno(c,j)-tfrz) &
                      + hfus*h2osoi_liq(c,j) !&
!                      + (cpliq-cpice)*h2osoi_ice(c,j)*tfrz !enthalpy reconciliation term
             if (j == 1 .and. h2osno(c) > 0._r8 .and. j == jtop(c)) then
                ocvts(c) = ocvts(c) - h2osno(c)*hfus
             end if
             t_soisno_bef(c,j) = t_soisno(c,j)
#if (defined CLMDEBUG)
          write(6,*) 'at c, level ', c, j, ' t_soisno_bef = ', t_soisno(c,j)
          if(abs(t_soisno(c,j)-288) > 150) write(6,*),'WARNING: Extreme t_soisno' !call endrun()
          write(6,*) 'cv, tk, dz, z = ', cv(c,j), tk(c,j), dz(c,j), z(c,j)
#endif
          end if
       end do
    end do

!!!!!!!!!!!!!!!!!!!
    ! 6!) Set up vector r and vectors a, b, c1 that define tridiagonal matrix

    ! Heat capacity and resistance of snow without snow layers (<1cm) is ignored during diffusion,
    ! but its capacity to absorb latent heat may be used during phase change.

    ! Set up interface depths, zx, heat capacities, cvx, solar source terms, phix, and temperatures, tx.
    do j = -nlevsno+1, nlevlak+nlevsoi
!dir$ prefervector
!dir$ concurrent
!cdir nodep
       do fc = 1,num_shlakec
          c = filter_shlakec(fc)

          jprime = j - nlevlak

          if (j >= jtop(c)) then
             if (j < 1) then !snow layer
                zx(c,j) = z(c,j)
                cvx(c,j) = cv(c,j)
                phix(c,j) = 0._r8
                tx(c,j) = t_soisno(c,j)
             else if (j <= nlevlak) then !lake layer
                zx(c,j) = z_lake(c,j)
                cvx(c,j) = cv_lake(c,j)
                phix(c,j) = phi(c,j)
                tx(c,j) = t_lake(c,j)
             else !soil layer
                zx(c,j) = zx(c,nlevlak) + dz_lake(c,nlevlak)/2._r8 + z(c,jprime)
                cvx(c,j) = cv(c,jprime)
                if (j == nlevlak + 1) then !top soil layer
                   phix(c,j) = phi_soil(c)
                else !middle or bottom soil layer
                   phix(c,j) = 0._r8
                end if
                tx(c,j) = t_soisno(c,jprime)
             end if
          end if

       end do
    end do

    ! Determine interface thermal conductivities, tkix

    do j = -nlevsno+1, nlevlak+nlevsoi
!dir$ prefervector
!dir$ concurrent
!cdir nodep
       do fc = 1,num_shlakec
          c = filter_shlakec(fc)

          jprime = j - nlevlak

          if (j >= jtop(c)) then
             if (j < 0) then !non-bottom snow layer
                tkix(c,j) = tk(c,j)
             else if (j == 0) then !bottom snow layer
                dzp = zx(c,j+1) - zx(c,j)
                tkix(c,j) = tk_lake(c,1)*tk(c,j)*dzp / &
                      (tk(c,j)*z_lake(c,1) + tk_lake(c,1)*(-z(c,j)) )
                ! tk(c,0) is the conductivity at the middle of that layer, as defined in SoilThermProp_Lake
             else if (j < nlevlak) then !non-bottom lake layer
                tkix(c,j) = ( tk_lake(c,j)*tk_lake(c,j+1) * (dz_lake(c,j+1)+dz_lake(c,j)) ) &
                           / ( tk_lake(c,j)*dz_lake(c,j+1) + tk_lake(c,j+1)*dz_lake(c,j) )
             else if (j == nlevlak) then !bottom lake layer
                dzp = zx(c,j+1) - zx(c,j)
                tkix(c,j) = (tktopsoillay(c)*tk_lake(c,j)*dzp / &
                    (tktopsoillay(c)*dz_lake(c,j)/2._r8 + tk_lake(c,j)*z(c,1) ) )
                    ! tktopsoillay is the conductivity at the middle of that layer, as defined in SoilThermProp_Lake
             else !soil layer
                tkix(c,j) = tk(c,jprime)
             end if
         end if

      end do 
   end do


    ! Determine heat diffusion through the layer interface and factor used in computing
    ! tridiagonal matrix and set up vector r and vectors a, b, c1 that define tridiagonal
    ! matrix and solve system

    do j = -nlevsno+1, nlevlak+nlevsoi
!dir$ prefervector
!dir$ concurrent
!cdir nodep
       do fc = 1,num_shlakec
          c = filter_shlakec(fc)
          if (j >= jtop(c)) then
             if (j < nlevlak+nlevsoi) then !top or interior layer
                factx(c,j) = dtime/cvx(c,j)
                fnx(c,j) = tkix(c,j)*(tx(c,j+1)-tx(c,j))/(zx(c,j+1)-zx(c,j))
             else !bottom soil layer
                factx(c,j) = dtime/cvx(c,j)
                fnx(c,j) = 0._r8 !not used
             end if
          end if
       enddo
    end do

    do j = -nlevsno+1,nlevlak+nlevsoi
!dir$ prefervector
!dir$ concurrent
!cdir nodep
       do fc = 1,num_shlakec
          c = filter_shlakec(fc)
          if (j >= jtop(c)) then
             if (j == jtop(c)) then !top layer
                dzp    = zx(c,j+1)-zx(c,j)
                a(c,j) = 0._r8
                b(c,j) = 1+(1._r8-cnfac)*factx(c,j)*tkix(c,j)/dzp
                c1(c,j) =  -(1._r8-cnfac)*factx(c,j)*tkix(c,j)/dzp
                r(c,j) = tx(c,j) + factx(c,j)*( fin(c) + phix(c,j) + cnfac*fnx(c,j) )
             else if (j < nlevlak+nlevsoi) then !middle layer
                dzm    = (zx(c,j)-zx(c,j-1))
                dzp    = (zx(c,j+1)-zx(c,j))
                a(c,j) =   - (1._r8-cnfac)*factx(c,j)* tkix(c,j-1)/dzm
                b(c,j) = 1._r8+ (1._r8-cnfac)*factx(c,j)*(tkix(c,j)/dzp + tkix(c,j-1)/dzm)
                c1(c,j) =   - (1._r8-cnfac)*factx(c,j)* tkix(c,j)/dzp
                r(c,j) = tx(c,j) + cnfac*factx(c,j)*( fnx(c,j) - fnx(c,j-1) ) + factx(c,j)*phix(c,j)
             else  !bottom soil layer
                dzm     = (zx(c,j)-zx(c,j-1))
                a(c,j) =   - (1._r8-cnfac)*factx(c,j)*tkix(c,j-1)/dzm
                b(c,j) = 1._r8+ (1._r8-cnfac)*factx(c,j)*tkix(c,j-1)/dzm
                c1(c,j) = 0._r8
                r(c,j) = tx(c,j) - cnfac*factx(c,j)*fnx(c,j-1)
             end if
          end if
       enddo
    end do
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


    ! 7!) Solve for tdsolution

    call Tridiagonal(lbc, ubc, -nlevsno + 1, nlevlak + nlevsoi, jtop, num_shlakec, filter_shlakec, &
                     a, b, c1, r, tx)
 
    ! Set t_soisno and t_lake
    do j = -nlevsno+1, nlevlak + nlevsoi
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

          jprime = j - nlevlak

          ! Don't do anything with invalid snow layers.
          if (j >= jtop(c)) then
             if (j < 1) then !snow layer
             t_soisno(c,j) = tx(c,j)
             else if (j <= nlevlak) then !lake layer
             t_lake(c,j)   = tx(c,j)
             else !soil layer
             t_soisno(c,jprime) = tx(c,j)
             end if
#if (defined CLMDEBUG)
             write(6,*) 'at c, level ', c, j, ' tx = ', tx(c,j)
             if(abs(tx(c,j)-288) > 150) write(6,*)'WARNING: extreme tx'!call endrun()
#endif
          end if
       end do
    end do

!!!!!!!!!!!!!!!!!!!!!!!

    ! 8!) Sum energy content and total energy into lake for energy check. Any errors will be from the
    !     Tridiagonal solution.

#if (defined CLMDEBUG)
    do j = 1, nlevlak
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

          esum1(c) = esum1(c) + (t_lake(c,j)-t_lake_bef(c,j))*cv_lake(c,j)
          esum2(c) = esum2(c) + (t_lake(c,j)-tfrz)*cv_lake(c,j)
       end do
    end do

    do j = -nlevsno+1, nlevsoi
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

          if (j >= jtop(c)) then
             esum1(c) = esum1(c) + (t_soisno(c,j)-t_soisno_bef(c,j))*cv(c,j)
             esum2(c) = esum2(c) + (t_soisno(c,j)-tfrz)*cv(c,j)
          end if
       end do
    end do

!dir$ concurrent
!cdir nodep
       do fp = 1, num_shlakep
          p = filter_shlakep(fp)
          c = pcolumn(p)
          ! Again assuming only one pft per column
!          esum1(c) = esum1(c) + lhabs(c)
          errsoi(c) = esum1(c)/dtime - eflx_soil_grnd(p)
                    ! eflx_soil_grnd includes all the solar radiation absorbed in the lake,
                    ! unlike eflx_gnet
          if(abs(errsoi(c)) > 1.e-5_r8) then
             write(6,*)'Primary soil energy conservation error in shlake column during Tridiagonal Solution,', &
                        'error (W/m^2):', c, errsoi(c)
             call endrun()
          end if
       end do
       ! This has to be done before convective mixing because the heat capacities for each layer
       ! will get scrambled.

#endif

!!!!!!!!!!!!!!!!!!!!!!!

    ! 9!) Phase change
  !  call PhaseChange_Lake(lbc, ubc, num_shlakec, filter_shlakec, cv, cv_lake, lhabs)
    call PhaseChange_Lake (snl,h2osno,dz,dz_lake,                            & 
                               t_soisno,h2osoi_liq,h2osoi_ice,               & 
                               lake_icefrac,t_lake, snowdp,                  & 
                               qflx_snomelt,eflx_snomelt,imelt,              &   
                               cv, cv_lake,                                  & 
                               lhabs)                                          

!!!!!!!!!!!!!!!!!!!!!!!

    ! 9.5!) Second energy check and water check.  Now check energy balance before and after phase
    !       change, considering the possibility of changed heat capacity during phase change, by
    !       using initial heat capacity in the first step, final heat capacity in the second step,
    !       and differences from tfrz only to avoid enthalpy correction for (cpliq-cpice)*melt*tfrz.
    !       Also check soil water sum.

#if (defined CLMDEBUG)
    do j = 1, nlevlak
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

          esum2(c) = esum2(c) - (t_lake(c,j)-tfrz)*cv_lake(c,j)
       end do
    end do

    do j = -nlevsno+1, nlevsoi
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

          if (j >= jtop(c)) then
             esum2(c) = esum2(c) - (t_soisno(c,j)-tfrz)*cv(c,j)
          end if
       end do
    end do

!dir$ concurrent
!cdir nodep
       do fp = 1, num_shlakep
          p = filter_shlakep(fp)
          c = pcolumn(p)
          ! Again assuming only one pft per column
          esum2(c) = esum2(c) - lhabs(c)
          errsoi(c) = esum2(c)/dtime
          if(abs(errsoi(c)) > 1.e-5_r8) then
             write(6,*)'Primary soil energy conservation error in shlake column during Phase Change, error (W/m^2):', &
                       c, errsoi(c)
             call endrun()
          end if
       end do

    ! Check soil water
    ! Sum soil water.
    do j = 1, nlevsoi
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)
          if (j == 1) wsum_end(c) = 0._r8
          wsum_end(c) = wsum_end(c) + h2osoi_ice(c,j) + h2osoi_liq(c,j)
          if (j == nlevsoi) then
             if (abs(wsum(c)-wsum_end(c))>1.e-7_r8) then
                write(6,*)'Soil water balance error during phase change in ShalLakeTemperature.', &
                          'column, error (kg/m^2):', c, wsum_end(c)-wsum(c)
                call endrun()
             end if
          end if
       end do
    end do

#endif

!!!!!!!!!!!!!!!!!!!!!!!!!!
    ! 10!) Convective mixing: make sure fracice*dz is conserved, heat content c*dz*T is conserved, and
    ! all ice ends up at the top. Done over all lakes even if frozen.
    ! Either an unstable density profile or ice in a layer below an incompletely frozen layer will trigger.

    !Recalculate density
    do j = 1, nlevlak
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)
          rhow(c,j) = (1._r8 - lake_icefrac(c,j)) * &
                      1000._r8*( 1.0_r8 - 1.9549e-05_r8*(abs(t_lake(c,j)-277._r8))**1.68_r8 ) &
                    + lake_icefrac(c,j)*denice
       end do
    end do

    do j = 1, nlevlak-1
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)
          qav(c) = 0._r8
          nav(c) = 0._r8
          iceav(c) = 0._r8
       end do

       do i = 1, j+1
!dir$ concurrent
!cdir nodep
          do fc = 1, num_shlakec
             c = filter_shlakec(fc)
             if (rhow(c,j) > rhow(c,j+1) .or. &
                (lake_icefrac(c,j) < 1._r8 .and. lake_icefrac(c,j+1) > 0._r8) ) then
#if (defined CLMDEBUG)
                if (i==1) write(6,*), 'Convective Mixing in column ', c, '.'
#endif
                qav(c) = qav(c) + dz_lake(c,i)*(t_lake(c,i)-tfrz) * & 
                        ((1._r8 - lake_icefrac(c,i))*cwat + lake_icefrac(c,i)*cice_eff)
!                tav(c) = tav(c) + t_lake(c,i)*dz_lake(c,i)
                iceav(c) = iceav(c) + lake_icefrac(c,i)*dz_lake(c,i)
                nav(c) = nav(c) + dz_lake(c,i)
             end if
          end do
       end do

!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)
          if (rhow(c,j) > rhow(c,j+1) .or. &
             (lake_icefrac(c,j) < 1._r8 .and. lake_icefrac(c,j+1) > 0._r8) ) then
             qav(c) = qav(c)/nav(c)
             iceav(c) = iceav(c)/nav(c)
             !If the average temperature is above freezing, put the extra energy into the water.
             !If it is below freezing, take it away from the ice.
             if (qav(c) > 0._r8) then
                tav_froz(c) = 0._r8 !Celsius
                tav_unfr(c) = qav(c) / ((1._r8 - iceav(c))*cwat)
             else if (qav(c) < 0._r8) then
                tav_froz(c) = qav(c) / (iceav(c)*cice_eff)
                tav_unfr(c) = 0._r8 !Celsius
             else
                tav_froz(c) = 0._r8
                tav_unfr(c) = 0._r8
             end if
          end if
       end do

       do i = 1, j+1
!dir$ concurrent
!cdir nodep
          do fc = 1, num_shlakec
             c = filter_shlakec(fc)
             if (nav(c) > 0._r8) then
!             if(0==1) then

                !Put all the ice at the top.!
                !If the average temperature is above freezing, put the extra energy into the water.
                !If it is below freezing, take it away from the ice.
                !For the layer with both ice & water, be careful to use the average temperature
                !that preserves the correct total heat content given what the heat capacity of that
                !layer will actually be.
                if (i == 1) zsum(c) = 0._r8
                if ((zsum(c)+dz_lake(c,i))/nav(c) <= iceav(c)) then
                   lake_icefrac(c,i) = 1._r8
                   t_lake(c,i) = tav_froz(c) + tfrz
                else if (zsum(c)/nav(c) < iceav(c)) then
                   lake_icefrac(c,i) = (iceav(c)*nav(c) - zsum(c)) / dz_lake(c,i)
                   ! Find average value that preserves correct heat content.
                   t_lake(c,i) = ( lake_icefrac(c,i)*tav_froz(c)*cice_eff &
                               + (1._r8 - lake_icefrac(c,i))*tav_unfr(c)*cwat ) &
                               / ( lake_icefrac(c,i)*cice_eff + (1-lake_icefrac(c,i))*cwat ) + tfrz
                else
                   lake_icefrac(c,i) = 0._r8
                   t_lake(c,i) = tav_unfr(c) + tfrz
                end if
                zsum(c) = zsum(c) + dz_lake(c,i)

                rhow(c,i) = (1._r8 - lake_icefrac(c,i)) * & 
                            1000._r8*( 1.0_r8 - 1.9549e-05_r8*(abs(t_lake(c,i)-277._r8))**1.68_r8 ) &
                          + lake_icefrac(c,i)*denice
             end if
          end do
       end do
    end do

!!!!!!!!!!!!!!!!!!!!!!!
    ! 11!) Re-evaluate thermal properties and sum energy content.
    ! For lake
    do j = 1, nlevlak
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

          cv_lake(c,j) = dz_lake(c,j) * (cwat*(1._r8-lake_icefrac(c,j)) + cice_eff*lake_icefrac(c,j))
#if (defined CLMDEBUG)
          write(6,*)'Lake Ice Fraction, c, level:', c, j, lake_icefrac(c,j)
#endif
       end do
    end do
    ! For snow / soil
  !  call SoilThermProp_Lake(lbc, ubc, num_shlakec, filter_shlakec, tk, cv, tktopsoillay)
  call SoilThermProp_Lake (snl,dz,zi,z,t_soisno,h2osoi_liq,h2osoi_ice,    &
                           tk, cv, tktopsoillay)


    ! Do as above to sum energy content
    do j = 1, nlevlak
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

!          ncvts(c) = ncvts(c) + cv_lake(c,j)*t_lake(c,j) &
          ncvts(c) = ncvts(c) + cv_lake(c,j)*(t_lake(c,j)-tfrz) &
                   + cfus*dz_lake(c,j)*(1._r8-lake_icefrac(c,j)) !&
!                   + (cwat-cice_eff)*lake_icefrac(c)*tfrz*dz_lake(c,j) !enthalpy reconciliation term
          fin(c) = fin(c) + phi(c,j)
       end do
    end do

    do j = -nlevsno + 1, nlevsoi
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

          if (j >= jtop(c)) then
!             ncvts(c) = ncvts(c) + cv(c,j)*t_soisno(c,j) &
             ncvts(c) = ncvts(c) + cv(c,j)*(t_soisno(c,j)-tfrz) &
                      + hfus*h2osoi_liq(c,j) !&
!                      + (cpliq-cpice)*h2osoi_ice(c,j)*tfrz !enthalpy reconciliation term
             if (j == 1 .and. h2osno(c) > 0._r8 .and. j == jtop(c)) then
                ncvts(c) = ncvts(c) - h2osno(c)*hfus
             end if
          end if
          if (j == 1) fin(c) = fin(c) + phi_soil(c)
       end do
    end do


    ! Check energy conservation.

!dir$ concurrent
!cdir nodep
    do fp = 1, num_shlakep
       p = filter_shlakep(fp)
       c = pcolumn(p)
       errsoi(c) = (ncvts(c)-ocvts(c)) / dtime - fin(c)
#ifndef CLMDEBUG
!       if (abs(errsoi(c)) < 0.10_r8) then ! else send to Balance Check and abort
       if (abs(errsoi(c)) < 10._r8) then ! else send to Balance Check and abort
#else
       if (abs(errsoi(c)) < 1._r8) then
#endif
          eflx_sh_tot(p) = eflx_sh_tot(p) - errsoi(c)
          eflx_sh_grnd(p) = eflx_sh_grnd(p) - errsoi(c)
          eflx_soil_grnd(p) = eflx_soil_grnd(p) + errsoi(c)
          eflx_gnet(p) = eflx_gnet(p) + errsoi(c)
!#if (defined CLMDEBUG)
!          if (abs(errsoi(c)) > 1.e-3_r8) then
          if (abs(errsoi(c)) > 1.e-1_r8) then
             write(6,*)'errsoi incorporated into sensible heat in ShalLakeTemperature: c, (W/m^2):', c, errsoi(c)
          end if
!#endif
          errsoi(c) = 0._r8
#if (defined CLMDEBUG)
       else
          write(6,*)'Soil Energy Balance Error at column, ', c, 'G, fintotal, column E tendency = ', &
             eflx_gnet(p), fin(c), (ncvts(c)-ocvts(c)) / dtime
#endif
       end if
    end do
    ! This loop assumes only one point per column.

  end subroutine ShalLakeTemperature

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: SoilThermProp_Lake
!
! !INTERFACE:
!  subroutine SoilThermProp_Lake (lbc, ubc,  num_shlakec, filter_shlakec, tk, cv, tktopsoillay)
 ! subroutine SoilThermProp_Lake (snl,watsat,tksatu,tkmg,tkdry,csol,         &
 !                                dz,zi,z,t_soisno,h2osoi_liq,h2osoi_ice,    &
 !                                tk, cv, tktopsoillay)
  subroutine SoilThermProp_Lake (snl,dz,zi,z,t_soisno,h2osoi_liq,h2osoi_ice,    &
                           tk, cv, tktopsoillay)

!
! !DESCRIPTION:
! Calculation of thermal conductivities and heat capacities of
! snow/soil layers
! (1) The volumetric heat capacity is calculated as a linear combination
!     in terms of the volumetric fraction of the constituent phases.
!
! (2) The thermal conductivity of soil is computed from the algorithm of
!     Johansen (as reported by Farouki 1981), and of snow is from the
!     formulation used in SNTHERM (Jordan 1991).
! The thermal conductivities at the interfaces between two neighboring
! layers (j, j+1) are derived from an assumption that the flux across
! the interface is equal to that from the node j to the interface and the
! flux from the interface to the node j+1.
!
! For lakes, the proper soil layers (not snow) should always be saturated.
!
! !USES:
    use lake_const 
#if (defined CLMDEBUG)
!    use abortutils,   only: endrun
#endif

    implicit none
!in

    integer , intent(in) :: snl(1)           ! number of snow layers
!    real(r8), intent(in) :: h2osno(1)        ! snow water (mm H2O)
   ! real(r8), intent(in) :: watsat(1,nlevsoi)      ! volumetric soil water at saturation (porosity)
   ! real(r8), intent(in) :: tksatu(1,nlevsoi)      ! thermal conductivity, saturated soil [W/m-K]
   ! real(r8), intent(in) :: tkmg(1,nlevsoi)        ! thermal conductivity, soil minerals  [W/m-K]
   ! real(r8), intent(in) :: tkdry(1,nlevsoi)       ! thermal conductivity, dry soil (W/m/Kelvin)
   ! real(r8), intent(in) :: csol(1,nlevsoi)        ! heat capacity, soil solids (J/m**3/Kelvin)
    real(r8), intent(in) :: dz(1,-nlevsno+1:nlevsoi)          ! layer thickness (m)
    real(r8), intent(in) :: zi(1,-nlevsno+0:nlevsoi)          ! interface level below a "z" level (m)
    real(r8), intent(in) :: z(1,-nlevsno+1:nlevsoi)           ! layer depth (m)
    real(r8), intent(in) :: t_soisno(1,-nlevsno+1:nlevsoi)    ! soil temperature (Kelvin)
    real(r8), intent(in) :: h2osoi_liq(1,-nlevsno+1:nlevsoi)  ! liquid water (kg/m2)
    real(r8), intent(in) :: h2osoi_ice(1,-nlevsno+1:nlevsoi)  ! ice lens (kg/m2)

!out
    real(r8), intent(out) :: cv(lbc:ubc,-nlevsno+1:nlevsoi) ! heat capacity [J/(m2 K)]
    real(r8), intent(out) :: tk(lbc:ubc,-nlevsno+1:nlevsoi) ! thermal conductivity [W/(m K)]
    real(r8), intent(out) :: tktopsoillay(lbc:ubc)          ! thermal conductivity [W/(m K)]
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! !CALLED FROM:
! subroutine ShalLakeTemperature in this module.
!
! !REVISION HISTORY:
! 15 September 1999: Yongjiu Dai; Initial code
! 15 December 1999:  Paul Houser and Jon Radakovich; F90 Revision
! 2/13/02, Peter Thornton: migrated to new data structures
! 7/01/03, Mariana Vertenstein: migrated to vector code
! 4/09, Zack Subin, adjustment for ShalLake code.
! 03/04/10,Hongping Gu and Jiming Jin, adjustment for coupling with WRF.
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! !LOCAL VARIABLES:
!
! local pointers to original implicit in scalars
!
!    integer , pointer :: clandunit(:)     ! column's landunit
!    integer , pointer :: ityplun(:)       ! landunit type
!
!EOP


! OTHER LOCAL VARIABLES:

    integer  :: l,c,j                     ! indices
    integer  :: fc                        ! lake filtered column indices
    real(r8) :: bw                        ! partial density of water (ice + liquid)
    real(r8) :: dksat                     ! thermal conductivity for saturated soil (j/(k s m))
    real(r8) :: dke                       ! kersten number
    real(r8) :: fl                        ! fraction of liquid or unfrozen water to total water
    real(r8) :: satw                      ! relative total water content of soil.
    real(r8) :: thk(lbc:ubc,-nlevsno+1:nlevsoi) ! thermal conductivity of layer

! Thermal conductivity of soil from Farouki (1981)

    do j = -nlevsno+1,nlevsoi
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

          ! Only examine levels from 1->nlevsoi
          if (j >= 1) then
!             l = clandunit(c)
!             if (ityplun(l) /= istwet .AND. ityplun(l) /= istice) then
              ! This could be altered later for allowing this to be over glaciers.

          ! Soil should be saturated.
#if (defined CLMDEBUG)
                satw = (h2osoi_liq(c,j)/denh2o + h2osoi_ice(c,j)/denice)/(dz(c,j)*watsat(c,j))
!                satw = min(1._r8, satw)
                if (satw < 0.999_r8) then
                   write(6,*)'WARNING: soil layer unsaturated in SoilThermProp_Lake, satw, j = ', satw, j
!                   call endrun()
                end if
          ! Could use denice because if it starts out frozen, the volume of water will go below sat.,
          ! since we're not yet doing excess ice.
          ! But take care of this in HydrologyLake.
#endif
                satw = 1._r8
                   fl = h2osoi_liq(c,j)/(h2osoi_ice(c,j)+h2osoi_liq(c,j))
                   if (t_soisno(c,j) >= tfrz) then       ! Unfrozen soil
                      dke = max(0._r8, log10(satw) + 1.0_r8)
                      dksat = tksatu(c,j)
                   else                               ! Frozen soil
                      dke = satw
                      dksat = tkmg(c,j)*0.249_r8**(fl*watsat(c,j))*2.29_r8**watsat(c,j)
                   endif
                   thk(c,j) = dke*dksat + (1._r8-dke)*tkdry(c,j)
!             else
!                thk(c,j) = tkwat
!                if (t_soisno(c,j) < tfrz) thk(c,j) = tkice
!             endif
          endif

          ! Thermal conductivity of snow, which from Jordan (1991) pp. 18
          ! Only examine levels from snl(c)+1 -> 0 where snl(c) < 1
          if (snl(c)+1 < 1 .AND. (j >= snl(c)+1) .AND. (j <= 0)) then
             bw = (h2osoi_ice(c,j)+h2osoi_liq(c,j))/dz(c,j)
             thk(c,j) = tkair + (7.75e-5_r8 *bw + 1.105e-6_r8*bw*bw)*(tkice-tkair)
          end if

       end do
    end do

    ! Thermal conductivity at the layer interface

    ! Have to correct for the fact that bottom snow layer and top soil layer border lake.
    ! For the first case, the snow layer conductivity for the middle of the layer will be returned.
    ! Because the interfaces are below the soil layers, the conductivity for the top soil layer
    ! will have to be returned separately.
    do j = -nlevsno+1,nlevsoi
!dir$ concurrent
!cdir nodep
       do fc = 1,num_shlakec
          c = filter_shlakec(fc)
          if (j >= snl(c)+1 .AND. j <= nlevsoi-1 .AND. j /= 0) then
             tk(c,j) = thk(c,j)*thk(c,j+1)*(z(c,j+1)-z(c,j)) &
                  /(thk(c,j)*(z(c,j+1)-zi(c,j))+thk(c,j+1)*(zi(c,j)-z(c,j)))
          else if (j == 0) then
             tk(c,j) = thk(c,j)
          else if (j == nlevsoi) then
             tk(c,j) = 0._r8
          end if
          ! For top soil layer.
          if (j == 1) tktopsoillay(c) = thk(c,j)
       end do
    end do

    ! Soil heat capacity, from de Vires (1963)

    do j = 1, nlevsoi
!dir$ concurrent
!cdir nodep
       do fc = 1,num_shlakec
          c = filter_shlakec(fc)
!          l = clandunit(c)
!          if (ityplun(l) /= istwet .AND. ityplun(l) /= istice) then
             cv(c,j) = csol(c,j)*(1-watsat(c,j))*dz(c,j) +   &
               (h2osoi_ice(c,j)*cpice + h2osoi_liq(c,j)*cpliq)
!          else
!             cv(c,j) = (h2osoi_ice(c,j)*cpice + h2osoi_liq(c,j)*cpliq)
!          endif
!          if (j == 1) then
!             if (snl(c)+1 == 1 .AND. h2osno(c) > 0._r8) then
!                cv(c,j) = cv(c,j) + cpice*h2osno(c)
!             end if
!          end if
       ! Won't worry about heat capacity for thin snow on lake with no snow layers.
       enddo
    end do

    ! Snow heat capacity

    do j = -nlevsno+1,0
!dir$ concurrent
!cdir nodep
       do fc = 1,num_shlakec
          c = filter_shlakec(fc)
          if (snl(c)+1 < 1 .and. j >= snl(c)+1) then
             cv(c,j) = cpliq*h2osoi_liq(c,j) + cpice*h2osoi_ice(c,j)
          end if
       end do
    end do

  end subroutine SoilThermProp_Lake


!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: PhaseChange_Lake
!
! !INTERFACE:
 ! subroutine PhaseChange_Lake (lbc, ubc, num_shlakec, filter_shlakec, cv, cv_lake, lhabs)
  subroutine PhaseChange_Lake (snl,h2osno,dz,dz_lake,                        & 
                               t_soisno,h2osoi_liq,h2osoi_ice,               & 
                               lake_icefrac,t_lake, snowdp,                  & 
                               qflx_snomelt,eflx_snomelt,imelt,              &   
                               cv, cv_lake,                                  & 
                               lhabs)                                          
!=============================================================================================
! !DESCRIPTION:
! Calculation of the phase change within snow, soil, & lake layers:
! (1) Check the conditions for which the phase change may take place,
!     i.e., the layer temperature is great than the freezing point
!     and the ice mass is not equal to zero (i.e. melting),
!     or the layer temperature is less than the freezing point
!     and the liquid water mass is greater than the allowable supercooled 
!    (i.e. freezing).
! (2) Assess the amount of phase change from the energy excess (or deficit)
!     after setting the layer temperature to freezing point, depending on
!     how much water or ice is available.
! (3) Re-adjust the ice and liquid mass, and the layer temperature: either to
!     the freezing point if enough water or ice is available to fully compensate,
!     or to a remaining temperature.
! The specific heats are assumed constant. Potential cycling errors resulting from
! this assumption will be trapped at the end of ShalLakeTemperature.
! !CALLED FROM:
! subroutine ShalLakeTemperature in this module
!
! !REVISION HISTORY:
! 04/2009 Zack Subin: Initial code
! 03/2010 Hongping Gu and Jiming Jin, Change for coupling with WRF
!==============================================================================================
! !USES:
    use lake_const 
!
! !ARGUMENTS:
    implicit none
!in: 

    integer , intent(in) :: snl(1)           !number of snow layers
    real(r8), intent(in) :: h2osno(1)        !snow water (mm H2O)
    real(r8), intent(in) :: dz(1,-nlevsno+1:nlevsoi)          !layer thickness (m)
    real(r8), intent(in) :: dz_lake(1,nlevlak)     !lake layer thickness (m)
    ! Needed in case snow height is less than critical value.

!inout: 

    real(r8), intent(inout) :: snowdp(1)        !snow height (m)
    real(r8), intent(inout) :: t_soisno(1,-nlevsno+1:nlevsoi)     !soil temperature (Kelvin)
    real(r8), intent(inout) :: h2osoi_liq(1,-nlevsno+1:nlevsoi)   !liquid water (kg/m2)
    real(r8), intent(inout) :: h2osoi_ice(1,-nlevsno+1:nlevsoi)   !ice lens (kg/m2)
    real(r8), intent(inout) :: lake_icefrac(1,nlevlak) ! mass fraction of lake layer that is frozen
    real(r8), intent(inout) :: t_lake(1,nlevlak)       ! lake temperature (Kelvin)
!out: 

    real(r8), intent(out) :: qflx_snomelt(1)  !snow melt (mm H2O /s)
    real(r8), intent(out) :: eflx_snomelt(1)  !snow melt heat flux (W/m**2)
    integer, intent(out)  :: imelt(1,-nlevsno+1:nlevsoi)        !flag for melting (=1), freezing (=2), Not=0 (new)
                                          !What's the sign of this? Is it just output?
    real(r8), intent(inout) :: cv(lbc:ubc,-nlevsno+1:nlevsoi)       ! heat capacity [J/(m2 K)]
    real(r8), intent(inout) :: cv_lake (lbc:ubc,1:nlevlak)          ! heat capacity [J/(m2 K)]
    real(r8), intent(out):: lhabs(lbc:ubc)                       ! total per-column latent heat abs. (J/m^2)


! OTHER LOCAL VARIABLES:

    integer  :: j,c,g                              !do loop index
    integer  :: fc                                 !lake filtered column indices
!    real(r8) :: dtime                              !land model time step (sec)
    real(r8) :: heatavail                          !available energy for melting or freezing (J/m^2)
    real(r8) :: heatrem                            !energy residual or loss after melting or freezing
    real(r8) :: melt                               !actual melting (+) or freezing (-) [kg/m2]
    real(r8), parameter :: smallnumber = 1.e-7_r8  !to prevent tiny residuals from rounding error
    logical  :: dophasechangeflag
!-----------------------------------------------------------------------

!    dtime = get_step_size()

    ! Initialization

!dir$ concurrent
!cdir nodep
    do fc = 1,num_shlakec
       c = filter_shlakec(fc)

       qflx_snomelt(c) = 0._r8
       eflx_snomelt(c) = 0._r8
       lhabs(c)        = 0._r8
    end do

    do j = -nlevsno+1,0
!dir$ concurrent
!cdir nodep
       do fc = 1,num_shlakec
          c = filter_shlakec(fc)

          if (j >= snl(c) + 1) imelt(c,j) = 0
       end do
    end do

    ! Check for case of snow without snow layers and top lake layer temp above freezing.

!dir$ concurrent
!cdir nodep
    do fc = 1,num_shlakec
       c = filter_shlakec(fc)

       if (snl(c) == 0 .and. h2osno(c) > 0._r8 .and. t_lake(c,1) > tfrz) then
          heatavail = (t_lake(c,1) - tfrz) * cv_lake(c,1)
          melt = min(h2osno(c), heatavail/hfus)
          heatrem = max(heatavail - melt*hfus, 0._r8)
                       !catch small negative value to keep t at tfrz
          t_lake(c,1) = tfrz + heatrem/(cv_lake(c,1))
          snowdp(c) = snowdp(c)*(1._r8 - melt/h2osno(c))
          h2osno(c) = h2osno(c) - melt
          lhabs(c) = lhabs(c) + melt*hfus
          qflx_snomelt(c) = qflx_snomelt(c) + melt
          ! Prevent tiny residuals
          if (h2osno(c) < smallnumber) h2osno(c) = 0._r8
          if (snowdp(c) < smallnumber) snowdp(c) = 0._r8
       end if
    end do

    ! Lake phase change

    do j = 1,nlevlak
!dir$ concurrent
!cdir nodep
       do fc = 1,num_shlakec
          c = filter_shlakec(fc)

          dophasechangeflag = .false.
          if (t_lake(c,j) > tfrz .and. lake_icefrac(c,j) > 0._r8) then ! melting
             dophasechangeflag = .true.
             heatavail = (t_lake(c,j) - tfrz) * cv_lake(c,j)
             melt = min(lake_icefrac(c,j)*denh2o*dz_lake(c,j), heatavail/hfus)
                        !denh2o is used because layer thickness is not adjusted for freezing
             heatrem = max(heatavail - melt*hfus, 0._r8)
                       !catch small negative value to keep t at tfrz
          else if (t_lake(c,j) < tfrz .and. lake_icefrac(c,j) < 1._r8) then !freezing
             dophasechangeflag = .true.
             heatavail = (t_lake(c,j) - tfrz) * cv_lake(c,j)
             melt = max(-(1._r8-lake_icefrac(c,j))*denh2o*dz_lake(c,j), heatavail/hfus)
                        !denh2o is used because layer thickness is not adjusted for freezing
             heatrem = min(heatavail - melt*hfus, 0._r8)
                       !catch small positive value to keep t at tfrz
          end if
          ! Update temperature and ice fraction.
          if (dophasechangeflag) then
             lake_icefrac(c,j) = lake_icefrac(c,j) - melt/(denh2o*dz_lake(c,j))
             lhabs(c) = lhabs(c) + melt*hfus
          ! Update heat capacity
             cv_lake(c,j) = cv_lake(c,j) + melt*(cpliq-cpice)
             t_lake(c,j) = tfrz + heatrem/cv_lake(c,j)
             ! Prevent tiny residuals
             if (lake_icefrac(c,j) > 1._r8 - smallnumber) lake_icefrac(c,j) = 1._r8
             if (lake_icefrac(c,j) < smallnumber)         lake_icefrac(c,j) = 0._r8
          end if
       end do
    end do

    ! Snow & soil phase change

    do j = -nlevsno+1,nlevsoi
!dir$ concurrent
!cdir nodep
       do fc = 1,num_shlakec
          c = filter_shlakec(fc)
          dophasechangeflag = .false.

          if (j >= snl(c) + 1) then

             if (t_soisno(c,j) > tfrz .and. h2osoi_ice(c,j) > 0._r8) then ! melting
                dophasechangeflag = .true.
                heatavail = (t_soisno(c,j) - tfrz) * cv(c,j)
                melt = min(h2osoi_ice(c,j), heatavail/hfus)
                heatrem = max(heatavail - melt*hfus, 0._r8)
                          !catch small negative value to keep t at tfrz
                if (j <= 0) then !snow
                   imelt(c,j) = 1
                   qflx_snomelt(c) = qflx_snomelt(c) + melt
                end if
             else if (t_soisno(c,j) < tfrz .and. h2osoi_liq(c,j) > 0._r8) then !freezing
                dophasechangeflag = .true.
                heatavail = (t_soisno(c,j) - tfrz) * cv(c,j)
                melt = max(-h2osoi_liq(c,j), heatavail/hfus)
                heatrem = min(heatavail - melt*hfus, 0._r8)
                          !catch small positive value to keep t at tfrz
                if (j <= 0) then !snow
                   imelt(c,j) = 2
                   qflx_snomelt(c) = qflx_snomelt(c) + melt
                   ! Does this works for both signs of melt in SnowHydrology? I think
                   ! qflx_snomelt(c) is just output.
                end if
             end if

             ! Update temperature and soil components.
             if (dophasechangeflag) then
                h2osoi_ice(c,j) = h2osoi_ice(c,j) - melt
                h2osoi_liq(c,j) = h2osoi_liq(c,j) + melt
                lhabs(c) = lhabs(c) + melt*hfus
             ! Update heat capacity
                cv(c,j) = cv(c,j) + melt*(cpliq-cpice)
                t_soisno(c,j) = tfrz + heatrem/cv(c,j)
                ! Prevent tiny residuals
                if (h2osoi_ice(c,j) < smallnumber) h2osoi_ice(c,j) = 0._r8
                if (h2osoi_liq(c,j) < smallnumber) h2osoi_liq(c,j) = 0._r8
             end if

         end if
      end do
   end do

   ! Update eflx_snomelt(c)
!dir$ concurrent
!cdir nodep
    do fc = 1,num_shlakec
       c = filter_shlakec(fc)
       eflx_snomelt(c) = qflx_snomelt(c)*hfus
    end do
!!!

   end subroutine PhaseChange_Lake


  subroutine ShalLakeHydrology(dz_lake,forc_rain,forc_snow,                      & !i
                               begwb,qflx_evap_tot,forc_t,do_capsnow,            &
                               t_grnd,qflx_evap_soi,                             &
                               qflx_snomelt,imelt,frac_iceold,                   & !i add by guhp
                               z,dz,zi,snl,h2osno,snowdp,lake_icefrac,t_lake,      & !i&o
                               endwb,snowage,snowice,snowliq,t_snow,             & !o
                               t_soisno,h2osoi_ice,h2osoi_liq,h2osoi_vol,        &
                               qflx_drain,qflx_surf,qflx_infl,qflx_qrgwl,        &
                               qcharge,qflx_prec_grnd,qflx_snowcap,              &
                               qflx_snowcap_col,qflx_snow_grnd_pft,              &
                               qflx_snow_grnd_col,qflx_rain_grnd,                &
                               qflx_evap_tot_col,soilalpha,zwt,fcov,             &
                               rootr_column,qflx_evap_grnd,qflx_sub_snow,        &
                               qflx_dew_snow,qflx_dew_grnd,qflx_rain_grnd_col)
                       
!==================================================================================
! !DESCRIPTION:
! Calculation of Shallow Lake Hydrology. Full hydrology of snow layers is
! done. However, there is no infiltration, and the water budget is balanced with 
! qflx_qrgwl. Lake water mass is kept constant. The soil is simply maintained at
! volumetric saturation if ice melting frees up pore space. Likewise, if the water
! portion alone at some point exceeds pore capacity, it is reduced. This is consistent
! with the possibility of initializing the soil layer with excess ice. The only
! real error with that is that the thermal conductivity will ignore the excess ice
! (and accompanying thickness change).
! 
! If snow layers are present over an unfrozen lake, and the top layer of the lake
! is capable of absorbing the latent heat without going below freezing, 
! the snow-water is runoff and the latent heat is subtracted from the lake.
!
! WARNING: This subroutine assumes lake columns have one and only one pft.
!
! Sequence is:
!  ShalLakeHydrology:
!    Do needed tasks from Hydrology1, Biogeophysics2, & top of Hydrology2.
!    -> SnowWater:             change of snow mass and snow water onto soil
!    -> SnowCompaction:        compaction of snow layers
!    -> CombineSnowLayers:     combine snow layers that are thinner than minimum
!    -> DivideSnowLayers:      subdivide snow layers that are thicker than maximum
!    Add water to soil if melting has left it with open pore space.
!    Cleanup and do water balance.
!    If snow layers are found above a lake with unfrozen top layer, whose top
!    layer has enough heat to melt all the snow ice without freezing, do so
!    and eliminate the snow layers.
!
! !REVISION HISTORY:
! Created by Zack Subin, 2009
! Changed by guhp for coupling,2010
!
!============================================================================================

! USES:
    use lake_const
    use SnowHydro_lake  , only : SnowCompaction, CombineSnowLayers, DivideSnowLayers, &
                                 SnowWater, BuildSnowFilter
!    use clm_time_manager, only : get_step_size, is_perpetual
 !   use globals, only : dtime
#if (defined CLMDEBUG)
!    use abortutils,   only: endrun
#endif
!
    implicit none

! in:

   ! integer , intent(in) :: clandunit(1)     ! column's landunit
   ! integer , intent(in) :: ityplun(1)       ! landunit type
   ! real(r8), intent(in) :: watsat(1,1:nlevsoi)      ! volumetric soil water at saturation (porosity)
    real(r8), intent(in) :: dz_lake(1,nlevlak)     ! layer thickness for lake (m)
    real(r8), intent(in) :: forc_rain(1)     ! rain rate [mm/s]
    real(r8), intent(in) :: forc_snow(1)     ! snow rate [mm/s]
    real(r8), intent(in) :: qflx_evap_tot(1) ! qflx_evap_soi + qflx_evap_veg + qflx_tran_veg
    real(r8), intent(in) :: forc_t(1)        ! atmospheric temperature (Kelvin)
#if (defined OFFLINE)
    real(r8), intent(in) :: flfall(1)        ! fraction of liquid water within falling precipitation
#endif
    logical , intent(in) :: do_capsnow(1)     ! true => do snow capping
    real(r8), intent(in) :: t_grnd(1)          ! ground temperature (Kelvin)
    real(r8), intent(in) :: qflx_evap_soi(1)   ! soil evaporation (mm H2O/s) (+ = to atm)
    real(r8), intent(in) :: qflx_snomelt(1)     !snow melt (mm H2O /s)
    integer,  intent(in) :: imelt(1,-nlevsno+1:nlevsoi)        !flag for melting (=1), freezing (=2), Not=0

!inout:

    real(r8), intent(inout) :: begwb(1)         ! water mass begining of the time step

! inout: 

    
    real(r8), intent(inout) :: z(1,-nlevsno+1:nlevsoi)           ! layer depth  (m)
    real(r8), intent(inout) :: dz(1,-nlevsno+1:nlevsoi)          ! layer thickness depth (m)
    real(r8), intent(inout) :: zi(1,-nlevsno+0:nlevsoi)          ! interface depth (m)
    integer , intent(inout) :: snl(1)           ! number of snow layers
    real(r8), intent(inout) :: h2osno(1)        ! snow water (mm H2O)
    real(r8), intent(inout) :: snowdp(1)        ! snow height (m)
    real(r8), intent(inout) :: lake_icefrac(1,nlevlak)  ! mass fraction of lake layer that is frozen
    real(r8), intent(inout) :: t_lake(1,nlevlak)        ! lake temperature (Kelvin)

    real(r8), intent(inout) :: frac_iceold(1,-nlevsno+1:nlevsoi)      ! fraction of ice relative to the tot water
! out: 


    real(r8), intent(out) :: endwb(1)         ! water mass end of the time step
    real(r8), intent(out) :: snowage(1)       ! non dimensional snow age [-]
    real(r8), intent(out) :: snowice(1)       ! average snow ice lens
    real(r8), intent(out) :: snowliq(1)       ! average snow liquid water
    real(r8), intent(out) :: t_snow(1)        ! vertically averaged snow temperature
    real(r8), intent(out) :: t_soisno(1,-nlevsno+1:nlevsoi)    ! snow temperature (Kelvin)
    real(r8), intent(out) :: h2osoi_ice(1,-nlevsno+1:nlevsoi)  ! ice lens (kg/m2)
    real(r8), intent(out) :: h2osoi_liq(1,-nlevsno+1:nlevsoi)  ! liquid water (kg/m2)
    real(r8), intent(out) :: h2osoi_vol(1,-nlevsno+1:nlevsoi)  ! volumetric soil water (0<=h2osoi_vol<=watsat)[m3/m3]
    real(r8), intent(out) :: qflx_drain(1)    ! sub-surface runoff (mm H2O /s)
    real(r8), intent(out) :: qflx_surf(1)     ! surface runoff (mm H2O /s)
    real(r8), intent(out) :: qflx_infl(1)     ! infiltration (mm H2O /s)
    real(r8), intent(out) :: qflx_qrgwl(1)    ! qflx_surf at glaciers, wetlands, lakes
    real(r8), intent(out) :: qcharge(1)       ! aquifer recharge rate (mm/s)
    real(r8), intent(out) :: qflx_prec_grnd(1)     ! water onto ground including canopy runoff [kg/(m2 s)]
    real(r8), intent(out) :: qflx_snowcap(1)       ! excess precipitation due to snow capping (mm H2O /s) [+]
    real(r8), intent(out) :: qflx_snowcap_col(1)   ! excess precipitation due to snow capping (mm H2O /s) [+]
    real(r8), intent(out) :: qflx_snow_grnd_pft(1) ! snow on ground after interception (mm H2O/s) [+]
    real(r8), intent(out) :: qflx_snow_grnd_col(1) ! snow on ground after interception (mm H2O/s) [+]
    real(r8), intent(out) :: qflx_rain_grnd(1)     ! rain on ground after interception (mm H2O/s) [+]
    real(r8), intent(out) :: qflx_evap_tot_col(1) !pft quantity averaged to the column (assuming one pft)
    real(r8) ,intent(out) :: soilalpha(1)     !factor that reduces ground saturated specific humidity (-)
    real(r8), intent(out) :: zwt(1)           !water table depth
    real(r8), intent(out) :: fcov(1)          !fractional area with water table at surface
    real(r8), intent(out) :: rootr_column(1,1:nlevsoi) !effective fraction of roots in each soil layer
    real(r8), intent(out) :: qflx_evap_grnd(1)  ! ground surface evaporation rate (mm H2O/s) [+]
    real(r8), intent(out) :: qflx_sub_snow(1)   ! sublimation rate from snow pack (mm H2O /s) [+]
    real(r8), intent(out) :: qflx_dew_snow(1)   ! surface dew added to snow pack (mm H2O /s) [+]
    real(r8), intent(out) :: qflx_dew_grnd(1)   ! ground surface dew formation (mm H2O /s) [+]
    real(r8), intent(out) :: qflx_rain_grnd_col(1)   !rain on ground after interception (mm H2O/s) [+]

! Block of biogeochem currently not used.
#ifndef SHLAKE
    real(r8), pointer :: sucsat(:,:)      ! minimum soil suction (mm)
    real(r8), pointer :: bsw(:,:)         ! Clapp and Hornberger "b"
    real(r8), pointer :: bsw2(:,:)        ! Clapp and Hornberger "b" for CN code
    real(r8), pointer :: psisat(:,:)      ! soil water potential at saturation for CN code (MPa)
    real(r8), pointer :: vwcsat(:,:)      ! volumetric water content at saturation for CN code (m3/m3)
    real(r8), pointer :: wf(:)            ! soil water as frac. of whc for top 0.5 m
    real(r8), pointer :: soilpsi(:,:)     ! soil water potential in each soil layer (MPa)
    real(r8) :: psi,vwc,fsat               ! temporary variables for soilpsi calculation
#if (defined DGVM) || (defined CN)
    real(r8) :: watdry                     ! temporary
    real(r8) :: rwat(lbc:ubc)              ! soil water wgted by depth to maximum depth of 0.5 m
    real(r8) :: swat(lbc:ubc)              ! same as rwat but at saturation
    real(r8) :: rz(lbc:ubc)                ! thickness of soil layers contributing to rwat (m)
    real(r8) :: tsw                        ! volumetric soil water to 0.5 m
    real(r8) :: stsw                       ! volumetric soil water to 0.5 m at saturation
#endif
#endif


! OTHER LOCAL VARIABLES:

    integer  :: p,fp,g,l,c,j,fc,jtop             ! indices
    integer  :: num_shlakesnowc                  ! number of column snow points
    integer  :: filter_shlakesnowc(ubc-lbc+1)    ! column filter for snow points
    integer  :: num_shlakenosnowc                ! number of column non-snow points
    integer  :: filter_shlakenosnowc(ubc-lbc+1)  ! column filter for non-snow points
!    real(r8) :: dtime                      ! land model time step (sec)
    integer  :: newnode                      ! flag when new snow node is set, (1=yes, 0=no)
    real(r8) :: dz_snowf                     ! layer thickness rate change due to precipitation [mm/s]
    real(r8) :: bifall                       ! bulk density of newly fallen dry snow [kg/m3]
    real(r8) :: fracsnow(lbp:ubp)            ! frac of precipitation that is snow
    real(r8) :: fracrain(lbp:ubp)            ! frac of precipitation that is rain
    real(r8) :: qflx_prec_grnd_snow(lbp:ubp) ! snow precipitation incident on ground [mm/s]
    real(r8) :: qflx_prec_grnd_rain(lbp:ubp) ! rain precipitation incident on ground [mm/s]
    real(r8) :: qflx_evap_soi_lim            ! temporary evap_soi limited by top snow layer content [mm/s]
    real(r8) :: h2osno_temp                  ! temporary h2osno [kg/m^2]
    real(r8), parameter :: snow_bd = 250._r8  !constant snow bulk density (only used in special case here) [kg/m^3]
    real(r8) :: sumsnowice(lbc:ubc)             ! sum of snow ice if snow layers found above unfrozen lake [kg/m&2]
    logical  :: unfrozen(lbc:ubc)            ! true if top lake layer is unfrozen with snow layers above
    real(r8) :: heatrem                      ! used in case above [J/m^2]
    real(r8) :: heatsum(lbc:ubc)             ! used in case above [J/m^2]
    real(r8) :: qflx_top_soil(1)     !net water input into soil from top (mm/s)

#if (defined CLMDEBUG)
    real(r8) :: snow_water(lbc:ubc)           ! temporary sum of snow water for Bal Check [kg/m^2]
#endif
!-----------------------------------------------------------------------


    ! Determine step size

!    dtime = get_step_size()

    ! Add soil water to water balance.
    do j = 1, nlevsoi
!dir$ concurrent
!cdir nodep
      do fc = 1, num_shlakec
         c = filter_shlakec(fc)
         begwb(c) = begwb(c) + h2osoi_ice(c,j) + h2osoi_liq(c,j)
      end do
    end do

!!!!!!!!!!!!!!!!!!!!!!!!!!!

    ! Do precipitation onto ground, etc., from Hydrology1.

!dir$ concurrent
!cdir nodep
    do fp = 1, num_shlakep
       p = filter_shlakep(fp)
       g = pgridcell(p)
!       l = plandunit(p)
       c = pcolumn(p)

       ! Precipitation onto ground (kg/(m2 s))
!       ! PET, 1/18/2005: Added new terms for mass balance correction
!       ! due to dynamic pft weight shifting (column-level h2ocan_loss)
!       ! Because the fractionation between rain and snow is indeterminate if
!       ! rain + snow = 0, I am adding this very small flux only to the rain
!       ! components.
       ! Not relevant unless PFTs are added to lake later.
!       if (frac_veg_nosno(p) == 0) then
          qflx_prec_grnd_snow(p) = forc_snow(g)
          qflx_prec_grnd_rain(p) = forc_rain(g) !+ h2ocan_loss(c)
!       else
!          qflx_prec_grnd_snow(p) = qflx_through_snow(p) + (qflx_candrip(p) * fracsnow(p))
!          qflx_prec_grnd_rain(p) = qflx_through_rain(p) + (qflx_candrip(p) * fracrain(p)) + h2ocan_loss(c)
!       end if
       qflx_prec_grnd(p) = qflx_prec_grnd_snow(p) + qflx_prec_grnd_rain(p)

       if (do_capsnow(c)) then
          qflx_snowcap(p) = qflx_prec_grnd_snow(p) + qflx_prec_grnd_rain(p)
          qflx_snow_grnd_pft(p) = 0._r8
          qflx_rain_grnd(p) = 0._r8
       else
          qflx_snowcap(p) = 0._r8
#if (defined OFFLINE)
          qflx_snow_grnd_pft(p) = qflx_prec_grnd(p)*(1._r8-flfall(g)) ! ice onto ground (mm/s)
          qflx_rain_grnd(p)     = qflx_prec_grnd(p)*flfall(g)      ! liquid water onto ground (mm/s)
#else
          qflx_snow_grnd_pft(p) = qflx_prec_grnd_snow(p)           ! ice onto ground (mm/s)
          qflx_rain_grnd(p)     = qflx_prec_grnd_rain(p)           ! liquid water onto ground (mm/s)
#endif
       end if
       ! Assuming one PFT; needed for below
       qflx_snow_grnd_col(c) = qflx_snow_grnd_pft(p)
       qflx_rain_grnd_col(c) = qflx_rain_grnd(p)

    end do ! (end pft loop)

    ! Determine snow height and snow water

!dir$ concurrent
!cdir nodep
    do fc = 1, num_shlakec
       c = filter_shlakec(fc)
!       l = clandunit(c)
       g = cgridcell(c)

       ! Use Alta relationship, Anderson(1976); LaChapelle(1961),
       ! U.S.Department of Agriculture Forest Service, Project F,
       ! Progress Rep. 1, Alta Avalanche Study Center:Snow Layer Densification.

       if (do_capsnow(c)) then
          dz_snowf = 0._r8
       else
          if (forc_t(g) > tfrz + 2._r8) then
             bifall=50._r8 + 1.7_r8*(17.0_r8)**1.5_r8
          else if (forc_t(g) > tfrz - 15._r8) then
             bifall=50._r8 + 1.7_r8*(forc_t(g) - tfrz + 15._r8)**1.5_r8
          else
             bifall=50._r8
          end if
          dz_snowf = qflx_snow_grnd_col(c)/bifall
          snowdp(c) = snowdp(c) + dz_snowf*dtime
          h2osno(c) = h2osno(c) + qflx_snow_grnd_col(c)*dtime  ! snow water equivalent (mm)
       end if

!       if (itype(l)==istwet .and. t_grnd(c)>tfrz) then
!          h2osno(c)=0._r8
!          snowdp(c)=0._r8
!          snowage(c)=0._r8
!       end if
       ! Take care of this later in function.

       ! When the snow accumulation exceeds 10 mm, initialize snow layer
       ! Currently, the water temperature for the precipitation is simply set
       ! as the surface air temperature

       newnode = 0    ! flag for when snow node will be initialized
       if (snl(c) == 0 .and. qflx_snow_grnd_col(c) > 0.0_r8 .and. snowdp(c) >= 0.01_r8) then
          newnode = 1
          snl(c) = -1
          dz(c,0) = snowdp(c)                       ! meter
          z(c,0) = -0.5_r8*dz(c,0)
          zi(c,-1) = -dz(c,0)
          snowage(c) = 0._r8                        ! snow age
          t_soisno(c,0) = min(tfrz, forc_t(g))      ! K
          h2osoi_ice(c,0) = h2osno(c)               ! kg/m2
          h2osoi_liq(c,0) = 0._r8                   ! kg/m2
          frac_iceold(c,0) = 1._r8
       end if

       ! The change of ice partial density of surface node due to precipitation.
       ! Only ice part of snowfall is added here, the liquid part will be added
       ! later.

       if (snl(c) < 0 .and. newnode == 0) then
          h2osoi_ice(c,snl(c)+1) = h2osoi_ice(c,snl(c)+1)+dtime*qflx_snow_grnd_col(c)
          dz(c,snl(c)+1) = dz(c,snl(c)+1)+dz_snowf*dtime
       end if

    end do

    ! Calculate sublimation and dew, adapted from HydrologyLake and Biogeophysics2.

!dir$ concurrent
!cdir nodep
    do fp = 1,num_shlakep
       p = filter_shlakep(fp)
       c = pcolumn(p)
       jtop = snl(c)+1

       ! Use column variables here
       qflx_evap_grnd(c) = 0._r8
       qflx_sub_snow(c) = 0._r8
       qflx_dew_snow(c) = 0._r8
       qflx_dew_grnd(c) = 0._r8

       if (jtop <= 0) then ! snow layers
          j = jtop
          ! Assign ground evaporation to sublimation from soil ice or to dew
          ! on snow or ground

          if (qflx_evap_soi(p) >= 0._r8) then
          ! for evaporation partitioning between liquid evap and ice sublimation, 
          ! use the ratio of liquid to (liquid+ice) in the top layer to determine split
          ! Since we're not limiting evap over lakes, but still can't remove more from top
          ! snow layer than there is there, create temp. limited evap_soi.
             qflx_evap_soi_lim = min(qflx_evap_soi(p), (h2osoi_liq(c,j)+h2osoi_ice(c,j))/dtime)
             if ((h2osoi_liq(c,j)+h2osoi_ice(c,j)) > 0._r8) then
                qflx_evap_grnd(c) = max(qflx_evap_soi_lim*(h2osoi_liq(c,j)/(h2osoi_liq(c,j)+h2osoi_ice(c,j))), 0._r8)
             else
                qflx_evap_grnd(c) = 0._r8
             end if
             qflx_sub_snow(c) = qflx_evap_soi_lim - qflx_evap_grnd(c)     
          else
             if (t_grnd(c) < tfrz) then
                qflx_dew_snow(c) = abs(qflx_evap_soi(p))
             else
                qflx_dew_grnd(c) = abs(qflx_evap_soi(p))
             end if
          end if
          ! Update the pft-level qflx_snowcap
          ! This was moved in from Hydrology2 to keep all pft-level
          ! calculations out of Hydrology2
          if (do_capsnow(c)) qflx_snowcap(p) = qflx_snowcap(p) + qflx_dew_snow(c) + qflx_dew_grnd(c)

       else ! No snow layers: do as in HydrologyLake but with actual clmtype variables
          if (qflx_evap_soi(p) >= 0._r8) then
             ! Sublimation: do not allow for more sublimation than there is snow
             ! after melt.  Remaining surface evaporation used for infiltration.
             qflx_sub_snow(c) = min(qflx_evap_soi(p), h2osno(c)/dtime)
             qflx_evap_grnd(c) = qflx_evap_soi(p) - qflx_sub_snow(c)
          else
             if (t_grnd(c) < tfrz-0.1_r8) then
                qflx_dew_snow(c) = abs(qflx_evap_soi(p))
             else
                qflx_dew_grnd(c) = abs(qflx_evap_soi(p))
             end if
          end if

          ! Update snow pack for dew & sub.
          h2osno_temp = h2osno(c)
          if (do_capsnow(c)) then
             h2osno(c) = h2osno(c) - qflx_sub_snow(c)*dtime
             qflx_snowcap(p) = qflx_snowcap(p) + qflx_dew_snow(c) + qflx_dew_grnd(c)
          else
             h2osno(c) = h2osno(c) + (-qflx_sub_snow(c)+qflx_dew_snow(c))*dtime
          end if
          if (h2osno_temp > 0._r8) then
             snowdp(c) = snowdp(c) * h2osno(c) / h2osno_temp
          else
             snowdp(c) = h2osno(c)/snow_bd !Assume a constant snow bulk density = 250.
          end if

#if (defined PERGRO)
          if (abs(h2osno(c)) < 1.e-10_r8) h2osno(c) = 0._r8
#else
          h2osno(c) = max(h2osno(c), 0._r8)
#endif

       end if

    qflx_snowcap_col(c) = qflx_snowcap(p)

    end do


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    ! Determine initial snow/no-snow filters (will be modified possibly by
    ! routines CombineSnowLayers and DivideSnowLayers below

    call BuildSnowFilter(lbc, ubc, num_shlakec, filter_shlakec,snl,       &            !i
         num_shlakesnowc, filter_shlakesnowc, num_shlakenosnowc, filter_shlakenosnowc) !o

    ! Determine the change of snow mass and the snow water onto soil

    call SnowWater(lbc, ubc, num_shlakesnowc, filter_shlakesnowc,         & !i 
                   num_shlakenosnowc, filter_shlakenosnowc,               & !i 
                   snl,do_capsnow,qflx_snomelt,qflx_rain_grnd,            & !i 
                   qflx_sub_snow,qflx_evap_grnd,                          & !i   
                   qflx_dew_snow,qflx_dew_grnd,dz,                        & !i   
                   h2osoi_ice,h2osoi_liq,                                 & !i&o 
                   qflx_top_soil)                                           !o                        


    ! Determine soil hydrology
    ! Here this consists only of making sure that soil is saturated even as it melts and 10%
    ! of pore space opens up. Conversely, if excess ice is melting and the liquid water exceeds the
    ! saturation value, then remove water.

    do j = 1,nlevsoi
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

          if (h2osoi_vol(c,j) < watsat(c,j)) then
             h2osoi_liq(c,j) = (watsat(c,j)*dz(c,j) - h2osoi_ice(c,j)/denice)*denh2o
          ! h2osoi_vol will be updated below, and this water addition will come from qflx_qrgwl
          else if (h2osoi_liq(c,j) > watsat(c,j)*denh2o*dz(c,j)) then
             h2osoi_liq(c,j) = watsat(c,j)*denh2o*dz(c,j)
          end if

       end do
    end do
!!!!!!!!!!

!    if (.not. is_perpetual()) then
    if (1==1) then

       ! Natural compaction and metamorphosis.

       call SnowCompaction(lbc, ubc, num_shlakesnowc, filter_shlakesnowc,   &!i
                           snl,imelt,frac_iceold,t_soisno,                  &!i
                           h2osoi_ice,h2osoi_liq,                           &!i
                           dz)                                               !&o

       ! Combine thin snow elements

       call CombineSnowLayers(lbc, ubc,                            & !i
                              num_shlakesnowc, filter_shlakesnowc, & !i&o
                              snl,h2osno,snowdp,dz,zi,             & !i&o
                              t_soisno,h2osoi_ice,h2osoi_liq,      & !i&o
                              z)  !o                              


       ! Divide thick snow elements

       call DivideSnowLayers(lbc, ubc,                             & !i
                             num_shlakesnowc, filter_shlakesnowc,  & !i&o
                             snl,dz,zi,t_soisno,                   & !i&o
                             h2osoi_ice,h2osoi_liq,                & !i&o
                             z)  !o


    else

       do fc = 1, num_shlakesnowc
          c = filter_shlakesnowc(fc)
          h2osno(c) = 0._r8
       end do
       do j = -nlevsno+1,0
          do fc = 1, num_shlakesnowc
             c = filter_shlakesnowc(fc)
             if (j >= snl(c)+1) then
                h2osno(c) = h2osno(c) + h2osoi_ice(c,j) + h2osoi_liq(c,j)
             end if
          end do
       end do

    end if

    ! Check for snow layers above lake with unfrozen top layer.  Mechanically,
    ! the snow will fall into the lake and melt or turn to ice.  If the top layer has
    ! sufficient heat to melt the snow without freezing, then that will be done.
    ! Otherwise, the top layer will undergo freezing, but only if the top layer will
    ! not freeze completely.  Otherwise, let the snow layers persist and melt by diffusion.
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

          if (t_lake(c,1) > tfrz .and. lake_icefrac(c,1) == 0._r8 .and. snl(c) < 0) then
             unfrozen(c) = .true.
          else
             unfrozen(c) = .false.
          end if
       end do

    do j = -nlevsno+1,0
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

          if (unfrozen(c)) then
             if (j == -nlevsno+1) then
                sumsnowice(c) = 0._r8
                heatsum(c) = 0._r8
             end if
             if (j >= snl(c)+1) then
                sumsnowice(c) = sumsnowice(c) + h2osoi_ice(c,j)
                heatsum(c) = heatsum(c) + h2osoi_ice(c,j)*cpice*(tfrz - t_soisno(c,j)) &
                           + h2osoi_liq(c,j)*cpliq*(tfrz - t_soisno(c,j))
             end if
          end if
       end do
    end do

!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)

          if (unfrozen(c)) then
             heatsum(c) = heatsum(c) + sumsnowice(c)*hfus
             heatrem = (t_lake(c,1) - tfrz)*cpliq*denh2o*dz_lake(c,1) - heatsum(c)

             if (heatrem + denh2o*dz_lake(c,1)*hfus > 0._r8) then            
                ! Remove snow and subtract the latent heat from the top layer.
                h2osno(c) = 0._r8
                snl(c) = 0
                ! The rest of the bookkeeping for the removed snow will be done below.
#if (defined CLMDEBUG)
                write(6,*)'Snow layers removed above unfrozen lake for column, snowice:', &
                          c, sumsnowice(c)
#endif
                if (heatrem > 0._r8) then ! simply subtract the heat from the layer
                   t_lake(c,1) = t_lake(c,1) - heatrem/(cpliq*denh2o*dz_lake(c,1))
                else !freeze part of the layer
                   t_lake(c,1) = tfrz
                   lake_icefrac(c,1) = -heatrem/(denh2o*dz_lake(c,1)*hfus)
                end if
             end if
          end if
       end do
!!!!!!!!!!!!

    ! Set snow age to zero if no snow

!dir$ concurrent
!cdir nodep
    do fc = 1, num_shlakesnowc
       c = filter_shlakesnowc(fc)
       if (snl(c) == 0) then
          snowage(c) = 0._r8
       end if
    end do

    ! Set empty snow layers to zero

    do j = -nlevsno+1,0
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakesnowc
          c = filter_shlakesnowc(fc)
          if (j <= snl(c) .and. snl(c) > -nlevsno) then
             h2osoi_ice(c,j) = 0._r8
             h2osoi_liq(c,j) = 0._r8
             t_soisno(c,j) = 0._r8
             dz(c,j) = 0._r8
             z(c,j) = 0._r8
             zi(c,j-1) = 0._r8
          end if
       end do
    end do

    ! Build new snow filter

    call BuildSnowFilter(lbc, ubc, num_shlakec, filter_shlakec, snl,&   !i
         num_shlakesnowc, filter_shlakesnowc, num_shlakenosnowc, filter_shlakenosnowc) !o

    ! Vertically average t_soisno and sum of h2osoi_liq and h2osoi_ice
    ! over all snow layers for history output

!dir$ concurrent
!cdir nodep
    do fc = 1, num_shlakesnowc
       c = filter_shlakesnowc(fc)
       t_snow(c)  = 0._r8
       snowice(c) = 0._r8
       snowliq(c) = 0._r8
    end do
!dir$ concurrent
!cdir nodep
    do fc = 1, num_shlakenosnowc
       c = filter_shlakenosnowc(fc)
       t_snow(c)  = spval
       snowice(c) = spval
       snowliq(c) = spval
    end do

    do j = -nlevsno+1, 0
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakesnowc
          c = filter_shlakesnowc(fc)
          if (j >= snl(c)+1) then
             t_snow(c)  = t_snow(c) + t_soisno(c,j)
             snowice(c) = snowice(c) + h2osoi_ice(c,j)
             snowliq(c) = snowliq(c) + h2osoi_liq(c,j)
          end if
       end do
    end do

    ! Determine ending water balance and volumetric soil water

!dir$ concurrent
!cdir nodep
    do fc = 1, num_shlakec
       
       c = filter_shlakec(fc)
       if (snl(c) < 0) t_snow(c) = t_snow(c)/abs(snl(c))
       endwb(c) = h2osno(c)
    end do

    do j = 1, nlevsoi
!dir$ concurrent
!cdir nodep
       do fc = 1, num_shlakec
          c = filter_shlakec(fc)
          endwb(c) = endwb(c) + h2osoi_ice(c,j) + h2osoi_liq(c,j)
          h2osoi_vol(c,j) = h2osoi_liq(c,j)/(dz(c,j)*denh2o) + h2osoi_ice(c,j)/(dz(c,j)*denice)
       end do
    end do

#if (defined CLMDEBUG)
    ! Check to make sure snow water adds up correctly.
    do j = -nlevsno+1,0
!dir$ concurrent
!cdir nodep
      do fc = 1, num_shlakec
         c = filter_shlakec(fc)
 
         jtop = snl(c)+1
         if(j == jtop) snow_water(c) = 0._r8
         if(j >= jtop) then
            snow_water(c) = snow_water(c) + h2osoi_ice(c,j) + h2osoi_liq(c,j)
            if(j == 0 .and. abs(snow_water(c)-h2osno(c))>1.e-7_r8) then
               write(6,*)'h2osno does not equal sum of snow layers in ShalLakeHydrology:', &
                         'column, h2osno, sum of snow layers =', c, h2osno(c), snow_water(c)
               call endrun()
            end if
         end if
      end do
    end do
#endif

!!!!!!!!!!!!!
    ! Do history variables and set special landunit runoff (adapted from end of HydrologyLake)
!dir$ concurrent
!cdir nodep
    do fp = 1,num_shlakep
       p = filter_shlakep(fp)
       c = pcolumn(p)
       g = pgridcell(p)

       qflx_infl(c)      = 0._r8
       qflx_surf(c)      = 0._r8
       qflx_drain(c)     = 0._r8
       rootr_column(c,:) = spval
       soilalpha(c)      = spval
       zwt(c)            = spval
       fcov(c)           = spval
       qcharge(c)        = spval
!       h2osoi_vol(c,:)   = spval

       ! Insure water balance using qflx_qrgwl
       qflx_qrgwl(c)     = forc_rain(g) + forc_snow(g) - qflx_evap_tot(p) - (endwb(c)-begwb(c))/dtime
#if (defined CLMDEBUG)
    write(6,*)'c, rain, snow, evap, endwb, begwb, qflx_qrgwl:', &
       c, forc_rain(g), forc_snow(g), qflx_evap_tot(p), endwb(c), begwb(c), qflx_qrgwl(c)
#endif

       ! The pft average must be done here for output to history tape
       qflx_evap_tot_col(c) = qflx_evap_tot(p)
    end do

!!!!!!!!!!!!!
!For now, bracket off the remaining biogeochem code.  May need to bring it back
!to do soil carbon and methane beneath lakes.
#if (defined CN)
#ifndef SHLAKE
    do j = 1, nlevsoi
!dir$ concurrent
!cdir nodep
       do fc = 1, num_soilc
          c = filter_soilc(fc)
          
          if (h2osoi_liq(c,j) > 0._r8) then
             vwc = h2osoi_liq(c,j)/(dz(c,j)*denh2o)
            
             ! the following limit set to catch very small values of 
             ! fractional saturation that can crash the calculation of psi
           
             fsat = max(vwc/vwcsat(c,j), 0.001_r8)
             psi = psisat(c,j) * (fsat)**bsw2(c,j)
             soilpsi(c,j) = min(max(psi,-15.0_r8),0._r8)
          else 
             soilpsi(c,j) = -15.0_r8
          end if
       end do
    end do
#endif
#endif

#if (defined DGVM) || (defined CN)
#ifndef SHLAKE
    ! Available soil water up to a depth of 0.5 m.
    ! Potentially available soil water (=whc) up to a depth of 0.5 m.
    ! Water content as fraction of whc up to a depth of 0.5 m.

!dir$ concurrent
!cdir nodep
    do c = lbc,ubc
       l = clandunit(c)
       if (ityplun(l) == istsoil) then
          rwat(c) = 0._r8
          swat(c) = 0._r8
          rz(c)   = 0._r8
       end if
    end do

    do j = 1, nlevsoi
!dir$ concurrent
!cdir nodep
       do c = lbc,ubc
          l = clandunit(c)
          if (ityplun(l) == istsoil) then
             if (z(c,j)+0.5_r8*dz(c,j) <= 0.5_r8) then
                watdry = watsat(c,j) * (316230._r8/sucsat(c,j)) ** (-1._r8/bsw(c,j))
                rwat(c) = rwat(c) + (h2osoi_vol(c,j)-watdry) * dz(c,j)
                swat(c) = swat(c) + (watsat(c,j)    -watdry) * dz(c,j)
                rz(c) = rz(c) + dz(c,j)
             end if
          end if
       end do
    end do

!dir$ concurrent
!cdir nodep
    do c = lbc,ubc
       l = clandunit(c)
       if (ityplun(l) == istsoil) then
          if (rz(c) /= 0._r8) then
             tsw  = rwat(c)/rz(c)
             stsw = swat(c)/rz(c)
          else
             watdry = watsat(c,1) * (316230._r8/sucsat(c,1)) ** (-1._r8/bsw(c,1))
             tsw = h2osoi_vol(c,1) - watdry
             stsw = watsat(c,1) - watdry
          end if
          wf(c) = tsw/stsw
       else
          wf(c) = 1.0_r8
       end if
    end do

#endif
#endif

  end subroutine ShalLakeHydrology

  subroutine QSat (T, p, es, esdT, qs, qsdT)
!
! !DESCRIPTION:
! Computes saturation mixing ratio and the change in saturation
! mixing ratio with respect to temperature.
! Reference:  Polynomial approximations from:
!             Piotr J. Flatau, et al.,1992:  Polynomial fits to saturation
!             vapor pressure.  Journal of Applied Meteorology, 31, 1507-1513.
!
! !USES:
    use  lake_const, only: r8,tfrz 
!
! !ARGUMENTS:
    implicit none
    real(r8), intent(in)  :: T        ! temperature (K)
    real(r8), intent(in)  :: p        ! surface atmospheric pressure (pa)
    real(r8), intent(out) :: es       ! vapor pressure (pa)
    real(r8), intent(out) :: esdT     ! d(es)/d(T)
    real(r8), intent(out) :: qs       ! humidity (kg/kg)
    real(r8), intent(out) :: qsdT     ! d(qs)/d(T)
!
! !CALLED FROM:
! subroutine Biogeophysics1 in module Biogeophysics1Mod
! subroutine BiogeophysicsLake in module BiogeophysicsLakeMod
! subroutine CanopyFluxesMod CanopyFluxesMod
!
! !REVISION HISTORY:
! 15 September 1999: Yongjiu Dai; Initial code
! 15 December 1999:  Paul Houser and Jon Radakovich; F90 Revision
!
!EOP
!
! !LOCAL VARIABLES:
!
    real(r8) :: T_limit
    real(r8) :: td,vp,vp1,vp2
!
! For water vapor (temperature range 0C-100C)
!
    real(r8), parameter :: a0 =  6.11213476
    real(r8), parameter :: a1 =  0.444007856
    real(r8), parameter :: a2 =  0.143064234e-01
    real(r8), parameter :: a3 =  0.264461437e-03
    real(r8), parameter :: a4 =  0.305903558e-05
    real(r8), parameter :: a5 =  0.196237241e-07
    real(r8), parameter :: a6 =  0.892344772e-10
    real(r8), parameter :: a7 = -0.373208410e-12
    real(r8), parameter :: a8 =  0.209339997e-15
!
! For derivative:water vapor
!
    real(r8), parameter :: b0 =  0.444017302
    real(r8), parameter :: b1 =  0.286064092e-01
    real(r8), parameter :: b2 =  0.794683137e-03
    real(r8), parameter :: b3 =  0.121211669e-04
    real(r8), parameter :: b4 =  0.103354611e-06
    real(r8), parameter :: b5 =  0.404125005e-09
    real(r8), parameter :: b6 = -0.788037859e-12
    real(r8), parameter :: b7 = -0.114596802e-13
    real(r8), parameter :: b8 =  0.381294516e-16
!
! For ice (temperature range -75C-0C)
!
    real(r8), parameter :: c0 =  6.11123516
    real(r8), parameter :: c1 =  0.503109514
    real(r8), parameter :: c2 =  0.188369801e-01
    real(r8), parameter :: c3 =  0.420547422e-03
    real(r8), parameter :: c4 =  0.614396778e-05
    real(r8), parameter :: c5 =  0.602780717e-07
    real(r8), parameter :: c6 =  0.387940929e-09
    real(r8), parameter :: c7 =  0.149436277e-11
    real(r8), parameter :: c8 =  0.262655803e-14
!
! For derivative:ice
!
    real(r8), parameter :: d0 =  0.503277922
    real(r8), parameter :: d1 =  0.377289173e-01
    real(r8), parameter :: d2 =  0.126801703e-02
    real(r8), parameter :: d3 =  0.249468427e-04
    real(r8), parameter :: d4 =  0.313703411e-06
    real(r8), parameter :: d5 =  0.257180651e-08
    real(r8), parameter :: d6 =  0.133268878e-10
    real(r8), parameter :: d7 =  0.394116744e-13
    real(r8), parameter :: d8 =  0.498070196e-16
!-----------------------------------------------------------------------

    T_limit = T - tfrz
    if (T_limit > 100.0) T_limit=100.0
    if (T_limit < -75.0) T_limit=-75.0

    td       = T_limit
    if (td >= 0.0) then
       es   = a0 + td*(a1 + td*(a2 + td*(a3 + td*(a4 &
            + td*(a5 + td*(a6 + td*(a7 + td*a8)))))))
       esdT = b0 + td*(b1 + td*(b2 + td*(b3 + td*(b4 &
            + td*(b5 + td*(b6 + td*(b7 + td*b8)))))))
    else
       es   = c0 + td*(c1 + td*(c2 + td*(c3 + td*(c4 &
            + td*(c5 + td*(c6 + td*(c7 + td*c8)))))))
       esdT = d0 + td*(d1 + td*(d2 + td*(d3 + td*(d4 &
            + td*(d5 + td*(d6 + td*(d7 + td*d8)))))))
    endif

    es    = es    * 100.            ! pa
    esdT  = esdT  * 100.            ! pa/K

    vp    = 1.0   / (p - 0.378*es)
    vp1   = 0.622 * vp
    vp2   = vp1   * vp

    qs    = es    * vp1             ! kg/kg
    qsdT  = esdT  * vp2 * p         ! 1 / K

  end subroutine QSat


  subroutine Tridiagonal (lbc, ubc, lbj, ubj, jtop, numf, filter, &
                          a, b, c, r, u)
!
! !DESCRIPTION:
! Tridiagonal matrix solution
!
! !USES:
  !  use shr_kind_mod, only: r8 => shr_kind_r8
    use lake_const, only: r8 
!
! !ARGUMENTS:
    implicit none
    integer , intent(in)    :: lbc, ubc               ! lbinning and ubing column indices
    integer , intent(in)    :: lbj, ubj               ! lbinning and ubing level indices
    integer , intent(in)    :: jtop(lbc:ubc)          ! top level for each column
    integer , intent(in)    :: numf                   ! filter dimension
    integer , intent(in)    :: filter(1:numf)         ! filter
    real(r8), intent(in)    :: a(lbc:ubc, lbj:ubj)    ! "a" left off diagonal of tridiagonal matrix
    real(r8), intent(in)    :: b(lbc:ubc, lbj:ubj)    ! "b" diagonal column for tridiagonal matrix
    real(r8), intent(in)    :: c(lbc:ubc, lbj:ubj)    ! "c" right off diagonal tridiagonal matrix
    real(r8), intent(in)    :: r(lbc:ubc, lbj:ubj)    ! "r" forcing term of tridiagonal matrix
    real(r8), intent(inout) :: u(lbc:ubc, lbj:ubj)    ! solution
!
! !CALLED FROM:
! subroutine BiogeophysicsLake in module BiogeophysicsLakeMod
! subroutine SoilTemperature in module SoilTemperatureMod
! subroutine SoilWater in module HydrologyMod
!
! !REVISION HISTORY:
! 15 September 1999: Yongjiu Dai; Initial code
! 15 December 1999:  Paul Houser and Jon Radakovich; F90 Revision
!  1 July 2003: Mariana Vertenstein; modified for vectorization
!
!EOP
!
! !OTHER LOCAL VARIABLES:
!
    integer  :: j,ci,fc                   !indices
    real(r8) :: gam(lbc:ubc,lbj:ubj)      !temporary
    real(r8) :: bet(lbc:ubc)              !temporary
!-----------------------------------------------------------------------

    ! Solve the matrix

!dir$ concurrent
!cdir nodep
    do fc = 1,numf
       ci = filter(fc)
       bet(ci) = b(ci,jtop(ci))
    end do

    do j = lbj, ubj
!dir$ prefervector
!dir$ concurrent
!cdir nodep
       do fc = 1,numf
          ci = filter(fc)
          if (j >= jtop(ci)) then
             if (j == jtop(ci)) then
                u(ci,j) = r(ci,j) / bet(ci)
             else
                gam(ci,j) = c(ci,j-1) / bet(ci)
                bet(ci) = b(ci,j) - a(ci,j) * gam(ci,j)
                u(ci,j) = (r(ci,j) - a(ci,j)*u(ci,j-1)) / bet(ci)
             end if
          end if
       end do
    end do

!Cray X1 unroll directive used here as work-around for compiler issue 2003/10/20
!dir$ unroll 0
    do j = ubj-1,lbj,-1
!dir$ prefervector
!dir$ concurrent
!cdir nodep
       do fc = 1,numf
          ci = filter(fc)
          if (j >= jtop(ci)) then
             u(ci,j) = u(ci,j) - gam(ci,j+1) * u(ci,j+1)
          end if
       end do
    end do

  end subroutine Tridiagonal

!=============================================
!02/18/2010  Created by Hongping Gu and Jiming Jin	
!For initializing lake model
!==============================================
 SUBROUTINE lakeini(IVGTYP,         ISLTYP,          HT,              SNOW,           & 
                    lakeminalt,     restart,         domainlakedepth,                 &
                    lakedepth2d,    savedtke12d,     snowdp2d,        h2osno2d,       & 
                    snl2d,          t_grnd2d,        t_lake3d,        lake_icefrac3d, &
                    z_lake3d,       dz_lake3d,       t_soisno3d,      h2osoi_ice3d,   &
                    h2osoi_liq3d,   h2osoi_vol3d,    z3d,             dz3d,           &
                    zi3d,           watsat3d,        csol3d,          tkmg3d,         &
                    tkdry3d,        tksatu3d,        lake,            ims,ime, jms,jme,&
                    ISWATER) ! ZMS 9/9/10

  use lake_const
  implicit none

  LOGICAL , INTENT(IN)      ::     restart
  INTEGER,  INTENT(IN   )   ::     ims,ime, jms,jme
  INTEGER, DIMENSION( ims:ime, jms:jme ), INTENT(IN)       :: IVGTYP,       &
                                                              ISLTYP
  REAL,    DIMENSION( ims:ime, jms:jme ), INTENT(IN)       :: HT
  REAL,    DIMENSION( ims:ime, jms:jme ), INTENT(INOUT)    :: SNOW
  real,    intent(in)                                      :: domainlakedepth,lakeminalt

  real,    dimension(ims:ime,jms:jme ),intent(out)                        :: lakedepth2d,    &
                                                                             savedtke12d
  real,    dimension(ims:ime,jms:jme ),intent(out)                        :: snowdp2d,       &
                                                                             h2osno2d,       &
                                                                             snl2d,          &
                                                                             t_grnd2d
                                                                              
  real,    dimension( ims:ime,1:nlevlak, jms:jme ),INTENT(out)            :: t_lake3d,       &
                                                                             lake_icefrac3d, &
                                                                             z_lake3d,       &
                                                                             dz_lake3d
  real,    dimension( ims:ime,-nlevsno+1:nlevsoi, jms:jme ),INTENT(out)   :: t_soisno3d,     &
                                                                             h2osoi_ice3d,   &
                                                                             h2osoi_liq3d,   &
                                                                             h2osoi_vol3d,   &
                                                                             z3d,            &
                                                                             dz3d
  real,    dimension( ims:ime,1:nlevsoi, jms:jme ),INTENT(out)            :: watsat3d,       &
                                                                             csol3d,         &
                                                                             tkmg3d,         &
                                                                             tkdry3d,        &
                                                                             tksatu3d
  real,    dimension( ims:ime,-nlevsno+0:nlevsoi, jms:jme ),INTENT(out)   :: zi3d            

  LOGICAL, DIMENSION( ims:ime, jms:jme ),intent(out)                      :: lake

  real,    dimension( ims:ime,1:nlevsoi, jms:jme )   :: bsw3d,    &
                                                        bsw23d,   &
                                                        psisat3d, &
                                                        vwcsat3d, &
                                                        watdry3d, &
                                                        watopt3d, &
                                                        hksat3d,  &
                                                        sucsat3d, &
                                                        clay3d,   &
                                                        sand3d   
  integer  :: n,i,j,k,ib,lev,bottom      ! indices
  real(r8),dimension(ims:ime,jms:jme )    :: bd2d               ! bulk density of dry soil material [kg/m^3]
  real(r8),dimension(ims:ime,jms:jme )    :: tkm2d              ! mineral conductivity
  real(r8),dimension(ims:ime,jms:jme )    :: xksat2d            ! maximum hydraulic conductivity of soil [mm/s]
  real(r8),dimension(ims:ime,jms:jme )    :: depthratio2d       ! ratio of lake depth to standard deep lake depth 
  real(r8),dimension(ims:ime,jms:jme )    :: clay2d             ! temporary
  real(r8),dimension(ims:ime,jms:jme )    :: sand2d             ! temporary
  integer, intent(in) :: ISWATER

  real(r8)                 :: scalez  = 0.025_r8   ! Soil layer thickness discretization (m)
  logical,parameter        :: arbinit = .true.
  real,parameter           :: defval  = -999.0
  integer                  :: isl
  integer                  :: numb_lak    ! for debug

  IF ( RESTART ) RETURN 

  DO j = jms,jme
  DO i = ims,ime
        snowdp2d(i,j)         = snow(i,j)*0.005               ! SNOW in kg/m^2 and snowdp in m
	h2osno2d(i,j)         = snow(i,j) ! mm 
  ENDDO
  ENDDO

! initialize all the grid with default value 
  DO j = jms,jme
  DO i = ims,ime

    lakedepth2d(i,j)             = defval
    snl2d(i,j)                   = defval
    do k = -nlevsno+1,nlevsoi
        h2osoi_liq3d(i,k,j)      = defval
        h2osoi_ice3d(i,k,j)      = defval
	t_soisno3d(i,k,j)        = defval
        z3d(i,k,j)               = defval 
        dz3d(i,k,j)              = defval                           
    enddo
    do k = 1,nlevlak 
	t_lake3d(i,k,j)          = defval
        lake_icefrac3d(i,k,j)    = defval
        z_lake3d(i,k,j)          = defval
        dz_lake3d(i,k,j)         = defval
    enddo

  ENDDO
  ENDDO

! judge whether the grid is lake grid
   print *,"ims,ime :",ims,ime
   print *,"jms,jme :",jms,jme
   numb_lak = 0
   print *,"numb_lak = ", numb_lak

       do i=ims,ime
         do j=jms,jme
            if(ivgtyp(i,j)==ISWATER.and.ht(i,j)>=lakeminalt) then 
                lake(i,j)  = .true.
                numb_lak   = numb_lak + 1
            else 
                lake(i,j)  = .false.
            end if
        end do
       end do
   print *, "the total number of lake grid is :", numb_lak
   print *, "whether the grid is lake grid"
! initialize lake grid 

  DO j = jms,jme
  DO i = ims,ime

     if ( lake(i,j) ) then

	t_soisno3d(i,:,j)      = 277.0
        t_lake3d(i,:,j)        = 277.0
        t_grnd2d(i,j)          = 277.0
        z3d(i,:,j)             = 0.0
        dz3d(i,:,j)            = 0.0
        zi3d(i,:,j)            = 0.0
        h2osoi_liq3d(i,:,j)    = 0.0
        h2osoi_ice3d(i,:,j)    = 0.0
        lake_icefrac3d(i,:,j)  = 0.0
        h2osoi_vol3d(i,:,j)    = 0.0
        snl2d(i,j)             = 0.0
        if ( domainlakedepth  > 0.0 ) then
            lakedepth2d(i,j)   = domainlakedepth
        else 
            lakedepth2d(i,j)   = spval
        endif

     endif

  ENDDO
  ENDDO 

  
#ifndef EXTRALAKELAYERS   
  dzlak(1) = 0.1_r8
  dzlak(2) = 1._r8
  dzlak(3) = 2._r8
  dzlak(4) = 3._r8
  dzlak(5) = 4._r8
  dzlak(6) = 5._r8
  dzlak(7) = 7._r8
  dzlak(8) = 7._r8
  dzlak(9) = 10.45_r8
  dzlak(10)= 10.45_r8

  zlak(1) =  0.05_r8
  zlak(2) =  0.6_r8
  zlak(3) =  2.1_r8
  zlak(4) =  4.6_r8
  zlak(5) =  8.1_r8
  zlak(6) = 12.6_r8
  zlak(7) = 18.6_r8
  zlak(8) = 25.6_r8
  zlak(9) = 34.325_r8
  zlak(10)= 44.775_r8
#else
  dzlak(1) =0.1_r8
  dzlak(2) =0.25_r8
  dzlak(3) =0.25_r8
  dzlak(4) =0.25_r8
  dzlak(5) =0.25_r8
  dzlak(6) =0.5_r8
  dzlak(7) =0.5_r8
  dzlak(8) =0.5_r8
  dzlak(9) =0.5_r8
  dzlak(10) =0.75_r8
  dzlak(11) =0.75_r8
  dzlak(12) =0.75_r8
  dzlak(13) =0.75_r8
  dzlak(14) =2_r8
  dzlak(15) =2_r8
  dzlak(16) =2.5_r8
  dzlak(17) =2.5_r8
  dzlak(18) =3.5_r8
  dzlak(19) =3.5_r8
  dzlak(20) =3.5_r8
  dzlak(21) =3.5_r8
  dzlak(22) =5.225_r8
  dzlak(23) =5.225_r8
  dzlak(24) =5.225_r8
  dzlak(25) =5.225_r8

  zlak(1) = dzlak(1)/2._r8
  do k = 2,nlevlak
     zlak(k) = zlak(k-1) + (dzlak(k-1)+dzlak(k))/2._r8
  end do
#endif

   ! "0" refers to soil surface and "nlevsoi" refers to the bottom of model soil

   do j = 1, nlevsoi
      zsoi(j) = scalez*(exp(0.5_r8*(j-0.5_r8))-1._r8)    !node depths
   enddo

   dzsoi(1) = 0.5_r8*(zsoi(1)+zsoi(2))             !thickness b/n two interfaces
   do j = 2,nlevsoi-1
      dzsoi(j)= 0.5_r8*(zsoi(j+1)-zsoi(j-1))
   enddo
   dzsoi(nlevsoi) = zsoi(nlevsoi)-zsoi(nlevsoi-1)

   zisoi(0) = 0._r8
   do j = 1, nlevsoi-1
      zisoi(j) = 0.5_r8*(zsoi(j)+zsoi(j+1))         !interface depths
   enddo
   zisoi(nlevsoi) = zsoi(nlevsoi) + 0.5_r8*dzsoi(nlevsoi)


!!!!!!!!!!!!!!!!!!begin to initialize lake variables!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  DO j = jms,jme
  DO i = ims,ime
      
     if ( lake(i,j) ) then

! 1: subroutine initTimeConst(lakedepth,z,zi,dz,dz_lake,z_lake,savedtke1,watsat3d,tkmg3d,tksatu3d,tkdry3d,csol3d)
                             ! Soil hydraulic and thermal properties
         isl = ISLTYP(i,j)   
         if (isl == 14 ) isl = isl + 1 
         do k = 1,nlevsoi
            sand3d(i,k,j)  = sand(isl)
            clay3d(i,k,j)  = clay(isl)
         enddo

         do k = 1,nlevsoi
            clay2d(i,j) = clay3d(i,k,j)
            sand2d(i,j) = sand3d(i,k,j)
            watsat3d(i,k,j) = 0.489_r8 - 0.00126_r8*sand2d(i,j)
            bd2d(i,j)    = (1._r8-watsat3d(i,k,j))*2.7e3_r8
            xksat2d(i,j) = 0.0070556_r8 *( 10._r8**(-0.884_r8+0.0153_r8*sand2d(i,j)) ) ! mm/s
            tkm2d(i,j) = (8.80_r8*sand2d(i,j)+2.92_r8*clay2d(i,j))/(sand2d(i,j)+clay2d(i,j))          ! W/(m K)

            bsw3d(i,k,j) = 2.91_r8 + 0.159_r8*clay2d(i,j)
            bsw23d(i,k,j) = -(3.10_r8 + 0.157_r8*clay2d(i,j) - 0.003_r8*sand2d(i,j))
            psisat3d(i,k,j) = -(exp((1.54_r8 - 0.0095_r8*sand2d(i,j) + 0.0063_r8*(100.0_r8-sand2d(i,j)  &
                              -clay2d(i,j)))*log(10.0_r8))*9.8e-5_r8)
            vwcsat3d(i,k,j) = (50.5_r8 - 0.142_r8*sand2d(i,j) - 0.037_r8*clay2d(i,j))/100.0_r8
            hksat3d(i,k,j) = xksat2d(i,j)
            sucsat3d(i,k,j) = 10._r8 * ( 10._r8**(1.88_r8-0.0131_r8*sand2d(i,j)) )
            tkmg3d(i,k,j) = tkm2d(i,j) ** (1._r8- watsat3d(i,k,j))
            tksatu3d(i,k,j) = tkmg3d(i,k,j)*0.57_r8**watsat3d(i,k,j)
            tkdry3d(i,k,j) = (0.135_r8*bd2d(i,j) + 64.7_r8) / (2.7e3_r8 - 0.947_r8*bd2d(i,j))
            csol3d(i,k,j) = (2.128_r8*sand2d(i,j)+2.385_r8*clay2d(i,j)) / (sand2d(i,j)+clay2d(i,j))*1.e6_r8  ! J/(m3 K)
            watdry3d(i,k,j) = watsat3d(i,k,j) * (316230._r8/sucsat3d(i,k,j)) ** (-1._r8/bsw3d(i,k,j))
            watopt3d(i,k,j) = watsat3d(i,k,j) * (158490._r8/sucsat3d(i,k,j)) ** (-1._r8/bsw3d(i,k,j))
         end do
         if (lakedepth2d(i,j) == spval) then
            lakedepth2d(i,j) = zlak(nlevlak) + 0.5_r8*dzlak(nlevlak)
            z_lake3d(i,1:nlevlak,j) = zlak(1:nlevlak)
            dz_lake3d(i,1:nlevlak,j) = dzlak(1:nlevlak)
         else
            depthratio2d(i,j) = lakedepth2d(i,j) / (zlak(nlevlak) + 0.5_r8*dzlak(nlevlak)) 
            z_lake3d(i,1,j) = zlak(1)
            dz_lake3d(i,1,j) = dzlak(1)
            dz_lake3d(i,2:nlevlak,j) = dzlak(2:nlevlak)*depthratio2d(i,j)
            z_lake3d(i,2:nlevlak,j) = zlak(2:nlevlak)*depthratio2d(i,j) + dz_lake3d(i,1,j)*(1._r8 - depthratio2d(i,j))
         end if
         z3d(i,1:nlevsoi,j) = zsoi(1:nlevsoi)
         zi3d(i,0:nlevsoi,j) = zisoi(0:nlevsoi)
         dz3d(i,1:nlevsoi,j) = dzsoi(1:nlevsoi)
         savedtke12d(i,j) = tkwat ! Initialize for first timestep.
   
 ! 2:  subroutine snowdp2levLake(snowdp,snl,z,dz,zi)

        if (snowdp2d(i,j) < 0.01_r8) then
           snl2d(i,j) = 0
           dz3d(i,-nlevsno+1:0,j) = 0._r8
           z3d (i,-nlevsno+1:0,j) = 0._r8
           zi3d(i,-nlevsno+0:0,j) = 0._r8
        else
           if ((snowdp2d(i,j) >= 0.01_r8) .and. (snowdp2d(i,j) <= 0.03_r8)) then
              snl2d(i,j) = -1
              dz3d(i,0,j)  = snowdp2d(i,j)
           else if ((snowdp2d(i,j) > 0.03_r8) .and. (snowdp2d(i,j) <= 0.04_r8)) then
              snl2d(i,j) = -2
              dz3d(i,-1,j) = snowdp2d(i,j)/2._r8
              dz3d(i, 0,j) = dz3d(i,-1,j)
           else if ((snowdp2d(i,j) > 0.04_r8) .and. (snowdp2d(i,j) <= 0.07_r8)) then
              snl2d(i,j) = -2
              dz3d(i,-1,j) = 0.02_r8
              dz3d(i, 0,j) = snowdp2d(i,j) - dz3d(i,-1,j)
           else if ((snowdp2d(i,j) > 0.07_r8) .and. (snowdp2d(i,j) <= 0.12_r8)) then
              snl2d(i,j) = -3
              dz3d(i,-2,j) = 0.02_r8
              dz3d(i,-1,j) = (snowdp2d(i,j) - 0.02_r8)/2._r8
              dz3d(i, 0,j) = dz3d(i,-1,j)
           else if ((snowdp2d(i,j) > 0.12_r8) .and. (snowdp2d(i,j) <= 0.18_r8)) then
              snl2d(i,j) = -3
              dz3d(i,-2,j) = 0.02_r8
              dz3d(i,-1,j) = 0.05_r8
              dz3d(i, 0,j) = snowdp2d(i,j) - dz3d(i,-2,j) - dz3d(i,-1,j)
           else if ((snowdp2d(i,j) > 0.18_r8) .and. (snowdp2d(i,j) <= 0.29_r8)) then
              snl2d(i,j) = -4
              dz3d(i,-3,j) = 0.02_r8
              dz3d(i,-2,j) = 0.05_r8
              dz3d(i,-1,j) = (snowdp2d(i,j) - dz3d(i,-3,j) - dz3d(i,-2,j))/2._r8
              dz3d(i, 0,j) = dz3d(i,-1,j)
           else if ((snowdp2d(i,j) > 0.29_r8) .and. (snowdp2d(i,j) <= 0.41_r8)) then
              snl2d(i,j) = -4
              dz3d(i,-3,j) = 0.02_r8
              dz3d(i,-2,j) = 0.05_r8
              dz3d(i,-1,j) = 0.11_r8
              dz3d(i, 0,j) = snowdp2d(i,j) - dz3d(i,-3,j) - dz3d(i,-2,j) - dz3d(i,-1,j)
           else if ((snowdp2d(i,j) > 0.41_r8) .and. (snowdp2d(i,j) <= 0.64_r8)) then
              snl2d(i,j) = -5
              dz3d(i,-4,j) = 0.02_r8
              dz3d(i,-3,j) = 0.05_r8
              dz3d(i,-2,j) = 0.11_r8
              dz3d(i,-1,j) = (snowdp2d(i,j) - dz3d(i,-4,j) - dz3d(i,-3,j) - dz3d(i,-2,j))/2._r8
              dz3d(i, 0,j) = dz3d(i,-1,j)
           else if (snowdp2d(i,j) > 0.64_r8) then
              snl2d(i,j) = -5
              dz3d(i,-4,j) = 0.02_r8
              dz3d(i,-3,j) = 0.05_r8
              dz3d(i,-2,j) = 0.11_r8
              dz3d(i,-1,j) = 0.23_r8
              dz3d(i, 0,j)=snowdp2d(i,j)-dz3d(i,-4,j)-dz3d(i,-3,j)-dz3d(i,-2,j)-dz3d(i,-1,j)
           endif
        end if
 
        do k = 0, snl2d(i,j)+1, -1
           z3d(i,k,j)    = zi3d(i,k,j) - 0.5_r8*dz3d(i,k,j)
           zi3d(i,k-1,j) = zi3d(i,k,j) - dz3d(i,k,j)
        end do

! 3:subroutine makearbinit

        if (snl2d(i,j) < 0) then
           do k = snl2d(i,j)+1, 0
                ! Be careful because there may be new snow layers with bad temperatures like 0 even if
                ! coming from init. con. file.
              if(arbinit .or. t_soisno3d(i,k,j) > 300 .or. t_soisno3d(i,k,j) < 200) t_soisno3d(i,k,j) = 250._r8
           enddo
        end if

        do k = 1, nlevsoi
           if(arbinit .or. t_soisno3d(i,k,j) > 1000 .or. t_soisno3d(i,k,j) < 0) t_soisno3d(i,k,j) = t_lake3d(i,nlevlak,j)
        end do

        do k = 1, nlevlak
           if(arbinit .or. lake_icefrac3d(i,k,j) > 1._r8 .or. lake_icefrac3d(i,k,j) < 0._r8) then
              if(t_lake3d(i,k,j) >= tfrz) then
                 lake_icefrac3d(i,k,j) = 0._r8
              else
                 lake_icefrac3d(i,k,j) = 1._r8
              end if
           end if
        end do
        
        do k = 1,nlevsoi
           if (arbinit .or. h2osoi_vol3d(i,k,j) > 10._r8 .or. h2osoi_vol3d(i,k,j) < 0._r8) h2osoi_vol3d(i,k,j) = 1.0_r8
           h2osoi_vol3d(i,k,j) = min(h2osoi_vol3d(i,k,j),watsat3d(i,k,j))

             ! soil layers
           if (t_soisno3d(i,k,j) <= tfrz) then
              h2osoi_ice3d(i,k,j)  = dz3d(i,k,j)*denice*h2osoi_vol3d(i,k,j)
              h2osoi_liq3d(i,k,j) = 0._r8
           else
              h2osoi_ice3d(i,k,j) = 0._r8
              h2osoi_liq3d(i,k,j) = dz3d(i,k,j)*denh2o*h2osoi_vol3d(i,k,j)
           endif
        enddo

        do k = -nlevsno+1, 0
           if (k > snl2d(i,j)) then
              h2osoi_ice3d(i,k,j) = dz3d(i,k,j)*bdsno
              h2osoi_liq3d(i,k,j) = 0._r8
           end if
        end do

    end if   !lake(i,j)
  ENDDO
  ENDDO

  END SUBROUTINE lakeini

END MODULE module_sf_lake
