MODULE mm5_input

   USE header_space

   LOGICAL , PARAMETER :: just_header = .FALSE.
   LOGICAL , PARAMETER :: all_data    = .TRUE.

   INTEGER , PARAMETER , PRIVATE ::  unit = 10
   INTEGER , PARAMETER , PRIVATE :: lunit =  9
   INTEGER , PARAMETER :: unit_outside = unit , lunit_outside = lunit

   TYPE mm53d
      REAL , DIMENSION(:,:,:) , POINTER :: data
      TYPE(small_header)                :: sh
   END TYPE mm53d

   TYPE mm52d
      REAL , DIMENSION(:,:)   , POINTER :: data
      TYPE(small_header)                :: sh
   END TYPE mm52d

   TYPE mm51d
      REAL , DIMENSION(:)     , POINTER :: data
      TYPE(small_header)                :: sh
   END TYPE mm51d

   TYPE(mm53d) , ALLOCATABLE , DIMENSION(:) :: all_mm5_3d
   TYPE(mm52d) , ALLOCATABLE , DIMENSION(:) :: all_mm5_2d
   TYPE(mm51d) , ALLOCATABLE , DIMENSION(:) :: all_mm5_1d

   TYPE(mm53d) , ALLOCATABLE , DIMENSION(:) :: all_wrf_3d
   TYPE(mm52d) , ALLOCATABLE , DIMENSION(:) :: all_wrf_2d
   TYPE(mm51d) , ALLOCATABLE , DIMENSION(:) :: all_wrf_1d
   
   INTEGER , DIMENSION(4) :: num_mm5 , num_wrf

CONTAINS

   SUBROUTINE read_mm5 ( kx_wrf )
   
      !  After this routine, a single time period of MM5 data is input, and
      !  the variables are stored in arrays, along with the small header
      !  associated with each of the arrays.  

      !  The dot arrays are interpolated to the new staggering locations.
      
      !  The arrays with X,Y spatial dimensions are transposed to have a similar
      !  index ordering to WRF.

      !  The arrays with a vertical index are inverted, also to follow the WRF convention.

      USE util

      IMPLICIT NONE

      INTEGER , INTENT(IN) :: kx_wrf

      INTEGER :: ok , foo , k , flag 
      
      CHARACTER (LEN=1) :: stag_loc

      REAL , DIMENSION(:,:,:,:) , ALLOCATABLE :: data

      !  Initialize the number of MM5 fields to zero.

      num_mm5 = 0
      num_wrf = 0

      !  Get all of the MM5 data for this time period.  Once we hit a END-OF-TIME
      !  flag, we exit out of this DO loop.

      get_data_for_this_time : DO 
      
         !  First, there is a flag.

         READ ( unit , IOSTAT = ok ) flag
   
         !  Error checks on the read.  If we are here, we have PREVIOUSLY found a big header flag,
         !  and a big header.  So, if we also get a small header flag, we assume that the
         !  I/O is going to be OK afterwards.  Which means, there are only error checks on this
         !  flag READ.

         !  Negative return values are EOF or EOR things.  We are hoping for the EOF
         !  result, as that would be an OK thing to have happen - just means we're all
         !  outta data to input from the MM5 file.

         IF      ( ok .LT. 0 ) THEN
            PRINT '(A)','END-OF-FILE detected in MM5.  Stopping fairly gracefully.'
            STOP 'EOF_in_MM5_input'
    
         !  Positive return codes are other weird errors.

         ELSE IF ( ok .GT. 0 ) THEN
            PRINT '(A)','Error reading MM5 small header flag.  This is a problem.'
            STOP 'ERROR_reading_small_header_flag'

         !  No errors with reading the flag - wow.

         ELSE

            !  Did we stumble upon a big header flag.  If the input file was cat'ed together
            !  from some smaller constituent parts, this could occur.  No biggy, we just
            !  read past the big header, and try again.

            IF ( flag .EQ. bh_flag ) THEN
               PRINT '(A)','Found an extra big header flag.  We are going to skip the big header.'
               READ (unit) foo
               CYCLE get_data_for_this_time

            !  If this is the EOT flag, that means that we have already read in all of the 
            !  data for this time.  We can EXIT this loop.

            ELSE IF ( flag .EQ. eot_flag ) THEN

               PRINT '(A)','Found the end of time flag, all data input for this time period.'
               EXIT  get_data_for_this_time

            !  Most of our time in this loop should be spent pulling in data in this portion
            !  of the IF test.

            ELSE IF ( flag .EQ. sh_flag ) THEN

               !  This is part two of the read - the small header.  For each field, the size
               !  of this record is the same.  

               READ ( unit ) num_dims_mm5 , start_dims_mm5 , end_dims_mm5 , xtime_mm5 , &
                             staggering_mm5 , ordering_mm5 , current_date_mm5 , &
                             name_mm5 , units_mm5 , description_mm5
  
               !  Store the data into a temporary location that can be transferred around
               !  ease, grace and facility.

               sh%num_dims     = num_dims_mm5
               sh%start_dims   = start_dims_mm5
               sh%end_dims     = end_dims_mm5
               sh%xtime        = xtime_mm5
               sh%staggering   = staggering_mm5
               sh%ordering     = ordering_mm5
               sh%current_date = current_date_mm5
               sh%name         = name_mm5
               sh%units        = units_mm5
               sh%description  = description_mm5

               !  In a C-grid, where would this variable be going?

               IF      ( name_mm5(1:2) .EQ. 'U ' ) THEN
                  stag_loc = 'U'
               ELSE IF ( name_mm5(1:2) .EQ. 'V ' ) THEN
                  stag_loc = 'V'
               ELSE
                  stag_loc = 'T'
               END IF

               !  Consistent with the WRF output format is the index ordering requirement.             

               IF ( sh%ordering(1:2) .EQ. 'YX' ) THEN
                  sh%ordering(1:2) = 'XY'
               END IF

               !  We need to read in part three of the 3-part data stream - now comes the
               !  gridded data.  But wait, is it 3d, 2d, 1d?  When we find out, we allocate
               !  some space and bump a counter.

               num_mm5(num_dims_mm5) = num_mm5(num_dims_mm5) + 1
               IF ( num_dims_mm5 .EQ. 3 ) THEN
                  num_wrf(num_dims_mm5) = num_wrf(num_dims_mm5) + 1
               END IF
  
               !  ALLOCATE the temporary space for input of the MM5-formatted data.

               ALLOCATE ( data(start_dims_mm5(1):end_dims_mm5(1) , &
                               start_dims_mm5(2):end_dims_mm5(2) , &
                               start_dims_mm5(3):end_dims_mm5(3) , &
                               start_dims_mm5(4):end_dims_mm5(4) ) )

               !  READ in the MM5 input data.

               READ ( unit ) data

               !  ALLOCATE space for the stored data.  We are going to transpose the data to the 
               !  WRF C-staggering right now, as well as the bottom up vertical orientation.  Note
               !  that for the 2-D arrays that have "YX" as the ordering, the indices are reversed.

               IF      ( num_dims_mm5 .EQ. 3 ) THEN
                  ALLOCATE ( all_mm5_3d(num_mm5(num_dims_mm5))%data(start_dims_mm5(2):end_dims_mm5(2) , &
                                                                    start_dims_mm5(1):end_dims_mm5(1) , &
                                                                    start_dims_mm5(3):end_dims_mm5(3) ) )
                  ALLOCATE ( all_wrf_3d(num_wrf(num_dims_mm5))%data(start_dims_mm5(2):end_dims_mm5(2) , &
                                                                    start_dims_mm5(1):end_dims_mm5(1) , &
                                                                    1:kx_wrf+1                        ) )
   
                  !  Store it semi-permanently in the "all" array of choice.  Remember, we are also saving the
                  !  header that accompanies this data.
   
                  CALL b2c3d ( data(:,:,:,1) , all_mm5_3d(num_mm5(num_dims_mm5))%data , &
                               end_dims_mm5(2) , end_dims_mm5(1) , end_dims_mm5(3) , stag_loc )
                  all_mm5_3d(num_mm5(num_dims_mm5))%sh   = sh
                  all_wrf_3d(num_wrf(num_dims_mm5))%sh   = sh
                  all_wrf_3d(num_wrf(num_dims_mm5))%sh%end_dims(3) = kx_wrf+1
                  all_wrf_3d(num_wrf(num_dims_mm5))%sh%ordering(1:2) = 'XY'

               ELSE IF ( ( num_dims_mm5 .EQ. 2 ) .AND. ( ordering_mm5(1:2) .EQ. 'YX' ) ) THEN
                  ALLOCATE ( all_mm5_2d(num_mm5(num_dims_mm5))%data(start_dims_mm5(2):end_dims_mm5(2) , &
                                                                    start_dims_mm5(1):end_dims_mm5(1) ) )
   
                  !  Store it semi-permanently in the "all" array of choice.  Remember, we are also saving the
                  !  header that accompanies this data.
   
                  CALL b2c2d ( data(:,:,1,1) , all_mm5_2d(num_mm5(num_dims_mm5))%data , &
                               end_dims_mm5(2) , end_dims_mm5(1) , stag_loc )
                  all_mm5_2d(num_mm5(num_dims_mm5))%sh   = sh

                  !  WRF can't handle negative terrain values.

                  IF ( all_mm5_2d(num_mm5(num_dims_mm5))%sh%name(1:8) .EQ. 'TERRAIN ' ) THEN
                     WHERE ( all_mm5_2d(num_mm5(num_dims_mm5))%data .LT. 1.e-3 )
                        all_mm5_2d(num_mm5(num_dims_mm5))%data = 0
                     END WHERE
                  END IF

               ELSE IF ( num_dims_mm5 .EQ. 2 ) THEN
!                 ALLOCATE ( all_mm5_2d(num_mm5(num_dims_mm5))%data(start_dims_mm5(1):end_dims_mm5(1) , &
!                                                                   start_dims_mm5(2):end_dims_mm5(2) ) )
   
                  !  Store it semi-permanently in the "all" array of choice.  Remember, we are also saving the
                  !  header that accompanies this data.
   
!                 all_mm5_2d(num_mm5(num_dims_mm5))%data = data(:,:,1,1)
!                 all_mm5_2d(num_mm5(num_dims_mm5))%sh   = sh
                  PRINT '(A,A,A)','Skipping this 2d array that is not "YX" ordering, ',TRIM(name_mm5),'.'
                  num_mm5(num_dims_mm5) = num_mm5(num_dims_mm5) - 1


               ELSE IF ( num_dims_mm5 .EQ. 1 ) THEN
                  ALLOCATE ( all_mm5_1d(num_mm5(num_dims_mm5))%data(start_dims_mm5(1):end_dims_mm5(1) ) )
   
                  !  Store it semi-permanently in the "all" array of choice.  Remember, we are also saving the
                  !  header that accompanies this data.  The only 1D array that we care about is the sigma
                  !  coordinates, so we invert that along with the rest of the vertically stacked 3d data.
   
                  DO k = 1 , end_dims_mm5(1)
                     all_mm5_1d(num_mm5(num_dims_mm5))%data(k) = data(end_dims_mm5(1)+1-k,1,1,1)
                  END DO
                  all_mm5_1d(num_mm5(num_dims_mm5))%sh   = sh

               END IF

               !  DEALLOCATE the temporary storage array.

               DEALLOCATE ( data ) 

            ELSE

               PRINT '(A)','"Great day in the the morning!", as Grandma used to say.'
               PRINT '(A)','What kind of data you trying to make me read?'

            END IF  !  Was this the right flag that we picked up?

         END IF  ! Was there an error on the read?      

      END DO get_data_for_this_time
      
   END SUBROUTINE read_mm5

   SUBROUTINE read_lowbdy
   
      !  The arrays with X,Y spatial dimensions are transposed to have a similar
      !  index ordering to WRF.

      USE util

      IMPLICIT NONE

      INTEGER :: ok , foo , k , flag 
      
      CHARACTER (LEN=1) :: stag_loc

      REAL , DIMENSION(:,:,:,:) , ALLOCATABLE :: data

      !  Get all of the MM5 data for this time period.  Once we hit a END-OF-TIME
      !  flag, we exit out of this DO loop.

      get_data_for_this_time : DO 
      
         !  First, there is a flag.

         READ ( lunit , IOSTAT = ok ) flag
   
         !  Error checks on the read.  If we are here, we have PREVIOUSLY found a big header flag,
         !  and a big header.  So, if we also get a small header flag, we assume that the
         !  I/O is going to be OK afterwards.  Which means, there are only error checks on this
         !  flag READ.

         !  Negative return values are EOF or EOR things.  We are hoping for the EOF
         !  result, as that would be an OK thing to have happen - just means we're all
         !  outta data to input from the MM5 file.

         IF      ( ok .LT. 0 ) THEN
            PRINT '(A)','END-OF-FILE detected in LOWBDY file.  Maybe only one time is available.'
            PRINT '(A)','REWINDING and trying again.'
            REWIND(lunit)
            CYCLE get_data_for_this_time
    
         !  Positive return codes are other weird errors.

         ELSE IF ( ok .GT. 0 ) THEN
            PRINT '(A)','Error reading LOWBDY small header flag.  This is a problem.'
            STOP 'ERROR_reading_small_header_flag'

         !  No errors with reading the flag - wow.

         ELSE

            !  Did we stumble upon a big header flag.  If the input file was cat'ed together
            !  from some smaller constituent parts, this could occur.  No biggy, we just
            !  read past the big header, and try again.

            IF ( flag .EQ. bh_flag ) THEN
               PRINT '(A)','Found an extra big header flag.  We are going to skip the big header.'
               READ (lunit) foo
               CYCLE get_data_for_this_time

            !  If this is the EOT flag, that means that we have already read in all of the 
            !  data for this time.  We can EXIT this loop.

            ELSE IF ( flag .EQ. eot_flag ) THEN

               PRINT '(A)','Found the end of time flag, all data input for this time period.'
               EXIT  get_data_for_this_time

            !  Most of our time in this loop should be spent pulling in data in this portion
            !  of the IF test.

            ELSE IF ( flag .EQ. sh_flag ) THEN

               !  This is part two of the read - the small header.  For each field, the size
               !  of this record is the same.  

               READ ( lunit ) num_dims_mm5 , start_dims_mm5 , end_dims_mm5 , xtime_mm5 , &
                             staggering_mm5 , ordering_mm5 , current_date_mm5 , &
                             name_mm5 , units_mm5 , description_mm5
  
               !  Store the data into a temporary location that can be transferred around
               !  ease, grace and facility.

               sh%num_dims     = num_dims_mm5
               sh%start_dims   = start_dims_mm5
               sh%end_dims     = end_dims_mm5
               sh%xtime        = xtime_mm5
               sh%staggering   = staggering_mm5
               sh%ordering     = ordering_mm5
               sh%current_date = current_date_mm5
               sh%name         = name_mm5
               sh%units        = units_mm5
               sh%description  = description_mm5

               !  In a C-grid, where would this variable be going?

               stag_loc = 'T'

               !  Consistent with the WRF output format is the index ordering requirement.             

               IF ( sh%ordering(1:2) .EQ. 'YX' ) THEN
                  sh%ordering(1:2) = 'XY'
               END IF

               !  We allocate some space and bump a counter.

               num_mm5(num_dims_mm5) = num_mm5(num_dims_mm5) + 1
  
               !  ALLOCATE the temporary space for input of the MM5-formatted data.

               ALLOCATE ( data(start_dims_mm5(1):end_dims_mm5(1) , &
                               start_dims_mm5(2):end_dims_mm5(2) , &
                               start_dims_mm5(3):end_dims_mm5(3) , &
                               start_dims_mm5(4):end_dims_mm5(4) ) )

               !  READ in the MM5 input data.

               READ ( lunit ) data

               !  ALLOCATE space for the stored data.  We are going to transpose the data to the 
               !  WRF C-staggering right now, as well as the bottom up vertical orientation.  Note
               !  that for the 2-D arrays that have "YX" as the ordering, the indices are reversed.

               IF ( ( num_dims_mm5 .EQ. 2 ) .AND. ( ordering_mm5(1:2) .EQ. 'YX' ) ) THEN
                  ALLOCATE ( all_mm5_2d(num_mm5(num_dims_mm5))%data(start_dims_mm5(2):end_dims_mm5(2) , &
                                                                    start_dims_mm5(1):end_dims_mm5(1) ) )
   
                  !  Store it semi-permanently in the "all" array of choice.  Remember, we are also saving the
                  !  header that accompanies this data.
   
                  CALL b2c2d ( data(:,:,1,1) , all_mm5_2d(num_mm5(num_dims_mm5))%data , &
                               end_dims_mm5(2) , end_dims_mm5(1) , stag_loc )
                  all_mm5_2d(num_mm5(num_dims_mm5))%sh   = sh

                  !  WRF can't handle negative terrain values.

                  IF ( all_mm5_2d(num_mm5(num_dims_mm5))%sh%name(1:8) .EQ. 'TERRAIN ' ) THEN
                     WHERE ( all_mm5_2d(num_mm5(num_dims_mm5))%data .LT. 1.e-3 )
                        all_mm5_2d(num_mm5(num_dims_mm5))%data = 0
                     END WHERE
                  END IF

               ELSE

                  PRINT '(A)','Pain and woe.  We found a non-2d field in the LOWBDY file.'
                  STOP 'LOWBDY_problems'

               END IF

               !  DEALLOCATE the temporary storage array.

               DEALLOCATE ( data ) 

            ELSE

               PRINT '(A)','"Great day in the the morning!", as Grandma used to say.'
               PRINT '(A)','What kind of data you trying to make me read?'

            END IF  !  Was this the right flag that we picked up?

         END IF  ! Was there an error on the read?      

      END DO get_data_for_this_time
      
   END SUBROUTINE read_lowbdy

   SUBROUTINE open_file ( unit_num , name )
   
      IMPLICIT NONE

      INTEGER , INTENT(IN) :: unit_num
      CHARACTER (LEN=132) , INTENT(IN) :: name
  
      LOGICAL :: file_exist

      INQUIRE ( FILE   = TRIM(name)    , &
                EXIST  = file_exist      )

      IF ( file_exist ) THEN
       
         OPEN ( UNIT   = unit_num      , &
                FILE   = TRIM(name)    , &
                STATUS = 'OLD'         , &
                FORM   = 'UNFORMATTED' , &
                ACCESS = 'SEQUENTIAL'    )
      ELSE

         PRINT '(A,A,A)','The MM5 input file, ',TRIM(name),', does not exist.'
         STOP 'MM5_input_does_not_exist'

      END IF
   
   END SUBROUTINE open_file

   SUBROUTINE get_header_info_only 

      IMPLICIT NONE

      INTEGER :: flag
      
      READ ( unit ) flag

      IF ( flag .NE. bh_flag ) THEN

         PRINT '(A)','Expected a big header flag at the beginning of this file.'
         PRINT '(A,I16,A)','Instead, we found a ',flag,'.'
         STOP 'NO_big_header_flag_found'

      END IF

      READ ( unit ) bhi_mm5 , bhr_mm5 , bhic_mm5 , bhrc_mm5

   END SUBROUTINE get_header_info_only

END MODULE mm5_input
