MODULE diags

   REAL , ALLOCATABLE , DIMENSION(:,:,:) :: zt_mm5 , zw_mm5 , zu_mm5 , zv_mm5
   REAL , ALLOCATABLE , DIMENSION(:,:,:) :: zt_wrf , zw_wrf , zu_wrf , zv_wrf
   LOGICAL :: z_mm5_is_allocated = .FALSE.
   LOGICAL :: z_wrf_is_allocated = .FALSE.

CONTAINS 

   SUBROUTINE compute_density ( ix , jx , kx_mm5 , kx_wrf )

      USE mm5_input
      USE constants

      IMPLICIT NONE

      INTEGER , INTENT(IN) :: ix , jx , kx_mm5 , kx_wrf

      INTEGER :: index_pressure , index_temperature , index_sigma
      INTEGER :: i , j , k , loop

      !  Find the total pressure.
  
      index_pressure = 0 
      find_pressure : DO loop = 1 , num_mm5(3) 
         IF ( all_mm5_3d(loop)%sh%name(1:8) .EQ. 'PRESSURE' ) THEN 
            index_pressure = loop
            EXIT find_pressure            
         END IF 
      END DO find_pressure

      IF ( index_pressure .EQ. 0 ) THEN
         PRINT '(A)','Could not find the index for the PRESSURE data for theta computation.'
         STOP 'COMPUTE_theta_no_pressure'
      END IF

      !  Find the temperature.
  
      index_temperature = 0 
      find_temperature : DO loop = 1 , num_mm5(3) 
         IF ( all_mm5_3d(loop)%sh%name(1:8) .EQ. 'T       ' ) THEN 
            index_temperature = loop
            EXIT find_temperature            
         END IF 
      END DO find_temperature

      IF ( index_temperature .EQ. 0 ) THEN
         PRINT '(A)','Could not find the index for the TEMPERATURE data for theta computation.'
         STOP 'COMPUTE_theta_no_temperature'
      END IF

      !  Allocate space for the density.

      num_mm5(3) = num_mm5(3) + 1
      ALLOCATE ( all_mm5_3d(num_mm5(3))%data(ix,jx,kx_mm5) )
      num_wrf(3) = num_wrf(3) + 1
      ALLOCATE ( all_wrf_3d(num_mm5(3))%data(ix,jx,kx_wrf+1) )
      all_wrf_3d(num_mm5(3))%data = 0

      !  Compute density.

      DO k = 1 , kx_mm5
         DO j = 1 , jx - 1
            DO i = 1 , ix - 1
               all_mm5_3d(num_mm5(3))%data(i,j,k) = all_mm5_3d(index_pressure)%data(i,j,k) / &
                 ( R * all_mm5_3d(index_temperature)%data(i,j,k) ) 
            END DO
         END DO
      END DO

      !  Fix the small header for the new total pressure field.

      all_mm5_3d(num_mm5(3))%sh = all_mm5_3d(index_temperature)%sh
      all_mm5_3d(num_mm5(3))%sh%name(1:8) = 'RHO     '
      all_mm5_3d(num_mm5(3))%sh%units(1:25) = 'kg m{-3}                 '
      all_mm5_3d(num_mm5(3))%sh%description(1:46) = 'Density                                       '
      all_wrf_3d(num_mm5(3))%sh = all_mm5_3d(num_mm5(3))%sh
      all_wrf_3d(num_mm5(3))%sh%end_dims(3) = kx_wrf+1
      
   END SUBROUTINE compute_density

   SUBROUTINE compute_theta ( ix , jx , kx_mm5 , kx_wrf )

      USE mm5_input
      USE constants

      IMPLICIT NONE

      INTEGER , INTENT(IN) :: ix , jx , kx_mm5 , kx_wrf

      INTEGER :: index_pressure , index_temperature , index_sigma
      INTEGER :: i , j , k , loop

      REAL , PARAMETER :: rovcp = R / Cp

      !  Find the total pressure.
  
      index_pressure = 0 
      find_pressure : DO loop = 1 , num_mm5(3) 
         IF ( all_mm5_3d(loop)%sh%name(1:8) .EQ. 'PRESSURE' ) THEN 
            index_pressure = loop
            EXIT find_pressure            
         END IF 
      END DO find_pressure

      IF ( index_pressure .EQ. 0 ) THEN
         PRINT '(A)','Could not find the index for the PRESSURE data for theta computation.'
         STOP 'COMPUTE_theta_no_pressure'
      END IF

      !  Find the temperature.
  
      index_temperature = 0 
      find_temperature : DO loop = 1 , num_mm5(3) 
         IF ( all_mm5_3d(loop)%sh%name(1:8) .EQ. 'T       ' ) THEN 
            index_temperature = loop
            EXIT find_temperature            
         END IF 
      END DO find_temperature

      IF ( index_temperature .EQ. 0 ) THEN
         PRINT '(A)','Could not find the index for the TEMPERATURE data for theta computation.'
         STOP 'COMPUTE_theta_no_temperature'
      END IF

      !  Find the sigma data.

      index_sigma  = 0
      find_sigma  : DO loop = 1 , num_mm5(1)
         IF ( all_mm5_1d(loop)%sh%name(1:8) .EQ. 'SIGMAH  ' ) THEN
            index_sigma    = loop
            EXIT find_sigma
         END IF
      END DO find_sigma

      IF ( index_sigma    .EQ. 0 ) THEN
         PRINT '(A)','Could not find the index for the SIGMA data for total pressure.'
         STOP 'COMPUTE_total_p_no_SIGMA'
      END IF

      !  Allocate space for the total pressure.

      num_mm5(3) = num_mm5(3) + 1
      ALLOCATE ( all_mm5_3d(num_mm5(3))%data(ix,jx,kx_mm5) )
      num_wrf(3) = num_wrf(3) + 1
      ALLOCATE ( all_wrf_3d(num_mm5(3))%data(ix,jx,kx_wrf+1) )
      all_wrf_3d(num_mm5(3))%data = 0

      !  Compute theta.

      DO k = 1 , kx_mm5
         DO j = 1 , jx - 1
            DO i = 1 , ix - 1
               all_mm5_3d(num_mm5(3))%data(i,j,k) = all_mm5_3d(index_temperature)%data(i,j,k) * &
                 ( 100000. / all_mm5_3d(index_pressure)%data(i,j,k) ) ** rovcp
            END DO
         END DO
      END DO

      !  Fix the small header for the new total pressure field.

      all_mm5_3d(num_mm5(3))%sh = all_mm5_3d(index_temperature)%sh
      all_mm5_3d(num_mm5(3))%sh%name(1:8) = 'THETA   '
      all_mm5_3d(num_mm5(3))%sh%units(1:25) = 'K                        '
      all_mm5_3d(num_mm5(3))%sh%description(1:46) = 'Potential temperature (dry)                   '
      all_wrf_3d(num_mm5(3))%sh = all_mm5_3d(num_mm5(3))%sh
      all_wrf_3d(num_mm5(3))%sh%end_dims(3) = kx_wrf+1
      
   END SUBROUTINE compute_theta

   SUBROUTINE compute_surface_p ( ix , jx , kx_mm5 , ptop )

      USE mm5_input

      IMPLICIT NONE

      INTEGER , INTENT(IN) :: ix , jx , kx_mm5 

      REAL    , INTENT(IN) :: ptop

      INTEGER :: index_pp , index_pstarcrs , index_sigma
      INTEGER :: i , j , k , loop
      REAL , DIMENSION(kx_mm5+1) :: sigma , sigmaf

      !  Find the reference pstar (reference p surface - reference p top).  
  
      index_pstarcrs = 0 
      find_pstarcrs : DO loop = 1 , num_mm5(2) 
         IF ( all_mm5_2d(loop)%sh%name(1:8) .EQ. 'PSTARCRS' ) THEN 
            index_pstarcrs = loop
            EXIT find_pstarcrs            
         END IF 
      END DO find_pstarcrs

      IF ( index_pstarcrs .EQ. 0 ) THEN
         PRINT '(A)','Could not find the index for the PSTARCRS data for surface pressure computation.'
         STOP 'COMPUTE_surface_p_no_PSTARCRS'
      END IF

      !  Find the pressure perturbation.
  
      index_pp = 0 
      find_pp : DO loop = 1 , num_mm5(3) 
         IF ( all_mm5_3d(loop)%sh%name(1:8) .EQ. 'PP      ' ) THEN 
            index_pp = loop
            EXIT find_pp
         END IF 
      END DO find_pp

      IF ( index_pp .EQ. 0 ) THEN
         PRINT '(A)','Could not find the index for the PP data for surface pressure.'
         STOP 'COMPUTE_surface_p_no_pp'
      END IF
   
      !  Find the sigma data.

      index_sigma  = 0
      find_sigma  : DO loop = 1 , num_mm5(1)
         IF ( all_mm5_1d(loop)%sh%name(1:8) .EQ. 'SIGMAH  ' ) THEN
            index_sigma    = loop
            EXIT find_sigma
         END IF
      END DO find_sigma

      IF ( index_sigma    .EQ. 0 ) THEN
         PRINT '(A)','Could not find the index for the SIGMA data for surface pressure.'
         STOP 'COMPUTE_surface_p_no_SIGMA'
      END IF

      !  Sigma on half and full levels.

      DO k = 1 , kx_mm5
         sigma(k) = all_mm5_1d(loop)%data(k)
      END DO

      sigmaf(1) = 1.
      sigmaf(kx_mm5+1) = 0.
      DO k = 2 , kx_mm5
         sigmaf(k) = 2. * sigma(k-1) - sigmaf(k-1)
      END DO

      !  Allocate space for the surface pressure.

      num_wrf(2) = num_wrf(2) + 1
      ALLOCATE ( all_wrf_2d(num_wrf(2))%data(ix,jx) )
      all_wrf_2d(num_wrf(2))%data = 0

      !  Compute total pressure.

      DO j = 1 , jx - 1
         DO i = 1 , ix - 1
            all_wrf_2d(num_wrf(2))%data(i,j) = all_mm5_2d(index_pstarcrs)%data(i,j) * sigmaf(1) + &
                                               all_mm5_3d(index_pp)%data(i,j,1) + ptop
         END DO
      END DO

      !  Fix the small header for the new total pressure field.

      all_wrf_2d(num_wrf(2))%sh = all_mm5_3d(index_pstarcrs)%sh
      all_wrf_2d(num_wrf(2))%sh%name(1:8) = 'PSURFACE'
      all_wrf_2d(num_wrf(2))%sh%description(1:46) = 'Total surface pressure                        '
      
   END SUBROUTINE compute_surface_p

   SUBROUTINE compute_total_press ( ix , jx , kx_mm5 , kx_wrf , ptop )

      USE mm5_input

      IMPLICIT NONE

      INTEGER , INTENT(IN) :: ix , jx , kx_mm5 , kx_wrf

      REAL    , INTENT(IN) :: ptop

      INTEGER :: index_pp , index_pstarcrs , index_sigma
      INTEGER :: i , j , k , loop

      !  Find the reference pstar (reference p surface - reference p top).  
  
      index_pstarcrs = 0 
      find_pstarcrs : DO loop = 1 , num_mm5(2) 
         IF ( all_mm5_2d(loop)%sh%name(1:8) .EQ. 'PSTARCRS' ) THEN 
            index_pstarcrs = loop
            EXIT find_pstarcrs            
         END IF 
      END DO find_pstarcrs

      IF ( index_pstarcrs .EQ. 0 ) THEN
         PRINT '(A)','Could not find the index for the PSTARCRS data for total pressure computation.'
         STOP 'COMPUTE_total_p_no_PSTARCRS'
      END IF

      !  Find the pressure perturbation.
  
      index_pp = 0 
      find_pp : DO loop = 1 , num_mm5(3) 
         IF ( all_mm5_3d(loop)%sh%name(1:8) .EQ. 'PP      ' ) THEN 
            index_pp = loop
            EXIT find_pp
         END IF 
      END DO find_pp

      IF ( index_pp .EQ. 0 ) THEN
         PRINT '(A)','Could not find the index for the PP data.'
         STOP 'COMPUTE_total_p_no_pp'
      END IF
   
      !  Find the sigma data.

      index_sigma  = 0
      find_sigma  : DO loop = 1 , num_mm5(1)
         IF ( all_mm5_1d(loop)%sh%name(1:8) .EQ. 'SIGMAH  ' ) THEN
            index_sigma    = loop
            EXIT find_sigma
         END IF
      END DO find_sigma

      IF ( index_sigma    .EQ. 0 ) THEN
         PRINT '(A)','Could not find the index for the SIGMA data for total pressure.'
         STOP 'COMPUTE_total_p_no_SIGMA'
      END IF

      !  Allocate space for the total pressure.

      num_mm5(3) = num_mm5(3) + 1
      ALLOCATE ( all_mm5_3d(num_mm5(3))%data(ix,jx,kx_mm5) )
      num_wrf(3) = num_wrf(3) + 1
      ALLOCATE ( all_wrf_3d(num_mm5(3))%data(ix,jx,kx_wrf+1) )
      all_wrf_3d(num_mm5(3))%data = 0

      !  Compute total pressure.

      DO k = 1 , kx_mm5
         DO j = 1 , jx - 1
            DO i = 1 , ix - 1
               all_mm5_3d(num_mm5(3))%data(i,j,k) = all_mm5_2d(index_pstarcrs)%data(i,j) * all_mm5_1d(index_sigma)%data(k) + &
                                                    all_mm5_3d(index_pp)%data(i,j,k) + ptop
            END DO
         END DO
      END DO

      !  Fix the small header for the new total pressure field.

      all_mm5_3d(num_mm5(3))%sh = all_mm5_3d(index_pp)%sh
      all_mm5_3d(num_mm5(3))%sh%name(1:8) = 'PRESSURE'
      all_mm5_3d(num_mm5(3))%sh%description(1:46) = 'Total pressure                                '
      all_wrf_3d(num_mm5(3))%sh = all_mm5_3d(num_mm5(3))%sh
      all_wrf_3d(num_mm5(3))%sh%end_dims(3) = kx_wrf+1

   END SUBROUTINE compute_total_press

   SUBROUTINE compute_height_mm5 ( ix , jx , kx , ptop , p0 , ts0 , tlp , tiso , &
                                   zeta_top , numzeta , dzetaw )

      USE constants 
      USE mm5_input 

      IMPLICIT NONE 

      INTEGER , INTENT(IN) :: ix , jx , kx
      REAL    , INTENT(IN) :: ptop , p0  , ts0 , tlp , tiso
      REAL    , INTENT(IN) :: zeta_top , dzetaw
      INTEGER , INTENT(IN) :: numzeta

      INTEGER :: i , j , k , loop
      
      INTEGER :: index_terrain , index_pstarcrs , index_sigma

      REAL , DIMENSION ( ix,jx ) :: terrain , pstar , terrain_dummy
      REAL , DIMENSION ( kx+1 ) :: sigmaf , sigma
      REAL :: p3d0 , xmin

      !  There are four height staggerings that we need to compute:
      !        1. half layers, center location          - zt_mm5
      !        2. full layers, center location (for w)  - zw_mm5
      !        3. half layers, u location               - zu_mm5
      !        4. half layers, v location               - zv_mm5

      !  If we have not yet allocated space for the different heights,
      !  we need to do so.  If we have already allocated the space and
      !  computed the values, we drop through this IF test and exit this
      !  routine.  This is an initialization.

      IF ( .NOT. z_mm5_is_allocated ) THEN

         ALLOCATE ( zt_mm5 ( ix,jx,kx+1) )
         ALLOCATE ( zw_mm5 ( ix,jx,kx+1) )
         ALLOCATE ( zu_mm5 ( ix,jx,kx+1) )
         ALLOCATE ( zv_mm5 ( ix,jx,kx+1) )
         z_mm5_is_allocated = .TRUE.

         !  Find the reference pstar (reference p surface - reference p top).  
     
         index_pstarcrs = 0 
         find_pstarcrs : DO loop = 1 , num_mm5(2) 
            IF ( all_mm5_2d(loop)%sh%name(1:8) .EQ. 'PSTARCRS' ) THEN 
               index_pstarcrs = loop
               DO j = 1 , jx - 1
                  DO i = 1 , ix - 1 
                     pstar(i,j) = all_mm5_2d(index_pstarcrs)%data(i,j)
                  END DO
               END DO
               EXIT find_pstarcrs            
            END IF 
         END DO find_pstarcrs
   
         IF ( index_pstarcrs .EQ. 0 ) THEN
            PRINT '(A)','Could not find the index for the PSTARCRS data.'
            STOP 'COMPUTE_height_no_PSTARCRS'
         END IF
   
         !  Find the terrain elevation.
   
         index_terrain  = 0
         find_terrain  : DO loop = 1 , num_mm5(2)
            IF ( all_mm5_2d(loop)%sh%name(1:8) .EQ. 'TERRAIN ' ) THEN
               index_terrain  = loop
               DO j = 1 , jx - 1
                  DO i = 1 , ix - 1 
                     terrain(i,j) = all_mm5_2d(index_terrain)%data(i,j)
                  END DO
               END DO
               EXIT find_terrain             
            END IF
         END DO find_terrain 
   
         IF ( index_terrain  .EQ. 0 ) THEN
            PRINT '(A)','Could not find the index for the TERRAIN data for MM5.'
            STOP 'COMPUTE_height_no_TERRAIN_MM5'
         END IF
   
         !  Find the sigma data.
   
         index_sigma  = 0
         find_sigma  : DO loop = 1 , num_mm5(1)
            IF ( all_mm5_1d(loop)%sh%name(1:8) .EQ. 'SIGMAH  ' ) THEN
               index_sigma    = loop
               DO k = 1 , kx
                  sigma(k) = all_mm5_1d(index_sigma)%data(k)
               END DO
               EXIT find_sigma
            END IF
         END DO find_sigma
   
         IF ( index_sigma    .EQ. 0 ) THEN
            PRINT '(A)','Could not find the index for the SIGMA data.'
            STOP 'COMPUTE_height_no_SIGMA'
         END IF
    
         !  Compute full sigma levels.
   
         sigmaf(1) = 1.
         sigmaf(kx+1) = 0.
         DO k = 2 , kx
            sigmaf(k) = 2. * sigma(k-1) - sigmaf(k-1)
         END DO
   
         !  1. zt_mm5, Z at half levels, at center points.
   
         xmin = 1.e6
         DO k = 1 , kx
            DO j = 1 , jx - 1
               DO i = 1 , ix - 1 
                  p3d0 = pstar(i,j) * sigma(k) + ptop
                  zt_mm5(i,j,k) = - ( R * tlp / ( 2. * g ) * ( LOG ( p3d0 / p0 ) ) **2 + &
                                  R * ts0 /        g   *   LOG ( p3d0 / p0 ) )
                  if(k.eq.kx) xmin=min(xmin,zt_mm5(i,j,k))
               END DO
            END DO
         END DO
   
         !  2. zw_mm5, Z at full levels, center points (w).
   
         DO k = 1 , kx + 1
            DO j = 1 , jx - 1
               DO i = 1 , ix - 1
                  p3d0 = pstar(i,j) * sigmaf(k) + ptop
                  zw_mm5(i,j,k) = - ( R * tlp / ( 2. * g ) * ( LOG ( p3d0 / p0 ) ) **2 + &
                                  R * ts0 /        g   *   LOG ( p3d0 / p0 ) )
               END DO
            END DO
         END DO
   
         !  3. zu_mm5, Z at half levels, u points.
   
         DO j = 1 , jx - 1
            DO i = 1 , ix - 2 
               terrain_dummy(i+1,j) = ( terrain(i,j) + terrain(i+1,j) ) * 0.5
            END DO
         END DO
         DO j = 1 , jx - 1
            terrain_dummy( 1, j) = terrain(   1,   j)
            terrain_dummy(ix, j) = terrain(ix-1,   j)
         END DO
   
         DO j = 1 , jx - 1
            DO i = 1 , ix
               pstar(i,j) = p0 * &
                                 EXP ( -1. * ts0 / tlp + &
                                       sqrt ( ( ts0 / tlp ) **2 - 2. * g * terrain_dummy(i,j) / ( tlp * R ) ) ) - &
                            ptop
            END DO
         END DO
               
         DO k = 1 , kx
            DO j = 1 , jx - 1
               DO i = 1 , ix
                  p3d0 = pstar(i,j) * sigma(k) + ptop
                  zu_mm5(i,j,k) = - ( R * tlp / ( 2. * g ) * ( LOG ( p3d0 / p0 ) ) **2 + &
                                  R * ts0 /        g   *   LOG ( p3d0 / p0 ) )
                  if(k.eq.kx) xmin=min(xmin,zu_mm5(i,j,k))
               END DO
            END DO
         END DO
   
         !  4. zv_mm5, Z at half levels, v points.
   
         DO j = 1 , jx - 2
            DO i = 1 , ix - 1 
               terrain_dummy(i,j+1) = ( terrain(i,j) + terrain(i,j+1) ) * 0.5
            END DO
         END DO
         DO i = 1 , ix - 1
            terrain_dummy( i, 1) = terrain(   i,   1)
            terrain_dummy( i,jx) = terrain(   i,jx-1)
         END DO
   
         DO j = 1 , jx - 1
            DO i = 1 , ix
               pstar(i,j) = p0 * &
                                 EXP ( -1. * ts0 / tlp + &
                                       sqrt ( ( ts0 / tlp ) **2 - 2. * g * terrain_dummy(i,j) / ( tlp * R ) ) ) - &
                            ptop
            END DO
         END DO
               
         DO k = 1 , kx
            DO j = 1 , jx - 1
               DO i = 1 , ix
                  p3d0 = pstar(i,j) * sigma(k) + ptop
                  zv_mm5(i,j,k) = - ( R * tlp / ( 2. * g ) * ( LOG ( p3d0 / p0 ) ) **2 + &
                                  R * ts0 /        g   *   LOG ( p3d0 / p0 ) )
                  if(k.eq.kx) xmin=min(xmin,zv_mm5(i,j,k))
               END DO
            END DO
         END DO

      END IF

      !  Now that we have the various heights in the MM5 domain, we need to check to
      !  see that the levels that we are requesting fit beneath our top level.

      IF ( xmin .LT. zeta_top ) THEN
         PRINT '(A)','TROUBLES with chosen levels.'
         PRINT '(A,F6.0,A)','The maximum allowable WRF height is ',xmin,' m.'
         PRINT '(A,F6.0,A)','With your namelist options of thickness and # or levels, we get a maximum height of ',zeta_top,' m.'
         STOP 'Lower_the_effective_max_height'
      ELSE
         PRINT '(A,F6.0,A)','Minimum height from MM5 top level is = ',xmin,' m.'
      END IF

   END SUBROUTINE compute_height_mm5

   SUBROUTINE compute_height_wrf ( ix , jx , kx , dzetaw , zeta_top )

      USE mm5_input 

      IMPLICIT NONE 

      INTEGER , INTENT(IN) :: ix , jx , kx
      REAL    , INTENT(IN) :: dzetaw
      REAL    , INTENT(IN) :: zeta_top

      INTEGER :: i , j , k , loop
      
      INTEGER :: index_terrain

      REAL , DIMENSION ( kx+1 ) :: zeta_full , zeta_half

      REAL , DIMENSION ( ix,jx ) :: terrain , terrain_dummy

      !  There are four height staggerings that we need to compute:
      !        1. half layers, center location          - zt_wrf
      !        2. full layers, center location (for w)  - zw_wrf
      !        3. half layers, u location               - zu_wrf
      !        4. half layers, v location               - zv_wrf

      !  If we have not yet allocated space for the different heights,
      !  we need to do so.  If we have already allocated the space and
      !  computed the values, we drop through this IF test and exit this
      !  routine.  This is an initialization.

      IF ( .NOT. z_wrf_is_allocated ) THEN

         ALLOCATE ( zt_wrf ( ix,jx,kx+1) )
         ALLOCATE ( zw_wrf ( ix,jx,kx+1) )
         ALLOCATE ( zu_wrf ( ix,jx,kx+1) )
         ALLOCATE ( zv_wrf ( ix,jx,kx+1) )
         z_wrf_is_allocated = .TRUE.
   
         !  Find the terrain elevation.
   
         index_terrain  = 0
         find_terrain  : DO loop = 1 , num_mm5(2)
            IF ( all_mm5_2d(loop)%sh%name(1:8) .EQ. 'TERRAIN ' ) THEN
               index_terrain  = loop
               DO j = 1 , jx - 1
                  DO i = 1 , ix - 1 
                     terrain(i,j) = all_mm5_2d(index_terrain)%data(i,j)
                  END DO
               END DO
               EXIT find_terrain             
            END IF
         END DO find_terrain 
   
         IF ( index_terrain  .EQ. 0 ) THEN
            PRINT '(A)','Could not find the index for the TERRAIN data for WRF.'
            STOP 'COMPUTE_height_no_TERRAIN_WRF'
         END IF

         !  Compute the zeta levels from the thickness and the number of levels.

         ALLOCATE ( all_wrf_1d(1)%data(kx+1) )
         zeta_full(1) = 0.
         all_wrf_1d(1)%data(1) = 0
         DO k = 2 , kx+1
            zeta_full(k)= zeta_full(k-1) + dzetaw
            all_wrf_1d(1)%data(k) = zeta_full(k)
         END DO 
         all_wrf_1d(1)%sh%name(1:8) = 'ZETAFULL'

         ALLOCATE ( all_wrf_1d(2)%data(kx+1) )
         DO k = 1 , kx
            zeta_half(k) = ( zeta_full(k) + zeta_full(k+1) ) * 0.5
            all_wrf_1d(2)%data(k) = zeta_half(k)
         END DO
         all_wrf_1d(2)%sh%name(1:8) = 'ZETAHALF'
         num_wrf(1) = 2
          
        
         !  1. zt_wrf, Z at half levels, at center points.
   
         DO k = 1 , kx
            DO j = 1 , jx - 1
               DO i = 1 , ix - 1 
                  zt_wrf(i,j,k) = zeta_half(k) / zeta_top * ( zeta_top - terrain(i,j) ) + terrain(i,j)
               END DO
            END DO
         END DO
   
         !  2. zw_wrf, Z at full levels, center points (w).
   
         DO k = 1 , kx + 1
            DO j = 1 , jx - 1
               DO i = 1 , ix - 1
                  zw_wrf(i,j,k) = zeta_full(k) / zeta_top * ( zeta_top - terrain(i,j) ) + terrain(i,j)
               END DO
            END DO
         END DO
   
         !  3. zu_wrf, Z at half levels, u points.
   
         DO j = 1 , jx - 1
            DO i = 1 , ix - 2 
               terrain_dummy(i+1,j) = ( terrain(i,j) + terrain(i+1,j) ) * 0.5
            END DO
         END DO
         DO j = 1 , jx - 1
            terrain_dummy( 1, j) = terrain(   1,   j)
            terrain_dummy(ix, j) = terrain(ix-1,   j)
         END DO
   
         DO k = 1 , kx
            DO j = 1 , jx - 1
               DO i = 1 , ix
                  zu_wrf(i,j,k) = zeta_half(k) / zeta_top * ( zeta_top - terrain_dummy(i,j) ) + terrain_dummy(i,j)
               END DO
            END DO
         END DO
   
         !  4. zv_wrf, Z at half levels, v points.
   
         DO j = 1 , jx - 2
            DO i = 1 , ix - 1 
               terrain_dummy(i,j+1) = ( terrain(i,j) + terrain(i,j+1) ) * 0.5
            END DO
         END DO
         DO i = 1 , ix - 1
            terrain_dummy( i, 1) = terrain(   i,   1)
            terrain_dummy( i,jx) = terrain(   i,jx-1)
         END DO
   
         DO k = 1 , kx
            DO j = 1 , jx
               DO i = 1 , ix - 1
                  zv_wrf(i,j,k) = zeta_half(k) / zeta_top * ( zeta_top - terrain_dummy(i,j) ) + terrain_dummy(i,j)
               END DO
            END DO
         END DO

      END IF

   END SUBROUTINE compute_height_wrf

   SUBROUTINE compute_2d ( ix , jx , kx , zetatop ) 

      USE mm5_input 
      USE header_space 

      IMPLICIT NONE 

      INTEGER , INTENT(IN) :: ix , jx , kx
      REAL , INTENT(IN) :: zetatop

      INTEGER :: i , j , k , loop , previous_max
      
      INTEGER :: index_mapx , index_mapd , index_latitude , index_longitude , &
                 index_terrain

      REAL :: xlonc , cone_fac
   
      !  Find the map scale factors.
   
      index_mapx  = 0
      find_mapx  : DO loop = 1 , num_mm5(2)
         IF ( all_mm5_2d(loop)%sh%name(1:8) .EQ. 'MAPFACCR' ) THEN
            index_mapx  = loop
            EXIT find_mapx             
         END IF
      END DO find_mapx 

      IF ( index_mapx  .EQ. 0 ) THEN
         PRINT '(A)','Could not find the index for the map factor cross data for MM5.'
         STOP 'COMPUTE_2d_no_mapfacx_MM5'
      END IF
   
      index_mapd  = 0
      find_mapd  : DO loop = 1 , num_mm5(2)
         IF ( all_mm5_2d(loop)%sh%name(1:8) .EQ. 'MAPFACDT' ) THEN
            index_mapd  = loop
            EXIT find_mapd             
         END IF
      END DO find_mapd 

      IF ( index_mapd  .EQ. 0 ) THEN
         PRINT '(A)','Could not find the index for the map factor dot data for MM5.'
         STOP 'COMPUTE_2d_no_mapfacd_MM5'
      END IF

      !  Allocate space for the three map factors.

      previous_max = num_wrf(2)
      num_wrf(2) = num_wrf(2) + 3

      ALLOCATE ( all_wrf_2d(previous_max+1)%data(ix,jx) )
      ALLOCATE ( all_wrf_2d(previous_max+2)%data(ix,jx) )
      ALLOCATE ( all_wrf_2d(previous_max+3)%data(ix,jx) )

      !  Fix the small header for the new map scale factor fields.

      all_wrf_2d(previous_max+1)%sh                   = all_mm5_2d(index_mapx)%sh
      all_wrf_2d(previous_max+1)%sh%name(1:8)         = 'MSFT    '
      all_wrf_2d(previous_max+1)%sh%units(1:25)       = 'km km{-1}                '
      all_wrf_2d(previous_max+1)%sh%description(1:46) = 'Map Scale Factor - central points             '

      all_wrf_2d(previous_max+2)%sh                   = all_mm5_2d(index_mapx)%sh
      all_wrf_2d(previous_max+2)%sh%name(1:8)         = 'MSFU    '
      all_wrf_2d(previous_max+2)%sh%units(1:25)       = 'km km{-1}                '
      all_wrf_2d(previous_max+2)%sh%description(1:46) = 'Map Scale Factor - u points                   '

      all_wrf_2d(previous_max+3)%sh                   = all_mm5_2d(index_mapx)%sh
      all_wrf_2d(previous_max+3)%sh%name(1:8)         = 'MSFV    '
      all_wrf_2d(previous_max+3)%sh%units(1:25)       = 'km km{-1}                '
      all_wrf_2d(previous_max+3)%sh%description(1:46) = 'Map Scale Factor - v points                   '

      all_wrf_2d(previous_max+1)%data = 0
      DO j = 1 , jx - 1
         DO i = 1 , ix - 1 
            all_wrf_2d(previous_max+1)%data(i,j) = all_mm5_2d(index_mapx)%data(i,j)
         END DO      
      END DO

      all_wrf_2d(previous_max+2)%data = 0
      DO j = 1 , jx - 1
         DO i = 1 , ix
            all_wrf_2d(previous_max+2)%data(i,j) = ( all_mm5_2d(index_mapd)%data(i,j  ) + & 
                                                     all_mm5_2d(index_mapd)%data(i,j+1) ) * 0.5
         END DO      
      END DO

      all_wrf_2d(previous_max+3)%data = 0
      DO j = 1 , jx 
         DO i = 1 , ix - 1
            all_wrf_2d(previous_max+3)%data(i,j) = ( all_mm5_2d(index_mapd)%data(i  ,j) + & 
                                                     all_mm5_2d(index_mapd)%data(i+1,j) ) * 0.5
         END DO      
      END DO
   
      !  Find the longitude.
   
      index_longitude  = 0
      find_longitude  : DO loop = 1 , num_mm5(2)
         IF ( all_mm5_2d(loop)%sh%name(1:8) .EQ. 'LONGICRS' ) THEN
            index_longitude  = loop
            EXIT find_longitude             
         END IF
      END DO find_longitude 

      IF ( index_longitude  .EQ. 0 ) THEN
         PRINT '(A)','Could not find the index for the LONGITUDE data for MM5.'
         STOP 'COMPUTE_2d_no_LONGITUDE_MM5'
      END IF

      xlonc = bhr_mm5(3,1)
      cone_fac = bhr_mm5(4,1)

      !  Allocate space for the density.

      previous_max = num_wrf(2)
      num_wrf(2) = num_wrf(2) + 2

      ALLOCATE ( all_wrf_2d(previous_max+1)%data(ix,jx) )
      ALLOCATE ( all_wrf_2d(previous_max+2)%data(ix,jx) )

      !  Fix the small header for the sin(alpha) and cos(alpha),
      !  where alpha is xlonc - longitude(i,j)

      all_wrf_2d(previous_max+1)%sh                   = all_mm5_2d(index_longitude)%sh
      all_wrf_2d(previous_max+1)%sh%name(1:8)         = 'SINA    '
      all_wrf_2d(previous_max+1)%sh%units(1:25)       = 'radians                  '
      all_wrf_2d(previous_max+1)%sh%description(1:46) = 'sin(xlonc-longitude)                          '

      all_wrf_2d(previous_max+2)%sh                   = all_mm5_2d(index_longitude)%sh
      all_wrf_2d(previous_max+2)%sh%name(1:8)         = 'COSA    '
      all_wrf_2d(previous_max+2)%sh%units(1:25)       = 'radians                  '
      all_wrf_2d(previous_max+2)%sh%description(1:46) = 'cos(xlonc-longitude)                          '
 
      !  Compute sin(alpha), cos(alpha).

      all_wrf_2d(previous_max+1)%data = 0
      all_wrf_2d(previous_max+2)%data = 0
      DO j = 1 , jx - 1
         DO i = 1 , ix
            all_wrf_2d(previous_max+1)%data(i,j) = &
            sin ( cone_fac * ( xlonc - all_mm5_2d(index_longitude)%data(i,j) ) * 3.14159265359 / 180. )
            all_wrf_2d(previous_max+2)%data(i,j) = &
            cos ( cone_fac * ( xlonc - all_mm5_2d(index_longitude)%data(i,j) ) * 3.14159265359 / 180. )
         END DO      
      END DO
   
      !  Find the latitude parameter.
   
      index_latitude  = 0
      find_latitude  : DO loop = 1 , num_mm5(2)
         IF ( all_mm5_2d(loop)%sh%name(1:8) .EQ. 'LATITCRS' ) THEN
            index_latitude  = loop
            EXIT find_latitude             
         END IF
      END DO find_latitude 

      IF ( index_latitude  .EQ. 0 ) THEN
         PRINT '(A)','Could not find the index for the LATITUDE data for MM5.'
         STOP 'COMPUTE_2d_no_LATITUDE_MM5'
      END IF

      !  Allocate space for the two components of Coriolis.

      previous_max = num_wrf(2)
      num_wrf(2) = num_wrf(2) + 2

      ALLOCATE ( all_wrf_2d(previous_max+1)%data(ix,jx) )
      ALLOCATE ( all_wrf_2d(previous_max+2)%data(ix,jx) )

      !  Fix the small header for the new Coriolis fields.

      all_wrf_2d(previous_max+1)%sh                   = all_mm5_2d(index_latitude)%sh
      all_wrf_2d(previous_max+1)%sh%name(1:8)         = 'E       '
      all_wrf_2d(previous_max+1)%sh%units(1:25)       = 's{-1}                    '
      all_wrf_2d(previous_max+1)%sh%description(1:46) = 'Coriolis Parameter                            '

      all_wrf_2d(previous_max+2)%sh                   = all_mm5_2d(index_latitude)%sh
      all_wrf_2d(previous_max+2)%sh%name(1:8)         = 'F       '
      all_wrf_2d(previous_max+2)%sh%units(1:25)       = 's{-1}                    '
      all_wrf_2d(previous_max+2)%sh%description(1:46) = 'Coriolis Parameter                            '

      !  Coriolis = 2 omega * sin(lat), omega = 2 pi / 86400

      DO j = 1 , jx - 1
         DO i = 1 , ix - 1 
            all_wrf_2d(previous_max+1)%data(i,j) = 4. * 3.14159265359 / 86400. * &
                                                   SIN ( all_mm5_2d(index_latitude)%data(i,j) * 3.14159265359 / 180. )
            all_wrf_2d(previous_max+2)%data(i,j) = 4. * 3.14159265359 / 86400. * &
                                                   COS ( all_mm5_2d(index_latitude)%data(i,j) * 3.14159265359 / 180. )
         END DO      
      END DO
   
      !  Find the terrain.
   
      index_terrain  = 0
      find_terrain  : DO loop = 1 , num_mm5(2)
         IF ( all_mm5_2d(loop)%sh%name(1:8) .EQ. 'TERRAIN ' ) THEN
            index_terrain  = loop
            EXIT find_terrain             
         END IF
      END DO find_terrain 

      IF ( index_terrain  .EQ. 0 ) THEN
         PRINT '(A)','Could not find the index for the TERRAIN data for MM5.'
         STOP 'COMPUTE_2d_no_TERRAIN_MM5'
      END IF

      !  Allocate space for dz/dzeta and dzeta/dz.

      previous_max = num_wrf(2)
      num_wrf(2) = num_wrf(2) + 2

      ALLOCATE ( all_wrf_2d(previous_max+1)%data(ix,jx) )
      ALLOCATE ( all_wrf_2d(previous_max+2)%data(ix,jx) )

      !  Fix the small header for dz/dzeta and dzeta/dz.

      all_wrf_2d(previous_max+1)%sh                   = all_mm5_2d(index_terrain)%sh
      all_wrf_2d(previous_max+1)%sh%name(1:8)         = 'ZETA_Z  '
      all_wrf_2d(previous_max+1)%sh%units(1:25)       = 'm m{-1}                  '
      all_wrf_2d(previous_max+1)%sh%description(1:46) = 'd(zeta) / d(z)                                '

      all_wrf_2d(previous_max+2)%sh                   = all_mm5_2d(index_terrain)%sh
      all_wrf_2d(previous_max+2)%sh%name(1:8)         = 'Z_ZETA  '
      all_wrf_2d(previous_max+2)%sh%units(1:25)       = 'm m{-1}                  '
      all_wrf_2d(previous_max+2)%sh%description(1:46) = 'd(z) / d(zeta)                                '

      !  Compute zeta_z and z_zeta.

      DO j = 1 , jx - 1
         DO i = 1 , ix - 1 
            all_wrf_2d(previous_max+1)%data(i,j) =  zetatop / ( zetatop - all_mm5_2d(index_terrain)%data(i,j) ) 
            all_wrf_2d(previous_max+2)%data(i,j) =  ( zetatop - all_mm5_2d(index_terrain)%data(i,j) ) / zetatop
         END DO      
      END DO
   
   END SUBROUTINE compute_2d

END MODULE diags
