!WRF:DRIVER_LAYER:DOMAIN_OBJECT
!
!  Following are the routines contained within this MODULE:

!  alloc_space_domain                1. Allocate the space for a single domain (constants
!                                       and null terminate pointers).
!                                    2. Connect the domains as a linked list.
!                                    3. Store all of the domain constants.
!                                    4. CALL alloc_space_field.

!  alloc_space_field                 1. Allocate space for the gridded data required for
!                                       each domain.

!  dealloc_space_domain              1. Reconnect linked list nodes since the current
!                                       node is removed.
!                                    2. CALL dealloc_space_field.
!                                    3. Deallocate single domain.

!  dealloc_space_field               1. Deallocate each of the fields for a particular
!                                       domain.

!  first_loc_integer                 1. Find the first incidence of a particular
!                                       domain identifier from an array of domain
!                                       identifiers.

MODULE module_domain

   USE module_driver_constants
   USE module_machine
   USE module_state_description
   USE module_wrf_error

   CHARACTER (LEN=80) program_name

   !  An entire domain.  This contains multiple meteorological fields by having
   !  arrays (such as "data_3d") of pointers for each field.  Also inside each
   !  domain is a link to a couple of other domains, one is just the "next"
   !  domain that is to be stored, the other is the next domain which happens to
   !  also be on the "same_level".

   TYPE domain_ptr
      TYPE(domain), POINTER :: ptr
   END TYPE domain_ptr

   TYPE domain

! SEE THE INCLUDE FILE FOR DEFINITIONS OF STATE FIELDS WITHIN THE DOMAIN DATA STRUTURE
#include <state_struct_items.inc>

#ifdef RSL
      INTEGER                                             :: comms( WRF_RSL_RK_NCOMMS )
#endif

      INTEGER                                             :: domdesc
      INTEGER                                             :: communicator
      INTEGER,POINTER                                     :: mapping(:,:)
      INTEGER,POINTER                                     :: i_start(:),i_end(:)
      INTEGER,POINTER                                     :: j_start(:),j_end(:)
      INTEGER                                             :: max_tiles
      INTEGER                                             :: num_tiles        ! taken out of namelist 20000908
      INTEGER                                             :: num_tiles_x      ! taken out of namelist 20000908
      INTEGER                                             :: num_tiles_y      ! taken out of namelist 20000908
      INTEGER                                             :: num_tiles_spec   ! place to store number of tiles computed from 
                                                                              ! externally specified params

      TYPE(domain_ptr) , DIMENSION( : ) , POINTER             :: parents                            
      TYPE(domain_ptr) , DIMENSION( : ) , POINTER             :: nests                            
      TYPE(domain) , POINTER                              :: sibling ! overlapped domains at same lev
      INTEGER                                             :: num_parents, num_nests, num_siblings
      INTEGER      , DIMENSION( max_parents )             :: child_of_parent
      INTEGER      , DIMENSION( max_nests )               :: active

      TYPE(domain) , POINTER                              :: next
      TYPE(domain) , POINTER                              :: same_level

   END TYPE domain

   !  Now that a "domain" TYPE exists, we can use it to store a few pointers
   !  to this type.  These are primarily for use in traversing the linked list.
   !  The "head_grid" is always the pointer to the first domain that is
   !  allocated.  This is available and is not to be changed.  The others are
   !  just temporary pointers.

   TYPE(domain) , POINTER :: head_grid , new_grid , next_grid , old_grid

   !  To facilitate an easy integration of each of the domains that are on the
   !  same level, we have an array for the head pointer for each level.  This
   !  removed the need to search through the linked list at each time step to
   !  find which domains are to be active.

   TYPE domain_levels
      TYPE(domain) , POINTER                              :: first_domain
   END TYPE domain_levels

   TYPE(domain_levels) , DIMENSION(max_levels)            :: head_for_each_level
   
CONTAINS

#ifndef DM_PARALLEL

! Default version ; Otherwise module containing interface to DM library will provide

   SUBROUTINE patch_domain( id , domdesc , parent_id , parent_domdesc , &
                            sd1 , ed1 , sp1 , ep1 , sm1 , em1 , &
                            sd2 , ed2 , sp2 , ep2 , sm2 , em2 , &
                            sd3 , ed3 , sp3 , ep3 , sm3 , em3 , &
                            sd4 , ed4 , sp4 , ep4 , sm4 , em4 , &
                            bdx , bdy )

   USE module_machine

   IMPLICIT NONE
   INTEGER, INTENT(IN)   :: sd1 , ed1 , sd2 , ed2 , sd3 , ed3 , sd4 , ed4 , bdx , bdy
   INTEGER, INTENT(OUT)  :: sp1 , ep1 , sp2 , ep2 , sp3 , ep3 , &
                            sm1 , em1 , sm2 , em2 , sm3 , em3
   INTEGER, INTENT(IN)   :: sp4 , ep4 , sm4 , em4
   INTEGER, INTENT(IN)   :: id , parent_id , parent_domdesc
   INTEGER, INTENT(INOUT)  :: domdesc

! this is a trivial version -- 1 patch per processor; 
! use version in module_dm to compute for DM
   sp1 = sd1 ; sp2 = sd2 ; sp3 = sd3
   ep1 = ed1 ; ep2 = ed2 ; ep3 = ed3
   SELECT CASE ( model_data_order )
      CASE ( DATA_ORDER_XYZ )
         sm1 = sp1 - bdx ; em1 = ep1 + bdx
         sm2 = sp2 - bdy ; em2 = ep2 + bdy
         sm3 = sp3       ; em3 = ep3
      CASE ( DATA_ORDER_YXZ )
         sm1 = sp1 - bdy ; em1 = ep1 + bdy
         sm2 = sp2 - bdx ; em2 = ep2 + bdx
         sm3 = sp3       ; em3 = ep3
      CASE ( DATA_ORDER_ZXY )
         sm1 = sp1       ; em1 = ep1
         sm2 = sp2 - bdx ; em2 = ep2 + bdx
         sm3 = sp3 - bdy ; em3 = ep3 + bdy
      CASE ( DATA_ORDER_ZYX )
         sm1 = sp1       ; em1 = ep1
         sm2 = sp2 - bdy ; em2 = ep2 + bdy
         sm3 = sp3 - bdx ; em3 = ep3 + bdx
      CASE ( DATA_ORDER_XZY )
         sm1 = sp1 - bdx ; em1 = ep1 + bdx
         sm2 = sp2       ; em2 = ep2
         sm3 = sp3 - bdy ; em3 = ep3 + bdy
      CASE ( DATA_ORDER_YZX )
         sm1 = sp1 - bdy ; em1 = ep1 + bdy
         sm2 = sp2       ; em2 = ep2
         sm3 = sp3 - bdx ; em3 = ep3 + bdx
   END SELECT
   RETURN
   END SUBROUTINE patch_domain

#endif

!

!  This subroutine is used to ALLOCATE the separate TYPE(domain) space for
!  each domain that is required.  This routine is called initially from a 
!  routine after the input information is available.  This routine is also
!  called during the model run whenever new domains are requested.

!  This routine uses the information contained in the MODULE declarations,
!  most important of which is the pointer to the head of the linked list
!  ("head_grid").  All of the data in the argument list is input that is
!  transferred to the domain for permanent storage.  The pointers are
!  nullified prior to use.

!  This routine calls the SUBROUTINE to ALLOCATE the space for each of the
!  required meteorological fields, alloc_space_fields.

   SUBROUTINE alloc_and_configure_domain ( domain_id , local_time , grid , parent, kid )
      
      IMPLICIT NONE

      !  Input data.

      INTEGER , INTENT(IN)                           :: domain_id
      INTEGER , INTENT(IN)                           :: local_time   ! time on grid
      TYPE( domain ) , POINTER                       :: grid
      TYPE( domain ) , POINTER                       :: parent
      INTEGER , INTENT(IN)                           :: kid    ! which kid of parent am I?

      INTEGER                     :: sd1 , ed1 , sp1 , ep1 , sm1 , em1
      INTEGER                     :: sd2 , ed2 , sp2 , ep2 , sm2 , em2
      INTEGER                     :: sd3 , ed3 , sp3 , ep3 , sm3 , em3
      INTEGER                     :: sd4 , ed4 , sp4 , ep4 , sm4 , em4

      !  Local data.
      TYPE(domain) , POINTER      :: new_grid
      INTEGER                     :: i
      INTEGER                     :: parent_id , parent_domdesc , new_domdesc
      INTEGER                     :: bdyzone_x , bdyzone_y


! This next step uses information that is listed in the registry as namelist_derived
! to properly size the domain and the patches; this in turn is stored in the new_grid
! data structure
      data_ordering : SELECT CASE ( model_data_order )
        CASE  ( DATA_ORDER_XYZ )

          CALL get_s_we( domain_id , sd1 )
          CALL get_e_we( domain_id , ed1 )
          CALL get_s_sn( domain_id , sd2 )
          CALL get_e_sn( domain_id , ed2 )
          CALL get_s_vert( domain_id , sd3 )
          CALL get_e_vert( domain_id , ed3 )

        CASE  ( DATA_ORDER_YXZ )

          CALL get_s_sn( domain_id , sd1 )
          CALL get_e_sn( domain_id , ed1 )
          CALL get_s_we( domain_id , sd2 )
          CALL get_e_we( domain_id , ed2 )
          CALL get_s_vert( domain_id , sd3 )
          CALL get_e_vert( domain_id , ed3 )

        CASE  ( DATA_ORDER_ZXY )

          CALL get_s_vert( domain_id , sd1 )
          CALL get_e_vert( domain_id , ed1 )
          CALL get_s_we( domain_id , sd2 )
          CALL get_e_we( domain_id , ed2 )
          CALL get_s_sn( domain_id , sd3 )
          CALL get_e_sn( domain_id , ed3 )

        CASE  ( DATA_ORDER_ZYX )

          CALL get_s_vert( domain_id , sd1 )
          CALL get_e_vert( domain_id , ed1 )
          CALL get_s_sn( domain_id , sd2 )
          CALL get_e_sn( domain_id , ed2 )
          CALL get_s_we( domain_id , sd3 )
          CALL get_e_we( domain_id , ed3 )

        CASE  ( DATA_ORDER_XZY )

          CALL get_s_we( domain_id , sd1 )
          CALL get_e_we( domain_id , ed1 )
          CALL get_s_vert( domain_id , sd2 )
          CALL get_e_vert( domain_id , ed2 )
          CALL get_s_sn( domain_id , sd3 )
          CALL get_e_sn( domain_id , ed3 )

        CASE  ( DATA_ORDER_YZX )

          CALL get_s_sn( domain_id , sd1 )
          CALL get_e_sn( domain_id , ed1 )
          CALL get_s_vert( domain_id , sd2 )
          CALL get_e_vert( domain_id , ed2 )
          CALL get_s_we( domain_id , sd3 )
          CALL get_e_we( domain_id , ed3 )

      END SELECT data_ordering

      if ( num_time_levels > 3 ) then
        WRITE ( wrf_err_message , * ) 'module_domain: alloc_and_configure_domain: Incorrect value for num_time_levels ', &
                                       num_time_levels
        CALL wrf_error_fatal ( TRIM ( wrf_err_message ) )
      endif

      sd4 = 1
      ed4 = num_time_levels
      sp4 = 1
      ep4 = num_time_levels
      sm4 = 1
      em4 = num_time_levels

      IF (ASSOCIATED(parent)) THEN
        parent_id = parent%id
        parent_domdesc = parent%domdesc
      ELSE
        parent_id = -1
        parent_domdesc = -1
      ENDIF

      CALL get_bdyzone_x( bdyzone_x )
      CALL get_bdyzone_y( bdyzone_y )

      ALLOCATE ( new_grid )
      ALLOCATE ( new_grid%parents( max_parents ) )
      ALLOCATE ( new_grid%nests( max_nests ) )
      NULLIFY( new_grid%sibling )
      DO i = 1, max_nests
         NULLIFY( new_grid%nests(i)%ptr )
      ENDDO
      NULLIFY  (new_grid%next)
      NULLIFY  (new_grid%same_level)
      NULLIFY  (new_grid%i_start)
      NULLIFY  (new_grid%j_start)
      NULLIFY  (new_grid%i_end)
      NULLIFY  (new_grid%j_end)

      CALL patch_domain( domain_id  , new_domdesc , parent_id, parent_domdesc , &
                         sd1 , ed1 , sp1 , ep1 , sm1 , em1 , &
                         sd2 , ed2 , sp2 , ep2 , sm2 , em2 , &
                         sd3 , ed3 , sp3 , ep3 , sm3 , em3 , &
                         sd4 , ed4 , sp4 , ep4 , sm4 , em4 , &
                         bdyzone_x  , bdyzone_y )

      CALL alloc_space_field ( new_grid, domain_id ,                   &
                               sd1, ed1, sd2, ed2, sd3, ed3, sd4, ed4, &
                               sm1, em1, sm2, em2, sm3, em3, sm4, em4  )

      new_grid%id = domain_id
      new_grid%domdesc = new_domdesc
      new_grid%total_time_steps = local_time
      new_grid%num_nests = 0
      new_grid%num_siblings = 0
      new_grid%num_parents = 0
      new_grid%max_tiles   = 0
      new_grid%num_tiles_spec   = 0
      IF ( domain_id .NE. 1 ) THEN
         new_grid%parents(1)%ptr => parent
         new_grid%num_parents = 1
         parent%nests(kid)%ptr => new_grid
         new_grid%child_of_parent(1) = kid    ! note assumption that nest can have only 1 parent
         parent%num_nests = parent%num_nests + 1
      END IF
 
! This next set of includes causes all but the namelist_derived variables to be
! properly assigned to the new_grid record

      !  The domain has a couple of pointers that we need to NULLIFY
      !  right away.

      new_grid%sd31                            = sd1 
      new_grid%ed31                            = ed1
      new_grid%sp31                            = sp1 
      new_grid%ep31                            = ep1 
      new_grid%sm31                            = sm1 
      new_grid%em31                            = em1
      new_grid%sd32                            = sd2 
      new_grid%ed32                            = ed2
      new_grid%sp32                            = sp2 
      new_grid%ep32                            = ep2 
      new_grid%sm32                            = sm2 
      new_grid%em32                            = em2
      new_grid%sd33                            = sd3 
      new_grid%ed33                            = ed3
      new_grid%sp33                            = sp3 
      new_grid%ep33                            = ep3 
      new_grid%sm33                            = sm3 
      new_grid%em33                            = em3
      new_grid%sd34                            = sd4
      new_grid%ed34                            = ed4
      new_grid%sp34                            = sp4
      new_grid%ep34                            = ep4
      new_grid%sm34                            = sm4
      new_grid%em34                            = em4

      SELECT CASE ( model_data_order )
         CASE  ( DATA_ORDER_XYZ )
            new_grid%sd21 = sd1 ; new_grid%sd22 = sd2 ; new_grid%sd23 = sd4 ;
            new_grid%ed21 = ed1 ; new_grid%ed22 = ed2 ; new_grid%ed23 = ed4 ;
            new_grid%sp21 = sp1 ; new_grid%sp22 = sp2 ; new_grid%sp23 = sp4 ;
            new_grid%ep21 = ep1 ; new_grid%ep22 = ep2 ; new_grid%ep23 = ep4 ;
            new_grid%sm21 = sm1 ; new_grid%sm22 = sm2 ; new_grid%sm23 = sm4 ;
            new_grid%em21 = em1 ; new_grid%em22 = em2 ; new_grid%em23 = em4 ;
            new_grid%sd11 = sd1 ; new_grid%sd12 = sd2 ;
            new_grid%ed11 = ed1 ; new_grid%ed12 = ed2 ;
            new_grid%sp11 = sp1 ; new_grid%sp12 = sp2 ;
            new_grid%ep11 = ep1 ; new_grid%ep12 = ep2 ;
            new_grid%sm11 = sm1 ; new_grid%sm12 = sm2 ;
            new_grid%em11 = em1 ; new_grid%em12 = em2 ;
         CASE  ( DATA_ORDER_YXZ )
            new_grid%sd21 = sd1 ; new_grid%sd22 = sd2 ; new_grid%sd23 = sd4 ;
            new_grid%ed21 = ed1 ; new_grid%ed22 = ed2 ; new_grid%ed23 = ed4 ;
            new_grid%sp21 = sp1 ; new_grid%sp22 = sp2 ; new_grid%sp23 = sp4 ;
            new_grid%ep21 = ep1 ; new_grid%ep22 = ep2 ; new_grid%ep23 = ep4 ;
            new_grid%sm21 = sm1 ; new_grid%sm22 = sm2 ; new_grid%sm23 = sm4 ;
            new_grid%em21 = em1 ; new_grid%em22 = em2 ; new_grid%em23 = em4 ;
            new_grid%sd11 = sd1 ; new_grid%sd12 = sd2 ;
            new_grid%ed11 = ed1 ; new_grid%ed12 = ed2 ;
            new_grid%sp11 = sp1 ; new_grid%sp12 = sp2 ;
            new_grid%ep11 = ep1 ; new_grid%ep12 = ep2 ;
            new_grid%sm11 = sm1 ; new_grid%sm12 = sm2 ;
            new_grid%em11 = em1 ; new_grid%em12 = em2 ;
         CASE  ( DATA_ORDER_ZXY )
            new_grid%sd21 = sd2 ; new_grid%sd22 = sd3 ; new_grid%sd23 = sd4 ;
            new_grid%ed21 = ed2 ; new_grid%ed22 = ed3 ; new_grid%ed23 = ed4 ;
            new_grid%sp21 = sp2 ; new_grid%sp22 = sp3 ; new_grid%sp23 = sp4 ;
            new_grid%ep21 = ep2 ; new_grid%ep22 = ep3 ; new_grid%ep23 = ep4 ;
            new_grid%sm21 = sm2 ; new_grid%sm22 = sm3 ; new_grid%sm23 = sm4 ;
            new_grid%em21 = em2 ; new_grid%em22 = em3 ; new_grid%em23 = em4 ;
            new_grid%sd11 = sd2 ; new_grid%sd12 = sd3 ;
            new_grid%ed11 = ed2 ; new_grid%ed12 = ed3 ;
            new_grid%sp11 = sp2 ; new_grid%sp12 = sp3 ;
            new_grid%ep11 = ep2 ; new_grid%ep12 = ep3 ;
            new_grid%sm11 = sm2 ; new_grid%sm12 = sm3 ;
            new_grid%em11 = em2 ; new_grid%em12 = em3 ;
         CASE  ( DATA_ORDER_ZYX )
            new_grid%sd21 = sd2 ; new_grid%sd22 = sd3 ; new_grid%sd23 = sd4 ;
            new_grid%ed21 = ed2 ; new_grid%ed22 = ed3 ; new_grid%ed23 = ed4 ;
            new_grid%sp21 = sp2 ; new_grid%sp22 = sp3 ; new_grid%sp23 = sp4 ;
            new_grid%ep21 = ep2 ; new_grid%ep22 = ep3 ; new_grid%ep23 = ep4 ;
            new_grid%sm21 = sm2 ; new_grid%sm22 = sm3 ; new_grid%sm23 = sm4 ;
            new_grid%em21 = em2 ; new_grid%em22 = em3 ; new_grid%em23 = em4 ;
            new_grid%sd11 = sd2 ; new_grid%sd12 = sd3 ;
            new_grid%ed11 = ed2 ; new_grid%ed12 = ed3 ;
            new_grid%sp11 = sp2 ; new_grid%sp12 = sp3 ;
            new_grid%ep11 = ep2 ; new_grid%ep12 = ep3 ;
            new_grid%sm11 = sm2 ; new_grid%sm12 = sm3 ;
            new_grid%em11 = em2 ; new_grid%em12 = em3 ;
         CASE  ( DATA_ORDER_XZY )
            new_grid%sd21 = sd1 ; new_grid%sd22 = sd3 ; new_grid%sd23 = sd4 ;
            new_grid%ed21 = ed1 ; new_grid%ed22 = ed3 ; new_grid%ed23 = ed4 ;
            new_grid%sp21 = sp1 ; new_grid%sp22 = sp3 ; new_grid%sp23 = sp4 ;
            new_grid%ep21 = ep1 ; new_grid%ep22 = ep3 ; new_grid%ep23 = ep4 ;
            new_grid%sm21 = sm1 ; new_grid%sm22 = sm3 ; new_grid%sm23 = sm4 ;
            new_grid%em21 = em1 ; new_grid%em22 = em3 ; new_grid%em23 = em4 ;
            new_grid%sd11 = sd1 ; new_grid%sd12 = sd3 ;
            new_grid%ed11 = ed1 ; new_grid%ed12 = ed3 ;
            new_grid%sp11 = sp1 ; new_grid%sp12 = sp3 ;
            new_grid%ep11 = ep1 ; new_grid%ep12 = ep3 ;
            new_grid%sm11 = sm1 ; new_grid%sm12 = sm3 ;
            new_grid%em11 = em1 ; new_grid%em12 = em3 ;
         CASE  ( DATA_ORDER_YZX )
            new_grid%sd21 = sd1 ; new_grid%sd22 = sd3 ; new_grid%sd23 = sd4 ;
            new_grid%ed21 = ed1 ; new_grid%ed22 = ed3 ; new_grid%ed23 = ed4 ;
            new_grid%sp21 = sp1 ; new_grid%sp22 = sp3 ; new_grid%sp23 = sp4 ;
            new_grid%ep21 = ep1 ; new_grid%ep22 = ep3 ; new_grid%ep23 = ep4 ;
            new_grid%sm21 = sm1 ; new_grid%sm22 = sm3 ; new_grid%sm23 = sm4 ;
            new_grid%em21 = em1 ; new_grid%em22 = em3 ; new_grid%em23 = em4 ;
            new_grid%sd11 = sd1 ; new_grid%sd12 = sd3 ;
            new_grid%ed11 = ed1 ; new_grid%ed12 = ed3 ;
            new_grid%sp11 = sp1 ; new_grid%sp12 = sp3 ;
            new_grid%ep11 = ep1 ; new_grid%ep12 = ep3 ;
            new_grid%sm11 = sm1 ; new_grid%sm12 = sm3 ;
            new_grid%em11 = em1 ; new_grid%em12 = em3 ;
      END SELECT

      CALL add_config_info_to_grid ( new_grid )           ! this is a mediation layer routine

! Some miscellaneous state that is in the Registry but not namelist data

      new_grid%tiled                           = .false.
      new_grid%patched                         = .false.
      NULLIFY(new_grid%mapping)

! This next set of includes causes all but the namelist_derived variables to be
! properly assigned to the new_grid record

      grid => new_grid

#ifdef DM_PARALLEL
      CALL get_dm_communicator ( grid%communicator )
      CALL define_comms( grid )
#endif

   END SUBROUTINE alloc_and_configure_domain

!

!  This routine ALLOCATEs the required space for the meteorological fields
!  for a specific domain.  The fields are simply ALLOCATEd as an -1.  They
!  are referenced as wind, temperature, moisture, etc. in routines that are
!  below this top-level of data allocation and management (in the solve routine
!  and below).

   SUBROUTINE alloc_space_field ( grid,   id,                                     &
                                  sd31, ed31, sd32, ed32, sd33, ed33, sd34, ed34, &
                                  sm31, em31, sm32, em32, sm33, em33, sm34, em34  )
      
      IMPLICIT NONE
 

      !  Input data.

      TYPE(domain)               , POINTER          :: grid
      INTEGER , INTENT(IN)            :: id
      INTEGER , INTENT(IN)            :: sd31, ed31, sd32, ed32, sd33, ed33, sd34, ed34
      INTEGER , INTENT(IN)            :: sm31, em31, sm32, em32, sm33, em33, sm34, em34
      !  Local data.
      INTEGER dyn_opt, idum1, idum2, num_soil_layers, spec_bdy_width
      INTEGER num_bytes_allocated
      CHARACTER (LEN=256) message

      INTEGER                              :: loop

      !  Each of the following loops is used to ALLOCATE the correct number of
      !  required fields.  The indices with the "_con" suffix (which are for the
      !  fields with the "_constant" suffix) refer to fields that do not have a
      !  time dependence, and therefore do not have more than a single time
      !  level in the spatial allocation.

      CALL get_dyn_opt( dyn_opt )
      CALL get_num_soil_layers( num_soil_layers )
      CALL get_spec_bdy_width( spec_bdy_width )

      CALL set_scalar_indices_from_config( id , idum1 , idum2 )

      num_bytes_allocated = 0 
      IF      ( dyn_opt == 1 ) THEN
        CALL wrf_error_fatal ( 'LEAP FROG DYNAMICS NO LONGER IN WRF' )
      ELSE IF ( dyn_opt == 2 ) THEN
        CALL wrf_message ( 'DYNAMICS OPTION: 2nd ORDER RUNGE-KUTTA' )
#include <rk_allocs.inc>
      ELSE IF ( dyn_opt == 3 ) THEN
        CALL wrf_message ( 'DYNAMICS OPTION: 3rd ORDER RUNGE-KUTTA' )
#include <rk_allocs.inc>
      ELSE
        WRITE( wrf_err_message , * )'Invalid specification of dynamics: dyn_opt = ',dyn_opt
        CALL wrf_error_fatal ( TRIM ( wrf_err_message ) )
      ENDIF
      WRITE(message,*)'alloc_space_field: domain ',id,' ',num_bytes_allocated
      CALL  wrf_message( message )

   END SUBROUTINE alloc_space_field

!

!  This routine is used to DEALLOCATE space for a single domain.  First
!  the pointers in the linked list are fixed (so the one in the middle can
!  be removed).  Second, the field data are all removed through a CALL to 
!  the dealloc_space_domain routine.  Finally, the pointer to the domain
!  itself is DEALLOCATEd.

   SUBROUTINE dealloc_space_domain ( id )
      
      IMPLICIT NONE

      !  Input data.

      INTEGER , INTENT(IN)            :: id

      !  Local data.

      TYPE(domain) , POINTER          :: grid
      LOGICAL                         :: found

      !  Initializations required to start the routine.

      grid => head_grid
      old_grid => head_grid
      found = .FALSE.

      !  The identity of the domain to delete is based upon the "id".
      !  We search all of the possible grids.  It is required to find a domain
      !  otherwise it is a fatal error.  

      find_grid : DO WHILE ( ASSOCIATED(grid) ) 
         IF ( grid%id == id ) THEN
            found = .TRUE.
            old_grid%next => grid%next
            CALL dealloc_space_field ( grid )
            DEALLOCATE(grid)
            EXIT find_grid
         END IF
         old_grid => grid
         grid     => grid%next
      END DO find_grid

      IF ( .NOT. found ) THEN
         WRITE ( wrf_err_message , * ) 'module_domain: dealloc_space_domain: Could not de-allocate grid id ',id
         CALL wrf_error_fatal ( TRIM( wrf_err_message ) ) 
      END IF

   END SUBROUTINE dealloc_space_domain

!

!  This routine DEALLOCATEs each gridded field for this domain.  For each type of
!  different array (1d, 2d, 3d, etc.), the space for each pointer is DEALLOCATEd
!  for every -1 (i.e., each different meteorological field).

   SUBROUTINE dealloc_space_field ( grid )
      
      IMPLICIT NONE

      !  Input data.

      TYPE(domain)              , POINTER :: grid

      !  Local data.

      INTEGER                             :: loop

   END SUBROUTINE dealloc_space_field

!

!  This routine is used to find a specific domain identifier in an array
!  of domain identifiers.

   FUNCTION first_loc_integer ( array , search ) RESULT ( loc ) 
 
      IMPLICIT NONE

      !  Input data.

      INTEGER , INTENT(IN) , DIMENSION(:) :: array
      INTEGER , INTENT(IN)                :: search

      !  Output data.

      INTEGER                             :: loc
      
      !  Local data.

      INTEGER :: loop

      loc = -1
      find : DO loop = 1 , SIZE(array)
         IF ( search == array(loop) ) THEN         
            loc = loop
            EXIT find
         END IF
      END DO find

   END FUNCTION first_loc_integer

   SUBROUTINE init_module_domain
   END SUBROUTINE init_module_domain

END MODULE module_domain
