!IDEAL:MODEL_LAYER:INITIALIZATION
!

!  This MODULE holds the routines which are used to perform various initializations
!  for the individual domains.  

!  This MODULE CONTAINS the following routines:

!  initialize_field_test - 1. Set different fields to different constant
!                             values.  This is only a test.  If the correct
!                             domain is not found (based upon the "id")
!                             then a fatal error is issued.               

MODULE module_initialize

   USE module_domain
!  USE module_io_domain
   USE module_state_description
   USE module_model_constants
   USE module_bc
   USE module_timing
   USE module_configure
#ifdef DM_PARALLEL
   USE module_dm
#endif


CONTAINS

!
!  This routine provides a test initialization for a specific domain.  The
!  input requests which "id" to choose (this is compared with the
!  "id" stored in every domain).  When the correct domain is found,
!  the test data are initialized.  If no domain is found with the required
!  "id", a fatal error is issued.

!  The only argument is the input of the "id" to which all processing
!  refers.  On output, the domain referenced by this "id" has all of
!  its fields initialized, but only in a testing sort of way.


   SUBROUTINE init_coords ( grid,                                     &
                            ht, rdx, rdy, zeta, zetaw, dzetaw, dzeta, &
                            rdzu, rdzw, z, zx, zy, &
                            z_zeta, zeta_z, fzm, fzp, ztop, &
                            cf1, cf2, cf3, &
                            cofrz, cofwr, dtseps,  &
                            ids, ide, jds, jde, kds, kde, &
                            ims, ime, jms, jme, kms, kme, &
                            its, ite, jts, jte, kts, kte )

   USE module_domain

   IMPLICIT NONE

   ! Input data

   TYPE (domain ), POINTER              :: grid

   INTEGER , INTENT(IN)                    :: ids, ide, jds, jde, kds, kde, &
                                              ims, ime, jms, jme, kms, kme, &
                                              its, ite, jts, jte, kts, kte
   REAL , DIMENSION( ims: , jms: ) , INTENT(IN) :: ht
   REAL , INTENT(IN) :: rdx, rdy, dtseps
   REAL , DIMENSION(ims: , kms: , jms: ) , INTENT(OUT) :: z, zx, zy
   REAL , DIMENSION(ims: , jms: ) , INTENT(OUT) :: z_zeta, zeta_z, cofwr
   ! WRF state 1d_constant
   REAL , DIMENSION( kms: ) , INTENT(OUT) :: zeta, zetaw, dzetaw, dzeta, rdzu, rdzw, fzm, fzp, cofrz
   ! WRF state data
   REAL , INTENT(OUT) :: cf1, cf2, cf3
   REAL , INTENT(IN) :: ztop

   ! Local data

   INTEGER               :: itf, jtf, ktf, i, j, k
   REAL                  :: cof1, cof2

   REAL , DIMENSION(ids:ide,kms:kme,jds:jde) :: zw

   ! Executable

   itf=min0(ite,ide-1)
   jtf=min0(jte,jde-1)
   ktf=min0(kte,kde-1)

   !________________________________________________________________________________
   !________________________________________________________________________________
   ! USER-DEFINED PART - SET DZETA OR ZETAW
   !
!   DO k=kts,ktf
!      dzetaw(k)=100.*float(k) ! arbitrary stretched function of k
!   ENDDO
   !________________________________________________________________________________
   DO k=kts,kte
      dzetaw(k)=ztop/(kte-kts)
   ENDDO
   !________________________________________________________________________________
   !________________________________________________________________________________
   zetaw(1)=0.
   zeta(1)=0.5*dzetaw(1)
   DO k=2,ktf
     zetaw(k)=zetaw(k-1)+dzetaw(k-1)
     zeta(k)=zeta(k-1)+0.5*(dzetaw(k)+dzetaw(k-1))
     dzeta(k)=zeta(k)-zeta(k-1)
   ENDDO
   zetaw(kte)=zetaw(ktf)+dzetaw(ktf)
   !________________________________________________________________________________
   !!!k=1
   !!!zeta(k)=0.5*zetaw(k+1)
   !!!DO k=kts+1,ktf
   !!!   dzetaw(k)=zetaw(k)-zetaw(k-1)
   !!!   zeta(k)=zeta(k-1)+0.5*(dzetaw(k)+dzetaw(k-1))
   !!!   dzeta(k)=zeta(k)-zeta(k-1)
   !!!ENDDO
   !!!zetatop=zetaw(kde)
   !________________________________________________________________________________
   !
   z = 0.
   zw = 0.
   DO k=1,ktf
      rdzw(k)=1./dzetaw(k)
      cofrz(k) = dtseps * rdzw(k)
   ENDDO
   DO k=2,ktf
      rdzu(k)=1./dzeta(k)
      fzp(k)=0.5*dzetaw(k  )*rdzu(k)
      fzm(k)=0.5*dzetaw(k-1)*rdzu(k)
   ENDDO
   DO i=max(ids,its-1),itf
     DO j=max(jds,jts-1),jtf
       z_zeta(i,j)=(ztop-ht(i,j))/ztop
       zeta_z(i,j)=ztop/(ztop-ht(i,j))
       cofwr(i,j)=.5 * dtseps * g * zeta_z(i,j)
       DO k=1,ktf
          z(i,k,j)=ht(i,j)+z_zeta(i,j)*zeta(k)
          zw(i,k,j)=ht(i,j)+z_zeta(i,j)*zetaw(k)
   ! map-scale factor not needed here because it is accounted for in full pgf
   !   and omega terms (i.e. z_x is relative to coordinate x not real x)
   ! this still assumes that terrain slope is zero at physical boundaries
   !  (may not be general enough for periodic bc's)
          if(i.ne.ids) then
               zx(i,k,j)=(zw(i,k,j)-zw(i-1,k,j))*rdx
          else
               zx(i,k,j) = 0.
          end if
          if(j.ne.jds) then
               zy(i,k,j)=(zw(i,k,j)-zw(i,k,j-1))*rdy
          else
               zy(i,k,j) = 0.
          end if
       ENDDO
     ENDDO
   ENDDO

   IF ( ite == ide ) THEN
     DO j=jts,jtf
     DO k=1,ktf
      zx(ide,k,j) = 0.
     ENDDO
     ENDDO
   ENDIF

   IF ( jte == jde ) THEN
     DO i=its,itf
     DO k=1,ktf
      zy(i,k,jde) = 0.
     ENDDO
     ENDDO
   ENDIF


   cof1 = (2. * dzeta(2) + dzeta(3))/(dzeta(2) + dzeta(3)) &
                        * dzetaw(1)/dzeta(2)
   cof2 =       dzeta(2)             /(dzeta(2) + dzeta(3)) &
                        * dzetaw(1)/dzeta(3)

   cf1  = fzp(2) + cof1
   cf2  = fzm(2) - cof1 - cof2
   cf3  = cof2

   END SUBROUTINE init_coords
!-----------------------------------------------------------------------------


   SUBROUTINE init_base_state ( grid ,                               &
                                rrb, rtb, pb, pib, pb8w, h_diabatic, &
                                msft, TSK, TMN, dzetaw, z, ht, z_zeta,    &
                                ids, ide, jds, jde, kds, kde,        &
                                ims, ime, jms, jme, kms, kme,        &
                                its, ite, jts, jte, kts, kte        )

   USE module_domain

   IMPLICIT NONE

   ! Input data
   TYPE (domain), POINTER           ::  grid
   INTEGER , INTENT(IN)                    :: ids, ide, jds, jde, kds, kde, &
                                              ims, ime, jms, jme, kms, kme, &
                                              its, ite, jts, jte, kts, kte

   REAL , DIMENSION( ims: , kms: , jms: ) , INTENT(OUT) :: rrb, rtb, pb, pib, pb8w, &
                                                           h_diabatic
   REAL , DIMENSION( ims: , jms: ) , INTENT(INOUT) :: TSK, TMN
   REAL , DIMENSION( ims: , jms: ) , INTENT(IN) :: msft, ht, z_zeta
   REAL , DIMENSION( ims: , kms: , jms: ) , INTENT(IN) :: z
   REAL , DIMENSION( kms: ) , INTENT(IN) :: dzetaw
   
   ! Local data

   REAL , DIMENSION(kms:kme) :: t, p, pi, rho, theta, rhotheta, dz
   REAL :: tslvl, xn2m, tav, p_surf, pi_surf
   INTEGER :: i, j, k, iter, it, itf, jtf, ktf
   REAL , DIMENSION(1:kme) :: u_tmp, qv_tmp, v_tmp, rh_tmp

   REAL :: pi_top, dzk, pitmp, thsfc, p_balance_tolerance
   LOGICAL :: integrate_up

   LOGICAL wrf_on_monitor
   EXTERNAL wrf_on_monitor

! space for reading in initial sounding

   INTEGER, PARAMETER :: npts=1000
   REAL , DIMENSION(npts) :: theta_in, qv_in, rel_hum_in, u_in, v_in, z_in 
   INTEGER :: npts_in
   LOGICAL :: rh_in
   REAL    :: p_surface, p_check

!  space for balanced jet initialization
!  we are going to set the base state equal to the southern column

   REAL, DIMENSION(1:kde-1,jds:jde-1) :: u_jet, rho_jet, theta_jet

! mods to allow parallelization
   REAL, DIMENSION(kds:kde) :: h_diabatic_col , rtb_col , rrb_col , pb_col , pib_col, pb8w_col

   !   constants

!   integrate_up = .false.
   integrate_up = .true.
   p_balance_tolerance = 1.e-03
   iter=10

   itf=min0(ite,ide-1)
   jtf=min0(jte,jde-1)
   ktf=min0(kte,kde-1)
   

   !   can set surface pressure as function of z(i,k,j)

    tslvl=300.
    xn2m=0.0001

    IF ( wrf_on_monitor() ) THEN

      DO j = jts,jte
        DO i = its,ite

          ! do the first column, and then copy results into the other
          ! columns to fill out the reference state
          IF ( i .eq. ids .and. j .eq. jds ) THEN

            call read_input_jet( u_jet, rho_jet, theta_jet, kde-1, jde-jds )

!  initialize the diabatic heating rate array to zero

            DO k=kts,ktf
              h_diabatic_col(k) = 0.
            ENDDO

            dz(k)=0.5*dzetaw(k)*z_zeta(i,j)
            DO k=kms+1,kme
              dz(k)=0.5*(dzetaw(k)+dzetaw(k-1))*z_zeta(i,j)
            ENDDO

            DO k=1,ktf
              theta(k) = theta_jet(k,j)
              rho(k) = rho_jet(k,j)
              p(k) = p1000mb*(R_d*rho_jet(k,j)*theta_jet(k,j)/p1000mb)**cpovcv
              pi(k)=(p(k)/p1000mb)**(rcp)
            ENDDO

  !  put the hydrostatically balanced fields in the state arrays

            DO k=1,ktf
              rtb_col(k)=rho_jet(k,j)*theta_jet(k,j)*z_zeta(i,j)/msft(i,j)
              rrb_col(k)=rho_jet(k,j)*z_zeta(i,j)/msft(i,j)
              pb_col(k)=p(k)
              pib_col(k)=pi(k)
            ENDDO

            IF ((i == ids) .and. (j == jds)) THEN
              write(6,*) ' initial base state profile, k, z, p, theta, rho '
              do k=1, ktf
                write(6,FMT="(1x,i3,4(1x,1pe10.3))") k,z(i,k,j),p(k),rhotheta(k)/rho(k),rho(k)
              enddo
            ENDIF

  ! compute some variables for physics

            thsfc = theta(1)
            pitmp=1.-g*ht(i,j)/(cp*theta(1))
!        TSK(I,J)=thsfc*pitmp
!        TMN(I,J)=TSK(I,J)-0.5
            pb8w_col(1)=p1000mb*pitmp**(cp/R_d)

            do k=2, ktf
              pitmp=pitmp-g*dzetaw(k-1)*z_zeta(i,j)/(cp*theta(k-1))
              pb8w_col(k) = p1000mb*(pitmp**(1./rcp))
            enddo

            pitmp=pitmp-g*dzetaw(kte-1)*z_zeta(i,j)/(cp*theta(kte-1))
            pb8w_col(kte) = p1000mb*(pitmp**(1./rcp))

          END IF
        END DO
      END DO
    END IF
    CALL wrf_dm_bcast_bytes( h_diabatic_col , (kte-kts+1)*RWORDSIZE )
    CALL wrf_dm_bcast_bytes( pb8w_col      , (kte-kts+1)*RWORDSIZE )
    CALL wrf_dm_bcast_bytes( rtb_col       , (kte-kts+1)*RWORDSIZE )
    CALL wrf_dm_bcast_bytes( rrb_col       , (kte-kts+1)*RWORDSIZE )
    CALL wrf_dm_bcast_bytes( pb_col        , (kte-kts+1)*RWORDSIZE )
    CALL wrf_dm_bcast_bytes( pib_col       , (kte-kts+1)*RWORDSIZE )

!  we've finished column (1,1), copy into the rest of the columns

    DO j=jts,jtf
    DO k=1,ktf
    DO i=its,itf
       rtb(i,k,j) =rtb_col(k)
       rrb(i,k,j) =rrb_col(k)
       pb(i,k,j)  =pb_col(k)
       pib(i,k,j) =pib_col(k)
       h_diabatic(i,k,j) =h_diabatic_col(k)
    ENDDO
    ENDDO
    ENDDO
   
    DO j=jts,jtf
    DO k=1,kte
    DO i=its,itf
       pb8w(i,k,j) = pb8w_col(k)
    ENDDO
    ENDDO
    ENDDO

!  DO j=jts,jtf
!  DO i=its,itf
!    TSK(I,J)=tsk(1,1)
!    TMN(I,J)=tmn(1,1)
!  ENDDO
!  ENDDO

   END SUBROUTINE init_base_state

!------------------------------------------------------------------------------------


   SUBROUTINE init_state ( rrb, rtb, pb,                  &
                           rr, rrp, rthp,                 &
                           rr2, rrp2, rthp2,              &
                           moist_1, moist_2, num3d_moist, &
                           chem_1, chem_2, num3d_chem,    &
                           ru_1, ru_2, rv_1, rv_2,        &
                           u_base, v_base, qv_base,       &
                           msft, dzetaw, z, ht, z_zeta,   &
                           dx, dy, TSK, TMN, cf1,cf2,cf3, &
                           config_flags,                  &
                           ids, ide, jds, jde, kds, kde,  &
                           ims, ime, jms, jme, kms, kme,  &
                           its, ite, jts, jte, kts, kte  )

   USE module_domain
   IMPLICIT NONE

   ! Input data
   TYPE (grid_config_rec_type)              :: config_flags

   INTEGER , INTENT(IN)                    :: ids, ide, jds, jde, kds, kde, &
                                              ims, ime, jms, jme, kms, kme, &
                                              its, ite, jts, jte, kts, kte
   INTEGER, INTENT(IN) :: num3d_moist, num3d_chem

   REAL , DIMENSION( ims: , kms: , jms: ) , INTENT(INOUT) :: rrb, rtb, pb
   REAL , DIMENSION( ims: , kms: , jms: ) , INTENT(OUT) :: rr, rrp, rthp
   REAL , DIMENSION( ims: , kms: , jms: ) , INTENT(OUT) :: rr2, rrp2, rthp2
   REAL , DIMENSION( ims: , kms: , jms: ) , INTENT(OUT) :: ru_1, ru_2, rv_1, rv_2
   REAL , DIMENSION( ims: , jms: )          , INTENT(IN) :: msft, ht, z_zeta
   REAL , DIMENSION( ims: , kms: , jms: ) , INTENT(IN) :: z
   REAL , DIMENSION( kms: ) , INTENT(IN)  :: dzetaw
   REAL , DIMENSION( kms: ) , INTENT(OUT) :: u_base, v_base, qv_base
   REAL ,                     INTENT(IN)  :: dx, dy, cf1, cf2, cf3
   REAL , DIMENSION( ims: , jms: ) , INTENT(INOUT) :: TSK, TMN

   REAL , DIMENSION( ims:,kms:,jms:, 1: ), INTENT(OUT) ::  &
      moist_1, moist_2
   REAL , DIMENSION( ims:,kms:,jms:, 1: ), INTENT(OUT) ::  &
      chem_1, chem_2

   ! Local data

   REAL , DIMENSION(kms:kme) :: rthb, rb, thp, rp, pi, p, &
                                pp, rho, theta, rhotheta, dz
   REAL :: xn2l, tslvl
   INTEGER :: i, j, k, iter, it, itf, jtf, ktf, imoist_num

   ! bubble-related local
   LOGICAL :: perturb, integrate_up, p_top
   REAL :: piov2, xcen, ycen, zcen, rcen, htbub, radbub, tpbub
   REAL :: radz
   REAL :: pii, tp
   INTEGER :: icen, jcen

   LOGICAL, EXTERNAL :: wrf_on_monitor

   ! read-in local
!  for initial state only
   REAL , DIMENSION(1:kde-1,1:ide) :: rbj,rtbj,rrj,rtj

   REAL , DIMENSION( 1:kme ) :: u_tmp, v_tmp, qv_tmp, temperature
   INTEGER :: iterm, itm
!   REAL temperature, pressure, qvs, check, moist_coef, delt, dzk, thsfc
   REAL pressure, qvs, es, tmppi, check, moist_coef, delt, dzk, thsfc

! space for reading in initial sounding

   INTEGER, PARAMETER :: npts = 1000
   REAL , DIMENSION(npts) :: theta_in, qv_in, u_in, v_in, z_in, &
     rel_hum_in, rel_hum, theta_v
   INTEGER :: npts_in
   LOGICAL :: rh_in, theta_adjust
   REAL :: p_surface, p_surf, pi_surf, p_check, p_balance_tolerance

!  space for balanced jet initialization

   REAL, DIMENSION(1:kde-1,jds:jde-1) :: u_jet, rho_jet, theta_jet
   logical :: rebalance


   rebalance = .true.
   integrate_up = .true.
   perturb = .true.
   p_balance_tolerance = 1.e-06

!   IF (bubble) THEN
     piov2=2.*atan(1.0)
     htbub=8000.    ! height (m) of center above z=0.
!     radbub=10000.   ! radius (m) of sphere for horizontal
     radbub=2000000.   ! radius (m) of sphere for horizontal
     radz  = 8000.   ! vertical radius
!     tpbub=10.0      ! max temp pert at center (K)
!     tpbub=0.0      ! max temp pert at center (K)
     tpbub=1.0      ! max temp pert at center (K)
!     icen=3*ide/4     ! center i
     icen=ide/4     ! center i
!     icen=ide-3     ! center i
!     icen=ide/2+12     ! center i
     jcen=jde/2     ! center j
!   ENDIF

   IF (config_flags%mp_physics /= 0)  THEN
     write(0,*) ' moist initialization, mp_physics = ',config_flags%mp_physics
   ELSE
     write(0,*) ' dry initialization, mp_physics = ',config_flags%mp_physics
   END IF

   itf=min0(ite,ide-1)
   jtf=min0(jte,jde-1)
   ktf=min0(kte,kde-1)

   IF ( wrf_on_monitor() ) THEN
     call read_input_jet( u_jet, rho_jet, theta_jet, kde-1, jde-jds )
   ENDIF

   CALL wrf_dm_bcast_bytes (     u_jet , (kde-1) * ((jde-1)-jds+1) * RWORDSIZE )
   CALL wrf_dm_bcast_bytes (   rho_jet , (kde-1) * ((jde-1)-jds+1) * RWORDSIZE )
   CALL wrf_dm_bcast_bytes ( theta_jet , (kde-1) * ((jde-1)-jds+1) * RWORDSIZE )

   ! iteration parameters

   iter = 30
   iterm = 20

   tslvl=300.
   xn2l=.0001

!  i = its  ! balance the first j plane and copy values into the other planes

   DO j = jts,jtf
   DO i = its,itf
     k = kms
     dz(k)=0.5*dzetaw(k)*z_zeta(i,j)
     DO k=kms+1,kme
       dz(k)=0.5*(dzetaw(k)+dzetaw(k-1))*z_zeta(i,j)
     ENDDO

     DO k=1,ktf
       if(z(i,k,j) .gt. 8000.) then
         rel_hum(k)=0.1
       else
         rel_hum(k)=(1.-0.90*(z(i,k,j)/8000.)**1.25)
       end if
       rel_hum(k) = min(0.7,rel_hum(k))
     ENDDO

     DO k=1,ktf
       rthb(k) = rtb(i,k,j)/z_zeta(i,j)*msft(i,j)
       rb(k) = rrb(i,k,j)/z_zeta(i,j)*msft(i,j)
     ENDDO

     IF (config_flags%mp_physics /= 0)  THEN

       DO k=1,ktf
         tmppi=(pb(i,k,j)/p1000mb)**rcp
         temperature(k) = tmppi*theta_jet(k,j)
         if (temperature(k) .gt. svpt0) then
            es  = 1000.*svp1*exp(svp2*(temperature(k)-svpt0)/(temperature(k)-svp3))
            qvs = ep_2*es/(pb(i,k,j)-es)
         else
            es  = 1000.*svp1*exp( 21.8745584*(temperature(k)-273.16)/(temperature(k)-7.66) )
            qvs = ep_2*es/(pb(i,k,j)-es)
         endif
         moist_1(i,k,j,P_QV) = rel_hum(k)*qvs
         moist_2(i,k,j,P_QV) = moist_1(i,k,j,P_QV)
       ENDDO

     ELSE

       DO k=1,ktf
         moist_1(i,k,j,P_QV) = 0.
         moist_2(i,k,j,P_QV) = 0.
       ENDDO
 
     ENDIF

!  compute the perturbation variables here

     DO k=1,ktf

       IF (perturb) then
         pii = 2.*asin(1.0)
       
         xcen=float(i-1)/float(ide-ids)
!        ycen=0.
!        xcen = 0.
         ycen=float(j-jde/2-1)*dy

         zcen=z(i,k,j)-htbub

         rcen=sqrt(                   &
                   +(ycen/radbub)**2  &
                   +(zcen/radz)**2   )
!  add in bubble if point lies inside of bubble radius, set flag to push
!  re-balancing pressure and density hydrostatically

         IF (config_flags%periodic_x)  THEN
            IF (rcen <= 1) THEN
              tp = tpbub*cos(rcen*piov2)*cos(rcen*piov2)*cos(xcen*2*pii+pii)
            else
              tp = 0.
            end if
         ELSE
            xcen = xcen*float(ide-ids)*dx/4000.e+03
            IF (rcen <= 1 .and. xcen <= 1) THEN
              tp = 2*tpbub*cos(rcen*piov2)*cos(rcen*piov2)*sin(xcen*2*pii)
            else
              tp = 0.
            end if
         END IF
       END IF

!      p(k) = p1000mb*(R_d*rho_jet(k,j)*theta_jet(j,k)/p1000mb)**cpovcv
!      pi(k)=(p(k)/p1000mb)**(rcp)
       rp(k)=rho_jet(k,j)-rb(k)
       thp(k) = rho_jet(k,j)*(theta_jet(k,j)+tp)-rthb(k)
       pp(k)=p1000mb*(r_d*rho_jet(k,j)*(theta_jet(k,j)+tp)/p1000mb)**cpovcv &
               -pb(i,k,j)
       theta_v(k) = theta_jet(k,j)+tp
       theta(k) = theta_v(k)/(1.+.61*moist_1(i,k,j,P_QV))
       p(k)=pp(k)+pb(i,k,j)
       pi(k)=(p(k)/p1000mb)**rcp 
       rho(k)=p(k)/R_d/(pi(k)*theta(k)*(1.+rvovrd*moist_1(i,k,j,P_QV)))

     ENDDO

!  re-balance perturbation variables

     if(rebalance) then

       DO k=ktf-1,1,-1

         it = 0
         p_check = 2*p_balance_tolerance

!  loop until converged on hydrostatic balance for moisture

         DO WHILE ( (it <= iter) .and. (p_check > p_balance_tolerance) )

           p_check = pp(k)
           pp(k)=pp(k+1)+0.5*(rp(k)+rp(k+1))*g*dz(k)            &
               +0.5*( rho(k  )*moist_1(i,k  ,j,P_QV)            &
                     +rho(k+1)*moist_1(i,k+1,j,P_QV) )*g*dz(k)

           p(k)=pp(k)+pb(i,k,j)
           pi(k)=(p(k)/p1000mb)**rcp 
           rho(k)=p(k)/R_d/(pi(k)*theta(k)*(1.+rvovrd*moist_1(i,k,j,P_QV)))
           rp(k)=rho(k)-rb(k)
           rhotheta(k)=rho(k)*theta(k)

           IF (config_flags%mp_physics /= 0)  THEN
             temperature(k) = ((pb(i,k,j)+pp(k))/p1000mb)**rcp*theta(k)
             if (temperature(k) .gt. svpt0) then
                es  = 1000.*svp1*exp(svp2*(temperature(k)-svpt0)/(temperature(k)-svp3))
                qvs = ep_2*es/((pb(i,k,j)+pp(k))-es)
             else
                es  = 1000.*svp1*exp( 21.8745584*(temperature(k)-273.16)/(temperature(k)-7.66) )
                qvs = ep_2*es/(pb(i,k,j)-es)
             endif
             moist_1(i,k,j,P_QV) = rel_hum(k)*qvs
             moist_2(i,k,j,P_QV) = moist_1(i,k,j,P_QV)
             theta(k) = theta_v(k)/(1.+.61*moist_1(i,k,j,P_QV))
           ENDIF

           p_check = abs(p_check - pp(k))
           it = it+1

         ENDDO
       ENDDO
     END IF

     TSK(I,J)=cf1*temperature(1)+cf2*temperature(2)+cf3*temperature(3)
     TMN(I,J)=TSK(I,J)-0.5

     DO k=1,ktf
!      rthp(i,k,j)=thp(k)*z_zeta(i,j)/msft(i,j)
       rthp(i,k,j)=(theta(k)*rho(k)-rthb(k))*z_zeta(i,j)/msft(i,j)
       rrp(i,k,j)=rp(k)*z_zeta(i,j)/msft(i,j)
       rr(i,k,j)= rrp(i,k,j) + rrb(i,k,j)
       rthp2(i,k,j)=rthp(i,k,j)
       rrp2(i,k,j)=rrp(i,k,j)
       rr2(i,k,j)= rr(i,k,j)
     ENDDO

   ENDDO
   ENDDO

!  set initial velocity field

   DO i=its,itf
   DO j=jts,jtf
   DO K = 1, ktf

     ru_1(i,k,j) = u_jet(k,j)*rr(i,k,j)
     ru_2(i,k,j) = ru_1(i,k,j)

     rv_1(i,k,j) = 0.
     rv_2(i,k,j) = rv_1(i,k,j)


   ENDDO
   ENDDO
   ENDDO

   IF ( ite .eq. ide ) THEN

     DO j=jts,jtf
     DO K = 1, ktf

       ru_1(ite,k,j) = ru_1(ite-1,k,j)
       ru_2(ite,k,j) = ru_2(ite-1,k,j)

     ENDDO
     ENDDO

   ENDIF

   IF ( jte .eq. jde ) THEN

     DO i=its,itf
     DO K = 1, ktf

       rv_1(i,k,jte) = rv_1(i,k,jte-1)
       rv_2(i,k,jte) = rv_2(i,k,jte-1)

     ENDDO
     ENDDO

   ENDIF

   DO K = 1, ktf
     u_base(k) = 0.
     v_base(k) = 0.
   ENDDO
   

!  convert to rho_theta_m, store off initial moisture

   IF(num3d_moist >= PARAM_FIRST_SCALAR) THEN

     DO i=its,itf
     DO j=jts,jtf
     DO K = 1, ktf

       rthp(i,k,j)=(rtb(i,k,j)+rthp(i,k,j))*  &
                   (1.+rvovrd*moist_1(i,k,j,P_QV)) &
                      - rtb(i,k,j)
       rthp2(i,k,j)=rthp(i,k,j)

     ENDDO
     ENDDO
     ENDDO

     DO k=1, ktf
       qv_base(k) = moist_1(its,k,jts,P_QV)
     ENDDO

   ENDIF

   DO i=its,itf
   DO j=jts,jtf
   DO K = 1, ktf

     chem_1(i,k,j,1) = k
     chem_2(i,k,j,1) = k

   ENDDO
   ENDDO
   ENDDO

   END SUBROUTINE init_state

!-------------------------------------------------------------------
! this is a wrapper for the solver-specific init_domain routines.
! Also dereferences the grid variables and passes them down as arguments.
! This is crucial, since the lower level routines may do message passing
! and this will get fouled up on machines that insist on passing down
! copies of assumed-shape arrays (by passing down as arguments, the 
! data are treated as assumed-size -- ie. f77 -- arrays and the copying
! business is avoided).  Fie on the F90 designers.  Fie and a pox.

   SUBROUTINE init_domain ( grid )

   IMPLICIT NONE

   !  Input data.
   TYPE (domain), POINTER :: grid 
   !  Local data.
   INTEGER                :: dyn_opt 
   INTEGER :: idum1, idum2

#ifdef DEREF_KLUDGE
   INTEGER     :: sm31 , em31 , sm32 , em32 , sm33 , em33
#endif

#ifdef DEREF_KLUDGE
   sm31             = grid%sm31
   em31             = grid%em31
   sm32             = grid%sm32
   em32             = grid%em32
   sm33             = grid%sm33
   em33             = grid%em33
#endif

   CALL get_dyn_opt( dyn_opt )
   
   CALL set_scalar_indices_from_config ( head_grid%id , idum1, idum2 )

   IF (      dyn_opt .eq. 1 &
        .or. dyn_opt .eq. 2 &
        .or. dyn_opt .eq. 3 &
                                       ) THEN
     CALL init_domain_rk( grid, &
!
#include <rk_actual_args.inc>
!
                        )

   ELSE
     WRITE(0,*)' init_domain: unknown or unimplemented dyn_opt = ',dyn_opt
     STOP
   ENDIF

   END SUBROUTINE init_domain

!-------------------------------------------------------------------

   SUBROUTINE init_domain_rk ( grid, &
!
# include <rk_dummy_args.inc>
!
)
   IMPLICIT NONE

   !  Input data.
   TYPE (domain), POINTER :: grid

# include <rk_dummy_arg_defines.inc>

   TYPE (grid_config_rec_type)              :: config_flags

   !  Local data
   INTEGER                             ::                       &
                                  ids, ide, jds, jde, kds, kde, &
                                  ims, ime, jms, jme, kms, kme, &
                                  its, ite, jts, jte, kts, kte, &
                                  i, j, k

   ! Local data for mountain
   INTEGER :: icm,jcm, ii, im1, jj, jm1, loop, error, fid
   REAL    :: xa,hm

   LOGICAL :: moisture_init

#ifdef DM_PARALLEL
#  ifdef RSL
#    include <rsl_rk_data_calls.inc>
#  endif
#endif

   SELECT CASE ( model_data_order )
         CASE ( DATA_ORDER_ZXY )
   kds = grid%sd31 ; kde = grid%ed31 ;
   ids = grid%sd32 ; ide = grid%ed32 ;
   jds = grid%sd33 ; jde = grid%ed33 ;

   kms = grid%sm31 ; kme = grid%em31 ;
   ims = grid%sm32 ; ime = grid%em32 ;
   jms = grid%sm33 ; jme = grid%em33 ;

   kts = grid%sp31 ; kte = grid%ep31 ;   ! note that tile is entire patch
   its = grid%sp32 ; ite = grid%ep32 ;   ! note that tile is entire patch
   jts = grid%sp33 ; jte = grid%ep33 ;   ! note that tile is entire patch
         CASE ( DATA_ORDER_XYZ )
   ids = grid%sd31 ; ide = grid%ed31 ;
   jds = grid%sd32 ; jde = grid%ed32 ;
   kds = grid%sd33 ; kde = grid%ed33 ;

   ims = grid%sm31 ; ime = grid%em31 ;
   jms = grid%sm32 ; jme = grid%em32 ;
   kms = grid%sm33 ; kme = grid%em33 ;

   its = grid%sp31 ; ite = grid%ep31 ;   ! note that tile is entire patch
   jts = grid%sp32 ; jte = grid%ep32 ;   ! note that tile is entire patch
   kts = grid%sp33 ; kte = grid%ep33 ;   ! note that tile is entire patch
         CASE ( DATA_ORDER_XZY )
   ids = grid%sd31 ; ide = grid%ed31 ;
   kds = grid%sd32 ; kde = grid%ed32 ;
   jds = grid%sd33 ; jde = grid%ed33 ;

   ims = grid%sm31 ; ime = grid%em31 ;
   kms = grid%sm32 ; kme = grid%em32 ;
   jms = grid%sm33 ; jme = grid%em33 ;

   its = grid%sp31 ; ite = grid%ep31 ;   ! note that tile is entire patch
   kts = grid%sp32 ; kte = grid%ep32 ;   ! note that tile is entire patch
   jts = grid%sp33 ; jte = grid%ep33 ;   ! note that tile is entire patch

   END SELECT


   CALL model_to_grid_config_rec ( grid%id , model_config_rec , config_flags )

! here we check to see if the boundary conditions are set properly

   call boundary_condition_check( config_flags, bdyzone, error, grid%id )

   moisture_init = .true.

    grid%itimestep=0

   hm = 00.
   xa = 5.

   icm = ide/2
   jcm = jde/2

#ifdef DM_PARALLEL
   call wrf_dm_bcast_bytes( icm , IWORDSIZE )
   call wrf_dm_bcast_bytes( jcm , IWORDSIZE )
#endif

    CALL set_mminlu('    ')
    CALL set_iswater(1,0)
    CALL set_cen_lat(1,40.)

    DO j = jts, jte
      DO i = its, ite
!CIO ht          ij
!CIO msft        ij
!CIO msfu        ij
!CIO msfv        ij
!CIO sina        ij
!CIO cosa        ij
!CIO e           ij
!CIO f           ij
!CIO r           kij
!CIO ru          kij
!CIO rv          kij
!CIO rom         kij
         ht(i,j)       = hm/(1.+(float(i-icm)/xa)**2+(float(j-jcm)/xa)**2)
!         ht(i,j)       = hm/(1.+(float(j-jcm)/xa)**2)
!         ht(i,j)       = hm/(1.+(float(i-icm)/xa)**2)
         msft(i,j)     = 1.
         msfu(i,j)     = 1.
         msfv(i,j)     = 1.
         sina(i,j)     = 0.
         cosa(i,j)     = 1.
         e(i,j)        = 0.
         f(i,j)        = 1.e-04
! values used only in some physics options
         snowc(i,j)    = 0.
         xlat(i,j)     = 40.
         xlong(i,j)    = -105.

         DO k = kts, kte
            r(i,k,j)    = 1.               ! constant
            ru_1(i,k,j) = 0.
            ru_2(i,k,j) = ru_1(i,k,j)
            rv_1(i,k,j) = 0.
            rv_2(i,k,j) = rv_1(i,k,j)
            rom_1(i,k,j) = 0.
            rom_2(i,k,j) = rom_1(i,k,j)
         END DO
      END DO
   END DO

# ifdef DM_PARALLEL

!
!
!                           * * * * *
!         *        * * *    * * * * *
!       * + *      * + *    * * + * *
!         *        * * *    * * * * *
!                           * * * * *

! ru_1                          x
! ru_2                          x
! rv_1                          x
! rv_2                          x
! rom_1                         x
! rom_2                         x
! r                             x
! rtp_1                         x
! rtp_2                         x
! rr_1                          x
! rr_2                          x
! rrp_1                         x
! rrp_2                         x
! rtb                           x
! rrb                           x
! pip                           x
! pib                           x
! z                             x
! zx                            x
! zy                            x

! ht                            x
! msft                          x
! msfu                          x
! msfv                          x
! sina                          x
! cosa                          x
! e                             x
! f                             x
! z_zeta                        x
! zeta_z                        x
! cofwr                         x
! moist_2                       x
! chem_2                        x
! moist_1                       x
! chem_1                        x

      CALL wrf_dm_halo ( grid%domdesc , grid%comms , HALO_RK_INIT )
# endif

   rdx = 1./dx
   rdy = 1./dy
   resm = (1.-epssm)/(1.+epssm)
   dts    = ( dt / float(grid%time_step_sound) )
   dtseps = .5 * dts * ( 1. + epssm )


! Need to read in some stuff

!CIO z        kij
!CIO zx       kij
!CIO zy       kij
!CIO rtp      kij
!CIO z_zeta   ij
!CIO zeta_z   ij
!CIO cofwr    ij
!CIO zeta     k
!CIO zetaw    k
!CIO dzetaw   k
!CIO dzeta    k
!CIO rdzu     k
!CIO rdzw     k
!CIO fzm      k
!CIO fzp      k
!CIO cofrz    k
!CIO ztop     s
!CIO cf1      s
!CIO cf2      s
!CIO cf3      s



   CALL init_coords (    grid,                                     &
                         ht, rdx, rdy, zeta, zetaw, dzetaw, dzeta, &
                         rdzu, rdzw, z, zx, zy, &
                         z_zeta, zeta_z, fzm, fzp, ztop, &
                         cf1, cf2, cf3, &
                         cofrz, cofwr, dtseps, &
                         ids, ide, jds, jde, kds, kde, &
                         ims, ime, jms, jme, kms, kme, &
                         its, ite, jts, jte, kts, kte )

!CIO rrb          kij
!CIO rtb          kij
!CIO pb           kij
!CIO pib          kij
!CIO h_diabatic   kij


CALL start_timing
   CALL init_base_state ( grid,                                &
                          rrb, rtb, pb, pib, pb8w, h_diabatic, &
                          msft, TSK, TMN, dzetaw, z, ht, z_zeta,    &
                          ids, ide, jds, jde, kds, kde,        &
                          ims, ime, jms, jme, kms, kme,        &
                          its, ite, jts, jte, kts, kte        )

CALL end_timing( 'module_initialize: init_base_state' )


! If we read all of the state, don't need this.

!CIO rrb         kij
!CIO rtb         kij
!CIO pb          kij
!CIO rr          kij
!CIO rtp         kij
!CIO rrp         kij
!CIO rthp        kij
!CIO ru          kij
!CIO rv          kij
!CIO u_base      k
!CIO v_base      k
!CIO qv_base     k
!CIO moist       kijf
!CIO chem        kijf


CALL start_timing
   CALL init_state (  rrb, rtb, pb,                   &
                      rr_1, rrp_1, rtp_1,             &
                      rr_2, rrp_2, rtp_2,             &
                      moist_1, moist_2, num_moist,    &
                      chem_1, chem_2, num_chem,       &
                      ru_1, ru_2, rv_1, rv_2,         &
                      u_base, v_base, qv_base,        &
                      msft, dzetaw, z, ht, z_zeta,    &
                      dx, dy, TSK, TMN, cf1,cf2,cf3,  &
                      config_flags,                   &
                      ids, ide, jds, jde, kds, kde,   &
                      ims, ime, jms, jme, kms, kme,   &
                      its, ite, jts, jte, kts, kte   )
CALL end_timing ('module_initialize: init_state')

!
   IF ( .not. input_from_file ) then
       
!   IF (moisture_init) THEN

!      DO j = jts, min(jte,jde-1)
!         DO i = its, min(ite,ide-1)
!            DO k = kts, min(kde-1,kte)
!               loop_3d_mbc   : DO loop = PARAM_FIRST_SCALAR , num_3d_moist
!                  moist_1(i,k,j,loop)=float (100 * loop + k)/rr_1(i,k,j)
!                  moist_2(i,k,j,loop)=float (100 * loop + k)/rr_2(i,k,j)
!               END DO loop_3d_mbc
!            ENDDO
!         ENDDO
!      ENDDO
!   ENDIF

    endif

     RETURN

   END SUBROUTINE init_domain_rk

   

   SUBROUTINE init_module_initialize
   END SUBROUTINE init_module_initialize

!-------------------------------------------------------------------------


 SUBROUTINE read_input_sounding( p_surface, theta, relh, qv, u, v, z, &
                                 npts, npts_in,  rh_in )

 IMPLICIT NONE
 INTEGER, INTENT(IN   ) :: npts
 REAL, DIMENSION(npts), INTENT(  OUT) :: theta, relh, qv, u, v, z
 LOGICAL, INTENT(  OUT) :: rh_in
 REAL, INTENT(  OUT) :: p_surface
 INTEGER, INTENT(  OUT) :: npts_in
 INTEGER :: io_status, i

 OPEN(unit=10, file='input_sounding', form='formatted', status='old' )
 REWIND(10) 
 
 io_status = 0
 npts_in = 0
 read(unit=10,iostat = io_status,fmt=*) p_surface
 p_surface = p_surface*100.
 rh_in = .false.
 relh = 0.
 

 DO WHILE ( io_status == 0 )
  read(unit=10,iostat = io_status,fmt=*) &
    z(npts_in+1),theta(npts_in+1),qv(npts_in+1),u(npts_in+1),v(npts_in+1)
  IF(io_status == 0) npts_in = npts_in+1
  qv(npts_in) = 0.001 * qv(npts_in)
  IF(npts_in+1 > npts) then
    write(6,*) ' more space needed for input sounding '
    stop
  END IF
 ENDDO

 write(6,*) ' input sounding '
 write(6,*) ' surface pressure ',p_surface
 do i=1,npts_in
   write(6,FMT="(1x,i3,5(1x,1pe10.3))") i, z(i),theta(i),u(i),v(i),qv(i)
 enddo

 END SUBROUTINE read_input_sounding

!----------------------------------------------

 REAL FUNCTION zfind( z,var,np,height )

 IMPLICIT NONE
 INTEGER, INTENT(IN) :: np
 REAL, INTENT(IN) :: height
 REAL, DIMENSION(np), INTENT(IN) :: z,var
 LOGICAL :: interp
 REAL :: w1, w2

 INTEGER :: ip

 interp = .false.
 ip = 2

 DO WHILE ( (.not. interp) .and. (ip <= np) )

   IF(   ((z(ip-1) >= height))                         .OR.  &  ! extrapolate down
         ((z(ip-1) <= height) .and. (z(ip) > height))  .OR.  &  ! interp between
         ( ip == np )                                   )  THEN ! extrapolate up
     w2 = (height-z(ip-1))/(z(ip)-z(ip-1))
     w1 = 1.-w2
     zfind = w1*var(ip-1) + w2*var(ip)
     interp = .true.
   END IF
   ip = ip+1

 ENDDO

 END FUNCTION zfind


 SUBROUTINE init_sound1( theta, qv, u, v, z, ktf,  &
                         theta_in, qv_in, u_in, v_in, z_in, npts  )

 IMPLICIT NONE
 INTEGER, INTENT(IN   ) :: ktf, npts
 REAL, DIMENSION(npts), INTENT(IN   ) :: theta_in, qv_in, u_in, v_in, z_in
 REAL, DIMENSION(ktf), INTENT(  OUT) :: theta, qv, u, v
 REAL, DIMENSION(ktf), INTENT(IN   ) :: z

 INTEGER :: k

 do k = 1, ktf

   theta(k) = zfind( z_in, theta_in, npts, z(k) )
   qv(k)    = zfind( z_in, qv_in   , npts, z(k) )
   u(k)     = zfind( z_in, u_in    , npts, z(k) )
   v(k)     = zfind( z_in, v_in    , npts, z(k) )

!  alternatively, a user could put different functions here
!  if other profiles were desired

 enddo

 END SUBROUTINE init_sound1

 SUBROUTINE read_input_jet( u, r, t, nz, ny )
 implicit none

 integer, intent(in) :: nz,ny
 real, dimension(nz,ny), intent(out) :: u,r,t
 integer :: ny_in, nz_in, j,k
 real, dimension(ny,nz) :: field_in

! this code assumes it is called on processor 0 only

   OPEN(unit=10, file='input_jet', form='unformatted', status='old' )
   REWIND(10) 
   read(10) ny_in,nz_in
   if((ny_in /= ny ) .or. (nz_in /= nz)) then
     write(0,*) ' error in input jet dimensions '
     write(0,*) ' ny, ny_input, nz, nz_input ', ny, ny_in, nz,nz_in
     write(0,*) ' error exit '
     stop
   end if
   read(10) field_in
   do j=1,ny
   do k=1,nz
     u(k,j) = field_in(j,k)
   enddo
   enddo
   read(10) field_in
   do j=1,ny
   do k=1,nz
     t(k,j) = field_in(j,k)
   enddo
   enddo

   read(10) field_in
   do j=1,ny
   do k=1,nz
     r(k,j) = field_in(j,k)
   enddo
   enddo

 end subroutine read_input_jet

END MODULE module_initialize
