!WRF:MODEL_LAYER:PHYSICS
!
SUBROUTINE microphysics_driver(th_phy, moist_new, moist_old, w,       &
                               rho, pi_phy, p_phy, RAINNC,            &
                               z, ht, dz8w, dt, config_flags, n_moist,&
                               z_zeta, dzetaw,                        &
                               ids,ide, jds,jde, kds,kde, 	      & 
                               ims,ime, jms,jme, kms,kme, 	      &
                               its,ite, jts,jte, kts,kte  	      )
!----------------------------------------------------------------------
   USE module_bc
   USE module_state_description
   USE module_model_constants
   USE module_wrf_error

! *** add new modules of schemes here

   USE module_mp_kessler
   USE module_mp_lin
   USE module_mp_ncloud3
    
!----------------------------------------------------------------------
   ! This driver calls subroutines for the microphys.
   !
   ! Schemes
   !
   ! 1. Kessler scheme
   ! 2. Lin et al. (1983), Ruttledge and Hibbs (1984)
   ! 
!----------------------------------------------------------------------
   IMPLICIT NONE
!======================================================================
! Grid structure in physics part of WRF
!----------------------------------------------------------------------  
! The horizontal velocities used in the physics are unstaggered
! relative to temperature/moisture variables. All predicted
! variables are carried at half levels except w, which is at full
! levels. Some arrays with names (*8w) are at w (full) levels.
!
!----------------------------------------------------------------------  
! In WRF, kms (smallest number) is the bottom level and kme (largest 
! number) is the top level.  In your scheme, if 1 is at the top level, 
! then you have to reverse the order in the k direction.
!                 
!         kme      -   half level (no data at this level)
!         kme    ----- full level
!         kme-1    -   half level
!         kme-1  ----- full level
!         .
!         .
!         .
!         kms+2    -   half level
!         kms+2  ----- full level
!         kms+1    -   half level
!         kms+1  ----- full level
!         kms      -   half level
!         kms    ----- full level
!
!======================================================================
! Definitions
!-----------
! Rho_d      dry density (kg/m^3)
! Theta_m    moist potential temperature (K)
! Qv         water vapor mixing ratio (kg/kg)
! Qc         cloud water mixing ratio (kg/kg)
! Qr         rain water mixing ratio (kg/kg)
! Qi         cloud ice mixing ratio (kg/kg)
! Qs         snow mixing ratio (kg/kg)
!----------------------------------------------------------------------
!-- th_phy	  potential temperature    (K)
!-- moist_new     updated moisture array   (kg/kg)
!-- moist_old     Old moisture array       (kg/kg)
!-- rho           density of air           (kg/m^3)
!-- pi		  exner function           (dimensionless)
!-- p		  pressure                 (Pa)
!-- RAINNC	  grid scale precipitation (mm)
!-- z		  Height above sea level   (m)
!-- dt		  Time step 		 (s)
!-- config_flags  flag for configuration      ! change ---  ?????   
!-- n_moist  	  number of water substances   (integer)
!-- G		  acceleration due to gravity  (m/s^2)
!-- CP		  heat capacity at constant pressure for dry air (J/kg/K)
!-- R_d		  gas constant for dry air (J/kg/K)
!-- R_v		  gas constant for water vapor (J/kg/K)
!-- XLS		  latent heat of sublimation   (J/kg)
!-- XLV		  latent heat of vaporization  (J/kg)
!-- XLF		  latent heat of melting       (J/kg)
!-- rhowater 	  water density 		     (kg/m^3)
!-- rhosnow 	  snow density 		     (kg/m^3)
!-- P_QV          species index for water vapor
!-- P_QC          species index for cloud water
!-- P_QR          species index for rain water
!-- P_QI          species index for cloud ice
!-- P_QS          species index for snow
!-- P_QG          species index for graupel
!-- ids           start index for i in domain
!-- ide           end index for i in domain
!-- jds           start index for j in domain
!-- jde           end index for j in domain
!-- kds           start index for k in domain
!-- kde           end index for k in domain
!-- ims           start index for i in memory
!-- ime           end index for i in memory
!-- jms           start index for j in memory
!-- jme           end index for j in memory
!-- kms           start index for k in memory
!-- kme           end index for k in memory
!-- its           start index for i in tile
!-- ite           end index for i in tile
!-- jts           start index for j in tile
!-- jte           end index for j in tile
!-- kts           start index for k in tile
!-- kte           end index for k in tile
!
!======================================================================

   TYPE(grid_config_rec_type),    INTENT(IN   )    :: config_flags
!
   INTEGER,      INTENT(IN   )    ::       ids,ide, jds,jde, kds,kde
   INTEGER,      INTENT(IN   )    ::       ims,ime, jms,jme, kms,kme
   INTEGER,      INTENT(IN   )    ::       its,ite, jts,jte, kts,kte
   INTEGER,      INTENT(IN   )    ::                         n_moist
!
   REAL, DIMENSION( ims:ime , kms:kme , jms:jme ),                    &
         INTENT(INOUT) ::                                     th_phy
!
   REAL, DIMENSION( ims:ime , kms:kme , jms:jme, n_moist ),           &
         INTENT(INOUT) ::                                  moist_new

   REAL, DIMENSION( ims:ime , kms:kme , jms:jme ),                    &
         INTENT(IN   ) ::                                          z, &
                                                                 rho, &
                                                                dz8w, &
                                                                   w, &
                                                              pi_phy, &
                                                               p_phy
!
   REAL, DIMENSION( ims:ime , kms:kme , jms:jme, n_moist ),           &
         INTENT(IN   ) ::                                  moist_old
!
   REAL, DIMENSION( ims:ime , jms:jme ),  INTENT(INOUT) ::    RAINNC
!

   REAL , DIMENSION( ims:ime , jms:jme ) , INTENT(IN)   :: ht

   REAL , DIMENSION( ims:ime , jms:jme ) ,  INTENT(IN ) :: z_zeta
   REAL , DIMENSION( kms:kme ) ,            INTENT(IN ) :: dzetaw

   REAL, INTENT(IN   ) :: dt
 
! LOCAL  VAR

   INTEGER :: i,j,k

!---------------------------------------------------------------------
!  check for microphysics type.  We need a clean way to 
!  specify these things!
!---------------------------------------------------------------------

   if (config_flags%mp_physics .eq. 0) return

!-----------
   IF ( n_moist >= PARAM_FIRST_SCALAR ) THEN

     micro_select: SELECT CASE(config_flags%mp_physics)

        CASE (KESSLERSCHEME)
             CALL wrf_debug ( 200 , 'microphysics_driver: calling kessler' )
             CALL kessler( th_phy,                       &
                     moist_new(ims,kms,jms,P_QV),        &
                     moist_new(ims,kms,jms,P_QC),        &
                     moist_old(ims,kms,jms,P_QC),        &
                     moist_new(ims,kms,jms,P_QR),        &
                     moist_old(ims,kms,jms,P_QR),        &
                     rho, pi_phy, RAINNC, dt, z, cp,     &
                     EP_2,SVP1,SVP2,SVP3,SVPT0,rhowater, &
                     dzetaw,                             &
                     ids,ide, jds,jde, kds,kde,          &
                     ims,ime, jms,jme, kms,kme,          &
                     its,ite, jts,jte, kts,kte           )

        CASE (LINSCHEME)
             CALL wrf_debug ( 200 , 'microphysics_driver: calling lin_et_al' )
             CALL lin_et_al( th_phy,                     &
                     moist_new(ims,kms,jms,P_QV),        &
                     moist_new(ims,kms,jms,P_QC),        &
                     moist_new(ims,kms,jms,P_QR),        &
                     moist_new(ims,kms,jms,P_QI),        &
                     moist_new(ims,kms,jms,P_QS),        &
                     moist_new(ims,kms,jms,P_QG),        &
                     moist_old(ims,kms,jms,P_QR),        &
                     moist_old(ims,kms,jms,P_QS),        &
                     moist_old(ims,kms,jms,P_QG),        &
                     rho, pi_phy, p_phy, RAINNC, dt, z,  &
                     ht, z_zeta, dzetaw, G, cp, R_d, R_v,&
                     XLS, XLV, XLF, rhowater, rhosnow,   &
                     EP_2,SVP1,SVP2,SVP3,SVPT0,          &
                     P_QI, P_QS, P_QG,                   &
                     PARAM_FIRST_SCALAR,                 &
                     ids,ide, jds,jde, kds,kde,          &
                     ims,ime, jms,jme, kms,kme,          & 
                     its,ite, jts,jte, kts,kte           )

        CASE (NCEPCLOUD3)
             CALL wrf_debug ( 200 , 'microphysics_driver: calling ncloud3' )
             CALL  ncloud3(th_phy,                       &
                     moist_new(ims,kms,jms,P_QV),        &
                     moist_new(ims,kms,jms,P_QC),        &
                     moist_new(ims,kms,jms,P_QR),        &
                     w, rho, pi_phy, p_phy, dz8w, RAINNC,&
                     dt,g, cp, cv, r_d, r_v, SVPT0,      &
                     ep_1, ep_2, epsilon,                &
                     XLS, XLV, XLF, rhoair0, rhowater,   &
                     cliq,cice,psat,                     &
                     ids,ide, jds,jde, kds,kde,          &
                     ims,ime, jms,jme, kms,kme,          &
                     its,ite, jts,jte, kts,kte           )

      CASE DEFAULT 

         WRITE( wrf_err_message , * ) 'microphysics_driver: no microphysics for n_moist = ',n_moist
         CALL wrf_error_fatal ( wrf_err_message )

      END SELECT micro_select 

   ELSE

      WRITE( wrf_err_message , * ) 'microphysics_driver: in microphysics for n_moist = ',n_moist
      CALL wrf_error_fatal ( wrf_err_message )

   ENDIF

   CALL wrf_debug ( 200 , 'microphysics_driver: returning from' )

   RETURN

   END SUBROUTINE microphysics_driver

