!WRF:MODEL_LAYER:DYNAMICS
!
MODULE module_big_step_utilities

   USE module_domain
   USE module_model_constants
   USE module_state_description
   USE module_configure

CONTAINS

!-------------------------------------------------------------------------------

SUBROUTINE buoyancy ( rrp, rrb, moist, rw_tend,     &
                      config_flags, n_moist,            &
                      ids, ide, jds, jde, kds, kde, &
                      ims, ime, jms, jme, kms, kme, &
                      its, ite, jts, jte, kts, kte )

   IMPLICIT NONE
   
   ! Input data

   TYPE(grid_config_rec_type) ,           INTENT(IN   ) :: config_flags   

   INTEGER ,          INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                       ims, ime, jms, jme, kms, kme, &
                                       its, ite, jts, jte, kts, kte 

   INTEGER ,          INTENT(IN   ) :: n_moist
   
   REAL, DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(INOUT) :: rw_tend

   REAL, DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: rrp, rrb

   REAL, DIMENSION( ims:ime, kms:kme , jms:jme , n_moist ), INTENT(IN   ) :: moist
                                              
   ! Local stuff

   REAL :: total_water_k, total_water_km1, dry_buoyancy, moist_correction
   
   INTEGER :: i, j, k, itf, jtf, ktf, ispe
   
   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)
   ktf=MIN(kte,kde-1)

! dry density contribution

   DO j=jts,jtf
   DO k=kts+1,ktf
   DO i=its,itf
         dry_buoyancy = - 0.5*g*(rrp(i,k,j)+rrp(i,k-1,j))
         rw_tend(i,k,j)=rw_tend(i,k,j) + dry_buoyancy
   ENDDO
   ENDDO
   ENDDO

! moist contribution  NOTE: need a better way to distinguish which
!  microphysics we're using (for example, we can't just sum all
!  the moisture variables, because some of them may be number concentrations -
!  e.g., double moment schemes)

   IF( n_moist >= PARAM_FIRST_SCALAR ) THEN

       DO j=jts,jtf
       DO k=kts+1,ktf
       DO i=its,itf
          total_water_k = 0.
          total_water_km1 = 0.
          DO ispe=PARAM_FIRST_SCALAR,n_moist
             total_water_k   = total_water_k   + moist(i,k,  j,ispe)
             total_water_km1 = total_water_km1 + moist(i,k-1,j,ispe)
          ENDDO
!         total_water_k   =   moist(i,k  ,j,1) &
!                           + moist(i,k  ,j,2) &
!                           + moist(i,k  ,j,3)
!         total_water_km1 =   moist(i,k-1,j,1) &
!                           + moist(i,k-1,j,2) &
!                           + moist(i,k-1,j,3)
          moist_correction =              - 0.5*g*(                    &
                   rrb(i,k  ,j)*total_water_k  /(1+total_water_k  )    &
                  +rrb(i,k-1,j)*total_water_km1/(1+total_water_km1)  )
          rw_tend(i,k,j) = rw_tend(i,k,j) + moist_correction
       ENDDO
       ENDDO
       ENDDO

   ENDIF

END SUBROUTINE buoyancy

!-------------------------------------------------------------------------------

SUBROUTINE calculate_full ( rfield, rfieldb, rfieldp,     &
                            ids, ide, jds, jde, kds, kde, &
                            ims, ime, jms, jme, kms, kme, &
                            its, ite, jts, jte, kts, kte )

   IMPLICIT NONE
   
   ! Input data
   
   INTEGER ,      INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                   ims, ime, jms, jme, kms, kme, &
                                   its, ite, jts, jte, kts, kte 
   
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: rfieldb, &
                                                                      rfieldp

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(OUT  ) :: rfield
   
   ! Local indices.
   
   INTEGER :: i, j, k, itf, jtf, ktf
   
   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)
   ktf=MIN(kte,kde-1)

   DO j=jts,jtf
   DO k=kts,ktf
   DO i=its,itf
      rfield(i,k,j)=rfieldb(i,k,j)+rfieldp(i,k,j)
   ENDDO
   ENDDO
   ENDDO

END SUBROUTINE calculate_full

!-------------------------------------------------------------------------------

SUBROUTINE calculate_pp ( rtb, rth, pb, pp, pib, pip,   &
                          msft, zeta_z,                 &
                          ids, ide, jds, jde, kds, kde, &
                          ims, ime, jms, jme, kms, kme, &
                          its, ite, jts, jte, kts, kte )

   IMPLICIT NONE
   
   ! Input data

   INTEGER ,    INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                 ims, ime, jms, jme, kms, kme, &
                                 its, ite, jts, jte, kts, kte
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: rtb, &
                                                                rth, &
                                                                pb,  &
                                                                pib

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(OUT  ) :: pp, &
                                                                pip

   REAL , DIMENSION( ims:ime , jms:jme ) ,         INTENT(IN   ) :: msft, &
                                                                zeta_z
   
   ! Local indices.
   
   INTEGER :: i, j, k, itf, jtf, ktf, kk
   REAL :: rhotheta, rhothetab, p, pi, pbase

     
   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)
   ktf=MIN(kte,kde-1)

     DO j=jts,jtf
     DO k=kts,ktf
     DO i=its,itf

           rhotheta =msft(i,j)*zeta_z(i,j)*rth(i,k,j)
           rhothetab=msft(i,j)*zeta_z(i,j)*rtb(i,k,j)
   ! it may also be usful to output p, total pressure, for physics
           p=p1000mb*(r_d*rhotheta/p1000mb)**cpovcv
           pi=p/(r_d*rhotheta)
           pbase=p1000mb*(r_d*rhothetab/p1000mb)**cpovcv
           pp(i,k,j)=p-pbase
           pip(i,k,j)=pi-pib(i,k,j)
     ENDDO
     ENDDO
     ENDDO

END SUBROUTINE calculate_pp

!-------------------------------------------------------------------------------


SUBROUTINE calculate_pim ( pip, pib, moist, n_moist,   &
                          ids, ide, jds, jde, kds, kde, &
                          ims, ime, jms, jme, kms, kme, &
                          its, ite, jts, jte, kts, kte )

   IMPLICIT NONE
   
   ! Input data

   INTEGER ,    INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                 ims, ime, jms, jme, kms, kme, &
                                 its, ite, jts, jte, kts, kte
   INTEGER ,    INTENT(IN   ) :: n_moist

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: pib

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(INOUT) :: pip

   REAL , DIMENSION( ims:ime, kms:kme , jms:jme , n_moist ), INTENT(IN   ) :: moist

   ! Local indices.
   
   INTEGER :: i, j, k, itf, jtf, ktf, kk, ispe

   LOGICAL :: moist_simulation
!  INTEGER :: microphysics_type

   REAL :: total_water,  rhod_over_rhom

     
   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)
   ktf=MIN(kte,kde-1)

   IF( n_moist >= PARAM_FIRST_SCALAR ) THEN

     moist_simulation = .true.

   ELSE ! dry simulation

     moist_simulation = .false.

   ENDIF


     DO j=jts,jtf
     DO k=kts,ktf
     DO i=its,itf

!         IF( microphysics_type == 1 ) THEN
          IF( moist_simulation ) THEN
             total_water =  0.
             DO ispe=PARAM_FIRST_SCALAR,n_moist
                total_water =  total_water + moist(i,k,j,ispe)
             ENDDO

!            total_water =  moist(i,k,j,1)+moist(i,k,j,2)+moist(i,k,j,3)
             rhod_over_rhom = 1./(1.+total_water)
          ELSE
             rhod_over_rhom = 1.  
          ENDIF

          pip(i,k,j)=rhod_over_rhom*(pip(i,k,j)+pib(i,k,j))

     ENDDO
     ENDDO
     ENDDO

END SUBROUTINE calculate_pim

!-------------------------------------------------------------------------------
SUBROUTINE calculate_p ( rth, pi, p,                   &
                         msft, zeta_z,                 &
                         ids, ide, jds, jde, kds, kde, &
                         ims, ime, jms, jme, kms, kme, &
                         its, ite, jts, jte, kts, kte )

   IMPLICIT NONE
   
   ! Input data

   INTEGER ,    INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                 ims, ime, jms, jme, kms, kme, &
                                 its, ite, jts, jte, kts, kte
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: rth
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(OUT  ) :: p, pi
   REAL , DIMENSION( ims:ime , jms:jme ) ,           INTENT(IN   ) :: msft, &
                                                                      zeta_z
   
   ! Local indices.
   
   INTEGER :: i, j, k, itf, jtf, ktf, kk
   REAL :: rhotheta, rhothetab

     
   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)
   ktf=MIN(kte,kde-1)

     DO j=jts,jtf
     DO k=kts,ktf
     DO i=its,itf
           rhotheta =msft(i,j)*zeta_z(i,j)*rth(i,k,j)
           p(i,k,j)=p1000mb*(r_d*rhotheta/p1000mb)**cpovcv
           pi(i,k,j)=p(i,k,j)/(r_d*rhotheta)

     ENDDO
     ENDDO
     ENDDO

END SUBROUTINE calculate_p


!-------------------------------------------------------------------
SUBROUTINE calculate_p8w ( p8w, P3D,Pb3D,Pwb3D,zeta,zetaw,dzetaw, &
                           ids, ide, jds, jde, kds, kde,          &
                           ims, ime, jms, jme, kms, kme,          &
                           its, ite, jts, jte, kts, kte           )

   IMPLICIT NONE
  
   ! Input data

   INTEGER ,    INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                 ims, ime, jms, jme, kms, kme, &
                                 its, ite, jts, jte, kts, kte

   REAL, DIMENSION( ims:ime, kms:kme, jms:jme )                 , &
         INTENT(IN   ) ::                                    P3D, &
                                                            Pb3D, &
                                                           Pwb3D

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(OUT  ) :: p8w

   REAL , DIMENSION( kms: ) , INTENT(IN)  ::                zeta, &
                                                           zetaw, &
                                                          dzetaw

   INTEGER :: i, j, k, itf, jtf, ktf
   
   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)
   ktf=MIN(kte,kde-1)

   DO j=jts,jtf
      DO K=2,ktf
         DO i=its,itf
            p8w(I,K,J) = Pwb3D(I,K,J)+((P3D(I,K-1,J)-Pb3D(I,K-1,J))*  &
                                    dzetaw(K  )+ &
                                    (P3D(I,K,J)-Pb3D(I,K,J))*         &
                                    dzetaw(K-1))/(dzetaw(K-1)+dzetaw(K))
         ENDDO
      ENDDO
   ENDDO

   DO j=jts,jtf
      DO i=its,itf
         p8w(I,1,J)=Pwb3D(I,1,J) + P3D(I,1,J)-Pb3D(I,1,J)+ &
                   ((P3D(I,1,J)-Pb3D(I,1,J)-P3D(I,2,J)+Pb3D(I,2,J))/ &
                    (zeta(1)-zeta(2))*(zetaw(1)-zeta(1)))

         p8w(I,ktf+1,J)=Pwb3D(I,ktf+1,J)+P3D(I,ktf,J)-Pb3D(I,ktf,J)+ &
                   ((P3D(I,ktf-1,J)-Pb3D(I,ktf-1,J)-  &
                     P3D(I,ktf,J)+Pb3D(I,ktf,J))/ &
                   (zeta(ktf-1)-zeta(ktf))*  &
                    (zetaw(ktf+1)-zeta(ktf)))
      ENDDO
   ENDDO

END SUBROUTINE calculate_p8w

!--------------------------------------------------------------------
SUBROUTINE calculate_temp(t_phy, pi, theta,                       &
                           ids, ide, jds, jde, kds, kde,          &
                           ims, ime, jms, jme, kms, kme,          &
                           its, ite, jts, jte, kts, kte           )

   IMPLICIT NONE

   ! Input data

   INTEGER ,    INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                 ims, ime, jms, jme, kms, kme, &
                                 its, ite, jts, jte, kts, kte

   REAL, DIMENSION( ims:ime, kms:kme, jms:jme )                 , &
         INTENT(IN   ) ::                                     pi, &
                                                           theta

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(OUT  )&
                                                         ::t_phy


   INTEGER :: i, j, k, itf, jtf, ktf

   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)
   ktf=MIN(kte,kde-1)

   DO j=jts,jtf
      DO K=kts,ktf
         DO i=its,itf
            t_phy(I,K,J) = theta(I,K,J)*pi(I,K,J)
         ENDDO
      ENDDO
   ENDDO

END SUBROUTINE calculate_temp

!-------------------------------------------------------------------
SUBROUTINE calculate_t8w ( t8w, t_phy, TSK, zeta,zetaw,dzetaw,    &
                           ids, ide, jds, jde, kds, kde,          &
                           ims, ime, jms, jme, kms, kme,          &
                           its, ite, jts, jte, kts, kte           )

   IMPLICIT NONE

   ! Input data

   INTEGER ,    INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                 ims, ime, jms, jme, kms, kme, &
                                 its, ite, jts, jte, kts, kte

   REAL, DIMENSION( ims:ime, jms:jme ), INTENT(IN   ) ::     TSK

   REAL, DIMENSION( ims:ime, kms:kme, jms:jme )                 , &
         INTENT(IN   ) ::                                   t_phy

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(OUT  ) :: t8w

   REAL , DIMENSION( kms: ) , INTENT(IN)  ::                zeta, &
                                                           zetaw, &
                                                          dzetaw

   INTEGER :: i, j, k, itf, jtf, ktf

   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)
   ktf=MIN(kte,kde-1)

   DO j=jts,jtf
      DO k=2,ktf
         DO i=its,itf
            t8w(I,K,J) = (t_phy(I,K-1,J)*dzetaw(K)+t_phy(I,K,J)*dzetaw(K-1))/ &
                         (dzetaw(K-1)+dzetaw(K))
         ENDDO
      ENDDO
   ENDDO
!
   DO j=jts,jtf
         DO i=its,itf
         t8w(I,ktf+1,J)=t_phy(I,ktf,J)+ (t_phy(I,ktf,J)-t_phy(I,ktf-1,J))/ &
                  (zeta(ktf)-zeta(ktf-1))*(zetaw(ktf+1)-zeta(ktf))
         t8w(I,1,J)=TSK(I,J)
      ENDDO
   ENDDO

END SUBROUTINE calculate_t8w

!-----------------------------------------------------------------
SUBROUTINE calculate_dz8w ( dz8w, zeta_z, dzetaw,         &
                             ids, ide, jds, jde, kds, kde, &
                             ims, ime, jms, jme, kms, kme, &
                             its, ite, jts, jte, kts, kte )

   IMPLICIT NONE

   ! Input data

   INTEGER ,    INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                 ims, ime, jms, jme, kms, kme, &
                                 its, ite, jts, jte, kts, kte
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(OUT  ) :: dz8w
   REAL , DIMENSION( ims:ime , jms:jme ) ,           INTENT(IN   ) :: zeta_z
   REAL , DIMENSION( kms:kme ) ,                     INTENT(IN   ) :: dzetaw

   ! Local indices.

   INTEGER :: i, j, k, itf, jtf, ktf

   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)
   ktf=MIN(kte,kde-1)

   DO j=jts,jtf
     DO k=kts,ktf
        DO i=its,itf
           dz8w(i,k,j)=dzetaw(K)/zeta_z(I,J)
        ENDDO
     ENDDO
   ENDDO

END SUBROUTINE calculate_dz8w

!--------------------------------------------------------
SUBROUTINE c2agrid_u ( uatc, uata,                    &
                       ids, ide, jds, jde, kds, kde,  &
                       ims, ime, jms, jme, kms, kme,  &
                       its, ite, jts, jte, kts, kte   )

   IMPLICIT NONE

   ! Input data

   INTEGER ,    INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                 ims, ime, jms, jme, kms, kme, &
                                 its, ite, jts, jte, kts, kte
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: uatc
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(INOUT) :: uata

   ! Local indices.

   INTEGER :: i, j, k, itf, jtf, ktf

   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)
   ktf=MIN(kte,kde-1)

   DO j=jts,jtf
   DO k=kts,ktf
   DO i=its,itf
      uata(i,k,j)=0.5*(uatc(i,k,j)+uatc(i+1,k,j))
   ENDDO
   ENDDO
   ENDDO

END SUBROUTINE c2agrid_u

!-------------------------------------------------------
SUBROUTINE c2agrid_v ( vatc, vata,                    &
                       ids, ide, jds, jde, kds, kde,  &
                       ims, ime, jms, jme, kms, kme,  &
                       its, ite, jts, jte, kts, kte   )

   IMPLICIT NONE

   ! Input data

   INTEGER ,    INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                 ims, ime, jms, jme, kms, kme, &
                                 its, ite, jts, jte, kts, kte
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: vatc
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(INOUT) :: vata

   ! Local indices.

   INTEGER :: i, j, k, itf, jtf, ktf

   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)
   ktf=MIN(kte,kde-1)

   DO j=jts,jtf
   DO k=kts,ktf
   DO i=its,itf
      vata(i,k,j)=0.5*(vatc(i,k,j)+vatc(i,k,j+1))
   ENDDO
   ENDDO
   ENDDO

END SUBROUTINE c2agrid_v
!---------------------------------------------------------
SUBROUTINE c2agrid_w ( watc, wata,                    &
                       ids, ide, jds, jde, kds, kde,  &
                       ims, ime, jms, jme, kms, kme,  &
                       its, ite, jts, jte, kts, kte   )

   IMPLICIT NONE

   ! Input data

   INTEGER ,    INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                 ims, ime, jms, jme, kms, kme, &
                                 its, ite, jts, jte, kts, kte
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: watc
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(INOUT) :: wata

   ! Local indices.

   INTEGER :: i, j, k, itf, jtf, ktf

   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)
   ktf=MIN(kte,kde-1)

   DO j=jts,jtf
   DO k=kts,ktf
   DO i=its,itf
      wata(i,k,j)=0.5*(watc(i,k,j)+watc(i,k+1,j))
   ENDDO
   ENDDO
   ENDDO

END SUBROUTINE c2agrid_w

!-----------------------------------------------------------
SUBROUTINE calculate_rb ( RHOBASE, rr, msft, zeta_z,    &
                          ids, ide, jds, jde, kds, kde, &
                          ims, ime, jms, jme, kms, kme, &
                          its, ite, jts, jte, kts, kte )

   IMPLICIT NONE

   ! Input data

   INTEGER ,    INTENT(IN   ) :: ids, ide, jds, jde, kds, kde
   INTEGER ,    INTENT(IN   ) :: ims, ime, jms, jme, kms, kme
   INTEGER ,    INTENT(IN   ) :: its, ite, jts, jte, kts, kte

   REAL , DIMENSION(ims:ime,kms:kme,jms:jme) , INTENT(OUT) :: RHOBASE
   REAL , DIMENSION(ims:ime,kms:kme,jms:jme) , INTENT(IN ) :: rr
   REAL , DIMENSION(ims:ime,jms:jme),          INTENT(IN ) :: zeta_z
   REAL , DIMENSION(ims:ime,jms:jme),          INTENT(IN ) :: msft

   ! Local indices.

   INTEGER :: i, j, k, itf, jtf, ktf

   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)
   ktf=MIN(kte,kde-1)

   DO j=jts,jtf
   DO k=kts,ktf
      DO i=its,itf
         RHOBASE(i,k,j)= rr(i,k,j)*msft(i,j)*zeta_z(i,j)
      ENDDO
   ENDDO
   ENDDO

END SUBROUTINE calculate_rb

!------------------------------------------------------------------

SUBROUTINE calculate_rw_w (rr, u, v, rom, rw, w,                 &
                                 cf1, cf2, cf3,                  &
                                 msft, zx, zy, z_zeta, fzm, fzp, &
                                 ids, ide, jds, jde, kds, kde,   &
                                 ims, ime, jms, jme, kms, kme,   &
                                 its, ite, jts, jte, kts, kte   )

   IMPLICIT NONE
   
   ! Input data
   
   INTEGER ,       INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                    ims, ime, jms, jme, kms, kme, &
                                    its, ite, jts, jte, kts, kte 
   
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) ,                 &
                                               INTENT(IN   ) :: u,   &
                                                                v,   &
                                                                rom, &
                                                                rr
   
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(OUT  ) :: rw, w
   
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: zx,  &
                                                                      zy
   
   REAL , DIMENSION( ims:ime , jms:jme ) ,         INTENT(IN   ) :: msft, &
                                                                    z_zeta
   
   REAL , DIMENSION( kms:kme ) ,                 INTENT(IN   ) :: fzm, &
                                                                  fzp

   REAL , INTENT(IN)   :: cf1, cf2, cf3
   REAL                :: rho_surface

   ! Local data
   
   INTEGER :: i, j, k, itf, jtf, ktf

   LOGICAL :: coupled

   coupled = .true.
   
   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)
   ktf=MIN(kte,kde-1)


   IF (.not. coupled) then

     DO j=jts,jtf

     DO k=kts+1,ktf
     DO i=its,itf

         w(i,k,j) = rom(i,k,j)*z_zeta(i,j)/                    &
                  (fzm(k)*rr(i,k,j)+fzp(k)*rr(i,k-1,j))        &
                            +0.5*msft(i,j)*(                   &
           zx(i,k,j  )*(fzm(k)*u(i  ,k,j)+fzp(k)*u(i  ,k-1,j)) &
          +zx(i+1,k,j)*(fzm(k)*u(i+1,k,j)+fzp(k)*u(i+1,k-1,j)) &
          +zy(i,k,j  )*(fzm(k)*v(i,k,j  )+fzp(k)*v(i,k-1,j  )) &
          +zy(i,k,j+1)*(fzm(k)*v(i,k,j+1)+fzp(k)*v(i,k-1,j+1)) )

          rw(i,k,j)=w(i,k,j)*(fzm(k)*rr(i,k,j)+fzp(k)*rr(i,k-1,j))

     ENDDO
     ENDDO

!
!  set lower b.c if necessary
!

     IF (kts == 1) THEN

     DO i=its,itf

       w(i,1,j) =                                                          &
         0.5*zx(i  ,1,j)*(cf1*u(i  ,1,j)+cf2*u(i  ,2,j)+cf3*u(i  ,3,j)) +  &
         0.5*zx(i+1,1,j)*(cf1*u(i+1,1,j)+cf2*u(i+1,2,j)+cf3*u(i+1,3,j)) +  &
         0.5*zy(i,1,j  )*(cf1*v(i,1,j  )+cf2*v(i,2,j  )+cf3*v(i,3,j  )) +  &
         0.5*zy(i,1,j+1)*(cf1*v(i,1,j+1)+cf2*v(i,2,j+1)+cf3*v(i,3,j+1))

       rho_surface = cf1*rr(i,1,j) + cf2*rr(i,2,j) + cf3*rr(i,3,j)
       rw(i,1,j) = w(i,1,j)*rho_surface

     ENDDO

     END IF

     ENDDO

   ELSE

!
!  NOTE :  here u and v are the coupled variables ru and rv !!!
!
     DO j=jts,jtf

     DO k=kts+1,ktf
     DO i=its,itf

         rw(i,k,j) = rom(i,k,j)*z_zeta(i,j)                      &
                            +0.5*msft(i,j)*(                     &
           zx(i  ,k,j)*(fzm(k)*u(i  ,k,j)+fzp(k)*u(i  ,k-1,j)) &
          +zx(i+1,k,j)*(fzm(k)*u(i+1,k,j)+fzp(k)*u(i+1,k-1,j)) &
          +zy(i,k,j  )*(fzm(k)*v(i,k,j  )+fzp(k)*v(i,k-1,j  )) &
          +zy(i,k,j+1)*(fzm(k)*v(i,k,j+1)+fzp(k)*v(i,k-1,j+1)) )

         w(i,k,j) = rw(i,k,j)/(fzm(k)*rr(i,k,j)+fzp(k)*rr(i,k-1,j))  

     ENDDO
     ENDDO

     IF (kts == 1) THEN

     DO i=its,itf

       rw(i,1,j) =                                                         &
         0.5*zx(i  ,1,j)*(cf1*u(i  ,1,j)+cf2*u(i  ,2,j)+cf3*u(i  ,3,j)) +  &
         0.5*zx(i+1,1,j)*(cf1*u(i+1,1,j)+cf2*u(i+1,2,j)+cf3*u(i+1,3,j)) +  &
         0.5*zy(i,1,j  )*(cf1*v(i,1,j  )+cf2*v(i,2,j  )+cf3*v(i,3,j  )) +  &
         0.5*zy(i,1,j+1)*(cf1*v(i,1,j+1)+cf2*v(i,2,j+1)+cf3*v(i,3,j+1))

       rho_surface = cf1*rr(i,1,j) + cf2*rr(i,2,j) + cf3*rr(i,3,j)
       w(i,1,j) = rw(i,1,j)/rho_surface

     ENDDO

     END IF

     ENDDO

   END IF

!
!  upper bc
!

   IF (kte == kde) THEN

   DO j=jts,jtf
   DO i=its,itf

      w(i,kte,j) = 0.
     rw(i,kte,j) = 0.

   ENDDO
   ENDDO

   END IF

END SUBROUTINE calculate_rw_w

!-------------------------------------------------------------------------------

SUBROUTINE diabatic_heating  ( h_diabatic, rtp_new, dt,          &
                               ids, ide, jds, jde, kds, kde,     &
                               ims, ime, jms, jme, kms, kme,     &
                               its, ite, jts, jte, kts, kte     )

   IMPLICIT NONE
   
   ! Input data
   
   INTEGER ,      INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                   ims, ime, jms, jme, kms, kme, &
                                   its, ite, jts, jte, kts, kte 

   REAL,          INTENT(IN   ) :: dt
   
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: rtp_new

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(INOUT) :: h_diabatic
   
   ! Local indices.
   
   INTEGER :: i, j, k, itf, jtf, ktf
   
! store off the diabatic heating rate.
! h_diababtic shoudl have the value of rho*theta' after the dry step
! is completed.  



   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)
   ktf=MIN(kte,kde-1)

   DO j=jts,jtf
   DO k=kts,ktf
   DO i=its,itf
         h_diabatic(i,k,j) = (rtp_new(i,k,j) - h_diabatic(i,k,j))/(dt)
   ENDDO
   ENDDO
   ENDDO

END SUBROUTINE diabatic_heating

!------------------------------------------------------------------------------

SUBROUTINE add_diabatic  ( rt_tend, h_diabatic,          &
                           ids, ide, jds, jde, kds, kde, &
                           ims, ime, jms, jme, kms, kme, &
                           its, ite, jts, jte, kts, kte )

   IMPLICIT NONE
   
   ! Input data
   
   INTEGER ,      INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                   ims, ime, jms, jme, kms, kme, &
                                   its, ite, jts, jte, kts, kte 

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(INOUT) :: rt_tend

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: h_diabatic
   
   ! Local indices.
   
   INTEGER :: i, j, k, itf, jtf, ktf
   
! store off the diabatic heating rate.
! h_diababtic shoudl have the value of rho*theta' after the dry step
! is completed.  



   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)
   ktf=MIN(kte,kde-1)

   DO j=jts,jtf
   DO k=kts,ktf
   DO i=its,itf
         rt_tend(i,k,j) = rt_tend(i,k,j) + h_diabatic(i,k,j)
   ENDDO
   ENDDO
   ENDDO

END SUBROUTINE add_diabatic

!------------------------------------------------------------------------------

SUBROUTINE coriolis ( ru, rv, rw, ru_tend, rv_tend, rw_tend, &
                      config_flags,                              &
                      f, e, sina, cosa, fzm, fzp,            &
                      ids, ide, jds, jde, kds, kde,          &
                      ims, ime, jms, jme, kms, kme,          &
                      its, ite, jts, jte, kts, kte          )

   IMPLICIT NONE
   
   ! Input data
   
   TYPE(grid_config_rec_type) ,           INTENT(IN   ) :: config_flags   

   INTEGER ,                 INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                              ims, ime, jms, jme, kms, kme, &
                                              its, ite, jts, jte, kts, kte

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(INOUT) :: ru_tend, &
                                                                rv_tend, &
                                                                rw_tend
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: ru, &
                                                                rv, &
                                                                rw

   REAL , DIMENSION( ims:ime , jms:jme ) ,         INTENT(IN   ) :: f,    &
                                                                    e,    &
                                                                    sina, &
                                                                    cosa

   REAL , DIMENSION( kms:kme ) ,                 INTENT(IN   ) :: fzm, &
                                                                  fzp
   
   ! Local indices.
   
   INTEGER :: i, j , k, ktf
   INTEGER :: i_start, i_end, j_start, j_end
   
   ktf=MIN(kte,kde-1)

! coriolis for u-momentum equation

   i_start = its
   i_end   = ite
   IF ( config_flags%open_xs .or. config_flags%specified .or. &
        config_flags%nested) i_start = MAX(ids+1,its)
   IF ( config_flags%open_xe .or. config_flags%specified .or. &
        config_flags%nested) i_end   = MIN(ide-1,ite)

   DO j = jts, MIN(jte,jde-1)

   DO k=kts,ktf
   DO i = i_start, i_end
   
     ru_tend(i,k,j)=ru_tend(i,k,j) + 0.5*(f(i,j)+f(i-1,j)) &
       *0.25*(rv(i-1,k,j+1)+rv(i,k,j+1)+rv(i-1,k,j)+rv(i,k,j)) &
           - 0.5*(e(i,j)+e(i-1,j))*0.5*(cosa(i,j)+cosa(i-1,j)) &
       *0.25*(rw(i-1,k+1,j)+rw(i-1,k,j)+rw(i,k+1,j)+rw(i,k,j))

   ENDDO
   ENDDO

   IF ( (config_flags%open_xs) .and. (its == ids) ) THEN

     DO k=kts,ktf
   
       ru_tend(its,k,j)=ru_tend(its,k,j) + 0.5*(f(its,j)+f(its,j))   &
         *0.25*(rv(its,k,j+1)+rv(its,k,j+1)+rv(its,k,j)+rv(its,k,j)) &
             - 0.5*(e(its,j)+e(its,j))*0.5*(cosa(its,j)+cosa(its,j)) &
         *0.25*(rw(its,k+1,j)+rw(its,k,j)+rw(its,k+1,j)+rw(its,k,j))

     ENDDO

   ENDIF

   IF ( (config_flags%open_xe) .and. (ite == ide) ) THEN

     DO k=kts,ktf
   
       ru_tend(ite,k,j)=ru_tend(ite,k,j) + 0.5*(f(ite-1,j)+f(ite-1,j)) &
         *0.25*(rv(ite-1,k,j+1)+rv(ite-1,k,j+1)+rv(ite-1,k,j)+rv(ite-1,k,j)) &
             - 0.5*(e(ite-1,j)+e(ite-1,j))*0.5*(cosa(ite-1,j)+cosa(ite-1,j)) &
         *0.25*(rw(ite-1,k+1,j)+rw(ite-1,k,j)+rw(ite-1,k+1,j)+rw(ite-1,k,j))

     ENDDO

   ENDIF

   ENDDO

!  coriolis term for v-momentum equation

   j_start = jts
   j_end   = jte

   IF ( config_flags%open_ys .or. config_flags%specified .or. &
        config_flags%nested) j_start = MAX(jds+1,jts)
   IF ( config_flags%open_ye .or. config_flags%specified .or. &
        config_flags%nested) j_end   = MIN(jde-1,jte)

   IF ( (config_flags%open_ys) .and. (jts == jds) ) THEN

     DO k=kts,ktf
     DO i=its,MIN(ide-1,ite)
   
        rv_tend(i,k,jts)=rv_tend(i,k,jts) - 0.5*(f(i,jts)+f(i,jts))    &
         *0.25*(ru(i,k,jts)+ru(i+1,k,jts)+ru(i,k,jts)+ru(i+1,k,jts))   &
             + 0.5*(e(i,jts)+e(i,jts))*0.5*(sina(i,jts)+sina(i,jts))   &
             *0.25*(rw(i,k+1,jts)+rw(i,k,jts)+rw(i,k+1,jts)+rw(i,k,jts)) 

     ENDDO
     ENDDO

   ENDIF

   DO j=j_start, j_end
   DO k=kts,ktf
   DO i=its,MIN(ide-1,ite)
   
      rv_tend(i,k,j)=rv_tend(i,k,j) - 0.5*(f(i,j)+f(i,j-1))    &
       *0.25*(ru(i,k,j)+ru(i+1,k,j)+ru(i,k,j-1)+ru(i+1,k,j-1)) &
           + 0.5*(e(i,j)+e(i,j-1))*0.5*(sina(i,j)+sina(i,j-1)) &
           *0.25*(rw(i,k+1,j-1)+rw(i,k,j-1)+rw(i,k+1,j)+rw(i,k,j)) 

   ENDDO
   ENDDO
   ENDDO


   IF ( (config_flags%open_ye) .and. (jte == jde) ) THEN

     DO k=kts,ktf
     DO i=its,MIN(ide-1,ite)
   
        rv_tend(i,k,jte)=rv_tend(i,k,jte) - 0.5*(f(i,jte-1)+f(i,jte-1))        &
         *0.25*(ru(i,k,jte-1)+ru(i+1,k,jte-1)+ru(i,k,jte-1)+ru(i+1,k,jte-1))   &
             + 0.5*(e(i,jte-1)+e(i,jte-1))*0.5*(sina(i,jte-1)+sina(i,jte-1))   &
             *0.25*(rw(i,k+1,jte-1)+rw(i,k,jte-1)+rw(i,k+1,jte-1)+rw(i,k,jte-1)) 

     ENDDO
     ENDDO

   ENDIF

! coriolis term for w-mometum 

   DO j=jts,MIN(jte, jde-1)
   DO k=kts+1,ktf
   DO i=its,MIN(ite, ide-1)

       rw_tend(i,k,j)=rw_tend(i,k,j) + e(i,j)*           &
          (cosa(i,j)*0.5*(fzm(k)*(ru(i,k,j)+ru(i+1,k,j)) &
          +fzp(k)*(ru(i,k-1,j)+ru(i+1,k-1,j)))           &
          -sina(i,j)*0.5*(fzm(k)*(rv(i,k,j)+rv(i,k,j+1)) & 
          +fzp(k)*(rv(i,k-1,j)+rv(i,k-1,j+1))))

   ENDDO
   ENDDO
   ENDDO

END SUBROUTINE coriolis

!------------------------------------------------------------------------------

SUBROUTINE curvature ( ru, rv, rw, u, v, w, ru_tend, rv_tend, rw_tend, &
                        config_flags,                                       &
                        msfu, msfv, fzm, fzp, rdx, rdy,                 &
                        ids, ide, jds, jde, kds, kde,                   &
                        ims, ime, jms, jme, kms, kme,                   &
                        its, ite, jts, jte, kts, kte                   )


   IMPLICIT NONE
   
   ! Input data

   TYPE(grid_config_rec_type) ,           INTENT(IN   ) :: config_flags   

   INTEGER ,                  INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                               ims, ime, jms, jme, kms, kme, &
                                               its, ite, jts, jte, kts, kte
   
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) ,                     &
                                               INTENT(INOUT) :: ru_tend, &
                                                                rv_tend, &
                                                                rw_tend

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) ,                     &
                                               INTENT(IN   ) :: ru,      &
                                                                rv,      &
                                                                rw,      &
                                                                u,       &
                                                                v,       &
                                                                w

   REAL , DIMENSION( ims:ime , jms:jme ) ,         INTENT(IN   ) :: msfu,    &
                                                                msfv

   REAL , DIMENSION( kms:kme ) ,                 INTENT(IN   ) :: fzm,     &
                                                                fzp

   REAL ,                                      INTENT(IN   ) :: rdx,     &
                                                                rdy
   
   ! Local data
   
!   INTEGER :: i, j, k, itf, jtf, ktf, kp1, im, ip, jm, jp
   INTEGER :: i, j, k, itf, jtf, ktf
   INTEGER :: i_start, i_end, j_start, j_end
!   INTEGER :: irmin, irmax, jrmin, jrmax

   REAL , DIMENSION( its-1:ite , kts:kte, jts-1:jte ) :: vxgm

      itf=MIN(ite,ide-1)
      jtf=MIN(jte,jde-1)
      ktf=MIN(kte,kde-1)

!   irmin = ims
!   irmax = ime
!   jrmin = jms
!   jrmax = jme
!   IF ( config_flags%open_xs ) irmin = ids
!   IF ( config_flags%open_xe ) irmax = ide-1
!   IF ( config_flags%open_ys ) jrmin = jds
!   IF ( config_flags%open_ye ) jrmax = jde-1
   
! Define v cross grad m at scalar points - vxgm(i,j)

   i_start = its-1
   i_end   = ite
   j_start = jts-1
   j_end   = jte

   IF ( ( config_flags%open_xs .or. config_flags%specified .or. &
        config_flags%nested) .and. (its == ids) ) i_start = its
   IF ( ( config_flags%open_xe .or. config_flags%specified .or. &
        config_flags%nested) .and. (ite == ide) ) i_end   = ite-1
   IF ( ( config_flags%open_ys .or. config_flags%specified .or. &
        config_flags%nested) .and. (jts == jds) ) j_start = jts
   IF ( ( config_flags%open_ye .or. config_flags%specified .or. &
        config_flags%nested) .and. (jte == jde) ) j_end   = jte-1

   DO j=j_start, j_end
   DO k=kts,ktf
   DO i=i_start, i_end
      vxgm(i,k,j)=0.5*(u(i,k,j)+u(i+1,k,j))*(msfv(i,j+1)-msfv(i,j))*rdy - &
                  0.5*(v(i,k,j)+v(i,k,j+1))*(msfu(i+1,j)-msfu(i,j))*rdx
   ENDDO
   ENDDO
   ENDDO

!  Pick up the boundary rows for open (radiation) lateral b.c.
!  Rather crude at present, we are assuming there is no
!    variation in this term at the boundary.

   IF ( ( config_flags%open_xs .or. config_flags%specified .or. &
        config_flags%nested) .and. (its == ids) ) THEN

     DO j = jts-1, jte
     DO k = kts, ktf
       vxgm(its-1,k,j) =  vxgm(its,k,j)
     ENDDO
     ENDDO

   ENDIF

   IF ( ( config_flags%open_xe .or. config_flags%specified .or. &
        config_flags%nested) .and. (ite == ide) ) THEN

     DO j = jts-1, jte
     DO k = kts, ktf
       vxgm(ite,k,j) =  vxgm(ite-1,k,j)
     ENDDO
     ENDDO

   ENDIF

   IF ( ( config_flags%open_ys .or. config_flags%specified .or. &
        config_flags%nested) .and. (jts == jds) ) THEN

     DO k = kts, ktf
     DO i = its-1, ite
       vxgm(i,k,jts-1) =  vxgm(i,k,jts)
     ENDDO
     ENDDO

   ENDIF

   IF ( ( config_flags%open_ye .or. config_flags%specified .or. &
        config_flags%nested) .and. (jte == jde) ) THEN

     DO k = kts, ktf
     DO i = its-1, ite
       vxgm(i,k,jte) =  vxgm(i,k,jte-1)
     ENDDO
     ENDDO

   ENDIF

!  curvature term for u momentum eqn.

   i_start = its
   IF ( config_flags%open_xs .or. config_flags%specified .or. &
        config_flags%nested) i_start = MAX ( ids+1 , its )
   IF ( config_flags%open_xe .or. config_flags%specified .or. &
        config_flags%nested) i_end   = MIN ( ide-1 , ite )

   DO j=jts,MIN(jde-1,jte)
   DO k=kts,ktf
   DO i=i_start,i_end

      ru_tend(i,k,j)=ru_tend(i,k,j) + 0.5*(vxgm(i,k,j)+vxgm(i-1,k,j)) &
              *0.25*(rv(i-1,k,j+1)+rv(i,k,j+1)+rv(i-1,k,j)+rv(i,k,j)) &
               - u(i,k,j)*reradius &
              *0.25*(rw(i-1,k+1,j)+rw(i-1,k,j)+rw(i,k+1,j)+rw(i,k,j))

   ENDDO
   ENDDO
   ENDDO

!  curvature term for v momentum eqn.

   j_start = jts
   IF ( config_flags%open_ys .or. config_flags%specified .or. &
        config_flags%nested) j_start = MAX ( jds+1 , jts )
   IF ( config_flags%open_ye .or. config_flags%specified .or. &
        config_flags%nested) j_end   = MIN ( jde-1 , jte ) 

   DO j=j_start,j_end
   DO k=kts,ktf
   DO i=its,MIN(ite,ide-1)

      rv_tend(i,k,j)=rv_tend(i,k,j) - 0.5*(vxgm(i,k,j)+vxgm(i,k,j-1)) &
              *0.25*(ru(i,k,j)+ru(i+1,k,j)+ru(i,k,j-1)+ru(i+1,k,j-1)) &
                    + v(i,k,j)*reradius                               &
              *0.25*(rw(i,k+1,j-1)+rw(i,k,j-1)+rw(i,k+1,j)+rw(i,k,j))

   ENDDO
   ENDDO
   ENDDO

!  curvature term for vertical momentum eqn.

   DO j=jts,MIN(jte,jde-1)
   DO k=MAX(2,kts),ktf
   DO i=its,MIN(ite,ide-1)

      rw_tend(i,k,j)=rw_tend(i,k,j) + reradius*                              &
    (0.5*(fzm(k)*(ru(i,k,j)+ru(i+1,k,j))+fzp(k)*(ru(i,k-1,j)+ru(i+1,k-1,j))) &
    *0.5*(fzm(k)*( u(i,k,j) +u(i+1,k,j))+fzp(k)*( u(i,k-1,j) +u(i+1,k-1,j)))     &
    +0.5*(fzm(k)*(rv(i,k,j)+rv(i,k,j+1))+fzp(k)*(rv(i,k-1,j)+rv(i,k-1,j+1))) &
    *0.5*(fzm(k)*( v(i,k,j) +v(i,k,j+1))+fzp(k)*( v(i,k-1,j) +v(i,k-1,j+1))))

   ENDDO
   ENDDO
   ENDDO

END SUBROUTINE curvature

!------------------------------------------------------------------------------

SUBROUTINE couple_theta_m ( rth, qv, couple,              &
                            ids, ide, jds, jde, kds, kde, &
                            ims, ime, jms, jme, kms, kme, &
                            its, ite, jts, jte, kts, kte )

   IMPLICIT NONE

   ! Input data

   INTEGER ,          INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                       ims, ime, jms, jme, kms, kme, &
                                       its, ite, jts, jte, kts, kte

   LOGICAL, INTENT(IN   ) :: couple

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: qv

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(INOUT) :: rth
   
   
   INTEGER :: i, j, k, itf, jtf, ktf

      itf=MIN(ite,ide-1)
      jtf=MIN(jte,jde-1)
      ktf=MIN(kte,kde-1)

   ! couple (or decouple) theta and rho*theta with (1+rvovrd*qv)

   IF( couple ) THEN

      DO j=jts,jtf
      DO k=kts,ktf
      DO i=its,itf
         rth(i,k,j) =rth(i,k,j) *(1.+rvovrd*qv(i,k,j))
      ENDDO
      ENDDO
      ENDDO

    ELSE

      DO j=jts,jtf
      DO k=kts,ktf
      DO i=its,itf
         rth(i,k,j) =rth(i,k,j) /(1.+rvovrd*qv(i,k,j))
      ENDDO
      ENDDO
      ENDDO
   
   ENDIF

END SUBROUTINE couple_theta_m

!-------------------------------------------------------------------------------

SUBROUTINE decouple ( rr, rfield, field, name, config_flags, &
                      fzm, fzp,                          &
                      ids, ide, jds, jde, kds, kde,      &
                      ims, ime, jms, jme, kms, kme,      &
                      its, ite, jts, jte, kts, kte      )

   IMPLICIT NONE

   ! Input data

   TYPE(grid_config_rec_type) ,           INTENT(IN   ) :: config_flags   

   INTEGER ,                                   INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                                                ims, ime, jms, jme, kms, kme, &
                                                                its, ite, jts, jte, kts, kte

   CHARACTER(LEN=1) ,                          INTENT(IN   ) :: name

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: rfield

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: rr
   
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(OUT  ) :: field
   
   REAL , DIMENSION( kms:kme ) , INTENT(IN   ) :: fzm, fzp
   
   ! Local data
   
   INTEGER :: i, j, k, itf, jtf, ktf
   
   ktf=MIN(kte,kde-1)
   
   IF (name .EQ. 'u')THEN
      itf=ite
      jtf=MIN(jte,jde-1)

      DO j=jts,jtf
      DO k=kts,ktf
      DO i=its,itf
         field(i,k,j)=rfield(i,k,j)/(0.5*(rr(i,k,j)+rr(i-1,k,j)))
      ENDDO
      ENDDO
      ENDDO

   ELSE IF (name .EQ. 'v')THEN
      itf=MIN(ite,ide-1)
      jtf=jte

      DO j=jts,jtf
      DO k=kts,ktf
        DO i=its,itf
             field(i,k,j)=rfield(i,k,j)/(0.5*(rr(i,k,j)+rr(i,k,j-1)))
        ENDDO
      ENDDO
      ENDDO

   ELSE IF (name .EQ. 'w')THEN
      itf=MIN(ite,ide-1)
      jtf=MIN(jte,jde-1)
      DO j=jts,jtf
      DO k=kts+1,ktf
      DO i=its,itf
         field(i,k,j)=rfield(i,k,j)/(fzm(k)*rr(i,k,j)+fzp(k)*rr(i,k-1,j))
      ENDDO
      ENDDO
      ENDDO

      DO j=jts,jtf
      DO i=its,itf
        field(i,kte,j) = 0.
      ENDDO
      ENDDO

   ELSE 
      itf=MIN(ite,ide-1)
      jtf=MIN(jte,jde-1)
   ! For theta we will decouple tb and tp and add them to give t afterwards
      DO j=jts,jtf
      DO k=kts,ktf
      DO i=its,itf
         field(i,k,j)=rfield(i,k,j)/rr(i,k,j)
      ENDDO
      ENDDO
      ENDDO
   
   ENDIF

END SUBROUTINE decouple

!-------------------------------------------------------------------------------

SUBROUTINE decouple_thm ( rr, rth, theta, config_flags,     &
                          moist, fzm, fzp, n_moist,     &
                          ids, ide, jds, jde, kds, kde, &
                          ims, ime, jms, jme, kms, kme, &
                          its, ite, jts, jte, kts, kte )

   IMPLICIT NONE

   ! Input data

   TYPE(grid_config_rec_type) ,           INTENT(IN   ) :: config_flags   

   INTEGER ,   INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                ims, ime, jms, jme, kms, kme, &
                                its, ite, jts, jte, kts, kte

   INTEGER ,   INTENT(IN   ) :: n_moist

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: rr, rth

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme, n_moist ) , &
                                                     INTENT(IN   ) :: moist
   
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(OUT  ) :: theta
   
   REAL , DIMENSION( kms:kme ) , INTENT(IN   ) :: fzm, fzp
   
   ! Local data
   
   INTEGER :: i, j, k, itf, jtf, ktf

   ktf=MIN(kte,kde-1)
   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)

   IF ( n_moist >= PARAM_FIRST_SCALAR ) THEN

      DO j=jts,jtf
      DO k=kts,ktf
      DO i=its,itf
         theta(i,k,j)=rth(i,k,j)/( rr(i,k,j)*(1.+rvovrd*moist(i,k,j,P_QV)) )
      ENDDO
      ENDDO
      ENDDO

    ELSE

      DO j=jts,jtf
      DO k=kts,ktf
      DO i=its,itf
         theta(i,k,j)=rth(i,k,j)/rr(i,k,j)
      ENDDO
      ENDDO
      ENDDO

    ENDIF
   
END SUBROUTINE decouple_thm

!-------------------------------------------------------------------------------

SUBROUTINE decouple_thmp ( rr, rthp, thp, config_flags,      &
                           moist, rrb, rtb,              &
                           fzm, fzp, n_moist,            &
                           ids, ide, jds, jde, kds, kde, &
                           ims, ime, jms, jme, kms, kme, &
                           its, ite, jts, jte, kts, kte )

   IMPLICIT NONE

   ! Input data

   TYPE(grid_config_rec_type) ,           INTENT(IN   ) :: config_flags   

   INTEGER ,   INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                ims, ime, jms, jme, kms, kme, &
                                its, ite, jts, jte, kts, kte

   INTEGER ,   INTENT(IN   ) :: n_moist

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: rr,   &
                                                                      rthp, &
                                                                      rrb,  &
                                                                      rtb

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme , n_moist ) , &
                                                     INTENT(IN   ) :: moist
   
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(OUT  ) :: thp
   
   REAL , DIMENSION( kms:kme ) , INTENT(IN   ) :: fzm, fzp
   
   ! Local data
   
   INTEGER :: i, j, k, itf, jtf, ktf

   ktf=MIN(kte,kde-1)
   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)

   IF ( n_moist >= PARAM_FIRST_SCALAR ) THEN

      DO j=jts,jtf
      DO k=kts,ktf
      DO i=its,itf
         thp(i,k,j)= (rthp(i,k,j)+rtb(i,k,j))/(1.+rvovrd*moist(i,k,j,P_QV)) &
                      /rr(i,k,j)  - rtb(i,k,j)/rrb(i,k,j)
      ENDDO
      ENDDO
      ENDDO

    ELSE

      DO j=jts,jtf
      DO k=kts,ktf
      DO i=its,itf
         thp(i,k,j)=(rthp(i,k,j)+rtb(i,k,j))/rr(i,k,j) - rtb(i,k,j)/rrb(i,k,j)
      ENDDO
      ENDDO
      ENDDO

    ENDIF
   
END SUBROUTINE decouple_thmp

!------------------------------------------------------------------------------

SUBROUTINE mix_theta_m ( rt_tend, th_mix, qv_mix, rho, &
                         qv, theta_m, n_moist,         &
                         ids, ide, jds, jde, kds, kde, &
                         ims, ime, jms, jme, kms, kme, &
                         its, ite, jts, jte, kts, kte )

   IMPLICIT NONE

   ! Input data

   INTEGER ,               INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                            ims, ime, jms, jme, kms, kme, &
                                            its, ite, jts, jte, kts, kte

   REAL , DIMENSION( ims:ime, kms:kme , jms:jme ) , INTENT(IN   ) :: th_mix,  &
                                                                    qv_mix,  &
                                                                    rho,     &
                                                                    theta_m

   INTEGER , INTENT(IN) :: n_moist


   REAL , DIMENSION( ims:ime, kms:kme , jms:jme ) , INTENT(IN   ) :: qv

   REAL , DIMENSION( ims:ime, kms:kme , jms:jme ) , INTENT(INOUT) :: rt_tend

   ! Local data
   
   INTEGER :: i, j, k, itf, jtf, ktf
   REAL    :: theta_d

   ktf=MIN(kte,kde-1)
   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)

   IF( n_moist >= PARAM_FIRST_SCALAR ) then

     DO j=jts,jtf
     DO k=kts,ktf
     DO i=its,itf
       theta_d = theta_m(i,k,j)/(1+rvovrd*qv(i,k,j))
       rt_tend(i,k,j) = rt_tend(i,k,j)  +            (      &
                        (1+rvovrd*qv(i,k,j))*th_mix(i,k,j)  &
                              + rvovrd*theta_d*qv_mix(i,k,j)  &
                                                         )
     ENDDO
     ENDDO
     ENDDO

   ELSE
   
     DO j=jts,jtf
     DO k=kts,ktf
     DO i=its,itf

       rt_tend(i,k,j) = rt_tend(i,k,j)  + th_mix(i,k,j)

     ENDDO
     ENDDO
     ENDDO

   ENDIF

END SUBROUTINE mix_theta_m

!-------------------------------------------------------------------------------

SUBROUTINE mix_qv ( qv_tend, qv_mix, rho,         &
                    ids, ide, jds, jde, kds, kde, &
                    ims, ime, jms, jme, kms, kme, &
                    its, ite, jts, jte, kts, kte )

   IMPLICIT NONE

   ! Input data

   INTEGER ,               INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                            ims, ime, jms, jme, kms, kme, &
                                            its, ite, jts, jte, kts, kte

   REAL , DIMENSION( ims:ime, kms:kme , jms:jme ) , INTENT(IN   ) :: qv_mix,  &
                                                                     rho

   REAL , DIMENSION( ims:ime, kms:kme , jms:jme ) , INTENT(INOUT) :: qv_tend

   ! Local data
   
   INTEGER :: i, j, k, itf, jtf, ktf

   ktf=MIN(kte,kde-1)
   itf=MIN(ite,ide-1)
   jtf=MIN(jte,jde-1)

     DO j=jts,jtf
     DO k=kts,ktf
     DO i=its,itf

       qv_tend(i,k,j) = qv_tend(i,k,j)  + qv_mix(i,k,j)

     ENDDO
     ENDDO
     ENDDO

END SUBROUTINE mix_qv

!-------------------------------------------------------------------------------

SUBROUTINE horizontal_diffusion (name, field, tendency, rr, config_flags, &
                                 msfu, msfv, msft, khdif, rdx, rdy,   &
                                 ids, ide, jds, jde, kds, kde,        &
                                 ims, ime, jms, jme, kms, kme,        &
                                 its, ite, jts, jte, kts, kte        )

   IMPLICIT NONE
   
   ! Input data
   
   TYPE(grid_config_rec_type), INTENT(IN   ) :: config_flags

   INTEGER ,        INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                     ims, ime, jms, jme, kms, kme, &
                                     its, ite, jts, jte, kts, kte

   CHARACTER(LEN=1) ,                          INTENT(IN   ) :: name

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: field

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(INOUT) :: tendency

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: rr

   REAL , DIMENSION( ims:ime , jms:jme ) ,         INTENT(IN   ) :: msfu,      &
                                                                msfv,      &
                                                                msft

   REAL ,                                      INTENT(IN   ) :: rdx,       &
                                                                rdy,       &
                                                                khdif

   ! Local data
   
   INTEGER :: i, j, k, itf, jtf, ktf

   INTEGER :: i_start, i_end, j_start, j_end

   REAL , DIMENSION( kms:kme ) :: difcoef

   REAL :: mrdx, mkrdxm, mkrdxp, &
           mrdy, mkrdym, mkrdyp, &
           rcoup, constc, constd, tmp

   ktf=MIN(kte,kde-1)
  
   constc=0.2
   constd=0.1

   DO k=kts,ktf
! upper absorbing layer (test version)
      tmp=constc-(constc-constd)*(1.-float(k-kts)/float(ktf-kts+2))
      difcoef(k)=(tanh(constc)-tanh(constd))/(tanh(constc)-tanh(tmp))
      difcoef(k)=amin1(difcoef(k),10.)
! turn off upper absorbing layer
      difcoef(k)=1.
   ENDDO

   IF (name .EQ. 'u') THEN

      i_start = its
      i_end   = ite
      j_start = jts
      j_end   = MIN(jte,jde-1)

      IF ( config_flags%open_xs .or. config_flags%specified .or. &
           config_flags%nested) i_start = MAX(ids+1,its)
      IF ( config_flags%open_xe .or. config_flags%specified .or. &
           config_flags%nested) i_end   = MIN(ide-1,ite)
      IF ( config_flags%open_ys .or. config_flags%specified .or. &
           config_flags%nested) j_start = MAX(jds+1,jts)
      IF ( config_flags%open_ye .or. config_flags%specified .or. &
           config_flags%nested) j_end   = MIN(jde-2,jte)

      DO j = j_start, j_end
      DO k=kts,ktf
      DO i = i_start, i_end

         mkrdxm=msft(i-1,j)*khdif*rdx
         mkrdxp=msft(i,j)*khdif*rdx
         mrdx=msfu(i,j)*rdx
         mkrdym=0.5*(msfu(i,j)+msfu(i,j-1))*khdif*rdy
         mkrdyp=0.5*(msfu(i,j)+msfu(i,j+1))*khdif*rdy
         mrdy=msfu(i,j)*rdy

            rcoup=0.5*(rr(i,k,j)+rr(i-1,k,j))
            tendency(i,k,j)=tendency(i,k,j)+rcoup*difcoef(k)*( &
                            mrdx*(mkrdxp*(field(i+1,k,j)-field(i  ,k,j))  &
                                 -mkrdxm*(field(i  ,k,j)-field(i-1,k,j))) &
                           +mrdy*(mkrdyp*(field(i,k,j+1)-field(i,k,j  ))  &
                                 -mkrdym*(field(i,k,j  )-field(i,k,j-1))))
      ENDDO
      ENDDO
      ENDDO
   
   ELSE IF (name .EQ. 'v')THEN

      i_start = its
      i_end   = MIN(ite,ide-1)
      j_start = jts
      j_end   = jte

      IF ( config_flags%open_xs .or. config_flags%specified .or. &
           config_flags%nested) i_start = MAX(ids+1,its)
      IF ( config_flags%open_xe .or. config_flags%specified .or. &
           config_flags%nested) i_end   = MIN(ide-2,ite)
      IF ( config_flags%open_ys .or. config_flags%specified .or. &
           config_flags%nested) j_start = MAX(jds+1,jts)
      IF ( config_flags%open_ye .or. config_flags%specified .or. &
           config_flags%nested) j_end   = MIN(jde-1,jte)

      DO j = j_start, j_end
      DO k=kts,ktf
      DO i = i_start, i_end

         mkrdxm=0.5*(msfv(i,j)+msfv(i-1,j))*khdif*rdx
         mkrdxp=0.5*(msfv(i,j)+msfv(i+1,j))*khdif*rdx
         mrdx=msfv(i,j)*rdx
         mkrdym=msft(i,j-1)*khdif*rdy
         mkrdyp=msft(i,j)*khdif*rdy
         mrdy=msfv(i,j)*rdy

            rcoup=0.5*(rr(i,k,j)+rr(i,k,j-1))
            tendency(i,k,j)=tendency(i,k,j)+rcoup*difcoef(k)*( &
                            mrdx*(mkrdxp*(field(i+1,k,j)-field(i  ,k,j))  &
                                 -mkrdxm*(field(i  ,k,j)-field(i-1,k,j))) &
                           +mrdy*(mkrdyp*(field(i,k,j+1)-field(i,k,j  ))  &
                                 -mkrdym*(field(i,k,j  )-field(i,k,j-1))))
      ENDDO
      ENDDO
      ENDDO
   
   ELSE IF (name .EQ. 'w')THEN

      i_start = its
      i_end   = MIN(ite,ide-1)
      j_start = jts
      j_end   = MIN(jte,jde-1)

      IF ( config_flags%open_xs .or. config_flags%specified .or. &
           config_flags%nested) i_start = MAX(ids+1,its)
      IF ( config_flags%open_xe .or. config_flags%specified .or. &
           config_flags%nested) i_end   = MIN(ide-2,ite)
      IF ( config_flags%open_ys .or. config_flags%specified .or. &
           config_flags%nested) j_start = MAX(jds+1,jts)
      IF ( config_flags%open_ye .or. config_flags%specified .or. &
           config_flags%nested) j_end   = MIN(jde-2,jte)


      DO j = j_start, j_end
      DO k=kts+1,ktf
      DO i = i_start, i_end

         mkrdxm=msfu(i,j)*khdif*rdx
         mkrdxp=msfu(i+1,j)*khdif*rdx
         mrdx=msft(i,j)*rdx
         mkrdym=msfv(i,j)*khdif*rdy
         mkrdyp=msfv(i,j+1)*khdif*rdy
         mrdy=msft(i,j)*rdy

            rcoup=0.5*(rr(i,k,j)+rr(i,k-1,j))
            tendency(i,k,j)=tendency(i,k,j)+rcoup*difcoef(k)*( &
                            mrdx*(mkrdxp*(field(i+1,k,j)-field(i  ,k,j)) &
                                 -mkrdxm*(field(i  ,k,j)-field(i-1,k,j))) &
                           +mrdy*(mkrdyp*(field(i,k,j+1)-field(i,k,j  )) &
                                 -mkrdym*(field(i,k,j  )-field(i,k,j-1))))
      ENDDO
      ENDDO
      ENDDO
   
   ELSE


      i_start = its
      i_end   = MIN(ite,ide-1)
      j_start = jts
      j_end   = MIN(jte,jde-1)

      IF ( config_flags%open_xs .or. config_flags%specified .or. &
           config_flags%nested) i_start = MAX(ids+1,its)
      IF ( config_flags%open_xe .or. config_flags%specified .or. &
           config_flags%nested) i_end   = MIN(ide-2,ite)
      IF ( config_flags%open_ys .or. config_flags%specified .or. &
           config_flags%nested) j_start = MAX(jds+1,jts)
      IF ( config_flags%open_ye .or. config_flags%specified .or. &
           config_flags%nested) j_end   = MIN(jde-2,jte)

      DO j = j_start, j_end
      DO k=kts,ktf
      DO i = i_start, i_end

         mkrdxm=msfu(i,j)*khdif*rdx
         mkrdxp=msfu(i+1,j)*khdif*rdx
         mrdx=msft(i,j)*rdx
         mkrdym=msfv(i,j)*khdif*rdy
         mkrdyp=msfv(i,j+1)*khdif*rdy
         mrdy=msft(i,j)*rdy

            rcoup=rr(i,k,j)
            tendency(i,k,j)=tendency(i,k,j)+rcoup*difcoef(k)*( &
                            mrdx*(mkrdxp*(field(i+1,k,j)-field(i  ,k,j))  &
                                 -mkrdxm*(field(i  ,k,j)-field(i-1,k,j))) &
                           +mrdy*(mkrdyp*(field(i,k,j+1)-field(i,k,j  ))  &
                                 -mkrdym*(field(i,k,j  )-field(i,k,j-1))))
      ENDDO
      ENDDO
      ENDDO
           
   ENDIF

END SUBROUTINE horizontal_diffusion

!-----------------------------------------------------------------------------------------

SUBROUTINE pressure_gradient ( pp, ru_tend, rv_tend, rw_tend, &
                               moist, config_flags,               &
                               msft, rdzu, rdzw, z_zeta,      &
                               zx, zy, fzm, fzp, rdx, rdy,    &
                               cf1, cf2, cf3, n_moist,        &
                               ids, ide, jds, jde, kds, kde,  &
                               ims, ime, jms, jme, kms, kme,  &
                               its, ite, jts, jte, kts, kte  )

   IMPLICIT NONE
   
   ! Input data
   
   TYPE(grid_config_rec_type), INTENT(IN   ) :: config_flags
   
   INTEGER ,    INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                 ims, ime, jms, jme, kms, kme, &
                                 its, ite, jts, jte, kts, kte

   INTEGER ,    INTENT(IN   ) :: n_moist

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) ,                      &
                                               INTENT(INOUT) :: ru_tend,  &
                                                                rv_tend,  &
                                                                rw_tend
 
   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: zx,   &
                                                                      zy

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(IN   ) :: pp

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme , n_moist ), INTENT(IN   ) :: moist

   REAL , DIMENSION( ims:ime , jms:jme ) ,         INTENT(IN   ) :: msft,   &
                                                                    z_zeta

   REAL , DIMENSION( kms:kme ) ,                 INTENT(IN   ) :: rdzu,  &
                                                                  rdzw,  &
                                                                  fzm,   &
                                                                  fzp
 
   REAL ,                                      INTENT(IN   ) :: rdx,      &
                                                                rdy,      &
                                                                cf1,      &
                                                                cf2,      &
                                                                cf3
   
   ! Local data
   
   INTEGER :: i, j, k, ktf, ispe
   INTEGER :: i_start, i_end, j_start, j_end

   LOGICAL :: moist_simulation

   REAL, dimension( its:ite, kms:kme ) :: ppzy
   REAL, dimension( its:ite, kms:kme ) :: ppzx
   REAL, dimension( its:ite, kms:kme ) :: rhod_over_rhom
   REAL, dimension( its:ite, jts:jte ) :: rhod_over_rhom_w
   REAL :: total_water
   
!

   moist_simulation = .false. 

   IF( n_moist >= PARAM_FIRST_SCALAR ) THEN

     moist_simulation = .true.

   ELSE ! dry simulation

      rhod_over_rhom = 1.

   ENDIF

   
   ktf=MIN(kte,kde-1)

!  u-momentum equation pressure gradient

      i_start = its
      i_end   = ite
      j_start = jts
      j_end   = MIN(jte,jde-1)


      IF ( config_flags%open_xs .or. config_flags%specified .or. &
           config_flags%nested) i_start = MAX(ids+1,its)
      IF ( config_flags%open_xe .or. config_flags%specified .or. &
           config_flags%nested) i_end   = MIN(ide-1,ite)



   j_loop_u : DO j = j_start, j_end

      IF( moist_simulation ) THEN

        DO k = kts, ktf
        DO i = i_start, i_end
          total_water = 0.
          DO ispe=PARAM_FIRST_SCALAR, n_moist
             total_water = total_water + 0.5*( moist(i,  k,j,ispe) + &
                                               moist(i-1,k,j,ispe) )
          ENDDO

          rhod_over_rhom(i,k) = 1./(1.+total_water)
        ENDDO
        ENDDO
      ELSE
        DO k = kts, ktf
        DO i = i_start, i_end
          rhod_over_rhom(i,k) = 1.
        ENDDO
        ENDDO
      ENDIF

     DO k = kts, ktf

      IF (k == 1) THEN
        DO i = i_start, i_end
          ppzx(i,k)  = 0.5*zx(i,k,j)*(cf1*(pp(i-1,k  ,j)+pp(i,k  ,j)) &
                                         +cf2*(pp(i-1,k+1,j)+pp(i,k+1,j)) &
                                         +cf3*(pp(i-1,k+2,j)+pp(i,k+2,j)))
          ppzx(i,k+1) = 0.5*zx(i,k+1,j)*(fzm(k+1)*(pp(i-1,k+1,j)+pp(i,k+1,j)) &
                                          +fzp(k+1)*(pp(i-1,k  ,j)+pp(i,k  ,j)))
        ENDDO
      ELSE IF (k == ktf) THEN
        DO i = i_start, i_end
          ppzx(i,k+1) = 0.
        ENDDO
      ELSE
        DO i = i_start, i_end
          ppzx(i,k+1) = 0.5*zx(i,k+1,j)*(fzm(k+1)*(pp(i-1,k+1,j)+pp(i,k+1,j)) &
                                          +fzp(k+1)*(pp(i-1,k  ,j)+pp(i,k  ,j)))
        ENDDO
      ENDIF

     ENDDO


     DO k=1,ktf
        DO i = i_start, i_end
         ru_tend(i,k,j)=ru_tend(i,k,j) + rhod_over_rhom(i,k)*(                     &
                           - rdx*(z_zeta(i,j)*pp(i,k,j)-z_zeta(i-1,j)*pp(i-1,k,j)) &
                           + rdzw(k)*(ppzx(i,k+1)-ppzx(i,k))                )
        ENDDO
     ENDDO

   ENDDO  j_loop_u

! v-momentum equation pressure gradient



      i_start = its
      i_end   = MIN(ite,ide-1)
      j_start = jts
      j_end   = jte

      IF ( config_flags%open_ys .or. config_flags%specified .or. &
           config_flags%nested) j_start = MAX(jds+1,jts)
      IF ( config_flags%open_ye .or. config_flags%specified .or. &
           config_flags%nested) j_end   = MIN(jde-1,jte)

 j_loop_v : DO j = j_start, j_end

      IF( moist_simulation ) THEN
        DO k=kts,ktf
        DO i = i_start, i_end
          total_water = 0.
          DO ispe=PARAM_FIRST_SCALAR, n_moist
             total_water = total_water + 0.5*( moist(i,k,j  ,ispe) + &
                                               moist(i,k,j-1,ispe) )
          ENDDO

          rhod_over_rhom(i,k)  = 1./(1.+total_water)
        ENDDO
        ENDDO
      ELSE
        DO k=kts,ktf
        DO i = i_start, i_end
          rhod_over_rhom(i,k) = 1.
        ENDDO
        ENDDO
      ENDIF

    DO k=kts,ktf

      IF (k == 1) THEN

        DO i = i_start, i_end
         ppzy(i,k) = 0.5*zy(i,k,j)*(cf1*(pp(i,k  ,j-1)+pp(i,k  ,j)) &
                                       +cf2*(pp(i,k+1,j-1)+pp(i,k+1,j)) &
                                       +cf3*(pp(i,k+2,j-1)+pp(i,k+2,j)))
         ppzy(i,k+1) = 0.5*zy(i,k+1,j)*(fzm(k+1)*(pp(i,k+1,j-1)+pp(i,k+1,j)) &
                                         +fzp(k+1)*(pp(i,k  ,j-1)+pp(i,k  ,j)))
        ENDDO
      ELSE IF (k == ktf) THEN

        DO i = i_start, i_end
         ppzy(i,k+1) = 0.
        ENDDO

      ELSE
        DO i = i_start, i_end
         ppzy(i,k+1) = 0.5*zy(i,k+1,j)*(fzm(k+1)*(pp(i,k+1,j-1)+pp(i,k+1,j)) &
                                         +fzp(k+1)*(pp(i,k  ,j-1)+pp(i,k  ,j)))
        ENDDO
      ENDIF

    ENDDO

    DO k=kts,ktf
    DO i = i_start, i_end
      rv_tend(i,k,j)=rv_tend(i,k,j) + rhod_over_rhom(i,k)*(                    &
                        - rdy*(z_zeta(i,j)*pp(i,k,j)-z_zeta(i,j-1)*pp(i,k,j-1))  &
                        + rdzw(k)*(ppzy(i,k+1)-ppzy(i,k))                 )
   
    ENDDO
    ENDDO

  ENDDO j_loop_v

! w-momentum equation pressure gradient


   i_start = its
   i_end   = MIN(ite,ide-1)
   j_start = jts
   j_end   = MIN(jte,jde-1)



   DO j = j_start, j_end

     IF( moist_simulation ) THEN
       DO k = 2, ktf
       DO i = i_start, i_end
         total_water = 0.
         DO ispe=PARAM_FIRST_SCALAR, n_moist
            total_water = total_water + 0.5*( moist(i,k  ,j,ispe) + &
                                              moist(i,k-1,j,ispe) )
         ENDDO

         rhod_over_rhom(i,k) = 1./(1.+total_water)
       ENDDO
       ENDDO
     ENDIF

     DO k = 2, ktf
     DO i = i_start, i_end
         rw_tend(i,k,j)=rw_tend(i,k,j) - rhod_over_rhom(i,k)*rdzu(k)/msft(i,j) &
                           *(pp(i,k,j)-pp(i,k-1,j)) 
     ENDDO
     ENDDO


   ENDDO

END SUBROUTINE pressure_gradient


!-------------------------------------------------------------------------------

SUBROUTINE vertical_diffusion_1 ( name, field, tendency, rr,    &
                                config_flags,                     &
                                z, zeta_z, rdzw, kvdif,       &
                                ids, ide, jds, jde, kds, kde, &
                                ims, ime, jms, jme, kms, kme, &
                                its, ite, jts, jte, kts, kte )


   IMPLICIT NONE
   
   ! Input data
   
   TYPE(grid_config_rec_type), INTENT(IN   ) :: config_flags

   INTEGER ,    INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                 ims, ime, jms, jme, kms, kme, &
                                 its, ite, jts, jte, kts, kte

   CHARACTER(LEN=1) ,                          INTENT(IN   ) :: name

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) ,                      &
                                               INTENT(IN   ) :: field,    &
                                                                rr,       &
                                                                z

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(INOUT) :: tendency

   REAL , DIMENSION( ims:ime , jms:jme ) ,         INTENT(IN   ) :: zeta_z

   REAL , DIMENSION( kms:kme ) ,                 INTENT(IN   ) :: rdzw

   REAL ,                                      INTENT(IN   ) :: kvdif
   
   ! Local data
   
   INTEGER :: i, j, k, itf, jtf, ktf
   INTEGER :: i_start, i_end, j_start, j_end

   REAL , DIMENSION(its:ite, jts:jte) :: vfluxm, vfluxp, zz
   REAL , DIMENSION(its:ite, 0:kte+1) :: vflux

   REAL :: rdz, rcoup

   ktf=MIN(kte,kde-1)
   

   IF      (name .EQ. 'u')THEN

   write(6,*) ' loop order not changed in subroutine vertical diffusion for u '

      i_start = its
      i_end   = ite
      j_start = jts
      j_end   = MIN(jte,jde-1)

      IF ( config_flags%open_xs .or. config_flags%specified .or. &
           config_flags%nested) i_start = MAX(ids+1,its)
      IF ( config_flags%open_xe .or. config_flags%specified .or. &
           config_flags%nested) i_end   = MIN(ide-1,ite)

         k = 1
         DO j = j_start, j_end
         DO i = i_start, i_end

            zz(i,j)=0.5*(zeta_z(i,j)+zeta_z(i-1,j))
            rdz=2./((z(i,k+1,j)-z(i,k,j))+(z(i-1,k+1,j)-z(i-1,k,j)))
            vfluxp(i,j)=kvdif*rdz*(field(i,k+1,j)-field(i,k,j))
            vfluxm(i,j)=vfluxp(i,j)
            rcoup=0.5*(rr(i,k,j)+rr(i-1,k,j))
            tendency(i,k,j)=tendency(i,k,j)+rcoup*zz(i,j)*rdzw(k)*(vfluxp(i,j)-vfluxm(i,j))

         ENDDO
         ENDDO
     DO k=kts+1,ktf-1  
         DO j = j_start, j_end
         DO i = i_start, i_end

            vfluxm(i,j)=vfluxp(i,j)
            rdz=2./((z(i,k+1,j)-z(i,k,j))+(z(i-1,k+1,j)-z(i-1,k,j)))
            vfluxp(i,j)=kvdif*rdz*(field(i,k+1,j)-field(i,k,j))
            rcoup=0.5*(rr(i,k,j)+rr(i-1,k,j))
            tendency(i,k,j)=tendency(i,k,j)+rcoup*zz(i,j)*rdzw(k)*(vfluxp(i,j)-vfluxm(i,j))

         ENDDO
         ENDDO

     ENDDO
       k = ktf
         DO j = j_start, j_end
         DO i = i_start, i_end

            vfluxm(i,j)=vfluxp(i,j)
            vfluxp(i,j)=0.
            rcoup=0.5*(rr(i,k,j)+rr(i-1,k,j))
            tendency(i,k,j)=tendency(i,k,j)+rcoup*zz(i,j)*rdzw(k)*(vfluxp(i,j)-vfluxm(i,j))

         ENDDO
         ENDDO
   
   ELSE IF (name .EQ. 'v')THEN

   write(6,*) ' loop order not changed in subroutine vertical diffusion for v '

      i_start = its
      i_end   = MIN(ite,ide-1)
      j_start = jts
      j_end   = jte

      IF ( config_flags%open_ys .or. config_flags%specified .or. &
           config_flags%nested) j_start = MAX(jds+1,jts)
      IF ( config_flags%open_ye .or. config_flags%specified .or. &
           config_flags%nested) j_end   = MIN(jde-1,jte)

       k = 1
         DO j = j_start, j_end
         DO i = i_start, i_end

            zz(i,j)=0.5*(zeta_z(i,j)+zeta_z(i,j-1))
            rdz=2./((z(i,k+1,j)-z(i,k,j))+(z(i,k+1,j-1)-z(i,k,j-1)))
            vfluxp(i,j)=kvdif*rdz*(field(i,k+1,j)-field(i,k,j))
            vfluxm(i,j)=vfluxp(i,j)
            rcoup=0.5*(rr(i,k,j)+rr(i,k,j-1))
            tendency(i,k,j)=tendency(i,k,j)+rcoup*zz(i,j)*rdzw(k)*(vfluxp(i,j)-vfluxm(i,j))

         ENDDO
         ENDDO
     DO k=kts+1,ktf-1
         DO j = j_start, j_end
         DO i = i_start, i_end

            vfluxm(i,j)=vfluxp(i,j)
            rdz=2./((z(i,k+1,j)-z(i,k,j))+(z(i,k+1,j-1)-z(i,k,j-1)))
            vfluxp(i,j)=kvdif*rdz*(field(i,k+1,j)-field(i,k,j))
            rcoup=0.5*(rr(i,k,j)+rr(i,k,j-1))
            tendency(i,k,j)=tendency(i,k,j)+rcoup*zz(i,j)*rdzw(k)*(vfluxp(i,j)-vfluxm(i,j))

         ENDDO
         ENDDO

     ENDDO
       k = ktf
         DO j = j_start, j_end
         DO i = i_start, i_end

            vfluxm(i,j)=vfluxp(i,j)
            vfluxp(i,j)=0.
            rcoup=0.5*(rr(i,k,j)+rr(i,k,j-1))
            tendency(i,k,j)=tendency(i,k,j)+rcoup*zz(i,j)*rdzw(k)*(vfluxp(i,j)-vfluxm(i,j))

         ENDDO
         ENDDO

   ELSE IF (name .EQ. 'w')THEN

   i_start = its
   i_end   = MIN(ite,ide-1)
   j_start = jts
   j_end   = MIN(jte,jde-1)

j_loop_w : DO j = j_start, j_end

     DO k=kts,ktf-1
       DO i = i_start, i_end
          vflux(i,k)=kvdif*zeta_z(i,j)*rdzw(k)*(field(i,k+1,j)-field(i,k,j))
       ENDDO
     ENDDO

     DO i = i_start, i_end
       vflux(i,0)=vflux(i,1)
     ENDDO

     DO i = i_start, i_end
       vflux(i,ktf)=0.
     ENDDO

     DO k=kts+1,ktf
       DO i = i_start, i_end
            rdz=1./(z(i,k,j)-z(i,k-1,j))
            rcoup=0.5*(rr(i,k,j)+rr(i,k-1,j))
            tendency(i,k,j)=tendency(i,k,j)+rcoup*rdz*(vflux(i,k)-vflux(i,k-1))
       ENDDO
     ENDDO

    ENDDO j_loop_w

   ELSE 

     i_start = its
     i_end   = MIN(ite,ide-1)
     j_start = jts
     j_end   = MIN(jte,jde-1)

j_loop_s : DO j = j_start, j_end

     DO k=kts,ktf-1
       DO i = i_start, i_end
         rdz=1./(z(i,k+1,j)-z(i,k,j))
         vflux(i,k)=kvdif*rdz*(field(i,k+1,j)-field(i,k,j))
       ENDDO
     ENDDO

     DO i = i_start, i_end
       vflux(i,0)=vflux(i,1)
     ENDDO

     DO i = i_start, i_end
       vflux(i,ktf)=0.
     ENDDO

     DO k=kts,ktf-1
       DO i = i_start, i_end
         rcoup=rr(i,k,j)
         tendency(i,k,j)=tendency(i,k,j)+rcoup*zeta_z(i,j)   &
                *rdzw(k)*(vflux(i,k)-vflux(i,k-1))
       ENDDO
     ENDDO

 ENDDO j_loop_s

   ENDIF

END SUBROUTINE vertical_diffusion_1


!-------------------------------------------------------------------------------


SUBROUTINE vertical_diffusion ( name, field, tendency, rr,    &
                                config_flags,                     &
                                z, zeta_z, rdzw, kvdif,       &
                                ids, ide, jds, jde, kds, kde, &
                                ims, ime, jms, jme, kms, kme, &
                                its, ite, jts, jte, kts, kte )


   IMPLICIT NONE
   
   ! Input data
   
   TYPE(grid_config_rec_type), INTENT(IN   ) :: config_flags

   INTEGER ,    INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                 ims, ime, jms, jme, kms, kme, &
                                 its, ite, jts, jte, kts, kte

   CHARACTER(LEN=1) ,                          INTENT(IN   ) :: name

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) ,                      &
                                               INTENT(IN   ) :: field,    &
                                                                rr,       &
                                                                z

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(INOUT) :: tendency

   REAL , DIMENSION( ims:ime , jms:jme ) ,         INTENT(IN   ) :: zeta_z

   REAL , DIMENSION( kms:kme ) ,                 INTENT(IN   ) :: rdzw

   REAL ,                                      INTENT(IN   ) :: kvdif
   
   ! Local data
   
   INTEGER :: i, j, k, itf, jtf, ktf
   INTEGER :: i_start, i_end, j_start, j_end

   REAL , DIMENSION(its:ite, jts:jte) :: vfluxm, vfluxp, zz
   REAL , DIMENSION(its:ite, 0:kte+1) :: vflux

   REAL :: rdz, rcoup

   ktf=MIN(kte,kde-1)
   
   IF      (name .EQ. 'u')THEN

   write(6,*) ' loop order not changed in subroutine vertical diffusion for u '

      i_start = its
      i_end   = ite
      j_start = jts
      j_end   = MIN(jte,jde-1)

      IF ( config_flags%open_xs .or. config_flags%specified .or. &
           config_flags%nested) i_start = MAX(ids+1,its)
      IF ( config_flags%open_xe .or. config_flags%specified .or. &
           config_flags%nested) i_end   = MIN(ide-1,ite)

         k = 1
         DO j = j_start, j_end
         DO i = i_start, i_end

            zz(i,j)=0.5*(zeta_z(i,j)+zeta_z(i-1,j))
            rdz=2./((z(i,k+1,j)-z(i,k,j))+(z(i-1,k+1,j)-z(i-1,k,j)))
            vfluxp(i,j)=kvdif*rdz*(field(i,k+1,j)-field(i,k,j))
            vfluxm(i,j)=vfluxp(i,j)
            rcoup=0.5*(rr(i,k,j)+rr(i-1,k,j))
            tendency(i,k,j)=tendency(i,k,j)+rcoup*zz(i,j)*rdzw(k)*(vfluxp(i,j)-vfluxm(i,j))

         ENDDO
         ENDDO
     DO k=kts+1,ktf-1  
         DO j = j_start, j_end
         DO i = i_start, i_end

            vfluxm(i,j)=vfluxp(i,j)
            rdz=2./((z(i,k+1,j)-z(i,k,j))+(z(i-1,k+1,j)-z(i-1,k,j)))
            vfluxp(i,j)=kvdif*rdz*(field(i,k+1,j)-field(i,k,j))
            rcoup=0.5*(rr(i,k,j)+rr(i-1,k,j))
            tendency(i,k,j)=tendency(i,k,j)+rcoup*zz(i,j)*rdzw(k)*(vfluxp(i,j)-vfluxm(i,j))

         ENDDO
         ENDDO

     ENDDO
       k = ktf
         DO j = j_start, j_end
         DO i = i_start, i_end

            vfluxm(i,j)=vfluxp(i,j)
            vfluxp(i,j)=0.
            rcoup=0.5*(rr(i,k,j)+rr(i-1,k,j))
            tendency(i,k,j)=tendency(i,k,j)+rcoup*zz(i,j)*rdzw(k)*(vfluxp(i,j)-vfluxm(i,j))

         ENDDO
         ENDDO
   
   ELSE IF (name .EQ. 'v')THEN

   write(6,*) ' loop order not changed in subroutine vertical diffusion for v '

      i_start = its
      i_end   = MIN(ite,ide-1)
      j_start = jts
      j_end   = jte

      IF ( config_flags%open_ys .or. config_flags%specified .or. &
           config_flags%nested) j_start = MAX(jds+1,jts)
      IF ( config_flags%open_ye .or. config_flags%specified .or. &
           config_flags%nested) j_end   = MIN(jde-1,jte)

       k = 1
         DO j = j_start, j_end
         DO i = i_start, i_end

            zz(i,j)=0.5*(zeta_z(i,j)+zeta_z(i,j-1))
            rdz=2./((z(i,k+1,j)-z(i,k,j))+(z(i,k+1,j-1)-z(i,k,j-1)))
            vfluxp(i,j)=kvdif*rdz*(field(i,k+1,j)-field(i,k,j))
            vfluxm(i,j)=vfluxp(i,j)
            rcoup=0.5*(rr(i,k,j)+rr(i,k,j-1))
            tendency(i,k,j)=tendency(i,k,j)+rcoup*zz(i,j)*rdzw(k)*(vfluxp(i,j)-vfluxm(i,j))

         ENDDO
         ENDDO
     DO k=kts+1,ktf-1
         DO j = j_start, j_end
         DO i = i_start, i_end

            vfluxm(i,j)=vfluxp(i,j)
            rdz=2./((z(i,k+1,j)-z(i,k,j))+(z(i,k+1,j-1)-z(i,k,j-1)))
            vfluxp(i,j)=kvdif*rdz*(field(i,k+1,j)-field(i,k,j))
            rcoup=0.5*(rr(i,k,j)+rr(i,k,j-1))
            tendency(i,k,j)=tendency(i,k,j)+rcoup*zz(i,j)*rdzw(k)*(vfluxp(i,j)-vfluxm(i,j))

         ENDDO
         ENDDO

     ENDDO
       k = ktf
         DO j = j_start, j_end
         DO i = i_start, i_end

            vfluxm(i,j)=vfluxp(i,j)
            vfluxp(i,j)=0.
            rcoup=0.5*(rr(i,k,j)+rr(i,k,j-1))
!CJM            tendency(i,j,k)=tendency(i,j,k)+rcoup*zz(i,j)*rdzw(k)*(vfluxp(i,j)-vfluxm(i,j))
            tendency(i,k,j)=tendency(i,k,j)+rcoup*zz(i,j)*rdzw(k)*(vfluxp(i,j)-vfluxm(i,j))

         ENDDO
         ENDDO

   ELSE IF (name .EQ. 'w')THEN

   
   i_start = its
   i_end   = MIN(ite,ide-1)
   j_start = jts
   j_end   = MIN(jte,jde-1)

j_loop_w : DO j = j_start, j_end

     DO k=kts,ktf-1
       DO i = i_start, i_end
          vflux(i,k)=kvdif*zeta_z(i,j)*rdzw(k)*(field(i,k+1,j)-field(i,k,j))
       ENDDO
     ENDDO

!     DO i = i_start, i_end
!       vflux(i,0)=vflux(i,1)
!     ENDDO

     DO i = i_start, i_end
       vflux(i,ktf)=0.
     ENDDO

     DO k=kts+1,ktf
       DO i = i_start, i_end
            rdz=1./(z(i,k,j)-z(i,k-1,j))
            rcoup=0.5*(rr(i,k,j)+rr(i,k-1,j))
            tendency(i,k,j)=tendency(i,k,j)+rcoup*rdz*(vflux(i,k)-vflux(i,k-1))
       ENDDO
     ENDDO

    ENDDO j_loop_w

   ELSE 

     i_start = its
     i_end   = MIN(ite,ide-1)
     j_start = jts
     j_end   = MIN(jte,jde-1)

j_loop_s : DO j = j_start, j_end

     DO k=kts,ktf-1
       DO i = i_start, i_end
         rdz=1./(z(i,k+1,j)-z(i,k,j))
         vflux(i,k)=kvdif*rdz*(field(i,k+1,j)-field(i,k,j))
       ENDDO
     ENDDO

     DO i = i_start, i_end
       vflux(i,0)=vflux(i,1)
     ENDDO

     DO i = i_start, i_end
       vflux(i,ktf)=0.
     ENDDO

     DO k=kts,ktf
       DO i = i_start, i_end
         rcoup=rr(i,k,j)
         tendency(i,k,j)=tendency(i,k,j)+rcoup*zeta_z(i,j)   &
                *rdzw(k)*(vflux(i,k)-vflux(i,k-1))
       ENDDO
     ENDDO

 ENDDO j_loop_s

   ENDIF

END SUBROUTINE vertical_diffusion


!-------------------------------------------------------------------------------

SUBROUTINE vertical_diffusion_qv ( name, field, tendency, rr,    &
                                   config_flags, base,               &
                                   z, zeta_z, rdzw, kvdif,       &
                                   ids, ide, jds, jde, kds, kde, &
                                   ims, ime, jms, jme, kms, kme, &
                                   its, ite, jts, jte, kts, kte )


   IMPLICIT NONE
   
   ! Input data
   
   TYPE(grid_config_rec_type), INTENT(IN   ) :: config_flags

   INTEGER ,    INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                 ims, ime, jms, jme, kms, kme, &
                                 its, ite, jts, jte, kts, kte

   CHARACTER(LEN=1) ,                          INTENT(IN   ) :: name

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) ,                      &
                                               INTENT(IN   ) :: field,    &
                                                                rr,       &
                                                                z

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(INOUT) :: tendency

   REAL , DIMENSION( ims:ime , jms:jme ) ,         INTENT(IN   ) :: zeta_z

   REAL , DIMENSION( kms:kme ) ,                 INTENT(IN   ) :: rdzw, &
                                                                  base

   REAL ,                                      INTENT(IN   ) :: kvdif
   
   ! Local data
   
   INTEGER :: i, j, k, itf, jtf, ktf
   INTEGER :: i_start, i_end, j_start, j_end

   REAL , DIMENSION(its:ite, 0:kte+1) :: vflux

   REAL :: rdz, rcoup

   ktf=MIN(kte,kde-1)
   
     i_start = its
     i_end   = MIN(ite,ide-1)
     j_start = jts
     j_end   = MIN(jte,jde-1)

j_loop_s : DO j = j_start, j_end

     DO k=kts,ktf-1
       DO i = i_start, i_end
         rdz=1./(z(i,k+1,j)-z(i,k,j))
         vflux(i,k)=kvdif*rdz*(field(i,k+1,j)-field(i,k,j)   &
                                    -base(k+1)    +base(k)  )
       ENDDO
     ENDDO

     DO i = i_start, i_end
       vflux(i,0)=vflux(i,1)
     ENDDO

     DO i = i_start, i_end
       vflux(i,ktf)=0.
     ENDDO

     DO k=kts,ktf
       DO i = i_start, i_end
         rcoup=rr(i,k,j)
         tendency(i,k,j)=tendency(i,k,j)+rcoup*zeta_z(i,j)   &
                *rdzw(k)*(vflux(i,k)-vflux(i,k-1))
       ENDDO
     ENDDO

 ENDDO j_loop_s

END SUBROUTINE vertical_diffusion_qv


!-------------------------------------------------------------------------------


SUBROUTINE vertical_diffusion_u ( field, tendency, rr,          &
                                  config_flags, u_base,             &
                                  z, zeta_z, rdzw, kvdif,       &
                                  ids, ide, jds, jde, kds, kde, &
                                  ims, ime, jms, jme, kms, kme, &
                                  its, ite, jts, jte, kts, kte )


   IMPLICIT NONE
   
   ! Input data
   
   TYPE(grid_config_rec_type), INTENT(IN   ) :: config_flags

   INTEGER ,    INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                 ims, ime, jms, jme, kms, kme, &
                                 its, ite, jts, jte, kts, kte

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) ,                      &
                                               INTENT(IN   ) :: field,    &
                                                                rr,       &
                                                                z

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(INOUT) :: tendency

   REAL , DIMENSION( ims:ime , jms:jme ) ,         INTENT(IN   ) :: zeta_z

   REAL , DIMENSION( kms:kme ) ,                 INTENT(IN   ) :: rdzw, u_base

   REAL ,                                      INTENT(IN   ) :: kvdif
   
   ! Local data
   
   INTEGER :: i, j, k, itf, jtf, ktf
   INTEGER :: i_start, i_end, j_start, j_end

   REAL , DIMENSION(its:ite, 0:kte+1) :: vflux

   REAL :: rdz, rcoup, zz

   ktf=MIN(kte,kde-1)

      i_start = its
      i_end   = ite
      j_start = jts
      j_end   = MIN(jte,jde-1)

      IF ( config_flags%open_xs .or. config_flags%specified .or. &
           config_flags%nested) i_start = MAX(ids+1,its)
      IF ( config_flags%open_xe .or. config_flags%specified .or. &
           config_flags%nested) i_end   = MIN(ide-1,ite)


j_loop_u : DO j = j_start, j_end

     DO k=kts,ktf-1
       DO i = i_start, i_end
         rdz=2./((z(i,k+1,j)-z(i,k,j))+(z(i-1,k+1,j)-z(i-1,k,j)))
         vflux(i,k)=kvdif*rdz*(field(i,k+1,j)-field(i,k,j)   &
                               -u_base(k+1)   +u_base(k)  )
       ENDDO
     ENDDO

     DO i = i_start, i_end
       vflux(i,0)=vflux(i,1)
     ENDDO

     DO i = i_start, i_end
       vflux(i,ktf)=0.
     ENDDO

     DO k=kts,ktf-1
       DO i = i_start, i_end
         rcoup=0.5*(rr(i,k,j)+rr(i-1,k,j))
         zz = 0.5*(zeta_z(i,j)+zeta_z(i-1,j))
         tendency(i,k,j)=tendency(i,k,j)+rcoup*zz   &
                *rdzw(k)*(vflux(i,k)-vflux(i,k-1))
       ENDDO
     ENDDO

 ENDDO j_loop_u
   
END SUBROUTINE vertical_diffusion_u

!-------------------------------------------------------------------------------


SUBROUTINE vertical_diffusion_v ( field, tendency, rr,          &
                                  config_flags, v_base,             &
                                  z, zeta_z, rdzw, kvdif,       &
                                  ids, ide, jds, jde, kds, kde, &
                                  ims, ime, jms, jme, kms, kme, &
                                  its, ite, jts, jte, kts, kte )


   IMPLICIT NONE
   
   ! Input data
   
   TYPE(grid_config_rec_type), INTENT(IN   ) :: config_flags

   INTEGER ,    INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                 ims, ime, jms, jme, kms, kme, &
                                 its, ite, jts, jte, kts, kte

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) ,                      &
                                               INTENT(IN   ) :: field,    &
                                                                rr,       &
                                                                z

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(INOUT) :: tendency

   REAL , DIMENSION( ims:ime , jms:jme ) ,         INTENT(IN   ) :: zeta_z

   REAL , DIMENSION( kms:kme ) ,                 INTENT(IN   ) :: rdzw, v_base

   REAL ,                                      INTENT(IN   ) :: kvdif
   
   ! Local data
   
   INTEGER :: i, j, k, itf, jtf, ktf
   INTEGER :: i_start, i_end, j_start, j_end

   REAL , DIMENSION(its:ite, 0:kte+1) :: vflux

   REAL :: rdz, rcoup, zz

   ktf=MIN(kte,kde-1)
   
      i_start = its
      i_end   = MIN(ite,ide-1)
      j_start = jts
      j_end   = jte

      IF ( config_flags%open_ys .or. config_flags%specified .or. &
           config_flags%nested) j_start = MAX(jds+1,jts)
      IF ( config_flags%open_ye .or. config_flags%specified .or. &
           config_flags%nested) j_end   = MIN(jde-1,jte)

j_loop_v : DO j = j_start, j_end

     DO k=kts,ktf-1
       DO i = i_start, i_end
         rdz=2./((z(i,k+1,j)-z(i,k,j))+(z(i,k+1,j-1)-z(i,k,j-1)))
         vflux(i,k)=kvdif*rdz*(field(i,k+1,j)-field(i,k,j)   &
                               -v_base(k+1)   +v_base(k)  )
       ENDDO
     ENDDO

     DO i = i_start, i_end
       vflux(i,0)=vflux(i,1)
     ENDDO

     DO i = i_start, i_end
       vflux(i,ktf)=0.
     ENDDO

     DO k=kts,ktf-1
       DO i = i_start, i_end
         rcoup=0.5*(rr(i,k,j)+rr(i,k,j-1))
         zz = 0.5*(zeta_z(i,j)+zeta_z(i,j-1))
         tendency(i,k,j)=tendency(i,k,j)+rcoup*zz   &
                *rdzw(k)*(vflux(i,k)-vflux(i,k-1))
       ENDDO
     ENDDO

 ENDDO j_loop_v
   
END SUBROUTINE vertical_diffusion_v

!-------------------------------------------------------------------------------


SUBROUTINE zero_tend ( tendency,                     &
                       ids, ide, jds, jde, kds, kde, &
                       ims, ime, jms, jme, kms, kme, &
                       its, ite, jts, jte, kts, kte )


   IMPLICIT NONE
   
   ! Input data
   
   INTEGER ,                                   INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                                                ims, ime, jms, jme, kms, kme, &
                                                                its, ite, jts, jte, kts, kte

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) , INTENT(INOUT) :: tendency

   ! Local data
   
   INTEGER :: i, j, k, itf, jtf, ktf

      DO j = jts, jte
      DO k = kts, kte
      DO i = its, ite
        tendency(i,k,j) = 0.
      ENDDO
      ENDDO
      ENDDO

      END SUBROUTINE zero_tend

!======================================================================
   SUBROUTINE phy_prep ( config_flags, rr, rth , th_phy ,             &
                         moist , p_phy , pi_phy , pb, pb8w,           & 
                         u_phy, v_phy, u, v, msft,                    &
                         RHOBASE, p8w, dz8w, t_phy, t8w,              &
                         dzetaw, zeta_z, zeta, zetaw, TSK,            &
                         fzm, fzp, n_moist,                           &
                         ids, ide, jds, jde, kds, kde,                &
                         ims, ime, jms, jme, kms, kme,                &
                         its, ite, jts, jte, kts, kte                 )
!----------------------------------------------------------------------
   IMPLICIT NONE
!----------------------------------------------------------------------

   TYPE(grid_config_rec_type) ,     INTENT(IN   ) :: config_flags

   INTEGER ,        INTENT(IN   ) ::   ids, ide, jds, jde, kds, kde, &
                                       ims, ime, jms, jme, kms, kme, &
                                       its, ite, jts, jte, kts, kte

   INTEGER ,       INTENT(INOUT) ::    n_moist

   REAL , DIMENSION( ims:ime, kms:kme ,jms:jme ), INTENT(INOUT) ::   &
						            RHOBASE

   REAL , DIMENSION( ims:ime, jms:jme ), INTENT(IN   )   ::     TSK

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) ,                 &
          INTENT(INOUT)                                  ::     rth, &
                                                             th_phy, &
                                                              u_phy, &
                                                              v_phy, &
                                                             pi_phy, &
                                                              p_phy, &
                                                                p8w, &
                                                              t_phy, &
                                                                t8w, &
                                                               dz8w

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) ,                 &
          INTENT(IN   )                                  ::      rr, &
                                                                  u, &
                                                                  v, &
                                                                 pb, &
                                                               pb8w


   REAL , DIMENSION( ims:ime, kms:kme, jms:jme, n_moist ),           &
          INTENT(IN)                                     ::   moist

   REAL , DIMENSION( ims:ime , jms:jme ) , INTENT(IN   ) ::    msft, &
                                                             zeta_z

   REAL , DIMENSION( kms:kme ) ,           INTENT(IN   ) ::     fzm,   &
                                                                fzp,   &
                                                             dzetaw,   &
                                                               zeta,   &
                                                              zetaw

   INTEGER :: i_start, i_end, j_start, j_end, k_start, k_end
   INTEGER :: i, j, k

!-----------------------------------------------------------------------

!  set up loop bounds for this grid's boundary conditions

    i_start = its
    i_end   = min( ite,ide-1 )
    j_start = jts
    j_end   = min( jte,jde-1 )

    IF( config_flags%nested .or. config_flags%specified ) THEN
      i_start = max( its,ids+1 )
      i_end   = min( ite,ide-2 )
      j_start = max( jts,jds+1 )
      j_end   = min( jte,jde-2 )
    ENDIF

! for dimention of ice phases

    k_start = kts
    k_end = min( kte, kde-1 )


   !  Decouple variables and compute diagnostics
   !  at the current time level.

   ! get theta*(1+rvovrd*qv) from rho_d*theta*(1+rvovrd*qv) for advection
   ! and for pressure calculation

     CALL decouple_thm ( rr, rth, th_phy,               &
                         config_flags, moist, fzm, fzp, &
                         n_moist,                       &
                         ids, ide, jds, jde, kds, kde,  &
                         ims, ime, jms, jme, kms, kme,  &
                         its, ite, jts, jte, kts, kte   )

     CALL calculate_p ( rth, pi_phy, p_phy,              &
                      msft, zeta_z,                     &
                      ids, ide, jds, jde, kds, kde,     &
                      ims, ime, jms, jme, kms, kme,     &
                      its, ite, jts, jte, kts, kte      )

     CALL c2agrid_u ( u, u_phy,                         &
                      ids, ide, jds, jde, kds, kde,     &
                      ims, ime, jms, jme, kms, kme,     &
                      its, ite, jts, jte, kts, kte      )

     CALL c2agrid_v ( v, v_phy,                         &
                      ids, ide, jds, jde, kds, kde,     &
                      ims, ime, jms, jme, kms, kme,     &
                      its, ite, jts, jte, kts, kte      )

     CALL calculate_p8w ( p8w, p_phy,pb,pb8w,zeta,      &
                      zetaw,dzetaw,                     &
                      ids, ide, jds, jde, kds, kde,     &
                      ims, ime, jms, jme, kms, kme,     &
                      its, ite, jts, jte, kts, kte      )

     CALL calculate_temp ( t_phy, pi_phy, th_phy,       &
                      ids, ide, jds, jde, kds, kde,     &
                      ims, ime, jms, jme, kms, kme,     &
                      its, ite, jts, jte, kts, kte      )

     CALL calculate_t8w ( t8w, t_phy, TSK, zeta,        &
                      zetaw,dzetaw,                     &
                      ids, ide, jds, jde, kds, kde,     &
                      ims, ime, jms, jme, kms, kme,     &
                      its, ite, jts, jte, kts, kte      )

     CALL calculate_dz8w ( dz8w, zeta_z, dzetaw,        &
                      ids, ide, jds, jde, kds, kde,     &
                      ims, ime, jms, jme, kms, kme,     &
                      its, ite, jts, jte, kts, kte      )
   
     CALL calculate_rb ( RHOBASE, rr, msft, zeta_z,     &
                      ids, ide, jds, jde, kds, kde,     &
                      ims, ime, jms, jme, kms, kme,     &
                      its, ite, jts, jte, kts, kte      )

END SUBROUTINE phy_prep

!------------------------------------------------------------
SUBROUTINE add_a2a(lvar,rvar,config_flags,                  &
                   ids,ide, jds, jde, kds, kde,             &
                   ims, ime, jms, jme, kms, kme,            &
                   its, ite, jts, jte, kts, kte             )
!------------------------------------------------------------
   IMPLICIT NONE
!------------------------------------------------------------
   TYPE(grid_config_rec_type),  INTENT(IN) :: config_flags

   INTEGER , INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                              ims, ime, jms, jme, kms, kme, &
                              its, ite, jts, jte, kts, kte

   REAL, DIMENSION(ims:ime,kms:kme,jms:jme),INTENT(IN   ) ::&
                                                      rvar
   REAL, DIMENSION(ims:ime,kms:kme,jms:jme),INTENT(INOUT) ::&
                                                      lvar

! LOCAL VARS
   INTEGER :: i,j,k,i_start,i_end,j_start,j_end,ktf

   i_start = its
   i_end   = MIN(ite,ide-1)
   j_start = jts
   j_end   = MIN(jte,jde-1)
   ktf = min(kte,kde-1)

   IF ( config_flags%open_xs .or. config_flags%specified .or. &
        config_flags%nested) i_start = MAX(ids+1,its)
   IF ( config_flags%open_xe .or. config_flags%specified .or. &
        config_flags%nested) i_end   = MIN(ide-2,ite)
   IF ( config_flags%open_ys .or. config_flags%specified .or. &
        config_flags%nested) j_start = MAX(jds+1,jts)
   IF ( config_flags%open_ye .or. config_flags%specified .or. &
        config_flags%nested) j_end   = MIN(jde-2,jte)

   DO j = j_start,j_end
   DO k = kts,ktf
   DO i = i_start,i_end
      lvar(i,k,j) = lvar(i,k,j) + rvar(i,k,j)
   ENDDO
   ENDDO
   ENDDO

END SUBROUTINE add_a2a

!------------------------------------------------------------
SUBROUTINE add_a2c_u(lvar,rvar,config_flags,                &
                   ids,ide, jds, jde, kds, kde,             &
                   ims, ime, jms, jme, kms, kme,            &
                   its, ite, jts, jte, kts, kte             )
!------------------------------------------------------------
   USE module_bc
!------------------------------------------------------------
   IMPLICIT NONE
!------------------------------------------------------------

   TYPE(grid_config_rec_type),  INTENT(IN   )    :: config_flags

   INTEGER , INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                              ims, ime, jms, jme, kms, kme, &
                              its, ite, jts, jte, kts, kte

   REAL, DIMENSION(ims:ime,kms:kme,jms:jme),INTENT(IN   ) ::&
                                                      rvar
   REAL, DIMENSION(ims:ime,kms:kme,jms:jme),INTENT(INOUT) ::&
                                                      lvar

! LOCAL VARS

   INTEGER :: i,j,k,i_start,i_end,j_start,j_end,ktf

   ktf=min(kte,kde-1)

   i_start = its
   i_end   = ite
   j_start = jts
   j_end   = MIN(jte,jde-1)

   IF ( config_flags%open_xs .or. config_flags%specified .or. &
        config_flags%nested) i_start = MAX(ids+1,its)
   IF ( config_flags%open_xe .or. config_flags%specified .or. &
        config_flags%nested) i_end   = MIN(ide-1,ite)
   IF ( config_flags%open_ys .or. config_flags%specified .or. &
        config_flags%nested) j_start = MAX(jds+1,jts)
   IF ( config_flags%open_ye .or. config_flags%specified .or. &
        config_flags%nested) j_end   = MIN(jde-2,jte)

   DO j = j_start,j_end
   DO k = kts,ktf
   DO i = i_start,i_end
      lvar(i,k,j) = lvar(i,k,j) + &
                       0.5*(rvar(i,k,j)+rvar(i-1,k,j))
   ENDDO
   ENDDO
   ENDDO

END SUBROUTINE add_a2c_u

!------------------------------------------------------------
SUBROUTINE add_a2c_v(lvar,rvar,config_flags,                &
                   ids,ide, jds, jde, kds, kde,             &
                   ims, ime, jms, jme, kms, kme,            &
                   its, ite, jts, jte, kts, kte             )
!------------------------------------------------------------
   USE module_bc
!------------------------------------------------------------
   IMPLICIT NONE
!------------------------------------------------------------

   TYPE(grid_config_rec_type),  INTENT(IN   )    :: config_flags

   INTEGER , INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                              ims, ime, jms, jme, kms, kme, &
                              its, ite, jts, jte, kts, kte

   REAL, DIMENSION(ims:ime,kms:kme,jms:jme),INTENT(IN   ) ::&
                                                      rvar
   REAL, DIMENSION(ims:ime,kms:kme,jms:jme),INTENT(INOUT) ::&
                                                      lvar

! LOCAL VARS

   INTEGER :: i,j,k,i_start,i_end,j_start,j_end,ktf

   ktf=min(kte,kde-1)

   i_start = its
   i_end   = MIN(ite,ide-1)
   j_start = jts
   j_end   = jte

   IF ( config_flags%open_xs .or. config_flags%specified .or. &
        config_flags%nested) i_start = MAX(ids+1,its)
   IF ( config_flags%open_xe .or. config_flags%specified .or. &
        config_flags%nested) i_end   = MIN(ide-2,ite)
   IF ( config_flags%open_ys .or. config_flags%specified .or. &
        config_flags%nested) j_start = MAX(jds+1,jts)
   IF ( config_flags%open_ye .or. config_flags%specified .or. &
        config_flags%nested) j_end   = MIN(jde-1,jte)

   DO j = j_start,j_end
   DO k = kts,kte
   DO i = i_start,i_end
      lvar(i,k,j) = lvar(i,k,j) + &
                     0.5*(rvar(i,k,j)+rvar(i,k,j-1))
   ENDDO
   ENDDO
   ENDDO

END SUBROUTINE add_a2c_v


   SUBROUTINE init_module_big_step
   END SUBROUTINE init_module_big_step

END MODULE module_big_step_utilities
