!IDEAL:MODEL_LAYER:INITIALIZATION
!

!  This MODULE holds the routines which are used to perform various initializations
!  for the individual domains.  

!  This MODULE CONTAINS the following routines:

!  initialize_field_test - 1. Set different fields to different constant
!                             values.  This is only a test.  If the correct
!                             domain is not found (based upon the "id")
!                             then a fatal error is issued.               

MODULE module_initialize

   USE module_domain
!  USE module_io_domain
   USE module_state_description
   USE module_model_constants
   USE module_bc
   USE module_timing
   USE module_configure
#ifdef DM_PARALLEL
   USE module_dm
#endif


CONTAINS

!
!  This routine provides a test initialization for a specific domain.  The
!  input requests which "id" to choose (this is compared with the
!  "id" stored in every domain).  When the correct domain is found,
!  the test data are initialized.  If no domain is found with the required
!  "id", a fatal error is issued.

!  The only argument is the input of the "id" to which all processing
!  refers.  On output, the domain referenced by this "id" has all of
!  its fields initialized, but only in a testing sort of way.


   SUBROUTINE init_coords ( grid,                                     &
                            ht, rdx, rdy, zeta, zetaw, dzetaw, dzeta, &
                            rdzu, rdzw, z, zx, zy, &
                            z_zeta, zeta_z, fzm, fzp, ztop, &
                            cf1, cf2, cf3, &
                            cofrz, cofwr, dtseps,  &
                            ids, ide, jds, jde, kds, kde, &
                            ims, ime, jms, jme, kms, kme, &
                            its, ite, jts, jte, kts, kte )

   USE module_domain

   IMPLICIT NONE

   ! Input data

   TYPE (domain ), POINTER              :: grid

   INTEGER , INTENT(IN)                    :: ids, ide, jds, jde, kds, kde, &
                                              ims, ime, jms, jme, kms, kme, &
                                              its, ite, jts, jte, kts, kte
   REAL , DIMENSION( ims: , jms: ) , INTENT(IN) :: ht
   REAL , INTENT(IN) :: rdx, rdy, dtseps
   REAL , DIMENSION(ims: , kms: , jms: ) , INTENT(OUT) :: z, zx, zy
   REAL , DIMENSION(ims: , jms: ) , INTENT(OUT) :: z_zeta, zeta_z, cofwr
   ! WRF state 1d_constant
   REAL , DIMENSION( kms: ) , INTENT(OUT) :: zeta, zetaw, dzetaw, dzeta, rdzu, rdzw, fzm, fzp, cofrz
   ! WRF state data
   REAL , INTENT(OUT) :: cf1, cf2, cf3
   REAL , INTENT(IN) :: ztop

   ! Local data

   INTEGER               :: itf, jtf, ktf, i, j, k
   REAL                  :: cof1, cof2

   REAL , DIMENSION(ids:ide,kms:kme,jds:jde) :: zw

   ! Executable

   itf=min0(ite,ide-1)
   jtf=min0(jte,jde-1)
   ktf=min0(kte,kde-1)

   !________________________________________________________________________________
   !________________________________________________________________________________
   ! USER-DEFINED PART - SET DZETA OR ZETAW
   !
!   DO k=kts,ktf
!      dzetaw(k)=100.*float(k) ! arbitrary stretched function of k
!   ENDDO
   !________________________________________________________________________________
   DO k=kts,kte
      dzetaw(k)=ztop/(kte-kts)
   ENDDO
   !________________________________________________________________________________
   !________________________________________________________________________________
   zetaw(1)=0.
   zeta(1)=0.5*dzetaw(1)
   DO k=2,ktf
     zetaw(k)=zetaw(k-1)+dzetaw(k-1)
     zeta(k)=zeta(k-1)+0.5*(dzetaw(k)+dzetaw(k-1))
     dzeta(k)=zeta(k)-zeta(k-1)
   ENDDO
   zetaw(kte)=zetaw(ktf)+dzetaw(ktf)
   !________________________________________________________________________________
   !!!k=1
   !!!zeta(k)=0.5*zetaw(k+1)
   !!!DO k=kts+1,ktf
   !!!   dzetaw(k)=zetaw(k)-zetaw(k-1)
   !!!   zeta(k)=zeta(k-1)+0.5*(dzetaw(k)+dzetaw(k-1))
   !!!   dzeta(k)=zeta(k)-zeta(k-1)
   !!!ENDDO
   !!!zetatop=zetaw(kde)
   !________________________________________________________________________________
   !
   z = 0.
   zw = 0.
   DO k=1,ktf
      rdzw(k)=1./dzetaw(k)
      cofrz(k) = dtseps * rdzw(k)
   ENDDO
   DO k=2,ktf
      rdzu(k)=1./dzeta(k)
      fzp(k)=0.5*dzetaw(k  )*rdzu(k)
      fzm(k)=0.5*dzetaw(k-1)*rdzu(k)
   ENDDO
   DO i=max(ids,its-1),itf
     DO j=max(jds,jts-1),jtf
       z_zeta(i,j)=(ztop-ht(i,j))/ztop
       zeta_z(i,j)=ztop/(ztop-ht(i,j))
       cofwr(i,j)=.5 * dtseps * g * zeta_z(i,j)
       DO k=1,ktf
          z(i,k,j)=ht(i,j)+z_zeta(i,j)*zeta(k)
          zw(i,k,j)=ht(i,j)+z_zeta(i,j)*zetaw(k)
   ! map-scale factor not needed here because it is accounted for in full pgf
   !   and omega terms (i.e. z_x is relative to coordinate x not real x)
   ! this still assumes that terrain slope is zero at physical boundaries
   !  (may not be general enough for periodic bc's)
          if(i.ne.ids) then
               zx(i,k,j)=(zw(i,k,j)-zw(i-1,k,j))*rdx
          else
               zx(i,k,j) = 0.
          end if
          if(j.ne.jds) then
               zy(i,k,j)=(zw(i,k,j)-zw(i,k,j-1))*rdy
          else
               zy(i,k,j) = 0.
          end if
       ENDDO
     ENDDO
   ENDDO

   IF ( ite == ide ) THEN
     DO j=jts,jtf
     DO k=1,ktf
      zx(ide,k,j) = 0.
     ENDDO
     ENDDO
   ENDIF

   IF ( jte == jde ) THEN
     DO i=its,itf
     DO k=1,ktf
      zy(i,k,jde) = 0.
     ENDDO
     ENDDO
   ENDIF


   cof1 = (2. * dzeta(2) + dzeta(3))/(dzeta(2) + dzeta(3)) &
                        * dzetaw(1)/dzeta(2)
   cof2 =       dzeta(2)             /(dzeta(2) + dzeta(3)) &
                        * dzetaw(1)/dzeta(3)

   cf1  = fzp(2) + cof1
   cf2  = fzm(2) - cof1 - cof2
   cf3  = cof2

   END SUBROUTINE init_coords
!-----------------------------------------------------------------------------


   SUBROUTINE init_base_state ( grid ,                               &
                                rrb, rtb, pb, pib, pb8w, h_diabatic, &
                                msft, TSK, TMN, dzetaw, z, ht, z_zeta,    &
                                u_frame, v_frame,                    &
                                ids, ide, jds, jde, kds, kde,        &
                                ims, ime, jms, jme, kms, kme,        &
                                its, ite, jts, jte, kts, kte        )

   USE module_domain

   IMPLICIT NONE

   ! Input data
   TYPE (domain), POINTER           ::  grid
   INTEGER , INTENT(IN)                    :: ids, ide, jds, jde, kds, kde, &
                                              ims, ime, jms, jme, kms, kme, &
                                              its, ite, jts, jte, kts, kte

   REAL , DIMENSION( ims: , kms: , jms: ) , INTENT(OUT) :: rrb, rtb, pb, pib, pb8w, &
                                                           h_diabatic
   REAL , DIMENSION( ims: , jms: ) , INTENT(INOUT) :: TSK, TMN
   REAL , DIMENSION( ims: , jms: ) , INTENT(IN) :: msft, ht, z_zeta
   REAL                            , INTENT(OUT):: u_frame, v_frame
   REAL , DIMENSION( ims: , kms: , jms: ) , INTENT(IN) :: z
   REAL , DIMENSION( kms: ) , INTENT(IN) :: dzetaw
   
   ! Local data

   REAL , DIMENSION(kms:kme) :: t, p, pi, rho, theta, rhotheta, dz
   REAL :: tslvl, xn2m, tav
   INTEGER :: i, j, k, iter, it, itf, jtf, ktf
   REAL , DIMENSION(1,kme) :: u_tmp, qv_tmp

   REAL :: pi_top, dzk, pitmp, thsfc
   LOGICAL :: integrate_up

   !   constants

!   integrate_up = .false.
   integrate_up = .true.
   iter=10

   itf=min0(ite,ide-1)
   jtf=min0(jte,jde-1)
   ktf=min0(kte,kde-1)
   
   !   can set surface pressure as function of z(i,k,j)

    tslvl=300.
    xn2m=0.0001



   DO i=its,itf
   DO j=jts,jtf

!  initialize the diabatic heating rate array to zero

     DO k=kts,ktf
       h_diabatic(i,k,j) = 0.
     ENDDO


! If we ever tile in the k dimension, this code will need a second look.
   
   !________________________________________________________________________________
   ! USER-DEFINED PART - SET BASE-STATE PROFILE theta(k) 
   !
   ! The base state profile is for a DRY atmosphere, moisture only
   !  appears in the perturbation state.
   !

   !   can set t as function of z(i,k,j)

! init_sound returns the initial state theta, qv and u for the
! standard squall line simulation

     CALL init_sound( theta, thsfc, qv_tmp, z(i,1:ktf,j), ht(i,j), u_tmp, ktf,u_frame,v_frame)
     tslvl = theta(1)

         DO k=kms,kme
!           theta(k)=tslvl*exp(xn2m*z(i,k,j)/g)
           if(k.ne.kms)dz(k)=0.5*(dzetaw(k)+dzetaw(k-1))*z_zeta(i,j)
           if(k.eq.kms)dz(k)=0.5*dzetaw(k)*z_zeta(i,j)
         ENDDO
   !________________________________________________________________________________


 IF ( integrate_up ) THEN


   !   get p, pi, r
         k=1

!       if(xn2m.eq.0.)then
   ! calculate p at lowest level from functional form if xn2m=0
!***         pi(k)=1.-g*z(i,k,j)/(cp*tslvl)
         pi(k)=1.-g*z(i,k,j)/(cp*theta(1))
         pitmp=1.-g*ht(i,j)/(cp*theta(1))
         TSK(I,J)=thsfc*pitmp
         TMN(I,J)=TSK(I,J)-0.5
!       else
   ! could alternatively use the following if xn2m is NOT zero
!         pi(k)=1.+g*g/(cp*xn2m*tslvl)*(exp(-xn2m*z(i,k,j)/g)-1.)
!       endif

         pb8w(i,k,j)=p1000mb*pitmp**(cp/R_d)
 
         p(k)=p1000mb*pi(k)**(cp/R_d)
         rho(k)=p(k)/R_d/(pi(k)*theta(k))
         rhotheta(k)=rho(k)*theta(k)
         DO k=2,ktf
           tav=0.5*(theta(k)+theta(k-1))
           pi(k)=pi(k-1)-g*dz(k)/(cp*tav)
           p(k)=p1000mb*pi(k)**(cp/R_d)
           rho(k)=p(k)/R_d/(pi(k)*theta(k))
           rhotheta(k)=rho(k)*theta(k)
   !!        check=p(k-1)-p(k)-0.5*(rho(k)+rho(k-1))*g*dz(k)
   !!        if(k.eq.10)print *,k,t(k),p(k),rho(k),check
         ENDDO
!
!  wcs - switched k and it loop to increase accuracy
!
         DO k=2,ktf
         DO it=1,iter
           pi(k)=pi(k-1)-2.*g*(z(i,k,j)-z(i,k-1,j))/(cp*(theta(k)+theta(k-1)))
           p(k) = p1000mb*(pi(k)**(1./rcp))
!             p(k)=p(k-1)-0.5*(rho(k)+rho(k-1))*g*dz(k)
!             pi(k)=(p(k)/p1000mb)**(rcp)
           if (it .eq. iter) pitmp=pitmp-g*dzetaw(k-1)*z_zeta(i,j)/(cp*theta(k-1))
           if (it .eq. iter) pb8w(i,k,j) = p1000mb*(pitmp**(1./rcp))
             rho(k)=p(k)/R_d/(pi(k)*theta(k))
             rhotheta(k)=rho(k)*theta(k)
   !!          check=p(k-1)-p(k)-0.5*(rho(k)+rho(k-1))*g*dz(k)
         ENDDO
         ENDDO

         pitmp=pitmp-g*dzetaw(kte-1)*z_zeta(i,j)/(cp*theta(kte-1))
         pb8w(i,kte,j) = p1000mb*(pitmp**(1./rcp))

     DO k=1,ktf
       rtb(i,k,j)=rhotheta(k)*z_zeta(i,j)/msft(i,j)
       rrb(i,k,j)=rho(k)*z_zeta(i,j)/msft(i,j)
       pb(i,k,j)=p(k)
       pib(i,k,j)=pi(k)
     ENDDO

   ELSE

  !  here we are integrating from the top down,
  !  as in joe klemp's model

    IF( (i==its) .and. (j==jts) ) THEN

      ! set the top pressure: this will not work with patched or tiled code

      pi(1)=1.-g*z(1,i,j)/(cp*theta(1))
      pi_top = pi(1)

        do k=2,ktf
          dzk = z(i,k,j)-z(i,k-1,j)
          pi_top = pi_top - 0.5*dzk*(g/cp)*(1/theta(k) + 1/theta(k-1))
        enddo
          dzk = z(i,ktf,j)-z(i,ktf-1,j)
          pi_top = pi_top - 0.5*dzk*(g/cp)*(1/theta(ktf))


    ENDIF

    dzk = z(i,ktf,j)-z(i,ktf-1,j)
    pi(ktf) = pi_top + 0.5*dzk*(g/cp)*(1/theta(ktf))

     DO k=ktf-1,1,-1
       dzk = z(i,k+1,j)-z(i,k,j)
       pi(k) = pi(k+1) + 0.5*dzk*(g/cp)*(1./theta(k+1) + 1./theta(k))
     ENDDO

     DO k=1,ktf
       rho(k) = pi(k)**(1./rcv)/((r_d/p1000mb)*theta(k))
       rhotheta(k) = rho(k)*theta(k)
       p(k) = p1000mb*(pi(k)**(cp/r_d))

       rtb(i,k,j)=rhotheta(k)*z_zeta(i,j)/msft(i,j)
       rrb(i,k,j)=rho(k)*z_zeta(i,j)/msft(i,j)
       pb(i,k,j)=p(k)
       pib(i,k,j)=pi(k)

     ENDDO

   ENDIF

     IF ((i == its) .and. (j == jts)) THEN
       write(6,*) ' initial base state profile, k, z, p, theta, rho '
       do k=1, ktf
         write(6,FMT="(1x,i3,4(1x,1pe10.3))") k,z(i,k,j),p(k),rhotheta(k)/rho(k),rho(k)
       enddo
     ENDIF

   ENDDO
   ENDDO
   
   END SUBROUTINE init_base_state

!------------------------------------------------------------------------------------


   SUBROUTINE init_state ( rrb, rtb, pb,                  &
                           rr, rrp, rthp,                 &
                           rr2, rrp2, rthp2,              &
                           moist_1, moist_2, num3d_moist, &
                           chem_1, chem_2, num3d_chem,    &
                           ru_1, ru_2, rv_1, rv_2,        &
                           u_base, v_base, qv_base,       &
                           u_frame, v_frame,              &
                           msft, dzetaw, z, ht, z_zeta,   &
                           dx, dy,                        &
                           ids, ide, jds, jde, kds, kde,  &
                           ims, ime, jms, jme, kms, kme,  &
                           its, ite, jts, jte, kts, kte  )

   USE module_domain
   IMPLICIT NONE

   ! Input data
   INTEGER , INTENT(IN)                    :: ids, ide, jds, jde, kds, kde, &
                                              ims, ime, jms, jme, kms, kme, &
                                              its, ite, jts, jte, kts, kte
   INTEGER, INTENT(IN) :: num3d_moist, num3d_chem

   REAL , DIMENSION( ims: , kms: , jms: ) , INTENT(INOUT) :: rrb, rtb, pb
   REAL , DIMENSION( ims: , kms: , jms: ) , INTENT(OUT) :: rr, rrp, rthp
   REAL , DIMENSION( ims: , kms: , jms: ) , INTENT(OUT) :: rr2, rrp2, rthp2
   REAL , DIMENSION( ims: , kms: , jms: ) , INTENT(OUT) :: ru_1, ru_2, rv_1, rv_2
   REAL , DIMENSION( ims: , jms: )          , INTENT(IN) :: msft, ht, z_zeta
   REAL , DIMENSION( ims: , kms: , jms: ) , INTENT(IN) :: z
   REAL , DIMENSION( kms: ) , INTENT(IN)  :: dzetaw
   REAL , DIMENSION( kms: ) , INTENT(OUT) :: u_base, v_base, qv_base
   REAL                     , INTENT(OUT) :: u_frame, v_frame
   REAL ,                     INTENT(IN)  :: dx, dy

   REAL , DIMENSION( ims:,kms:,jms:, 1: ), INTENT(OUT) ::  &
      moist_1, moist_2
   REAL , DIMENSION( ims:,kms:,jms:, 1: ), INTENT(OUT) ::  &
      chem_1, chem_2

   ! Local data

   REAL , DIMENSION(kms:kme) :: thb, rb, thp, rp, pi, p, &
                                pp, rho, theta, rhotheta, dz
   REAL :: xn2l, tslvl
   INTEGER :: i, j, k, iter, it, itf, jtf, ktf, imoist_num

   ! bubble-related local
   LOGICAL :: bubble, integrate_up, p_top
   LOGICAL :: bubble1, bubble2
   REAL :: piov2, xcen, ycen, zcen, rcen, htbub, radbub, tpbub
   REAL :: radz
   INTEGER :: icen, jcen

   ! read-in local
!  for initial state only
   REAL , DIMENSION(1:kde-1,1:ide) :: rbj,rtbj,rrj,rtj

   REAL , DIMENSION( 1:kme ) :: u_tmp, qv_tmp, rel_hum
   INTEGER :: iterm, itm
   REAL temperature, pressure, qvs, es, check, moist_coef, delt, dzk, thsfc


   integrate_up = .true.
   bubble=.true.
   bubble1 = .false.
   bubble2 = .true.

   IF (bubble) THEN
     piov2=2.*atan(1.0)
     htbub=1500.    ! height (m) of center above z=0.
!     radbub=10000.   ! radius (m) of sphere for horizontal
     radbub=4000.   ! radius (m) of sphere for horizontal
     radz  = 1500.   ! vertical radius
!     tpbub=10.0      ! max temp pert at center (K)
!     tpbub=0.0      ! max temp pert at center (K)
     tpbub=3.0      ! max temp pert at center (K)
     icen=ide/2     ! center i
!     icen=ide-3     ! center i
!     icen=ide/2+12     ! center i
     jcen=jde/2     ! center j
   ENDIF

   itf=min0(ite,ide-1)
   jtf=min0(jte,jde-1)
   ktf=min0(kte,kde-1)

   !   constants
         iter = 10
         iterm = 20

         tslvl=300.
         xn2l=.0001
!         xn2l=.0000



   DO i=its,itf
   DO j=jts,jtf

! If this code is ever tiled in k dimension, it will need a second look.

     DO k=1,ktf
       thb(k)=rtb(i,k,j)/z_zeta(i,j)*msft(i,j)
       rb(k)=rrb(i,k,j)/z_zeta(i,j)*msft(i,j)
     ENDDO

   !________________________________________________________________________________
   ! USER-DEFINED PART - SET BASE-STATE PROFILE theta(k)
   !
   !   can set surface pressure as function of z(i,k,j)
   !   can set t as function of z(i,k,j)
         DO k=kms,kme
!           theta(k)=tslvl*exp(xn2l*z(i,k,j)/g)
           if(k.ne.kms)dz(k)=0.5*(dzetaw(k)+dzetaw(k-1))*z_zeta(i,j)
           if(k.eq.kms)dz(k)=0.5*dzetaw(k)*z_zeta(i,j)
         ENDDO
   !________________________________________________________________________________

   !________________________________________________________________________________
   ! USER-DEFINED PART - SET BASE-STATE MOISTURE PROFILE
   !
   !   can set qv as function of z(i,k,j), start by setting to zero

       IF(num3d_moist >= PARAM_FIRST_SCALAR) THEN

         DO imoist_num=PARAM_FIRST_SCALAR,num3d_moist
            DO k=kms,kme
               moist_1(i,k,j,imoist_num) = 0.   ! here we need to put the function of z
               moist_2(i,k,j,imoist_num) = moist_1(i,k,j,imoist_num)
            ENDDO
         ENDDO

       ENDIF

!  same call as in subroutine init_base, here we use the moisture
!    to initialize the moist perturbation

    CALL init_sound( theta, thsfc, rel_hum, z(i,1:ktf,j), ht(i,j), u_tmp, ktf,u_frame,v_frame)
    tslvl = theta(1)


   !________________________________________________________________________________


! If we ever tile in the k dimension, this code will need a second look.

   !   get p, pi, r

!   IF( integrate_up ) THEN

         k=1

!       if(xn2l.eq.0.)then
   ! calculate p at lowest level from functional form if xn2l=0.
         pi(k)=1.-g*z(i,k,j)/(cp*theta(1))
!       else
   ! could alternatively use the following if xn2l is NOT zero
!         pi(k)=1.+g*g/(cp*xn2l*tslvl)*(exp(-xn2l*z(i,k,j)/g)-1.)
!       endif

         p(k)=p1000mb*pi(k)**(cp/R_d)
         rho(k)=p(k)/R_d/(pi(k)*theta(k))
         pp(k)=p(k)-pb(i,k,j)
         rp(k)=rho(k)-rb(k)
         rhotheta(k)=rho(k)*theta(k)
         DO k=2,ktf
           pp(k)=pp(k-1)-rp(k-1)*g*dz(k)
           p(k)=pp(k)+pb(i,k,j)
           pi(k)=(p(k)/p1000mb)**rcp
           rho(k)=p(k)/R_d/(pi(k)*theta(k))
           rp(k)=rho(k)-rb(k)
           rhotheta(k)=rho(k)*theta(k)
   !!        check=p(k-1)-p(k)-0.5*(rho(k)+rho(k-1))*g*dz(k)
         ENDDO

!    ELSE
!     ENDIF

!
! wcs - switched k and it loops to increase accuracy

! wcs - added moisture initialization - 6 may 1999

     MOIST_LOOP:  DO itm = 1, iterm

!************
!         rhotheta(1)=rho(1)*theta(1)
!         p(1)=p1000mb*(r_d*rhotheta(1)  &
!             *(1+rvovrd*moist_1(i,1,j,P_QV))/p1000mb)**cpovcv
!         pp(1) = p(1) - pb(1,i,j)
!*************


        rho(1) = ((p(1)/p1000mb)**(cvovcp))*       &
                 (p1000mb/R_d)/                   &
             (theta(1)*(1.+rvovrd*moist_1(i,1,j,P_QV)))

        rp(1) = rho(1) - rb(1)
        rhotheta(1)=rho(1)*theta(1)

!         rhotheta(1)=rho(1)*theta(1)
!         p(1)=p1000mb*(r_d*rhotheta(1)  &
!             *(1+rvovrd*moist_1(1,i,j,P_QV))/p1000mb)**cpovcv
!         pp(1) = p(1) - pb(1,i,j)
!         rp(1) = rho(1) - rb(1)

         DO k=2,ktf
           DO it=1,2*iter

             pp(k)=pp(k-1)-0.5*(rp(k)+rp(k-1))*g*dz(k)            &
                 -0.5*( rho(k  )*moist_1(i,k  ,j,P_QV)            &
                       +rho(k-1)*moist_1(i,k-1,j,P_QV) )*g*dz(k)

             p(k)=pp(k)+pb(i,k,j)

             pi(k)=(p(k)/p1000mb)**rcp 

            rho(k)=p(k)/R_d/(pi(k)*theta(k)*(1.+rvovrd*moist_1(i,k,j,P_QV)))

!             rho(k) = ((p(k)/p1000mb)**(cvovcp))*       &
!                 (p1000mb/R_d)/                         &
!                (theta(k)*(1.+rvovrd*moist_1(i,k,j,P_QV)))

             rp(k)=rho(k)-rb(k)
             
             rhotheta(k)=rho(k)*theta(k)
   !!          check=p(k-1)-p(k)-0.5*(rho(k)+rho(k-1))*g*dz(k)
           ENDDO
         ENDDO

!  add in a bubble?

     DO k=1,ktf

           thp(k)=rhotheta(k)-thb(k)
           IF (bubble1) THEN
   !         bubble-center coordinate (xcen,ycen,zcen) = (0,0,0)

!             xcen=abs(float(i-icen)*dx)
!             ycen=0.

            xcen = 0.
            ycen=abs(float(j-jcen)*dy)

             zcen=z(i,k,j)-htbub
             rcen=sqrt( (xcen/radbub)**2  &
                       +(ycen/radbub)**2  &
                       +(zcen/radz)**2   )
             IF (rcen <= 1) THEN
               rp(k)=rp(k)-rho(k)/theta(k)*tpbub*cos(rcen*piov2)*  &
                           cos(rcen*piov2) 
!               delt = tpbub*cos(rcen*piov2)*cos(rcen*piov2)
!               rp(k)= (-rrb(i,k,j)*delt)/theta(k)
!                thp(k) = delt

             ENDIF
           ENDIF

       rthp(i,k,j)=thp(k)*z_zeta(i,j)/msft(i,j)
       rrp(i,k,j)=rp(k)*z_zeta(i,j)/msft(i,j)
       rr(i,k,j)= rrp(i,k,j) + rrb(i,k,j)
       rthp2(i,k,j)=rthp(i,k,j)
       rrp2(i,k,j)=rrp(i,k,j)
       rr2(i,k,j)= rr(i,k,j)

     ENDDO


! calc saturation vapor pressure and reset mixing ratio

        DO k = 1,ktf
!***********
!          pi(k)=(pb(i,k,j)/p1000mb)**rcp 
          pi(k)=(p(k)/p1000mb)**rcp 
          temperature = pi(k)*theta(k)
!***********
          pressure = p(k)
!          pressure = pb(i,k,j)
!         qvs = 380.*exp(17.27*(temperature-273.)/(temperature-36.))/pressure
          es  = 1000.*svp1*exp(svp2*(temperature-svpt0)/(temperature-svp3))
          qvs = ep_2*es/(pressure-es)
          moist_1(i,k,j,P_QV) = min(0.014,rel_hum(k)*qvs)
          moist_2(i,k,j,P_QV) = moist_1(i,k,j,P_QV)
        ENDDO

     ENDDO moist_loop


     IF (bubble2) THEN

     !    bubble-center coordinate (xcen,ycen,zcen) = (0,0,0)
         DO k=1,ktf

!           xcen=abs(float(i-icen)*dx)
!           ycen=0.
           xcen = 0.
           ycen=abs(float(j-jcen)*dy)

           zcen=z(i,k,j)-htbub
           rcen=sqrt( (xcen/radbub)**2  &
                     +(ycen/radbub)**2  &
                     +(zcen/radz)**2   )
           IF (rcen <= 1) THEN
             delt = tpbub*cos(rcen*piov2)*cos(rcen*piov2)
           !  thp(k) = delt
             theta(k) = theta(k) + delt
           ENDIF

         ENDDO

         DO k=ktf-1,1,-1
           DO it=1,2*iter
             rhotheta(k)=rho(k)*theta(k)           
            ! pp(k)=pp(k-1)-0.5*(rp(k)+rp(k-1))*g*dz(k)            &
            !     -0.5*( rho(k  )*moist_1(i,k  ,j,P_QV)               &
            !           +rho(k-1)*moist_1(i,k-1,j,P_QV) )*g*dz(k)
             pp(k)=pp(k+1)+0.5*(rp(k)+rp(k+1))*g*dz(k+1)          &
                 +0.5*( rho(k  )*moist_1(i,k  ,j,P_QV)               &
                       +rho(k+1)*moist_1(i,k+1,j,P_QV) )*g*dz(k+1)
             p(k)=pp(k)+pb(i,k,j)
             pi(k)=(p(k)/p1000mb)**rcp 
             rho(k)=p(k)/R_d/(pi(k)*theta(k)*(1.+rvovrd*moist_1(i,k,j,P_QV)))
             rp(k)=rho(k)-rb(k)
             rhotheta(k)=rho(k)*theta(k)          
           ENDDO

!           rthp(i,k,j)=thp(k)*z_zeta(i,j)/msft(i,j)
           rthp (i,k,j) = rhotheta(k)*z_zeta(i,j)/msft(i,j) - rtb(i,k,j)
           rrp(i,k,j)=rp(k)*z_zeta(i,j)/msft(i,j)
           rr(i,k,j)= rrp(i,k,j) + rrb(i,k,j)
           rthp2(i,k,j)=rthp(i,k,j)
           rrp2(i,k,j)=rrp(i,k,j)
           rr2(i,k,j)= rr(i,k,j)

         ENDDO


      ENDIF

   IF( (i == its) .and. (j == jts) ) THEN

    write(6,*) ' perturbation state profile, k, z, qv, pp, rp, rthp '
    do k=1,ktf
      write(6,*) k,z(i,k,j), moist_1(i,k,j,P_QV), pp(k), rp(k), rthp(i,k,j)
    enddo

   ENDIF


   ENDDO
   ENDDO


!  set initial velocity field

    DO i=its,itf
    DO j=jts,jtf
    DO K = 1, ktf

!     ru_1(i,k,j) = u_tmp(k)*rr(i,k,j)
!     ru_2(i,k,j) = ru_1(i,k,j)
!     rv_1(i,k,j) = 0.
!     rv_2(i,k,j) = 0.

     rv_1(i,k,j) = u_tmp(k)*rr(i,k,j)
     rv_2(i,k,j) = rv_1(i,k,j)
     ru_1(i,k,j) = 0.
     ru_2(i,k,j) = 0.


    ENDDO
    ENDDO
    ENDDO

    IF ( ite .eq. ide ) THEN

      DO j=jts,jtf
      DO K = 1, ktf

        ru_1(ite,k,j) = 0.
        ru_2(ite,k,j) = 0.

!     ru_1(ite,k,j) = ru_1(ite-1,k,j)
!     ru_2(ite,k,j) = ru_2(ite-1,k,j)

      ENDDO
      ENDDO

    ENDIF

    IF ( jte .eq. jde ) THEN

      DO i=its,itf
      DO K = 1, ktf

!       rv_1(i,k,jte) = 0.
!       rv_2(i,k,jte) = 0.

        rv_1(i,k,jte) = rv_1(i,k,jte-1)
        rv_2(i,k,jte) = rv_2(i,k,jte-1)

      ENDDO
      ENDDO

    ENDIF

    DO K = 1, ktf
!      u_base(k) = u_tmp(k)
!      v_base(k) = 0.
      u_base(k) = 0.
      v_base(k) = u_tmp(k)
!      write(6,*) ' k, u_base, ru ',k,u_base(k),ru_1(1,k,1),rv_1(1,k,1)
    ENDDO
   

!  convert to rho_theta_m, store off initial moisture



       IF(num3d_moist >= PARAM_FIRST_SCALAR) THEN

!        IF(num3d_moist == 3) THEN   ! kessler model (*,*,*,1) is qv, 2, and 3 are 
!                                    ! qc and qr
           DO i=its,itf
           DO j=jts,jtf
           DO K = 1, ktf

!             rthp(i,k,j)=rthp(i,k,j)*(1.+rvovrd*moist_1(i,k,j,P_QV))
!             rthp2(i,k,j)=rthp2(i,k,j)*(1.+rvovrd*moist_1(i,k,j,P_QV))

             rthp(i,k,j)=(rtb(i,k,j)+rthp(i,k,j))*  &
                         (1.+rvovrd*moist_1(i,k,j,P_QV)) &
                            - rtb(i,k,j)
             rthp2(i,k,j)=rthp(i,k,j)

           ENDDO
           ENDDO
           ENDDO

           DO k=1, ktf
             qv_base(k) = moist_1(its,k,jts,P_QV)
           ENDDO


!        ELSE

!          write(6,*) ' number of moist variables is not 0 or 3 '
!          write(6,*) ' error stop in init_state '
!          stop

!        ENDIF

       ENDIF

           DO i=its,itf
           DO j=jts,jtf
           DO K = 1, ktf

          chem_1(i,k,j,1) = 0.
          chem_2(i,k,j,1) = 0.

           ENDDO
           ENDDO
           ENDDO

   END SUBROUTINE init_state

!-------------------------------------------------------------------
! this is a wrapper for the solver-specific init_domain routines.
! Also dereferences the grid variables and passes them down as arguments.
! This is crucial, since the lower level routines may do message passing
! and this will get fouled up on machines that insist on passing down
! copies of assumed-shape arrays (by passing down as arguments, the 
! data are treated as assumed-size -- ie. f77 -- arrays and the copying
! business is avoided).  Fie on the F90 designers.  Fie and a pox.

   SUBROUTINE init_domain ( grid )

   IMPLICIT NONE

   !  Input data.
   TYPE (domain), POINTER :: grid 
   !  Local data.
   INTEGER                :: dyn_opt 
   INTEGER :: idum1, idum2

#ifdef DEREF_KLUDGE
   INTEGER     :: sm31 , em31 , sm32 , em32 , sm33 , em33
#endif

#ifdef DEREF_KLUDGE
   sm31             = grid%sm31
   em31             = grid%em31
   sm32             = grid%sm32
   em32             = grid%em32
   sm33             = grid%sm33
   em33             = grid%em33
#endif

   CALL get_dyn_opt( dyn_opt )
   
   CALL set_scalar_indices_from_config ( head_grid%id , idum1, idum2 )

   IF (      dyn_opt .eq. 1 &
        .or. dyn_opt .eq. 2 &
        .or. dyn_opt .eq. 3 &
                                       ) THEN
     CALL init_domain_rk( grid, &
!
#include <rk_actual_args.inc>
!
                        )

   ELSE
     WRITE(0,*)' init_domain: unknown or unimplemented dyn_opt = ',dyn_opt
     STOP
   ENDIF

   END SUBROUTINE init_domain

!-------------------------------------------------------------------

   SUBROUTINE init_domain_rk ( grid, &
!
# include <rk_dummy_args.inc>
!
)
   IMPLICIT NONE

   !  Input data.
   TYPE (domain), POINTER :: grid

# include <rk_dummy_arg_defines.inc>

   TYPE (grid_config_rec_type)              :: config_flags

   !  Local data
   INTEGER                             ::                       &
                                  ids, ide, jds, jde, kds, kde, &
                                  ims, ime, jms, jme, kms, kme, &
                                  its, ite, jts, jte, kts, kte, &
                                  i, j, k

   ! Local data for mountain
   INTEGER :: icm,jcm, ii, im1, jj, jm1, loop, error, fid
   REAL    :: xa,hm

   LOGICAL :: moisture_init

#ifdef DM_PARALLEL
#  ifdef RSL
#    include <rsl_rk_data_calls.inc>
#  endif
#endif

   SELECT CASE ( model_data_order )
         CASE ( DATA_ORDER_ZXY )
   kds = grid%sd31 ; kde = grid%ed31 ;
   ids = grid%sd32 ; ide = grid%ed32 ;
   jds = grid%sd33 ; jde = grid%ed33 ;

   kms = grid%sm31 ; kme = grid%em31 ;
   ims = grid%sm32 ; ime = grid%em32 ;
   jms = grid%sm33 ; jme = grid%em33 ;

   kts = grid%sp31 ; kte = grid%ep31 ;   ! note that tile is entire patch
   its = grid%sp32 ; ite = grid%ep32 ;   ! note that tile is entire patch
   jts = grid%sp33 ; jte = grid%ep33 ;   ! note that tile is entire patch
         CASE ( DATA_ORDER_XYZ )
   ids = grid%sd31 ; ide = grid%ed31 ;
   jds = grid%sd32 ; jde = grid%ed32 ;
   kds = grid%sd33 ; kde = grid%ed33 ;

   ims = grid%sm31 ; ime = grid%em31 ;
   jms = grid%sm32 ; jme = grid%em32 ;
   kms = grid%sm33 ; kme = grid%em33 ;

   its = grid%sp31 ; ite = grid%ep31 ;   ! note that tile is entire patch
   jts = grid%sp32 ; jte = grid%ep32 ;   ! note that tile is entire patch
   kts = grid%sp33 ; kte = grid%ep33 ;   ! note that tile is entire patch
         CASE ( DATA_ORDER_XZY )
   ids = grid%sd31 ; ide = grid%ed31 ;
   kds = grid%sd32 ; kde = grid%ed32 ;
   jds = grid%sd33 ; jde = grid%ed33 ;

   ims = grid%sm31 ; ime = grid%em31 ;
   kms = grid%sm32 ; kme = grid%em32 ;
   jms = grid%sm33 ; jme = grid%em33 ;

   its = grid%sp31 ; ite = grid%ep31 ;   ! note that tile is entire patch
   kts = grid%sp32 ; kte = grid%ep32 ;   ! note that tile is entire patch
   jts = grid%sp33 ; jte = grid%ep33 ;   ! note that tile is entire patch

   END SELECT


   CALL model_to_grid_config_rec ( grid%id , model_config_rec , config_flags )

! here we check to see if the boundary conditions are set properly

   call boundary_condition_check( config_flags, bdyzone, error, grid%id )

   moisture_init = .true.

    grid%itimestep=0

   hm = 00.
   xa = 5.

   icm = ide/2
   jcm = jde/2

#ifdef DM_PARALLEL
   call wrf_dm_bcast_bytes( icm , IWORDSIZE )
   call wrf_dm_bcast_bytes( jcm , IWORDSIZE )
#endif

    CALL set_mminlu('    ')
    CALL set_iswater(1,0)
    CALL set_cen_lat(1,40.)
    CALL set_cen_lon(1,-105.)
    CALL set_truelat1(1,0.)
    CALL set_truelat2(1,0.)
    CALL set_map_proj(1,0)

    DO j = jts, jte
      DO i = its, ite
!CIO ht          ij
!CIO msft        ij
!CIO msfu        ij
!CIO msfv        ij
!CIO sina        ij
!CIO cosa        ij
!CIO e           ij
!CIO f           ij
!CIO r           kij
!CIO ru          kij
!CIO rv          kij
!CIO rom         kij
!        ht(i,j)       = hm/(1.+(float(i-icm)/xa)**2+(float(j-jcm)/xa)**2)
          ht(i,j)       = hm/(1.+(float(j-jcm)/xa)**2)
!         ht(i,j)       = hm/(1.+(float(i-icm)/xa)**2)
         msft(i,j)     = 1.
         msfu(i,j)     = 1.
         msfv(i,j)     = 1.
         sina(i,j)     = 0.
         cosa(i,j)     = 1.
         e(i,j)        = 0.
         f(i,j)        = 0.
! values used only in some physics options
         snowc(i,j)    = 0.
         xlat(i,j)     = 40.
         xlong(i,j)    = -105.

         DO k = kts, kte
            r(i,k,j)    = 1.               ! constant
            ru_1(i,k,j) = 0.
            ru_2(i,k,j) = ru_1(i,k,j)
            rv_1(i,k,j) = 0.
            rv_2(i,k,j) = rv_1(i,k,j)
            rom_1(i,k,j) = 0.
            rom_2(i,k,j) = rom_1(i,k,j)
         END DO
      END DO
   END DO

# ifdef DM_PARALLEL

!
!
!                           * * * * *
!         *        * * *    * * * * *
!       * + *      * + *    * * + * *
!         *        * * *    * * * * *
!                           * * * * *

! ru_1                          x
! ru_2                          x
! rv_1                          x
! rv_2                          x
! rom_1                         x
! rom_2                         x
! r                             x
! rtp_1                         x
! rtp_2                         x
! rr_1                          x
! rr_2                          x
! rrp_1                         x
! rrp_2                         x
! rtb                           x
! rrb                           x
! pip                           x
! pib                           x
! z                             x
! zx                            x
! zy                            x

! ht                            x
! msft                          x
! msfu                          x
! msfv                          x
! sina                          x
! cosa                          x
! e                             x
! f                             x
! z_zeta                        x
! zeta_z                        x
! cofwr                         x
! moist_2                       x
! chem_2                        x
! moist_1                       x
! chem_1                        x

      CALL wrf_dm_halo ( grid%domdesc , grid%comms , HALO_RK_INIT )
# endif

   rdx = 1./dx
   rdy = 1./dy
   resm = (1.-epssm)/(1.+epssm)
   dts    = ( dt / float(grid%time_step_sound) )
   dtseps = .5 * dts * ( 1. + epssm )


! Need to read in some stuff

!CIO z        kij
!CIO zx       kij
!CIO zy       kij
!CIO rtp      kij
!CIO z_zeta   ij
!CIO zeta_z   ij
!CIO cofwr    ij
!CIO zeta     k
!CIO zetaw    k
!CIO dzetaw   k
!CIO dzeta    k
!CIO rdzu     k
!CIO rdzw     k
!CIO fzm      k
!CIO fzp      k
!CIO cofrz    k
!CIO ztop     s
!CIO cf1      s
!CIO cf2      s
!CIO cf3      s


   print*,'P_QI',P_QV,P_QC,P_QR,P_QI,P_QS,P_QG

   CALL init_coords (    grid,                                     &
                         ht, rdx, rdy, zeta, zetaw, dzetaw, dzeta, &
                         rdzu, rdzw, z, zx, zy, &
                         z_zeta, zeta_z, fzm, fzp, ztop, &
                         cf1, cf2, cf3, &
                         cofrz, cofwr, dtseps, &
                         ids, ide, jds, jde, kds, kde, &
                         ims, ime, jms, jme, kms, kme, &
                         its, ite, jts, jte, kts, kte )

!CIO rrb          kij
!CIO rtb          kij
!CIO pb           kij
!CIO pib          kij
!CIO h_diabatic   kij


CALL start_timing
   CALL init_base_state ( grid,                                &
                          rrb, rtb, pb, pib, pb8w, h_diabatic, &
                          msft, TSK, TMN, dzetaw, z, ht, z_zeta,    &
                          u_frame, v_frame,                    &
                          ids, ide, jds, jde, kds, kde,        &
                          ims, ime, jms, jme, kms, kme,        &
                          its, ite, jts, jte, kts, kte        )

CALL end_timing( 'module_initialize: init_base_state' )


! If we read all of the state, don't need this.

!CIO rrb         kij
!CIO rtb         kij
!CIO pb          kij
!CIO rr          kij
!CIO rtp         kij
!CIO rrp         kij
!CIO rthp        kij
!CIO ru          kij
!CIO rv          kij
!CIO u_base      k
!CIO v_base      k
!CIO qv_base     k
!CIO moist       kijf
!CIO chem        kijf


CALL start_timing
   CALL init_state (  rrb, rtb, pb,                   &
                      rr_1, rrp_1, rtp_1,             &
                      rr_2, rrp_2, rtp_2,             &
                      moist_1, moist_2, num_moist,    &
                      chem_1, chem_2, num_chem,       &
                      ru_1, ru_2, rv_1, rv_2,         &
                      u_base, v_base, qv_base,        &
                      u_frame, v_frame,               &
                      msft, dzetaw, z, ht, z_zeta,    &
                      dx, dy,                         &
                      ids, ide, jds, jde, kds, kde,   &
                      ims, ime, jms, jme, kms, kme,   &
                      its, ite, jts, jte, kts, kte   )
CALL end_timing ('module_initialize: init_state')

!
   IF ( .not. input_from_file ) then
       
!   IF (moisture_init) THEN

!      DO j = jts, min(jte,jde-1)
!         DO i = its, min(ite,ide-1)
!            DO k = kts, min(kde-1,kte)
!               loop_3d_mbc   : DO loop = PARAM_FIRST_SCALAR , num_3d_moist
!                  moist_1(i,k,j,loop)=float (100 * loop + k)/rr_1(i,k,j)
!                  moist_2(i,k,j,loop)=float (100 * loop + k)/rr_2(i,k,j)
!               END DO loop_3d_mbc
!            ENDDO
!         ENDDO
!      ENDDO
!   ENDIF

    endif

     RETURN

   END SUBROUTINE init_domain_rk

   

   SUBROUTINE init_module_initialize
   END SUBROUTINE init_module_initialize

!-------------------------------------------------------------------------

  SUBROUTINE init_sound(t,thsfc,qv,zg,sfch,u,n,u_frame,v_frame)

  IMPLICIT NONE

  integer  :: n
  real, dimension(n), intent(out) ::   t,qv,u
  real, dimension(n), intent(in ) ::   zg

  real, intent(out) ::   thsfc
  real, intent(out) ::   u_frame,v_frame
  REAL, INTENT(IN)  :: sfch

  integer :: nz1,k
  real    :: ztr, alpha,vs,thetar,ttr,qv0m,thetas,zts,cc1,cc2
  real    :: z, ztp

      nz1 = n
      ZTR=12000.
      ALPHA=4.
      VS=12.0
!      VS=0.0
      u_frame=0.
      v_frame=VS
      THETAR=343.
      TTR=213.
      QV0M=0.014
      THETAS=300.5
      ZTS=2500.
      CC1 = 1.
      CC2 = 0.

      DO K=1,NZ1
        z = zg(k)
        ztp = cc1*z + cc2*z**2
        if(ztp .gt. ztr) then
          t(k) = thetar*exp(9.8*(ztp-ztr)/(1003.*ttr))
          qv(k) = 0.25
        else
          t(k) = 300.+43.*(ztp/ztr)**1.25
          qv(k) = (1.-0.75*(ztp/ztr)**1.25)
          if(t(k).lt.thetas) t(k)=thetas
        end if
        u(k) = vs*ztp/zts
        if(ztp .ge. zts) u(k) = vs
      enddo
      thsfc = 300.+43.*(sfch/ztr)**1.25
      do k=1,nz1
        u(k) = u(k) - vs
!         u(k) = 0.
!        t(k) = 300.
!        qv(k) = 0.
      enddo

!      write(6,*) ' initial conditions '

!      do k=1,nz1
!        write(6,*) k,zg(k),t(k),u(k),qv(k)
!      enddo

      END SUBROUTINE init_sound


END MODULE module_initialize
