!WRF:MODEL_LAYER:PHYSICS
!

MODULE module_microphysics

   USE module_big_step_utilities
   USE module_state_description
   USE module_bc

   !  this module contains subroutines for the microphysical parameterizations.
   !
   !  there is a routine to prepare the fields for time-split microphysics
   !
   !  there is a driver routine the calls the appropriate parameterization
   !  
   !  the only parameterization in here presently is a kessler warm-rain micro

CONTAINS

!------------------------------------------------------------------------------

   SUBROUTINE moist_physics_prep( rt_new, rt_old, rtp_new, rtp_old, &
                                  t_new, t_old, rr_new, rr_old,     &
                                  moist_new, moist_old, moist_sav,  &
                                  rtb,                              &
                                  rtp_sav, pi, p, msft, zeta_z,     &
                                  fzm, fzp, config_flags, n_moist,      &
                                  ids,ide, jds,jde, kds,kde,  & ! domain dims
                                  ims,ime, jms,jme, kms,kme,  & ! memory dims
                                  its,ite, jts,jte, kts,kte )   ! tile   dims

   IMPLICIT NONE

! Here we construct full fields
! here for uncoupled density (density without the 
! stretching and map scale factor)
! and the provisional value for the new theta.

   TYPE(grid_config_rec_type),    INTENT(IN   )    :: config_flags

   INTEGER,      INTENT(IN   )    :: ids,ide, jds,jde, kds,kde
   INTEGER,      INTENT(IN   )    :: ims,ime, jms,jme, kms,kme
   INTEGER,      INTENT(IN   )    :: its,ite, jts,jte, kts,kte
   INTEGER,      INTENT(IN   )    :: n_moist

   REAL, DIMENSION( ims:ime , kms:kme, jms:jme ),       &
         INTENT(IN   ) ::                                &
                                              rtb

   REAL, DIMENSION( ims:ime , kms:kme, jms:jme, n_moist ),  &
         INTENT(IN   ) ::                                    &
                                              moist_new,     &
                                              moist_old,     &
                                              moist_sav

   REAL, DIMENSION( ims:ime , kms:kme, jms:jme ),       &
         INTENT(INOUT) ::                                &
                                              rt_new,    &
                                              rt_old,    &
                                             rtp_new,    &
                                             rtp_old,    &
                                             rtp_sav,    &
                                              rr_new,    &
                                              rr_old,    &
                                               t_old

   REAL, DIMENSION( ims:ime , kms:kme, jms:jme ),       &
         INTENT(  OUT) ::                                &
                                              t_new,     &
                                                  p,     &
                                                 pi

   REAL, DIMENSION( ims:ime , jms:jme ),    &
         INTENT(IN   ) ::                   &
                                    msft  , &
                                    zeta_z

   REAL, DIMENSION( kms:kme ), INTENT(IN   ) :: fzm, fzp

!  need to switch names to old and new so that they make sense

!  local variables

   INTEGER :: i_start, i_end, j_start, j_end, k_start, k_end
   INTEGER :: i, j, k

!--------------------------------------------------------------------

!  set up loop bounds for this grid's boundary conditions


    i_start = its    
    i_end   = min( ite,ide-1 )
    j_start = jts    
    j_end   = min( jte,jde-1 )

    IF( config_flags%nested .or. config_flags%specified ) THEN
      i_start = max( its,ids+1 )
      i_end   = min( ite,ide-2 )
      j_start = max( jts,jds+1 )
      j_end   = min( jte,jde-2 )
    ENDIF

    k_start = kts
    k_end = min( kte, kde-1 )

           !  compute full pi or p at the new time-level
           !  (needed for physics)

     DO j = j_start, j_end
     DO k = k_start, k_end
     DO i = i_start, i_end
         rt_new(i,k,j) = rtb(i,k,j)+ rtp_new(i,k,j)
     ENDDO
     ENDDO
     ENDDO

     CALL calculate_p ( rt_new, pi, p,                &
                        msft, zeta_z,                 &
                        ids, ide, jds, jde, kds, kde, &
                        ims, ime, jms, jme, kms, kme, &
                        its, ite, jts, jte, kts, kte )

     CALL decouple_thm ( rr_new, rt_new, t_new,         &
                         config_flags, moist_new, fzm, fzp, &
                         n_moist,                       &
                         ids, ide, jds, jde, kds, kde,  &
                         ims, ime, jms, jme, kms, kme,  &
                         its, ite, jts, jte, kts, kte  )

     CALL decouple_thm ( rr_old, rt_old, t_old,         &
                         config_flags, moist_old, fzm, fzp, &
                         n_moist,                       &
                         ids, ide, jds, jde, kds, kde,  &
                         ims, ime, jms, jme, kms, kme,  &
                         its, ite, jts, jte, kts, kte  )

   !  while we're at it, decouple the density from the vertical
   !  stretching and map-scale factor

     DO j = j_start, j_end
     DO k = k_start, k_end
     DO i = i_start, i_end
         rr_new(i,k,j) = rr_new(i,k,j)*msft(i,j)*zeta_z(i,j)
         rr_old(i,k,j) = rr_old(i,k,j)*msft(i,j)*zeta_z(i,j)
     ENDDO
     ENDDO
     ENDDO

   END SUBROUTINE moist_physics_prep

!------------------------------------------------------------------------------

   SUBROUTINE moist_physics_finish( rt_new, rt_old, rtp_new, rtp_old, &
                                    t_new, rr_new, rr_old,            &
                                    moist_new, rtb, msft, zeta_z,     &
                                    config_flags, n_moist,                &
                                    ids,ide, jds,jde, kds,kde,  & ! domain dims
                                    ims,ime, jms,jme, kms,kme,  & ! memory dims
                                    its,ite, jts,jte, kts,kte )   ! tile   dims

   IMPLICIT NONE

!  The physics routines have returned a new value for theta.
!  here we replace the dynamical value of rho_d*theta'(1+rvovrd*qv) with
!  that computed from the updated values of theta and qv.  We
!  also recouple the density with zeta_z and the map-scale factor.

   TYPE(grid_config_rec_type),    INTENT(IN   )    :: config_flags

   INTEGER,      INTENT(IN   )    :: ids,ide, jds,jde, kds,kde
   INTEGER,      INTENT(IN   )    :: ims,ime, jms,jme, kms,kme
   INTEGER,      INTENT(IN   )    :: its,ite, jts,jte, kts,kte
   INTEGER,      INTENT(IN   )    :: n_moist

   REAL, DIMENSION( ims:ime , kms:kme, jms:jme ),       &
         INTENT(IN   ) ::                                &
                                              rtb

   REAL, DIMENSION( ims:ime , kms:kme, jms:jme, n_moist ),  &
         INTENT(IN   ) ::                                    &
                                              moist_new


                                              

   REAL, DIMENSION( ims:ime , kms:kme, jms:jme ),       &
         INTENT(INOUT) ::                                &
                                              rt_new,    &
                                              rt_old,    &
                                             rtp_new,    &
                                             rtp_old,    &
                                              rr_new,    &
                                              rr_old

   REAL, DIMENSION( ims:ime , kms:kme, jms:jme ),       &
         INTENT(IN   ) ::                                &
                                              t_new

   REAL, DIMENSION( ims:ime , jms:jme ),    &
         INTENT(IN   ) ::                   &
                                    msft  , &
                                    zeta_z

!  need to switch names to old and new so that they make sense

!  local variables

   INTEGER :: i_start, i_end, j_start, j_end, k_start, k_end
   INTEGER :: i, j, k

!--------------------------------------------------------------------

!  set up loop bounds for this grid's boundary conditions

    i_start = its    
    i_end   = min( ite,ide-1 )
    j_start = jts    
    j_end   = min( jte,jde-1 )

    IF( config_flags%nested .or. config_flags%specified ) THEN
      i_start = max( its,ids+1 )
      i_end   = min( ite,ide-2 )
      j_start = max( jts,jds+1 )
      j_end   = min( jte,jde-2 )
    ENDIF

    k_start = kts
    k_end = min( kte, kde-1 )

!  at this point we should be almost finished with
!  the timestep, and we don't need to re-couple rr_old,
!  but we'll compute it anyway so that we have it.
!  also note that we are not recomputing any of the pressure
!  diagnostics (pi, p, pi', or p') for the updated timelevel

!  we shouldn't be in this routine of n_moist is 0 (dry model)
!  but we'll trap it anyway (habit I guess)

     IF( n_moist >= PARAM_FIRST_SCALAR ) THEN

       DO j = j_start, j_end
       DO k = k_start, k_end
       DO i = i_start, i_end
           rr_new(i,k,j) = rr_new(i,k,j)/(msft(i,j)*zeta_z(i,j))
           rr_old(i,k,j) = rr_old(i,k,j)/(msft(i,j)*zeta_z(i,j)) 
           rt_new(i,k,j) = t_new(i,k,j)*rr_new(i,k,j)*(1+rvovrd*moist_new(i,k,j,P_QV))
           rtp_new(i,k,j) = rt_new(i,k,j) - rtb(i,k,j)
       ENDDO
       ENDDO
       ENDDO

     ELSE

       DO j = j_start, j_end
       DO k = k_start, k_end
       DO i = i_start, i_end
           rr_new(i,k,j) = rr_new(i,k,j)/(msft(i,j)*zeta_z(i,j))
           rr_old(i,k,j) = rr_old(i,k,j)/(msft(i,j)*zeta_z(i,j)) 
           rt_new(i,k,j) = t_new(i,k,j)*rr_new(i,k,j)            
           rtp_new(i,k,j) = rt_new(i,k,j)-rtb(i,k,j)
       ENDDO
       ENDDO
       ENDDO

     ENDIF

   END SUBROUTINE moist_physics_finish

END MODULE module_microphysics
