!WRF:MODEL_LAYER:DYNAMICS
!

MODULE module_rk

   USE module_model_constants
   USE module_advect
   USE module_big_step_utilities
   USE module_state_description

CONTAINS

!------------------------------------------------------------------------

SUBROUTINE rk_step_prep ( config_flags,                             &
                           ru,  rv,  rom,  rthp,  rrp, rr,           &
                           u , v , rw , w , rth , theta , thp ,      &
                           moist,                                    &
                           rtb, rb, pb, pib, pp , pip ,              &
                           zx, zy, msft, zeta_z, z_zeta,             &
                           fzm, fzp,                                 &
                           cf1, cf2, cf3, n_moist,                   &
                           ids, ide, jds, jde, kds, kde,             &
                           ims, ime, jms, jme, kms, kme,             &
                           its, ite, jts, jte, kts, kte             )

   IMPLICIT NONE


   !  Input data.

   TYPE(grid_config_rec_type   ) ,   INTENT(IN   ) :: config_flags

   INTEGER ,       INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                    ims, ime, jms, jme, kms, kme, &
                                    its, ite, jts, jte, kts, kte

   INTEGER ,       INTENT(IN   ) :: n_moist

   REAL , DIMENSION(  ims:ime , kms:kme, jms:jme ) ,                     &
                                               INTENT(OUT  ) :: u,       &
                                                                v,       &
                                                                rw,      &
                                                                w,       &
                                                                rth,     &
                                                                theta,   &
                                                                thp,     &
                                                                pp,      &
                                                                pip

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme  ) , INTENT(IN   ) :: zx ,  & 
                                                                      zy

   REAL , DIMENSION( ims:ime , kms:kme , jms:jme ) ,                     &
                                               INTENT(IN   ) :: rr,      &
                                                                ru,      &
                                                                rv,      &
                                                                rom,     &
                                                                rrp,     &
                                                                rtb,     &
                                                                rb,      &
                                                                pb,      &
                                                                pib

   REAL , DIMENSION( ims:ime , kms:kme, jms:jme ) ,                     &
                                               INTENT(INOUT) :: rthp

   REAL , DIMENSION( ims:ime, kms:kme, jms:jme, n_moist ), INTENT(   IN) :: &
                                                           moist

   REAL , DIMENSION( ims:ime , jms:jme ) ,    INTENT(IN   ) :: msft,   &
                                                               zeta_z, &
                                                               z_zeta

   REAL , DIMENSION( kms:kme ) ,                 INTENT(IN   ) :: fzm, &
                                                                  fzp

   REAL,  INTENT(IN) :: cf1, cf2, cf3

   integer :: k


   !  Decouple variables and compute diagnostics 
   !  at the current time level.

   CALL decouple ( rr, ru, u, 'u', config_flags,     &
                   fzm, fzp,                     &
                   ids, ide, jds, jde, kds, kde, &
                   ims, ime, jms, jme, kms, kme, &
                   its, ite, jts, jte, kts, kte )

   CALL decouple ( rr, rv, v, 'v', config_flags,     &
                   fzm, fzp,                     &
                   ids, ide, jds, jde, kds, kde, &
                   ims, ime, jms, jme, kms, kme, &
                   its, ite, jts, jte, kts, kte )

   CALL calculate_rw_w ( rr, ru, rv, rom, rw, w,         &
                         cf1, cf2, cf3,                  &
                         msft, zx, zy, z_zeta, fzm, fzp, &
                         ids, ide, jds, jde, kds, kde,   &
                         ims, ime, jms, jme, kms, kme,   &
                         its, ite, jts, jte, kts, kte   )

   CALL calculate_full ( rth, rtb, rthp,  &
                            ids, ide, jds, jde, kds, kde,    &
                            ims, ime, jms, jme, kms, kme,    &
                            its, ite, jts, jte, kts, kte    )

   ! get theta*(1+rvovrd*qv) from rho_d*theta*(1+rvovrd*qv) for advection
   ! and for pressure calculation

   CALL decouple ( rr, rth, theta, 't', config_flags,  &
                   fzm, fzp,                       &
                   ids, ide, jds, jde, kds, kde,   &
                   ims, ime, jms, jme, kms, kme,   &
                   its, ite, jts, jte, kts, kte   )

   ! get theta' from rho_d*theta'*(1+rvovrd*qv) for mixing

   CALL decouple_thmp ( rr, rthp, thp, config_flags,      &
                        moist, rb, rtb,               &
                        fzm, fzp, n_moist,            &
                        ids, ide, jds, jde, kds, kde, &
                        ims, ime, jms, jme, kms, kme, &
                        its, ite, jts, jte, kts, kte )

   ! calculate p, p', pi and pi' from rho*theta_m

   CALL calculate_pp ( rtb, rth, pb, pp, pib, pip,   &
                       msft, zeta_z,                 &
                       ids, ide, jds, jde, kds, kde, &
                       ims, ime, jms, jme, kms, kme, &
                       its, ite, jts, jte, kts, kte )

   !  Modify pi for moisture pi -> rhod_over_rhom*pi
   !  NOTE: pip will now contain full pi*rhod_over_rhom for use in small steps

   CALL calculate_pim (  pip, pib, moist, n_moist,     &
                         ids, ide, jds, jde, kds, kde, &
                         ims, ime, jms, jme, kms, kme, &
                         its, ite, jts, jte, kts, kte )

END SUBROUTINE rk_step_prep

!-------------------------------------------------------------------------------

SUBROUTINE rk_tendency ( config_flags, rk_step,                          &
                          ru_tend , rv_tend , rw_tend , rt_tend , r_tend,  &
                          ru,    rv,  rom,    rthp,   rrp,                 &
                          ru_tendf, rv_tendf, rw_tendf, rt_tendf, r_tendf, &
                          rr , u , v , rw , w ,                            &
                          rth , theta , thp , moist,                       &
                          th_mix, qv_mix, h_diabatic,                      &
                          u_base, v_base, qv_base,                         &
                          r, pp , rtb,  rb, zx, zy, z,                     &
                          msfu, msfv, msft, f, e, sina, cosa,              &
                          fzm, fzp, rdzu, rdzw, zeta_z, z_zeta,            &
                          rdx, rdy, khdif, kvdif,                          &
                          cf1, cf2, cf3, n_moist,                          &
                          ids, ide, jds, jde, kds, kde,                    &
                          ims, ime, jms, jme, kms, kme,                    &
                          its, ite, jts, jte, kts, kte                    )

   IMPLICIT NONE

   !  Input data.

   TYPE(grid_config_rec_type)    ,                              INTENT(IN   ) :: config_flags

   INTEGER ,               INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                            ims, ime, jms, jme, kms, kme, &
                                            its, ite, jts, jte, kts, kte
   INTEGER ,               INTENT(IN   ) :: n_moist, rk_step

   REAL , DIMENSION( ims:ime , kms:kme, jms:jme  ) , INTENT(IN   ) :: ru,      &
                                                                      rv,      &
                                                                      rom,     & 
                                                                      rthp,    &
                                                                      rrp,     &
                                                                      rtb,     &
                                                                      rb,      &
                                                                      h_diabatic

   REAL , DIMENSION( ims:ime , kms:kme, jms:jme  ) , INTENT(OUT  ) :: ru_tend, &
                                                                      rv_tend, &
                                                                      rw_tend, &
                                                                      rt_tend, &
                                                                      r_tend,  &
                                                                      th_mix,  &
                                                                      qv_mix 

   REAL , DIMENSION( ims:ime , kms:kme, jms:jme  ) , INTENT(INOUT) ::  r_tendf

   REAL , DIMENSION( ims:ime , kms:kme, jms:jme  ) , INTENT(OUT  ) :: ru_tendf, &
                                                                      rv_tendf, &
                                                                      rw_tendf, &
                                                                      rt_tendf

   REAL , DIMENSION( ims:ime , kms:kme, jms:jme , n_moist ) , INTENT(IN   ) :: &
                                                         moist

   REAL , DIMENSION( ims:ime , kms:kme, jms:jme  ) , INTENT(IN   ) :: r,       &
                                                                      rr,      &
                                                                      u,       &
                                                                      v,       &
                                                                      rw,      &
                                                                      w,       &
                                                                      rth,     &
                                                                      theta,   &
                                                                      thp,     &
                                                                      pp,      &
                                                                      zx,      &
                                                                      zy,      &
                                                                      z

   REAL , DIMENSION( ims:ime , jms:jme ) ,         INTENT(IN   ) :: msfu,    &
                                                                    msfv,    &
                                                                    msft,    &
                                                                    f,       &
                                                                    e,       &
                                                                    sina,    &
                                                                    cosa,    &
                                                                    zeta_z,  &
                                                                    z_zeta


   REAL , DIMENSION( kms:kme ) ,                 INTENT(IN   ) :: fzm,     &
                                                                  fzp,     &
                                                                  rdzu,    &
                                                                  rdzw,    &
                                                                  u_base,  &
                                                                  v_base,  &
                                                                  qv_base

   REAL ,                                      INTENT(IN   ) :: rdx,     &
                                                                rdy,     &
                                                                khdif,   &
                                                                kvdif,   &
                                                                cf1,     &
                                                                cf2,     &
                                                                cf3

   REAL    :: kdift, khdq, kvdq
   INTEGER :: i,j,k


   CALL zero_tend ( ru_tend,                      &
                    ids, ide, jds, jde, kds, kde, &
                    ims, ime, jms, jme, kms, kme, &
                    its, ite, jts, jte, kts, kte )

   CALL zero_tend ( rv_tend,                      &
                    ids, ide, jds, jde, kds, kde, &
                    ims, ime, jms, jme, kms, kme, &
                    its, ite, jts, jte, kts, kte )

   CALL zero_tend ( rw_tend,                      &
                    ids, ide, jds, jde, kds, kde, &
                    ims, ime, jms, jme, kms, kme, &
                    its, ite, jts, jte, kts, kte )

   CALL zero_tend ( rt_tend,                      &
                    ids, ide, jds, jde, kds, kde, &
                    ims, ime, jms, jme, kms, kme, &
                    its, ite, jts, jte, kts, kte )

   CALL zero_tend ( r_tend,                       &
                    ids, ide, jds, jde, kds, kde, &
                    ims, ime, jms, jme, kms, kme, &
                    its, ite, jts, jte, kts, kte )

   !  advection tendencies

   CALL advect_u ( u, u , ru_tend, ru, rv, rom,    &
                     rr, config_flags,             &
                     msfu, msfv, msft,             &
                     fzm, fzp, rdx, rdy, rdzw,     &
                     ids, ide, jds, jde, kds, kde, &
                     ims, ime, jms, jme, kms, kme, &
                     its, ite, jts, jte, kts, kte )

   CALL advect_v  ( v, v , rv_tend, ru, rv, rom,   &
                     rr, config_flags,             &
                     msfu, msfv, msft,             &
                     fzm, fzp, rdx, rdy, rdzw,     &
                     ids, ide, jds, jde, kds, kde, &
                     ims, ime, jms, jme, kms, kme, &
                     its, ite, jts, jte, kts, kte )

   CALL advect_w  ( w, w, rw_tend, ru, rv, rom,    &
                     rr, config_flags,             &
                     msfu, msfv, msft,             &
                     fzm, fzp, rdx, rdy, rdzu,     &
                     ids, ide, jds, jde, kds, kde, &
                     ims, ime, jms, jme, kms, kme, &
                     its, ite, jts, jte, kts, kte )

!-------------------

!  theta flux divergence

    CALL advect_scalar ( theta, theta, rt_tend, ru, rv, rom,  &
                         rr, config_flags,                    &
                         msfu, msfv, msft, fzm, fzp,          &
                         rdx, rdy, rdzw,                      &
                         ids, ide, jds, jde, kds, kde,        &
                         ims, ime, jms, jme, kms, kme,        &
                         its, ite, jts, jte, kts, kte        )

   !  The rho tendency (flux divergence)
   
   ! note: decoupled rho is identically one
   !r=1.

   ! NOTE!!!  - always use advect_scalar, the second order centered scheme
   !            and not a higher order scheme.

   CALL  mass_flux_divergence( r, r, r_tend, ru, rv, rom,            &
                               rr, config_flags,                     &
                               msfu, msfv, msft, fzm, fzp,           &
                               rdx, rdy, rdzw,                       &
                               ids, ide, jds, jde, kds, kde,         &
                               ims, ime, jms, jme, kms, kme,         &
                               its, ite, jts, jte, kts, kte         )

    CALL pressure_gradient ( pp, ru_tend,  rv_tend,  rw_tend,  &
                             moist, config_flags,                  &
                             msft, rdzu, rdzw, z_zeta,         &
                             zx, zy, fzm, fzp, rdx, rdy,       &
                             cf1, cf2, cf3, n_moist,           &
                             ids, ide, jds, jde, kds, kde,     &
                             ims, ime, jms, jme, kms, kme,     &
                             its, ite, jts, jte, kts, kte    )

   CALL coriolis ( ru, rv, rw,                   &
                   ru_tend,  rv_tend,  rw_tend,  &
                   config_flags,                     &
                   f, e, sina, cosa, fzm, fzp,   &
                   ids, ide, jds, jde, kds, kde, &
                   ims, ime, jms, jme, kms, kme, &
                   its, ite, jts, jte, kts, kte )

   CALL curvature ( ru, rv, rw, u, v, w,            &
                    ru_tend,  rv_tend,  rw_tend,    &
                    config_flags,                       &
                    msfu, msfv, fzm, fzp, rdx, rdy, &
                    ids, ide, jds, jde, kds, kde,   &
                    ims, ime, jms, jme, kms, kme,   &
                    its, ite, jts, jte, kts, kte   )

   CALL buoyancy ( rrp, rb, moist, rw_tend,      &
                   config_flags, n_moist,            &
                   ids, ide, jds, jde, kds, kde, &
                   ims, ime, jms, jme, kms, kme, &
                   its, ite, jts, jte, kts, kte )

!------------------------------------------------------
!**************************************************************
!
!  Next, the terms that we integrate only with forward-in-time
!  (evaluate with time t variables).
!
!**************************************************************

   forward_step: IF( rk_step == 1 ) THEN

   diff_opt1 : IF (config_flags%diff_opt .eq. 1) THEN
   
!  move to init_zero_tendency

!  CALL zero_tend ( ru_tendf,                     &
!                   ids, ide, jds, jde, kds, kde, &
!                   ims, ime, jms, jme, kms, kme, &
!                   its, ite, jts, jte, kts, kte )

!  CALL zero_tend ( rv_tendf,                     &
!                   ids, ide, jds, jde, kds, kde, &
!                   ims, ime, jms, jme, kms, kme, &
!                   its, ite, jts, jte, kts, kte )

!  CALL zero_tend ( rw_tendf,                     &
!                   ids, ide, jds, jde, kds, kde, &
!                   ims, ime, jms, jme, kms, kme, &
!                   its, ite, jts, jte, kts, kte )

!  CALL zero_tend ( rt_tendf,                     &
!                   ids, ide, jds, jde, kds, kde, &
!                   ims, ime, jms, jme, kms, kme, &
!                   its, ite, jts, jte, kts, kte )

!  CALL zero_tend ( r_tendf,                      &
!                   ids, ide, jds, jde, kds, kde, &
!                   ims, ime, jms, jme, kms, kme, &
!                   its, ite, jts, jte, kts, kte )

!  CALL zero_tend ( th_mix,                       &
!                   ids, ide, jds, jde, kds, kde, &
!                   ims, ime, jms, jme, kms, kme, &
!                   its, ite, jts, jte, kts, kte )

!  CALL zero_tend ( qv_mix,                       &
!                   ids, ide, jds, jde, kds, kde, &
!                   ims, ime, jms, jme, kms, kme, &
!                   its, ite, jts, jte, kts, kte )

   CALL horizontal_diffusion ('u', u, ru_tendf, rr, config_flags,    &
                                   msfu, msfv, msft,             &
                                   khdif, rdx, rdy,              &
                                   ids, ide, jds, jde, kds, kde, &
                                   ims, ime, jms, jme, kms, kme, &
                                   its, ite, jts, jte, kts, kte )

   CALL horizontal_diffusion ('v', v, rv_tendf, rr, config_flags,    &
                                   msfu, msfv, msft,             &
                                   khdif, rdx, rdy,              &
                                   ids, ide, jds, jde, kds, kde, &
                                   ims, ime, jms, jme, kms, kme, &
                                   its, ite, jts, jte, kts, kte )

   CALL horizontal_diffusion ('w', w, rw_tendf, rr, config_flags,    &
                                   msfu, msfv, msft,             &
                                   khdif, rdx, rdy,              &
                                   ids, ide, jds, jde, kds, kde, &
                                   ims, ime, jms, jme, kms, kme, &
                                   its, ite, jts, jte, kts, kte )
   IF (config_flags%bl_pbl_physics .eq. 0) THEN
   CALL vertical_diffusion_u ( u, ru_tendf, rr, config_flags,    &
                               u_base,                       &
                               z, zeta_z, rdzw, kvdif,       &
                               ids, ide, jds, jde, kds, kde, &
                               ims, ime, jms, jme, kms, kme, &
                               its, ite, jts, jte, kts, kte )

   CALL vertical_diffusion_v ( v, rv_tendf, rr, config_flags,    &
                               v_base,                       &
                               z, zeta_z, rdzw, kvdif,       &
                               ids, ide, jds, jde, kds, kde, &
                               ims, ime, jms, jme, kms, kme, &
                               its, ite, jts, jte, kts, kte )

   CALL vertical_diffusion ('w', w, rw_tendf, rr, config_flags,    &
                                 z, zeta_z, rdzw, kvdif,       &
                                 ids, ide, jds, jde, kds, kde, &
                                 ims, ime, jms, jme, kms, kme, &
                                 its, ite, jts, jte, kts, kte )
   ENDIF

   !  Theta tendency computations.

!  this is a surrogate for a prandtl number = 3.0

   khdq = 3.*khdif
   kvdq = 3.*kvdif

   CALL horizontal_diffusion ('t', thp, th_mix,  rr, config_flags,        &
                                   msfu, msfv, msft, khdq , rdx, rdy, &
                                   ids, ide, jds, jde, kds, kde,      &
                                   ims, ime, jms, jme, kms, kme,      &
                                   its, ite, jts, jte, kts, kte      )

   IF (config_flags%bl_pbl_physics .eq. 0) &
   CALL vertical_diffusion ('t', thp, th_mix , rr, config_flags,      &
                                 z, zeta_z, rdzw, kvdq ,          &
                                 ids, ide, jds, jde, kds, kde,    &
                                 ims, ime, jms, jme, kms, kme,    &
                                 its, ite, jts, jte, kts, kte    )

   IF( n_moist >= PARAM_FIRST_SCALAR ) then

!
!   Here we are diffusing the perturbation qv, by first
!   diffusing the full qv and subtracting off diffusion 
!   of the base-state qv.  This is rather ugly but we can
!   work on a more satisfactory way of dealing with things later.
!   Diffusion of qv' is a common appraoch in idealized cloud-model
!   studies.
!   WCS 19 May 1999

     CALL horizontal_diffusion ('t', moist(ims,kms,jms,P_QV), qv_mix,   &
     		                     rr, config_flags,                  &
                                     msfu, msfv, msft, khdq , rdx, rdy, &
                                     ids, ide, jds, jde, kds, kde,      &
                                     ims, ime, jms, jme, kms, kme,      &
                                     its, ite, jts, jte, kts, kte      )

     IF (config_flags%bl_pbl_physics .eq. 0) &
       CALL vertical_diffusion_qv ('t', moist(ims,kms,jms,P_QV), qv_mix, &
 	                                rr, config_flags,                &
                                        qv_base,                         &
                                        z, zeta_z, rdzw, kvdq ,          &
                                        ids, ide, jds, jde, kds, kde,    &
                                        ims, ime, jms, jme, kms, kme,    &
                                        its, ite, jts, jte, kts, kte    )

     ENDIF


   CALL mix_theta_m ( rt_tendf, th_mix, qv_mix,     &
                      rr, moist(ims,kms,jms,P_QV), theta, n_moist,    &
                      ids, ide, jds, jde, kds, kde, &
                      ims, ime, jms, jme, kms, kme, &
                      its, ite, jts, jte, kts, kte )

   END IF diff_opt1

   CALL add_diabatic( rt_tendf, h_diabatic,         &
                      ids, ide, jds, jde, kds, kde, &
                      ims, ime, jms, jme, kms, kme, &
                      its, ite, jts, jte, kts, kte )

   END IF forward_step



END SUBROUTINE rk_tendency

!-------------------------------------------------------------------------------
SUBROUTINE rk_addtend_dry ( ru_tend, rv_tend, rw_tend, rt_tend, r_tend, &
                          ru_tendf, rv_tendf, rw_tendf, rt_tendf, r_tendf,&
                          ids,ide, jds,jde, kds,kde,  & ! domain dims
                          ims,ime, jms,jme, kms,kme,  & ! memory dims
                          ips,ipe, jps,jpe, kps,kpe,  & ! patch  dims
                          its,ite, jts,jte, kts,kte )
   IMPLICIT NONE

   !  Input data.

   INTEGER ,               INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                            ims, ime, jms, jme, kms, kme, &
                                            ips, ipe, jps, jpe, kps, kpe, &
                                            its, ite, jts, jte, kts, kte

   REAL , DIMENSION( ims:ime , kms:kme, jms:jme  ) , INTENT(INOUT) :: ru_tend, &
                                                                      rv_tend, &
                                                                      rw_tend, &
                                                                      rt_tend, &
                                                                      r_tend
   REAL , DIMENSION( ims:ime , kms:kme, jms:jme  ) , INTENT(IN   ) :: ru_tendf, &
                                                                      rv_tendf, &
                                                                      rw_tendf, &
                                                                      rt_tendf, &
                                                                      r_tendf
! Local
   INTEGER :: i, j, k


!  Finally, add the forward-step tendency to the rk_tendency
!
!  we'll just do it here as opposed to in a subroutine

   DO j = jts,jte
   DO k = kts,kte
   DO i = its,ite
     ru_tend(i,k,j) = ru_tend(i,k,j) + ru_tendf(i,k,j)
     rv_tend(i,k,j) = rv_tend(i,k,j) + rv_tendf(i,k,j)
     rw_tend(i,k,j) = rw_tend(i,k,j) + rw_tendf(i,k,j)
     rt_tend(i,k,j) = rt_tend(i,k,j) + rt_tendf(i,k,j)
      r_tend(i,k,j) =  r_tend(i,k,j) +  r_tendf(i,k,j)
   ENDDO
   ENDDO
   ENDDO

END SUBROUTINE rk_addtend_dry
!-------------------------------------------------------------------------------

SUBROUTINE rk_scalar_tend ( num_3d, isc, config_flags,    &
                             rk_step, dt,                 &
                             ru, rv, rom,                  &
                             rr_old, rr_new,               &
                             scalar_old, scalar,           &
                             qv_mix, moist_step,           &
                             fzm, fzp, scalar_tends,       &
                             advect_tend,                  &
                             msfu, msfv, msft,             &
                             rdx, rdy, rdzu, rdzw, khdif,  &
                             z, zeta_z, kvdif,             &
                             ids, ide, jds, jde, kds, kde, &
                             ims, ime, jms, jme, kms, kme, &
                             its, ite, jts, jte, kts, kte )

   IMPLICIT NONE

   !  Input data.

   TYPE(grid_config_rec_type)    ,           INTENT(IN   ) :: config_flags

   INTEGER ,                INTENT(IN   ) :: num_3d, rk_step, isc
   INTEGER ,                INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                             ims, ime, jms, jme, kms, kme, &
                                             its, ite, jts, jte, kts, kte

   LOGICAL , INTENT(IN   ) :: moist_step

   REAL, DIMENSION(ims:ime, kms:kme, jms:jme ),                       &
         INTENT(INOUT)                                  :: scalar,    &
                                                           scalar_old

   REAL, DIMENSION(ims:ime, kms:kme, jms:jme ),                         &
         INTENT(  OUT)                                                  &
                                                        :: scalar_tends
   REAL, DIMENSION(ims:ime, kms:kme, jms:jme  ) :: advect_tend


   REAL, DIMENSION(ims:ime, kms:kme, jms:jme  ), INTENT(IN   ) ::      ru,     &
                                                                      rv,     &
                                                                      rom,    &
                                                                      rr_old, &
                                                                      rr_new, &
                                                                      qv_mix, &
                                                                      z

   REAL , DIMENSION( kms:kme ) ,                 INTENT(IN   ) :: fzm,  &
                                                                  fzp,  &
                                                                  rdzu, &
                                                                  rdzw

   REAL , DIMENSION( ims:ime , jms:jme ) ,       INTENT(IN   ) :: msfu,    &
                                                                  msfv,    &
                                                                  msft,    &
                                                                  zeta_z

   REAL ,                                        INTENT(IN   ) :: rdx,     &
                                                                  rdy,     &
                                                                  khdif,   &
                                                                  kvdif

   REAL ,                                        INTENT(IN   ) :: dt


   ! Local data
  
   INTEGER :: i,j,k

   REAL    :: khdq, kvdq, tendency

!  this is a surrogate for a prandtl number of 3.0

   khdq = 3.*khdif
   kvdq = 3.*kvdif

   CALL zero_tend ( advect_tend(ims,kms,jms),     &
                    ids, ide, jds, jde, kds, kde, &
                    ims, ime, jms, jme, kms, kme, &
                    its, ite, jts, jte, kts, kte )

   CALL advect_scalar ( scalar(ims,kms,jms),        &
                        scalar(ims,kms,jms),        &
                        advect_tend(ims,kms,jms),      &
                        ru, rv, rom, rr_new,           &
                        config_flags,                  &
                        msfu, msfv, msft, fzm, fzp,    &
                        rdx, rdy, rdzw,                &
                        ids, ide, jds, jde, kds, kde,  &
                        ims, ime, jms, jme, kms, kme,  &
                        its, ite, jts, jte, kts, kte  )

   rk_step_1: IF( rk_step == 1 ) THEN

   diff_opt1 : IF (config_flags%diff_opt .eq. 1) THEN

     !  zero out the non-advective tendency. this is
     !  computed using time (t) data and saved for use
     !  during both predictor and corrector parts of the rk step.

   theta_mix: IF( moist_step .and. (isc == P_QV) ) THEN

     CALL mix_qv(  scalar_tends(ims,kms,jms), &
                     qv_mix, rr_old,               &
                     ids, ide, jds, jde, kds, kde, &
                     ims, ime, jms, jme, kms, kme, &
                     its, ite, jts, jte, kts, kte )

   ELSE

   
     CALL horizontal_diffusion ( 'm', scalar(ims,kms,jms),               &
                                      scalar_tends(ims,kms,jms), rr_old, &
                                      config_flags,                         &
                                      msfu, msfv, msft, khdq , rdx, rdy,    &
                                      ids, ide, jds, jde, kds, kde,         &
                                      ims, ime, jms, jme, kms, kme,         &
                                      its, ite, jts, jte, kts, kte         )

     IF (config_flags%bl_pbl_physics .eq. 0) &
     CALL vertical_diffusion (  'm', scalar(ims,kms,jms),               &
                                     scalar_tends(ims,kms,jms), rr_old, &
                                     config_flags,                         &
                                     z, zeta_z, rdzw, kvdq ,               &
                                     ids, ide, jds, jde, kds, kde,         &
                                     ims, ime, jms, jme, kms, kme,         &
                                     its, ite, jts, jte, kts, kte         )

    ENDIF theta_mix

    END IF diff_opt1

    ENDIF rk_step_1

END SUBROUTINE rk_scalar_tend

!-------------------------------------------------------------------------------

SUBROUTINE rk_update_scalar( scalar_1, scalar_2, sc_tend,  &
                              advect_tend,config_flags,     &
                              rr_old, rr_new, dt,           &
                              rk_step, num_3d,spec_zone,   &
                              ids, ide, jds, jde, kds, kde, &
                              ims, ime, jms, jme, kms, kme, &
                              its, ite, jts, jte, kts, kte )

   IMPLICIT NONE

   !  Input data.

   TYPE(grid_config_rec_type   ) ,   INTENT(IN   ) :: config_flags

   INTEGER ,                INTENT(IN   ) :: num_3d, rk_step, spec_zone
   INTEGER ,                INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                             ims, ime, jms, jme, kms, kme, &
                                             its, ite, jts, jte, kts, kte

   REAL,                    INTENT(IN   ) :: dt

   REAL, DIMENSION(ims:ime, kms:kme, jms:jme ),                &
         INTENT(INOUT)                                  :: scalar_1,  &
                                                           scalar_2,  &
                                                           sc_tend

   REAL, DIMENSION(ims:ime, kms:kme, jms:jme ),                &
         INTENT(IN)                                  :: advect_tend

   REAL, DIMENSION(ims:ime, kms:kme, jms:jme  ), INTENT(IN   ) ::     rr_old, &
                                                                      rr_new

   REAL, DIMENSION(its:ite, kts:kte, jts:jte  ) :: tendency

   INTEGER :: i,j,k,i_start,i_end,j_start,j_end,k_start,k_end
   INTEGER :: i_start_spc,i_end_spc,j_start_spc,j_end_spc,k_start_spc,k_end_spc

!
    DO  j = jts, min(jte,jde-1)
    DO  k = kts, min(kte,kde-1)
    DO  i = its, min(ite,ide-1)
        tendency(i,k,j) = 0.
    ENDDO
    ENDDO
    ENDDO


!
!  set loop limits.

      i_start = its
      i_end   = ite
      j_start = jts
      j_end   = jte
      k_start = kts
      k_end   = kte-1
      IF(j_end == jde) j_end = j_end - 1
      IF(i_end == ide) i_end = i_end - 1

      i_start_spc = i_start
      i_end_spc   = i_end
      j_start_spc = j_start
      j_end_spc   = j_end
      k_start_spc = k_start
      k_end_spc   = k_end

    IF( config_flags%nested .or. config_flags%specified ) THEN
      i_start = max( its,ids+spec_zone )
      i_end   = min( ite,ide-spec_zone-1 )
      j_start = max( jts,jds+spec_zone )
      j_end   = min( jte,jde-spec_zone-1 )
      k_start = kts
      k_end   = min( kte, kde-1 )
    ENDIF

    DO  j = j_start,j_end
    DO  k = k_start,k_end
    DO  i = i_start,i_end
        tendency(i,k,j) = advect_tend(i,k,j)
    ENDDO
    ENDDO
    ENDDO

    DO  j = j_start_spc,j_end_spc
    DO  k = k_start_spc,k_end_spc
    DO  i = i_start_spc,i_end_spc
        tendency(i,k,j) = tendency(i,k,j) + sc_tend(i,k,j)
    ENDDO
    ENDDO
    ENDDO


    IF ( rk_step == 1 ) THEN

      !  replace t-dt values (in scalar_1) with t values scalar_2,
      !  then compute new values by adding tendency to values at t

      DO  j = j_start_spc,j_end_spc
      DO  k = k_start_spc,k_end_spc
      DO  i = i_start_spc,i_end_spc

        scalar_1(i,k,j) = scalar_2(i,k,j)
        tendency(i,k,j)    = tendency(i,k,j) +  sc_tend(i,k,j) + advect_tend(i,k,j)
        scalar_2(i,k,j) = (rr_old(i,k,j)*scalar_1(i,k,j)   &
                                + dt*tendency(i,k,j))/rr_new(i,k,j)

      ENDDO
      ENDDO
      ENDDO

    ELSE

      !  just compute new values, scalar_1 already at time t.

      DO  j = j_start_spc,j_end_spc
      DO  k = k_start_spc,k_end_spc
      DO  i = i_start_spc,i_end_spc

        scalar_2(i,k,j) = (rr_old(i,k,j)*scalar_1(i,k,j)   &
                                + dt*tendency(i,k,j))/rr_new(i,k,j)

      ENDDO
      ENDDO
      ENDDO

    END IF

END SUBROUTINE rk_update_scalar

!-----------------------------------------------------------------------
SUBROUTINE init_zero_tendency(ru_tendf, rv_tendf, rw_tendf, rt_tendf,  &
                              r_tendf, tke_tendf, th_mix, qv_mix,      &
                              moist_tendf,chem_tendf,                  &
                              n_moist,n_chem,rk_step,                 &
                              ids, ide, jds, jde, kds, kde,            &
                              ims, ime, jms, jme, kms, kme,            &
                              its, ite, jts, jte, kts, kte             )
!-----------------------------------------------------------------------
   IMPLICIT NONE
!-----------------------------------------------------------------------

   INTEGER ,       INTENT(IN   ) :: ids, ide, jds, jde, kds, kde, &
                                    ims, ime, jms, jme, kms, kme, &
                                    its, ite, jts, jte, kts, kte

   INTEGER ,       INTENT(IN   ) :: n_moist,n_chem,rk_step

   REAL , DIMENSION( ims:ime , kms:kme, jms:jme  ) , INTENT(INOUT) ::  &
						             ru_tendf, &
                                                             rv_tendf, &
                                                             rw_tendf, &
                                                             rt_tendf, &
                                                              r_tendf, &
                                                            tke_tendf, &
                                                               th_mix, &
                                                               qv_mix

   REAL , DIMENSION(ims:ime, kms:kme, jms:jme, n_moist),INTENT(INOUT)::&
                                                          moist_tendf

   REAL , DIMENSION(ims:ime, kms:kme, jms:jme, n_chem ),INTENT(INOUT)::&
                                                          chem_tendf

! LOCAL VARS

   INTEGER :: im, ic

   CALL zero_tend ( ru_tendf,                        &
                    ids, ide, jds, jde, kds, kde,    &
                    ims, ime, jms, jme, kms, kme,    &
                    its, ite, jts, jte, kts, kte     )

   CALL zero_tend ( rv_tendf,                        &
                    ids, ide, jds, jde, kds, kde,    &
                    ims, ime, jms, jme, kms, kme,    &
                    its, ite, jts, jte, kts, kte     )

   CALL zero_tend ( rw_tendf,                        &
                    ids, ide, jds, jde, kds, kde,    &
                    ims, ime, jms, jme, kms, kme,    &
                    its, ite, jts, jte, kts, kte     )

   CALL zero_tend ( rt_tendf,                        &
                    ids, ide, jds, jde, kds, kde,    &
                    ims, ime, jms, jme, kms, kme,    &
                    its, ite, jts, jte, kts, kte     )

   CALL zero_tend ( r_tendf,                         &
                    ids, ide, jds, jde, kds, kde,    &
                    ims, ime, jms, jme, kms, kme,    &
                    its, ite, jts, jte, kts, kte     )

   CALL zero_tend ( tke_tendf,                       &
                    ids, ide, jds, jde, kds, kde,    &
                    ims, ime, jms, jme, kms, kme,    &
                    its, ite, jts, jte, kts, kte     )

   CALL zero_tend ( th_mix,                          &
                    ids, ide, jds, jde, kds, kde,    &
                    ims, ime, jms, jme, kms, kme,    &
                    its, ite, jts, jte, kts, kte     )

   CALL zero_tend ( qv_mix,                          &
                    ids, ide, jds, jde, kds, kde,    &
                    ims, ime, jms, jme, kms, kme,    &
                    its, ite, jts, jte, kts, kte     )

   DO im=PARAM_FIRST_SCALAR,n_moist
      CALL zero_tend ( moist_tendf(ims,kms,jms,im),  &
                       ids, ide, jds, jde, kds, kde, &
                       ims, ime, jms, jme, kms, kme, &
                       its, ite, jts, jte, kts, kte  )
   ENDDO

   DO ic=PARAM_FIRST_SCALAR,n_chem
      CALL zero_tend ( chem_tendf(ims,kms,jms,ic),   &
                       ids, ide, jds, jde, kds, kde, &
                       ims, ime, jms, jme, kms, kme, &
                       its, ite, jts, jte, kts, kte  )
   ENDDO

END SUBROUTINE init_zero_tendency

!===================================================================
SUBROUTINE update_phy_ten(rt_tendf,ru_tendf,rv_tendf,moist_tendf,  &
                      RTHRATEN,RTHBLTEN,RTHCUTEN,RUBLTEN,RVBLTEN,  &
                      RQVBLTEN,RQCBLTEN,RQIBLTEN,                  &
                      RQVCUTEN,RQCCUTEN,RQRCUTEN,RQICUTEN,RQSCUTEN,&
		      n_moist,config_flags,rk_step,		   &
                      ids, ide, jds, jde, kds, kde,                &
                      ims, ime, jms, jme, kms, kme,                &
                      its, ite, jts, jte, kts, kte                 )
!-------------------------------------------------------------------
   IMPLICIT NONE
!-------------------------------------------------------------------

   TYPE(grid_config_rec_type   ) ,   INTENT(IN   ) :: config_flags

   INTEGER , INTENT(IN)        ::  ids, ide, jds, jde, kds, kde,   &
                                   ims, ime, jms, jme, kms, kme,   &
                                   its, ite, jts, jte, kts, kte,   &
				   n_moist,rk_step

   REAL , DIMENSION(ims:ime , kms:kme, jms:jme),INTENT(INOUT) ::   &
                                                         ru_tendf, &
                                                         rv_tendf, &
                                                         rt_tendf

   REAL , DIMENSION(ims:ime, kms:kme, jms:jme, n_moist),           &
          INTENT(INOUT)     ::                        moist_tendf

   REAL, DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(IN  ) ::   &
                                                       RTHRATEN, &
                                                       RTHBLTEN, &
                                                       RTHCUTEN, &
                                                        RUBLTEN, &
                                                        RVBLTEN, &
                                                       RQVBLTEN, &
                                                       RQCBLTEN, &
                                                       RQIBLTEN, &
                                                       RQVCUTEN, &
                                                       RQCCUTEN, &
                                                       RQRCUTEN, &
                                                       RQICUTEN, &
                                                       RQSCUTEN
!------------------------------------------------------------------

!  set up loop bounds for this grid's boundary conditions

   if (config_flags%ra_lw_physics .gt. 0 .or.                  &
       config_flags%ra_sw_physics .gt. 0)                      &
      CALL phy_ra_ten(config_flags,rt_tendf,RTHRATEN,          &
                      ids, ide, jds, jde, kds, kde,            &
                      ims, ime, jms, jme, kms, kme,            &
                      its, ite, jts, jte, kts, kte             )

   if (config_flags%bl_pbl_physics .gt. 0)                     &
      CALL phy_bl_ten(config_flags,rk_step,n_moist,           &
                      rt_tendf,ru_tendf,rv_tendf,moist_tendf,  &
                      RTHBLTEN,RUBLTEN,RVBLTEN,                &
		      RQVBLTEN,RQCBLTEN,RQIBLTEN,              &
                      ids, ide, jds, jde, kds, kde,            &
                      ims, ime, jms, jme, kms, kme,            &
                      its, ite, jts, jte, kts, kte             )

   if (config_flags%cu_physics .gt. 0)                         &
      CALL phy_cu_ten(config_flags,rk_step,n_moist,rt_tendf,  &
                      RTHCUTEN,RQVCUTEN,RQCCUTEN,RQRCUTEN,     &
                      RQICUTEN,RQSCUTEN,moist_tendf,           &
                      ids, ide, jds, jde, kds, kde,            &
                      ims, ime, jms, jme, kms, kme,            &
                      its, ite, jts, jte, kts, kte             )

END SUBROUTINE update_phy_ten

!=================================================================
SUBROUTINE phy_ra_ten(config_flags,rt_tendf,RTHRATEN,            &
                      ids, ide, jds, jde, kds, kde,              &
                      ims, ime, jms, jme, kms, kme,              &
                      its, ite, jts, jte, kts, kte               )
!-----------------------------------------------------------------
   IMPLICIT NONE
!-----------------------------------------------------------------
   TYPE(grid_config_rec_type  ) , INTENT(IN   ) :: config_flags

   INTEGER , INTENT(IN)        ::  ids, ide, jds, jde, kds, kde, &
                                   ims, ime, jms, jme, kms, kme, &
                                   its, ite, jts, jte, kts, kte

   REAL, DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(IN  ) ::   &
                                                       RTHRATEN

   REAL, DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(OUT ) ::   &
                                                       rt_tendf

! LOCAL VARS

   INTEGER :: i,j,k

   CALL add_a2a(rt_tendf,RTHRATEN,config_flags,                  &
                ids,ide, jds, jde, kds, kde,                     &
                ims, ime, jms, jme, kms, kme,                    &
                its, ite, jts, jte, kts, kte                     )

END SUBROUTINE phy_ra_ten

!=================================================================
SUBROUTINE phy_bl_ten(config_flags,rk_step,n_moist,             &
                      rt_tendf,ru_tendf,rv_tendf,moist_tendf,    &
                      RTHBLTEN,RUBLTEN,RVBLTEN,                  & 
		      RQVBLTEN,RQCBLTEN,RQIBLTEN,                &
                      ids, ide, jds, jde, kds, kde,              &
                      ims, ime, jms, jme, kms, kme,              &
                      its, ite, jts, jte, kts, kte               )
!-----------------------------------------------------------------
   IMPLICIT NONE
!-----------------------------------------------------------------
   TYPE(grid_config_rec_type) ,     INTENT(IN   ) :: config_flags

   INTEGER , INTENT(IN)        ::  ids, ide, jds, jde, kds, kde, &
                                   ims, ime, jms, jme, kms, kme, &
                                   its, ite, jts, jte, kts, kte, &
 				   n_moist, rk_step

   REAL , DIMENSION(ims:ime, kms:kme, jms:jme, n_moist),         &
          INTENT(INOUT)     ::                      moist_tendf

   REAL, DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(IN  ) ::   & 
                                                       RTHBLTEN, &
                                                        RUBLTEN, &
                                                        RVBLTEN, &
		                                       RQVBLTEN, &
						       RQCBLTEN, &
     						       RQIBLTEN

   REAL, DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(INOUT)::   &
                                                       rt_tendf, &
                                                       ru_tendf, &
                                                       rv_tendf
! LOCAL VARS

   INTEGER :: i,j,k,IBGN,IEND,JBGN,JEND

!-----------------------------------------------------------------

   SELECT CASE(config_flags%bl_pbl_physics)

      CASE (MRFSCHEME)

           CALL add_a2a(rt_tendf,RTHBLTEN,config_flags,          &
                ids,ide, jds, jde, kds, kde,                     &
                ims, ime, jms, jme, kms, kme,                    &
                its, ite, jts, jte, kts, kte                     )

           CALL add_a2c_u(ru_tendf,RUBLTEN,config_flags,         &
                ids,ide, jds, jde, kds, kde,                     &
                ims, ime, jms, jme, kms, kme,                    &
                its, ite, jts, jte, kts, kte                     )

           CALL add_a2c_v(rv_tendf,RVBLTEN,config_flags,         &
                ids,ide, jds, jde, kds, kde,                     &
                ims, ime, jms, jme, kms, kme,                    &
                its, ite, jts, jte, kts, kte                     )

           CALL add_a2a(moist_tendf(ims,kms,jms,P_QV),RQVBLTEN,  &
                config_flags,                                    &
                ids,ide, jds, jde, kds, kde,                     &
                ims, ime, jms, jme, kms, kme,                    &
                its, ite, jts, jte, kts, kte                     )

           CALL add_a2a(moist_tendf(ims,kms,jms,P_QC),RQCBLTEN,  &
                config_flags,                                    &
                ids,ide, jds, jde, kds, kde,                     &
                ims, ime, jms, jme, kms, kme,                    &
                its, ite, jts, jte, kts, kte                     )
     
           CALL add_a2a(moist_tendf(ims,kms,jms,P_QI),RQIBLTEN,  &
                config_flags,                                    &
                ids,ide, jds, jde, kds, kde,                     &
                ims, ime, jms, jme, kms, kme,                    &
                its, ite, jts, jte, kts, kte                     )

      CASE DEFAULT

       print*,'The pbl scheme does not exist'

   END SELECT

END SUBROUTINE phy_bl_ten

!=================================================================
SUBROUTINE phy_cu_ten(config_flags,rk_step,n_moist,rt_tendf,    &
                      RTHCUTEN,RQVCUTEN,RQCCUTEN,RQRCUTEN,       &
                      RQICUTEN,RQSCUTEN,moist_tendf,		 &
                      ids, ide, jds, jde, kds, kde,              &
                      ims, ime, jms, jme, kms, kme,              &
                      its, ite, jts, jte, kts, kte               )
!-----------------------------------------------------------------
   IMPLICIT NONE
!-----------------------------------------------------------------
   TYPE(grid_config_rec_type  ) , INTENT(IN   ) :: config_flags

   INTEGER , INTENT(IN)        ::  ids, ide, jds, jde, kds, kde, &
                                   ims, ime, jms, jme, kms, kme, &
                                   its, ite, jts, jte, kts, kte, &
 				   n_moist, rk_step

   REAL , DIMENSION(ims:ime, kms:kme, jms:jme, n_moist),         &
          INTENT(INOUT)     ::                      moist_tendf

   REAL, DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(IN  ) ::   &
                                                       RTHCUTEN, &
                                                       RQVCUTEN, &
                                                       RQCCUTEN, &
                                                       RQRCUTEN, &
                                                       RQICUTEN, &
                                                       RQSCUTEN

   REAL, DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(OUT ) ::   &
					     	       rt_tendf

! LOCAL VARS

   INTEGER :: i,j,k

   SELECT CASE (config_flags%cu_physics)   

   CASE (KFSCHEME)
        CALL add_a2a(rt_tendf,RTHCUTEN,config_flags,             &
                ids,ide, jds, jde, kds, kde,                     &
                ims, ime, jms, jme, kms, kme,                    &
                its, ite, jts, jte, kts, kte                     )

        CALL add_a2a(moist_tendf(ims,kms,jms,P_QV),RQVCUTEN,     &
                config_flags,                                    &
                ids,ide, jds, jde, kds, kde,                     &
                ims, ime, jms, jme, kms, kme,                    &
                its, ite, jts, jte, kts, kte                     )

        CALL add_a2a(moist_tendf(ims,kms,jms,P_QC),RQCCUTEN,     &
                config_flags,                                    &
                ids,ide, jds, jde, kds, kde,                     &
                ims, ime, jms, jme, kms, kme,                    &
                its, ite, jts, jte, kts, kte                     )

        CALL add_a2a(moist_tendf(ims,kms,jms,P_QR),RQRCUTEN,     &
                config_flags,                                    &
                ids,ide, jds, jde, kds, kde,                     &
                ims, ime, jms, jme, kms, kme,                    &
                its, ite, jts, jte, kts, kte                     )

        if (P_QI .ge. PARAM_FIRST_SCALAR)                                         &
        CALL add_a2a(moist_tendf(ims,kms,jms,P_QI),RQICUTEN,     &
                config_flags,                                    &
                ids,ide, jds, jde, kds, kde,                     &
                ims, ime, jms, jme, kms, kme,                    &
                its, ite, jts, jte, kts, kte                     )

        if (P_QS .ge. PARAM_FIRST_SCALAR)                                         &
        CALL add_a2a(moist_tendf(ims,kms,jms,P_QS),RQSCUTEN,     &
                config_flags,                                    &
                ids,ide, jds, jde, kds, kde,                     &
                ims, ime, jms, jme, kms, kme,                    &
                its, ite, jts, jte, kts, kte                     )

   CASE (BMJSCHEME)
        CALL add_a2a(rt_tendf,RTHCUTEN,                          &
                config_flags,                                    &
                ids,ide, jds, jde, kds, kde,                     &
                ims, ime, jms, jme, kms, kme,                    &
                its, ite, jts, jte, kts, kte                     )

        CALL add_a2a(moist_tendf(ims,kms,jms,P_QV),RQVCUTEN,     &
                config_flags,                                    &
                ids,ide, jds, jde, kds, kde,                     &
                ims, ime, jms, jme, kms, kme,                    &
                its, ite, jts, jte, kts, kte                     )

   CASE DEFAULT

   END SELECT

END SUBROUTINE phy_cu_ten

!----------------------------------------------------------------------
SUBROUTINE advance_ppt(RTHCUTEN,RQVCUTEN,RQCCUTEN,RQRCUTEN,           &
                     RQICUTEN,RQSCUTEN,RAINC,RAINCV,NCA,              &
                     config_flags,                                    &
                     ids,ide, jds,jde, kds,kde,                       &
                     ims,ime, jms,jme, kms,kme,                       &
                     its,ite, jts,jte, kts,kte                        )     
!----------------------------------------------------------------------
   USE module_bc
   USE module_state_description
   USE module_cu_kf
!----------------------------------------------------------------------
   IMPLICIT NONE
!----------------------------------------------------------------------
   TYPE(grid_config_rec_type),  INTENT(IN   )    :: config_flags

   INTEGER,      INTENT(IN   )    ::                             &
                                      ids,ide, jds,jde, kds,kde, &
                                      ims,ime, jms,jme, kms,kme, &
                                      its,ite, jts,jte, kts,kte


   REAL, DIMENSION( ims:ime, kms:kme, jms:jme ),                 &
         INTENT(INOUT)  ::                             RTHCUTEN, &
                                                       RQVCUTEN, &
                                                       RQCCUTEN, &
                                                       RQRCUTEN, &
                                                       RQICUTEN, &
                                                       RQSCUTEN

   REAL, DIMENSION( ims:ime , jms:jme ),                         &
          INTENT(INOUT) ::                                RAINC, &
                                                         RAINCV

   INTEGER, DIMENSION( ims:ime , jms:jme ),                      &
         INTENT(INOUT) ::                     	  	    NCA

! LOCAL  VAR

   INTEGER :: i,j,k,i_start,i_end,j_start,j_end,k_start,k_end

!-----------------------------------------------------------------

   IF (config_flags%cu_physics .eq. 0) return

! SET START AND END POINTS FOR TILES

   i_start = its
   i_end   = min( ite,ide-1 )
   j_start = jts
   j_end   = min( jte,jde-1 )
!
   IF( config_flags%nested .or. config_flags%specified ) THEN
     i_start = max( its,ids+1 )
     i_end   = min( ite,ide-2 )
     j_start = max( jts,jds+1 )
     j_end   = min( jte,jde-2 )
   ENDIF
!
   k_start = kts
   k_end = min( kte, kde-1 )

! Update total cumulus scheme precipitation

! in mm  

   DO J = j_start,j_end
   DO i = i_start,i_end
      RAINC(I,J)=RAINC(I,J)+RAINCV(I,J)
   ENDDO
   ENDDO

   SELECT CASE (config_flags%cu_physics)

   CASE (KFSCHEME)

        DO J = j_start,j_end
        DO i = i_start,i_end

           IF ( NCA(I,J).GT. 0 ) THEN

              IF ( NCA(I,J) .eq. 1 ) THEN

              ! set tendency to zero
                 RAINCV(I,J)=0.
                 DO k = k_start,k_end
                    RTHCUTEN(i,k,j)=0.
                    RQVCUTEN(i,k,j)=0.
                    RQCCUTEN(i,k,j)=0.
                    RQRCUTEN(i,k,j)=0.
                    if (P_QI .ge. PARAM_FIRST_SCALAR) RQICUTEN(i,k,j)=0.
                    if (P_QS .ge. PARAM_FIRST_SCALAR) RQSCUTEN(i,k,j)=0.
                 ENDDO
              ENDIF

              NCA(I,J)=NCA(I,J)-1 ! Decrease NCA

           ENDIF
!
        ENDDO
        ENDDO

   CASE DEFAULT

   END SELECT

END SUBROUTINE advance_ppt


END MODULE module_rk



