!WRF:MODEL_LAYER:DYNAMICS
!

!  SMALL_STEP code for the geometric height coordinate model
!
!---------------------------------------------------------------------------

MODULE module_small_step

   USE module_configure
   USE module_model_constants

   !  This module comprises the small time step code.  The following 
   !  routines are in alphabetical order in the CONTAINS section of
   !  this module.  The routines are broken into apparently small pieces
   !  to remove any need for inter-process communications.

   !  ROUTINE                  CALLS MADE           CALLED BY
   !-----------------------------------------------------------------------
   !  advance_uv               NONE                 solve (outside of module)
   !  advance_w                NONE                 solve (outside of module)
   !  calc_coef_w              NONE                 solve (outside of module)
   !  small_step_decouple      NONE                 solve (outside of module)
   !  small_step_recouple      NONE                 solve (outside of module)
   !  time_filter              NONE                 solve (outside of module)
   !  save_old_time            NONE                 solve (outside of module)
   !  time_filter_scalars      NONE                 solve (outside of module)
   !  sumflux                  NONE                 solve (outside of module)
   !  time_filter_dry          time_filter          solve (outside of module)

CONTAINS

!----------------------------------------------------------------------


      SUBROUTINE advance_uv( ru, rv, config_flags,       &
                             ru_tend, rv_tend,           &
                             du, dv,                     &
                             zx, zy, pip, pib, zz, msft, &
                             fzm, fzp, rdzw , rt,        &
                             rdx, rdy, dts,              &
                             cf1, cf2, cf3,              &
                             spec_zone,                  &
                             ids,ide, jds,jde, kds,kde,  & ! domain dims
                             ims,ime, jms,jme, kms,kme,  & ! memory dims
                             its,ite, jts,jte, kts,kte )   ! tile   dims

      IMPLICIT NONE  ! religion first

! stuff coming in

      TYPE(grid_config_rec_type),    INTENT(IN   )    :: config_flags

      INTEGER,      INTENT(IN   )    :: ids,ide, jds,jde, kds,kde
      INTEGER,      INTENT(IN   )    :: ims,ime, jms,jme, kms,kme
      INTEGER,      INTENT(IN   )    :: its,ite, jts,jte, kts,kte
      INTEGER,      INTENT(IN   )    :: spec_zone

      REAL, DIMENSION( ims:ime , kms:kme, jms:jme ), &
            INTENT(INOUT) ::                          &
                                                  ru, &
                                                  du, &
                                                  rv, &
                                                  dv, &
                                                  rt

      REAL, DIMENSION( ims:ime , kms:kme, jms:jme ), &
            INTENT(IN   ) ::                          &
                                             ru_tend, &
                                             rv_tend, &
                                             zx,      &
                                             zy,      &
                                             pip,     &
                                             pib

      REAL, DIMENSION( ims:ime , jms:jme ),    INTENT(IN   ) :: zz,     &
                                                                msft

      REAL, DIMENSION( kms:kme ),              INTENT(IN   ) :: fzp,    &
                                                                fzm ,   &
                                                                rdzw

      REAL,                                    INTENT(IN   ) :: rdx,    &
                                                                rdy,    &
                                                                dts,    &
					                        cf1,    &
					                        cf2,    &
						                cf3
     

!  Local 3d array from the stack (note tile size)

!      REAL, DIMENSION(its-1:ite+1, jts-1:jte+1, kts:kte)    :: dpzxy
      REAL, DIMENSION(its-1:ite+1, 2)    :: dpzxyu
      REAL, DIMENSION(its-1:ite+1, 2)    :: dpzxyv
      INTEGER :: kp0, kp1,ktmp


      INTEGER :: i,j,k, i_start, i_end, j_start, j_end, k_start, k_end
      INTEGER :: i_endu, j_endv, k_endw
      INTEGER :: i_start_up, i_end_up, j_start_up, j_end_up
      INTEGER :: i_start_vp, i_end_vp, j_start_vp, j_end_vp
      INTEGER :: i_start_u_tend, i_end_u_tend, j_start_v_tend, j_end_v_tend

!  now, the real work.
!  set the loop bounds taking into account boundary conditions.

    IF( config_flags%nested .or. config_flags%specified ) THEN
      i_start = max( its,ids+spec_zone )
      i_end   = min( ite,ide-spec_zone-1 )
      j_start = max( jts,jds+spec_zone )
      j_end   = min( jte,jde-spec_zone-1 )
      k_start = kts
      k_end   = min( kte, kde-1 )
     
      i_endu = min( ite,ide-1 )
      j_endv = min( jte,jde-1 )
      k_endw = k_end
    ELSE
      i_start = its
      i_end   = ite
      j_start = jts
      j_end   = jte
      k_start = kts
      k_end   = kte-1
     
      i_endu = i_end
      j_endv = j_end
      k_endw = k_end

      IF(j_end == jde) j_end = j_end - 1
      IF(i_end == ide) i_end = i_end - 1
    ENDIF

      i_start_up = i_start
      i_end_up   = i_endu
      j_start_up = j_start
      j_end_up   = j_end

      i_start_vp = i_start
      i_end_vp   = i_end
      j_start_vp = j_start
      j_end_vp   = j_endv

      IF ( (config_flags%open_xs   .or.     &
!           config_flags%specified .or.     &
            config_flags%symmetric_xs   )   &
            .and. (its == ids) )            &
                 i_start_up = i_start_up + 1

      IF ( (config_flags%open_xe    .or.  &
!           config_flags%specified  .or.  &
            config_flags%symmetric_xe   ) &
             .and. (ite == ide) )         &
                 i_end_up   = i_end_up - 1

      IF ( (config_flags%open_ys    .or.   &
!           config_flags%specified  .or.   &
            config_flags%symmetric_ys   )  &
                     .and. (jts == jds) )  &
                 j_start_vp = j_start_vp + 1

      IF ( (config_flags%open_ye     .or. &
!           config_flags%specified   .or. &
            config_flags%symmetric_ye   ) &
            .and. (jte == jde) )          &
                 j_end_vp   = j_end_vp - 1

      i_start_u_tend = i_start
      i_end_u_tend   = i_endu
      j_start_v_tend = j_start
      j_end_v_tend   = j_endv

      IF ( config_flags%symmetric_xs .and. (its == ids) ) &
                     i_start_u_tend = i_start_u_tend+1
      IF ( config_flags%symmetric_xe .and. (ite == ide) ) &
                     i_end_u_tend = i_end_u_tend-1
      IF ( config_flags%symmetric_ys .and. (jts == jds) ) &
                     j_start_v_tend = j_start_v_tend+1
      IF ( config_flags%symmetric_ye .and. (jte == jde) ) &
                     j_end_v_tend = j_end_v_tend-1

!  start real calculations.
!  first, u

      u_outer_j_loop: DO j = j_start, j_end

        kp0 = 1
        kp1 = 2

      u_outer_k_loop: DO k = k_start, k_end


        DO i = i_start, i_endu
          du(i,k,j) = ru(i,k,j)
        ENDDO

        DO i = i_start_u_tend, i_end_u_tend
          ru(i,k,j) = ru(i,k,j) + dts*ru_tend(i,k,j)
        ENDDO

      IF(k == k_start) THEN
        DO i = i_start_up, i_end_up

          dpzxyu(i,kp0) =  0.5*msft(i,j)*zx(i,1,j)*(  &
            zz(i  ,j)*(   cf1*rt(i  ,1,j)            &
                        + cf2*rt(i  ,2,j)            &
                        + cf3*rt(i  ,3,j) )          &
          + zz(i-1,j)*(   cf1*rt(i-1,1,j)            &
                        + cf2*rt(i-1,2,j)            &
                        + cf3*rt(i-1,3,j) )  )
        ENDDO
       END IF

       IF( k < k_end ) THEN

        DO i = i_start_up, i_end_up

            dpzxyu(i,kp1) =  0.5*msft(i,j)*zx(i,k+1,j)*(                 &
              fzm(k+1)*(zz(i,j)*rt(i,k+1,j) + zz(i-1,j)*rt(i-1,k+1,j))  &
            + fzp(k+1)*(zz(i,j)*rt(i,k  ,j) + zz(i-1,j)*rt(i-1,k  ,j)) )

          ENDDO

       ELSE

         DO i = i_start_up, i_end_up
           dpzxyu(i,kp1) = 0.
         ENDDO

       END IF

      DO i = i_start_up, i_end_up

            ru(i,k,j) = ru(i,k,j)                                   &
                   - 0.5*c2*dts*                                    &
                                ( pip(i  ,k,j)+pip(i-1,k,j)) *(     &
                  rdx*(msft(i,j)*rt(i,k,j)-msft(i-1,j)*rt(i-1,k,j)) &
                      +rdzw(k)*(dpzxyu(i,kp1)-dpzxyu(i,kp0)) )
      ENDDO

      DO i = i_start, i_endu
          du(i,k,j) = ru(i,k,j) - du(i,k,j)
      ENDDO

        ktmp = kp1
        kp1 = kp0
        kp0 = ktmp

   ENDDO u_outer_k_loop
   ENDDO u_outer_j_loop

! now v

      v_outer_j_loop: DO j = j_start, j_endv

        kp0 = 1
        kp1 = 2

      v_outer_k_loop: DO k = k_start, k_end


        DO i = i_start, i_end 
          dv(i,k,j) = rv(i,k,j)
        ENDDO

        IF ( (j >= j_start_v_tend) .and. (j <= j_end_v_tend)) then
          DO i = i_start, i_end 
            rv(i,k,j) = rv(i,k,j) + dts*rv_tend(i,k,j)
          ENDDO
        ENDIF

        IF (     ( j >= j_start_vp)  &
            .and.( j <= j_end_vp  ) )  THEN

          IF ( k == k_start ) THEN
        
          DO i = i_start, i_end
              dpzxyv(i,kp0) =  0.5*msft(i,j)*zy(i,1,j)*(  &
                zz(i,j  )*(   cf1*rt(i,1,j  )            &
                            + cf2*rt(i,2,j  )            &
                            + cf3*rt(i,3,j  ) )          &
              + zz(i,j-1)*(   cf1*rt(i,1,j-1)            &
                            + cf2*rt(i,2,j-1)            &
                            + cf3*rt(i,3,j-1) )  )
          ENDDO

        END IF

        IF ( k < k_end ) THEN

          DO i = i_start, i_end
            dpzxyv(i,kp1) =  0.5*msft(i,j)*zy(i,k+1,j)*(                 &
              fzm(k+1)*(zz(i,j)*rt(i,k+1,j) + zz(i,j-1)*rt(i,k+1,j-1))  &
            + fzp(k+1)*(zz(i,j)*rt(i,k  ,j) + zz(i,j-1)*rt(i,k  ,j-1)) )
          ENDDO

        ELSE

          DO i = i_start, i_end
              dpzxyv(i,kp1) = 0.
          ENDDO

        END IF


        DO i = i_start, i_end 
            rv(i,k,j) = rv(i,k,j)                                   &
                   - 0.5*c2*dts*                                    &
                                ( pip(i,k,j  )+pip(i,k,j-1))*(      &
                 rdy*(msft(i,j)*rt(i,k,j)-msft(i,j-1)*rt(i,k,j-1))  &
                      + rdzw(k)*(dpzxyv(i,kp1)-dpzxyv(i,kp0)  ) ) 
        ENDDO

      END IF

        DO i = i_start, i_end 
            dv(i,k,j) = rv(i,k,j) - dv(i,k,j)
        ENDDO

       ktmp = kp1
       kp1 = kp0
       kp0 = ktmp


    ENDDO  v_outer_k_loop
    ENDDO  v_outer_j_loop

END SUBROUTINE advance_uv

!---------------------------------------------------------------------

      SUBROUTINE advance_w( ru, du, rv, dv, rw, rw_tend,                 &
                            r, r_tend, rt, rt_tend, rtold, msft,         &
                            theta, a, alpha, gamma,                      &
                            cofwz, coftz, cofwt, rdzw, fzm, fzp,         &
                            cofwr , cofrz ,                              &
                            zx, zy, zz, rdx, rdy, dts, smdiv, resm,      &
                            iteration, iter_end, config_flags,           &
                            spec_zone,                                   &
                            ids,ide, jds,jde, kds,kde,  & ! domain dims
                            ims,ime, jms,jme, kms,kme,  & ! memory dims
                            its,ite, jts,jte, kts,kte   & ! tile   dims
                                                            )
      IMPLICIT NONE ! religion first
      
      real*8 time0,time1,time2,time3
      real*8 taccum(0:63)
      external rtc
      real*8 rtc

      common/timer/taccum

! stuff coming in

      INTEGER,      INTENT(IN   )    :: ids,ide, jds,jde, kds,kde
      INTEGER,      INTENT(IN   )    :: ims,ime, jms,jme, kms,kme
      INTEGER,      INTENT(IN   )    :: its,ite, jts,jte, kts,kte
      INTEGER,      INTENT(IN   )    :: spec_zone

      INTEGER,      INTENT(IN   )    :: iteration, iter_end
      TYPE(grid_config_rec_type),    INTENT(IN   )    :: config_flags

      REAL, DIMENSION( ims:ime , kms:kme , jms:jme ), &
            INTENT(INOUT) ::                          &
                                             r,       &
                                             rt,      &
                                             rw ,     &
                                             rtold


      REAL, DIMENSION(  ims:ime , kms:kme, jms:jme ), &
            INTENT(IN   ) :: ru,                      &
                                             du,      &
                                             rv,      &
                                             dv,      &
                                             r_tend,  &
                                             rt_tend, &
                                             rw_tend, &
                                             theta,   &
                                             zx ,     &
                                             zy ,     &
                                             alpha,   &
                                             gamma,   &
                                             a,       &
                                             cofwz,   &
                                             coftz,   &
                                             cofwt



      REAL, DIMENSION( ims:ime , jms:jme ), &
            INTENT(IN   )  ::               &
                                   zz,      &
                                   msft,    &
                                   cofwr

      REAL, DIMENSION( kms:kme ),  INTENT(IN   )  :: fzp,     &
                                                     fzm,     &
                                                     rdzw,    &
                                                     cofrz

      REAL,   INTENT(IN   )  :: rdx,     &
                                rdy,     &
                                dts,     &
                                smdiv,   &
                                resm

!  Stack based 3d data, tile size.

      REAL, DIMENSION( its:ite, kts:kte, jts:jte)                :: rs,      &
                                                                    ts 

      REAL, DIMENSION( its:ite ) :: coftztmp 
      INTEGER :: i,j,k, i_start, i_end, j_start, j_end, k_start, k_end, kk
      INTEGER :: ij, ijp, ijm


!  set loop limits.

    IF( config_flags%nested .or. config_flags%specified ) THEN
      i_start = max( its,ids+spec_zone )
      i_end   = min( ite,ide-spec_zone-1 )
      j_start = max( jts,jds+spec_zone )
      j_end   = min( jte,jde-spec_zone-1 )
      k_start = kts
      k_end   = min( kte, kde-1 )
    ELSE
      i_start = its
      i_end   = ite
      j_start = jts
      j_end   = jte
      k_start = kts
      k_end   = kte-1
      IF(j_end == jde) j_end = j_end - 1
      IF(i_end == ide) i_end = i_end - 1
    ENDIF

! start by removing div damping from rho*theta
    DO j = j_start, j_end
      DO k = k_start, k_end

        IF ( iteration /= 1 ) THEN
        DO i = i_start, i_end
          rt(i,k,j) = rt(i,k,j) &
             - smdiv*(rt(i,k,j) - rtold(i,k,j)                      )
        ENDDO
        END IF

        DO i = i_start, i_end

          rtold(i,k,j) = rt(i,k,j)

!
!  explicit portion of rho timestep
!
            rs(i,k,j) = r(i,k,j) + dts*r_tend(i,k,j)         &
                   - dts*msft(i,j)*(                         &
                         rdx*(ru(i+1,k,j)-ru(i,k,j))         &
                       + rdy*(rv(i,k,j+1)-rv(i,k,j)) )       &
                  - cofrz(k)*resm*(rw(i,k+1,j)-rw(i,k,j))
!
!  explicit portion of rho_theta timestep

            ts(i,k,j) = rt(i,k,j) + dts*rt_tend(i,k,j) - .5*dts*msft(i,j)*(  &
             rdx*(  ru(i+1,k,j)*(theta(i+1,k,j)+theta(i,k,j))    &
                  - ru(i,k,j)*(theta(i,k,j)+theta(i-1,k,j)) )  &
            +rdy*(  rv(i,k,j+1)*(theta(i,k,j+1)+theta(i,k,j))    &
                  - rv(i,k,j)*(theta(i,k,j)+theta(i,k,j-1)) )) &
                       - resm*rdzw(k)*(coftz(i,k+1,j)*rw(i,k+1,j)  &
                                     - coftz(i,k,j)*rw(i,k,j))  
        ENDDO
      ENDDO

      DO k = k_start+1,k_end
        DO i = i_start, i_end

            rw(i,k,j) =                                                           &
                   rw(i,k,j)                                                      &
                 + dts*rw_tend(i,k,j)                                             &
                 - cofwz(i,k,j)*                                                  &
                   ( (ts(i,k,j)-ts(i,k-1,j)) + resm*(rt(i,k,j)-rt(i,k-1,j)) )     &
                 - cofwr(i,j)*                                                    &
                   ( (rs(i,k,j)+rs(i,k-1,j)) + resm*(r(i,k,j)+ r(i,k-1,j)))       &
                 + cofwt(i,k,j)*(ts(i,k,j)+resm*rt(i,k,j))                        &
                 + cofwt(i,k-1,j)*(ts(i,k-1,j)+resm*rt(i,k-1,j))                  &
                 -.25*(                                                           &
                   zx(i+1,k,j)*(zz(i,j)                                           &
                  +zz(i+1,j))*(fzm(k)*du(i+1,k,j)+fzp(k)*du(i+1,k-1,j))           &
                  +zx(i,k,j)*(zz(i,j)                                             &
                  +zz(i-1,j))*(fzm(k)*du(i,k,j)+fzp(k)*du(i,k-1,j))               &
                      )                                                           &
                 -.25*(                                                           &
                   zy(i,k,j+1)*(zz(i,j)                                           &
                  +zz(i,j+1))*(fzm(k)*dv(i,k,j+1)+fzp(k)*dv(i,k-1,j+1))           &
                  +zy(i,k,j)*(zz(i,j)                                             &
                  +zz(i,j-1))*(fzm(k)*dv(i,k,j)+fzp(k)*dv(i,k-1,j))               &
                      )
            rw(i,k,j) = (rw(i,k,j)-a(i,k,j)*rw(i,k-1,j))*alpha(i,k,j)

        ENDDO
      ENDDO
!
!  implicit solution for rw, rr and rt
!

       DO k = k_end,k_start,-1
       DO i = i_start, i_end
            rw(i,k,j) = rw(i,k,j) - gamma(i,k,j)*rw(i,k+1,j)
            r(i,k,j)  = rs(i,k,j)  - cofrz(k) *(rw(i,k+1,j)-rw(i,k,j))
            rt(i,k,j) = ts(i,k,j) - rdzw(k)*( coftz(i,k+1,j)*rw(i,k+1,j)  &
                                             -coftz(i,k,j)*rw(i,k,j))
       ENDDO
       ENDDO

! add div damping back into rho*theta 
! if another small timestep is next

       IF(iteration /= iter_end)  THEN
         DO k = k_end,k_start,-1
         DO i = i_start, i_end
           rt(i,k,j) = rt(i,k,j) &
             + smdiv*(rt(i,k,j) - rtold(i,k,j))
         ENDDO
         ENDDO
       ENDIF

      ENDDO

      END SUBROUTINE advance_w



!----------------------------------------------------------------------


      SUBROUTINE calc_coef_w( alpha, gamma, a, cofwz, coftz, cofwt,     &
                              dtseps, dts,zz, rdzu, fzm, fzp,           &
                              cofwr, cofrz, rdzw,                       &
                              pi, theta, pibar, rb, rho_theta,          &
                              config_flags, spec_zone,                  &
                              ids,ide, jds,jde, kds,kde,  & ! domain dims
                              ims,ime, jms,jme, kms,kme,  & ! memory dims
                              its,ite, jts,jte, kts,kte   & ! tile   dims
                                                   )

      IMPLICIT NONE  ! religion first

!  passed in through the call

      INTEGER,      INTENT(IN   )    :: ids,ide, jds,jde, kds,kde
      INTEGER,      INTENT(IN   )    :: ims,ime, jms,jme, kms,kme
      INTEGER,      INTENT(IN   )    :: its,ite, jts,jte, kts,kte
      INTEGER,      INTENT(IN   )    :: spec_zone

      TYPE(grid_config_rec_type),    INTENT(IN   )    :: config_flags

      REAL, DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(IN   ) :: pi,     &
                                                                   theta,  &
                                                                   pibar,  &
                                                                   rb,     &
                                                                   rho_theta

      REAL, DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(INOUT) :: alpha,  &
                                                                   gamma,  &
                                                                   a,      &
                                                                   cofwz,  &
                                                                   coftz,  &
                                                                   cofwt

      REAL, DIMENSION(ims:ime, jms:jme),          INTENT(IN   ) :: zz,     &
                                                                   cofwr

      REAL, DIMENSION(kms:kme),                   INTENT(IN   ) :: rdzu,   &
                                                                   rdzw,   &
                                                                   fzm,    &
                                                                   fzp,    &
                                                                   cofrz

      REAL,                                       INTENT(IN   ) :: dtseps, &
                                                                   dts

!  Local stack data.

      REAL, DIMENSION(ims:ime, jms:jme)                         :: b , c

      INTEGER :: i, j, k, i_start, i_end, j_start, j_end, k_start, k_end
      INTEGER :: ij, ijp, ijm
                                                 

!  coefficients for implicit w, rho and rho_theta solution

    IF( config_flags%nested .or. config_flags%specified ) THEN
      i_start = max( its,ids+spec_zone )
      i_end   = min( ite,ide-spec_zone-1 )
      j_start = max( jts,jds+spec_zone )
      j_end   = min( jte,jde-spec_zone-1 )
      k_start = kts
      k_end   = min( kte, kde-1 )

    ELSE
      i_start = its
      i_end   = ite
      j_start = jts
      j_end   = jte
      k_start = kts
      k_end   = kte-1

      IF(j_end == jde) j_end = j_end - 1
      IF(i_end == ide) i_end = i_end - 1
    ENDIF

      outer_j_loop:  DO j = j_start, j_end

      DO i = i_start, i_end

        gamma(i,1,j)       = 0.
        coftz(i,1,j)       = 0.
        coftz(i,k_end+1,j) = 0.
        cofwz(i,1,j)       = 0.
        cofwz(i,k_end+1,j) = 0.
        cofwt(i,k_end+1,j) = 0.
        a(i,1,j) = 0.
        b(i,j) = 1.
        c(i,j) = 0.

      ENDDO


      DO k = 2, k_end
      DO i = i_start, i_end
          cofwz(i,k,j) = dtseps*c2*zz(i,j)**2*rdzu(k)                        &
                  *(fzm(k)*pi(i,k  ,j)  &
                   +fzp(k)*pi(i,k-1,j) ) 

          coftz(i,k,j) = dtseps*(fzm(k)*theta(i,k,j)+fzp(k)*theta(i,k-1,j))
      ENDDO
      ENDDO
        

      DO k = 1, k_end
      DO i = i_start, i_end
           cofwt(i,k,j) = .5*dtseps*rcv*zz(i,j)*g*rb(i,k,j)  &
                         *pi(i,k,j)/          &
                          ((rho_theta(i,k,j))*pibar(i,k,j))
      ENDDO
      ENDDO
            
      DO k = 2,k_end
      DO i = i_start, i_end

                a(i,k,j) =    - cofwz(i,k  ,j)* coftz(i,k-1,j)*rdzw (k-1) &
                              + cofwr(i,j    )* cofrz(k-1  )              &
                              - cofwt(i,k-1,j)* coftz(i,k-1,j)*rdzw (k-1)
                b(i,j) = 1. + cofwz(i,k  ,j)*(coftz(i,k  ,j)*rdzw (k  )     &
                                             +coftz(i,k  ,j)*rdzw (k-1))    &
                            - coftz(i,k,j)*(cofwt(i,k  ,j)*rdzw (k  )   &
                                           -cofwt(i,k-1,j)*rdzw (k-1))    &
                            + cofwr(    i,j)*(cofrz(k    )-cofrz(k-1))
                c(i,j) =    - cofwz(i,k  ,j)* coftz(i,k+1,j)*rdzw (k  )     &
                            - cofwr(    i,j)* cofrz(k    )                &
                            + cofwt(i,k,j)* coftz(i,k+1,j)*rdzw (k  )

               alpha(i,k,j) = 1./(b(i,j)-a(i,k,j)*gamma(i,k-1,j))
               gamma(i,k,j) = c(i,j)*alpha(i,k,j)

      ENDDO
      ENDDO

      ENDDO outer_j_loop


      END SUBROUTINE calc_coef_w


!-----------------------------------------------------------------------


      SUBROUTINE small_step_decouple( ru_1,rv_1,rw_1,r_1,rt_1,    &
                                      ru_2,rv_2,rw_2,r_2,rt_2,    &
                                      rw_tend, rt_tend,           &
                                      rtold,                      &
                                      zz, smdiv, dts,             &
                                      ids,ide, jds,jde, kds,kde,  & 
                                      ims,ime, jms,jme, kms,kme,  &
                                      its,ite, jts,jte, kts,kte   &
                                                                   )

      IMPLICIT NONE  ! religion first

! declarations for the stuff coming in

      INTEGER,      INTENT(IN   )    :: ids,ide, jds,jde, kds,kde
      INTEGER,      INTENT(IN   )    :: ims,ime, jms,jme, kms,kme
      INTEGER,      INTENT(IN   )    :: its,ite, jts,jte, kts,kte

      REAL, DIMENSION(ims:ime, kms:kme, jms:jme),INTENT(INOUT) :: ru_1,   &
                                                                  rv_1,   &
                                                                  rw_1,   &
                                                                  r_1,    &
                                                                  rt_1

      REAL, DIMENSION(ims:ime, kms:kme, jms:jme),INTENT(INOUT) :: ru_2,   &
                                                                  rv_2,   &
                                                                  rw_2,   &
                                                                   r_2,   &
                                                                  rt_2

      REAL, DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(INOUT) :: rtold

      REAL, DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(INOUT) :: rw_tend, &
                                                                   rt_tend

      REAL, DIMENSION(ims:ime, jms:jme)         , INTENT(IN   ) :: zz

      REAL,                                       INTENT(IN   ) :: smdiv,   &
                                                                   dts

!  Local data.

      INTEGER   :: i,j,k
      INTEGER   :: i_start, i_end, j_start, j_end
      INTEGER   :: i_endu, j_endv, k_endw

!  subtract out the middle time level for small steps in u and v
!  w, rho_theta and rho, and reset w tendency to be omega tendency.

      i_start = its
      i_end   = ite
      j_start = jts
      j_end   = jte

      i_endu = i_end
      j_endv = j_end

      IF(i_end == ide) i_end = i_end - 1
      IF(j_end == jde) j_end = j_end - 1

      DO j = j_start, j_end
      DO k = kms, kme-1
      DO i = i_start, i_endu
        ru_1(i,k,j) = ru_1(i,k,j) - ru_2(i,k,j)
      ENDDO
      ENDDO
      ENDDO

      DO j = j_start, j_endv
      DO k = kms, kme-1
      DO i = i_start, i_end
        rv_1(i,k,j) = rv_1(i,k,j) - rv_2(i,k,j)
      ENDDO
      ENDDO
      ENDDO

      DO j = j_start, j_end
      DO k = kms, kme
      DO i = i_start, i_end
        rw_1(i,k,j) = rw_1(i,k,j) - rw_2(i,k,j)
        rw_tend(i,k,j) = zz(i,j)*rw_tend(i,k,j)
      ENDDO
      ENDDO
      ENDDO

!  here we also put the divergence damping increment into p 
!  for use in the horizontal momentum update

      DO j = j_start, j_end
      DO k = kms, kme-1
      DO i = i_start, i_end
         r_1(i,k,j)    =  r_1(i,k,j) -  r_2(i,k,j)
        rt_1(i,k,j)    = rt_1(i,k,j) - rt_2(i,k,j) 
        rtold(i,k,j)   = rt_1(i,k,j) - dts*rt_tend(i,k,j) 
        rt_1(i,k,j)    = rt_1(i,k,j) + smdiv*(        &
                rt_1(i,k,j) - rtold(i,k,j) - dts*rt_tend(i,k,j)  )
      ENDDO
      ENDDO
      ENDDO

      END SUBROUTINE small_step_decouple

!-------------------------------------------------------------------------


  SUBROUTINE small_step_recouple( dt,                         &
                                  ru_1,rv_1,rw_1,r_1,rt_1,    &
                                  ru_2,rv_2,rw_2,r_2,rt_2,    &
                                  ru_m,rv_m,rw_m,h_diabatic,  &
                                  ids,ide, jds,jde, kds,kde,  &
                                  ims,ime, jms,jme, kms,kme,  &
                                  its,ite, jts,jte, kts,kte   &
                                                                   )

      IMPLICIT NONE  ! religion first

!  stuff passed in

      INTEGER,                  INTENT(IN   ) :: ids,ide, jds,jde, kds,kde
      INTEGER,                  INTENT(IN   ) :: ims,ime, jms,jme, kms,kme
      INTEGER,                  INTENT(IN   ) :: its,ite, jts,jte, kts,kte

      REAL,                     INTENT(IN   ) :: dt

      REAL,   DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(INOUT) :: ru_1, &
                                                                     rv_1, &
                                                                     rw_1, &
                                                                      r_1, &
                                                                     rt_1

      REAL,   DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(INOUT) :: ru_2, &
                                                                     rv_2, &
                                                                     rw_2, &
                                                                      r_2, &
                                                                     rt_2

      REAL,   DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(INOUT) :: ru_m, &
                                                                     rv_m, &
                                                                     rw_m

      REAL,   DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(OUT) :: h_diabatic

! local stuff

      INTEGER         :: i,j,k
      INTEGER :: i_start, i_end, j_start, j_end, i_endu, j_endv

      i_start = its
      i_end   = ite
      j_start = jts
      j_end   = jte

      i_endu = i_end
      j_endv = j_end

      IF(i_end == ide) i_end = i_end - 1
      IF(j_end == jde) j_end = j_end - 1

      DO j = j_start, j_end
      DO k = kms, kme-1
      DO i = i_start, i_endu
        ru_1(i,k,j) = ru_1(i,k,j) + ru_2(i,k,j)
        ru_m(i,k,j) = ru_m(i,k,j) + ru_2(i,k,j)
      ENDDO
      ENDDO
      ENDDO

      DO j = j_start, j_endv
      DO k = kms, kme-1
      DO i = i_start, i_end
        rv_1(i,k,j) = rv_1(i,k,j) + rv_2(i,k,j)
        rv_m(i,k,j) = rv_m(i,k,j) + rv_2(i,k,j)
      ENDDO
      ENDDO
      ENDDO

      DO j = j_start, j_end
      DO k = kms, kme
      DO i = i_start, i_end
        rw_1(i,k,j) = rw_1(i,k,j) + rw_2(i,k,j)
        rw_m(i,k,j) = rw_m(i,k,j) + rw_2(i,k,j)
      ENDDO
      ENDDO
      ENDDO

      DO j = j_start, j_end
      DO k = kms, kme-1
      DO i = i_start, i_end
         r_1(i,k,j)        =  r_1(i,k,j) +  r_2(i,k,j)
        rt_1(i,k,j)        = rt_1(i,k,j) + rt_2(i,k,j) &
                             - 2*dt*h_diabatic(i,k,j)
        h_diabatic(i,k,j)  = rt_1(i,k,j)
      ENDDO
      ENDDO
      ENDDO

      END SUBROUTINE small_step_recouple

!-------------------------------------------------------------------------


      SUBROUTINE time_filter( f_old, f_middle, f_new, fld, &
                              ids,ide, jds,jde, kds,kde,   &
                              ims,ime, jms,jme, kms,kme,   &
                              its,ite, jts,jte, kts,kte    &
                                                            )

!  this routine does the asselin time filtering for the leapfrog
!  time integration scheme and switches the new and middle time
!  level fields

      IMPLICIT NONE  ! religion first

! declarations for the stuff coming in

      INTEGER,            INTENT(IN   ) :: ids,ide, jds,jde, kds,kde
      INTEGER,            INTENT(IN   ) :: ims,ime, jms,jme, kms,kme
      INTEGER,            INTENT(IN   ) :: its,ite, jts,jte, kts,kte

      REAL, DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(IN   ) :: f_old

      REAL, DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(INOUT) :: f_middle, &
                                                                   f_new

      CHARACTER (LEN=1),                        INTENT(IN   ) :: fld

! local stuff

      INTEGER :: i,j,k,i_start,j_start,k_start,i_end,j_end,k_end
      REAL :: f_temp

!  set start and end, and check for physical boundary conditions

      i_start = its
      i_end   = ite
      j_start = jts
      j_end   = jte
      k_start = kts
      k_end   = kte

      
      IF(( i_end == ide ) .and. (fld /= 'u')) i_end = i_end - 1
      IF(( j_end == jde ) .and. (fld /= 'v')) j_end = j_end - 1

      DO j = j_start, j_end
      DO k = k_start, k_end
      DO i = i_start, i_end

        f_temp       = f_middle(i,k,j) + asselin *   &
               (f_old(i,k,j) - 2*f_middle(i,k,j)+f_new(i,k,j))
        f_middle(i,k,j) = f_new(i,k,j)
        f_new(i,k,j)    = f_temp

      ENDDO
      ENDDO
      ENDDO

      END SUBROUTINE time_filter


!---------------------------------------------------------------------


      SUBROUTINE time_filter_scalars( num_3d,                      &
                                      f_old, f_middle, f_new,      &
                                      ids,ide, jds,jde, kds,kde,   &
                                      ims,ime, jms,jme, kms,kme,   &
                                      its,ite, jts,jte, kts,kte    &
                                                            )

!  this routine does the asselin time filtering for the leapfrog
!  time integration scheme and switches the new and middle time
!  level fields.  this is the routine used for an arbitrary number
!  of scalars (all assumed to be defined at a pressure/density point).

      IMPLICIT NONE  ! religion first

! declarations for the stuff coming in

      INTEGER,            INTENT(IN   ) :: ids,ide, jds,jde, kds,kde
      INTEGER,            INTENT(IN   ) :: ims,ime, jms,jme, kms,kme
      INTEGER,            INTENT(IN   ) :: its,ite, jts,jte, kts,kte
      INTEGER,            INTENT(IN   ) :: num_3d

      REAL, DIMENSION( ims:ime, kms:kme, jms:jme, num_3d),           &
            INTENT(IN   )                                  :: f_old

      REAL, DIMENSION( ims:ime, kms:kme, jms:jme, num_3d),              &
            INTENT(INOUT)                                  :: f_middle, &
                                                              f_new

! local stuff

      INTEGER :: i,j,k,i_start,j_start,k_start,i_end,j_end,k_end,is
      REAL    :: f_temp

!  set start and end, and check for physical boundary conditions

      i_start = its
      i_end   = ite
      j_start = jts
      j_end   = jte
      k_start = kts
      k_end   = kte

      
      IF(( i_end == ide )) i_end = i_end - 1
      IF(( j_end == jde )) j_end = j_end - 1

      DO is = 1, num_3d
      DO j = j_start, j_end
      DO k = k_start, k_end
      DO i = i_start, i_end

        f_temp       = f_middle(i,k,j,is) + asselin *   &
               (f_old(i,k,j,is) - 2*f_middle(i,k,j,is)+f_new(i,k,j,is))
        f_middle(i,k,j,is) = f_new(i,k,j,is)
        f_new(i,k,j,is)    = f_temp

      ENDDO
      ENDDO
      ENDDO
      ENDDO

      END SUBROUTINE time_filter_scalars


!---------------------------------------------------------------------

      SUBROUTINE save_old_time(    ru,rv,rw,r,rt,              &
                                   u_old,v_old,w_old,          &
                                   r_old,rt_old,               &
                                   ids,ide, jds,jde, kds,kde,  &
                                   ims,ime, jms,jme, kms,kme,  &
                                   its,ite, jts,jte, kts,kte   &
                                                                )

      IMPLICIT NONE  ! religion first

! declarations for the stuff coming in

      INTEGER,      INTENT(IN   )    :: ids,ide, jds,jde, kds,kde
      INTEGER,      INTENT(IN   )    :: ims,ime, jms,jme, kms,kme
      INTEGER,      INTENT(IN   )    :: its,ite, jts,jte, kts,kte

      REAL, DIMENSION(ims:ime, kms:kme, jms:jme),    INTENT(INOUT) :: ru, &
                                                                      rv, &
                                                                      rw, &
                                                                      r,  &
                                                                      rt

      REAL, DIMENSION(ims:ime, kms:kme, jms:jme) , INTENT(  OUT) :: u_old, &
                                                                    v_old, &
                                                                    w_old, &
                                                                    r_old, &
                                                                    rt_old

      INTEGER   :: i,j,k

!  store off the old time level for the Asselin time filter
!  in the leapfrog scheme

      DO  j = jts, jte
      DO  k = kts, kte
      DO  i = its, ite
        u_old(i,k,j)  = ru(i,k,j)
        v_old(i,k,j)  = rv(i,k,j)
        w_old(i,k,j)  = rw(i,k,j)
        r_old(i,k,j)  = r(i,k,j)
        rt_old(i,k,j) = rt(i,k,j)
      ENDDO
      ENDDO
      ENDDO

      RETURN
      END SUBROUTINE save_old_time

!---------------------------------------------------------------------

      SUBROUTINE sumflux ( flag,                                   &
                           ru, rv, rom,                            &
                           ru_m, rv_m, rom_m, epssm,               &
                           iteration , number_of_small_timesteps,  &
                           ids,ide, jds,jde, kds,kde,              &
                           ims,ime, jms,jme, kms,kme,              &
                           its,ite, jts,jte, kts,kte               &
                                                                 )

      IMPLICIT NONE  ! religion first

! declarations for the stuff coming in

      INTEGER,      INTENT(IN   )    :: flag, iteration
      INTEGER,      INTENT(IN   )    :: number_of_small_timesteps
      INTEGER,      INTENT(IN   )    :: ids,ide, jds,jde, kds,kde
      INTEGER,      INTENT(IN   )    :: ims,ime, jms,jme, kms,kme
      INTEGER,      INTENT(IN   )    :: its,ite, jts,jte, kts,kte

      REAL, DIMENSION(ims:ime, kms:kme, jms:jme),  INTENT(IN   ) :: ru, &
                                                                    rv, &
                                                                    rom

      REAL, DIMENSION(ims:ime, kms:kme, jms:jme) , INTENT(INOUT) :: ru_m, &
                                                                    rv_m, &
                                                                    rom_m
      REAL, INTENT(IN   )  ::  epssm
      INTEGER   :: i,j,k

      IF (flag == 1)THEN

        IF (iteration == 1 )THEN
          DO  j = jts, jte
          DO  k = kts, kte
          DO  i = its, ite
            ru_m(i,k,j)  = 0.
            rv_m(i,k,j)  = 0.
            rom_m(i,k,j)  = 0.
          ENDDO
          ENDDO
          ENDDO
        ENDIF

        DO  j = jts, min(jde-1,jte)
        DO  k = kts, min(kde-1,kte)
        DO  i = its, ite
          ru_m(i,k,j)  = ru_m(i,k,j) + ru(i,k,j)
        ENDDO
        ENDDO
        ENDDO

        DO  j = jts, jte
        DO  k = kts, min(kde-1,kte)
        DO  i = its, min(ide-1,ite)
          rv_m(i,k,j)  = rv_m(i,k,j) + rv(i,k,j)
        ENDDO
        ENDDO
        ENDDO

        DO  j = jts, min(jde-1,jte)
        DO  k = kts, kte
        DO  i = its, min(ide-1,ite)
          rom_m(i,k,j)  = rom_m(i,k,j) + 0.5*(1.-epssm)*rom(i,k,j)
        ENDDO
        ENDDO
        ENDDO

      ENDIF

      IF (flag == 2)THEN
        DO  j = jts, min(jte, jde-1)
        DO  k = kts, kte
        DO  i = its, min(ite, ide-1)
          rom_m(i,k,j)  = rom_m(i,k,j) + 0.5*(1.+epssm)*rom(i,k,j)
        ENDDO
        ENDDO
        ENDDO
      ENDIF

      IF ((iteration == number_of_small_timesteps) .and. (flag == 2)) THEN

        DO  j = jts, min(jde-1,jte)
        DO  k = kts, min(kde-1,kte)
        DO  i = its, ite
          ru_m(i,k,j)  = ru_m(i,k,j) / number_of_small_timesteps
        ENDDO
        ENDDO
        ENDDO

        DO  j = jts, jte
        DO  k = kts, min(kde-1,kte)
        DO  i = its, min(ide-1,ite)
          rv_m(i,k,j)  = rv_m(i,k,j) / number_of_small_timesteps
        ENDDO
        ENDDO
        ENDDO

        DO  j = jts, min(jde-1,jte)
        DO  k = kts, kte
        DO  i = its, min(ide-1,ite)
          rom_m(i,k,j)  = rom_m(i,k,j) / number_of_small_timesteps
        ENDDO
        ENDDO
        ENDDO
      ENDIF

      END SUBROUTINE sumflux 

!---------------------------------------------------------------------

      SUBROUTINE time_filter_dry( ru_old,  ru_2,  ru_1,         &
                                  rv_old,  rv_2,  rv_1,         &
                                  rom_old, rom_2, rom_1,        &
                                  rr_old,  rrp_2, rrp_1,        &
                                  rtp_old, rtp_2, rtp_1,        &
                                  ids, ide, jds, jde, kds, kde, &
                                  ims, ime, jms, jme, kms, kme, &
                                  its, ite, jts, jte, kts, kte )
!
!  a wrapper to call the time fliter for each variable in the dry model
!

      IMPLICIT NONE  ! religion first

! declarations for the stuff coming in

      INTEGER,            INTENT(IN   ) :: ids,ide, jds,jde, kds,kde
      INTEGER,            INTENT(IN   ) :: ims,ime, jms,jme, kms,kme
      INTEGER,            INTENT(IN   ) :: its,ite, jts,jte, kts,kte

      REAL, DIMENSION( ims:ime , kms:kme, jms:jme ), INTENT(IN   ) :: &
       ru_old, rv_old, rom_old, rr_old, rtp_old

      REAL, DIMENSION( ims:ime , kms:kme, jms:jme ), INTENT(INOUT) :: &
       ru_2, ru_1, rv_2, rv_1, rom_2, rom_1, rrp_2, rrp_1, rtp_2, rtp_1

      CALL time_filter(    ru_old, ru_2, ru_1, 'u',       &
                           ids, ide, jds, jde, kds, kde,  &
                           ims, ime, jms, jme, kms, kme,  &
                           its, ite, jts, jte, kts, kte  )

      CALL time_filter(    rv_old, rv_2, rv_1, 'v',       &
                           ids, ide, jds, jde, kds, kde,  &
                           ims, ime, jms, jme, kms, kme,  &
                           its, ite, jts, jte, kts, kte  )

      CALL time_filter(    rom_old, rom_2, rom_1, 'w',    &
                           ids, ide, jds, jde, kds, kde,  &
                           ims, ime, jms, jme, kms, kme,  &
                           its, ite, jts, jte, kts, kte  )

      CALL time_filter(    rr_old, rrp_2, rrp_1, 'r',     &
                           ids, ide, jds, jde, kds, kde,  &
                           ims, ime, jms, jme, kms, kme,  &
                           its, ite, jts, jte, kts, kte  )

      CALL time_filter(    rtp_old, rtp_2, rtp_1, 't',    &
                           ids, ide, jds, jde, kds, kde,  &
                           ims, ime, jms, jme, kms, kme,  &
                           its, ite, jts, jte, kts, kte  )

  END SUBROUTINE time_filter_dry



!---------------------------------------------------------------------

      SUBROUTINE init_module_small_step
      END SUBROUTINE init_module_small_step

!**********************************************************************
!
!  routines for the RK solver
!
!**********************************************************************

      SUBROUTINE rk_small_step_decouple( rk_step,             &
                                          ru_1,rv_1,rw_1,r_1,rt_1,    &
                                          ru_2,rv_2,rw_2,r_2,rt_2,    &
                                          ru_mid, rv_mid, rw_mid,     &
                                          r_mid, rt_mid,              &
                                          rw_tend, rt_tend,           &
                                          rtold,                      &
                                          zz, smdiv, dts,             &
                                          ids,ide, jds,jde, kds,kde,  & 
                                          ims,ime, jms,jme, kms,kme,  &
                                          its,ite, jts,jte, kts,kte   &
                                                                   )

      IMPLICIT NONE  ! religion first

! declarations for the stuff coming in

      INTEGER,      INTENT(IN   )    :: ids,ide, jds,jde, kds,kde
      INTEGER,      INTENT(IN   )    :: ims,ime, jms,jme, kms,kme
      INTEGER,      INTENT(IN   )    :: its,ite, jts,jte, kts,kte

      INTEGER,      INTENT(IN   )    :: rk_step

      REAL, DIMENSION(ims:ime, kms:kme, jms:jme),INTENT(INOUT) :: ru_1,   &
                                                                  rv_1,   &
                                                                  rw_1,   &
                                                                  r_1,    &
                                                                  rt_1

      REAL, DIMENSION(ims:ime, kms:kme, jms:jme),INTENT(INOUT) :: ru_2,   &
                                                                  rv_2,   &
                                                                  rw_2,   &
                                                                   r_2,   &
                                                                  rt_2

      REAL, DIMENSION(ims:ime, kms:kme, jms:jme),INTENT(  OUT) :: ru_mid, &
                                                                  rv_mid, &
                                                                  rw_mid, &
                                                                   r_mid, &
                                                                  rt_mid

      REAL, DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(INOUT) :: rtold

      REAL, DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(INOUT) :: rw_tend, &
                                                                   rt_tend

      REAL, DIMENSION(ims:ime, jms:jme)         , INTENT(IN   ) :: zz

      REAL,                                       INTENT(IN   ) :: smdiv,   &
                                                                   dts

!  Local data.

      INTEGER   :: i,j,k
      INTEGER   :: i_start, i_end, j_start, j_end
      INTEGER   :: i_endu, j_endv, k_endw

!  transform variables to perturbation form for small steps in u and v
!  w, rho_theta and rho, and reset w tendency to be omega tendency.

!  In this rk version, the time level t data is in *_2 in the first pass,
!  and in *_1 in subsequent passes.  
!  We are always advancing the *_2 data, sooooo...
!  in the first rk_step (rk_step == 1) we set the *_1 = *_2 (i.e., we are
!  replacing the t-dt data with the time t data) and zero out the *_2 fields
!  (because we are subtracting out the starting point time t data in our 
!  perturbation form of the small timestep in the first step).  
!  In subsequent passes, we want to subtract the
!  the midpoint  predictor from the starting time value (t), and we need to hang
!  on to the predictor, so we stick the predictor (*_2) into the *_mid variables,
!  and subtract it off from the *_1 (time level t) variables to initialize
!  the *_2 variable for advancing over the small timesteps.

      i_start = its
      i_end   = ite
      j_start = jts
      j_end   = jte

      i_endu = i_end
      j_endv = j_end

      IF(i_end == ide) i_end = i_end - 1
      IF(j_end == jde) j_end = j_end - 1

   IF( rk_step == 1 ) THEN

      DO j = j_start, j_end
      DO k = kms, kme-1
      DO i = i_start, i_endu
        ru_1(i,k,j) = ru_2(i,k,j)
        ru_2(i,k,j) = 0.
      ENDDO
      ENDDO
      ENDDO

      DO j = j_start, j_endv
      DO k = kms, kme-1
      DO i = i_start, i_end
        rv_1(i,k,j) = rv_2(i,k,j)
        rv_2(i,k,j) = 0.
      ENDDO
      ENDDO
      ENDDO

      DO j = j_start, j_end
      DO k = kms, kme
      DO i = i_start, i_end

        rw_1(i,k,j) = rw_2(i,k,j)
        rw_2(i,k,j) = 0.
        rw_tend(i,k,j) = zz(i,j)*rw_tend(i,k,j)
      ENDDO
      ENDDO
      ENDDO

!  here we also put the divergence damping increment into p 
!  for use in the horizontal momentum update

      DO j = j_start, j_end
      DO k = kms, kme-1
      DO i = i_start, i_end
         r_1(i,k,j)    =  r_2(i,k,j)
        rt_1(i,k,j)    = rt_2(i,k,j) 
         r_2(i,k,j)    = 0.
        rt_2(i,k,j)    = 0.
        rtold(i,k,j)   = rt_2(i,k,j) - dts*rt_tend(i,k,j) 
        rt_2(i,k,j)    = rt_2(i,k,j) + smdiv*(        &
                rt_2(i,k,j) - rtold(i,k,j) - dts*rt_tend(i,k,j)  )
      ENDDO
      ENDDO
      ENDDO

    ELSE

      DO j = j_start, j_end
      DO k = kms, kme-1
      DO i = i_start, i_endu
        ru_mid(i,k,j) = ru_2(i,k,j)
        ru_2(i,k,j) = ru_1(i,k,j) - ru_mid(i,k,j)
      ENDDO
      ENDDO
      ENDDO

      DO j = j_start, j_endv
      DO k = kms, kme-1
      DO i = i_start, i_end
        rv_mid(i,k,j) = rv_2(i,k,j)
        rv_2(i,k,j) = rv_1(i,k,j) - rv_mid(i,k,j)
      ENDDO
      ENDDO
      ENDDO

      DO j = j_start, j_end
      DO k = kms, kme
      DO i = i_start, i_end
        rw_mid(i,k,j) = rw_2(i,k,j)
        rw_2(i,k,j) = rw_1(i,k,j) - rw_mid(i,k,j)
        rw_tend(i,k,j) = zz(i,j)*rw_tend(i,k,j)
      ENDDO
      ENDDO
      ENDDO

!  here we also put the divergence damping increment into p 
!  for use in the horizontal momentum update

      DO j = j_start, j_end
      DO k = kms, kme-1
      DO i = i_start, i_end
        r_mid(i,k,j)  = r_2(i,k,j)
         r_2(i,k,j)    = r_1(i,k,j) - r_mid(i,k,j)
        rt_mid(i,k,j) = rt_2(i,k,j)
        rt_2(i,k,j)    = rt_1(i,k,j) - rt_mid(i,k,j)
        rtold(i,k,j)   = rt_2(i,k,j) - dts*rt_tend(i,k,j) 
        rt_2(i,k,j)    = rt_2(i,k,j) + smdiv*(        &
                rt_2(i,k,j) - rtold(i,k,j) - dts*rt_tend(i,k,j)  )
      ENDDO
      ENDDO
      ENDDO

    END IF

    END SUBROUTINE rk_small_step_decouple

!-------------------------------------------------------------------------

  SUBROUTINE rk_small_step_recouple( dt, rk_step,               &
                                      rk_order,                   &
                                      ru_1,rv_1,rw_1,r_1,rt_1,    &
                                      ru_2,rv_2,rw_2,r_2,rt_2,    &
                                      ru_mid, rv_mid, rw_mid,     &
                                      r_mid, rt_mid,              &
                                      ru_m,rv_m,rw_m,h_diabatic,  &
                                      ids,ide, jds,jde, kds,kde,  &
                                      ims,ime, jms,jme, kms,kme,  &
                                      its,ite, jts,jte, kts,kte   &
                                                                 )

      IMPLICIT NONE  ! religion first

!  stuff passed in

      INTEGER,                  INTENT(IN   ) :: ids,ide, jds,jde, kds,kde
      INTEGER,                  INTENT(IN   ) :: ims,ime, jms,jme, kms,kme
      INTEGER,                  INTENT(IN   ) :: its,ite, jts,jte, kts,kte

      INTEGER,                  INTENT(IN   ) :: rk_step, rk_order

      REAL,                     INTENT(IN   ) :: dt

      REAL,   DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(INOUT) :: ru_1, &
                                                                     rv_1, &
                                                                     rw_1, &
                                                                      r_1, &
                                                                     rt_1

      REAL,   DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(INOUT) :: ru_2, &
                                                                     rv_2, &
                                                                     rw_2, &
                                                                      r_2, &
                                                                     rt_2

      REAL,   DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(IN   ) :: ru_mid, &
                                                                     rv_mid, &
                                                                     rw_mid, &
                                                                      r_mid, &
                                                                     rt_mid

      REAL,   DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(INOUT) :: ru_m, &
                                                                     rv_m, &
                                                                     rw_m

      REAL,   DIMENSION(ims:ime, kms:kme, jms:jme), INTENT(OUT) :: h_diabatic

! local stuff

      INTEGER         :: i,j,k
      INTEGER :: i_start, i_end, j_start, j_end, i_endu, j_endv

      i_start = its
      i_end   = ite
      j_start = jts
      j_end   = jte

      i_endu = i_end
      j_endv = j_end

   IF( rk_step == 1 ) THEN     

      IF(i_end == ide) i_end = i_end - 1
      IF(j_end == jde) j_end = j_end - 1

      DO j = j_start, j_end
      DO k = kms, kme-1
      DO i = i_start, i_endu
        ru_2(i,k,j) = ru_2(i,k,j) + ru_1(i,k,j)
        ru_m(i,k,j) = ru_m(i,k,j) + ru_1(i,k,j)
      ENDDO
      ENDDO
      ENDDO

      DO j = j_start, j_endv
      DO k = kms, kme-1
      DO i = i_start, i_end
        rv_2(i,k,j) = rv_2(i,k,j) + rv_1(i,k,j)
        rv_m(i,k,j) = rv_m(i,k,j) + rv_1(i,k,j)
      ENDDO
      ENDDO
      ENDDO

      DO j = j_start, j_end
      DO k = kms, kme
      DO i = i_start, i_end
        rw_2(i,k,j) = rw_2(i,k,j) + rw_1(i,k,j)
        rw_m(i,k,j) = rw_m(i,k,j) + rw_1(i,k,j)
      ENDDO
      ENDDO
      ENDDO

      DO j = j_start, j_end
      DO k = kms, kme-1
      DO i = i_start, i_end
         r_2(i,k,j)        =  r_2(i,k,j) +  r_1(i,k,j)
        rt_2(i,k,j)        = rt_2(i,k,j) + rt_1(i,k,j) &
                                - dt*h_diabatic(i,k,j)
      ENDDO
      ENDDO
      ENDDO

   ELSE

      IF(i_end == ide) i_end = i_end - 1
      IF(j_end == jde) j_end = j_end - 1

      DO j = j_start, j_end
      DO k = kms, kme-1
      DO i = i_start, i_endu
        ru_2(i,k,j) = ru_2(i,k,j) + ru_mid(i,k,j)
        ru_m(i,k,j) = ru_m(i,k,j) + ru_mid(i,k,j)
      ENDDO
      ENDDO
      ENDDO

      DO j = j_start, j_endv
      DO k = kms, kme-1
      DO i = i_start, i_end
        rv_2(i,k,j) = rv_2(i,k,j) + rv_mid(i,k,j)
        rv_m(i,k,j) = rv_m(i,k,j) + rv_mid(i,k,j)
      ENDDO
      ENDDO
      ENDDO

      DO j = j_start, j_end
      DO k = kms, kme
      DO i = i_start, i_end
        rw_2(i,k,j) = rw_2(i,k,j) + rw_mid(i,k,j)
        rw_m(i,k,j) = rw_m(i,k,j) + rw_mid(i,k,j)
      ENDDO
      ENDDO
      ENDDO

      DO j = j_start, j_end
      DO k = kms, kme-1
      DO i = i_start, i_end
         r_2(i,k,j)        =  r_2(i,k,j) +  r_mid(i,k,j)
        rt_2(i,k,j)        = rt_2(i,k,j) + rt_mid(i,k,j) &
                                - dt*h_diabatic(i,k,j)
      ENDDO
      ENDDO
      ENDDO


  END IF

  IF (rk_step == rk_order) THEN

      ! reset the diabatic tendency to the current diabatic-updated
      ! value of theta

      DO j = j_start, j_end
      DO k = kms, kme-1
      DO i = i_start, i_end
        h_diabatic(i,k,j)  = rt_2(i,k,j)
      ENDDO
      ENDDO
      ENDDO

   END IF



 END SUBROUTINE rk_small_step_recouple

 subroutine dump_small( a,ivar,                   &
                        ims,ime,jms,jme,kms,kme,  &
                        ids,ide,jds,jde,kds,kde  )
 implicit none
 integer :: ivar
 integer :: ims,ime,jms,jme,kms,kme
 integer :: ids,ide,jds,jde,kds,kde
 real, dimension(ims:ime,kms:kme,jms:jme) :: a

 integer i,j,k

 if(ivar == 1)  then

! this is v

 write(14) ids,ide-1,kds,kde-1,jds,jde
 do j=jds,jde
 do k=kds,kde-1
 do i=ids,ide-1
   write(14) i,k,j,a(i,k,j)
 enddo
 enddo
 enddo

 else if(ivar == 2) then

! this is w

 write(11) ids,ide-1,kds,kde,jds,jde-1
 do j=jds,jde-1
 do k=kds,kde
 do i=ids,ide-1
   write(11) i,k,j,a(i,k,j)
 enddo
 enddo
 enddo

 else if( ivar == 3 ) then

! this is rt

 write(12) ids,ide-1,kds,kde-1,jds,jde-1
 do j=jds,jde-1
 do k=kds,kde-1
 do i=ids,ide-1
   write(12) i,k,j,a(i,k,j)
 enddo
 enddo
 enddo

 else if( ivar == 4 ) then

! this is rt

 write(13) ids,ide-1,kds,kde-1,jds,jde-1
 do j=jds,jde-1
 do k=kds,kde-1
 do i=ids,ide-1
   write(13) i,k,j,a(i,k,j)
 enddo
 enddo
 enddo

 else

  write(6,*) ' unknown variable, error exit ',ivar
  stop

 end if

  end subroutine dump_small


END MODULE module_small_step


