      SUBROUTINE STNDRD(LUNIT,MSGIN,LMSGOT,MSGOT)

!$$$  SUBPROGRAM DOCUMENTATION BLOCK
!
! SUBPROGRAM:    STNDRD
!   PRGMMR: ATOR             ORG: NP12       DATE: 2004-08-18
!
! ABSTRACT: THIS SUBROUTINE READS AN INPUT NCEP BUFR MESSAGE CONTAINED
!   WITHIN ARRAY MSGIN AND, USING THE BUFR TABLES INFORMATION ASSOCIATED
!   WITH LOGICAL UNIT LUNIT, OUTPUTS A "STANDARDIZED" VERSION OF THIS
!   SAME MESSAGE WITHIN ARRAY MSGOT.  THIS "STANDARDIZATION" INVOLVES
!   REMOVING ALL OCCURRENCES OF NCEP BUFRLIB-SPECIFIC BYTE COUNTERS AND
!   BIT PADS IN SECTION 4 AS WELL AS REPLACING THE TOP-LEVEL TABLE A FXY
!   NUMBER IN SECTION 3 WITH AN EQUIVALENT SEQUENCE OF LOWER-LEVEL
!   TABLE B, TABLE C, TABLE D AND/OR REPLICATION FXY NUMBERS WHICH
!   DIRECTLY CONSTITUTE THAT TABLE A FXY NUMBER AND WHICH THEMSELVES ARE
!   ALL WMO-STANDARD.  THE RESULT IS THAT THE OUTPUT MESSAGE IN MSGOT IS
!   NOW ENTIRELY COMPLIANT WITH WMO FM-94 BUFR REGULATIONS (I.E. IT IS
!   NOW "STANDARD"). IT IS IMPORTANT TO NOTE THAT THE SEQUENCE EXPANSION
!   WITHIN SECTION 3 MAY CAUSE THE FINAL "STANDARDIZED" BUFR MESSAGE TO
!   BE LONGER THAN THE ORIGINAL INPUT NCEP BUFR MESSAGE BY AS MANY AS
!   1000 BYTES, SO THE USER MUST ALLOW FOR ENOUGH SPACE TO ACCOMODATE
!   SUCH AN EXPANSION WITHIN THE MSGOT ARRAY.
!
! PROGRAM HISTORY LOG:
! 2004-08-18  J. ATOR    -- ORIGINAL AUTHOR
!                           THIS SUBROUTINE IS MODELED AFTER SUBROUTINE
!                           STANDARD; HOWEVER, IT USES SUBROUTINE RESTD
!                           TO EXPAND SECTION 3 AS MANY LEVELS AS
!                           NECESSARY IN ORDER TO ATTAIN TRUE WMO
!                           STANDARDIZATION (WHEREAS STANDARD ONLY
!                           EXPANDED THE TOP-LEVEL TABLE A FXY NUMBER
!                           ONE LEVEL DEEP), AND IT ALSO CONTAINS AN
!                           EXTRA INPUT ARGUMENT LMSGOT WHICH PREVENTS
!                           OVERFLOW OF THE MSGOT ARRAY
! 2005-11-29  J. ATOR    -- USE GETLENS AND IUPBS01; ENSURE THAT BYTE 4
!                           OF SECTION 4 IS ZEROED OUT IN MSGOT; CHECK
!                           EDITION NUMBER OF BUFR MESSAGE BEFORE 
!                           PADDING TO AN EVEN BYTE COUNT
!
! USAGE:    CALL STNDRD (LUNIT, MSGIN, LMSGOT, MSGOT)
!   INPUT ARGUMENT LIST:
!     LUNIT    - INTEGER: FORTRAN LOGICAL UNIT NUMBER FOR BUFR FILE
!     MSGIN    - INTEGER: *-WORD ARRAY CONTAINING BUFR MESSAGE IN NCEP
!                BUFR
!     LMSGOT   - INTEGER: DIMENSIONED SIZE (IN INTEGER WORDS) OF MSGOT;
!                USED BY THE SUBROUTINE TO ENSURE THAT IT DOES NOT
!                OVERFLOW THE MSGOT ARRAY
!
!   OUTPUT ARGUMENT LIST:
!     MSGOT    - INTEGER: *-WORD ARRAY CONTAINING INPUT BUFR MESSAGE
!                NOW IN STANDARDIZED BUFR
!
! REMARKS:
!    MSGIN AND MSGOT MUST BE SEPARATE ARRAYS.
!
!    THIS ROUTINE CALLS:        BORT     GETLENS  ISTDESC  IUPB
!                               IUPBS01  MVB      NUMTAB   PKB
!                               PKC      RESTD    STATUS   UPB
!                               UPC
!    THIS ROUTINE IS CALLED BY: MSGWRT
!                               Also called by application programs.
!
! ATTRIBUTES:
!   LANGUAGE: FORTRAN 77
!   MACHINE:  PORTABLE TO ALL PLATFORMS
!
!$$$

      INCLUDE 'bufrlib.prm'

      DIMENSION ICD(MAXNC)

      COMMON /HRDWRD/ NBYTW,NBITW,NREV,IORD(8)

      DIMENSION MSGIN(*),MSGOT(*)

      CHARACTER*128 BORT_STR
      CHARACTER*8   SUBSET
      CHARACTER*4   SEVN
      CHARACTER*1   TAB

!-----------------------------------------------------------------------
!-----------------------------------------------------------------------

!  LUNIT MUST POINT TO AN OPEN BUFR FILE
!  -------------------------------------

      CALL STATUS(LUNIT,LUN,IL,IM)
      IF(IL.EQ.0) GOTO 900

!  IDENTIFY THE SECTION LENGTHS AND ADDRESSES IN MSGIN
!  ---------------------------------------------------

      CALL GETLENS(MSGIN,5,LEN0,LEN1,LEN2,LEN3,LEN4,LEN5)

      IAD3 = LEN0+LEN1+LEN2
      IAD4 = IAD3+LEN3

      LENN = LEN0+LEN1+LEN2+LEN3+LEN4+LEN5

      LENM = IUPBS01(MSGIN,'LENM')

      IF(LENN.NE.LENM) GOTO 901

      MBIT = (LENN-4)*8
      CALL UPC(SEVN,4,MSGIN,MBIT)
      IF(SEVN.NE.'7777') GOTO 902

!  COPY SECTIONS 0 THROUGH PART OF SECTION 3 INTO MSGOT
!  ----------------------------------------------------

      MXBYTO = (LMSGOT*NBYTW) - 8

      LBYTO = IAD3+7
      IF(LBYTO.GT.MXBYTO) GOTO 905
      CALL MVB(MSGIN,1,MSGOT,1,LBYTO)

!  REWRITE NEW SECTION 3 IN A "STANDARD" FORM
!  ------------------------------------------

      NSUB = IUPB(MSGIN,IAD3+ 5,16)
      ISUB = IUPB(MSGIN,IAD3+10,16)
      IBIT = (IAD3+7)*8

      CALL NUMTAB(LUN,ISUB,SUBSET,TAB,ITAB)
      IF(ITAB.EQ.0) GOTO 903

      IF (ISTDESC(ISUB).EQ.0) THEN

!         ISUB IS A NON-STANDARD TABLE A DESCRIPTOR AND NEEDS
!         TO BE EXPANDED INTO AN EQUIVALENT STANDARD SEQUENCE  

          CALL RESTD(LUN,ISUB,NCD,ICD)
      ELSE

!         ISUB IS ALREADY A STANDARD DESCRIPTOR, SO JUST COPY
!         IT "AS IS" INTO THE NEW SECTION 3 (I.E. NO EXPANSION
!         IS NECESSARY!)

          NCD = 1
          ICD(NCD) = ISUB
      ENDIF

!     USE THE EDITION NUMBER TO DETERMINE THE LENGTH OF THE
!     NEW SECTION 3

      LEN3 = 7+(NCD*2)
      IBEN = IUPBS01(MSGIN,'BEN')
      IF(IBEN.LT.4) THEN
          LEN3 = LEN3+1
      ENDIF
      LBYTO = LBYTO + LEN3 - 7
      IF(LBYTO.GT.MXBYTO) GOTO 905

!     STORE THE DESCRIPTORS INTO THE NEW SECTION 3

      DO N=1,NCD
          CALL PKB(ICD(N),16,MSGOT,IBIT)
      ENDDO

!     DEPENDING ON THE EDITION NUMBER, PAD OUT THE NEW SECTION 3 WITH AN
!     ADDITIONAL ZEROED-OUT BYTE IN ORDER TO ENSURE AN EVEN BYTE COUNT

      IF(IBEN.LT.4) THEN
          CALL PKB(0,8,MSGOT,IBIT)
      ENDIF

!     STORE THE LENGTH OF THE NEW SECTION 3

      IBIT = IAD3*8
      CALL PKB(LEN3,24,MSGOT,IBIT)

!  NOW THE TRICKY PART - NEW SECTION 4
!  -----------------------------------

      NAD4 = IAD3+LEN3

      IBIT = (IAD4+4)*8
      JBIT = (NAD4+4)*8

      LBYTO = LBYTO + 4

!     COPY THE SUBSETS, MINUS THE BYTE COUNTERS AND BIT PADS,
!     INTO THE NEW SECTION 4

      DO 10 I=1,NSUB
      CALL UPB(LSUB,16,MSGIN,IBIT)

      DO L=1,LSUB-2
      CALL UPB(NVAL,8,MSGIN,IBIT)
      LBYTO = LBYTO + 1
      IF(LBYTO.GT.MXBYTO) GOTO 905
      CALL PKB(NVAL,8,MSGOT,JBIT)
      ENDDO

      DO K=1,8
      KBIT = IBIT-K-8
      CALL UPB(KVAL,8,MSGIN,KBIT)
      IF(KVAL.EQ.K) THEN
         JBIT = JBIT-K-8
         GOTO 10
      ENDIF
      ENDDO
      GOTO 904

10    ENDDO

!     FROM THIS POINT ON, WE WILL NEED (AT MOST) 6 MORE BYTES OF SPACE
!     WITHIN MSGOT IN ORDER TO BE ABLE TO STORE THE ENTIRE STANDARDIZED
!     MESSAGE (I.E. WE WILL NEED (AT MOST) 2 MORE ZEROED-OUT BYTES IN
!     SECTION 4 PLUS THE 4 BYTES '7777' IN SECTION 5), SO DO A FINAL
!     MSGOT OVERFLOW CHECK NOW.

      IF(LBYTO+6.GT.MXBYTO) GOTO 905

!     PAD THE NEW SECTION 4 WITH ZEROES UP TO THE NEXT WHOLE BYTE
!     BOUNDARY.

      DO WHILE(.NOT.(MOD(JBIT,8).EQ.0))
         CALL PKB(0,1,MSGOT,JBIT)
      ENDDO

!     DEPENDING ON THE EDITION NUMBER, WE MAY NEED TO FURTHER PAD THE
!     NEW SECTION 4 WITH AN ADDITIONAL ZEROED-OUT BYTE IN ORDER TO
!     ENSURE THAT THE PADDING IS UP TO AN EVEN BYTE BOUNDARY.

      IF( (IBEN.LT.4) .AND. (MOD(JBIT/8,2).NE.0) ) THEN
         CALL PKB(0,8,MSGOT,JBIT)
      ENDIF

      IBIT = NAD4*8
      LEN4 = JBIT/8 - NAD4
      CALL PKB(LEN4,24,MSGOT,IBIT)
      CALL PKB(0,8,MSGOT,IBIT)

!  FINISH THE NEW MESSAGE WITH AN UPDATED SECTION 0 BYTE COUNT
!  -----------------------------------------------------------

      IBIT = 32
      LENN = LEN0+LEN1+LEN2+LEN3+LEN4+LEN5
      CALL PKB(LENN,24,MSGOT,IBIT)

      CALL PKC('7777',4,MSGOT,JBIT)

!  EXITS
!  -----

      RETURN
900   CALL BORT('BUFRLIB: STNDRD - BUFR FILE IS CLOSED, IT MUST BE'// &
       ' OPEN')
901   WRITE(BORT_STR,'("BUFRLIB: STNDRD - INPUT MESSAGE LENGTH FROM'// &
       ' SECTION 0",I6," DOES NOT EQUAL SUM OF ALL INDIVIDUAL SECTION'// &
       ' LENGTHS (",I6,")")') LENM,LENN
      CALL BORT(BORT_STR)
902   WRITE(BORT_STR,'("BUFRLIB: STNDRD - INPUT MESSAGE DOES NOT '// &
       'END WITH ""7777"" (ENDS WITH ",A)') SEVN
      CALL BORT(BORT_STR)
903   CALL BORT('BUFRLIB: STNDRD - INPUT MESSAGE TABLE D SUBSET '// &
       'DESCRIPTOR NOT FOUND IN INTERNAL TABLE D')
904   CALL BORT('BUFRLIB: STNDRD - BIT MISMATCH COPYING SECTION 4 '// &
       'FROM INPUT TO OUTPUT (STANDARD) MESSAGE')
905   CALL BORT('BUFRLIB: STNDRD - OVERFLOW OF OUTPUT (STANDARD) '// &
       'MESSAGE ARRAY; TRY A LARGER DIMENSION FOR THIS ARRAY')
      END SUBROUTINE STNDRD
