!
! CRTM_Model_Profiles
!
! Module to define a set of climatological model profiles
!
!
! CREATION HISTORY:
!       Written by:     Paul van Delst, CIMSS/SSEC 13-Nov-2001
!                       paul.vandelst@noaa.gov
!       Modified by:    Yong Chen, NESDIS/CIRA 22-Aug-2007
!                       Yong.Chen@noaa.gov
!                       Paul van Delst, 22-Oct-2007
!                       paul.vandelst@noaa.gov
!


MODULE CRTM_Model_Profiles

  ! -----------------
  ! Environment setup
  ! -----------------
  ! Module use
  USE Type_Kinds            , ONLY: fp
  USE CRTM_Atmosphere_Define, ONLY: N_MODELS => N_VALID_CLIMATOLOGY_MODELS, &
                                    TROPICAL              , &
                                    MIDLATITUDE_SUMMER    , &
                                    MIDLATITUDE_WINTER    , &
                                    SUBARCTIC_SUMMER      , &
                                    SUBARCTIC_WINTER      , &
                                    US_STANDARD_ATMOSPHERE

  ! Disable implicit typing
  IMPLICIT NONE


  ! ------------
  ! Visibilities
  ! ------------
  PRIVATE
  ! Parameters
  PUBLIC :: MODEL_LEVEL_PRESSURE
  ! Procedures
  PUBLIC :: CRTM_Get_Model_Profile


  ! -----------------
  ! Module parameters
  ! -----------------
  CHARACTER(*), PARAMETER :: MODULE_RCS_ID = &
  '$Id: CRTM_Model_Profiles.f90 3707 2009-05-05 18:35:36Z paul.vandelst@noaa.gov $'

  ! Model dimension info
  INTEGER, PARAMETER :: N_MODEL_LEVELS    = 101
  INTEGER, PARAMETER :: N_MODEL_LAYERS    = N_MODEL_LEVELS-1
  INTEGER, PARAMETER :: N_MODEL_ABSORBERS = 2

  ! The absorber indices
  INTEGER, PARAMETER :: H2O = 1
  INTEGER, PARAMETER :: O3  = 2

  REAL(fp), PARAMETER :: MODEL_LEVEL_PRESSURE(0:N_MODEL_LAYERS) = &
  (/  0.00500000_fp,    0.01606500_fp,    0.03838300_fp,    0.07687900_fp,    0.13695400_fp, &
      0.22441200_fp,    0.34540400_fp,    0.50637400_fp,    0.71402300_fp,    0.97527400_fp, &
      1.29724000_fp,    1.68720000_fp,    2.15257300_fp,    2.70089700_fp,    3.33981200_fp, &
      4.07703800_fp,    4.92036200_fp,    5.87762300_fp,    6.95669500_fp,    8.16548000_fp, &
      9.51188900_fp,   11.00383500_fp,   12.64922300_fp,   14.45593600_fp,   16.43183300_fp, &
     18.58473200_fp,   20.92240800_fp,   23.45258300_fp,   26.18291800_fp,   29.12100900_fp, &
     32.27437800_fp,   35.65046700_fp,   39.25663300_fp,   43.10014400_fp,   47.18817100_fp, &
     51.52778600_fp,   56.12595300_fp,   60.98953000_fp,   66.12525900_fp,   71.53976800_fp, &
     77.23956000_fp,   83.23101600_fp,   89.52039000_fp,   96.11380300_fp,  103.01724400_fp, &
    110.23656500_fp,  117.77748100_fp,  125.64556200_fp,  133.84624000_fp,  142.38479600_fp, &
    151.26636600_fp,  160.49593900_fp,  170.07834800_fp,  180.01827900_fp,  190.32026000_fp, &
    200.98866500_fp,  212.02771200_fp,  223.44146100_fp,  235.23381400_fp,  247.40851400_fp, &
    259.96914200_fp,  272.91912000_fp,  286.26170600_fp,  300.00000000_fp,  314.13693600_fp, &
    328.67528600_fp,  343.61765900_fp,  358.96650300_fp,  374.72409800_fp,  390.89256600_fp, &
    407.47386100_fp,  424.46977600_fp,  441.88194100_fp,  459.71182100_fp,  477.96072200_fp, &
    496.62978500_fp,  515.71998900_fp,  535.23215300_fp,  555.16693500_fp,  575.52483200_fp, &
    596.30618200_fp,  617.51116300_fp,  639.13979700_fp,  661.19194600_fp,  683.66731600_fp, &
    706.56546000_fp,  729.88577200_fp,  753.62749400_fp,  777.78971600_fp,  802.37137600_fp, &
    827.37125900_fp,  852.78800300_fp,  878.62009600_fp,  904.86588000_fp,  931.52354900_fp, &
    958.59115400_fp,  986.06660100_fp, 1013.94765500_fp, 1042.23194000_fp, 1070.91694000_fp, &
   1100.00000000_fp /)
  
  
  ! ----------------------
  ! Module data statements
  ! ----------------------
  REAL(fp), SAVE :: Level_Temperature(0:N_MODEL_LAYERS,N_MODELS)
  DATA Level_Temperature(:,TROPICAL) / &
    178.174_fp,  192.328_fp,  210.111_fp,  225.459_fp,  239.120_fp, &
    251.516_fp,  258.971_fp,  264.536_fp,  268.260_fp,  269.940_fp, &
    267.898_fp,  263.813_fp,  259.762_fp,  256.009_fp,  252.506_fp, &
    249.223_fp,  246.228_fp,  243.425_fp,  240.822_fp,  238.354_fp, &
    236.044_fp,  233.852_fp,  231.779_fp,  229.858_fp,  228.013_fp, &
    226.216_fp,  224.456_fp,  222.760_fp,  221.135_fp,  219.623_fp, &
    218.157_fp,  216.716_fp,  215.232_fp,  213.323_fp,  211.116_fp, &
    208.960_fp,  206.863_fp,  204.840_fp,  202.874_fp,  201.054_fp, &
    199.289_fp,  197.557_fp,  195.862_fp,  195.130_fp,  196.031_fp, &
    196.910_fp,  199.292_fp,  201.792_fp,  204.249_fp,  206.692_fp, &
    209.083_fp,  211.535_fp,  214.055_fp,  216.524_fp,  218.876_fp, &
    221.164_fp,  223.408_fp,  225.700_fp,  227.958_fp,  230.178_fp, &
    232.509_fp,  234.797_fp,  237.043_fp,  239.252_fp,  241.422_fp, &
    243.553_fp,  245.698_fp,  247.807_fp,  249.880_fp,  251.983_fp, &
    254.067_fp,  256.118_fp,  258.148_fp,  260.155_fp,  262.131_fp, &
    264.092_fp,  266.071_fp,  268.020_fp,  269.939_fp,  271.870_fp, &
    273.782_fp,  275.665_fp,  277.531_fp,  279.397_fp,  281.235_fp, &
    283.047_fp,  284.395_fp,  285.475_fp,  286.540_fp,  287.590_fp, &
    289.118_fp,  290.683_fp,  292.227_fp,  293.750_fp,  295.281_fp, &
    296.790_fp,  298.280_fp,  299.749_fp,  301.199_fp,  302.630_fp, &
    304.043_fp /
  DATA Level_Temperature(:,MIDLATITUDE_SUMMER) / &
    166.103_fp,  181.157_fp,  202.859_fp,  222.255_fp,  239.663_fp, &   
    252.229_fp,  261.666_fp,  268.978_fp,  272.709_fp,  275.659_fp, &   
    275.105_fp,  270.621_fp,  265.928_fp,  261.515_fp,  257.445_fp, &   
    253.718_fp,  250.248_fp,  247.060_fp,  244.068_fp,  241.271_fp, &   
    238.659_fp,  236.454_fp,  234.345_fp,  232.403_fp,  230.575_fp, &   
    228.818_fp,  227.618_fp,  226.594_fp,  225.605_fp,  224.701_fp, &   
    223.884_fp,  223.178_fp,  222.456_fp,  221.710_fp,  221.003_fp, &   
    220.320_fp,  219.654_fp,  218.993_fp,  218.317_fp,  217.695_fp, &   
    217.154_fp,  216.627_fp,  216.116_fp,  215.700_fp,  215.700_fp, &   
    215.700_fp,  215.700_fp,  215.700_fp,  215.700_fp,  215.700_fp, &   
    215.700_fp,  215.730_fp,  215.767_fp,  216.038_fp,  218.372_fp, &   
    220.660_fp,  222.920_fp,  225.181_fp,  227.399_fp,  229.604_fp, &   
    231.820_fp,  233.995_fp,  236.134_fp,  238.241_fp,  240.310_fp, &   
    242.374_fp,  244.466_fp,  246.522_fp,  248.550_fp,  250.576_fp, &   
    252.568_fp,  254.527_fp,  256.478_fp,  258.399_fp,  260.290_fp, &   
    262.111_fp,  263.867_fp,  265.596_fp,  267.301_fp,  269.024_fp, &   
    270.722_fp,  272.394_fp,  274.060_fp,  275.718_fp,  277.352_fp, &   
    278.963_fp,  280.560_fp,  282.137_fp,  283.691_fp,  285.218_fp, &   
    286.393_fp,  287.551_fp,  288.694_fp,  289.823_fp,  290.949_fp, &   
    292.059_fp,  293.155_fp,  294.236_fp,  295.303_fp,  296.356_fp, &   
    297.395_fp /
  DATA Level_Temperature(:,MIDLATITUDE_WINTER) / &
    200.964_fp,  216.091_fp,  227.925_fp,  237.836_fp,  246.210_fp, &  
    253.447_fp,  259.898_fp,  263.297_fp,  265.617_fp,  264.331_fp, &  
    258.371_fp,  252.296_fp,  246.806_fp,  241.773_fp,  237.138_fp, &  
    232.901_fp,  228.974_fp,  225.394_fp,  222.050_fp,  219.798_fp, &  
    218.606_fp,  217.468_fp,  216.770_fp,  216.126_fp,  215.508_fp, &  
    215.407_fp,  215.317_fp,  215.230_fp,  215.200_fp,  215.200_fp, &  
    215.200_fp,  215.200_fp,  215.200_fp,  215.200_fp,  215.200_fp, &  
    215.200_fp,  215.200_fp,  215.200_fp,  215.363_fp,  215.611_fp, &  
    215.854_fp,  216.092_fp,  216.324_fp,  216.551_fp,  216.773_fp, &  
    216.988_fp,  217.199_fp,  217.406_fp,  217.607_fp,  217.805_fp, &  
    217.998_fp,  218.188_fp,  218.374_fp,  218.557_fp,  218.736_fp, &  
    218.911_fp,  219.083_fp,  219.251_fp,  219.417_fp,  219.580_fp, &  
    220.181_fp,  222.085_fp,  223.955_fp,  225.794_fp,  227.652_fp, &  
    229.477_fp,  231.270_fp,  233.065_fp,  234.839_fp,  236.584_fp, &  
    238.315_fp,  240.046_fp,  241.750_fp,  243.426_fp,  245.108_fp, &  
    246.771_fp,  248.408_fp,  250.028_fp,  251.653_fp,  253.253_fp, &  
    254.830_fp,  256.399_fp,  257.966_fp,  259.509_fp,  261.030_fp, &  
    262.193_fp,  263.071_fp,  263.936_fp,  264.789_fp,  265.636_fp, &  
    266.477_fp,  267.306_fp,  268.124_fp,  268.933_fp,  269.738_fp, &  
    270.532_fp,  271.316_fp,  272.089_fp,  272.852_fp,  273.605_fp, &  
    274.348_fp /
  DATA Level_Temperature(:,SUBARCTIC_SUMMER) / &
    162.600_fp,  176.793_fp,  198.862_fp,  219.134_fp,  237.464_fp, &   
    254.166_fp,  265.996_fp,  272.936_fp,  275.498_fp,  277.137_fp, &   
    276.306_fp,  274.243_fp,  271.376_fp,  267.499_fp,  262.519_fp, &   
    257.948_fp,  253.686_fp,  249.780_fp,  246.155_fp,  242.879_fp, &   
    239.836_fp,  237.823_fp,  235.897_fp,  234.239_fp,  232.785_fp, &   
    231.387_fp,  230.336_fp,  229.438_fp,  228.572_fp,  227.634_fp, &   
    226.602_fp,  225.672_fp,  225.200_fp,  225.200_fp,  225.200_fp, &   
    225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp, &   
    225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp, &   
    225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp, &   
    225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp, &   
    225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp,  225.200_fp, &   
    225.200_fp,  226.105_fp,  228.344_fp,  230.542_fp,  232.719_fp, &   
    234.915_fp,  237.074_fp,  239.195_fp,  241.311_fp,  243.391_fp, &   
    245.437_fp,  247.492_fp,  249.535_fp,  251.545_fp,  253.541_fp, &   
    255.569_fp,  257.566_fp,  259.532_fp,  261.175_fp,  262.673_fp, &   
    264.149_fp,  265.604_fp,  267.058_fp,  268.491_fp,  269.903_fp, &   
    271.305_fp,  272.712_fp,  274.099_fp,  275.466_fp,  276.825_fp, &   
    278.180_fp,  279.516_fp,  280.835_fp,  282.152_fp,  283.486_fp, &   
    284.801_fp,  286.099_fp,  287.379_fp,  288.643_fp,  289.889_fp, &   
    291.120_fp /
  DATA Level_Temperature(:,SUBARCTIC_WINTER) / &
    214.590_fp,  231.481_fp,  244.770_fp,  248.281_fp,  250.442_fp, &   
    255.518_fp,  259.144_fp,  259.262_fp,  255.110_fp,  249.389_fp, &   
    244.240_fp,  239.569_fp,  235.403_fp,  231.542_fp,  227.948_fp, &   
    224.653_fp,  221.850_fp,  220.096_fp,  218.457_fp,  217.428_fp, &   
    216.448_fp,  215.538_fp,  214.690_fp,  213.877_fp,  213.104_fp, &   
    212.364_fp,  211.653_fp,  211.345_fp,  211.756_fp,  212.155_fp, &   
    212.541_fp,  212.915_fp,  213.277_fp,  213.629_fp,  213.971_fp, &   
    214.303_fp,  214.627_fp,  214.942_fp,  215.249_fp,  215.547_fp, &   
    215.839_fp,  216.123_fp,  216.401_fp,  216.673_fp,  216.938_fp, &   
    217.198_fp,  217.200_fp,  217.200_fp,  217.200_fp,  217.200_fp, &   
    217.200_fp,  217.200_fp,  217.200_fp,  217.200_fp,  217.200_fp, &   
    217.200_fp,  217.200_fp,  217.200_fp,  217.200_fp,  217.200_fp, &   
    217.200_fp,  217.200_fp,  217.457_fp,  218.476_fp,  219.477_fp, &   
    220.460_fp,  222.270_fp,  224.190_fp,  226.077_fp,  227.963_fp, &   
    229.873_fp,  231.752_fp,  233.601_fp,  235.457_fp,  237.298_fp, &   
    239.109_fp,  240.893_fp,  242.699_fp,  244.477_fp,  246.229_fp, &   
    247.892_fp,  249.174_fp,  250.437_fp,  251.682_fp,  252.835_fp, &   
    253.620_fp,  254.394_fp,  255.157_fp,  255.909_fp,  256.660_fp, &   
    257.400_fp,  258.129_fp,  258.849_fp,  258.826_fp,  258.408_fp, &   
    257.995_fp,  257.588_fp,  257.187_fp,  256.790_fp,  256.399_fp, &   
    256.013_fp /
  DATA Level_Temperature(:,US_STANDARD_ATMOSPHERE) / &
    190.195_fp,  203.667_fp,  215.175_fp,  226.809_fp,  237.789_fp, &   
    247.507_fp,  256.483_fp,  263.555_fp,  268.955_fp,  270.636_fp, &   
    267.045_fp,  261.561_fp,  256.386_fp,  251.670_fp,  247.321_fp, &   
    243.261_fp,  239.551_fp,  236.057_fp,  232.759_fp,  229.832_fp, &   
    228.503_fp,  227.233_fp,  226.133_fp,  225.245_fp,  224.392_fp, &   
    223.595_fp,  222.847_fp,  222.126_fp,  221.425_fp,  220.733_fp, &   
    220.065_fp,  219.420_fp,  218.797_fp,  218.196_fp,  217.614_fp, &   
    217.106_fp,  216.700_fp,  216.700_fp,  216.700_fp,  216.700_fp, &   
    216.700_fp,  216.700_fp,  216.700_fp,  216.700_fp,  216.700_fp, &   
    216.700_fp,  216.700_fp,  216.700_fp,  216.700_fp,  216.700_fp, &   
    216.700_fp,  216.700_fp,  216.700_fp,  216.700_fp,  216.700_fp, &   
    216.723_fp,  216.757_fp,  216.790_fp,  218.296_fp,  220.415_fp, &   
    222.495_fp,  224.553_fp,  226.585_fp,  228.580_fp,  230.577_fp, &   
    232.588_fp,  234.564_fp,  236.514_fp,  238.474_fp,  240.401_fp, &   
    242.296_fp,  244.201_fp,  246.087_fp,  247.943_fp,  249.783_fp, &   
    251.627_fp,  253.442_fp,  255.229_fp,  257.021_fp,  258.798_fp, &   
    260.549_fp,  262.275_fp,  264.015_fp,  265.730_fp,  267.420_fp, &   
    269.095_fp,  270.776_fp,  272.433_fp,  274.067_fp,  275.689_fp, &   
    277.314_fp,  278.917_fp,  280.497_fp,  282.066_fp,  283.643_fp, &   
    285.200_fp,  286.736_fp,  288.251_fp,  289.746_fp,  291.221_fp, &   
    292.678_fp /

  REAL(fp), SAVE :: Level_Absorber(0:N_MODEL_LAYERS,N_MODEL_ABSORBERS,N_MODELS)
  DATA Level_Absorber(:,H2O,TROPICAL) / &
    8.77980825E-04_fp, 1.63665427E-03_fp, 2.41525886E-03_fp, 3.01358863E-03_fp, 3.42673415E-03_fp, &
    3.69727056E-03_fp, 3.73182898E-03_fp, 3.73182898E-03_fp, 3.73182898E-03_fp, 3.70485491E-03_fp, & 
    3.62553040E-03_fp, 3.52250711E-03_fp, 3.42918702E-03_fp, 3.30368092E-03_fp, 3.18356224E-03_fp, & 
    3.07218031E-03_fp, 2.96914223E-03_fp, 2.87229788E-03_fp, 2.78234404E-03_fp, 2.69709168E-03_fp, & 
    2.61724153E-03_fp, 2.54151913E-03_fp, 2.46345214E-03_fp, 2.37323353E-03_fp, 2.28665909E-03_fp, & 
    2.20863685E-03_fp, 2.14020216E-03_fp, 2.07426306E-03_fp, 2.01766533E-03_fp, 1.99628659E-03_fp, & 
    1.90185294E-03_fp, 1.79636495E-03_fp, 1.75789394E-03_fp, 1.71098213E-03_fp, 1.65816603E-03_fp, & 
    1.63469692E-03_fp, 1.61839270E-03_fp, 1.61712589E-03_fp, 1.61712589E-03_fp, 1.65651334E-03_fp, & 
    1.69871321E-03_fp, 1.73942166E-03_fp, 1.77895400E-03_fp, 1.81305422E-03_fp, 1.83851684E-03_fp, & 
    1.86338111E-03_fp, 2.07865778E-03_fp, 2.31078836E-03_fp, 2.60268897E-03_fp, 3.11383907E-03_fp, & 
    3.61397564E-03_fp, 4.29053420E-03_fp, 5.15159631E-03_fp, 5.99495710E-03_fp, 9.54264560E-03_fp, & 
    1.36728122E-02_fp, 1.77218010E-02_fp, 2.69138794E-02_fp, 3.64198574E-02_fp, 4.62694696E-02_fp, & 
    7.10905559E-02_fp, 9.54559132E-02_fp, 1.19808735E-01_fp, 1.65311675E-01_fp, 2.10009241E-01_fp, & 
    2.53925406E-01_fp, 3.23805339E-01_fp, 3.93088062E-01_fp, 4.61199718E-01_fp, 5.57061296E-01_fp, & 
    6.58709582E-01_fp, 7.58695269E-01_fp, 8.89551039E-01_fp, 1.04316485E+00_fp, 1.19433850E+00_fp, & 
    1.36357009E+00_fp, 1.59234548E+00_fp, 1.81758872E+00_fp, 2.03938326E+00_fp, 2.24071288E+00_fp, & 
    2.43503683E+00_fp, 2.62646230E+00_fp, 2.96715923E+00_fp, 3.68750198E+00_fp, 4.39735364E+00_fp, & 
    5.09695512E+00_fp, 6.07752900E+00_fp, 7.20936332E+00_fp, 8.32520255E+00_fp, 9.42539639E+00_fp, & 
    1.01510542E+01_fp, 1.08244053E+01_fp, 1.14885020E+01_fp, 1.21559103E+01_fp, 1.31774892E+01_fp, & 
    1.41852861E+01_fp, 1.51795783E+01_fp, 1.61606176E+01_fp, 1.71286637E+01_fp, 1.80839545E+01_fp, & 
    1.90267237E+01_fp /
  DATA Level_Absorber(:,O3,TROPICAL) / &
    4.76282998E-01_fp, 2.63575307E-01_fp, 2.38323078E-01_fp, 4.34252158E-01_fp, 7.32103525E-01_fp, &
    1.05832797E+00_fp, 1.49900320E+00_fp, 1.96708261E+00_fp, 2.51463612E+00_fp, 3.08189624E+00_fp, &   
    3.82225579E+00_fp, 4.75582031E+00_fp, 5.80609455E+00_fp, 6.90465318E+00_fp, 7.85315879E+00_fp, &   
    8.62916701E+00_fp, 9.17872204E+00_fp, 9.64583838E+00_fp, 9.76328654E+00_fp, 9.83182051E+00_fp, &   
    9.68129051E+00_fp, 9.45808993E+00_fp, 9.15268309E+00_fp, 8.60873569E+00_fp, 8.08675973E+00_fp, &   
    7.46417712E+00_fp, 6.70969487E+00_fp, 5.98272612E+00_fp, 5.26763918E+00_fp, 4.51144304E+00_fp, &   
    3.87334450E+00_fp, 3.28178900E+00_fp, 2.66325567E+00_fp, 2.20360475E+00_fp, 1.86393581E+00_fp, &   
    1.62600430E+00_fp, 1.41629416E+00_fp, 1.19079017E+00_fp, 9.69572281E-01_fp, 7.60019680E-01_fp, &   
    5.56473977E-01_fp, 4.22289982E-01_fp, 3.16357103E-01_fp, 2.34147587E-01_fp, 1.90916468E-01_fp, &   
    1.48701241E-01_fp, 1.37969522E-01_fp, 1.30953035E-01_fp, 1.23887238E-01_fp, 1.16261319E-01_fp, &   
    1.08799712E-01_fp, 1.02767939E-01_fp, 9.82121846E-02_fp, 9.37500852E-02_fp, 8.87007788E-02_fp, &   
    8.35895445E-02_fp, 7.85787706E-02_fp, 7.42657223E-02_fp, 7.00914630E-02_fp, 6.60152554E-02_fp, &   
    6.25765013E-02_fp, 5.92008847E-02_fp, 5.59111488E-02_fp, 5.39198530E-02_fp, 5.19638019E-02_fp, &   
    5.00419464E-02_fp, 4.83436243E-02_fp, 4.66785655E-02_fp, 4.50416508E-02_fp, 4.40871060E-02_fp, &   
    4.33155387E-02_fp, 4.25565913E-02_fp, 4.18230593E-02_fp, 4.11113222E-02_fp, 4.04108911E-02_fp, &   
    3.97271370E-02_fp, 3.90712599E-02_fp, 3.84255092E-02_fp, 3.77896455E-02_fp, 3.71872689E-02_fp, &   
    3.65994968E-02_fp, 3.60204917E-02_fp, 3.55648316E-02_fp, 3.54061032E-02_fp, 3.52496865E-02_fp, &   
    3.50955285E-02_fp, 3.47584482E-02_fp, 3.43210595E-02_fp, 3.38898521E-02_fp, 3.34646906E-02_fp, &   
    3.29662458E-02_fp, 3.24653767E-02_fp, 3.19713915E-02_fp, 3.14763682E-02_fp, 3.07596938E-02_fp, &   
    3.00526880E-02_fp, 2.93551562E-02_fp, 2.86669218E-02_fp, 2.79878026E-02_fp, 2.73176317E-02_fp, &   
    2.66562451E-02_fp /

  DATA Level_Absorber(:,H2O,MIDLATITUDE_SUMMER) / &
    8.80599931E-04_fp, 1.47572005E-03_fp, 1.97813347E-03_fp, 2.38351250E-03_fp, 2.72802818E-03_fp, &   
    3.00292035E-03_fp, 3.19132561E-03_fp, 3.32180485E-03_fp, 3.37724401E-03_fp, 3.42084323E-03_fp, &   
    3.42028608E-03_fp, 3.39397328E-03_fp, 3.32997533E-03_fp, 3.25260801E-03_fp, 3.17150369E-03_fp, &   
    3.13410940E-03_fp, 3.10449317E-03_fp, 3.08824187E-03_fp, 3.06740482E-03_fp, 3.03934609E-03_fp, &   
    3.01055985E-03_fp, 2.97174541E-03_fp, 2.93462097E-03_fp, 2.88485311E-03_fp, 2.83070490E-03_fp, &   
    2.77866814E-03_fp, 2.72916030E-03_fp, 2.68161015E-03_fp, 2.63574105E-03_fp, 2.57094018E-03_fp, &   
    2.48649721E-03_fp, 2.42662857E-03_fp, 2.34999817E-03_fp, 2.25339352E-03_fp, 2.19271705E-03_fp, &   
    2.13957036E-03_fp, 2.08783721E-03_fp, 2.04260672E-03_fp, 2.01023719E-03_fp, 1.98452690E-03_fp, &   
    1.96920466E-03_fp, 1.96410491E-03_fp, 1.97853791E-03_fp, 1.99496639E-03_fp, 2.02268300E-03_fp, &   
    2.04974830E-03_fp, 2.07583594E-03_fp, 2.10129201E-03_fp, 2.29281897E-03_fp, 2.67060180E-03_fp, &   
    3.04024471E-03_fp, 3.67850181E-03_fp, 4.36794406E-03_fp, 5.46397037E-03_fp, 1.02532782E-02_fp, &   
    1.49470824E-02_fp, 2.22350362E-02_fp, 3.65409017E-02_fp, 5.05732548E-02_fp, 7.11067223E-02_fp, &   
    1.03253429E-01_fp, 1.34809905E-01_fp, 1.67179038E-01_fp, 2.01107829E-01_fp, 2.34436101E-01_fp, &   
    2.71873055E-01_fp, 3.18610664E-01_fp, 3.64549797E-01_fp, 4.14550309E-01_fp, 4.86968165E-01_fp, &   
    5.58187485E-01_fp, 6.28241915E-01_fp, 7.17769497E-01_fp, 8.07854932E-01_fp, 8.96509358E-01_fp, &   
    1.00673091E+00_fp, 1.13686247E+00_fp, 1.26498487E+00_fp, 1.40046198E+00_fp, 1.68417177E+00_fp, &   
    1.96361553E+00_fp, 2.23889119E+00_fp, 2.56503717E+00_fp, 2.93826004E+00_fp, 3.30604729E+00_fp, &   
    3.66852373E+00_fp, 4.24303573E+00_fp, 4.84696946E+00_fp, 5.44236840E+00_fp, 6.03073164E+00_fp, &   
    6.69658337E+00_fp, 7.35322328E+00_fp, 8.00083839E+00_fp, 8.65543046E+00_fp, 9.43033313E+00_fp, &   
    1.01947817E+01_fp, 1.09489865E+01_fp, 1.16931386E+01_fp, 1.24274348E+01_fp, 1.31520557E+01_fp, &   
    1.38671785E+01_fp /
  DATA Level_Absorber(:,O3,MIDLATITUDE_SUMMER) / &
    5.28761628E-01_fp, 1.96792365E-01_fp, 2.54518897E-01_fp, 4.75536727E-01_fp, 7.92056224E-01_fp, &   
    1.15672290E+00_fp, 1.48711958E+00_fp, 1.78681001E+00_fp, 2.33267719E+00_fp, 2.85792951E+00_fp, &   
    3.51791554E+00_fp, 4.36402526E+00_fp, 5.39690129E+00_fp, 6.48151796E+00_fp, 7.56018675E+00_fp, &   
    8.25159185E+00_fp, 8.73449876E+00_fp, 8.83901351E+00_fp, 8.75395997E+00_fp, 8.39305957E+00_fp, &   
    8.02923474E+00_fp, 7.57159438E+00_fp, 7.13388001E+00_fp, 6.75296026E+00_fp, 6.40472425E+00_fp, &   
    6.07006743E+00_fp, 5.70200930E+00_fp, 5.33504602E+00_fp, 4.98105597E+00_fp, 4.53413503E+00_fp, &   
    3.99106853E+00_fp, 3.60604361E+00_fp, 3.25661050E+00_fp, 2.94597039E+00_fp, 2.65143375E+00_fp, &   
    2.37328364E+00_fp, 2.15148077E+00_fp, 1.92042065E+00_fp, 1.66020348E+00_fp, 1.40703933E+00_fp, &   
    1.16068975E+00_fp, 9.52782106E-01_fp, 8.13550699E-01_fp, 6.92511553E-01_fp, 6.47949041E-01_fp, &   
    6.04433707E-01_fp, 5.62490237E-01_fp, 5.21562211E-01_fp, 4.89261010E-01_fp, 4.66483670E-01_fp, &   
    4.44197104E-01_fp, 3.97334483E-01_fp, 3.45605470E-01_fp, 2.97181021E-01_fp, 2.69526368E-01_fp, &   
    2.42423177E-01_fp, 2.18830070E-01_fp, 2.03628367E-01_fp, 1.88717304E-01_fp, 1.73248858E-01_fp, &   
    1.56581717E-01_fp, 1.40220595E-01_fp, 1.27885420E-01_fp, 1.21531632E-01_fp, 1.15290303E-01_fp, &   
    1.09042505E-01_fp, 1.02657654E-01_fp, 9.63818824E-02_fp, 9.05268627E-02_fp, 8.62821781E-02_fp, &   
    8.21077443E-02_fp, 7.80015891E-02_fp, 7.39311148E-02_fp, 6.99229303E-02_fp, 6.59784159E-02_fp, &   
    6.27189244E-02_fp, 6.00970366E-02_fp, 5.75156292E-02_fp, 5.50040365E-02_fp, 5.30191697E-02_fp, &   
    5.10641484E-02_fp, 4.91382877E-02_fp, 4.73518022E-02_fp, 4.56961666E-02_fp, 4.40646437E-02_fp, &   
    4.24566797E-02_fp, 4.10229808E-02_fp, 3.96358393E-02_fp, 3.82683009E-02_fp, 3.69259339E-02_fp, &   
    3.59937718E-02_fp, 3.50745060E-02_fp, 3.41678744E-02_fp, 3.32825328E-02_fp, 3.24819665E-02_fp, &   
    3.16922005E-02_fp, 3.09130175E-02_fp, 3.01442203E-02_fp, 2.93856052E-02_fp, 2.86369860E-02_fp, &   
    2.78981793E-02_fp /
    
  DATA Level_Absorber(:,H2O,MIDLATITUDE_WINTER) / &
    8.74332246E-04_fp, 1.49718079E-03_fp, 1.95195793E-03_fp, 2.35711591E-03_fp, 2.65468785E-03_fp, &  
    2.85767582E-03_fp, 3.00096567E-03_fp, 3.04944903E-03_fp, 3.08308098E-03_fp, 3.10985748E-03_fp, &  
    3.10985748E-03_fp, 3.10985748E-03_fp, 3.09351587E-03_fp, 3.07299566E-03_fp, 3.05427505E-03_fp, &  
    3.03702602E-03_fp, 3.02095600E-03_fp, 3.00616912E-03_fp, 2.99230417E-03_fp, 2.97922469E-03_fp, &  
    2.96686654E-03_fp, 2.95506941E-03_fp, 2.94405213E-03_fp, 2.93351392E-03_fp, 2.92340138E-03_fp, &  
    2.91367493E-03_fp, 2.90431481E-03_fp, 2.89529602E-03_fp, 2.87835857E-03_fp, 2.85745031E-03_fp, &  
    2.83684182E-03_fp, 2.82482238E-03_fp, 2.81705932E-03_fp, 2.80603976E-03_fp, 2.79887173E-03_fp, &  
    2.79887173E-03_fp, 2.79887173E-03_fp, 2.79887173E-03_fp, 2.79887173E-03_fp, 2.79887173E-03_fp, &  
    2.79887173E-03_fp, 2.79887173E-03_fp, 2.81434083E-03_fp, 2.84255984E-03_fp, 2.87009063E-03_fp, &  
    2.89695072E-03_fp, 2.92319027E-03_fp, 2.94883936E-03_fp, 2.97391660E-03_fp, 3.01152488E-03_fp, &  
    3.05970643E-03_fp, 3.10686597E-03_fp, 3.32681239E-03_fp, 3.55402722E-03_fp, 3.91087891E-03_fp, &  
    4.78254265E-03_fp, 5.63707400E-03_fp, 7.47526348E-03_fp, 1.15169670E-02_fp, 1.54824914E-02_fp, &  
    1.97084441E-02_fp, 2.48533895E-02_fp, 2.99050194E-02_fp, 3.51272761E-02_fp, 4.51472902E-02_fp, &  
    5.49921348E-02_fp, 6.46667487E-02_fp, 8.45843276E-02_fp, 1.07466432E-01_fp, 1.29966066E-01_fp, &  
    1.62099986E-01_fp, 2.12027416E-01_fp, 2.61145512E-01_fp, 3.09475916E-01_fp, 3.63203514E-01_fp, &  
    4.17297509E-01_fp, 4.70548251E-01_fp, 5.28053350E-01_fp, 6.04854643E-01_fp, 6.80491245E-01_fp, &  
    7.54990531E-01_fp, 8.54663471E-01_fp, 9.85885149E-01_fp, 1.11518275E+00_fp, 1.24259726E+00_fp, &  
    1.35998736E+00_fp, 1.46918467E+00_fp, 1.57682899E+00_fp, 1.68295208E+00_fp, 1.78567781E+00_fp, &  
    1.88517492E+00_fp, 1.98329553E+00_fp, 2.08006757E+00_fp, 2.18395930E+00_fp, 2.30730386E+00_fp, &  
    2.42898440E+00_fp, 2.54903439E+00_fp, 2.66748424E+00_fp, 2.78436530E+00_fp, 2.89970630E+00_fp, &  
    3.01353544E+00_fp /
  DATA Level_Absorber(:,O3,MIDLATITUDE_WINTER) / &
    5.13824109E-01_fp, 2.41632954E-01_fp, 3.01139649E-01_fp, 4.80917555E-01_fp, 7.91364237E-01_fp, &  
    1.18908931E+00_fp, 1.64984950E+00_fp, 2.25519636E+00_fp, 2.88203086E+00_fp, 3.80482427E+00_fp, &  
    4.62171912E+00_fp, 5.64746315E+00_fp, 6.42547774E+00_fp, 6.95559661E+00_fp, 7.13618951E+00_fp, &  
    7.16580449E+00_fp, 7.11413005E+00_fp, 6.99974493E+00_fp, 6.86599321E+00_fp, 6.65957726E+00_fp, &  
    6.38140684E+00_fp, 6.11586435E+00_fp, 5.93419689E+00_fp, 5.76476456E+00_fp, 5.60217603E+00_fp, &  
    5.44579517E+00_fp, 5.29530398E+00_fp, 5.15030072E+00_fp, 4.92238564E+00_fp, 4.65345678E+00_fp, &  
    4.38838356E+00_fp, 4.13446410E+00_fp, 3.88989110E+00_fp, 3.65366241E+00_fp, 3.38741275E+00_fp, &  
    3.05569077E+00_fp, 2.73064175E+00_fp, 2.41212341E+00_fp, 2.13743042E+00_fp, 1.88945447E+00_fp, &  
    1.67691975E+00_fp, 1.48641733E+00_fp, 1.32538682E+00_fp, 1.18927599E+00_fp, 1.07098983E+00_fp, &  
    9.84619020E-01_fp, 9.00243622E-01_fp, 8.58883440E-01_fp, 8.18564495E-01_fp, 7.79803354E-01_fp, &  
    7.42464818E-01_fp, 7.05918287E-01_fp, 6.40673216E-01_fp, 5.74770590E-01_fp, 5.11627432E-01_fp, &  
    4.55289027E-01_fp, 4.00057942E-01_fp, 3.49484701E-01_fp, 3.07909396E-01_fp, 2.67117714E-01_fp, &  
    2.30569056E-01_fp, 2.05080137E-01_fp, 1.80053518E-01_fp, 1.55870755E-01_fp, 1.39525396E-01_fp, &  
    1.23465785E-01_fp, 1.07683866E-01_fp, 9.82162019E-02_fp, 9.08257439E-02_fp, 8.35588159E-02_fp, &  
    7.68041071E-02_fp, 7.08774194E-02_fp, 6.50468045E-02_fp, 5.93096935E-02_fp, 5.57481739E-02_fp, &  
    5.26523318E-02_fp, 4.96047496E-02_fp, 4.65702815E-02_fp, 4.34473838E-02_fp, 4.03718448E-02_fp, &  
    3.73425513E-02_fp, 3.52424995E-02_fp, 3.42842253E-02_fp, 3.33400020E-02_fp, 3.24095305E-02_fp, &  
    3.15057165E-02_fp, 3.06253763E-02_fp, 2.97575562E-02_fp, 2.89020001E-02_fp, 2.84289367E-02_fp, &  
    2.83112407E-02_fp, 2.81951729E-02_fp, 2.80807004E-02_fp, 2.79853632E-02_fp, 2.79347499E-02_fp, &  
    2.78848194E-02_fp, 2.78355580E-02_fp, 2.77869531E-02_fp, 2.77389920E-02_fp, 2.76916629E-02_fp, &  
    2.76449541E-02_fp /

  DATA Level_Absorber(:,H2O,SUBARCTIC_SUMMER) / &
    8.69383728E-04_fp, 1.36116616E-03_fp, 1.76469831E-03_fp, 2.10026204E-03_fp, 2.44573806E-03_fp, &  
    2.68348563E-03_fp, 2.86237283E-03_fp, 2.99606196E-03_fp, 3.04567272E-03_fp, 3.07754036E-03_fp, &  
    3.10655579E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, &  
    3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, &  
    3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10332744E-03_fp, 3.09229744E-03_fp, &  
    3.08169754E-03_fp, 3.07164176E-03_fp, 3.06200913E-03_fp, 3.05271704E-03_fp, 3.03799773E-03_fp, &  
    3.01659788E-03_fp, 3.00831971E-03_fp, 2.99845479E-03_fp, 2.98689086E-03_fp, 2.95285888E-03_fp, &  
    2.91662195E-03_fp, 2.88155558E-03_fp, 2.84738628E-03_fp, 2.81403006E-03_fp, 2.76435656E-03_fp, &  
    2.70130163E-03_fp, 2.63110641E-03_fp, 2.55605408E-03_fp, 2.51179113E-03_fp, 2.49757023E-03_fp, &  
    2.48788599E-03_fp, 2.48788599E-03_fp, 2.48788599E-03_fp, 2.48788599E-03_fp, 2.48788599E-03_fp, &  
    2.55305414E-03_fp, 2.66197293E-03_fp, 2.77071777E-03_fp, 3.13346408E-03_fp, 3.48887207E-03_fp, &  
    4.22682497E-03_fp, 5.83106127E-03_fp, 7.40422819E-03_fp, 1.09552220E-02_fp, 1.69722497E-02_fp, &  
    2.28773144E-02_fp, 3.34187290E-02_fp, 5.08394269E-02_fp, 6.79481093E-02_fp, 9.32770644E-02_fp, &  
    1.45896166E-01_fp, 1.97605412E-01_fp, 2.48431246E-01_fp, 3.24174426E-01_fp, 3.98706765E-01_fp, &  
    4.72005572E-01_fp, 5.61681354E-01_fp, 6.58406539E-01_fp, 7.53580570E-01_fp, 8.61983503E-01_fp, &  
    1.02204873E+00_fp, 1.17961877E+00_fp, 1.33475601E+00_fp, 1.52343275E+00_fp, 1.72391468E+00_fp, &  
    1.92138205E+00_fp, 2.11943114E+00_fp, 2.36062938E+00_fp, 2.59829098E+00_fp, 2.83249127E+00_fp, &  
    3.08782910E+00_fp, 3.40062556E+00_fp, 3.70897345E+00_fp, 4.01296376E+00_fp, 4.31618141E+00_fp, &  
    4.62075034E+00_fp, 4.92110566E+00_fp, 5.21733292E+00_fp, 5.57739905E+00_fp, 6.06579203E+00_fp, &  
    6.54759618E+00_fp, 7.02294405E+00_fp, 7.49195601E+00_fp, 7.95475623E+00_fp, 8.41145844E+00_fp, &  
    8.86217434E+00_fp /
  DATA Level_Absorber(:,O3,SUBARCTIC_SUMMER) / &
    6.02447863E-01_fp, 1.85384866E-01_fp, 2.45755187E-01_fp, 4.43875328E-01_fp, 7.61277121E-01_fp, &  
    1.05158664E+00_fp, 1.34585211E+00_fp, 1.65291517E+00_fp, 2.07443467E+00_fp, 2.48432665E+00_fp, &  
    3.12568202E+00_fp, 3.95253307E+00_fp, 4.85091650E+00_fp, 5.83269455E+00_fp, 6.90940812E+00_fp, &  
    7.44284770E+00_fp, 7.78765316E+00_fp, 7.73487051E+00_fp, 7.58385625E+00_fp, 7.21986676E+00_fp, &  
    6.85994658E+00_fp, 6.36681745E+00_fp, 5.89515953E+00_fp, 5.61600841E+00_fp, 5.47413698E+00_fp, &  
    5.33779770E+00_fp, 5.11691393E+00_fp, 4.86911782E+00_fp, 4.63008203E+00_fp, 4.40678739E+00_fp, &  
    4.20034843E+00_fp, 3.86871267E+00_fp, 3.57850376E+00_fp, 3.33060533E+00_fp, 2.98547464E+00_fp, &  
    2.63590625E+00_fp, 2.29762991E+00_fp, 2.01200498E+00_fp, 1.79748565E+00_fp, 1.58901363E+00_fp, &  
    1.38625538E+00_fp, 1.21632217E+00_fp, 1.07152004E+00_fp, 9.65303396E-01_fp, 8.96710707E-01_fp, &  
    8.30054162E-01_fp, 7.65667586E-01_fp, 7.02740825E-01_fp, 6.58995663E-01_fp, 6.17019138E-01_fp, &  
    5.76716277E-01_fp, 5.37801027E-01_fp, 4.99725104E-01_fp, 4.65860708E-01_fp, 4.32681386E-01_fp, &  
    3.98552843E-01_fp, 3.61453664E-01_fp, 3.25072988E-01_fp, 2.87804478E-01_fp, 2.49240970E-01_fp, &  
    2.11395041E-01_fp, 1.82920946E-01_fp, 1.67893374E-01_fp, 1.53134955E-01_fp, 1.38232663E-01_fp, &  
    1.22272790E-01_fp, 1.06588886E-01_fp, 9.11729285E-02_fp, 8.69304911E-02_fp, 8.27823305E-02_fp, &  
    7.87028231E-02_fp, 7.50455252E-02_fp, 7.16194928E-02_fp, 6.82484027E-02_fp, 6.48620845E-02_fp, &  
    6.12800313E-02_fp, 5.77538175E-02_fp, 5.42820463E-02_fp, 5.15875999E-02_fp, 4.92297424E-02_fp, &  
    4.69073391E-02_fp, 4.46667180E-02_fp, 4.30751373E-02_fp, 4.15068937E-02_fp, 3.99614901E-02_fp, &  
    3.84894396E-02_fp, 3.71657147E-02_fp, 3.58608157E-02_fp, 3.45743575E-02_fp, 3.33635638E-02_fp, &  
    3.22617141E-02_fp, 3.11751081E-02_fp, 3.01034363E-02_fp, 2.89659113E-02_fp, 2.76858761E-02_fp, &  
    2.64231096E-02_fp, 2.51772645E-02_fp, 2.39480253E-02_fp, 2.27350665E-02_fp, 2.15380901E-02_fp, &  
    2.03568032E-02_fp /

  DATA Level_Absorber(:,H2O,SUBARCTIC_WINTER) / &
    8.84696872E-04_fp, 1.54955175E-03_fp, 2.03052785E-03_fp, 2.47065038E-03_fp, 2.74191225E-03_fp, &   
    2.92146556E-03_fp, 3.03039715E-03_fp, 3.06709313E-03_fp, 3.10012079E-03_fp, 3.10985748E-03_fp, &   
    3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, &   
    3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, &   
    3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, &   
    3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.10985748E-03_fp, 3.09146619E-03_fp, &   
    3.07147389E-03_fp, 3.05208862E-03_fp, 3.03329059E-03_fp, 3.01505782E-03_fp, 2.99733188E-03_fp, &   
    2.98010536E-03_fp, 2.96333121E-03_fp, 2.94701069E-03_fp, 2.93111777E-03_fp, 2.91562533E-03_fp, &   
    2.90051362E-03_fp, 2.88576944E-03_fp, 2.87137086E-03_fp, 2.85730694E-03_fp, 2.84353388E-03_fp, &   
    2.83008447E-03_fp, 2.81700923E-03_fp, 2.80423099E-03_fp, 2.79170619E-03_fp, 2.77943253E-03_fp, &   
    2.77862415E-03_fp, 3.14321091E-03_fp, 3.50019707E-03_fp, 4.03488780E-03_fp, 4.91460361E-03_fp, &   
    5.77677700E-03_fp, 7.22889321E-03_fp, 9.30815524E-03_fp, 1.13476641E-02_fp, 1.33261166E-02_fp, &   
    1.52411067E-02_fp, 1.71209366E-02_fp, 1.87015151E-02_fp, 1.94619961E-02_fp, 2.02090171E-02_fp, &   
    2.09429787E-02_fp, 3.85915946E-02_fp, 5.87588759E-02_fp, 7.85852755E-02_fp, 9.68791925E-02_fp, &   
    1.12588983E-01_fp, 1.28041819E-01_fp, 1.43244161E-01_fp, 1.71432368E-01_fp, 2.04090211E-01_fp, &   
    2.36234320E-01_fp, 2.67877346E-01_fp, 3.27052478E-01_fp, 3.85437353E-01_fp, 4.42936821E-01_fp, &   
    5.00201056E-01_fp, 5.60201978E-01_fp, 6.19315101E-01_fp, 6.77561459E-01_fp, 7.32076449E-01_fp, &   
    7.71902320E-01_fp, 8.11157232E-01_fp, 8.49853863E-01_fp, 8.87881652E-01_fp, 9.15307184E-01_fp, &   
    9.42350623E-01_fp, 9.69019926E-01_fp, 9.95322689E-01_fp, 9.85632908E-01_fp, 9.56886655E-01_fp, &   
    9.28528213E-01_fp, 9.00549779E-01_fp, 8.72944270E-01_fp, 8.45704377E-01_fp, 8.18823407E-01_fp, &   
    7.92294783E-01_fp /
  DATA Level_Absorber(:,O3,SUBARCTIC_WINTER) / &
    6.64488655E-01_fp, 2.70387123E-01_fp, 4.89988085E-01_fp, 6.44061875E-01_fp, 8.95052643E-01_fp, &   
    1.31605255E+00_fp, 1.82244415E+00_fp, 2.41243879E+00_fp, 2.87476351E+00_fp, 3.67618430E+00_fp, &   
    4.54517916E+00_fp, 5.26145086E+00_fp, 5.80774687E+00_fp, 6.07829864E+00_fp, 6.24554631E+00_fp, &   
    6.21897368E+00_fp, 6.16447409E+00_fp, 6.02603072E+00_fp, 5.89135176E+00_fp, 5.68564390E+00_fp, &   
    5.48967750E+00_fp, 5.30381111E+00_fp, 5.12709769E+00_fp, 4.95779587E+00_fp, 4.85867873E+00_fp, &   
    4.79704146E+00_fp, 4.73772631E+00_fp, 4.67583708E+00_fp, 4.60726127E+00_fp, 4.54086129E+00_fp, &   
    4.42972331E+00_fp, 4.24271853E+00_fp, 4.10758583E+00_fp, 3.98549190E+00_fp, 3.81449433E+00_fp, &   
    3.59662883E+00_fp, 3.27299705E+00_fp, 2.94629363E+00_fp, 2.61411140E+00_fp, 2.31486882E+00_fp, &   
    2.04760764E+00_fp, 1.81770663E+00_fp, 1.63250734E+00_fp, 1.46370947E+00_fp, 1.33084427E+00_fp, &   
    1.20110129E+00_fp, 1.07496781E+00_fp, 9.51699345E-01_fp, 8.42396537E-01_fp, 7.43729115E-01_fp, &   
    6.47186109E-01_fp, 5.52641076E-01_fp, 4.60067038E-01_fp, 3.93909310E-01_fp, 3.76229324E-01_fp, &   
    3.58901894E-01_fp, 3.41887263E-01_fp, 3.25172172E-01_fp, 3.08776655E-01_fp, 2.86854038E-01_fp, &   
    2.58462510E-01_fp, 2.30592267E-01_fp, 2.01994858E-01_fp, 1.70228779E-01_fp, 1.39024942E-01_fp, &   
    1.08366609E-01_fp, 9.61530320E-02_fp, 8.71327802E-02_fp, 7.82649952E-02_fp, 6.97853309E-02_fp, &   
    6.19016706E-02_fp, 5.41469585E-02_fp, 4.65179519E-02_fp, 4.31724147E-02_fp, 4.14266931E-02_fp, &   
    3.97084331E-02_fp, 3.80169583E-02_fp, 3.65604959E-02_fp, 3.51271963E-02_fp, 3.37156327E-02_fp, &   
    3.23437308E-02_fp, 3.10974815E-02_fp, 2.98696723E-02_fp, 2.86598662E-02_fp, 2.74879278E-02_fp, &   
    2.64305466E-02_fp, 2.53883245E-02_fp, 2.43609248E-02_fp, 2.33525872E-02_fp, 2.27333876E-02_fp, &   
    2.21228147E-02_fp, 2.15206889E-02_fp, 2.09268386E-02_fp, 2.03303186E-02_fp, 1.97360877E-02_fp, &   
    1.91498734E-02_fp, 1.85715146E-02_fp, 1.80008647E-02_fp, 1.74377726E-02_fp, 1.68821001E-02_fp, &   
    1.63337111E-02_fp /

  DATA Level_Absorber(:,H2O,US_STANDARD_ATMOSPHERE) / &
    8.86997040E-04_fp, 1.52427987E-03_fp, 2.01101378E-03_fp, 2.40599752E-03_fp, 2.72437587E-03_fp, &   
    2.96236227E-03_fp, 3.10396312E-03_fp, 3.19368814E-03_fp, 3.23610094E-03_fp, 3.25981037E-03_fp, &   
    3.25671227E-03_fp, 3.23196162E-03_fp, 3.19285497E-03_fp, 3.13971111E-03_fp, 3.10625749E-03_fp, &   
    3.08100578E-03_fp, 3.06248483E-03_fp, 3.04448043E-03_fp, 3.02081059E-03_fp, 2.99803553E-03_fp, &   
    2.97440450E-03_fp, 2.95184624E-03_fp, 2.92511340E-03_fp, 2.89196774E-03_fp, 2.86016092E-03_fp, &   
    2.82977471E-03_fp, 2.80069281E-03_fp, 2.77267144E-03_fp, 2.73864263E-03_fp, 2.68478764E-03_fp, &   
    2.64118787E-03_fp, 2.59714199E-03_fp, 2.54483821E-03_fp, 2.50568261E-03_fp, 2.47311126E-03_fp, &   
    2.44672175E-03_fp, 2.42271064E-03_fp, 2.40621873E-03_fp, 2.39238370E-03_fp, 2.38458004E-03_fp, &   
    2.38110201E-03_fp, 2.38850647E-03_fp, 2.39914434E-03_fp, 2.42737549E-03_fp, 2.45492999E-03_fp, &   
    2.71901083E-03_fp, 2.99417069E-03_fp, 3.24509817E-03_fp, 3.47714919E-03_fp, 3.78040842E-03_fp, &   
    4.95999293E-03_fp, 6.11455674E-03_fp, 7.57663188E-03_fp, 9.42311177E-03_fp, 1.12322375E-02_fp, &   
    1.42466434E-02_fp, 1.78602949E-02_fp, 2.14039606E-02_fp, 2.73154771E-02_fp, 3.41753036E-02_fp, &   
    4.09074844E-02_fp, 5.42724979E-02_fp, 7.17133280E-02_fp, 8.88417820E-02_fp, 1.15945509E-01_fp, &   
    1.56046471E-01_fp, 1.95454032E-01_fp, 2.34255117E-01_fp, 2.72750760E-01_fp, 3.10602956E-01_fp, &   
    3.47828689E-01_fp, 4.06535547E-01_fp, 4.70306893E-01_fp, 5.33055556E-01_fp, 6.01901014E-01_fp, &   
    6.85096335E-01_fp, 7.66994753E-01_fp, 8.47628708E-01_fp, 9.65099457E-01_fp, 1.09450413E+00_fp, &   
    1.22196301E+00_fp, 1.34952968E+00_fp, 1.52006224E+00_fp, 1.68809431E+00_fp, 1.85367915E+00_fp, &   
    2.03383153E+00_fp, 2.26692831E+00_fp, 2.49671002E+00_fp, 2.72324444E+00_fp, 2.94770972E+00_fp, &   
    3.17163654E+00_fp, 3.39246541E+00_fp, 3.61025923E+00_fp, 3.83453885E+00_fp, 4.08727861E+00_fp, &   
    4.33660869E+00_fp, 4.58259770E+00_fp, 4.82530792E+00_fp, 5.06480362E+00_fp, 5.30114363E+00_fp, &   
    5.53438577E+00_fp /
  DATA Level_Absorber(:,O3,US_STANDARD_ATMOSPHERE) / &
    4.73303997E-01_fp, 2.74146902E-01_fp, 2.80243651E-01_fp, 5.10480276E-01_fp, 8.31073636E-01_fp, &   
    1.12571713E+00_fp, 1.58104620E+00_fp, 2.16173433E+00_fp, 2.87091990E+00_fp, 3.74371439E+00_fp, &   
    4.73885897E+00_fp, 5.61330664E+00_fp, 6.34570524E+00_fp, 7.09761422E+00_fp, 7.50525354E+00_fp, &   
    7.77591671E+00_fp, 7.81936233E+00_fp, 7.80536994E+00_fp, 7.56992926E+00_fp, 7.33375237E+00_fp, &   
    7.02220367E+00_fp, 6.72479815E+00_fp, 6.44285088E+00_fp, 6.17639446E+00_fp, 5.92070090E+00_fp, &   
    5.68739719E+00_fp, 5.47387081E+00_fp, 5.26813104E+00_fp, 5.03222902E+00_fp, 4.69211312E+00_fp, &   
    4.38133099E+00_fp, 4.07406867E+00_fp, 3.74952993E+00_fp, 3.39674005E+00_fp, 3.03656514E+00_fp, &   
    2.78144823E+00_fp, 2.52642430E+00_fp, 2.23528370E+00_fp, 1.96713487E+00_fp, 1.74480839E+00_fp, &   
    1.53398068E+00_fp, 1.34350343E+00_fp, 1.16379657E+00_fp, 1.01995634E+00_fp, 8.79563704E-01_fp, &   
    7.81925294E-01_fp, 6.89400605E-01_fp, 6.15902146E-01_fp, 5.56537854E-01_fp, 4.99365854E-01_fp, &   
    4.53753714E-01_fp, 4.09109075E-01_fp, 3.72418996E-01_fp, 3.45262678E-01_fp, 3.18655731E-01_fp, &   
    2.88188006E-01_fp, 2.55989671E-01_fp, 2.24414924E-01_fp, 1.95655516E-01_fp, 1.68400469E-01_fp, &   
    1.41652577E-01_fp, 1.23541576E-01_fp, 1.10965251E-01_fp, 9.86141761E-02_fp, 8.73600481E-02_fp, &   
    7.74538294E-02_fp, 6.77189026E-02_fp, 5.91969666E-02_fp, 5.63118453E-02_fp, 5.34749481E-02_fp, &   
    5.06850024E-02_fp, 4.80182173E-02_fp, 4.54157757E-02_fp, 4.28550687E-02_fp, 4.08112252E-02_fp, &   
    3.98355325E-02_fp, 3.88750495E-02_fp, 3.79293958E-02_fp, 3.69055940E-02_fp, 3.58639488E-02_fp, &   
    3.48379664E-02_fp, 3.38620749E-02_fp, 3.36773243E-02_fp, 3.34952826E-02_fp, 3.33158922E-02_fp, &   
    3.31308212E-02_fp, 3.29213219E-02_fp, 3.27148022E-02_fp, 3.25112010E-02_fp, 3.21398618E-02_fp, &   
    3.13748034E-02_fp, 3.06203294E-02_fp, 2.98762248E-02_fp, 2.91576054E-02_fp, 2.84997705E-02_fp, &   
    2.78508104E-02_fp, 2.72105464E-02_fp, 2.65788165E-02_fp, 2.59554534E-02_fp, 2.53403040E-02_fp, &   
    2.47332178E-02_fp /


CONTAINS


!--------------------------------------------------------------------------------
!:sdoc+:
!
! NAME:
!       CRTM_Get_Model_Profile
!
! PURPOSE:
!       Subroutine to retrieve climatological model profile levels for
!       supplementing a user input profile for use in the CRTM. 
!
!       Currently the only absorbers available are water vapour and ozone.
!
! CALLING SEQUENCE:
!       CALL CRTM_Get_Model_Profile( Pressure   , &
!                                    Temperature, &
!                                    Absorber   , &
!                                    Model=Model  )
!                                    
!
! OUTPUT ARGUMENTS:
!       Pressure:        Model profile pressure levels.
!                        UNITS:      hectoPascals
!                        TYPE:       REAL(fp)
!                        DIMENSION:  Rank-1 (0:)
!                        ATTRIBUTES: INTENT(OUT)
!
!       Temperature:     Model profile level temperatures.
!                        UNITS:      Kelvin
!                        TYPE:       REAL(fp)
!                        DIMENSION:  Rank-1 (0:)
!                        ATTRIBUTES: INTENT(OUT)
!
!       Absorber:        Model profile level absorber amounts. Only two
!                        absorbers are supported:
!                          Absorber #1 is water vapour
!                          Absorber #2 is ozone
!                        Note that the number and order of the absorbers will
!                        change in the future.
!                        UNITS:      g/kg (H2O) and ppmv (O3)
!                        TYPE:       REAL(fp)
!                        DIMENSION:  Rank-2 (0:,2)
!                        ATTRIBUTES: INTENT(OUT)
!
! OPTIONAL INPUT ARGUMENTS:
!       Model:           Integer flag defining the climatological profile
!                        to retrieve. Valid values are:
!                          1: Tropical        
!                          2: Midlatitude summer   
!                          3: Midlatitude winter   
!                          4: Subarctic summer    
!                          5: Subarctic winter    
!                          6: U.S. Standard Atmosphere (DEFAULT)
!                        If no model is specified, or if an invalid model
!                        is specified, the U.S. Standard Atmosphere model
!                        is used.
!                        UNITS:      N/A
!                        TYPE:       INTEGER
!                        DIMENSION:  Scalar
!                        ATTRIBUTES: INTENT(IN), OPTIONAL
!
! COMMENTS:
!       - The model profiles are specified at the AIRS 101 pressure levels. If
!         the output arguments are larger than this, only the 101 defined 
!         levels are filled with profile data.
!       - Currently the only absorbers available are water vapour and ozone.
!
!:sdoc-:
!--------------------------------------------------------------------------------

  SUBROUTINE CRTM_Get_Model_Profile( p, t, a, Model )
    ! Arguments
    REAL(fp),           INTENT(OUT) :: p(0:), t(0:), a(0:,:) 
    INTEGER , OPTIONAL, INTENT(IN)  :: Model            
    ! Local Variables
    INTEGER :: i, k
    
    ! Use US Std Atm if passed model is invalid
    i = US_STANDARD_ATMOSPHERE
    IF ( PRESENT(Model) ) THEN
      i = Model
      IF ( i < 1 .OR. i > N_MODELS ) i = US_STANDARD_ATMOSPHERE
    END IF

    ! Get the requireds number of levels    
    k = MIN(SIZE(p)-1,N_MODEL_LAYERS)
    p = MODEL_LEVEL_PRESSURE(0:k)
    t = Level_Temperature(0:k,i)
    a = Level_Absorber(0:k,:,i)
    
  END SUBROUTINE CRTM_Get_Model_Profile

END MODULE CRTM_Model_Profiles
