MODULE all_io

   USE header_data
   USE util
   USE diags
   USE date_pack
   USE wrfsi_maxdims
   USE wrf_metadata

   TYPE input_fields_3d
      REAL , POINTER , DIMENSION(:,:,:) :: array
      TYPE(sh)                          :: small_header
   END TYPE input_fields_3d

   TYPE input_fields_2d
      REAL , POINTER , DIMENSION(:,:)   :: array
      TYPE(sh)                          :: small_header
   END TYPE input_fields_2d

   TYPE input_fields_1d
      REAL , POINTER , DIMENSION(:)     :: array
      TYPE(sh)                          :: small_header
   END TYPE input_fields_1d

   TYPE(input_fields_3d) , ALLOCATABLE , DIMENSION(:) :: allp_3d
   TYPE(input_fields_2d) , ALLOCATABLE , DIMENSION(:) :: all_2d
   TYPE(input_fields_1d) , ALLOCATABLE , DIMENSION(:) :: all_1d

   TYPE input_and_bdy_fields_3d
      REAL , POINTER , DIMENSION(:,:,:) :: array
      TYPE(sh)                          :: small_header
   END TYPE input_and_bdy_fields_3d
   TYPE(input_and_bdy_fields_3d) , ALLOCATABLE , DIMENSION(:) :: allm_3d

   INTEGER :: nump_3d , numm_3d , num_2d , num_1d
   INTEGER :: iprog_no_old

   CHARACTER(LEN=19) :: sh_date

   INTEGER , DIMENSION(:) , ALLOCATABLE :: output_levels
   TYPE (wrfvar_metadata) :: var_up(max_fg_variables)
   TYPE (wrfvar_metadata) :: var_sfc(max_fg_variables)

   INTEGER :: index_t , index_u , index_v , index_height , index_rh
   INTEGER :: index_qc , index_qr , index_qs , index_qi , index_qg , index_qn
   INTEGER :: index_sfc_qc , index_sfc_qr , index_sfc_qs , index_sfc_qi , index_sfc_qg , index_sfc_qn
   INTEGER :: index_sfc_t , index_sfc_u , index_sfc_v , index_sfc_rh , index_terrain , index_land_use , index_mapfaccr , &
              index_mapfacdt , index_coriolis , index_latitcrs , index_longicrs , index_latitdot , index_longidot , &
              index_snowcovr , index_psealvld , index_psealvlc , index_tseasfc , index_vegfr
   INTEGER :: index_p

CONTAINS

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   SUBROUTINE read_data ( unit_fg , file_fg , count_fg , kx )

      IMPLICIT NONE

      !  Input variables.

      INTEGER :: unit_fg , count_fg
      CHARACTER(LEN=132) , DIMENSION(100) :: file_fg

      !  Output variables.

      INTEGER :: kx

      !  Local variables.

      INTEGER :: loop3 , loop2 , loop1
      INTEGER :: flag , ok , bhi_i1

      !  Loop over all of the variables.

      loop3 = 0
      loop2 = 0
      loop1 = 0

      var_loop : DO

         !  Start off with a flag, as ususal.  We are here after a big header
         !  flag, so we can just look for the small header without trying to 
         !  do too many error tests for which flag is expected.
   
         READ ( unit_fg , IOSTAT = ok ) flag
   
         IF      ( ( ok .LT. 0 ) .AND. ( count_fg .EQ. 100 ) ) THEN
            PRINT '(A,I8,A)','Unexpected EOF in first-guess data: ',ok,'.'
            PRINT '(A)','Exhausted list of input file names in the namelist.input file.'
            PRINT '(A)','You may have asked for a time period that is not available.'
            STOP 'EOF_FG_read_all_files'
         ELSE IF ( ( ok .LT. 0 ) .AND. ( LEN_TRIM ( file_fg(count_fg+1) ) .EQ. 0 ) ) THEN
            PRINT '(A,I8,A)','Unexpected EOF in first-guess data: ',ok,'.'
            PRINT '(A)','No subsequent file specified for input in the namelist.input file.'
            PRINT '(A)','You may have asked for a time period that is not available.'
            STOP 'EOF_in_FG_data'
         ELSE IF ( ( ok .LT. 0 ) .AND. ( LEN_TRIM ( file_fg(count_fg+1) ) .NE. 0 ) ) THEN
            PRINT '(A,I8,A)','Unexpected EOF in first-guess data: ',ok,'.'
            PRINT '(A,A,A)','Moving to next file in the namelist.input file: ',TRIM ( file_fg(count_fg+1) ) ,'.'
            CLOSE ( unit_fg )
            count_fg = count_fg + 1
            CALL get_fg_file ( file_fg(count_fg) ,  unit_fg )
            CYCLE var_loop
         END IF

         IF ( flag .EQ. bh_flag ) THEN
            PRINT '(A,I1,A)','Wrong flag, wanted small header flag: ',flag,'.'
            PRINT '(A)','Assuming that this is an instance of multiple input files or cat''ed first-guess input files.'
            READ ( unit_fg ) bhi_i1
            IF ( ( bhi_i1 .EQ. 2 ) .OR. ( bhi_i1 .EQ. 3 ) .OR. ( bhi_i1 .EQ. 8 ) ) THEN
               CYCLE var_loop
            ELSE
               PRINT '(A,I8,A)','This did not turn out to be first-guess data, program ID = ',bhi_i1,'.'
               STOP 'Messed_up_multiple_input_files'
            END IF
         ELSE IF ( flag .EQ. eot_flag ) THEN
            PRINT '(A)','Found the end of the time period.'
            EXIT var_loop
         END IF
   
         !  We have the right flag, so get the header.
   
         READ ( unit_fg ) small_header%num_dims , small_header%start_dims , small_header%end_dims , &
                        small_header%xtime , small_header%staggering , small_header%ordering , &
                        small_header%current_date , small_header%name , small_header%units , &
                        small_header%description
   
         !  Let's allocate space for this one array.  Then read it in.  The 3d array will be
         !  input as a surface field and a 3d array (the pressure levels).

         IF      ( small_header%num_dims .eq. 3 ) THEN
            
            !  Increment the count of the fields.
          
            loop3 = loop3 + 1

            !  Do not place surface fields in 2d arrays
            ! loop2 = loop2 + 1
    
            !  Number of pressure levels, INCLUDING the surface (for vinterp).

            kx = small_header%end_dims(3)

            !  Allocate space for the data in the input array area.

            ALLOCATE ( allp_3d(loop3)%array(small_header%end_dims(1),small_header%end_dims(2),small_header%end_dims(3)) )
            !ALLOCATE ( all_2d(loop2)%array(small_header%end_dims(1),small_header%end_dims(2)) )

            !  Use this space to input the data. First level data are surface.

            READ ( unit_fg ) allp_3d(loop3)%array
            !READ ( unit_fg ) all_2d(loop2)%array , allp_3d(loop3)%array
       
            !  Assign the small_header data to this variable's small_header storage.  We need to
            !  modify the 2d header to reflect the 2d nature of the data.            

            !all_2d(loop2)%small_header = small_header
            !all_2d(loop2)%small_header%num_dims = 2
            !all_2d(loop2)%small_header%ordering(3:3) = ' '
            !all_2d(loop2)%small_header%name = TRIM ( all_2d(loop2)%small_header%name ) // 'SFC'

            allp_3d(loop3)%small_header = small_header

            !  Check to see if this 3d data contains any of the non-traditional hydrometeor fields that may
            !  be used with lateral boundary conditions in the model.

         ELSE IF ( small_header%num_dims .eq. 2 ) THEN
            
            !  Discard some data, if found from MM5 stream

            IF (small_header%name(1:6) .EQ. 'SOICAT' .OR. small_header%name(1:6) .EQ. 'VEGCAT' ) THEN
               
            READ ( unit_fg ) 

            ELSE

            !  Increment the count of the fields.

            loop2 = loop2 + 1
    
            !  Allocate space for the data in the input array area.

            ALLOCATE ( all_2d(loop2)%array(small_header%end_dims(1),small_header%end_dims(2)) )

            !  Use this space to input the data.

            READ ( unit_fg ) all_2d(loop2)%array

            !  Assign the small_header data to this variable's small_header storage.

            all_2d(loop2)%small_header = small_header
           
            !  Did we find a "real" SST yet?  If so, we need to set the SST mean flag
            !  to .FALSE. - which inplies that we can do physically reasonable variable
            !  SST computations.

            END IF

         ELSE IF ( small_header%num_dims .eq. 1 ) THEN
            
            !  Increment the count of the fields.
          
            loop1 = loop1 + 1
    
            !  Allocate space for the data in the input array area.

            ALLOCATE ( all_1d(loop1)%array(small_header%end_dims(1)) )

            !  Use this space to input the data.

            READ ( unit_fg ) all_1d(loop1)%array

            !  Assign the small_header data to this variable's small_header storage.

            all_1d(loop1)%small_header = small_header
   
         END IF
   
      END DO var_loop

      !  A few values that we want out of here: the date, and the number of fields.

      sh_date = small_header%current_date(1:19)

      nump_3d = loop3
      num_2d = loop2
      num_1d = loop1

   END SUBROUTINE read_data

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   SUBROUTINE do_namelist ( input_file ,  &
                            start_year , start_month , start_day , start_hour , &
                            start_minute , start_second , start_frac , &
                            end_year ,   end_month ,   end_day ,   end_hour , &
                            end_minute ,   end_second ,   end_frac , &
                            interval ,  less_than_24h , &
                            ptop , wrth2o , &
                            psfc_method , &
                            simulation_name, user_desc )

      IMPLICIT NONE

      !  Input variables.
   
      !  RECORD0
      
      CHARACTER(LEN=132) , DIMENSION(100) :: input_file
      NAMELIST /RECORD0/ input_file

      !  RECORD1
      
      INTEGER :: start_year , start_month , start_day , start_hour , &
                 start_minute , start_second , start_frac
      INTEGER ::   end_year ,   end_month ,   end_day ,   end_hour , &
                 end_minute ,   end_second ,   end_frac
      INTEGER :: interval
      LOGICAL :: less_than_24h

      NAMELIST /RECORD1/ start_year , start_month , start_day ,   &
                         start_hour ,  start_minute , start_second , start_frac ,  &
                         end_year ,   end_month ,   end_day ,     &
                         end_hour ,  end_minute ,   end_second ,   end_frac ,  &
                         interval , less_than_24h
      
      !  RECORD2
      
      REAL    :: ptop
      LOGICAL :: wrth2o
      INTEGER :: psfc_method

      NAMELIST /RECORD2/ ptop, wrth2o, psfc_method
      
      !  RECORD3 (new)

      CHARACTER(LEN= 80)                  :: simulation_name, user_desc
      NAMELIST /RECORD3/ simulation_name, user_desc
      
      !  Local variables.

      LOGICAL :: is_it_there = .FALSE.
      INTEGER , PARAMETER :: unit_nml=10

      !  Initialize the variable "input_file" to blank

      input_file = '                                                                                ' // &
                   '                                                    '

      !  Default value for variable "psfc_method"
      
      psfc_method = 0
      
      !     Does the file exist?
      
      INQUIRE ( FILE = 'namelist.input' , EXIST = is_it_there )
      
      IF ( is_it_there ) THEN
      
         !  The file exists, get a unit number.
      
         OPEN ( FILE   = 'namelist.input' , &
                UNIT   =  unit_nml        , &
                STATUS = 'OLD'            , &
                FORM   = 'FORMATTED'      , &
                ACTION = 'READ'           , &
                ACCESS = 'SEQUENTIAL'     )
   
         !  File is opened, so read it.
   
         READ (unit_nml , NML = RECORD0 )
!        WRITE (6    , NML = RECORD0 )
         READ (unit_nml , NML = RECORD1 )
         WRITE (6    , NML = RECORD1 )
         READ (unit_nml , NML = RECORD2 )
         WRITE (6    , NML = RECORD2 )
         READ (unit_nml , NML = RECORD3 )
         WRITE (6    , NML = RECORD3 )
   
      ELSE
         PRINT '(A)','Could not find the namelist: "namelist.input".'
         STOP 'No_namelist_found'
      END IF
   
      CLOSE ( unit_nml )
   
   END SUBROUTINE do_namelist

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
   
   SUBROUTINE get_fg_file ( input_file , unit_fg )
   
      IMPLICIT NONE
   
      !  Input variables.

      CHARACTER(LEN=132) , INTENT(IN) :: input_file

      INTEGER , INTENT(IN) :: unit_fg

      !  Local variables.

      LOGICAL :: is_it_there = .FALSE.
   
   
      !  Does the file exist?
   
      INQUIRE ( FILE = TRIM(input_file) , EXIST = is_it_there )
   
      IF ( is_it_there ) THEN
         OPEN ( FILE   = TRIM(input_file) , &
                UNIT   =  unit_fg         ,  &
                STATUS = 'OLD'            , &
                FORM   = 'UNFORMATTED'    , &
                ACCESS = 'SEQUENTIAL'     )
         PRINT '(A)','Opened pressure level data for input.'
      
      ELSE
         PRINT '(A,A,A)','Could not find file ',TRIM(input_file),'.'
         STOP 'No_fg_file_found'
      
      END IF
   
   END SUBROUTINE get_fg_file
   
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
   
   SUBROUTINE open_out_file ( file_unit , output_prefix, output_date, domain_id )
   
      IMPLICIT NONE
      
      !  Input variables.

      INTEGER , INTENT(IN) :: file_unit , domain_id
      
      !  Local variables

      LOGICAL :: is_it_there = .FALSE.
      CHARACTER(LEN=  2) :: id_string
      CHARACTER(LEN= 20) :: output_prefix
      CHARACTER(LEN=132) :: output_name
      CHARACTER(LEN= 19) :: output_date

      WRITE ( id_string , '(i2.2)' ) domain_id
      output_name = TRIM(output_prefix) // '.d' //id_string // '.' // output_date(1:19)

      OPEN ( FILE   =  TRIM(output_name) , &
             UNIT   =  file_unit         , &
             STATUS = 'UNKNOWN'          , &
             FORM   = 'UNFORMATTED'      , &
             ACCESS = 'SEQUENTIAL'       )
             PRINT '(A)','Opened ', TRIM(output_name), 'file for output.'

   END SUBROUTINE open_out_file

!KWM START
   SUBROUTINE average_surface_T_over_time ( input_file , imx , jmx , avgsfct , end_date)
!
! Time average of the surface air T from the pressure-level input file.  This subroutine
! should be called only if "less_than_24h" is .FALSE.  To keep from introducing a bias
! when the number of time periods do not fit cleanly into days (for example, a data set with
! times 00Z, 12Z and 00Z will have a strong bias for the 00Z conditions if all three times
! go into the average) the data are first read into a holding array, and when that holding 
! array has one day's worth of data, those data are summed and added to the average array.
!
     IMPLICIT NONE
     CHARACTER ( LEN = 132 ) , DIMENSION ( 100 ) , INTENT( in ) :: input_file
     INTEGER , INTENT( in )                                     :: imx
     INTEGER , INTENT( in )                                     :: jmx
     REAL    , DIMENSION ( jmx , imx ) , INTENT ( out )         :: avgsfct
     INTEGER                                                    :: flag
     INTEGER                                                    :: idum
     INTEGER                                                    :: ierr
     INTEGER                                                    :: i
     INTEGER                                                    :: icount
     INTEGER                                                    :: dcount
     INTEGER                                                    :: file_count
     INTEGER                                                    :: idts
     INTEGER                                                    :: intv
     INTEGER                                                    :: last_intv
     REAL                                                       :: xdum
     INTEGER , PARAMETER                                        :: nhold = 48
     INTEGER , PARAMETER                                        :: iunit = 60
     REAL    , DIMENSION ( imx , jmx , nhold )                  :: holdt
     CHARACTER ( len = 19 ) , INTENT ( in )                     :: end_date
     CHARACTER ( len = 24 )                                     :: startdate
     CHARACTER ( len = 24 )                                     :: lastdate
     LOGICAL                                                    :: is_it_there
     REAL    , ALLOCATABLE, DIMENSION ( : , : )                 :: tmp2d, sfctmp

     startdate = "0000-00-00_00:00:00.0000"
     avgsfct = 0.
     ALLOCATE(sfctmp(imx,jmx))

     file_loop : DO file_count = 1, 100

        IF ( input_file ( file_count ) == "" ) THEN
           EXIT file_loop
        END IF

        INQUIRE ( FILE = TRIM ( input_file ( file_count ) ) , EXIST = is_it_there )

        IF ( .NOT. is_it_there ) THEN
           EXIT file_loop
        END IF

        OPEN ( FILE   = TRIM ( input_file ( file_count ) ) , &
             UNIT   =  iunit                            , &
             STATUS = 'OLD'                             , &
             FORM   = 'UNFORMATTED'                     , &
             ACCESS = 'SEQUENTIAL'                      , &
             ACTION = 'READ'                            )

print *, 'opening file = ', input_file ( file_count )
print *, 'imx, jmx, nhold = ',imx, jmx, nhold

        var_loop : DO

           READ ( iunit , iostat=ierr ) flag
           IF ( ierr /= 0 ) EXIT var_loop

           IF ( flag == bh_flag ) THEN

              ! Just skip the big_header info
              READ ( iunit ) idum

           ELSE IF ( flag == sh_flag ) THEN

              READ ( iunit ) small_header%num_dims , small_header%start_dims , small_header%end_dims , &
                   small_header%xtime , small_header%staggering , small_header%ordering , &
                   small_header%current_date , small_header%name , small_header%units , &
                   small_header%description

              IF ( small_header%name == "T" .and. small_header%current_date .LE. end_date ) THEN
                 PRINT * , " Read time " , small_header%current_date ( 1 : 19 )

                 IF ( startdate ( 1 : 4 ) == "0000" ) THEN

                    startdate = small_header%current_date
                    lastdate = startdate
                    holdt = 0.0
                    idts = 0
                    intv = 0
                    icount = 0
                    dcount = 0

                 ELSE

                    last_intv = intv
                    CALL geth_idts ( small_header%current_date ( 1 : 19 ) , startdate ( 1 : 19 ) , idts )
                    CALL geth_idts ( small_header%current_date ( 1 : 19 ) ,  lastdate ( 1 : 19 ) , intv )

                    IF ( ( last_intv > 0 ) .AND. ( intv /= last_intv ) ) THEN

                       PRINT * , " Intervals in input file not consistent "
                       STOP " STOP in subroutine 'average_surface_T_over_time' "

                    END IF

                 END IF

                 icount = icount + 1

                 IF ( icount > nhold ) THEN

                    PRINT * , " ************************************************************************** "
                    PRINT * , " NHOLD exceeded in subroutine 'average_surface_T_over_time': NHOLD = ", NHOLD
                    PRINT * , " This could happen if the input time interval is less than " , &
                         ( 1440 / float ( NHOLD ) ) , " minutes "
                    PRINT * , " or if the input time interval does not fit evenly into days. "
                    PRINT * , " If either of these is the case, try increasing NHOLD "
                    STOP 

                 END IF

                 IF ( ( bhi( 1,1) .EQ. 2 ) .AND. &
                      ( bhi( 8,1) .EQ. 1 ) .AND. &
                      ( bhi(15,1) .EQ. 0 ) ) THEN
                    ALLOCATE(tmp2d(small_header%end_dims(1), small_header%end_dims(2)))
                    READ ( iunit ) tmp2d
                    holdt ( : , : , icount ) = tmp2d ( bhi(11,1) + 1 : bhi(11,1) + imx , bhi(12,1) + 1 : bhi(12,1) + jmx )
                    DEALLOCATE(tmp2d)
                 else
                    READ ( iunit ) holdt( : , : , icount )
                 endif
                 holdt ( : , jmx , icount ) = holdt ( : , jmx - 1 , icount )
                 holdt ( imx , : , icount ) = holdt ( imx - 1 , : , icount )

                 IF ( ( idts > 0 ) .AND. ( MOD ( ( idts + intv ) , 86400 ) == 0 ) ) THEN

                    PRINT * , " Summing data to avgsfct at time " , small_header%current_date ( 1 : 19 )
                    dcount = dcount + icount

                    DO i = 1 , icount
                       sfctmp = sfctmp + holdt( : , : , i )
                    END DO

                    icount = 0
                    holdt = 0.0

                 END IF

                 lastdate = small_header%current_date

              ELSE

                 READ ( iunit ) xdum

              END IF


           ELSE IF ( flag == 2 ) THEN

              ! We've hit the end of time.  Nothing in particular to do.
              IF ( small_header%current_date .GE. end_date ) EXIT var_loop

           ELSE

              PRINT * , " Unrecognized FLAG in subroutine average_surface_T_over_time "
              PRINT * , " FLAG = " , flag
              STOP

           END IF

        END DO var_loop

        CLOSE ( iunit )

     END DO file_loop

     WRITE ( * , ' ( " Using " , i4 , " input T fields in our T average. " ) ' ) dcount

     IF ( icount > 0 ) THEN
        WRITE ( * , ' ( " Skipping " , i4 , " input T fields off the end of our input data set for our T average. " ) ' ) icount
     END IF

     sfctmp = sfctmp / float ( dcount )

!    invert i and j, and output to avgsfct

     CALL invert2d ( avgsfct , imx , jmx , sfctmp )

     DEALLOCATE (sfctmp)

   END SUBROUTINE average_surface_T_over_time
!KWM END

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

!dis   
!dis    Open Source License/Disclaimer, Forecast Systems Laboratory
!dis    NOAA/OAR/FSL, 325 Broadway Boulder, CO 80305
!dis    
!dis    This software is distributed under the Open Source Definition,
!dis    which may be found at http://www.opensource.org/osd.html.
!dis    
!dis    In particular, redistribution and use in source and binary forms,
!dis    with or without modification, are permitted provided that the
!dis    following conditions are met:
!dis    
!dis    - Redistributions of source code must retain this notice, this
!dis    list of conditions and the following disclaimer.
!dis    
!dis    - Redistributions in binary form must provide access to this
!dis    notice, this list of conditions and the following disclaimer, and
!dis    the underlying source code.
!dis    
!dis    - All modifications to this software must be clearly documented,
!dis    and are solely the responsibility of the agent making the
!dis    modifications.
!dis    
!dis    - If significant modifications or enhancements are made to this
!dis    software, the FSL Software Policy Manager
!dis    (softwaremgr@fsl.noaa.gov) should be notified.
!dis    
!dis    THIS SOFTWARE AND ITS DOCUMENTATION ARE IN THE PUBLIC DOMAIN
!dis    AND ARE FURNISHED "AS IS."  THE AUTHORS, THE UNITED STATES
!dis    GOVERNMENT, ITS INSTRUMENTALITIES, OFFICERS, EMPLOYEES, AND
!dis    AGENTS MAKE NO WARRANTY, EXPRESS OR IMPLIED, AS TO THE USEFULNESS
!dis    OF THE SOFTWARE AND DOCUMENTATION FOR ANY PURPOSE.  THEY ASSUME
!dis    NO RESPONSIBILITY (1) FOR THE USE OF THE SOFTWARE AND
!dis    DOCUMENTATION; OR (2) TO PROVIDE TECHNICAL SUPPORT TO USERS.
!dis   
!dis 

  SUBROUTINE proc_make_variable_metadata(domain_id,date,time)

    ! This subroutine populates the variable metadata records
    ! found in module_gridded_data.
    ! 
    ! HISTORY
    ! =======
    ! Nov 2000 - Brent L. Shaw, NOAA/FSL
    !            Original Version
    ! Apr 2001 - Brent L. Shaw, NOAA/FSL
    !            A few mods to support hinterp re-structure and
    !            LSM support
    ! Jul 2003 - Adapted to process MM5 data

    IMPLICIT NONE

    INTEGER, INTENT(IN)    :: domain_id
    INTEGER, INTENT(IN)    :: date  ! YYYYJJJ
    REAL, INTENT(IN)       :: time  ! Seconds since 00UTC of date
    INTEGER :: i, nz
   
    nz = bhi(12,bhi(1,1))

    ! We need to make the metadata record for each of the variables
    ! by running through each of them, first by 3d then by 2d

    ! Set the defaults for the 3d variables and change by exception

    var_up(:)%domain_id         = domain_id
    var_up(:)%ndim              = allp_3d(1)%small_header%num_dims
    !  use actual non-expanded domain dimension, not input data dimesion
    var_up(:)%dim_val(1)        = bhi(17,1)
    var_up(:)%dim_val(2)        = bhi(16,1)
    var_up(:)%dim_val(3)        = allp_3d(1)%small_header%end_dims(3)
    var_up(:)%dim_desc(1)       = 'W-E '
    var_up(:)%dim_desc(2)       = 'S-N '
    var_up(:)%dim_desc(3)       = 'VERT'
    var_up(:)%start_index(1)    = 1
    var_up(:)%start_index(2)    = 1
    var_up(:)%start_index(3)    = 1
    var_up(:)%stop_index(1)     = bhi(17,1)
    var_up(:)%stop_index(2)     = bhi(16,1)
    var_up(:)%stop_index(3)     = allp_3d(1)%small_header%end_dims(3)
    var_up(:)%h_stagger_index   = n_ind ! Non-staggered
    var_up(:)%v_stagger_index   = 1
    var_up(:)%array_order       = '+X+Y+Z  '
    var_up(:)%field_type        = 'REAL'
    var_up(:)%field_source_prog = 'SI      '
    var_up(:)%description       = '                                                                                '
    var_up(:)%source_desc       = '                                                                                '
   
    header_3d : DO i = 1 , nump_3d
        var_up(i)%name              = allp_3d(i)%small_header%name(1:8)
        var_up(i)%units(1:16)       = allp_3d(i)%small_header%units(1:16)
        var_up(i)%description(1:46) = allp_3d(i)%small_header%description(1:46)
        var_up(i)%source_desc(1:3)  = 'MM5'
        var_up(i)%vt_date_start     = date
        var_up(i)%vt_date_stop      = date
        var_up(i)%vt_time_start     = time
        var_up(i)%vt_time_stop      = time
        var_up(i)%field_time_type   = 'INSTANT '
! all 3d fields are expected to be at non-staggered grid: n_ind
!       if (allp_3d(i)%small_header%name(1:1) .EQ. 'U' .or. allp_3d(i)%small_header%name(1:1) .EQ. 'V' ) then
            var_up(i)%h_stagger_index   = n_ind
!       else
!           var_up(i)%h_stagger_index   = t_ind
!       end if
    END DO header_3d

    ! Set the defaults for the 2d variables and change by exception

    var_up(:)%domain_id = domain_id
    var_sfc(:)%ndim              = all_2d(1)%small_header%num_dims
    var_sfc(:)%dim_val(1)        = bhi(17,1)
    var_sfc(:)%dim_val(2)        = bhi(16,1)
    var_sfc(:)%dim_desc(1)       = 'W-E '
    var_sfc(:)%dim_desc(2)       = 'S-N '
    var_sfc(:)%start_index(1)    = 1
    var_sfc(:)%start_index(2)    = 1
    var_sfc(:)%stop_index(1)     = bhi(17,1)
    var_sfc(:)%stop_index(2)     = bhi(16,1)
    var_sfc(:)%h_stagger_index   = t_ind
    var_sfc(:)%v_stagger_index   = 1
    var_sfc(:)%array_order       = '+X+Y    '
    var_sfc(:)%field_type        = 'REAL'
    var_sfc(:)%field_source_prog = 'SI      '
    var_sfc(:)%description       = '                                                                                '
    var_sfc(:)%source_desc       = '                                                                                '

    header_2d : DO i = 1 , num_2d
        var_sfc(i)%name              = all_2d(i)%small_header%name(1:8)
        var_sfc(i)%units(1:16)       = all_2d(i)%small_header%units(1:16)
        var_sfc(i)%description(1:46) = all_2d(i)%small_header%description(1:46)
        var_sfc(i)%source_desc(1:3)  = 'MM5'
        var_sfc(i)%vt_date_start     = date
        var_sfc(i)%vt_date_stop      = date
        var_sfc(i)%vt_time_start     = time
        var_sfc(i)%vt_time_stop      = time
        var_sfc(i)%field_time_type = 'INSTANT '
        IF ( (var_sfc(i)%name(1:8) .EQ. 'LATITDOT').OR.&
             (var_sfc(i)%name(1:8) .EQ. 'LONGIDOT').OR.&
             (var_sfc(i)%name(1:8) .EQ. 'MAPFACDT').OR.&
             (var_sfc(i)%name(1:8) .EQ. 'CORIOLIS')  ) THEN
          ! the above variables are on non-staggered grid (dot-point): n_grid
          var_sfc(i)%h_stagger_index = n_ind
        END IF
    END DO header_2d

  END SUBROUTINE proc_make_variable_metadata

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

!dis   
!dis    Open Source License/Disclaimer, Forecast Systems Laboratory
!dis    NOAA/OAR/FSL, 325 Broadway Boulder, CO 80305
!dis    
!dis    This software is distributed under the Open Source Definition,
!dis    which may be found at http://www.opensource.org/osd.html.
!dis    
!dis    In particular, redistribution and use in source and binary forms,
!dis    with or without modification, are permitted provided that the
!dis    following conditions are met:
!dis    
!dis    - Redistributions of source code must retain this notice, this
!dis    list of conditions and the following disclaimer.
!dis    
!dis    - Redistributions in binary form must provide access to this
!dis    notice, this list of conditions and the following disclaimer, and
!dis    the underlying source code.
!dis    
!dis    - All modifications to this software must be clearly documented,
!dis    and are solely the responsibility of the agent making the
!dis    modifications.
!dis    
!dis    - If significant modifications or enhancements are made to this
!dis    software, the FSL Software Policy Manager
!dis    (softwaremgr@fsl.noaa.gov) should be notified.
!dis    
!dis    THIS SOFTWARE AND ITS DOCUMENTATION ARE IN THE PUBLIC DOMAIN
!dis    AND ARE FURNISHED "AS IS."  THE AUTHORS, THE UNITED STATES
!dis    GOVERNMENT, ITS INSTRUMENTALITIES, OFFICERS, EMPLOYEES, AND
!dis    AGENTS MAKE NO WARRANTY, EXPRESS OR IMPLIED, AS TO THE USEFULNESS
!dis    OF THE SOFTWARE AND DOCUMENTATION FOR ANY PURPOSE.  THEY ASSUME
!dis    NO RESPONSIBILITY (1) FOR THE USE OF THE SOFTWARE AND
!dis    DOCUMENTATION; OR (2) TO PROVIDE TECHNICAL SUPPORT TO USERS.
!dis   
!dis 

SUBROUTINE proc_output_variable(time, output_unit)

    ! This subroutine writes out the variables to output file
    !
    ! HISTORY
    ! =======
    !            
    ! Jul 2003 - Adapted from WRF SI to process MM5 data

  IMPLICIT NONE
  CHARACTER(LEN=19),INTENT(IN) :: time
  CHARACTER(LEN=200) :: outfile_name
  CHARACTER (LEN=2) :: domain_id_string
  INTEGER  :: i,nz, output_file, output_unit, file_index, open_status
  TYPE(wrfvar_metadata) :: var_dummy
  REAL :: current_min_top = 0
  LOGICAL :: have_soilhgt
  LOGICAL :: keep_sfc
  REAL, ALLOCATABLE :: diffarr(:,:)
  INTEGER :: zstart, zend
  INTEGER :: loop
  INTEGER , DIMENSION(:) , ALLOCATABLE :: output_levels

  nz = bhi(12,bhi(1,1))

  !  If the data are expanded, we need to chop down the size on the assignment for
  !  the horizontal dimensions.

  ! Fill output_levels array with 1-D pressure array

  ALLOCATE ( output_levels (nz) )

  DO loop = 1 , num_1d
     IF ( all_1d(loop)%small_header%name(1:8) .EQ. 'PRESSURE' ) THEN
          output_levels = all_1d(loop)%array
     END IF
  END DO
  output_levels (1) = 200100

  have_soilhgt = .false. 

  ! If this is isobaric data, we need to make sure real data
  ! is in the 2001 mb level, which is always present by default
  ! We will check the critical values for interpolation...namely
  ! T, RH, and HGT.  If any of those has the 1000mb data replicated
  ! then we will throw away the first level for all variables.

  IF (MAXVAL(output_levels) .GT. 100000)THEN
    keep_sfc = .TRUE.
    ALLOCATE(diffarr(dom_meta%xdim,dom_meta%ydim))
    check_critical: DO i = 1, nump_3d
      IF ((var_up(i)%name(1:8) .EQ. 'T       ').OR. &
          (var_up(i)%name(1:8) .EQ. 'HGT     ').OR. &
          (var_up(i)%name(1:8) .EQ. 'RH      ').OR. &
          (var_up(i)%name(1:8) .EQ. 'U       ').OR. &
          (var_up(i)%name(1:8) .EQ. 'V       ')) THEN
        diffarr = allp_3d(i)%array(:,:,2) - allp_3d(i)%array(:,:,1)
        IF ( ( NINT(MAXVAL(diffarr)) .EQ. 0 ) .AND. &
           ( NINT(MINVAL(diffarr)) .EQ. 0) ) THEN
          ! 1st (2001) and second levels are same, so
          ! sfc data is invalid
          keep_sfc = .FALSE.
        ENDIF
        IF (.NOT. keep_sfc) EXIT check_critical
      ENDIF
    ENDDO check_critical
    DEALLOCATE(diffarr)
    zend = dom_meta%zdim
    IF (.NOT. keep_sfc) THEN
      print *, 'ISOBARIC DATA: SFC T, HGT, RH, U, or V missing,'
      print *, '  IGNORING ALL SFC VALUES IN 3D ARRAYS'
      dom_meta%zdim = dom_meta%zdim - 1
      zstart = 2
    ELSE
      zstart = 1
    ENDIF
    print *, 'ISOBARIC OUTPUT LEVELS:', output_levels(zstart:zend)
  ENDIF

  ! Put the domain metadata as the first record of this file.  There is a 
  ! different file for each time period.  

  PRINT '(A)', 'Writing domain metadata...'
  WRITE(output_unit) dom_meta%id, dom_meta%parent_id, &
    dom_meta%dyn_init_src, dom_meta%static_init_src, &
    dom_meta%vt_date, dom_meta%vt_time, dom_meta%origin_parent_x, &
    dom_meta%origin_parent_y, dom_meta%ratio_to_parent, &
    dom_meta%delta_x, dom_meta%delta_y, dom_meta%top_level, &
    dom_meta%origin_parent_z, dom_meta%corner_lats, dom_meta%corner_lons, &
    dom_meta%xdim, dom_meta%ydim, dom_meta%zdim

  ! If this is isobaric data, then output the pressure levels
  ! used as a 1D array.
  IF (MAXVAL(output_levels).GT.100000) THEN  

    PRINT '(A)', 'Writing output levels...'
    print *, 'pressure_id = ', pressure_id
    var_dummy%name = pressure_id
    var_dummy%units = 'Pa              '
    var_dummy%description = 'Pressure levels used for vertical coordinate ' // &
                       '(200100 = sfc)'
    var_dummy%dim_desc(1) = 'PRES'
    var_dummy%source_desc ='User defined pressure levels'
    var_dummy%domain_id = dom_meta%id
    var_dummy%ndim = 1
    var_dummy%dim_val(1) = dom_meta%zdim
    var_dummy%dim_val(2) = 0
    var_dummy%dim_val(3) = 0
    var_dummy%start_index(1) = 1
    var_dummy%stop_index(1) = dom_meta%zdim
    var_dummy%h_stagger_index = 0
    var_dummy%v_stagger_index = 1             
    var_dummy%array_order = '-P      '
    var_dummy%field_type = 'REAL'
    var_dummy%field_source_prog = 'SI  '
    var_dummy%field_time_type = 'CONSTANT'
    var_dummy%vt_date_start = dom_meta%vt_date
    var_dummy%vt_time_start = dom_meta%vt_time
    var_dummy%vt_date_stop = dom_meta%vt_date
    var_dummy%vt_time_stop = dom_meta%vt_time

    ! Output the levels.

    WRITE (output_unit) var_dummy%name, var_dummy%units, var_dummy%description, &
     var_dummy%domain_id, var_dummy%ndim, var_dummy%dim_val, var_dummy%dim_desc, &
     var_dummy%start_index, var_dummy%stop_index, var_dummy%h_stagger_index, &
     var_dummy%v_stagger_index, var_dummy%array_order, var_dummy%field_type, &
     var_dummy%field_source_prog, var_dummy%source_desc, var_dummy%field_time_type, &
     var_dummy%vt_date_start, var_dummy%vt_time_start, var_dummy%vt_date_stop, &
     var_dummy%vt_time_stop

    WRITE (output_unit) REAL(output_levels(zstart:zend))
 
  ENDIF
  ! Now loop through 3-d and 2-d variables and output their metadata followed
  ! by the arrays

  PRINT '(A)', 'Writing 3-d variables...'
  loop_3d: DO i = 1, nump_3d
    var_up(i)%dim_val(3) = dom_meta%zdim
    var_up(i)%stop_index(3) = dom_meta%zdim 
    WRITE (output_unit) var_up(i)%name, var_up(i)%units, &
      var_up(i)%description, var_up(i)%domain_id, var_up(i)%ndim, &
      var_up(i)%dim_val, var_up(i)%dim_desc, var_up(i)%start_index, &
      var_up(i)%stop_index, var_up(i)%h_stagger_index, &
      var_up(i)%v_stagger_index, var_up(i)%array_order, &
      var_up(i)%field_type, var_up(i)%field_source_prog, &
      var_up(i)%source_desc, var_up(i)%field_time_type, &
      var_up(i)%vt_date_start, var_up(i)%vt_time_start, &
      var_up(i)%vt_date_stop, var_up(i)%vt_time_stop
  
!   dum3d = all_3d(:,:,:,i)
    WRITE (output_unit) allp_3d(i)%array(:,:,zstart:zend)
    PRINT '(A,A)', 'Wrote ', var_up(i)%name
 
    ! Check to see if this is GPH so we can determine the minimum height of the top
    ! level to help the user in setting model top in the zeta coordinate system.  

!  IF (var_up(i)%name(1:8).EQ. height_id) THEN
!    current_min_top = MINVAL(dum3d(:,:,nz))
!    !IF (verbose) THEN 
!      PRINT '(A,F10.2)', 'Minimum height of top level for this time = ',current_min_top
!    !END IF
!    IF (current_min_top .LT. min_top_height) THEN
!        min_top_height = current_min_top
!        PRINT '(A,F10.2)', 'New min top = ', min_top_height
!    END IF
!  ENDIF
  END DO loop_3d

  PRINT '(A)', 'Writing 2-d variables...'
  loop_2d: DO i = 1, num_2d
    IF  (var_sfc(i)%name .EQ. 'LANDSEA ') THEN
      ! We do not want to write out the source models land mask
      CYCLE loop_2d
    ENDIF
    WRITE (output_unit) var_sfc(i)%name, var_sfc(i)%units, &
    var_sfc(i)%description, var_sfc(i)%domain_id, var_sfc(i)%ndim, &
    var_sfc(i)%dim_val, var_sfc(i)%dim_desc, var_sfc(i)%start_index, &
    var_sfc(i)%stop_index, var_sfc(i)%h_stagger_index, &
    var_sfc(i)%v_stagger_index, var_sfc(i)%array_order, &
    var_sfc(i)%field_type, var_sfc(i)%field_source_prog, &
    var_sfc(i)%source_desc, var_sfc(i)%field_time_type, &
    var_sfc(i)%vt_date_start, var_sfc(i)%vt_time_start, &
    var_sfc(i)%vt_date_stop, var_sfc(i)%vt_time_stop 
    !dum2d = all_2d(:,:,i)
    WRITE (output_unit) all_2d(i)%array
    PRINT '(A,A)', 'Wrote ', var_sfc(i)%name
!   if ( i .ge. 14 ) print *, all_2d(i)%array(20,15)
    IF (var_sfc(i)%name .EQ. 'SOILHGT ') have_soilhgt = .true.
  END DO loop_2d

  ! If we did not have a soil height field explicitly read in, then output
  ! the one we generated

! IF ( (.NOT.have_soilhgt).AND.(ALLOCATED(soilhgt_bg))) THEN
!   IF (MAXVAL(soilhgt_bg) .GT. -9999.) THEN
!     var_dummy%name = soilhgt_id
!     var_dummy%units = 'm      '
!     var_dummy%description = 'Terrain height of source data        '
!     var_dummy%domain_id = dom_meta%id
!     var_dummy%ndim=2
!     var_dummy%dim_val(1) = dom_meta%xdim
!     var_dummy%dim_val(2) = dom_meta%ydim
!     var_dummy%dim_val(3:var_maxdims) = 1
!     var_dummy%dim_desc(1) = 'E-W'
!     var_dummy%dim_desc(2) = 'N-S'
!     var_dummy%dim_desc(3:var_maxdims) = '    '
!     var_dummy%start_index(1) = 1
!     var_dummy%start_index(2) = 1
!     var_dummy%start_index(3:var_maxdims) = 0
!     var_dummy%stop_index(1) = dom_meta%xdim
!     var_dummy%stop_index(2) = dom_meta%ydim
!     var_dummy%stop_index(3:var_maxdims) = 0
!     var_dummy%v_stagger_index =  0
!     var_dummy%array_order = '+X+Y  '
!     var_dummy%field_type = 'REAL'
!     var_dummy%field_source_prog = 'SI'
!     var_dummy%source_desc = 'Horizon. interpolation by SI'
!     var_dummy%field_time_type = 'CONSTANT'
!     var_dummy%vt_date_start = dom_meta%vt_date
!     var_dummy%vt_time_start = dom_meta%vt_time
!     var_dummy%vt_date_stop = dom_meta%vt_date
!     var_dummy%vt_time_stop = dom_meta%vt_time

!     WRITE(output_unit) var_dummy
!     WRITE(output_unit) soilhgt_bg
!     DEALLOCATE(soilhgt_bg)
!   ENDIF
! ENDIF
  !CLOSE (output_file)
  PRINT '(A)', 'Done with variable output.'
END subroutine proc_output_variable

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

SUBROUTINE proc_store_global_metadata (out_unit, output_prefix)

  IMPLICIT NONE
  INTEGER  :: out_unit 
  CHARACTER (LEN=200) :: metafile
  CHARACTER (LEN= 20) :: output_prefix

  metafile = TRIM(output_prefix) // '.global.metadata'

  OPEN ( FILE   =  TRIM(metafile)    , &
         UNIT   =  out_unit          , &
         STATUS = 'UNKNOWN'          , &
         FORM   = 'UNFORMATTED'      , &
         ACCESS = 'SEQUENTIAL'       )
  PRINT '(A)','Opened ', TRIM(metafile), 'file for writing.'

  ! Write out the global metadata.  We cannot just write the 
  ! struture, because the byte-swap conversion will not be done 
  ! properly if you are on a little-endian machine.  So, write
  ! each element of the structure

  WRITE (out_unit)    global_meta%simulation_name, &
                      global_meta%user_desc, &
                      global_meta%si_version,  &
                      global_meta%anal_version, &
                      global_meta%wrf_version, &
                      global_meta%post_version, &
                      global_meta%map_projection, &
                      global_meta%moad_known_lat, &
                      global_meta%moad_known_lon, &
                      global_meta%moad_known_loc, &
                      global_meta%moad_stand_lats, &
                      global_meta%moad_stand_lons, &
                      global_meta%moad_delta_x, &
                      global_meta%moad_delta_y, &
                      global_meta%horiz_stagger_type, &
                      global_meta%num_stagger_xy, &
                      global_meta%stagger_dir_x, &
                      global_meta%stagger_dir_y, &
                      global_meta%num_stagger_z, &
                      global_meta%stagger_dir_z, &
                      global_meta%vertical_coord, &
                      global_meta%num_domains, &
                      global_meta%init_date, &
                      global_meta%init_time, &
                      global_meta%end_date, &
                      global_meta%end_time, &
                      global_meta%lu_source, & 
                      global_meta%lu_water, &
                      global_meta%lu_ice, & 
                      global_meta%st_water
  CLOSE (out_unit)

END SUBROUTINE proc_store_global_metadata

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   SUBROUTINE prepare_data ( avgsfct , wrth2o , less_than_24h , psfc_method , julday )

      !  We need this routine to do a few things before writing the data in SI format
      !   - cut down from expanded domain (do it here or in writing)
      !   - reverse I and J 
      !   - change variable name in pointer, small_header
      !   - may need to get a few more variables (such as sfcp)

      IMPLICIT NONE

      !  Input variables.

      REAL , DIMENSION(:,:)   :: avgsfct
      LOGICAL, INTENT ( IN )  :: wrth2o, less_than_24h
      INTEGER, INTENT ( IN )  :: psfc_method , julday

      !  Local variables.

      REAL , ALLOCATABLE , DIMENSION(:,:,:) :: t, u, v, height, rh, q
      REAL , ALLOCATABLE , DIMENSION(:,:,:) :: scr3d1, scr3d2

      REAL , ALLOCATABLE , DIMENSION(:,:)   :: scr2d1, scr2d2

      REAL , ALLOCATABLE , DIMENSION(:,:,:)   :: vegfmo

      REAL , ALLOCATABLE , DIMENSION(:,:)   :: terrain, mapfaccr, mapfac_u, mapfac_v, &
                                               proj_angel, greenmax, greenmin, &
                                               albedo, greenfac, &
                                               psealvlc, sfc_p

      REAL , ALLOCATABLE , DIMENSION(:)     :: p

      INTEGER :: kx, imon

      !  Local variables.

      INTEGER :: loop , months
      INTEGER :: is , js , ie , je, imx, jmx, imxm, jmxm
      INTEGER :: num2d_new, i, j, k

      LOGICAL :: fndsoilhgt , fndtopsoil , fndskintemp , fndvegfrc

      REAL    :: omega , deg_to_rad , conv_fac
 
      omega = 7.292e-5
      deg_to_rad = 57.29578
      conv_fac   = 1./deg_to_rad

      print *, 'Num of 3D, 2D, 1D vars : ', nump_3d, num_2d, num_1d
      imx  = bhi(16,1)
      jmx  = bhi(17,1)
      imxm = imx - 1
      jmxm = jmx - 1
      kx   = bhi(12,bhi(1,1))
     
      !  If the data are expanded, we need to chop down the size on the assignment for
      !  the horizontal dimensions.

      IF ( ( bhi( 1,1) .EQ. 2 ) .AND. &
           ( bhi( 8,1) .EQ. 1 ) .AND. &
           ( bhi(15,1) .EQ. 0 ) ) THEN
         is = 1         + bhi(11,1)
         js = 1         + bhi(12,1)
         ie = bhi(16,1) + bhi(11,1)
         je = bhi(17,1) + bhi(12,1)
      ELSE
         is = 1
         js = 1
         ie = bhi(16,1)
         je = bhi(17,1)
      END IF

      fndsoilhgt = .false.
      fndtopsoil = .false.
      fndskintemp= .false.
      fndvegfrc  = .false.

      !  Assign the 3d data.

      ALLOCATE ( Q           ( JMX , IMX , KX ) )
      ALLOCATE ( HEIGHT      ( JMX , IMX , KX ) )
      ALLOCATE ( RH          ( JMX , IMX , KX ) )
      ALLOCATE ( T           ( JMX , IMX , KX ) )
      ALLOCATE ( U           ( JMX , IMX , KX ) )
      ALLOCATE ( V           ( JMX , IMX , KX ) )
      ALLOCATE ( SCR3D1      ( IMX , JMX , KX ) )
      ALLOCATE ( SCR3D2      ( IMX , JMX , KX ) )
      ALLOCATE ( P           ( KX ) )

      DO loop = 1 , nump_3d
         IF      ( allp_3d(loop)%small_header%name(1:8) .EQ. 'T       ' ) THEN
            scr3d1 = allp_3d(loop)%array(is:ie,js:je,:)
            CALL invert ( t      , bhi(16,1) , bhi(17,1) , kx , scr3d1 )
            deallocate ( allp_3d(loop)%array )
            allocate ( allp_3d(loop)%array(jmx,imx,kx) )
            allp_3d(loop)%array = t
            index_t = loop
         ELSE IF ( allp_3d(loop)%small_header%name(1:8) .EQ. 'U       ' ) THEN
            scr3d1 = allp_3d(loop)%array(is:ie,js:je,:)
            CALL invert ( u      , bhi(16,1) , bhi(17,1) , kx , scr3d1 )
            deallocate ( allp_3d(loop)%array )
            allocate ( allp_3d(loop)%array(jmx,imx,kx) )
            allp_3d(loop)%array = u
            index_u = loop
         ELSE IF ( allp_3d(loop)%small_header%name(1:8) .EQ. 'V       ' ) THEN
            scr3d1 = allp_3d(loop)%array(is:ie,js:je,:) 
            CALL invert ( v      , bhi(16,1) , bhi(17,1) , kx , scr3d1 )
            deallocate ( allp_3d(loop)%array )
            allocate ( allp_3d(loop)%array(jmx,imx,kx) )
            allp_3d(loop)%array = v
            index_v = loop
         ELSE IF ( allp_3d(loop)%small_header%name(1:8) .EQ. 'H       ' ) THEN
            scr3d1 = allp_3d(loop)%array(is:ie,js:je,:)
            CALL invert ( height , bhi(16,1) , bhi(17,1) , kx , scr3d1 )
            deallocate ( allp_3d(loop)%array )
            allocate ( allp_3d(loop)%array(jmx,imx,kx) )
            allp_3d(loop)%array = height
            allp_3d(loop)%small_header%name(1:8) = height_id
            index_height = loop
         ELSE IF ( allp_3d(loop)%small_header%name(1:8) .EQ. 'RH      ' ) THEN
            scr3d1 = allp_3d(loop)%array(is:ie,js:je,:)
            CALL invert ( rh     , bhi(16,1) , bhi(17,1) , kx , scr3d1 )
            deallocate ( allp_3d(loop)%array )
            allocate ( allp_3d(loop)%array(jmx,imx,kx) )
            allp_3d(loop)%array = rh
            index_rh = loop
         ELSE IF ( allp_3d(loop)%small_header%name(1:8) .EQ. 'CLW     ' ) THEN
            scr3d1 = allp_3d(loop)%array(is:ie,js:je,:)
            CALL invert ( scr3d2 , bhi(16,1) , bhi(17,1) , kx , scr3d1 )
            deallocate ( allp_3d(loop)%array )
            allocate ( allp_3d(loop)%array(jmx,imx,kx) )
            allp_3d(loop)%array = scr3d2
            allp_3d(loop)%small_header%name(1:8) = qliquid_id
         ELSE IF ( allp_3d(loop)%small_header%name(1:8) .EQ. 'RNW     ' ) THEN
            scr3d1 = allp_3d(loop)%array(is:ie,js:je,:)
            CALL invert ( scr3d2 , bhi(16,1) , bhi(17,1) , kx , scr3d1 )
            deallocate ( allp_3d(loop)%array )
            allocate ( allp_3d(loop)%array(jmx,imx,kx) )
            allp_3d(loop)%array = scr3d2
            allp_3d(loop)%small_header%name(1:8) = qrain_id
         ELSE IF ( allp_3d(loop)%small_header%name(1:8) .EQ. 'ICE     ' ) THEN
            scr3d1 = allp_3d(loop)%array(is:ie,js:je,:)
            CALL invert ( scr3d2 , bhi(16,1) , bhi(17,1) , kx , scr3d1 )
            deallocate ( allp_3d(loop)%array )
            allocate ( allp_3d(loop)%array(jmx,imx,kx) )
            allp_3d(loop)%array = scr3d2
            allp_3d(loop)%small_header%name(1:8) = qice_id
         ELSE IF ( allp_3d(loop)%small_header%name(1:8) .EQ. 'SNOW    ' ) THEN
            scr3d1 = allp_3d(loop)%array(is:ie,js:je,:)
            CALL invert ( scr3d2 , bhi(16,1) , bhi(17,1) , kx , scr3d1 )
            deallocate ( allp_3d(loop)%array )
            allocate ( allp_3d(loop)%array(jmx,imx,kx) )
            allp_3d(loop)%array = scr3d2
            allp_3d(loop)%small_header%name(1:8) = qsnow_id
         ELSE IF ( allp_3d(loop)%small_header%name(1:8) .EQ. 'GRAUPEL ' ) THEN
            scr3d1 = allp_3d(loop)%array(is:ie,js:je,:)
            CALL invert ( scr3d2 , bhi(16,1) , bhi(17,1) , kx , scr3d1 )
            deallocate ( allp_3d(loop)%array )
            allocate ( allp_3d(loop)%array(jmx,imx,kx) )
            allp_3d(loop)%array = scr3d2
            allp_3d(loop)%small_header%name(1:8) = qgraupel_id
         ELSE IF ( allp_3d(loop)%small_header%name(1:8) .EQ. 'NCI     ' ) THEN
            scr3d1 = allp_3d(loop)%array(is:ie,js:je,:)
            CALL invert ( scr3d2 , bhi(16,1) , bhi(17,1) , kx , scr3d1 )
            deallocate ( allp_3d(loop)%array )
            allocate ( allp_3d(loop)%array(jmx,imx,kx) )
            allp_3d(loop)%array = scr3d2
            allp_3d(loop)%small_header%name(1:8) = qnci_id
         ELSE
            PRINT '(A,A,A)','Unnecessary 3d array: ',allp_3d(loop)%small_header%name(1:8),'.'
         END IF
      END DO
      DEALLOCATE ( SCR3D1 )
      DEALLOCATE ( SCR3D2 )

      !  Assign the 2d data.

      ALLOCATE ( scr2d1 ( IMX, JMX ) )
      ALLOCATE ( scr2d2 ( JMX, IMX ) )
      ALLOCATE ( terrain ( JMX, IMX ) )
      ALLOCATE ( psealvlc ( JMX, IMX ) )

      num2d_new = 0

      !  First switch I and J for all 2D arrays

      DO loop = 1 , num_2d
         scr2d1  = all_2d(loop)%array(is:ie,js:je)
         CALL invert2d ( scr2d2 , bhi(16,1) , bhi(17,1) , scr2d1 )
         deallocate ( all_2d(loop)%array )
         allocate ( all_2d(loop)%array(jmx,imx) )
         all_2d(loop)%array = scr2d2
      END DO

      DEALLOCATE ( scr2d1 )

      !  Next, change variable names, and check for required variables for WRF

      DO loop = 1 , num_2d
         IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'TERRAIN ' ) THEN
            all_2d(loop)%small_header%name(1:8) = terrain_id
            index_terrain = loop
            terrain = all_2d(loop)%array
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'LAND USE' ) THEN
            all_2d(loop)%small_header%name(1:8) = landusec_id          ! dominent categories
            index_land_use = loop
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'SOILINDX' ) THEN
!           all_2d(loop)%small_header%name(1:8) = 'SOILCTOP'           ! dominent categories
            all_2d(loop)%small_header%name(1:8) = soilcat_id           ! dominent categories
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'MAPFACCR' ) THEN
            all_2d(loop)%small_header%name(1:8) = map_factor_m_id
            index_mapfaccr = loop
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'MAPFACDT' ) THEN
            all_2d(loop)%small_header%name(1:8) = map_factor_id
            index_mapfacdt = loop
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'CORIOLIS' ) THEN
            all_2d(loop)%small_header%name(1:8) = hcoriolis_id
            index_coriolis = loop
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'LATITCRS' ) THEN
            all_2d(loop)%small_header%name(1:8) = latitudem_id
            index_latitcrs = loop
            num2d_new = num2d_new + 1
            scr2d2 = all_2d(loop)%array
            dom_meta%corner_lats(1,1) = scr2d2(1,1) 
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'LONGICRS' ) THEN
            all_2d(loop)%small_header%name(1:8) = longitudem_id
            index_longicrs = loop
            num2d_new = num2d_new + 1
            scr2d2 = all_2d(loop)%array
            dom_meta%corner_lons(1,1) = scr2d2(1,1)
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'LATITDOT' ) THEN
            all_2d(loop)%small_header%name(1:8) = latitude_id
            index_latitdot = loop
            num2d_new = num2d_new + 1
            scr2d2 = all_2d(loop)%array
            dom_meta%corner_lats(1,2) = scr2d2(1,1)
            dom_meta%corner_lats(1,3) = scr2d2(1,1)
            dom_meta%corner_lats(1,4) = scr2d2(1,1)
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'LONGIDOT' ) THEN
            all_2d(loop)%small_header%name(1:8) = longitude_id
            index_longidot = loop
            num2d_new = num2d_new + 1
            scr2d2 = all_2d(loop)%array
            dom_meta%corner_lons(1,2) = scr2d2(1,1)
            dom_meta%corner_lons(1,3) = scr2d2(1,1)
            dom_meta%corner_lons(1,4) = scr2d2(1,1)
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'PSEALVLD' ) THEN
            all_2d(loop)%small_header%name(1:8) = press_msl_id
            index_psealvld = loop
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'PSEALVLC' ) THEN
            all_2d(loop)%small_header%name(1:8) = press_mslm_id
            index_psealvlc = loop
            num2d_new = num2d_new + 1
            psealvlc = all_2d(loop)%array
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'TSEASFC ' ) THEN
            all_2d(loop)%small_header%name(1:8) = sst_id
            index_tseasfc = loop
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'TEMPGRD ' ) THEN
            all_2d(loop)%small_header%name(1:8) = 'TMN     '
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'WEASD   ' ) THEN
            all_2d(loop)%small_header%name(1:8) = 'SNOW    '
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'SOILHGT ' ) THEN
            !  if found, this is pre-3.4 data
            all_2d(loop)%small_header%name(1:8) = 'SOILHGT '
            fndsoilhgt = .true.
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'ALBSNOMX' ) THEN
            all_2d(loop)%small_header%name(1:8) = 'ALBEDOMX'
            num2d_new = num2d_new + 1
! need to add green fraction and monthly albedo data later
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'SOILM010' ) THEN
            all_2d(loop)%small_header%name(1:8) = soilm0_10_id
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'SOILM040' ) THEN
            all_2d(loop)%small_header%name(1:8) = soilm10_40_id
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'SOILM100' ) THEN
            all_2d(loop)%small_header%name(1:8) = soilm40_100_id
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'SOILM200' ) THEN
            all_2d(loop)%small_header%name(1:8) = soilm100_200_id
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'SOILM200' ) THEN
! need a way to identify this field
            all_2d(loop)%small_header%name(1:8) = soilm10_200_id
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'SOILT010' ) THEN
            all_2d(loop)%small_header%name(1:8) = soilt0_10_id
            num2d_new = num2d_new + 1
            fndtopsoil = .true.
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'SOILT040' ) THEN
            all_2d(loop)%small_header%name(1:8) = soilt10_40_id
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'SOILT100' ) THEN
            all_2d(loop)%small_header%name(1:8) = soilt40_100_id
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'SOILT200' ) THEN
            all_2d(loop)%small_header%name(1:8) = soilt100_200_id
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'SOILT200' ) THEN
! need a way to identify this field
            all_2d(loop)%small_header%name(1:8) = soilt10_200_id
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:8) .EQ. 'SKINTEMP' ) THEN
            all_2d(loop)%small_header%name(1:8) = tskin_id
            fndskintemp = .true.
            num2d_new = num2d_new + 1
         ELSE IF ( all_2d(loop)%small_header%name(1:6) .EQ. 'VEGFRC' ) THEN
            read(all_2d(loop)%small_header%name(7:8),'(I2)') imon
            IF ( imon .eq. 1) THEN
                 allocate ( vegfmo(jmx, imx, 12) )
                 num2d_new = num2d_new + 1
!                index_vegfr = loop
            END IF
            fndvegfrc = .true.
            vegfmo(:,:,imon) = all_2d(loop)%array
            PRINT '(A,I2)', 'Found and read 2d VEGFRC array for month ', imon
         ELSE
            PRINT '(A,A,A)','Found extra 2d array : ',TRIM( all_2d(loop)%small_header%name(1:8) ),'.'
         END IF
      END DO

      DEALLOCATE ( SCR2D2 )
      print *, 'These many 2 D fields names have been changed : ', num2d_new, num_2d

      !  Assign the 1d data.

      DO loop = 1 , num_1d
         IF      ( all_1d(loop)%small_header%name(1:8) .EQ. 'PRESSURE' ) THEN
            p      = all_1d(loop)%array
            index_p = loop
         ELSE
            PRINT '(A,A,A)','Unnecessary 1d array: ',all_1d(loop)%small_header%name(1:8),'.'
         END IF
      END DO

      !  Add a couple 2D fields here

      !  sfc pressure. Note that all arrays are bottom_up and has surface data at bottom level

      CALL mxratprs (rh, t, p, jmxm, imxm, kx, WRTH2O , q)
      print *, 'RH, T, Q, P : ', rh(20,10,2), t(20,10,2), q(20,10,2), p(2) 

      !  Fill last row and column.

      q(:,imx,:) = q(:,imxm,:)
      q(jmx,:,:) = q(jmxm,:,:)

      !do k = 2, kx
      !do i = 1, imx
      !do j = 1, jmx
      !   if (q(j,i,k) .lt. 0.) print *, 'neg q at j,i,k ',j,i,k,q(j,i,k)
      !end do
      !end do
      !end do

      PRINT '(A)','Mixing ratio computed on pressure.'

      !  Compute the surface pressure.

      ALLOCATE ( sfc_p ( JMX, IMX ) )
      CALL sfcprs ( t, q, height, psealvlc, terrain, avgsfct, p, jmx, imx, kx, sfc_p , psfc_method , &
                    less_than_24h )

      PRINT '(A)','Surface pressure field computed.'

      !  Increment num_2d

      num_2d = num_2d + 1
      print *, 'num_2d is now ', num_2d
      all_2d(num_2d)%small_header = all_2d(index_psealvlc)%small_header
      all_2d(num_2d)%small_header%end_dims(1) = jmx
      all_2d(num_2d)%small_header%end_dims(2) = imx
      all_2d(num_2d)%small_header%name(1:8) = press_sfc_id
      all_2d(num_2d)%small_header%description(1:46) = 'SURFACE PRESSURE ESTIMATED IN INTERPF         '
      ALLOCATE ( all_2d(num_2d)%array( jmx, imx ) )
      all_2d(num_2d)%array = sfc_p
      print *, 'SFC_P : ', sfc_p(20,15), all_2d(num_2d)%array(20,15)

      if ( .not. fndsoilhgt ) then

      num_2d = num_2d + 1
      print *, 'num_2d is now ', num_2d
      all_2d(num_2d)%small_header = all_2d(index_terrain)%small_header
      all_2d(num_2d)%small_header%end_dims(1) = jmx
      all_2d(num_2d)%small_header%end_dims(2) = imx
      all_2d(num_2d)%small_header%name(1:8) = soilhgt_id
      all_2d(num_2d)%small_header%description(1:46) = 'TERRAIN HEIGHT FROM TERRESTRIAL INPUT         '
      ALLOCATE ( all_2d(num_2d)%array( jmx, imx ) )
      all_2d(num_2d)%array = terrain

      end if

      if ( .not. fndtopsoil ) then

      num_2d = num_2d + 1
      print *, 'num_2d is now ', num_2d
      all_2d(num_2d)%small_header = all_2d(index_terrain)%small_header
      all_2d(num_2d)%small_header%end_dims(1) = jmx
      all_2d(num_2d)%small_header%end_dims(2) = imx
      all_2d(num_2d)%small_header%name(1:8) = soilt0_10_id
      all_2d(num_2d)%small_header%units(1:25) = 'K                        '
      all_2d(num_2d)%small_header%description(1:46) = 'GROUND TEMPERATURE FROM AVERAGED SFC T        '
      ALLOCATE ( all_2d(num_2d)%array( jmx, imx ) )
      all_2d(num_2d)%array = avgsfct

      end if

      if ( .not. fndskintemp ) then

      num_2d = num_2d + 1
      print *, 'num_2d is now ', num_2d
      all_2d(num_2d)%small_header = all_2d(index_terrain)%small_header
      all_2d(num_2d)%small_header%end_dims(1) = jmx
      all_2d(num_2d)%small_header%end_dims(2) = imx
      all_2d(num_2d)%small_header%name(1:8) = tskin_id
      all_2d(num_2d)%small_header%units(1:25) = 'K                        '
      all_2d(num_2d)%small_header%description(1:46) = 'SKIN TEMPERATURE FROM SURFACE T               '
      ALLOCATE ( all_2d(num_2d)%array( jmx, imx ) )
      all_2d(num_2d)%array = t(:,:,1)

      end if

      !  Vertical Coriolis parameter on cross point 

      num_2d = num_2d + 1
      print *, 'num_2d is now ', num_2d
      all_2d(num_2d)%small_header = all_2d(index_coriolis)%small_header
      all_2d(num_2d)%small_header%end_dims(1) = jmx
      all_2d(num_2d)%small_header%end_dims(2) = imx
      all_2d(num_2d)%small_header%name(1:8) = vcoriolis_id
      all_2d(num_2d)%small_header%description(1:46) = 'Vertical Component of Coriolis on Mass Grid   '
      ALLOCATE ( all_2d(num_2d)%array( jmx, imx ) )
      all_2d(num_2d)%array = 2. * omega * cos( all_2d(index_latitcrs)%array * conv_fac )

      ALLOCATE ( proj_angel ( jmx , imx ) )
      ALLOCATE ( mapfac_u   ( jmx , imx ) )
      ALLOCATE ( mapfac_v   ( jmx , imx ) )

      CALL get_new_fields ( proj_angel , mapfac_u , mapfac_v , &
                            all_2d(index_latitdot)%array , all_2d(index_longidot)%array , &
                            jmx , imx )

      !  sinalpha

      num_2d = num_2d + 1
      print *, 'num_2d is now ', num_2d
      all_2d(num_2d)%small_header = all_2d(index_coriolis)%small_header
      all_2d(num_2d)%small_header%end_dims(1) = jmx
      all_2d(num_2d)%small_header%end_dims(2) = imx
      all_2d(num_2d)%small_header%name(1:8) = sinalpha_id
      all_2d(num_2d)%small_header%description(1:46) = 'Sine of angel true north and Y on Mass Grid   '
      ALLOCATE ( all_2d(num_2d)%array( jmx, imx ) )
      all_2d(num_2d)%array = sin( proj_angel * conv_fac )

      !  cosalpha

      num_2d = num_2d + 1
      print *, 'num_2d is now ', num_2d
      all_2d(num_2d)%small_header = all_2d(index_coriolis)%small_header
      all_2d(num_2d)%small_header%end_dims(1) = jmx
      all_2d(num_2d)%small_header%end_dims(2) = imx
      all_2d(num_2d)%small_header%name(1:8) = cosalpha_id
      all_2d(num_2d)%small_header%description(1:46) = 'Cosine of angel true north and Y on Mass Grid '
      ALLOCATE ( all_2d(num_2d)%array( jmx, imx ) )
      all_2d(num_2d)%array = cos( proj_angel * conv_fac )

      !  mapfac_u

      num_2d = num_2d + 1
      print *, 'num_2d is now ', num_2d
      all_2d(num_2d)%small_header = all_2d(index_mapfaccr)%small_header
      all_2d(num_2d)%small_header%end_dims(1) = jmx
      all_2d(num_2d)%small_header%end_dims(2) = imx
      all_2d(num_2d)%small_header%name(1:8) = map_factor_u_id
      all_2d(num_2d)%small_header%description(1:46) = 'Map Factor on u grid                          '
      ALLOCATE ( all_2d(num_2d)%array( jmx, imx ) )
      all_2d(num_2d)%array = mapfac_u

      !  mapfac_v

      num_2d = num_2d + 1
      print *, 'num_2d is now ', num_2d
      all_2d(num_2d)%small_header = all_2d(index_mapfaccr)%small_header
      all_2d(num_2d)%small_header%end_dims(1) = jmx
      all_2d(num_2d)%small_header%end_dims(2) = imx
      all_2d(num_2d)%small_header%name(1:8) = map_factor_v_id
      all_2d(num_2d)%small_header%description(1:46) = 'Map Factor on v grid                          '
      ALLOCATE ( all_2d(num_2d)%array( jmx, imx ) )
      all_2d(num_2d)%array = mapfac_v

!     pstardot = pstarcrs
!     CALL crs2dot (pstardot, imx, jmx)

      !  Time-interpolated VEGFRC

      IF ( fndvegfrc ) THEN

         ALLOCATE ( scr2d2 ( JMX, IMX ) )
         ALLOCATE ( scr2d1 ( JMX, IMX ) )
         CALL setvegfr ( jmxm, imxm, julday, vegfmo, scr2d2 )
         scr2d2(:,imx) = scr2d2(:,imxm)
         scr2d2(jmx,:) = scr2d2(jmxm,:)
         
         num_2d = num_2d + 1
         print *, 'num_2d is now ', num_2d
         all_2d(num_2d)%small_header = all_2d(index_land_use)%small_header
         all_2d(num_2d)%small_header%end_dims(1) = jmx
         all_2d(num_2d)%small_header%end_dims(2) = imx
         all_2d(num_2d)%small_header%name(1:8) = vegfra_id
         all_2d(num_2d)%small_header%description(1:46) = 'Time-interpolated monthly greenness fraction  '
         ALLOCATE ( all_2d(num_2d)%array( jmx, imx ) )
         all_2d(num_2d)%array = scr2d2
         print *, 'VEGFRA : ', scr2d2(20,15), all_2d(num_2d)%array(20,15)

      !  Estimate max and min greenness

         DO i = 1, imxm
         DO j = 1, jmxm
            scr2d1 (j, i) = 1.
            scr2d2 (j, i) = 100.
            DO months = 1, 12
               scr2d1 (j,i) = AMAX1( scr2d1(j,i),vegfmo(j,i,months) )
               scr2d2 (j,i) = AMIN1( scr2d2(j,i),vegfmo(j,i,months) )
            END DO
         END DO
         END DO

         scr2d2(:,imx) = scr2d2(:,imxm)
         scr2d2(jmx,:) = scr2d2(jmxm,:)
         scr2d1(:,imx) = scr2d1(:,imxm)
         scr2d1(jmx,:) = scr2d1(jmxm,:)

         num_2d = num_2d + 1
         print *, 'num_2d is now ', num_2d
         all_2d(num_2d)%small_header = all_2d(index_land_use)%small_header
         all_2d(num_2d)%small_header%end_dims(1) = jmx
         all_2d(num_2d)%small_header%end_dims(2) = imx
         all_2d(num_2d)%small_header%name(1:8) = vegfr_max_id
         all_2d(num_2d)%small_header%description(1:46) = 'Maximum annual greenness fraction             '
         ALLOCATE ( all_2d(num_2d)%array( jmx, imx ) )
         all_2d(num_2d)%array = scr2d1
         print *, 'SHDMAX : ', scr2d1(20,15), all_2d(num_2d)%array(20,15)

         num_2d = num_2d + 1
         print *, 'num_2d is now ', num_2d
         all_2d(num_2d)%small_header = all_2d(index_land_use)%small_header
         all_2d(num_2d)%small_header%end_dims(1) = jmx
         all_2d(num_2d)%small_header%end_dims(2) = imx
         all_2d(num_2d)%small_header%name(1:8) = vegfr_min_id
         all_2d(num_2d)%small_header%description(1:46) = 'Minimum annual greenness fraction             '
         ALLOCATE ( all_2d(num_2d)%array( jmx, imx ) )
         all_2d(num_2d)%array = scr2d2
         print *, 'SHDMIN : ', scr2d2(20,15), all_2d(num_2d)%array(20,15)
      END IF

      DEALLOCATE ( HEIGHT )
      DEALLOCATE ( RH )
      DEALLOCATE ( P )
      DEALLOCATE ( T )
      DEALLOCATE ( TERRAIN )
      DEALLOCATE ( PSEALVLC )
      DEALLOCATE ( SFC_P )
      DEALLOCATE ( PROJ_ANGEL )
      DEALLOCATE ( MAPFAC_U )
      DEALLOCATE ( MAPFAC_V )
      IF ( ALLOCATED(scr2d1) ) DEALLOCATE ( scr2d1 )
      IF ( ALLOCATED(scr2d2) ) DEALLOCATE ( scr2d2 )

   END SUBROUTINE prepare_data

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

END MODULE all_io
