!dis   
!dis    Open Source License/Disclaimer, Forecast Systems Laboratory
!dis    NOAA/OAR/FSL, 325 Broadway Boulder, CO 80305
!dis    
!dis    This software is distributed under the Open Source Definition,
!dis    which may be found at http://www.opensource.org/osd.html.
!dis    
!dis    In particular, redistribution and use in source and binary forms,
!dis    with or without modification, are permitted provided that the
!dis    following conditions are met:
!dis    
!dis    - Redistributions of source code must retain this notice, this
!dis    list of conditions and the following disclaimer.
!dis    
!dis    - Redistributions in binary form must provide access to this
!dis    notice, this list of conditions and the following disclaimer, and
!dis    the underlying source code.
!dis    
!dis    - All modifications to this software must be clearly documented,
!dis    and are solely the responsibility of the agent making the
!dis    modifications.
!dis    
!dis    - If significant modifications or enhancements are made to this
!dis    software, the FSL Software Policy Manager
!dis    (softwaremgr@fsl.noaa.gov) should be notified.
!dis    
!dis    THIS SOFTWARE AND ITS DOCUMENTATION ARE IN THE PUBLIC DOMAIN
!dis    AND ARE FURNISHED "AS IS."  THE AUTHORS, THE UNITED STATES
!dis    GOVERNMENT, ITS INSTRUMENTALITIES, OFFICERS, EMPLOYEES, AND
!dis    AGENTS MAKE NO WARRANTY, EXPRESS OR IMPLIED, AS TO THE USEFULNESS
!dis    OF THE SOFTWARE AND DOCUMENTATION FOR ANY PURPOSE.  THEY ASSUME
!dis    NO RESPONSIBILITY (1) FOR THE USE OF THE SOFTWARE AND
!dis    DOCUMENTATION; OR (2) TO PROVIDE TECHNICAL SUPPORT TO USERS.
!dis   
!dis 

MODULE vinterp_utils

  USE wrfsi_io , ONLY : DryRun
  USE physical_constants
  USE diagnostic_vars
  ! Module that contains arrays and routines for linear and logarithmic
  ! interpolation 

  
  IMPLICIT NONE
  
CONTAINS
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
   SUBROUTINE compute_lin_weights(p_mid_mb,p_lower_mb,p_upper_mb, &
                                  weight_bot, weight_top)

    ! Computes the weighting coefficient for the top bounding level
    ! of two pressure levels to interpolate to a level in-between using
    ! linear interpolation.
   
    IMPLICIT NONE
    REAL, INTENT(IN)             :: p_mid_mb  ! Desired pressure level
    REAL, INTENT(IN)             :: p_lower_mb ! Lower bounding pressure
    REAL, INTENT(IN)             :: p_upper_mb ! Upper bounding pressure
    REAL, INTENT(OUT)            :: weight_bot ! Weight given to bottom level
    REAL, INTENT(OUT)            :: weight_top ! Weight given to top level

    weight_bot = (p_mid_mb - p_upper_mb) / (p_lower_mb - p_upper_mb) 
    weight_top = 1.0 - weight_bot
    
  END SUBROUTINE compute_lin_weights
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE compute_log_weights(p_mid_mb,p_lower_mb,p_upper_mb, &
                                 weight_bot, weight_top)
    
    ! Computes weighting coefficient for upper pressure level that bounds
    ! a desired pressure level for logarithmic interpolation.
    
    ! NOTE: Pressures must be in mb!!!
    
    IMPLICIT NONE
    REAL, INTENT(IN)             :: p_mid_mb  ! Desired pressure level
    REAL, INTENT(IN)             :: p_lower_mb ! Lower bounding pressure
    REAL, INTENT(IN)             :: p_upper_mb ! Upper bounding pressure
    REAL, INTENT(OUT)            :: weight_bot ! Weight given to bottom level
    REAL, INTENT(OUT)            :: weight_top ! Weight given to top level
    
    weight_bot = ( ALOG(p_mid_mb) - ALOG(p_upper_mb) ) / &
                 ( ALOG(p_lower_mb) - ALOG(p_upper_mb) ) 
    weight_top = 1.0 - weight_bot

  END SUBROUTINE compute_log_weights
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE vinterp_3d(var_in, trap_bot_ind, trap_top_ind, &
                        weight_top, below_ground_value, var_out, &
                        nx, ny, nzin, nzout)

    ! Interpolates var_in to var_out using array of trapping indices
    ! and top weight values.  Allows the user to set a default
    ! 2d array to use for "below ground" values

    IMPLICIT NONE
    INTEGER, INTENT(IN)            :: nx
    INTEGER, INTENT(IN)            :: ny
    INTEGER, INTENT(IN)            :: nzin
    INTEGER, INTENT(IN)            :: nzout
    REAL, INTENT(IN)               :: var_in (nx,ny,nzin)
    INTEGER, INTENT(IN)            :: trap_bot_ind(nx,ny,nzout)
    INTEGER, INTENT(IN)            :: trap_top_ind(nx,ny,nzout)
    REAL, INTENT(IN)               :: weight_top(nx,ny,nzout)
    REAL, INTENT(IN)               :: below_ground_value(nx,ny)
    REAL, INTENT(OUT)              :: var_out(nx,ny,nzout)
    INTEGER                        :: i,j,k

    DO j = 1,ny 
      DO i = 1,nx
        DO k= 1,nzout

          ! Is level below ground?  If so, zero it out
          IF (trap_bot_ind(i,j,k).LT. 1) THEN
            var_out(i,j,k) = below_ground_value(i,j)
          
          ! Is it above model top? If so, replicate top value
          ELSE IF (trap_bot_ind(i,j,k).EQ.nzin) THEN
            var_out(i,j,k) = var_in(i,j,nzin)

          ELSE
            var_out(i,j,k) = weight_top(i,j,k) * &
                             var_in(i,j,trap_top_ind(i,j,k)) + &
                             (1.-weight_top(i,j,k)) * &
                             var_in(i,j,trap_bot_ind(i,j,k))
          ENDIF
        ENDDO
      ENDDO
    ENDDO
 
    RETURN
  END SUBROUTINE vinterp_3d
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE compute_eta_3d(nx,ny,nz,p,t,z,rh,ptop,ztopo,eta,mu,psfc_in,k_sfc)
    ! Subroutine to compute eta on a 3D grid.  Also computes mu as a function
    ! of the terrain passed in.

    IMPLICIT NONE
    INTEGER, INTENT(IN)              :: nx,ny,nz    ! Dimensions of array
    REAL, INTENT(IN)                 :: p(nx,ny,nz) ! 3D Pressure (Pa)
    REAL, INTENT(IN)                 :: t(nx,ny,nz) ! 3D Temperature (K)
    REAL, INTENT(IN)                 :: z(nx,ny,nz) ! 3D GPH (gpm)
    REAL, INTENT(IN)                 :: rh(nx,ny,nz)! 3D RH (%)
    REAL, INTENT(IN)                 :: ptop        ! Top pressure (Pa)
    REAL, INTENT(IN)                 :: ztopo(nx,ny)! Topography
    REAL, INTENT(OUT)                :: eta(nx,ny,nz) ! Eta values
    REAL, INTENT(OUT)                :: mu(nx,ny)    ! Mu

    REAL, ALLOCATABLE                :: rho(:)
    REAL, ALLOCATABLE                :: qv(:)
    REAL, ALLOCATABLE                :: pdry(:)
    REAL, ALLOCATABLE                :: iqvp(:)
    REAL                             :: tsfc, psfc, qvsfc, dz, qvbar,iqvp_sfc
    REAL                             :: wgt0, wgt1 
    LOGICAL                          :: found_level
!WW
    INTEGER, INTENT(IN)              :: k_sfc(nx,ny)  ! sfc k index found in driver
    REAL, INTENT(IN)                 :: psfc_in(nx,ny)! psfc from mm5 input

    INTEGER :: i,j,k

    ALLOCATE (rho(nz))      ! Density for each level in a column
    ALLOCATE (qv(nz))       ! Water vapor mixing ratio
    ALLOCATE (pdry(nz))     ! Dry air pressure (iqvp removed)
    ALLOCATE (iqvp(nz))     ! Integrated (downward) vapor pressure

    DO j = 1, ny
      DO i = 1, nx

        ! Compute density and qv in the column
        DO k = 1,nz
          rho(k)= compute_density(t(i,j,k),p(i,j,k))
          qv(k) = compute_vapor_mixing_ratio(t(i,j,k),p(i,j,k),rh(i,j,k),.true.)
        ENDDO

        ! Compute the dry pressure values and iqvp for the column
        ! (pdry+iqvp = p)

        CALL compute_column_pdry(nz,p(i,j,:),rho,qv,z(i,j,:),pdry,iqvp)

        ! Compute mu at this point using the terrain height. Mu is just
        ! the dry surface pressure, which would be the model-adjusted
        ! surface pressure minus the integrated water vapor.

!WW
!       IF (ztopo(i,j) .LE. z(i,j,1) ) THEN

!         ! WRF terrain below lowest BG model level.  Make crude adjustment
!         ! using 10 Pa per meter to get surface pressure and assume a
!         ! constant qv for moisture.  Assume a standard 6.5 K/km lapse rate
!         ! for temperature     
!         k = 1
!         dz = z(i,j,1)-ztopo(i,j)
!         psfc = p(i,j,1) + dz*10.
!         qvbar = qv(1)
!         tsfc = t(i,j,1) + dz*.0065
!       ELSE
!         found_level = .false.
!         find_level: DO k = 2, nz
!           IF ( (ztopo(i,j) .LE. z(i,j,k)) .AND. &
!                (ztopo(i,j) .GT. z(i,j,k-1)) ) THEN
!             found_level = .true.
!             EXIT find_level
!           ENDIF
!         ENDDO find_level
!         IF (.NOT. found_level) THEN
!           PRINT *, 'COMPUTE_ETA_3D: This should not happen.'
!           PRINT *, 'Could not find where to insert WRF surface.'
!           PRINT *, 'I/J = ',i,j
!           PRINT *, 'Topo = ', ztopo(i,j)
!           PRINT *, 'Z in column = ', z(i,j,:)
!           STOP 'level_problem'
!         ENDIF
!         ! Determine pressure, temperature, and qv
!         psfc = EXP ( &
!                      ( ztopo(i,j)*ALOG(p(i,j,k-1)/p(i,j,k)) - &
!                             z(i,j,k)*ALOG(p(i,j,k-1)) + &
!                             z(i,j,k-1)*ALOG(p(i,j,k))  ) / &
!                             (z(i,j,k-1) - z(i,j,k)) )

!         IF ( (p(i,j,k-1)-p(i,j,k)) .GE. 1.) THEN
!           CALL compute_lin_weights(psfc, p(i,j,k-1), p(i,j,k), &
!                wgt0, wgt1)
!         ELSE
!           wgt0 = 0.0
!           wgt1 = 1.0
!         ENDIF

          k = k_sfc(i,j)
          psfc = psfc_in(i,j)
          tsfc = t(i,j,k)
          qvsfc= qv(k)
          qvbar= qvsfc
!         tsfc = t(i,j,k-1)*wgt0 + t(i,j,k)*wgt1
!         qvsfc = qv(k-1)*wgt0 + qv(k)*wgt1
!         qvbar = (qvsfc + qv(k))*0.5
          dz = z(i,j,k) - ztopo(i,j)
!       ENDIF
        
        ! Now compute the integrated vapor pressure between
        ! the surface and the next lowest level above, then add to
        ! iqvp array.  
!       iqvp_sfc = iqvp(k)+g*qvbar*rho(k)*dz/(1.+qvbar)
        iqvp_sfc = iqvp(k)
        mu(i,j) = psfc - iqvp_sfc - ptop
    
        ! We now have everything we need to compute eta for the column
        DO k = 1,nz
          eta(i,j,k) = (pdry(k) - ptop) / mu(i,j)
        ENDDO
        IF ((i .EQ. nx/2).AND.(j .EQ. ny/2 .and. .not.DryRun)) THEN
          ! Some diagnostic prints from the center of the column
          PRINT *, 'Eta values on background grid center column:'
          PRINT *, 'TOPO/PSFC/MU = ',ztopo(i,j), psfc, mu(i,j), k_sfc(i,j)
          PRINT *, 'Z          PRESS      PRESSDRY   IQVAPORP   ETA'
          PRINT *, '---------- ---------- ---------- ---------- ----------'
          DO k=1,nz
            PRINT '(F10.1,1x,F10.1,1x,F10.1,1x,F10.6,1x,F10.6)',z(i,j,k), &
              p(i,j,k), &
              pdry(k),iqvp(k),eta(i,j,k)
          ENDDO  
            PRINT *, ' '
        ENDIF                       
      ENDDO
    ENDDO
    DEALLOCATE (rho)
    DEALLOCATE (qv)
    DEALLOCATE (pdry)
    DEALLOCATE (iqvp)
    RETURN
  END SUBROUTINE compute_eta_3d
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE compute_column_pdry(nz,p,rho,qv,z,pdry,pvapor)

    ! Given full pressure in Pascals (p), dry air density in kg/m^3 (rho),
    ! water vapor mixing ratio in kg/kg (qv), compute the dry pressure value
    ! at each grid point in the column of nz elements.
    ! This routine assumes the pressure array decreases monotonically from
    ! 1->nz and that the rho and qv arrays match the pressure array.

    IMPLICIT NONE
    INTEGER, INTENT(IN)                    :: nz
    REAL, INTENT(IN)                       :: p (nz)
    REAL, INTENT(IN)                       :: rho (nz)
    REAL, INTENT(IN)                       :: qv(nz)
    REAL, INTENT(IN)                       :: z(nz)
    REAL, INTENT(OUT)                      :: pdry(nz)
    REAL, INTENT(OUT)                      :: pvapor(nz)

    INTEGER                                :: kbot, k
    REAL                                   :: qv_mean,dz

    ! Set top vapor pressure to zero and top dry pressure equal to
    ! top total pressure

    pvapor(nz) = 0.
    pdry(nz) = p(nz)

    ! Integrate moisture downward

    main_loop:  DO kbot = nz-1, 1, -1

      ! Initialize for upcoming sums.
      pvapor(kbot) = 0.

      ! Integrate downward
      down_to_here:  DO k = nz-1,kbot,-1
        ! Compute delta-Z and mean Qv for this layer
        dz = z(k+1) - z(k)
        qv_mean = (qv(k) + qv(k+1)) * 0.5

        ! Compute pvapor for this layer and sum with previous layer
        pvapor(kbot) = pvapor(kbot)+g*qv_mean*rho(k)*dz/(1.+qv_mean)
      ENDDO down_to_here

      ! Subtract the integrated vapor pressure from the total pressure
      pdry(kbot) = p(kbot) - pvapor(kbot)
    ENDDO main_loop

  END SUBROUTINE compute_column_pdry                            
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE interp_press_to_z(press_levels, heights, new_level, press_out, &
                                  nx,ny,nz)

    ! Given a 1D array of pressure levels and a 3D array of heights (m) at
    ! those levels, this routine interpolates the pressure to the desired
    ! new_level.

    ! Pressures are in Pa, heights are in m!

    IMPLICIT NONE
    INTEGER, INTENT(IN)               :: nx,ny,nz
    REAL, INTENT(IN)                  :: press_levels(nz)
    REAL, INTENT(IN)                  :: heights(nx,ny,nz)
    REAL, INTENT(IN)                  :: new_level
    REAL, INTENT(OUT)                 :: press_out (nx,ny)
    INTEGER                           :: i,j,k, ktop
    REAL                              :: ptop,pbot,ztop,zbot

    nsloop: DO j = 1, ny
      ewloop: DO i = 1, nx
        vertloop: DO k = 2, nz
          IF (heights(nx,ny,nz) .ge. new_level) THEN
             ktop = k
             EXIT vertloop
          ENDIF
        ENDDO vertloop
        ztop = heights(i,j,k)
        zbot = heights(i,j,k-1)
        ptop = press_levels(k)*0.01    ! Convert to mb!
        pbot = press_levels(k-1)*0.01
        press_out(i,j) = EXP ( &
                              ( new_level*ALOG(pbot/ptop) - &
                              ztop*ALOG(pbot) + &
                              zbot*ALOG(ptop)  ) / &
                              (zbot - ztop) ) * 100.  ! convert back to Pa
      ENDDO ewloop
    ENDDO nsloop
    RETURN
  END SUBROUTINE interp_press_to_z    
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE interp_eta2eta_lin(nx,ny,nz_in,nz_out,eta_in, eta_out, &
                 mu, ptop, data_in, data_out, varname)

    ! Interpolates one column of data from one set of eta surfaces to
    ! another
 
    INTEGER, INTENT(IN)                :: nx
    INTEGER, INTENT(IN)                :: ny 
    INTEGER, INTENT(IN)                :: nz_in
    INTEGER, INTENT(IN)                :: nz_out
    REAL, INTENT(IN)                   :: eta_in(nx,ny,nz_in)
    REAL, INTENT(IN)                   :: eta_out(nz_out)
    REAL, INTENT(IN)                   :: mu(nx,ny)
    REAL, INTENT(IN)                   :: ptop
    REAL, INTENT(IN)                   :: data_in(nx,ny,nz_in)
    REAL, INTENT(OUT)                  :: data_out(nx,ny,nz_out)
    CHARACTER (LEN=8),INTENT(IN)       :: varname

    INTEGER                            :: i,j
    INTEGER                            :: k,kk
    REAL                               :: desired_eta
    REAL                               :: dvaldeta
    REAL                               :: wgt0
  
    data_out(:,:,:) = -99999.9
    DO j = 1, ny
      DO i = 1, nx

        output_loop: DO k = 1, nz_out

          desired_eta = eta_out(k)
          IF (desired_eta .GT. eta_in(i,j,1)) THEN

            IF ((desired_eta - eta_in(i,j,1)).LT. 0.0001) THEN
               data_out(i,j,k) = data_in(i,j,1)
            ELSE
              IF (varname .EQ. 'EXTRAPDN') THEN
                ! Extrapolate downward because desired eta level is below
                ! the lowest level in our input data.  Extrapolate using simple
                ! 1st derivative of value with respect to eta for the bottom 2
                ! input layers.
 
                ! Add a check to make sure we are not using the gradient of 
                ! a very thin layer
  
                IF ( (eta_in(i,j,1)-eta_in(i,j,2)) .GT. 0.001) THEN
                  dvaldeta = (data_in(i,j,1) - data_in(i,j,2)) / &
                            (eta_in(i,j,1)  - eta_in(i,j,2) )
                ELSE
                  dvaldeta = (data_in(i,j,1) - data_in(i,j,3)) / &
                            (eta_in(i,j,1)  - eta_in(i,j,3) )  
                ENDIF
                data_out(i,j,k) = data_in(i,j,1) + &
                              dvaldeta * (desired_eta-eta_in(i,j,1)) 
              ELSE
                data_out(i,j,k) = data_in(i,j,1)
              ENDIF
            ENDIF
          ELSE IF (desired_eta .LE. eta_in(i,j,nz_in)) THEN
            IF ( abs(eta_in(i,j,nz_in) - desired_eta) .LT. 0.0001) THEN
               data_out(i,j,k) = data_in(i,j,nz_in)
            ELSE
              IF (varname .EQ. 'THETA   ') THEN
                ! Extrapolate upward
                IF ( (eta_in(i,j,nz_in-1)-eta_in(i,j,nz_in)) .GT. 0.0005) THEN
                  dvaldeta = (data_in(i,j,nz_in) - data_in(i,j,nz_in-1)) / &
                             (eta_in(i,j,nz_in)  - eta_in(i,j,nz_in-1))
                ELSE
                  dvaldeta = (data_in(i,j,nz_in) - data_in(i,j,nz_in-2)) / &
                             (eta_in(i,j,nz_in)  - eta_in(i,j,nz_in-2)) 
                ENDIF
                data_out(i,j,k) =  data_in(i,j,nz_in) + &
                                   dvaldeta * (desired_eta-eta_in(i,j,nz_in)) 
              ELSE
                data_out(i,j,k) = data_in(i,j,nz_in)
              ENDIF
            ENDIF
          ELSE
            ! We can trap between two levels and linearly interpolate

            input_loop:  DO kk = 1, nz_in-1
              IF (desired_eta .EQ. eta_in(i,j,kk) )THEN
                data_out(i,j,k) = data_in(i,j,kk)
                EXIT input_loop
              ELSE IF ( (desired_eta .LT. eta_in(i,j,kk)) .AND. &
                        (desired_eta .GT. eta_in(i,j,kk+1)) ) THEN
                wgt0 = (desired_eta - eta_in(i,j,kk+1)) / &
                       (eta_in(i,j,kk)-eta_in(i,j,kk+1))
                data_out(i,j,k) = wgt0*data_in(i,j,kk) + &
                                  (1.-wgt0)*data_in(i,j,kk+1)
                EXIT input_loop

              ENDIF        
           
            ENDDO input_loop
          ENDIF
        ENDDO output_loop
      ENDDO
    ENDDO
    RETURN
  END SUBROUTINE interp_eta2eta_lin 
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
END MODULE vinterp_utils  
