SUBROUTINE DA_Read_Obs (iunit, ob, xb)
!-----------------------------------------------------------------------------!
!  History:
!
!  Additions:
!             03/20/2003      Profiler and Buoy Obs            S. R. H. Rizvi
!
!  Purpose: Read the header of a MM5 3D-VAR 2.0 GTS observation file
!-----------------------------------------------------------------------------!

   implicit none

   INTEGER,        INTENT (in)  :: iunit
   TYPE (xb_type)               :: xb          ! First guess model state.
   TYPE (ob_type), INTENT(inout) :: ob

   CHARACTER (LEN =  10)        :: fmt_name
   CHARACTER (LEN =   3)        :: cfm

   CHARACTER (LEN = 160)        :: fmt_info,    &
                                   fmt_loc, & 
                                   fmt_each
   CHARACTER (LEN = 40)         :: obs_platform

   INTEGER                      :: i, j, iost, nlevels, fm

   TYPE (multi_level_type)      :: platform

   TYPE (count_obs_type)        :: count_obs  ! Initialized in Define_Structure

   LOGICAL                      :: connected
   LOGICAL                      :: inside_halo
   LOGICAL                      :: outside
   LOGICAL                      :: discard, cycle_report

   INTEGER                      :: nsurfaces, nuppers,   &
                                   nsound_innov, nsurface_innov, surface_level

   REAL                         :: height_error, u_comp, v_comp

   INTEGER                      :: total_obs, &
                                   num_sound, &
                                   num_sound_sfc, &
                                   num_synop, &
                                   num_pilot, &
                                   num_satob, &
                                   num_satem, &
                                   num_airep, &
                                   num_metar, &
                                   num_ships, &
                                   num_gpspw, &
                                   num_ssmi_retrieval, &
                                   num_ssmi_tb      , &
                                   num_ssmt1, &
                                   num_ssmt2, &
                                   num_pseudo, num_qscat, &
                                   num_profiler, num_buoy

! -------------------------------------------------------------------
! Initialize the counters:

   count_obs % num_synop = count_obs_number_type(0, 0, 0, 0)
   count_obs % num_sound = count_obs_number_type(0, 0, 0, 0)
   count_obs % num_satob = count_obs_number_type(0, 0, 0, 0)
   count_obs % num_pilot = count_obs_number_type(0, 0, 0, 0)
   count_obs % num_satem = count_obs_number_type(0, 0, 0, 0)
   count_obs % num_airep = count_obs_number_type(0, 0, 0, 0)
   count_obs % num_metar = count_obs_number_type(0, 0, 0, 0)
   count_obs % num_ships = count_obs_number_type(0, 0, 0, 0)
   count_obs % num_gpspw = count_obs_number_type(0, 0, 0, 0)
   count_obs % num_ssmi_retrieval = count_obs_number_type(0, 0, 0, 0)
   count_obs % num_ssmi_tb = count_obs_number_type(0, 0, 0, 0)
   count_obs % num_ssmt1 = count_obs_number_type(0, 0, 0, 0)
   count_obs % num_ssmt2 = count_obs_number_type(0, 0, 0, 0)
   count_obs % num_Radar = count_obs_number_type(0, 0, 0, 0)
   count_obs % num_qscat = count_obs_number_type(0, 0, 0, 0)
   count_obs % num_other = count_obs_number_type(0, 0, 0, 0)
   count_obs % num_profiler = count_obs_number_type(0, 0, 0, 0)
   count_obs % num_buoy = count_obs_number_type(0, 0, 0, 0)

!----------------------------------------------------------------------------!

   nsurfaces = 0
   nuppers   = 0

   nsound_innov   =  0
   nsurface_innov = 0

!----------------------------------------------------------------------------!

   total_obs = 0
   num_sound = 0
   num_sound_sfc = 0
   num_synop = 0
   num_pilot = 0
   num_satob = 0
   num_satem = 0
   num_airep = 0
   num_metar = 0
   num_ships = 0
   num_gpspw = 0
   num_ssmi_retrieval = 0
   num_ssmi_tb       = 0   
   num_ssmt1 = 0
   num_ssmt2 = 0
   num_pseudo = 0
   num_qscat = 0
   num_profiler = 0
   num_buoy = 0

!----------------------------------------------------------------------------!

   INQUIRE(UNIT = iunit, OPENED = connected)

   IF(.NOT. connected) RETURN

!----------------------------------------------------------------------------!

!  READ FORMATS
!  ------------

   READ (iunit, FMT = '(A,1X,A)', IOSTAT = iost) &
        fmt_name, fmt_info, &
        fmt_name, fmt_loc,  &
        fmt_name, fmt_each

   IF (iost /= 0) THEN
      write(unit=*, fmt='(/A,I3,A/)') &
           ' ERROR IN INPUT FILE UNIT ',iunit, &
           ' FOR GTS OBSERVATIONS CANNOT BE FOUND OR CANNOT BE OPENED'
      RETURN
   ENDIF

! write(unit=*, fmt='(2a)') &
!      'fmt_info=', fmt_info, &
!      'fmt_loc =', fmt_loc,  &
!      'fmt_each=', fmt_each

!----------------------------------------------------------------------------!

!  SKIP UNITS
!  ----------

   READ (iunit, FMT = '(A)') fmt_name

!  LOOP OVER RECORDS
!  -----------------

   reports: &
   DO
!     READ STATION GENERAL INFO
!     =============================

      discard   = .false.

      READ (iunit, FMT = fmt_info, IOSTAT = iost) &
                   platform%info%platform,    &
                   platform%info%date_char,   &
                   platform%info%name,        &
                   platform%info%levels,      &
                   platform%info%lat,         &
                   platform%info%lon,         &
                   platform%info%elv,         &
                   platform%info%id

!-----Fix funny wind direction at South Pole
      if(platform%info%lat < -89.9999) then
         platform%info%lon = 0.0
      endif

      cfm = trim(platform%info%platform (4:6))
!      read(platform%info%platform (4:6), '(I3)') fm
      read(cfm, '(I3)') fm

      IF (iost /= 0) THEN
          WRITE (0,'(/,A,I3,/)') ' END OF OBS UNIT: ',iunit
          WRITE (0,'(A,I3)')     ' IOSTAT == ',iost
         EXIT reports
      ENDIF

!     READ MODEL LOCATION
!     =========================

      READ (iunit, FMT = fmt_loc) platform%loc%slp, platform%loc%pw

      total_obs = total_obs + 1

! == levels < 1 and not GPSPW, go back to reports

      IF ((platform%info%levels < 1) .AND.            &
          (index(platform%info%platform, 'GPSPW') <= 0)) then
           cycle reports
      ENDIF

      call DA_ll_to_xy (platform%info, platform%loc,   &
                        xb, outside, inside_halo )

!     READ EACH LEVELS
!     ----------------

loop_level: DO i = 1, platform%info%levels

      platform%each (i) = each_level_type(missing_r, missing, -1.0, & ! height
                       field_type(missing_r, missing, missing_r), & ! u
                       field_type(missing_r, missing, missing_r), & ! v
                       field_type(missing_r, missing, missing_r), & ! p
                       field_type(missing_r, missing, missing_r), & ! t
                       field_type(missing_r, missing, missing_r), & ! q
                       field_type(missing_r, missing, missing_r), & ! rh
                       field_type(missing_r, missing, missing_r), & ! td
                       field_type(missing_r, missing, missing_r))  ! speed 

       READ (UNIT = iunit, FMT = TRIM (fmt_each)) &
              platform%each (i)%p,            &
              platform%each (i)%speed,        &
      ! Here the 'direction' is stored in platform%each (i)%v:
              platform%each (i)%v,            &
              platform%each (i)%height,       &
              platform%each (i)%height_qc,    &
              height_error,                       &
              platform%each (i)%t,            &
              platform%each (i)%td,           &
              platform%each (i)%rh

!      write(unit=*, FMT = '( a, i6)') 'i=', i
!      write(unit=*, FMT = TRIM (fmt_each)) &
!             platform%each (i)%p,            &
!             platform%each (i)%speed,        &
!             platform%each (i)%v,            &
!             platform%each (i)%height,       &
!             platform%each (i)%height_qc,    &
!             height_error,                   &
!             platform%each (i)%t,            &
!             platform%each (i)%td,           &
!             platform%each (i)%rh

! Uncomment the following if errors in reported heights (test later):             
!      if(fm == 97 .or. fm == 96 .or. fm == 42) then
!         platform%each (i)%height_qc = -5 ! Aircraft heights are not above surface.
!      end if

      ! To convert the wind speed and direction to 
      !      the model wind components: u, v

       if (platform%each (i)%speed%qc /= missing_data .and. &
           platform%each (i)%v%qc /= missing_data) then

           call FFDDUV (platform%each (i)%speed%inv, &
                        platform%each (i)%v%inv,     &
                        U_comp, V_comp, platform%info%lon, 1)
           platform%each (i)%u = platform%each (i)%speed
           platform%each (i)%v = platform%each (i)%speed
           platform%each (i)%u%inv = U_comp
           platform%each (i)%v%inv = V_comp
       else
           platform%each (i)%u%inv = missing_r
           platform%each (i)%v%inv = missing_r
           platform%each (i)%u%error = missing_r
           platform%each (i)%v%error = missing_r
           platform%each (i)%u%qc  = missing_data
           platform%each (i)%v%qc  = missing_data
       endif

      ENDDO loop_level

      nlevels = platform%info%levels

      if( nlevels > max_ob_levels) then
         nlevels = max_ob_levels

         write(unit=*, fmt='(/a/)') &
              'WARNING   WARNING   WARNING   WARNING   WARNING :'

         write(unit=*, fmt='(/4(2a,2x),a,2f8.2,2x,2(a,f9.2,2x),2(a,i4,2x)/)') &
              'Station: ', trim(platform%info%name), &
              'ID: ', trim(platform%info%id), &
              'Platfrom: ', trim(platform%info%platform), &
              'Date: ', trim(platform%info%date_char), &
              'At Loc(lat, lon): ', platform%info%lat, platform%info%lon, &
              'At elv: ', platform%info%elv, &
              'with pstar: ', platform%info%pstar, &
              'Has level: ', platform%info%levels, &
              'which is great than max_ob_levels: ', max_ob_levels

             write (unit=*, FMT = '(A,1X,A,1X,A,1X,I4,2f9.3,f9.1,1X,A)') &
             platform%info%name,        &
             platform%info%platform,    &
             platform%info%date_char,   &
             platform%info%levels,      &
             platform%info%lat,         &
             platform%info%lon,         &
             platform%info%elv,         &
             platform%info%id

         platform%info%levels = nlevels
      else if( nlevels < 1) then
         if ( fm /= 111 ) then
            cycle reports
         endif
      endif

      if ( fm == 111 ) discard = .false.  ! Don't reject GPS as below ground.

! Counting the OBS thrown away

      CALL DA_Obs_Count(count_obs, fm, cycle_report, &
                        outside, discard, &
                        inside_halo)

      if (cycle_report) then
         cycle reports
      endif

!     write the obs info

!     write (unit=*, FMT = fmt_info) &
!            platform%info%platform,    &
!            platform%info%date_char,   &
!            platform%info%name,        &
!            platform%info%levels,      &
!            platform%info%lat,         &
!            platform%info%lon,         &
!            platform%info%elv,         &
!            platform%info%id

!     write (unit=*, FMT = '(a, 2f12.4)') &
!           'platform%loc%x, y=', platform%loc%x, platform%loc%y

!     write (unit=*, FMT = '(a, l)') &
!           'outside=', outside

!     write (unit=*, FMT = '(a, 2f12.4)') &
!           'platform%info%lat, lon=', platform%info%lat, platform%info%lon

      CALL DA_Obs_Proc_Station(platform, xb, nsound_innov, nsurface_innov)

      nlevels = platform%info%levels

      SELECT CASE(fm)

!     if(index(platform%info%source, 'SYNOP') > 0) then

         CASE (12) ;

         if(.not.use_SynopObs) cycle reports

         num_synop = num_synop + 1

         if(num_synop > ob%num_synop) then
            write(unit=*, fmt='(/a/)') &
                 'WARNING   WARNING   WARNING   WARNING   WARNING :'

            write(unit=*, fmt='(A,I6,A,I6)') 'SYNOP #=',num_synop, &
                      ' > ob%num_synop=', ob%num_synop
            num_synop = ob%num_synop
         endif

         ob%synop(num_synop)%info = platform%info
         ob%synop(num_synop)%loc  = platform%loc
         
         ob%synop(num_synop)%h = platform%each(1)%height
         ob%synop(num_synop)%u = platform%each(1)%u
         ob%synop(num_synop)%v = platform%each(1)%v
         ob%synop(num_synop)%t = platform%each(1)%t
         ob%synop(num_synop)%q = platform%each(1)%q
         ob%synop(num_synop)%p = platform%each(1)%p

!     else if(index(platform%info%source, 'SHIP') > 0 .or. &
!             index(platform%info%source, 'BOUY') > 0) then

         CASE (13, 17) ;                  ! ships          

         if(.not.use_ShipsObs) cycle reports

         num_ships  = num_ships  + 1

         if(num_ships > ob%num_ships) then
            write(unit=*, fmt='(/a/)') &
                 'WARNING   WARNING   WARNING   WARNING   WARNING :'

            write(unit=*, fmt='(A,I6,A,I6)') 'SHIPS #=',num_ships, &
                      ' > ob%num_ships=', ob%num_ships
            num_ships = ob%num_ships
         endif

         ob%ships(num_ships)%info = platform%info
         ob%ships(num_ships)%loc  = platform%loc
         
         ob%ships(num_ships)%h = platform%each(1)%height
         ob%ships(num_ships)%u = platform%each(1)%u
         ob%ships(num_ships)%v = platform%each(1)%v
         ob%ships(num_ships)%t = platform%each(1)%t
         ob%ships(num_ships)%p = platform%each(1)%p
         ob%ships(num_ships)%q = platform%each(1)%q

!     else if(index(platform%info%source, 'METAR') > 0) then

         CASE (15:16) ;

         if(.not.use_MetarObs) cycle reports

         num_metar = num_metar + 1

         if(num_metar > ob%num_metar) then
            write(unit=*, fmt='(/a/)') &
                 'WARNING   WARNING   WARNING   WARNING   WARNING :'

            write(unit=*, fmt='(A,I6,A,I6)') 'METAR #=',num_metar, &
                      ' > ob%num_metar=', ob%num_metar
            num_metar = ob%num_metar
         endif

         ob%metar(num_metar)%info = platform%info
         ob%metar(num_metar)%loc  = platform%loc

         ob%metar(num_metar)%h = platform%each(1)%height
         ob%metar(num_metar)%u = platform%each(1)%u
         ob%metar(num_metar)%v = platform%each(1)%v
         ob%metar(num_metar)%t = platform%each(1)%t
         ob%metar(num_metar)%p = platform%each(1)%p
         ob%metar(num_metar)%q = platform%each(1)%q

!     else if(index(platform%info%source, 'PILOT') > 0) then

         CASE (32:34) ;

         if(.not.use_PilotObs) cycle reports

         num_pilot = num_pilot + 1

         if(num_pilot > ob%num_pilot) then
            write(unit=*, fmt='(/a/)') &
                 'WARNING   WARNING   WARNING   WARNING   WARNING :'

            write(unit=*, fmt='(A,I6,A,I6)') 'PILOT #=',num_pilot, &
                      ' > ob%num_pilot=', ob%num_pilot
            num_pilot = ob%num_pilot
         endif

         ob%pilot(num_pilot)%info = platform%info
         ob%pilot(num_pilot)%loc  = platform%loc

         allocate (ob%pilot(num_pilot)%p (1:nlevels))
         allocate (ob%pilot(num_pilot)%zk(1:nlevels))
         allocate (ob%pilot(num_pilot)%u (1:nlevels))
         allocate (ob%pilot(num_pilot)%v (1:nlevels))

         do i = 1, nlevels
           ob%pilot(num_pilot)%p(i) = platform%each(i)%p%inv
           ob%pilot(num_pilot)%u(i) = platform%each(i)%u
           ob%pilot(num_pilot)%v(i) = platform%each(i)%v
         enddo

!     else if(index(platform%info%source, 'FM-35 TEMP') > 0) then

         CASE (35:38) ;

         if(.not.use_SoundObs) cycle reports

         num_sound = num_sound + 1

         if(num_sound > ob%num_sound) then
            write(unit=*, fmt='(/a/)') &
                 'WARNING   WARNING   WARNING   WARNING   WARNING :'

            write(unit=*, fmt='(A,I6,A,I6)') 'SOUND #=',num_sound, &
                      ' > ob%num_sound=', ob%num_sound
            num_sound = ob%num_sound
         endif

         ob%sonde_sfc(num_sound)%info = platform%info
         ob%sonde_sfc(num_sound)%loc  = platform%loc

!--------Search to see if we have surface obs.

         surface_level = 0

         do i = 1, nlevels
!-----------if(elevation and height are the same, it is surface.)
            if(abs(platform%info%elv - platform%each(i)%height) < 0.1) then
               surface_level = i

!--------------Save surface pressure.
               ob%sonde_sfc(num_sound)%h = platform%each(i)%height
               ob%sonde_sfc(num_sound)%u = platform%each(i)%u
               ob%sonde_sfc(num_sound)%v = platform%each(i)%v
               ob%sonde_sfc(num_sound)%t = platform%each(i)%t
               ob%sonde_sfc(num_sound)%q = platform%each(i)%q
               ob%sonde_sfc(num_sound)%p = platform%each(i)%p

               exit
            endif
         enddo

! ............................
! discarded the sound_sfc data:

!         surface_level = 0
! ............................

         ob%sound(num_sound)%info = platform%info
         ob%sound(num_sound)%loc  = platform%loc

         if(surface_level > 0) then
            nlevels = nlevels - 1
            ob%sound(num_sound)%info%levels = nlevels

            surface_level = 1
            num_sound_sfc = num_sound_sfc + 1
         else
            ob%sonde_sfc(num_sound)%h = missing_r
            ob%sonde_sfc(num_sound)%u%inv   = missing_r
            ob%sonde_sfc(num_sound)%u%qc    = missing
            ob%sonde_sfc(num_sound)%u%error = abs(missing_r)
            ob%sonde_sfc(num_sound)%v = ob%sonde_sfc(num_sound)%u
            ob%sonde_sfc(num_sound)%t = ob%sonde_sfc(num_sound)%u
            ob%sonde_sfc(num_sound)%p = ob%sonde_sfc(num_sound)%u
            ob%sonde_sfc(num_sound)%q = ob%sonde_sfc(num_sound)%u
         endif

         allocate (ob%sound(num_sound)%h (1:nlevels))
         allocate (ob%sound(num_sound)%p (1:nlevels))
         allocate (ob%sound(num_sound)%zk(1:nlevels))
         allocate (ob%sound(num_sound)%u (1:nlevels))
         allocate (ob%sound(num_sound)%v (1:nlevels))
         allocate (ob%sound(num_sound)%t (1:nlevels))
         allocate (ob%sound(num_sound)%q (1:nlevels))

         j = 0
         do i = 1, nlevels+surface_level
           if(i == surface_level) cycle

           j=j+1

           ob%sound(num_sound)%h(j) = platform%each(i)%height
           ob%sound(num_sound)%p(j) = platform%each(i)%p%inv
           ob%sound(num_sound)%u(j) = platform%each(i)%u
           ob%sound(num_sound)%v(j) = platform%each(i)%v
           ob%sound(num_sound)%t(j) = platform%each(i)%t
           ob%sound(num_sound)%q(j) = platform%each(i)%q
         enddo

!     else if(index(platform%info%source, 'SATEM') > 0) then

         CASE (86)    ;

         if(.not.use_SatemObs) cycle reports

         num_satem = num_satem + 1

         if(num_satem > ob%num_satem) then
            write(unit=*, fmt='(/a/)') &
                 'WARNING   WARNING   WARNING   WARNING   WARNING :'

            write(unit=*, fmt='(A,I6,A,I6)') 'SATEM #=',num_satem, &
                      ' > ob%num_satem=', ob%num_satem
            num_satem = ob%num_satem
         endif

         ob%satem(num_satem)%info = platform%info
         ob%satem(num_satem)%loc  = platform%loc

!--------Reject cloudy Satem obs.

         if(platform%loc%pw%inv > 10.) then
            num_satem = num_satem - 1
            cycle reports
         endif

!        The Satem ref_p is put in the SLP position in OBSPROC data stream.

         ob%satem(num_satem)%ref_p= platform%loc%slp%inv

         allocate (ob%satem(num_satem)%p        (1:nlevels))
         allocate (ob%satem(num_satem)%thickness(1:nlevels))

         ob%satem(num_satem)%p(1) = platform%each(1)%p%inv
         ob%satem(num_satem)%thickness(1) = platform%each(1)%t

!        Splitting the reported Satem data into smaller layers.

         do i = 2, nlevels
           ob%satem(num_satem)%p(i) = platform%each(i)%p%inv
           ob%satem(num_satem)%thickness(i) = platform%each(i)%t

           if(ABS(platform%each(i)%t%inv - missing_r) > 1.  .and. &
              ABS(platform%each(i-1)%t%inv - missing_r) > 1.) then
              ob%satem(num_satem)%thickness(i)%inv =            &
              platform%each(i)%t%inv - platform%each(i-1)%t%inv
           else
              ob%satem(num_satem)%thickness(i)%inv = missing_r
           endif
         enddo

!--------Thickness error (m):

         do i = nlevels, 2, -1
            ob%satem(num_satem)%thickness(i)%error = &
            sqrt( ob%satem(num_satem)%thickness(i  )%error ** 2 + &
                  ob%satem(num_satem)%thickness(i-1)%error ** 2) / gravity
         end do

         ob%satem(num_satem)%thickness(1)%error = &
                        sqrt( ob%satem(num_satem)%thickness(1)%error ** 2 + &
                        platform%loc%pw%error ** 2) / gravity

!     else if(index(platform%info%source, 'SATOB') > 0) then

         CASE (88)    ;

         if(.not.use_SatobObs) cycle reports

         num_satob = num_satob + 1

         if(num_satob > ob%num_satob) then
            write(unit=*, fmt='(/a/)') &
                 'WARNING   WARNING   WARNING   WARNING   WARNING :'

            write(unit=*, fmt='(A,I6,A,I6)') 'SATOB #=',num_satob, &
                      ' > ob%num_satob=', ob%num_satob
            num_satob = ob%num_satob
         endif

         ob%satob(num_satob)%info = platform%info
         ob%satob(num_satob)%loc  = platform%loc
         ob%satob(num_satob)%h = platform%each(1)%height
         ob%satob(num_satob)%p = platform%each(1)%p%inv
         ob%satob(num_satob)%u = platform%each(1)%u
         ob%satob(num_satob)%v = platform%each(1)%v

!     else if(index(platform%info%source, 'AIREP') > 0) then

         CASE (42,96:97) ;

         if(.not.use_AirepObs) cycle reports

         num_airep = num_airep + 1

         if(num_airep > ob%num_airep) then
            write(unit=*, fmt='(/a/)') &
                 'WARNING   WARNING   WARNING   WARNING   WARNING :'

            write(unit=*, fmt='(A,I6,A,I6)') 'AIREP #=',num_airep, &
                      ' > ob%num_airep=', ob%num_airep
            num_airep = ob%num_airep
         endif

         ob%airep(num_airep)%info = platform%info
         ob%airep(num_airep)%loc  = platform%loc

         allocate (ob%airep(num_airep)%h        (1:nlevels))
         allocate (ob%airep(num_airep)%p        (1:nlevels))
         allocate (ob%airep(num_airep)%zk       (1:nlevels))
         allocate (ob%airep(num_airep)%u        (1:nlevels))
         allocate (ob%airep(num_airep)%v        (1:nlevels))
         allocate (ob%airep(num_airep)%t        (1:nlevels))

         do i = 1, nlevels
           ob%airep(num_airep)%h(i) = platform%each(i)%height
           ob%airep(num_airep)%p(i) = platform%each(i)%p%inv
           ob%airep(num_airep)%u(i) = platform%each(i)%u
           ob%airep(num_airep)%v(i) = platform%each(i)%v
           ob%airep(num_airep)%t(i) = platform%each(i)%t
         enddo

!     else if(index(platform%info%source, 'GPSPW') > 0) then

         CASE (111) ;
         
         if(.not.use_GpspwObs) cycle reports

         if(num_gpspw >= ob%num_gpspw) cycle reports

         num_gpspw = num_gpspw + 1

         if(num_gpspw > ob%num_gpspw) then
            write(unit=*, fmt='(/a/)') &
                 'WARNING   WARNING   WARNING   WARNING   WARNING :'

            write(unit=*, fmt='(A,I6,A,I6)') 'GPSPW #=',num_gpspw, &
                      ' > ob%num_gpspw=', ob%num_gpspw
            num_gpspw = ob%num_gpspw
         endif

         ob%gpspw(num_gpspw)%info = platform%info
         ob%gpspw(num_gpspw)%loc  = platform%loc
         ob%gpspw(num_gpspw)%tpw  = platform%loc%pw

!        SSM/T1 temperatures:
         CASE (121) ;

         if(.not.use_ssmt1obs) cycle reports

         num_ssmt1 = num_ssmt1 + 1

         if(num_ssmt1 > ob%num_ssmt1) then
            write(unit=*, fmt='(/a/)') &
                 'WARNING   WARNING   WARNING   WARNING   WARNING :'

            write(unit=*, fmt='(A,I6,A,I6)') 'SSMT1 #=',num_ssmt1, &
                      ' > ob%num_ssmt1=', ob%num_ssmt1
            num_ssmt1 = ob%num_ssmt1
         endif

         ob%ssmt1(num_ssmt1)%info = platform%info
         ob%ssmt1(num_ssmt1)%loc  = platform%loc

         allocate (ob%ssmt1(num_ssmt1)%h (1:nlevels))
         allocate (ob%ssmt1(num_ssmt1)%p (1:nlevels))
         allocate (ob%ssmt1(num_ssmt1)%t (1:nlevels))
         allocate (ob%ssmt1(num_ssmt1)%zk(1:nlevels))

         do i = 1, nlevels
           ob%ssmt1(num_ssmt1)%h(i) = platform%each(i)%height
           ob%ssmt1(num_ssmt1)%p(i) = platform%each(i)%p%inv
           ob%ssmt1(num_ssmt1)%t(i) = platform%each(i)%t
         enddo
         
!        SSM/T2 relative humidities:
         CASE (122) ;

         if(.not.use_ssmt2obs) cycle reports

         num_ssmt2 = num_ssmt2 + 1

         if(num_ssmt2 > ob%num_ssmt2) then
            write(unit=*, fmt='(/a/)') &
                 'WARNING   WARNING   WARNING   WARNING   WARNING :'

            write(unit=*, fmt='(A,I6,A,I6)') 'SSMT2 #=',num_ssmt2, &
                      ' > ob%num_ssmt2=', ob%num_ssmt2
            num_ssmt2 = ob%num_ssmt2
         endif

         ob%ssmt2(num_ssmt2)%info = platform%info
         ob%ssmt2(num_ssmt2)%loc  = platform%loc

         allocate (ob%ssmt2(num_ssmt2)%h (1:nlevels))
         allocate (ob%ssmt2(num_ssmt2)%p (1:nlevels))
         allocate (ob%ssmt2(num_ssmt2)%zk(1:nlevels))
         allocate (ob%ssmt2(num_ssmt2)%rh(1:nlevels))

         do i = 1, nlevels
           ob%ssmt2(num_ssmt2)% h(i) = platform%each(i)%height
           ob%ssmt2(num_ssmt2)% p(i) = platform%each(i)%p%inv
           ob%ssmt2(num_ssmt2)%rh(i) = platform%each(i)%rh
         enddo

!        Scatterometer:

         CASE (281)    ;

         if(.not.use_qscatobs) cycle reports

         num_qscat  = num_qscat  + 1

         if(num_qscat > ob%num_qscat) then
            write(unit=*, fmt='(/a/)') &
                 'WARNING   WARNING   WARNING   WARNING   WARNING :'

            write(unit=*, fmt='(A,I6,A,I6)') 'QSCAT #=',num_qscat, &
                      ' > ob%num_qscat=', ob%num_qscat
            num_qscat = ob%num_qscat
         endif
         
         ob%qscat(num_qscat)%info = platform%info
         ob%qscat(num_qscat)%loc  = platform%loc
         
         ob%qscat(num_qscat)%h = platform%each(1)%height
         ob%qscat(num_qscat)%u = platform%each(1)%u
         ob%qscat(num_qscat)%v = platform%each(1)%v

!!!DALE  Impose minimum observation error = 1.0m/s for Quikscat data:
         ob%qscat(num_qscat)%u%error = max(platform%each(1)%u%error,1.0)
         ob%qscat(num_qscat)%v%error = max(platform%each(1)%v%error,1.0)
!!!DALE 

         CASE (132) ;

         if(.not.use_ProfilerObs) cycle reports

         num_profiler = num_profiler + 1

         if(num_profiler > ob%num_profiler) then
            write(unit=*, fmt='(/a/)') &
                 'WARNING   WARNING   WARNING   WARNING   WARNING :'

            write(unit=*, fmt='(A,I6,A,I6)') 'profiler #=',num_profiler, &
                      ' > ob%num_profiler=', ob%num_profiler
            num_profiler = ob%num_profiler
         endif

         ob%profiler(num_profiler)%info = platform%info
         ob%profiler(num_profiler)%loc  = platform%loc

         allocate (ob%profiler(num_profiler)%p (1:nlevels))
         allocate (ob%profiler(num_profiler)%zk(1:nlevels))
         allocate (ob%profiler(num_profiler)%u (1:nlevels))
         allocate (ob%profiler(num_profiler)%v (1:nlevels))

         do i = 1, nlevels
           ob%profiler(num_profiler)%p(i) = platform%each(i)%p%inv
           ob%profiler(num_profiler)%u(i) = platform%each(i)%u
           ob%profiler(num_profiler)%v(i) = platform%each(i)%v
         enddo

         CASE (18,19) ;             ! bouy

         if(.not.use_BuoyObs) cycle reports

         num_buoy  = num_buoy  + 1

         if(num_buoy > ob%num_buoy) then
            write(unit=*, fmt='(/a/)') &
                 'WARNING   WARNING   WARNING   WARNING   WARNING :'

            write(unit=*, fmt='(A,I6,A,I6)') 'BUOY  #=',num_buoy, &
                      ' > ob%num_buoy=', ob%num_buoy
            num_buoy = ob%num_buoy
         endif

         ob%buoy(num_buoy)%info = platform%info
         ob%buoy(num_buoy)%loc  = platform%loc

         ob%buoy(num_buoy)%h = platform%each(1)%height
         ob%buoy(num_buoy)%u = platform%each(1)%u
         ob%buoy(num_buoy)%v = platform%each(1)%v
         ob%buoy(num_buoy)%t = platform%each(1)%t
         ob%buoy(num_buoy)%p = platform%each(1)%p
         ob%buoy(num_buoy)%q = platform%each(1)%q

         CASE DEFAULT;

            write(unit=*, fmt='(/a/)') &
                 'WARNING   WARNING   WARNING   WARNING   WARNING :'

            write(unit=*, fmt='(a)') 'unsaved obs found:'

            write(unit=*, fmt='(2a)') &
                 'platform%info%platform=', platform%info%platform
 
            write(unit=*, fmt='(a, i3)') &
                 'platform%info%levels=', platform%info%levels

      END SELECT

   ENDDO reports

!------------------------------------------------------------------------
!--Check the numbers again, make sure we have the right number.
!------------------------------------------------------------------------

   if(num_sound < ob%num_sound) ob%num_sound = num_sound
   if(num_synop < ob%num_synop) ob%num_synop = num_synop
   if(num_pilot < ob%num_pilot) ob%num_pilot = num_pilot
   if(num_satem < ob%num_satem) ob%num_satem = num_satem
   if(num_satob < ob%num_satob) ob%num_satob = num_satob
   if(num_airep < ob%num_airep) ob%num_airep = num_airep
   if(num_gpspw < ob%num_gpspw) ob%num_gpspw = num_gpspw
   if(num_metar < ob%num_metar) ob%num_metar = num_metar
   if(num_ships < ob%num_ships) ob%num_ships = num_ships
   if(num_ssmi_retrieval < ob%num_ssmi_retrieval) ob%num_ssmi_retrieval = num_ssmi_retrieval
   if(num_ssmi_tb        < ob%num_ssmi_tb       ) ob%num_ssmi_tb = num_ssmi_tb
   if(num_ssmt1 < ob%num_ssmt1) ob%num_ssmt1 = num_ssmt1
   if(num_ssmt2 < ob%num_ssmt2) ob%num_ssmt2 = num_ssmt2
   if(num_qscat < ob%num_qscat) ob%num_qscat = num_qscat
   if(num_profiler < ob%num_profiler) ob%num_profiler = num_profiler
   if(num_buoy < ob%num_buoy) ob%num_buoy = num_buoy

!  PRINT OUT
!  =============

   if(print_detail >= 1) then
 
      WRITE (UNIT = 6 , FMT = '(2(A,I5,A))')    &
       "Remove  ",nsurfaces," surface stations.", &
       "Remove  ",nuppers,  " upper-air records."

      WRITE(6,'(A,I6,A,I6)') 'DA_INNOVATION PROCESSED: nsound =', &
            nsound_innov,'  nsurface =', nsurface_innov

      write(6, fmt='(a)')  ' '

      write(6, fmt='(5x,a,i6,a)') &
           'Read:  ', num_sound, ' SOUND reports,', &
           'Read:  ', num_synop, ' SYNOP reports,', &
           'Read:  ', num_pilot, ' PILOT reports,', &
           'Read:  ', num_satem, ' SATEM reports,', &
           'Read:  ', num_satob, ' SATOB reports,', & 
           'Read:  ', num_airep, ' AIREP reports,', &
           'Read:  ', num_gpspw, ' GPSPW reports,', &
           'Read:  ', num_metar, ' METAR reports,', &
           'Read:  ', num_ships , ' SHIP  reports,', &
           'Read:  ', num_ssmi_retrieval , ' SSMI_RETRIEVAL reports,', &
           'Read:  ', num_ssmi_tb       , ' SSMI_TB       reports,', &
           'Read:  ', num_ssmt1, ' SSMT1 reports,', &
           'Read:  ', num_ssmt2, ' SSMT2 reports,', &
           'Read:  ', num_qscat, ' QSCAT reports,', &
           'Read:  ', num_profiler, ' Profiler reports,', &
           'Read:  ', num_buoy, ' Buoy reports,', &
           'Read:  ', total_obs, ' Total Observations.', &
           'There are ', total_obs - num_sound - num_synop &
                                   - num_pilot - num_satem &
                                   - num_satob - num_airep &
                                   - num_metar - num_ships &
                                   - num_ssmi_retrieval  &
                                   - num_ssmi_tb - num_ssmt1 - num_ssmt2 &
                                   - num_gpspw - num_qscat - num_profiler - num_buoy, &
                                   '  Observations unsaved.'

   end if

   call proc_sum_count_obs( count_obs)
   
   ob%num_synop_glo = count_obs%num_synop%num_used
   ob%num_sound_glo = count_obs%num_sound%num_used
   ob%num_satem_glo = count_obs%num_satem%num_used
   ob%num_satob_glo = count_obs%num_satob%num_used
   ob%num_metar_glo = count_obs%num_metar%num_used
   ob%num_airep_glo = count_obs%num_airep%num_used
   ob%num_ships_glo = count_obs%num_ships%num_used
   ob%num_pilot_glo = count_obs%num_pilot%num_used
   ob%num_gpspw_glo = count_obs%num_gpspw%num_used
   ob%num_ssmi_retrieval_glo = count_obs%num_ssmi_retrieval%num_used
   ob%num_ssmi_tb_glo = count_obs%num_ssmi_tb%num_used
   ob%num_ssmt1_glo = count_obs%num_ssmt1%num_used
   ob%num_ssmt2_glo = count_obs%num_ssmt2%num_used
   ob%num_qscat_glo = count_obs%num_qscat%num_used
   ob%num_profiler_glo = count_obs%num_profiler%num_used
   ob%num_buoy_glo = count_obs%num_buoy%num_used
  

   
   write(6, fmt='(/,25x,a)') &
                        '     used   outdomain  max_er_chk   missing' 
   write(6, fmt='(a,4i11)') &
              '    SYNOP_diag         : ', count_obs %num_synop, &
              '    SOUND_diag         : ', count_obs %num_sound, &
              '    SATEM_diag         : ', count_obs %num_satem, &
              '    SATOB_diag         : ', count_obs %num_satob, &
              '    METAR_diag         : ', count_obs %num_metar, &
              '    AIREP_diag         : ', count_obs %num_airep, &
              '    SHIPS_diag         : ', count_obs %num_ships, &
              '    PILOT_diag         : ', count_obs %num_pilot, &
              '    GPSPW_diag         : ', count_obs %num_gpspw, &
              '    SSMI_RETRIEVAL_diag: ', count_obs %num_ssmi_retrieval, &
              '    SSMI_TB_diag       : ', count_obs %num_ssmi_tb, &
              '    SSMT1_diag         : ', count_obs %num_ssmt1, &
              '    SSMT2_diag         : ', count_obs %num_ssmt2, &
              '    QSCAT_diag         : ', count_obs %num_qscat, &
              '    PROFILER_diag      : ', count_obs %num_profiler, &
              '    BUOY_diag          : ', count_obs %num_buoy

   write(6,'(/a,i11/)') "==> number of SOUND_sfc data available =", num_sound_sfc

END SUBROUTINE DA_Read_Obs

