SUBROUTINE DA_Read_Radar (iunit, ob, xb, xp)

!-----------------------------------------------------------------------------!
! Read the Radar observation file
!-----------------------------------------------------------------------------!

   implicit none

   INTEGER,        INTENT (in)   :: iunit
   TYPE (xb_type)                :: xb          ! First guess model state.
   TYPE (ob_type), INTENT (out)  :: ob

   INTEGER                       :: lvl, i, j, ii, iost, nlevels, fm

   TYPE (Radar_multi_level_type) :: platform

   LOGICAL                       :: connected
   LOGICAL                       :: outside, corrected, &
                                    obs_missing, discard, cycle_report

   REAL                          :: x, y, xc, yc
   REAL                          :: dx, dxm, dy, dym

   REAL     :: model_lowest_heightc , model_lowest_heightd , &
               model_highest_heightc, model_highest_heightd, &
               zdiff_bottom_crs, zdiff_bottom_dot

   TYPE (xpose_type), INTENT(IN) :: xp       ! Domain decomposition vars.
   REAL, DIMENSION(xp%kms:xp%kme):: v_h      ! Model value h at ob hor. location.
   INTEGER                       :: k

! -------------------------------------------------------------------
! Initialize the counters:

   ob % num_Radar = 0

!----------------------------------------------------------------------------!

   INQUIRE ( UNIT = iunit, OPENED = connected )

   IF ( .NOT. connected ) RETURN

   WRITE (0,'(A)')  &
   '--------------------------------------------------------------------------'

!  READ FORMATS
!  ------------

!  LOOP OVER RECORDS
!  -----------------

   reports: &
   DO
!     READ STATION GENERAL INFO
!     =============================

      discard   = .false.

      READ (UNIT = iunit, IOSTAT = iost, &
                   FMT = '(A12,3X,A19,2X,2(F12.3,2X),F8.1,2X,I6)' ) &
                   platform % info % platform,  &
                   platform % info % date_char, &
                   platform % info % lat,       &
                   platform % info % lon,       &
                   platform % info % elv,       &
                   platform % info % levels

      IF (iost /= 0) THEN
         WRITE (0,'(/,A,I3,/)') ' END OF UNIT: ',iunit
         WRITE (0,'(A,I3)')     ' IOSTAT == ',iost
         EXIT reports
      ENDIF

      ob % total_obs = ob % total_obs + 1

      platform % loc % x = -1.0
      platform % loc % y = -1.0
     
      read(platform % info % platform (4:6), '(I3)') fm

! 1.0 First, get the (x, y) coordinates

      CALL llxy (platform%info%lat, platform%info%lon, x, y)

      xc = x - .5
      yc = y - .5
      platform%loc%x = xc
      platform%loc%y = yc

!     READ EACH LEVELS
!     ----------------

      lvl = 1

   loop_level: DO ii = 1, platform % info % levels

      if( lvl > max_levels ) lvl = max_levels

      platform%each(lvl) = Radar_each_level_type(missing_r,missing,  & ! height
                       -1.0,                                         & ! zk
                       field_type(missing_r, missing, missing_r),    & ! rv
                       field_type(missing_r, missing, missing_r) )    ! rf

      READ (UNIT = iunit, FMT = '( 3X, F12.1, 2(F12.3,I4,F12.3,2X) )' ) &
                          platform % each (lvl) % height,           &
                          platform % each (lvl) % rv % inv,         &
                          platform % each (lvl) % rv % qc,          &
                          platform % each (lvl) % rv % error,       &
                          platform % each (lvl) % rf % inv,         &
                          platform % each (lvl) % rf % qc,          &
                          platform % each (lvl) % rf % error


! Processing the obs read in at the level = lvl

      obs_missing = .false.

      if (platform % each (lvl) % rv % inv == missing_r .or. &
          platform % each (lvl) % rv % error == missing_r ) then

         platform % each (lvl) % rv % qc = missing_data
         obs_missing = .true.

      else

      outside = .false.

      platform % info % pstar = missing_r
      call toGrid (xc, 1, mix, i, dx, dxm)
      call toGrid (yc, 1, mjy, j, dy, dym)

      IF ((i > xb%ims) .AND. (i < xb%ime ) .AND. &
          (j > xb%jms) .AND. (j < xb%jme )) THEN

         if(def_sub_domain .and. &
            (x_start_sub_domain > x .or. &
             y_start_sub_domain > y .or. &
             x_end_sub_domain   < x .or. &
             y_end_sub_domain   < y)) then
            platform % each (lvl) % rv % qc = outside_of_domain
            platform % each (lvl) % rf % qc = outside_of_domain
            platform % each (lvl) % zk = -1.0
            outside = .true.
         else

! 2.0 Pstar at obs site

     platform % info % pstar = dym*(dxm*xb % psac(i,j  ) + dx*xb % psac(i+1,j  )) &
                               + dy *(dxm*xb % psac(i,j+1) + dx*xb % psac(i+1,j+1))

   do k=xp%kts,xp%kte
      v_h(k) = dym*(dxm*xb%h(i,j  ,k) + dx*xb%h(i+1,j  ,k)) &
             + dy *(dxm*xb%h(i,j+1,k) + dx*xb%h(i+1,j+1,k))
   enddo
      v_h(1) = dym*(dxm*xb%h(i,j  ,1) + dx*xb%h(i+1,j  ,1)) &
             + dy *(dxm*xb%h(i,j+1,1) + dx*xb%h(i+1,j+1,1))

            platform % loc % i = i
            platform % loc % j = j
            platform % loc % dx = dx
            platform % loc % dy = dy
            platform % loc % dxm = dxm
            platform % loc % dym = dym

! 3.0 OBS vertical model's indices at (xj, yi)

            call h_to_zk ( platform % each (lvl) % height, v_h, xp,   &
                           platform % each (lvl) % zk)

! Set the qc_index beased on the model background

            if (platform % each (lvl) % zk < 1.0) then
               platform % each (lvl) % height_qc = below_model_surface
            else if (platform % each (lvl) % zk > mkz) then
               platform % each (lvl) % height_qc = above_model_lid
            endif

! 2.0 far_below_model_surface

            if (platform % each (lvl) % zk < 1.0 ) then
                zdiff_bottom_crs = platform % each (lvl) % height &
                                   - v_h(1)
                zdiff_bottom_crs = min(0.,zdiff_bottom_crs)

                IF (ABS (zdiff_bottom_crs) .GT. Max_StHeight_Diff) THEN
                   platform % each (lvl) % height_qc = far_below_model_surface
                ENDIF

             endif

          endif

       ELSE

          platform % each (lvl) % rv % qc = outside_of_domain
          platform % each (lvl) % rf % qc = outside_of_domain
          platform % each (lvl) % zk = -1.0
          outside = .true.

       ENDIF

! Exit when zk < 1.0

       if (.NOT. outside) lvl = lvl + 1

       end if
 
   ENDDO loop_level

       nlevels = lvl - 1

       if (nlevels == 0) discard = .true.

       SELECT CASE ( fm )
 
          CASE (128)

          if( discard .or. .not.Use_RadarObs) cycle reports
   
          ob % num_Radar = ob % num_Radar + 1
   
          if(ob % num_Radar > max_Radar_input) then
                write(0,'(A,I6,A,I6)') 'Radar #=',ob % num_Radar, &
                          ' > max_Radar_input=', max_Radar_input
                STOP
          endif

          ob % Radar ( ob % num_Radar ) % info = platform % info
          ob % Radar ( ob % num_Radar ) % loc  = platform % loc

          ob % Radar ( ob % num_Radar ) % info % levels = nlevels
 
          allocate (ob % Radar ( ob % num_Radar ) % height   (1:nlevels))
          allocate (ob % Radar ( ob % num_Radar ) % height_qc(1:nlevels))
          allocate (ob % Radar ( ob % num_Radar ) % zk       (1:nlevels))
          allocate (ob % Radar ( ob % num_Radar ) % rv       (1:nlevels))
          allocate (ob % Radar ( ob % num_Radar ) % rf       (1:nlevels))

          if (nlevels > max_levels) write(0,'(A,2I8)') &
            ' Radar=> nlevels > max_levels:',nlevels, max_levels
    
          do i = 1, nlevels
    
            ob % Radar ( ob % num_Radar ) % height(i) = &
                                            platform % each(i) % height
            ob % Radar ( ob % num_Radar ) % height_qc(i) = &
                                            platform % each(i) % height_qc
            ob % Radar ( ob % num_Radar ) % zk(i) = &
                                            platform % each(i) % zk
            ob % Radar ( ob % num_Radar ) % rv(i) = platform % each(i) % rv
            ob % Radar ( ob % num_Radar ) % rf(i) = platform % each(i) % rf

          enddo

          CASE DEFAULT;

          write(unit=*, fmt='(a)') 'Warining: unsaved obs found:'

          write(unit=*, fmt='(2a)') &
               'platform % info % platform=', platform % info % platform

          write(unit=*, fmt='(a, i3)') &
               'platform % info % levels=', platform % info % levels

       END SELECT

    ENDDO reports

!  PRINT OUT
!  =============
 
    write(unit=*, fmt='(a)')  ' '

    write(unit=*, fmt='(5x,a,i6,a)') &
         'Read:  ', ob % num_Radar, ' Radar reports,'

END SUBROUTINE DA_Read_Radar
