
;   Script to produce standard plots for a WRF real-data run,
;   with the ARW coordinate dynamics option.

load "WRFOptions.ncl"
;load "$NCARG_ROOT/lib/ncarg/nclscripts/csm/gsn_code.ncl"
load "gsn_code.ncl"
load "WRFPlot.ncl"
load "WRFUserARW.ncl"

begin
;
; The WRF ARW input file.  
; This needs to have a ".nc" appended, so just do it.

  a = addfile("../WRFV2/run/wrf_real_input_em.d01.2000-01-24_12:00:00.nc","r")

;  We generate plots, but what kind do we prefer?

   type = "x11"
;  type = "pdf"
;  type = "ps"
;  type = "ncgm"

  wks = gsn_open_wks(type,"wrf_real_input")

; Debug information.

  debug  = False
;  debug  = True
  icount = 0

; Basic Plot Information                             

    res@MainTitle                   = "REAL-TIME WRF : INPUT"


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;  What times and how many time steps are in the data set?

  FirstTime = True
  times  = wrf_user_list_times(a)  ; get times in the file
  ntimes = dimsizes(times)         ; number of times in the file

; This is the big loop over all of the time periods to process.

  do it = 0,ntimes-1,2

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;Save some Time Information and create the Map Background 

    time = it
    if (FirstTime) then
      times_sav = times(it)
    end if
    res@TimeLabel                  = times(it)

    map = wrf_map(wks,a,res)                   

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; First get the variables we will need        

    ; Get MU (analogous to Pstar), and convert from pascal to mb
    if( isfilevar(a,"MU") .and. isfilevar(a,"MUB") )
      mub = wrf_user_getvar(a,"MUB",time)
      mu = wrf_user_getvar(a,"MU",time)
      mu = (mub + mu)*0.01
    end if
    if( isfilevar(a,"MU0") )
      mu0 = wrf_user_getvar(a,"MU0",time)
    end if

    ; Landuse
    if(isfilevar(a,"LU_INDEX"))
      land = wrf_user_getvar(a,"LU_INDEX",time)
    end if

    ; Soil categories
    if(isfilevar(a,"ISLTYP"))
      isoil = wrf_user_getvar(a,"ISLTYP",time)
    end if

    ; Vegetation Fraction
    if(isfilevar(a,"VEGFRA"))
      vgf = wrf_user_getvar(a,"VEGFRA",time)
    end if

    ; Soil temperature
    if(isfilevar(a,"TSLB"))
      tsoil = wrf_user_getvar(a,"TSLB",time)
    else
      if(isfilevar(a,"ST000010"))
        ts1 = wrf_user_getvar(a,"ST000010",time)
      end if
      if(isfilevar(a,"ST010040"))
        ts2 = wrf_user_getvar(a,"ST010040",time)
      end if
      if(isfilevar(a,"ST040100"))
        ts3 = wrf_user_getvar(a,"ST040100",time)
      end if
      if(isfilevar(a,"ST100200"))
        ts4 = wrf_user_getvar(a,"ST100200",time)
      end if
    end if

    ; Soil moisture - only available from LSM option
    if(isfilevar(a,"SMOIS"))
      sf_sfc_phys  = a@SF_SURFACE_PHYSICS
      if ( sf_sfc_phys .eq. 2 ) then
        soilm = wrf_user_getvar(a,"SMOIS",time)
      end if
    else
      if(isfilevar(a,"SM000010"))
        sm1 = wrf_user_getvar(a,"SM000010",time)
      end if
      if(isfilevar(a,"SM010040"))
        sm2 = wrf_user_getvar(a,"SM010040",time)
      end if
      if(isfilevar(a,"SM040100"))
        sm3 = wrf_user_getvar(a,"SM040100",time)
      end if
      if(isfilevar(a,"SM100200"))
        sm4 = wrf_user_getvar(a,"SM100200",time)
      end if
    end if

    if(isfilevar(a,"Z"))
      z   = wrf_user_getvar(a,"Z",time)      ; height
    end if
    th  = wrf_user_getvar(a,"T",time)      ; pert theta
      th  = th + 300.
    qv  = wrf_user_getvar(a,"QVAPOR",time) ; Qv
    tsk = wrf_user_getvar(a,"TSK",time)    ; Surface Temp
    u  = wrf_user_getvar(a,"ua",time)      ; u averaged to mass points
    v  = wrf_user_getvar(a,"va",time)      ; v averaged to mass points
    ter = wrf_user_getvar(a,"HGT",time)    ; terrain height 

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  ;       ************************************************************

  ;       1) MU (pstar)
  ;          Surface T + Surface wind barbs


    ;       Plotting options for MU / MU0                

    if (isvar("mu"))
      opts_mu = res                          
      opts_mu@FieldTitle         = "MU"
      opts_mu@UnitLabel          = "mb"
      opts_mu@ContourParameters  = (/ 5./)
      contour_mu  = wrf_contour(a,wks,mu,opts_mu)
      print_opts("opts_mu",  opts_mu,  debug)
      wrf_map_overlay(wks,map,(/contour_mu/),False)
      print_header(icount,debug)
      delete(opts_mu)
    else  
      if (isvar("mu0"))
        opts_m0 = res                          
        opts_m0@ContourParameters  = (/ 1000./)
        contour_m0  = wrf_contour(a,wks,mu0,opts_m0)
        print_opts("opts_m0",  opts_m0,  debug)
        wrf_map_overlay(wks,map,(/contour_m0/),False)
        print_header(icount,debug)
        delete(opts_m0)
      end if
    end if


    ;       Plotting options for T                

    opts_t = res          
    opts_t@ContourParameters   = (/ 5./) 
    opts_t@cnFillOn            = True 
    opts_t@cnLinesOn           = True
    opts_t@cnLineLabelsOn      = True
    opts_t@cnLineLabelBackgroundColor = -1
    opts_t@gsnSpreadColorEnd   = -3  ; End third from the last color in color map
    contour_t  = wrf_contour(a,wks,tsk,opts_t)
    print_opts("opts_t",  opts_t,  debug)

  ;       Plotting options for Wind Vectors                 

    u_plane = u(0,:,:)*1.94386 ; Turn wind into knots - Typical of a surface map
    v_plane = v(0,:,:)*1.94386
    opts_vct = res          
    opts_vct@FieldTitle        = "Winds"
    opts_vct@UnitLabel         = "kts"
    opts_vct@NumVectors        = 47
    vector      = wrf_vector(a,wks,u_plane, v_plane,opts_vct)
    print_opts("opts_vct", opts_vct, debug)

  ;       MAKE PLOTS                                       

    wrf_map_overlay(wks,map,(/contour_t,vector/),False)
    print_header(icount,debug)

; Delete options and fields, so we don't have carry over
delete(opts_t)
delete(opts_vct)
delete(u_plane)
delete(v_plane)

  ;       ************************************************************

  ;       2) Other 2D fields

  ; TERRAIN

    opts_ter = res
    opts_ter@ContourParameters = (/ 100. /)
    contour_ter  = wrf_contour(a,wks,ter,opts_ter)
    print_opts("opts_ter",  opts_ter,  debug)
    wrf_map_overlay(wks,map,(/contour_ter/),False)
    print_header(icount,debug)

    delete(opts_ter)


  ; LANDUSE , SOIL & VEGETATION FRACTION
    opts_ln = res
    opts_ln@cnFillOn          = True 
    opts_ln@cnRasterModeOn    = True
    opts_ln@gsnSpreadColorStart= 32   
    print_opts("opts_ln",  opts_ln,  debug)

    if (isvar("land"))
      contour_ld  = wrf_contour(a,wks,land,opts_ln)
      wrf_map_overlay(wks,map,(/contour_ld/),False)
      print_header(icount,debug)
    end if

    if (isvar("isoil"))
      contour_sl  = wrf_contour(a,wks,isoil,opts_ln)
      wrf_map_overlay(wks,map,(/contour_sl/),False)
      print_header(icount,debug)
    end if

    if (isvar("vgf"))
      opts_ln@gsnSpreadColorStart= 0   
      opts_ln@ContourParameters = (/ 10. /);
      contour_vf  = wrf_contour(a,wks,vgf,opts_ln)
      wrf_map_overlay(wks,map,(/contour_vf/),False)
      print_header(icount,debug)
    end if

    delete(opts_ln)

  ; SOIL Temperature

    if (isvar("tsoil"))
      dimsl = dimsizes(tsoil)
      do level = 0, dimsl(0)-1
        slevel = level+1
        tsoil_plane  = tsoil(level,:,:)
        opts_ts = res
        opts_ts@cnFillOn         = True
        opts_ts@gsnSpreadColorEnd = -10   
        opts_ts@PlotLevelID      = "Level  " + slevel
        contour_ts     = wrf_contour(a,wks,tsoil_plane,opts_ts)
        print_opts("opts_ts",  opts_ts,  debug)
        wrf_map_overlay(wks,map,(/contour_ts/),False)
        print_header(icount,debug)
      end do 
      delete(opts_ts)
      delete(contour_ts)
    end if
    if (isvar("ts1"))
      opts_ts = res
      opts_ts@cnFillOn         = True
      opts_ts@ContourParameters  = (/ 240.,300.,5./)
      opts_ts@gsnSpreadColorEnd = -10   
      print_opts("opts_ts",  opts_ts,  debug)
      contour_ts     = wrf_contour(a,wks,ts1,opts_ts)
      wrf_map_overlay(wks,map,(/contour_ts/),False)
      print_header(icount,debug)
      if (isvar("ts2"))
        delete(contour_ts)
        contour_ts     = wrf_contour(a,wks,ts2,opts_ts)
        wrf_map_overlay(wks,map,(/contour_ts/),False)
        print_header(icount,debug)
      end if
      if (isvar("ts3"))
        delete(contour_ts)
        contour_ts     = wrf_contour(a,wks,ts3,opts_ts)
        wrf_map_overlay(wks,map,(/contour_ts/),False)
        print_header(icount,debug)
      end if
      if (isvar("ts4"))
        delete(contour_ts)
        contour_ts     = wrf_contour(a,wks,ts4,opts_ts)
        wrf_map_overlay(wks,map,(/contour_ts/),False)
        print_header(icount,debug)
      end if
      delete(opts_ts)
      delete(contour_ts)
    end if

  ; SOIL Moisture

    if (isvar("soilm"))
      dimsl = dimsizes(soilm)
      do level = 0, dimsl(0)-1
        slevel = level+1
        msoil_plane  = soilm(level,:,:)
        opts_ms = res
        opts_ms@cnFillOn         = True
        opts_ms@gsnSpreadColorEnd = -10   
        opts_ms@PlotLevelID      = "Level  " + slevel
        contour_ms     = wrf_contour(a,wks,msoil_plane,opts_ms)
        print_opts("opts_ms",  opts_ms,  debug)
        wrf_map_overlay(wks,map,(/contour_ms/),False)
        print_header(icount,debug)
      end do 
      delete(opts_ms)
      delete(contour_ms)
    end if
    if (isvar("sm1"))
      opts_ms = res
      opts_ms@cnFillOn         = True
      opts_ms@gsnSpreadColorEnd = -10   
      print_opts("opts_ms",  opts_ms,  debug)
      contour_ms     = wrf_contour(a,wks,sm1,opts_ms)
      wrf_map_overlay(wks,map,(/contour_ms/),False)
      print_header(icount,debug)
      if (isvar("sm2"))
        delete(contour_ms)
        contour_ms     = wrf_contour(a,wks,sm2,opts_ms)
        wrf_map_overlay(wks,map,(/contour_ms/),False)
        print_header(icount,debug)
      end if
      if (isvar("sm3"))
        delete(contour_ms)
        contour_ms     = wrf_contour(a,wks,sm3,opts_ms)
        wrf_map_overlay(wks,map,(/contour_ms/),False)
        print_header(icount,debug)
      end if
      if (isvar("sm4"))
        delete(contour_ms)
        contour_ms     = wrf_contour(a,wks,sm4,opts_ms)
        wrf_map_overlay(wks,map,(/contour_ms/),False)
        print_header(icount,debug)
      end if
      delete(opts_ms)
      delete(contour_ms)
    end if


  ;       ************************************************************

  ;       3) 3D fields

    dimsv = dimsizes(th)
    do level =0,dimsv(0)-1,5
    display_level = level + 1


  ; Height
    if (isvar("z")) then
      opts_ht= res
      opts_ht@FieldTitle          = z@description
      opts_ht@PlotLevelID         = "Eta Level  " + display_level
      opts_ht@cnLineColor         = "NavyBlue"
      opts_ht@gsnContourLineThicknessesScale = 2.0
      opts_ht@ContourParameters   = (/ 2.0 /)
      z_plane  = z(level,:,:)
      contour_ht     = wrf_contour(a,wks,z_plane,opts_z)
      print_opts("opts_ht",  opts_ht,  debug)
      wrf_map_overlay(wks,map,(/contour_ht/),False)
      print_header(icount,debug)
      delete(opts_ht)
      delete(z_plane)
    end if 

  ; Theta
    opts_th = res
    opts_th@FieldTitle          = "Theta"
    opts_th@PlotLevelID         = "Eta Level  " + display_level
    opts_th@cnLineColor         = "Red"
    opts_th@gsnContourLineThicknessesScale = 2.0
    opts_th@ContourParameters   = (/ 5.0 /)
    th_plane = th(level,:,:)
    contour_th     = wrf_contour(a,wks,th_plane,opts_th)
    print_opts("opts_th",  opts_th,  debug)
    wrf_map_overlay(wks,map,(/contour_th/),False)
    print_header(icount,debug)
    delete(opts_th)
    delete(th_plane)

  ; Qv
    opts_qv = res
    opts_qv@FieldTitle          = qv@description
    opts_qv@PlotLevelID         = "Eta Level  " + display_level
    opts_qv@cnLineColor         = "Blue"
    opts_qv@gsnContourLineThicknessesScale = 2.0
    opts_qv@ContourParameters   = (/ 2.0 /)
    qv_plane = qv(level,:,:)*1000.
    contour_qv     = wrf_contour(a,wks,qv_plane,opts_qv)
    print_opts("opts_qv",  opts_qv,  debug)
    wrf_map_overlay(wks,map,(/contour_qv/),False)
    print_header(icount,debug)
    delete(opts_qv)
    delete(qv_plane)

  ; Wind Vectors and Speed
    u_plane  = u(level,:,:)
    v_plane  = v(level,:,:)
    spd     = (u_plane*u_plane + v_plane*v_plane)^(0.5) ; m/sec
    u_plane  = u_plane*1.94386    ; kts
    v_plane  = v_plane*1.94386    ; kts
    opts_spd = res
    opts_spd@FieldTitle        = "Wind Speed"
    opts_spd@UnitLabel         = "m/s"
    opts_spd@PlotLevelID       = "Eta Level  " + display_level
    opts_spd@cnLineColor       = "MediumSeaGreen"
    opts_spd@gsnContourLineThicknessesScale = 2.0
    opts_spd@ContourParameters = (/ 10. /)
    opts_vct = res
    opts_vct@FieldTitle        = "Winds"
    opts_vct@UnitLabel         = "kts"
    opts_vct@NumVectors        = 47
    contour_spd    = wrf_contour(a,wks,spd,opts_spd)
    print_opts("opts_spd",  opts_spd,  debug)
    vector         =  wrf_vector(a,wks,u_plane, v_plane,opts_vct)
    print_opts("opts_vct", opts_vct, debug)
    wrf_map_overlay(wks,map,(/contour_spd, vector/),False)
    print_header(icount,debug)
    delete(opts_spd)
    delete(opts_vct)
    delete(u_plane)
    delete(v_plane)
    delete(spd)

    end do


  ;       ************************************************************
      times_sav = times(it)
      FirstTime = False
  end do  ;       end of the time loop

end
