;
;	Script to produce standard plots for a WRF real-data run,
;	with the MASS coordinate dynamics option.

load "wrf_plot.ncl"
load "wrf_user_mass.ncl"
load "gsn_code.ncl"
load "skewt_func.ncl"

;	The WRF MASS input file.  Do not remove ".nc" in
;	the following line

a = addfile("../../WRFV2/test/em_real/wrfout_d01_2000-01-24_12:00:00.nc","r")

;	We generate plots, but what kind do we prefer?

wks = wrf_open_ncgm("wrf_mass_plots")   ; output to ncgm
;wks = wrf_open_X11()                ; output to screen
;wks = wrf_open_PS("wrf_plots")     ; output to postscript

;frame(wks)  ; allow for window resize before beginning plots

;	What times and how many times are in the data set?

times = wrf_user_list_times(a)  ; get times in  the file
ntimes = dimsizes(times)  ; number of times in the file

;	The specific pressure levels that we want the data
;	interpolated to.

pressure_levels = (/ 850., 700., 500., 300./)   ; pressure levels to plot
nlevels = dimsizes(pressure_levels)  ; number of pressure levels

;	This is the big loop over all of the time periods to process.

do it = 0, ntimes-1,2
;do it = 0,2
	time = it
	if (it.eq.0) then
		time_save = times(it)
	end if
	hours = it*3.

	; 	************************************************************

	;	1) SLP, sfc temp, wind barbs

	;	SLP, smooth it

	slvl = wrf_user_getvar(a,"slvl",time) ; psl amd tsl
	wrf_user_filter2d(slvl, 3) ; filter the fields

	;	Turn temperature into Fahrenheit, typical of a surface map.
	tc = wrf_user_getvar(a,"tc",time)
	tc = 1.8*tc+32.
	tc_plane = tc(0,:,:)

	;	Get u and v, put them in knots, again typical of a surface chart.

	u   = wrf_user_getvar(a,"ua",time) ; ua is u averaged to mass points
	v   = wrf_user_getvar(a,"va",time) ; va is v averaged to mass points
	u = u*1.94386
	v = v*1.94386
	u_plane = u(0,:,:)
	v_plane = v(0,:,:)

	;	Plotting options for the temperature.

	opts_tc = True
	opts_tc@MainTitle = "Surface T (F, color) SLP (mb) and winds (kts)"
	opts_tc@UnitLabel = "T(F)"
	opts_tc@TimeLabel = times(it) + " = " + time_save + " + " + hours + " h"
	opts_tc@LevelLabel = "WRF MASS Forecast"
	opts_tc@ContourParameters = (/ -20., 95., 5./);
	opts_tc@LineThickness = 0.5
	opts_tc@cnInfoLabelOn = False
	opts_tc@cnLineLabelsOn = False
	opts_tc@lbAutoManage = False
	opts_tc@lbBottomMarginF = .4
	opts_tc@lbTitleFontHeightF = .02
	opts_tc@lbLabelStrings = (/ -20., 0., 20., 40., 60., 80./)
	opts_tc@lbLabelFontHeightF = .02

	;	Plotting options for the sea level pressure

	opts_psl = True
	opts_psl@NoTitles = False
	opts_psl@MainTitle = " sea-level pressure (mb) "
	opts_psl@TimeLabel = times(it) + " = " + time_save + " + " + hours + " h"
	opts_psl@cnLineColor = "navy blue"
	opts_psl@cnInfoLabelOn = True
	opts_psl@cnHighLabelsOn = True
	opts_psl@cnLowLabelsOn = True
	opts_psl@cnLineLabelAngleF = 0.
	opts_psl@cnLineLabelFontHeightF = .015
	opts_psl@cnLineLabelFontColor = "navy blue"
	opts_psl@cnLineLabelPerimOn = True
	opts_psl@cnLineLabelPerimColor = "navy blue"
	opts_psl@cnLineLabelBackgroundColor = -1
	opts_psl@cnHighLabelBackgroundColor = -1
	opts_psl@cnLowLabelBackgroundColor = -1   
	opts_psl@ContourParameters = (/ 900., 1100., 4. /)
	opts_psl@LineThickness = 2.0

	;	Plotting options for the overlayed wind vectors.

	opts_vct = True
	opts_vct@NumVectors = 47
	opts_vct@WindBarbsOn = True
	opts_vct@NoTitles = True
	opts_vct@vcWindBarbColor = "black"
	opts_vct@vcRefAnnoOn = False


	;	Map options

;	opts_mp = False
	opts_mp = True
	opts_mp@mpGeophysicalLineThicknessF = 0.5
	opts_mp@mpGeophysicalLineColor = "gray"
	opts_mp@mpUSStateLineThicknessF = 0.5
	opts_mp@mpUSStateLineColor = "gray"
	opts_mp@mpNationalLineThicknessF = 0.5
	opts_mp@mpNationalLineColor = "gray"
	opts_mp@mpGridLineThicknessF = 0.5
	opts_mp@mpGridLineColor = "gray"
	opts_mp@mpLimbLineThicknessF = 0.5
	opts_mp@mpLimbLineColor = "gray"
	opts_mp@mpPerimOn = True
	opts_mp@mpPerimLineThicknessF = 1.0
	opts_mp@mpPerimLineColor = "gray"
	opts_mp@mpGridSpacingF = 10
	opts_mp@mpGridMaskMode = 3

	map = wrf_new_map(wks,a,opts_mp)

	;	... and yet more map options galore!

	opts_map = True
	opts_map@LabelFont = "HELVETICA-BOLD"
	opts_map@LabelFontHeight = .01
	wrf_maplabel(wks,map,opts_map)

	;	Now that we have the slp, temp, and wind values,
	;	we can overlay them to make a plot.

	contour_tc     = wrf_new_fill_contour(wks,tc_plane,opts_tc)
	contour_psl    = wrf_new_line_contour(wks,slvl(:,:),opts_psl)
	vector         = wrf_new_vector(wks,u_plane, v_plane,opts_vct)

	wrf_mapoverlay(map,contour_tc)
	wrf_mapoverlay(map,contour_psl)
	wrf_mapoverlay(map,vector)

	;	Put on the map last, and then close the metacode frame.

	draw(map)
	frame(wks)

	NhlRemoveOverlay(map,contour_psl,False)

	; 	************************************************************

	;	2) 2d plots of rain and terrain elevation

  delete(map)

	;	Get the convective, non-convective and total precipitation, 
	;	and get these for the accumulated and tendency values.

	rain_resolved = wrf_user_getvar(a,"RAINNC",time)
	rain_param    = wrf_user_getvar(a,"RAINC",time)
	rain_tot      = rain_resolved+rain_param
	if( it .eq. 0 ) then
		rain_tend = rain_tot
	end if
	rain_tend     = rain_tot - rain_tend
	rain_tot_save = rain_tot
	rain_tot      = rain_tot+0.1
	rain_tend     = rain_tend+0.1

	;	Plotting options for the total precip

	opts_r = True
	opts_r@MainTitle = "Total Precip (color, mm)"
	opts_r@UnitLabel = "Rainfall (mm)"
	opts_r@lbBottomMarginF = .5
	opts_r@TimeLabel = times(it) + " = " + time_save + " + " + hours + " h"
	opts_r@LevelLabel = "WRF MASS Forecast"
	opts_r@cnLevelSelectionMode = "ExplicitLevels"
	opts_r@cnLevels = (/ .1, .2, .4, .8, 1.6, 3.2, 6.4, 12.8, 25.6, 51.2/)
	opts_r@cnFillColors = (/"white","white","DarkOliveGreen1","DarkOliveGreen3","chartreuse","chartreuse3","green","forest green","yellow","orange","red","violet"/)
	opts_r@cnConstFLabelOn = False
	opts_r@lbAutoManage = False
	opts_r@lbTitleFontHeightF = .02
	opts_r@lbLabelFontHeightF = .02
	opts_r@lbBottomMarginF = .4

	;	Plotting options for the 6h precipitation

	opts_rt = opts_r
	opts_rt@MainTitle = "Surface 6 Hour Precip (color, mm), SLP (mb)"

	;	Plotting options for the resolved precip

	opts_prc = opts_r
	opts_prc@MainTitle = "Resolved precip (color, mm)"

	;	Plotting options for the parameterized precip

	opts_parc = opts_r
	opts_parc@MainTitle = "Surface Parameterized Precip (color, mm)"

	;	Plotting options for the map

	opts_mp = True
	opts_mp@mpGeophysicalLineThicknessF = 0.5
	opts_mp@mpGeophysicalLineColor = "gray"
	opts_mp@mpUSStateLineThicknessF = 0.5
	opts_mp@mpUSStateLineColor = "gray"
	opts_mp@mpNationalLineThicknessF = 0.5
	opts_mp@mpNationalLineColor = "gray"
	opts_mp@mpGridLineThicknessF = 0.5
	opts_mp@mpGridLineColor = "gray"
	opts_mp@mpLimbLineThicknessF = 0.5
	opts_mp@mpLimbLineColor = "gray"
	opts_mp@mpPerimLineThicknessF = 1.0
	opts_mp@mpPerimLineColor = "gray"
	opts_mp@mpFillOn = False
	
	opts_map = True
	opts_map@LabelFont = "HELVETICA-BOLD"
	opts_map@LabelFontHeight = .01
	map_base = wrf_new_map(wks,a,opts_mp)
	wrf_maplabel(wks,map_base,opts_map)

	;	Build the contours for the different precip plots.

	contour_tot  = wrf_new_fill_contour(wks, rain_tot,opts_r)
	contour_tend = wrf_new_fill_contour(wks, rain_tend,opts_rt)
	contour_res  = wrf_new_fill_contour(wks,rain_resolved,opts_prc)
	contour_prm  = wrf_new_fill_contour(wks,rain_param,opts_parc)

	;	Plotting options for the SLP we are hanging onto.

	opts_psl@LineThickness = 1.0
	opts_psl@ContourParameters = (/ 900., 1100., 2. /)
	opts_psl@cnLineColor = "blue"
	opts_psl@cnInfoLabelOn = False
	opts_psl@cnLineLabelFontHeightF = .01
	opts_psl@cnLineLabelPerimOn = True
	opts_psl@cnLineLabelPerimColor = "blue"
	opts_psl@cnLineLabelFontColor = "blue"
	opts_psl@cnLineLabelBackgroundColor = 0
	opts_psl@cnHighLabelFontColor = "blue"
	opts_psl@cnHighLabelBackgroundColor = -1
	opts_psl@cnLowLabelFontColor = "blue"
	opts_psl@cnLowLabelBackgroundColor = -1

	;	Build the contours for the SLP

	contour_psl  = wrf_new_line_contour(wks,slvl(:,:),opts_psl)

	;	Bookkeeping, just to allow the tendency at the next time step

	rain_tend = rain_tot_save

	;	6h precip, SLP and map plot

	wrf_mapoverlay(map_base,contour_tend)
	wrf_mapoverlay(map_base,contour_psl)
	draw(map_base)
	frame(wks)
	NhlRemoveOverlay(map_base,contour_tend,False)
	NhlRemoveOverlay(map_base,contour_psl,False)
	
	;	Total precip + map

	wrf_mapoverlay(map_base,contour_tot)
	draw(map_base)
	frame(wks)
	NhlRemoveOverlay(map_base,contour_tot,False)

	;	Parameterized precip + map

	wrf_mapoverlay(map_base,contour_prm)
	draw(map_base)
	frame(wks)
	NhlRemoveOverlay(map_base,contour_prm,False)

	delete(map_base)
	delete(opts_psl)

	; 	************************************************************

	;	3) 3d plots: height, rh, vert vel, temp

	p   = wrf_user_getvar(a, "p",time) ; pressure is our vertical coordinate
	z   = wrf_user_getvar(a, "Z",time) ; grid point height
	rh = wrf_user_getvar(a,"rh",time) ; relative humidity
	w = wrf_user_getvar(a,"wa",time) ; vertical velocity, averaged to half levels
	w = 100.*w ; vert vel in cm/s
	tc = (tc-32.)*.55555 ; temp back in Celcius

	;	Terrain elevation

	ter = wrf_user_getvar(a,"HGT",time) ; need terrain height sometimes

	;	Loop over the selected pressure levels to plot

	do level = 0,nlevels-1

		;	For this loop, what pressure level are we plotting

		pressure = pressure_levels(level)

		;	Vertically interpolate the selected 3d fields to this
		;	pressure surface.

		z_plane  = wrf_user_intrp3d( z,p,ter,"h",pressure,0.)
		tc_plane = wrf_user_intrp3d(tc,p,ter,"h",pressure,0.)
		u_plane  = wrf_user_intrp3d( u,p,ter,"h",pressure,0.) ; u,v still in kts from above
		v_plane  = wrf_user_intrp3d( v,p,ter,"h",pressure,0.)
		rh_plane  = wrf_user_intrp3d( rh,p,ter,"h",pressure,0.)
		w_plane  = wrf_user_intrp3d( w,p,ter,"h",pressure,0.)
		
		;	Plotting options for temperature

		delete(opts_tc) ; zap the old temp stuff, start anew
		opts_tc = True
		opts_tc@MainTitle = pressure + " mb Height (m, blue), T(C, red), and Winds (kts) "
		opts_tc@UnitLabel = "T(C)"
		opts_tc@TimeLabel = times(it) + " = " + time_save + " + " + hours + " h"
		opts_tc@LevelLabel = "WRF MASS Forecast"
		opts_tc@cnLineColor = "red"
		opts_tc@cnLineDashPattern = "1"
		opts_tc@cnInfoLabelOn = True
		opts_tc@cnLineLabelAngleF = 0.
		opts_tc@cnLineLabelFontColor = "red"
		opts_tc@cnLineLabelFontHeightF = .015
		opts_tc@cnLineLabelPerimOn = True
		opts_tc@cnLineLabelPerimColor = "red"
		opts_tc@cnHighLabelFontColor = "red"
		opts_tc@cnLowLabelFontColor = "red"
		opts_tc@LineThickness = 2.0
		opts_tc@ContourParameters = (/ 5.0 /)
		
		;	Plotting options for relative humidity
		
		opts_rh = True
		opts_rh@MainTitle = pressure + " mb Height (m, blue), T (C, red), RH (color) and Winds (kts)"
		opts_rh@UnitLabel = "RH (%)"
		opts_rh@TimeLabel = times(it) + " = " + time_save + " + " + hours + " h"
		opts_rh@LevelLabel = "WRF MASS Forecast"
		opts_rh@ContourParameters = (/ 10., 90., 10. /)
		opts_rh@cnFillColors = (/"white","white","white","white","chartreuse","green","green3","green4","forest green","PaleGreen4"/)
		opts_rh@cnLineLabelBackgroundColor = -1
		opts_rh@lbAutoManage = False
		opts_rh@lbTitleFontHeightF = .02
		opts_rh@lbLabelFontHeightF = .02
		opts_rh@lbBottomMarginF = .4
		
		;	Plotting options for vertical velocity

		opts_w = True
		opts_w@MainTitle = "Height (dm,line), w(color) and winds(kts) "
		opts_w@UnitLabel = "w (cm/s)"
		opts_w@TimeLabel = times(it) + " = " + time_save + " + " + hours + " h"
		opts_w@LevelLabel = "WRF MASS Forecast"
		opts_w@ContourParameters = (/ -25., 25., 2. /)
		
		;	Plotting options for the wind speed

		opts_spd = True
		opts_spd@MainTitle = "300 mb Height (m, blue), Wind Speed (m/sec, green) and Winds (kts)"
		opts_spd@UnitLabel = "wpsd(kts)"
		opts_spd@TimeLabel = times(it) + " = " + time_save + " + " + hours + " h"
		opts_spd@LevelLabel = "WRF MASS Forecast"
		opts_spd@LineThickness = 3.0
		opts_spd@cnLineColor = "medium sea green"
		opts_spd@cnLineLabelFontColor = "medium sea green"
		opts_spd@cnLineLabelAngleF = 0.
		opts_spd@cnLineLabelFontHeightF = .015
		opts_spd@cnLineLabelPerimOn = True
		opts_spd@cnLineLabelPerimColor = "medium sea green"
		opts_spd@cnHighLabelBackgroundColor = -1
		opts_spd@cnHighLabelFontColor = "medium sea green"
		opts_spd@cnLowLabelBackgroundColor = -1
		opts_spd@cnLowLabelFontColor = "medium sea green"
		opts_spd@ContourParameters = (/ 10. /)
		opts_spd@cnInfoLabelOn = True
		
		;	Plotting options for geopotential height

		opts_z = True
		opts_z@NoTitles = True
		opts_z@cnLineColor = "navy blue"
		opts_z@cnInfoLabelOn = False
		opts_z@cnHighLabelsOn = True
		opts_z@cnLowLabelsOn = True
		opts_z@cnLineLabelAngleF = 0.
		opts_z@cnLineLabelFontHeightF = .015
		opts_z@cnLineLabelPerimOn = True
		opts_z@cnLineLabelPerimColor = "navy blue"
		opts_z@cnLineLabelFontColor = "navy blue"
		opts_z@cnHighLabelBackgroundColor = -1
		opts_z@cnHighLabelFontColor = "navy blue"
		opts_z@cnLowLabelBackgroundColor = -1
		opts_z@cnLowLabelFontColor = "navy blue"
		opts_z@LineThickness = 3.

		;	Plotting options for the wind barbs
		
		opts_vct = True
		opts_vct@NumVectors = 47
		opts_vct@WindBarbsOn = True
		opts_vct@NoTitles = True
		opts_vct@vcWindBarbColor = "black"
		
		;	Plotting options for the map background

		opts_mp = True
		opts_mp@mpGeophysicalLineThicknessF = 0.5
		opts_mp@mpGeophysicalLineColor = "gray"
		opts_mp@mpUSStateLineThicknessF = 0.5
		opts_mp@mpUSStateLineColor = "gray"
		opts_mp@mpNationalLineThicknessF = 0.5
		opts_mp@mpNationalLineColor = "gray"
		opts_mp@mpGridLineThicknessF = 0.5
		opts_mp@mpGridLineColor = "gray"
		opts_mp@mpLimbLineThicknessF = 0.5
		opts_mp@mpLimbLineColor = "gray"
		opts_mp@mpPerimLineThicknessF = 0.5
		opts_mp@mpPerimLineColor = "gray"
		
		map = wrf_new_map(wks,a,opts_mp)
		
		opts_map = True
		opts_map@LabelFont = "HELVETICA-BOLD"
		opts_map@LabelFontHeight = .01
		wrf_maplabel(wks,map,opts_map)
		
		;	OK, what do we plot out of all of that mess?  Compute
		;	the contour levels that we are going to have.

		if( level .eq. 0 ) then

			;	Plot 850 mb: temp, rh, height, barbs

			opts_z@ContourParameters = (/ 20.0 /)
			opts_tc@NoTitles = True
			contour_rh     = wrf_new_fill_contour(wks,rh_plane,opts_rh)
			contour_tc     = wrf_new_line_contour(wks,tc_plane,opts_tc)
			contour_height = wrf_new_line_contour(wks, z_plane,opts_z)
			vector         = wrf_new_vector(wks,u_plane, v_plane,opts_vct)

		else
			if( level .eq. 1 ) then

				;	700 mb plot: temp, height, barbs

				opts_z@ContourParameters = (/ 30.0 /)
				contour_tc     = wrf_new_line_contour(wks,tc_plane,opts_tc)
				contour_height = wrf_new_line_contour(wks, z_plane,opts_z)
				vector         = wrf_new_vector(wks,u_plane, v_plane,opts_vct)
			else

				;	500 and 300 mb plot: temp, height, barbs

				opts_z@ContourParameters = (/ 60.0 /)
				contour_tc     = wrf_new_line_contour(wks,tc_plane,opts_tc)
				contour_height = wrf_new_line_contour(wks, z_plane,opts_z)
				vector         = wrf_new_vector(wks,u_plane, v_plane,opts_vct)
			end if
		end if

		;	Same logic as for contour levels, given the level that we are
		;	processing, we overlay the map with what fields.
		
		if( level .eq. 0 ) then
			wrf_mapoverlay(map,contour_rh)
			wrf_mapoverlay(map,contour_tc)
			wrf_mapoverlay(map,contour_height)
			wrf_mapoverlay(map,vector)
		else
			if( level .lt. nlevels-1 ) then
				wrf_mapoverlay(map,contour_tc)
				wrf_mapoverlay(map,contour_height)
				wrf_mapoverlay(map,vector)
			else
				map = wrf_new_map(wks,a,opts_mp)
				wrf_maplabel(wks,map,opts_map)
				u_plane = u_plane*0.514 ; speed is in m/s, but barbs are in kts
				v_plane = v_plane*0.514
				spd = (u_plane*u_plane + v_plane*v_plane)^(0.5)
				contour_spd = wrf_new_line_contour(wks,spd,opts_spd)
				wrf_mapoverlay(map,contour_spd)
				wrf_mapoverlay(map,contour_height)
				wrf_mapoverlay(map,vector)
			end if
		end if
		
		draw(map)
		frame(wks)
	
	end do	;	end of the pressure level loop
	
	;	remove the dummy arrays

	delete(opts_tc)
	delete(tc_plane)
	delete(rh_plane)
	delete(u_plane)
	delete(v_plane)
	delete(contour_tc)
	delete(contour_rh)

	; 	************************************************************

	;	4)	xsection plots

	aspect_ratio = .75
	
	do ip = 1, 2
	
		;	                                     |
		;	                                     |
		;	Two different xsections, angle=90 is |, angle=0 is ------
		;	                                     |
		;	                                     |
			
		if(ip .eq. 1) then
			angle = 90.
		else
			angle = 0.
		end if

		;	Build planes of data for temp and rh
	
		dimsrh = dimsizes(rh)
		plane = new(2,float)
		plane = (/ dimsrh(2)/2, dimsrh(1)/2 /)
		
		rh_plane = wrf_user_intrp3d(rh,z,ter,"v",plane,angle)
		tc_plane = wrf_user_intrp3d(tc,z,ter,"v",plane,angle)
		
		;	Plotting options for rh

		opts_rh = True
		if(ip .eq. 1) then
			opts_rh@MainTitle = "West-East Relative Humidity and T (C) "
		else
			opts_rh@MainTitle = "South-North Relative Humidity and T (C) "
		end if
		opts_rh@UnitLabel = "RH (%)"
		opts_rh@TimeLabel = time_save + " + " + hours + " h"
		opts_rh@ContourParameters = (/ 10., 90., 10. /)
		opts_rh@LevelLabel = "WRF MASS Forecast"
		opts_rh@AspectRatio = aspect_ratio
                opts_rh@cnFillColors = (/"white","white","white","white","chartreuse","green","green3","green4","forest green","PaleGreen4"/)
                opts_rh@cnFillOn = True
                opts_rh@cnMissingValPerimOn = True
                opts_rh@cnMissingValFillColor = 0
                opts_rh@cnMissingValFillPattern = 11
                opts_rh@AspectRatio = aspect_ratio
                opts_rh@tmYLMode          = "Explicit"     ; Set tick mark mode.
                opts_rh@tmYLValues        = (/ 0, 10, 20, 30, 40, 50, 60, 70, 80, 90, 100 /)
                opts_rh@tmYLLabels        = (/ 0,  2,  4,  6,  8, 10, 12, 14, 16, 18,  20 /)
                opts_rh@tiXAxisString   = "No of Grid Points"  ; Label for the X axis
                opts_rh@tiYAxisString   = "Height (km)"      ; Label for the Y axis
                opts_rh@tiXAxisFont     = "Helvetica"        ; Font for X axis label
                opts_rh@tiYAxisFont     = "Helvetica"        ; Font for Y axis label
                opts_rh@tiXAxisFontHeightF = 0.020
                opts_rh@tiYAxisFontHeightF = 0.020
		
		;	Plotting options for temperature

		opts_tc = True
		opts_tc@NoTitles = True
		opts_tc@ContourParameters = (/ 5. /)
		
		;	Get the contour info for the rh and temp
		
		contour_tc = wrf_new_line_contour(wks, tc_plane,opts_tc)
		contour_rh = wrf_new_fill_contour(wks,rh_plane,opts_rh)

		;	Overlay the fields on the background
		
		overlay(contour_rh,contour_tc)
		draw(contour_rh)
		frame(wks)

		;	Remove the dummy stuff

		delete(contour_tc)
		delete(contour_rh)
		delete(tc_plane)
		delete(rh_plane)
		
	end do	;	end of xsection plot rotation

	; 	************************************************************

	;	5)	skew-T plots

	;	We already have temp, get the rest of the cast assembled
		
	qv  = wrf_user_getvar(a,"QVAPOR",time)
	td =  wrf_user_getvar(a,"td",time)
        uv = wrf_user_getvar(a,"umeta",time) ; u,v not rotated to grid
        u =  uv(0,:,:,:)*1.94386
        v =  uv(1,:,:,:)*1.94386
		
	do ip = 1, 2
		
		;	Define a few skew-T plotting options

		skewtOpts               = True
		dataOpts                = True
		dataOpts@Parcel         = 0
		dataOpts@WspdWdir       = False   ; wind speed and dir [else: u,v]
		dataOpts@PlotWindH      = False  ; plot wind barbs at h lvls [pibal; special]
		dataOpts@HspdHdir       = True   ; wind speed and dir [else: u,v]
		
		;	Choose the lat/lon of the skew-T plots	
		
		if (ip .eq. 1) then
			locr = wrf_user_find_ij_lat_long(a, 39.77, 104.87 )
			loc = floattointeger(locr)
			loc_str = "Skew-T at Denver valid at " + times(it)
		end if
		if (ip .eq. 2) then
			locr = wrf_user_find_ij_lat_long(a, 38.58, 77.0 )
			loc = floattointeger(locr)
			loc_str = "Skew-T at DC valid at " + times(it)
		end if
		
		;	Get the skew-T background

		skewtOpts@tiMainString    = loc_str
		skewt_bkgd    = skewT_BackGround (wks, skewtOpts)
		
		;	Draw the skew-T plot

		draw (skewt_bkgd)
		skewT_data = skewT_PlotData(wks, skewt_bkgd, p(:,loc(0), loc(1)), \
		tc(:,loc(0), loc(1)), \
		td(:,loc(0), loc(1)), \
		z(:,loc(0), loc(1)), \
		-u(:,loc(0), loc(1)), \
		-v(:,loc(0), loc(1)), \
		dataOpts              )
		
		;	Close the frame

		frame(wks)

		;	Delete the temporary arrays

		delete(skewtOpts)
		delete(dataOpts)
		delete(skewT_data)
		delete(skewt_bkgd)
	
	end do

	delete(opts_tc) ; zap the old temp stuff, start anew
	
	; 	************************************************************

end do 	;	end of the time loop
