;
;	Script to produce standard plots for a WRF real-data run,
;	with the MASS coordinate dynamics option.

load "wrf_plot.ncl"
load "wrf_user_mass.ncl"
load "gsn_code.ncl"

;	The WRF MASS input file.  This needs to have a ".nc" appended,
;	so just do it.

a = addfile("../../WRFV2/test/em_real/wrfout_d01_2000-01-24_12:00:00.nc","r")

;	We generate plots, but what kind do we prefer?

wks = wrf_open_ncgm("wrf_sfc")   ; output to ncgm
;wks = wrf_open_X11()                ; output to screen
;wks = wrf_open_PS("wrf_plots")     ; output to postscript

;	What times and how many times are in the data set?

times = wrf_user_list_times(a)  ; get times in  the file
ntimes = dimsizes(times)  ; number of times in the file

;	This is the big loop over all of the time periods to process.

do it = 0, ntimes-1
;do it = 0,1
	time = it
	if (it.eq.0) then
		time_save = times(it)
	end if
	hours = it*6.

	; 	************************************************************

	;	1) SLP, sfc temp, wind barbs

	;	SLP, smooth it

	slvl = wrf_user_getvar(a,"slvl",time) ; psl amd tsl
	wrf_user_filter2d(slvl, 3) ; filter the fields

	;	Turn temperature into Fahrenheit, typical of a surface map.

	tc = wrf_user_getvar(a,"tc",time)
	tc = 1.8*tc+32.
	tc_plane = tc(0,:,:)

	;	Turn temperature into Fahrenheit, typical of a surface map.

	rh = wrf_user_getvar(a,"rh",time)
	rh_plane = rh(0,:,:)

	;	Get u and v, put them in knots, again typical of a surface chart.

	u   = wrf_user_getvar(a,"ua",time) ; ua is u averaged to mass points
	v   = wrf_user_getvar(a,"va",time) ; va is v averaged to mass points
	u = u*1.94386
	v = v*1.94386
	u_plane = u(0,:,:)
	v_plane = v(0,:,:)

	;	Plotting options for the temperature.

	opts_tc = True
	opts_tc@MainTitle = "Surface T (F, color) SLP (mb) and winds (kts)"
	opts_tc@UnitLabel = "T(F)"
	opts_tc@TimeLabel = times(it) + " = " + time_save + " + " + hours + " h"
	opts_tc@LevelLabel = "WRF MASS Run"
	opts_tc@ContourParameters = (/ 20., 100., 5./);
	opts_tc@LineThickness = 0.5
	opts_tc@cnInfoLabelOn = False
	opts_tc@cnLineLabelsOn = False
	opts_tc@lbAutoManage = False
	opts_tc@lbBottomMarginF = .4
	opts_tc@lbTitleFontHeightF = .02
	opts_tc@lbLabelStrings = (/ 20., 40., 60., 80., 100./)
	opts_tc@lbLabelFontHeightF = .02

	;	Plotting options for the sea level pressure

	opts_psl = True
	opts_psl@NoTitles = False
	opts_psl@MainTitle = " sea-level pressure (mb) "
	opts_psl@TimeLabel = times(it) + " = " + time_save + " + " + hours + " h"
	opts_psl@cnLineColor = "navy blue"
	opts_psl@cnInfoLabelOn = True
	opts_psl@cnHighLabelsOn = True
	opts_psl@cnLowLabelsOn = True
	opts_psl@cnLineLabelAngleF = 0.
	opts_psl@cnLineLabelFontHeightF = .015
	opts_psl@cnLineLabelFontColor = "navy blue"
	opts_psl@cnLineLabelPerimOn = True
	opts_psl@cnLineLabelPerimColor = "navy blue"
	opts_psl@cnLineLabelBackgroundColor = -1
	opts_psl@cnHighLabelBackgroundColor = -1
	opts_psl@cnLowLabelBackgroundColor = -1   
	opts_psl@ContourParameters = (/ 900., 1100., 2. /)
	opts_psl@LineThickness = 2.0

	;	Plotting options for the overlayed wind vectors.

	opts_vct = True
	opts_vct@NumVectors = 41
	opts_vct@WindBarbsOn = True
	opts_vct@NoTitles = True
	opts_vct@vcWindBarbColor = "black"
	opts_vct@vcRefAnnoOn = False

	;	Plotting options for the RH.

	opts_rh = True
	opts_rh@MainTitle = "Surface RH (%, color) and winds (kts)"
	opts_rh@UnitLabel = "RH(%)"
	opts_rh@TimeLabel = times(it) + " = " + time_save + " + " + hours + " h"
	opts_rh@LevelLabel = "WRF MASS Run"
	opts_rh@ContourParameters = (/ 10., 90., 10./);
        opts_rh@cnFillColors = (/"white","white","white","white","chartreuse","green","green3","green4","yellowgreen","springgreen4","DarkOliveGreen"/)
	opts_rh@LineThickness = 0.5
	opts_rh@cnInfoLabelOn = False
	opts_rh@cnLineLabelsOn = False
	opts_rh@lbAutoManage = False
	opts_rh@lbBottomMarginF = .4
	opts_rh@lbTitleFontHeightF = .02
	opts_rh@lbLabelFontHeightF = .02

	;	Map options

;	opts_mp = False
	opts_mp = True
	opts_mp@mpGeophysicalLineThicknessF = 0.5
	opts_mp@mpGeophysicalLineColor = "gray"
	opts_mp@mpUSStateLineThicknessF = 0.5
	opts_mp@mpUSStateLineColor = "gray"
	opts_mp@mpNationalLineThicknessF = 0.5
	opts_mp@mpNationalLineColor = "gray"
	opts_mp@mpGridLineThicknessF = 0.5
	opts_mp@mpGridLineColor = "gray"
	opts_mp@mpLimbLineThicknessF = 0.5
	opts_mp@mpLimbLineColor = "gray"
	opts_mp@mpPerimOn = True
	opts_mp@mpPerimLineThicknessF = 1.0
	opts_mp@mpPerimLineColor = "gray"
	opts_mp@mpGridSpacingF = 10
	opts_mp@mpGridMaskMode = 3

	map = wrf_new_map(wks,a,opts_mp)

	;	... and yet more map options galore!

	opts_map = True
	opts_map@LabelFont = "HELVETICA-BOLD"
	opts_map@LabelFontHeight = .01
	wrf_maplabel(wks,map,opts_map)

	;	Now that we have the slp, temp, and wind values,
	;	we can overlay them to make a plot.

	contour_tc     = wrf_new_fill_contour(wks,tc_plane,opts_tc)
	contour_psl    = wrf_new_line_contour(wks,slvl(:,:),opts_psl)
	vector         = wrf_new_vector(wks,u_plane, v_plane,opts_vct)

	wrf_mapoverlay(map,contour_tc)
	wrf_mapoverlay(map,contour_psl)
	wrf_mapoverlay(map,vector)

	;	Put on the map last, and then close the metacode frame.

	draw(map)
	frame(wks)

	NhlRemoveOverlay(map,contour_psl,False)
	NhlRemoveOverlay(map,contour_tc,False)

        ;       Plot sfc RH and wind vectors

	contour_rh     = wrf_new_fill_contour(wks,rh_plane,opts_rh)
	vector         = wrf_new_vector(wks,u_plane, v_plane,opts_vct)

	wrf_mapoverlay(map,contour_rh)
	wrf_mapoverlay(map,vector)

	;	Put on the map last, and then close the metacode frame.

	draw(map)
	frame(wks)

	NhlRemoveOverlay(map,contour_rh,False)

	; 	************************************************************

	;	2) 2d plots of rain and terrain elevation

        delete(map)

	;	Get the convective, non-convective and total precipitation, 
	;	and get these for the accumulated and tendency values.

	rain_resolved = wrf_user_getvar(a,"RAINNC",time)
	rain_param    = wrf_user_getvar(a,"RAINC",time)
	rain_tot      = rain_resolved+rain_param
	if( it .eq. 0 ) then
		rain_save = rain_tot
		rainc_save = rain_param
		rainnc_save = rain_resolved
	end if
	rain_tend     = rain_tot - rain_save
	rainc_tend    = rain_param - rainc_save
	rainnc_tend   = rain_resolved - rainnc_save
	rain_save     = rain_tot
	rainc_save    = rain_param
	rainnc_save   = rain_resolved

	;	Terrain elevation

	ter = wrf_user_getvar(a,"HGT",time) ; need terrain height sometimes

	;	Plotting options for the total precip

	opts_r = True
	opts_r@MainTitle = "Total Precip (color, mm)"
	opts_r@UnitLabel = "Rainfall (mm)"
	opts_r@lbBottomMarginF = .5
	opts_r@TimeLabel = times(it) + " = " + time_save + " + " + hours + " h"
	opts_r@LevelLabel = "WRF MASS Run"
	opts_r@cnLevelSelectionMode = "ExplicitLevels"
	opts_r@cnLevels = (/ .1, .2, .4, .8, 1.6, 3.2, 6.4, 12.8, 25.6, 51.2/)
	opts_r@cnFillColors = (/"white","white","DarkOliveGreen1","DarkOliveGreen3","chartreuse","chartreuse3","green","forest green","yellow","orange","red","violet"/)
	opts_r@cnConstFLabelOn = False
	opts_r@lbAutoManage = False
	opts_r@lbTitleFontHeightF = .02
	opts_r@lbLabelFontHeightF = .02
	opts_r@lbBottomMarginF = .4

	;	Plotting options for the 6h precipitation

	opts_rt = opts_r
	opts_rt@MainTitle = "Surface 3 Hourly Precip (color, mm), SLP (mb)"

	;	Plotting options for the resolved precip

	opts_nc = opts_r
	opts_nc@MainTitle = "Resolved precip (color, mm)"

	;	Plotting options for the parameterized precip

	opts_c = opts_r
	opts_c@MainTitle = "Surface Parameterized Precip (color, mm)"

	;	Plotting options for the map

	opts_mp = True
	opts_mp@mpGeophysicalLineThicknessF = 0.5
	opts_mp@mpGeophysicalLineColor = "gray"
	opts_mp@mpUSStateLineThicknessF = 0.5
	opts_mp@mpUSStateLineColor = "gray"
	opts_mp@mpNationalLineThicknessF = 0.5
	opts_mp@mpNationalLineColor = "gray"
	opts_mp@mpGridLineThicknessF = 0.5
	opts_mp@mpGridLineColor = "gray"
	opts_mp@mpLimbLineThicknessF = 0.5
	opts_mp@mpLimbLineColor = "gray"
	opts_mp@mpPerimLineThicknessF = 1.0
	opts_mp@mpPerimLineColor = "gray"
	opts_mp@mpFillOn = False
	
	opts_map = True
	opts_map@LabelFont = "HELVETICA-BOLD"
	opts_map@LabelFontHeight = .01
	map_base = wrf_new_map(wks,a,opts_mp)
	wrf_maplabel(wks,map_base,opts_map)

	;	Build the contours for the different precip plots.

	contour_tot  = wrf_new_fill_contour(wks, rain_tot,opts_r)
	contour_tend = wrf_new_fill_contour(wks, rain_tend,opts_rt)
	contour_res  = wrf_new_fill_contour(wks,rainc_tend,opts_c)
	contour_prm  = wrf_new_fill_contour(wks,rainnc_tend,opts_nc)

	;	Plotting options for the SLP we are hanging onto.

	opts_psl@LineThickness = 1.0
	opts_psl@ContourParameters = (/ 900., 1100., 2. /)
	opts_psl@cnLineColor = "blue"
	opts_psl@cnInfoLabelOn = False
	opts_psl@cnLineLabelFontHeightF = .01
	opts_psl@cnLineLabelPerimOn = True
	opts_psl@cnLineLabelPerimColor = "blue"
	opts_psl@cnLineLabelFontColor = "blue"
	opts_psl@cnLineLabelBackgroundColor = 0
	opts_psl@cnHighLabelFontColor = "blue"
	opts_psl@cnHighLabelBackgroundColor = -1
	opts_psl@cnLowLabelFontColor = "blue"
	opts_psl@cnLowLabelBackgroundColor = -1

	;	Build the contours for the SLP

	contour_psl  = wrf_new_line_contour(wks,slvl(:,:),opts_psl)

	;	6h precip, SLP and map plot

	wrf_mapoverlay(map_base,contour_tend)
	wrf_mapoverlay(map_base,contour_psl)
	draw(map_base)
	frame(wks)
	NhlRemoveOverlay(map_base,contour_tend,False)
	NhlRemoveOverlay(map_base,contour_psl,False)
	
	;	Total precip + map

	wrf_mapoverlay(map_base,contour_tot)
	draw(map_base)
	frame(wks)
	NhlRemoveOverlay(map_base,contour_tot,False)

	;	Parameterized precip + map

	wrf_mapoverlay(map_base,contour_prm)
	draw(map_base)
	frame(wks)
	NhlRemoveOverlay(map_base,contour_prm,False)

	;	Parameterized precip + map

	wrf_mapoverlay(map_base,contour_res)
	draw(map_base)
	frame(wks)
	NhlRemoveOverlay(map_base,contour_res,False)

	delete(map_base)
	delete(opts_psl)

	; 	************************************************************

end do 	;	end of the time loop
