MODULE wrfsi_variables

! This module is used to define all recognized variable names for the
! SI so it can determine how to interpolate, etc.

  USE wrf_metadata
  IMPLICIT NONE

  PUBLIC

  ! Parameters for status flags

  INTEGER, PARAMETER          :: VAR_KNOWN   = 0
  INTEGER, PARAMETER          :: VAR_UNKNOWN = 1

  TYPE variable_def_struct
  
    CHARACTER (LEN=8)         :: name
    CHARACTER (LEN=80)        :: description
    CHARACTER (LEN=16)        :: units_out
    LOGICAL                   :: convert_units
    LOGICAL                   :: masked_field
    LOGICAL                   :: lsm_hinterp
    LOGICAL                   :: si_output_var
    INTEGER                   :: valid_mask
    INTEGER                   :: hinterp_stagger
    INTEGER                   :: vinterp_stagger
    REAL                      :: minimum_value
    REAL                      :: maximum_value
    REAL                      :: missing_value
    REAL                      :: default_value
    REAL                      :: conversion_factor

  END TYPE variable_def_struct

CONTAINS

  SUBROUTINE return_variable_def (name,vardef,status)

    ! Given an SI 8-character variable string, check to see
    ! if the variable is recognized and return some basic info
    ! about it.  This is the new way of determining horizontal
    ! interpolation method, staggering, etc.  This subroutine
    ! must contain an entry in its main CASE statement for any 
    ! variable you would like to input.

    IMPLICIT NONE
 
    CHARACTER (LEN=8), INTENT(IN)             :: name
    TYPE (variable_def_struct), INTENT(OUT)   :: vardef
    INTEGER, INTENT(OUT)                      :: status
    CHARACTER (LEN=42) :: routine_name


    routine_name = 'module_wrfsi_variables:return_variable_def'
    ! Initialize status flag to recongized and set up default values
    ! for those things we can do, which allows us to override by
    ! exception in the CASE statments below

    status = VAR_KNOWN

    ! Atmospheric variables

    PRINT *, 'Checking for NAME in atmospheric list : ', name

    IF (name .EQ. 'T       ') THEN  
      ! Temperature
      vardef%name          = 'T       '
      vardef%description   = "Temperature                            "
      vardef%units_out     = "K               "
      vardef%convert_units = .FALSE.
      vardef%masked_field  = .FALSE.
      vardef%lsm_hinterp   = .FALSE.
      vardef%si_output_var = .FALSE.
      vardef%valid_mask    = 2
      vardef%hinterp_stagger = n_ind
      vardef%vinterp_stagger = t_ind
      vardef%minimum_value = 100.
      vardef%maximum_value = 500.
      vardef%missing_value = rmissing
      vardef%default_value = 273.
      vardef%conversion_factor = 1.0

    ELSEIF( (name .EQ. 'HGT     ') .OR.&
            (name .EQ. 'HEIGHT  ')) THEN  
      ! Height
      vardef%name          = 'HGT     '
      vardef%description   = "Height                                 "
      vardef%units_out     = "m               "
      vardef%convert_units = .FALSE.
      vardef%masked_field  = .FALSE.
      vardef%lsm_hinterp   = .FALSE.
      vardef%si_output_var = .FALSE.
      vardef%valid_mask    = 2
      vardef%hinterp_stagger = n_ind
      vardef%vinterp_stagger = t_ind
      vardef%minimum_value = -2000.
      vardef%maximum_value = 100000.
      vardef%missing_value = rmissing
      vardef%default_value = 0.
      vardef%conversion_factor = 1.0

    ELSEIF( (name .EQ. 'RH      ') .OR. &
            (name .EQ. 'RELHUM  ') ) THEN  
      ! Relative Humidity
      vardef%name          = 'RH      '
      vardef%description   = "Relative humidity (percentage)         "
      vardef%units_out     = "                "
      vardef%convert_units = .FALSE.
      vardef%masked_field  = .FALSE.
      vardef%lsm_hinterp   = .FALSE.
      vardef%si_output_var = .FALSE.
      vardef%valid_mask    = 2
      vardef%hinterp_stagger = n_ind
      vardef%vinterp_stagger = t_ind
      vardef%minimum_value = 0.01
      vardef%maximum_value = 110.
      vardef%missing_value = rmissing
      vardef%default_value = 10.
      vardef%conversion_factor = 1.0

    ELSEIF( (name .EQ. 'QVAPOR  ') .OR. &
            (name .EQ. 'MIXRAT  ') ) THEN
      ! Water vapor mixing ratio
      vardef%name          = 'QVAPOR  '
      vardef%description   = "Water vapor mixing ratio               "
      vardef%units_out     = "kg kg-1         "
      vardef%convert_units = .FALSE.
      vardef%masked_field  = .FALSE.
      vardef%lsm_hinterp   = .FALSE.
      vardef%si_output_var = .TRUE.
      vardef%valid_mask    = 2
      vardef%hinterp_stagger = n_ind
      vardef%vinterp_stagger = t_ind
      vardef%minimum_value = 1e-20
      vardef%maximum_value = 1.000
      vardef%missing_value = rmissing
      vardef%default_value = 0.001
      vardef%conversion_factor = 1.0

    ELSEIF(name .EQ. 'VPTMP   ') THEN 
      ! Virtual potential temperature
      vardef%name          = "VPTMP   "
      vardef%description   = "Virtual potential temperature          "
      vardef%units_out     = "K               "
      vardef%convert_units = .FALSE.
      vardef%masked_field  = .FALSE.
      vardef%lsm_hinterp   = .FALSE.
      vardef%si_output_var = .FALSE.
      vardef%valid_mask    = 2
      vardef%hinterp_stagger = n_ind
      vardef%vinterp_stagger = t_ind
      vardef%minimum_value = 90.0
      vardef%maximum_value = 600.
      vardef%missing_value = rmissing
      vardef%default_value = 273.
      vardef%conversion_factor = 1.0

    ELSEIF(name .EQ. 'U       ') THEN 
      ! U-component of wind velocity
      vardef%name          = "U       "
      vardef%description   = "U-component wind velocity              "
      vardef%units_out     = "m s-1           "
      vardef%convert_units = .FALSE.
      vardef%masked_field  = .FALSE.
      vardef%lsm_hinterp   = .FALSE.
      vardef%si_output_var = .TRUE.
      vardef%valid_mask    = 2
      vardef%hinterp_stagger = n_ind
      vardef%vinterp_stagger = u_ind
      vardef%minimum_value = -200.
      vardef%maximum_value = 200.
      vardef%missing_value = rmissing
      vardef%default_value =  0.
      vardef%conversion_factor = 1.0

    ELSEIF(name .EQ. 'V       ') THEN
      ! V-component of wind velocity
      vardef%name          = "V       "
      vardef%description   = "V-component wind velocity              "
      vardef%units_out     = "m s-1           "
      vardef%convert_units = .FALSE.
      vardef%masked_field  = .FALSE.
      vardef%lsm_hinterp   = .FALSE.
      vardef%si_output_var = .TRUE.
      vardef%valid_mask    = 2
      vardef%hinterp_stagger = n_ind
      vardef%vinterp_stagger = v_ind
      vardef%minimum_value = -200.
      vardef%maximum_value = 200.
      vardef%missing_value = rmissing
      vardef%default_value =  0.
      vardef%conversion_factor = 1.0

    ELSEIF(name .EQ. 'PRESSURE') THEN
      ! Pressure
      vardef%name          = "PRESSURE"
      vardef%description   = "Pressure                               "
      vardef%units_out     = "Pa              "
      vardef%convert_units = .FALSE.
      vardef%masked_field  = .FALSE.
      vardef%lsm_hinterp   = .FALSE.
      vardef%si_output_var = .FALSE.
      vardef%valid_mask    = 2
      vardef%hinterp_stagger = n_ind
      vardef%vinterp_stagger = t_ind
      vardef%minimum_value = 0.
      vardef%maximum_value = 110000.
      vardef%missing_value = rmissing
      vardef%default_value =  101300.0
      vardef%conversion_factor = 1.0

    ELSEIF(name .EQ. 'PSFC    ') THEN
      ! Surface pressure
      vardef%name          = "PSFC    "
      vardef%description   = "Atmospheric surface pressure           "
      vardef%units_out     = "Pa              "
      vardef%convert_units = .FALSE.
      vardef%masked_field  = .FALSE.
      vardef%lsm_hinterp   = .FALSE.
      vardef%si_output_var = .FALSE.
      vardef%valid_mask    = 2
      vardef%hinterp_stagger = n_ind
      vardef%vinterp_stagger = t_ind
      vardef%minimum_value =  50000.
      vardef%maximum_value =  110000.
      vardef%missing_value = rmissing
      vardef%default_value =  101300.0
      vardef%conversion_factor = 1.0

    ELSEIF(name .EQ. 'PMSL    ') THEN
      ! Sea level pressure
      vardef%name          = "PMSL    "
      vardef%description   = "Pressure reduced to mean sea level     "
      vardef%units_out     = "Pa              "
      vardef%convert_units = .FALSE.
      vardef%masked_field  = .FALSE.
      vardef%lsm_hinterp   = .FALSE.
      vardef%si_output_var = .FALSE.
      vardef%valid_mask    = 2
      vardef%hinterp_stagger = n_ind
      vardef%vinterp_stagger = t_ind
      vardef%minimum_value = 80000.
      vardef%maximum_value = 110000.
      vardef%missing_value = rmissing
      vardef%default_value =  101300.
      vardef%conversion_factor = 1.0

    ELSEIF( (name .EQ. 'QLIQUID ') .OR. &
            (name .EQ. 'CLW     ') ) THEN
      ! Cloud liquid mixing ratio
      vardef%name          = "QLIQUID "
      vardef%description   = "Cloud water mixing ratio               "
      vardef%units_out     = "kg kg-1         "
      vardef%convert_units = .FALSE.
      vardef%masked_field  = .FALSE.
      vardef%lsm_hinterp   = .FALSE.
      vardef%si_output_var = .TRUE.
      vardef%valid_mask    = 2
      vardef%hinterp_stagger = n_ind
      vardef%vinterp_stagger = t_ind
      vardef%minimum_value = 0.
      vardef%maximum_value = 0.100  
      vardef%missing_value = rmissing
      vardef%default_value =  0.0     
      vardef%conversion_factor = 1.0
      
    ELSEIF( (name .EQ. 'QICE    ') .OR. &
            (name .EQ. 'ICE     ') ) THEN 
      ! Cloud ice mixing ratio
      vardef%name          = "QICE    "
      vardef%description   = "Cloud ice mixing ratio                 "
      vardef%units_out     = "kg kg-1         "
      vardef%convert_units = .FALSE.
      vardef%masked_field  = .FALSE.
      vardef%lsm_hinterp   = .FALSE.
      vardef%si_output_var = .TRUE.
      vardef%valid_mask    = 2
      vardef%hinterp_stagger = n_ind
      vardef%vinterp_stagger = t_ind
      vardef%minimum_value = 0.
      vardef%maximum_value = 0.100  
      vardef%missing_value = rmissing
      vardef%default_value =  0.0     
      vardef%conversion_factor = 1.0

    ELSEIF(name .EQ. 'QRAIN   ') THEN
      ! Rain mixing ratio
      vardef%name          = "QRAIN   "
      vardef%description   = "Rain water mixing ratio                "
      vardef%units_out     = "kg kg-1         "
      vardef%convert_units = .FALSE.
      vardef%masked_field  = .FALSE.
      vardef%lsm_hinterp   = .FALSE.
      vardef%si_output_var = .TRUE.
      vardef%valid_mask    = 2
      vardef%hinterp_stagger = n_ind
      vardef%vinterp_stagger = t_ind
      vardef%minimum_value = 0.
      vardef%maximum_value = 0.100  
      vardef%missing_value = rmissing
      vardef%default_value =  0.0     
      vardef%conversion_factor = 1.0 

    ELSEIF(name .EQ. 'QSNOW   ') THEN
      ! Snow mixing ratio
      vardef%name          = "QSNOW   "
      vardef%description   = "Snow mixing ratio                      "
      vardef%units_out     = "kg kg-1         "
      vardef%convert_units = .FALSE.
      vardef%masked_field  = .FALSE.
      vardef%lsm_hinterp   = .FALSE.
      vardef%si_output_var = .TRUE.
      vardef%valid_mask    = 2
      vardef%hinterp_stagger = n_ind
      vardef%vinterp_stagger = t_ind
      vardef%minimum_value = 0.
      vardef%maximum_value = 0.100  
      vardef%missing_value = rmissing
      vardef%default_value =  0.0     
      vardef%conversion_factor = 1.
 
    ELSEIF(name .EQ. 'QGRAUPEL') THEN
      ! Graupel mixing ratio
      vardef%name          = "QGRAUPEL"
      vardef%description   = "Graupel mixing ratio                   "
      vardef%units_out     = "kg kg-1         "
      vardef%convert_units = .FALSE.
      vardef%masked_field  = .FALSE.
      vardef%lsm_hinterp   = .FALSE.
      vardef%si_output_var = .TRUE.
      vardef%valid_mask    = 2
      vardef%hinterp_stagger = n_ind
      vardef%vinterp_stagger = t_ind
      vardef%minimum_value = 0.
      vardef%maximum_value = 0.100  
      vardef%missing_value = rmissing
      vardef%default_value =  0.0     
      vardef%conversion_factor = 1.
 
    ELSEIF (name .EQ. 'QNI     ') THEN
      ! Ice number concentration
      vardef%name          = "QNI     "
      vardef%description   = "Ice crystal number concentration       "
      vardef%units_out     = "m-3             "
      vardef%convert_units = .FALSE.
      vardef%masked_field  = .FALSE.
      vardef%lsm_hinterp   = .FALSE.
      vardef%si_output_var = .TRUE.
      vardef%valid_mask    = 2
      vardef%hinterp_stagger = n_ind
      vardef%vinterp_stagger = t_ind
      vardef%minimum_value = 0.
      vardef%maximum_value = 1.e7
      vardef%missing_value = rmissing
      vardef%default_value =  0.0     
      vardef%conversion_factor = 1.

    ELSEIF (name .EQ. 'QVG     ') THEN
      ! Water vapor mixing ratio at ground
      vardef%name          = "QVG     "
      vardef%description   = "Water vapor mixing ratio at ground     "
      vardef%units_out     = "kg kg-1         "
      vardef%convert_units = .FALSE.
      vardef%masked_field  = .FALSE.
      vardef%lsm_hinterp   = .FALSE.
      vardef%si_output_var = .TRUE.
      vardef%valid_mask    = 2
      vardef%hinterp_stagger = n_ind
      vardef%vinterp_stagger = t_ind
      vardef%minimum_value = 0.
      vardef%maximum_value = 0.100 
      vardef%missing_value = rmissing
      vardef%default_value =  0.0     
      vardef%conversion_factor = 1.

    ELSE 
      status = VAR_UNKNOWN
      PRINT *, 'Not found in atmsopheric list'

    ENDIF      
    
    IF (status .EQ. VAR_UNKNOWN) THEN
      PRINT *, 'Checking for NAME in LSM list: ', name 
      
      ! reset status
      status = VAR_KNOWN 
      IF (name .EQ. 'SOILHGT ') THEN
        ! Terrain height of ground.  Generally from the
        ! input model data, with interpolation using
        ! LSM_HINTERP_METHOD, so that it can be used
        ! to adjust soil temperatures to the WRF grid.
        vardef%name          = "SOILHGT "
        vardef%description   = "Terrain height of input LSM data       "
        vardef%units_out     = "m               "
        vardef%convert_units = .FALSE.
        vardef%masked_field  = .TRUE.
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .TRUE.
        vardef%valid_mask    = 1
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = -1000.
        vardef%maximum_value = 20000.
        vardef%missing_value = rmissing
        vardef%default_value =  0.0     
        vardef%conversion_factor = 1.

      ELSEIF ( (name .EQ. 'WEASD   ') .OR. &
               (name .EQ. 'SNOW    ') .OR. &
               (name .EQ. 'SNOWRUC ') ) THEN        
        ! Water equivalent snow depth
        vardef%name          = "SNOW    "
        vardef%description   = "Water equivalent snow depth            "
        vardef%units_out     = "kg m-2          "
        IF (name .NE. 'SNOWRUC ') THEN
          vardef%convert_units = .FALSE.
          vardef%conversion_factor = 1.0
        ELSE
          vardef%convert_units = .TRUE.
          vardef%conversion_factor = 1000. ! Convert from meters
        ENDIF
        vardef%masked_field  = .TRUE.
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .TRUE.
        vardef%valid_mask    = 1
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = 0.
        vardef%maximum_value = 100000.
        vardef%missing_value = rmissing
        vardef%default_value =  0.0     

      ELSEIF ( (name .EQ. 'SNOWH   ') .OR. &
               (name .EQ. 'SNOWDEP ')) THEN
        ! Physical snow depth
        vardef%name          = "SNOWH   "
        vardef%description   = "Physical depth of snow cover           "
        vardef%units_out     = "m               "
        vardef%convert_units = .FALSE.
        vardef%masked_field  = .TRUE.
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .TRUE.
        vardef%valid_mask    = 1
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = 0.
        vardef%maximum_value = 100.  
        vardef%missing_value = rmissing
        vardef%default_value =  0.0     
        vardef%conversion_factor = 1.
       
      ELSEIF ( (name .EQ. 'SKINTEMP') .OR. &
               (name .EQ. 'TSKIN   ') ) THEN
        ! Skin temperature
        vardef%name          = "SKINTEMP"
        vardef%description   = "Skin temperature                       "
        vardef%units_out     = "K               "
        vardef%convert_units = .FALSE.
        vardef%masked_field  = .TRUE.
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .TRUE.
        vardef%valid_mask    = 2
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = 100.
        vardef%maximum_value = 400.  
        vardef%missing_value = rmissing
        vardef%default_value =  0.0     
        vardef%conversion_factor = 1.

      ELSEIF (name .EQ. 'LANDSEA ') THEN
        ! Background land mask
        vardef%name          = "LANDSEA "
        vardef%description   = "Land mask for input LSM data           "
        vardef%units_out     = "                "
        vardef%convert_units = .FALSE.
        vardef%masked_field  = .TRUE. 
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .FALSE.
        vardef%valid_mask    = 2
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = 0.
        vardef%maximum_value = 1.    
        vardef%missing_value = rmissing
        vardef%default_value =  1.0     
        vardef%conversion_factor = 1.

      ELSEIF (name .EQ. 'SEAICE  ') THEN
        ! Sea ice flag
        vardef%name          = "SEAICE  "
        vardef%description   = "Flag indicating ice covered water      "
        vardef%units_out     = "                "
        vardef%convert_units = .FALSE.
        vardef%masked_field  = .TRUE. 
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .TRUE.
        vardef%valid_mask    = 0
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = 0.
        vardef%maximum_value = 1.    
        vardef%missing_value = rmissing
        vardef%default_value =  0.0     
        vardef%conversion_factor = 1.

      ELSEIF( (name .EQ. 'CANOPY  ') .OR. &
              (name .EQ. 'CANWAT  ') ) THEN
        ! Canopy water content.  "CANOPY" indicates
        ! it is from old AGRMET dataset and needs conversion
        ! from m to kg m-2
        vardef%name          = "CANWAT  "
        vardef%description   = "Canopy water content                   "
        vardef%units_out     = "kg m-2          "
        IF (name .EQ. 'CANOPY  ') THEN
          vardef%convert_units = .TRUE.
          vardef%conversion_factor = 1000.
        ELSE
          vardef%convert_units = .FALSE.
          vardef%conversion_factor = 1.0
        ENDIF
        vardef%masked_field  = .TRUE. 
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .TRUE.
        vardef%valid_mask    = 1
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = 0.
        vardef%maximum_value = 100.    
        vardef%missing_value = rmissing
        vardef%default_value =  0.0     
        vardef%conversion_factor = 1.

      ELSEIF (name .EQ. 'SOILCATZ') THEN
        ! Zobler soil categorization
        vardef%name          = "SOILCATZ"
        vardef%description   = "Zobler dominant soil type category     "
        vardef%units_out     = "                "
        vardef%convert_units = .FALSE.
        vardef%masked_field  = .TRUE. 
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .TRUE.
        vardef%valid_mask    = 1
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = 0.
        vardef%maximum_value = 16.    
        vardef%missing_value = rmissing
        vardef%default_value =  0.0     
        vardef%conversion_factor = 1.
 
      ELSEIF (name .EQ. 'SOILCATS') THEN
        ! Statsgo soil categorization
        vardef%name          = "SOILCATS"
        vardef%description   = "Statsgo dominant soil type category    "
        vardef%units_out     = "                "
        vardef%convert_units = .FALSE.
        vardef%masked_field  = .TRUE. 
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .TRUE.
        vardef%valid_mask    = 1
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = 0.
        vardef%maximum_value = 16.    
        vardef%missing_value = rmissing
        vardef%default_value =  0.0     
        vardef%conversion_factor = 1.

      ELSEIF (name .EQ. 'VEGCATU ') THEN
        ! USGS 24-cat dominant category
        vardef%name          = "VEGCATU "
        vardef%description   = "USGS 24-cat dominant surface type      "
        vardef%units_out     = "                "
        vardef%convert_units = .FALSE.
        vardef%masked_field  = .TRUE. 
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .TRUE.
        vardef%valid_mask    = 2
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = 1.
        vardef%maximum_value = 24.    
        vardef%missing_value = rmissing
        vardef%default_value =  0.0     
        vardef%conversion_factor = 1.

      ELSEIF (name .EQ. 'VEGCATS ') THEN
        ! SiB 16-cat dominant veg category
        vardef%name          = "VEGCATS "
        vardef%description   = "SiB 16-cat dominant surface type       "
        vardef%units_out     = "                "
        vardef%convert_units = .FALSE.
        vardef%masked_field  = .TRUE. 
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .TRUE.
        vardef%valid_mask    = 2
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = 1.
        vardef%maximum_value = 16.    
        vardef%missing_value = rmissing
        vardef%default_value =  0.     
        vardef%conversion_factor = 1.

      ELSEIF(name .EQ. 'RHOSN   ') THEN
        ! Density of snowpack
        vardef%name          = "RHOSN   "
        vardef%description   = "Density of snow layer                  "
        vardef%units_out     = "kg m-3          "
        vardef%convert_units = .FALSE.
        vardef%masked_field  = .TRUE. 
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .TRUE.
        vardef%valid_mask    = 1
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = 0.
        vardef%maximum_value = 10000. 
        vardef%missing_value = rmissing
        vardef%default_value =  0.     
        vardef%conversion_factor = 1.

      ELSEIF ( (name .EQ. 'SST     ') .OR. &
               (name .EQ. 'TSEASFC ') ) THEN
        ! Sea surface temperature
        vardef%name          = "SST     "
        vardef%description   = "Sea surface temperature                "
        vardef%units_out     = "K               "
        vardef%convert_units = .FALSE.
        vardef%masked_field  = .TRUE. 
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .TRUE.
        vardef%valid_mask    = 0
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = 265.
        vardef%maximum_value = 350.   
        vardef%missing_value = rmissing
        vardef%default_value =  290.   
        vardef%conversion_factor = 1.

      ELSEIF (name .EQ. 'TSNOW   ') THEN
        ! Temperature of snow layer
        vardef%name          = "TSNOW   "
        vardef%description   = "Temperature of snow layer              "
        vardef%units_out     = "K               "
        vardef%convert_units = .FALSE.
        vardef%masked_field  = .TRUE. 
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .TRUE.
        vardef%valid_mask    = 1
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = 100.
        vardef%maximum_value = 300.   
        vardef%missing_value = rmissing
        vardef%default_value =  273.   
        vardef%conversion_factor = 1.

      ! The rest of these are for LSM field where
      ! layer indicator is part of the name.  For now, we
      ! will take it on faith that if the characters being
      ! matched results in a true value, then the remaining 
      ! characters are correct
      ELSEIF (name(1:2) .EQ. 'ST') THEN
        ! Soil temperature of a layer indicated as
        ! STtttbbb, where ttt is top level of layer in cm
        ! and bbb is bottom level of layer in cm
        vardef%name          = name
        vardef%description   = "Temperature of soil layer              "
        vardef%units_out     = "K               "
        vardef%convert_units = .FALSE.
        vardef%masked_field  = .TRUE. 
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .TRUE.
        vardef%valid_mask    = 1
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = 100.
        vardef%maximum_value = 400.   
        vardef%missing_value = rmissing
        vardef%default_value =  290.   
        vardef%conversion_factor = 1.

      ELSEIF(name(1:5) .EQ. 'SOILT') THEN
        ! Temperature of a soil level, indicated
        ! as SOILTlll, where lll is the level in cm
        vardef%name          = name
        vardef%description   = "Temperature of soil level              "
        vardef%units_out     = "K               "
        vardef%convert_units = .FALSE.
        vardef%masked_field  = .TRUE. 
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .TRUE.
        vardef%valid_mask    = 1
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = 100.
        vardef%maximum_value = 400.   
        vardef%missing_value = rmissing
        vardef%default_value =  290.   
        vardef%conversion_factor = 1.

      ELSEIF(name(1:2) .EQ. 'SM'   ) THEN
        ! Total soil moisture content of a layer
        !  Variable name contains layer boundaries
        !  in cm as SMtttbbb
        vardef%name          = name
        vardef%description   = "Total volumetric soil moisture (layer) "
        vardef%units_out     = "                "
        vardef%convert_units = .FALSE.
        vardef%masked_field  = .TRUE. 
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .TRUE.
        vardef%valid_mask    = 1
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = 0.
        vardef%maximum_value = 1. 
        vardef%missing_value = rmissing
        vardef%default_value =  0.0    
        vardef%conversion_factor = 1.
         
      ELSEIF(name(1:5) .EQ. 'SOILM') THEN
        ! Total soil moisture content of a level
        !  Variable name contains level value      
        !  in cm as SOILMlll
        vardef%name          = name
        vardef%description   = "Total volumetric soil moisture (level) "
        vardef%units_out     = "                "
        vardef%convert_units = .FALSE.
        vardef%masked_field  = .TRUE. 
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .TRUE.
        vardef%valid_mask    = 1
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = 0.
        vardef%maximum_value = 1. 
        vardef%missing_value = rmissing
        vardef%default_value =  0.0    
        vardef%conversion_factor = 1.

      ELSEIF(name(1:2) .EQ. 'SW') THEN
        ! Total liquid soil moisture content of a layer
        !  Variable name contains layer boundaries 
        !  in cm as SWtttbbb

        vardef%name          = name
        vardef%description   = "Liquid volumetric soil moisture (layer)"
        vardef%units_out     = "                "
        vardef%convert_units = .FALSE.
        vardef%masked_field  = .TRUE. 
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .TRUE.
        vardef%valid_mask    = 1
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = 0.
        vardef%maximum_value = 1. 
        vardef%missing_value = rmissing
        vardef%default_value =  0.0    
        vardef%conversion_factor = 1.

      ELSEIF(name(1:5) .EQ. 'SOILW') THEN
        ! Total soil moisture content of a level
        !  Variable name contains level value      
        !  in cm as SOILWlll
        vardef%name          = name
        vardef%description   = "Liquid volumetric soil moisture (level)"
        vardef%units_out     = "                "
        vardef%convert_units = .FALSE.
        vardef%masked_field  = .TRUE. 
        vardef%lsm_hinterp   = .TRUE.
        vardef%si_output_var = .TRUE.
        vardef%valid_mask    = 1
        vardef%hinterp_stagger = t_ind
        vardef%vinterp_stagger = t_ind
        vardef%minimum_value = 0.
        vardef%maximum_value = 1. 
        vardef%missing_value = rmissing
        vardef%default_value =  0.0    
        vardef%conversion_factor = 1.

      ELSE
        status = VAR_UNKNOWN
      ENDIF

    ENDIF
    IF (status .EQ. VAR_UNKNOWN) THEN
      PRINT *, 'WARNING in ', routine_name
      PRINT *, 'Variable name not recognized: ', name
    ENDIF
    RETURN

  END SUBROUTINE return_variable_def

END MODULE wrfsi_variables
