!dis   
!dis    Open Source License/Disclaimer, Forecast Systems Laboratory
!dis    NOAA/OAR/FSL, 325 Broadway Boulder, CO 80305
!dis    
!dis    This software is distributed under the Open Source Definition,
!dis    which may be found at http://www.opensource.org/osd.html.
!dis    
!dis    In particular, redistribution and use in source and binary forms,
!dis    with or without modification, are permitted provided that the
!dis    following conditions are met:
!dis    
!dis    - Redistributions of source code must retain this notice, this
!dis    list of conditions and the following disclaimer.
!dis    
!dis    - Redistributions in binary form must provide access to this
!dis    notice, this list of conditions and the following disclaimer, and
!dis    the underlying source code.
!dis    
!dis    - All modifications to this software must be clearly documented,
!dis    and are solely the responsibility of the agent making the
!dis    modifications.
!dis    
!dis    - If significant modifications or enhancements are made to this
!dis    software, the FSL Software Policy Manager
!dis    (softwaremgr@fsl.noaa.gov) should be notified.
!dis    
!dis    THIS SOFTWARE AND ITS DOCUMENTATION ARE IN THE PUBLIC DOMAIN
!dis    AND ARE FURNISHED "AS IS."  THE AUTHORS, THE UNITED STATES
!dis    GOVERNMENT, ITS INSTRUMENTALITIES, OFFICERS, EMPLOYEES, AND
!dis    AGENTS MAKE NO WARRANTY, EXPRESS OR IMPLIED, AS TO THE USEFULNESS
!dis    OF THE SOFTWARE AND DOCUMENTATION FOR ANY PURPOSE.  THEY ASSUME
!dis    NO RESPONSIBILITY (1) FOR THE USE OF THE SOFTWARE AND
!dis    DOCUMENTATION; OR (2) TO PROVIDE TECHNICAL SUPPORT TO USERS.
!dis   
!dis 

! Module containing routines to vertically interpolate pressure level data
! to the Zeta (scaled height) coordinate for WRF, where zeta is defined as:
!
!     Zeta =    Zt *  (Z - Zg)
!                     --------
!                     (Zt - Zg)
!
! where Zt is the model top, Z is the atmospheric height, and Zg is the terrain
! height.
!
! This version assumes that the incoming atmospheric data is on non-staggered grids, and that
! any 2d LSM fields (e.g., Soil Moisture, Soil Temp, Snow cover, etc.) are already staggered
! to the mass grid (t_ind in the metadata).
!
! HISTORY
! =======
! Nov 2000 - Original Release - Brent Shaw, NOAA/FSL
! May 2001 - Numerous mods - Brent Shaw, NOAA/FSL
!          -- Changed to do all vertical interpolation on non-staggered grid
!          -- Consitent with new version of hinterp

MODULE vinterp_p2z

  USE wrfsi_io
  USE wrf_metadata
  USE vinterp_domain
  USE vinterp_setup
  USE date_pack
  USE grid_utils
  USE physical_constants
  USE diagnostic_vars
  USE wrfsi_static

  ! The following variables are going to be used to set up
  ! the vertical interpolation.  The far right dimension in each
  ! one is to account for the staggered grid, as we will need heights
  ! for the U and V components of wind when doing the vertical interp. 
  ! We will also need map_factor for doing the balance portion, so it
  ! is in here now.

  PRIVATE
  REAL, ALLOCATABLE		:: dzetadz(:,:)
  REAL, ALLOCATABLE		:: dzdzeta(:,:)
  REAL, ALLOCATABLE             :: soilhgt(:,:)   ! Source data terrain height
  REAL, ALLOCATABLE             :: height(:,:,:)
  INTEGER, ALLOCATABLE          :: trap_top(:,:,:)
  INTEGER, ALLOCATABLE          :: trap_bot(:,:,:)
  REAL, ALLOCATABLE		:: press_zeta(:,:,:)
  REAL, ALLOCATABLE             :: press_sfc(:,:)
  REAL, ALLOCATABLE             :: press_msl(:,:)
  REAL, ALLOCATABLE		:: rho_zeta(:,:,:)
  REAL, ALLOCATABLE		:: temp_zeta(:,:,:)
  REAL, ALLOCATABLE		:: uwind_zeta(:,:,:)
  REAL, ALLOCATABLE		:: vwind_zeta(:,:,:)
  REAL, ALLOCATABLE             :: rh_zeta(:,:,:)
  REAL, ALLOCATABLE 		:: qv_zeta(:,:,:)
  REAL, ALLOCATABLE             :: theta_zeta(:,:,:)
  REAL, ALLOCATABLE		:: dum3d(:,:,:)
  REAL, ALLOCATABLE             :: dum2d(:,:)
  REAL, ALLOCATABLE             :: dum3ds(:,:,:)
  REAL, ALLOCATABLE             :: dum2ds(:,:)
  REAL, ALLOCATABLE		:: weight_top(:,:,:)
  REAL 				:: ztop
  CHARACTER(LEN=8)              :: processed_var_list(100)
  INTEGER                       :: num_processed
  TYPE(wrfvar_metadata)         :: var_meta_out

  PUBLIC p2z_driver
                                   
CONTAINS

  SUBROUTINE p2z_driver
    PRINT '(A)', 'P2Z_DRIVER: Calling p2z_setup'
    CALL p2z_setup
    PRINT '(A)', 'P2Z_DRIVER: Calling p2z_compute_weights'
    CALL p2z_compute_weights
    PRINT '(A)', 'P2Z_DRIVER: Calling p2z_interp_state_variables'
    CALL p2z_interp_state_variables
    PRINT '(A)', 'P2Z_DRIVER: Calling p2z_make_wrf_variables'
    CALL p2z_make_wrf_variables
    PRINT '(A)', 'P2z_DRIVER: Calling p2z_process_other_variables'
    CALL p2z_process_other_variables
    PRINT '(A)', 'P2Z_DRIVER: Calling p2z_cleanup'
    CALL p2z_cleanup
  
  END SUBROUTINE p2z_driver
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  SUBROUTINE p2z_setup

    ! This routine sets up for the vertical pressure to zeta interpolation
    ! by getting all of the mandatory variables

    IMPLICIT NONE
    INTEGER :: status, i, j
    REAL :: deltax, deltay
    REAL, ALLOCATABLE :: hgt_column1(:), hgt_column2(:)
    LOGICAL :: reuse_domain
    reuse_domain = .false.

    num_processed = 0
    processed_var_list(:) = '        '
    nx = dom_meta%xdim
    ny = dom_meta%ydim
    input_nz = dom_meta%zdim 
    nsh = global_meta%num_stagger_xy
    IF (setup_info%vstagger_output) THEN 
      nsv_out = 2
    ELSE
      nsv_out = 1
    ENDIF
    nsv_in = global_meta%num_stagger_z 
    ztop = output_levels(output_nz, zstag_full_index)

    ! Get deltax
    
    deltax = dom_meta%delta_x 
    deltay = dom_meta%delta_y

    ! Initialize the output file (opens it and outputs the
    ! domain metadata)

    dom_out = dom_meta
    dom_out%zdim = output_nz
    dom_out%top_level = ztop
    CALL init_new_output_file(dom_out,setup_info%output_prefix,setup_info%current_date)

    ! Check and see if this exact domain was previously processed, because
    ! if it is we can save a little I/O and processing time by re-using the
    ! static data
     
    IF ( (dom_meta%id .NE. dom_old%id).OR. &
         (dom_meta%xdim .NE. dom_old%xdim).OR. &
         (dom_meta%ydim .NE. dom_old%ydim).OR. &
         (dom_meta%zdim .NE. dom_old%zdim)) THEN 

      IF (ALLOCATED(dzetadz)) DEALLOCATE(dzetadz)
      IF (ALLOCATED(dzdzeta)) DEALLOCATE(dzdzeta)

      ! Compute the dz/dzeta array and its inverse.  

      ALLOCATE(dzdzeta(nx,ny))
      ALLOCATE(dzetadz(nx,ny))
      dzetadz(:,:) = ztop / (ztop - terrain_hgt_t(:,:))
      dzdzeta(:,:) = 1./dzetadz(:,:)

    ELSE
      reuse_domain = .true.
    ENDIF

    ! We also need to get the input source terrain, usually identified
    ! as TOPOSOIL.

    IF ((ALLOCATED(soilhgt)).AND.(.NOT.reuse_domain)) THEN
      DEALLOCATE(soilhgt)
      ALLOCATE(soilhgt(nx,ny))
    ELSE IF (.NOT.ALLOCATED(soilhgt)) THEN
      ALLOCATE(soilhgt(nx,ny))
    ENDIF
    soilhgt(:,:) = 0.
    PRINT '(A,A)', 'P2Z_SETUP: Getting source terrain using ID: ', soilhgt_id
    CALL get_variable(setup_info%input_prefix, soilhgt_id,dom_meta%id, &
                      setup_info%current_date,status)
    IF (status.NE.0) THEN
      PRINT *,'Error: No variable named ', soilhgt_id, ' found!'
      PRINT *,'  Substituting WRF terrain for soil height'
      soilhgt(:,:)=terrain_hgt_n(:,:)
    ELSE
      soilhgt(:,:)=real_array(:,:,1,1,1)
      DEALLOCATE(real_array)
    ENDIF                 
   
    num_processed = num_processed + 1
    processed_var_list(num_processed) = soilhgt_id

    ! Now we have to have the height field on pressure surfaces for 
    ! all staggers
    PRINT '(A,A)', 'P2Z_SETUP: Getting heights using ID: ', height_id
    CALL get_variable(setup_info%input_prefix, height_id,dom_meta%id, &
                      setup_info%current_date,status)
    IF (status.NE.0) THEN
      PRINT *,'Error: No variable named ', height_id, ' found!'
      STOP 'no_height_data'
    ENDIF
    ! Make sure this data is from the newer version of hinterp, which keeps
    ! all met variables on the non-staggered grid
    IF (var_info%h_stagger_index .NE. n_ind) THEN
      PRINT *, 'Height field appears to be staggered.  This version of vinterp'
      PRINT *, 'expects all atmospheric fields to be on non-staggered grid.  Please'
      PRINT *, 're-process your data with the newer version of hinterp and try again.'
      STOP 'P2Z_SETUP'
    ENDIF
    IF ((ALLOCATED(height)).AND.(.NOT.reuse_domain)) THEN
      DEALLOCATE(height)
      ALLOCATE(height(nx,ny,input_nz))
    ELSE IF (.NOT.ALLOCATED(height)) THEN
      ALLOCATE(height(nx,ny,input_nz))
    ENDIF
    height(:,:,:)=real_array(:,:,:,1,1) 
    DEALLOCATE(real_array)                
    
    num_processed = num_processed+1
    processed_var_list(num_processed) = height_id
    ! Replace the bottom level of the height array which is artifically
    ! set to the same value as the lowest mandatory pressure level to
    ! the soil height and then we can deallocate the soilhgt
    height(:,:,1)=soilhgt(:,:)
    DEALLOCATE(soilhgt)
    num_processed = num_processed + 1
    processed_var_list(num_processed) = height_id

    ! Get the input pressure levels

    IF ( (ALLOCATED(input_levels)).AND.(.NOT.reuse_domain)) THEN
      DEALLOCATE(input_levels)
      ALLOCATE(input_levels(input_nz,nsv_in))
    ELSE IF (.NOT.ALLOCATED(input_levels)) THEN
      ALLOCATE(input_levels(input_nz,nsv_in))
    ENDIF
    PRINT '(A,A)','P2Z_SETUP: Getting pressure levels using ID:',pressure_id
    CALL get_variable(setup_info%input_prefix,pressure_id,dom_meta%id, &
                      setup_info%current_date,status)
    IF (status.NE.0) THEN
      PRINT *,'Error: No variable named ', pressure_id, ' found!'
      STOP 'no_pressure_data'
    ENDIF
    input_levels(:,:) =real_array(:,:,1,1,1)
    num_processed = num_processed + 1 
    processed_var_list(num_processed) = pressure_id
    DEALLOCATE(real_array)
 
    ! Get the MSLP data
    ALLOCATE(press_msl(nx,ny))
    CALL get_variable(setup_info%input_prefix,press_msl_id,dom_meta%id, &
                      setup_info%current_date,status)
    IF (status.NE.0) THEN
      PRINT '(A)', ' MSLP data not found using ' //press_msl_id
      STOP 'no_mslp_data'
    ENDIF
    press_msl = real_array(:,:,1,1,1)
    DEALLOCATE(real_array)
    num_processed = num_processed+1
    processed_var_list(num_processed) = press_msl_id

    ! Output static variables specific to zeta coordinate system
    CALL output_zeta_static  

    ! Save current domain info into dom_old so we can check for 
    ! same domain next time around.

    dom_old = dom_meta

      
    RETURN
  END SUBROUTINE p2z_setup
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  SUBROUTINE p2z_compute_weights
  ! This subroutine uses the heights corresponding to the input pressure
  ! level data and surface data, the terrain height, and the output zeta
  ! levels to find the indices in each input column that trap each zeta
  ! level in each output column and computes weights to use in the interpolation
  ! between those two levels. It has logic in it to account for the cases
  ! where the height of the output zeta level is below the terrain height of the
  ! input FG model.  The pressure of each zeta level is also computed during
  ! this routine by interpolating the log of pressure linearly.  In the cases
  ! where the surface of the FG model is one of the bounding levels, the
  ! MSLP field is used and the weights and heights are adjusted accordingly.

    INTEGER :: i,j,ki,ko,kp
    REAL :: zeta1,zeta2,zeta1p,zeta2p,zeta1p_value,zeta2p_value,weight_top_p,&
            deltaz
    LOGICAL :: found_trap

    ! Allocate  some variables. 
 
    ALLOCATE(trap_top (nx,ny,output_nz))
    ALLOCATE(trap_bot (nx,ny,output_nz))
    ALLOCATE(weight_top(nx,ny,output_nz))
    ALLOCATE(press_zeta (nx,ny,output_nz))
    trap_top(:,:,:) = -2   ! Use as a missing flag
    trap_bot(:,:,:) = -2
    weight_top(:,:,:) = -2.   

    ! Loop over the ouput domain and find the "trapping" k-indices in each
    ! column that bound the desired output layer.  We only need to know 
    ! one of the traps, so lets use the top trap.

    find_traps_koloop: DO ko=1,output_nz  ! Loop through output column
      find_traps_jloop: DO j = 1, ny       ! Loop over N-S direction
        find_traps_iloop: DO i = 1,nx      ! Loop over E-W direction
          
          found_trap = .false.

          ! Now loop through the input pressure levels, skipping level 1 which
          ! is the surface data, which is typically "derived" anyway 
          ! when using NCEP model output (I originally tried using this by
          ! simple downward extrapolation when the surface layer was higher 
          ! than the WRF zeta level, but in the case of Eta, this did not 
          ! give good results.

          find_traps_kiloop: DO ki=2,input_nz 

            ! Compute zeta2 at this input point
            zeta2 = ztop*( (height(i,j,ki)-terrain_hgt_n(i,j)) / &
                            (ztop - terrain_hgt_n(i,j)) )

            ! See if this zeta is above the desired data.

            IF (zeta2 .GE. output_levels(ko,zstag_half_index)) THEN
              trap_top(i,j,ko) = ki
              trap_bot(i,j,ko) = ki - 1
              found_trap = .true.
              IF (ki.GT.2)THEN
                ! This is the easy case because our desired level is bounded
                ! by two pressure levels

                zeta1 =  ztop*( (height(i,j,ki-1)-terrain_hgt_n(i,j)) / &
                        (ztop - terrain_hgt_n(i,j)) )

              ELSE

                ! The lowest pressure level is higher than our desired zeta, which
                ! presents some complications for trapping below our desired zeta

                ! PRINT'(A,I3,A,3I4)','Zeta level ',ko, & 
                !  'is below height of lowest press level at i/j/m:',i,j,m
                ! Compute the zeta of the level below our desired zeta (in this
                ! case we are computing the zeta value of the FG model surface
                ! height)
                zeta1 =  ztop*( (height(i,j,1)-terrain_hgt_n(i,j)) / &
                        (ztop - terrain_hgt_n(i,j)) )
                ! Hopefully, the FG surface data provides the lower bound, but
                ! if not we will make the surface the top boundary

                IF (zeta1.GE.output_levels(ko,zstag_half_index))THEN
                  ! Well, even the FG surface height is above our zeta level, 
                  ! so reset the top bound, and its zeta bound to give
                  ! it 100% weighting.  The interpolation code will have 
                  ! some exception handling to extrapolate some of the variables
     
                  !PRINT '(A)', 'Zeta level is also below the FG model surface.'
                  trap_top(i,j,ko) = 1
                  ! The next line gives the values at trap_top 100% weighting
                  ! by setting the zeta value of trap top = desired zeta
                  zeta2 = output_levels(ko,zstag_half_index)
                  trap_bot(i,j,ko) = 1
                ENDIF
              ENDIF

              ! Go ahead and interpolate pressure
                 
              zeta2p = zeta2
              zeta1p = zeta1 
              IF (trap_top(i,j,ko).GT.2)THEN
                zeta2p_value = input_levels(ki,1)
                zeta1p_value = input_levels(ki-1,1)
              ELSE 
                ! Set zeta2p to zeta of lowest pressure level 
                zeta2p =  ztop*( (height(i,j,2)-terrain_hgt_n(i,j)) / &
                         (ztop - terrain_hgt_n(i,j)) )

                IF (terrain_hgt_n(i,j).GE.0) THEN
                  ! Zeta1p should be reset to zeta at sea level
                  zeta1p = ztop*( (0.-terrain_hgt_n(i,j)) / &
                          (ztop - terrain_hgt_n(i,j)) )
                  zeta2p_value =  input_levels(ki,1)
                  zeta1p_value = press_msl(i,j)
                ELSE
                  !PRINT '(A,3I4)', 'Below sea-level and lowest p level at i/j/m = ',&
                  !      i,j,m
                  
                  ! This should be a rare occurrence, and for now we will simply
                  ! extrapolate down from either the lowest pressure level
                  ! (if it is below sea-level) or the MSLP using a pressure 
                  ! lapse rate of 12.5 Pa/m (Wallace and Hobbs, p. 60)

                  zeta2p = output_levels(ko,zstag_half_index) ! to give 100% weighting to
                                                              ! value we will compute
                  IF (height(i,j,2).LT.0) THEN
                    deltaz = height(i,j,2) - terrain_hgt_n(i,j)
                    zeta2p_value = input_levels(2,1)
                  ELSE 
                    deltaz = -terrain_hgt_n(i,j)
                    zeta2p_value = press_msl(i,j)
                  ENDIF
                  zeta2p_value = zeta2p_value + deltaz * dPdz
                  zeta1p_value = 1050. ! Does not matter!
                  zeta1p = -50 ! Does not matter!
                ENDIF
              ENDIF 
              IF (zeta2p .EQ.zeta1p) THEN 
                 weight_top_p = 1.
              ELSE
                weight_top_p = 1. - &
                  ABS ((zeta2p-output_levels(ko,zstag_half_index))/(zeta2p-zeta1p))
              ENDIF
              press_zeta(i,j,ko) =EXP ( weight_top_p*ALOG(zeta2p_value) + &
                                      (1.-weight_top_p)*ALOG(zeta1p_value) ) 
                  
              IF (zeta2 .EQ. zeta1) THEN
                weight_top(i,j,ko) = 1.0
              ELSE                          
                weight_top(i,j,ko) = 1. - &
                  ABS ((zeta2-output_levels(ko,zstag_half_index))/(zeta2-zeta1))
              ENDIF
              EXIT find_traps_kiloop
            ENDIF
          ENDDO find_traps_kiloop
          IF (.NOT.found_trap)THEN
            PRINT '(A)','P2Z_COMPUTE_WEIGHTS: No trap found! ' // &
                    ' Model top may be too high!'
            PRINT '(A,3I6)', 'I/J/KO: ', i,j,ko
            PRINT '(A,F10.2)', 'Zeta2 = ',zeta2
            PRINT '(A,F10.2)', 'Model top = ',ztop
            STOP 'no_trap_found'
          ENDIF
        ENDDO find_traps_iloop
      ENDDO find_traps_jloop
    ENDDO find_traps_koloop
   
   IF (setup_info%verbose) THEN
      PRINT '(A)', 'P2Z_COMPUTE_WEIGHTS: Diagnostics from domain center: '
      PRINT '(A)', '----------------------------------------'
      PRINT '(A)', ' KO     ZETA   TRAPT  WEIGHTT PRESSURE  '
      PRINT '(A)', '----------------------------------------'
      diag_print_ko: DO ko = 1,output_nz
        PRINT '(I3,F12.1,I5,2F10.3)', ko, output_levels(ko,zstag_half_index),&
           trap_top(nx/2,ny/2,ko),weight_top(nx/2,ny/2,ko),press_zeta(nx/2,ny/2,ko)
      ENDDO diag_print_ko
      PRINT '(A,F10.2)', 'TERRAIN HEIGHT: ', terrain_hgt_n(nx/2,ny/2)
      PRINT '(A,F10.2)', 'SOURCE TERRHGT: ', height(nx/2,ny/2,1)
      PRINT '(A,F10.2)', 'ZTOP: ', ztop
      PRINT '(A)', '---------------------------'
      PRINT '(A)', ' KI     HEIGHT      ZETA'
      PRINT '(A)', '---------------------------'
      diag_print_ki: DO ki = 1,input_nz
        zeta =  ztop*( (height(nx/2,ny/2,ki)-terrain_hgt_n(nx/2,ny/2)) / &
                            (ztop - terrain_hgt_n(nx/2,ny/2)) )
        PRINT '(I3,F12.2,F12.2)', ki, height(nx/2,ny/2,ki),zeta
      ENDDO diag_print_ki
    ENDIF

    ! Finally, since we have now computed the pressure values on the 
    ! zeta surfaces, lets write those out as well.  Because pressure
    ! was computed on the non-staggered grid, but WRF needs pressure on the
    ! mass grid, we need to stagger it first.  ONLY STAGGER IF 
    ! OUTPUT_VARS <=1 (WRF R-K support)
 
    IF (setup_info%output_vars .LE. 2) THEN 
      ALLOCATE(dum3ds(nx,ny,output_nz))
      CALL arakawa_c_n2t(press_zeta, nx, ny, output_nz, dum3ds)
      press_zeta = dum3ds
      DEALLOCATE(dum3ds)
      var_meta_out%h_stagger_index = t_ind
    ELSE
      var_meta_out%h_stagger_index = n_ind
    ENDIF
    var_meta_out%name = pressure_id   
    var_meta_out%units = 'Pa'
    var_meta_out%description = 'Pressure on zeta point'
    var_meta_out%domain_id = dom_meta%id
    var_meta_out%ndim = 3
    var_meta_out%dim_val(1) = nx
    var_meta_out%dim_val(2) = ny
    var_meta_out%dim_val(3) = output_nz
    var_meta_out%dim_desc(1) = 'E-W'
    var_meta_out%dim_desc(2) = 'N-S'
    var_meta_out%dim_desc(3) = 'VERT'
    var_meta_out%start_index(1) = 1
    var_meta_out%start_index(2) = 1
    var_meta_out%start_index(3) = 1
    var_meta_out%stop_index(1) = nx
    var_meta_out%stop_index(2) = ny
    var_meta_out%stop_index(3) = output_nz
    var_meta_out%v_stagger_index = zstag_half_index
    var_meta_out%array_order = '+X+Y+Z'
    var_meta_out%field_type = 'REAL'
    var_meta_out%field_source_prog = 'SI'
    var_meta_out%source_desc = 'Vertical interpolation by SI'
    var_meta_out%field_time_type = 'INSTANT'
    var_meta_out%vt_date_start = dom_meta%vt_date
    var_meta_out%vt_time_start = dom_meta%vt_time
    var_meta_out%vt_date_stop = dom_meta%vt_date
    var_meta_out%vt_time_stop = dom_meta%vt_time
    CALL output_variable_metadata(var_meta_out)
    WRITE (domfile_out) press_zeta
    RETURN
  END SUBROUTINE p2z_compute_weights
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE p2z_interp_state_variables

    ! This subroutine will interpolate the basic state variables 
    ! temperature, relative_humidity, u wind component, and v wind component
    ! to the zeta surfaces.  Note that the pressure on zeta levels has
    ! already been done in the compute_weights routine.  

    IMPLICIT NONE
    INTEGER :: i,j,k
    ! Allocate the arrays for the state variables
    ALLOCATE (temp_zeta(nx,ny,output_nz))
    ALLOCATE (uwind_zeta(nx,ny,output_nz))
    ALLOCATE (vwind_zeta(nx,ny,output_nz))
    ALLOCATE (rh_zeta(nx,ny,output_nz))
    ALLOCATE (dum3d(nx,ny,input_nz))
    ALLOCATE (dum3ds(nx,ny,output_nz))
  
    ! First, do the temperature and RH so we have those available to 
    ! compute SFC pressure if needed.

    PRINT '(A,A)', 'P2Z_INTERP_STATE_VARIABLES: Interpolating ', t_id
    CALL get_variable(setup_info%input_prefix,t_id,dom_meta%id, setup_info%current_date,status)
    IF (status.NE.0) THEN
      PRINT '(A)', 'Temperature data not found using ' // t_id
      STOP 'no_temperature_data'
    ENDIF
    IF (var_info%h_stagger_index .NE. n_ind) THEN
      PRINT *,'Temperature appears to be staggered.  Please re-process the'
      PRINT *,'grib-prep output with the newer version of hinterp.'
      STOP
    ENDIF
    dum3d = real_array(:,:,:,1,1)
    DEALLOCATE(real_array)
    num_processed = num_processed+1
    processed_var_list(num_processed) = t_id
    var_meta_out = var_info
    CALL linear_interpolate(dum3d,temp_zeta)
    IF((setup_info%output_vars .EQ. 1).OR.(setup_info%output_vars .EQ. 2))THEN
      CALL arakawa_c_n2t(temp_zeta,nx,ny,output_nz,dum3ds)
      temp_zeta = dum3ds 
      var_meta_out%h_stagger_index = t_ind
    ELSE
      var_meta_out%h_stagger_index = n_ind
    ENDIF
    IF ( (setup_info%output_vars .EQ. 2).OR.(setup_info%output_vars.EQ.4))THEN
      var_meta_out%dim_val(3) = output_nz
      var_meta_out%stop_index(3) = output_nz
      CALL output_variable_metadata(var_meta_out)
      WRITE(domfile_out) temp_zeta
    ENDIF

    ! Relative Humidity.  Later, we will add a check so if RH not present,
    ! we can look for qv or mr and compute RH.

    PRINT '(A,A)', 'P2Z_INTERP_STATE_VARIABLES: Interpolating ' , rh_id
    CALL get_variable(setup_info%input_prefix,rh_id,dom_meta%id, setup_info%current_date,status)
    IF (status.NE.0) THEN
      PRINT '(A)', 'Relative humidity data not found using ' // rh_id
      STOP 'no_rh_data'
    ENDIF
    IF (var_info%h_stagger_index .NE. n_ind) THEN
      PRINT *,'Humidity appears to be staggered.  Please re-process the'
      PRINT *,'grib-prep output with the newer version of hinterp.'
      STOP
    ENDIF
    dum3d = real_array(:,:,:,1,1)
    DEALLOCATE(real_array)
    num_processed = num_processed+1
    processed_var_list(num_processed) = rh_id
    var_meta_out = var_info
    CALL linear_interpolate(dum3d,rh_zeta)
    IF((setup_info%output_vars .EQ. 1).OR.(setup_info%output_vars .EQ. 2))THEN 
      CALL arakawa_c_n2t(rh_zeta,nx,ny,output_nz,dum3ds)
      rh_zeta = dum3ds
      var_meta_out%h_stagger_index = t_ind 
    ELSE
      var_meta_out%h_stagger_index = n_ind
    ENDIF
    WHERE (rh_zeta .LT. 0.) rh_zeta = 0.1  ! To be safe
    WHERE (rh_zeta .GT. 100.) rh_zeta = 100.  
    IF ( (setup_info%output_vars .EQ.2).OR.(setup_info%output_vars .EQ.4))THEN
      var_meta_out%dim_val(3) = output_nz
      var_meta_out%stop_index(3) = output_nz
      CALL output_variable_metadata(var_meta_out)
      WRITE(domfile_out) rh_zeta
    ENDIF

    ! U wind component
    PRINT '(A,A)', 'P2Z_INTERP_STATE_VARIABLES: Interpolating ' , u_id
    CALL get_variable(setup_info%input_prefix,u_id,dom_meta%id, setup_info%current_date,status)
    IF (status.NE.0) THEN
      PRINT '(A)', 'U-component wind data not found using ' // u_id
      STOP 'no_u_data'
    ENDIF
    IF (var_info%h_stagger_index .NE. n_ind) THEN
      PRINT *,'U wind appears to be staggered.  Please re-process the'
      PRINT *,'grib-prep output with the newer version of hinterp.'
      STOP
    ENDIF
    dum3d = real_array(:,:,:,1,1)
    DEALLOCATE(real_array)
    num_processed = num_processed+1
    processed_var_list(num_processed) = u_id
    var_meta_out = var_info 
    CALL linear_interpolate(dum3d,uwind_zeta)
    var_meta_out%dim_val(3) = output_nz
    var_meta_out%stop_index(3) = output_nz
    IF((setup_info%output_vars .EQ. 1).OR.(setup_info%output_vars .EQ. 2))THEN
      CALL arakawa_c_n2u(uwind_zeta,nx,ny,output_nz,dum3ds)
      uwind_zeta = dum3ds
      var_meta_out%h_stagger_index = u_ind
    ELSE
      var_meta_out%h_stagger_index = n_ind
    ENDIF
    CALL output_variable_metadata(var_meta_out)
    WRITE (domfile_out) uwind_zeta

    ! V wind component
    PRINT '(A,A)', 'P2Z_INTERP_STATE_VARIABLES: Interpolating ' , v_id
    CALL get_variable(setup_info%input_prefix,v_id,dom_meta%id, setup_info%current_date,status)
    IF (status.NE.0) THEN
      PRINT '(A)', 'V-component wind data not found using ' // v_id
      STOP 'no_v_data'
    ENDIF
    IF (var_info%h_stagger_index .NE. n_ind) THEN
      PRINT *,'V wind appears to be staggered.  Please re-process the'
      PRINT *,'grib-prep output with the newer version of hinterp.'
      STOP
    ENDIF
    dum3d = real_array(:,:,:,1,1)
    DEALLOCATE(real_array)
    num_processed = num_processed+1
    processed_var_list(num_processed) = v_id
    CALL linear_interpolate(dum3d,vwind_zeta)
    var_meta_out = var_info
    var_meta_out%dim_val(3) = output_nz
    var_meta_out%stop_index(3) = output_nz
    IF((setup_info%output_vars .EQ. 1).OR.(setup_info%output_vars .EQ. 2))THEN
      CALL arakawa_c_n2v(vwind_zeta,nx,ny,output_nz,dum3ds)
      vwind_zeta = dum3ds
      var_meta_out%h_stagger_index = v_ind
    ELSE
      var_meta_out%h_stagger_index = n_ind
    ENDIF
    CALL output_variable_metadata(var_meta_out)
    WRITE(domfile_out) vwind_zeta

    IF (setup_info%verbose) THEN
      PRINT '(A)', 'P2Z_INTERP_STATE_VARIABLES: Diagnostic Print:'
      PRINT '(A)', '---------------------------------------------'
      PRINT '(A)', 'LEV   ZETA   PRESS    TEMP   RH     U     V'
      PRINT '(A)', '---------------------------------------------'
      DO k=1,output_nz
        PRINT '(I3,2F9.1,4F6.1)',k,output_levels(k,zstag_half_index), &
           press_zeta(nx/2,ny/2,k),temp_zeta(nx/2,ny/2,k),rh_zeta(nx/2,ny/2,k), &
           uwind_zeta(nx/2,ny/2,k), vwind_zeta(nx/2,ny/2,k)
      ENDDO
    ENDIF
    DEALLOCATE(dum3d)
    DEALLOCATE(dum3ds)
  END SUBROUTINE p2z_interp_state_variables
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE p2z_make_wrf_variables
  
    ! Derives any variables needed for initialization of WRF.  For now, the WRF 
    ! prototype is the Runge-Kutta version and we are supplying non-coupled
    ! variables so WRF can do its own balance just prior to initialization.

    INTEGER 	:: i,j,k
    REAL	:: tv_k, tv_sum_k, tv_mean_k, z_layer_top, thick_m, ptop_pa

    ALLOCATE ( press_sfc (nx,ny) ) ! Surface pressure field 
    ALLOCATE ( rho_zeta (nx,ny,output_nz) ) ! density
    ALLOCATE ( theta_zeta (nx,ny,output_nz ) ) ! potential temperature
    ALLOCATE ( qv_zeta (nx,ny,output_nz) )  ! water vapor mixing ratio

    diag_k_loop: DO k = 1, output_nz
      diag_j_loop: DO j = 1, ny
        diag_i_loop: DO i = 1, nx
  
          rho_zeta(i,j,k) =compute_density(temp_zeta(i,j,k),press_zeta(i,j,k))
          qv_zeta(i,j,k)=compute_vapor_mixing_ratio(temp_zeta(i,j,k), &
               press_zeta(i,j,k),rh_zeta(i,j,k),setup_info%rh_wrt_liquid)
          theta_zeta(i,j,k) = compute_theta(temp_zeta(i,j,k),press_zeta(i,j,k))
    
        ENDDO diag_i_loop
      ENDDO diag_j_loop
    ENDDO diag_k_loop
    

   ! Compute surface pressure using the third zeta level(arbitrary) as the top pressure
   ! and using the temperature and mixing ratio at the three lowest levels
   ! to compute and average Tv.
          
   diag_sfcp_j_loop: DO j = 1, ny
     diag_sfcp_i_loop: DO i= 1, nx
       ptop_pa = press_zeta(i,j,3)
       z_layer_top = output_levels(3,zstag_half_index)/ztop*(ztop-terrain_hgt_t(i,j)) + &
                          terrain_hgt_t(i,j)
       thick_m = z_layer_top - terrain_hgt_t(i,j)
       tv_sum_k = 0.
       mean_tv_loop: DO k=1,3
        tv_k =  compute_virtual_temp(temp_zeta(i,j,k),qv_zeta(i,j,k))
         tv_sum_k = tv_sum_k + tv_k
       ENDDO mean_tv_loop
       tv_mean_k = tv_sum_k / 3.
       press_sfc(i,j)= compute_lower_pressure(ptop_pa, thick_m, tv_mean_k)
     ENDDO diag_sfcp_i_loop
   ENDDO diag_sfcp_j_loop

   ! Output the variables

   ! Output THETA
   var_meta_out%name = theta_id
   var_meta_out%units = 'K'
   var_meta_out%description = 'Pot. temp on zeta points'
   var_meta_out%domain_id = dom_meta%id
   var_meta_out%ndim = 3
   var_meta_out%dim_val(1) = nx
   var_meta_out%dim_val(2) = ny
   var_meta_out%dim_val(3) = output_nz
   var_meta_out%dim_desc(1) = 'E-W'
   var_meta_out%dim_desc(2) = 'N-S'
   var_meta_out%dim_desc(3) = 'VERT'
   var_meta_out%start_index(1) = 1
   var_meta_out%start_index(2) = 1
   var_meta_out%start_index(3) = 1
   var_meta_out%stop_index(1) = nx
   var_meta_out%stop_index(2) = ny
   var_meta_out%stop_index(3) = output_nz
   IF((setup_info%output_vars .EQ. 1).OR.(setup_info%output_vars .EQ. 2))THEN 
     var_meta_out%h_stagger_index = t_ind
   ELSE
     var_meta_out%h_stagger_index = n_ind
   ENDIF
   var_meta_out%v_stagger_index = zstag_half_index
   var_meta_out%array_order = '+X+Y+Z'
   var_meta_out%field_type = 'REAL'
   var_meta_out%field_source_prog = 'SI'
   var_meta_out%source_desc = 'Derived from vertically interpolated temp/press'
   var_meta_out%field_time_type = 'INSTANT'
   var_meta_out%vt_date_start = dom_meta%vt_date
   var_meta_out%vt_time_start = dom_meta%vt_time
   var_meta_out%vt_date_stop = dom_meta%vt_date
   var_meta_out%vt_time_stop = dom_meta%vt_time
   CALL output_variable_metadata(var_meta_out)
   WRITE (domfile_out) theta_zeta
   num_processed = num_processed + 1
   processed_var_list(num_processed) = theta_id

   ! Output RHO
   var_meta_out%name = rho_id
   var_meta_out%units = 'kg m{-3}'
   var_meta_out%description = 'Density of dry air on zeta points'
   var_meta_out%domain_id = dom_meta%id
   var_meta_out%ndim = 3
   var_meta_out%dim_val(1) = nx
   var_meta_out%dim_val(2) = ny
   var_meta_out%dim_val(3) = output_nz
   var_meta_out%dim_desc(1) = 'E-W'
   var_meta_out%dim_desc(2) = 'N-S'
   var_meta_out%dim_desc(3) = 'VERT'
   var_meta_out%start_index(1) = 1
   var_meta_out%start_index(2) = 1
   var_meta_out%start_index(3) = 1
   var_meta_out%stop_index(1) = nx
   var_meta_out%stop_index(2) = ny
   var_meta_out%stop_index(3) = output_nz
   IF((setup_info%output_vars .EQ. 1).OR.(setup_info%output_vars .EQ. 2))THEN
     var_meta_out%h_stagger_index = t_ind
   ELSE
     var_meta_out%h_stagger_index = n_ind
   ENDIF
   var_meta_out%v_stagger_index = zstag_half_index
   var_meta_out%array_order = '+X+Y+Z'
   var_meta_out%field_type = 'REAL'
   var_meta_out%field_source_prog = 'SI'
   var_meta_out%source_desc = 'Derived from vertically interpolated temp/press'
   var_meta_out%field_time_type = 'INSTANT'
   var_meta_out%vt_date_start = dom_meta%vt_date
   var_meta_out%vt_time_start = dom_meta%vt_time
   var_meta_out%vt_date_stop = dom_meta%vt_date
   var_meta_out%vt_time_stop = dom_meta%vt_time
   CALL output_variable_metadata(var_meta_out)
   WRITE (domfile_out) rho_zeta
   num_processed = num_processed + 1
   processed_var_list(num_processed) = rho_id

   ! Output mixing ratio
   var_meta_out%name = qvapor_id
   var_meta_out%units = 'kg kg{-1}'
   var_meta_out%description = 'Water vapor mixing ratio on zeta points'
   var_meta_out%domain_id = dom_meta%id
   var_meta_out%ndim = 3
   var_meta_out%dim_val(1) = nx
   var_meta_out%dim_val(2) = ny
   var_meta_out%dim_val(3) = output_nz
   var_meta_out%dim_desc(1) = 'E-W'
   var_meta_out%dim_desc(2) = 'N-S'
   var_meta_out%dim_desc(3) = 'VERT'
   var_meta_out%start_index(1) = 1
   var_meta_out%start_index(2) = 1
   var_meta_out%start_index(3) = 1
   var_meta_out%stop_index(1) = nx
   var_meta_out%stop_index(2) = ny
   var_meta_out%stop_index(3) = output_nz
   IF((setup_info%output_vars .EQ. 1).OR.(setup_info%output_vars .EQ. 2))THEN
     var_meta_out%h_stagger_index = t_ind
   ELSE 
     var_meta_out%h_stagger_index = n_ind
   ENDIF
   var_meta_out%v_stagger_index = zstag_half_index
   var_meta_out%array_order = '+X+Y+Z'
   var_meta_out%field_type = 'REAL'
   var_meta_out%field_source_prog = 'SI'
   var_meta_out%source_desc = 'Derived from vertically interpolated temp/press/RH'
   var_meta_out%field_time_type = 'INSTANT'
   var_meta_out%vt_date_start = dom_meta%vt_date
   var_meta_out%vt_time_start = dom_meta%vt_time
   var_meta_out%vt_date_stop = dom_meta%vt_date
   var_meta_out%vt_time_stop = dom_meta%vt_time
   CALL output_variable_metadata(var_meta_out)
   WRITE (domfile_out) qv_zeta
   num_processed = num_processed + 1
   processed_var_list(num_processed) = qvapor_id

   ! Output surface pressure
   var_meta_out%name = press_sfc_id
   var_meta_out%units = 'Pa'
   var_meta_out%description = 'Surface pressure on WRF terrain'
   var_meta_out%domain_id = dom_meta%id
   var_meta_out%ndim = 2
   var_meta_out%dim_val(1) = nx
   var_meta_out%dim_val(2) = ny
   var_meta_out%dim_val(3) = 0            
   var_meta_out%dim_desc(1) = 'E-W'
   var_meta_out%dim_desc(2) = 'N-S'
   var_meta_out%dim_desc(3) = '   '
   var_meta_out%start_index(1) = 1
   var_meta_out%start_index(2) = 1
   var_meta_out%start_index(3) = 0
   var_meta_out%stop_index(1) = nx
   var_meta_out%stop_index(2) = ny
   var_meta_out%stop_index(3) = 0  
   IF((setup_info%output_vars .EQ. 1).OR.(setup_info%output_vars .EQ. 2))THEN
     ALLOCATE(dum2ds(nx,ny))
     var_meta_out%h_stagger_index = t_ind
     CALL arakawa_c_n2t(press_sfc,nx,ny,1,dum2ds) 
     press_sfc= dum2ds
     DEALLOCATE(dum2ds)
   ELSE
     var_meta_out%h_stagger_index = n_ind
   ENDIF
   var_meta_out%v_stagger_index = 0                     
   var_meta_out%array_order = '+X+Y+Z'
   var_meta_out%field_type = 'REAL'
   var_meta_out%field_source_prog = 'SI'
   var_meta_out%source_desc = 'Derived '
   var_meta_out%field_time_type = 'INSTANT'
   var_meta_out%vt_date_start = dom_meta%vt_date
   var_meta_out%vt_time_start = dom_meta%vt_time
   var_meta_out%vt_date_stop = dom_meta%vt_date
   var_meta_out%vt_time_stop = dom_meta%vt_time
   CALL output_variable_metadata(var_meta_out)
   WRITE (domfile_out) press_sfc
   num_processed = num_processed + 1
   processed_var_list(num_processed) = press_sfc_id



   IF (setup_info%verbose) THEN
     PRINT '(A)', 'Diagnostic printout from p2z_make_wrf_variables...'
     PRINT '(A)', '---------------------------'
     PRINT '(A)', ' K   RHO    THETA      QV'
     PRINT '(A)', '--------------------------'
     DO k=1,output_nz
       PRINT '(I3,1x,F6.4,2x,F5.1,2x,F8.6)',k,rho_zeta(nx/2,ny/2,k), &
           theta_zeta(nx/2,ny/2,k), qv_zeta(nx/2,ny/2,k)
     ENDDO
     PRINT '(A,F8.1)', 'Surface Pressure = ' , press_sfc(nx/2,ny/2)
   ENDIF

   DEALLOCATE(press_sfc)
   DEALLOCATE(qv_zeta)
   DEALLOCATE(theta_zeta)
   DEALLOCATE(rho_zeta)
   DEALLOCATE(uwind_zeta)
   DEALLOCATE(vwind_zeta)
   DEALLOCATE(press_zeta)
   DEALLOCATE(temp_zeta)
   DEALLOCATE(rh_zeta)

  END SUBROUTINE p2z_make_wrf_variables
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE p2z_cleanup
 
  DEALLOCATE(weight_top)
  DEALLOCATE(trap_top)
  DEALLOCATE(trap_bot)
  DEALLOCATE(height)
  DEALLOCATE(press_msl)
  IF (setup_info%current_date .EQ. setup_info%ending_date) THEN
    DEALLOCATE(terrain_hgt_n)
    DEALLOCATE(terrain_hgt_t)
    DEALLOCATE(dzetadz)
    DEALLOCATE(dzdzeta)
!     DEALLOCATE(dzdx)
!     DEALLOCATE(dzdy)
  ENDIF 
   CLOSE(domfile_out)
  END SUBROUTINE p2z_cleanup

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE linear_interpolate(data_in, data_out)
  
  ! This subroutine does linear vertical interpolation using the 
  !trap_top and weight_top arrays computed in the p2z_compute_weights
  ! subroutine.
  
    IMPLICIT NONE
    INTEGER :: i,j,k
    REAL, INTENT(IN)  :: data_in(:,:,:)
    REAL, INTENT(INOUT) :: data_out(:,:,:)
    REAL :: delta_z
    vinterp_z_loop : DO k=1,output_nz
      vinterp_y_loop: DO j=1,ny
        vinterp_x_loop: DO i=1,nx

          data_out(i,j,k) =data_in(i,j,trap_top(i,j,k))* &
                         weight_top(i,j,k) + &
                         data_in(i,j,trap_bot(i,j,k))*&
                         (1.-weight_top(i,j,k))
          IF (trap_top(i,j,k).EQ.1) THEN     
            IF (TRIM(var_info%name) .EQ. 'T') THEN
              ! This is temperature and the output level is below
              ! The input model surface value, so adjust by
              ! extrapolating using a standard lapse rate.
              !PRINT '(A,2I4)', 'Extrapolating T at ',i,j
              !PRINT '(A,F6.1)', '...Sfc value = ', data_out(i,j,k)
              delta_z = height(i,j,1) - &
                      (output_levels(k,zstag_half_index)/ztop * &
                      (ztop-terrain_hgt_n(i,j)) + &
                      terrain_hgt_n(i,j))
              !PRINT '(A,F6.1)', '...Delta-Z (m) = ', delta_z
              data_out(i,j,k) = data_out(i,j,k) + delta_z * dTdz 
              PRINT '(A,F6.1)', '...Extrapolated Value = ', data_out(i,j,k)
            ENDIF 
          ENDIF
        ENDDO vinterp_x_loop
      ENDDO vinterp_y_loop
    ENDDO vinterp_z_loop
    IF (setup_info%verbose) THEN
       PRINT '(A,F10.5,F10.5)', 'LINEAR_INTERPOLATE: Min/Max Input Values: ',&
            MINVAL(data_in),MAXVAL(data_in)
       PRINT '(A,F10.5,F10.5)', 'LINEAR_INTERPOLATE: Min/Max Output Values: ',&
            MINVAL(data_out),MAXVAL(data_out)
    ENDIF
  END SUBROUTINE linear_interpolate
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE output_zeta_static
  
    ! Outputs required "static" fields that are specific to the zeta coordinate
    ! version of WRF

    IMPLICIT NONE
    INTEGER :: status

    ! Write half zeta levels

    var_meta_out%name = zetahalf_id
    var_meta_out%units = 'm {scaled}'
    var_meta_out%description = 'Zeta value on vertical half levels (mass)'
    var_meta_out%domain_id = dom_meta%id
    var_meta_out%ndim=1
    var_meta_out%dim_val(1) = output_nz
    var_meta_out%dim_val(2) = 0
    var_meta_out%dim_val(3) = 0
    var_meta_out%dim_val(4) = 0
    var_meta_out%dim_val(5) = 0
    var_meta_out%dim_desc(1) = 'VERT'
    var_meta_out%dim_desc(2) = '    '
    var_meta_out%dim_desc(3) = '    '
    var_meta_out%dim_desc(4) = '    '  
    var_meta_out%dim_desc(5) = '    '
    var_meta_out%start_index(1) = 1 
    var_meta_out%start_index(2) = 0 
    var_meta_out%start_index(3) = 0 
    var_meta_out%start_index(4) = 0 
    var_meta_out%start_index(5) = 0
    var_meta_out%stop_index(1) = output_nz
    var_meta_out%stop_index(2) = 0
    var_meta_out%stop_index(3) = 0
    var_meta_out%stop_index(4) = 0
    var_meta_out%stop_index(5) = 0
    var_meta_out%h_stagger_index = 0
    var_meta_out%v_stagger_index = zstag_half_index
    var_meta_out%array_order = '+Z      '
    var_meta_out%field_type = 'REAL'
    var_meta_out%field_source_prog = 'SI'
    var_meta_out%source_desc = 'User defined levels generated by WRFSI/vinterp'
    var_meta_out%field_time_type = 'CONSTANT'
    var_meta_out%vt_date_start = dom_meta%vt_date
    var_meta_out%vt_time_start = dom_meta%vt_time
    var_meta_out%vt_date_stop = dom_meta%vt_date
    var_meta_out%vt_time_stop = dom_meta%vt_time
    CALL output_variable_metadata(var_meta_out)
    WRITE(domfile_out) output_levels(:,zstag_half_index)
    num_processed = num_processed + 1
    processed_var_list(num_processed) = zetahalf_id

    ! Output full zeta levels
    var_meta_out%name = zetafull_id
    var_meta_out%description = 'Zeta value on full half levels (w)'
    var_meta_out%v_stagger_index = zstag_full_index
    CALL output_variable_metadata(var_meta_out)
    WRITE(domfile_out) output_levels(:,zstag_full_index)
    num_processed = num_processed + 1
    processed_var_list(num_processed) = zetafull_id

    ! Write dzetadz and dzdzeta

    var_meta_out%name = dzetadz_id
    var_meta_out%units = 'm m{-1}'
    var_meta_out%description = 'dzeta/dz on mass grid'
    var_meta_out%domain_id = dom_meta%id
    var_meta_out%ndim=2
    var_meta_out%dim_val(1) = nx
    var_meta_out%dim_val(2) = ny
    var_meta_out%dim_val(3) = 0
    var_meta_out%dim_val(4) = 0
    var_meta_out%dim_val(5) = 0
    var_meta_out%dim_desc(1) = 'W-E '
    var_meta_out%dim_desc(2) = 'S-N '
    var_meta_out%dim_desc(3) = '    '
    var_meta_out%dim_desc(4) = '    '  
    var_meta_out%dim_desc(5) = '    '
    var_meta_out%start_index(1) = 1 
    var_meta_out%start_index(2) = 1 
    var_meta_out%start_index(3) = 0 
    var_meta_out%start_index(4) = 0 
    var_meta_out%start_index(5) = 0
    var_meta_out%stop_index(1) = nx
    var_meta_out%stop_index(2) = ny
    var_meta_out%stop_index(3) = 0
    var_meta_out%stop_index(4) = 0
    var_meta_out%stop_index(5) = 0
    var_meta_out%h_stagger_index = t_ind
    var_meta_out%v_stagger_index = 0
    var_meta_out%array_order = '+X+Y    '
    var_meta_out%field_type = 'REAL'
    var_meta_out%field_source_prog = 'SI'
    var_meta_out%source_desc = 'Generated by WRFSI/vinterp'
    var_meta_out%field_time_type = 'CONSTANT'
    var_meta_out%vt_date_start = dom_meta%vt_date
    var_meta_out%vt_time_start = dom_meta%vt_time
    var_meta_out%vt_date_stop = dom_meta%vt_date
    var_meta_out%vt_time_stop = dom_meta%vt_time
    CALL output_variable_metadata(var_meta_out)
    WRITE(domfile_out) dzetadz
    num_processed = num_processed + 1
    processed_var_list(num_processed) = dzetadz_id
   
    var_meta_out%name = dzdzeta_id
    var_meta_out%description = 'dz/dzeta on mass grid'
    CALL output_variable_metadata(var_meta_out)
    WRITE(domfile_out) dzdzeta
    num_processed = num_processed + 1
    processed_var_list(num_processed) = dzdzeta_id

    ! Output dzdx and dzdy -COMMENTED OUT ON 25 MAY 01 - BLS 
    ! var_meta_out%name = dzdx_id
    ! var_meta_out%units = 'm m{-1}'
    ! var_meta_out%description = 'dz/dx on u grid'
    ! var_meta_out%domain_id = dom_meta%id
    ! var_meta_out%ndim=3
    ! var_meta_out%dim_val(1) = nx
    ! var_meta_out%dim_val(2) = ny
    ! var_meta_out%dim_val(3) = output_nz
    ! var_meta_out%dim_val(4) = 0
    ! var_meta_out%dim_val(5) = 0
    ! var_meta_out%dim_desc(1) = 'W-E '
    ! var_meta_out%dim_desc(2) = 'S-N '
    ! var_meta_out%dim_desc(3) = 'VERT'
    ! var_meta_out%dim_desc(4) = '    '  
    ! var_meta_out%dim_desc(5) = '    '
    ! var_meta_out%start_index(1) = 1 
    ! var_meta_out%start_index(2) = 1 
    ! var_meta_out%start_index(3) = 1 
    ! var_meta_out%start_index(4) = 0 
    ! var_meta_out%start_index(5) = 0
    ! var_meta_out%stop_index(1) = nx
    ! var_meta_out%stop_index(2) = ny
    ! var_meta_out%stop_index(3) = output_nz
    ! var_meta_out%stop_index(4) = 0
    ! var_meta_out%stop_index(5) = 0
    ! var_meta_out%h_stagger_index = u_ind
    ! var_meta_out%v_stagger_index = zstag_half_index
    ! var_meta_out%array_order = '+X+Y+Z  '
    ! var_meta_out%field_type = 'REAL'
    ! var_meta_out%field_source_prog = 'SI'
    ! var_meta_out%source_desc = 'Generated by WRFSI/vinterp'
    ! var_meta_out%field_time_type = 'CONSTANT'
    ! var_meta_out%vt_date_start = dom_meta%vt_date
    ! var_meta_out%vt_time_start = dom_meta%vt_time
    ! var_meta_out%vt_date_stop = dom_meta%vt_date
    ! var_meta_out%vt_time_stop = dom_meta%vt_time
    ! CALL output_variable_metadata(var_meta_out)
    ! WRITE(domfile_out) dzdx
    ! num_processed = num_processed + 1
    ! processed_var_list(num_processed) = dzdx_id

    ! var_meta_out%name = dzdy_id
    ! var_meta_out%description = 'dz/dy on v grid'
    ! var_meta_out%h_stagger_index = v_ind
    ! CALL output_variable_metadata(var_meta_out)
    ! WRITE(domfile_out) dzdy
    ! num_processed = num_processed + 1
    ! processed_var_list(num_processed) = dzdy_id

  END SUBROUTINE output_zeta_static

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  SUBROUTINE p2z_process_other_variables
  ! This routine scans the file looking for any variables that have not been
  ! processed yet.  It looks at the dimensions, etc. to determine if the 
  ! variable needs to be vertically interpolated and/or destaggered into
  ! multiple output fields.  If not, it just passes them on, metadata included
  ! into the new output file.  

    IMPLICIT NONE
    INTEGER :: v,s
    CHARACTER (LEN=1) :: stag_string
    LOGICAL :: processed
    !  First, reset the file pointer to the beginning of the input file by
    ! calling get_domain_metadata.  The API should have something like 
    ! get_first_variable to eventually replace this kludge method.

    ALLOCATE (dum3d (nx,ny,output_nz))
    CALL get_domain_metadata(dom_meta%id, setup_info%current_date,setup_info%input_prefix)

    PRINT *, 'Here is what we have processed so far: ', processed_var_list
    ! Now, we can loop through calling read_next_variable and using the
    ! status to determine if we have reached the end of the file (module_wrf_io 
    ! sets the status=2 when EOF is reached)

    status = 0                          
    CALL read_next_variable
    filescan_loop: DO WHILE (status .EQ. 0)
   
      ! Compare the variable name with those in the processed_var list
      processed = .false.
      check_var_loop: DO v = 1,num_processed
        IF (var_info%name .EQ. processed_var_list(v))THEN
          processed = .true.  
          EXIT check_var_loop    
        ENDIF
      ENDDO check_var_loop
      IF (processed) THEN
        CALL read_next_variable
        CYCLE filescan_loop
      ENDIF
      PRINT '(2A)', 'Other Variable: ', var_info%name
      ! We have a variable that has not been processed, so figure out
      ! what we need to do with it.

      IF ((var_info%ndim .EQ. 3).AND.(var_info%array_order.EQ.'+X+Y+Z  '))THEN
  
        ! Standard 3D variable on 1 grid, needs vertical interpolation
        CALL linear_interpolate(real_array(:,:,:,1,1),dum3d)
        var_meta_out = var_info
        var_meta_out%dim_val(3) = output_nz
        var_meta_out%stop_index(3) = output_nz
        IF ((var_meta_out%h_stagger_index .EQ. n_ind).AND. & 
            (setup_info%output_vars .LE. 2) )THEN
          ALLOCATE(dum3ds(nx,ny,output_nz))
          IF ( (var_meta_out%name(1:1) .NE. 'U') .AND. &
               (var_meta_out%name(1:1) .NE. 'V')) THEN
             CALL arakawa_c_n2t(dum3d,nx,ny,output_nz,dum3ds)
             var_meta_out%h_stagger_index = t_ind
          ELSE IF (var_meta_out%name(1:1) .EQ. 'U') THEN
             CALL arakawa_c_n2u(dum3d,nx,ny,output_nz,dum3ds)
             var_meta_out%h_stagger_index = u_ind
          ELSE IF (var_meta_out%name(1:1) .EQ. 'V') THEN
             CALL arakawa_c_n2v(dum3d,nx,ny,output_nz,dum3ds)
             var_meta_out%h_stagger_index = v_ind
          ENDIF
          dum3d = dum3ds
          DEALLOCATE(dum3ds)
        ENDIF
        CALL output_variable_metadata(var_meta_out)
        WRITE (domfile_out) dum3d

      ELSE IF ((var_info%ndim .EQ. 4).AND. &
               (var_info%array_order.EQ.'+X+Y+Z+S'))THEN
    
        ! Standard 3D variable on all grids, needs vertical interpolation
        stag_interp_loop: DO s = 1,var_info%dim_val(4)
          CALL linear_interpolate(real_array(:,:,:,s,1),dum3d)
          var_meta_out = var_info
          ! Replace the last character of the variable name with stagger index
          WRITE (stag_string, '(I1)') s
          var_meta_out%name(8:8) = stag_string
          var_meta_out%description = TRIM(var_meta_out%description) // &
              ' on stagger number ' // stag_string
          var_meta_out%ndim = 3
          var_meta_out%dim_val(3) = output_nz
          var_meta_out%stop_index(3) = output_nz
          var_meta_out%dim_val(4) = 0
          var_meta_out%start_index(4) = 0
          var_meta_out%stop_index(4) = 0
          var_meta_out%h_stagger_index = s
          var_meta_out%array_order = '+X+Y+Z  '
          var_meta_out%dim_desc(4) = '    '
          CALL output_variable_metadata(var_meta_out)
          WRITE (domfile_out) dum3d 
        ENDDO stag_interp_loop

      ELSE IF ((var_info%ndim .EQ. 3).AND.(var_info%array_order .EQ. '+X+Y+S  '))THEN
        ! Staggered 2D variable
        stag_output_loop: DO s=1,var_info%dim_val(3)
          var_meta_out = var_info
  
          ! Replace the last character of the variable name with stagger index

          WRITE (stag_string, '(I1)') s
          var_meta_out%name(8:8) = stag_string
          var_meta_out%description = TRIM(var_meta_out%description) // &
              ' on stagger number ' // stag_string
          var_meta_out%ndim = 2
          var_meta_out%dim_val(3) = 0               
          var_meta_out%stop_index(3) = 0             
          var_meta_out%h_stagger_index = s
          var_meta_out%array_order = '+X+Y    '
          var_meta_out%dim_desc(3) = '    '
          CALL output_variable_metadata(var_meta_out)
          WRITE (domfile_out) real_array(:,:,s,1,1)
        ENDDO stag_output_loop
      ELSE IF ( var_info%ndim .EQ. 2) THEN
        ALLOCATE(dum2d(nx,ny))  
        dum2d = real_array(:,:,1,1,1)
        ! Pass this variable right on through
        var_meta_out = var_info
  
        IF ((var_meta_out%h_stagger_index .EQ. n_ind).AND. &
            (setup_info%output_vars .LE. 2) ) THEN
          ALLOCATE (dum2ds(nx,ny))
          IF ( (var_meta_out%name(1:1) .NE. 'U') .AND. &
               (var_meta_out%name(1:1) .NE. 'V')) THEN
             CALL arakawa_c_n2t(dum2d,nx,ny,1,dum2ds)
             var_meta_out%h_stagger_index = t_ind
          ELSE IF (var_meta_out%name(1:1) .EQ. 'U') THEN
             CALL arakawa_c_n2u(dum2d,nx,ny,1,dum2ds)
             var_meta_out%h_stagger_index = u_ind
          ELSE IF (var_meta_out%name(1:1) .EQ. 'V') THEN
             CALL arakawa_c_n2v(dum2d,nx,ny,1,dum2ds)
             var_meta_out%h_stagger_index = v_ind
          ENDIF
          dum2d = dum2ds
          DEALLOCATE (dum2ds)
        ENDIF
        CALL output_variable_metadata(var_meta_out)
        WRITE (domfile_out) dum2d
        DEALLOCATE(dum2d)
      ELSE IF ( var_info%ndim .EQ. 1) THEN

        ! Pass this variable right on through
        var_meta_out = var_info
        CALL output_variable_metadata(var_meta_out)
        WRITE (domfile_out) real_array(:,1,1,1,1)

      ELSE IF (var_info%ndim .EQ. 0) THEN

         ! Pass this variable right on through
         var_meta_out = var_info
         CALL output_variable_metadata(var_meta_out)
         WRITE (domfile_out) real_array(1,1,1,1,1)

      ELSE 

         PRINT '(A)', 'Unsure how to process this variable...omitting from output:'
         PRINT '(3A,I1,2A)', 'NAME: ', var_info%name, ' NDIM: ', var_info%ndim,&
              ' ARRAY ORDER: ', var_info%array_order
      ENDIF

      CALL read_next_variable
    ENDDO  filescan_loop
    PRINT '(A)', 'P2Z_PROCESS_OTHER_VARIABLES: Reached end of file.'
    DEALLOCATE (dum3d)
    RETURN
  END SUBROUTINE p2z_process_other_variables
 
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
END MODULE vinterp_p2z

 
